c iridreg.for, version number can be found at the end of this comment.
c-----------------------------------------------------------------------
C
C This file contains the D-region models of Friedrich and Torkar (2001)
C (subroutine F00 and block data statement).
C The subroutine DRegion of Danilov et al. (1995) was moved to IRIFUN,
C because of consistent problems of some Fortran compilers wit the long
C BLOCK DATA statement. 
C
C !!!USER NOTE!!! If your compiler has problems with this subroutine you 
C can compile IRI without this file. But you first have to comment out  
C the following two line in IRISUB: 
C            call F00(HEIGHT,LATI,DAYNR,XHI,F107D,EDENS,IERROR)
C            if(ierror.eq.0.or.ierror.eq.2) outf(1,kk)=edens
c
c-----------------------------------------------------------------------
C Corrections/Version Numbers:
C-Version-mm/dd/yy-description (person reporting correction)
C 2001.01 05/07/01 initial version
C 2001.02 07/11/01 new version of F00 (as provided by K. Torkar)
C 2002.01 28/10/02 replace TAB/6 blanks, PARAMETER () (D. Simpson)
C 2007.00 05/18/07 Release of IRI-2007
C 2012.00 12/29/11 Release of IRI-2012; no change in iridreg.for
C 2012.00 01/18/12 Moved subroutine DRegion (Danilov model) to IRIFUN
C
C 2020.00 08/02/22 Updated to FIRI-2018
c-----------------------------------------------------------------------
C
C
      SUBROUTINE F00(HGT,GLAT1,IDAY,ZANG,F107T,EDENS,IERROR)
C---------------------------------------------------------------------
C     PURPOSE:
C     THIS SUBROUTINE COMPUTES "FIRI" ELECTRON DENSITIES
C
C     COMMON BLOCK REQUIRED:
C       REAL EDEN,TABHE,TABLA,TABMO,TABZA,TABFL
C       COMMON/FIRCOM/EDEN(81,5,12,11,3),
C      1              TABHE(81),TABLA(5),TABMO(12),TABZA(11),TABFL(3)
C
C       ARRAY EDEN contains LOG10(tabulated electron density,
C       ordered in (height,latitude,month,zenithangle,f107)
C       Quantity      Minimum  Maximum  Number of steps
C       Height        60       140      81
C       Latitude       0        60       5
C       Month          1        12      12
C       Zenith angle   0       130      11
C       F10.7         75       200       3
C
C     PARAMETERS:
C        HGT   height in km (input, REAL)
C        GLAT1 latitude in degrees, north is positive (input, REAL)
C        IDAY  day of the year (input, INTEGER)
C        ZANG  solar zenith angle in degrees (input, REAL)
C        F107T 10.7cm flux in Ja (input, REAL)
C        EDENS model electron density in m**-3 (output, REAL)
C        IERROR  Error code (INTEGER, output)
C
C       Error code
C         0         no error
C         1         model undefined for given combination of input
C                   parameters, output is set to zero
C         2         input parameters outside valid range, output is invalid
C         3         both error conditions detected, output is zero
C
C     USAGE
C        CALL F00(HGT,GLAT1,IDAY,ZANG,F107T,EDENS,IERROR)
C
C     SUBROUTINES AND FUNCTION SUBPROGRAMS REQUIRED
C        none
C
C     Reference: Friedrich, M., Pock, C., & Torkar, K. (2018), 
C                FIRI-2018, an updated empirical model of the lower ionosphere. 
C                Journal of Geophysical Research: Space Physics, 123, 67376751. 
C                https://doi.org/10.1029/2018JA025437

C     WRITTEN BY K. TORKAR, IWF GRAZ
C     Klaus.Torkar@oeaw.ac.at
C
C     LAST MODIFICATION:  23.05.2022
C
C     VERSION: 2.0
C
C     ------------------------------------------------------------------
C
      REAL HGT,GLAT1,ZANG,F107T,EDENS,F107L
      INTEGER IDAY,IERROR
C
      PARAMETER (NHGT=81)
      PARAMETER (NLAT=5)
      PARAMETER (NMON=12)
      PARAMETER (NZEN=11)
      PARAMETER (NF10=3)
C
      REAL EDEN,TABHE,TABLA,TABMO,TABZA,TABFL
      COMMON/FIRCOM/EDEN(81,5,12,11,3),
     1              TABHE(81),TABLA(5),TABMO(12),TABZA(11),TABFL(3)
C
      INTEGER MON,I,J,L,M,ISTEPJ,I1,I2,J1,J2,K1,K2,L1,L2,M1,M2
      INTEGER TABM(12)
      REAL EDENI(2,2,2,2),EDENIJ(2,2,2),EDENIJK(2,2),EDENIJKL(2)
      REAL STEPJ,DAY1,H1,DEG1,XHI1,FLX1,EL
C
      DATA TABM/0,31,59,90,120,151,181,212,243,273,304,334/
      DATA STEPJ,ISTEPJ/15.0,15/
C
C     INDICES:
C     I=HEIGHT, J=LATITUDE, K=MONTH, L=ZANG, M=F10.7
C
C     CHECK INPUT
C
      IERROR=0
      F107L=ALOG10(MIN(1000.0,MAX(1.0,F107T)))
      IF (HGT.LT.TABHE(1).OR.HGT.GT.TABHE(NHGT).OR.
     1    GLAT1.GT.TABLA(NLAT).OR.GLAT1.LT.-TABLA(NLAT).OR.
     2    IDAY.LT.1.OR.IDAY.GT.366.OR.
     3    ZANG.LT.TABZA(1).OR.ZANG.GT.TABZA(NZEN).OR.
     4    F107L.LT.TABFL(1).OR.F107L.GT.TABFL(NF10)) IERROR=2
C
C     assume height table is in 1 km steps from 60 to 140 km
      I=MIN0(NHGT-1,IFIX(HGT)-59)
      IF(I.LT.1)I=1
      H1=HGT-TABHE(I)
      I1=I
      I2=I+1
C
C     assume latitude table is in 15 deg steps from 0 to 60 deg
      J=MAX0(1,MIN0(NLAT-1,IFIX(ABS(GLAT1))/ISTEPJ))
      DEG1=(ABS(GLAT1)-TABLA(J))/STEPJ
      J1=J
      J2=J+1
C
C     assume month table is given for each month
      MON=12
      DO WHILE (TABM(MON).GT.IDAY)
        MON=MON-1
      END DO
      DAY1=FLOAT(IDAY-TABM(MON)-15)/30.0
      IF (DAY1.LT.0.0) MON=MON-1
      IF(MON.GE.1.AND.MON.LE.11)THEN
        K1=MON
        K2=MON+1
      ELSE
        K1=12
        K2=1
      END IF
C
C     assume zenith angle table has 11 entries between 0 and 130 deg
      DO L=2,NZEN-1
        IF(ZANG.LT.TABZA(L))GOTO 1
      END DO
      L=NZEN
1     L=L-1
      L1=L
      L2=L+1
      XHI1=(ZANG-TABZA(L1))/(TABZA(L2)-TABZA(L1))
C
C     assume solar activity table has 3 entries
      F107L=MIN(TABFL(3),MAX(TABFL(1),F107L))
      IF(F107L.LT.TABFL(NF10-1))THEN
        M1=1
        M2=2
      ELSE
        M1=2
        M2=3
      END IF
      FLX1=(F107L-TABFL(M1))/(TABFL(M2)-TABFL(M1))
C
C     ADJUST SOUTHERN LATITUDES TO NORTH AND MONTH+6
C
      IF(GLAT1.LT.0.0)THEN
        K1=K1+6
        IF(K1.GT.12)K1=K1-12
        K2=K2+6
        IF(K2.GT.12)K2=K2-12
      END IF
C
C     EDEN(hgt,lat,mon,zang,f107)
C          I   J   K   L    M
C
      DO M=M1,M2
C
        MH=M+1-M1
C       INTERPOLATE IN HEIGHT I
        DO L=L1,L2
          IF(EDEN(I1,J1,K1,L,M).EQ.0.0.OR.
     1       EDEN(I2,J1,K1,L,M).EQ.0.0.OR.
     2       EDEN(I1,J2,K1,L,M).EQ.0.0.OR.
     3       EDEN(I2,J2,K1,L,M).EQ.0.0.OR.
     4       EDEN(I1,J1,K2,L,M).EQ.0.0.OR.
     5       EDEN(I2,J1,K2,L,M).EQ.0.0.OR.
     6       EDEN(I1,J2,K2,L,M).EQ.0.0.OR.
     7       EDEN(I2,J2,K2,L,M).EQ.0.0)THEN
               EDENS=0.0
               IERROR=IERROR+1
               RETURN
          END IF
          IF(HGT.LT.TABHE(1))THEN
            EDENI(1,1,L+1-L1,MH)=EDEN(I1,J1,K1,L,M)
            EDENI(2,1,L+1-L1,MH)=EDEN(I1,J2,K1,L,M)
            EDENI(1,2,L+1-L1,MH)=EDEN(I1,J1,K2,L,M)
            EDENI(2,2,L+1-L1,MH)=EDEN(I1,J2,K2,L,M)
          ELSE IF(HGT.GT.TABHE(NHGT))THEN
            EDENI(1,1,L+1-L1,MH)=EDEN(I2,J1,K1,L,M)
            EDENI(2,1,L+1-L1,MH)=EDEN(I2,J2,K1,L,M)
            EDENI(1,2,L+1-L1,MH)=EDEN(I2,J1,K2,L,M)
            EDENI(2,2,L+1-L1,MH)=EDEN(I2,J2,K2,L,M)
          ELSE
            EDENI(1,1,L+1-L1,MH)=EDEN(I1,J1,K1,L,M)+
     1        H1*(EDEN(I2,J1,K1,L,M)-EDEN(I1,J1,K1,L,M))
            EDENI(2,1,L+1-L1,MH)=EDEN(I1,J2,K1,L,M)+
     1        H1*(EDEN(I2,J2,K1,L,M)-EDEN(I1,J2,K1,L,M))
            EDENI(1,2,L+1-L1,MH)=EDEN(I1,J1,K2,L,M)+
     1        H1*(EDEN(I2,J1,K2,L,M)-EDEN(I1,J1,K2,L,M))
            EDENI(2,2,L+1-L1,MH)=EDEN(I1,J2,K2,L,M)+
     1        H1*(EDEN(I2,J2,K2,L,M)-EDEN(I1,J2,K2,L,M))
          END IF
        END DO
C
C       INTERPOLATE IN LATITUDE J
        DO L=1,2
          IF(ABS(GLAT1).GT.TABLA(NLAT))THEN
            EDENIJ(1,L,MH)=EDENI(2,1,L,MH)
            EDENIJ(2,L,MH)=EDENI(2,2,L,MH)
          ELSE
            EDENIJ(1,L,MH)=EDENI(1,1,L,MH)+
     1        DEG1*(EDENI(2,1,L,MH)-EDENI(1,1,L,MH))
            EDENIJ(2,L,MH)=EDENI(1,2,L,MH)+
     1        DEG1*(EDENI(2,2,L,MH)-EDENI(1,2,L,MH))
          END IF
        END DO
C
C       INTERPOLATE IN MONTH K
        EDENIJK(1,MH)=EDENIJ(1,1,MH)+
     &     DAY1*(EDENIJ(2,1,MH)-EDENIJ(1,1,MH))
        EDENIJK(2,MH)=EDENIJ(1,2,MH)+
     &     DAY1*(EDENIJ(2,2,MH)-EDENIJ(1,2,MH))
C
C       INTERPOLATE IN ZENITH ANGLE L
        EDENIJKL(MH)=EDENIJK(1,MH)+XHI1*(EDENIJK(2,MH)-EDENIJK(1,MH))
      END DO
C
      EL=EDENIJKL(1)+FLX1*(EDENIJKL(2)-EDENIJKL(1))
C
      EDENS=10.**EL
C
      RETURN
      END
C
C
      BLOCK DATA
C
C     PURPOSE:
C     DEFINES TABLES OF FIRI(2018) IN
C     ARRAY EDEN(height,latitude,month,zenithangle,f107)
C     Quantity      Minimum  Maximum  Number of steps
C     Height        60       140      81
C     Latitude       0        60       5
C     Month          1        12      12
C     Zenith angle   0       130      11
C     F10.7         75       200       3
C
C     WRITTEN BY K. TORKAR, IWF GRAZ
C     Klaus.Torkar@oeaw.ac.at
C
C     LAST MODIFICATION:  23.05.2022
C
C     VERSION: 2.0
C
C     ------------------------------------------------------------------
C
      PARAMETER (NHGT=81)
      PARAMETER (NLAT=5)
      PARAMETER (NMON=12)
      PARAMETER (NZEN=11)
      PARAMETER (NF10=3)
C
      REAL EDEN,TABHE,TABLA,TABMO,TABZA,TABFL
      COMMON/FIRCOM/EDEN(81,5,12,11,3),
     1              TABHE(81),TABLA(5),TABMO(12),TABZA(11),TABFL(3)
C
C      INTEGER I,L
C
C     altitudes in km
      DATA (TABHE(I),I=1,81)/
     * 60.,61.,62.,63.,64.,65.,66.,67.,68.,69.,
     * 70.,71.,72.,73.,74.,75.,76.,77.,78.,79.,
     * 80.,81.,82.,83.,84.,85.,86.,87.,88.,89.,
     * 90.,91.,92.,93.,94.,95.,96.,97.,98.,99.,
     * 100.,101.,102.,103.,104.,105.,106.,107.,108.,109.,
     * 110.,111.,112.,113.,114.,115.,116.,117.,118.,119.,
     * 120.,121.,122.,123.,124.,125.,126.,127.,128.,129.,
     * 130.,131.,132.,133.,134.,135.,136.,137.,138.,139.,140./
C
C     latitudes in degree
      DATA (TABLA(I),I=1,5)/
     * 0.,15.,30.,45.,60./
C
C     months
      DATA (TABMO(I),I=1,12)/
     * 1.,2.,3.,4.,5.,6.,7.,8.,9.,10.,11.,12./
C
C     solar zenith angles in degree
      DATA (TABZA(I),I=1,11)/
     * 0.,30.,45.,60.,75.,80.,85.,90.,95.,100.,130./
C
C     log10(F10.7) for 75,130,200 Jy
      DATA (TABFL(I),I=1,3)/
     * 1.87506, 2.11394, 2.30103/
C
C     log10 electron densities, ordered as
C     I,J,K,L,M = Height,Latitude,Month,Zenithangle,F10.7
C     8 heights in each line
C     11 zenith angles in each DATA statement
C     innermost loop: J (5 latitudes)
C     next loop:      K (12 months)
C     next loop:      M (3 F10.7-fluxes)
C     outermost loop: I (11 groups of heights)
      DATA ((EDEN(I,1,1,L,1),I=1,8),L=1,NZEN)/
     * 7.044,7.218,7.416,7.627,7.834,8.016,8.171,8.290,
     * 7.015,7.163,7.324,7.511,7.712,7.902,8.066,8.197,
     * 6.999,7.132,7.262,7.401,7.562,7.736,7.904,8.046,
     * 6.988,7.121,7.243,7.354,7.451,7.545,7.644,7.761,
     * 6.967,7.110,7.247,7.369,7.469,7.533,7.561,7.561,
     * 6.957,7.107,7.253,7.386,7.488,7.553,7.577,7.567,
     * 6.943,7.104,7.264,7.407,7.517,7.590,7.614,7.597,
     * 0.000,0.000,0.000,0.000,6.058,6.310,6.499,6.634,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,1,L,1),I=1,8),L=1,NZEN)/
     * 7.019,7.180,7.368,7.578,7.792,7.987,8.152,8.283,
     * 6.998,7.136,7.288,7.470,7.672,7.870,8.042,8.183,
     * 6.983,7.111,7.236,7.374,7.531,7.707,7.881,8.030,
     * 6.978,7.108,7.228,7.338,7.431,7.518,7.614,7.726,
     * 6.957,7.098,7.232,7.348,7.443,7.500,7.527,7.523,
     * 6.939,7.086,7.228,7.352,7.451,7.509,7.530,7.515,
     * 6.930,7.089,7.245,7.384,7.484,7.544,7.562,7.543,
     * 0.000,0.000,0.000,0.000,0.000,6.195,6.373,6.504,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,1,L,1),I=1,8),L=1,NZEN)/
     * 7.118,7.261,7.430,7.624,7.820,8.003,8.161,8.292,
     * 7.096,7.222,7.360,7.524,7.708,7.890,8.054,8.190,
     * 7.081,7.195,7.308,7.429,7.571,7.731,7.893,8.036,
     * 7.062,7.178,7.285,7.380,7.467,7.548,7.644,7.759,
     * 7.031,7.155,7.271,7.375,7.454,7.503,7.523,7.521,
     * 7.016,7.145,7.268,7.379,7.459,7.505,7.518,7.504,
     * 6.992,7.130,7.264,7.383,7.466,7.515,7.527,7.506,
     * 0.000,0.000,0.000,0.000,0.000,6.164,6.327,6.454,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,1,L,1),I=1,8),L=1,NZEN)/
     * 7.352,7.481,7.629,7.791,7.951,8.099,8.235,8.351,
     * 7.315,7.420,7.541,7.687,7.842,7.993,8.133,8.252,
     * 7.290,7.375,7.464,7.572,7.699,7.839,7.979,8.104,
     * 7.266,7.347,7.420,7.489,7.555,7.630,7.725,7.833,
     * 7.219,7.307,7.387,7.455,7.502,7.531,7.538,7.534,
     * 7.190,7.282,7.367,7.440,7.489,7.512,7.511,7.493,
     * 7.161,7.260,7.355,7.436,7.486,7.508,7.503,7.473,
     * 0.000,0.000,0.000,0.000,6.013,6.196,6.339,6.454,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,1,L,1),I=1,8),L=1,NZEN)/
     * 7.662,7.781,7.901,8.022,8.138,8.249,8.361,8.458,
     * 7.592,7.688,7.796,7.914,8.033,8.149,8.261,8.366,
     * 7.534,7.599,7.675,7.772,7.883,7.999,8.115,8.222,
     * 7.495,7.539,7.576,7.618,7.674,7.752,7.850,7.956,
     * 7.436,7.483,7.521,7.547,7.560,7.562,7.558,7.558,
     * 7.400,7.450,7.492,7.521,7.532,7.527,7.508,7.484,
     * 7.362,7.421,7.471,7.504,7.516,7.505,7.478,7.443,
     * 0.000,0.000,0.000,0.000,6.033,6.208,6.326,6.427,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,2,L,1),I=1,8),L=1,NZEN)/
     * 7.015,7.184,7.373,7.586,7.794,7.982,8.140,8.264,
     * 6.991,7.136,7.293,7.475,7.674,7.867,8.035,8.169,
     * 6.976,7.110,7.239,7.378,7.531,7.702,7.872,8.018,
     * 6.965,7.100,7.224,7.339,7.436,7.527,7.623,7.734,
     * 6.944,7.090,7.229,7.356,7.458,7.523,7.555,7.555,
     * 6.933,7.087,7.235,7.369,7.476,7.543,7.571,7.562,
     * 6.920,7.084,7.246,7.392,7.506,7.577,7.605,7.591,
     * 0.000,0.000,0.000,0.000,6.040,6.291,6.481,6.617,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,2,L,1),I=1,8),L=1,NZEN)/
     * 7.006,7.165,7.346,7.556,7.768,7.962,8.127,8.259,
     * 6.987,7.125,7.275,7.452,7.650,7.844,8.018,8.157,
     * 6.973,7.103,7.228,7.364,7.515,7.684,7.855,8.003,
     * 6.966,7.097,7.219,7.331,7.426,7.512,7.604,7.710,
     * 6.945,7.087,7.222,7.344,7.438,7.500,7.525,7.525,
     * 6.929,7.077,7.220,7.349,7.448,7.510,7.531,7.522,
     * 6.918,7.078,7.235,7.375,7.482,7.542,7.563,7.546,
     * 0.000,0.000,0.000,0.000,0.000,6.203,6.383,6.515,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,2,L,1),I=1,8),L=1,NZEN)/
     * 7.109,7.253,7.416,7.606,7.800,7.981,8.140,8.268,
     * 7.088,7.213,7.348,7.511,7.690,7.869,8.033,8.169,
     * 7.074,7.190,7.302,7.423,7.559,7.715,7.873,8.015,
     * 7.060,7.176,7.285,7.381,7.466,7.545,7.634,7.741,
     * 7.028,7.154,7.271,7.377,7.459,7.513,7.535,7.531,
     * 7.012,7.143,7.270,7.381,7.464,7.517,7.532,7.520,
     * 6.990,7.131,7.268,7.391,7.478,7.530,7.544,7.524,
     * 0.000,0.000,0.000,0.000,0.000,6.189,6.357,6.485,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,2,L,1),I=1,8),L=1,NZEN)/
     * 7.316,7.442,7.583,7.741,7.902,8.053,8.190,8.309,
     * 7.287,7.390,7.505,7.643,7.795,7.948,8.089,8.211,
     * 7.266,7.353,7.440,7.541,7.662,7.799,7.938,8.065,
     * 7.243,7.330,7.407,7.476,7.541,7.612,7.699,7.802,
     * 7.200,7.294,7.379,7.452,7.505,7.540,7.551,7.549,
     * 7.177,7.272,7.365,7.443,7.499,7.529,7.533,7.517,
     * 7.150,7.256,7.359,7.446,7.504,7.534,7.534,7.511,
     * 0.000,0.000,0.000,0.000,6.027,6.219,6.369,6.490,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,2,L,1),I=1,8),L=1,NZEN)/
     * 7.564,7.671,7.787,7.909,8.030,8.149,8.268,8.371,
     * 7.510,7.595,7.692,7.806,7.927,8.049,8.167,8.279,
     * 7.472,7.531,7.596,7.681,7.785,7.901,8.021,8.136,
     * 7.442,7.491,7.530,7.570,7.618,7.684,7.773,7.876,
     * 7.388,7.442,7.487,7.522,7.543,7.553,7.553,7.552,
     * 7.355,7.413,7.463,7.502,7.523,7.527,7.518,7.498,
     * 7.320,7.388,7.445,7.489,7.512,7.512,7.496,7.466,
     * 0.000,0.000,0.000,0.000,6.030,6.218,6.346,6.454,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,3,L,1),I=1,8),L=1,NZEN)/
     * 6.973,7.136,7.319,7.527,7.737,7.930,8.095,8.226,
     * 6.952,7.096,7.248,7.425,7.621,7.816,7.989,8.130,
     * 6.940,7.077,7.207,7.341,7.490,7.657,7.827,7.976,
     * 6.929,7.068,7.196,7.314,7.416,7.504,7.593,7.701,
     * 6.909,7.058,7.202,7.332,7.438,7.509,7.543,7.544,
     * 6.898,7.054,7.208,7.347,7.456,7.529,7.559,7.553,
     * 6.885,7.052,7.218,7.369,7.487,7.563,7.594,7.584,
     * 0.000,0.000,0.000,0.000,6.012,6.266,6.456,6.594,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,3,L,1),I=1,8),L=1,NZEN)/
     * 6.985,7.141,7.319,7.520,7.730,7.922,8.089,8.222,
     * 6.966,7.105,7.252,7.424,7.617,7.809,7.984,8.125,
     * 6.956,7.088,7.215,7.347,7.492,7.655,7.820,7.969,
     * 6.947,7.082,7.206,7.322,7.417,7.503,7.590,7.691,
     * 6.926,7.071,7.210,7.336,7.435,7.501,7.532,7.534,
     * 6.915,7.067,7.216,7.349,7.452,7.518,7.545,7.536,
     * 6.900,7.062,7.223,7.366,7.480,7.549,7.574,7.560,
     * 0.000,0.000,0.000,0.000,0.000,6.226,6.411,6.547,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,3,L,1),I=1,8),L=1,NZEN)/
     * 7.094,7.233,7.393,7.574,7.763,7.940,8.097,8.226,
     * 7.077,7.201,7.333,7.486,7.659,7.832,7.993,8.127,
     * 7.065,7.183,7.295,7.413,7.542,7.688,7.840,7.980,
     * 7.056,7.176,7.285,7.387,7.472,7.547,7.625,7.717,
     * 7.030,7.160,7.282,7.393,7.479,7.538,7.565,7.563,
     * 7.012,7.147,7.279,7.396,7.486,7.545,7.565,7.558,
     * 6.997,7.143,7.286,7.416,7.513,7.572,7.594,7.579,
     * 0.000,0.000,0.000,0.000,6.017,6.247,6.422,6.555,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,3,L,1),I=1,8),L=1,NZEN)/
     * 7.277,7.393,7.523,7.675,7.831,7.982,8.121,8.243,
     * 7.255,7.354,7.461,7.588,7.734,7.882,8.024,8.148,
     * 7.239,7.327,7.414,7.510,7.619,7.747,7.882,8.008,
     * 7.223,7.313,7.396,7.471,7.536,7.600,7.677,7.767,
     * 7.192,7.290,7.385,7.470,7.532,7.574,7.595,7.595,
     * 7.171,7.277,7.379,7.470,7.537,7.581,7.595,7.585,
     * 7.149,7.264,7.378,7.478,7.551,7.596,7.608,7.592,
     * 0.000,0.000,0.000,0.000,6.073,6.284,6.450,6.583,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,3,L,1),I=1,8),L=1,NZEN)/
     * 7.459,7.550,7.654,7.771,7.895,8.017,8.140,8.260,
     * 7.426,7.501,7.583,7.684,7.800,7.923,8.047,8.166,
     * 7.404,7.464,7.522,7.594,7.683,7.791,7.912,8.030,
     * 7.384,7.441,7.490,7.534,7.578,7.631,7.704,7.795,
     * 7.345,7.411,7.469,7.520,7.557,7.581,7.595,7.598,
     * 7.319,7.391,7.457,7.513,7.554,7.576,7.581,7.574,
     * 7.293,7.375,7.452,7.517,7.562,7.584,7.587,7.571,
     * 0.000,0.000,0.000,0.000,6.072,6.262,6.445,6.572,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,4,L,1),I=1,8),L=1,NZEN)/
     * 6.945,7.106,7.287,7.492,7.705,7.901,8.071,8.206,
     * 6.927,7.072,7.223,7.396,7.591,7.787,7.965,8.110,
     * 6.916,7.055,7.186,7.322,7.467,7.634,7.804,7.956,
     * 6.905,7.047,7.178,7.300,7.404,7.490,7.580,7.682,
     * 6.885,7.037,7.183,7.318,7.425,7.498,7.533,7.539,
     * 6.874,7.033,7.189,7.333,7.444,7.522,7.553,7.545,
     * 6.860,7.030,7.199,7.355,7.475,7.556,7.588,7.576,
     * 0.000,0.000,0.000,0.000,0.000,6.251,6.443,6.582,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,4,L,1),I=1,8),L=1,NZEN)/
     * 6.977,7.132,7.306,7.504,7.707,7.897,8.062,8.194,
     * 6.960,7.099,7.245,7.412,7.600,7.786,7.958,8.098,
     * 6.949,7.084,7.211,7.342,7.480,7.638,7.800,7.946,
     * 6.938,7.075,7.203,7.320,7.421,7.503,7.588,7.683,
     * 6.917,7.065,7.208,7.339,7.443,7.513,7.547,7.551,
     * 6.905,7.060,7.212,7.351,7.462,7.534,7.564,7.560,
     * 6.892,7.056,7.222,7.373,7.491,7.570,7.601,7.589,
     * 0.000,0.000,0.000,0.000,6.020,6.272,6.464,6.604,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,4,L,1),I=1,8),L=1,NZEN)/
     * 7.079,7.215,7.364,7.537,7.717,7.890,8.045,8.173,
     * 7.066,7.190,7.318,7.460,7.623,7.787,7.945,8.078,
     * 7.058,7.177,7.289,7.404,7.524,7.658,7.801,7.933,
     * 7.049,7.171,7.287,7.390,7.476,7.551,7.622,7.701,
     * 7.027,7.161,7.292,7.409,7.499,7.565,7.596,7.600,
     * 7.014,7.155,7.295,7.421,7.520,7.585,7.615,7.609,
     * 7.000,7.152,7.302,7.443,7.549,7.620,7.648,7.639,
     * 0.000,0.000,0.000,0.000,6.080,6.326,6.514,6.653,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,4,L,1),I=1,8),L=1,NZEN)/
     * 7.215,7.323,7.438,7.573,7.719,7.867,8.010,8.136,
     * 7.201,7.301,7.400,7.512,7.639,7.779,7.920,8.047,
     * 7.195,7.291,7.380,7.470,7.564,7.673,7.794,7.917,
     * 7.189,7.292,7.386,7.470,7.541,7.604,7.664,7.731,
     * 7.170,7.283,7.392,7.490,7.569,7.625,7.658,7.666,
     * 7.156,7.276,7.396,7.504,7.590,7.649,7.679,7.678,
     * 7.141,7.271,7.405,7.526,7.621,7.684,7.713,7.708,
     * 0.000,0.000,0.000,0.000,6.127,6.362,6.550,6.697,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,4,L,1),I=1,8),L=1,NZEN)/
     * 7.328,7.405,7.490,7.587,7.700,7.822,7.951,8.075,
     * 7.314,7.387,7.456,7.534,7.630,7.742,7.868,7.991,
     * 7.304,7.374,7.436,7.497,7.566,7.651,7.755,7.870,
     * 7.296,7.369,7.436,7.493,7.545,7.593,7.647,7.711,
     * 7.281,7.367,7.452,7.530,7.594,7.643,7.679,7.697,
     * 7.265,7.360,7.454,7.542,7.613,7.665,7.699,7.710,
     * 7.249,7.355,7.463,7.562,7.642,7.699,7.733,7.738,
     * 0.000,0.000,0.000,0.000,6.111,6.336,6.527,6.687,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,5,L,1),I=1,8),L=1,NZEN)/
     * 6.959,7.122,7.306,7.518,7.730,7.925,8.093,8.225,
     * 6.940,7.084,7.238,7.418,7.616,7.814,7.989,8.131,
     * 6.927,7.064,7.197,7.335,7.487,7.657,7.829,7.979,
     * 6.917,7.058,7.187,7.307,7.412,7.501,7.595,7.704,
     * 6.896,7.047,7.192,7.325,7.434,7.505,7.540,7.544,
     * 6.886,7.043,7.198,7.339,7.452,7.525,7.556,7.551,
     * 6.872,7.040,7.208,7.361,7.483,7.560,7.591,7.581,
     * 0.000,0.000,0.000,0.000,6.013,6.268,6.460,6.598,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,5,L,1),I=1,8),L=1,NZEN)/
     * 6.998,7.155,7.331,7.531,7.733,7.918,8.076,8.203,
     * 6.976,7.116,7.264,7.434,7.621,7.808,7.976,8.112,
     * 6.962,7.095,7.223,7.357,7.500,7.658,7.821,7.966,
     * 6.950,7.086,7.214,7.331,7.432,7.521,7.610,7.708,
     * 6.930,7.077,7.220,7.353,7.458,7.534,7.573,7.574,
     * 6.919,7.074,7.227,7.370,7.482,7.560,7.596,7.588,
     * 6.908,7.074,7.241,7.395,7.520,7.600,7.636,7.626,
     * 0.000,0.000,0.000,0.000,6.081,6.342,6.540,6.682,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,5,L,1),I=1,8),L=1,NZEN)/
     * 7.071,7.206,7.351,7.514,7.688,7.855,8.008,8.135,
     * 7.062,7.187,7.313,7.449,7.602,7.761,7.914,8.046,
     * 7.052,7.175,7.290,7.401,7.515,7.641,7.777,7.905,
     * 7.043,7.168,7.285,7.392,7.483,7.556,7.624,7.696,
     * 7.025,7.162,7.296,7.420,7.518,7.588,7.627,7.632,
     * 7.016,7.161,7.306,7.440,7.546,7.620,7.656,7.654,
     * 7.005,7.162,7.319,7.467,7.585,7.663,7.702,7.696,
     * 0.000,0.000,0.000,0.000,6.145,6.406,6.606,6.753,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,5,L,1),I=1,8),L=1,NZEN)/
     * 7.144,7.250,7.354,7.471,7.603,7.743,7.884,8.017,
     * 7.138,7.243,7.339,7.438,7.548,7.670,7.804,7.933,
     * 7.142,7.250,7.346,7.436,7.522,7.611,7.713,7.823,
     * 7.143,7.258,7.360,7.457,7.539,7.604,7.660,7.709,
     * 7.131,7.258,7.381,7.495,7.589,7.660,7.702,7.716,
     * 7.123,7.255,7.390,7.515,7.619,7.692,7.733,7.740,
     * 7.114,7.259,7.409,7.549,7.663,7.743,7.787,7.790,
     * 0.000,0.000,0.000,0.000,6.166,6.423,6.627,6.786,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,5,L,1),I=1,8),L=1,NZEN)/
     * 7.189,7.270,7.345,7.419,7.509,7.614,7.737,7.867,
     * 7.186,7.267,7.338,7.406,7.480,7.569,7.676,7.797,
     * 7.179,7.265,7.339,7.406,7.469,7.534,7.613,7.709,
     * 7.189,7.283,7.371,7.449,7.522,7.584,7.643,7.693,
     * 7.186,7.290,7.398,7.499,7.592,7.668,7.728,7.764,
     * 7.175,7.291,7.411,7.524,7.627,7.708,7.766,7.795,
     * 7.168,7.295,7.430,7.558,7.671,7.759,7.817,7.841,
     * 0.000,0.000,0.000,0.000,6.122,6.374,6.587,6.762,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,6,L,1),I=1,8),L=1,NZEN)/
     * 7.000,7.169,7.362,7.578,7.792,7.984,8.144,8.271,
     * 6.975,7.121,7.281,7.472,7.676,7.872,8.043,8.180,
     * 6.961,7.095,7.227,7.370,7.534,7.710,7.884,8.031,
     * 6.949,7.086,7.212,7.330,7.432,7.527,7.633,7.750,
     * 6.930,7.076,7.217,7.347,7.450,7.519,7.552,7.555,
     * 6.917,7.071,7.222,7.360,7.468,7.539,7.568,7.562,
     * 6.904,7.068,7.232,7.383,7.498,7.577,7.605,7.591,
     * 0.000,0.000,0.000,0.000,6.050,6.305,6.496,6.632,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,6,L,1),I=1,8),L=1,NZEN)/
     * 7.035,7.199,7.382,7.587,7.788,7.968,8.121,8.241,
     * 7.009,7.150,7.301,7.479,7.671,7.856,8.019,8.149,
     * 6.990,7.122,7.250,7.387,7.538,7.702,7.867,8.008,
     * 6.976,7.109,7.235,7.352,7.452,7.546,7.645,7.753,
     * 6.957,7.101,7.244,7.375,7.484,7.557,7.596,7.597,
     * 6.948,7.100,7.253,7.394,7.511,7.587,7.623,7.618,
     * 6.938,7.102,7.268,7.422,7.549,7.633,7.668,7.659,
     * 0.000,0.000,0.000,0.000,6.144,6.413,6.614,6.756,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,6,L,1),I=1,8),L=1,NZEN)/
     * 7.074,7.208,7.351,7.514,7.683,7.849,7.999,8.123,
     * 7.065,7.190,7.314,7.451,7.601,7.757,7.907,8.037,
     * 7.056,7.178,7.294,7.406,7.518,7.642,7.773,7.901,
     * 7.045,7.172,7.288,7.397,7.487,7.563,7.631,7.703,
     * 7.031,7.169,7.306,7.431,7.534,7.609,7.651,7.658,
     * 7.022,7.168,7.317,7.453,7.566,7.648,7.687,7.690,
     * 7.015,7.174,7.339,7.489,7.615,7.701,7.745,7.744,
     * 0.000,0.000,0.000,0.000,6.203,6.475,6.684,6.835,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,6,L,1),I=1,8),L=1,NZEN)/
     * 7.088,7.197,7.298,7.403,7.523,7.655,7.795,7.930,
     * 7.090,7.199,7.298,7.395,7.494,7.603,7.727,7.854,
     * 7.099,7.214,7.316,7.411,7.497,7.579,7.668,7.763,
     * 7.100,7.218,7.332,7.435,7.522,7.590,7.645,7.686,
     * 7.093,7.227,7.358,7.480,7.584,7.662,7.710,7.729,
     * 7.090,7.229,7.376,7.511,7.626,7.710,7.758,7.771,
     * 7.083,7.235,7.398,7.547,7.674,7.766,7.816,7.826,
     * 0.000,0.000,0.000,0.000,6.196,6.468,6.684,6.849,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,6,L,1),I=1,8),L=1,NZEN)/
     * 7.069,7.158,7.235,7.310,7.383,7.472,7.585,7.714,
     * 7.069,7.157,7.240,7.314,7.382,7.458,7.551,7.663,
     * 7.078,7.173,7.264,7.346,7.418,7.486,7.556,7.631,
     * 7.090,7.193,7.294,7.389,7.478,7.556,7.625,7.680,
     * 7.092,7.207,7.329,7.449,7.560,7.653,7.727,7.774,
     * 7.089,7.216,7.348,7.483,7.603,7.702,7.772,7.810,
     * 7.082,7.221,7.373,7.523,7.655,7.760,7.829,7.859,
     * 0.000,0.000,0.000,0.000,6.126,6.396,6.620,6.801,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,7,L,1),I=1,8),L=1,NZEN)/
     * 7.028,7.201,7.402,7.621,7.833,8.022,8.179,8.300,
     * 6.998,7.146,7.312,7.508,7.716,7.911,8.079,8.211,
     * 6.983,7.115,7.247,7.398,7.565,7.749,7.921,8.065,
     * 6.971,7.104,7.229,7.345,7.447,7.548,7.658,7.782,
     * 6.950,7.094,7.233,7.360,7.462,7.530,7.561,7.561,
     * 6.938,7.089,7.237,7.374,7.480,7.550,7.577,7.567,
     * 6.925,7.086,7.247,7.394,7.513,7.587,7.611,7.597,
     * 0.000,0.000,0.000,0.000,6.074,6.328,6.518,6.653,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,7,L,1),I=1,8),L=1,NZEN)/
     * 7.057,7.224,7.416,7.622,7.823,8.001,8.152,8.268,
     * 7.026,7.169,7.326,7.508,7.704,7.889,8.051,8.178,
     * 7.007,7.138,7.266,7.406,7.563,7.734,7.898,8.038,
     * 6.992,7.124,7.248,7.362,7.464,7.562,7.667,7.781,
     * 6.972,7.117,7.258,7.391,7.495,7.570,7.606,7.609,
     * 6.964,7.116,7.268,7.411,7.523,7.602,7.635,7.629,
     * 6.955,7.117,7.283,7.439,7.561,7.646,7.680,7.671,
     * 0.000,0.000,0.000,0.000,6.174,6.444,6.646,6.788,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,7,L,1),I=1,8),L=1,NZEN)/
     * 7.082,7.217,7.361,7.526,7.698,7.864,8.014,8.139,
     * 7.073,7.198,7.322,7.463,7.614,7.773,7.924,8.054,
     * 7.064,7.185,7.299,7.413,7.527,7.654,7.791,7.919,
     * 7.054,7.179,7.298,7.403,7.496,7.572,7.642,7.717,
     * 7.040,7.178,7.315,7.440,7.544,7.619,7.662,7.669,
     * 7.032,7.178,7.326,7.464,7.580,7.661,7.701,7.702,
     * 7.026,7.185,7.350,7.502,7.629,7.716,7.759,7.759,
     * 0.000,0.000,0.000,0.000,6.234,6.509,6.720,6.871,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,7,L,1),I=1,8),L=1,NZEN)/
     * 7.083,7.193,7.291,7.398,7.519,7.652,7.794,7.932,
     * 7.087,7.195,7.293,7.390,7.488,7.600,7.727,7.856,
     * 7.096,7.211,7.311,7.408,7.493,7.575,7.664,7.761,
     * 7.098,7.216,7.329,7.431,7.518,7.586,7.642,7.684,
     * 7.093,7.225,7.358,7.483,7.586,7.664,7.713,7.732,
     * 7.092,7.233,7.379,7.516,7.630,7.716,7.765,7.778,
     * 7.087,7.240,7.402,7.553,7.682,7.775,7.829,7.840,
     * 0.000,0.000,0.000,0.000,6.224,6.500,6.720,6.887,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,7,L,1),I=1,8),L=1,NZEN)/
     * 7.051,7.139,7.217,7.288,7.364,7.454,7.568,7.700,
     * 7.051,7.143,7.223,7.296,7.365,7.441,7.538,7.652,
     * 7.063,7.159,7.249,7.330,7.403,7.470,7.540,7.616,
     * 7.076,7.178,7.281,7.378,7.468,7.547,7.613,7.665,
     * 7.080,7.200,7.324,7.444,7.557,7.652,7.725,7.769,
     * 7.079,7.207,7.346,7.482,7.604,7.704,7.774,7.810,
     * 7.075,7.215,7.374,7.523,7.659,7.764,7.835,7.866,
     * 0.000,0.000,0.000,0.000,6.154,6.431,6.660,6.843,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,8,L,1),I=1,8),L=1,NZEN)/
     * 7.018,7.192,7.391,7.612,7.826,8.014,8.173,8.295,
     * 6.990,7.137,7.303,7.500,7.706,7.904,8.073,8.206,
     * 6.974,7.108,7.242,7.390,7.556,7.739,7.914,8.058,
     * 6.964,7.098,7.224,7.341,7.444,7.542,7.654,7.774,
     * 6.943,7.087,7.228,7.357,7.458,7.526,7.561,7.561,
     * 6.931,7.083,7.233,7.370,7.476,7.546,7.577,7.567,
     * 6.918,7.081,7.244,7.392,7.510,7.583,7.611,7.594,
     * 0.000,0.000,0.000,0.000,6.068,6.323,6.513,6.649,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,8,L,1),I=1,8),L=1,NZEN)/
     * 7.045,7.210,7.399,7.607,7.810,7.992,8.146,8.267,
     * 7.016,7.158,7.314,7.499,7.694,7.883,8.047,8.177,
     * 6.998,7.130,7.259,7.398,7.557,7.729,7.894,8.036,
     * 6.986,7.118,7.243,7.360,7.462,7.556,7.659,7.774,
     * 6.967,7.111,7.253,7.382,7.490,7.561,7.596,7.600,
     * 6.958,7.110,7.261,7.401,7.516,7.591,7.623,7.617,
     * 6.949,7.111,7.277,7.428,7.553,7.636,7.669,7.659,
     * 0.000,0.000,0.000,0.000,6.154,6.421,6.620,6.762,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,8,L,1),I=1,8),L=1,NZEN)/
     * 7.092,7.228,7.377,7.548,7.726,7.894,8.046,8.172,
     * 7.081,7.205,7.332,7.476,7.636,7.799,7.954,8.086,
     * 7.072,7.192,7.305,7.421,7.542,7.675,7.818,7.949,
     * 7.063,7.186,7.303,7.409,7.500,7.576,7.651,7.734,
     * 7.048,7.185,7.320,7.442,7.545,7.615,7.655,7.663,
     * 7.042,7.187,7.333,7.467,7.579,7.653,7.692,7.691,
     * 7.034,7.190,7.351,7.501,7.621,7.706,7.746,7.741,
     * 0.000,0.000,0.000,0.000,6.224,6.494,6.700,6.849,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,8,L,1),I=1,8),L=1,NZEN)/
     * 7.140,7.247,7.348,7.465,7.598,7.743,7.888,8.024,
     * 7.140,7.245,7.341,7.438,7.549,7.675,7.811,7.944,
     * 7.143,7.247,7.342,7.432,7.519,7.611,7.717,7.831,
     * 7.148,7.261,7.363,7.458,7.540,7.604,7.659,7.710,
     * 7.141,7.266,7.391,7.508,7.605,7.674,7.718,7.733,
     * 7.137,7.271,7.408,7.535,7.640,7.718,7.760,7.768,
     * 7.132,7.280,7.431,7.574,7.695,7.779,7.826,7.831,
     * 0.000,0.000,0.000,0.000,6.248,6.519,6.732,6.894,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,8,L,1),I=1,8),L=1,NZEN)/
     * 7.158,7.240,7.312,7.384,7.469,7.576,7.704,7.841,
     * 7.157,7.239,7.310,7.377,7.450,7.536,7.649,7.777,
     * 7.161,7.246,7.324,7.390,7.453,7.519,7.599,7.698,
     * 7.179,7.275,7.365,7.447,7.523,7.588,7.645,7.694,
     * 7.177,7.287,7.400,7.508,7.603,7.681,7.740,7.772,
     * 7.176,7.295,7.420,7.540,7.649,7.735,7.794,7.820,
     * 7.170,7.302,7.445,7.582,7.703,7.795,7.856,7.878,
     * 0.000,0.000,0.000,0.000,6.210,6.480,6.703,6.880,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,9,L,1),I=1,8),L=1,NZEN)/
     * 6.989,7.157,7.351,7.569,7.781,7.975,8.138,8.265,
     * 6.963,7.110,7.271,7.459,7.666,7.863,8.035,8.172,
     * 6.950,7.085,7.219,7.362,7.525,7.702,7.875,8.022,
     * 6.939,7.077,7.206,7.324,7.428,7.523,7.626,7.741,
     * 6.919,7.068,7.211,7.341,7.446,7.519,7.552,7.552,
     * 6.909,7.063,7.215,7.356,7.464,7.539,7.565,7.559,
     * 6.895,7.060,7.226,7.378,7.498,7.573,7.602,7.589,
     * 0.000,0.000,0.000,0.000,6.042,6.297,6.489,6.626,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,9,L,1),I=1,8),L=1,NZEN)/
     * 7.011,7.173,7.357,7.563,7.771,7.958,8.119,8.245,
     * 6.988,7.129,7.282,7.462,7.657,7.849,8.019,8.156,
     * 6.975,7.108,7.236,7.372,7.526,7.694,7.862,8.009,
     * 6.965,7.099,7.226,7.343,7.445,7.535,7.631,7.740,
     * 6.947,7.092,7.234,7.364,7.470,7.540,7.577,7.578,
     * 6.936,7.089,7.241,7.379,7.492,7.565,7.596,7.590,
     * 6.924,7.088,7.254,7.403,7.526,7.606,7.637,7.627,
     * 0.000,0.000,0.000,0.000,6.093,6.353,6.549,6.689,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,9,L,1),I=1,8),L=1,NZEN)/
     * 7.101,7.241,7.396,7.572,7.755,7.929,8.083,8.210,
     * 7.088,7.212,7.341,7.491,7.659,7.830,7.988,8.121,
     * 7.080,7.199,7.312,7.428,7.555,7.697,7.845,7.981,
     * 7.071,7.193,7.306,7.409,7.499,7.576,7.655,7.745,
     * 7.054,7.188,7.316,7.435,7.532,7.598,7.632,7.637,
     * 7.045,7.186,7.324,7.453,7.557,7.627,7.658,7.656,
     * 7.035,7.186,7.339,7.481,7.597,7.671,7.704,7.695,
     * 0.000,0.000,0.000,0.000,6.172,6.429,6.626,6.769,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,9,L,1),I=1,8),L=1,NZEN)/
     * 7.230,7.338,7.456,7.594,7.743,7.893,8.036,8.161,
     * 7.219,7.319,7.417,7.531,7.665,7.806,7.951,8.081,
     * 7.219,7.315,7.404,7.494,7.592,7.705,7.831,7.955,
     * 7.217,7.318,7.408,7.491,7.564,7.627,7.693,7.765,
     * 7.206,7.318,7.428,7.530,7.615,7.672,7.706,7.714,
     * 7.198,7.317,7.438,7.550,7.643,7.705,7.736,7.736,
     * 7.185,7.319,7.454,7.580,7.684,7.751,7.785,7.780,
     * 0.000,0.000,0.000,0.000,6.245,6.497,6.696,6.847,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,9,L,1),I=1,8),L=1,NZEN)/
     * 7.339,7.418,7.499,7.596,7.709,7.833,7.962,8.086,
     * 7.331,7.398,7.467,7.544,7.642,7.757,7.885,8.012,
     * 7.321,7.389,7.448,7.508,7.577,7.666,7.777,7.898,
     * 7.326,7.397,7.463,7.521,7.572,7.622,7.680,7.748,
     * 7.320,7.410,7.498,7.579,7.648,7.703,7.739,7.756,
     * 7.311,7.409,7.508,7.599,7.678,7.737,7.773,7.782,
     * 7.302,7.412,7.525,7.633,7.723,7.788,7.826,7.831,
     * 0.000,0.000,0.000,6.018,6.261,6.509,6.714,6.879,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,10,L,1),I=1,8),L=1,NZEN)/
     * 6.974,7.140,7.331,7.543,7.756,7.950,8.116,8.245,
     * 6.952,7.098,7.254,7.439,7.639,7.837,8.010,8.151,
     * 6.939,7.077,7.209,7.349,7.503,7.677,7.850,7.999,
     * 6.928,7.068,7.198,7.317,7.420,7.510,7.608,7.719,
     * 6.908,7.058,7.203,7.334,7.442,7.512,7.546,7.547,
     * 6.898,7.054,7.208,7.348,7.460,7.532,7.562,7.556,
     * 6.885,7.052,7.218,7.370,7.490,7.567,7.597,7.584,
     * 0.000,0.000,0.000,0.000,6.024,6.279,6.470,6.608,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,10,L,1),I=1,8),L=1,NZEN)/
     * 6.990,7.148,7.327,7.536,7.745,7.938,8.104,8.237,
     * 6.970,7.111,7.261,7.435,7.632,7.826,8.001,8.142,
     * 6.960,7.093,7.221,7.353,7.504,7.671,7.840,7.989,
     * 6.951,7.086,7.212,7.328,7.430,7.517,7.608,7.713,
     * 6.931,7.077,7.219,7.346,7.447,7.519,7.551,7.554,
     * 6.921,7.074,7.224,7.362,7.468,7.541,7.569,7.561,
     * 6.908,7.071,7.234,7.383,7.500,7.573,7.604,7.591,
     * 0.000,0.000,0.000,0.000,6.028,6.279,6.469,6.607,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,10,L,1),I=1,8),L=1,NZEN)/
     * 7.109,7.249,7.411,7.594,7.783,7.961,8.116,8.247,
     * 7.093,7.217,7.348,7.506,7.679,7.856,8.017,8.154,
     * 7.082,7.200,7.310,7.430,7.562,7.712,7.866,8.007,
     * 7.074,7.193,7.304,7.406,7.492,7.570,7.654,7.751,
     * 7.053,7.183,7.310,7.423,7.512,7.572,7.600,7.602,
     * 7.042,7.179,7.315,7.436,7.530,7.594,7.619,7.611,
     * 7.027,7.175,7.320,7.455,7.559,7.625,7.651,7.637,
     * 0.000,0.000,0.000,0.000,6.098,6.339,6.524,6.661,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,10,L,1),I=1,8),L=1,NZEN)/
     * 7.313,7.432,7.569,7.722,7.879,8.028,8.166,8.285,
     * 7.288,7.389,7.501,7.636,7.784,7.934,8.075,8.197,
     * 7.274,7.364,7.451,7.550,7.668,7.799,7.937,8.065,
     * 7.266,7.354,7.437,7.512,7.581,7.652,7.734,7.831,
     * 7.244,7.347,7.443,7.530,7.600,7.645,7.668,7.671,
     * 7.228,7.337,7.442,7.538,7.611,7.659,7.678,7.670,
     * 7.216,7.333,7.452,7.561,7.643,7.695,7.714,7.699,
     * 0.000,0.000,0.000,0.000,6.201,6.430,6.610,6.750,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,10,L,1),I=1,8),L=1,NZEN)/
     * 7.522,7.622,7.734,7.854,7.976,8.095,8.212,8.323,
     * 7.483,7.563,7.654,7.764,7.885,8.007,8.126,8.245,
     * 7.458,7.518,7.582,7.662,7.763,7.879,8.001,8.117,
     * 7.441,7.495,7.541,7.587,7.638,7.702,7.789,7.889,
     * 7.418,7.484,7.546,7.600,7.644,7.673,7.690,7.695,
     * 7.401,7.474,7.546,7.609,7.659,7.690,7.701,7.694,
     * 7.383,7.467,7.551,7.625,7.681,7.715,7.725,7.710,
     * 0.000,0.000,0.000,6.047,6.256,6.471,6.649,6.794,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,11,L,1),I=1,8),L=1,NZEN)/
     * 6.994,7.161,7.351,7.566,7.776,7.967,8.128,8.255,
     * 6.971,7.116,7.274,7.456,7.657,7.853,8.024,8.161,
     * 6.957,7.093,7.224,7.362,7.518,7.690,7.861,8.008,
     * 6.946,7.084,7.211,7.328,7.428,7.520,7.616,7.726,
     * 6.925,7.074,7.215,7.345,7.446,7.516,7.549,7.552,
     * 6.915,7.070,7.221,7.360,7.468,7.539,7.568,7.556,
     * 6.902,7.067,7.232,7.382,7.498,7.573,7.602,7.586,
     * 0.000,0.000,0.000,0.000,6.030,6.283,6.474,6.612,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,11,L,1),I=1,8),L=1,NZEN)/
     * 6.993,7.150,7.331,7.541,7.753,7.949,8.117,8.252,
     * 6.975,7.114,7.264,7.440,7.639,7.833,8.009,8.151,
     * 6.963,7.094,7.220,7.354,7.506,7.675,7.848,7.998,
     * 6.955,7.089,7.213,7.326,7.423,7.511,7.602,7.709,
     * 6.935,7.079,7.217,7.343,7.439,7.504,7.536,7.535,
     * 6.926,7.077,7.225,7.358,7.459,7.523,7.548,7.540,
     * 6.909,7.071,7.232,7.374,7.485,7.552,7.576,7.563,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.407,6.542,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,11,L,1),I=1,8),L=1,NZEN)/
     * 7.116,7.257,7.422,7.611,7.805,7.985,8.145,8.272,
     * 7.097,7.219,7.356,7.517,7.697,7.876,8.040,8.176,
     * 7.083,7.199,7.312,7.432,7.570,7.725,7.883,8.026,
     * 7.070,7.187,7.294,7.394,7.480,7.559,7.648,7.755,
     * 7.048,7.176,7.297,7.403,7.486,7.542,7.568,7.566,
     * 7.030,7.162,7.292,7.405,7.492,7.548,7.568,7.556,
     * 7.016,7.161,7.301,7.428,7.521,7.579,7.598,7.581,
     * 0.000,0.000,0.000,0.000,6.019,6.245,6.419,6.552,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,11,L,1),I=1,8),L=1,NZEN)/
     * 7.364,7.494,7.641,7.802,7.960,8.106,8.238,8.356,
     * 7.329,7.435,7.557,7.701,7.856,8.007,8.146,8.269,
     * 7.307,7.393,7.483,7.592,7.719,7.860,7.999,8.125,
     * 7.288,7.371,7.446,7.516,7.584,7.662,7.758,7.865,
     * 7.254,7.345,7.429,7.506,7.562,7.596,7.610,7.608,
     * 7.231,7.327,7.419,7.501,7.560,7.593,7.599,7.584,
     * 7.208,7.317,7.421,7.512,7.577,7.609,7.614,7.591,
     * 0.000,0.000,0.000,0.000,6.123,6.327,6.489,6.617,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,11,L,1),I=1,8),L=1,NZEN)/
     * 7.668,7.788,7.910,8.032,8.148,8.259,8.372,8.470,
     * 7.600,7.698,7.809,7.929,8.051,8.168,8.279,8.387,
     * 7.546,7.613,7.693,7.794,7.909,8.028,8.146,8.258,
     * 7.511,7.557,7.599,7.648,7.711,7.796,7.898,8.009,
     * 7.465,7.516,7.562,7.598,7.623,7.635,7.640,7.645,
     * 7.437,7.495,7.546,7.588,7.613,7.621,7.615,7.595,
     * 7.409,7.474,7.536,7.584,7.612,7.619,7.605,7.576,
     * 0.000,0.000,0.000,6.010,6.187,6.393,6.534,6.648,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,12,L,1),I=1,8),L=1,NZEN)/
     * 7.029,7.202,7.395,7.609,7.817,8.003,8.159,8.282,
     * 7.003,7.149,7.312,7.497,7.697,7.888,8.055,8.187,
     * 6.988,7.121,7.251,7.390,7.548,7.722,7.893,8.036,
     * 6.976,7.111,7.234,7.348,7.447,7.536,7.637,7.751,
     * 6.956,7.101,7.238,7.364,7.462,7.530,7.558,7.555,
     * 6.944,7.097,7.245,7.377,7.484,7.550,7.574,7.564,
     * 6.931,7.095,7.255,7.398,7.513,7.583,7.611,7.594,
     * 0.000,0.000,0.000,0.000,6.050,6.303,6.492,6.628,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,12,L,1),I=1,8),L=1,NZEN)/
     * 7.011,7.170,7.353,7.566,7.780,7.976,8.143,8.274,
     * 6.990,7.128,7.280,7.462,7.660,7.859,8.033,8.175,
     * 6.975,7.104,7.230,7.367,7.523,7.697,7.871,8.022,
     * 6.970,7.102,7.222,7.335,7.429,7.514,7.609,7.720,
     * 6.950,7.093,7.227,7.346,7.443,7.501,7.528,7.525,
     * 6.933,7.081,7.224,7.351,7.449,7.510,7.533,7.521,
     * 6.923,7.084,7.241,7.380,7.485,7.547,7.566,7.549,
     * 0.000,0.000,0.000,0.000,0.000,6.199,6.378,6.510,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,12,L,1),I=1,8),L=1,NZEN)/
     * 7.120,7.265,7.430,7.624,7.820,8.003,8.162,8.292,
     * 7.098,7.222,7.361,7.524,7.709,7.891,8.055,8.191,
     * 7.083,7.198,7.309,7.433,7.573,7.733,7.894,8.038,
     * 7.068,7.184,7.291,7.386,7.472,7.552,7.647,7.757,
     * 7.037,7.161,7.279,7.383,7.460,7.510,7.534,7.532,
     * 7.021,7.152,7.277,7.386,7.467,7.518,7.531,7.518,
     * 6.998,7.138,7.275,7.394,7.479,7.529,7.541,7.522,
     * 0.000,0.000,0.000,0.000,0.000,6.183,6.350,6.478,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,12,L,1),I=1,8),L=1,NZEN)/
     * 7.372,7.506,7.655,7.817,7.976,8.124,8.259,8.371,
     * 7.333,7.440,7.565,7.713,7.868,8.019,8.158,8.276,
     * 7.306,7.394,7.484,7.593,7.724,7.866,8.006,8.131,
     * 7.284,7.365,7.438,7.506,7.574,7.651,7.750,7.860,
     * 7.239,7.326,7.406,7.474,7.524,7.551,7.562,7.557,
     * 7.211,7.305,7.391,7.464,7.514,7.538,7.538,7.520,
     * 7.184,7.284,7.381,7.462,7.515,7.537,7.533,7.506,
     * 0.000,0.000,0.000,0.000,6.049,6.237,6.385,6.502,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,12,L,1),I=1,8),L=1,NZEN)/
     * 7.712,7.835,7.957,8.077,8.191,8.299,8.410,8.503,
     * 7.634,7.736,7.850,7.971,8.088,8.202,8.314,8.416,
     * 7.568,7.637,7.721,7.825,7.940,8.058,8.172,8.274,
     * 7.525,7.568,7.608,7.655,7.720,7.806,7.909,8.018,
     * 7.467,7.514,7.552,7.580,7.595,7.598,7.597,7.602,
     * 7.433,7.484,7.527,7.558,7.571,7.568,7.551,7.527,
     * 7.399,7.458,7.508,7.545,7.558,7.550,7.527,7.492,
     * 0.000,0.000,0.000,0.000,6.096,6.276,6.397,6.498,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,1,L,2),I=1,8),L=1,NZEN)/
     * 7.025,7.215,7.432,7.661,7.878,8.068,8.227,8.351,
     * 6.992,7.148,7.327,7.534,7.752,7.952,8.124,8.260,
     * 6.976,7.112,7.248,7.404,7.585,7.779,7.960,8.110,
     * 6.967,7.102,7.226,7.342,7.445,7.552,7.677,7.814,
     * 6.954,7.101,7.239,7.365,7.464,7.536,7.569,7.574,
     * 6.949,7.102,7.251,7.386,7.495,7.562,7.595,7.588,
     * 6.943,7.106,7.267,7.415,7.534,7.605,7.636,7.626,
     * 0.000,0.000,0.000,0.000,6.101,6.360,6.558,6.704,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,1,L,2),I=1,8),L=1,NZEN)/
     * 6.985,7.163,7.372,7.607,7.834,8.036,8.209,8.342,
     * 6.959,7.107,7.277,7.485,7.707,7.917,8.099,8.245,
     * 6.943,7.074,7.207,7.361,7.546,7.743,7.933,8.092,
     * 6.942,7.073,7.196,7.309,7.414,7.513,7.635,7.774,
     * 6.929,7.073,7.208,7.330,7.424,7.488,7.521,7.523,
     * 6.919,7.067,7.210,7.341,7.440,7.504,7.530,7.526,
     * 6.914,7.076,7.234,7.374,7.479,7.548,7.571,7.559,
     * 0.000,0.000,0.000,0.000,0.000,6.230,6.416,6.560,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,1,L,2),I=1,8),L=1,NZEN)/
     * 7.074,7.237,7.430,7.648,7.859,8.052,8.219,8.352,
     * 7.045,7.181,7.339,7.531,7.738,7.936,8.110,8.252,
     * 7.030,7.148,7.267,7.408,7.577,7.765,7.943,8.097,
     * 7.014,7.132,7.239,7.340,7.436,7.537,7.662,7.803,
     * 6.992,7.118,7.237,7.343,7.424,7.478,7.501,7.506,
     * 6.982,7.113,7.237,7.352,7.435,7.486,7.505,7.497,
     * 6.964,7.104,7.240,7.360,7.451,7.502,7.518,7.504,
     * 0.000,0.000,0.000,0.000,0.000,6.179,6.351,6.490,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,1,L,2),I=1,8),L=1,NZEN)/
     * 7.313,7.457,7.625,7.805,7.983,8.147,8.297,8.421,
     * 7.272,7.385,7.525,7.692,7.868,8.038,8.191,8.324,
     * 7.244,7.336,7.433,7.558,7.711,7.875,8.034,8.177,
     * 7.223,7.308,7.381,7.453,7.531,7.627,7.752,7.888,
     * 7.184,7.273,7.354,7.425,7.474,7.505,7.517,7.522,
     * 7.160,7.254,7.340,7.415,7.465,7.493,7.496,7.481,
     * 7.133,7.237,7.333,7.414,7.468,7.494,7.491,7.470,
     * 0.000,0.000,0.000,0.000,6.016,6.205,6.355,6.481,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,1,L,2),I=1,8),L=1,NZEN)/
     * 7.610,7.745,7.885,8.026,8.163,8.299,8.425,8.541,
     * 7.537,7.645,7.774,7.914,8.054,8.191,8.317,8.443,
     * 7.478,7.549,7.640,7.759,7.895,8.035,8.174,8.300,
     * 7.439,7.483,7.523,7.577,7.653,7.761,7.891,8.028,
     * 7.383,7.432,7.471,7.500,7.514,7.519,7.523,7.542,
     * 7.351,7.402,7.445,7.477,7.490,7.488,7.472,7.454,
     * 7.317,7.376,7.428,7.464,7.476,7.470,7.447,7.417,
     * 0.000,0.000,0.000,0.000,6.009,6.167,6.287,6.403,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,2,L,2),I=1,8),L=1,NZEN)/
     * 6.995,7.178,7.388,7.618,7.837,8.031,8.196,8.326,
     * 6.966,7.119,7.292,7.497,7.712,7.916,8.092,8.231,
     * 6.951,7.089,7.225,7.377,7.552,7.745,7.926,8.080,
     * 6.943,7.081,7.207,7.325,7.429,7.534,7.652,7.786,
     * 6.931,7.080,7.221,7.350,7.452,7.525,7.559,7.566,
     * 6.926,7.082,7.232,7.371,7.479,7.551,7.585,7.582,
     * 6.919,7.085,7.250,7.399,7.518,7.595,7.627,7.621,
     * 0.000,0.000,0.000,0.000,6.082,6.342,6.540,6.688,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,2,L,2),I=1,8),L=1,NZEN)/
     * 6.971,7.145,7.350,7.582,7.807,8.011,8.182,8.318,
     * 6.947,7.095,7.261,7.464,7.683,7.891,8.073,8.219,
     * 6.934,7.067,7.199,7.351,7.525,7.720,7.906,8.065,
     * 6.929,7.063,7.187,7.302,7.405,7.507,7.622,7.756,
     * 6.918,7.062,7.199,7.323,7.422,7.487,7.522,7.525,
     * 6.906,7.057,7.203,7.335,7.439,7.505,7.533,7.530,
     * 6.902,7.064,7.223,7.367,7.477,7.545,7.571,7.561,
     * 0.000,0.000,0.000,0.000,0.000,6.238,6.426,6.570,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,2,L,2),I=1,8),L=1,NZEN)/
     * 7.063,7.223,7.413,7.626,7.836,8.028,8.193,8.328,
     * 7.038,7.171,7.326,7.513,7.716,7.912,8.086,8.229,
     * 7.023,7.142,7.261,7.398,7.563,7.744,7.920,8.073,
     * 7.011,7.129,7.239,7.341,7.435,7.531,7.646,7.780,
     * 6.989,7.116,7.235,7.344,7.429,7.485,7.512,7.519,
     * 6.978,7.112,7.242,7.356,7.442,7.497,7.518,7.512,
     * 6.962,7.105,7.244,7.367,7.462,7.518,7.536,7.524,
     * 0.000,0.000,0.000,0.000,0.000,6.205,6.380,6.520,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,2,L,2),I=1,8),L=1,NZEN)/
     * 7.277,7.415,7.576,7.753,7.930,8.092,8.242,8.369,
     * 7.244,7.354,7.485,7.646,7.817,7.984,8.138,8.267,
     * 7.220,7.312,7.405,7.523,7.666,7.825,7.982,8.122,
     * 7.201,7.288,7.365,7.440,7.514,7.601,7.714,7.843,
     * 7.166,7.261,7.346,7.422,7.478,7.513,7.528,7.532,
     * 7.144,7.244,7.336,7.417,7.474,7.508,7.515,7.505,
     * 7.123,7.233,7.336,7.425,7.486,7.520,7.523,7.505,
     * 0.000,0.000,0.000,0.000,6.029,6.227,6.385,6.515,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,2,L,2),I=1,8),L=1,NZEN)/
     * 7.515,7.638,7.770,7.911,8.047,8.182,8.315,8.432,
     * 7.456,7.552,7.667,7.800,7.941,8.077,8.215,8.339,
     * 7.415,7.478,7.555,7.659,7.787,7.924,8.062,8.198,
     * 7.386,7.434,7.476,7.523,7.584,7.673,7.790,7.918,
     * 7.337,7.390,7.437,7.474,7.497,7.509,7.513,7.523,
     * 7.307,7.365,7.417,7.458,7.480,7.487,7.480,7.466,
     * 7.276,7.342,7.402,7.448,7.472,7.475,7.462,7.437,
     * 0.000,0.000,0.000,0.000,6.005,6.174,6.303,6.423,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,3,L,2),I=1,8),L=1,NZEN)/
     * 6.949,7.125,7.327,7.556,7.778,7.981,8.151,8.287,
     * 6.926,7.077,7.242,7.442,7.656,7.864,8.046,8.193,
     * 6.915,7.053,7.189,7.338,7.507,7.695,7.881,8.039,
     * 6.907,7.047,7.178,7.300,7.405,7.504,7.620,7.747,
     * 6.895,7.046,7.192,7.327,7.435,7.511,7.546,7.557,
     * 6.890,7.049,7.204,7.348,7.461,7.536,7.572,7.574,
     * 6.883,7.053,7.221,7.376,7.497,7.580,7.614,7.610,
     * 0.000,0.000,0.000,0.000,6.053,6.315,6.514,6.664,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,3,L,2),I=1,8),L=1,NZEN)/
     * 6.948,7.118,7.315,7.543,7.766,7.969,8.143,8.281,
     * 6.926,7.072,7.235,7.430,7.646,7.855,8.038,8.186,
     * 6.916,7.051,7.183,7.328,7.495,7.685,7.869,8.028,
     * 6.910,7.046,7.174,7.292,7.394,7.494,7.604,7.730,
     * 6.897,7.046,7.186,7.316,7.417,7.490,7.524,7.530,
     * 6.893,7.047,7.197,7.335,7.442,7.513,7.543,7.543,
     * 6.883,7.048,7.212,7.359,7.476,7.550,7.581,7.576,
     * 0.000,0.000,0.000,0.000,6.009,6.260,6.453,6.601,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,3,L,2),I=1,8),L=1,NZEN)/
     * 7.047,7.202,7.383,7.590,7.796,7.985,8.150,8.284,
     * 7.026,7.158,7.306,7.483,7.681,7.872,8.045,8.187,
     * 7.014,7.134,7.252,7.384,7.539,7.711,7.884,8.035,
     * 7.008,7.129,7.241,7.343,7.437,7.524,7.628,7.747,
     * 6.990,7.121,7.246,7.359,7.448,7.510,7.540,7.546,
     * 6.978,7.115,7.248,7.370,7.463,7.525,7.551,7.547,
     * 6.968,7.116,7.265,7.395,7.497,7.559,7.585,7.576,
     * 0.000,0.000,0.000,0.000,6.025,6.262,6.445,6.589,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,3,L,2),I=1,8),L=1,NZEN)/
     * 7.234,7.363,7.513,7.681,7.853,8.014,8.162,8.291,
     * 7.209,7.315,7.436,7.584,7.747,7.909,8.062,8.194,
     * 7.193,7.286,7.376,7.485,7.613,7.760,7.911,8.050,
     * 7.178,7.272,7.355,7.433,7.503,7.581,7.676,7.789,
     * 7.155,7.257,7.353,7.436,7.503,7.548,7.570,7.577,
     * 7.139,7.246,7.352,7.445,7.514,7.558,7.576,7.573,
     * 7.122,7.239,7.355,7.457,7.535,7.579,7.595,7.585,
     * 0.000,0.000,0.000,0.000,6.075,6.291,6.465,6.606,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,3,L,2),I=1,8),L=1,NZEN)/
     * 7.408,7.515,7.635,7.768,7.904,8.038,8.167,8.293,
     * 7.371,7.453,7.549,7.669,7.801,7.937,8.071,8.199,
     * 7.344,7.408,7.474,7.558,7.668,7.793,7.927,8.057,
     * 7.326,7.384,7.434,7.481,7.532,7.600,7.692,7.803,
     * 7.291,7.359,7.418,7.472,7.510,7.536,7.551,7.560,
     * 7.268,7.342,7.410,7.468,7.510,7.535,7.543,7.538,
     * 7.246,7.329,7.407,7.474,7.522,7.547,7.553,7.540,
     * 0.000,0.000,0.000,0.000,6.045,6.241,6.405,6.547,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,4,L,2),I=1,8),L=1,NZEN)/
     * 6.921,7.093,7.291,7.518,7.744,7.949,8.125,8.265,
     * 6.901,7.051,7.214,7.410,7.624,7.834,8.020,8.171,
     * 6.890,7.030,7.168,7.313,7.480,7.669,7.855,8.017,
     * 6.883,7.026,7.160,7.285,7.392,7.490,7.599,7.727,
     * 6.871,7.025,7.174,7.312,7.423,7.499,7.540,7.551,
     * 6.867,7.028,7.185,7.332,7.448,7.528,7.566,7.565,
     * 6.860,7.031,7.203,7.361,7.487,7.569,7.608,7.604,
     * 0.000,0.000,0.000,0.000,6.037,6.300,6.501,6.651,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,4,L,2),I=1,8),L=1,NZEN)/
     * 6.939,7.107,7.302,7.523,7.744,7.944,8.116,8.253,
     * 6.919,7.066,7.225,7.415,7.626,7.831,8.011,8.159,
     * 6.909,7.046,7.179,7.321,7.484,7.667,7.847,8.004,
     * 6.901,7.041,7.170,7.291,7.394,7.490,7.599,7.722,
     * 6.889,7.039,7.184,7.318,7.424,7.501,7.539,7.550,
     * 6.883,7.039,7.194,7.336,7.451,7.527,7.565,7.566,
     * 6.875,7.043,7.210,7.365,7.486,7.570,7.606,7.605,
     * 0.000,0.000,0.000,0.000,6.046,6.307,6.506,6.658,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,4,L,2),I=1,8),L=1,NZEN)/
     * 7.029,7.179,7.348,7.546,7.745,7.931,8.094,8.229,
     * 7.014,7.144,7.286,7.452,7.637,7.822,7.992,8.135,
     * 7.006,7.129,7.245,7.371,7.513,7.672,7.837,7.984,
     * 7.000,7.125,7.240,7.348,7.440,7.524,7.615,7.722,
     * 6.987,7.123,7.253,7.375,7.471,7.536,7.575,7.582,
     * 6.979,7.122,7.263,7.393,7.496,7.564,7.597,7.597,
     * 6.971,7.124,7.279,7.420,7.531,7.605,7.640,7.636,
     * 0.000,0.000,0.000,0.000,6.087,6.339,6.535,6.685,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,4,L,2),I=1,8),L=1,NZEN)/
     * 7.169,7.285,7.416,7.569,7.731,7.889,8.038,8.165,
     * 7.155,7.260,7.368,7.494,7.638,7.793,7.943,8.076,
     * 7.149,7.248,7.340,7.436,7.545,7.668,7.807,7.939,
     * 7.148,7.250,7.345,7.430,7.506,7.574,7.646,7.728,
     * 7.133,7.246,7.356,7.459,7.540,7.599,7.633,7.645,
     * 7.125,7.247,7.366,7.477,7.567,7.627,7.660,7.664,
     * 7.114,7.248,7.379,7.503,7.602,7.667,7.700,7.699,
     * 0.000,0.000,0.000,0.000,6.128,6.370,6.564,6.719,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,4,L,2),I=1,8),L=1,NZEN)/
     * 7.271,7.358,7.454,7.566,7.690,7.821,7.954,8.079,
     * 7.256,7.332,7.409,7.500,7.610,7.733,7.865,7.992,
     * 7.248,7.317,7.381,7.447,7.527,7.625,7.742,7.863,
     * 7.239,7.315,7.379,7.439,7.493,7.546,7.608,7.682,
     * 7.228,7.316,7.402,7.480,7.545,7.597,7.634,7.655,
     * 7.216,7.313,7.407,7.495,7.569,7.623,7.659,7.672,
     * 7.203,7.310,7.417,7.519,7.602,7.660,7.696,7.705,
     * 0.000,0.000,0.000,0.000,6.083,6.313,6.509,6.676,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,5,L,2),I=1,8),L=1,NZEN)/
     * 6.936,7.111,7.314,7.545,7.771,7.975,8.149,8.286,
     * 6.914,7.065,7.231,7.432,7.651,7.862,8.046,8.194,
     * 6.902,7.043,7.179,7.329,7.503,7.695,7.882,8.043,
     * 6.895,7.037,7.169,7.292,7.400,7.504,7.620,7.752,
     * 6.883,7.034,7.183,7.320,7.431,7.507,7.546,7.554,
     * 6.878,7.037,7.194,7.339,7.457,7.532,7.572,7.571,
     * 6.871,7.042,7.211,7.369,7.493,7.576,7.614,7.610,
     * 0.000,0.000,0.000,0.000,6.055,6.318,6.517,6.668,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,5,L,2),I=1,8),L=1,NZEN)/
     * 6.961,7.133,7.331,7.551,7.769,7.964,8.130,8.263,
     * 6.936,7.084,7.246,7.440,7.650,7.851,8.029,8.173,
     * 6.922,7.058,7.191,7.339,7.503,7.688,7.868,8.024,
     * 6.913,7.051,7.180,7.301,7.410,7.509,7.619,7.748,
     * 6.901,7.050,7.196,7.333,7.444,7.523,7.562,7.574,
     * 6.897,7.053,7.209,7.356,7.475,7.554,7.594,7.597,
     * 6.893,7.060,7.230,7.388,7.515,7.603,7.642,7.642,
     * 0.000,0.000,0.000,0.000,6.106,6.376,6.582,6.736,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,5,L,2),I=1,8),L=1,NZEN)/
     * 7.021,7.166,7.331,7.519,7.714,7.895,8.057,8.190,
     * 7.010,7.140,7.277,7.435,7.613,7.792,7.958,8.099,
     * 7.001,7.125,7.244,7.367,7.500,7.649,7.807,7.952,
     * 6.994,7.122,7.240,7.349,7.445,7.526,7.611,7.710,
     * 6.985,7.123,7.260,7.384,7.488,7.560,7.602,7.615,
     * 6.982,7.129,7.275,7.411,7.523,7.599,7.636,7.643,
     * 6.977,7.135,7.295,7.445,7.568,7.649,7.693,7.692,
     * 0.000,0.000,0.000,0.000,6.152,6.420,6.628,6.785,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,5,L,2),I=1,8),L=1,NZEN)/
     * 7.096,7.209,7.322,7.454,7.601,7.752,7.899,8.034,
     * 7.090,7.195,7.298,7.409,7.531,7.669,7.812,7.948,
     * 7.096,7.201,7.302,7.395,7.491,7.593,7.708,7.828,
     * 7.100,7.213,7.320,7.418,7.502,7.570,7.631,7.689,
     * 7.095,7.222,7.347,7.463,7.559,7.631,7.676,7.694,
     * 7.091,7.223,7.360,7.488,7.594,7.669,7.713,7.724,
     * 7.087,7.233,7.384,7.527,7.644,7.727,7.774,7.781,
     * 0.000,0.000,0.000,0.000,6.166,6.429,6.640,6.806,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,5,L,2),I=1,8),L=1,NZEN)/
     * 7.124,7.211,7.290,7.376,7.477,7.591,7.718,7.847,
     * 7.122,7.206,7.280,7.354,7.437,7.534,7.649,7.772,
     * 7.120,7.205,7.279,7.349,7.416,7.487,7.575,7.676,
     * 7.132,7.225,7.312,7.392,7.468,7.533,7.593,7.648,
     * 7.131,7.239,7.346,7.449,7.544,7.621,7.682,7.721,
     * 7.128,7.244,7.360,7.477,7.582,7.666,7.726,7.759,
     * 7.120,7.248,7.385,7.514,7.630,7.720,7.782,7.808,
     * 0.000,0.000,0.000,0.000,6.093,6.351,6.569,6.751,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,6,L,2),I=1,8),L=1,NZEN)/
     * 6.979,7.163,7.377,7.609,7.836,8.034,8.200,8.331,
     * 6.950,7.104,7.281,7.490,7.715,7.923,8.102,8.243,
     * 6.937,7.074,7.212,7.368,7.555,7.754,7.940,8.095,
     * 6.929,7.066,7.194,7.316,7.425,7.537,7.663,7.803,
     * 6.916,7.064,7.207,7.341,7.448,7.522,7.559,7.569,
     * 6.911,7.066,7.219,7.360,7.474,7.551,7.585,7.582,
     * 6.903,7.069,7.235,7.389,7.514,7.591,7.627,7.618,
     * 0.000,0.000,0.000,0.000,6.092,6.354,6.554,6.702,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,6,L,2),I=1,8),L=1,NZEN)/
     * 7.002,7.181,7.389,7.614,7.827,8.017,8.176,8.301,
     * 6.970,7.120,7.291,7.491,7.703,7.901,8.074,8.210,
     * 6.952,7.086,7.220,7.374,7.547,7.738,7.917,8.068,
     * 6.940,7.075,7.202,7.322,7.431,7.537,7.661,7.797,
     * 6.927,7.076,7.221,7.355,7.467,7.543,7.586,7.597,
     * 6.926,7.080,7.236,7.382,7.502,7.582,7.623,7.625,
     * 6.921,7.088,7.257,7.417,7.545,7.635,7.674,7.672,
     * 0.000,0.000,0.000,0.000,6.171,6.447,6.656,6.811,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,6,L,2),I=1,8),L=1,NZEN)/
     * 7.023,7.168,7.331,7.516,7.707,7.887,8.046,8.179,
     * 7.013,7.143,7.279,7.437,7.610,7.787,7.950,8.090,
     * 7.004,7.129,7.247,7.368,7.500,7.648,7.803,7.946,
     * 6.997,7.125,7.245,7.354,7.449,7.533,7.618,7.715,
     * 6.990,7.130,7.269,7.398,7.506,7.583,7.626,7.641,
     * 6.987,7.136,7.285,7.426,7.542,7.627,7.669,7.679,
     * 6.986,7.147,7.312,7.467,7.597,7.686,7.733,7.738,
     * 0.000,0.000,0.000,0.000,6.211,6.489,6.705,6.867,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,6,L,2),I=1,8),L=1,NZEN)/
     * 7.038,7.150,7.258,7.378,7.510,7.655,7.803,7.942,
     * 7.042,7.152,7.253,7.359,7.468,7.593,7.727,7.861,
     * 7.053,7.166,7.270,7.369,7.462,7.552,7.652,7.758,
     * 7.056,7.176,7.290,7.395,7.483,7.554,7.613,7.662,
     * 7.056,7.190,7.325,7.448,7.555,7.633,7.684,7.706,
     * 7.058,7.201,7.345,7.485,7.602,7.688,7.738,7.755,
     * 7.056,7.211,7.372,7.525,7.655,7.748,7.803,7.818,
     * 0.000,0.000,0.000,0.000,6.197,6.474,6.696,6.869,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,6,L,2),I=1,8),L=1,NZEN)/
     * 7.005,7.094,7.176,7.253,7.338,7.435,7.552,7.682,
     * 7.006,7.097,7.179,7.254,7.329,7.411,7.510,7.626,
     * 7.017,7.112,7.203,7.286,7.361,7.433,7.508,7.587,
     * 7.031,7.134,7.234,7.332,7.423,7.503,7.575,7.632,
     * 7.038,7.154,7.276,7.398,7.511,7.607,7.682,7.731,
     * 7.038,7.166,7.302,7.436,7.558,7.659,7.732,7.773,
     * 7.036,7.174,7.328,7.479,7.614,7.722,7.793,7.825,
     * 0.000,0.000,0.000,0.000,6.096,6.372,6.602,6.790,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,7,L,2),I=1,8),L=1,NZEN)/
     * 7.010,7.200,7.419,7.655,7.879,8.074,8.237,8.363,
     * 6.976,7.132,7.314,7.534,7.758,7.964,8.138,8.275,
     * 6.958,7.095,7.235,7.400,7.591,7.795,7.978,8.129,
     * 6.949,7.084,7.210,7.330,7.441,7.559,7.694,7.838,
     * 6.937,7.082,7.223,7.355,7.460,7.533,7.569,7.577,
     * 6.931,7.084,7.234,7.375,7.487,7.562,7.591,7.588,
     * 6.923,7.088,7.252,7.403,7.522,7.602,7.633,7.626,
     * 0.000,0.000,0.000,0.000,6.117,6.378,6.576,6.724,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,7,L,2),I=1,8),L=1,NZEN)/
     * 7.026,7.211,7.422,7.651,7.864,8.051,8.207,8.330,
     * 6.989,7.142,7.317,7.525,7.740,7.936,8.105,8.239,
     * 6.968,7.102,7.238,7.398,7.578,7.769,7.950,8.099,
     * 6.956,7.089,7.214,7.335,7.448,7.560,7.689,7.828,
     * 6.944,7.090,7.234,7.368,7.478,7.557,7.599,7.610,
     * 6.943,7.095,7.250,7.396,7.514,7.594,7.634,7.637,
     * 6.938,7.103,7.272,7.431,7.562,7.648,7.688,7.685,
     * 0.000,0.000,0.000,0.000,6.202,6.479,6.689,6.843,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,7,L,2),I=1,8),L=1,NZEN)/
     * 7.032,7.179,7.345,7.530,7.724,7.903,8.063,8.195,
     * 7.021,7.152,7.291,7.449,7.625,7.804,7.968,8.108,
     * 7.013,7.136,7.254,7.376,7.512,7.664,7.823,7.966,
     * 7.005,7.132,7.251,7.360,7.459,7.542,7.631,7.732,
     * 7.000,7.140,7.278,7.407,7.513,7.594,7.637,7.650,
     * 6.997,7.145,7.295,7.438,7.554,7.640,7.683,7.691,
     * 6.998,7.158,7.323,7.479,7.611,7.702,7.750,7.755,
     * 0.000,0.000,0.000,0.000,6.241,6.523,6.742,6.904,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,7,L,2),I=1,8),L=1,NZEN)/
     * 7.034,7.145,7.254,7.372,7.506,7.654,7.805,7.948,
     * 7.038,7.148,7.250,7.354,7.465,7.591,7.729,7.866,
     * 7.050,7.163,7.268,7.366,7.457,7.548,7.649,7.759,
     * 7.055,7.174,7.286,7.390,7.481,7.551,7.610,7.660,
     * 7.057,7.191,7.324,7.451,7.557,7.636,7.687,7.710,
     * 7.060,7.203,7.350,7.490,7.606,7.694,7.746,7.763,
     * 7.060,7.215,7.376,7.532,7.661,7.759,7.815,7.830,
     * 0.000,0.000,0.000,0.000,6.224,6.506,6.733,6.907,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,7,L,2),I=1,8),L=1,NZEN)/
     * 6.986,7.076,7.155,7.234,7.316,7.415,7.537,7.673,
     * 6.987,7.080,7.162,7.236,7.313,7.395,7.498,7.618,
     * 7.000,7.098,7.189,7.272,7.347,7.416,7.492,7.574,
     * 7.017,7.120,7.222,7.321,7.415,7.493,7.562,7.617,
     * 7.027,7.148,7.272,7.394,7.510,7.606,7.680,7.727,
     * 7.029,7.159,7.297,7.435,7.560,7.662,7.734,7.772,
     * 7.028,7.169,7.326,7.481,7.617,7.726,7.799,7.832,
     * 0.000,0.000,0.000,0.000,6.123,6.406,6.642,6.831,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,8,L,2),I=1,8),L=1,NZEN)/
     * 6.999,7.189,7.409,7.644,7.869,8.065,8.229,8.355,
     * 6.966,7.123,7.306,7.523,7.747,7.955,8.130,8.267,
     * 6.950,7.087,7.227,7.392,7.583,7.784,7.970,8.122,
     * 6.943,7.078,7.205,7.326,7.437,7.552,7.688,7.828,
     * 6.928,7.076,7.218,7.351,7.456,7.529,7.566,7.574,
     * 6.923,7.078,7.230,7.371,7.483,7.558,7.591,7.588,
     * 6.917,7.082,7.247,7.400,7.522,7.598,7.633,7.624,
     * 0.000,0.000,0.000,0.000,6.111,6.372,6.571,6.719,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,8,L,2),I=1,8),L=1,NZEN)/
     * 7.012,7.196,7.409,7.635,7.852,8.042,8.202,8.326,
     * 6.978,7.130,7.304,7.514,7.731,7.931,8.103,8.239,
     * 6.959,7.094,7.230,7.389,7.571,7.765,7.947,8.098,
     * 6.949,7.082,7.210,7.330,7.441,7.555,7.684,7.824,
     * 6.939,7.085,7.228,7.363,7.473,7.548,7.593,7.600,
     * 6.936,7.089,7.244,7.388,7.508,7.586,7.626,7.625,
     * 6.933,7.097,7.265,7.423,7.550,7.638,7.676,7.673,
     * 0.000,0.000,0.000,0.000,6.181,6.455,6.663,6.817,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,8,L,2),I=1,8),L=1,NZEN)/
     * 7.043,7.192,7.364,7.557,7.753,7.935,8.096,8.227,
     * 7.029,7.160,7.302,7.467,7.652,7.834,8.001,8.141,
     * 7.020,7.144,7.262,7.388,7.530,7.688,7.852,8.000,
     * 7.015,7.140,7.256,7.367,7.460,7.550,7.646,7.755,
     * 7.009,7.146,7.282,7.407,7.513,7.590,7.631,7.644,
     * 7.008,7.154,7.301,7.438,7.554,7.633,7.673,7.679,
     * 7.005,7.163,7.325,7.477,7.604,7.690,7.735,7.738,
     * 0.000,0.000,0.000,0.000,6.232,6.508,6.722,6.882,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,8,L,2),I=1,8),L=1,NZEN)/
     * 7.091,7.201,7.316,7.448,7.598,7.755,7.911,8.051,
     * 7.092,7.196,7.300,7.409,7.535,7.677,7.827,7.968,
     * 7.096,7.201,7.300,7.393,7.490,7.595,7.718,7.845,
     * 7.104,7.216,7.322,7.418,7.502,7.569,7.632,7.695,
     * 7.105,7.233,7.359,7.475,7.573,7.648,7.693,7.712,
     * 7.104,7.239,7.378,7.507,7.617,7.696,7.740,7.754,
     * 7.105,7.253,7.407,7.553,7.674,7.762,7.813,7.823,
     * 0.000,0.000,0.000,0.000,6.249,6.524,6.744,6.914,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,8,L,2),I=1,8),L=1,NZEN)/
     * 7.097,7.180,7.255,7.339,7.436,7.555,7.693,7.837,
     * 7.095,7.181,7.253,7.324,7.405,7.504,7.629,7.766,
     * 7.102,7.189,7.263,7.331,7.399,7.472,7.564,7.673,
     * 7.119,7.218,7.305,7.392,7.468,7.535,7.596,7.650,
     * 7.124,7.236,7.347,7.456,7.556,7.635,7.694,7.730,
     * 7.126,7.247,7.372,7.494,7.604,7.692,7.753,7.783,
     * 7.124,7.258,7.398,7.538,7.661,7.756,7.821,7.845,
     * 0.000,0.000,0.000,0.000,6.181,6.456,6.685,6.869,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,9,L,2),I=1,8),L=1,NZEN)/
     * 6.967,7.150,7.362,7.600,7.826,8.026,8.193,8.326,
     * 6.940,7.093,7.269,7.481,7.702,7.912,8.093,8.236,
     * 6.925,7.064,7.202,7.360,7.543,7.743,7.930,8.086,
     * 6.918,7.057,7.188,7.310,7.421,7.530,7.654,7.793,
     * 6.906,7.056,7.202,7.335,7.443,7.518,7.556,7.566,
     * 6.901,7.057,7.212,7.356,7.470,7.547,7.582,7.579,
     * 6.894,7.062,7.230,7.384,7.510,7.587,7.624,7.618,
     * 0.000,0.000,0.000,0.000,6.084,6.347,6.546,6.696,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,9,L,2),I=1,8),L=1,NZEN)/
     * 6.977,7.154,7.361,7.587,7.810,8.007,8.174,8.305,
     * 6.949,7.098,7.269,7.471,7.690,7.896,8.075,8.217,
     * 6.936,7.071,7.207,7.359,7.535,7.730,7.914,8.069,
     * 6.928,7.065,7.193,7.313,7.424,7.529,7.650,7.786,
     * 6.918,7.066,7.210,7.344,7.453,7.530,7.567,7.577,
     * 6.914,7.069,7.222,7.366,7.479,7.560,7.598,7.597,
     * 6.908,7.074,7.241,7.398,7.522,7.607,7.644,7.640,
     * 0.000,0.000,0.000,0.000,6.119,6.386,6.591,6.743,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,9,L,2),I=1,8),L=1,NZEN)/
     * 7.054,7.206,7.383,7.586,7.787,7.973,8.135,8.268,
     * 7.036,7.168,7.315,7.489,7.681,7.868,8.038,8.181,
     * 7.029,7.150,7.271,7.399,7.549,7.717,7.886,8.036,
     * 7.023,7.146,7.260,7.369,7.464,7.554,7.656,7.773,
     * 7.014,7.149,7.280,7.402,7.501,7.573,7.609,7.619,
     * 7.011,7.152,7.294,7.427,7.534,7.606,7.643,7.645,
     * 7.005,7.160,7.317,7.463,7.579,7.657,7.694,7.693,
     * 0.000,0.000,0.000,0.000,6.180,6.444,6.649,6.803,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,9,L,2),I=1,8),L=1,NZEN)/
     * 7.186,7.304,7.435,7.592,7.758,7.921,8.073,8.208,
     * 7.172,7.276,7.384,7.516,7.668,7.827,7.984,8.124,
     * 7.175,7.271,7.363,7.462,7.574,7.708,7.853,7.992,
     * 7.172,7.273,7.368,7.452,7.530,7.601,7.680,7.774,
     * 7.170,7.283,7.394,7.498,7.584,7.647,7.682,7.693,
     * 7.162,7.287,7.409,7.523,7.618,7.683,7.717,7.722,
     * 7.158,7.294,7.430,7.558,7.665,7.736,7.771,7.772,
     * 0.000,0.000,0.000,0.000,6.247,6.505,6.710,6.870,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,9,L,2),I=1,8),L=1,NZEN)/
     * 7.283,7.368,7.461,7.576,7.706,7.842,7.982,8.118,
     * 7.268,7.343,7.417,7.510,7.627,7.759,7.900,8.036,
     * 7.263,7.331,7.391,7.459,7.542,7.649,7.779,7.915,
     * 7.266,7.340,7.405,7.465,7.521,7.577,7.647,7.734,
     * 7.269,7.357,7.445,7.528,7.600,7.656,7.696,7.716,
     * 7.261,7.359,7.459,7.553,7.635,7.695,7.734,7.745,
     * 7.256,7.367,7.481,7.590,7.683,7.750,7.791,7.799,
     * 0.000,0.000,0.000,0.000,6.235,6.488,6.698,6.869,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,10,L,2),I=1,8),L=1,NZEN)/
     * 6.952,7.131,7.339,7.571,7.797,8.000,8.171,8.305,
     * 6.927,7.080,7.250,7.456,7.675,7.885,8.067,8.213,
     * 6.914,7.054,7.192,7.345,7.522,7.715,7.902,8.061,
     * 6.907,7.047,7.179,7.301,7.409,7.514,7.635,7.769,
     * 6.894,7.046,7.193,7.329,7.439,7.514,7.553,7.560,
     * 6.890,7.049,7.205,7.348,7.466,7.543,7.576,7.576,
     * 6.884,7.052,7.222,7.379,7.501,7.584,7.617,7.613,
     * 0.000,0.000,0.000,0.000,6.066,6.327,6.527,6.677,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,10,L,2),I=1,8),L=1,NZEN)/
     * 6.953,7.126,7.331,7.558,7.783,7.987,8.160,8.297,
     * 6.931,7.079,7.245,7.445,7.662,7.872,8.055,8.203,
     * 6.920,7.055,7.190,7.339,7.514,7.704,7.891,8.051,
     * 6.914,7.052,7.180,7.300,7.405,7.508,7.624,7.756,
     * 6.903,7.051,7.194,7.326,7.433,7.504,7.544,7.553,
     * 6.899,7.054,7.206,7.346,7.459,7.532,7.568,7.568,
     * 6.891,7.057,7.222,7.375,7.496,7.574,7.608,7.605,
     * 0.000,0.000,0.000,0.000,6.054,6.313,6.511,6.662,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,10,L,2),I=1,8),L=1,NZEN)/
     * 7.063,7.220,7.402,7.611,7.817,8.007,8.172,8.307,
     * 7.042,7.174,7.323,7.506,7.705,7.897,8.070,8.213,
     * 7.031,7.151,7.269,7.405,7.562,7.737,7.911,8.064,
     * 7.026,7.146,7.258,7.363,7.458,7.551,7.659,7.784,
     * 7.013,7.145,7.271,7.387,7.482,7.544,7.578,7.587,
     * 7.007,7.147,7.284,7.407,7.508,7.572,7.601,7.601,
     * 6.998,7.148,7.296,7.435,7.543,7.610,7.641,7.635,
     * 0.000,0.000,0.000,0.000,6.105,6.355,6.548,6.696,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,10,L,2),I=1,8),L=1,NZEN)/
     * 7.270,7.405,7.559,7.732,7.905,8.067,8.213,8.345,
     * 7.245,7.353,7.479,7.633,7.801,7.969,8.123,8.260,
     * 7.229,7.319,7.415,7.528,7.667,7.822,7.979,8.120,
     * 7.221,7.313,7.396,7.475,7.550,7.637,7.743,7.866,
     * 7.208,7.312,7.409,7.498,7.571,7.618,7.646,7.653,
     * 7.197,7.308,7.414,7.512,7.589,7.639,7.660,7.657,
     * 7.186,7.309,7.429,7.540,7.625,7.680,7.703,7.693,
     * 0.000,0.000,0.000,0.000,6.203,6.438,6.625,6.774,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,10,L,2),I=1,8),L=1,NZEN)/
     * 7.471,7.583,7.713,7.852,7.991,8.126,8.263,8.382,
     * 7.426,7.515,7.623,7.753,7.894,8.036,8.172,8.303,
     * 7.400,7.463,7.535,7.634,7.758,7.896,8.038,8.178,
     * 7.383,7.439,7.486,7.538,7.599,7.685,7.799,7.926,
     * 7.364,7.431,7.495,7.552,7.596,7.628,7.648,7.661,
     * 7.352,7.427,7.497,7.564,7.616,7.649,7.663,7.660,
     * 7.335,7.420,7.506,7.583,7.643,7.678,7.691,7.682,
     * 0.000,0.000,0.000,6.015,6.231,6.451,6.635,6.787,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,11,L,2),I=1,8),L=1,NZEN)/
     * 6.973,7.154,7.362,7.593,7.818,8.016,8.184,8.316,
     * 6.946,7.099,7.271,7.475,7.694,7.900,8.079,8.222,
     * 6.933,7.071,7.207,7.360,7.537,7.729,7.915,8.071,
     * 6.925,7.064,7.193,7.313,7.421,7.524,7.644,7.777,
     * 6.913,7.063,7.207,7.339,7.448,7.518,7.556,7.563,
     * 6.907,7.064,7.219,7.360,7.470,7.547,7.579,7.579,
     * 6.902,7.069,7.235,7.389,7.510,7.591,7.624,7.618,
     * 0.000,0.000,0.000,0.000,6.072,6.334,6.532,6.681,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,11,L,2),I=1,8),L=1,NZEN)/
     * 6.956,7.129,7.334,7.566,7.792,7.998,8.172,8.314,
     * 6.935,7.082,7.248,7.450,7.669,7.879,8.064,8.213,
     * 6.923,7.057,7.190,7.340,7.516,7.709,7.898,8.059,
     * 6.918,7.055,7.180,7.298,7.401,7.502,7.618,7.754,
     * 6.907,7.053,7.194,7.322,7.422,7.493,7.527,7.534,
     * 6.903,7.056,7.207,7.343,7.451,7.519,7.550,7.545,
     * 6.894,7.058,7.219,7.367,7.483,7.554,7.583,7.577,
     * 0.000,0.000,0.000,0.000,6.008,6.259,6.450,6.597,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,11,L,2),I=1,8),L=1,NZEN)/
     * 7.071,7.228,7.419,7.632,7.843,8.034,8.200,8.335,
     * 7.046,7.180,7.335,7.522,7.725,7.921,8.095,8.239,
     * 7.033,7.151,7.268,7.408,7.571,7.753,7.931,8.086,
     * 7.023,7.141,7.248,7.351,7.446,7.543,7.661,7.797,
     * 7.009,7.138,7.259,7.372,7.458,7.516,7.544,7.550,
     * 6.995,7.130,7.260,7.378,7.470,7.529,7.550,7.547,
     * 6.988,7.134,7.278,7.407,7.504,7.565,7.588,7.578,
     * 0.000,0.000,0.000,0.000,6.027,6.261,6.443,6.587,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,11,L,2),I=1,8),L=1,NZEN)/
     * 7.326,7.468,7.637,7.816,7.992,8.154,8.306,8.428,
     * 7.287,7.401,7.539,7.707,7.882,8.050,8.205,8.333,
     * 7.263,7.353,7.451,7.577,7.731,7.894,8.054,8.195,
     * 7.247,7.330,7.406,7.482,7.560,7.660,7.784,7.920,
     * 7.218,7.310,7.398,7.473,7.532,7.570,7.590,7.596,
     * 7.200,7.298,7.391,7.475,7.537,7.574,7.582,7.573,
     * 7.182,7.291,7.398,7.490,7.558,7.595,7.604,7.586,
     * 0.000,0.000,0.000,0.000,6.126,6.337,6.505,6.643,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,11,L,2),I=1,8),L=1,NZEN)/
     * 7.617,7.752,7.893,8.035,8.173,8.305,8.431,8.551,
     * 7.545,7.656,7.786,7.929,8.072,8.210,8.348,8.465,
     * 7.489,7.561,7.657,7.780,7.922,8.067,8.206,8.332,
     * 7.453,7.501,7.547,7.605,7.689,7.804,7.939,8.078,
     * 7.413,7.465,7.511,7.549,7.577,7.593,7.606,7.629,
     * 7.389,7.446,7.499,7.543,7.571,7.582,7.578,7.566,
     * 7.363,7.430,7.492,7.542,7.574,7.583,7.575,7.551,
     * 0.000,0.000,0.000,0.000,6.163,6.348,6.503,6.633,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,12,L,2),I=1,8),L=1,NZEN)/
     * 7.011,7.198,7.412,7.642,7.860,8.054,8.215,8.343,
     * 6.979,7.135,7.310,7.520,7.736,7.938,8.113,8.250,
     * 6.964,7.101,7.237,7.392,7.572,7.765,7.947,8.100,
     * 6.955,7.091,7.217,7.333,7.437,7.546,7.668,7.805,
     * 6.942,7.090,7.230,7.358,7.460,7.529,7.566,7.571,
     * 6.937,7.092,7.242,7.378,7.487,7.558,7.588,7.585,
     * 6.930,7.095,7.259,7.409,7.526,7.602,7.633,7.624,
     * 0.000,0.000,0.000,0.000,6.092,6.353,6.550,6.698,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,12,L,2),I=1,8),L=1,NZEN)/
     * 6.976,7.152,7.361,7.591,7.820,8.026,8.198,8.334,
     * 6.951,7.098,7.267,7.473,7.695,7.906,8.089,8.236,
     * 6.936,7.068,7.200,7.354,7.536,7.734,7.923,8.084,
     * 6.934,7.067,7.190,7.305,7.408,7.509,7.630,7.767,
     * 6.922,7.067,7.204,7.327,7.423,7.488,7.519,7.525,
     * 6.912,7.062,7.207,7.338,7.441,7.506,7.532,7.529,
     * 6.907,7.070,7.229,7.372,7.481,7.547,7.572,7.564,
     * 0.000,0.000,0.000,0.000,0.000,6.234,6.422,6.566,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,12,L,2),I=1,8),L=1,NZEN)/
     * 7.076,7.237,7.430,7.646,7.859,8.051,8.218,8.355,
     * 7.048,7.182,7.340,7.531,7.738,7.936,8.110,8.252,
     * 7.033,7.150,7.268,7.410,7.579,7.765,7.944,8.099,
     * 7.020,7.138,7.246,7.346,7.442,7.538,7.661,7.801,
     * 6.997,7.123,7.241,7.348,7.431,7.486,7.512,7.515,
     * 6.988,7.120,7.246,7.359,7.446,7.496,7.516,7.510,
     * 6.970,7.112,7.252,7.372,7.464,7.515,7.534,7.521,
     * 0.000,0.000,0.000,0.000,0.000,6.199,6.373,6.514,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,12,L,2),I=1,8),L=1,NZEN)/
     * 7.335,7.481,7.650,7.832,8.009,8.171,8.324,8.446,
     * 7.292,7.406,7.549,7.720,7.896,8.065,8.218,8.349,
     * 7.262,7.354,7.452,7.582,7.738,7.904,8.063,8.205,
     * 7.243,7.325,7.398,7.471,7.552,7.653,7.781,7.920,
     * 7.203,7.293,7.372,7.443,7.496,7.527,7.541,7.547,
     * 7.180,7.275,7.363,7.438,7.491,7.520,7.522,7.509,
     * 7.158,7.260,7.359,7.440,7.497,7.523,7.523,7.502,
     * 0.000,0.000,0.000,0.000,6.054,6.249,6.403,6.530,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,12,L,2),I=1,8),L=1,NZEN)/
     * 7.658,7.797,7.940,8.081,8.219,8.353,8.476,8.593,
     * 7.577,7.694,7.827,7.971,8.114,8.250,8.375,8.499,
     * 7.511,7.587,7.688,7.815,7.956,8.100,8.237,8.364,
     * 7.468,7.513,7.557,7.616,7.704,7.822,7.959,8.098,
     * 7.415,7.463,7.503,7.532,7.550,7.558,7.566,7.594,
     * 7.385,7.436,7.480,7.513,7.529,7.529,7.517,7.500,
     * 7.354,7.413,7.465,7.503,7.520,7.516,7.496,7.467,
     * 0.000,0.000,0.000,0.000,6.073,6.236,6.360,6.477,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,1,L,3),I=1,8),L=1,NZEN)/
     * 7.069,7.268,7.492,7.728,7.949,8.143,8.304,8.432,
     * 7.031,7.197,7.385,7.604,7.826,8.031,8.205,8.343,
     * 7.009,7.152,7.298,7.466,7.661,7.862,8.047,8.201,
     * 6.990,7.132,7.263,7.385,7.501,7.624,7.762,7.911,
     * 6.959,7.114,7.261,7.398,7.510,7.585,7.630,7.645,
     * 6.950,7.110,7.265,7.409,7.527,7.608,7.650,7.656,
     * 6.945,7.113,7.279,7.432,7.556,7.638,7.679,7.683,
     * 0.000,0.000,0.000,0.000,6.159,6.429,6.637,6.801,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,1,L,3),I=1,8),L=1,NZEN)/
     * 7.011,7.201,7.425,7.668,7.901,8.109,8.283,8.420,
     * 6.977,7.136,7.320,7.544,7.778,7.993,8.179,8.328,
     * 6.954,7.094,7.238,7.407,7.610,7.822,8.018,8.182,
     * 6.942,7.082,7.212,7.335,7.448,7.570,7.712,7.867,
     * 6.912,7.063,7.207,7.340,7.445,7.518,7.560,7.576,
     * 6.897,7.051,7.203,7.341,7.455,7.528,7.565,7.573,
     * 6.893,7.058,7.222,7.370,7.485,7.559,7.595,7.596,
     * 0.000,0.000,0.000,0.000,6.024,6.278,6.476,6.637,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,1,L,3),I=1,8),L=1,NZEN)/
     * 7.067,7.249,7.465,7.700,7.924,8.123,8.295,8.431,
     * 7.029,7.179,7.359,7.576,7.800,8.007,8.187,8.335,
     * 7.006,7.133,7.266,7.430,7.627,7.832,8.022,8.182,
     * 6.984,7.107,7.223,7.335,7.444,7.570,7.724,7.887,
     * 6.951,7.083,7.208,7.321,7.415,7.475,7.511,7.527,
     * 6.938,7.075,7.207,7.328,7.421,7.482,7.508,7.515,
     * 6.922,7.066,7.206,7.334,7.431,7.488,7.515,7.517,
     * 0.000,0.000,0.000,0.000,0.000,6.200,6.384,6.538,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,1,L,3),I=1,8),L=1,NZEN)/
     * 7.304,7.473,7.663,7.862,8.050,8.222,8.375,8.503,
     * 7.249,7.385,7.549,7.740,7.933,8.112,8.277,8.410,
     * 7.216,7.315,7.430,7.582,7.762,7.945,8.115,8.266,
     * 7.188,7.275,7.355,7.438,7.536,7.663,7.817,7.973,
     * 7.148,7.241,7.324,7.401,7.457,7.495,7.516,7.535,
     * 7.122,7.221,7.312,7.391,7.449,7.483,7.492,7.489,
     * 7.100,7.204,7.305,7.389,7.450,7.482,7.488,7.475,
     * 0.000,0.000,0.000,0.000,6.021,6.220,6.380,6.519,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,1,L,3),I=1,8),L=1,NZEN)/
     * 7.613,7.774,7.934,8.089,8.235,8.365,8.499,8.625,
     * 7.515,7.655,7.811,7.971,8.125,8.270,8.403,8.532,
     * 7.430,7.524,7.648,7.799,7.957,8.114,8.267,8.393,
     * 7.377,7.426,7.480,7.557,7.669,7.812,7.968,8.118,
     * 7.319,7.371,7.411,7.446,7.466,7.481,7.501,7.551,
     * 7.286,7.341,7.387,7.424,7.442,7.447,7.440,7.434,
     * 7.252,7.313,7.366,7.407,7.427,7.427,7.410,7.392,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.420,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,2,L,3),I=1,8),L=1,NZEN)/
     * 7.037,7.229,7.448,7.685,7.909,8.106,8.273,8.404,
     * 7.004,7.166,7.347,7.565,7.787,7.995,8.174,8.317,
     * 6.984,7.128,7.272,7.435,7.625,7.825,8.014,8.171,
     * 6.966,7.110,7.244,7.369,7.484,7.599,7.735,7.882,
     * 6.935,7.092,7.240,7.380,7.494,7.575,7.619,7.637,
     * 6.926,7.088,7.246,7.394,7.514,7.597,7.641,7.648,
     * 6.922,7.092,7.259,7.417,7.542,7.627,7.669,7.678,
     * 0.000,0.000,0.000,0.000,6.140,6.410,6.619,6.784,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,2,L,3),I=1,8),L=1,NZEN)/
     * 6.994,7.182,7.399,7.641,7.874,8.082,8.257,8.397,
     * 6.964,7.121,7.304,7.520,7.751,7.966,8.152,8.301,
     * 6.946,7.085,7.228,7.393,7.587,7.796,7.990,8.153,
     * 6.929,7.070,7.202,7.326,7.439,7.556,7.694,7.845,
     * 6.898,7.051,7.197,7.332,7.442,7.514,7.561,7.576,
     * 6.884,7.040,7.194,7.336,7.449,7.529,7.568,7.577,
     * 6.881,7.046,7.210,7.360,7.477,7.555,7.594,7.597,
     * 0.000,0.000,0.000,0.000,6.028,6.285,6.485,6.646,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,2,L,3),I=1,8),L=1,NZEN)/
     * 7.054,7.233,7.444,7.675,7.895,8.093,8.262,8.402,
     * 7.020,7.167,7.344,7.553,7.773,7.979,8.159,8.304,
     * 6.998,7.125,7.260,7.415,7.606,7.806,7.994,8.154,
     * 6.980,7.105,7.221,7.332,7.440,7.558,7.701,7.858,
     * 6.946,7.080,7.208,7.326,7.418,7.481,7.520,7.537,
     * 6.935,7.072,7.207,7.331,7.425,7.489,7.521,7.528,
     * 6.920,7.066,7.209,7.340,7.438,7.503,7.533,7.532,
     * 0.000,0.000,0.000,0.000,0.000,6.224,6.411,6.567,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,2,L,3),I=1,8),L=1,NZEN)/
     * 7.263,7.425,7.608,7.803,7.991,8.163,8.318,8.448,
     * 7.217,7.345,7.500,7.686,7.875,8.054,8.215,8.351,
     * 7.186,7.286,7.396,7.539,7.710,7.888,8.057,8.205,
     * 7.167,7.256,7.338,7.420,7.512,7.626,7.769,7.919,
     * 7.126,7.225,7.317,7.396,7.460,7.500,7.524,7.540,
     * 7.107,7.211,7.308,7.393,7.457,7.495,7.511,7.510,
     * 7.089,7.198,7.304,7.400,7.467,7.505,7.516,7.508,
     * 0.000,0.000,0.000,0.000,6.033,6.240,6.407,6.550,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,2,L,3),I=1,8),L=1,NZEN)/
     * 7.501,7.653,7.810,7.966,8.114,8.262,8.394,8.515,
     * 7.420,7.545,7.691,7.848,8.004,8.152,8.287,8.418,
     * 7.360,7.441,7.545,7.683,7.836,7.992,8.140,8.276,
     * 7.320,7.375,7.423,7.487,7.578,7.702,7.850,8.000,
     * 7.270,7.327,7.376,7.418,7.446,7.465,7.480,7.510,
     * 7.240,7.302,7.357,7.400,7.431,7.442,7.442,7.436,
     * 7.208,7.277,7.340,7.391,7.420,7.428,7.422,7.406,
     * 0.000,0.000,0.000,0.000,0.000,6.146,6.297,6.433,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,3,L,3),I=1,8),L=1,NZEN)/
     * 6.989,7.174,7.384,7.621,7.850,8.054,8.229,8.367,
     * 6.962,7.122,7.297,7.506,7.731,7.943,8.127,8.277,
     * 6.945,7.092,7.234,7.393,7.578,7.774,7.967,8.130,
     * 6.927,7.075,7.213,7.344,7.461,7.573,7.699,7.843,
     * 6.898,7.056,7.210,7.355,7.472,7.557,7.606,7.626,
     * 6.890,7.053,7.216,7.368,7.492,7.579,7.625,7.640,
     * 6.885,7.058,7.231,7.391,7.521,7.611,7.656,7.667,
     * 0.000,0.000,0.000,0.000,6.110,6.382,6.592,6.758,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,3,L,3),I=1,8),L=1,NZEN)/
     * 6.969,7.151,7.361,7.598,7.830,8.038,8.214,8.354,
     * 6.943,7.097,7.273,7.484,7.710,7.926,8.113,8.265,
     * 6.926,7.069,7.210,7.369,7.553,7.756,7.949,8.113,
     * 6.909,7.054,7.188,7.315,7.430,7.542,7.672,7.816,
     * 6.878,7.032,7.183,7.323,7.437,7.517,7.564,7.584,
     * 6.868,7.030,7.187,7.334,7.453,7.536,7.578,7.590,
     * 6.862,7.030,7.198,7.352,7.476,7.562,7.602,7.609,
     * 0.000,0.000,0.000,0.000,6.043,6.306,6.511,6.675,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,3,L,3),I=1,8),L=1,NZEN)/
     * 7.035,7.206,7.408,7.632,7.848,8.045,8.215,8.352,
     * 7.006,7.149,7.317,7.516,7.730,7.932,8.112,8.257,
     * 6.989,7.115,7.245,7.394,7.574,7.766,7.951,8.111,
     * 6.975,7.103,7.221,7.334,7.435,7.544,7.673,7.817,
     * 6.947,7.084,7.217,7.337,7.438,7.505,7.545,7.564,
     * 6.933,7.075,7.215,7.344,7.448,7.516,7.553,7.563,
     * 6.926,7.077,7.227,7.363,7.472,7.544,7.579,7.585,
     * 0.000,0.000,0.000,0.000,6.032,6.279,6.474,6.634,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,3,L,3),I=1,8),L=1,NZEN)/
     * 7.215,7.363,7.534,7.723,7.907,8.077,8.231,8.364,
     * 7.178,7.299,7.441,7.614,7.796,7.971,8.131,8.274,
     * 7.158,7.257,7.360,7.489,7.643,7.812,7.978,8.126,
     * 7.142,7.240,7.325,7.410,7.494,7.591,7.715,7.852,
     * 7.115,7.219,7.319,7.412,7.483,7.534,7.563,7.578,
     * 7.101,7.210,7.320,7.417,7.493,7.545,7.569,7.573,
     * 7.086,7.206,7.324,7.431,7.512,7.564,7.585,7.585,
     * 0.000,0.000,0.000,0.000,6.077,6.301,6.484,6.637,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,3,L,3),I=1,8),L=1,NZEN)/
     * 7.377,7.510,7.657,7.811,7.961,8.106,8.248,8.371,
     * 7.322,7.425,7.551,7.699,7.851,8.001,8.145,8.282,
     * 7.283,7.358,7.441,7.556,7.694,7.845,7.995,8.134,
     * 7.259,7.320,7.374,7.431,7.503,7.599,7.724,7.862,
     * 7.222,7.293,7.356,7.414,7.456,7.488,7.510,7.529,
     * 7.201,7.276,7.346,7.409,7.457,7.486,7.500,7.503,
     * 7.180,7.264,7.344,7.416,7.467,7.496,7.507,7.502,
     * 0.000,0.000,0.000,0.000,6.005,6.209,6.381,6.533,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,4,L,3),I=1,8),L=1,NZEN)/
     * 6.959,7.140,7.347,7.583,7.814,8.023,8.201,8.343,
     * 6.935,7.093,7.266,7.472,7.698,7.912,8.101,8.254,
     * 6.919,7.067,7.211,7.368,7.549,7.748,7.941,8.107,
     * 6.901,7.052,7.193,7.326,7.445,7.554,7.679,7.820,
     * 6.872,7.034,7.191,7.339,7.459,7.546,7.597,7.618,
     * 6.865,7.031,7.196,7.351,7.479,7.567,7.618,7.631,
     * 6.861,7.036,7.211,7.374,7.508,7.599,7.649,7.658,
     * 0.000,0.000,0.000,0.000,6.094,6.366,6.577,6.745,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,4,L,3),I=1,8),L=1,NZEN)/
     * 6.958,7.137,7.346,7.578,7.804,8.009,8.184,8.326,
     * 6.934,7.089,7.261,7.468,7.687,7.898,8.083,8.234,
     * 6.917,7.061,7.203,7.360,7.538,7.733,7.923,8.087,
     * 6.898,7.045,7.182,7.312,7.430,7.539,7.662,7.803,
     * 6.867,7.025,7.180,7.323,7.442,7.528,7.575,7.598,
     * 6.857,7.020,7.182,7.335,7.460,7.549,7.596,7.611,
     * 6.853,7.025,7.196,7.357,7.489,7.581,7.627,7.640,
     * 0.000,0.000,0.000,0.000,6.079,6.350,6.562,6.730,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,4,L,3),I=1,8),L=1,NZEN)/
     * 7.014,7.176,7.364,7.580,7.791,7.985,8.153,8.292,
     * 6.992,7.132,7.289,7.474,7.678,7.876,8.053,8.201,
     * 6.980,7.108,7.235,7.378,7.540,7.719,7.898,8.055,
     * 6.966,7.097,7.219,7.335,7.435,7.537,7.650,7.780,
     * 6.942,7.084,7.222,7.350,7.456,7.531,7.576,7.595,
     * 6.934,7.082,7.227,7.365,7.476,7.556,7.598,7.612,
     * 6.928,7.086,7.242,7.390,7.510,7.590,7.632,7.641,
     * 0.000,0.000,0.000,0.000,6.093,6.355,6.562,6.728,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,4,L,3),I=1,8),L=1,NZEN)/
     * 7.140,7.271,7.424,7.597,7.775,7.944,8.100,8.233,
     * 7.121,7.233,7.357,7.505,7.672,7.843,8.004,8.145,
     * 7.112,7.216,7.315,7.425,7.555,7.703,7.860,8.006,
     * 7.108,7.215,7.313,7.403,7.487,7.567,7.660,7.768,
     * 7.091,7.210,7.325,7.429,7.517,7.581,7.622,7.642,
     * 7.085,7.210,7.333,7.447,7.543,7.609,7.648,7.661,
     * 7.078,7.214,7.350,7.476,7.577,7.648,7.687,7.696,
     * 0.000,0.000,0.000,0.000,6.129,6.379,6.581,6.748,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,4,L,3),I=1,8),L=1,NZEN)/
     * 7.218,7.321,7.442,7.582,7.729,7.875,8.016,8.150,
     * 7.193,7.281,7.374,7.494,7.631,7.776,7.924,8.063,
     * 7.179,7.254,7.326,7.410,7.514,7.641,7.783,7.923,
     * 7.169,7.247,7.315,7.380,7.442,7.509,7.594,7.698,
     * 7.157,7.248,7.334,7.418,7.488,7.544,7.587,7.614,
     * 7.145,7.244,7.340,7.433,7.512,7.571,7.611,7.630,
     * 7.133,7.243,7.354,7.457,7.544,7.607,7.648,7.662,
     * 0.000,0.000,0.000,0.000,6.039,6.277,6.481,6.657,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,5,L,3),I=1,8),L=1,NZEN)/
     * 6.975,7.159,7.373,7.609,7.841,8.050,8.224,8.363,
     * 6.948,7.108,7.284,7.497,7.723,7.940,8.127,8.278,
     * 6.932,7.079,7.224,7.385,7.572,7.774,7.968,8.132,
     * 6.914,7.063,7.202,7.335,7.453,7.570,7.699,7.846,
     * 6.885,7.044,7.200,7.347,7.468,7.553,7.603,7.624,
     * 6.877,7.043,7.207,7.361,7.488,7.575,7.625,7.637,
     * 6.873,7.046,7.220,7.384,7.515,7.607,7.656,7.664,
     * 0.000,0.000,0.000,0.000,6.112,6.385,6.596,6.763,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,5,L,3),I=1,8),L=1,NZEN)/
     * 6.981,7.163,7.375,7.605,7.828,8.028,8.198,8.334,
     * 6.950,7.108,7.282,7.490,7.710,7.918,8.101,8.247,
     * 6.930,7.074,7.217,7.376,7.558,7.755,7.943,8.104,
     * 6.910,7.055,7.193,7.324,7.441,7.554,7.685,7.830,
     * 6.880,7.038,7.193,7.340,7.459,7.549,7.601,7.624,
     * 6.873,7.036,7.200,7.354,7.483,7.576,7.626,7.644,
     * 6.870,7.043,7.215,7.380,7.516,7.614,7.663,7.677,
     * 0.000,0.000,0.000,0.000,6.140,6.421,6.638,6.810,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,5,L,3),I=1,8),L=1,NZEN)/
     * 7.002,7.161,7.341,7.546,7.753,7.944,8.112,8.247,
     * 6.986,7.126,7.276,7.455,7.649,7.841,8.015,8.161,
     * 6.973,7.103,7.230,7.367,7.518,7.689,7.862,8.016,
     * 6.958,7.092,7.217,7.335,7.436,7.532,7.639,7.760,
     * 6.939,7.084,7.226,7.360,7.471,7.554,7.603,7.625,
     * 6.936,7.087,7.239,7.382,7.501,7.590,7.636,7.653,
     * 6.933,7.095,7.259,7.414,7.541,7.632,7.684,7.695,
     * 0.000,0.000,0.000,0.000,6.157,6.434,6.653,6.825,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,5,L,3),I=1,8),L=1,NZEN)/
     * 7.060,7.180,7.313,7.465,7.630,7.796,7.956,8.095,
     * 7.052,7.163,7.274,7.401,7.547,7.705,7.864,8.009,
     * 7.057,7.168,7.270,7.374,7.484,7.606,7.744,7.882,
     * 7.058,7.175,7.283,7.387,7.476,7.554,7.627,7.705,
     * 7.052,7.181,7.310,7.429,7.533,7.611,7.661,7.686,
     * 7.050,7.187,7.327,7.455,7.568,7.650,7.699,7.718,
     * 7.050,7.199,7.350,7.497,7.620,7.706,7.759,7.776,
     * 0.000,0.000,0.000,0.000,6.166,6.436,6.656,6.834,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,5,L,3),I=1,8),L=1,NZEN)/
     * 7.059,7.150,7.244,7.353,7.478,7.617,7.763,7.904,
     * 7.053,7.142,7.222,7.310,7.418,7.540,7.679,7.820,
     * 7.049,7.139,7.216,7.291,7.369,7.461,7.574,7.701,
     * 7.060,7.153,7.243,7.329,7.410,7.480,7.548,7.617,
     * 7.058,7.168,7.278,7.384,7.482,7.565,7.631,7.675,
     * 7.055,7.170,7.293,7.412,7.522,7.609,7.675,7.712,
     * 7.052,7.179,7.318,7.451,7.570,7.664,7.729,7.762,
     * 0.000,0.000,0.000,0.000,6.048,6.314,6.539,6.731,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,6,L,3),I=1,8),L=1,NZEN)/
     * 7.021,7.214,7.435,7.677,7.906,8.108,8.277,8.411,
     * 6.987,7.150,7.339,7.560,7.788,8.001,8.183,8.326,
     * 6.967,7.111,7.259,7.431,7.627,7.835,8.026,8.186,
     * 6.950,7.094,7.230,7.360,7.480,7.605,7.747,7.899,
     * 6.920,7.076,7.228,7.371,7.489,7.571,7.619,7.637,
     * 6.912,7.073,7.233,7.384,7.506,7.594,7.638,7.648,
     * 6.906,7.076,7.246,7.406,7.535,7.627,7.669,7.678,
     * 0.000,0.000,0.000,0.000,6.150,6.422,6.633,6.798,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,6,L,3),I=1,8),L=1,NZEN)/
     * 7.023,7.215,7.431,7.667,7.886,8.080,8.243,8.368,
     * 6.986,7.147,7.328,7.544,7.765,7.969,8.144,8.285,
     * 6.962,7.104,7.247,7.412,7.603,7.804,7.991,8.148,
     * 6.939,7.082,7.216,7.347,7.465,7.590,7.730,7.879,
     * 6.910,7.065,7.218,7.365,7.487,7.573,7.628,7.650,
     * 6.902,7.063,7.227,7.383,7.513,7.605,7.658,7.672,
     * 6.900,7.069,7.243,7.410,7.548,7.647,7.696,7.710,
     * 0.000,0.000,0.000,0.000,6.205,6.492,6.713,6.885,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,6,L,3),I=1,8),L=1,NZEN)/
     * 7.004,7.161,7.341,7.544,7.746,7.934,8.099,8.235,
     * 6.990,7.128,7.278,7.452,7.645,7.833,8.006,8.151,
     * 6.976,7.107,7.233,7.368,7.518,7.685,7.856,8.009,
     * 6.961,7.095,7.221,7.339,7.443,7.539,7.642,7.762,
     * 6.944,7.091,7.234,7.372,7.486,7.574,7.626,7.650,
     * 6.940,7.094,7.248,7.396,7.522,7.614,7.669,7.688,
     * 6.944,7.107,7.275,7.437,7.573,7.668,7.725,7.740,
     * 0.000,0.000,0.000,0.000,6.216,6.504,6.732,6.908,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,6,L,3),I=1,8),L=1,NZEN)/
     * 7.000,7.118,7.237,7.375,7.530,7.692,7.853,8.000,
     * 7.002,7.117,7.226,7.341,7.473,7.617,7.770,7.917,
     * 7.012,7.129,7.239,7.343,7.446,7.553,7.675,7.801,
     * 7.013,7.136,7.253,7.364,7.457,7.534,7.602,7.665,
     * 7.012,7.149,7.286,7.415,7.527,7.611,7.667,7.697,
     * 7.017,7.161,7.308,7.452,7.576,7.666,7.722,7.746,
     * 7.020,7.176,7.339,7.496,7.629,7.728,7.787,7.809,
     * 0.000,0.000,0.000,0.000,6.196,6.481,6.712,6.896,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,6,L,3),I=1,8),L=1,NZEN)/
     * 6.934,7.028,7.115,7.206,7.310,7.433,7.574,7.723,
     * 6.933,7.029,7.115,7.195,7.285,7.387,7.512,7.651,
     * 6.945,7.044,7.136,7.221,7.305,7.387,7.476,7.578,
     * 6.957,7.062,7.165,7.265,7.363,7.445,7.523,7.587,
     * 6.963,7.082,7.207,7.332,7.448,7.547,7.628,7.683,
     * 6.965,7.094,7.230,7.371,7.496,7.601,7.679,7.724,
     * 6.966,7.108,7.260,7.416,7.554,7.663,7.739,7.778,
     * 0.000,0.000,0.000,0.000,6.050,6.332,6.570,6.767,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,7,L,3),I=1,8),L=1,NZEN)/
     * 7.053,7.253,7.481,7.721,7.949,8.148,8.313,8.442,
     * 7.014,7.180,7.374,7.602,7.832,8.042,8.219,8.359,
     * 6.990,7.133,7.284,7.463,7.665,7.877,8.066,8.220,
     * 6.971,7.114,7.248,7.376,7.498,7.630,7.781,7.935,
     * 6.943,7.096,7.245,7.385,7.502,7.582,7.628,7.647,
     * 6.932,7.092,7.250,7.398,7.518,7.605,7.647,7.656,
     * 6.928,7.093,7.262,7.420,7.548,7.638,7.679,7.686,
     * 0.000,0.000,0.000,0.000,6.175,6.447,6.657,6.821,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,7,L,3),I=1,8),L=1,NZEN)/
     * 7.050,7.246,7.470,7.705,7.924,8.115,8.274,8.400,
     * 7.008,7.171,7.360,7.580,7.802,8.005,8.178,8.314,
     * 6.979,7.121,7.267,7.438,7.638,7.841,8.026,8.180,
     * 6.955,7.097,7.231,7.360,7.482,7.614,7.759,7.913,
     * 6.927,7.081,7.234,7.378,7.498,7.588,7.638,7.663,
     * 6.921,7.081,7.243,7.398,7.528,7.621,7.670,7.684,
     * 6.917,7.086,7.260,7.425,7.563,7.661,7.713,7.724,
     * 0.000,0.000,0.000,0.000,6.236,6.524,6.746,6.918,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,7,L,3),I=1,8),L=1,NZEN)/
     * 7.014,7.173,7.355,7.561,7.766,7.953,8.118,8.252,
     * 6.998,7.137,7.290,7.469,7.662,7.853,8.026,8.171,
     * 6.985,7.115,7.240,7.376,7.533,7.705,7.877,8.032,
     * 6.970,7.102,7.228,7.346,7.450,7.551,7.659,7.783,
     * 6.954,7.100,7.244,7.381,7.496,7.584,7.637,7.664,
     * 6.950,7.104,7.260,7.408,7.534,7.628,7.683,7.702,
     * 6.955,7.118,7.286,7.448,7.584,7.687,7.742,7.760,
     * 0.000,0.000,0.000,0.000,6.246,6.539,6.769,6.946,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,7,L,3),I=1,8),L=1,NZEN)/
     * 6.996,7.113,7.233,7.370,7.527,7.692,7.857,8.007,
     * 6.999,7.113,7.222,7.336,7.470,7.616,7.773,7.924,
     * 7.009,7.127,7.233,7.339,7.441,7.548,7.672,7.803,
     * 7.011,7.135,7.252,7.361,7.453,7.530,7.600,7.666,
     * 7.012,7.149,7.288,7.417,7.528,7.613,7.670,7.700,
     * 7.019,7.164,7.313,7.456,7.581,7.672,7.730,7.755,
     * 7.023,7.180,7.345,7.503,7.636,7.738,7.799,7.822,
     * 0.000,0.000,0.000,0.000,6.224,6.514,6.747,6.934,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,7,L,3),I=1,8),L=1,NZEN)/
     * 6.915,7.009,7.095,7.185,7.287,7.412,7.558,7.714,
     * 6.916,7.011,7.095,7.177,7.266,7.370,7.499,7.644,
     * 6.928,7.029,7.122,7.208,7.288,7.370,7.460,7.564,
     * 6.943,7.050,7.154,7.255,7.352,7.434,7.510,7.571,
     * 6.952,7.073,7.200,7.327,7.446,7.546,7.625,7.677,
     * 6.956,7.087,7.225,7.369,7.498,7.605,7.680,7.724,
     * 6.959,7.103,7.259,7.415,7.557,7.669,7.746,7.784,
     * 0.000,0.000,0.000,0.000,6.077,6.367,6.610,6.808,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,8,L,3),I=1,8),L=1,NZEN)/
     * 7.043,7.242,7.469,7.711,7.939,8.139,8.305,8.435,
     * 7.005,7.171,7.363,7.590,7.822,8.033,8.211,8.351,
     * 6.982,7.126,7.277,7.452,7.656,7.866,8.057,8.212,
     * 6.964,7.107,7.242,7.373,7.494,7.624,7.772,7.927,
     * 6.935,7.089,7.240,7.382,7.498,7.582,7.628,7.645,
     * 6.926,7.085,7.245,7.395,7.518,7.601,7.647,7.656,
     * 6.920,7.087,7.258,7.417,7.543,7.634,7.679,7.683,
     * 0.000,0.000,0.000,0.000,6.169,6.441,6.652,6.816,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,8,L,3),I=1,8),L=1,NZEN)/
     * 7.037,7.232,7.456,7.693,7.914,8.108,8.271,8.397,
     * 6.996,7.159,7.348,7.570,7.794,8.000,8.175,8.317,
     * 6.971,7.113,7.260,7.433,7.630,7.835,8.024,8.179,
     * 6.950,7.092,7.226,7.355,7.479,7.606,7.754,7.908,
     * 6.920,7.076,7.227,7.372,7.493,7.578,7.632,7.653,
     * 6.914,7.073,7.235,7.390,7.517,7.609,7.661,7.675,
     * 6.912,7.080,7.253,7.418,7.553,7.651,7.702,7.712,
     * 0.000,0.000,0.000,0.000,6.215,6.500,6.722,6.893,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,8,L,3),I=1,8),L=1,NZEN)/
     * 7.027,7.191,7.380,7.592,7.800,7.990,8.155,8.292,
     * 7.008,7.148,7.305,7.492,7.694,7.889,8.064,8.210,
     * 6.994,7.123,7.250,7.391,7.555,7.736,7.913,8.070,
     * 6.980,7.112,7.235,7.350,7.458,7.562,7.679,7.814,
     * 6.964,7.108,7.250,7.386,7.497,7.582,7.635,7.657,
     * 6.963,7.114,7.266,7.411,7.536,7.625,7.676,7.694,
     * 6.963,7.123,7.289,7.447,7.579,7.674,7.729,7.744,
     * 0.000,0.000,0.000,0.000,6.237,6.524,6.749,6.925,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,8,L,3),I=1,8),L=1,NZEN)/
     * 7.056,7.176,7.307,7.461,7.631,7.804,7.970,8.118,
     * 7.054,7.166,7.276,7.404,7.554,7.718,7.883,8.034,
     * 7.057,7.167,7.268,7.371,7.483,7.611,7.758,7.904,
     * 7.062,7.179,7.288,7.389,7.477,7.555,7.631,7.716,
     * 7.062,7.193,7.321,7.442,7.548,7.628,7.678,7.704,
     * 7.063,7.202,7.343,7.476,7.591,7.676,7.727,7.747,
     * 7.069,7.218,7.376,7.524,7.650,7.744,7.797,7.814,
     * 0.000,0.000,0.000,0.000,6.249,6.533,6.760,6.941,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,8,L,3),I=1,8),L=1,NZEN)/
     * 7.029,7.119,7.205,7.310,7.437,7.583,7.740,7.898,
     * 7.026,7.112,7.193,7.277,7.383,7.511,7.661,7.818,
     * 7.030,7.119,7.196,7.272,7.353,7.444,7.563,7.701,
     * 7.047,7.147,7.239,7.329,7.409,7.481,7.551,7.620,
     * 7.050,7.164,7.277,7.391,7.494,7.577,7.643,7.683,
     * 7.054,7.176,7.303,7.429,7.545,7.635,7.701,7.737,
     * 7.056,7.188,7.332,7.474,7.602,7.701,7.767,7.798,
     * 0.000,0.000,0.000,0.000,6.136,6.418,6.654,6.848,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,9,L,3),I=1,8),L=1,NZEN)/
     * 7.008,7.201,7.422,7.665,7.895,8.100,8.271,8.404,
     * 6.977,7.139,7.326,7.546,7.777,7.990,8.174,8.319,
     * 6.957,7.102,7.251,7.420,7.617,7.825,8.017,8.177,
     * 6.939,7.085,7.223,7.355,7.476,7.599,7.739,7.891,
     * 6.910,7.067,7.221,7.365,7.485,7.568,7.615,7.634,
     * 6.901,7.064,7.227,7.379,7.503,7.594,7.638,7.648,
     * 6.897,7.067,7.239,7.402,7.532,7.623,7.666,7.675,
     * 0.000,0.000,0.000,0.000,6.142,6.415,6.626,6.792,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,9,L,3),I=1,8),L=1,NZEN)/
     * 6.999,7.188,7.409,7.647,7.874,8.075,8.244,8.379,
     * 6.966,7.125,7.311,7.527,7.755,7.968,8.150,8.297,
     * 6.946,7.089,7.235,7.402,7.597,7.801,7.993,8.154,
     * 6.926,7.072,7.208,7.338,7.458,7.579,7.721,7.872,
     * 6.900,7.055,7.208,7.352,7.473,7.557,7.609,7.631,
     * 6.891,7.052,7.213,7.367,7.494,7.583,7.633,7.647,
     * 6.888,7.056,7.228,7.392,7.523,7.620,7.666,7.678,
     * 0.000,0.000,0.000,0.000,6.153,6.432,6.648,6.818,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,9,L,3),I=1,8),L=1,NZEN)/
     * 7.040,7.209,7.408,7.626,7.839,8.033,8.200,8.338,
     * 7.017,7.160,7.321,7.519,7.729,7.928,8.105,8.254,
     * 7.003,7.132,7.260,7.409,7.582,7.770,7.953,8.111,
     * 6.989,7.118,7.240,7.356,7.462,7.569,7.696,7.838,
     * 6.971,7.112,7.250,7.378,7.487,7.568,7.613,7.635,
     * 6.964,7.113,7.260,7.400,7.518,7.599,7.645,7.661,
     * 6.963,7.120,7.280,7.433,7.557,7.643,7.689,7.700,
     * 0.000,0.000,0.000,0.000,6.186,6.460,6.677,6.847,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,9,L,3),I=1,8),L=1,NZEN)/
     * 7.157,7.291,7.447,7.624,7.807,7.981,8.141,8.282,
     * 7.139,7.250,7.376,7.532,7.708,7.884,8.051,8.196,
     * 7.136,7.240,7.340,7.454,7.591,7.749,7.913,8.065,
     * 7.134,7.240,7.337,7.427,7.513,7.599,7.701,7.823,
     * 7.127,7.245,7.362,7.470,7.562,7.627,7.671,7.690,
     * 7.124,7.248,7.377,7.497,7.595,7.667,7.707,7.720,
     * 7.122,7.258,7.399,7.533,7.641,7.718,7.761,7.769,
     * 0.000,0.000,0.000,0.000,6.248,6.514,6.728,6.899,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,9,L,3),I=1,8),L=1,NZEN)/
     * 7.228,7.332,7.453,7.596,7.750,7.902,8.052,8.195,
     * 7.207,7.292,7.387,7.509,7.655,7.809,7.965,8.111,
     * 7.196,7.270,7.339,7.424,7.537,7.676,7.830,7.984,
     * 7.199,7.275,7.343,7.408,7.472,7.545,7.643,7.764,
     * 7.198,7.289,7.380,7.467,7.543,7.603,7.648,7.675,
     * 7.190,7.291,7.394,7.491,7.577,7.642,7.686,7.706,
     * 7.189,7.298,7.415,7.528,7.624,7.696,7.743,7.756,
     * 0.000,0.000,0.000,0.000,6.192,6.452,6.671,6.852,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,10,L,3),I=1,8),L=1,NZEN)/
     * 6.992,7.181,7.398,7.638,7.869,8.075,8.247,8.382,
     * 6.964,7.124,7.306,7.521,7.749,7.963,8.148,8.296,
     * 6.946,7.092,7.239,7.405,7.594,7.797,7.990,8.151,
     * 6.927,7.076,7.214,7.346,7.465,7.582,7.716,7.864,
     * 6.899,7.057,7.212,7.357,7.477,7.561,7.612,7.629,
     * 6.890,7.055,7.216,7.370,7.495,7.586,7.631,7.642,
     * 6.886,7.058,7.232,7.392,7.524,7.615,7.663,7.669,
     * 0.000,0.000,0.000,0.000,6.124,6.396,6.607,6.773,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,10,L,3),I=1,8),L=1,NZEN)/
     * 6.975,7.161,7.375,7.616,7.850,8.057,8.234,8.372,
     * 6.947,7.104,7.284,7.500,7.731,7.946,8.134,8.285,
     * 6.930,7.073,7.217,7.383,7.573,7.779,7.973,8.138,
     * 6.913,7.058,7.194,7.323,7.440,7.558,7.694,7.846,
     * 6.883,7.039,7.192,7.333,7.449,7.535,7.583,7.604,
     * 6.875,7.036,7.196,7.347,7.470,7.555,7.603,7.615,
     * 6.869,7.039,7.208,7.369,7.495,7.586,7.633,7.640,
     * 0.000,0.000,0.000,0.000,6.088,6.359,6.570,6.738,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,10,L,3),I=1,8),L=1,NZEN)/
     * 7.052,7.228,7.433,7.658,7.876,8.072,8.241,8.378,
     * 7.023,7.168,7.337,7.542,7.760,7.963,8.143,8.290,
     * 7.007,7.134,7.264,7.419,7.602,7.799,7.985,8.146,
     * 6.993,7.121,7.240,7.351,7.460,7.573,7.708,7.859,
     * 6.971,7.109,7.243,7.365,7.469,7.541,7.585,7.603,
     * 6.962,7.107,7.250,7.383,7.491,7.566,7.607,7.618,
     * 6.955,7.108,7.263,7.404,7.520,7.598,7.638,7.643,
     * 0.000,0.000,0.000,0.000,6.112,6.371,6.576,6.741,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,10,L,3),I=1,8),L=1,NZEN)/
     * 7.256,7.408,7.588,7.779,7.965,8.136,8.294,8.421,
     * 7.217,7.341,7.490,7.669,7.857,8.036,8.196,8.339,
     * 7.195,7.293,7.401,7.540,7.705,7.881,8.053,8.205,
     * 7.188,7.281,7.369,7.454,7.544,7.655,7.791,7.939,
     * 7.168,7.275,7.379,7.472,7.551,7.605,7.639,7.658,
     * 7.159,7.270,7.383,7.486,7.569,7.627,7.654,7.660,
     * 7.151,7.276,7.397,7.514,7.604,7.665,7.693,7.694,
     * 0.000,0.000,0.000,0.000,6.206,6.449,6.645,6.807,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,10,L,3),I=1,8),L=1,NZEN)/
     * 7.447,7.590,7.745,7.903,8.055,8.199,8.342,8.464,
     * 7.383,7.499,7.637,7.793,7.953,8.106,8.255,8.388,
     * 7.341,7.417,7.515,7.645,7.799,7.959,8.114,8.259,
     * 7.319,7.376,7.432,7.496,7.583,7.703,7.850,8.002,
     * 7.297,7.368,7.433,7.494,7.546,7.583,7.611,7.641,
     * 7.284,7.360,7.437,7.507,7.563,7.602,7.623,7.629,
     * 7.268,7.356,7.442,7.524,7.589,7.630,7.649,7.647,
     * 0.000,0.000,0.000,0.000,6.191,6.420,6.613,6.778,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,11,L,3),I=1,8),L=1,NZEN)/
     * 7.014,7.205,7.422,7.661,7.889,8.091,8.260,8.393,
     * 6.984,7.144,7.326,7.543,7.768,7.980,8.162,8.307,
     * 6.964,7.109,7.254,7.420,7.607,7.812,8.002,8.162,
     * 6.946,7.092,7.228,7.358,7.476,7.591,7.725,7.874,
     * 6.916,7.073,7.226,7.368,7.485,7.568,7.615,7.632,
     * 6.908,7.071,7.232,7.382,7.504,7.590,7.635,7.645,
     * 6.903,7.074,7.244,7.404,7.533,7.623,7.666,7.675,
     * 0.000,0.000,0.000,0.000,6.130,6.401,6.611,6.777,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,11,L,3),I=1,8),L=1,NZEN)/
     * 6.979,7.165,7.382,7.624,7.860,8.070,8.249,8.390,
     * 6.951,7.109,7.289,7.507,7.738,7.955,8.144,8.295,
     * 6.932,7.076,7.218,7.384,7.576,7.785,7.982,8.149,
     * 6.917,7.061,7.195,7.321,7.436,7.555,7.691,7.844,
     * 6.887,7.042,7.192,7.329,7.442,7.520,7.566,7.585,
     * 6.881,7.040,7.197,7.343,7.460,7.542,7.583,7.592,
     * 6.872,7.040,7.207,7.361,7.484,7.563,7.606,7.613,
     * 0.000,0.000,0.000,0.000,6.043,6.304,6.508,6.672,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,11,L,3),I=1,8),L=1,NZEN)/
     * 7.062,7.241,7.455,7.685,7.906,8.105,8.275,8.414,
     * 7.029,7.176,7.352,7.566,7.785,7.992,8.173,8.322,
     * 7.009,7.135,7.267,7.427,7.618,7.821,8.010,8.171,
     * 6.991,7.116,7.232,7.343,7.451,7.572,7.720,7.878,
     * 6.967,7.102,7.231,7.352,7.445,7.513,7.552,7.571,
     * 6.951,7.091,7.229,7.353,7.454,7.521,7.555,7.565,
     * 6.946,7.095,7.242,7.379,7.483,7.553,7.584,7.589,
     * 0.000,0.000,0.000,0.000,6.036,6.281,6.475,6.635,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,11,L,3),I=1,8),L=1,NZEN)/
     * 7.316,7.485,7.674,7.872,8.057,8.226,8.383,8.511,
     * 7.266,7.398,7.563,7.755,7.945,8.124,8.290,8.419,
     * 7.232,7.330,7.448,7.602,7.781,7.964,8.134,8.287,
     * 7.209,7.298,7.381,7.465,7.564,7.693,7.847,8.004,
     * 7.179,7.276,7.368,7.451,7.515,7.560,7.586,7.608,
     * 7.163,7.264,7.362,7.452,7.520,7.562,7.580,7.579,
     * 7.148,7.258,7.366,7.466,7.538,7.582,7.598,7.591,
     * 0.000,0.000,0.000,0.000,6.130,6.350,6.529,6.681,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,11,L,3),I=1,8),L=1,NZEN)/
     * 7.620,7.781,7.942,8.100,8.245,8.389,8.516,8.639,
     * 7.524,7.667,7.824,7.986,8.141,8.289,8.423,8.552,
     * 7.441,7.537,7.665,7.820,7.983,8.141,8.297,8.425,
     * 7.391,7.445,7.503,7.586,7.707,7.856,8.015,8.166,
     * 7.347,7.405,7.455,7.497,7.530,7.555,7.584,7.639,
     * 7.323,7.384,7.440,7.489,7.523,7.541,7.545,7.545,
     * 7.296,7.366,7.431,7.487,7.523,7.538,7.537,7.525,
     * 0.000,0.000,0.000,0.000,6.128,6.324,6.490,6.638,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,12,L,3),I=1,8),L=1,NZEN)/
     * 7.054,7.250,7.472,7.711,7.932,8.129,8.292,8.422,
     * 7.018,7.182,7.370,7.588,7.811,8.017,8.194,8.334,
     * 6.997,7.140,7.286,7.452,7.644,7.849,8.035,8.191,
     * 6.977,7.121,7.253,7.377,7.494,7.614,7.753,7.901,
     * 6.947,7.102,7.251,7.389,7.502,7.582,7.625,7.642,
     * 6.939,7.098,7.256,7.402,7.518,7.605,7.644,7.654,
     * 6.934,7.102,7.269,7.424,7.548,7.634,7.675,7.681,
     * 0.000,0.000,0.000,0.000,6.151,6.421,6.630,6.794,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,12,L,3),I=1,8),L=1,NZEN)/
     * 6.999,7.189,7.409,7.655,7.890,8.099,8.275,8.414,
     * 6.968,7.126,7.312,7.534,7.765,7.984,8.170,8.321,
     * 6.946,7.087,7.230,7.400,7.599,7.812,8.009,8.174,
     * 6.934,7.074,7.205,7.330,7.446,7.564,7.706,7.861,
     * 6.904,7.056,7.202,7.335,7.444,7.518,7.561,7.578,
     * 6.889,7.046,7.199,7.339,7.455,7.529,7.568,7.576,
     * 6.887,7.052,7.218,7.368,7.482,7.562,7.596,7.600,
     * 0.000,0.000,0.000,0.000,6.026,6.281,6.481,6.642,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,12,L,3),I=1,8),L=1,NZEN)/
     * 7.069,7.253,7.468,7.702,7.925,8.125,8.295,8.433,
     * 7.031,7.181,7.360,7.576,7.800,8.009,8.190,8.336,
     * 7.009,7.135,7.267,7.432,7.629,7.834,8.024,8.185,
     * 6.990,7.113,7.228,7.338,7.447,7.571,7.723,7.885,
     * 6.957,7.089,7.215,7.331,7.421,7.485,7.519,7.538,
     * 6.945,7.082,7.215,7.335,7.430,7.491,7.522,7.529,
     * 6.929,7.074,7.216,7.343,7.442,7.503,7.531,7.534,
     * 0.000,0.000,0.000,0.000,0.000,6.219,6.405,6.562,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,12,L,3),I=1,8),L=1,NZEN)/
     * 7.329,7.500,7.693,7.891,8.077,8.247,8.403,8.529,
     * 7.271,7.409,7.576,7.771,7.963,8.141,8.303,8.438,
     * 7.232,7.334,7.451,7.611,7.793,7.976,8.147,8.293,
     * 7.206,7.293,7.374,7.459,7.560,7.692,7.850,8.009,
     * 7.168,7.259,7.345,7.421,7.479,7.517,7.538,7.561,
     * 7.144,7.241,7.334,7.416,7.475,7.509,7.521,7.517,
     * 7.124,7.227,7.329,7.416,7.479,7.510,7.518,7.508,
     * 0.000,0.000,0.000,0.000,6.059,6.263,6.427,6.569,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,12,L,3),I=1,8),L=1,NZEN)/
     * 7.667,7.832,7.992,8.146,8.292,8.420,8.553,8.678,
     * 7.563,7.711,7.870,8.031,8.186,8.330,8.462,8.592,
     * 7.467,7.570,7.702,7.861,8.024,8.179,8.330,8.458,
     * 7.407,7.458,7.517,7.604,7.730,7.881,8.039,8.191,
     * 7.351,7.402,7.445,7.480,7.504,7.522,7.551,7.615,
     * 7.319,7.375,7.423,7.460,7.483,7.489,7.485,7.482,
     * 7.289,7.349,7.405,7.448,7.471,7.474,7.463,7.446,
     * 0.000,0.000,0.000,0.000,6.040,6.214,6.362,6.501,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,1,L,1),I=9,16),L=1,NZEN)/
     * 8.380,8.453,8.519,8.597,8.664,8.726,8.769,8.807,
     * 8.293,8.378,8.447,8.531,8.604,8.671,8.716,8.758,
     * 8.159,8.255,8.336,8.429,8.510,8.582,8.635,8.685,
     * 7.878,7.992,8.097,8.213,8.312,8.403,8.472,8.536,
     * 7.546,7.546,7.584,7.687,7.810,7.936,8.043,8.145,
     * 7.538,7.509,7.488,7.504,7.548,7.626,7.725,7.838,
     * 7.559,7.522,7.489,7.481,7.480,7.481,7.482,7.499,
     * 6.744,6.842,6.934,7.036,7.129,7.204,7.263,7.319,
     * 0.000,0.000,0.000,6.291,6.559,6.772,6.926,7.061,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,1,L,1),I=9,16),L=1,NZEN)/
     * 8.380,8.461,8.534,8.617,8.687,8.754,8.800,8.837,
     * 8.291,8.375,8.456,8.544,8.622,8.691,8.742,8.785,
     * 8.150,8.251,8.342,8.442,8.527,8.604,8.661,8.713,
     * 7.846,7.965,8.078,8.203,8.311,8.405,8.480,8.549,
     * 7.511,7.510,7.543,7.643,7.766,7.896,8.010,8.115,
     * 7.489,7.466,7.450,7.473,7.517,7.593,7.687,7.799,
     * 7.507,7.474,7.449,7.451,7.457,7.466,7.467,7.480,
     * 6.616,6.722,6.827,6.944,7.052,7.142,7.211,7.270,
     * 0.000,0.000,0.000,6.151,6.438,6.667,6.833,6.970,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,1,L,1),I=9,16),L=1,NZEN)/
     * 8.392,8.478,8.561,8.656,8.738,8.811,8.865,8.913,
     * 8.300,8.394,8.480,8.580,8.669,8.747,8.806,8.858,
     * 8.156,8.264,8.360,8.469,8.566,8.651,8.718,8.777,
     * 7.879,8.002,8.120,8.251,8.366,8.467,8.549,8.622,
     * 7.512,7.527,7.578,7.692,7.827,7.964,8.078,8.187,
     * 7.482,7.465,7.463,7.498,7.551,7.634,7.732,7.840,
     * 7.473,7.449,7.437,7.452,7.469,7.485,7.495,7.506,
     * 6.570,6.687,6.808,6.942,7.063,7.162,7.234,7.285,
     * 0.000,0.000,0.000,6.114,6.419,6.662,6.834,6.963,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,1,L,1),I=9,16),L=1,NZEN)/
     * 8.446,8.536,8.633,8.740,8.836,8.924,8.990,9.045,
     * 8.352,8.450,8.549,8.664,8.764,8.856,8.927,8.986,
     * 8.216,8.322,8.428,8.549,8.659,8.757,8.836,8.902,
     * 7.947,8.070,8.188,8.326,8.450,8.565,8.656,8.737,
     * 7.538,7.575,7.657,7.794,7.938,8.078,8.200,8.308,
     * 7.472,7.469,7.487,7.552,7.640,7.755,7.868,7.982,
     * 7.444,7.430,7.433,7.466,7.499,7.530,7.554,7.577,
     * 6.570,6.693,6.825,6.970,7.100,7.202,7.271,7.307,
     * 0.000,0.000,0.000,6.126,6.444,6.698,6.872,6.986,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,1,L,1),I=9,16),L=1,NZEN)/
     * 8.549,8.646,8.757,8.878,8.986,9.080,9.150,9.207,
     * 8.460,8.563,8.675,8.801,8.913,9.013,9.088,9.151,
     * 8.325,8.436,8.554,8.687,8.809,8.919,9.001,9.069,
     * 8.067,8.189,8.321,8.469,8.605,8.730,8.828,8.911,
     * 7.585,7.661,7.779,7.936,8.097,8.251,8.378,8.492,
     * 7.472,7.491,7.544,7.647,7.772,7.915,8.046,8.168,
     * 7.418,7.420,7.447,7.503,7.562,7.620,7.662,7.708,
     * 6.550,6.688,6.835,6.998,7.145,7.258,7.331,7.362,
     * 0.000,0.000,0.000,6.105,6.451,6.723,6.906,7.015,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,2,L,1),I=9,16),L=1,NZEN)/
     * 8.355,8.434,8.501,8.583,8.651,8.714,8.756,8.793,
     * 8.272,8.355,8.429,8.516,8.590,8.657,8.705,8.744,
     * 8.133,8.231,8.316,8.413,8.495,8.569,8.623,8.672,
     * 7.852,7.968,8.076,8.196,8.299,8.389,8.457,8.522,
     * 7.538,7.538,7.570,7.669,7.792,7.919,8.026,8.128,
     * 7.533,7.504,7.484,7.500,7.540,7.613,7.708,7.820,
     * 7.554,7.518,7.485,7.479,7.478,7.477,7.478,7.494,
     * 6.728,6.828,6.921,7.025,7.119,7.196,7.255,7.313,
     * 0.000,0.000,0.000,6.277,6.547,6.759,6.915,7.051,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,2,L,1),I=9,16),L=1,NZEN)/
     * 8.359,8.437,8.510,8.597,8.669,8.733,8.778,8.816,
     * 8.267,8.354,8.434,8.525,8.602,8.671,8.720,8.762,
     * 8.124,8.224,8.315,8.416,8.504,8.581,8.638,8.688,
     * 7.828,7.946,8.059,8.185,8.291,8.387,8.460,8.527,
     * 7.511,7.511,7.541,7.636,7.757,7.885,7.996,8.099,
     * 7.496,7.471,7.455,7.475,7.516,7.588,7.679,7.788,
     * 7.513,7.480,7.454,7.454,7.459,7.465,7.466,7.476,
     * 6.628,6.734,6.836,6.952,7.057,7.145,7.212,7.270,
     * 0.000,0.000,0.000,6.162,6.445,6.672,6.836,6.973,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,2,L,1),I=9,16),L=1,NZEN)/
     * 8.369,8.454,8.539,8.631,8.710,8.782,8.832,8.877,
     * 8.277,8.371,8.458,8.557,8.642,8.719,8.773,8.821,
     * 8.134,8.242,8.339,8.446,8.540,8.623,8.686,8.742,
     * 7.855,7.976,8.093,8.222,8.335,8.434,8.511,8.582,
     * 7.523,7.533,7.579,7.685,7.812,7.942,8.053,8.156,
     * 7.498,7.481,7.476,7.508,7.555,7.628,7.718,7.821,
     * 7.494,7.468,7.454,7.467,7.480,7.492,7.497,7.507,
     * 6.602,6.717,6.834,6.962,7.077,7.170,7.238,7.287,
     * 0.000,0.000,0.000,6.145,6.443,6.679,6.848,6.976,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,2,L,1),I=9,16),L=1,NZEN)/
     * 8.405,8.496,8.594,8.700,8.793,8.874,8.936,8.989,
     * 8.316,8.413,8.512,8.622,8.719,8.808,8.873,8.931,
     * 8.181,8.285,8.390,8.509,8.615,8.709,8.784,8.848,
     * 7.913,8.033,8.153,8.290,8.408,8.519,8.605,8.681,
     * 7.549,7.579,7.648,7.770,7.907,8.040,8.153,8.258,
     * 7.499,7.495,7.509,7.561,7.634,7.731,7.833,7.940,
     * 7.484,7.470,7.472,7.499,7.527,7.550,7.563,7.577,
     * 6.607,6.730,6.859,6.999,7.124,7.223,7.289,7.326,
     * 0.000,0.000,0.000,6.170,6.482,6.729,6.899,7.014,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,2,L,1),I=9,16),L=1,NZEN)/
     * 8.467,8.567,8.679,8.798,8.904,8.996,9.064,9.122,
     * 8.380,8.484,8.596,8.722,8.833,8.930,9.002,9.065,
     * 8.248,8.361,8.479,8.609,8.727,8.832,8.913,8.982,
     * 7.987,8.111,8.242,8.390,8.520,8.642,8.736,8.820,
     * 7.570,7.628,7.727,7.876,8.023,8.163,8.288,8.397,
     * 7.488,7.502,7.543,7.627,7.731,7.849,7.966,8.078,
     * 7.444,7.446,7.469,7.518,7.567,7.613,7.641,7.670,
     * 6.580,6.717,6.862,7.020,7.161,7.269,7.339,7.370,
     * 0.000,0.000,0.000,6.154,6.487,6.752,6.931,7.039,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,3,L,1),I=9,16),L=1,NZEN)/
     * 8.325,8.405,8.475,8.562,8.632,8.696,8.740,8.776,
     * 8.238,8.327,8.404,8.495,8.572,8.640,8.687,8.727,
     * 8.098,8.201,8.288,8.391,8.475,8.551,8.605,8.653,
     * 7.815,7.935,8.046,8.170,8.275,8.370,8.439,8.501,
     * 7.529,7.527,7.554,7.648,7.768,7.894,8.005,8.107,
     * 7.525,7.500,7.477,7.492,7.527,7.597,7.688,7.799,
     * 7.549,7.514,7.483,7.477,7.474,7.475,7.473,7.487,
     * 6.707,6.808,6.903,7.009,7.105,7.184,7.245,7.303,
     * 0.000,0.000,0.000,6.258,6.528,6.743,6.900,7.037,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,3,L,1),I=9,16),L=1,NZEN)/
     * 8.323,8.403,8.481,8.567,8.640,8.704,8.747,8.781,
     * 8.235,8.326,8.407,8.499,8.577,8.645,8.692,8.732,
     * 8.090,8.195,8.288,8.388,8.474,8.553,8.606,8.655,
     * 7.803,7.922,8.034,8.160,8.266,8.362,8.434,8.496,
     * 7.520,7.518,7.544,7.633,7.749,7.872,7.981,8.082,
     * 7.513,7.488,7.472,7.487,7.519,7.578,7.660,7.762,
     * 7.528,7.498,7.470,7.468,7.469,7.472,7.469,7.479,
     * 6.661,6.765,6.866,6.978,7.080,7.162,7.226,7.282,
     * 0.000,0.000,0.000,6.200,6.477,6.697,6.858,6.993,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,3,L,1),I=9,16),L=1,NZEN)/
     * 8.328,8.415,8.496,8.590,8.669,8.737,8.784,8.823,
     * 8.235,8.333,8.418,8.516,8.599,8.674,8.725,8.769,
     * 8.098,8.204,8.301,8.407,8.500,8.580,8.639,8.689,
     * 7.822,7.938,8.051,8.177,8.289,8.384,8.458,8.523,
     * 7.555,7.556,7.586,7.672,7.780,7.900,8.003,8.100,
     * 7.537,7.518,7.509,7.532,7.569,7.629,7.705,7.799,
     * 7.548,7.524,7.505,7.510,7.519,7.523,7.520,7.522,
     * 6.672,6.784,6.894,7.014,7.121,7.206,7.267,7.315,
     * 0.000,0.000,0.000,6.222,6.507,6.732,6.893,7.020,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,3,L,1),I=9,16),L=1,NZEN)/
     * 8.343,8.438,8.533,8.638,8.728,8.805,8.860,8.907,
     * 8.260,8.358,8.455,8.564,8.659,8.740,8.800,8.852,
     * 8.123,8.234,8.338,8.453,8.556,8.644,8.710,8.769,
     * 7.870,7.987,8.107,8.238,8.355,8.458,8.536,8.608,
     * 7.595,7.612,7.658,7.755,7.869,7.988,8.092,8.189,
     * 7.571,7.566,7.573,7.609,7.657,7.721,7.795,7.884,
     * 7.568,7.554,7.551,7.570,7.587,7.599,7.600,7.603,
     * 6.707,6.830,6.955,7.087,7.201,7.291,7.350,7.387,
     * 0.000,0.000,0.000,6.278,6.573,6.807,6.970,7.087,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,3,L,1),I=9,16),L=1,NZEN)/
     * 8.362,8.466,8.578,8.696,8.799,8.888,8.949,9.003,
     * 8.276,8.387,8.500,8.623,8.729,8.822,8.889,8.947,
     * 8.148,8.265,8.383,8.513,8.626,8.725,8.798,8.864,
     * 7.903,8.024,8.156,8.300,8.427,8.538,8.623,8.701,
     * 7.609,7.647,7.716,7.831,7.958,8.080,8.185,8.286,
     * 7.569,7.580,7.608,7.665,7.733,7.814,7.896,7.986,
     * 7.556,7.558,7.575,7.612,7.647,7.670,7.680,7.684,
     * 6.697,6.840,6.985,7.130,7.260,7.360,7.420,7.451,
     * 0.000,0.000,0.000,6.294,6.610,6.860,7.030,7.142,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,4,L,1),I=9,16),L=1,NZEN)/
     * 8.305,8.389,8.463,8.550,8.622,8.688,8.729,8.764,
     * 8.222,8.313,8.393,8.483,8.561,8.630,8.676,8.716,
     * 8.080,8.186,8.277,8.380,8.466,8.541,8.595,8.641,
     * 7.799,7.919,8.033,8.158,8.264,8.359,8.429,8.491,
     * 7.526,7.520,7.546,7.638,7.757,7.885,7.995,8.098,
     * 7.520,7.495,7.475,7.490,7.524,7.590,7.681,7.790,
     * 7.544,7.511,7.480,7.475,7.472,7.473,7.471,7.482,
     * 6.696,6.799,6.894,7.002,7.099,7.178,7.240,7.299,
     * 0.000,0.000,0.000,6.251,6.521,6.736,6.893,7.030,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,4,L,1),I=9,16),L=1,NZEN)/
     * 8.296,8.376,8.455,8.540,8.611,8.677,8.719,8.751,
     * 8.207,8.300,8.381,8.472,8.549,8.618,8.664,8.701,
     * 8.067,8.172,8.264,8.367,8.451,8.527,8.580,8.623,
     * 7.793,7.910,8.021,8.146,8.252,8.343,8.411,8.472,
     * 7.538,7.536,7.557,7.641,7.752,7.872,7.978,8.075,
     * 7.536,7.512,7.492,7.506,7.537,7.598,7.679,7.780,
     * 7.558,7.527,7.497,7.493,7.490,7.488,7.484,7.490,
     * 6.719,6.822,6.919,7.025,7.121,7.198,7.257,7.312,
     * 0.000,0.000,0.000,6.275,6.544,6.757,6.912,7.045,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,4,L,1),I=9,16),L=1,NZEN)/
     * 8.275,8.365,8.447,8.540,8.617,8.683,8.728,8.761,
     * 8.187,8.282,8.370,8.469,8.550,8.622,8.668,8.708,
     * 8.050,8.157,8.254,8.360,8.448,8.526,8.581,8.627,
     * 7.797,7.905,8.014,8.140,8.245,8.339,8.406,8.467,
     * 7.591,7.590,7.608,7.677,7.772,7.878,7.974,8.066,
     * 7.589,7.571,7.558,7.573,7.596,7.638,7.697,7.779,
     * 7.611,7.586,7.564,7.562,7.560,7.557,7.546,7.544,
     * 6.772,6.880,6.982,7.091,7.188,7.264,7.318,7.366,
     * 0.000,0.000,6.016,6.334,6.603,6.815,6.966,7.094,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,4,L,1),I=9,16),L=1,NZEN)/
     * 8.246,8.345,8.444,8.550,8.640,8.716,8.765,8.804,
     * 8.161,8.270,8.370,8.481,8.573,8.654,8.707,8.752,
     * 8.032,8.144,8.254,8.372,8.474,8.559,8.619,8.670,
     * 7.814,7.916,8.029,8.159,8.272,8.370,8.440,8.501,
     * 7.666,7.671,7.694,7.755,7.833,7.924,8.010,8.093,
     * 7.668,7.662,7.661,7.683,7.706,7.734,7.768,7.826,
     * 7.689,7.674,7.665,7.673,7.678,7.676,7.663,7.650,
     * 6.829,6.952,7.069,7.189,7.292,7.371,7.423,7.463,
     * 0.000,0.000,6.127,6.433,6.698,6.907,7.056,7.178,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,4,L,1),I=9,16),L=1,NZEN)/
     * 8.191,8.310,8.428,8.552,8.658,8.745,8.802,8.847,
     * 8.110,8.233,8.354,8.481,8.592,8.682,8.743,8.792,
     * 7.989,8.113,8.241,8.375,8.490,8.587,8.655,8.711,
     * 7.795,7.905,8.029,8.170,8.297,8.404,8.483,8.549,
     * 7.713,7.737,7.774,7.840,7.913,7.992,8.067,8.141,
     * 7.717,7.731,7.751,7.788,7.819,7.841,7.862,7.899,
     * 7.736,7.741,7.752,7.776,7.793,7.794,7.779,7.759,
     * 6.839,6.988,7.129,7.270,7.389,7.476,7.530,7.566,
     * 0.000,0.000,6.079,6.450,6.751,6.980,7.139,7.262,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,5,L,1),I=9,16),L=1,NZEN)/
     * 8.325,8.405,8.478,8.562,8.632,8.695,8.736,8.770,
     * 8.241,8.327,8.407,8.497,8.570,8.639,8.684,8.722,
     * 8.101,8.204,8.291,8.391,8.477,8.551,8.603,8.650,
     * 7.822,7.941,8.052,8.175,8.282,8.373,8.441,8.501,
     * 7.529,7.527,7.557,7.654,7.775,7.903,8.012,8.114,
     * 7.525,7.497,7.477,7.492,7.531,7.603,7.697,7.808,
     * 7.547,7.514,7.480,7.475,7.472,7.473,7.471,7.485,
     * 6.712,6.813,6.908,7.014,7.109,7.187,7.247,7.304,
     * 0.000,0.000,0.000,6.267,6.536,6.750,6.906,7.041,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,5,L,1),I=9,16),L=1,NZEN)/
     * 8.300,8.379,8.452,8.536,8.606,8.666,8.708,8.738,
     * 8.217,8.306,8.384,8.472,8.547,8.613,8.657,8.691,
     * 8.084,8.186,8.273,8.372,8.455,8.527,8.577,8.619,
     * 7.821,7.936,8.043,8.164,8.267,8.353,8.417,8.474,
     * 7.564,7.560,7.587,7.677,7.789,7.908,8.010,8.106,
     * 7.566,7.539,7.518,7.531,7.565,7.629,7.716,7.821,
     * 7.595,7.561,7.527,7.517,7.512,7.507,7.501,7.511,
     * 6.797,6.897,6.989,7.089,7.176,7.245,7.297,7.349,
     * 0.000,0.000,6.058,6.368,6.627,6.830,6.978,7.109,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,5,L,1),I=9,16),L=1,NZEN)/
     * 8.236,8.325,8.409,8.500,8.578,8.642,8.683,8.713,
     * 8.154,8.247,8.337,8.432,8.512,8.581,8.627,8.662,
     * 8.019,8.125,8.221,8.326,8.416,8.491,8.542,8.581,
     * 7.782,7.886,7.993,8.115,8.221,8.310,8.375,8.431,
     * 7.624,7.621,7.633,7.692,7.775,7.875,7.966,8.055,
     * 7.637,7.618,7.602,7.610,7.626,7.659,7.710,7.787,
     * 7.672,7.645,7.617,7.609,7.603,7.593,7.576,7.572,
     * 6.873,6.978,7.073,7.173,7.260,7.326,7.374,7.421,
     * 0.000,0.000,6.148,6.453,6.708,6.906,7.050,7.177,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,5,L,1),I=9,16),L=1,NZEN)/
     * 8.133,8.240,8.348,8.461,8.556,8.634,8.684,8.719,
     * 8.053,8.167,8.276,8.396,8.492,8.575,8.627,8.668,
     * 7.933,8.048,8.160,8.286,8.388,8.474,8.532,8.579,
     * 7.767,7.849,7.949,8.073,8.184,8.282,8.349,8.405,
     * 7.718,7.721,7.731,7.768,7.815,7.878,7.942,8.017,
     * 7.734,7.727,7.723,7.740,7.751,7.759,7.767,7.801,
     * 7.776,7.762,7.749,7.751,7.747,7.737,7.716,7.701,
     * 6.921,7.044,7.156,7.267,7.362,7.434,7.485,7.533,
     * 0.000,0.000,6.255,6.546,6.795,6.991,7.133,7.262,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,5,L,1),I=9,16),L=1,NZEN)/
     * 7.996,8.128,8.263,8.403,8.523,8.615,8.679,8.719,
     * 7.923,8.057,8.192,8.336,8.457,8.554,8.622,8.667,
     * 7.820,7.950,8.084,8.232,8.360,8.464,8.534,8.587,
     * 7.749,7.825,7.922,8.047,8.164,8.266,8.343,8.403,
     * 7.791,7.819,7.848,7.891,7.926,7.954,7.983,8.025,
     * 7.815,7.835,7.854,7.885,7.902,7.899,7.887,7.882,
     * 7.851,7.859,7.866,7.884,7.890,7.880,7.858,7.836,
     * 6.922,7.074,7.211,7.343,7.454,7.537,7.594,7.645,
     * 0.000,0.000,6.240,6.574,6.848,7.058,7.211,7.349,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,6,L,1),I=9,16),L=1,NZEN)/
     * 8.363,8.437,8.506,8.585,8.653,8.713,8.753,8.786,
     * 8.281,8.365,8.437,8.523,8.594,8.657,8.702,8.740,
     * 8.146,8.243,8.326,8.421,8.501,8.573,8.624,8.669,
     * 7.869,7.986,8.092,8.209,8.309,8.397,8.463,8.524,
     * 7.541,7.544,7.583,7.688,7.813,7.938,8.044,8.141,
     * 7.533,7.504,7.484,7.502,7.548,7.629,7.728,7.841,
     * 7.554,7.518,7.485,7.477,7.476,7.475,7.477,7.494,
     * 6.742,6.841,6.934,7.036,7.128,7.203,7.260,7.313,
     * 0.000,0.000,0.000,6.297,6.565,6.775,6.928,7.060,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,6,L,1),I=9,16),L=1,NZEN)/
     * 8.330,8.406,8.472,8.552,8.618,8.678,8.716,8.747,
     * 8.249,8.333,8.405,8.489,8.562,8.624,8.665,8.698,
     * 8.122,8.217,8.299,8.395,8.472,8.541,8.590,8.630,
     * 7.866,7.979,8.083,8.197,8.294,8.378,8.439,8.493,
     * 7.586,7.588,7.623,7.720,7.837,7.954,8.053,8.145,
     * 7.591,7.563,7.540,7.555,7.595,7.670,7.761,7.869,
     * 7.625,7.587,7.549,7.538,7.529,7.522,7.516,7.533,
     * 6.869,6.965,7.051,7.145,7.225,7.287,7.333,7.382,
     * 0.000,0.000,6.148,6.450,6.701,6.897,7.037,7.164,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,6,L,1),I=9,16),L=1,NZEN)/
     * 8.223,8.312,8.394,8.485,8.563,8.626,8.667,8.694,
     * 8.144,8.238,8.324,8.422,8.499,8.568,8.611,8.644,
     * 8.014,8.119,8.217,8.320,8.405,8.480,8.531,8.568,
     * 7.788,7.890,7.997,8.116,8.219,8.310,8.372,8.425,
     * 7.651,7.647,7.656,7.713,7.796,7.892,7.982,8.068,
     * 7.672,7.652,7.633,7.639,7.654,7.685,7.739,7.820,
     * 7.717,7.689,7.659,7.647,7.635,7.620,7.602,7.596,
     * 6.956,7.059,7.149,7.242,7.321,7.380,7.422,7.469,
     * 0.000,0.000,6.258,6.553,6.797,6.986,7.122,7.250,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,6,L,1),I=9,16),L=1,NZEN)/
     * 8.054,8.170,8.284,8.406,8.507,8.590,8.639,8.673,
     * 7.976,8.096,8.211,8.336,8.439,8.525,8.581,8.619,
     * 7.867,7.980,8.094,8.223,8.330,8.420,8.480,8.526,
     * 7.734,7.804,7.895,8.018,8.130,8.228,8.300,8.359,
     * 7.733,7.738,7.745,7.779,7.816,7.864,7.921,7.991,
     * 7.768,7.763,7.759,7.775,7.783,7.785,7.786,7.806,
     * 7.817,7.806,7.792,7.795,7.789,7.777,7.754,7.739,
     * 6.987,7.109,7.216,7.323,7.413,7.482,7.532,7.587,
     * 0.000,6.074,6.344,6.633,6.876,7.064,7.203,7.338,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,6,L,1),I=9,16),L=1,NZEN)/
     * 7.851,7.998,8.144,8.303,8.433,8.538,8.606,8.649,
     * 7.790,7.933,8.079,8.237,8.369,8.479,8.551,8.601,
     * 7.722,7.839,7.970,8.121,8.255,8.364,8.443,8.500,
     * 7.730,7.790,7.865,7.970,8.076,8.171,8.248,8.312,
     * 7.807,7.837,7.864,7.906,7.934,7.950,7.963,7.989,
     * 7.832,7.852,7.870,7.905,7.924,7.927,7.917,7.911,
     * 7.871,7.883,7.892,7.916,7.928,7.925,7.907,7.895,
     * 6.961,7.111,7.243,7.372,7.481,7.567,7.632,7.700,
     * 0.000,6.127,6.398,6.693,6.944,7.140,7.288,7.435,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,7,L,1),I=9,16),L=1,NZEN)/
     * 8.390,8.458,8.526,8.602,8.667,8.726,8.768,8.800,
     * 8.310,8.388,8.457,8.539,8.608,8.672,8.716,8.755,
     * 8.177,8.268,8.349,8.439,8.516,8.587,8.638,8.684,
     * 7.902,8.015,8.117,8.229,8.328,8.415,8.481,8.540,
     * 7.548,7.555,7.602,7.711,7.835,7.960,8.064,8.161,
     * 7.538,7.509,7.488,7.510,7.560,7.646,7.748,7.862,
     * 7.559,7.520,7.487,7.479,7.478,7.477,7.480,7.501,
     * 6.763,6.860,6.951,7.051,7.141,7.214,7.268,7.320,
     * 0.000,0.000,6.002,6.317,6.583,6.792,6.943,7.073,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,7,L,1),I=9,16),L=1,NZEN)/
     * 8.356,8.426,8.493,8.569,8.635,8.693,8.730,8.762,
     * 8.272,8.354,8.425,8.508,8.577,8.639,8.680,8.715,
     * 8.149,8.240,8.324,8.412,8.491,8.558,8.606,8.647,
     * 7.897,8.009,8.109,8.221,8.316,8.399,8.459,8.512,
     * 7.597,7.602,7.644,7.747,7.864,7.980,8.078,8.171,
     * 7.601,7.571,7.550,7.565,7.612,7.693,7.788,7.897,
     * 7.635,7.597,7.558,7.546,7.536,7.528,7.524,7.548,
     * 6.899,6.994,7.078,7.168,7.246,7.305,7.349,7.396,
     * 0.000,0.000,6.183,6.482,6.731,6.923,7.060,7.186,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,7,L,1),I=9,16),L=1,NZEN)/
     * 8.240,8.325,8.407,8.498,8.574,8.639,8.679,8.708,
     * 8.160,8.255,8.341,8.436,8.515,8.583,8.625,8.659,
     * 8.033,8.137,8.231,8.338,8.423,8.497,8.547,8.587,
     * 7.807,7.911,8.018,8.138,8.239,8.330,8.392,8.445,
     * 7.662,7.657,7.669,7.731,7.817,7.918,8.008,8.096,
     * 7.684,7.664,7.644,7.650,7.664,7.702,7.760,7.847,
     * 7.734,7.703,7.671,7.659,7.644,7.629,7.610,7.608,
     * 6.992,7.094,7.181,7.271,7.346,7.402,7.442,7.487,
     * 0.000,6.012,6.304,6.595,6.835,7.020,7.153,7.279,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,7,L,1),I=9,16),L=1,NZEN)/
     * 8.058,8.178,8.290,8.414,8.516,8.597,8.649,8.683,
     * 7.979,8.101,8.220,8.346,8.448,8.535,8.591,8.629,
     * 7.867,7.983,8.100,8.230,8.339,8.431,8.492,8.539,
     * 7.733,7.806,7.900,8.026,8.142,8.245,8.315,8.376,
     * 7.737,7.743,7.752,7.785,7.825,7.877,7.938,8.012,
     * 7.776,7.773,7.769,7.784,7.793,7.794,7.797,7.822,
     * 7.832,7.821,7.808,7.810,7.802,7.789,7.765,7.750,
     * 7.025,7.147,7.251,7.355,7.442,7.508,7.556,7.611,
     * 0.000,6.101,6.384,6.676,6.916,7.102,7.238,7.374,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,7,L,1),I=9,16),L=1,NZEN)/
     * 7.841,7.992,8.141,8.300,8.433,8.542,8.612,8.656,
     * 7.782,7.929,8.077,8.238,8.374,8.483,8.557,8.608,
     * 7.710,7.830,7.965,8.121,8.259,8.371,8.451,8.508,
     * 7.713,7.773,7.850,7.961,8.071,8.173,8.253,8.321,
     * 7.800,7.829,7.856,7.899,7.931,7.952,7.969,8.001,
     * 7.831,7.851,7.870,7.906,7.928,7.934,7.926,7.924,
     * 7.879,7.894,7.904,7.930,7.942,7.939,7.922,7.910,
     * 7.003,7.151,7.282,7.408,7.516,7.600,7.663,7.732,
     * 0.000,6.165,6.446,6.742,6.991,7.184,7.330,7.477,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.052,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,8,L,1),I=9,16),L=1,NZEN)/
     * 8.384,8.458,8.523,8.601,8.665,8.725,8.767,8.800,
     * 8.302,8.384,8.455,8.537,8.606,8.671,8.714,8.753,
     * 8.171,8.264,8.346,8.436,8.514,8.585,8.637,8.682,
     * 7.894,8.010,8.112,8.229,8.325,8.410,8.476,8.536,
     * 7.546,7.552,7.597,7.706,7.830,7.955,8.059,8.157,
     * 7.535,7.507,7.488,7.508,7.558,7.641,7.742,7.856,
     * 7.559,7.520,7.487,7.479,7.478,7.479,7.480,7.501,
     * 6.758,6.856,6.948,7.049,7.139,7.211,7.267,7.320,
     * 0.000,0.000,0.000,6.312,6.579,6.788,6.940,7.070,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,8,L,1),I=9,16),L=1,NZEN)/
     * 8.356,8.429,8.497,8.577,8.643,8.704,8.744,8.776,
     * 8.276,8.359,8.432,8.516,8.586,8.650,8.692,8.730,
     * 8.148,8.243,8.326,8.419,8.496,8.568,8.616,8.659,
     * 7.890,8.005,8.108,8.222,8.317,8.402,8.464,8.522,
     * 7.590,7.595,7.634,7.737,7.856,7.974,8.075,8.171,
     * 7.592,7.562,7.541,7.560,7.604,7.683,7.779,7.889,
     * 7.623,7.585,7.550,7.538,7.529,7.524,7.521,7.541,
     * 6.873,6.970,7.056,7.149,7.229,7.291,7.338,7.386,
     * 0.000,0.000,6.149,6.452,6.704,6.900,7.041,7.168,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,8,L,1),I=9,16),L=1,NZEN)/
     * 8.271,8.359,8.442,8.532,8.608,8.675,8.716,8.748,
     * 8.193,8.287,8.372,8.468,8.549,8.617,8.663,8.699,
     * 8.065,8.170,8.266,8.368,8.456,8.530,8.582,8.625,
     * 7.830,7.938,8.045,8.167,8.269,8.360,8.426,8.483,
     * 7.653,7.650,7.668,7.738,7.832,7.938,8.033,8.124,
     * 7.671,7.651,7.631,7.638,7.660,7.702,7.766,7.856,
     * 7.714,7.684,7.653,7.642,7.630,7.618,7.603,7.604,
     * 6.968,7.069,7.159,7.251,7.329,7.387,7.429,7.475,
     * 0.000,0.000,6.268,6.564,6.809,6.998,7.134,7.259,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,8,L,1),I=9,16),L=1,NZEN)/
     * 8.145,8.257,8.365,8.481,8.577,8.657,8.707,8.743,
     * 8.067,8.183,8.296,8.415,8.513,8.597,8.651,8.694,
     * 7.947,8.067,8.182,8.308,8.415,8.504,8.565,8.613,
     * 7.773,7.862,7.968,8.098,8.211,8.315,8.388,8.448,
     * 7.734,7.738,7.747,7.789,7.843,7.913,7.988,8.071,
     * 7.762,7.756,7.750,7.765,7.774,7.785,7.802,7.849,
     * 7.818,7.802,7.786,7.785,7.779,7.766,7.745,7.733,
     * 7.028,7.146,7.248,7.352,7.438,7.503,7.548,7.597,
     * 0.000,6.125,6.385,6.668,6.906,7.092,7.229,7.360,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,8,L,1),I=9,16),L=1,NZEN)/
     * 7.979,8.121,8.260,8.408,8.531,8.630,8.695,8.738,
     * 7.912,8.055,8.198,8.347,8.471,8.575,8.642,8.690,
     * 7.814,7.949,8.089,8.243,8.372,8.480,8.555,8.611,
     * 7.748,7.823,7.920,8.049,8.170,8.278,8.357,8.424,
     * 7.794,7.817,7.841,7.885,7.923,7.962,8.004,8.062,
     * 7.834,7.848,7.862,7.892,7.909,7.913,7.909,7.917,
     * 7.884,7.889,7.892,7.909,7.915,7.909,7.890,7.875,
     * 7.035,7.177,7.303,7.426,7.529,7.607,7.663,7.720,
     * 6.031,6.195,6.460,6.752,7.000,7.193,7.337,7.474,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.036,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,9,L,1),I=9,16),L=1,NZEN)/
     * 8.359,8.434,8.503,8.585,8.653,8.714,8.756,8.790,
     * 8.275,8.358,8.434,8.521,8.592,8.659,8.703,8.742,
     * 8.139,8.237,8.323,8.418,8.499,8.571,8.624,8.670,
     * 7.860,7.978,8.085,8.205,8.306,8.395,8.461,8.522,
     * 7.538,7.540,7.577,7.681,7.805,7.931,8.038,8.137,
     * 7.530,7.504,7.484,7.502,7.545,7.623,7.721,7.834,
     * 7.554,7.518,7.485,7.479,7.478,7.477,7.478,7.494,
     * 6.737,6.837,6.930,7.033,7.126,7.201,7.259,7.314,
     * 0.000,0.000,0.000,6.291,6.559,6.770,6.924,7.057,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,9,L,1),I=9,16),L=1,NZEN)/
     * 8.341,8.421,8.493,8.575,8.648,8.710,8.752,8.787,
     * 8.260,8.347,8.425,8.514,8.588,8.654,8.702,8.741,
     * 8.127,8.227,8.315,8.413,8.495,8.569,8.620,8.666,
     * 7.857,7.974,8.082,8.203,8.304,8.394,8.459,8.521,
     * 7.566,7.568,7.602,7.699,7.817,7.940,8.044,8.144,
     * 7.565,7.540,7.519,7.535,7.576,7.649,7.741,7.852,
     * 7.593,7.558,7.526,7.517,7.514,7.511,7.508,7.522,
     * 6.802,6.902,6.994,7.094,7.182,7.251,7.304,7.356,
     * 0.000,0.000,6.057,6.369,6.630,6.835,6.983,7.113,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,9,L,1),I=9,16),L=1,NZEN)/
     * 8.311,8.399,8.481,8.574,8.654,8.721,8.766,8.804,
     * 8.228,8.323,8.412,8.509,8.591,8.663,8.712,8.754,
     * 8.098,8.203,8.300,8.406,8.496,8.573,8.629,8.679,
     * 7.847,7.959,8.069,8.196,8.300,8.393,8.462,8.525,
     * 7.628,7.627,7.651,7.731,7.834,7.947,8.047,8.143,
     * 7.635,7.614,7.598,7.612,7.639,7.692,7.765,7.859,
     * 7.666,7.636,7.610,7.603,7.599,7.593,7.582,7.586,
     * 6.886,6.991,7.087,7.186,7.272,7.338,7.386,7.433,
     * 0.000,0.000,6.158,6.465,6.722,6.922,7.066,7.192,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,9,L,1),I=9,16),L=1,NZEN)/
     * 8.271,8.371,8.471,8.579,8.671,8.750,8.800,8.842,
     * 8.195,8.303,8.403,8.517,8.611,8.693,8.749,8.794,
     * 8.072,8.185,8.295,8.414,8.514,8.601,8.663,8.717,
     * 7.854,7.962,8.077,8.208,8.323,8.419,8.495,8.560,
     * 7.711,7.716,7.738,7.803,7.887,7.985,8.075,8.165,
     * 7.724,7.714,7.708,7.726,7.747,7.780,7.826,7.900,
     * 7.760,7.739,7.723,7.726,7.724,7.717,7.702,7.695,
     * 6.977,7.093,7.199,7.305,7.395,7.461,7.506,7.547,
     * 0.000,6.008,6.290,6.590,6.845,7.041,7.182,7.304,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,9,L,1),I=9,16),L=1,NZEN)/
     * 8.206,8.324,8.446,8.575,8.685,8.774,8.837,8.885,
     * 8.135,8.258,8.380,8.513,8.624,8.721,8.785,8.838,
     * 8.021,8.151,8.277,8.416,8.533,8.635,8.707,8.766,
     * 7.837,7.952,8.078,8.223,8.349,8.459,8.540,8.611,
     * 7.769,7.790,7.823,7.888,7.965,8.051,8.134,8.217,
     * 7.785,7.792,7.803,7.832,7.859,7.885,7.916,7.972,
     * 7.825,7.822,7.823,7.837,7.844,7.840,7.823,7.812,
     * 7.018,7.158,7.283,7.406,7.507,7.582,7.627,7.666,
     * 0.000,6.144,6.406,6.702,6.959,7.157,7.298,7.419,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,10,L,1),I=9,16),L=1,NZEN)/
     * 8.340,8.422,8.492,8.575,8.645,8.709,8.751,8.785,
     * 8.258,8.345,8.421,8.510,8.583,8.651,8.697,8.737,
     * 8.118,8.220,8.306,8.405,8.488,8.562,8.616,8.662,
     * 7.838,7.956,8.066,8.188,8.292,8.384,8.452,8.514,
     * 7.534,7.533,7.567,7.665,7.787,7.914,8.023,8.124,
     * 7.528,7.502,7.482,7.498,7.536,7.610,7.705,7.817,
     * 7.552,7.516,7.485,7.477,7.476,7.477,7.477,7.491,
     * 6.721,6.821,6.916,7.021,7.116,7.193,7.252,7.309,
     * 0.000,0.000,0.000,6.274,6.543,6.756,6.911,7.047,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,10,L,1),I=9,16),L=1,NZEN)/
     * 8.338,8.421,8.495,8.582,8.656,8.722,8.766,8.804,
     * 8.252,8.342,8.422,8.516,8.595,8.663,8.712,8.754,
     * 8.111,8.215,8.306,8.408,8.495,8.574,8.629,8.679,
     * 7.831,7.950,8.063,8.186,8.297,8.390,8.461,8.526,
     * 7.542,7.541,7.572,7.666,7.786,7.912,8.021,8.122,
     * 7.536,7.512,7.495,7.512,7.546,7.614,7.702,7.811,
     * 7.559,7.525,7.496,7.494,7.492,7.494,7.492,7.504,
     * 6.721,6.824,6.922,7.029,7.126,7.204,7.264,7.320,
     * 0.000,0.000,0.000,6.272,6.543,6.758,6.914,7.048,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,10,L,1),I=9,16),L=1,NZEN)/
     * 8.347,8.437,8.521,8.615,8.695,8.766,8.817,8.859,
     * 8.263,8.358,8.446,8.545,8.631,8.706,8.761,8.808,
     * 8.126,8.233,8.331,8.439,8.531,8.613,8.675,8.729,
     * 7.859,7.977,8.091,8.219,8.330,8.429,8.504,8.572,
     * 7.594,7.596,7.629,7.719,7.834,7.954,8.061,8.161,
     * 7.588,7.568,7.557,7.577,7.614,7.675,7.754,7.853,
     * 7.607,7.578,7.557,7.557,7.562,7.563,7.560,7.565,
     * 6.778,6.886,6.989,7.100,7.198,7.275,7.331,7.380,
     * 0.000,0.000,6.019,6.341,6.615,6.828,6.981,7.109,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,10,L,1),I=9,16),L=1,NZEN)/
     * 8.384,8.475,8.570,8.674,8.765,8.845,8.903,8.954,
     * 8.305,8.403,8.500,8.607,8.702,8.787,8.849,8.903,
     * 8.177,8.287,8.388,8.503,8.607,8.697,8.766,8.828,
     * 7.937,8.053,8.170,8.300,8.415,8.517,8.599,8.674,
     * 7.669,7.686,7.734,7.831,7.946,8.064,8.165,8.265,
     * 7.654,7.643,7.646,7.678,7.726,7.795,7.875,7.971,
     * 7.675,7.655,7.644,7.655,7.666,7.670,7.666,7.670,
     * 6.874,6.992,7.106,7.223,7.324,7.400,7.450,7.487,
     * 0.000,0.000,6.176,6.482,6.751,6.960,7.106,7.222,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,10,L,1),I=9,16),L=1,NZEN)/
     * 8.419,8.519,8.631,8.748,8.852,8.944,9.011,9.069,
     * 8.346,8.451,8.561,8.684,8.791,8.886,8.956,9.020,
     * 8.231,8.342,8.458,8.585,8.698,8.799,8.877,8.947,
     * 8.001,8.125,8.250,8.389,8.514,8.627,8.716,8.800,
     * 7.707,7.751,7.826,7.947,8.073,8.199,8.306,8.410,
     * 7.687,7.693,7.716,7.770,7.840,7.929,8.015,8.118,
     * 7.693,7.688,7.695,7.722,7.744,7.761,7.767,7.782,
     * 6.928,7.067,7.197,7.328,7.436,7.516,7.563,7.592,
     * 0.000,0.000,6.212,6.554,6.845,7.070,7.223,7.334,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,11,L,1),I=9,16),L=1,NZEN)/
     * 8.352,8.428,8.499,8.579,8.648,8.711,8.755,8.792,
     * 8.266,8.352,8.426,8.514,8.589,8.656,8.701,8.742,
     * 8.126,8.227,8.313,8.410,8.493,8.567,8.621,8.669,
     * 7.845,7.963,8.072,8.192,8.295,8.387,8.457,8.520,
     * 7.536,7.536,7.569,7.668,7.790,7.917,8.026,8.128,
     * 7.530,7.504,7.484,7.498,7.538,7.612,7.708,7.820,
     * 7.554,7.518,7.485,7.479,7.478,7.479,7.478,7.494,
     * 6.724,6.824,6.918,7.022,7.116,7.193,7.254,7.311,
     * 0.000,0.000,0.000,6.274,6.544,6.758,6.913,7.049,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,11,L,1),I=9,16),L=1,NZEN)/
     * 8.352,8.437,8.513,8.598,8.672,8.740,8.785,8.824,
     * 8.260,8.355,8.436,8.528,8.607,8.678,8.729,8.770,
     * 8.121,8.227,8.318,8.422,8.508,8.588,8.645,8.696,
     * 7.827,7.950,8.064,8.190,8.299,8.396,8.470,8.537,
     * 7.523,7.522,7.552,7.647,7.769,7.898,8.010,8.113,
     * 7.514,7.489,7.473,7.492,7.525,7.591,7.676,7.783,
     * 7.530,7.497,7.469,7.470,7.473,7.479,7.479,7.489,
     * 6.656,6.761,6.863,6.977,7.080,7.166,7.232,7.290,
     * 0.000,0.000,0.000,6.196,6.476,6.699,6.862,6.998,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,11,L,1),I=9,16),L=1,NZEN)/
     * 8.375,8.464,8.548,8.644,8.728,8.802,8.854,8.903,
     * 8.287,8.382,8.471,8.572,8.660,8.739,8.797,8.848,
     * 8.146,8.252,8.353,8.462,8.559,8.644,8.711,8.769,
     * 7.871,7.993,8.110,8.242,8.355,8.460,8.538,8.612,
     * 7.557,7.564,7.603,7.703,7.827,7.956,8.069,8.174,
     * 7.532,7.516,7.510,7.538,7.587,7.660,7.752,7.859,
     * 7.551,7.524,7.509,7.517,7.529,7.539,7.542,7.549,
     * 6.670,6.783,6.896,7.019,7.130,7.219,7.284,7.335,
     * 0.000,0.000,0.000,6.223,6.512,6.742,6.906,7.034,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,11,L,1),I=9,16),L=1,NZEN)/
     * 8.450,8.539,8.634,8.740,8.834,8.919,8.983,9.038,
     * 8.366,8.459,8.558,8.668,8.767,8.856,8.925,8.983,
     * 8.231,8.337,8.442,8.560,8.667,8.763,8.838,8.904,
     * 7.978,8.097,8.213,8.347,8.469,8.578,8.665,8.747,
     * 7.610,7.643,7.716,7.841,7.975,8.111,8.224,8.332,
     * 7.564,7.559,7.571,7.624,7.700,7.803,7.907,8.019,
     * 7.563,7.546,7.544,7.568,7.591,7.612,7.623,7.642,
     * 6.737,6.859,6.983,7.115,7.232,7.321,7.380,7.417,
     * 0.000,0.000,0.000,6.312,6.612,6.848,7.010,7.123,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,11,L,1),I=9,16),L=1,NZEN)/
     * 8.560,8.656,8.764,8.882,8.987,9.080,9.147,9.205,
     * 8.481,8.582,8.691,8.813,8.921,9.018,9.091,9.154,
     * 8.360,8.467,8.581,8.709,8.825,8.931,9.010,9.081,
     * 8.117,8.237,8.363,8.505,8.635,8.754,8.847,8.933,
     * 7.674,7.747,7.858,8.006,8.160,8.303,8.426,8.539,
     * 7.584,7.600,7.645,7.738,7.857,7.983,8.106,8.232,
     * 7.552,7.550,7.568,7.610,7.653,7.699,7.733,7.780,
     * 6.774,6.905,7.037,7.180,7.306,7.399,7.459,7.490,
     * 0.000,0.000,6.000,6.365,6.681,6.929,7.097,7.205,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,12,L,1),I=9,16),L=1,NZEN)/
     * 8.373,8.445,8.515,8.594,8.661,8.724,8.767,8.803,
     * 8.289,8.372,8.442,8.527,8.599,8.666,8.714,8.756,
     * 8.151,8.247,8.330,8.424,8.506,8.578,8.632,8.682,
     * 7.869,7.985,8.091,8.209,8.309,8.400,8.468,8.533,
     * 7.543,7.544,7.581,7.683,7.805,7.932,8.040,8.141,
     * 7.535,7.507,7.486,7.504,7.545,7.623,7.720,7.834,
     * 7.559,7.520,7.487,7.481,7.480,7.481,7.480,7.497,
     * 6.738,6.837,6.930,7.032,7.125,7.201,7.260,7.317,
     * 0.000,0.000,0.000,6.286,6.555,6.768,6.923,7.058,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,12,L,1),I=9,16),L=1,NZEN)/
     * 8.373,8.456,8.530,8.616,8.687,8.753,8.799,8.839,
     * 8.283,8.373,8.451,8.543,8.622,8.692,8.744,8.788,
     * 8.144,8.248,8.337,8.438,8.526,8.604,8.661,8.713,
     * 7.841,7.961,8.075,8.200,8.309,8.407,8.481,8.550,
     * 7.513,7.512,7.546,7.643,7.768,7.898,8.012,8.118,
     * 7.495,7.470,7.455,7.478,7.521,7.595,7.690,7.802,
     * 7.512,7.479,7.455,7.458,7.463,7.471,7.472,7.485,
     * 6.623,6.729,6.834,6.951,7.058,7.147,7.215,7.275,
     * 0.000,0.000,0.000,6.161,6.446,6.674,6.840,6.977,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,12,L,1),I=9,16),L=1,NZEN)/
     * 8.392,8.480,8.565,8.661,8.745,8.818,8.873,8.922,
     * 8.301,8.398,8.486,8.587,8.675,8.754,8.814,8.867,
     * 8.158,8.266,8.365,8.476,8.571,8.659,8.726,8.785,
     * 7.876,8.000,8.120,8.251,8.366,8.470,8.553,8.627,
     * 7.523,7.535,7.588,7.702,7.836,7.973,8.088,8.196,
     * 7.495,7.480,7.477,7.511,7.564,7.645,7.744,7.853,
     * 7.491,7.465,7.451,7.467,7.484,7.499,7.508,7.521,
     * 6.596,6.713,6.832,6.964,7.084,7.181,7.252,7.304,
     * 0.000,0.000,0.000,6.144,6.447,6.686,6.857,6.985,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,12,L,1),I=9,16),L=1,NZEN)/
     * 8.467,8.556,8.652,8.760,8.858,8.944,9.011,9.065,
     * 8.373,8.472,8.570,8.685,8.786,8.878,8.949,9.009,
     * 8.238,8.346,8.452,8.572,8.682,8.780,8.859,8.926,
     * 7.974,8.097,8.213,8.352,8.477,8.590,8.682,8.763,
     * 7.561,7.602,7.686,7.822,7.967,8.108,8.230,8.341,
     * 7.499,7.495,7.514,7.578,7.668,7.784,7.899,8.014,
     * 7.476,7.460,7.461,7.493,7.526,7.555,7.576,7.605,
     * 6.618,6.740,6.871,7.013,7.140,7.239,7.306,7.342,
     * 0.000,0.000,0.000,6.180,6.495,6.744,6.915,7.029,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,12,L,1),I=9,16),L=1,NZEN)/
     * 8.593,8.689,8.800,8.918,9.024,9.118,9.187,9.242,
     * 8.508,8.608,8.719,8.843,8.956,9.054,9.128,9.188,
     * 8.375,8.485,8.602,8.734,8.854,8.962,9.043,9.111,
     * 8.127,8.247,8.377,8.523,8.657,8.779,8.876,8.960,
     * 7.636,7.718,7.841,7.999,8.159,8.311,8.439,8.553,
     * 7.517,7.536,7.593,7.702,7.838,7.977,8.113,8.236,
     * 7.466,7.468,7.494,7.548,7.604,7.664,7.709,7.764,
     * 6.620,6.754,6.896,7.053,7.193,7.301,7.370,7.400,
     * 0.000,0.000,0.000,6.184,6.522,6.786,6.964,7.071,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,1,L,2),I=9,16),L=1,NZEN)/
     * 8.444,8.520,8.591,8.677,8.751,8.818,8.869,8.917,
     * 8.364,8.448,8.525,8.615,8.692,8.765,8.818,8.871,
     * 8.228,8.328,8.413,8.515,8.599,8.679,8.739,8.800,
     * 7.947,8.070,8.180,8.302,8.408,8.503,8.579,8.655,
     * 7.570,7.590,7.651,7.777,7.912,8.046,8.160,8.274,
     * 7.569,7.547,7.534,7.564,7.627,7.723,7.838,7.970,
     * 7.600,7.569,7.544,7.543,7.546,7.549,7.556,7.592,
     * 6.829,6.940,7.041,7.150,7.242,7.314,7.374,7.444,
     * 0.000,0.000,6.125,6.452,6.719,6.923,7.076,7.225,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,1,L,2),I=9,16),L=1,NZEN)/
     * 8.447,8.530,8.608,8.697,8.775,8.847,8.901,8.951,
     * 8.355,8.448,8.532,8.629,8.711,8.788,8.845,8.899,
     * 8.220,8.325,8.421,8.526,8.618,8.701,8.766,8.829,
     * 7.911,8.042,8.161,8.293,8.404,8.506,8.588,8.670,
     * 7.521,7.540,7.598,7.725,7.866,8.005,8.125,8.243,
     * 7.507,7.490,7.484,7.522,7.585,7.682,7.798,7.931,
     * 7.532,7.508,7.490,7.500,7.511,7.521,7.531,7.561,
     * 6.687,6.806,6.921,7.045,7.153,7.240,7.312,7.385,
     * 0.000,0.000,0.000,6.299,6.584,6.807,6.972,7.124,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,1,L,2),I=9,16),L=1,NZEN)/
     * 8.458,8.550,8.639,8.738,8.826,8.908,8.970,9.029,
     * 8.368,8.467,8.561,8.667,8.759,8.845,8.911,8.976,
     * 8.225,8.337,8.440,8.555,8.658,8.751,8.827,8.897,
     * 7.943,8.078,8.201,8.339,8.462,8.571,8.659,8.746,
     * 7.510,7.546,7.627,7.772,7.924,8.071,8.195,8.319,
     * 7.481,7.475,7.481,7.531,7.609,7.716,7.838,7.969,
     * 7.482,7.466,7.461,7.484,7.507,7.527,7.547,7.581,
     * 6.621,6.750,6.880,7.020,7.142,7.240,7.315,7.381,
     * 0.000,0.000,0.000,6.236,6.540,6.779,6.951,7.095,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,1,L,2),I=9,16),L=1,NZEN)/
     * 8.530,8.634,8.744,8.863,8.972,9.075,9.159,9.227,
     * 8.439,8.550,8.662,8.788,8.903,9.010,9.098,9.174,
     * 8.300,8.422,8.540,8.676,8.798,8.914,9.010,9.094,
     * 8.026,8.164,8.301,8.453,8.591,8.722,8.834,8.935,
     * 7.544,7.614,7.734,7.903,8.071,8.235,8.377,8.510,
     * 7.470,7.476,7.513,7.606,7.732,7.881,8.029,8.175,
     * 7.447,7.442,7.455,7.497,7.540,7.583,7.626,7.690,
     * 6.609,6.743,6.885,7.039,7.174,7.280,7.357,7.413,
     * 0.000,0.000,0.000,6.231,6.555,6.808,6.986,7.118,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,1,L,2),I=9,16),L=1,NZEN)/
     * 8.653,8.772,8.903,9.044,9.172,9.292,9.380,9.453,
     * 8.561,8.685,8.819,8.968,9.104,9.226,9.322,9.398,
     * 8.427,8.561,8.700,8.856,9.000,9.132,9.237,9.324,
     * 8.169,8.311,8.466,8.639,8.798,8.948,9.070,9.174,
     * 7.604,7.725,7.888,8.083,8.279,8.460,8.621,8.765,
     * 7.454,7.492,7.579,7.731,7.908,8.095,8.269,8.431,
     * 7.401,7.413,7.453,7.522,7.600,7.677,7.767,7.875,
     * 6.548,6.702,6.867,7.045,7.204,7.329,7.418,7.475,
     * 0.000,0.000,0.000,6.192,6.544,6.822,7.017,7.144,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,2,L,2),I=9,16),L=1,NZEN)/
     * 8.423,8.502,8.574,8.662,8.737,8.806,8.857,8.903,
     * 8.338,8.428,8.505,8.599,8.678,8.752,8.807,8.857,
     * 8.203,8.306,8.393,8.497,8.585,8.666,8.726,8.787,
     * 7.918,8.045,8.158,8.284,8.392,8.487,8.563,8.639,
     * 7.563,7.577,7.634,7.757,7.892,8.027,8.142,8.257,
     * 7.561,7.543,7.529,7.559,7.615,7.707,7.821,7.953,
     * 7.595,7.565,7.539,7.541,7.542,7.545,7.552,7.586,
     * 6.814,6.925,7.028,7.138,7.232,7.305,7.365,7.436,
     * 0.000,0.000,6.111,6.438,6.706,6.911,7.064,7.215,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,2,L,2),I=9,16),L=1,NZEN)/
     * 8.424,8.507,8.586,8.677,8.754,8.826,8.879,8.927,
     * 8.331,8.427,8.510,8.607,8.689,8.765,8.822,8.875,
     * 8.192,8.302,8.395,8.502,8.592,8.677,8.741,8.802,
     * 7.892,8.023,8.142,8.271,8.385,8.488,8.566,8.646,
     * 7.523,7.538,7.593,7.714,7.852,7.991,8.109,8.227,
     * 7.512,7.495,7.488,7.523,7.581,7.674,7.787,7.916,
     * 7.537,7.513,7.495,7.503,7.510,7.520,7.527,7.555,
     * 6.698,6.816,6.929,7.051,7.157,7.242,7.310,7.382,
     * 0.000,0.000,0.000,6.307,6.589,6.810,6.972,7.124,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,2,L,2),I=9,16),L=1,NZEN)/
     * 8.433,8.525,8.613,8.711,8.795,8.875,8.936,8.991,
     * 8.342,8.443,8.535,8.638,8.729,8.813,8.877,8.937,
     * 8.200,8.313,8.415,8.529,8.628,8.719,8.790,8.858,
     * 7.916,8.048,8.171,8.307,8.425,8.533,8.620,8.702,
     * 7.519,7.549,7.623,7.757,7.901,8.044,8.165,8.285,
     * 7.496,7.488,7.492,7.537,7.605,7.704,7.818,7.944,
     * 7.502,7.486,7.477,7.496,7.516,7.531,7.546,7.574,
     * 6.650,6.777,6.902,7.036,7.153,7.245,7.316,7.380,
     * 0.000,0.000,0.000,6.264,6.560,6.792,6.960,7.102,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,2,L,2),I=9,16),L=1,NZEN)/
     * 8.475,8.575,8.681,8.797,8.900,8.994,9.068,9.134,
     * 8.385,8.493,8.601,8.722,8.829,8.930,9.007,9.078,
     * 8.244,8.365,8.481,8.609,8.726,8.833,8.919,8.997,
     * 7.975,8.110,8.240,8.389,8.522,8.642,8.743,8.836,
     * 7.548,7.600,7.700,7.853,8.009,8.162,8.293,8.415,
     * 7.494,7.497,7.522,7.595,7.696,7.826,7.954,8.087,
     * 7.485,7.477,7.486,7.522,7.555,7.586,7.611,7.652,
     * 6.644,6.777,6.914,7.061,7.188,7.287,7.357,7.410,
     * 0.000,0.000,0.000,6.266,6.580,6.824,6.995,7.124,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,2,L,2),I=9,16),L=1,NZEN)/
     * 8.542,8.656,8.781,8.914,9.033,9.143,9.226,9.298,
     * 8.454,8.574,8.699,8.837,8.963,9.076,9.166,9.243,
     * 8.316,8.445,8.578,8.725,8.859,8.980,9.077,9.162,
     * 8.055,8.194,8.344,8.506,8.653,8.792,8.905,9.007,
     * 7.561,7.650,7.787,7.963,8.138,8.306,8.451,8.588,
     * 7.462,7.487,7.547,7.660,7.805,7.959,8.111,8.256,
     * 7.422,7.430,7.463,7.520,7.577,7.636,7.687,7.759,
     * 6.568,6.720,6.878,7.047,7.195,7.309,7.387,7.436,
     * 0.000,0.000,0.000,6.210,6.548,6.814,6.998,7.122,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,3,L,2),I=9,16),L=1,NZEN)/
     * 8.390,8.474,8.550,8.641,8.718,8.789,8.839,8.884,
     * 8.306,8.399,8.481,8.577,8.660,8.733,8.788,8.838,
     * 8.168,8.278,8.369,8.474,8.564,8.646,8.708,8.766,
     * 7.882,8.012,8.130,8.257,8.368,8.467,8.545,8.620,
     * 7.552,7.564,7.613,7.731,7.866,8.002,8.118,8.236,
     * 7.556,7.535,7.523,7.549,7.600,7.688,7.798,7.930,
     * 7.587,7.560,7.535,7.537,7.538,7.540,7.545,7.577,
     * 6.792,6.905,7.009,7.121,7.217,7.292,7.354,7.425,
     * 0.000,0.000,6.090,6.418,6.687,6.894,7.049,7.200,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,3,L,2),I=9,16),L=1,NZEN)/
     * 8.386,8.474,8.554,8.647,8.724,8.795,8.845,8.891,
     * 8.302,8.398,8.482,8.580,8.663,8.738,8.793,8.843,
     * 8.158,8.267,8.364,8.472,8.565,8.647,8.710,8.768,
     * 7.865,7.996,8.115,8.247,8.359,8.459,8.538,8.612,
     * 7.530,7.541,7.590,7.705,7.839,7.975,8.092,8.208,
     * 7.526,7.511,7.501,7.530,7.578,7.657,7.760,7.885,
     * 7.552,7.530,7.508,7.514,7.520,7.525,7.530,7.555,
     * 6.730,6.847,6.957,7.075,7.177,7.258,7.323,7.392,
     * 0.000,0.000,6.006,6.344,6.619,6.833,6.992,7.142,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,3,L,2),I=9,16),L=1,NZEN)/
     * 8.392,8.482,8.572,8.669,8.753,8.829,8.884,8.933,
     * 8.303,8.401,8.496,8.598,8.686,8.767,8.826,8.879,
     * 8.162,8.274,8.378,8.491,8.587,8.675,8.740,8.802,
     * 7.875,8.005,8.127,8.259,8.375,8.480,8.561,8.638,
     * 7.548,7.565,7.618,7.731,7.861,7.994,8.108,8.218,
     * 7.533,7.522,7.524,7.558,7.612,7.694,7.795,7.912,
     * 7.555,7.538,7.525,7.537,7.549,7.557,7.560,7.578,
     * 6.720,6.842,6.961,7.086,7.194,7.277,7.340,7.401,
     * 0.000,0.000,0.000,6.338,6.622,6.842,7.001,7.141,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,3,L,2),I=9,16),L=1,NZEN)/
     * 8.396,8.496,8.596,8.707,8.802,8.886,8.949,9.006,
     * 8.311,8.416,8.519,8.633,8.732,8.822,8.888,8.950,
     * 8.174,8.287,8.400,8.523,8.630,8.726,8.801,8.869,
     * 7.911,8.039,8.169,8.306,8.427,8.539,8.624,8.707,
     * 7.584,7.614,7.682,7.802,7.933,8.065,8.179,8.289,
     * 7.563,7.563,7.577,7.622,7.685,7.770,7.865,7.977,
     * 7.567,7.558,7.559,7.583,7.603,7.619,7.624,7.641,
     * 6.740,6.872,7.003,7.138,7.253,7.339,7.399,7.447,
     * 0.000,0.000,6.014,6.365,6.660,6.888,7.048,7.176,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,3,L,2),I=9,16),L=1,NZEN)/
     * 8.401,8.510,8.627,8.752,8.860,8.955,9.024,9.087,
     * 8.315,8.430,8.550,8.678,8.791,8.890,8.963,9.029,
     * 8.185,8.308,8.432,8.565,8.686,8.792,8.871,8.946,
     * 7.926,8.062,8.197,8.350,8.481,8.601,8.696,8.783,
     * 7.580,7.630,7.721,7.857,8.001,8.136,8.252,8.366,
     * 7.538,7.554,7.589,7.658,7.740,7.844,7.943,8.057,
     * 7.530,7.536,7.557,7.598,7.634,7.662,7.676,7.698,
     * 6.689,6.837,6.986,7.141,7.271,7.370,7.434,7.475,
     * 0.000,0.000,0.000,6.347,6.657,6.898,7.065,7.187,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,4,L,2),I=9,16),L=1,NZEN)/
     * 8.370,8.458,8.538,8.630,8.707,8.778,8.828,8.872,
     * 8.285,8.383,8.469,8.566,8.649,8.722,8.778,8.826,
     * 8.148,8.262,8.356,8.462,8.553,8.637,8.698,8.755,
     * 7.863,7.994,8.114,8.245,8.356,8.457,8.535,8.607,
     * 7.547,7.557,7.604,7.720,7.855,7.991,8.109,8.224,
     * 7.550,7.533,7.520,7.545,7.593,7.680,7.789,7.920,
     * 7.582,7.558,7.533,7.533,7.535,7.536,7.542,7.570,
     * 6.781,6.895,7.000,7.113,7.209,7.285,7.348,7.420,
     * 0.000,0.000,6.081,6.410,6.679,6.886,7.041,7.193,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,4,L,2),I=9,16),L=1,NZEN)/
     * 8.359,8.448,8.528,8.619,8.697,8.767,8.817,8.858,
     * 8.271,8.370,8.456,8.554,8.635,8.710,8.763,8.810,
     * 8.134,8.244,8.341,8.448,8.540,8.620,8.681,8.736,
     * 7.852,7.982,8.101,8.232,8.340,8.439,8.515,8.586,
     * 7.547,7.557,7.603,7.712,7.841,7.972,8.086,8.198,
     * 7.550,7.535,7.524,7.549,7.596,7.675,7.778,7.903,
     * 7.584,7.559,7.537,7.538,7.540,7.540,7.542,7.567,
     * 6.787,6.903,7.009,7.123,7.218,7.292,7.352,7.419,
     * 0.000,0.000,6.090,6.418,6.686,6.892,7.045,7.193,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,4,L,2),I=9,16),L=1,NZEN)/
     * 8.338,8.430,8.519,8.617,8.701,8.774,8.825,8.869,
     * 8.248,8.352,8.445,8.546,8.636,8.714,8.768,8.815,
     * 8.111,8.224,8.327,8.439,8.534,8.620,8.682,8.737,
     * 7.841,7.966,8.085,8.216,8.331,8.429,8.507,8.576,
     * 7.581,7.591,7.628,7.726,7.842,7.964,8.071,8.178,
     * 7.586,7.574,7.568,7.592,7.630,7.691,7.774,7.881,
     * 7.617,7.597,7.580,7.586,7.587,7.585,7.582,7.594,
     * 6.817,6.937,7.047,7.162,7.258,7.331,7.386,7.446,
     * 0.000,0.000,6.119,6.446,6.714,6.920,7.070,7.210,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,4,L,2),I=9,16),L=1,NZEN)/
     * 8.281,8.381,8.483,8.592,8.685,8.765,8.819,8.864,
     * 8.192,8.304,8.408,8.521,8.618,8.701,8.759,8.809,
     * 8.061,8.179,8.290,8.410,8.514,8.604,8.669,8.726,
     * 7.827,7.942,8.060,8.194,8.309,8.410,8.483,8.554,
     * 7.649,7.662,7.694,7.766,7.857,7.957,8.049,8.142,
     * 7.659,7.656,7.660,7.686,7.712,7.747,7.794,7.868,
     * 7.687,7.677,7.671,7.681,7.686,7.683,7.669,7.665,
     * 6.860,6.991,7.113,7.236,7.337,7.411,7.462,7.510,
     * 0.000,0.000,6.114,6.475,6.757,6.969,7.120,7.255,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,4,L,2),I=9,16),L=1,NZEN)/
     * 8.197,8.313,8.432,8.557,8.664,8.753,8.812,8.861,
     * 8.112,8.236,8.357,8.484,8.594,8.687,8.749,8.801,
     * 7.985,8.110,8.240,8.373,8.488,8.585,8.655,8.713,
     * 7.778,7.893,8.019,8.164,8.287,8.393,8.472,8.543,
     * 7.674,7.703,7.744,7.816,7.894,7.976,8.052,8.129,
     * 7.684,7.700,7.723,7.762,7.793,7.818,7.843,7.887,
     * 7.709,7.716,7.729,7.754,7.770,7.769,7.754,7.739,
     * 6.836,6.990,7.136,7.277,7.393,7.476,7.529,7.569,
     * 0.000,0.000,6.102,6.481,6.780,7.005,7.162,7.292,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,5,L,2),I=9,16),L=1,NZEN)/
     * 8.390,8.474,8.552,8.640,8.715,8.785,8.835,8.878,
     * 8.306,8.402,8.484,8.579,8.658,8.731,8.785,8.833,
     * 8.171,8.278,8.372,8.477,8.564,8.646,8.706,8.763,
     * 7.888,8.018,8.135,8.265,8.371,8.469,8.545,8.618,
     * 7.552,7.566,7.619,7.740,7.874,8.011,8.127,8.239,
     * 7.553,7.535,7.523,7.551,7.604,7.695,7.807,7.939,
     * 7.587,7.560,7.535,7.535,7.536,7.538,7.543,7.577,
     * 6.796,6.910,7.014,7.125,7.220,7.295,7.356,7.426,
     * 0.000,0.000,6.100,6.427,6.695,6.899,7.053,7.203,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,5,L,2),I=9,16),L=1,NZEN)/
     * 8.363,8.448,8.526,8.614,8.690,8.758,8.805,8.846,
     * 8.283,8.375,8.458,8.554,8.634,8.706,8.755,8.799,
     * 8.150,8.258,8.349,8.455,8.542,8.621,8.677,8.729,
     * 7.879,8.006,8.122,8.249,8.353,8.448,8.519,8.587,
     * 7.573,7.586,7.637,7.750,7.880,8.008,8.117,8.226,
     * 7.581,7.563,7.549,7.576,7.627,7.712,7.817,7.943,
     * 7.618,7.593,7.567,7.565,7.563,7.558,7.561,7.589,
     * 6.865,6.978,7.080,7.185,7.273,7.339,7.392,7.456,
     * 0.000,0.000,6.193,6.512,6.770,6.965,7.110,7.255,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,5,L,2),I=9,16),L=1,NZEN)/
     * 8.297,8.393,8.481,8.579,8.661,8.732,8.780,8.820,
     * 8.214,8.315,8.409,8.512,8.597,8.674,8.724,8.768,
     * 8.077,8.190,8.291,8.403,8.500,8.581,8.639,8.689,
     * 7.820,7.942,8.060,8.191,8.302,8.400,8.472,8.536,
     * 7.613,7.619,7.649,7.734,7.841,7.956,8.059,8.162,
     * 7.631,7.619,7.608,7.628,7.655,7.707,7.781,7.884,
     * 7.677,7.655,7.636,7.633,7.627,7.619,7.608,7.619,
     * 6.918,7.035,7.138,7.243,7.329,7.392,7.441,7.499,
     * 0.000,0.000,6.252,6.564,6.817,7.010,7.150,7.291,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,5,L,2),I=9,16),L=1,NZEN)/
     * 8.149,8.260,8.370,8.488,8.584,8.665,8.718,8.759,
     * 8.070,8.186,8.296,8.417,8.517,8.602,8.660,8.704,
     * 7.945,8.063,8.176,8.301,8.406,8.495,8.557,8.610,
     * 7.760,7.852,7.957,8.083,8.195,8.294,8.364,8.429,
     * 7.700,7.706,7.720,7.763,7.817,7.883,7.951,8.032,
     * 7.723,7.721,7.719,7.738,7.748,7.758,7.771,7.813,
     * 7.773,7.763,7.753,7.756,7.752,7.738,7.717,7.708,
     * 6.952,7.082,7.197,7.311,7.403,7.470,7.518,7.572,
     * 0.000,6.054,6.328,6.625,6.872,7.059,7.197,7.334,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,5,L,2),I=9,16),L=1,NZEN)/
     * 7.973,8.102,8.235,8.376,8.497,8.595,8.660,8.707,
     * 7.897,8.028,8.162,8.303,8.424,8.524,8.595,8.645,
     * 7.789,7.916,8.047,8.191,8.317,8.420,8.496,8.554,
     * 7.708,7.787,7.881,8.003,8.113,8.209,8.284,8.347,
     * 7.752,7.782,7.813,7.855,7.888,7.913,7.935,7.973,
     * 7.781,7.803,7.824,7.856,7.871,7.867,7.853,7.848,
     * 7.822,7.833,7.842,7.861,7.866,7.852,7.829,7.810,
     * 6.919,7.076,7.216,7.350,7.457,7.536,7.590,7.645,
     * 0.000,0.000,6.224,6.589,6.869,7.078,7.230,7.374,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,6,L,2),I=9,16),L=1,NZEN)/
     * 8.429,8.509,8.580,8.665,8.738,8.804,8.853,8.898,
     * 8.349,8.437,8.514,8.604,8.681,8.752,8.804,8.853,
     * 8.217,8.317,8.405,8.506,8.590,8.667,8.727,8.783,
     * 7.939,8.064,8.176,8.298,8.400,8.496,8.571,8.642,
     * 7.565,7.586,7.649,7.778,7.914,8.047,8.160,8.270,
     * 7.561,7.543,7.531,7.563,7.627,7.726,7.841,7.973,
     * 7.595,7.565,7.539,7.539,7.540,7.543,7.550,7.589,
     * 6.828,6.939,7.041,7.150,7.242,7.312,7.370,7.437,
     * 0.000,0.000,6.134,6.459,6.724,6.927,7.077,7.224,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,6,L,2),I=9,16),L=1,NZEN)/
     * 8.392,8.474,8.546,8.629,8.703,8.767,8.814,8.854,
     * 8.317,8.402,8.482,8.571,8.646,8.715,8.764,8.808,
     * 8.188,8.290,8.378,8.475,8.561,8.635,8.690,8.740,
     * 7.929,8.052,8.163,8.282,8.383,8.473,8.542,8.607,
     * 7.598,7.620,7.680,7.800,7.929,8.054,8.160,8.265,
     * 7.606,7.587,7.574,7.604,7.663,7.759,7.867,7.992,
     * 7.648,7.620,7.592,7.588,7.581,7.577,7.579,7.619,
     * 6.938,7.048,7.144,7.244,7.324,7.383,7.430,7.491,
     * 0.000,0.000,6.284,6.595,6.845,7.032,7.171,7.312,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,6,L,2),I=9,16),L=1,NZEN)/
     * 8.282,8.379,8.465,8.563,8.646,8.717,8.764,8.801,
     * 8.204,8.305,8.396,8.500,8.585,8.659,8.709,8.750,
     * 8.071,8.183,8.284,8.396,8.490,8.571,8.628,8.676,
     * 7.825,7.946,8.063,8.193,8.302,8.398,8.467,8.531,
     * 7.639,7.645,7.674,7.756,7.860,7.972,8.073,8.175,
     * 7.668,7.654,7.641,7.657,7.683,7.735,7.810,7.917,
     * 7.723,7.700,7.676,7.670,7.659,7.646,7.633,7.644,
     * 7.002,7.115,7.213,7.310,7.389,7.444,7.488,7.545,
     * 0.000,6.052,6.360,6.663,6.906,7.089,7.223,7.363,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,6,L,2),I=9,16),L=1,NZEN)/
     * 8.067,8.186,8.299,8.426,8.528,8.615,8.670,8.710,
     * 7.986,8.106,8.222,8.351,8.458,8.545,8.606,8.651,
     * 7.869,7.986,8.101,8.231,8.340,8.432,8.496,8.550,
     * 7.718,7.798,7.895,8.019,8.132,8.233,8.309,8.372,
     * 7.714,7.722,7.734,7.770,7.810,7.863,7.920,7.996,
     * 7.755,7.756,7.755,7.772,7.779,7.781,7.781,7.812,
     * 7.813,7.806,7.797,7.800,7.793,7.778,7.754,7.744,
     * 7.017,7.147,7.258,7.367,7.454,7.517,7.564,7.625,
     * 0.000,6.135,6.417,6.711,6.950,7.130,7.266,7.408,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,6,L,2),I=9,16),L=1,NZEN)/
     * 7.816,7.959,8.104,8.261,8.396,8.508,8.582,8.635,
     * 7.752,7.892,8.034,8.189,8.324,8.439,8.517,8.574,
     * 7.680,7.794,7.920,8.067,8.196,8.309,8.389,8.455,
     * 7.684,7.745,7.819,7.919,8.016,8.104,8.173,8.239,
     * 7.767,7.800,7.828,7.870,7.897,7.908,7.916,7.935,
     * 7.798,7.820,7.840,7.874,7.892,7.893,7.881,7.875,
     * 7.842,7.857,7.869,7.892,7.902,7.896,7.879,7.868,
     * 6.957,7.112,7.247,7.378,7.486,7.567,7.629,7.701,
     * 0.000,6.138,6.421,6.722,6.972,7.163,7.309,7.460,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.004,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,7,L,2),I=9,16),L=1,NZEN)/
     * 8.456,8.532,8.600,8.681,8.752,8.818,8.867,8.911,
     * 8.378,8.462,8.535,8.622,8.696,8.766,8.818,8.866,
     * 8.248,8.343,8.427,8.526,8.606,8.684,8.742,8.798,
     * 7.972,8.094,8.201,8.319,8.422,8.513,8.586,8.658,
     * 7.575,7.602,7.672,7.803,7.938,8.069,8.182,8.290,
     * 7.569,7.547,7.538,7.574,7.643,7.747,7.863,7.995,
     * 7.600,7.569,7.544,7.543,7.544,7.547,7.557,7.599,
     * 6.848,6.958,7.058,7.165,7.255,7.323,7.379,7.445,
     * 0.000,0.000,6.155,6.479,6.743,6.944,7.093,7.237,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,7,L,2),I=9,16),L=1,NZEN)/
     * 8.421,8.495,8.566,8.648,8.720,8.785,8.830,8.870,
     * 8.340,8.425,8.500,8.588,8.664,8.731,8.781,8.825,
     * 8.216,8.314,8.400,8.497,8.579,8.652,8.706,8.757,
     * 7.960,8.082,8.189,8.306,8.405,8.493,8.560,8.627,
     * 7.610,7.636,7.704,7.829,7.959,8.083,8.186,8.292,
     * 7.617,7.597,7.586,7.620,7.684,7.785,7.895,8.022,
     * 7.661,7.632,7.601,7.596,7.589,7.587,7.591,7.636,
     * 6.970,7.078,7.171,7.268,7.345,7.402,7.447,7.507,
     * 0.000,6.004,6.321,6.629,6.876,7.061,7.196,7.336,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,7,L,2),I=9,16),L=1,NZEN)/
     * 8.300,8.393,8.479,8.575,8.658,8.729,8.777,8.814,
     * 8.221,8.320,8.413,8.514,8.599,8.674,8.723,8.765,
     * 8.091,8.203,8.304,8.415,8.507,8.588,8.644,8.693,
     * 7.847,7.969,8.086,8.215,8.323,8.418,8.488,8.552,
     * 7.650,7.658,7.690,7.776,7.886,8.001,8.101,8.204,
     * 7.682,7.668,7.654,7.669,7.698,7.756,7.838,7.949,
     * 7.738,7.715,7.689,7.683,7.672,7.656,7.644,7.658,
     * 7.038,7.151,7.246,7.341,7.416,7.468,7.509,7.566,
     * 0.000,6.100,6.406,6.707,6.945,7.123,7.255,7.394,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,7,L,2),I=9,16),L=1,NZEN)/
     * 8.075,8.197,8.314,8.438,8.544,8.629,8.686,8.727,
     * 7.995,8.118,8.238,8.367,8.473,8.563,8.623,8.670,
     * 7.873,7.995,8.113,8.246,8.356,8.451,8.517,8.572,
     * 7.720,7.804,7.905,8.035,8.152,8.253,8.331,8.401,
     * 7.720,7.728,7.741,7.778,7.823,7.880,7.944,8.026,
     * 7.766,7.765,7.765,7.782,7.788,7.791,7.796,7.833,
     * 7.828,7.821,7.811,7.815,7.807,7.789,7.765,7.756,
     * 7.056,7.184,7.293,7.399,7.483,7.543,7.589,7.650,
     * 0.000,6.159,6.454,6.752,6.990,7.169,7.303,7.446,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,7,L,2),I=9,16),L=1,NZEN)/
     * 7.814,7.962,8.112,8.274,8.410,8.524,8.600,8.651,
     * 7.750,7.895,8.044,8.203,8.342,8.457,8.537,8.596,
     * 7.671,7.792,7.925,8.079,8.213,8.327,8.411,8.480,
     * 7.667,7.730,7.807,7.916,8.021,8.117,8.195,8.266,
     * 7.760,7.792,7.820,7.864,7.894,7.913,7.926,7.954,
     * 7.796,7.820,7.840,7.876,7.898,7.901,7.892,7.890,
     * 7.849,7.867,7.880,7.905,7.917,7.912,7.893,7.883,
     * 6.999,7.153,7.286,7.415,7.521,7.600,7.661,7.733,
     * 0.000,6.172,6.468,6.771,7.019,7.208,7.351,7.503,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.060,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,8,L,2),I=9,16),L=1,NZEN)/
     * 8.450,8.525,8.597,8.679,8.751,8.817,8.866,8.911,
     * 8.371,8.456,8.531,8.618,8.695,8.764,8.817,8.866,
     * 8.240,8.339,8.424,8.521,8.604,8.681,8.739,8.796,
     * 7.964,8.087,8.195,8.316,8.416,8.511,8.583,8.655,
     * 7.573,7.598,7.667,7.798,7.932,8.064,8.174,8.287,
     * 7.566,7.547,7.536,7.572,7.639,7.741,7.858,7.990,
     * 7.597,7.569,7.541,7.543,7.544,7.547,7.556,7.597,
     * 6.844,6.954,7.055,7.163,7.253,7.322,7.378,7.444,
     * 0.000,0.000,6.149,6.474,6.739,6.940,7.089,7.235,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,8,L,2),I=9,16),L=1,NZEN)/
     * 8.424,8.500,8.571,8.656,8.730,8.795,8.842,8.885,
     * 8.343,8.430,8.507,8.596,8.672,8.742,8.794,8.839,
     * 8.216,8.316,8.404,8.502,8.585,8.661,8.718,8.771,
     * 7.956,8.079,8.189,8.307,8.408,8.499,8.567,8.636,
     * 7.603,7.627,7.695,7.821,7.951,8.078,8.184,8.291,
     * 7.608,7.588,7.577,7.612,7.676,7.776,7.888,8.016,
     * 7.649,7.620,7.592,7.588,7.584,7.583,7.587,7.629,
     * 6.944,7.053,7.150,7.249,7.330,7.390,7.437,7.499,
     * 0.000,0.000,6.287,6.599,6.850,7.038,7.176,7.318,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,8,L,2),I=9,16),L=1,NZEN)/
     * 8.334,8.425,8.513,8.610,8.694,8.765,8.814,8.855,
     * 8.252,8.354,8.445,8.545,8.634,8.709,8.762,8.806,
     * 8.125,8.238,8.337,8.448,8.541,8.623,8.682,8.733,
     * 7.873,7.998,8.116,8.247,8.355,8.451,8.524,8.591,
     * 7.642,7.653,7.693,7.790,7.906,8.026,8.131,8.235,
     * 7.669,7.653,7.642,7.662,7.697,7.761,7.848,7.963,
     * 7.721,7.696,7.672,7.667,7.660,7.649,7.640,7.659,
     * 7.014,7.127,7.225,7.322,7.400,7.456,7.499,7.557,
     * 0.000,6.063,6.373,6.677,6.921,7.104,7.239,7.377,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,8,L,2),I=9,16),L=1,NZEN)/
     * 8.174,8.291,8.405,8.524,8.624,8.707,8.764,8.805,
     * 8.095,8.217,8.330,8.455,8.559,8.646,8.705,8.752,
     * 7.971,8.096,8.217,8.347,8.455,8.548,8.615,8.669,
     * 7.775,7.877,7.993,8.129,8.250,8.352,8.431,8.500,
     * 7.717,7.724,7.740,7.791,7.856,7.938,8.022,8.117,
     * 7.751,7.749,7.746,7.763,7.776,7.791,7.819,7.883,
     * 7.814,7.804,7.790,7.791,7.785,7.771,7.749,7.744,
     * 7.058,7.184,7.292,7.398,7.481,7.541,7.584,7.640,
     * 0.000,6.163,6.447,6.743,6.982,7.162,7.296,7.435,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,8,L,2),I=9,16),L=1,NZEN)/
     * 7.978,8.121,8.266,8.415,8.541,8.644,8.712,8.762,
     * 7.905,8.052,8.196,8.349,8.478,8.583,8.656,8.710,
     * 7.799,7.939,8.082,8.237,8.372,8.482,8.560,8.623,
     * 7.711,7.795,7.898,8.032,8.155,8.267,8.348,8.421,
     * 7.754,7.780,7.807,7.852,7.892,7.936,7.981,8.047,
     * 7.800,7.817,7.833,7.862,7.880,7.883,7.880,7.896,
     * 7.855,7.863,7.868,7.886,7.891,7.882,7.863,7.853,
     * 7.031,7.179,7.308,7.432,7.533,7.607,7.661,7.723,
     * 0.000,6.183,6.473,6.777,7.028,7.217,7.359,7.502,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.051,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,9,L,2),I=9,16),L=1,NZEN)/
     * 8.426,8.504,8.578,8.665,8.738,8.806,8.856,8.900,
     * 8.346,8.434,8.512,8.603,8.681,8.753,8.805,8.853,
     * 8.209,8.314,8.402,8.502,8.588,8.667,8.727,8.785,
     * 7.930,8.057,8.170,8.295,8.397,8.492,8.567,8.641,
     * 7.563,7.584,7.644,7.771,7.905,8.039,8.152,8.267,
     * 7.561,7.543,7.531,7.563,7.623,7.720,7.834,7.965,
     * 7.592,7.565,7.539,7.541,7.542,7.543,7.552,7.588,
     * 6.822,6.934,7.036,7.146,7.238,7.310,7.369,7.437,
     * 0.000,0.000,6.126,6.452,6.718,6.921,7.072,7.220,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,9,L,2),I=9,16),L=1,NZEN)/
     * 8.405,8.488,8.566,8.656,8.732,8.802,8.852,8.896,
     * 8.326,8.420,8.500,8.596,8.675,8.749,8.801,8.851,
     * 8.194,8.301,8.391,8.496,8.583,8.663,8.724,8.779,
     * 7.920,8.048,8.162,8.289,8.393,8.490,8.564,8.637,
     * 7.578,7.596,7.657,7.779,7.913,8.045,8.156,8.267,
     * 7.582,7.564,7.554,7.584,7.645,7.739,7.849,7.979,
     * 7.618,7.593,7.567,7.567,7.567,7.567,7.572,7.607,
     * 6.872,6.984,7.087,7.193,7.281,7.349,7.403,7.468,
     * 0.000,0.000,6.194,6.515,6.775,6.973,7.119,7.265,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,9,L,2),I=9,16),L=1,NZEN)/
     * 8.378,8.469,8.556,8.655,8.737,8.814,8.866,8.913,
     * 8.294,8.394,8.487,8.588,8.677,8.755,8.813,8.864,
     * 8.163,8.276,8.374,8.486,8.581,8.667,8.731,8.789,
     * 7.898,8.025,8.145,8.276,8.388,8.489,8.565,8.639,
     * 7.620,7.636,7.683,7.790,7.914,8.040,8.152,8.261,
     * 7.632,7.617,7.610,7.637,7.682,7.757,7.854,7.973,
     * 7.674,7.652,7.631,7.631,7.630,7.626,7.624,7.646,
     * 6.934,7.049,7.153,7.259,7.346,7.410,7.460,7.520,
     * 0.000,0.000,6.265,6.581,6.837,7.031,7.174,7.314,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,9,L,2),I=9,16),L=1,NZEN)/
     * 8.321,8.429,8.534,8.649,8.744,8.829,8.889,8.940,
     * 8.244,8.359,8.467,8.585,8.686,8.773,8.837,8.892,
     * 8.119,8.240,8.358,8.482,8.587,8.681,8.751,8.814,
     * 7.884,8.009,8.134,8.275,8.394,8.500,8.581,8.658,
     * 7.697,7.711,7.748,7.833,7.938,8.053,8.156,8.262,
     * 7.715,7.708,7.709,7.732,7.765,7.814,7.883,7.981,
     * 7.758,7.743,7.732,7.737,7.736,7.731,7.721,7.726,
     * 7.008,7.133,7.244,7.355,7.444,7.508,7.553,7.604,
     * 0.000,6.128,6.392,6.686,6.933,7.121,7.258,7.391,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,9,L,2),I=9,16),L=1,NZEN)/
     * 8.244,8.366,8.495,8.630,8.747,8.843,8.913,8.970,
     * 8.170,8.300,8.430,8.567,8.686,8.787,8.861,8.920,
     * 8.050,8.190,8.324,8.468,8.593,8.700,8.780,8.847,
     * 7.844,7.975,8.115,8.268,8.401,8.520,8.609,8.689,
     * 7.732,7.759,7.804,7.887,7.985,8.092,8.188,8.285,
     * 7.753,7.763,7.778,7.811,7.845,7.885,7.938,8.021,
     * 7.798,7.798,7.801,7.818,7.827,7.823,7.811,7.809,
     * 7.022,7.165,7.293,7.418,7.517,7.588,7.635,7.682,
     * 0.000,6.044,6.341,6.695,6.973,7.179,7.325,7.459,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,10,L,2),I=9,16),L=1,NZEN)/
     * 8.407,8.489,8.564,8.654,8.729,8.800,8.849,8.894,
     * 8.322,8.417,8.496,8.590,8.672,8.745,8.799,8.847,
     * 8.187,8.294,8.384,8.488,8.577,8.658,8.719,8.777,
     * 7.905,8.033,8.149,8.276,8.383,8.483,8.558,8.632,
     * 7.558,7.573,7.629,7.750,7.886,8.022,8.137,8.253,
     * 7.558,7.540,7.527,7.555,7.612,7.706,7.817,7.948,
     * 7.590,7.563,7.537,7.539,7.540,7.543,7.550,7.583,
     * 6.806,6.919,7.023,7.134,7.228,7.302,7.362,7.432,
     * 0.000,0.000,6.107,6.435,6.702,6.906,7.060,7.209,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,10,L,2),I=9,16),L=1,NZEN)/
     * 8.402,8.491,8.569,8.662,8.740,8.813,8.866,8.914,
     * 8.320,8.414,8.500,8.598,8.681,8.757,8.814,8.867,
     * 8.179,8.291,8.386,8.494,8.585,8.668,8.732,8.792,
     * 7.894,8.025,8.145,8.277,8.386,8.487,8.567,8.642,
     * 7.552,7.569,7.623,7.744,7.881,8.017,8.134,8.251,
     * 7.551,7.536,7.526,7.557,7.611,7.699,7.810,7.938,
     * 7.582,7.558,7.538,7.542,7.545,7.549,7.556,7.586,
     * 6.791,6.907,7.015,7.129,7.227,7.302,7.365,7.434,
     * 0.000,0.000,6.088,6.419,6.689,6.896,7.051,7.200,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,10,L,2),I=9,16),L=1,NZEN)/
     * 8.412,8.506,8.593,8.695,8.780,8.861,8.918,8.972,
     * 8.328,8.428,8.522,8.626,8.718,8.800,8.864,8.922,
     * 8.192,8.304,8.408,8.521,8.620,8.710,8.780,8.847,
     * 7.916,8.047,8.169,8.304,8.419,8.525,8.609,8.691,
     * 7.588,7.607,7.666,7.786,7.919,8.054,8.172,8.287,
     * 7.587,7.574,7.572,7.606,7.660,7.746,7.850,7.973,
     * 7.614,7.595,7.579,7.587,7.594,7.600,7.605,7.630,
     * 6.826,6.946,7.058,7.175,7.274,7.350,7.409,7.472,
     * 0.000,0.000,6.128,6.459,6.732,6.941,7.094,7.235,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,10,L,2),I=9,16),L=1,NZEN)/
     * 8.453,8.553,8.656,8.772,8.872,8.965,9.035,9.100,
     * 8.372,8.480,8.587,8.706,8.811,8.907,8.981,9.050,
     * 8.248,8.363,8.477,8.604,8.716,8.818,8.900,8.977,
     * 7.994,8.127,8.256,8.399,8.525,8.640,8.735,8.825,
     * 7.662,7.699,7.774,7.904,8.045,8.179,8.300,8.419,
     * 7.647,7.644,7.655,7.704,7.776,7.876,7.986,8.113,
     * 7.676,7.663,7.658,7.674,7.690,7.700,7.708,7.735,
     * 6.909,7.036,7.157,7.281,7.383,7.460,7.514,7.565,
     * 0.000,0.000,6.165,6.534,6.823,7.041,7.195,7.329,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,10,L,2),I=9,16),L=1,NZEN)/
     * 8.493,8.606,8.732,8.864,8.982,9.090,9.174,9.248,
     * 8.420,8.539,8.664,8.801,8.921,9.034,9.121,9.200,
     * 8.302,8.430,8.557,8.700,8.829,8.948,9.041,9.130,
     * 8.061,8.202,8.347,8.503,8.643,8.775,8.883,8.986,
     * 7.690,7.760,7.872,8.028,8.185,8.335,8.464,8.594,
     * 7.659,7.673,7.711,7.790,7.895,8.019,8.147,8.283,
     * 7.670,7.672,7.685,7.718,7.748,7.776,7.801,7.850,
     * 6.932,7.074,7.210,7.348,7.462,7.546,7.602,7.649,
     * 0.000,0.000,6.197,6.582,6.889,7.121,7.280,7.410,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,11,L,2),I=9,16),L=1,NZEN)/
     * 8.417,8.497,8.570,8.659,8.735,8.805,8.855,8.900,
     * 8.334,8.422,8.502,8.595,8.676,8.749,8.805,8.854,
     * 8.196,8.298,8.390,8.495,8.583,8.663,8.723,8.783,
     * 7.912,8.040,8.154,8.280,8.389,8.485,8.561,8.636,
     * 7.560,7.575,7.631,7.754,7.891,8.025,8.140,8.257,
     * 7.561,7.540,7.529,7.557,7.614,7.707,7.820,7.951,
     * 7.592,7.565,7.539,7.541,7.542,7.545,7.552,7.585,
     * 6.809,6.921,7.024,7.135,7.229,7.303,7.364,7.435,
     * 0.000,0.000,6.107,6.435,6.703,6.908,7.062,7.211,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,11,L,2),I=9,16),L=1,NZEN)/
     * 8.417,8.505,8.586,8.680,8.759,8.831,8.885,8.934,
     * 8.328,8.426,8.511,8.609,8.695,8.773,8.829,8.884,
     * 8.190,8.301,8.398,8.507,8.600,8.685,8.750,8.813,
     * 7.893,8.026,8.146,8.279,8.395,8.496,8.578,8.658,
     * 7.532,7.548,7.604,7.727,7.866,8.006,8.127,8.245,
     * 7.530,7.512,7.504,7.535,7.589,7.675,7.782,7.912,
     * 7.554,7.530,7.510,7.518,7.526,7.535,7.543,7.571,
     * 6.726,6.844,6.956,7.077,7.182,7.265,7.332,7.405,
     * 0.000,0.000,0.000,6.341,6.621,6.838,6.999,7.150,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,11,L,2),I=9,16),L=1,NZEN)/
     * 8.443,8.535,8.624,8.727,8.816,8.896,8.959,9.016,
     * 8.354,8.455,8.549,8.657,8.750,8.837,8.902,8.965,
     * 8.214,8.328,8.432,8.549,8.651,8.745,8.818,8.889,
     * 7.933,8.068,8.194,8.331,8.450,8.559,8.648,8.735,
     * 7.552,7.577,7.644,7.774,7.918,8.061,8.183,8.305,
     * 7.533,7.522,7.527,7.570,7.640,7.739,7.857,7.987,
     * 7.558,7.541,7.531,7.548,7.564,7.578,7.590,7.617,
     * 6.719,6.843,6.966,7.096,7.209,7.297,7.366,7.430,
     * 0.000,0.000,0.000,6.344,6.633,6.858,7.022,7.165,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,11,L,2),I=9,16),L=1,NZEN)/
     * 8.535,8.637,8.744,8.862,8.970,9.070,9.152,9.222,
     * 8.448,8.559,8.668,8.792,8.904,9.011,9.095,9.171,
     * 8.318,8.438,8.555,8.686,8.806,8.917,9.011,9.095,
     * 8.056,8.194,8.327,8.474,8.608,8.736,8.843,8.944,
     * 7.615,7.677,7.788,7.948,8.107,8.262,8.398,8.529,
     * 7.561,7.564,7.594,7.674,7.785,7.924,8.064,8.207,
     * 7.567,7.558,7.564,7.597,7.628,7.661,7.692,7.746,
     * 6.776,6.909,7.043,7.183,7.305,7.398,7.465,7.519,
     * 0.000,0.000,6.057,6.417,6.721,6.957,7.122,7.253,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,11,L,2),I=9,16),L=1,NZEN)/
     * 8.663,8.779,8.908,9.047,9.172,9.288,9.376,9.450,
     * 8.582,8.705,8.836,8.979,9.109,9.232,9.323,9.404,
     * 8.459,8.589,8.726,8.877,9.016,9.144,9.244,9.336,
     * 8.219,8.358,8.509,8.672,8.827,8.971,9.090,9.196,
     * 7.693,7.811,7.968,8.156,8.337,8.513,8.666,8.814,
     * 7.566,7.601,7.681,7.822,7.987,8.164,8.329,8.493,
     * 7.536,7.543,7.573,7.629,7.692,7.761,7.838,7.947,
     * 6.777,6.923,7.072,7.227,7.364,7.469,7.546,7.601,
     * 0.000,0.000,6.061,6.441,6.768,7.023,7.202,7.331,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,12,L,2),I=9,16),L=1,NZEN)/
     * 8.438,8.516,8.587,8.672,8.747,8.815,8.867,8.915,
     * 8.353,8.443,8.518,8.610,8.688,8.761,8.815,8.867,
     * 8.220,8.321,8.408,8.508,8.595,8.675,8.737,8.796,
     * 7.938,8.063,8.174,8.298,8.403,8.498,8.574,8.651,
     * 7.568,7.586,7.646,7.771,7.907,8.041,8.156,8.270,
     * 7.566,7.545,7.534,7.563,7.623,7.721,7.834,7.966,
     * 7.597,7.569,7.544,7.543,7.546,7.547,7.556,7.591,
     * 6.824,6.934,7.036,7.146,7.239,7.312,7.371,7.442,
     * 0.000,0.000,6.121,6.448,6.715,6.919,7.072,7.222,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,12,L,2),I=9,16),L=1,NZEN)/
     * 8.438,8.525,8.604,8.695,8.773,8.846,8.901,8.951,
     * 8.349,8.443,8.528,8.627,8.710,8.787,8.846,8.901,
     * 8.212,8.323,8.417,8.525,8.616,8.701,8.768,8.829,
     * 7.906,8.038,8.157,8.291,8.403,8.508,8.590,8.671,
     * 7.523,7.540,7.599,7.724,7.865,8.007,8.129,8.250,
     * 7.511,7.496,7.489,7.525,7.588,7.684,7.801,7.935,
     * 7.537,7.515,7.497,7.504,7.516,7.528,7.537,7.566,
     * 6.694,6.813,6.927,7.051,7.159,7.246,7.317,7.391,
     * 0.000,0.000,0.000,6.310,6.593,6.815,6.979,7.131,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,12,L,2),I=9,16),L=1,NZEN)/
     * 8.458,8.550,8.642,8.742,8.830,8.913,8.978,9.038,
     * 8.369,8.470,8.564,8.670,8.764,8.852,8.921,8.985,
     * 8.228,8.342,8.444,8.561,8.663,8.759,8.835,8.906,
     * 7.940,8.076,8.201,8.340,8.462,8.573,8.664,8.752,
     * 7.521,7.557,7.636,7.780,7.932,8.080,8.208,8.329,
     * 7.496,7.488,7.494,7.544,7.621,7.728,7.851,7.983,
     * 7.498,7.484,7.477,7.498,7.521,7.541,7.560,7.595,
     * 6.646,6.774,6.904,7.042,7.163,7.260,7.334,7.400,
     * 0.000,0.000,0.000,6.266,6.567,6.803,6.975,7.118,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,12,L,2),I=9,16),L=1,NZEN)/
     * 8.553,8.658,8.768,8.891,9.002,9.107,9.191,9.261,
     * 8.465,8.577,8.690,8.817,8.933,9.044,9.133,9.210,
     * 8.328,8.451,8.568,8.706,8.831,8.949,9.046,9.132,
     * 8.059,8.197,8.333,8.487,8.627,8.760,8.873,8.976,
     * 7.571,7.646,7.769,7.943,8.110,8.274,8.419,8.556,
     * 7.497,7.504,7.543,7.639,7.769,7.922,8.072,8.221,
     * 7.480,7.474,7.486,7.527,7.570,7.615,7.660,7.728,
     * 6.659,6.794,6.934,7.086,7.219,7.324,7.400,7.457,
     * 0.000,0.000,0.000,6.291,6.610,6.861,7.038,7.170,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,12,L,2),I=9,16),L=1,NZEN)/
     * 8.706,8.826,8.958,9.099,9.230,9.349,9.440,9.512,
     * 8.618,8.745,8.879,9.027,9.164,9.291,9.386,9.463,
     * 8.489,8.623,8.764,8.921,9.066,9.200,9.304,9.389,
     * 8.233,8.382,8.538,8.710,8.871,9.022,9.146,9.249,
     * 7.668,7.801,7.973,8.170,8.364,8.548,8.711,8.857,
     * 7.502,7.547,7.645,7.808,7.992,8.185,8.363,8.531,
     * 7.452,7.465,7.505,7.577,7.656,7.741,7.842,7.965,
     * 6.621,6.774,6.936,7.110,7.265,7.388,7.477,7.536,
     * 0.000,0.000,0.000,6.281,6.629,6.901,7.092,7.221,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,1,L,3),I=9,16),L=1,NZEN)/
     * 8.525,8.604,8.673,8.764,8.840,8.914,8.971,9.026,
     * 8.449,8.536,8.613,8.707,8.788,8.866,8.926,8.987,
     * 8.320,8.425,8.511,8.615,8.702,8.787,8.854,8.924,
     * 8.052,8.179,8.292,8.416,8.523,8.623,8.702,8.790,
     * 7.659,7.695,7.775,7.915,8.056,8.191,8.310,8.433,
     * 7.650,7.641,7.640,7.687,7.762,7.872,7.996,8.145,
     * 7.674,7.659,7.644,7.655,7.662,7.666,7.680,7.738,
     * 6.947,7.074,7.189,7.308,7.401,7.469,7.530,7.616,
     * 0.000,0.000,6.332,6.670,6.934,7.130,7.279,7.446,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,1,L,3),I=9,16),L=1,NZEN)/
     * 8.527,8.611,8.692,8.785,8.864,8.941,9.002,9.059,
     * 8.440,8.538,8.622,8.721,8.807,8.888,8.952,9.015,
     * 8.310,8.422,8.517,8.626,8.720,8.809,8.880,8.952,
     * 8.014,8.151,8.270,8.407,8.519,8.626,8.713,8.806,
     * 7.591,7.629,7.712,7.856,8.005,8.150,8.277,8.408,
     * 7.571,7.568,7.575,7.629,7.711,7.825,7.955,8.104,
     * 7.587,7.579,7.573,7.595,7.611,7.626,7.643,7.694,
     * 6.785,6.922,7.051,7.186,7.297,7.382,7.457,7.548,
     * 0.000,0.000,6.136,6.497,6.783,6.999,7.163,7.333,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,1,L,3),I=9,16),L=1,NZEN)/
     * 8.539,8.632,8.723,8.827,8.916,9.005,9.076,9.142,
     * 8.452,8.555,8.649,8.760,8.856,8.948,9.024,9.095,
     * 8.314,8.430,8.535,8.656,8.760,8.862,8.944,9.023,
     * 8.040,8.181,8.309,8.451,8.573,8.689,8.787,8.883,
     * 7.555,7.618,7.730,7.896,8.057,8.212,8.342,8.479,
     * 7.516,7.521,7.543,7.616,7.718,7.849,7.988,8.138,
     * 7.511,7.510,7.517,7.554,7.585,7.613,7.644,7.706,
     * 6.689,6.835,6.980,7.132,7.259,7.358,7.438,7.523,
     * 0.000,0.000,6.020,6.400,6.708,6.943,7.116,7.277,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,1,L,3),I=9,16),L=1,NZEN)/
     * 8.616,8.724,8.836,8.962,9.078,9.186,9.272,9.347,
     * 8.529,8.643,8.759,8.892,9.012,9.126,9.219,9.299,
     * 8.395,8.520,8.644,8.785,8.914,9.036,9.137,9.229,
     * 8.123,8.270,8.413,8.571,8.715,8.853,8.971,9.079,
     * 7.586,7.692,7.842,8.033,8.210,8.380,8.528,8.677,
     * 7.490,7.514,7.573,7.699,7.855,8.023,8.186,8.347,
     * 7.468,7.475,7.501,7.557,7.610,7.666,7.728,7.826,
     * 6.665,6.816,6.971,7.136,7.277,7.386,7.470,7.545,
     * 0.000,0.000,0.000,6.373,6.701,6.952,7.133,7.282,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,1,L,3),I=9,16),L=1,NZEN)/
     * 8.742,8.870,9.005,9.150,9.283,9.406,9.499,9.573,
     * 8.656,8.787,8.927,9.080,9.219,9.347,9.447,9.525,
     * 8.526,8.664,8.810,8.973,9.122,9.259,9.369,9.455,
     * 8.264,8.421,8.582,8.761,8.928,9.085,9.213,9.318,
     * 7.655,7.816,8.005,8.216,8.418,8.611,8.780,8.934,
     * 7.451,7.520,7.645,7.839,8.040,8.245,8.432,8.611,
     * 7.391,7.419,7.475,7.563,7.658,7.760,7.882,8.026,
     * 6.572,6.742,6.922,7.112,7.279,7.415,7.516,7.592,
     * 0.000,0.000,0.000,6.295,6.655,6.937,7.138,7.282,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,2,L,3),I=9,16),L=1,NZEN)/
     * 8.502,8.583,8.658,8.748,8.826,8.901,8.957,9.012,
     * 8.424,8.515,8.595,8.691,8.774,8.851,8.912,8.972,
     * 8.297,8.400,8.492,8.597,8.688,8.773,8.840,8.909,
     * 8.024,8.154,8.268,8.399,8.505,8.607,8.687,8.774,
     * 7.647,7.680,7.756,7.893,8.035,8.172,8.288,8.417,
     * 7.645,7.637,7.634,7.676,7.749,7.854,7.978,8.125,
     * 7.669,7.654,7.640,7.651,7.658,7.661,7.674,7.729,
     * 6.930,7.059,7.174,7.294,7.389,7.458,7.520,7.608,
     * 0.000,0.000,6.316,6.654,6.920,7.115,7.265,7.433,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,2,L,3),I=9,16),L=1,NZEN)/
     * 8.503,8.589,8.670,8.764,8.843,8.919,8.977,9.033,
     * 8.417,8.514,8.599,8.700,8.786,8.866,8.927,8.988,
     * 8.283,8.395,8.492,8.601,8.696,8.785,8.855,8.924,
     * 7.992,8.129,8.249,8.387,8.500,8.604,8.690,8.779,
     * 7.589,7.625,7.704,7.845,7.991,8.133,8.256,8.386,
     * 7.574,7.573,7.578,7.628,7.705,7.814,7.939,8.086,
     * 7.589,7.582,7.576,7.596,7.611,7.622,7.639,7.685,
     * 6.795,6.931,7.058,7.191,7.299,7.382,7.454,7.543,
     * 0.000,0.000,6.144,6.503,6.785,6.998,7.159,7.329,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,2,L,3),I=9,16),L=1,NZEN)/
     * 8.509,8.603,8.693,8.796,8.886,8.970,9.036,9.098,
     * 8.423,8.525,8.622,8.730,8.825,8.913,8.984,9.051,
     * 8.286,8.403,8.507,8.626,8.730,8.826,8.904,8.980,
     * 8.007,8.147,8.276,8.415,8.537,8.648,8.741,8.836,
     * 7.560,7.615,7.717,7.876,8.032,8.180,8.309,8.438,
     * 7.529,7.533,7.552,7.617,7.709,7.829,7.961,8.107,
     * 7.529,7.527,7.531,7.562,7.590,7.612,7.637,7.692,
     * 6.717,6.860,7.000,7.145,7.266,7.359,7.434,7.516,
     * 0.000,0.000,0.000,0.000,6.724,6.951,7.119,7.280,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,2,L,3),I=9,16),L=1,NZEN)/
     * 8.556,8.662,8.769,8.890,8.998,9.099,9.179,9.248,
     * 8.470,8.582,8.695,8.820,8.934,9.039,9.124,9.201,
     * 8.336,8.459,8.578,8.714,8.836,8.948,9.042,9.128,
     * 8.066,8.209,8.347,8.501,8.639,8.766,8.875,8.978,
     * 7.578,7.662,7.794,7.970,8.140,8.297,8.437,8.576,
     * 7.511,7.526,7.571,7.673,7.803,7.953,8.100,8.252,
     * 7.501,7.506,7.526,7.573,7.613,7.654,7.696,7.767,
     * 6.696,6.844,6.993,7.149,7.280,7.381,7.457,7.527,
     * 0.000,0.000,6.016,6.398,6.713,6.955,7.128,7.274,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,2,L,3),I=9,16),L=1,NZEN)/
     * 8.631,8.749,8.878,9.016,9.140,9.256,9.342,9.418,
     * 8.542,8.668,8.800,8.944,9.074,9.193,9.289,9.366,
     * 8.409,8.545,8.682,8.834,8.973,9.102,9.206,9.296,
     * 8.152,8.298,8.453,8.620,8.775,8.921,9.040,9.149,
     * 7.584,7.714,7.883,8.078,8.270,8.446,8.599,8.749,
     * 7.448,7.492,7.582,7.734,7.907,8.090,8.256,8.423,
     * 7.403,7.425,7.470,7.540,7.613,7.684,7.767,7.877,
     * 6.583,6.744,6.915,7.093,7.248,7.369,7.459,7.526,
     * 0.000,0.000,0.000,6.292,6.636,6.904,7.094,7.234,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,3,L,3),I=9,16),L=1,NZEN)/
     * 8.470,8.556,8.635,8.727,8.807,8.882,8.939,8.992,
     * 8.391,8.488,8.572,8.670,8.754,8.833,8.893,8.951,
     * 8.262,8.370,8.467,8.576,8.667,8.753,8.819,8.888,
     * 7.986,8.121,8.240,8.371,8.484,8.585,8.667,8.752,
     * 7.636,7.665,7.732,7.867,8.008,8.146,8.266,8.393,
     * 7.635,7.628,7.626,7.666,7.731,7.832,7.954,8.101,
     * 7.659,7.647,7.634,7.645,7.651,7.654,7.666,7.717,
     * 6.906,7.037,7.154,7.275,7.372,7.444,7.507,7.595,
     * 0.000,0.000,6.293,6.633,6.900,7.096,7.247,7.417,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,3,L,3),I=9,16),L=1,NZEN)/
     * 8.464,8.551,8.635,8.730,8.811,8.887,8.942,8.995,
     * 8.384,8.482,8.569,8.669,8.756,8.835,8.895,8.953,
     * 8.247,8.361,8.458,8.570,8.664,8.751,8.818,8.885,
     * 7.962,8.098,8.220,8.356,8.471,8.573,8.657,8.741,
     * 7.596,7.627,7.697,7.831,7.975,8.114,8.234,8.362,
     * 7.588,7.587,7.589,7.629,7.694,7.790,7.908,8.050,
     * 7.606,7.597,7.589,7.606,7.618,7.626,7.637,7.682,
     * 6.824,6.959,7.084,7.213,7.317,7.394,7.462,7.549,
     * 0.000,0.000,6.184,6.534,6.811,7.019,7.176,7.344,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,3,L,3),I=9,16),L=1,NZEN)/
     * 8.463,8.556,8.649,8.749,8.837,8.918,8.978,9.035,
     * 8.379,8.480,8.575,8.684,8.776,8.861,8.926,8.988,
     * 8.244,8.360,8.465,8.583,8.682,8.774,8.847,8.918,
     * 7.960,8.098,8.224,8.363,8.481,8.589,8.676,8.764,
     * 7.578,7.619,7.699,7.838,7.982,8.122,8.241,8.367,
     * 7.561,7.566,7.578,7.630,7.706,7.809,7.928,8.066,
     * 7.580,7.576,7.575,7.599,7.617,7.628,7.643,7.682,
     * 6.783,6.922,7.054,7.189,7.300,7.382,7.449,7.528,
     * 0.000,0.000,6.137,6.493,6.779,6.992,7.150,7.308,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,3,L,3),I=9,16),L=1,NZEN)/
     * 8.473,8.575,8.680,8.794,8.892,8.983,9.050,9.114,
     * 8.390,8.497,8.605,8.724,8.829,8.923,8.996,9.065,
     * 8.256,8.374,8.492,8.619,8.732,8.833,8.913,8.990,
     * 7.991,8.130,8.264,8.409,8.536,8.652,8.745,8.837,
     * 7.601,7.655,7.751,7.898,8.048,8.190,8.309,8.435,
     * 7.574,7.583,7.610,7.675,7.762,7.873,7.989,8.123,
     * 7.579,7.580,7.591,7.623,7.649,7.668,7.686,7.727,
     * 6.787,6.932,7.072,7.215,7.332,7.418,7.482,7.545,
     * 0.000,0.000,6.122,6.485,6.781,7.004,7.165,7.308,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,3,L,3),I=9,16),L=1,NZEN)/
     * 8.484,8.597,8.719,8.846,8.958,9.058,9.133,9.201,
     * 8.400,8.518,8.643,8.775,8.893,8.996,9.075,9.149,
     * 8.269,8.397,8.527,8.667,8.792,8.903,8.990,9.071,
     * 8.006,8.157,8.295,8.454,8.593,8.719,8.820,8.917,
     * 7.569,7.651,7.776,7.942,8.102,8.253,8.380,8.510,
     * 7.511,7.538,7.590,7.684,7.798,7.933,8.058,8.195,
     * 7.503,7.517,7.548,7.599,7.642,7.679,7.708,7.760,
     * 6.689,6.847,7.005,7.166,7.301,7.402,7.472,7.529,
     * 0.000,0.000,0.000,0.000,6.702,6.951,7.126,7.265,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,4,L,3),I=9,16),L=1,NZEN)/
     * 8.450,8.541,8.620,8.715,8.797,8.871,8.928,8.979,
     * 8.373,8.473,8.558,8.659,8.744,8.823,8.882,8.938,
     * 8.241,8.356,8.451,8.564,8.657,8.743,8.810,8.874,
     * 7.965,8.103,8.225,8.361,8.474,8.575,8.656,8.739,
     * 7.629,7.655,7.721,7.854,7.997,8.135,8.254,8.382,
     * 7.630,7.623,7.622,7.660,7.724,7.823,7.945,8.091,
     * 7.654,7.643,7.631,7.641,7.649,7.650,7.661,7.711,
     * 6.894,7.025,7.145,7.267,7.365,7.437,7.501,7.589,
     * 0.000,0.000,6.285,6.625,6.891,7.089,7.240,7.409,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,4,L,3),I=9,16),L=1,NZEN)/
     * 8.432,8.523,8.604,8.700,8.781,8.856,8.910,8.960,
     * 8.352,8.451,8.539,8.640,8.726,8.804,8.863,8.917,
     * 8.220,8.331,8.432,8.543,8.636,8.722,8.787,8.850,
     * 7.946,8.082,8.203,8.339,8.449,8.550,8.630,8.710,
     * 7.611,7.639,7.705,7.834,7.972,8.107,8.223,8.348,
     * 7.612,7.609,7.609,7.648,7.712,7.806,7.922,8.063,
     * 7.635,7.628,7.616,7.630,7.638,7.639,7.648,7.693,
     * 6.881,7.014,7.133,7.256,7.354,7.425,7.487,7.572,
     * 0.000,0.000,6.272,6.612,6.878,7.075,7.226,7.392,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,4,L,3),I=9,16),L=1,NZEN)/
     * 8.404,8.502,8.593,8.695,8.781,8.858,8.914,8.965,
     * 8.320,8.427,8.523,8.629,8.720,8.803,8.863,8.918,
     * 8.186,8.304,8.411,8.527,8.625,8.713,8.782,8.845,
     * 7.918,8.052,8.179,8.314,8.431,8.535,8.615,8.696,
     * 7.609,7.640,7.700,7.821,7.953,8.083,8.198,8.317,
     * 7.612,7.612,7.617,7.656,7.711,7.793,7.895,8.024,
     * 7.638,7.632,7.627,7.642,7.649,7.652,7.656,7.688,
     * 6.879,7.014,7.136,7.260,7.359,7.429,7.488,7.564,
     * 0.000,0.000,6.258,6.597,6.866,7.063,7.213,7.370,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,4,L,3),I=9,16),L=1,NZEN)/
     * 8.351,8.455,8.562,8.674,8.771,8.855,8.913,8.964,
     * 8.266,8.381,8.491,8.607,8.709,8.795,8.859,8.915,
     * 8.137,8.261,8.377,8.502,8.608,8.703,8.772,8.837,
     * 7.890,8.020,8.149,8.290,8.410,8.514,8.595,8.674,
     * 7.655,7.682,7.733,7.831,7.944,8.060,8.164,8.272,
     * 7.665,7.671,7.681,7.717,7.757,7.811,7.884,7.986,
     * 7.693,7.693,7.694,7.710,7.720,7.717,7.710,7.722,
     * 6.902,7.045,7.175,7.304,7.405,7.475,7.527,7.588,
     * 0.000,0.000,6.217,6.587,6.868,7.074,7.222,7.370,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,4,L,3),I=9,16),L=1,NZEN)/
     * 8.276,8.393,8.517,8.646,8.757,8.848,8.912,8.965,
     * 8.189,8.317,8.441,8.574,8.687,8.785,8.852,8.911,
     * 8.059,8.193,8.325,8.465,8.585,8.686,8.763,8.829,
     * 7.821,7.958,8.098,8.254,8.381,8.496,8.583,8.662,
     * 7.641,7.681,7.741,7.839,7.946,8.052,8.148,8.244,
     * 7.649,7.673,7.702,7.748,7.793,7.836,7.889,7.969,
     * 7.673,7.688,7.707,7.737,7.756,7.759,7.751,7.751,
     * 6.827,6.990,7.142,7.288,7.405,7.487,7.541,7.593,
     * 0.000,0.000,6.131,6.467,6.797,7.030,7.191,7.334,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,5,L,3),I=9,16),L=1,NZEN)/
     * 8.470,8.556,8.635,8.726,8.806,8.879,8.934,8.985,
     * 8.394,8.488,8.572,8.671,8.754,8.832,8.890,8.946,
     * 8.264,8.373,8.470,8.578,8.668,8.752,8.818,8.884,
     * 7.991,8.126,8.246,8.377,8.488,8.587,8.668,8.751,
     * 7.636,7.667,7.738,7.874,8.017,8.155,8.273,8.399,
     * 7.633,7.628,7.626,7.668,7.736,7.841,7.964,8.110,
     * 7.659,7.647,7.634,7.643,7.651,7.654,7.664,7.718,
     * 6.911,7.042,7.159,7.280,7.376,7.446,7.509,7.596,
     * 0.000,0.000,6.304,6.643,6.908,7.103,7.254,7.420,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,5,L,3),I=9,16),L=1,NZEN)/
     * 8.438,8.523,8.602,8.695,8.773,8.846,8.898,8.946,
     * 8.361,8.457,8.543,8.640,8.721,8.798,8.853,8.905,
     * 8.235,8.344,8.439,8.547,8.637,8.719,8.781,8.840,
     * 7.972,8.105,8.224,8.354,8.459,8.557,8.633,8.710,
     * 7.637,7.669,7.741,7.874,8.009,8.140,8.252,8.373,
     * 7.643,7.638,7.637,7.677,7.744,7.845,7.962,8.102,
     * 7.672,7.661,7.648,7.656,7.659,7.658,7.667,7.718,
     * 6.959,7.090,7.204,7.320,7.409,7.472,7.528,7.609,
     * 0.000,6.031,6.375,6.705,6.962,7.149,7.291,7.453,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,5,L,3),I=9,16),L=1,NZEN)/
     * 8.360,8.457,8.550,8.652,8.737,8.814,8.867,8.912,
     * 8.282,8.386,8.483,8.588,8.679,8.758,8.815,8.867,
     * 8.148,8.267,8.373,8.488,8.587,8.671,8.735,8.794,
     * 7.890,8.023,8.147,8.283,8.398,8.497,8.576,8.650,
     * 7.638,7.662,7.712,7.821,7.945,8.069,8.178,8.292,
     * 7.654,7.654,7.656,7.686,7.731,7.801,7.896,8.020,
     * 7.694,7.689,7.678,7.686,7.688,7.682,7.680,7.709,
     * 6.976,7.108,7.223,7.338,7.426,7.486,7.538,7.613,
     * 0.000,6.048,6.384,6.710,6.964,7.149,7.289,7.446,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,5,L,3),I=9,16),L=1,NZEN)/
     * 8.218,8.334,8.446,8.567,8.668,8.754,8.810,8.854,
     * 8.138,8.259,8.375,8.500,8.604,8.694,8.755,8.804,
     * 8.011,8.137,8.255,8.386,8.496,8.590,8.657,8.717,
     * 7.800,7.912,8.032,8.169,8.289,8.391,8.470,8.541,
     * 7.699,7.715,7.740,7.800,7.875,7.962,8.049,8.149,
     * 7.725,7.731,7.736,7.759,7.777,7.796,7.830,7.904,
     * 7.776,7.774,7.770,7.779,7.777,7.764,7.748,7.748,
     * 6.993,7.133,7.255,7.373,7.464,7.526,7.575,7.641,
     * 0.000,6.021,6.323,6.686,6.953,7.144,7.286,7.441,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,5,L,3),I=9,16),L=1,NZEN)/
     * 8.038,8.176,8.315,8.462,8.586,8.688,8.755,8.804,
     * 7.959,8.097,8.238,8.387,8.514,8.620,8.693,8.749,
     * 7.835,7.978,8.119,8.275,8.406,8.516,8.597,8.663,
     * 7.700,7.804,7.923,8.063,8.188,8.298,8.378,8.455,
     * 7.711,7.747,7.783,7.835,7.884,7.928,7.977,8.044,
     * 7.742,7.770,7.795,7.830,7.848,7.850,7.846,7.865,
     * 7.783,7.799,7.812,7.834,7.841,7.829,7.809,7.799,
     * 6.909,7.074,7.219,7.355,7.462,7.537,7.592,7.656,
     * 0.000,0.000,6.182,6.512,6.852,7.080,7.242,7.404,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,6,L,3),I=9,16),L=1,NZEN)/
     * 8.510,8.589,8.663,8.751,8.827,8.899,8.953,9.004,
     * 8.437,8.525,8.602,8.697,8.776,8.851,8.909,8.966,
     * 8.312,8.415,8.504,8.606,8.694,8.775,8.839,8.904,
     * 8.043,8.172,8.284,8.411,8.516,8.614,8.693,8.773,
     * 7.652,7.691,7.773,7.915,8.056,8.190,8.306,8.428,
     * 7.645,7.637,7.636,7.685,7.762,7.874,7.999,8.146,
     * 7.669,7.654,7.640,7.651,7.656,7.659,7.674,7.736,
     * 6.944,7.073,7.189,7.306,7.399,7.465,7.524,7.608,
     * 0.000,0.000,6.339,6.676,6.938,7.132,7.278,7.443,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,6,L,3),I=9,16),L=1,NZEN)/
     * 8.469,8.547,8.623,8.710,8.785,8.856,8.907,8.953,
     * 8.391,8.482,8.561,8.656,8.736,8.809,8.862,8.913,
     * 8.271,8.375,8.467,8.569,8.655,8.733,8.793,8.852,
     * 8.019,8.149,8.261,8.386,8.487,8.580,8.653,8.728,
     * 7.666,7.706,7.788,7.924,8.059,8.186,8.293,8.411,
     * 7.670,7.665,7.663,7.709,7.785,7.892,8.011,8.151,
     * 7.705,7.690,7.675,7.681,7.680,7.680,7.690,7.753,
     * 7.034,7.161,7.271,7.380,7.462,7.519,7.568,7.645,
     * 0.000,6.129,6.468,6.791,7.039,7.218,7.353,7.512,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.049,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,6,L,3),I=9,16),L=1,NZEN)/
     * 8.347,8.442,8.535,8.636,8.721,8.797,8.849,8.891,
     * 8.270,8.375,8.470,8.576,8.665,8.744,8.799,8.846,
     * 8.141,8.258,8.366,8.478,8.575,8.661,8.721,8.778,
     * 7.892,8.024,8.148,8.284,8.396,8.494,8.569,8.643,
     * 7.665,7.687,7.737,7.841,7.963,8.084,8.191,8.302,
     * 7.690,7.689,7.687,7.716,7.759,7.829,7.924,8.052,
     * 7.740,7.732,7.718,7.722,7.719,7.708,7.703,7.734,
     * 7.060,7.189,7.298,7.405,7.484,7.538,7.583,7.657,
     * 0.000,6.167,6.496,6.811,7.053,7.228,7.361,7.517,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.040,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,6,L,3),I=9,16),L=1,NZEN)/
     * 8.133,8.257,8.375,8.504,8.612,8.702,8.760,8.803,
     * 8.051,8.180,8.299,8.431,8.542,8.637,8.699,8.750,
     * 7.928,8.056,8.178,8.314,8.427,8.525,8.596,8.655,
     * 7.743,7.846,7.961,8.099,8.220,8.328,8.407,8.484,
     * 7.712,7.727,7.747,7.795,7.854,7.927,8.006,8.104,
     * 7.756,7.764,7.769,7.790,7.801,7.810,7.828,7.886,
     * 7.815,7.816,7.812,7.819,7.815,7.800,7.780,7.780,
     * 7.057,7.197,7.314,7.427,7.514,7.572,7.619,7.690,
     * 0.000,6.185,6.498,6.808,7.048,7.224,7.356,7.511,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.004,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,6,L,3),I=9,16),L=1,NZEN)/
     * 7.872,8.026,8.178,8.344,8.484,8.599,8.677,8.732,
     * 7.797,7.951,8.105,8.270,8.411,8.532,8.614,8.675,
     * 7.697,7.833,7.976,8.138,8.279,8.398,8.486,8.558,
     * 7.652,7.730,7.825,7.950,8.067,8.172,8.257,8.336,
     * 7.724,7.761,7.793,7.839,7.873,7.897,7.922,7.971,
     * 7.756,7.784,7.808,7.845,7.865,7.867,7.861,7.870,
     * 7.800,7.821,7.837,7.864,7.874,7.869,7.853,7.850,
     * 6.946,7.108,7.249,7.383,7.489,7.567,7.629,7.708,
     * 0.000,6.110,6.431,6.749,7.002,7.192,7.337,7.497,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.020,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,7,L,3),I=9,16),L=1,NZEN)/
     * 8.535,8.612,8.682,8.768,8.842,8.913,8.967,9.019,
     * 8.464,8.548,8.624,8.715,8.792,8.866,8.924,8.981,
     * 8.340,8.439,8.524,8.625,8.711,8.790,8.854,8.919,
     * 8.076,8.202,8.311,8.433,8.536,8.631,8.709,8.792,
     * 7.663,7.710,7.798,7.942,8.081,8.214,8.327,8.450,
     * 7.650,7.643,7.646,7.697,7.781,7.898,8.023,8.168,
     * 7.676,7.661,7.646,7.655,7.660,7.666,7.681,7.747,
     * 6.966,7.094,7.207,7.323,7.413,7.478,7.535,7.617,
     * 0.000,6.015,6.362,6.698,6.959,7.150,7.295,7.457,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,7,L,3),I=9,16),L=1,NZEN)/
     * 8.495,8.571,8.645,8.729,8.803,8.873,8.924,8.971,
     * 8.417,8.507,8.585,8.675,8.752,8.825,8.880,8.931,
     * 8.300,8.403,8.490,8.589,8.673,8.751,8.811,8.870,
     * 8.053,8.181,8.292,8.409,8.510,8.602,8.675,8.750,
     * 7.682,7.728,7.817,7.955,8.089,8.215,8.321,8.439,
     * 7.681,7.675,7.677,7.730,7.810,7.923,8.042,8.182,
     * 7.718,7.705,7.688,7.693,7.691,7.692,7.704,7.774,
     * 7.066,7.193,7.300,7.408,7.487,7.540,7.587,7.664,
     * 0.000,6.169,6.506,6.827,7.071,7.247,7.380,7.537,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.091,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,7,L,3),I=9,16),L=1,NZEN)/
     * 8.363,8.459,8.548,8.650,8.734,8.811,8.862,8.905,
     * 8.286,8.393,8.487,8.592,8.681,8.759,8.815,8.862,
     * 8.162,8.281,8.383,8.499,8.594,8.678,8.740,8.797,
     * 7.917,8.050,8.173,8.307,8.418,8.516,8.591,8.663,
     * 7.678,7.703,7.756,7.868,7.991,8.114,8.222,8.334,
     * 7.705,7.705,7.702,7.731,7.779,7.857,7.956,8.086,
     * 7.760,7.748,7.734,7.736,7.732,7.721,7.717,7.752,
     * 7.098,7.226,7.332,7.436,7.513,7.564,7.608,7.682,
     * 0.000,6.217,6.544,6.855,7.094,7.265,7.395,7.551,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.089,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,7,L,3),I=9,16),L=1,NZEN)/
     * 8.141,8.266,8.388,8.518,8.627,8.716,8.776,8.820,
     * 8.060,8.189,8.316,8.450,8.559,8.653,8.718,8.768,
     * 7.934,8.067,8.192,8.330,8.446,8.545,8.617,8.677,
     * 7.747,7.854,7.973,8.116,8.241,8.350,8.435,8.510,
     * 7.717,7.733,7.755,7.806,7.870,7.950,8.035,8.139,
     * 7.766,7.775,7.779,7.800,7.813,7.825,7.847,7.913,
     * 7.831,7.831,7.827,7.835,7.828,7.813,7.793,7.795,
     * 7.095,7.234,7.349,7.459,7.543,7.599,7.644,7.717,
     * 0.000,6.234,6.549,6.856,7.092,7.264,7.395,7.550,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.055,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,7,L,3),I=9,16),L=1,NZEN)/
     * 7.871,8.030,8.189,8.357,8.499,8.615,8.693,8.749,
     * 7.797,7.958,8.117,8.285,8.429,8.551,8.634,8.696,
     * 7.689,7.833,7.983,8.152,8.297,8.419,8.509,8.583,
     * 7.635,7.715,7.816,7.949,8.076,8.190,8.281,8.367,
     * 7.716,7.752,7.786,7.833,7.871,7.903,7.936,7.996,
     * 7.755,7.782,7.808,7.848,7.870,7.877,7.873,7.888,
     * 7.809,7.831,7.849,7.876,7.888,7.883,7.869,7.868,
     * 6.987,7.148,7.287,7.418,7.523,7.600,7.660,7.741,
     * 0.000,6.176,6.488,6.803,7.052,7.237,7.379,7.541,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.077,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,8,L,3),I=9,16),L=1,NZEN)/
     * 8.530,8.608,8.679,8.766,8.842,8.913,8.967,9.018,
     * 8.458,8.543,8.619,8.712,8.790,8.866,8.922,8.980,
     * 8.332,8.433,8.520,8.622,8.708,8.789,8.854,8.918,
     * 8.068,8.195,8.304,8.428,8.534,8.628,8.707,8.789,
     * 7.661,7.706,7.793,7.936,8.076,8.208,8.323,8.446,
     * 7.650,7.643,7.644,7.695,7.777,7.892,8.017,8.163,
     * 7.674,7.659,7.644,7.655,7.662,7.666,7.681,7.745,
     * 6.962,7.090,7.205,7.321,7.412,7.477,7.534,7.617,
     * 0.000,6.009,6.357,6.693,6.954,7.146,7.291,7.454,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,8,L,3),I=9,16),L=1,NZEN)/
     * 8.495,8.575,8.649,8.738,8.815,8.886,8.937,8.987,
     * 8.423,8.511,8.590,8.684,8.764,8.838,8.894,8.947,
     * 8.302,8.404,8.495,8.595,8.682,8.763,8.824,8.887,
     * 8.050,8.178,8.289,8.413,8.515,8.609,8.684,8.763,
     * 7.673,7.720,7.807,7.948,8.084,8.212,8.324,8.440,
     * 7.672,7.668,7.670,7.722,7.803,7.916,8.036,8.177,
     * 7.706,7.693,7.677,7.686,7.686,7.688,7.700,7.767,
     * 7.041,7.169,7.280,7.390,7.472,7.529,7.578,7.657,
     * 0.000,6.133,6.473,6.798,7.048,7.228,7.364,7.523,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.060,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,8,L,3),I=9,16),L=1,NZEN)/
     * 8.401,8.495,8.586,8.688,8.773,8.849,8.903,8.951,
     * 8.325,8.430,8.524,8.630,8.718,8.799,8.856,8.908,
     * 8.201,8.318,8.422,8.535,8.631,8.717,8.781,8.842,
     * 7.951,8.085,8.209,8.344,8.455,8.554,8.631,8.709,
     * 7.672,7.704,7.767,7.889,8.018,8.146,8.255,8.374,
     * 7.695,7.693,7.693,7.728,7.783,7.868,7.973,8.108,
     * 7.743,7.732,7.720,7.726,7.724,7.718,7.718,7.761,
     * 7.075,7.205,7.314,7.421,7.501,7.556,7.602,7.677,
     * 0.000,6.181,6.512,6.829,7.073,7.250,7.383,7.540,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.072,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,8,L,3),I=9,16),L=1,NZEN)/
     * 8.246,8.366,8.481,8.606,8.709,8.797,8.855,8.902,
     * 8.169,8.293,8.414,8.540,8.646,8.739,8.802,8.855,
     * 8.043,8.174,8.302,8.434,8.548,8.646,8.717,8.779,
     * 7.820,7.946,8.075,8.220,8.345,8.455,8.538,8.616,
     * 7.717,7.735,7.764,7.835,7.926,8.029,8.131,8.241,
     * 7.755,7.760,7.766,7.789,7.810,7.841,7.894,7.989,
     * 7.817,7.816,7.808,7.816,7.813,7.799,7.785,7.793,
     * 7.099,7.235,7.350,7.461,7.545,7.602,7.647,7.717,
     * 0.000,6.214,6.534,6.845,7.086,7.263,7.395,7.548,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.071,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,8,L,3),I=9,16),L=1,NZEN)/
     * 8.051,8.199,8.348,8.502,8.630,8.737,8.808,8.860,
     * 7.973,8.127,8.277,8.436,8.569,8.680,8.755,8.813,
     * 7.850,8.007,8.160,8.325,8.463,8.579,8.663,8.730,
     * 7.706,7.817,7.948,8.102,8.236,8.358,8.449,8.533,
     * 7.714,7.745,7.778,7.837,7.896,7.965,8.041,8.137,
     * 7.761,7.784,7.804,7.839,7.859,7.871,7.884,7.931,
     * 7.815,7.829,7.840,7.862,7.869,7.862,7.847,7.848,
     * 7.020,7.177,7.312,7.440,7.541,7.613,7.668,7.740,
     * 0.000,6.196,6.502,6.815,7.066,7.253,7.394,7.548,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.078,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,9,L,3),I=9,16),L=1,NZEN)/
     * 8.505,8.587,8.661,8.749,8.827,8.901,8.956,9.007,
     * 8.430,8.520,8.601,8.695,8.775,8.852,8.911,8.968,
     * 8.303,8.409,8.499,8.602,8.691,8.774,8.840,8.905,
     * 8.034,8.165,8.280,8.405,8.512,8.610,8.691,8.774,
     * 7.650,7.688,7.766,7.907,8.047,8.183,8.299,8.423,
     * 7.643,7.637,7.636,7.681,7.757,7.868,7.991,8.138,
     * 7.669,7.654,7.640,7.651,7.656,7.661,7.675,7.733,
     * 6.939,7.068,7.184,7.303,7.396,7.464,7.524,7.608,
     * 0.000,0.000,6.331,6.669,6.932,7.126,7.274,7.439,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,9,L,3),I=9,16),L=1,NZEN)/
     * 8.482,8.568,8.646,8.738,8.818,8.894,8.949,9.001,
     * 8.408,8.504,8.586,8.685,8.769,8.845,8.905,8.962,
     * 8.282,8.390,8.484,8.591,8.684,8.767,8.833,8.898,
     * 8.017,8.151,8.269,8.397,8.504,8.604,8.683,8.767,
     * 7.647,7.686,7.767,7.909,8.048,8.182,8.300,8.422,
     * 7.647,7.641,7.644,7.691,7.768,7.879,8.000,8.145,
     * 7.675,7.664,7.652,7.662,7.669,7.673,7.685,7.743,
     * 6.968,7.099,7.215,7.332,7.423,7.488,7.545,7.628,
     * 0.000,6.032,6.379,6.713,6.972,7.163,7.307,7.470,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,9,L,3),I=9,16),L=1,NZEN)/
     * 8.446,8.543,8.632,8.733,8.822,8.902,8.960,9.015,
     * 8.373,8.473,8.567,8.674,8.766,8.849,8.913,8.973,
     * 8.243,8.359,8.462,8.578,8.677,8.767,8.836,8.905,
     * 7.980,8.117,8.242,8.379,8.493,8.597,8.679,8.762,
     * 7.650,7.690,7.763,7.896,8.034,8.168,8.284,8.406,
     * 7.660,7.661,7.665,7.708,7.776,7.873,7.986,8.126,
     * 7.698,7.689,7.680,7.692,7.698,7.699,7.707,7.752,
     * 6.997,7.129,7.245,7.362,7.452,7.515,7.569,7.647,
     * 0.000,6.066,6.408,6.738,6.994,7.183,7.325,7.483,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,9,L,3),I=9,16),L=1,NZEN)/
     * 8.398,8.508,8.615,8.734,8.834,8.923,8.988,9.045,
     * 8.323,8.440,8.552,8.675,8.779,8.871,8.940,9.003,
     * 8.198,8.326,8.447,8.574,8.687,8.785,8.861,8.931,
     * 7.956,8.095,8.227,8.373,8.500,8.610,8.698,8.785,
     * 7.706,7.736,7.797,7.911,8.040,8.169,8.283,8.401,
     * 7.722,7.726,7.734,7.773,7.824,7.898,7.994,8.118,
     * 7.765,7.762,7.757,7.769,7.776,7.773,7.773,7.799,
     * 7.052,7.189,7.310,7.427,7.517,7.580,7.629,7.695,
     * 0.000,6.119,6.455,6.783,7.040,7.230,7.370,7.518,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.040,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,9,L,3),I=9,16),L=1,NZEN)/
     * 8.324,8.452,8.584,8.723,8.842,8.944,9.018,9.081,
     * 8.250,8.386,8.519,8.662,8.785,8.891,8.968,9.037,
     * 8.130,8.277,8.418,8.566,8.695,8.808,8.891,8.968,
     * 7.902,8.054,8.206,8.369,8.509,8.630,8.726,8.817,
     * 7.702,7.745,7.815,7.932,8.061,8.191,8.301,8.418,
     * 7.720,7.737,7.761,7.807,7.862,7.931,8.020,8.135,
     * 7.765,7.773,7.783,7.806,7.822,7.824,7.821,7.839,
     * 7.017,7.169,7.305,7.435,7.538,7.610,7.662,7.723,
     * 0.000,6.069,6.381,6.746,7.026,7.231,7.379,7.525,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.058,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,10,L,3),I=9,16),L=1,NZEN)/
     * 8.487,8.571,8.648,8.741,8.820,8.893,8.949,9.002,
     * 8.411,8.503,8.586,8.683,8.768,8.845,8.904,8.962,
     * 8.281,8.390,8.482,8.589,8.680,8.764,8.832,8.898,
     * 8.009,8.142,8.259,8.390,8.498,8.600,8.680,8.764,
     * 7.643,7.674,7.749,7.887,8.028,8.166,8.285,8.409,
     * 7.640,7.632,7.632,7.674,7.744,7.850,7.974,8.121,
     * 7.664,7.652,7.638,7.649,7.656,7.659,7.671,7.726,
     * 6.921,7.051,7.169,7.289,7.384,7.454,7.516,7.603,
     * 0.000,0.000,6.312,6.651,6.916,7.112,7.262,7.429,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,10,L,3),I=9,16),L=1,NZEN)/
     * 8.482,8.569,8.653,8.748,8.831,8.908,8.966,9.020,
     * 8.402,8.501,8.589,8.690,8.776,8.857,8.919,8.979,
     * 8.270,8.384,8.481,8.592,8.688,8.776,8.845,8.914,
     * 7.994,8.131,8.256,8.388,8.501,8.604,8.688,8.775,
     * 7.620,7.656,7.734,7.874,8.020,8.160,8.279,8.406,
     * 7.616,7.611,7.616,7.661,7.733,7.838,7.961,8.107,
     * 7.637,7.629,7.621,7.637,7.648,7.654,7.668,7.719,
     * 6.888,7.022,7.144,7.270,7.370,7.443,7.508,7.595,
     * 0.000,0.000,6.273,6.617,6.886,7.087,7.240,7.408,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,10,L,3),I=9,16),L=1,NZEN)/
     * 8.489,8.583,8.675,8.780,8.870,8.955,9.019,9.080,
     * 8.409,8.512,8.608,8.719,8.813,8.902,8.971,9.036,
     * 8.277,8.395,8.500,8.619,8.722,8.818,8.893,8.968,
     * 8.005,8.145,8.274,8.410,8.531,8.641,8.732,8.824,
     * 7.623,7.668,7.754,7.900,8.047,8.189,8.313,8.441,
     * 7.617,7.619,7.630,7.682,7.761,7.868,7.990,8.134,
     * 7.639,7.635,7.632,7.652,7.669,7.679,7.694,7.742,
     * 6.892,7.029,7.155,7.284,7.387,7.463,7.527,7.608,
     * 0.000,0.000,6.274,6.620,6.895,7.100,7.252,7.411,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,10,L,3),I=9,16),L=1,NZEN)/
     * 8.533,8.638,8.745,8.864,8.970,9.068,9.146,9.216,
     * 8.458,8.569,8.679,8.803,8.913,9.015,9.097,9.172,
     * 8.335,8.458,8.575,8.707,8.823,8.933,9.022,9.106,
     * 8.084,8.228,8.360,8.509,8.641,8.762,8.865,8.964,
     * 7.685,7.749,7.858,8.014,8.167,8.314,8.442,8.576,
     * 7.661,7.670,7.698,7.769,7.870,7.997,8.126,8.273,
     * 7.687,7.686,7.693,7.720,7.744,7.761,7.783,7.838,
     * 6.958,7.100,7.232,7.365,7.471,7.549,7.609,7.678,
     * 0.000,0.000,6.281,6.663,6.956,7.170,7.325,7.475,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,10,L,3),I=9,16),L=1,NZEN)/
     * 8.581,8.700,8.828,8.965,9.088,9.203,9.293,9.367,
     * 8.508,8.630,8.762,8.905,9.032,9.150,9.243,9.327,
     * 8.394,8.527,8.662,8.810,8.944,9.068,9.171,9.262,
     * 8.157,8.302,8.454,8.617,8.764,8.902,9.019,9.130,
     * 7.700,7.808,7.958,8.141,8.307,8.467,8.609,8.753,
     * 7.639,7.670,7.732,7.846,7.987,8.139,8.288,8.444,
     * 7.648,7.660,7.686,7.733,7.775,7.816,7.867,7.951,
     * 6.937,7.092,7.240,7.389,7.510,7.600,7.665,7.731,
     * 0.000,0.000,6.257,6.657,6.970,7.204,7.369,7.515,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.076,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,11,L,3),I=9,16),L=1,NZEN)/
     * 8.495,8.579,8.653,8.747,8.825,8.899,8.955,9.008,
     * 8.418,8.511,8.591,8.688,8.771,8.850,8.910,8.969,
     * 8.289,8.396,8.487,8.595,8.685,8.770,8.837,8.906,
     * 8.017,8.148,8.264,8.396,8.502,8.603,8.686,8.771,
     * 7.645,7.678,7.753,7.891,8.033,8.170,8.288,8.415,
     * 7.643,7.634,7.634,7.676,7.747,7.853,7.977,8.124,
     * 7.666,7.654,7.640,7.651,7.658,7.661,7.674,7.728,
     * 6.925,7.054,7.171,7.291,7.386,7.456,7.519,7.606,
     * 0.000,0.000,6.312,6.651,6.917,7.114,7.264,7.432,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,11,L,3),I=9,16),L=1,NZEN)/
     * 8.498,8.587,8.670,8.765,8.848,8.926,8.987,9.043,
     * 8.415,8.513,8.601,8.702,8.790,8.873,8.937,9.000,
     * 8.282,8.395,8.495,8.608,8.703,8.792,8.864,8.936,
     * 7.995,8.133,8.254,8.394,8.509,8.615,8.703,8.792,
     * 7.599,7.636,7.715,7.858,8.006,8.150,8.274,8.404,
     * 7.592,7.590,7.594,7.639,7.709,7.813,7.935,8.083,
     * 7.606,7.601,7.593,7.613,7.627,7.639,7.655,7.705,
     * 6.823,6.959,7.086,7.218,7.325,7.406,7.477,7.567,
     * 0.000,0.000,6.184,6.540,6.819,7.029,7.188,7.358,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,11,L,3),I=9,16),L=1,NZEN)/
     * 8.522,8.616,8.709,8.813,8.906,8.994,9.065,9.130,
     * 8.439,8.542,8.640,8.750,8.847,8.940,9.015,9.085,
     * 8.303,8.422,8.527,8.648,8.754,8.853,8.936,9.016,
     * 8.028,8.171,8.300,8.442,8.565,8.680,8.776,8.873,
     * 7.591,7.643,7.740,7.895,8.050,8.200,8.329,8.463,
     * 7.564,7.569,7.587,7.653,7.747,7.871,8.005,8.154,
     * 7.584,7.583,7.585,7.615,7.640,7.661,7.685,7.735,
     * 6.786,6.930,7.066,7.208,7.325,7.414,7.487,7.571,
     * 0.000,0.000,6.144,6.509,6.802,7.022,7.187,7.348,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,11,L,3),I=9,16),L=1,NZEN)/
     * 8.620,8.726,8.837,8.960,9.074,9.181,9.268,9.340,
     * 8.539,8.652,8.767,8.895,9.014,9.126,9.217,9.296,
     * 8.410,8.535,8.657,8.795,8.919,9.040,9.139,9.230,
     * 8.155,8.297,8.436,8.590,8.732,8.865,8.980,9.089,
     * 7.654,7.751,7.893,8.075,8.245,8.407,8.550,8.696,
     * 7.581,7.600,7.651,7.764,7.906,8.064,8.220,8.380,
     * 7.586,7.589,7.608,7.655,7.695,7.740,7.787,7.875,
     * 6.830,6.980,7.126,7.278,7.405,7.501,7.575,7.648,
     * 0.000,0.000,6.181,6.558,6.864,7.098,7.266,7.415,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,11,L,3),I=9,16),L=1,NZEN)/
     * 8.754,8.877,9.010,9.155,9.287,9.406,9.496,9.571,
     * 8.676,8.805,8.942,9.090,9.225,9.353,9.450,9.528,
     * 8.557,8.694,8.836,8.993,9.137,9.272,9.378,9.467,
     * 8.313,8.468,8.625,8.795,8.957,9.107,9.231,9.340,
     * 7.745,7.903,8.086,8.286,8.479,8.662,8.826,8.982,
     * 7.565,7.628,7.748,7.927,8.117,8.313,8.493,8.672,
     * 7.524,7.549,7.595,7.670,7.751,7.840,7.952,8.097,
     * 6.793,6.954,7.119,7.290,7.437,7.553,7.642,7.717,
     * 0.000,0.000,6.146,6.543,6.878,7.137,7.322,7.469,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,6.044,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,12,L,3),I=9,16),L=1,NZEN)/
     * 8.518,8.597,8.670,8.758,8.836,8.912,8.968,9.022,
     * 8.440,8.530,8.608,8.702,8.783,8.863,8.922,8.983,
     * 8.312,8.415,8.506,8.609,8.699,8.783,8.850,8.920,
     * 8.042,8.171,8.284,8.411,8.518,8.617,8.700,8.784,
     * 7.654,7.690,7.769,7.908,8.050,8.186,8.303,8.430,
     * 7.648,7.641,7.638,7.683,7.758,7.868,7.992,8.140,
     * 7.671,7.659,7.644,7.655,7.662,7.666,7.678,7.737,
     * 6.940,7.069,7.184,7.303,7.397,7.465,7.527,7.614,
     * 0.000,0.000,6.327,6.665,6.930,7.126,7.275,7.442,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,2,12,L,3),I=9,16),L=1,NZEN)/
     * 8.520,8.608,8.689,8.784,8.866,8.942,9.002,9.060,
     * 8.438,8.534,8.619,8.720,8.807,8.889,8.953,9.017,
     * 8.307,8.418,8.514,8.625,8.721,8.809,8.880,8.954,
     * 8.009,8.147,8.268,8.406,8.521,8.629,8.716,8.808,
     * 7.591,7.630,7.713,7.858,8.009,8.153,8.279,8.411,
     * 7.575,7.574,7.580,7.634,7.715,7.828,7.957,8.107,
     * 7.592,7.584,7.580,7.601,7.619,7.632,7.649,7.699,
     * 6.792,6.929,7.058,7.193,7.304,7.389,7.464,7.555,
     * 0.000,0.000,6.148,6.508,6.793,7.007,7.170,7.341,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,3,12,L,3),I=9,16),L=1,NZEN)/
     * 8.541,8.635,8.728,8.831,8.924,9.013,9.085,9.153,
     * 8.455,8.558,8.655,8.767,8.863,8.957,9.034,9.107,
     * 8.319,8.435,8.541,8.663,8.770,8.871,8.954,9.036,
     * 8.038,8.181,8.310,8.454,8.578,8.694,8.794,8.894,
     * 7.563,7.627,7.739,7.906,8.069,8.221,8.356,8.492,
     * 7.530,7.535,7.557,7.630,7.731,7.862,8.002,8.152,
     * 7.528,7.527,7.534,7.569,7.600,7.628,7.660,7.722,
     * 6.714,6.861,7.005,7.155,7.282,7.379,7.459,7.544,
     * 0.000,0.000,6.052,6.431,6.736,6.969,7.141,7.304,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,4,12,L,3),I=9,16),L=1,NZEN)/
     * 8.642,8.752,8.865,8.991,9.109,9.221,9.310,9.385,
     * 8.558,8.672,8.790,8.922,9.045,9.163,9.258,9.337,
     * 8.425,8.551,8.676,8.818,8.947,9.073,9.177,9.266,
     * 8.160,8.304,8.448,8.608,8.753,8.893,9.014,9.125,
     * 7.616,7.728,7.883,8.074,8.253,8.424,8.578,8.728,
     * 7.518,7.543,7.608,7.737,7.897,8.068,8.234,8.398,
     * 7.500,7.507,7.534,7.589,7.643,7.701,7.768,7.871,
     * 6.717,6.869,7.023,7.187,7.326,7.434,7.517,7.593,
     * 0.000,0.000,6.048,6.437,6.761,7.010,7.189,7.339,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,5,12,L,3),I=9,16),L=1,NZEN)/
     * 8.797,8.928,9.063,9.211,9.346,9.467,9.559,9.633,
     * 8.716,8.848,8.990,9.141,9.283,9.412,9.509,9.587,
     * 8.590,8.730,8.877,9.041,9.192,9.329,9.434,9.523,
     * 8.337,8.494,8.656,8.836,9.003,9.159,9.289,9.391,
     * 7.733,7.901,8.095,8.306,8.508,8.702,8.872,9.026,
     * 7.507,7.586,7.723,7.928,8.132,8.341,8.530,8.713,
     * 7.446,7.478,7.536,7.626,7.726,7.837,7.968,8.125,
     * 6.659,6.826,7.002,7.188,7.351,7.483,7.584,7.662,
     * 0.000,0.000,0.000,6.394,6.749,7.026,7.224,7.370,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000,
     * 0.000,0.000,0.000,0.000,0.000,0.000,0.000,0.000/
C
      DATA ((EDEN(I,1,1,L,1),I=17,24),L=1,NZEN)/
     * 8.830,8.851,8.858,8.858,8.841,8.836,8.873,8.930,
     * 8.786,8.815,8.829,8.836,8.824,8.823,8.857,8.910,
     * 8.721,8.757,8.782,8.800,8.797,8.801,8.833,8.883,
     * 8.587,8.639,8.682,8.720,8.736,8.753,8.786,8.836,
     * 8.230,8.319,8.402,8.486,8.550,8.612,8.660,8.721,
     * 7.945,8.055,8.158,8.271,8.368,8.468,8.534,8.613,
     * 7.537,7.608,7.701,7.814,7.934,8.075,8.188,8.311,
     * 7.369,7.418,7.461,7.501,7.570,7.666,7.783,7.924,
     * 7.177,7.283,7.372,7.454,7.554,7.668,7.789,7.925,
     * 0.000,0.000,6.199,6.432,6.638,6.910,7.164,7.412,
     * 0.000,0.000,0.000,0.000,0.000,6.075,6.446,6.805/
C
      DATA ((EDEN(I,2,1,L,1),I=17,24),L=1,NZEN)/
     * 8.862,8.886,8.894,8.896,8.878,8.873,8.920,8.986,
     * 8.816,8.843,8.858,8.865,8.852,8.848,8.890,8.950,
     * 8.751,8.788,8.812,8.829,8.826,8.831,8.867,8.922,
     * 8.603,8.660,8.705,8.745,8.762,8.783,8.819,8.872,
     * 8.204,8.295,8.378,8.464,8.531,8.599,8.654,8.724,
     * 7.905,8.013,8.118,8.227,8.325,8.427,8.504,8.594,
     * 7.502,7.549,7.619,7.715,7.827,7.958,8.090,8.232,
     * 7.317,7.360,7.396,7.432,7.502,7.604,7.730,7.883,
     * 7.081,7.177,7.257,7.334,7.439,7.566,7.706,7.866,
     * 0.000,0.000,0.000,6.181,6.406,6.711,7.003,7.294,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.247,6.647/
C
      DATA ((EDEN(I,3,1,L,1),I=17,24),L=1,NZEN)/
     * 8.946,8.972,8.983,8.986,8.970,8.967,9.018,9.087,
     * 8.895,8.927,8.944,8.951,8.940,8.943,8.988,9.050,
     * 8.821,8.860,8.886,8.901,8.899,8.903,8.946,9.005,
     * 8.680,8.737,8.779,8.813,8.828,8.849,8.894,8.955,
     * 8.277,8.366,8.449,8.530,8.594,8.659,8.724,8.804,
     * 7.944,8.045,8.147,8.251,8.349,8.451,8.541,8.645,
     * 7.529,7.577,7.651,7.751,7.868,8.015,8.158,8.313,
     * 7.315,7.336,7.349,7.366,7.428,7.535,7.677,7.852,
     * 7.054,7.121,7.170,7.223,7.318,7.451,7.614,7.803,
     * 0.000,0.000,0.000,6.113,6.316,6.608,6.922,7.241,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.164,6.589/
C
      DATA ((EDEN(I,4,1,L,1),I=17,24),L=1,NZEN)/
     * 9.080,9.107,9.116,9.117,9.108,9.120,9.171,9.236,
     * 9.028,9.059,9.073,9.079,9.075,9.088,9.135,9.197,
     * 8.951,8.989,9.012,9.025,9.024,9.047,9.093,9.154,
     * 8.800,8.854,8.892,8.920,8.935,8.968,9.020,9.087,
     * 8.400,8.490,8.569,8.642,8.700,8.768,8.845,8.935,
     * 8.087,8.187,8.287,8.389,8.484,8.585,8.685,8.799,
     * 7.616,7.680,7.768,7.886,8.026,8.188,8.337,8.499,
     * 7.317,7.313,7.306,7.313,7.382,7.511,7.688,7.901,
     * 7.050,7.081,7.097,7.129,7.224,7.375,7.570,7.798,
     * 0.000,0.000,0.000,6.113,6.310,6.605,6.940,7.285,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.198,6.656/
C
      DATA ((EDEN(I,5,1,L,1),I=17,24),L=1,NZEN)/
     * 9.241,9.267,9.283,9.295,9.302,9.335,9.391,9.458,
     * 9.189,9.221,9.236,9.252,9.262,9.297,9.349,9.413,
     * 9.116,9.151,9.171,9.191,9.208,9.248,9.300,9.364,
     * 8.972,9.019,9.054,9.084,9.111,9.163,9.222,9.293,
     * 8.584,8.669,8.739,8.807,8.867,8.953,9.039,9.137,
     * 8.277,8.379,8.477,8.575,8.668,8.785,8.893,9.014,
     * 7.768,7.849,7.953,8.092,8.242,8.417,8.571,8.738,
     * 7.360,7.342,7.324,7.330,7.414,7.571,7.787,8.041,
     * 7.058,7.062,7.053,7.074,7.179,7.359,7.597,7.872,
     * 0.000,0.000,0.000,6.125,6.330,6.646,7.016,7.402,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.270,6.780/
C
      DATA ((EDEN(I,1,2,L,1),I=17,24),L=1,NZEN)/
     * 8.815,8.835,8.840,8.841,8.824,8.817,8.856,8.915,
     * 8.772,8.799,8.811,8.818,8.806,8.802,8.839,8.894,
     * 8.707,8.741,8.764,8.781,8.778,8.781,8.816,8.868,
     * 8.571,8.621,8.662,8.701,8.715,8.734,8.769,8.821,
     * 8.214,8.301,8.380,8.462,8.525,8.587,8.637,8.702,
     * 7.928,8.036,8.139,8.245,8.342,8.440,8.510,8.592,
     * 7.528,7.595,7.684,7.792,7.907,8.044,8.161,8.287,
     * 7.363,7.412,7.453,7.491,7.558,7.654,7.772,7.913,
     * 7.167,7.273,7.362,7.442,7.541,7.655,7.777,7.914,
     * 0.000,0.000,6.185,6.418,6.622,6.891,7.148,7.398,
     * 0.000,0.000,0.000,0.000,0.000,6.064,6.436,6.795/
C
      DATA ((EDEN(I,2,2,L,1),I=17,24),L=1,NZEN)/
     * 8.838,8.858,8.866,8.867,8.851,8.846,8.894,8.961,
     * 8.790,8.816,8.828,8.836,8.825,8.825,8.867,8.928,
     * 8.725,8.762,8.786,8.804,8.803,8.809,8.847,8.903,
     * 8.578,8.631,8.672,8.711,8.729,8.751,8.790,8.846,
     * 8.186,8.273,8.352,8.434,8.502,8.567,8.626,8.699,
     * 7.892,7.997,8.096,8.200,8.297,8.397,8.477,8.571,
     * 7.495,7.542,7.609,7.703,7.811,7.944,8.077,8.221,
     * 7.316,7.358,7.392,7.425,7.493,7.594,7.720,7.874,
     * 7.084,7.182,7.263,7.341,7.446,7.573,7.714,7.873,
     * 0.000,0.000,0.000,6.209,6.426,6.727,7.015,7.302,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.278,6.673/
C
      DATA ((EDEN(I,3,2,L,1),I=17,24),L=1,NZEN)/
     * 8.906,8.933,8.944,8.947,8.937,8.936,8.988,9.058,
     * 8.857,8.887,8.903,8.913,8.905,8.912,8.959,9.023,
     * 8.783,8.821,8.845,8.862,8.862,8.874,8.920,8.982,
     * 8.639,8.693,8.735,8.771,8.789,8.816,8.864,8.927,
     * 8.245,8.329,8.409,8.488,8.553,8.621,8.689,8.771,
     * 7.920,8.019,8.118,8.219,8.315,8.419,8.512,8.618,
     * 7.526,7.570,7.638,7.733,7.848,7.987,8.132,8.288,
     * 7.319,7.342,7.358,7.377,7.440,7.547,7.689,7.862,
     * 7.068,7.139,7.192,7.248,7.345,7.478,7.637,7.822,
     * 0.000,0.000,0.000,6.138,6.338,6.628,6.940,7.254,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.223,6.641/
C
      DATA ((EDEN(I,4,2,L,1),I=17,24),L=1,NZEN)/
     * 9.024,9.053,9.067,9.072,9.069,9.085,9.138,9.206,
     * 8.971,9.006,9.024,9.035,9.033,9.056,9.106,9.171,
     * 8.895,8.937,8.963,8.981,8.984,9.014,9.064,9.128,
     * 8.743,8.800,8.842,8.877,8.898,8.937,8.992,9.062,
     * 8.347,8.435,8.517,8.596,8.660,8.735,8.814,8.906,
     * 8.039,8.139,8.236,8.342,8.439,8.547,8.650,8.765,
     * 7.599,7.647,7.721,7.827,7.955,8.114,8.269,8.436,
     * 7.339,7.340,7.337,7.347,7.415,7.540,7.710,7.916,
     * 7.082,7.121,7.143,7.179,7.273,7.419,7.606,7.824,
     * 0.000,0.000,0.000,6.159,6.356,6.648,6.977,7.314,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.259,6.706/
C
      DATA ((EDEN(I,5,2,L,1),I=17,24),L=1,NZEN)/
     * 9.160,9.192,9.208,9.227,9.240,9.278,9.338,9.409,
     * 9.107,9.144,9.164,9.187,9.198,9.242,9.299,9.368,
     * 9.032,9.074,9.102,9.126,9.146,9.194,9.252,9.321,
     * 8.883,8.940,8.980,9.019,9.052,9.109,9.174,9.250,
     * 8.491,8.582,8.661,8.740,8.809,8.903,8.992,9.094,
     * 8.182,8.289,8.392,8.502,8.603,8.728,8.841,8.966,
     * 7.710,7.778,7.873,8.003,8.158,8.341,8.501,8.673,
     * 7.371,7.358,7.345,7.354,7.437,7.589,7.796,8.040,
     * 7.090,7.104,7.105,7.132,7.236,7.410,7.638,7.900,
     * 0.000,0.000,6.004,6.168,6.371,6.681,7.043,7.418,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.325,6.821/
C
      DATA ((EDEN(I,1,3,L,1),I=17,24),L=1,NZEN)/
     * 8.794,8.811,8.815,8.814,8.797,8.792,8.834,8.896,
     * 8.752,8.774,8.785,8.792,8.779,8.776,8.815,8.873,
     * 8.686,8.717,8.738,8.754,8.751,8.753,8.790,8.846,
     * 8.549,8.598,8.636,8.670,8.686,8.703,8.742,8.798,
     * 8.194,8.278,8.352,8.429,8.492,8.552,8.607,8.676,
     * 7.905,8.013,8.111,8.213,8.305,8.401,8.476,8.563,
     * 7.517,7.578,7.663,7.766,7.875,8.004,8.126,8.258,
     * 7.354,7.402,7.443,7.478,7.543,7.638,7.755,7.898,
     * 7.154,7.261,7.349,7.428,7.525,7.639,7.761,7.899,
     * 0.000,0.000,6.167,6.399,6.603,6.867,7.125,7.377,
     * 0.000,0.000,0.000,0.000,0.000,6.050,6.422,6.781/
C
      DATA ((EDEN(I,2,3,L,1),I=17,24),L=1,NZEN)/
     * 8.802,8.818,8.823,8.824,8.808,8.804,8.854,8.923,
     * 8.757,8.779,8.792,8.799,8.788,8.789,8.834,8.898,
     * 8.688,8.719,8.740,8.757,8.754,8.761,8.803,8.863,
     * 8.544,8.592,8.629,8.663,8.680,8.701,8.746,8.807,
     * 8.164,8.247,8.321,8.397,8.459,8.525,8.588,8.664,
     * 7.861,7.964,8.058,8.158,8.248,8.345,8.431,8.529,
     * 7.497,7.544,7.612,7.703,7.803,7.929,8.063,8.208,
     * 7.328,7.369,7.403,7.433,7.497,7.594,7.718,7.868,
     * 7.106,7.205,7.286,7.361,7.461,7.583,7.718,7.871,
     * 0.000,0.000,6.056,6.283,6.489,6.769,7.048,7.322,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.341,6.721/
C
      DATA ((EDEN(I,3,3,L,1),I=17,24),L=1,NZEN)/
     * 8.847,8.869,8.878,8.883,8.873,8.877,8.934,9.008,
     * 8.798,8.824,8.839,8.849,8.844,8.853,8.906,8.975,
     * 8.725,8.760,8.784,8.804,8.806,8.821,8.871,8.939,
     * 8.573,8.622,8.661,8.696,8.716,8.743,8.796,8.865,
     * 8.183,8.261,8.332,8.407,8.469,8.538,8.613,8.701,
     * 7.890,7.987,8.077,8.171,8.263,8.361,8.458,8.567,
     * 7.531,7.562,7.615,7.696,7.792,7.917,8.065,8.225,
     * 7.348,7.375,7.395,7.416,7.478,7.580,7.714,7.877,
     * 7.117,7.196,7.257,7.318,7.413,7.538,7.686,7.856,
     * 0.000,0.000,6.034,6.242,6.442,6.720,7.015,7.308,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.328,6.723/
C
      DATA ((EDEN(I,4,3,L,1),I=17,24),L=1,NZEN)/
     * 8.938,8.967,8.982,8.992,8.991,9.010,9.071,9.148,
     * 8.887,8.921,8.941,8.957,8.960,8.981,9.040,9.113,
     * 8.813,8.854,8.881,8.905,8.914,8.942,9.000,9.073,
     * 8.663,8.717,8.762,8.804,8.829,8.869,8.932,9.008,
     * 8.270,8.351,8.425,8.504,8.571,8.648,8.733,8.831,
     * 7.968,8.058,8.149,8.246,8.341,8.448,8.557,8.678,
     * 7.608,7.637,7.694,7.781,7.891,8.031,8.191,8.360,
     * 7.404,7.414,7.417,7.429,7.490,7.601,7.753,7.935,
     * 7.166,7.221,7.259,7.304,7.396,7.530,7.696,7.887,
     * 0.000,0.000,0.000,6.267,6.461,6.743,7.052,7.363,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.394,6.807/
C
      DATA ((EDEN(I,5,3,L,1),I=17,24),L=1,NZEN)/
     * 9.039,9.074,9.094,9.114,9.128,9.166,9.238,9.321,
     * 8.987,9.025,9.051,9.076,9.090,9.136,9.205,9.285,
     * 8.911,8.958,8.989,9.020,9.043,9.092,9.161,9.242,
     * 8.760,8.822,8.869,8.915,8.953,9.012,9.087,9.173,
     * 8.369,8.456,8.537,8.624,8.700,8.796,8.895,9.004,
     * 8.073,8.167,8.261,8.371,8.478,8.604,8.727,8.859,
     * 7.692,7.723,7.783,7.884,8.012,8.180,8.355,8.537,
     * 7.458,7.455,7.450,7.460,7.532,7.665,7.846,8.059,
     * 7.207,7.242,7.261,7.298,7.396,7.550,7.747,7.973,
     * 0.000,0.000,6.121,6.298,6.494,6.789,7.128,7.470,
     * 0.000,0.000,0.000,0.000,0.000,6.045,6.494,6.945/
C
      DATA ((EDEN(I,1,4,L,1),I=17,24),L=1,NZEN)/
     * 8.782,8.795,8.797,8.795,8.777,8.769,8.814,8.879,
     * 8.738,8.759,8.767,8.772,8.759,8.754,8.796,8.857,
     * 8.674,8.702,8.720,8.735,8.730,8.732,8.772,8.829,
     * 8.538,8.583,8.620,8.651,8.665,8.681,8.722,8.780,
     * 8.183,8.264,8.337,8.410,8.469,8.528,8.585,8.656,
     * 7.897,8.003,8.100,8.197,8.285,8.378,8.456,8.545,
     * 7.513,7.573,7.655,7.756,7.860,7.984,8.109,8.243,
     * 7.349,7.397,7.436,7.470,7.532,7.625,7.742,7.884,
     * 7.148,7.254,7.341,7.419,7.514,7.626,7.746,7.882,
     * 0.000,0.000,6.164,6.396,6.599,6.858,7.115,7.365,
     * 0.000,0.000,0.000,0.000,0.000,6.050,6.418,6.773/
C
      DATA ((EDEN(I,2,4,L,1),I=17,24),L=1,NZEN)/
     * 8.767,8.781,8.783,8.781,8.765,8.761,8.813,8.883,
     * 8.721,8.741,8.749,8.753,8.742,8.742,8.790,8.857,
     * 8.653,8.680,8.698,8.710,8.707,8.712,8.759,8.823,
     * 8.515,8.559,8.590,8.621,8.634,8.653,8.702,8.767,
     * 8.156,8.237,8.306,8.375,8.433,8.496,8.562,8.641,
     * 7.881,7.982,8.074,8.166,8.245,8.338,8.424,8.521,
     * 7.514,7.567,7.640,7.734,7.831,7.949,8.080,8.221,
     * 7.358,7.402,7.437,7.467,7.528,7.622,7.740,7.885,
     * 7.159,7.261,7.344,7.418,7.512,7.623,7.745,7.883,
     * 0.000,0.000,6.173,6.403,6.604,6.862,7.120,7.371,
     * 0.000,0.000,0.000,0.000,0.000,6.071,6.438,6.793/
C
      DATA ((EDEN(I,3,4,L,1),I=17,24),L=1,NZEN)/
     * 8.778,8.794,8.798,8.801,8.792,8.799,8.862,8.942,
     * 8.730,8.751,8.761,8.770,8.764,8.774,8.834,8.910,
     * 8.656,8.684,8.700,8.717,8.717,8.732,8.790,8.865,
     * 8.510,8.552,8.582,8.612,8.628,8.655,8.716,8.792,
     * 8.144,8.215,8.278,8.344,8.399,8.460,8.540,8.631,
     * 7.865,7.955,8.038,8.125,8.201,8.288,8.390,8.501,
     * 7.550,7.580,7.632,7.707,7.791,7.896,8.042,8.197,
     * 7.403,7.438,7.464,7.488,7.544,7.635,7.753,7.898,
     * 7.199,7.291,7.364,7.432,7.521,7.631,7.754,7.894,
     * 0.000,0.000,6.185,6.403,6.599,6.859,7.122,7.377,
     * 0.000,0.000,0.000,0.000,0.000,6.111,6.476,6.828/
C
      DATA ((EDEN(I,4,4,L,1),I=17,24),L=1,NZEN)/
     * 8.827,8.847,8.856,8.866,8.866,8.884,8.961,9.053,
     * 8.778,8.802,8.817,8.833,8.837,8.858,8.931,9.019,
     * 8.703,8.735,8.757,8.781,8.792,8.817,8.889,8.974,
     * 8.545,8.588,8.622,8.655,8.679,8.714,8.789,8.877,
     * 8.164,8.237,8.299,8.364,8.423,8.494,8.588,8.693,
     * 7.893,7.970,8.048,8.131,8.208,8.298,8.415,8.540,
     * 7.639,7.647,7.673,7.729,7.804,7.906,8.066,8.233,
     * 7.491,7.515,7.532,7.549,7.600,7.688,7.804,7.946,
     * 7.273,7.352,7.415,7.475,7.561,7.669,7.793,7.933,
     * 0.000,6.026,6.226,6.432,6.623,6.890,7.159,7.419,
     * 0.000,0.000,0.000,0.000,6.058,6.202,6.568,6.920/
C
      DATA ((EDEN(I,5,4,L,1),I=17,24),L=1,NZEN)/
     * 8.872,8.897,8.912,8.931,8.943,8.974,9.069,9.176,
     * 8.823,8.851,8.870,8.896,8.912,8.946,9.037,9.139,
     * 8.748,8.784,8.810,8.842,8.865,8.905,8.995,9.096,
     * 8.597,8.647,8.687,8.733,8.770,8.823,8.917,9.020,
     * 8.203,8.274,8.335,8.408,8.476,8.562,8.676,8.797,
     * 7.947,8.012,8.084,8.170,8.255,8.358,8.494,8.637,
     * 7.740,7.731,7.743,7.788,7.860,7.967,8.146,8.329,
     * 7.586,7.599,7.608,7.623,7.676,7.771,7.897,8.048,
     * 7.350,7.419,7.472,7.529,7.618,7.736,7.873,8.026,
     * 0.000,6.091,6.277,6.472,6.663,6.941,7.225,7.500,
     * 0.000,0.000,0.000,0.000,6.153,6.307,6.685,7.051/
C
      DATA ((EDEN(I,1,5,L,1),I=17,24),L=1,NZEN)/
     * 8.787,8.800,8.802,8.799,8.778,8.768,8.815,8.881,
     * 8.744,8.765,8.772,8.775,8.761,8.755,8.797,8.858,
     * 8.680,8.709,8.726,8.739,8.733,8.732,8.771,8.828,
     * 8.548,8.594,8.626,8.658,8.669,8.684,8.723,8.780,
     * 8.197,8.278,8.350,8.423,8.479,8.534,8.589,8.658,
     * 7.915,8.019,8.114,8.213,8.298,8.388,8.464,8.551,
     * 7.519,7.584,7.671,7.774,7.879,8.001,8.122,8.254,
     * 7.352,7.399,7.437,7.470,7.531,7.623,7.738,7.879,
     * 7.157,7.261,7.345,7.421,7.514,7.622,7.740,7.874,
     * 0.000,0.000,6.185,6.416,6.616,6.873,7.125,7.369,
     * 0.000,0.000,0.000,0.000,0.000,6.068,6.430,6.779/
C
      DATA ((EDEN(I,2,5,L,1),I=17,24),L=1,NZEN)/
     * 8.753,8.764,8.764,8.760,8.743,8.738,8.790,8.861,
     * 8.711,8.726,8.732,8.734,8.720,8.718,8.768,8.837,
     * 8.645,8.669,8.682,8.694,8.688,8.692,8.740,8.806,
     * 8.517,8.555,8.585,8.613,8.623,8.641,8.691,8.757,
     * 8.185,8.262,8.327,8.394,8.445,8.502,8.566,8.642,
     * 7.923,8.022,8.112,8.201,8.277,8.359,8.440,8.532,
     * 7.543,7.608,7.691,7.790,7.888,7.998,8.122,8.254,
     * 7.395,7.440,7.477,7.508,7.567,7.656,7.768,7.905,
     * 7.222,7.326,7.411,7.485,7.572,7.673,7.781,7.902,
     * 0.000,6.053,6.301,6.533,6.731,6.973,7.208,7.432,
     * 0.000,0.000,0.000,0.000,6.024,6.163,6.511,6.842/
C
      DATA ((EDEN(I,3,5,L,1),I=17,24),L=1,NZEN)/
     * 8.725,8.735,8.734,8.733,8.721,8.727,8.796,8.884,
     * 8.680,8.693,8.696,8.700,8.691,8.698,8.764,8.847,
     * 8.606,8.627,8.635,8.644,8.641,8.649,8.714,8.794,
     * 8.469,8.504,8.524,8.549,8.559,8.578,8.645,8.726,
     * 8.129,8.201,8.256,8.314,8.358,8.410,8.492,8.584,
     * 7.872,7.963,8.044,8.122,8.187,8.261,8.359,8.466,
     * 7.579,7.614,7.669,7.745,7.823,7.914,8.053,8.198,
     * 7.463,7.505,7.539,7.566,7.617,7.695,7.793,7.914,
     * 7.290,7.393,7.478,7.551,7.633,7.724,7.818,7.923,
     * 0.000,6.102,6.344,6.573,6.766,7.010,7.239,7.455,
     * 0.000,0.000,0.000,0.000,6.132,6.261,6.590,6.898/
C
      DATA ((EDEN(I,4,5,L,1),I=17,24),L=1,NZEN)/
     * 8.733,8.743,8.743,8.747,8.744,8.760,8.852,8.959,
     * 8.688,8.703,8.708,8.716,8.717,8.736,8.821,8.921,
     * 8.605,8.628,8.641,8.653,8.657,8.676,8.757,8.851,
     * 8.447,8.481,8.505,8.529,8.542,8.567,8.648,8.740,
     * 8.085,8.153,8.211,8.267,8.309,8.362,8.457,8.560,
     * 7.853,7.922,7.996,8.072,8.135,8.204,8.316,8.433,
     * 7.687,7.686,7.700,7.739,7.788,7.859,8.004,8.152,
     * 7.575,7.616,7.649,7.674,7.714,7.772,7.845,7.935,
     * 7.376,7.483,7.570,7.647,7.726,7.804,7.878,7.959,
     * 0.000,6.164,6.390,6.613,6.804,7.055,7.271,7.469,
     * 0.000,0.000,0.000,6.161,6.316,6.440,6.739,7.015/
C
      DATA ((EDEN(I,5,5,L,1),I=17,24),L=1,NZEN)/
     * 8.739,8.751,8.754,8.763,8.769,8.797,8.912,9.039,
     * 8.691,8.711,8.719,8.732,8.740,8.770,8.877,8.994,
     * 8.620,8.647,8.662,8.683,8.697,8.730,8.831,8.943,
     * 8.444,8.485,8.515,8.547,8.570,8.608,8.706,8.813,
     * 8.077,8.139,8.197,8.256,8.309,8.368,8.479,8.593,
     * 7.896,7.937,7.993,8.064,8.130,8.204,8.329,8.457,
     * 7.818,7.806,7.801,7.814,7.844,7.902,8.052,8.201,
     * 7.690,7.734,7.769,7.798,7.837,7.889,7.951,8.026,
     * 7.467,7.578,7.672,7.756,7.838,7.915,7.982,8.050,
     * 6.045,6.256,6.473,6.689,6.882,7.143,7.355,7.546,
     * 0.000,0.000,6.130,6.308,6.463,6.591,6.869,7.121/
C
      DATA ((EDEN(I,1,6,L,1),I=17,24),L=1,NZEN)/
     * 8.805,8.818,8.820,8.816,8.794,8.783,8.830,8.897,
     * 8.763,8.785,8.792,8.794,8.778,8.770,8.811,8.872,
     * 8.700,8.730,8.747,8.759,8.750,8.748,8.785,8.840,
     * 8.570,8.615,8.651,8.680,8.689,8.700,8.737,8.790,
     * 8.227,8.306,8.380,8.453,8.509,8.559,8.611,8.676,
     * 7.947,8.053,8.149,8.248,8.332,8.419,8.490,8.573,
     * 7.535,7.609,7.701,7.809,7.918,8.044,8.160,8.285,
     * 7.360,7.405,7.443,7.475,7.536,7.627,7.742,7.882,
     * 7.173,7.274,7.357,7.430,7.521,7.627,7.742,7.874,
     * 0.000,0.000,6.219,6.450,6.648,6.900,7.146,7.385,
     * 0.000,0.000,0.000,0.000,0.000,6.096,6.452,6.795/
C
      DATA ((EDEN(I,2,6,L,1),I=17,24),L=1,NZEN)/
     * 8.760,8.772,8.769,8.765,8.745,8.740,8.790,8.860,
     * 8.717,8.732,8.736,8.737,8.721,8.717,8.767,8.835,
     * 8.656,8.679,8.691,8.700,8.692,8.692,8.740,8.805,
     * 8.534,8.571,8.599,8.627,8.635,8.648,8.696,8.760,
     * 8.224,8.298,8.362,8.429,8.477,8.528,8.587,8.659,
     * 7.970,8.070,8.158,8.249,8.323,8.401,8.476,8.560,
     * 7.579,7.657,7.753,7.860,7.961,8.078,8.193,8.316,
     * 7.426,7.473,7.512,7.544,7.603,7.690,7.798,7.930,
     * 7.276,7.380,7.466,7.540,7.623,7.716,7.813,7.922,
     * 0.000,6.143,6.396,6.629,6.822,7.056,7.278,7.487,
     * 0.000,0.000,0.000,0.000,6.123,6.255,6.586,6.899/
C
      DATA ((EDEN(I,3,6,L,1),I=17,24),L=1,NZEN)/
     * 8.703,8.710,8.705,8.702,8.688,8.692,8.765,8.856,
     * 8.656,8.668,8.666,8.666,8.653,8.656,8.725,8.811,
     * 8.591,8.608,8.614,8.618,8.612,8.618,8.684,8.766,
     * 8.461,8.493,8.512,8.533,8.538,8.554,8.621,8.702,
     * 8.144,8.212,8.269,8.326,8.363,8.410,8.488,8.575,
     * 7.907,8.000,8.083,8.160,8.221,8.287,8.381,8.482,
     * 7.609,7.649,7.714,7.796,7.872,7.959,8.089,8.223,
     * 7.513,7.560,7.599,7.628,7.675,7.743,7.827,7.931,
     * 7.366,7.478,7.570,7.646,7.724,7.802,7.878,7.962,
     * 0.000,6.221,6.474,6.713,6.904,7.134,7.333,7.515,
     * 0.000,0.000,0.000,6.096,6.256,6.373,6.668,6.938/
C
      DATA ((EDEN(I,4,6,L,1),I=17,24),L=1,NZEN)/
     * 8.685,8.691,8.686,8.684,8.678,8.691,8.795,8.914,
     * 8.637,8.649,8.648,8.651,8.646,8.660,8.751,8.855,
     * 8.551,8.571,8.577,8.584,8.580,8.591,8.674,8.769,
     * 8.402,8.438,8.460,8.481,8.487,8.505,8.586,8.678,
     * 8.060,8.132,8.192,8.248,8.283,8.327,8.417,8.514,
     * 7.850,7.919,7.993,8.069,8.125,8.182,8.285,8.392,
     * 7.729,7.730,7.743,7.779,7.824,7.886,8.018,8.149,
     * 7.640,7.694,7.738,7.769,7.803,7.844,7.889,7.947,
     * 7.465,7.589,7.695,7.782,7.854,7.912,7.953,7.994,
     * 6.034,6.273,6.518,6.754,6.945,7.185,7.359,7.510,
     * 0.000,0.000,6.154,6.352,6.506,6.615,6.861,7.076/
C
      DATA ((EDEN(I,5,6,L,1),I=17,24),L=1,NZEN)/
     * 8.670,8.681,8.679,8.682,8.681,8.704,8.838,8.984,
     * 8.626,8.644,8.648,8.656,8.659,8.681,8.801,8.932,
     * 8.533,8.562,8.576,8.592,8.598,8.620,8.724,8.837,
     * 8.358,8.400,8.432,8.461,8.479,8.508,8.606,8.710,
     * 8.031,8.090,8.150,8.212,8.257,8.312,8.415,8.519,
     * 7.917,7.944,7.992,8.058,8.118,8.184,8.297,8.409,
     * 7.883,7.877,7.873,7.879,7.892,7.933,8.065,8.193,
     * 7.766,7.833,7.889,7.929,7.962,7.993,8.020,8.053,
     * 7.575,7.714,7.835,7.935,8.012,8.066,8.094,8.113,
     * 6.193,6.429,6.675,6.911,7.103,7.348,7.511,7.644,
     * 0.000,6.148,6.358,6.557,6.715,6.822,7.037,7.215/
C
      DATA ((EDEN(I,1,7,L,1),I=17,24),L=1,NZEN)/
     * 8.819,8.834,8.835,8.829,8.807,8.796,8.842,8.908,
     * 8.777,8.799,8.807,8.808,8.790,8.781,8.823,8.883,
     * 8.716,8.745,8.763,8.773,8.764,8.760,8.796,8.850,
     * 8.587,8.633,8.667,8.697,8.704,8.714,8.749,8.801,
     * 8.246,8.326,8.401,8.475,8.529,8.577,8.627,8.689,
     * 7.967,8.073,8.171,8.271,8.355,8.441,8.509,8.588,
     * 7.546,7.625,7.720,7.831,7.944,8.070,8.182,8.304,
     * 7.366,7.410,7.447,7.479,7.541,7.632,7.747,7.887,
     * 7.184,7.283,7.364,7.436,7.526,7.632,7.746,7.877,
     * 0.000,0.000,6.240,6.470,6.666,6.915,7.159,7.395,
     * 0.000,0.000,0.000,0.000,0.000,6.112,6.466,6.807/
C
      DATA ((EDEN(I,2,7,L,1),I=17,24),L=1,NZEN)/
     * 8.777,8.789,8.787,8.782,8.762,8.755,8.804,8.872,
     * 8.735,8.750,8.755,8.755,8.738,8.733,8.781,8.848,
     * 8.672,8.698,8.710,8.719,8.710,8.710,8.755,8.819,
     * 8.554,8.592,8.622,8.648,8.655,8.667,8.713,8.775,
     * 8.247,8.323,8.390,8.458,8.506,8.554,8.611,8.679,
     * 7.998,8.100,8.191,8.280,8.357,8.431,8.502,8.583,
     * 7.597,7.684,7.785,7.895,8.000,8.117,8.228,8.346,
     * 7.441,7.488,7.528,7.561,7.619,7.706,7.813,7.943,
     * 7.298,7.402,7.488,7.562,7.644,7.735,7.828,7.933,
     * 0.000,6.185,6.436,6.671,6.862,7.091,7.307,7.509,
     * 0.000,0.000,0.000,0.000,6.162,6.293,6.618,6.923/
C
      DATA ((EDEN(I,3,7,L,1),I=17,24),L=1,NZEN)/
     * 8.718,8.725,8.721,8.718,8.705,8.708,8.780,8.870,
     * 8.674,8.684,8.684,8.683,8.671,8.674,8.741,8.826,
     * 8.608,8.625,8.634,8.638,8.632,8.638,8.702,8.782,
     * 8.482,8.515,8.537,8.557,8.563,8.579,8.644,8.722,
     * 8.172,8.244,8.303,8.358,8.398,8.444,8.518,8.601,
     * 7.939,8.035,8.119,8.200,8.260,8.328,8.418,8.513,
     * 7.625,7.674,7.746,7.835,7.916,8.005,8.130,8.259,
     * 7.532,7.581,7.622,7.653,7.701,7.768,7.850,7.952,
     * 7.396,7.509,7.603,7.681,7.757,7.833,7.904,7.981,
     * 6.017,6.275,6.530,6.769,6.960,7.184,7.375,7.547,
     * 0.000,0.000,0.000,6.151,6.310,6.425,6.711,6.972/
C
      DATA ((EDEN(I,4,7,L,1),I=17,24),L=1,NZEN)/
     * 8.695,8.701,8.696,8.695,8.689,8.705,8.811,8.932,
     * 8.648,8.662,8.661,8.664,8.659,8.674,8.765,8.871,
     * 8.565,8.585,8.592,8.600,8.596,8.609,8.691,8.785,
     * 8.420,8.457,8.480,8.501,8.508,8.528,8.607,8.696,
     * 8.085,8.159,8.220,8.277,8.315,8.358,8.445,8.538,
     * 7.872,7.947,8.026,8.105,8.164,8.220,8.320,8.423,
     * 7.742,7.748,7.767,7.809,7.860,7.926,8.054,8.182,
     * 7.665,7.722,7.769,7.802,7.835,7.874,7.916,7.969,
     * 7.503,7.631,7.741,7.830,7.901,7.954,7.989,8.020,
     * 6.086,6.333,6.586,6.823,7.015,7.249,7.412,7.551,
     * 0.000,6.012,6.223,6.429,6.583,6.688,6.922,7.123/
C
      DATA ((EDEN(I,5,7,L,1),I=17,24),L=1,NZEN)/
     * 8.677,8.689,8.687,8.690,8.690,8.713,8.852,9.003,
     * 8.635,8.653,8.657,8.666,8.668,8.691,8.816,8.950,
     * 8.545,8.573,8.589,8.606,8.612,8.632,8.736,8.850,
     * 8.370,8.416,8.447,8.478,8.499,8.527,8.623,8.726,
     * 8.047,8.111,8.174,8.240,8.288,8.339,8.439,8.540,
     * 7.931,7.966,8.020,8.091,8.154,8.219,8.329,8.437,
     * 7.899,7.896,7.896,7.906,7.924,7.969,8.099,8.223,
     * 7.800,7.870,7.928,7.969,8.002,8.032,8.057,8.089,
     * 7.621,7.766,7.892,7.996,8.072,8.120,8.139,8.146,
     * 6.255,6.498,6.754,6.993,7.186,7.425,7.575,7.693,
     * 6.022,6.230,6.447,6.653,6.807,6.911,7.111,7.271/
C
      DATA ((EDEN(I,1,8,L,1),I=17,24),L=1,NZEN)/
     * 8.818,8.833,8.834,8.828,8.806,8.796,8.842,8.908,
     * 8.776,8.798,8.806,8.806,8.789,8.781,8.822,8.882,
     * 8.713,8.744,8.761,8.772,8.763,8.760,8.795,8.850,
     * 8.584,8.630,8.665,8.695,8.704,8.712,8.748,8.801,
     * 8.240,8.321,8.397,8.470,8.524,8.574,8.624,8.688,
     * 7.961,8.064,8.165,8.264,8.350,8.437,8.506,8.587,
     * 7.543,7.620,7.714,7.824,7.935,8.064,8.178,8.301,
     * 7.365,7.410,7.447,7.479,7.541,7.633,7.748,7.889,
     * 7.181,7.281,7.363,7.436,7.526,7.633,7.748,7.880,
     * 0.000,0.000,6.232,6.462,6.660,6.910,7.155,7.394,
     * 0.000,0.000,0.000,0.000,0.000,6.104,6.460,6.804/
C
      DATA ((EDEN(I,2,8,L,1),I=17,24),L=1,NZEN)/
     * 8.792,8.806,8.806,8.802,8.782,8.775,8.822,8.889,
     * 8.749,8.769,8.775,8.777,8.761,8.756,8.802,8.867,
     * 8.687,8.714,8.729,8.740,8.732,8.732,8.776,8.837,
     * 8.565,8.605,8.637,8.667,8.675,8.687,8.732,8.792,
     * 8.249,8.326,8.394,8.465,8.515,8.565,8.622,8.690,
     * 7.993,8.096,8.187,8.282,8.360,8.437,8.509,8.591,
     * 7.586,7.668,7.765,7.874,7.978,8.093,8.206,8.326,
     * 7.431,7.478,7.518,7.551,7.610,7.698,7.807,7.939,
     * 7.280,7.384,7.469,7.543,7.628,7.722,7.820,7.931,
     * 0.000,6.143,6.394,6.628,6.820,7.054,7.278,7.489,
     * 0.000,0.000,0.000,0.000,6.125,6.257,6.588,6.900/
C
      DATA ((EDEN(I,3,8,L,1),I=17,24),L=1,NZEN)/
     * 8.761,8.774,8.774,8.775,8.763,8.766,8.833,8.917,
     * 8.717,8.734,8.738,8.742,8.733,8.736,8.798,8.877,
     * 8.652,8.674,8.685,8.697,8.693,8.701,8.760,8.835,
     * 8.523,8.560,8.585,8.612,8.621,8.641,8.701,8.774,
     * 8.204,8.277,8.341,8.402,8.448,8.498,8.570,8.651,
     * 7.951,8.049,8.138,8.222,8.291,8.364,8.451,8.544,
     * 7.625,7.679,7.755,7.847,7.937,8.034,8.160,8.292,
     * 7.518,7.565,7.605,7.636,7.687,7.761,7.853,7.965,
     * 7.374,7.483,7.573,7.650,7.730,7.813,7.895,7.986,
     * 0.000,6.239,6.488,6.724,6.916,7.146,7.351,7.539,
     * 0.000,0.000,0.000,6.098,6.257,6.377,6.680,6.959/
C
      DATA ((EDEN(I,4,8,L,1),I=17,24),L=1,NZEN)/
     * 8.758,8.770,8.772,8.777,8.776,8.793,8.889,8.999,
     * 8.714,8.731,8.738,8.748,8.749,8.767,8.850,8.947,
     * 8.643,8.668,8.681,8.697,8.703,8.722,8.798,8.887,
     * 8.492,8.531,8.557,8.585,8.599,8.626,8.700,8.785,
     * 8.148,8.226,8.288,8.349,8.396,8.447,8.533,8.625,
     * 7.916,8.002,8.087,8.170,8.237,8.305,8.406,8.510,
     * 7.726,7.739,7.770,7.825,7.889,7.969,8.103,8.239,
     * 7.644,7.694,7.735,7.766,7.807,7.861,7.924,8.003,
     * 7.481,7.599,7.698,7.781,7.855,7.921,7.975,8.031,
     * 6.084,6.321,6.567,6.800,6.990,7.224,7.410,7.575,
     * 0.000,0.000,6.170,6.367,6.525,6.634,6.900,7.137/
C
      DATA ((EDEN(I,5,8,L,1),I=17,24),L=1,NZEN)/
     * 8.758,8.772,8.776,8.785,8.794,8.823,8.947,9.083,
     * 8.716,8.737,8.746,8.761,8.771,8.803,8.914,9.036,
     * 8.646,8.674,8.693,8.716,8.730,8.762,8.861,8.969,
     * 8.473,8.517,8.549,8.583,8.607,8.642,8.735,8.835,
     * 8.129,8.203,8.270,8.339,8.390,8.450,8.551,8.655,
     * 7.947,8.004,8.076,8.159,8.230,8.303,8.418,8.532,
     * 7.864,7.864,7.871,7.898,7.940,8.005,8.146,8.283,
     * 7.776,7.832,7.880,7.915,7.954,7.999,8.048,8.106,
     * 7.604,7.730,7.839,7.930,8.007,8.070,8.114,8.152,
     * 6.234,6.468,6.709,6.943,7.134,7.374,7.551,7.700,
     * 0.000,6.174,6.378,6.574,6.727,6.835,7.071,7.272/
C
      DATA ((EDEN(I,1,9,L,1),I=17,24),L=1,NZEN)/
     * 8.808,8.823,8.823,8.819,8.798,8.787,8.833,8.898,
     * 8.766,8.787,8.794,8.796,8.781,8.774,8.815,8.875,
     * 8.701,8.731,8.748,8.761,8.753,8.752,8.789,8.844,
     * 8.570,8.615,8.650,8.680,8.692,8.703,8.741,8.796,
     * 8.221,8.304,8.376,8.450,8.507,8.559,8.612,8.679,
     * 7.939,8.044,8.142,8.240,8.328,8.417,8.490,8.574,
     * 7.532,7.602,7.692,7.799,7.908,8.035,8.153,8.281,
     * 7.361,7.407,7.445,7.477,7.539,7.632,7.748,7.890,
     * 7.170,7.272,7.356,7.430,7.523,7.632,7.750,7.884,
     * 0.000,0.000,6.208,6.436,6.636,6.890,7.141,7.385,
     * 0.000,0.000,0.000,0.000,0.000,6.083,6.445,6.794/
C
      DATA ((EDEN(I,2,9,L,1),I=17,24),L=1,NZEN)/
     * 8.806,8.821,8.823,8.821,8.802,8.795,8.842,8.909,
     * 8.764,8.786,8.795,8.800,8.786,8.783,8.827,8.889,
     * 8.699,8.728,8.747,8.760,8.756,8.758,8.799,8.858,
     * 8.566,8.611,8.648,8.679,8.690,8.706,8.750,8.809,
     * 8.225,8.306,8.378,8.451,8.508,8.562,8.620,8.691,
     * 7.956,8.060,8.155,8.252,8.335,8.420,8.496,8.583,
     * 7.559,7.629,7.719,7.825,7.930,8.050,8.170,8.298,
     * 7.402,7.448,7.487,7.519,7.580,7.670,7.782,7.919,
     * 7.227,7.330,7.415,7.489,7.578,7.680,7.790,7.913,
     * 0.000,6.049,6.294,6.525,6.721,6.967,7.205,7.434,
     * 0.000,0.000,0.000,0.000,6.022,6.165,6.511,6.841/
C
      DATA ((EDEN(I,3,9,L,1),I=17,24),L=1,NZEN)/
     * 8.824,8.843,8.849,8.853,8.842,8.844,8.903,8.979,
     * 8.781,8.805,8.817,8.826,8.819,8.825,8.878,8.948,
     * 8.711,8.743,8.763,8.782,8.782,8.792,8.842,8.909,
     * 8.571,8.617,8.651,8.683,8.697,8.718,8.772,8.840,
     * 8.223,8.303,8.373,8.444,8.499,8.556,8.626,8.706,
     * 7.957,8.058,8.149,8.243,8.324,8.409,8.496,8.592,
     * 7.606,7.657,7.731,7.826,7.921,8.034,8.165,8.302,
     * 7.474,7.516,7.551,7.580,7.636,7.721,7.827,7.957,
     * 7.301,7.402,7.484,7.557,7.643,7.740,7.842,7.957,
     * 0.000,6.122,6.358,6.585,6.778,7.022,7.258,7.481,
     * 0.000,0.000,0.000,0.000,6.157,6.291,6.623,6.936/
C
      DATA ((EDEN(I,4,9,L,1),I=17,24),L=1,NZEN)/
     * 8.868,8.891,8.903,8.914,8.913,8.929,9.006,9.097,
     * 8.825,8.853,8.871,8.889,8.892,8.911,8.981,9.064,
     * 8.753,8.790,8.813,8.837,8.847,8.869,8.933,9.011,
     * 8.611,8.660,8.699,8.737,8.760,8.792,8.859,8.936,
     * 8.247,8.327,8.396,8.469,8.528,8.593,8.675,8.766,
     * 7.982,8.078,8.170,8.262,8.344,8.434,8.535,8.643,
     * 7.698,7.724,7.777,7.857,7.945,8.054,8.198,8.346,
     * 7.583,7.619,7.649,7.676,7.727,7.807,7.907,8.028,
     * 7.409,7.504,7.581,7.652,7.735,7.829,7.927,8.036,
     * 6.001,6.221,6.446,6.666,6.855,7.100,7.333,7.551,
     * 0.000,0.000,6.006,6.184,6.340,6.470,6.795,7.100/
C
      DATA ((EDEN(I,5,9,L,1),I=17,24),L=1,NZEN)/
     * 8.914,8.942,8.960,8.982,8.995,9.025,9.123,9.232,
     * 8.872,8.905,8.928,8.955,8.972,9.005,9.095,9.196,
     * 8.807,8.848,8.880,8.914,8.938,8.975,9.060,9.154,
     * 8.665,8.720,8.768,8.817,8.854,8.902,8.986,9.078,
     * 8.296,8.375,8.448,8.529,8.600,8.679,8.778,8.883,
     * 8.043,8.132,8.223,8.321,8.413,8.517,8.635,8.757,
     * 7.806,7.819,7.859,7.930,8.019,8.135,8.295,8.455,
     * 7.697,7.729,7.757,7.784,7.840,7.925,8.033,8.158,
     * 7.520,7.611,7.686,7.757,7.843,7.942,8.047,8.160,
     * 6.146,6.352,6.565,6.781,6.972,7.222,7.464,7.689,
     * 0.000,0.000,6.169,6.341,6.494,6.627,6.953,7.255/
C
      DATA ((EDEN(I,1,10,L,1),I=17,24),L=1,NZEN)/
     * 8.804,8.819,8.821,8.818,8.798,8.787,8.832,8.896,
     * 8.760,8.782,8.792,8.794,8.781,8.775,8.816,8.875,
     * 8.695,8.726,8.745,8.758,8.753,8.753,8.790,8.846,
     * 8.561,8.608,8.645,8.676,8.689,8.704,8.743,8.799,
     * 8.208,8.291,8.366,8.441,8.500,8.557,8.611,8.679,
     * 7.924,8.030,8.128,8.228,8.318,8.410,8.484,8.571,
     * 7.525,7.591,7.679,7.784,7.892,8.020,8.141,8.271,
     * 7.358,7.405,7.445,7.479,7.542,7.636,7.753,7.896,
     * 7.162,7.267,7.353,7.430,7.525,7.636,7.756,7.892,
     * 0.000,0.000,6.185,6.416,6.619,6.877,7.132,7.381,
     * 0.000,0.000,0.000,0.000,0.000,6.066,6.433,6.788/
C
      DATA ((EDEN(I,2,10,L,1),I=17,24),L=1,NZEN)/
     * 8.824,8.842,8.847,8.846,8.828,8.822,8.868,8.934,
     * 8.781,8.806,8.816,8.823,8.812,8.809,8.851,8.911,
     * 8.713,8.747,8.767,8.784,8.781,8.784,8.823,8.880,
     * 8.575,8.624,8.661,8.696,8.712,8.729,8.770,8.827,
     * 8.211,8.295,8.369,8.446,8.507,8.564,8.623,8.694,
     * 7.915,8.022,8.118,8.218,8.307,8.400,8.480,8.572,
     * 7.530,7.587,7.668,7.768,7.873,7.997,8.124,8.262,
     * 7.366,7.411,7.447,7.479,7.541,7.633,7.749,7.891,
     * 7.162,7.264,7.347,7.422,7.517,7.629,7.752,7.892,
     * 0.000,0.000,6.158,6.386,6.588,6.851,7.112,7.367,
     * 0.000,0.000,0.000,0.000,0.000,6.055,6.423,6.779/
C
      DATA ((EDEN(I,3,10,L,1),I=17,24),L=1,NZEN)/
     * 8.886,8.910,8.921,8.925,8.913,8.913,8.965,9.033,
     * 8.841,8.870,8.886,8.896,8.889,8.894,8.941,9.005,
     * 8.772,8.810,8.835,8.855,8.857,8.866,8.910,8.970,
     * 8.627,8.679,8.722,8.759,8.777,8.799,8.846,8.908,
     * 8.249,8.335,8.414,8.492,8.553,8.616,8.682,8.760,
     * 7.952,8.056,8.150,8.252,8.342,8.436,8.524,8.623,
     * 7.583,7.629,7.700,7.795,7.899,8.026,8.163,8.309,
     * 7.418,7.454,7.483,7.509,7.569,7.662,7.782,7.928,
     * 7.212,7.301,7.372,7.440,7.531,7.643,7.770,7.913,
     * 0.000,0.000,6.187,6.404,6.600,6.863,7.130,7.391,
     * 0.000,0.000,0.000,0.000,0.000,6.107,6.472,6.826/
C
      DATA ((EDEN(I,4,10,L,1),I=17,24),L=1,NZEN)/
     * 8.988,9.018,9.034,9.044,9.042,9.059,9.116,9.189,
     * 8.943,8.978,8.999,9.015,9.014,9.038,9.091,9.158,
     * 8.874,8.919,8.949,8.972,8.980,9.005,9.056,9.122,
     * 8.733,8.794,8.839,8.879,8.903,8.937,8.991,9.058,
     * 8.353,8.440,8.522,8.603,8.667,8.736,8.810,8.893,
     * 8.068,8.171,8.273,8.379,8.476,8.577,8.671,8.773,
     * 7.686,7.728,7.800,7.901,8.016,8.155,8.299,8.449,
     * 7.511,7.533,7.550,7.570,7.631,7.732,7.865,8.023,
     * 7.309,7.379,7.433,7.489,7.577,7.695,7.833,7.989,
     * 0.000,6.092,6.289,6.492,6.682,6.946,7.222,7.491,
     * 0.000,0.000,0.000,0.000,6.067,6.219,6.594,6.958/
C
      DATA ((EDEN(I,5,10,L,1),I=17,24),L=1,NZEN)/
     * 9.109,9.145,9.166,9.186,9.201,9.241,9.307,9.385,
     * 9.066,9.106,9.132,9.158,9.174,9.218,9.280,9.352,
     * 8.999,9.049,9.083,9.115,9.139,9.185,9.244,9.314,
     * 8.867,8.933,8.985,9.031,9.067,9.124,9.186,9.258,
     * 8.503,8.601,8.691,8.781,8.852,8.937,9.018,9.107,
     * 8.218,8.327,8.438,8.556,8.660,8.773,8.874,8.982,
     * 7.811,7.873,7.965,8.091,8.238,8.403,8.548,8.699,
     * 7.607,7.619,7.631,7.654,7.725,7.845,8.001,8.183,
     * 7.410,7.465,7.505,7.557,7.651,7.786,7.948,8.128,
     * 6.052,6.230,6.412,6.606,6.800,7.070,7.366,7.656,
     * 0.000,0.000,0.000,6.048,6.207,6.375,6.771,7.158/
C
      DATA ((EDEN(I,1,11,L,1),I=17,24),L=1,NZEN)/
     * 8.811,8.829,8.833,8.832,8.814,8.804,8.846,8.907,
     * 8.768,8.792,8.804,8.809,8.796,8.792,8.831,8.888,
     * 8.703,8.736,8.757,8.773,8.768,8.770,8.806,8.860,
     * 8.568,8.617,8.656,8.692,8.705,8.721,8.759,8.813,
     * 8.214,8.296,8.374,8.455,8.515,8.576,8.628,8.694,
     * 7.927,8.035,8.135,8.238,8.332,8.429,8.501,8.585,
     * 7.528,7.595,7.683,7.790,7.902,8.035,8.154,8.283,
     * 7.361,7.410,7.451,7.486,7.552,7.647,7.764,7.907,
     * 7.165,7.271,7.358,7.437,7.535,7.647,7.769,7.906,
     * 0.000,0.000,6.185,6.416,6.619,6.883,7.140,7.390,
     * 0.000,0.000,0.000,0.000,0.000,6.063,6.434,6.792/
C
      DATA ((EDEN(I,2,11,L,1),I=17,24),L=1,NZEN)/
     * 8.847,8.867,8.874,8.874,8.857,8.850,8.896,8.962,
     * 8.800,8.825,8.838,8.844,8.832,8.830,8.871,8.931,
     * 8.735,8.772,8.796,8.815,8.813,8.818,8.854,8.908,
     * 8.591,8.644,8.687,8.723,8.740,8.759,8.798,8.853,
     * 8.203,8.292,8.371,8.453,8.518,8.581,8.639,8.710,
     * 7.888,7.994,8.094,8.198,8.292,8.392,8.474,8.568,
     * 7.512,7.560,7.632,7.728,7.835,7.966,8.097,8.240,
     * 7.337,7.380,7.416,7.448,7.512,7.609,7.731,7.880,
     * 7.110,7.210,7.291,7.367,7.468,7.589,7.723,7.875,
     * 0.000,0.000,6.034,6.263,6.472,6.756,7.037,7.314,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.320,6.700/
C
      DATA ((EDEN(I,3,11,L,1),I=17,24),L=1,NZEN)/
     * 8.934,8.960,8.971,8.974,8.958,8.960,9.008,9.073,
     * 8.886,8.917,8.934,8.942,8.931,8.936,8.979,9.040,
     * 8.814,8.853,8.878,8.896,8.894,8.903,8.944,9.002,
     * 8.671,8.729,8.774,8.811,8.827,8.850,8.894,8.952,
     * 8.266,8.354,8.436,8.516,8.579,8.644,8.709,8.786,
     * 7.962,8.070,8.170,8.274,8.371,8.470,8.557,8.656,
     * 7.563,7.603,7.669,7.759,7.866,7.998,8.140,8.293,
     * 7.369,7.397,7.418,7.439,7.500,7.600,7.731,7.892,
     * 7.130,7.208,7.268,7.329,7.423,7.546,7.692,7.859,
     * 0.000,0.000,6.027,6.231,6.429,6.708,7.003,7.298,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.302,6.694/
C
      DATA ((EDEN(I,4,11,L,1),I=17,24),L=1,NZEN)/
     * 9.074,9.103,9.115,9.121,9.112,9.129,9.177,9.240,
     * 9.025,9.060,9.076,9.089,9.085,9.102,9.147,9.206,
     * 8.954,8.996,9.021,9.038,9.041,9.064,9.108,9.167,
     * 8.811,8.870,8.912,8.945,8.963,8.996,9.045,9.108,
     * 8.424,8.515,8.597,8.674,8.734,8.802,8.872,8.954,
     * 8.124,8.231,8.336,8.446,8.540,8.641,8.732,8.834,
     * 7.673,7.734,7.820,7.935,8.067,8.224,8.364,8.515,
     * 7.432,7.438,7.441,7.453,7.516,7.631,7.786,7.972,
     * 7.197,7.245,7.277,7.318,7.409,7.543,7.710,7.903,
     * 0.000,0.000,6.127,6.314,6.509,6.787,7.094,7.404,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.392,6.807/
C
      DATA ((EDEN(I,5,11,L,1),I=17,24),L=1,NZEN)/
     * 9.241,9.275,9.293,9.310,9.323,9.363,9.412,9.473,
     * 9.197,9.232,9.254,9.276,9.292,9.334,9.379,9.436,
     * 9.130,9.172,9.199,9.225,9.245,9.293,9.338,9.394,
     * 8.998,9.054,9.095,9.135,9.166,9.220,9.270,9.332,
     * 8.640,8.735,8.816,8.891,8.958,9.043,9.115,9.197,
     * 8.344,8.461,8.569,8.675,8.769,8.882,8.974,9.077,
     * 7.847,7.943,8.062,8.209,8.361,8.529,8.665,8.809,
     * 7.497,7.496,7.496,7.514,7.596,7.738,7.926,8.147,
     * 7.263,7.292,7.309,7.346,7.447,7.604,7.803,8.030,
     * 0.000,6.077,6.232,6.409,6.611,6.906,7.235,7.570,
     * 0.000,0.000,0.000,0.000,0.000,6.103,6.549,7.000/
C
      DATA ((EDEN(I,1,12,L,1),I=17,24),L=1,NZEN)/
     * 8.826,8.846,8.853,8.852,8.835,8.829,8.867,8.925,
     * 8.783,8.809,8.822,8.829,8.817,8.815,8.850,8.905,
     * 8.717,8.753,8.776,8.793,8.790,8.793,8.826,8.878,
     * 8.582,8.633,8.677,8.713,8.728,8.746,8.781,8.832,
     * 8.227,8.314,8.395,8.479,8.542,8.602,8.651,8.714,
     * 7.941,8.050,8.155,8.264,8.360,8.458,8.526,8.606,
     * 7.535,7.604,7.697,7.809,7.926,8.064,8.179,8.303,
     * 7.367,7.416,7.458,7.498,7.565,7.662,7.779,7.920,
     * 7.174,7.280,7.368,7.449,7.549,7.662,7.784,7.920,
     * 0.000,0.000,6.194,6.427,6.633,6.902,7.157,7.406,
     * 0.000,0.000,0.000,0.000,0.000,6.075,6.445,6.804/
C
      DATA ((EDEN(I,2,12,L,1),I=17,24),L=1,NZEN)/
     * 8.864,8.885,8.894,8.895,8.878,8.873,8.919,8.985,
     * 8.816,8.843,8.858,8.864,8.853,8.849,8.889,8.949,
     * 8.750,8.788,8.813,8.830,8.827,8.832,8.868,8.922,
     * 8.606,8.662,8.707,8.745,8.762,8.784,8.820,8.873,
     * 8.207,8.296,8.382,8.466,8.533,8.601,8.656,8.726,
     * 7.909,8.018,8.120,8.230,8.328,8.429,8.506,8.596,
     * 7.505,7.551,7.623,7.719,7.828,7.961,8.092,8.235,
     * 7.322,7.366,7.402,7.437,7.504,7.605,7.730,7.882,
     * 7.089,7.186,7.265,7.342,7.445,7.569,7.708,7.866,
     * 0.000,0.000,0.000,6.196,6.415,6.716,7.006,7.294,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.261,6.656/
C
      DATA ((EDEN(I,3,12,L,1),I=17,24),L=1,NZEN)/
     * 8.954,8.981,8.993,8.994,8.978,8.977,9.026,9.092,
     * 8.905,8.937,8.954,8.961,8.949,8.951,8.994,9.055,
     * 8.831,8.872,8.897,8.912,8.908,8.912,8.954,9.011,
     * 8.690,8.748,8.793,8.827,8.842,8.865,8.907,8.965,
     * 8.288,8.379,8.464,8.545,8.609,8.673,8.736,8.814,
     * 7.957,8.062,8.165,8.269,8.366,8.467,8.555,8.656,
     * 7.544,7.593,7.667,7.768,7.889,8.031,8.172,8.325,
     * 7.336,7.358,7.372,7.390,7.450,7.553,7.692,7.862,
     * 7.077,7.147,7.198,7.252,7.347,7.477,7.634,7.816,
     * 0.000,0.000,0.000,6.147,6.346,6.636,6.944,7.255,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.192,6.607/
C
      DATA ((EDEN(I,4,12,L,1),I=17,24),L=1,NZEN)/
     * 9.101,9.127,9.136,9.136,9.128,9.137,9.185,9.248,
     * 9.050,9.080,9.094,9.100,9.094,9.107,9.151,9.211,
     * 8.975,9.013,9.035,9.048,9.046,9.065,9.109,9.168,
     * 8.828,8.882,8.917,8.946,8.958,8.991,9.040,9.104,
     * 8.434,8.524,8.603,8.676,8.733,8.799,8.871,8.958,
     * 8.119,8.224,8.324,8.429,8.520,8.620,8.716,8.824,
     * 7.646,7.714,7.806,7.929,8.069,8.229,8.373,8.529,
     * 7.353,7.352,7.348,7.357,7.424,7.549,7.720,7.927,
     * 7.095,7.130,7.149,7.184,7.277,7.424,7.611,7.829,
     * 0.000,0.000,0.000,6.171,6.367,6.657,6.984,7.319,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.247,6.694/
C
      DATA ((EDEN(I,5,12,L,1),I=17,24),L=1,NZEN)/
     * 9.278,9.304,9.319,9.330,9.338,9.372,9.422,9.484,
     * 9.229,9.258,9.272,9.290,9.301,9.335,9.382,9.441,
     * 9.156,9.194,9.213,9.234,9.249,9.288,9.335,9.393,
     * 9.019,9.066,9.099,9.130,9.158,9.209,9.261,9.326,
     * 8.646,8.732,8.802,8.867,8.928,9.009,9.087,9.178,
     * 8.345,8.453,8.552,8.647,8.737,8.850,8.949,9.061,
     * 7.833,7.926,8.041,8.181,8.328,8.498,8.640,8.795,
     * 7.400,7.386,7.373,7.383,7.468,7.623,7.834,8.081,
     * 7.116,7.124,7.120,7.146,7.249,7.423,7.650,7.912,
     * 0.000,0.000,6.057,6.220,6.424,6.733,7.090,7.461,
     * 0.000,0.000,0.000,0.000,0.000,0.000,6.351,6.844/
C
      DATA ((EDEN(I,1,1,L,2),I=17,24),L=1,NZEN)/
     * 8.952,8.987,9.007,9.017,9.012,9.020,9.055,9.110,
     * 8.912,8.954,8.981,9.000,8.999,9.009,9.040,9.089,
     * 8.849,8.902,8.940,8.969,8.976,8.990,9.017,9.062,
     * 8.721,8.792,8.853,8.905,8.927,8.951,8.978,9.020,
     * 8.380,8.494,8.606,8.714,8.788,8.851,8.887,8.934,
     * 8.101,8.240,8.379,8.523,8.639,8.740,8.791,8.850,
     * 7.662,7.778,7.914,8.070,8.223,8.384,8.476,8.574,
     * 7.519,7.604,7.687,7.760,7.845,7.938,8.034,8.144,
     * 7.371,7.522,7.659,7.781,7.897,7.999,8.086,8.176,
     * 0.000,6.232,6.529,6.805,7.025,7.285,7.496,7.690,
     * 0.000,0.000,0.000,6.158,6.355,6.497,6.818,7.114/
C
      DATA ((EDEN(I,2,1,L,2),I=17,24),L=1,NZEN)/
     * 8.989,9.025,9.048,9.059,9.054,9.060,9.107,9.173,
     * 8.943,8.984,9.012,9.031,9.028,9.038,9.076,9.134,
     * 8.880,8.933,8.971,9.002,9.005,9.020,9.053,9.103,
     * 8.741,8.815,8.876,8.927,8.953,8.982,9.012,9.057,
     * 8.356,8.472,8.583,8.693,8.770,8.840,8.883,8.939,
     * 8.061,8.201,8.338,8.478,8.595,8.700,8.761,8.833,
     * 7.614,7.703,7.819,7.960,8.104,8.261,8.373,8.493,
     * 7.459,7.538,7.613,7.682,7.766,7.863,7.969,8.092,
     * 7.266,7.407,7.534,7.651,7.771,7.886,7.994,8.110,
     * 0.000,0.000,6.258,6.536,6.777,7.075,7.325,7.564,
     * 0.000,0.000,0.000,0.000,6.084,6.246,6.605,6.946/
C
      DATA ((EDEN(I,3,1,L,2),I=17,24),L=1,NZEN)/
     * 9.073,9.113,9.134,9.148,9.145,9.157,9.208,9.277,
     * 9.024,9.069,9.098,9.114,9.116,9.130,9.173,9.234,
     * 8.953,9.007,9.044,9.070,9.077,9.093,9.132,9.187,
     * 8.818,8.891,8.949,8.993,9.016,9.043,9.083,9.137,
     * 8.427,8.545,8.653,8.752,8.825,8.890,8.945,9.012,
     * 8.099,8.233,8.364,8.498,8.611,8.715,8.791,8.878,
     * 7.638,7.731,7.851,7.995,8.149,8.311,8.436,8.570,
     * 7.437,7.493,7.543,7.593,7.671,7.776,7.903,8.053,
     * 7.214,7.323,7.417,7.507,7.619,7.746,7.881,8.033,
     * 0.000,0.000,6.182,6.435,6.656,6.940,7.219,7.493,
     * 0.000,0.000,0.000,0.000,0.000,6.101,6.485,6.860/
C
      DATA ((EDEN(I,4,1,L,2),I=17,24),L=1,NZEN)/
     * 9.279,9.314,9.334,9.345,9.347,9.368,9.410,9.467,
     * 9.227,9.269,9.293,9.308,9.313,9.337,9.374,9.425,
     * 9.156,9.205,9.238,9.260,9.270,9.296,9.331,9.381,
     * 9.016,9.084,9.132,9.169,9.190,9.225,9.266,9.320,
     * 8.631,8.749,8.849,8.933,8.995,9.060,9.121,9.194,
     * 8.312,8.453,8.587,8.709,8.808,8.904,8.986,9.079,
     * 7.779,7.902,8.046,8.214,8.383,8.551,8.676,8.812,
     * 7.451,7.484,7.517,7.560,7.651,7.785,7.951,8.145,
     * 7.212,7.285,7.346,7.418,7.532,7.679,7.849,8.041,
     * 0.000,6.010,6.211,6.423,6.639,6.934,7.236,7.541,
     * 0.000,0.000,0.000,0.000,0.000,6.084,6.499,6.915/
C
      DATA ((EDEN(I,5,1,L,2),I=17,24),L=1,NZEN)/
     * 9.501,9.541,9.568,9.590,9.609,9.642,9.680,9.730,
     * 9.451,9.494,9.524,9.548,9.569,9.606,9.641,9.687,
     * 9.382,9.430,9.465,9.492,9.517,9.559,9.593,9.639,
     * 9.251,9.312,9.356,9.394,9.430,9.481,9.522,9.575,
     * 8.886,8.990,9.072,9.146,9.213,9.297,9.363,9.441,
     * 8.579,8.717,8.835,8.940,9.035,9.148,9.235,9.334,
     * 8.000,8.150,8.317,8.487,8.648,8.816,8.945,9.085,
     * 7.507,7.530,7.555,7.600,7.709,7.874,8.081,8.317,
     * 7.219,7.266,7.304,7.367,7.493,7.671,7.885,8.126,
     * 0.000,6.058,6.227,6.420,6.650,6.971,7.311,7.662,
     * 0.000,0.000,0.000,0.000,0.000,6.084,6.547,7.022/
C
      DATA ((EDEN(I,1,2,L,2),I=17,24),L=1,NZEN)/
     * 8.937,8.970,8.989,9.002,8.995,9.002,9.039,9.096,
     * 8.897,8.937,8.964,8.980,8.982,8.990,9.023,9.075,
     * 8.833,8.884,8.921,8.951,8.957,8.972,9.001,9.048,
     * 8.704,8.774,8.834,8.885,8.908,8.933,8.962,9.006,
     * 8.362,8.475,8.583,8.690,8.765,8.829,8.867,8.917,
     * 8.082,8.220,8.357,8.497,8.612,8.713,8.767,8.830,
     * 7.651,7.761,7.894,8.045,8.193,8.351,8.448,8.551,
     * 7.511,7.596,7.678,7.749,7.832,7.924,8.021,8.133,
     * 7.361,7.512,7.649,7.769,7.883,7.986,8.074,8.165,
     * 0.000,6.215,6.513,6.788,7.007,7.265,7.479,7.675,
     * 0.000,0.000,0.000,6.147,6.343,6.482,6.804,7.101/
C
      DATA ((EDEN(I,2,2,L,2),I=17,24),L=1,NZEN)/
     * 8.962,8.997,9.018,9.029,9.026,9.035,9.082,9.149,
     * 8.915,8.956,8.983,9.000,9.002,9.012,9.052,9.110,
     * 8.853,8.904,8.944,8.974,8.981,8.998,9.032,9.083,
     * 8.713,8.783,8.844,8.896,8.920,8.949,8.982,9.030,
     * 8.335,8.447,8.554,8.662,8.739,8.809,8.855,8.914,
     * 8.044,8.181,8.312,8.449,8.565,8.670,8.734,8.808,
     * 7.606,7.692,7.806,7.945,8.086,8.242,8.357,8.479,
     * 7.453,7.531,7.605,7.671,7.754,7.851,7.958,8.082,
     * 7.266,7.409,7.536,7.653,7.773,7.889,7.997,8.113,
     * 0.000,6.004,6.293,6.565,6.796,7.086,7.334,7.569,
     * 0.000,0.000,0.000,0.000,6.119,6.275,6.630,6.967/
C
      DATA ((EDEN(I,3,2,L,2),I=17,24),L=1,NZEN)/
     * 9.032,9.071,9.094,9.108,9.109,9.126,9.177,9.247,
     * 8.983,9.027,9.055,9.076,9.079,9.097,9.143,9.205,
     * 8.911,8.963,9.001,9.030,9.037,9.061,9.103,9.161,
     * 8.774,8.845,8.902,8.951,8.976,9.009,9.051,9.108,
     * 8.390,8.501,8.606,8.708,8.784,8.853,8.911,8.981,
     * 8.068,8.198,8.328,8.460,8.574,8.681,8.760,8.849,
     * 7.626,7.715,7.829,7.970,8.119,8.280,8.407,8.543,
     * 7.436,7.493,7.546,7.597,7.677,7.783,7.910,8.059,
     * 7.223,7.335,7.433,7.527,7.640,7.766,7.899,8.046,
     * 0.000,0.000,6.209,6.452,6.671,6.952,7.229,7.500,
     * 0.000,0.000,0.000,0.000,0.000,6.152,6.533,6.902/
C
      DATA ((EDEN(I,4,2,L,2),I=17,24),L=1,NZEN)/
     * 9.182,9.222,9.246,9.262,9.271,9.302,9.350,9.413,
     * 9.130,9.175,9.204,9.228,9.238,9.270,9.314,9.372,
     * 9.059,9.112,9.149,9.177,9.192,9.229,9.271,9.328,
     * 8.914,8.986,9.041,9.084,9.114,9.158,9.204,9.265,
     * 8.528,8.645,8.749,8.842,8.912,8.988,9.054,9.133,
     * 8.215,8.351,8.481,8.612,8.721,8.828,8.915,9.013,
     * 7.716,7.816,7.942,8.095,8.259,8.432,8.567,8.712,
     * 7.446,7.481,7.514,7.556,7.643,7.773,7.935,8.123,
     * 7.218,7.295,7.358,7.430,7.542,7.685,7.850,8.036,
     * 0.000,6.013,6.221,6.437,6.652,6.940,7.241,7.541,
     * 0.000,0.000,0.000,0.000,0.000,6.124,6.533,6.940/
C
      DATA ((EDEN(I,5,2,L,2),I=17,24),L=1,NZEN)/
     * 9.348,9.390,9.420,9.451,9.476,9.524,9.574,9.635,
     * 9.299,9.344,9.375,9.408,9.435,9.487,9.533,9.591,
     * 9.224,9.278,9.315,9.352,9.384,9.437,9.483,9.541,
     * 9.085,9.155,9.205,9.253,9.295,9.356,9.409,9.473,
     * 8.705,8.819,8.915,9.004,9.078,9.173,9.248,9.335,
     * 8.394,8.534,8.663,8.787,8.896,9.019,9.116,9.222,
     * 7.852,7.979,8.129,8.305,8.481,8.665,8.806,8.956,
     * 7.462,7.481,7.504,7.546,7.651,7.813,8.016,8.248,
     * 7.197,7.246,7.286,7.348,7.472,7.647,7.858,8.094,
     * 0.000,6.023,6.198,6.395,6.619,6.934,7.273,7.620,
     * 0.000,0.000,0.000,0.000,0.000,6.082,6.542,7.009/
C
      DATA ((EDEN(I,1,3,L,2),I=17,24),L=1,NZEN)/
     * 8.916,8.947,8.964,8.977,8.968,8.975,9.015,9.076,
     * 8.874,8.912,8.938,8.957,8.957,8.963,9.000,9.055,
     * 8.812,8.859,8.896,8.925,8.929,8.943,8.976,9.026,
     * 8.681,8.749,8.806,8.856,8.880,8.904,8.936,8.984,
     * 8.338,8.449,8.554,8.656,8.733,8.795,8.837,8.891,
     * 8.058,8.194,8.326,8.462,8.575,8.676,8.735,8.803,
     * 7.637,7.742,7.871,8.015,8.156,8.307,8.410,8.519,
     * 7.502,7.586,7.665,7.734,7.814,7.906,8.003,8.117,
     * 7.348,7.499,7.635,7.753,7.865,7.967,8.055,8.148,
     * 0.000,6.197,6.495,6.768,6.988,7.238,7.455,7.653,
     * 0.000,0.000,0.000,6.134,6.329,6.467,6.789,7.086/
C
      DATA ((EDEN(I,2,3,L,2),I=17,24),L=1,NZEN)/
     * 8.922,8.954,8.973,8.985,8.983,8.992,9.040,9.108,
     * 8.879,8.917,8.943,8.964,8.965,8.978,9.021,9.081,
     * 8.812,8.860,8.896,8.926,8.934,8.952,8.990,9.045,
     * 8.675,8.740,8.797,8.846,8.872,8.900,8.938,8.991,
     * 8.309,8.416,8.517,8.620,8.697,8.766,8.817,8.879,
     * 8.009,8.142,8.269,8.399,8.512,8.619,8.689,8.768,
     * 7.603,7.691,7.804,7.939,8.074,8.223,8.340,8.464,
     * 7.463,7.540,7.612,7.674,7.754,7.848,7.953,8.076,
     * 7.284,7.427,7.555,7.669,7.784,7.896,7.999,8.109,
     * 0.000,6.078,6.361,6.633,6.856,7.124,7.363,7.587,
     * 0.000,0.000,0.000,6.004,6.200,6.348,6.691,7.012/
C
      DATA ((EDEN(I,3,3,L,2),I=17,24),L=1,NZEN)/
     * 8.969,9.005,9.028,9.045,9.047,9.068,9.123,9.195,
     * 8.920,8.961,8.989,9.012,9.018,9.041,9.090,9.156,
     * 8.850,8.899,8.937,8.969,8.981,9.008,9.055,9.117,
     * 8.701,8.769,8.825,8.875,8.903,8.936,8.984,9.047,
     * 8.320,8.424,8.522,8.623,8.700,8.772,8.836,8.911,
     * 8.030,8.155,8.277,8.405,8.517,8.625,8.708,8.800,
     * 7.617,7.690,7.789,7.914,8.045,8.195,8.329,8.470,
     * 7.458,7.518,7.575,7.627,7.705,7.807,7.927,8.068,
     * 7.265,7.385,7.489,7.587,7.698,7.817,7.939,8.072,
     * 0.000,6.045,6.302,6.551,6.765,7.036,7.297,7.547,
     * 0.000,0.000,0.000,0.000,6.113,6.272,6.634,6.979/
C
      DATA ((EDEN(I,4,3,L,2),I=17,24),L=1,NZEN)/
     * 9.048,9.088,9.115,9.139,9.153,9.191,9.251,9.326,
     * 8.995,9.041,9.073,9.100,9.120,9.159,9.215,9.285,
     * 8.921,8.975,9.014,9.050,9.072,9.116,9.171,9.239,
     * 8.775,8.845,8.903,8.957,8.993,9.043,9.101,9.172,
     * 8.387,8.492,8.591,8.690,8.768,8.852,8.929,9.016,
     * 8.085,8.205,8.323,8.449,8.563,8.678,8.776,8.882,
     * 7.674,7.746,7.844,7.973,8.117,8.284,8.429,8.582,
     * 7.484,7.521,7.556,7.594,7.672,7.788,7.932,8.100,
     * 7.278,7.366,7.441,7.519,7.626,7.758,7.905,8.068,
     * 0.000,6.065,6.288,6.511,6.721,6.999,7.284,7.561,
     * 0.000,0.000,0.000,0.000,6.085,6.251,6.638,7.015/
C
      DATA ((EDEN(I,5,3,L,2),I=17,24),L=1,NZEN)/
     * 9.133,9.178,9.212,9.248,9.275,9.337,9.409,9.492,
     * 9.078,9.128,9.165,9.205,9.238,9.298,9.366,9.445,
     * 9.002,9.059,9.101,9.146,9.183,9.247,9.314,9.392,
     * 8.853,8.926,8.985,9.045,9.092,9.166,9.238,9.320,
     * 8.465,8.571,8.672,8.776,8.865,8.969,9.061,9.162,
     * 8.165,8.284,8.405,8.538,8.661,8.795,8.909,9.031,
     * 7.732,7.801,7.902,8.043,8.205,8.394,8.557,8.726,
     * 7.497,7.516,7.536,7.571,7.660,7.802,7.982,8.189,
     * 7.269,7.330,7.378,7.441,7.555,7.713,7.900,8.108,
     * 0.000,6.065,6.254,6.460,6.675,6.971,7.294,7.615,
     * 0.000,0.000,0.000,0.000,6.030,6.227,6.659,7.086/
C
      DATA ((EDEN(I,1,4,L,2),I=17,24),L=1,NZEN)/
     * 8.902,8.930,8.947,8.958,8.950,8.955,8.998,9.062,
     * 8.860,8.896,8.920,8.939,8.938,8.942,8.981,9.039,
     * 8.798,8.843,8.877,8.906,8.913,8.923,8.958,9.010,
     * 8.669,8.735,8.788,8.837,8.860,8.880,8.915,8.965,
     * 8.327,8.435,8.537,8.637,8.712,8.772,8.817,8.873,
     * 8.049,8.183,8.313,8.442,8.553,8.653,8.715,8.785,
     * 7.631,7.734,7.861,8.003,8.138,8.286,8.392,8.503,
     * 7.495,7.580,7.657,7.723,7.802,7.893,7.990,8.105,
     * 7.340,7.491,7.626,7.743,7.853,7.953,8.040,8.131,
     * 0.000,6.191,6.489,6.763,6.982,7.229,7.444,7.641,
     * 0.000,0.000,0.000,6.133,6.328,6.467,6.785,7.079/
C
      DATA ((EDEN(I,2,4,L,2),I=17,24),L=1,NZEN)/
     * 8.887,8.914,8.930,8.943,8.939,8.948,8.997,9.066,
     * 8.842,8.877,8.899,8.919,8.920,8.930,8.975,9.038,
     * 8.778,8.819,8.851,8.880,8.887,8.904,8.946,9.004,
     * 8.645,8.704,8.755,8.803,8.826,8.852,8.894,8.951,
     * 8.297,8.401,8.496,8.595,8.669,8.735,8.790,8.855,
     * 8.025,8.156,8.276,8.402,8.507,8.607,8.677,8.756,
     * 7.621,7.715,7.833,7.968,8.097,8.239,8.354,8.474,
     * 7.490,7.569,7.642,7.704,7.782,7.875,7.977,8.096,
     * 7.336,7.482,7.611,7.723,7.832,7.934,8.025,8.121,
     * 0.000,6.187,6.480,6.749,6.966,7.214,7.433,7.634,
     * 0.000,0.000,0.000,6.132,6.326,6.465,6.787,7.084/
C
      DATA ((EDEN(I,3,4,L,2),I=17,24),L=1,NZEN)/
     * 8.898,8.927,8.946,8.964,8.966,8.987,9.047,9.125,
     * 8.849,8.884,8.908,8.932,8.939,8.962,9.018,9.091,
     * 8.776,8.819,8.851,8.881,8.895,8.917,8.972,9.041,
     * 8.633,8.692,8.740,8.787,8.815,8.849,8.904,8.974,
     * 8.273,8.368,8.458,8.552,8.624,8.695,8.764,8.843,
     * 7.993,8.113,8.226,8.342,8.443,8.545,8.634,8.729,
     * 7.628,7.698,7.794,7.912,8.030,8.162,8.296,8.434,
     * 7.506,7.573,7.634,7.688,7.761,7.854,7.961,8.086,
     * 7.342,7.473,7.589,7.692,7.797,7.902,8.001,8.106,
     * 0.000,6.171,6.445,6.702,6.914,7.167,7.398,7.612,
     * 0.000,0.000,0.000,6.122,6.308,6.447,6.775,7.080/
C
      DATA ((EDEN(I,4,4,L,2),I=17,24),L=1,NZEN)/
     * 8.894,8.927,8.952,8.979,8.995,9.032,9.112,9.208,
     * 8.843,8.879,8.908,8.940,8.960,8.997,9.073,9.163,
     * 8.764,8.809,8.844,8.882,8.908,8.950,9.022,9.109,
     * 8.607,8.663,8.710,8.759,8.796,8.844,8.920,9.007,
     * 8.226,8.316,8.400,8.489,8.565,8.648,8.739,8.838,
     * 7.952,8.052,8.155,8.262,8.361,8.470,8.582,8.698,
     * 7.672,7.704,7.762,7.852,7.953,8.080,8.236,8.395,
     * 7.553,7.601,7.644,7.685,7.750,7.843,7.953,8.084,
     * 7.370,7.480,7.576,7.664,7.764,7.871,7.979,8.095,
     * 0.000,6.171,6.413,6.652,6.859,7.121,7.368,7.598,
     * 0.000,0.000,0.000,6.142,6.317,6.457,6.795,7.112/
C
      DATA ((EDEN(I,5,4,L,2),I=17,24),L=1,NZEN)/
     * 8.893,8.930,8.960,8.999,9.030,9.083,9.188,9.306,
     * 8.836,8.875,8.910,8.951,8.987,9.042,9.142,9.253,
     * 8.755,8.800,8.838,8.887,8.926,8.987,9.085,9.193,
     * 8.599,8.655,8.707,8.767,8.820,8.889,8.990,9.099,
     * 8.199,8.279,8.358,8.448,8.533,8.633,8.751,8.875,
     * 7.944,8.024,8.111,8.216,8.318,8.441,8.579,8.721,
     * 7.729,7.736,7.767,7.832,7.924,8.052,8.233,8.415,
     * 7.599,7.630,7.657,7.689,7.756,7.858,7.985,8.133,
     * 7.392,7.482,7.558,7.635,7.735,7.857,7.988,8.130,
     * 0.000,6.163,6.379,6.603,6.808,7.085,7.358,7.617,
     * 0.000,0.000,0.000,6.128,6.303,6.455,6.820,7.165/
C
      DATA ((EDEN(I,1,5,L,2),I=17,24),L=1,NZEN)/
     * 8.908,8.936,8.952,8.960,8.954,8.955,9.000,9.066,
     * 8.867,8.903,8.925,8.942,8.938,8.945,8.985,9.043,
     * 8.806,8.850,8.884,8.911,8.917,8.923,8.958,9.010,
     * 8.679,8.742,8.796,8.842,8.864,8.884,8.917,8.966,
     * 8.344,8.449,8.549,8.649,8.720,8.778,8.822,8.876,
     * 8.067,8.201,8.329,8.459,8.567,8.662,8.722,8.790,
     * 7.641,7.749,7.879,8.023,8.159,8.305,8.408,8.517,
     * 7.500,7.582,7.659,7.724,7.802,7.892,7.989,8.103,
     * 7.349,7.498,7.631,7.746,7.854,7.951,8.036,8.125,
     * 0.000,6.215,6.511,6.785,7.001,7.245,7.454,7.647,
     * 0.000,0.000,0.000,6.153,6.348,6.486,6.799,7.086/
C
      DATA ((EDEN(I,2,5,L,2),I=17,24),L=1,NZEN)/
     * 8.871,8.897,8.910,8.921,8.915,8.925,8.975,9.043,
     * 8.830,8.861,8.882,8.899,8.899,8.907,8.955,9.020,
     * 8.768,8.807,8.837,8.862,8.870,8.885,8.929,8.990,
     * 8.642,8.699,8.748,8.794,8.815,8.840,8.883,8.942,
     * 8.323,8.423,8.514,8.609,8.677,8.739,8.792,8.855,
     * 8.066,8.192,8.313,8.434,8.535,8.626,8.693,8.767,
     * 7.652,7.759,7.886,8.024,8.154,8.287,8.395,8.507,
     * 7.526,7.607,7.682,7.745,7.822,7.912,8.010,8.125,
     * 7.398,7.545,7.676,7.788,7.892,7.984,8.061,8.142,
     * 6.005,6.307,6.606,6.879,7.092,7.325,7.521,7.696,
     * 0.000,0.000,0.000,6.233,6.425,6.553,6.856,7.129/
C
      DATA ((EDEN(I,3,5,L,2),I=17,24),L=1,NZEN)/
     * 8.843,8.867,8.881,8.896,8.899,8.917,8.984,9.069,
     * 8.795,8.825,8.842,8.862,8.868,8.890,8.952,9.031,
     * 8.724,8.759,8.783,8.809,8.820,8.842,8.902,8.977,
     * 8.589,8.641,8.679,8.722,8.746,8.776,8.837,8.911,
     * 8.251,8.347,8.428,8.513,8.579,8.643,8.715,8.794,
     * 7.994,8.114,8.223,8.332,8.421,8.514,8.600,8.691,
     * 7.654,7.727,7.826,7.945,8.053,8.172,8.301,8.432,
     * 7.563,7.637,7.704,7.760,7.829,7.911,8.001,8.106,
     * 7.430,7.573,7.699,7.807,7.904,7.990,8.061,8.134,
     * 6.021,6.309,6.598,6.866,7.076,7.313,7.512,7.687,
     * 0.000,0.000,6.054,6.288,6.470,6.595,6.887,7.149/
C
      DATA ((EDEN(I,4,5,L,2),I=17,24),L=1,NZEN)/
     * 8.780,8.804,8.820,8.845,8.863,8.898,8.998,9.113,
     * 8.730,8.758,8.778,8.804,8.826,8.863,8.955,9.061,
     * 8.644,8.677,8.700,8.731,8.751,8.789,8.875,8.973,
     * 8.476,8.525,8.560,8.601,8.628,8.671,8.756,8.852,
     * 8.110,8.196,8.270,8.350,8.408,8.480,8.575,8.677,
     * 7.879,7.969,8.062,8.161,8.241,8.327,8.439,8.554,
     * 7.705,7.723,7.760,7.822,7.895,7.985,8.131,8.277,
     * 7.628,7.693,7.750,7.796,7.850,7.912,7.980,8.060,
     * 7.467,7.602,7.722,7.825,7.916,7.993,8.053,8.112,
     * 6.044,6.300,6.570,6.824,7.028,7.275,7.470,7.640,
     * 0.000,0.000,6.171,6.392,6.565,6.683,6.958,7.200/
C
      DATA ((EDEN(I,5,5,L,2),I=17,24),L=1,NZEN)/
     * 8.732,8.760,8.781,8.813,8.844,8.895,9.029,9.174,
     * 8.676,8.707,8.730,8.764,8.797,8.849,8.973,9.108,
     * 8.593,8.631,8.660,8.696,8.730,8.784,8.902,9.029,
     * 8.397,8.450,8.494,8.541,8.579,8.632,8.746,8.868,
     * 8.022,8.090,8.159,8.236,8.307,8.382,8.503,8.628,
     * 7.862,7.905,7.968,8.051,8.129,8.217,8.351,8.486,
     * 7.797,7.796,7.803,7.825,7.861,7.928,8.085,8.239,
     * 7.699,7.758,7.810,7.854,7.904,7.963,8.026,8.099,
     * 7.505,7.635,7.751,7.853,7.945,8.024,8.087,8.145,
     * 6.076,6.315,6.566,6.809,7.013,7.274,7.476,7.653,
     * 0.000,6.011,6.222,6.433,6.604,6.728,6.994,7.226/
C
      DATA ((EDEN(I,1,6,L,2),I=17,24),L=1,NZEN)/
     * 8.927,8.956,8.972,8.977,8.968,8.972,9.018,9.084,
     * 8.888,8.924,8.946,8.961,8.957,8.960,9.000,9.058,
     * 8.826,8.872,8.905,8.930,8.934,8.940,8.973,9.023,
     * 8.701,8.766,8.819,8.865,8.884,8.900,8.931,8.977,
     * 8.373,8.479,8.581,8.679,8.747,8.801,8.840,8.891,
     * 8.101,8.237,8.364,8.495,8.602,8.692,8.747,8.811,
     * 7.662,7.779,7.913,8.061,8.202,8.347,8.444,8.546,
     * 7.509,7.590,7.666,7.731,7.808,7.899,7.996,8.110,
     * 7.366,7.512,7.643,7.755,7.862,7.957,8.039,8.126,
     * 0.000,6.252,6.548,6.818,7.033,7.272,7.476,7.663,
     * 0.000,0.000,0.000,6.185,6.380,6.515,6.822,7.104/
C
      DATA ((EDEN(I,2,6,L,2),I=17,24),L=1,NZEN)/
     * 8.879,8.903,8.917,8.926,8.919,8.925,8.974,9.042,
     * 8.838,8.869,8.887,8.902,8.900,8.909,8.956,9.021,
     * 8.778,8.816,8.844,8.870,8.873,8.885,8.929,8.989,
     * 8.660,8.716,8.763,8.806,8.826,8.847,8.890,8.946,
     * 8.358,8.459,8.552,8.644,8.708,8.764,8.813,8.871,
     * 8.114,8.241,8.363,8.483,8.580,8.664,8.725,8.792,
     * 7.696,7.817,7.953,8.099,8.232,8.367,8.466,8.568,
     * 7.560,7.641,7.718,7.783,7.859,7.950,8.045,8.156,
     * 7.453,7.601,7.734,7.845,7.944,8.029,8.095,8.163,
     * 6.090,6.398,6.700,6.976,7.185,7.408,7.591,7.751,
     * 0.000,0.000,6.083,6.335,6.525,6.644,6.931,7.186/
C
      DATA ((EDEN(I,3,6,L,2),I=17,24),L=1,NZEN)/
     * 8.820,8.842,8.852,8.866,8.868,8.886,8.956,9.045,
     * 8.774,8.798,8.813,8.829,8.833,8.851,8.917,8.999,
     * 8.707,8.738,8.760,8.783,8.792,8.813,8.875,8.952,
     * 8.581,8.628,8.666,8.704,8.724,8.754,8.815,8.888,
     * 8.267,8.357,8.439,8.520,8.579,8.639,8.708,8.783,
     * 8.029,8.151,8.260,8.369,8.453,8.537,8.619,8.705,
     * 7.683,7.766,7.872,7.995,8.102,8.214,8.333,8.454,
     * 7.612,7.691,7.764,7.823,7.888,7.963,8.040,8.131,
     * 7.505,7.656,7.790,7.901,7.995,8.070,8.123,8.174,
     * 6.127,6.427,6.729,7.005,7.213,7.436,7.605,7.747,
     * 0.000,0.000,6.167,6.413,6.593,6.704,6.964,7.189/
C
      DATA ((EDEN(I,4,6,L,2),I=17,24),L=1,NZEN)/
     * 8.730,8.750,8.761,8.782,8.799,8.834,8.948,9.076,
     * 8.676,8.699,8.713,8.736,8.752,8.787,8.887,9.000,
     * 8.584,8.614,8.633,8.656,8.670,8.701,8.791,8.894,
     * 8.425,8.473,8.510,8.547,8.569,8.602,8.690,8.788,
     * 8.074,8.162,8.241,8.318,8.371,8.430,8.523,8.621,
     * 7.866,7.954,8.045,8.144,8.217,8.293,8.398,8.504,
     * 7.742,7.760,7.796,7.854,7.919,7.997,8.132,8.263,
     * 7.691,7.768,7.836,7.888,7.935,7.980,8.021,8.071,
     * 7.554,7.707,7.844,7.957,8.041,8.097,8.125,8.144,
     * 6.130,6.407,6.692,6.960,7.166,7.401,7.555,7.678,
     * 0.000,6.093,6.344,6.580,6.753,6.855,7.076,7.257/
C
      DATA ((EDEN(I,5,6,L,2),I=17,24),L=1,NZEN)/
     * 8.664,8.690,8.708,8.736,8.763,8.811,8.968,9.137,
     * 8.610,8.640,8.662,8.691,8.720,8.767,8.910,9.063,
     * 8.499,8.542,8.571,8.603,8.628,8.669,8.793,8.927,
     * 8.294,8.352,8.402,8.452,8.487,8.530,8.646,8.768,
     * 7.970,8.031,8.097,8.177,8.242,8.314,8.429,8.546,
     * 7.882,7.911,7.959,8.031,8.102,8.182,8.306,8.428,
     * 7.862,7.864,7.871,7.884,7.903,7.948,8.086,8.219,
     * 7.776,7.858,7.929,7.983,8.026,8.062,8.089,8.120,
     * 7.612,7.771,7.912,8.030,8.117,8.172,8.195,8.205,
     * 6.223,6.487,6.764,7.025,7.232,7.477,7.630,7.749,
     * 0.000,6.203,6.447,6.676,6.846,6.954,7.157,7.316/
C
      DATA ((EDEN(I,1,7,L,2),I=17,24),L=1,NZEN)/
     * 8.942,8.971,8.986,8.994,8.982,8.983,9.030,9.096,
     * 8.902,8.939,8.961,8.975,8.967,8.972,9.012,9.069,
     * 8.843,8.888,8.921,8.945,8.946,8.953,8.985,9.034,
     * 8.719,8.784,8.837,8.881,8.897,8.915,8.944,8.988,
     * 8.392,8.501,8.603,8.701,8.765,8.817,8.854,8.903,
     * 8.122,8.257,8.388,8.520,8.624,8.712,8.765,8.826,
     * 7.677,7.796,7.934,8.086,8.229,8.375,8.468,8.566,
     * 7.516,7.596,7.672,7.737,7.816,7.906,8.003,8.117,
     * 7.378,7.522,7.651,7.763,7.869,7.964,8.045,8.130,
     * 0.000,6.275,6.569,6.838,7.051,7.289,7.490,7.674,
     * 0.000,0.000,0.000,6.203,6.397,6.529,6.834,7.114/
C
      DATA ((EDEN(I,2,7,L,2),I=17,24),L=1,NZEN)/
     * 8.896,8.921,8.934,8.943,8.935,8.941,8.988,9.055,
     * 8.855,8.887,8.906,8.920,8.915,8.923,8.969,9.034,
     * 8.795,8.836,8.864,8.889,8.892,8.905,8.947,9.005,
     * 8.680,8.739,8.785,8.828,8.847,8.867,8.907,8.961,
     * 8.386,8.487,8.581,8.673,8.738,8.788,8.834,8.889,
     * 8.144,8.274,8.394,8.518,8.614,8.695,8.752,8.815,
     * 7.719,7.847,7.987,8.138,8.274,8.408,8.502,8.599,
     * 7.576,7.658,7.736,7.801,7.879,7.970,8.065,8.176,
     * 7.477,7.624,7.757,7.869,7.967,8.049,8.112,8.176,
     * 6.131,6.440,6.742,7.018,7.225,7.444,7.621,7.775,
     * 0.000,0.000,6.123,6.377,6.565,6.684,6.964,7.211/
C
      DATA ((EDEN(I,3,7,L,2),I=17,24),L=1,NZEN)/
     * 8.834,8.856,8.869,8.882,8.884,8.902,8.972,9.059,
     * 8.790,8.815,8.831,8.847,8.852,8.869,8.934,9.015,
     * 8.726,8.758,8.780,8.805,8.814,8.834,8.894,8.969,
     * 8.603,8.652,8.690,8.730,8.750,8.779,8.837,8.908,
     * 8.295,8.390,8.474,8.556,8.613,8.675,8.739,8.810,
     * 8.064,8.187,8.299,8.410,8.495,8.579,8.656,8.737,
     * 7.704,7.796,7.911,8.039,8.149,8.262,8.377,8.491,
     * 7.634,7.714,7.790,7.851,7.917,7.991,8.068,8.156,
     * 7.537,7.690,7.826,7.939,8.031,8.102,8.150,8.194,
     * 6.176,6.481,6.787,7.065,7.271,7.489,7.650,7.782,
     * 0.000,0.000,6.222,6.469,6.648,6.757,7.009,7.223/
C
      DATA ((EDEN(I,4,7,L,2),I=17,24),L=1,NZEN)/
     * 8.747,8.767,8.780,8.801,8.817,8.853,8.969,9.100,
     * 8.694,8.719,8.734,8.757,8.774,8.808,8.908,9.022,
     * 8.606,8.637,8.657,8.680,8.695,8.726,8.815,8.915,
     * 8.453,8.502,8.539,8.577,8.599,8.633,8.718,8.813,
     * 8.109,8.201,8.281,8.359,8.416,8.474,8.563,8.657,
     * 7.896,7.992,8.090,8.193,8.269,8.344,8.443,8.544,
     * 7.758,7.783,7.826,7.894,7.966,8.050,8.180,8.307,
     * 7.718,7.798,7.869,7.924,7.971,8.015,8.053,8.099,
     * 7.594,7.751,7.891,8.006,8.090,8.142,8.163,8.173,
     * 6.183,6.468,6.763,7.033,7.237,7.467,7.610,7.721,
     * 0.000,6.158,6.417,6.659,6.831,6.928,7.138,7.305/
C
      DATA ((EDEN(I,5,7,L,2),I=17,24),L=1,NZEN)/
     * 8.682,8.707,8.726,8.753,8.781,8.828,8.989,9.162,
     * 8.630,8.661,8.682,8.711,8.739,8.788,8.934,9.090,
     * 8.524,8.566,8.595,8.628,8.653,8.695,8.818,8.949,
     * 8.323,8.385,8.435,8.484,8.519,8.562,8.674,8.793,
     * 7.998,8.066,8.141,8.224,8.293,8.361,8.472,8.585,
     * 7.901,7.939,7.998,8.080,8.156,8.237,8.356,8.473,
     * 7.880,7.886,7.896,7.917,7.942,7.995,8.132,8.262,
     * 7.810,7.895,7.969,8.024,8.068,8.103,8.130,8.159,
     * 7.660,7.824,7.970,8.092,8.178,8.227,8.241,8.239,
     * 6.283,6.559,6.845,7.111,7.317,7.555,7.695,7.798,
     * 6.047,6.285,6.537,6.772,6.943,7.045,7.233,7.374/
C
      DATA ((EDEN(I,1,8,L,2),I=17,24),L=1,NZEN)/
     * 8.941,8.970,8.985,8.994,8.982,8.983,9.029,9.094,
     * 8.901,8.938,8.960,8.975,8.967,8.972,9.011,9.068,
     * 8.841,8.887,8.919,8.943,8.946,8.950,8.982,9.032,
     * 8.716,8.781,8.835,8.879,8.895,8.911,8.941,8.987,
     * 8.388,8.496,8.598,8.696,8.763,8.815,8.854,8.903,
     * 8.116,8.253,8.383,8.515,8.620,8.709,8.762,8.824,
     * 7.673,7.792,7.928,8.079,8.220,8.368,8.462,8.563,
     * 7.515,7.596,7.672,7.737,7.816,7.906,8.004,8.118,
     * 7.376,7.521,7.650,7.763,7.869,7.964,8.047,8.133,
     * 0.000,6.267,6.560,6.831,7.044,7.283,7.487,7.673,
     * 0.000,0.000,0.000,6.197,6.389,6.522,6.829,7.111/
C
      DATA ((EDEN(I,2,8,L,2),I=17,24),L=1,NZEN)/
     * 8.913,8.939,8.952,8.961,8.954,8.960,9.006,9.072,
     * 8.872,8.905,8.927,8.941,8.939,8.946,8.990,9.052,
     * 8.812,8.853,8.884,8.910,8.914,8.926,8.965,9.021,
     * 8.692,8.753,8.803,8.848,8.866,8.884,8.922,8.976,
     * 8.389,8.492,8.588,8.683,8.748,8.802,8.847,8.902,
     * 8.139,8.271,8.394,8.521,8.620,8.704,8.761,8.825,
     * 7.707,7.831,7.970,8.119,8.254,8.388,8.483,8.582,
     * 7.569,7.651,7.728,7.794,7.872,7.962,8.058,8.168,
     * 7.459,7.607,7.740,7.853,7.953,8.039,8.106,8.175,
     * 6.095,6.401,6.702,6.977,7.186,7.410,7.595,7.757,
     * 0.000,0.000,6.088,6.339,6.530,6.652,6.937,7.192/
C
      DATA ((EDEN(I,3,8,L,2),I=17,24),L=1,NZEN)/
     * 8.880,8.907,8.922,8.939,8.941,8.957,9.022,9.104,
     * 8.836,8.866,8.887,8.907,8.911,8.928,8.987,9.063,
     * 8.771,8.809,8.836,8.864,8.873,8.893,8.948,9.018,
     * 8.646,8.700,8.743,8.786,8.811,8.838,8.892,8.959,
     * 8.330,8.430,8.517,8.606,8.670,8.731,8.792,8.861,
     * 8.081,8.207,8.325,8.439,8.531,8.619,8.693,8.771,
     * 7.710,7.806,7.925,8.059,8.177,8.302,8.416,8.531,
     * 7.624,7.703,7.778,7.839,7.909,7.990,8.074,8.171,
     * 7.519,7.668,7.801,7.913,8.009,8.088,8.146,8.203,
     * 6.150,6.452,6.752,7.025,7.233,7.456,7.630,7.777,
     * 0.000,0.000,6.175,6.420,6.600,6.712,6.980,7.213/
C
      DATA ((EDEN(I,4,8,L,2),I=17,24),L=1,NZEN)/
     * 8.829,8.856,8.875,8.899,8.917,8.951,9.053,9.170,
     * 8.781,8.810,8.833,8.861,8.881,8.917,9.004,9.105,
     * 8.707,8.744,8.770,8.803,8.825,8.862,8.940,9.031,
     * 8.553,8.606,8.646,8.691,8.720,8.760,8.836,8.922,
     * 8.206,8.302,8.386,8.470,8.533,8.601,8.685,8.773,
     * 7.970,8.081,8.188,8.298,8.383,8.469,8.566,8.665,
     * 7.753,7.789,7.850,7.940,8.031,8.130,8.263,8.394,
     * 7.703,7.777,7.845,7.900,7.956,8.016,8.076,8.146,
     * 7.576,7.724,7.856,7.967,8.055,8.120,8.159,8.193,
     * 6.189,6.470,6.752,7.019,7.224,7.454,7.620,7.756,
     * 0.000,6.123,6.373,6.608,6.781,6.888,7.129,7.331/
C
      DATA ((EDEN(I,5,8,L,2),I=17,24),L=1,NZEN)/
     * 8.788,8.816,8.838,8.870,8.902,8.954,9.092,9.243,
     * 8.741,8.773,8.799,8.833,8.866,8.920,9.044,9.179,
     * 8.663,8.702,8.733,8.771,8.804,8.858,8.967,9.085,
     * 8.476,8.534,8.580,8.628,8.665,8.716,8.818,8.927,
     * 8.121,8.210,8.293,8.382,8.449,8.525,8.632,8.740,
     * 7.936,8.010,8.100,8.203,8.291,8.382,8.500,8.617,
     * 7.850,7.861,7.887,7.933,7.994,8.080,8.226,8.367,
     * 7.788,7.862,7.928,7.980,8.031,8.084,8.135,8.192,
     * 7.645,7.793,7.924,8.035,8.123,8.188,8.227,8.256,
     * 6.270,6.535,6.808,7.070,7.275,7.515,7.682,7.816,
     * 6.013,6.239,6.476,6.701,6.871,6.982,7.206,7.388/
C
      DATA ((EDEN(I,1,9,L,2),I=17,24),L=1,NZEN)/
     * 8.929,8.959,8.974,8.981,8.972,8.975,9.019,9.083,
     * 8.889,8.925,8.947,8.963,8.957,8.963,9.002,9.059,
     * 8.827,8.874,8.907,8.933,8.934,8.943,8.976,9.026,
     * 8.701,8.767,8.821,8.866,8.884,8.904,8.935,8.982,
     * 8.368,8.475,8.577,8.676,8.745,8.802,8.843,8.896,
     * 8.094,8.230,8.359,8.490,8.597,8.690,8.747,8.812,
     * 7.657,7.771,7.903,8.052,8.193,8.339,8.438,8.543,
     * 7.510,7.592,7.668,7.734,7.813,7.904,8.002,8.116,
     * 7.364,7.511,7.643,7.757,7.864,7.962,8.046,8.135,
     * 0.000,6.240,6.534,6.806,7.021,7.263,7.471,7.662,
     * 0.000,0.000,0.000,6.170,6.366,6.501,6.813,7.101/
C
      DATA ((EDEN(I,2,9,L,2),I=17,24),L=1,NZEN)/
     * 8.926,8.955,8.972,8.981,8.976,8.981,9.026,9.091,
     * 8.887,8.924,8.947,8.965,8.961,8.970,9.011,9.071,
     * 8.823,8.869,8.903,8.931,8.934,8.950,8.987,9.040,
     * 8.698,8.762,8.814,8.861,8.883,8.902,8.940,8.992,
     * 8.368,8.475,8.575,8.674,8.744,8.802,8.849,8.906,
     * 8.105,8.241,8.366,8.494,8.598,8.689,8.750,8.818,
     * 7.677,7.790,7.922,8.068,8.205,8.346,8.448,8.555,
     * 7.540,7.622,7.699,7.764,7.842,7.932,8.030,8.142,
     * 7.408,7.555,7.687,7.800,7.905,7.999,8.077,8.158,
     * 6.008,6.306,6.605,6.878,7.090,7.325,7.524,7.703,
     * 0.000,0.000,0.000,6.237,6.430,6.559,6.859,7.133/
C
      DATA ((EDEN(I,3,9,L,2),I=17,24),L=1,NZEN)/
     * 8.945,8.978,8.998,9.012,9.015,9.032,9.089,9.163,
     * 8.901,8.941,8.967,8.989,8.996,9.012,9.062,9.129,
     * 8.835,8.882,8.916,8.948,8.959,8.980,9.027,9.089,
     * 8.700,8.763,8.813,8.862,8.886,8.914,8.962,9.024,
     * 8.359,8.463,8.561,8.657,8.727,8.790,8.850,8.917,
     * 8.094,8.226,8.347,8.472,8.575,8.668,8.742,8.822,
     * 7.697,7.791,7.910,8.046,8.173,8.311,8.428,8.548,
     * 7.585,7.660,7.731,7.791,7.866,7.954,8.050,8.161,
     * 7.451,7.593,7.719,7.828,7.930,8.021,8.099,8.178,
     * 6.052,6.340,6.627,6.892,7.103,7.340,7.543,7.724,
     * 0.000,0.000,6.089,6.324,6.510,6.638,6.932,7.197/
C
      DATA ((EDEN(I,4,9,L,2),I=17,24),L=1,NZEN)/
     * 8.976,9.014,9.041,9.068,9.080,9.111,9.188,9.280,
     * 8.933,8.975,9.008,9.039,9.056,9.089,9.158,9.240,
     * 8.861,8.911,8.950,8.988,9.009,9.042,9.103,9.178,
     * 8.720,8.785,8.841,8.896,8.930,8.970,9.032,9.104,
     * 8.358,8.461,8.555,8.652,8.728,8.803,8.877,8.957,
     * 8.090,8.211,8.333,8.457,8.562,8.666,8.756,8.849,
     * 7.753,7.816,7.912,8.032,8.154,8.287,8.422,8.558,
     * 7.659,7.722,7.783,7.836,7.905,7.990,8.085,8.194,
     * 7.518,7.646,7.761,7.862,7.960,8.053,8.135,8.218,
     * 6.124,6.386,6.654,6.906,7.112,7.355,7.564,7.750,
     * 0.000,0.000,6.221,6.442,6.619,6.744,7.041,7.309/
C
      DATA ((EDEN(I,5,9,L,2),I=17,24),L=1,NZEN)/
     * 9.011,9.053,9.087,9.124,9.151,9.198,9.298,9.409,
     * 8.965,9.013,9.051,9.093,9.127,9.171,9.262,9.363,
     * 8.899,8.953,8.999,9.049,9.085,9.136,9.219,9.311,
     * 8.757,8.827,8.888,8.951,9.001,9.059,9.140,9.228,
     * 8.381,8.482,8.579,8.683,8.770,8.862,8.955,9.050,
     * 8.116,8.233,8.350,8.477,8.592,8.714,8.823,8.934,
     * 7.822,7.867,7.944,8.057,8.182,8.325,8.479,8.632,
     * 7.728,7.781,7.833,7.883,7.956,8.052,8.161,8.284,
     * 7.576,7.692,7.794,7.890,7.991,8.095,8.193,8.292,
     * 6.202,6.441,6.692,6.935,7.143,7.397,7.626,7.831,
     * 0.000,6.081,6.290,6.493,6.669,6.804,7.115,7.395/
C
      DATA ((EDEN(I,1,10,L,2),I=17,24),L=1,NZEN)/
     * 8.925,8.954,8.970,8.981,8.972,8.975,9.017,9.079,
     * 8.884,8.920,8.944,8.960,8.957,8.963,9.001,9.057,
     * 8.822,8.868,8.902,8.930,8.934,8.943,8.976,9.027,
     * 8.693,8.759,8.814,8.863,8.884,8.904,8.936,8.984,
     * 8.354,8.464,8.565,8.668,8.739,8.798,8.840,8.894,
     * 8.077,8.212,8.344,8.477,8.587,8.684,8.743,8.809,
     * 7.648,7.757,7.888,8.034,8.176,8.326,8.427,8.534,
     * 7.506,7.590,7.668,7.735,7.815,7.906,8.004,8.119,
     * 7.356,7.505,7.639,7.755,7.865,7.965,8.051,8.142,
     * 0.000,6.215,6.513,6.786,7.002,7.249,7.462,7.658,
     * 0.000,0.000,0.000,6.151,6.347,6.482,6.800,7.093/
C
      DATA ((EDEN(I,2,10,L,2),I=17,24),L=1,NZEN)/
     * 8.947,8.978,8.997,9.006,9.000,9.006,9.052,9.117,
     * 8.905,8.944,8.970,8.986,8.988,8.995,9.034,9.092,
     * 8.841,8.890,8.925,8.954,8.962,8.972,9.007,9.059,
     * 8.707,8.775,8.832,8.881,8.904,8.925,8.960,9.010,
     * 8.356,8.465,8.569,8.671,8.746,8.806,8.853,8.910,
     * 8.067,8.202,8.332,8.464,8.575,8.673,8.738,8.811,
     * 7.645,7.745,7.870,8.012,8.150,8.298,8.407,8.523,
     * 7.506,7.587,7.661,7.726,7.804,7.895,7.994,8.109,
     * 7.344,7.490,7.621,7.735,7.846,7.948,8.039,8.135,
     * 0.000,6.179,6.471,6.743,6.959,7.211,7.433,7.637,
     * 0.000,0.000,0.000,6.120,6.315,6.457,6.778,7.076/
C
      DATA ((EDEN(I,3,10,L,2),I=17,24),L=1,NZEN)/
     * 9.011,9.048,9.070,9.087,9.085,9.099,9.149,9.217,
     * 8.966,9.010,9.037,9.057,9.063,9.080,9.124,9.184,
     * 8.900,8.953,8.991,9.023,9.034,9.052,9.092,9.147,
     * 8.761,8.832,8.889,8.939,8.965,8.993,9.035,9.090,
     * 8.392,8.504,8.610,8.712,8.787,8.852,8.907,8.971,
     * 8.097,8.228,8.359,8.490,8.600,8.700,8.774,8.856,
     * 7.681,7.770,7.887,8.025,8.162,8.313,8.434,8.559,
     * 7.535,7.605,7.670,7.729,7.806,7.902,8.008,8.132,
     * 7.366,7.497,7.613,7.717,7.825,7.931,8.032,8.139,
     * 0.000,6.195,6.463,6.720,6.933,7.188,7.421,7.639,
     * 0.000,0.000,0.000,6.132,6.319,6.462,6.788,7.092/
C
      DATA ((EDEN(I,4,10,L,2),I=17,24),L=1,NZEN)/
     * 9.147,9.190,9.216,9.234,9.244,9.275,9.327,9.394,
     * 9.103,9.152,9.184,9.208,9.220,9.252,9.298,9.359,
     * 9.037,9.095,9.138,9.170,9.186,9.221,9.265,9.321,
     * 8.904,8.981,9.040,9.089,9.118,9.160,9.205,9.261,
     * 8.529,8.645,8.753,8.853,8.925,8.994,9.055,9.124,
     * 8.242,8.377,8.514,8.650,8.760,8.863,8.940,9.024,
     * 7.787,7.882,8.007,8.157,8.309,8.471,8.595,8.724,
     * 7.614,7.668,7.720,7.772,7.852,7.958,8.082,8.224,
     * 7.442,7.549,7.644,7.735,7.841,7.956,8.072,8.198,
     * 6.043,6.283,6.527,6.765,6.974,7.234,7.482,7.716,
     * 0.000,0.000,0.000,6.191,6.371,6.522,6.866,7.191/
C
      DATA ((EDEN(I,5,10,L,2),I=17,24),L=1,NZEN)/
     * 9.301,9.348,9.378,9.409,9.437,9.489,9.546,9.614,
     * 9.259,9.309,9.346,9.382,9.410,9.464,9.514,9.576,
     * 9.196,9.256,9.300,9.340,9.374,9.429,9.476,9.534,
     * 9.073,9.151,9.212,9.266,9.311,9.371,9.421,9.480,
     * 8.716,8.840,8.950,9.049,9.125,9.210,9.276,9.349,
     * 8.418,8.566,8.709,8.847,8.959,9.070,9.154,9.243,
     * 7.930,8.053,8.203,8.382,8.560,8.735,8.859,8.986,
     * 7.688,7.731,7.777,7.832,7.926,8.056,8.209,8.380,
     * 7.510,7.598,7.676,7.761,7.875,8.011,8.158,8.314,
     * 6.157,6.374,6.596,6.823,7.038,7.317,7.590,7.852,
     * 0.000,0.000,6.057,6.245,6.434,6.613,6.986,7.344/
C
      DATA ((EDEN(I,1,11,L,2),I=17,24),L=1,NZEN)/
     * 8.933,8.965,8.982,8.994,8.985,8.991,9.031,9.090,
     * 8.892,8.931,8.956,8.975,8.971,8.978,9.013,9.067,
     * 8.829,8.878,8.914,8.943,8.950,8.959,8.991,9.039,
     * 8.700,8.770,8.827,8.876,8.899,8.922,8.953,8.999,
     * 8.360,8.471,8.577,8.681,8.755,8.818,8.858,8.909,
     * 8.081,8.220,8.352,8.489,8.603,8.703,8.759,8.823,
     * 7.652,7.760,7.893,8.042,8.189,8.341,8.441,8.545,
     * 7.510,7.595,7.674,7.743,7.825,7.917,8.015,8.128,
     * 7.358,7.509,7.645,7.764,7.876,7.978,8.065,8.157,
     * 0.000,6.215,6.511,6.786,7.005,7.256,7.470,7.667,
     * 0.000,0.000,0.000,6.147,6.344,6.482,6.803,7.098/
C
      DATA ((EDEN(I,2,11,L,2),I=17,24),L=1,NZEN)/
     * 8.971,9.004,9.025,9.037,9.029,9.035,9.081,9.146,
     * 8.924,8.965,8.991,9.009,9.007,9.015,9.054,9.111,
     * 8.863,8.916,8.955,8.985,8.991,9.007,9.039,9.088,
     * 8.727,8.799,8.858,8.908,8.931,8.957,8.989,9.037,
     * 8.353,8.467,8.575,8.681,8.759,8.823,8.868,8.926,
     * 8.041,8.177,8.311,8.448,8.563,8.668,8.733,8.808,
     * 7.625,7.715,7.834,7.972,8.113,8.266,8.380,8.500,
     * 7.477,7.557,7.630,7.696,7.777,7.871,7.973,8.093,
     * 7.293,7.438,7.567,7.682,7.798,7.908,8.010,8.119,
     * 0.000,6.060,6.346,6.620,6.844,7.118,7.358,7.584,
     * 0.000,0.000,0.000,0.000,6.185,6.335,6.676,6.997/
C
      DATA ((EDEN(I,3,11,L,2),I=17,24),L=1,NZEN)/
     * 9.061,9.100,9.122,9.135,9.131,9.144,9.192,9.257,
     * 9.014,9.059,9.086,9.104,9.106,9.119,9.161,9.219,
     * 8.944,8.998,9.036,9.064,9.069,9.088,9.126,9.179,
     * 8.809,8.884,8.945,8.993,9.017,9.046,9.083,9.135,
     * 8.414,8.529,8.637,8.739,8.814,8.879,8.933,8.998,
     * 8.115,8.252,8.386,8.520,8.633,8.735,8.807,8.889,
     * 7.665,7.748,7.861,7.995,8.136,8.290,8.415,8.548,
     * 7.490,7.553,7.611,7.664,7.742,7.843,7.959,8.096,
     * 7.291,7.411,7.516,7.614,7.724,7.841,7.959,8.089,
     * 0.000,6.055,6.306,6.552,6.766,7.038,7.298,7.549,
     * 0.000,0.000,0.000,0.000,6.107,6.268,6.625,6.967/
C
      DATA ((EDEN(I,4,11,L,2),I=17,24),L=1,NZEN)/
     * 9.271,9.311,9.331,9.345,9.349,9.375,9.413,9.467,
     * 9.227,9.271,9.298,9.316,9.323,9.349,9.383,9.432,
     * 9.159,9.213,9.247,9.273,9.287,9.315,9.347,9.394,
     * 9.027,9.100,9.151,9.193,9.219,9.255,9.292,9.341,
     * 8.650,8.771,8.878,8.965,9.031,9.095,9.149,9.214,
     * 8.347,8.495,8.635,8.765,8.866,8.962,9.034,9.116,
     * 7.826,7.946,8.092,8.255,8.422,8.586,8.703,8.828,
     * 7.563,7.606,7.648,7.695,7.780,7.898,8.042,8.209,
     * 7.356,7.445,7.521,7.602,7.711,7.841,7.984,8.142,
     * 0.000,6.174,6.397,6.622,6.835,7.114,7.388,7.657,
     * 0.000,0.000,0.000,0.000,6.137,6.312,6.693,7.065/
C
      DATA ((EDEN(I,5,11,L,2),I=17,24),L=1,NZEN)/
     * 9.503,9.545,9.575,9.605,9.629,9.667,9.700,9.743,
     * 9.460,9.505,9.540,9.572,9.598,9.639,9.667,9.707,
     * 9.398,9.449,9.491,9.525,9.556,9.600,9.628,9.666,
     * 9.275,9.345,9.396,9.444,9.483,9.536,9.569,9.612,
     * 8.941,9.055,9.149,9.230,9.302,9.384,9.437,9.499,
     * 8.648,8.796,8.925,9.039,9.137,9.243,9.315,9.396,
     * 8.082,8.247,8.424,8.605,8.765,8.925,9.037,9.155,
     * 7.642,7.682,7.726,7.783,7.890,8.040,8.220,8.422,
     * 7.422,7.493,7.556,7.634,7.755,7.910,8.086,8.278,
     * 6.096,6.289,6.487,6.700,6.925,7.229,7.528,7.826,
     * 0.000,0.000,0.000,0.000,6.193,6.406,6.822,7.236/
C
      DATA ((EDEN(I,1,12,L,2),I=17,24),L=1,NZEN)/
     * 8.948,8.981,9.001,9.013,9.005,9.012,9.049,9.105,
     * 8.907,8.948,8.976,8.992,8.992,9.001,9.033,9.084,
     * 8.844,8.896,8.934,8.963,8.969,8.984,9.012,9.057,
     * 8.716,8.788,8.847,8.897,8.921,8.944,8.972,9.015,
     * 8.375,8.490,8.599,8.706,8.780,8.842,8.879,8.928,
     * 8.095,8.237,8.374,8.515,8.630,8.731,8.783,8.843,
     * 7.660,7.774,7.909,8.064,8.214,8.373,8.468,8.568,
     * 7.517,7.602,7.683,7.756,7.839,7.932,8.029,8.141,
     * 7.368,7.519,7.656,7.777,7.892,7.994,8.080,8.170,
     * 0.000,6.227,6.523,6.800,7.019,7.277,7.489,7.684,
     * 0.000,0.000,0.000,6.156,6.352,6.494,6.814,7.110/
C
      DATA ((EDEN(I,2,12,L,2),I=17,24),L=1,NZEN)/
     * 8.989,9.026,9.048,9.059,9.054,9.058,9.104,9.170,
     * 8.944,8.986,9.013,9.031,9.029,9.038,9.076,9.133,
     * 8.880,8.934,8.973,9.002,9.006,9.021,9.053,9.103,
     * 8.743,8.816,8.879,8.930,8.955,8.981,9.011,9.056,
     * 8.358,8.475,8.587,8.696,8.773,8.840,8.884,8.940,
     * 8.065,8.205,8.343,8.481,8.598,8.702,8.764,8.835,
     * 7.618,7.707,7.824,7.964,8.107,8.263,8.375,8.495,
     * 7.464,7.544,7.619,7.687,7.770,7.866,7.971,8.093,
     * 7.274,7.416,7.543,7.659,7.778,7.892,7.998,8.111,
     * 0.000,0.000,6.277,6.551,6.788,7.080,7.329,7.565,
     * 0.000,0.000,0.000,0.000,6.107,6.264,6.618,6.954/
C
      DATA ((EDEN(I,3,12,L,2),I=17,24),L=1,NZEN)/
     * 9.082,9.122,9.145,9.154,9.150,9.162,9.211,9.278,
     * 9.035,9.079,9.107,9.125,9.123,9.135,9.177,9.236,
     * 8.964,9.018,9.055,9.080,9.085,9.100,9.138,9.191,
     * 8.829,8.905,8.962,9.008,9.030,9.058,9.095,9.147,
     * 8.441,8.559,8.669,8.769,8.842,8.907,8.959,9.024,
     * 8.111,8.247,8.383,8.516,8.629,8.731,8.806,8.891,
     * 7.653,7.748,7.870,8.012,8.164,8.330,8.452,8.584,
     * 7.457,7.515,7.567,7.617,7.695,7.798,7.920,8.066,
     * 7.239,7.351,7.447,7.539,7.650,7.773,7.903,8.048,
     * 0.000,0.000,6.221,6.472,6.689,6.970,7.243,7.510,
     * 0.000,0.000,0.000,0.000,0.000,6.140,6.516,6.880/
C
      DATA ((EDEN(I,4,12,L,2),I=17,24),L=1,NZEN)/
     * 9.313,9.346,9.364,9.375,9.378,9.396,9.433,9.485,
     * 9.266,9.306,9.327,9.342,9.347,9.366,9.399,9.447,
     * 9.197,9.244,9.274,9.296,9.305,9.329,9.361,9.406,
     * 9.059,9.127,9.174,9.208,9.231,9.262,9.298,9.348,
     * 8.682,8.799,8.898,8.981,9.040,9.106,9.162,9.230,
     * 8.364,8.507,8.641,8.764,8.861,8.953,9.030,9.117,
     * 7.824,7.952,8.102,8.275,8.443,8.605,8.724,8.854,
     * 7.498,7.535,7.572,7.618,7.708,7.838,7.997,8.183,
     * 7.266,7.345,7.411,7.486,7.599,7.740,7.901,8.082,
     * 0.000,6.071,6.277,6.493,6.708,6.999,7.291,7.585,
     * 0.000,0.000,0.000,0.000,0.000,6.156,6.560,6.963/
C
      DATA ((EDEN(I,5,12,L,2),I=17,24),L=1,NZEN)/
     * 9.563,9.602,9.631,9.653,9.670,9.699,9.730,9.772,
     * 9.517,9.558,9.590,9.614,9.634,9.666,9.694,9.733,
     * 9.451,9.500,9.535,9.563,9.587,9.624,9.650,9.689,
     * 9.325,9.385,9.430,9.469,9.503,9.550,9.584,9.629,
     * 8.979,9.083,9.161,9.234,9.299,9.378,9.435,9.504,
     * 8.683,8.820,8.937,9.040,9.131,9.238,9.315,9.404,
     * 8.104,8.267,8.439,8.606,8.759,8.918,9.035,9.163,
     * 7.574,7.604,7.638,7.689,7.800,7.962,8.162,8.390,
     * 7.302,7.357,7.403,7.472,7.596,7.766,7.967,8.192,
     * 0.000,6.167,6.345,6.544,6.773,7.089,7.413,7.745,
     * 0.000,0.000,0.000,0.000,0.000,6.207,6.654,7.109/
C
      DATA ((EDEN(I,1,1,L,3),I=17,24),L=1,NZEN)/
     * 9.068,9.107,9.131,9.142,9.137,9.147,9.179,9.230,
     * 9.036,9.085,9.119,9.137,9.137,9.149,9.174,9.218,
     * 8.984,9.047,9.096,9.128,9.132,9.147,9.167,9.202,
     * 8.870,8.959,9.033,9.089,9.114,9.137,9.153,9.182,
     * 8.558,8.701,8.840,8.964,9.042,9.098,9.116,9.143,
     * 8.294,8.469,8.642,8.813,8.936,9.027,9.056,9.089,
     * 7.843,8.003,8.186,8.386,8.565,8.727,8.790,8.855,
     * 7.722,7.852,7.982,8.095,8.196,8.280,8.347,8.417,
     * 7.630,7.836,8.034,8.204,8.334,8.418,8.457,8.484,
     * 6.234,6.594,6.958,7.284,7.522,7.761,7.914,8.037,
     * 0.000,6.040,6.385,6.698,6.918,7.044,7.296,7.507/
C
      DATA ((EDEN(I,2,1,L,3),I=17,24),L=1,NZEN)/
     * 9.104,9.144,9.168,9.179,9.175,9.185,9.229,9.293,
     * 9.066,9.118,9.149,9.167,9.166,9.177,9.212,9.264,
     * 9.015,9.079,9.126,9.153,9.161,9.175,9.201,9.243,
     * 8.892,8.984,9.057,9.113,9.140,9.167,9.186,9.219,
     * 8.535,8.681,8.818,8.942,9.023,9.087,9.114,9.149,
     * 8.257,8.429,8.599,8.766,8.889,8.986,9.026,9.072,
     * 7.781,7.917,8.081,8.264,8.436,8.600,8.684,8.771,
     * 7.654,7.777,7.899,8.005,8.104,8.192,8.267,8.349,
     * 7.514,7.710,7.897,8.060,8.194,8.291,8.353,8.409,
     * 0.000,6.327,6.677,7.003,7.261,7.538,7.732,7.901,
     * 0.000,0.000,6.098,6.399,6.625,6.771,7.065,7.327/
C
      DATA ((EDEN(I,3,1,L,3),I=17,24),L=1,NZEN)/
     * 9.189,9.234,9.257,9.270,9.269,9.285,9.333,9.398,
     * 9.151,9.203,9.233,9.252,9.254,9.269,9.308,9.363,
     * 9.090,9.153,9.197,9.222,9.229,9.247,9.279,9.326,
     * 8.970,9.057,9.125,9.172,9.197,9.222,9.251,9.295,
     * 8.609,8.751,8.881,8.993,9.066,9.128,9.167,9.216,
     * 8.292,8.455,8.622,8.777,8.894,8.990,9.046,9.110,
     * 7.801,7.942,8.110,8.297,8.472,8.638,8.737,8.841,
     * 7.611,7.710,7.805,7.891,7.985,8.084,8.185,8.300,
     * 7.434,7.595,7.745,7.880,8.008,8.120,8.215,8.313,
     * 0.000,6.235,6.552,6.849,7.098,7.367,7.595,7.808,
     * 0.000,0.000,0.000,6.196,6.417,6.577,6.905,7.210/
C
      DATA ((EDEN(I,4,1,L,3),I=17,24),L=1,NZEN)/
     * 9.398,9.436,9.457,9.470,9.475,9.497,9.535,9.589,
     * 9.358,9.401,9.427,9.445,9.452,9.476,9.508,9.554,
     * 9.297,9.348,9.384,9.408,9.421,9.448,9.477,9.519,
     * 9.171,9.244,9.299,9.339,9.364,9.397,9.429,9.473,
     * 8.816,8.950,9.061,9.150,9.212,9.274,9.323,9.381,
     * 8.510,8.676,8.830,8.960,9.058,9.149,9.215,9.289,
     * 7.956,8.120,8.310,8.508,8.684,8.839,8.943,9.055,
     * 7.612,7.684,7.757,7.834,7.943,8.076,8.225,8.394,
     * 7.409,7.528,7.639,7.751,7.880,8.017,8.154,8.301,
     * 6.029,6.290,6.541,6.794,7.028,7.310,7.571,7.826,
     * 0.000,0.000,0.000,6.093,6.307,6.501,6.870,7.227/
C
      DATA ((EDEN(I,5,1,L,3),I=17,24),L=1,NZEN)/
     * 9.623,9.663,9.692,9.717,9.736,9.769,9.804,9.851,
     * 9.578,9.622,9.655,9.683,9.705,9.739,9.770,9.812,
     * 9.517,9.566,9.605,9.636,9.663,9.702,9.731,9.771,
     * 9.397,9.462,9.510,9.553,9.591,9.639,9.673,9.718,
     * 9.062,9.172,9.262,9.338,9.406,9.486,9.543,9.609,
     * 8.774,8.921,9.047,9.153,9.250,9.359,9.435,9.519,
     * 8.189,8.375,8.565,8.742,8.900,9.057,9.171,9.294,
     * 7.650,7.707,7.765,7.838,7.962,8.127,8.322,8.541,
     * 7.386,7.472,7.550,7.646,7.785,7.954,8.143,8.350,
     * 6.082,6.289,6.498,6.725,6.969,7.292,7.598,7.909,
     * 0.000,0.000,0.000,0.000,6.185,6.423,6.850,7.280/
C
      DATA ((EDEN(I,1,2,L,3),I=17,24),L=1,NZEN)/
     * 9.052,9.091,9.113,9.125,9.120,9.128,9.162,9.215,
     * 9.020,9.069,9.100,9.122,9.122,9.130,9.158,9.204,
     * 8.967,9.030,9.075,9.108,9.116,9.130,9.151,9.189,
     * 8.853,8.941,9.014,9.073,9.098,9.120,9.137,9.168,
     * 8.540,8.679,8.816,8.940,9.019,9.076,9.098,9.127,
     * 8.276,8.446,8.617,8.785,8.910,9.002,9.034,9.070,
     * 7.830,7.985,8.163,8.356,8.535,8.694,8.763,8.832,
     * 7.714,7.843,7.971,8.081,8.181,8.265,8.332,8.404,
     * 7.618,7.825,8.022,8.190,8.319,8.403,8.443,8.471,
     * 6.217,6.580,6.941,7.266,7.504,7.742,7.898,8.022,
     * 0.000,6.025,6.372,6.684,6.904,7.026,7.280,7.492/
C
      DATA ((EDEN(I,2,2,L,3),I=17,24),L=1,NZEN)/
     * 9.074,9.115,9.137,9.151,9.147,9.159,9.204,9.268,
     * 9.037,9.087,9.117,9.138,9.140,9.150,9.186,9.239,
     * 8.986,9.049,9.096,9.130,9.139,9.153,9.179,9.222,
     * 8.861,8.949,9.022,9.082,9.106,9.132,9.154,9.191,
     * 8.511,8.651,8.786,8.911,8.993,9.054,9.084,9.123,
     * 8.237,8.403,8.571,8.736,8.860,8.958,9.000,9.049,
     * 7.769,7.904,8.066,8.248,8.419,8.580,8.667,8.757,
     * 7.646,7.768,7.888,7.991,8.090,8.178,8.255,8.339,
     * 7.509,7.707,7.894,8.057,8.192,8.290,8.352,8.408,
     * 6.006,6.356,6.706,7.028,7.276,7.547,7.738,7.904,
     * 0.000,0.000,6.132,6.433,6.657,6.801,7.091,7.347/
C
      DATA ((EDEN(I,3,2,L,3),I=17,24),L=1,NZEN)/
     * 9.146,9.187,9.213,9.228,9.229,9.251,9.300,9.367,
     * 9.105,9.158,9.186,9.211,9.215,9.236,9.277,9.335,
     * 9.044,9.105,9.147,9.179,9.191,9.214,9.249,9.299,
     * 8.919,9.006,9.075,9.127,9.156,9.189,9.221,9.266,
     * 8.563,8.701,8.830,8.945,9.022,9.089,9.131,9.182,
     * 8.256,8.415,8.577,8.735,8.856,8.957,9.016,9.081,
     * 7.781,7.917,8.080,8.262,8.437,8.605,8.706,8.812,
     * 7.603,7.703,7.800,7.888,7.984,8.085,8.186,8.301,
     * 7.437,7.602,7.755,7.893,8.022,8.134,8.227,8.321,
     * 0.000,6.233,6.552,6.853,7.102,7.371,7.599,7.808,
     * 0.000,0.000,0.000,6.239,6.454,6.620,6.945,7.245/
C
      DATA ((EDEN(I,4,2,L,3),I=17,24),L=1,NZEN)/
     * 9.300,9.341,9.367,9.386,9.397,9.431,9.476,9.535,
     * 9.257,9.306,9.337,9.360,9.375,9.409,9.447,9.500,
     * 9.192,9.254,9.291,9.324,9.343,9.379,9.415,9.464,
     * 9.065,9.145,9.205,9.254,9.286,9.330,9.368,9.417,
     * 8.707,8.842,8.959,9.060,9.131,9.205,9.258,9.321,
     * 8.403,8.566,8.722,8.865,8.975,9.075,9.145,9.224,
     * 7.872,8.018,8.187,8.380,8.559,8.727,8.840,8.959,
     * 7.593,7.665,7.739,7.815,7.921,8.051,8.196,8.358,
     * 7.400,7.522,7.634,7.746,7.874,8.008,8.142,8.284,
     * 6.016,6.274,6.533,6.793,7.025,7.307,7.566,7.816,
     * 0.000,0.000,0.000,6.124,6.335,6.519,6.884,7.234/
C
      DATA ((EDEN(I,5,2,L,3),I=17,24),L=1,NZEN)/
     * 9.466,9.511,9.542,9.575,9.604,9.652,9.698,9.757,
     * 9.422,9.471,9.506,9.542,9.572,9.622,9.664,9.717,
     * 9.359,9.414,9.456,9.495,9.531,9.583,9.623,9.673,
     * 9.233,9.306,9.360,9.413,9.458,9.519,9.563,9.618,
     * 8.880,9.004,9.104,9.198,9.277,9.369,9.434,9.507,
     * 8.579,8.739,8.881,9.008,9.119,9.235,9.319,9.411,
     * 8.012,8.182,8.368,8.565,8.743,8.918,9.041,9.172,
     * 7.578,7.632,7.690,7.763,7.886,8.051,8.244,8.458,
     * 7.338,7.425,7.506,7.603,7.742,7.912,8.099,8.302,
     * 6.007,6.223,6.440,6.673,6.916,7.230,7.538,7.847,
     * 0.000,0.000,0.000,0.000,6.159,6.386,6.814,7.239/
C
      DATA ((EDEN(I,1,3,L,3),I=17,24),L=1,NZEN)/
     * 9.030,9.067,9.087,9.100,9.095,9.101,9.138,9.195,
     * 8.997,9.044,9.076,9.096,9.095,9.102,9.134,9.183,
     * 8.944,9.005,9.050,9.083,9.093,9.102,9.126,9.167,
     * 8.830,8.914,8.986,9.045,9.071,9.090,9.111,9.146,
     * 8.514,8.651,8.784,8.908,8.988,9.043,9.068,9.101,
     * 8.252,8.420,8.585,8.749,8.874,8.967,9.004,9.045,
     * 7.813,7.963,8.135,8.320,8.494,8.651,8.725,8.800,
     * 7.702,7.831,7.957,8.064,8.161,8.245,8.313,8.387,
     * 7.602,7.809,8.005,8.171,8.299,8.382,8.423,8.454,
     * 6.195,6.557,6.921,7.245,7.480,7.714,7.872,7.999,
     * 0.000,6.009,6.356,6.669,6.889,7.010,7.265,7.478/
C
      DATA ((EDEN(I,2,3,L,3),I=17,24),L=1,NZEN)/
     * 9.032,9.070,9.093,9.109,9.104,9.117,9.162,9.227,
     * 8.998,9.045,9.079,9.101,9.101,9.115,9.153,9.208,
     * 8.942,9.002,9.046,9.081,9.092,9.106,9.137,9.183,
     * 8.817,8.900,8.972,9.031,9.059,9.085,9.112,9.153,
     * 8.482,8.615,8.743,8.867,8.950,9.012,9.046,9.088,
     * 8.194,8.360,8.520,8.681,8.807,8.907,8.955,9.009,
     * 7.765,7.899,8.059,8.238,8.401,8.563,8.652,8.743,
     * 7.650,7.771,7.889,7.990,8.086,8.173,8.250,8.335,
     * 7.525,7.722,7.908,8.069,8.199,8.293,8.351,8.403,
     * 6.071,6.416,6.772,7.090,7.328,7.581,7.764,7.919,
     * 0.000,0.000,6.208,6.511,6.732,6.865,7.144,7.386/
C
      DATA ((EDEN(I,3,3,L,3),I=17,24),L=1,NZEN)/
     * 9.077,9.118,9.145,9.163,9.166,9.191,9.243,9.312,
     * 9.037,9.085,9.119,9.145,9.153,9.177,9.221,9.282,
     * 8.975,9.037,9.081,9.119,9.132,9.159,9.198,9.253,
     * 8.841,8.926,8.993,9.052,9.081,9.118,9.155,9.205,
     * 8.484,8.614,8.739,8.857,8.940,9.007,9.055,9.112,
     * 8.206,8.363,8.518,8.673,8.796,8.898,8.961,9.030,
     * 7.756,7.876,8.024,8.190,8.351,8.516,8.625,8.737,
     * 7.615,7.718,7.818,7.907,8.002,8.100,8.197,8.306,
     * 7.469,7.640,7.799,7.941,8.068,8.175,8.258,8.341,
     * 6.013,6.325,6.647,6.949,7.187,7.445,7.658,7.848,
     * 0.000,0.000,6.081,6.362,6.581,6.728,7.036,7.313/
C
      DATA ((EDEN(I,4,3,L,3),I=17,24),L=1,NZEN)/
     * 9.160,9.206,9.233,9.257,9.273,9.315,9.372,9.443,
     * 9.116,9.168,9.201,9.234,9.253,9.294,9.344,9.409,
     * 9.050,9.113,9.158,9.198,9.222,9.265,9.313,9.373,
     * 8.918,9.002,9.069,9.127,9.165,9.218,9.266,9.325,
     * 8.552,8.680,8.800,8.914,8.995,9.075,9.137,9.207,
     * 8.255,8.405,8.551,8.702,8.824,8.934,9.014,9.099,
     * 7.799,7.918,8.064,8.237,8.408,8.582,8.705,8.832,
     * 7.609,7.682,7.756,7.828,7.925,8.042,8.170,8.314,
     * 7.439,7.572,7.695,7.811,7.935,8.059,8.176,8.299,
     * 6.032,6.301,6.576,6.843,7.072,7.342,7.588,7.817,
     * 0.000,0.000,6.021,6.258,6.464,6.624,6.968,7.290/
C
      DATA ((EDEN(I,5,3,L,3),I=17,24),L=1,NZEN)/
     * 9.249,9.297,9.331,9.367,9.400,9.464,9.531,9.611,
     * 9.202,9.256,9.295,9.336,9.370,9.432,9.495,9.569,
     * 9.134,9.196,9.244,9.291,9.330,9.394,9.454,9.524,
     * 8.998,9.080,9.144,9.208,9.260,9.331,9.393,9.465,
     * 8.627,8.753,8.869,8.981,9.072,9.173,9.253,9.340,
     * 8.327,8.474,8.620,8.770,8.897,9.023,9.123,9.228,
     * 7.835,7.956,8.103,8.284,8.470,8.663,8.806,8.954,
     * 7.576,7.627,7.682,7.747,7.856,8.003,8.176,8.369,
     * 7.372,7.469,7.557,7.654,7.785,7.941,8.109,8.289,
     * 0.000,6.222,6.455,6.698,6.932,7.230,7.526,7.812,
     * 0.000,0.000,0.000,6.067,6.279,6.488,6.891,7.281/
C
      DATA ((EDEN(I,1,4,L,3),I=17,24),L=1,NZEN)/
     * 9.014,9.050,9.071,9.084,9.076,9.080,9.120,9.181,
     * 8.983,9.028,9.059,9.079,9.078,9.082,9.116,9.168,
     * 8.929,8.987,9.031,9.064,9.071,9.080,9.107,9.150,
     * 8.815,8.897,8.967,9.026,9.050,9.068,9.091,9.127,
     * 8.504,8.637,8.765,8.888,8.967,9.023,9.050,9.085,
     * 8.241,8.406,8.568,8.729,8.852,8.945,8.984,9.027,
     * 7.806,7.953,8.125,8.309,8.473,8.628,8.705,8.783,
     * 7.696,7.824,7.948,8.052,8.148,8.232,8.301,8.377,
     * 7.595,7.801,7.996,8.160,8.286,8.368,8.408,8.438,
     * 6.189,6.554,6.915,7.240,7.474,7.701,7.859,7.985,
     * 0.000,6.008,6.353,6.668,6.886,7.007,7.259,7.469/
C
      DATA ((EDEN(I,2,4,L,3),I=17,24),L=1,NZEN)/
     * 8.993,9.028,9.048,9.062,9.060,9.072,9.118,9.184,
     * 8.959,9.002,9.031,9.054,9.056,9.067,9.107,9.165,
     * 8.901,8.957,8.999,9.035,9.045,9.057,9.091,9.141,
     * 8.783,8.862,8.927,8.986,9.015,9.037,9.068,9.112,
     * 8.464,8.593,8.717,8.838,8.920,8.981,9.019,9.064,
     * 8.208,8.366,8.523,8.677,8.798,8.894,8.943,8.996,
     * 7.780,7.921,8.084,8.261,8.418,8.572,8.660,8.748,
     * 7.674,7.796,7.915,8.015,8.110,8.197,8.273,8.356,
     * 7.572,7.772,7.959,8.118,8.243,8.328,8.375,8.414,
     * 6.166,6.525,6.880,7.202,7.435,7.666,7.830,7.964,
     * 0.000,0.000,6.325,6.635,6.853,6.974,7.233,7.452/
C
      DATA ((EDEN(I,3,4,L,3),I=17,24),L=1,NZEN)/
     * 8.999,9.035,9.059,9.080,9.085,9.109,9.167,9.242,
     * 8.960,9.003,9.034,9.063,9.071,9.095,9.147,9.214,
     * 8.896,8.951,8.992,9.030,9.046,9.071,9.117,9.178,
     * 8.765,8.841,8.904,8.964,8.996,9.030,9.075,9.131,
     * 8.426,8.548,8.665,8.780,8.863,8.934,8.987,9.046,
     * 8.159,8.307,8.454,8.599,8.718,8.823,8.891,8.962,
     * 7.756,7.872,8.016,8.176,8.325,8.473,8.583,8.694,
     * 7.654,7.762,7.867,7.957,8.048,8.139,8.226,8.321,
     * 7.537,7.719,7.889,8.035,8.158,8.251,8.313,8.369,
     * 6.121,6.454,6.789,7.095,7.327,7.567,7.751,7.907,
     * 0.000,0.000,6.262,6.552,6.764,6.893,7.168,7.406/
C
      DATA ((EDEN(I,4,4,L,3),I=17,24),L=1,NZEN)/
     * 9.000,9.038,9.065,9.093,9.112,9.151,9.229,9.322,
     * 8.955,9.000,9.034,9.069,9.093,9.132,9.202,9.286,
     * 8.887,8.941,8.985,9.030,9.057,9.099,9.164,9.241,
     * 8.739,8.811,8.872,8.934,8.975,9.025,9.088,9.162,
     * 8.376,8.489,8.594,8.707,8.796,8.880,8.955,9.036,
     * 8.100,8.230,8.361,8.500,8.618,8.733,8.826,8.921,
     * 7.757,7.833,7.939,8.076,8.211,8.358,8.495,8.632,
     * 7.657,7.739,7.820,7.893,7.977,8.071,8.169,8.278,
     * 7.513,7.665,7.806,7.932,8.048,8.149,8.230,8.310,
     * 6.096,6.384,6.683,6.963,7.189,7.443,7.654,7.839,
     * 0.000,0.000,6.222,6.474,6.673,6.809,7.108,7.374/
C
      DATA ((EDEN(I,5,4,L,3),I=17,24),L=1,NZEN)/
     * 9.001,9.042,9.075,9.114,9.147,9.200,9.304,9.418,
     * 8.952,8.997,9.035,9.080,9.117,9.170,9.266,9.372,
     * 8.877,8.932,8.977,9.030,9.072,9.130,9.220,9.320,
     * 8.729,8.800,8.864,8.935,8.992,9.060,9.149,9.246,
     * 8.336,8.439,8.537,8.649,8.750,8.854,8.957,9.064,
     * 8.061,8.175,8.293,8.425,8.549,8.683,8.805,8.929,
     * 7.763,7.808,7.887,8.004,8.136,8.292,8.460,8.627,
     * 7.642,7.700,7.759,7.819,7.904,8.014,8.137,8.275,
     * 7.466,7.588,7.699,7.806,7.924,8.046,8.164,8.284,
     * 6.034,6.282,6.543,6.802,7.028,7.307,7.559,7.789,
     * 0.000,0.000,6.100,6.325,6.522,6.681,7.023,7.338/
C
      DATA ((EDEN(I,1,5,L,3),I=17,24),L=1,NZEN)/
     * 9.021,9.056,9.075,9.087,9.079,9.082,9.125,9.188,
     * 8.989,9.033,9.063,9.082,9.078,9.084,9.119,9.172,
     * 8.937,8.995,9.037,9.068,9.074,9.080,9.107,9.150,
     * 8.825,8.905,8.974,9.030,9.053,9.068,9.090,9.126,
     * 8.518,8.651,8.779,8.898,8.974,9.027,9.052,9.086,
     * 8.260,8.422,8.585,8.746,8.865,8.953,8.990,9.031,
     * 7.817,7.971,8.144,8.327,8.494,8.647,8.721,8.796,
     * 7.700,7.827,7.950,8.054,8.148,8.232,8.302,8.378,
     * 7.604,7.808,8.001,8.163,8.287,8.367,8.405,8.433,
     * 6.213,6.576,6.937,7.261,7.492,7.718,7.871,7.993,
     * 0.000,6.031,6.375,6.688,6.906,7.023,7.270,7.475/
C
      DATA ((EDEN(I,2,5,L,3),I=17,24),L=1,NZEN)/
     * 8.977,9.010,9.027,9.040,9.039,9.048,9.094,9.160,
     * 8.943,8.984,9.011,9.033,9.033,9.043,9.086,9.146,
     * 8.889,8.942,8.981,9.015,9.025,9.035,9.072,9.124,
     * 8.778,8.853,8.916,8.974,8.998,9.022,9.054,9.099,
     * 8.485,8.612,8.732,8.848,8.925,8.982,9.017,9.060,
     * 8.244,8.402,8.554,8.706,8.822,8.909,8.954,9.002,
     * 7.815,7.967,8.137,8.316,8.471,8.616,8.697,8.777,
     * 7.710,7.833,7.954,8.055,8.150,8.238,8.313,8.394,
     * 7.633,7.835,8.024,8.183,8.303,8.379,8.413,8.437,
     * 6.276,6.644,7.006,7.329,7.558,7.777,7.919,8.028,
     * 0.000,6.075,6.422,6.734,6.949,7.061,7.301,7.497/
C
      DATA ((EDEN(I,3,5,L,3),I=17,24),L=1,NZEN)/
     * 8.941,8.971,8.989,9.008,9.012,9.035,9.100,9.182,
     * 8.900,8.938,8.964,8.989,8.999,9.019,9.076,9.150,
     * 8.837,8.884,8.919,8.953,8.969,8.991,9.044,9.111,
     * 8.713,8.780,8.837,8.892,8.923,8.955,9.004,9.065,
     * 8.401,8.517,8.626,8.735,8.813,8.880,8.935,8.995,
     * 8.152,8.299,8.442,8.579,8.689,8.786,8.852,8.921,
     * 7.776,7.897,8.040,8.199,8.336,8.473,8.580,8.684,
     * 7.706,7.820,7.931,8.023,8.110,8.192,8.265,8.344,
     * 7.619,7.812,7.992,8.143,8.259,8.334,8.370,8.395,
     * 6.236,6.587,6.937,7.254,7.483,7.707,7.860,7.979,
     * 0.000,6.087,6.411,6.709,6.916,7.029,7.270,7.467/
C
      DATA ((EDEN(I,4,5,L,3),I=17,24),L=1,NZEN)/
     * 8.879,8.906,8.927,8.953,8.972,9.010,9.109,9.223,
     * 8.836,8.870,8.896,8.928,8.951,8.991,9.078,9.180,
     * 8.757,8.802,8.835,8.873,8.898,8.937,9.014,9.104,
     * 8.602,8.663,8.714,8.767,8.804,8.849,8.922,9.005,
     * 8.244,8.354,8.452,8.551,8.629,8.706,8.787,8.871,
     * 8.002,8.128,8.252,8.378,8.478,8.577,8.673,8.769,
     * 7.768,7.823,7.904,8.014,8.121,8.234,8.368,8.498,
     * 7.720,7.817,7.910,7.987,8.058,8.124,8.181,8.243,
     * 7.601,7.777,7.940,8.079,8.185,8.257,8.292,8.317,
     * 6.189,6.501,6.823,7.120,7.344,7.584,7.746,7.873,
     * 0.000,6.148,6.442,6.712,6.908,7.021,7.258,7.452/
C
      DATA ((EDEN(I,5,5,L,3),I=17,24),L=1,NZEN)/
     * 8.832,8.861,8.882,8.914,8.946,8.998,9.132,9.278,
     * 8.783,8.817,8.843,8.879,8.911,8.966,9.087,9.219,
     * 8.707,8.750,8.784,8.826,8.863,8.919,9.030,9.149,
     * 8.516,8.581,8.634,8.692,8.738,8.797,8.899,9.008,
     * 8.121,8.219,8.313,8.415,8.495,8.583,8.693,8.804,
     * 7.915,8.003,8.105,8.223,8.325,8.430,8.554,8.677,
     * 7.801,7.824,7.862,7.928,8.008,8.111,8.266,8.416,
     * 7.727,7.810,7.890,7.959,8.026,8.092,8.153,8.218,
     * 7.566,7.726,7.874,8.004,8.112,8.192,8.242,8.281,
     * 6.135,6.414,6.710,6.988,7.213,7.476,7.659,7.809,
     * 0.000,6.095,6.358,6.608,6.802,6.932,7.178,7.383/
C
      DATA ((EDEN(I,1,6,L,3),I=17,24),L=1,NZEN)/
     * 9.040,9.076,9.095,9.103,9.092,9.095,9.139,9.203,
     * 9.009,9.053,9.080,9.099,9.095,9.098,9.133,9.187,
     * 8.959,9.016,9.058,9.086,9.090,9.095,9.120,9.163,
     * 8.849,8.932,8.998,9.052,9.071,9.085,9.104,9.137,
     * 8.548,8.684,8.811,8.929,9.000,9.047,9.069,9.099,
     * 8.294,8.459,8.622,8.782,8.897,8.980,9.012,9.050,
     * 7.842,8.003,8.182,8.373,8.539,8.689,8.757,8.826,
     * 7.711,7.837,7.959,8.063,8.158,8.242,8.312,8.388,
     * 7.623,7.824,8.014,8.175,8.297,8.374,8.410,8.436,
     * 6.253,6.614,6.974,7.296,7.525,7.746,7.893,8.010,
     * 0.000,6.067,6.407,6.722,6.937,7.056,7.296,7.495/
C
      DATA ((EDEN(I,2,6,L,3),I=17,24),L=1,NZEN)/
     * 8.984,9.016,9.032,9.044,9.039,9.050,9.096,9.160,
     * 8.950,8.990,9.015,9.034,9.034,9.042,9.085,9.146,
     * 8.899,8.950,8.988,9.020,9.025,9.038,9.075,9.127,
     * 8.794,8.868,8.931,8.985,9.009,9.028,9.059,9.103,
     * 8.522,8.648,8.766,8.883,8.954,9.004,9.035,9.074,
     * 8.293,8.451,8.603,8.754,8.863,8.945,8.984,9.025,
     * 7.864,8.028,8.207,8.391,8.552,8.694,8.766,8.837,
     * 7.745,7.869,7.991,8.094,8.190,8.278,8.352,8.431,
     * 7.689,7.891,8.082,8.240,8.356,8.426,8.449,8.461,
     * 6.363,6.733,7.100,7.425,7.652,7.860,7.989,8.083,
     * 0.000,6.170,6.521,6.835,7.049,7.154,7.378,7.555/
C
      DATA ((EDEN(I,3,6,L,3),I=17,24),L=1,NZEN)/
     * 8.917,8.943,8.959,8.977,8.982,9.002,9.071,9.158,
     * 8.877,8.910,8.932,8.955,8.961,8.982,9.043,9.121,
     * 8.819,8.861,8.893,8.926,8.940,8.960,9.014,9.084,
     * 8.702,8.768,8.819,8.872,8.901,8.932,8.981,9.042,
     * 8.411,8.528,8.633,8.737,8.812,8.875,8.927,8.983,
     * 8.185,8.336,8.475,8.610,8.715,8.806,8.868,8.931,
     * 7.806,7.935,8.087,8.249,8.383,8.509,8.608,8.703,
     * 7.753,7.872,7.988,8.083,8.168,8.244,8.307,8.374,
     * 7.694,7.895,8.083,8.237,8.348,8.413,8.431,8.436,
     * 6.341,6.703,7.066,7.392,7.617,7.828,7.953,8.039,
     * 0.000,6.188,6.527,6.831,7.036,7.135,7.346,7.507/
C
      DATA ((EDEN(I,4,6,L,3),I=17,24),L=1,NZEN)/
     * 8.826,8.848,8.862,8.885,8.903,8.943,9.056,9.184,
     * 8.779,8.809,8.827,8.853,8.872,8.909,9.006,9.116,
     * 8.695,8.734,8.762,8.792,8.811,8.844,8.927,9.021,
     * 8.545,8.608,8.656,8.705,8.735,8.774,8.850,8.935,
     * 8.202,8.312,8.415,8.511,8.581,8.646,8.726,8.808,
     * 7.977,8.101,8.228,8.351,8.445,8.532,8.623,8.711,
     * 7.799,7.853,7.930,8.035,8.136,8.238,8.361,8.478,
     * 7.779,7.887,7.990,8.073,8.138,8.188,8.220,8.253,
     * 7.684,7.878,8.057,8.206,8.307,8.358,8.361,8.347,
     * 6.273,6.603,6.941,7.254,7.476,7.706,7.827,7.908,
     * 0.000,6.294,6.608,6.894,7.090,7.189,7.373,7.506/
C
      DATA ((EDEN(I,5,6,L,3),I=17,24),L=1,NZEN)/
     * 8.762,8.787,8.804,8.829,8.854,8.903,9.063,9.235,
     * 8.714,8.746,8.769,8.797,8.823,8.871,9.015,9.168,
     * 8.611,8.658,8.691,8.727,8.752,8.795,8.914,9.041,
     * 8.405,8.478,8.537,8.594,8.635,8.680,8.786,8.899,
     * 8.040,8.137,8.235,8.338,8.418,8.498,8.606,8.712,
     * 7.904,7.976,8.070,8.183,8.281,8.377,8.494,8.607,
     * 7.859,7.882,7.916,7.966,8.027,8.110,8.251,8.384,
     * 7.799,7.904,8.002,8.080,8.141,8.184,8.209,8.233,
     * 7.669,7.857,8.031,8.177,8.279,8.335,8.346,8.337,
     * 6.270,6.580,6.903,7.200,7.425,7.674,7.808,7.901,
     * 6.008,6.281,6.571,6.844,7.035,7.149,7.333,7.464/
C
      DATA ((EDEN(I,1,7,L,3),I=17,24),L=1,NZEN)/
     * 9.055,9.090,9.106,9.116,9.105,9.107,9.151,9.215,
     * 9.025,9.069,9.096,9.112,9.106,9.109,9.144,9.197,
     * 8.975,9.032,9.075,9.101,9.101,9.107,9.132,9.173,
     * 8.868,8.949,9.015,9.069,9.085,9.096,9.114,9.146,
     * 8.569,8.705,8.835,8.950,9.019,9.062,9.082,9.111,
     * 8.315,8.483,8.647,8.807,8.919,8.997,9.027,9.062,
     * 7.860,8.024,8.206,8.399,8.569,8.714,8.779,8.844,
     * 7.719,7.844,7.967,8.071,8.167,8.251,8.321,8.396,
     * 7.635,7.835,8.024,8.184,8.305,8.382,8.417,8.441,
     * 6.277,6.640,6.996,7.316,7.545,7.765,7.910,8.023,
     * 0.000,6.085,6.429,6.741,6.958,7.073,7.311,7.507/
C
      DATA ((EDEN(I,2,7,L,3),I=17,24),L=1,NZEN)/
     * 9.003,9.034,9.052,9.062,9.057,9.065,9.109,9.172,
     * 8.969,9.009,9.034,9.054,9.051,9.058,9.100,9.160,
     * 8.919,8.970,9.008,9.040,9.044,9.054,9.089,9.140,
     * 8.816,8.892,8.954,9.007,9.028,9.046,9.075,9.117,
     * 8.550,8.678,8.798,8.912,8.982,9.027,9.056,9.092,
     * 8.323,8.482,8.636,8.790,8.897,8.973,9.009,9.047,
     * 7.892,8.063,8.245,8.435,8.595,8.735,8.802,8.867,
     * 7.762,7.888,8.012,8.115,8.212,8.300,8.374,8.452,
     * 7.714,7.916,8.107,8.266,8.382,8.448,8.469,8.476,
     * 6.407,6.778,7.144,7.469,7.693,7.897,8.021,8.108,
     * 0.000,6.210,6.562,6.880,7.091,7.194,7.411,7.581/
C
      DATA ((EDEN(I,3,7,L,3),I=17,24),L=1,NZEN)/
     * 8.932,8.960,8.977,8.994,8.997,9.018,9.086,9.172,
     * 8.894,8.929,8.951,8.974,8.983,9.000,9.060,9.137,
     * 8.837,8.882,8.915,8.948,8.961,8.981,9.034,9.102,
     * 8.726,8.791,8.845,8.899,8.928,8.956,9.003,9.062,
     * 8.444,8.561,8.668,8.774,8.849,8.909,8.957,9.009,
     * 8.223,8.375,8.515,8.654,8.759,8.847,8.905,8.962,
     * 7.833,7.970,8.129,8.295,8.431,8.561,8.654,8.742,
     * 7.778,7.900,8.018,8.115,8.201,8.277,8.339,8.405,
     * 7.729,7.931,8.122,8.278,8.388,8.449,8.462,8.460,
     * 6.394,6.763,7.128,7.455,7.679,7.884,8.000,8.076,
     * 0.000,6.239,6.584,6.890,7.095,7.191,7.393,7.543/
C
      DATA ((EDEN(I,4,7,L,3),I=17,24),L=1,NZEN)/
     * 8.842,8.865,8.881,8.904,8.923,8.964,9.079,9.209,
     * 8.798,8.828,8.849,8.875,8.895,8.932,9.029,9.139,
     * 8.718,8.758,8.786,8.817,8.837,8.871,8.951,9.044,
     * 8.572,8.637,8.686,8.736,8.766,8.807,8.880,8.962,
     * 8.240,8.354,8.457,8.556,8.624,8.693,8.768,8.844,
     * 8.012,8.146,8.276,8.402,8.499,8.584,8.670,8.753,
     * 7.820,7.879,7.967,8.083,8.190,8.295,8.413,8.525,
     * 7.808,7.920,8.028,8.113,8.180,8.228,8.258,8.288,
     * 7.726,7.924,8.108,8.259,8.359,8.406,8.402,8.378,
     * 6.328,6.667,7.012,7.328,7.551,7.776,7.886,7.954,
     * 6.057,6.362,6.685,6.977,7.172,7.269,7.441,7.560/
C
      DATA ((EDEN(I,5,7,L,3),I=17,24),L=1,NZEN)/
     * 8.779,8.804,8.820,8.846,8.872,8.923,9.086,9.262,
     * 8.733,8.767,8.789,8.818,8.846,8.894,9.040,9.196,
     * 8.635,8.683,8.716,8.753,8.778,8.822,8.939,9.066,
     * 8.439,8.513,8.571,8.628,8.668,8.714,8.817,8.926,
     * 8.076,8.179,8.283,8.390,8.471,8.545,8.649,8.751,
     * 7.930,8.012,8.118,8.238,8.340,8.435,8.547,8.654,
     * 7.878,7.907,7.946,8.007,8.077,8.169,8.306,8.434,
     * 7.835,7.944,8.046,8.127,8.189,8.234,8.261,8.285,
     * 7.718,7.912,8.093,8.243,8.344,8.395,8.396,8.375,
     * 6.334,6.654,6.981,7.289,7.516,7.757,7.878,7.954,
     * 6.080,6.364,6.665,6.941,7.137,7.242,7.410,7.524/
C
      DATA ((EDEN(I,1,8,L,3),I=17,24),L=1,NZEN)/
     * 9.055,9.089,9.106,9.116,9.103,9.107,9.151,9.214,
     * 9.024,9.068,9.096,9.112,9.106,9.109,9.143,9.196,
     * 8.973,9.031,9.070,9.101,9.101,9.107,9.132,9.173,
     * 8.865,8.947,9.014,9.065,9.085,9.096,9.115,9.147,
     * 8.565,8.701,8.831,8.946,9.016,9.058,9.080,9.109,
     * 8.311,8.478,8.642,8.802,8.915,8.994,9.025,9.061,
     * 7.854,8.019,8.199,8.392,8.561,8.707,8.774,8.840,
     * 7.719,7.844,7.966,8.071,8.167,8.251,8.322,8.398,
     * 7.633,7.833,8.023,8.183,8.306,8.383,8.418,8.444,
     * 6.269,6.631,6.988,7.309,7.538,7.757,7.904,8.020,
     * 0.000,6.078,6.423,6.734,6.950,7.067,7.307,7.506/
C
      DATA ((EDEN(I,2,8,L,3),I=17,24),L=1,NZEN)/
     * 9.021,9.054,9.073,9.083,9.077,9.086,9.129,9.191,
     * 8.988,9.030,9.057,9.076,9.075,9.083,9.122,9.179,
     * 8.937,8.991,9.031,9.063,9.066,9.077,9.109,9.158,
     * 8.832,8.909,8.974,9.028,9.049,9.065,9.093,9.134,
     * 8.556,8.684,8.810,8.924,8.994,9.043,9.071,9.106,
     * 8.321,8.483,8.642,8.796,8.905,8.982,9.018,9.057,
     * 7.881,8.048,8.228,8.415,8.577,8.717,8.785,8.853,
     * 7.757,7.883,8.007,8.111,8.208,8.295,8.368,8.446,
     * 7.700,7.902,8.093,8.253,8.370,8.440,8.464,8.476,
     * 6.373,6.744,7.106,7.431,7.657,7.866,7.997,8.092,
     * 0.000,0.000,6.531,6.847,7.059,7.165,7.388,7.565/
C
      DATA ((EDEN(I,3,8,L,3),I=17,24),L=1,NZEN)/
     * 8.983,9.015,9.035,9.053,9.056,9.076,9.139,9.220,
     * 8.945,8.984,9.011,9.037,9.043,9.060,9.115,9.186,
     * 8.887,8.938,8.976,9.010,9.022,9.044,9.092,9.154,
     * 8.773,8.845,8.905,8.961,8.990,9.018,9.061,9.115,
     * 8.482,8.605,8.720,8.831,8.908,8.967,9.012,9.062,
     * 8.245,8.400,8.546,8.692,8.802,8.893,8.946,9.001,
     * 7.847,7.990,8.153,8.325,8.469,8.607,8.698,8.786,
     * 7.774,7.894,8.012,8.110,8.200,8.281,8.349,8.422,
     * 7.716,7.916,8.103,8.259,8.373,8.440,8.463,8.472,
     * 6.375,6.736,7.099,7.422,7.647,7.858,7.986,8.077,
     * 0.000,6.206,6.543,6.846,7.053,7.156,7.372,7.539/
C
      DATA ((EDEN(I,4,8,L,3),I=17,24),L=1,NZEN)/
     * 8.930,8.960,8.983,9.012,9.031,9.069,9.169,9.284,
     * 8.890,8.927,8.955,8.988,9.010,9.048,9.131,9.228,
     * 8.824,8.871,8.907,8.949,8.975,9.013,9.083,9.166,
     * 8.680,8.746,8.802,8.860,8.897,8.941,9.004,9.077,
     * 8.349,8.466,8.571,8.677,8.756,8.832,8.899,8.969,
     * 8.108,8.250,8.387,8.521,8.625,8.723,8.803,8.882,
     * 7.828,7.904,8.016,8.151,8.274,8.394,8.512,8.624,
     * 7.804,7.912,8.017,8.104,8.179,8.244,8.295,8.349,
     * 7.717,7.907,8.084,8.232,8.336,8.396,8.410,8.408,
     * 6.343,6.675,7.017,7.327,7.552,7.776,7.907,7.999,
     * 6.038,6.338,6.651,6.939,7.137,7.239,7.442,7.594/
C
      DATA ((EDEN(I,5,8,L,3),I=17,24),L=1,NZEN)/
     * 8.890,8.920,8.943,8.978,9.010,9.066,9.204,9.354,
     * 8.849,8.886,8.914,8.952,8.988,9.043,9.165,9.297,
     * 8.777,8.825,8.861,8.907,8.945,9.000,9.103,9.214,
     * 8.599,8.668,8.725,8.784,8.829,8.886,8.976,9.072,
     * 8.236,8.351,8.454,8.561,8.643,8.729,8.824,8.919,
     * 8.011,8.130,8.256,8.390,8.494,8.598,8.706,8.809,
     * 7.862,7.903,7.968,8.065,8.170,8.285,8.425,8.558,
     * 7.823,7.924,8.019,8.099,8.170,8.233,8.284,8.335,
     * 7.712,7.892,8.058,8.200,8.306,8.373,8.399,8.408,
     * 6.331,6.645,6.965,7.263,7.490,7.734,7.880,7.987,
     * 6.055,6.329,6.619,6.889,7.085,7.201,7.404,7.557/
C
      DATA ((EDEN(I,1,9,L,3),I=17,24),L=1,NZEN)/
     * 9.044,9.078,9.098,9.106,9.095,9.099,9.141,9.202,
     * 9.012,9.056,9.084,9.102,9.098,9.102,9.136,9.188,
     * 8.960,9.018,9.060,9.090,9.093,9.100,9.125,9.167,
     * 8.850,8.932,9.000,9.052,9.075,9.088,9.108,9.142,
     * 8.544,8.679,8.809,8.928,8.999,9.047,9.070,9.102,
     * 8.287,8.454,8.616,8.778,8.894,8.977,9.012,9.051,
     * 7.837,7.995,8.172,8.359,8.529,8.680,8.751,8.822,
     * 7.712,7.838,7.961,8.066,8.162,8.246,8.317,8.394,
     * 7.620,7.823,8.014,8.176,8.300,8.380,8.417,8.445,
     * 6.238,6.601,6.960,7.283,7.515,7.738,7.889,8.010,
     * 0.000,6.049,6.396,6.708,6.926,7.044,7.290,7.494/
C
      DATA ((EDEN(I,2,9,L,3),I=17,24),L=1,NZEN)/
     * 9.038,9.074,9.093,9.103,9.099,9.106,9.149,9.210,
     * 9.006,9.052,9.078,9.102,9.100,9.109,9.146,9.200,
     * 8.953,9.011,9.052,9.086,9.091,9.102,9.132,9.178,
     * 8.840,8.922,8.989,9.045,9.068,9.089,9.115,9.154,
     * 8.538,8.672,8.799,8.918,8.993,9.046,9.076,9.113,
     * 8.292,8.456,8.616,8.774,8.889,8.974,9.013,9.056,
     * 7.847,8.006,8.182,8.366,8.532,8.680,8.755,8.829,
     * 7.730,7.856,7.978,8.082,8.178,8.264,8.337,8.416,
     * 7.650,7.852,8.042,8.203,8.325,8.402,8.436,8.459,
     * 6.287,6.651,7.013,7.335,7.565,7.782,7.927,8.039,
     * 0.000,6.087,6.433,6.748,6.964,7.079,7.317,7.511/
C
      DATA ((EDEN(I,3,9,L,3),I=17,24),L=1,NZEN)/
     * 9.053,9.092,9.114,9.135,9.137,9.155,9.209,9.281,
     * 9.018,9.065,9.098,9.122,9.130,9.148,9.194,9.255,
     * 8.959,9.019,9.062,9.099,9.112,9.133,9.172,9.225,
     * 8.838,8.917,8.982,9.040,9.068,9.096,9.133,9.182,
     * 8.521,8.651,8.773,8.890,8.969,9.028,9.071,9.120,
     * 8.268,8.426,8.581,8.734,8.851,8.943,8.997,9.053,
     * 7.841,7.981,8.145,8.321,8.478,8.628,8.720,8.811,
     * 7.742,7.859,7.974,8.071,8.165,8.252,8.329,8.412,
     * 7.656,7.849,8.030,8.183,8.302,8.382,8.422,8.452,
     * 6.284,6.635,6.983,7.297,7.526,7.753,7.908,8.030,
     * 0.000,6.139,6.463,6.766,6.977,7.096,7.336,7.534/
C
      DATA ((EDEN(I,4,9,L,3),I=17,24),L=1,NZEN)/
     * 9.088,9.130,9.160,9.188,9.200,9.233,9.307,9.396,
     * 9.052,9.101,9.137,9.172,9.189,9.222,9.286,9.364,
     * 8.989,9.049,9.096,9.137,9.161,9.193,9.247,9.313,
     * 8.860,8.940,9.009,9.072,9.108,9.147,9.198,9.258,
     * 8.516,8.640,8.759,8.874,8.961,9.033,9.091,9.153,
     * 8.253,8.406,8.554,8.703,8.823,8.927,8.999,9.071,
     * 7.859,7.971,8.114,8.277,8.426,8.578,8.692,8.804,
     * 7.776,7.876,7.975,8.061,8.150,8.238,8.320,8.408,
     * 7.675,7.847,8.009,8.149,8.264,8.350,8.404,8.449,
     * 6.301,6.616,6.938,7.234,7.459,7.694,7.866,8.006,
     * 0.000,6.225,6.518,6.794,6.995,7.119,7.374,7.588/
C
      DATA ((EDEN(I,5,9,L,3),I=17,24),L=1,NZEN)/
     * 9.127,9.173,9.208,9.244,9.272,9.322,9.419,9.528,
     * 9.087,9.141,9.183,9.226,9.256,9.306,9.391,9.488,
     * 9.028,9.092,9.144,9.197,9.233,9.282,9.358,9.443,
     * 8.895,8.979,9.051,9.122,9.174,9.229,9.300,9.377,
     * 8.530,8.652,8.770,8.890,8.987,9.076,9.155,9.236,
     * 8.260,8.406,8.550,8.703,8.833,8.955,9.049,9.142,
     * 7.885,7.977,8.106,8.263,8.419,8.585,8.723,8.857,
     * 7.792,7.876,7.960,8.040,8.133,8.237,8.342,8.455,
     * 7.668,7.818,7.959,8.087,8.206,8.310,8.392,8.466,
     * 6.298,6.587,6.882,7.163,7.391,7.645,7.850,8.024,
     * 0.000,6.205,6.464,6.717,6.915,7.059,7.345,7.592/
C
      DATA ((EDEN(I,1,10,L,3),I=17,24),L=1,NZEN)/
     * 9.039,9.075,9.095,9.103,9.095,9.101,9.140,9.199,
     * 9.006,9.052,9.080,9.099,9.098,9.102,9.135,9.185,
     * 8.954,9.012,9.056,9.086,9.093,9.102,9.127,9.168,
     * 8.840,8.923,8.993,9.047,9.075,9.090,9.111,9.146,
     * 8.531,8.667,8.797,8.918,8.995,9.047,9.071,9.104,
     * 8.272,8.438,8.601,8.765,8.885,8.973,9.009,9.049,
     * 7.825,7.979,8.155,8.345,8.512,8.667,8.741,8.814,
     * 7.708,7.836,7.961,8.066,8.163,8.247,8.317,8.393,
     * 7.613,7.818,8.011,8.175,8.301,8.382,8.421,8.451,
     * 6.215,6.576,6.941,7.264,7.497,7.726,7.881,8.005,
     * 0.000,6.028,6.375,6.688,6.906,7.026,7.276,7.485/
C
      DATA ((EDEN(I,2,10,L,3),I=17,24),L=1,NZEN)/
     * 9.060,9.099,9.119,9.131,9.125,9.133,9.175,9.237,
     * 9.027,9.074,9.105,9.126,9.127,9.134,9.169,9.221,
     * 8.972,9.033,9.078,9.113,9.119,9.129,9.156,9.199,
     * 8.854,8.939,9.011,9.067,9.091,9.111,9.135,9.172,
     * 8.530,8.666,8.798,8.919,8.997,9.054,9.083,9.120,
     * 8.258,8.422,8.587,8.748,8.869,8.961,9.003,9.050,
     * 7.813,7.959,8.130,8.314,8.479,8.635,8.717,8.800,
     * 7.699,7.823,7.944,8.046,8.141,8.226,8.298,8.377,
     * 7.589,7.790,7.979,8.140,8.266,8.352,8.398,8.435,
     * 6.169,6.525,6.881,7.203,7.437,7.673,7.839,7.975,
     * 0.000,0.000,6.325,6.633,6.853,6.980,7.238,7.455/
C
      DATA ((EDEN(I,3,10,L,3),I=17,24),L=1,NZEN)/
     * 9.125,9.168,9.190,9.206,9.207,9.227,9.274,9.338,
     * 9.089,9.138,9.172,9.195,9.200,9.217,9.257,9.312,
     * 9.032,9.096,9.141,9.175,9.188,9.209,9.241,9.288,
     * 8.907,8.993,9.063,9.121,9.146,9.176,9.206,9.249,
     * 8.564,8.701,8.832,8.950,9.028,9.090,9.129,9.175,
     * 8.281,8.444,8.605,8.762,8.882,8.976,9.030,9.088,
     * 7.830,7.969,8.132,8.311,8.477,8.637,8.732,8.828,
     * 7.701,7.813,7.923,8.019,8.114,8.206,8.290,8.383,
     * 7.579,7.761,7.933,8.082,8.206,8.299,8.361,8.416,
     * 6.162,6.495,6.831,7.134,7.366,7.608,7.793,7.950,
     * 0.000,0.000,6.294,6.579,6.797,6.931,7.203,7.438/
C
      DATA ((EDEN(I,4,10,L,3),I=17,24),L=1,NZEN)/
     * 9.268,9.311,9.336,9.358,9.368,9.402,9.451,9.515,
     * 9.229,9.282,9.314,9.343,9.356,9.390,9.431,9.486,
     * 9.174,9.237,9.282,9.317,9.338,9.372,9.408,9.457,
     * 9.054,9.141,9.208,9.260,9.293,9.332,9.368,9.414,
     * 8.703,8.840,8.966,9.075,9.147,9.213,9.260,9.313,
     * 8.423,8.589,8.751,8.905,9.020,9.112,9.172,9.237,
     * 7.932,8.074,8.243,8.431,8.608,8.770,8.872,8.974,
     * 7.754,7.846,7.938,8.025,8.123,8.230,8.338,8.456,
     * 7.620,7.772,7.915,8.047,8.169,8.275,8.361,8.443,
     * 6.247,6.539,6.835,7.117,7.344,7.596,7.804,7.988,
     * 0.000,6.043,6.308,6.565,6.773,6.918,7.217,7.485/
C
      DATA ((EDEN(I,5,10,L,3),I=17,24),L=1,NZEN)/
     * 9.423,9.467,9.501,9.533,9.563,9.617,9.670,9.735,
     * 9.386,9.438,9.477,9.515,9.545,9.599,9.644,9.701,
     * 9.333,9.394,9.441,9.484,9.520,9.575,9.616,9.667,
     * 9.221,9.303,9.368,9.427,9.474,9.533,9.575,9.625,
     * 8.889,9.027,9.146,9.247,9.324,9.408,9.464,9.524,
     * 8.601,8.771,8.931,9.074,9.183,9.291,9.362,9.435,
     * 8.074,8.244,8.435,8.642,8.825,8.994,9.100,9.207,
     * 7.796,7.873,7.955,8.040,8.153,8.286,8.429,8.584,
     * 7.641,7.768,7.887,8.007,8.137,8.269,8.393,8.517,
     * 6.303,6.566,6.831,7.094,7.327,7.602,7.845,8.070,
     * 0.000,6.042,6.269,6.513,6.731,6.916,7.256,7.572/
C
      DATA ((EDEN(I,1,11,L,3),I=17,24),L=1,NZEN)/
     * 9.047,9.085,9.106,9.119,9.110,9.118,9.154,9.210,
     * 9.014,9.062,9.092,9.115,9.111,9.120,9.149,9.197,
     * 8.962,9.024,9.068,9.101,9.107,9.119,9.141,9.181,
     * 8.848,8.934,9.006,9.065,9.088,9.107,9.126,9.159,
     * 8.537,8.675,8.809,8.933,9.009,9.062,9.085,9.116,
     * 8.276,8.443,8.612,8.779,8.900,8.991,9.025,9.063,
     * 7.829,7.984,8.161,8.352,8.527,8.684,8.754,8.825,
     * 7.712,7.840,7.967,8.075,8.173,8.258,8.326,8.400,
     * 7.616,7.823,8.018,8.184,8.311,8.394,8.434,8.463,
     * 6.215,6.576,6.939,7.264,7.499,7.730,7.887,8.012,
     * 0.000,6.022,6.373,6.685,6.904,7.026,7.279,7.490/
C
      DATA ((EDEN(I,2,11,L,3),I=17,24),L=1,NZEN)/
     * 9.085,9.128,9.148,9.160,9.154,9.163,9.206,9.268,
     * 9.050,9.097,9.129,9.149,9.146,9.155,9.189,9.241,
     * 8.999,9.062,9.110,9.142,9.150,9.162,9.187,9.228,
     * 8.875,8.965,9.038,9.094,9.121,9.144,9.165,9.200,
     * 8.532,8.673,8.808,8.931,9.011,9.071,9.099,9.136,
     * 8.236,8.404,8.571,8.735,8.859,8.957,9.001,9.050,
     * 7.793,7.930,8.095,8.276,8.444,8.608,8.693,8.780,
     * 7.672,7.796,7.916,8.019,8.116,8.201,8.275,8.356,
     * 7.540,7.739,7.927,8.089,8.220,8.313,8.369,8.418,
     * 6.059,6.413,6.765,7.085,7.326,7.581,7.765,7.922,
     * 0.000,0.000,6.199,6.503,6.726,6.861,7.138,7.379/
C
      DATA ((EDEN(I,3,11,L,3),I=17,24),L=1,NZEN)/
     * 9.181,9.223,9.246,9.258,9.257,9.271,9.316,9.378,
     * 9.140,9.191,9.224,9.243,9.247,9.260,9.297,9.349,
     * 9.081,9.144,9.188,9.219,9.226,9.243,9.274,9.319,
     * 8.961,9.051,9.119,9.172,9.199,9.225,9.252,9.293,
     * 8.593,8.734,8.867,8.980,9.056,9.116,9.155,9.201,
     * 8.306,8.473,8.641,8.799,8.917,9.010,9.063,9.121,
     * 7.819,7.953,8.113,8.288,8.457,8.619,8.719,8.821,
     * 7.663,7.768,7.871,7.962,8.057,8.151,8.243,8.346,
     * 7.511,7.683,7.844,7.986,8.112,8.215,8.293,8.370,
     * 6.023,6.344,6.673,6.976,7.207,7.464,7.675,7.864,
     * 0.000,0.000,6.109,6.379,6.593,6.744,7.045,7.317/
C
      DATA ((EDEN(I,4,11,L,3),I=17,24),L=1,NZEN)/
     * 9.393,9.432,9.455,9.470,9.478,9.504,9.540,9.590,
     * 9.356,9.401,9.430,9.451,9.462,9.488,9.518,9.561,
     * 9.296,9.356,9.394,9.421,9.437,9.466,9.492,9.531,
     * 9.180,9.261,9.318,9.363,9.391,9.427,9.454,9.493,
     * 8.835,8.973,9.090,9.181,9.249,9.310,9.351,9.401,
     * 8.543,8.717,8.878,9.016,9.117,9.207,9.263,9.325,
     * 7.998,8.161,8.350,8.547,8.723,8.877,8.973,9.073,
     * 7.721,7.802,7.885,7.966,8.068,8.186,8.313,8.454,
     * 7.550,7.686,7.811,7.932,8.057,8.177,8.286,8.400,
     * 6.174,6.453,6.725,6.992,7.221,7.489,7.722,7.941,
     * 0.000,0.000,6.106,6.348,6.557,6.729,7.063,7.376/
C
      DATA ((EDEN(I,5,11,L,3),I=17,24),L=1,NZEN)/
     * 9.623,9.666,9.700,9.731,9.757,9.792,9.821,9.863,
     * 9.586,9.634,9.672,9.706,9.734,9.771,9.795,9.831,
     * 9.532,9.586,9.630,9.669,9.701,9.742,9.764,9.797,
     * 9.424,9.493,9.550,9.601,9.643,9.693,9.719,9.754,
     * 9.116,9.235,9.334,9.420,9.494,9.573,9.616,9.667,
     * 8.842,9.000,9.137,9.252,9.352,9.453,9.513,9.580,
     * 8.270,8.468,8.672,8.858,9.016,9.167,9.263,9.364,
     * 7.785,7.859,7.936,8.021,8.143,8.293,8.461,8.645,
     * 7.588,7.698,7.800,7.910,8.044,8.190,8.340,8.499,
     * 6.273,6.519,6.758,7.005,7.246,7.544,7.809,8.068,
     * 0.000,0.000,6.082,6.303,6.530,6.747,7.125,7.493/
C
      DATA ((EDEN(I,1,12,L,3),I=17,24),L=1,NZEN)/
     * 9.063,9.103,9.124,9.136,9.130,9.139,9.172,9.225,
     * 9.031,9.080,9.111,9.131,9.132,9.141,9.167,9.213,
     * 8.979,9.042,9.087,9.121,9.127,9.141,9.161,9.198,
     * 8.865,8.954,9.027,9.085,9.108,9.130,9.146,9.177,
     * 8.554,8.696,8.832,8.957,9.034,9.087,9.107,9.136,
     * 8.290,8.464,8.634,8.804,8.927,9.017,9.048,9.082,
     * 7.841,8.000,8.180,8.379,8.557,8.717,8.782,8.849,
     * 7.720,7.850,7.979,8.090,8.190,8.275,8.342,8.413,
     * 7.626,7.833,8.030,8.199,8.328,8.411,8.450,8.478,
     * 6.228,6.590,6.952,7.279,7.516,7.753,7.908,8.031,
     * 0.000,6.036,6.383,6.695,6.915,7.038,7.291,7.502/
C
      DATA ((EDEN(I,2,12,L,3),I=17,24),L=1,NZEN)/
     * 9.105,9.148,9.171,9.182,9.177,9.185,9.228,9.291,
     * 9.068,9.118,9.149,9.171,9.169,9.176,9.210,9.262,
     * 9.017,9.080,9.127,9.158,9.165,9.176,9.201,9.243,
     * 8.894,8.985,9.060,9.115,9.145,9.167,9.186,9.219,
     * 8.540,8.684,8.822,8.947,9.028,9.087,9.114,9.150,
     * 8.260,8.433,8.605,8.771,8.896,8.988,9.028,9.074,
     * 7.786,7.921,8.087,8.270,8.443,8.603,8.688,8.775,
     * 7.661,7.785,7.907,8.012,8.110,8.196,8.270,8.351,
     * 7.523,7.720,7.907,8.069,8.202,8.298,8.357,8.411,
     * 0.000,6.344,6.696,7.021,7.273,7.545,7.737,7.904,
     * 0.000,0.000,6.120,6.424,6.649,6.793,7.082,7.338/
C
      DATA ((EDEN(I,3,12,L,3),I=17,24),L=1,NZEN)/
     * 9.201,9.247,9.269,9.279,9.277,9.289,9.335,9.399,
     * 9.164,9.214,9.245,9.262,9.263,9.277,9.314,9.367,
     * 9.104,9.167,9.205,9.235,9.241,9.256,9.287,9.333,
     * 8.984,9.072,9.140,9.189,9.213,9.240,9.266,9.306,
     * 8.625,8.767,8.899,9.009,9.082,9.141,9.178,9.226,
     * 8.308,8.474,8.641,8.796,8.913,9.007,9.062,9.123,
     * 7.817,7.960,8.129,8.315,8.492,8.656,8.753,8.854,
     * 7.634,7.734,7.831,7.918,8.011,8.108,8.205,8.316,
     * 7.462,7.626,7.778,7.914,8.040,8.149,8.238,8.330,
     * 0.000,6.267,6.587,6.889,7.134,7.400,7.623,7.828,
     * 0.000,0.000,0.000,6.245,6.460,6.619,6.938,7.233/
C
      DATA ((EDEN(I,4,12,L,3),I=17,24),L=1,NZEN)/
     * 9.432,9.470,9.490,9.502,9.505,9.525,9.560,9.609,
     * 9.393,9.435,9.462,9.477,9.486,9.506,9.535,9.577,
     * 9.334,9.388,9.420,9.445,9.457,9.481,9.506,9.545,
     * 9.212,9.288,9.338,9.377,9.402,9.434,9.462,9.502,
     * 8.868,9.000,9.109,9.195,9.258,9.320,9.364,9.417,
     * 8.564,8.732,8.885,9.014,9.107,9.197,9.258,9.326,
     * 8.005,8.177,8.370,8.567,8.741,8.892,8.991,9.095,
     * 7.663,7.740,7.817,7.896,8.003,8.132,8.275,8.435,
     * 7.469,7.593,7.708,7.824,7.952,8.083,8.210,8.346,
     * 6.084,6.357,6.614,6.869,7.101,7.380,7.631,7.874,
     * 0.000,0.000,0.000,6.178,6.391,6.580,6.937,7.281/
C
      DATA ((EDEN(I,5,12,L,3),I=17,24),L=1,NZEN)/
     * 9.685,9.725,9.755,9.779,9.798,9.825,9.854,9.893,
     * 9.644,9.688,9.721,9.748,9.770,9.799,9.823,9.858,
     * 9.585,9.636,9.674,9.706,9.732,9.766,9.787,9.820,
     * 9.472,9.535,9.585,9.628,9.664,9.708,9.734,9.771,
     * 9.153,9.261,9.349,9.423,9.493,9.566,9.614,9.672,
     * 8.878,9.022,9.145,9.250,9.345,9.446,9.512,9.587,
     * 8.302,8.494,8.683,8.856,9.007,9.157,9.259,9.370,
     * 7.727,7.790,7.857,7.935,8.060,8.221,8.409,8.618,
     * 7.479,7.573,7.659,7.759,7.896,8.057,8.232,8.421,
     * 6.185,6.410,6.626,6.858,7.098,7.414,7.704,7.996,
     * 0.000,0.000,0.000,6.114,6.333,6.565,6.973,7.380/
C
      DATA ((EDEN(I,1,1,L,1),I=25,32),L=1,NZEN)/
     * 9.012,9.126,9.269,9.437,9.595,9.747,9.905,10.051,
     * 8.988,9.097,9.234,9.397,9.544,9.684,9.835,9.985,
     * 8.956,9.061,9.193,9.351,9.483,9.602,9.733,9.876,
     * 8.908,9.010,9.139,9.294,9.419,9.523,9.625,9.741,
     * 8.801,8.908,9.041,9.200,9.325,9.440,9.552,9.666,
     * 8.708,8.827,8.973,9.144,9.272,9.394,9.514,9.636,
     * 8.447,8.607,8.793,9.004,9.153,9.297,9.438,9.575,
     * 8.081,8.257,8.441,8.634,8.834,9.036,9.224,9.386,
     * 8.070,8.227,8.392,8.568,8.753,8.946,9.125,9.283,
     * 7.656,7.910,8.184,8.483,8.667,8.845,9.029,9.206,
     * 7.155,7.510,7.883,8.282,8.446,8.606,8.772,8.936/
C
      DATA ((EDEN(I,2,1,L,1),I=25,32),L=1,NZEN)/
     * 9.077,9.199,9.348,9.521,9.696,9.860,10.016,10.160,
     * 9.033,9.147,9.288,9.453,9.612,9.764,9.916,10.062,
     * 9.000,9.106,9.240,9.397,9.533,9.660,9.799,9.939,
     * 8.947,9.049,9.178,9.329,9.450,9.550,9.649,9.757,
     * 8.812,8.925,9.064,9.224,9.353,9.469,9.577,9.688,
     * 8.701,8.831,8.985,9.160,9.298,9.422,9.540,9.657,
     * 8.390,8.569,8.771,8.995,9.161,9.314,9.460,9.600,
     * 8.055,8.242,8.448,8.661,8.873,9.081,9.268,9.426,
     * 8.037,8.219,8.416,8.618,8.820,9.018,9.196,9.345,
     * 7.584,7.882,8.194,8.523,8.745,8.939,9.119,9.288,
     * 7.041,7.439,7.848,8.273,8.502,8.684,8.848,9.004/
C
      DATA ((EDEN(I,3,1,L,1),I=25,32),L=1,NZEN)/
     * 9.177,9.295,9.436,9.597,9.764,9.920,10.067,10.200,
     * 9.134,9.243,9.376,9.528,9.679,9.824,9.969,10.104,
     * 9.085,9.191,9.319,9.466,9.594,9.715,9.843,9.975,
     * 9.035,9.139,9.266,9.410,9.525,9.618,9.710,9.817,
     * 8.900,9.017,9.155,9.308,9.431,9.537,9.631,9.726,
     * 8.764,8.902,9.059,9.231,9.369,9.488,9.593,9.698,
     * 8.482,8.668,8.872,9.090,9.254,9.397,9.523,9.643,
     * 8.047,8.253,8.483,8.714,8.933,9.137,9.316,9.463,
     * 8.005,8.212,8.442,8.668,8.880,9.077,9.246,9.382,
     * 7.559,7.880,8.207,8.539,8.795,8.993,9.163,9.317,
     * 7.008,7.425,7.845,8.267,8.552,8.752,8.911,9.052/
C
      DATA ((EDEN(I,4,1,L,1),I=25,32),L=1,NZEN)/
     * 9.320,9.424,9.546,9.682,9.821,9.952,10.080,10.203,
     * 9.276,9.375,9.491,9.621,9.744,9.862,9.987,10.110,
     * 9.231,9.328,9.441,9.567,9.671,9.766,9.870,9.988,
     * 9.170,9.270,9.385,9.512,9.608,9.684,9.758,9.849,
     * 9.039,9.157,9.288,9.427,9.533,9.620,9.696,9.776,
     * 8.924,9.063,9.212,9.368,9.486,9.584,9.666,9.750,
     * 8.672,8.855,9.047,9.245,9.388,9.507,9.606,9.702,
     * 8.131,8.366,8.610,8.840,9.048,9.233,9.389,9.514,
     * 8.036,8.274,8.519,8.749,8.953,9.133,9.281,9.397,
     * 7.628,7.966,8.299,8.622,8.874,9.057,9.207,9.336,
     * 7.108,7.551,7.983,8.401,8.665,8.841,8.976,9.092/
C
      DATA ((EDEN(I,5,1,L,1),I=25,32),L=1,NZEN)/
     * 9.537,9.629,9.730,9.840,9.948,10.052,10.155,10.259,
     * 9.489,9.576,9.672,9.775,9.870,9.963,10.066,10.164,
     * 9.439,9.525,9.619,9.718,9.793,9.864,9.949,10.045,
     * 9.374,9.465,9.562,9.663,9.730,9.780,9.834,9.908,
     * 9.243,9.356,9.472,9.587,9.664,9.723,9.776,9.839,
     * 9.143,9.275,9.409,9.540,9.625,9.693,9.753,9.818,
     * 8.910,9.085,9.259,9.427,9.534,9.621,9.695,9.772,
     * 8.308,8.572,8.814,9.029,9.213,9.369,9.495,9.596,
     * 8.155,8.428,8.681,8.902,9.089,9.244,9.366,9.461,
     * 7.783,8.150,8.496,8.817,9.017,9.164,9.284,9.393,
     * 7.280,7.762,8.218,8.641,8.834,8.965,9.066,9.157/
C
      DATA ((EDEN(I,1,2,L,1),I=25,32),L=1,NZEN)/
     * 8.999,9.115,9.261,9.431,9.591,9.745,9.901,10.051,
     * 8.974,9.086,9.226,9.391,9.540,9.683,9.833,9.981,
     * 8.944,9.050,9.185,9.345,9.480,9.599,9.731,9.874,
     * 8.896,9.000,9.132,9.288,9.415,9.520,9.624,9.739,
     * 8.785,8.894,9.031,9.193,9.322,9.438,9.550,9.664,
     * 8.691,8.814,8.963,9.138,9.268,9.391,9.512,9.634,
     * 8.428,8.592,8.782,8.997,9.150,9.296,9.436,9.574,
     * 8.073,8.249,8.436,8.631,8.832,9.035,9.222,9.385,
     * 8.061,8.220,8.387,8.565,8.751,8.944,9.124,9.281,
     * 7.644,7.900,8.176,8.477,8.664,8.845,9.026,9.202,
     * 7.146,7.501,7.874,8.273,8.443,8.604,8.770,8.936/
C
      DATA ((EDEN(I,2,2,L,1),I=25,32),L=1,NZEN)/
     * 9.053,9.175,9.325,9.499,9.672,9.833,9.988,10.133,
     * 9.014,9.128,9.271,9.436,9.592,9.739,9.889,10.035,
     * 8.981,9.089,9.224,9.382,9.514,9.633,9.765,9.906,
     * 8.924,9.030,9.162,9.317,9.441,9.542,9.640,9.751,
     * 8.791,8.907,9.049,9.213,9.343,9.459,9.568,9.680,
     * 8.680,8.813,8.970,9.149,9.287,9.413,9.531,9.649,
     * 8.379,8.559,8.763,8.988,9.154,9.309,9.453,9.593,
     * 8.046,8.234,8.440,8.651,8.862,9.069,9.257,9.415,
     * 8.043,8.223,8.418,8.617,8.817,9.015,9.193,9.344,
     * 7.587,7.880,8.187,8.513,8.731,8.922,9.100,9.270,
     * 7.061,7.453,7.857,8.278,8.494,8.670,8.835,8.993/
C
      DATA ((EDEN(I,3,2,L,1),I=25,32),L=1,NZEN)/
     * 9.151,9.269,9.412,9.575,9.734,9.883,10.027,10.163,
     * 9.109,9.221,9.356,9.511,9.654,9.790,9.931,10.063,
     * 9.065,9.173,9.304,9.454,9.577,9.688,9.810,9.940,
     * 9.010,9.116,9.245,9.392,9.506,9.596,9.686,9.789,
     * 8.870,8.990,9.130,9.287,9.413,9.518,9.613,9.712,
     * 8.738,8.878,9.037,9.212,9.350,9.468,9.576,9.683,
     * 8.458,8.645,8.849,9.069,9.234,9.378,9.504,9.629,
     * 8.054,8.257,8.482,8.706,8.921,9.124,9.303,9.451,
     * 8.018,8.220,8.441,8.658,8.866,9.060,9.229,9.368,
     * 7.567,7.882,8.205,8.535,8.785,8.980,9.153,9.308,
     * 7.052,7.462,7.875,8.293,8.550,8.736,8.892,9.032/
C
      DATA ((EDEN(I,4,2,L,1),I=25,32),L=1,NZEN)/
     * 9.293,9.399,9.524,9.663,9.795,9.921,10.048,10.169,
     * 9.253,9.355,9.474,9.608,9.722,9.833,9.953,10.073,
     * 9.210,9.310,9.427,9.557,9.656,9.742,9.842,9.956,
     * 9.147,9.250,9.368,9.499,9.594,9.667,9.740,9.829,
     * 9.011,9.131,9.263,9.405,9.511,9.598,9.676,9.760,
     * 8.892,9.031,9.182,9.341,9.461,9.557,9.644,9.733,
     * 8.613,8.801,8.998,9.202,9.350,9.473,9.578,9.681,
     * 8.137,8.363,8.598,8.821,9.024,9.209,9.368,9.498,
     * 8.052,8.278,8.515,8.737,8.938,9.119,9.270,9.391,
     * 7.647,7.976,8.300,8.618,8.860,9.040,9.190,9.328,
     * 7.144,7.574,7.995,8.407,8.650,8.818,8.954,9.076/
C
      DATA ((EDEN(I,5,2,L,1),I=25,32),L=1,NZEN)/
     * 9.492,9.588,9.694,9.807,9.912,10.012,10.116,10.219,
     * 9.448,9.540,9.641,9.749,9.838,9.926,10.023,10.124,
     * 9.401,9.493,9.592,9.697,9.770,9.833,9.913,10.009,
     * 9.337,9.432,9.535,9.642,9.710,9.758,9.811,9.884,
     * 9.203,9.318,9.437,9.557,9.635,9.697,9.753,9.821,
     * 9.097,9.232,9.369,9.505,9.592,9.662,9.726,9.797,
     * 8.850,9.029,9.208,9.383,9.495,9.585,9.665,9.748,
     * 8.295,8.546,8.781,8.991,9.174,9.332,9.464,9.572,
     * 8.169,8.427,8.668,8.882,9.064,9.220,9.347,9.448,
     * 7.786,8.139,8.475,8.790,8.991,9.140,9.267,9.382,
     * 7.306,7.772,8.214,8.629,8.810,8.941,9.050,9.151/
C
      DATA ((EDEN(I,1,3,L,1),I=25,32),L=1,NZEN)/
     * 8.983,9.102,9.250,9.424,9.591,9.747,9.905,10.055,
     * 8.957,9.071,9.214,9.382,9.538,9.683,9.836,9.985,
     * 8.925,9.034,9.172,9.335,9.476,9.598,9.732,9.875,
     * 8.875,8.983,9.118,9.278,9.410,9.517,9.622,9.738,
     * 8.763,8.877,9.018,9.184,9.318,9.435,9.547,9.661,
     * 8.667,8.795,8.950,9.130,9.265,9.389,9.510,9.631,
     * 8.405,8.574,8.770,8.990,9.148,9.294,9.434,9.571,
     * 8.058,8.236,8.426,8.624,8.826,9.030,9.218,9.381,
     * 8.046,8.206,8.376,8.555,8.743,8.936,9.116,9.273,
     * 7.625,7.883,8.160,8.463,8.658,8.841,9.020,9.196,
     * 7.131,7.486,7.859,8.257,8.436,8.599,8.764,8.928/
C
      DATA ((EDEN(I,2,3,L,1),I=25,32),L=1,NZEN)/
     * 9.016,9.140,9.292,9.467,9.636,9.793,9.948,10.092,
     * 8.986,9.104,9.249,9.418,9.571,9.718,9.867,10.013,
     * 8.947,9.059,9.198,9.361,9.497,9.616,9.747,9.888,
     * 8.889,8.999,9.136,9.296,9.427,9.529,9.630,9.743,
     * 8.759,8.879,9.024,9.192,9.328,9.445,9.556,9.668,
     * 8.643,8.781,8.943,9.128,9.271,9.397,9.517,9.639,
     * 8.368,8.549,8.753,8.981,9.150,9.301,9.442,9.580,
     * 8.036,8.221,8.424,8.633,8.842,9.049,9.236,9.396,
     * 8.034,8.208,8.396,8.589,8.784,8.980,9.158,9.310,
     * 7.593,7.873,8.168,8.484,8.698,8.884,9.063,9.236,
     * 7.092,7.467,7.856,8.266,8.470,8.640,8.803,8.961/
C
      DATA ((EDEN(I,3,3,L,1),I=25,32),L=1,NZEN)/
     * 9.104,9.226,9.372,9.539,9.692,9.835,9.979,10.113,
     * 9.066,9.182,9.322,9.482,9.619,9.745,9.881,10.018,
     * 9.026,9.139,9.274,9.430,9.551,9.654,9.770,9.899,
     * 8.953,9.065,9.199,9.353,9.475,9.567,9.657,9.762,
     * 8.805,8.930,9.075,9.240,9.373,9.482,9.584,9.690,
     * 8.690,8.832,8.993,9.173,9.317,9.436,9.546,9.659,
     * 8.396,8.585,8.792,9.017,9.193,9.340,9.473,9.604,
     * 8.057,8.249,8.464,8.680,8.890,9.092,9.273,9.424,
     * 8.035,8.220,8.426,8.631,8.831,9.023,9.195,9.338,
     * 7.596,7.887,8.189,8.503,8.742,8.932,9.105,9.267,
     * 7.107,7.490,7.880,8.281,8.512,8.688,8.844,8.991/
C
      DATA ((EDEN(I,4,3,L,1),I=25,32),L=1,NZEN)/
     * 9.242,9.357,9.489,9.636,9.766,9.889,10.013,10.135,
     * 9.204,9.314,9.441,9.583,9.696,9.801,9.918,10.041,
     * 9.162,9.270,9.395,9.533,9.632,9.714,9.810,9.924,
     * 9.099,9.207,9.332,9.469,9.568,9.640,9.713,9.804,
     * 8.940,9.063,9.201,9.350,9.466,9.556,9.641,9.733,
     * 8.808,8.952,9.107,9.274,9.406,9.509,9.602,9.702,
     * 8.537,8.725,8.924,9.134,9.291,9.421,9.534,9.649,
     * 8.132,8.334,8.556,8.771,8.973,9.161,9.327,9.466,
     * 8.084,8.281,8.496,8.703,8.896,9.076,9.234,9.363,
     * 7.665,7.964,8.262,8.563,8.808,8.990,9.149,9.295,
     * 7.208,7.600,7.988,8.376,8.597,8.758,8.898,9.030/
C
      DATA ((EDEN(I,5,3,L,1),I=25,32),L=1,NZEN)/
     * 9.417,9.525,9.643,9.769,9.876,9.975,10.082,10.188,
     * 9.377,9.481,9.594,9.715,9.805,9.889,9.986,10.091,
     * 9.333,9.436,9.547,9.664,9.740,9.801,9.878,9.975,
     * 9.268,9.373,9.486,9.604,9.678,9.728,9.781,9.857,
     * 9.119,9.241,9.367,9.497,9.585,9.651,9.714,9.790,
     * 8.996,9.138,9.282,9.429,9.530,9.608,9.679,9.761,
     * 8.723,8.911,9.101,9.291,9.418,9.521,9.611,9.707,
     * 8.282,8.501,8.718,8.918,9.099,9.263,9.406,9.527,
     * 8.201,8.419,8.635,8.833,9.008,9.166,9.303,9.417,
     * 7.800,8.116,8.420,8.712,8.924,9.081,9.219,9.347,
     * 7.378,7.793,8.190,8.573,8.736,8.863,8.981,9.097/
C
      DATA ((EDEN(I,1,4,L,1),I=25,32),L=1,NZEN)/
     * 8.970,9.092,9.244,9.420,9.597,9.760,9.921,10.071,
     * 8.943,9.060,9.205,9.376,9.540,9.693,9.850,10.000,
     * 8.910,9.021,9.161,9.326,9.473,9.603,9.742,9.889,
     * 8.859,8.968,9.106,9.268,9.405,9.514,9.621,9.739,
     * 8.746,8.862,9.005,9.174,9.313,9.432,9.543,9.657,
     * 8.652,8.783,8.940,9.123,9.261,9.387,9.507,9.627,
     * 8.392,8.564,8.762,8.985,9.146,9.294,9.432,9.569,
     * 8.044,8.222,8.414,8.615,8.819,9.023,9.212,9.374,
     * 8.029,8.188,8.360,8.542,8.731,8.924,9.104,9.260,
     * 7.611,7.867,8.143,8.444,8.646,8.828,9.008,9.183,
     * 7.119,7.470,7.839,8.234,8.423,8.589,8.754,8.916/
C
      DATA ((EDEN(I,2,4,L,1),I=25,32),L=1,NZEN)/
     * 8.978,9.104,9.257,9.434,9.602,9.756,9.911,10.059,
     * 8.948,9.068,9.216,9.388,9.544,9.688,9.838,9.985,
     * 8.910,9.026,9.169,9.336,9.478,9.601,9.734,9.878,
     * 8.853,8.967,9.107,9.272,9.409,9.515,9.618,9.736,
     * 8.739,8.861,9.009,9.181,9.318,9.433,9.542,9.656,
     * 8.633,8.769,8.931,9.116,9.260,9.385,9.502,9.623,
     * 8.375,8.551,8.751,8.974,9.143,9.291,9.429,9.564,
     * 8.047,8.224,8.420,8.621,8.824,9.027,9.214,9.374,
     * 8.031,8.189,8.364,8.547,8.735,8.927,9.106,9.260,
     * 7.617,7.871,8.143,8.440,8.648,8.832,9.009,9.185,
     * 7.137,7.485,7.849,8.238,8.429,8.592,8.755,8.918/
C
      DATA ((EDEN(I,3,4,L,1),I=25,32),L=1,NZEN)/
     * 9.045,9.173,9.326,9.499,9.653,9.796,9.943,10.080,
     * 9.008,9.132,9.280,9.447,9.585,9.708,9.843,9.981,
     * 8.959,9.079,9.222,9.385,9.517,9.624,9.742,9.877,
     * 8.887,9.006,9.147,9.309,9.443,9.541,9.635,9.745,
     * 8.738,8.866,9.015,9.184,9.334,9.448,9.553,9.663,
     * 8.625,8.769,8.934,9.118,9.274,9.399,9.513,9.631,
     * 8.362,8.545,8.748,8.971,9.155,9.304,9.440,9.573,
     * 8.058,8.229,8.433,8.640,8.845,9.046,9.231,9.387,
     * 8.041,8.193,8.379,8.568,8.759,8.949,9.125,9.274,
     * 7.624,7.874,8.138,8.421,8.665,8.856,9.030,9.200,
     * 7.167,7.504,7.852,8.220,8.442,8.612,8.771,8.927/
C
      DATA ((EDEN(I,4,4,L,1),I=25,32),L=1,NZEN)/
     * 9.163,9.293,9.441,9.604,9.749,9.882,10.013,10.141,
     * 9.123,9.248,9.389,9.546,9.674,9.790,9.914,10.043,
     * 9.075,9.196,9.333,9.485,9.599,9.689,9.790,9.911,
     * 8.979,9.099,9.235,9.385,9.513,9.600,9.681,9.780,
     * 8.808,8.938,9.082,9.242,9.392,9.501,9.597,9.698,
     * 8.673,8.819,8.980,9.154,9.322,9.446,9.554,9.664,
     * 8.406,8.590,8.787,8.998,9.198,9.351,9.480,9.606,
     * 8.100,8.258,8.469,8.684,8.891,9.088,9.266,9.417,
     * 8.076,8.217,8.411,8.611,8.803,8.990,9.161,9.304,
     * 7.666,7.909,8.158,8.419,8.700,8.899,9.068,9.229,
     * 7.254,7.579,7.906,8.246,8.483,8.653,8.804,8.951/
C
      DATA ((EDEN(I,5,4,L,1),I=25,32),L=1,NZEN)/
     * 9.296,9.428,9.569,9.719,9.858,9.979,10.098,10.213,
     * 9.253,9.379,9.514,9.657,9.779,9.883,9.995,10.112,
     * 9.207,9.328,9.459,9.597,9.703,9.780,9.871,9.981,
     * 9.131,9.253,9.382,9.519,9.624,9.687,9.749,9.834,
     * 8.923,9.055,9.194,9.340,9.481,9.578,9.657,9.746,
     * 8.781,8.931,9.085,9.246,9.407,9.519,9.612,9.710,
     * 8.512,8.697,8.887,9.083,9.280,9.422,9.538,9.653,
     * 8.205,8.357,8.566,8.776,8.972,9.156,9.322,9.461,
     * 8.175,8.314,8.507,8.702,8.884,9.057,9.214,9.346,
     * 7.757,7.999,8.236,8.477,8.765,8.964,9.124,9.275,
     * 7.391,7.712,8.025,8.341,8.563,8.717,8.854,8.990/
C
      DATA ((EDEN(I,1,5,L,1),I=25,32),L=1,NZEN)/
     * 8.973,9.097,9.250,9.427,9.612,9.783,9.949,10.099,
     * 8.945,9.062,9.208,9.379,9.552,9.713,9.876,10.029,
     * 8.908,9.019,9.159,9.323,9.476,9.613,9.760,9.910,
     * 8.858,8.966,9.102,9.263,9.401,9.512,9.622,9.745,
     * 8.746,8.860,9.001,9.168,9.309,9.429,9.540,9.651,
     * 8.655,8.784,8.939,9.119,9.259,9.385,9.503,9.623,
     * 8.399,8.568,8.763,8.983,9.146,9.294,9.431,9.565,
     * 8.039,8.216,8.409,8.610,8.814,9.018,9.206,9.367,
     * 8.018,8.175,8.347,8.529,8.717,8.910,9.090,9.246,
     * 7.610,7.861,8.132,8.427,8.633,8.817,8.996,9.170,
     * 7.119,7.464,7.827,8.216,8.412,8.578,8.742,8.905/
C
      DATA ((EDEN(I,2,5,L,1),I=25,32),L=1,NZEN)/
     * 8.957,9.083,9.236,9.414,9.582,9.736,9.894,10.046,
     * 8.930,9.052,9.202,9.375,9.545,9.700,9.861,10.015,
     * 8.894,9.012,9.157,9.326,9.477,9.611,9.755,9.906,
     * 8.844,8.958,9.100,9.266,9.405,9.511,9.619,9.743,
     * 8.737,8.856,9.001,9.171,9.309,9.423,9.530,9.642,
     * 8.639,8.769,8.925,9.105,9.251,9.374,9.488,9.607,
     * 8.399,8.565,8.757,8.974,9.140,9.284,9.417,9.549,
     * 8.058,8.227,8.415,8.609,8.807,9.006,9.191,9.351,
     * 8.032,8.173,8.334,8.504,8.686,8.875,9.053,9.209,
     * 7.650,7.876,8.121,8.395,8.599,8.779,8.955,9.130,
     * 7.161,7.483,7.824,8.193,8.387,8.552,8.717,8.884/
C
      DATA ((EDEN(I,3,5,L,1),I=25,32),L=1,NZEN)/
     * 8.993,9.129,9.288,9.468,9.635,9.785,9.939,10.080,
     * 8.950,9.080,9.233,9.407,9.564,9.701,9.847,9.992,
     * 8.895,9.020,9.169,9.339,9.494,9.621,9.758,9.905,
     * 8.825,8.948,9.094,9.261,9.415,9.522,9.625,9.748,
     * 8.690,8.818,8.968,9.139,9.303,9.423,9.528,9.640,
     * 8.584,8.722,8.882,9.064,9.237,9.367,9.483,9.603,
     * 8.353,8.525,8.719,8.935,9.130,9.281,9.414,9.548,
     * 8.048,8.194,8.389,8.593,8.795,8.998,9.186,9.347,
     * 8.032,8.148,8.315,8.494,8.678,8.869,9.050,9.206,
     * 7.659,7.867,8.092,8.343,8.593,8.782,8.960,9.133,
     * 7.190,7.480,7.785,8.118,8.359,8.540,8.708,8.875/
C
      DATA ((EDEN(I,4,5,L,1),I=25,32),L=1,NZEN)/
     * 9.084,9.229,9.393,9.571,9.757,9.917,10.066,10.201,
     * 9.038,9.175,9.329,9.498,9.667,9.811,9.955,10.091,
     * 8.960,9.088,9.234,9.395,9.550,9.668,9.794,9.932,
     * 8.846,8.970,9.110,9.267,9.438,9.554,9.653,9.767,
     * 8.672,8.800,8.943,9.103,9.297,9.439,9.552,9.663,
     * 8.558,8.695,8.847,9.018,9.227,9.382,9.506,9.625,
     * 8.304,8.467,8.645,8.841,9.093,9.283,9.429,9.565,
     * 8.033,8.137,8.337,8.566,8.787,8.999,9.195,9.360,
     * 8.037,8.113,8.285,8.486,8.685,8.882,9.067,9.224,
     * 7.650,7.827,8.014,8.225,8.550,8.787,8.972,9.146,
     * 7.267,7.509,7.760,8.037,8.324,8.527,8.696,8.856/
C
      DATA ((EDEN(I,5,5,L,1),I=25,32),L=1,NZEN)/
     * 9.178,9.330,9.491,9.660,9.866,10.035,10.178,10.310,
     * 9.123,9.264,9.414,9.572,9.760,9.921,10.066,10.196,
     * 9.064,9.196,9.337,9.487,9.652,9.783,9.916,10.049,
     * 8.928,9.052,9.186,9.328,9.499,9.614,9.706,9.813,
     * 8.711,8.835,8.968,9.110,9.315,9.472,9.589,9.696,
     * 8.585,8.717,8.857,9.007,9.226,9.405,9.536,9.654,
     * 8.348,8.498,8.655,8.823,9.092,9.306,9.461,9.595,
     * 8.103,8.176,8.353,8.588,8.815,9.026,9.218,9.382,
     * 8.108,8.154,8.301,8.507,8.710,8.903,9.085,9.241,
     * 7.712,7.864,8.018,8.189,8.519,8.794,8.991,9.160,
     * 7.339,7.538,7.738,7.956,8.259,8.519,8.700,8.860/
C
      DATA ((EDEN(I,1,6,L,1),I=25,32),L=1,NZEN)/
     * 8.989,9.113,9.266,9.444,9.633,9.809,9.974,10.128,
     * 8.958,9.074,9.220,9.390,9.568,9.736,9.900,10.056,
     * 8.918,9.027,9.165,9.327,9.483,9.628,9.782,9.934,
     * 8.866,8.971,9.105,9.263,9.401,9.513,9.626,9.755,
     * 8.760,8.871,9.008,9.171,9.309,9.428,9.537,9.647,
     * 8.672,8.795,8.946,9.121,9.259,9.384,9.501,9.619,
     * 8.425,8.588,8.776,8.990,9.150,9.296,9.430,9.562,
     * 8.041,8.217,8.409,8.609,8.812,9.015,9.201,9.362,
     * 8.016,8.170,8.341,8.521,8.709,8.901,9.081,9.236,
     * 7.620,7.865,8.130,8.420,8.625,8.806,8.986,9.158,
     * 7.130,7.469,7.827,8.210,8.403,8.569,8.733,8.892/
C
      DATA ((EDEN(I,2,6,L,1),I=25,32),L=1,NZEN)/
     * 8.955,9.079,9.232,9.408,9.578,9.736,9.896,10.049,
     * 8.927,9.049,9.198,9.371,9.548,9.711,9.877,10.033,
     * 8.893,9.010,9.155,9.323,9.482,9.627,9.780,9.935,
     * 8.845,8.957,9.097,9.262,9.402,9.513,9.629,9.762,
     * 8.748,8.862,9.003,9.169,9.306,9.419,9.522,9.633,
     * 8.660,8.783,8.931,9.106,9.248,9.368,9.481,9.598,
     * 8.451,8.608,8.790,8.999,9.153,9.288,9.415,9.544,
     * 8.077,8.240,8.421,8.609,8.801,8.996,9.178,9.337,
     * 8.038,8.165,8.314,8.476,8.651,8.837,9.015,9.172,
     * 7.689,7.899,8.130,8.391,8.579,8.750,8.929,9.104,
     * 7.197,7.499,7.821,8.174,8.359,8.520,8.686,8.851/
C
      DATA ((EDEN(I,3,6,L,1),I=25,32),L=1,NZEN)/
     * 8.968,9.108,9.271,9.454,9.635,9.798,9.955,10.103,
     * 8.918,9.050,9.207,9.384,9.567,9.730,9.893,10.049,
     * 8.868,8.995,9.146,9.317,9.489,9.634,9.787,9.943,
     * 8.801,8.923,9.069,9.237,9.399,9.516,9.632,9.768,
     * 8.676,8.798,8.943,9.111,9.284,9.407,9.513,9.625,
     * 8.594,8.725,8.879,9.057,9.229,9.358,9.471,9.589,
     * 8.365,8.525,8.707,8.914,9.113,9.267,9.398,9.531,
     * 8.047,8.174,8.363,8.563,8.764,8.967,9.157,9.321,
     * 8.048,8.140,8.289,8.454,8.629,8.817,9.000,9.159,
     * 7.683,7.854,8.045,8.268,8.527,8.721,8.900,9.076,
     * 7.188,7.437,7.703,8.004,8.275,8.476,8.655,8.826/
C
      DATA ((EDEN(I,4,6,L,1),I=25,32),L=1,NZEN)/
     * 9.050,9.207,9.382,9.571,9.790,9.972,10.132,10.277,
     * 8.977,9.118,9.277,9.451,9.655,9.833,9.996,10.143,
     * 8.880,9.010,9.158,9.321,9.507,9.664,9.821,9.977,
     * 8.783,8.906,9.046,9.202,9.391,9.527,9.645,9.777,
     * 8.620,8.740,8.877,9.033,9.243,9.404,9.525,9.641,
     * 8.505,8.630,8.772,8.934,9.160,9.339,9.472,9.598,
     * 8.284,8.430,8.593,8.776,9.041,9.247,9.400,9.539,
     * 8.011,8.080,8.263,8.496,8.723,8.941,9.145,9.319,
     * 8.028,8.062,8.204,8.399,8.596,8.796,8.988,9.155,
     * 7.640,7.765,7.905,8.075,8.412,8.688,8.892,9.075,
     * 7.262,7.439,7.630,7.854,8.166,8.418,8.606,8.780/
C
      DATA ((EDEN(I,5,6,L,1),I=25,32),L=1,NZEN)/
     * 9.142,9.312,9.492,9.680,9.923,10.117,10.280,10.432,
     * 9.074,9.228,9.392,9.564,9.797,9.995,10.159,10.307,
     * 8.959,9.093,9.236,9.388,9.581,9.755,9.924,10.079,
     * 8.822,8.944,9.075,9.216,9.406,9.553,9.678,9.814,
     * 8.627,8.740,8.863,8.998,9.213,9.399,9.540,9.659,
     * 8.521,8.637,8.762,8.901,9.128,9.335,9.487,9.616,
     * 8.317,8.443,8.579,8.729,8.999,9.241,9.413,9.555,
     * 8.084,8.111,8.247,8.480,8.718,8.940,9.146,9.324,
     * 8.117,8.111,8.208,8.395,8.591,8.787,8.975,9.143,
     * 7.748,7.838,7.934,8.058,8.363,8.665,8.878,9.059,
     * 7.354,7.474,7.601,7.759,8.008,8.342,8.570,8.752/
C
      DATA ((EDEN(I,1,7,L,1),I=25,32),L=1,NZEN)/
     * 9.000,9.124,9.276,9.454,9.644,9.821,9.988,10.139,
     * 8.967,9.083,9.228,9.398,9.577,9.747,9.912,10.066,
     * 8.927,9.035,9.172,9.333,9.489,9.636,9.793,9.947,
     * 8.875,8.978,9.110,9.266,9.403,9.515,9.630,9.760,
     * 8.771,8.879,9.014,9.174,9.310,9.428,9.537,9.647,
     * 8.684,8.805,8.952,9.125,9.262,9.385,9.501,9.619,
     * 8.440,8.599,8.785,8.995,9.153,9.297,9.430,9.561,
     * 8.045,8.221,8.412,8.611,8.813,9.015,9.200,9.361,
     * 8.018,8.171,8.340,8.519,8.707,8.898,9.077,9.232,
     * 7.628,7.871,8.133,8.421,8.623,8.804,8.983,9.156,
     * 7.139,7.477,7.832,8.213,8.402,8.566,8.729,8.892/
C
      DATA ((EDEN(I,2,7,L,1),I=25,32),L=1,NZEN)/
     * 8.965,9.088,9.239,9.413,9.584,9.744,9.903,10.056,
     * 8.939,9.059,9.207,9.379,9.555,9.720,9.887,10.043,
     * 8.905,9.020,9.162,9.329,9.488,9.636,9.792,9.948,
     * 8.857,8.968,9.105,9.267,9.406,9.518,9.636,9.772,
     * 8.765,8.876,9.013,9.175,9.310,9.421,9.524,9.634,
     * 8.678,8.797,8.941,9.112,9.252,9.370,9.482,9.597,
     * 8.476,8.628,8.805,9.009,9.158,9.292,9.418,9.545,
     * 8.089,8.251,8.430,8.615,8.805,8.998,9.179,9.337,
     * 8.045,8.168,8.312,8.470,8.642,8.826,9.004,9.161,
     * 7.704,7.906,8.130,8.385,8.569,8.741,8.917,9.093,
     * 7.214,7.508,7.823,8.170,8.351,8.510,8.675,8.842/
C
      DATA ((EDEN(I,3,7,L,1),I=25,32),L=1,NZEN)/
     * 8.981,9.120,9.282,9.465,9.648,9.813,9.971,10.119,
     * 8.931,9.062,9.217,9.393,9.578,9.745,9.909,10.063,
     * 8.882,9.007,9.156,9.325,9.497,9.647,9.804,9.958,
     * 8.817,8.937,9.080,9.245,9.405,9.522,9.642,9.780,
     * 8.699,8.817,8.958,9.123,9.290,9.411,9.515,9.627,
     * 8.620,8.746,8.895,9.068,9.236,9.363,9.474,9.590,
     * 8.396,8.551,8.728,8.930,9.122,9.272,9.401,9.531,
     * 8.065,8.189,8.373,8.568,8.765,8.965,9.154,9.318,
     * 8.060,8.145,8.286,8.444,8.616,8.801,8.983,9.143,
     * 7.705,7.866,8.047,8.261,8.513,8.704,8.883,9.061,
     * 7.211,7.449,7.706,7.999,8.263,8.460,8.639,8.812/
C
      DATA ((EDEN(I,4,7,L,1),I=25,32),L=1,NZEN)/
     * 9.070,9.229,9.406,9.598,9.819,10.000,10.159,10.304,
     * 8.993,9.135,9.295,9.469,9.678,9.859,10.020,10.167,
     * 8.894,9.022,9.169,9.330,9.519,9.681,9.842,9.997,
     * 8.799,8.920,9.058,9.213,9.398,9.533,9.654,9.790,
     * 8.640,8.756,8.889,9.042,9.249,9.408,9.527,9.641,
     * 8.531,8.652,8.790,8.948,9.167,9.342,9.474,9.597,
     * 8.313,8.454,8.613,8.793,9.050,9.252,9.401,9.539,
     * 8.029,8.093,8.270,8.496,8.719,8.936,9.138,9.313,
     * 8.046,8.070,8.202,8.388,8.579,8.776,8.969,9.137,
     * 7.668,7.780,7.907,8.068,8.396,8.667,8.870,9.055,
     * 7.294,7.456,7.633,7.846,8.148,8.395,8.583,8.758/
C
      DATA ((EDEN(I,5,7,L,1),I=25,32),L=1,NZEN)/
     * 9.166,9.342,9.528,9.721,9.963,10.154,10.314,10.467,
     * 9.096,9.253,9.420,9.596,9.833,10.029,10.194,10.342,
     * 8.973,9.106,9.250,9.403,9.599,9.782,9.954,10.105,
     * 8.836,8.955,9.085,9.224,9.414,9.560,9.689,9.829,
     * 8.644,8.755,8.874,9.007,9.221,9.404,9.542,9.660,
     * 8.545,8.657,8.778,8.914,9.137,9.338,9.488,9.615,
     * 8.344,8.467,8.599,8.748,9.011,9.246,9.413,9.554,
     * 8.118,8.142,8.280,8.505,8.731,8.945,9.146,9.322,
     * 8.138,8.119,8.205,8.381,8.570,8.762,8.950,9.119,
     * 7.780,7.852,7.933,8.045,8.343,8.639,8.850,9.034,
     * 7.390,7.491,7.599,7.743,7.988,8.317,8.541,8.724/
C
      DATA ((EDEN(I,1,8,L,1),I=25,32),L=1,NZEN)/
     * 8.999,9.122,9.274,9.451,9.639,9.812,9.978,10.128,
     * 8.967,9.083,9.227,9.397,9.573,9.740,9.904,10.056,
     * 8.928,9.036,9.173,9.334,9.488,9.632,9.784,9.934,
     * 8.876,8.980,9.113,9.270,9.406,9.517,9.629,9.757,
     * 8.771,8.880,9.016,9.177,9.314,9.431,9.540,9.650,
     * 8.684,8.806,8.954,9.128,9.264,9.387,9.503,9.621,
     * 8.438,8.599,8.785,8.997,9.153,9.299,9.433,9.564,
     * 8.048,8.224,8.415,8.614,8.816,9.018,9.204,9.365,
     * 8.022,8.177,8.346,8.524,8.711,8.903,9.082,9.237,
     * 7.629,7.873,8.138,8.427,8.629,8.809,8.989,9.161,
     * 7.139,7.478,7.836,8.219,8.408,8.572,8.734,8.896/
C
      DATA ((EDEN(I,2,8,L,1),I=25,32),L=1,NZEN)/
     * 8.981,9.103,9.252,9.426,9.595,9.754,9.916,10.063,
     * 8.955,9.073,9.219,9.389,9.560,9.720,9.883,10.036,
     * 8.921,9.034,9.174,9.339,9.491,9.632,9.783,9.933,
     * 8.873,8.982,9.118,9.279,9.413,9.521,9.634,9.763,
     * 8.776,8.887,9.024,9.186,9.319,9.431,9.533,9.643,
     * 8.687,8.807,8.953,9.124,9.262,9.382,9.493,9.607,
     * 8.459,8.614,8.793,8.999,9.155,9.295,9.422,9.550,
     * 8.087,8.251,8.432,8.619,8.811,9.005,9.187,9.345,
     * 8.049,8.178,8.326,8.488,8.661,8.847,9.024,9.181,
     * 7.693,7.905,8.138,8.401,8.587,8.759,8.935,9.111,
     * 7.199,7.502,7.823,8.176,8.363,8.524,8.689,8.854/
C
      DATA ((EDEN(I,3,8,L,1),I=25,32),L=1,NZEN)/
     * 9.023,9.156,9.313,9.490,9.661,9.818,9.971,10.116,
     * 8.976,9.102,9.251,9.421,9.591,9.745,9.901,10.049,
     * 8.929,9.049,9.192,9.356,9.512,9.647,9.792,9.941,
     * 8.866,8.981,9.120,9.280,9.427,9.535,9.644,9.772,
     * 8.747,8.863,9.002,9.164,9.319,9.434,9.536,9.644,
     * 8.649,8.774,8.920,9.091,9.253,9.380,9.489,9.605,
     * 8.431,8.589,8.768,8.972,9.152,9.294,9.422,9.550,
     * 8.089,8.224,8.407,8.599,8.792,8.989,9.175,9.336,
     * 8.079,8.178,8.322,8.480,8.651,8.835,9.014,9.171,
     * 7.714,7.892,8.090,8.318,8.554,8.737,8.912,9.088,
     * 7.219,7.477,7.753,8.061,8.308,8.495,8.668,8.840/
C
      DATA ((EDEN(I,4,8,L,1),I=25,32),L=1,NZEN)/
     * 9.127,9.276,9.442,9.624,9.818,9.981,10.129,10.263,
     * 9.061,9.195,9.347,9.514,9.693,9.850,9.997,10.134,
     * 8.992,9.115,9.257,9.413,9.573,9.704,9.841,9.981,
     * 8.884,9.000,9.134,9.284,9.451,9.566,9.669,9.790,
     * 8.726,8.842,8.975,9.126,9.313,9.449,9.556,9.663,
     * 8.621,8.745,8.885,9.045,9.242,9.391,9.509,9.624,
     * 8.377,8.527,8.693,8.880,9.114,9.292,9.430,9.562,
     * 8.088,8.178,8.362,8.573,8.780,8.983,9.176,9.342,
     * 8.081,8.131,8.275,8.454,8.637,8.828,9.014,9.176,
     * 7.719,7.859,8.012,8.195,8.503,8.730,8.914,9.092,
     * 7.346,7.546,7.758,8.003,8.269,8.463,8.630,8.797/
C
      DATA ((EDEN(I,5,8,L,1),I=25,32),L=1,NZEN)/
     * 9.231,9.392,9.562,9.740,9.951,10.120,10.259,10.392,
     * 9.169,9.313,9.468,9.630,9.833,10.003,10.146,10.275,
     * 9.087,9.216,9.354,9.501,9.674,9.822,9.962,10.095,
     * 8.942,9.059,9.185,9.321,9.492,9.614,9.715,9.831,
     * 8.762,8.874,8.996,9.130,9.325,9.477,9.590,9.694,
     * 8.646,8.764,8.891,9.031,9.238,9.410,9.536,9.650,
     * 8.418,8.554,8.699,8.858,9.106,9.312,9.458,9.588,
     * 8.163,8.217,8.377,8.592,8.801,9.003,9.193,9.358,
     * 8.178,8.193,8.309,8.481,8.657,8.839,9.018,9.178,
     * 7.821,7.927,8.039,8.178,8.484,8.731,8.919,9.092,
     * 7.436,7.580,7.729,7.908,8.191,8.440,8.616,8.785/
C
      DATA ((EDEN(I,1,9,L,1),I=25,32),L=1,NZEN)/
     * 8.989,9.111,9.263,9.439,9.622,9.790,9.952,10.102,
     * 8.960,9.076,9.220,9.390,9.560,9.719,9.880,10.033,
     * 8.924,9.033,9.171,9.334,9.483,9.620,9.766,9.914,
     * 8.873,8.979,9.114,9.273,9.408,9.517,9.627,9.749,
     * 8.764,8.876,9.016,9.180,9.316,9.434,9.544,9.656,
     * 8.675,8.800,8.953,9.130,9.266,9.390,9.509,9.627,
     * 8.424,8.589,8.781,8.997,9.153,9.300,9.436,9.569,
     * 8.050,8.227,8.419,8.617,8.820,9.023,9.209,9.371,
     * 8.028,8.185,8.355,8.535,8.722,8.914,9.093,9.249,
     * 7.625,7.874,8.144,8.439,8.637,8.819,8.999,9.172,
     * 7.134,7.479,7.842,8.231,8.417,8.581,8.744,8.905/
C
      DATA ((EDEN(I,2,9,L,1),I=25,32),L=1,NZEN)/
     * 9.000,9.121,9.270,9.443,9.612,9.771,9.928,10.076,
     * 8.976,9.092,9.235,9.403,9.566,9.719,9.876,10.026,
     * 8.940,9.050,9.189,9.351,9.494,9.622,9.762,9.907,
     * 8.890,8.998,9.134,9.293,9.423,9.527,9.631,9.750,
     * 8.780,8.894,9.033,9.198,9.331,9.444,9.549,9.659,
     * 8.685,8.810,8.961,9.136,9.273,9.395,9.509,9.625,
     * 8.439,8.602,8.789,9.002,9.164,9.305,9.436,9.566,
     * 8.073,8.242,8.429,8.622,8.820,9.019,9.204,9.363,
     * 8.046,8.189,8.350,8.521,8.703,8.891,9.069,9.224,
     * 7.657,7.887,8.137,8.415,8.617,8.795,8.973,9.145,
     * 7.160,7.482,7.822,8.190,8.391,8.558,8.724,8.884/
C
      DATA ((EDEN(I,3,9,L,1),I=25,32),L=1,NZEN)/
     * 9.076,9.201,9.349,9.517,9.677,9.824,9.971,10.109,
     * 9.040,9.156,9.297,9.458,9.602,9.735,9.876,10.016,
     * 8.996,9.108,9.244,9.400,9.531,9.642,9.765,9.901,
     * 8.927,9.037,9.171,9.325,9.457,9.555,9.651,9.764,
     * 8.801,8.916,9.054,9.214,9.354,9.464,9.564,9.670,
     * 8.700,8.827,8.976,9.146,9.294,9.413,9.522,9.634,
     * 8.449,8.613,8.798,9.006,9.179,9.321,9.448,9.576,
     * 8.100,8.254,8.442,8.637,8.832,9.028,9.211,9.368,
     * 8.076,8.201,8.362,8.533,8.710,8.896,9.072,9.225,
     * 7.694,7.910,8.142,8.399,8.624,8.804,8.979,9.148,
     * 7.233,7.528,7.838,8.174,8.388,8.553,8.714,8.876/
C
      DATA ((EDEN(I,4,9,L,1),I=25,32),L=1,NZEN)/
     * 9.207,9.337,9.485,9.647,9.802,9.940,10.073,10.198,
     * 9.165,9.286,9.424,9.577,9.715,9.840,9.968,10.095,
     * 9.105,9.218,9.348,9.493,9.613,9.711,9.821,9.946,
     * 9.028,9.138,9.264,9.406,9.528,9.612,9.694,9.795,
     * 8.866,8.981,9.112,9.259,9.403,9.510,9.602,9.700,
     * 8.757,8.885,9.028,9.187,9.338,9.455,9.556,9.663,
     * 8.499,8.663,8.842,9.038,9.219,9.361,9.483,9.604,
     * 8.157,8.292,8.478,8.673,8.866,9.057,9.236,9.389,
     * 8.144,8.251,8.408,8.576,8.747,8.925,9.096,9.245,
     * 7.752,7.949,8.156,8.384,8.639,8.830,8.999,9.164,
     * 7.381,7.653,7.933,8.235,8.424,8.572,8.722,8.876/
C
      DATA ((EDEN(I,5,9,L,1),I=25,32),L=1,NZEN)/
     * 9.354,9.488,9.632,9.784,9.930,10.056,10.172,10.283,
     * 9.307,9.431,9.564,9.705,9.836,9.951,10.066,10.178,
     * 9.258,9.373,9.498,9.630,9.737,9.827,9.929,10.040,
     * 9.178,9.288,9.407,9.534,9.634,9.699,9.762,9.851,
     * 8.991,9.105,9.228,9.359,9.490,9.581,9.658,9.745,
     * 8.879,9.005,9.139,9.282,9.422,9.524,9.612,9.706,
     * 8.615,8.776,8.944,9.122,9.295,9.425,9.533,9.645,
     * 8.287,8.411,8.587,8.768,8.944,9.118,9.283,9.427,
     * 8.265,8.359,8.506,8.661,8.818,8.980,9.139,9.280,
     * 7.889,8.076,8.262,8.464,8.706,8.881,9.037,9.195,
     * 7.527,7.778,8.029,8.295,8.479,8.613,8.749,8.896/
C
      DATA ((EDEN(I,1,10,L,1),I=25,32),L=1,NZEN)/
     * 8.986,9.107,9.258,9.433,9.607,9.768,9.929,10.077,
     * 8.960,9.075,9.219,9.388,9.550,9.701,9.857,10.008,
     * 8.926,9.035,9.174,9.337,9.481,9.610,9.749,9.894,
     * 8.877,8.984,9.119,9.279,9.412,9.520,9.626,9.745,
     * 8.766,8.880,9.020,9.186,9.320,9.438,9.549,9.662,
     * 8.674,8.801,8.955,9.135,9.269,9.394,9.513,9.632,
     * 8.416,8.584,8.778,8.997,9.155,9.301,9.438,9.573,
     * 8.056,8.234,8.425,8.623,8.826,9.029,9.216,9.378,
     * 8.038,8.197,8.367,8.547,8.734,8.926,9.106,9.262,
     * 7.625,7.880,8.154,8.453,8.650,8.832,9.011,9.185,
     * 7.133,7.484,7.852,8.246,8.429,8.591,8.755,8.916/
C
      DATA ((EDEN(I,2,10,L,1),I=25,32),L=1,NZEN)/
     * 9.025,9.146,9.294,9.467,9.637,9.797,9.955,10.101,
     * 8.996,9.110,9.252,9.417,9.574,9.720,9.871,10.020,
     * 8.960,9.069,9.205,9.364,9.502,9.623,9.755,9.896,
     * 8.906,9.012,9.146,9.303,9.433,9.537,9.637,9.750,
     * 8.784,8.899,9.039,9.203,9.339,9.454,9.562,9.672,
     * 8.679,8.810,8.965,9.145,9.284,9.407,9.524,9.642,
     * 8.413,8.586,8.783,9.003,9.165,9.314,9.450,9.583,
     * 8.050,8.225,8.422,8.626,8.832,9.036,9.223,9.382,
     * 8.040,8.200,8.377,8.562,8.751,8.943,9.122,9.276,
     * 7.618,7.877,8.153,8.453,8.664,8.848,9.027,9.200,
     * 7.124,7.474,7.839,8.227,8.433,8.603,8.766,8.923/
C
      DATA ((EDEN(I,3,10,L,1),I=25,32),L=1,NZEN)/
     * 9.124,9.242,9.383,9.545,9.698,9.844,9.987,10.125,
     * 9.090,9.200,9.335,9.489,9.626,9.755,9.891,10.025,
     * 9.051,9.156,9.285,9.434,9.555,9.660,9.776,9.905,
     * 8.988,9.093,9.220,9.367,9.486,9.580,9.669,9.773,
     * 8.853,8.967,9.103,9.258,9.387,9.496,9.594,9.696,
     * 8.734,8.866,9.017,9.189,9.329,9.447,9.556,9.665,
     * 8.465,8.639,8.833,9.046,9.211,9.353,9.481,9.607,
     * 8.089,8.262,8.462,8.667,8.871,9.069,9.250,9.404,
     * 8.064,8.221,8.405,8.593,8.782,8.971,9.144,9.291,
     * 7.645,7.902,8.171,8.458,8.691,8.879,9.053,9.218,
     * 7.166,7.506,7.856,8.223,8.453,8.626,8.783,8.934/
C
      DATA ((EDEN(I,4,10,L,1),I=25,32),L=1,NZEN)/
     * 9.280,9.390,9.519,9.663,9.793,9.918,10.044,10.166,
     * 9.243,9.347,9.468,9.605,9.719,9.828,9.947,10.067,
     * 9.203,9.303,9.420,9.552,9.649,9.735,9.833,9.947,
     * 9.139,9.238,9.353,9.483,9.578,9.650,9.724,9.814,
     * 8.988,9.097,9.221,9.359,9.472,9.563,9.646,9.736,
     * 8.884,9.009,9.146,9.297,9.416,9.515,9.606,9.704,
     * 8.606,8.774,8.955,9.149,9.298,9.422,9.533,9.646,
     * 8.194,8.369,8.564,8.759,8.948,9.131,9.298,9.440,
     * 8.148,8.306,8.487,8.666,8.843,9.018,9.177,9.314,
     * 7.749,8.003,8.261,8.529,8.752,8.926,9.085,9.243,
     * 7.306,7.644,7.983,8.332,8.528,8.680,8.822,8.963/
C
      DATA ((EDEN(I,5,10,L,1),I=25,32),L=1,NZEN)/
     * 9.476,9.578,9.691,9.811,9.917,10.017,10.120,10.222,
     * 9.436,9.531,9.636,9.748,9.838,9.926,10.022,10.124,
     * 9.394,9.485,9.585,9.692,9.764,9.827,9.907,10.003,
     * 9.339,9.429,9.527,9.632,9.696,9.743,9.796,9.871,
     * 9.201,9.302,9.408,9.521,9.596,9.658,9.717,9.791,
     * 9.093,9.208,9.328,9.454,9.538,9.609,9.678,9.758,
     * 8.850,9.004,9.161,9.323,9.429,9.520,9.607,9.701,
     * 8.371,8.555,8.736,8.907,9.069,9.225,9.368,9.494,
     * 8.307,8.475,8.640,8.796,8.948,9.096,9.235,9.358,
     * 7.928,8.187,8.438,8.690,8.860,9.003,9.142,9.283,
     * 7.521,7.865,8.199,8.531,8.650,8.760,8.881,9.011/
C
      DATA ((EDEN(I,1,11,L,1),I=25,32),L=1,NZEN)/
     * 8.994,9.113,9.260,9.433,9.599,9.756,9.913,10.061,
     * 8.970,9.083,9.225,9.392,9.546,9.691,9.844,9.993,
     * 8.938,9.046,9.182,9.344,9.482,9.605,9.739,9.883,
     * 8.889,8.995,9.128,9.287,9.417,9.523,9.626,9.743,
     * 8.779,8.891,9.029,9.193,9.324,9.440,9.552,9.665,
     * 8.686,8.811,8.963,9.139,9.272,9.395,9.515,9.635,
     * 8.426,8.592,8.785,9.002,9.155,9.300,9.439,9.576,
     * 8.067,8.244,8.433,8.629,8.831,9.034,9.221,9.383,
     * 8.052,8.212,8.380,8.557,8.744,8.937,9.116,9.273,
     * 7.636,7.892,8.168,8.469,8.659,8.841,9.020,9.196,
     * 7.140,7.494,7.866,8.263,8.438,8.599,8.765,8.928/
C
      DATA ((EDEN(I,2,11,L,1),I=25,32),L=1,NZEN)/
     * 9.052,9.173,9.322,9.494,9.669,9.832,9.988,10.136,
     * 9.014,9.128,9.268,9.433,9.590,9.738,9.890,10.035,
     * 8.986,9.092,9.226,9.382,9.516,9.637,9.768,9.907,
     * 8.930,9.035,9.166,9.320,9.445,9.547,9.646,9.756,
     * 8.800,8.915,9.055,9.218,9.350,9.465,9.575,9.684,
     * 8.678,8.811,8.969,9.149,9.289,9.416,9.536,9.655,
     * 8.396,8.574,8.776,8.999,9.167,9.318,9.458,9.594,
     * 8.047,8.230,8.433,8.642,8.853,9.059,9.247,9.406,
     * 8.037,8.210,8.401,8.598,8.796,8.992,9.171,9.322,
     * 7.589,7.871,8.169,8.488,8.709,8.897,9.076,9.248,
     * 7.071,7.446,7.835,8.243,8.468,8.646,8.810,8.969/
C
      DATA ((EDEN(I,3,11,L,1),I=25,32),L=1,NZEN)/
     * 9.161,9.275,9.413,9.571,9.732,9.883,10.027,10.163,
     * 9.121,9.229,9.359,9.510,9.654,9.791,9.930,10.066,
     * 9.080,9.184,9.310,9.457,9.580,9.691,9.812,9.942,
     * 9.030,9.132,9.256,9.399,9.513,9.604,9.693,9.795,
     * 8.880,8.995,9.130,9.284,9.412,9.521,9.619,9.718,
     * 8.769,8.901,9.053,9.222,9.359,9.476,9.582,9.688,
     * 8.458,8.641,8.842,9.060,9.228,9.376,9.506,9.630,
     * 8.070,8.260,8.476,8.695,8.906,9.108,9.288,9.438,
     * 8.036,8.219,8.427,8.635,8.837,9.029,9.200,9.342,
     * 7.589,7.883,8.185,8.500,8.751,8.946,9.120,9.277,
     * 7.077,7.459,7.846,8.243,8.507,8.696,8.854,9.000/
C
      DATA ((EDEN(I,4,11,L,1),I=25,32),L=1,NZEN)/
     * 9.321,9.422,9.541,9.675,9.805,9.930,10.055,10.178,
     * 9.282,9.378,9.492,9.619,9.732,9.843,9.961,10.083,
     * 9.242,9.335,9.446,9.570,9.665,9.752,9.851,9.963,
     * 9.185,9.280,9.391,9.515,9.605,9.676,9.748,9.836,
     * 9.048,9.157,9.279,9.413,9.516,9.602,9.680,9.763,
     * 8.946,9.072,9.209,9.357,9.468,9.562,9.647,9.735,
     * 8.675,8.845,9.026,9.217,9.357,9.476,9.578,9.681,
     * 8.173,8.379,8.597,8.809,9.006,9.189,9.350,9.483,
     * 8.104,8.304,8.516,8.720,8.908,9.084,9.237,9.363,
     * 7.707,8.005,8.303,8.600,8.826,9.001,9.153,9.295,
     * 7.210,7.604,7.994,8.380,8.606,8.770,8.907,9.035/
C
      DATA ((EDEN(I,5,11,L,1),I=25,32),L=1,NZEN)/
     * 9.546,9.632,9.727,9.830,9.930,10.026,10.127,10.227,
     * 9.505,9.585,9.674,9.771,9.855,9.940,10.034,10.134,
     * 9.461,9.539,9.626,9.719,9.785,9.847,9.924,10.017,
     * 9.403,9.483,9.571,9.665,9.724,9.770,9.821,9.891,
     * 9.286,9.381,9.481,9.584,9.650,9.705,9.760,9.825,
     * 9.186,9.298,9.413,9.530,9.604,9.668,9.729,9.797,
     * 8.958,9.109,9.261,9.412,9.508,9.591,9.666,9.746,
     * 8.376,8.602,8.809,8.997,9.164,9.314,9.444,9.553,
     * 8.260,8.480,8.682,8.865,9.028,9.177,9.304,9.412,
     * 7.894,8.204,8.500,8.783,8.952,9.090,9.217,9.340,
     * 7.435,7.851,8.248,8.627,8.767,8.885,8.996,9.106/
C
      DATA ((EDEN(I,1,12,L,1),I=25,32),L=1,NZEN)/
     * 9.008,9.123,9.268,9.437,9.597,9.750,9.905,10.055,
     * 8.984,9.094,9.233,9.397,9.545,9.687,9.838,9.989,
     * 8.953,9.058,9.192,9.351,9.484,9.604,9.735,9.878,
     * 8.905,9.007,9.138,9.294,9.419,9.524,9.626,9.742,
     * 8.795,8.903,9.038,9.198,9.326,9.441,9.552,9.666,
     * 8.703,8.824,8.972,9.144,9.273,9.395,9.515,9.637,
     * 8.442,8.604,8.791,9.004,9.155,9.300,9.439,9.577,
     * 8.078,8.254,8.440,8.634,8.833,9.036,9.223,9.386,
     * 8.066,8.224,8.390,8.566,8.752,8.943,9.123,9.280,
     * 7.651,7.906,8.180,8.480,8.665,8.845,9.026,9.202,
     * 7.153,7.508,7.880,8.278,8.444,8.604,8.770,8.936/
C
      DATA ((EDEN(I,2,12,L,1),I=25,32),L=1,NZEN)/
     * 9.075,9.196,9.344,9.517,9.693,9.860,10.016,10.160,
     * 9.032,9.145,9.285,9.449,9.610,9.762,9.916,10.059,
     * 9.000,9.106,9.239,9.396,9.532,9.660,9.797,9.938,
     * 8.947,9.050,9.178,9.330,9.452,9.552,9.651,9.759,
     * 8.814,8.927,9.066,9.226,9.354,9.470,9.580,9.690,
     * 8.703,8.833,8.987,9.163,9.299,9.425,9.543,9.658,
     * 8.393,8.572,8.775,8.998,9.163,9.318,9.463,9.601,
     * 8.052,8.239,8.444,8.657,8.868,9.076,9.263,9.421,
     * 8.034,8.213,8.410,8.611,8.812,9.009,9.188,9.337,
     * 7.581,7.875,8.185,8.512,8.737,8.930,9.108,9.279,
     * 7.044,7.435,7.839,8.259,8.491,8.674,8.839,8.996/
C
      DATA ((EDEN(I,3,12,L,1),I=25,32),L=1,NZEN)/
     * 9.180,9.295,9.434,9.593,9.759,9.916,10.063,10.200,
     * 9.136,9.244,9.375,9.526,9.676,9.821,9.965,10.100,
     * 9.090,9.193,9.320,9.466,9.595,9.714,9.842,9.975,
     * 9.043,9.144,9.267,9.410,9.523,9.615,9.705,9.809,
     * 8.908,9.023,9.158,9.310,9.435,9.540,9.634,9.728,
     * 8.773,8.909,9.063,9.234,9.371,9.490,9.596,9.699,
     * 8.490,8.674,8.874,9.090,9.256,9.400,9.526,9.644,
     * 8.051,8.252,8.479,8.708,8.926,9.131,9.310,9.458,
     * 8.011,8.211,8.436,8.660,8.870,9.065,9.235,9.372,
     * 7.565,7.879,8.199,8.525,8.782,8.983,9.151,9.307,
     * 7.015,7.422,7.833,8.247,8.535,8.738,8.897,9.038/
C
      DATA ((EDEN(I,4,12,L,1),I=25,32),L=1,NZEN)/
     * 9.329,9.430,9.550,9.684,9.821,9.952,10.080,10.203,
     * 9.287,9.384,9.497,9.625,9.746,9.863,9.986,10.109,
     * 9.243,9.337,9.448,9.572,9.674,9.767,9.872,9.987,
     * 9.184,9.281,9.393,9.517,9.612,9.687,9.761,9.851,
     * 9.057,9.170,9.297,9.433,9.537,9.623,9.698,9.778,
     * 8.944,9.077,9.221,9.373,9.488,9.585,9.668,9.751,
     * 8.695,8.872,9.059,9.252,9.389,9.507,9.606,9.701,
     * 8.149,8.377,8.614,8.838,9.042,9.226,9.382,9.509,
     * 8.058,8.285,8.521,8.743,8.942,9.120,9.269,9.387,
     * 7.650,7.978,8.301,8.616,8.862,9.044,9.193,9.327,
     * 7.133,7.562,7.983,8.392,8.649,8.824,8.958,9.077/
C
      DATA ((EDEN(I,5,12,L,1),I=25,32),L=1,NZEN)/
     * 9.558,9.645,9.741,9.846,9.951,10.052,10.155,10.256,
     * 9.511,9.593,9.685,9.783,9.874,9.965,10.065,10.163,
     * 9.463,9.543,9.632,9.726,9.798,9.867,9.949,10.044,
     * 9.401,9.486,9.577,9.672,9.735,9.784,9.837,9.909,
     * 9.276,9.381,9.488,9.597,9.668,9.726,9.779,9.840,
     * 9.180,9.303,9.427,9.549,9.630,9.696,9.753,9.819,
     * 8.955,9.118,9.280,9.437,9.537,9.622,9.696,9.771,
     * 8.341,8.597,8.830,9.037,9.215,9.366,9.490,9.591,
     * 8.181,8.440,8.679,8.889,9.068,9.219,9.341,9.438,
     * 7.825,8.176,8.508,8.816,9.003,9.146,9.269,9.380,
     * 7.327,7.791,8.231,8.641,8.817,8.943,9.046,9.140/
C
      DATA ((EDEN(I,1,1,L,2),I=25,32),L=1,NZEN)/
     * 9.190,9.302,9.443,9.609,9.772,9.939,10.100,10.249,
     * 9.164,9.269,9.404,9.564,9.715,9.871,10.027,10.179,
     * 9.130,9.230,9.358,9.512,9.644,9.774,9.916,10.061,
     * 9.085,9.180,9.304,9.454,9.574,9.682,9.787,9.899,
     * 9.000,9.095,9.218,9.369,9.489,9.601,9.706,9.809,
     * 8.926,9.029,9.161,9.322,9.441,9.557,9.668,9.775,
     * 8.688,8.827,8.995,9.192,9.329,9.461,9.588,9.709,
     * 8.272,8.422,8.583,8.758,8.943,9.133,9.308,9.461,
     * 8.275,8.393,8.522,8.670,8.835,9.010,9.175,9.318,
     * 7.883,8.091,8.329,8.601,8.755,8.916,9.081,9.240,
     * 7.402,7.704,8.035,8.402,8.536,8.678,8.826,8.975/
C
      DATA ((EDEN(I,2,1,L,2),I=25,32),L=1,NZEN)/
     * 9.264,9.385,9.534,9.707,9.886,10.059,10.217,10.359,
     * 9.215,9.326,9.465,9.628,9.792,9.958,10.114,10.259,
     * 9.176,9.278,9.409,9.563,9.701,9.840,9.986,10.129,
     * 9.124,9.220,9.343,9.490,9.607,9.710,9.810,9.915,
     * 9.014,9.115,9.243,9.395,9.516,9.628,9.731,9.829,
     * 8.921,9.035,9.175,9.340,9.467,9.584,9.692,9.795,
     * 8.628,8.788,8.974,9.184,9.334,9.474,9.604,9.725,
     * 8.233,8.395,8.575,8.769,8.966,9.162,9.338,9.487,
     * 8.237,8.381,8.543,8.717,8.899,9.080,9.243,9.378,
     * 7.804,8.059,8.336,8.639,8.831,9.002,9.169,9.321,
     * 7.283,7.632,8.003,8.399,8.594,8.753,8.901,9.039/
C
      DATA ((EDEN(I,3,1,L,2),I=25,32),L=1,NZEN)/
     * 9.367,9.484,9.625,9.786,9.956,10.117,10.266,10.398,
     * 9.315,9.423,9.554,9.706,9.864,10.019,10.167,10.298,
     * 9.263,9.364,9.488,9.632,9.764,9.894,10.034,10.167,
     * 9.212,9.310,9.432,9.573,9.684,9.781,9.879,9.985,
     * 9.097,9.204,9.332,9.479,9.596,9.698,9.788,9.872,
     * 8.981,9.104,9.249,9.411,9.539,9.651,9.748,9.838,
     * 8.719,8.886,9.075,9.280,9.430,9.561,9.674,9.777,
     * 8.223,8.408,8.616,8.828,9.033,9.227,9.396,9.533,
     * 8.197,8.372,8.568,8.767,8.958,9.139,9.293,9.416,
     * 7.768,8.052,8.349,8.660,8.882,9.061,9.213,9.352,
     * 7.232,7.609,7.997,8.398,8.645,8.822,8.964,9.088/
C
      DATA ((EDEN(I,4,1,L,2),I=25,32),L=1,NZEN)/
     * 9.541,9.636,9.749,9.877,10.016,10.153,10.283,10.402,
     * 9.494,9.583,9.689,9.809,9.933,10.058,10.187,10.306,
     * 9.447,9.533,9.635,9.751,9.852,9.952,10.062,10.175,
     * 9.389,9.477,9.581,9.698,9.788,9.861,9.932,10.019,
     * 9.281,9.384,9.500,9.627,9.724,9.800,9.865,9.929,
     * 9.185,9.304,9.436,9.578,9.684,9.767,9.835,9.902,
     * 8.959,9.119,9.290,9.468,9.594,9.693,9.774,9.850,
     * 8.355,8.573,8.795,9.007,9.197,9.366,9.507,9.619,
     * 8.243,8.448,8.658,8.858,9.038,9.199,9.332,9.435,
     * 7.845,8.150,8.456,8.758,8.971,9.131,9.259,9.378,
     * 7.326,7.733,8.138,8.536,8.762,8.915,9.032,9.132/
C
      DATA ((EDEN(I,5,1,L,2),I=25,32),L=1,NZEN)/
     * 9.792,9.867,9.952,10.044,10.153,10.257,10.358,10.455,
     * 9.745,9.815,9.894,9.980,10.073,10.166,10.266,10.363,
     * 9.696,9.764,9.840,9.922,9.992,10.064,10.143,10.236,
     * 9.637,9.710,9.790,9.875,9.932,9.974,10.017,10.081,
     * 9.527,9.621,9.719,9.817,9.880,9.923,9.956,9.998,
     * 9.441,9.552,9.666,9.779,9.850,9.897,9.932,9.976,
     * 9.232,9.382,9.532,9.679,9.768,9.829,9.875,9.927,
     * 8.564,8.810,9.035,9.229,9.391,9.525,9.630,9.713,
     * 8.373,8.613,8.831,9.021,9.181,9.315,9.420,9.502,
     * 8.008,8.344,8.664,8.963,9.125,9.244,9.342,9.436,
     * 7.489,7.940,8.372,8.775,8.937,9.043,9.125,9.199/
C
      DATA ((EDEN(I,1,2,L,2),I=25,32),L=1,NZEN)/
     * 9.179,9.292,9.436,9.604,9.768,9.935,10.097,10.247,
     * 9.151,9.259,9.396,9.559,9.712,9.867,10.027,10.176,
     * 9.118,9.220,9.350,9.506,9.641,9.772,9.912,10.058,
     * 9.073,9.170,9.296,9.447,9.571,9.679,9.785,9.898,
     * 8.986,9.083,9.210,9.363,9.485,9.599,9.705,9.807,
     * 8.910,9.017,9.152,9.316,9.437,9.554,9.666,9.774,
     * 8.669,8.813,8.986,9.187,9.325,9.460,9.586,9.707,
     * 8.262,8.414,8.577,8.754,8.940,9.131,9.307,9.459,
     * 8.266,8.385,8.517,8.665,8.831,9.007,9.172,9.315,
     * 7.870,8.081,8.321,8.595,8.753,8.912,9.078,9.238,
     * 7.391,7.694,8.026,8.395,8.534,8.675,8.823,8.971/
C
      DATA ((EDEN(I,2,2,L,2),I=25,32),L=1,NZEN)/
     * 9.240,9.361,9.510,9.682,9.858,10.029,10.189,10.329,
     * 9.193,9.305,9.445,9.609,9.769,9.930,10.087,10.228,
     * 9.156,9.260,9.390,9.545,9.678,9.809,9.949,10.093,
     * 9.101,9.200,9.327,9.477,9.597,9.701,9.802,9.908,
     * 8.992,9.096,9.227,9.383,9.507,9.619,9.722,9.821,
     * 8.899,9.016,9.159,9.327,9.457,9.574,9.684,9.787,
     * 8.616,8.778,8.966,9.178,9.328,9.468,9.598,9.719,
     * 8.225,8.388,8.570,8.762,8.959,9.154,9.331,9.479,
     * 8.240,8.383,8.543,8.715,8.895,9.076,9.240,9.378,
     * 7.805,8.054,8.328,8.629,8.816,8.987,9.151,9.306,
     * 7.299,7.642,8.008,8.401,8.584,8.740,8.887,9.028/
C
      DATA ((EDEN(I,3,2,L,2),I=25,32),L=1,NZEN)/
     * 9.338,9.455,9.597,9.759,9.923,10.080,10.229,10.360,
     * 9.289,9.398,9.532,9.685,9.835,9.982,10.127,10.260,
     * 9.239,9.343,9.470,9.618,9.742,9.866,9.998,10.128,
     * 9.185,9.286,9.410,9.553,9.664,9.758,9.850,9.951,
     * 9.068,9.177,9.308,9.459,9.577,9.678,9.769,9.856,
     * 8.954,9.079,9.226,9.392,9.520,9.630,9.728,9.823,
     * 8.692,8.861,9.050,9.258,9.409,9.540,9.654,9.761,
     * 8.226,8.409,8.613,8.819,9.020,9.212,9.382,9.521,
     * 8.205,8.375,8.564,8.755,8.942,9.120,9.276,9.401,
     * 7.770,8.049,8.343,8.652,8.870,9.048,9.202,9.343,
     * 7.266,7.636,8.019,8.416,8.640,8.805,8.944,9.069/
C
      DATA ((EDEN(I,4,2,L,2),I=25,32),L=1,NZEN)/
     * 9.493,9.594,9.713,9.847,9.981,10.116,10.247,10.366,
     * 9.447,9.542,9.655,9.782,9.901,10.022,10.146,10.266,
     * 9.401,9.493,9.602,9.725,9.823,9.916,10.022,10.137,
     * 9.340,9.434,9.544,9.668,9.757,9.828,9.898,9.983,
     * 9.224,9.331,9.452,9.585,9.682,9.759,9.827,9.897,
     * 9.122,9.245,9.382,9.529,9.636,9.721,9.794,9.867,
     * 8.866,9.033,9.213,9.402,9.533,9.637,9.726,9.811,
     * 8.326,8.537,8.753,8.959,9.148,9.320,9.466,9.584,
     * 8.232,8.430,8.637,8.834,9.014,9.179,9.318,9.427,
     * 7.839,8.137,8.437,8.738,8.947,9.106,9.241,9.365,
     * 7.340,7.737,8.132,8.526,8.737,8.885,9.005,9.112/
C
      DATA ((EDEN(I,5,2,L,2),I=25,32),L=1,NZEN)/
     * 9.709,9.796,9.892,9.996,10.103,10.209,10.314,10.415,
     * 9.660,9.741,9.832,9.929,10.020,10.114,10.215,10.317,
     * 9.609,9.689,9.777,9.872,9.942,10.009,10.093,10.185,
     * 9.547,9.630,9.721,9.817,9.877,9.921,9.966,10.031,
     * 9.429,9.530,9.636,9.744,9.811,9.858,9.898,9.949,
     * 9.335,9.454,9.576,9.697,9.772,9.825,9.869,9.923,
     * 9.111,9.270,9.430,9.588,9.682,9.750,9.807,9.870,
     * 8.490,8.730,8.950,9.143,9.309,9.450,9.566,9.661,
     * 8.335,8.569,8.784,8.975,9.138,9.280,9.394,9.485,
     * 7.959,8.288,8.604,8.903,9.075,9.202,9.314,9.419,
     * 7.465,7.906,8.330,8.731,8.889,9.003,9.097,9.188/
C
      DATA ((EDEN(I,1,3,L,2),I=25,32),L=1,NZEN)/
     * 9.162,9.279,9.426,9.597,9.768,9.939,10.103,10.251,
     * 9.134,9.245,9.385,9.551,9.710,9.871,10.030,10.179,
     * 9.100,9.204,9.338,9.497,9.637,9.771,9.916,10.061,
     * 9.054,9.154,9.283,9.439,9.566,9.676,9.784,9.897,
     * 8.965,9.066,9.196,9.354,9.480,9.595,9.702,9.805,
     * 8.887,8.999,9.140,9.308,9.433,9.552,9.664,9.772,
     * 8.643,8.793,8.972,9.179,9.323,9.459,9.585,9.706,
     * 8.249,8.402,8.569,8.749,8.938,9.129,9.306,9.458,
     * 8.250,8.371,8.506,8.658,8.825,9.001,9.166,9.308,
     * 7.850,8.064,8.306,8.583,8.747,8.908,9.073,9.230,
     * 7.376,7.679,8.011,8.379,8.528,8.669,8.817,8.964/
C
      DATA ((EDEN(I,2,3,L,2),I=25,32),L=1,NZEN)/
     * 9.200,9.322,9.472,9.646,9.821,9.988,10.149,10.290,
     * 9.166,9.280,9.422,9.588,9.748,9.904,10.063,10.209,
     * 9.123,9.230,9.365,9.524,9.660,9.790,9.932,10.074,
     * 9.066,9.170,9.301,9.457,9.582,9.689,9.790,9.899,
     * 8.960,9.068,9.203,9.363,9.492,9.604,9.708,9.809,
     * 8.864,8.986,9.134,9.308,9.439,9.558,9.668,9.775,
     * 8.603,8.765,8.955,9.171,9.323,9.462,9.588,9.709,
     * 8.217,8.379,8.560,8.750,8.946,9.140,9.317,9.467,
     * 8.230,8.368,8.521,8.688,8.863,9.042,9.206,9.345,
     * 7.810,8.047,8.310,8.601,8.784,8.950,9.114,9.271,
     * 7.328,7.655,8.006,8.387,8.560,8.708,8.856,8.997/
C
      DATA ((EDEN(I,3,3,L,2),I=25,32),L=1,NZEN)/
     * 9.289,9.409,9.554,9.718,9.875,10.031,10.178,10.312,
     * 9.243,9.356,9.494,9.651,9.793,9.934,10.076,10.209,
     * 9.200,9.308,9.439,9.592,9.712,9.828,9.952,10.082,
     * 9.128,9.235,9.364,9.514,9.631,9.726,9.818,9.918,
     * 9.002,9.116,9.252,9.409,9.537,9.642,9.737,9.831,
     * 8.906,9.033,9.183,9.353,9.486,9.598,9.699,9.797,
     * 8.624,8.797,8.993,9.208,9.366,9.500,9.620,9.733,
     * 8.226,8.400,8.595,8.795,8.991,9.183,9.354,9.496,
     * 8.215,8.370,8.545,8.725,8.905,9.083,9.241,9.372,
     * 7.794,8.050,8.323,8.618,8.825,8.997,9.155,9.301,
     * 7.317,7.660,8.019,8.399,8.600,8.755,8.896,9.028/
C
      DATA ((EDEN(I,4,3,L,2),I=25,32),L=1,NZEN)/
     * 9.419,9.532,9.663,9.809,9.944,10.080,10.210,10.334,
     * 9.373,9.480,9.604,9.744,9.861,9.979,10.108,10.230,
     * 9.324,9.427,9.548,9.684,9.782,9.873,9.978,10.094,
     * 9.257,9.362,9.482,9.617,9.709,9.781,9.853,9.940,
     * 9.115,9.229,9.359,9.502,9.609,9.693,9.769,9.849,
     * 8.999,9.130,9.276,9.434,9.554,9.647,9.729,9.815,
     * 8.743,8.917,9.104,9.304,9.446,9.558,9.657,9.755,
     * 8.282,8.472,8.679,8.880,9.069,9.248,9.404,9.534,
     * 8.239,8.413,8.602,8.788,8.964,9.132,9.277,9.396,
     * 7.832,8.104,8.382,8.668,8.884,9.051,9.194,9.329,
     * 7.380,7.742,8.108,8.481,8.675,8.818,8.945,9.065/
C
      DATA ((EDEN(I,5,3,L,2),I=25,32),L=1,NZEN)/
     * 9.587,9.695,9.812,9.938,10.052,10.163,10.278,10.382,
     * 9.536,9.638,9.750,9.870,9.966,10.063,10.170,10.278,
     * 9.481,9.582,9.691,9.806,9.882,9.950,10.036,10.141,
     * 9.412,9.513,9.623,9.738,9.806,9.853,9.903,9.976,
     * 9.270,9.385,9.505,9.630,9.709,9.765,9.816,9.882,
     * 9.158,9.290,9.427,9.567,9.656,9.720,9.777,9.847,
     * 8.898,9.074,9.253,9.435,9.549,9.633,9.707,9.790,
     * 8.405,8.618,8.826,9.015,9.185,9.339,9.473,9.586,
     * 8.320,8.523,8.722,8.903,9.067,9.216,9.344,9.450,
     * 7.924,8.223,8.514,8.798,8.989,9.133,9.259,9.382,
     * 7.496,7.892,8.276,8.651,8.797,8.913,9.022,9.132/
C
      DATA ((EDEN(I,1,4,L,2),I=25,32),L=1,NZEN)/
     * 9.151,9.272,9.422,9.597,9.780,9.952,10.118,10.268,
     * 9.122,9.236,9.379,9.547,9.715,9.882,10.046,10.196,
     * 9.086,9.192,9.328,9.489,9.637,9.777,9.928,10.074,
     * 9.037,9.140,9.271,9.429,9.561,9.674,9.784,9.901,
     * 8.949,9.052,9.185,9.344,9.475,9.592,9.698,9.801,
     * 8.872,8.986,9.130,9.301,9.430,9.549,9.661,9.769,
     * 8.630,8.783,8.965,9.175,9.321,9.459,9.585,9.705,
     * 8.237,8.392,8.563,8.746,8.936,9.128,9.305,9.457,
     * 8.233,8.354,8.491,8.645,8.812,8.989,9.154,9.295,
     * 7.837,8.049,8.289,8.564,8.736,8.897,9.061,9.220,
     * 7.366,7.666,7.994,8.359,8.516,8.660,8.809,8.954/
C
      DATA ((EDEN(I,2,4,L,2),I=25,32),L=1,NZEN)/
     * 9.159,9.282,9.434,9.608,9.781,9.946,10.108,10.255,
     * 9.126,9.243,9.388,9.557,9.715,9.872,10.031,10.179,
     * 9.086,9.197,9.335,9.498,9.641,9.774,9.918,10.063,
     * 9.030,9.137,9.272,9.432,9.564,9.674,9.781,9.895,
     * 8.939,9.049,9.187,9.351,9.480,9.592,9.695,9.798,
     * 8.851,8.971,9.118,9.293,9.428,9.547,9.655,9.762,
     * 8.608,8.767,8.953,9.166,9.318,9.455,9.579,9.698,
     * 8.233,8.390,8.565,8.749,8.939,9.130,9.306,9.456,
     * 8.228,8.351,8.492,8.647,8.815,8.991,9.154,9.295,
     * 7.833,8.045,8.285,8.558,8.736,8.898,9.063,9.218,
     * 7.372,7.672,7.998,8.359,8.519,8.663,8.808,8.952/
C
      DATA ((EDEN(I,3,4,L,2),I=25,32),L=1,NZEN)/
     * 9.225,9.352,9.503,9.674,9.835,9.989,10.139,10.277,
     * 9.185,9.304,9.448,9.612,9.753,9.892,10.033,10.172,
     * 9.132,9.247,9.387,9.547,9.677,9.794,9.922,10.059,
     * 9.062,9.175,9.311,9.469,9.598,9.698,9.794,9.901,
     * 8.937,9.054,9.194,9.356,9.495,9.606,9.704,9.803,
     * 8.838,8.969,9.122,9.298,9.442,9.559,9.664,9.767,
     * 8.584,8.754,8.946,9.162,9.329,9.466,9.587,9.704,
     * 8.227,8.382,8.569,8.761,8.953,9.144,9.319,9.467,
     * 8.219,8.344,8.498,8.663,8.834,9.010,9.172,9.309,
     * 7.819,8.036,8.273,8.539,8.751,8.922,9.080,9.234,
     * 7.372,7.671,7.989,8.338,8.528,8.678,8.823,8.962/
C
      DATA ((EDEN(I,4,4,L,2),I=25,32),L=1,NZEN)/
     * 9.322,9.456,9.608,9.774,9.927,10.073,10.213,10.340,
     * 9.270,9.397,9.542,9.701,9.834,9.964,10.102,10.231,
     * 9.211,9.332,9.471,9.625,9.738,9.837,9.952,10.079,
     * 9.109,9.229,9.366,9.519,9.639,9.725,9.807,9.904,
     * 8.948,9.074,9.216,9.374,9.513,9.616,9.704,9.796,
     * 8.824,8.964,9.120,9.292,9.444,9.559,9.656,9.757,
     * 8.561,8.740,8.934,9.144,9.323,9.462,9.579,9.693,
     * 8.226,8.375,8.573,8.775,8.970,9.159,9.331,9.475,
     * 8.215,8.338,8.510,8.690,8.867,9.042,9.202,9.336,
     * 7.817,8.037,8.268,8.519,8.771,8.954,9.114,9.262,
     * 7.412,7.708,8.015,8.342,8.555,8.709,8.848,8.983/
C
      DATA ((EDEN(I,5,4,L,2),I=25,32),L=1,NZEN)/
     * 9.435,9.578,9.730,9.889,10.036,10.173,10.297,10.413,
     * 9.376,9.511,9.655,9.807,9.935,10.056,10.183,10.303,
     * 9.312,9.441,9.580,9.727,9.831,9.921,10.029,10.150,
     * 9.217,9.346,9.483,9.628,9.726,9.788,9.850,9.938,
     * 9.004,9.140,9.283,9.435,9.564,9.652,9.725,9.808,
     * 8.866,9.017,9.174,9.339,9.486,9.589,9.672,9.764,
     * 8.599,8.787,8.980,9.181,9.359,9.489,9.594,9.702,
     * 8.287,8.439,8.641,8.843,9.031,9.208,9.369,9.503,
     * 8.270,8.402,8.583,8.766,8.937,9.102,9.252,9.378,
     * 7.858,8.088,8.317,8.555,8.823,9.010,9.164,9.305,
     * 7.487,7.793,8.097,8.408,8.618,8.762,8.891,9.018/
C
      DATA ((EDEN(I,1,5,L,2),I=25,32),L=1,NZEN)/
     * 9.157,9.280,9.432,9.609,9.800,9.980,10.148,10.297,
     * 9.126,9.240,9.384,9.553,9.730,9.905,10.072,10.226,
     * 9.086,9.192,9.328,9.489,9.642,9.793,9.951,10.101,
     * 9.037,9.138,9.268,9.424,9.558,9.674,9.789,9.911,
     * 8.950,9.052,9.182,9.340,9.472,9.590,9.696,9.797,
     * 8.875,8.987,9.128,9.297,9.428,9.549,9.659,9.766,
     * 8.641,8.790,8.969,9.176,9.323,9.460,9.585,9.702,
     * 8.235,8.389,8.561,8.745,8.935,9.127,9.303,9.454,
     * 8.225,8.343,8.480,8.633,8.800,8.977,9.142,9.282,
     * 7.838,8.045,8.280,8.551,8.724,8.885,9.050,9.206,
     * 7.367,7.661,7.983,8.342,8.505,8.651,8.797,8.939/
C
      DATA ((EDEN(I,2,5,L,2),I=25,32),L=1,NZEN)/
     * 9.137,9.260,9.411,9.587,9.760,9.928,10.092,10.242,
     * 9.110,9.229,9.376,9.547,9.721,9.891,10.059,10.211,
     * 9.073,9.185,9.326,9.491,9.644,9.790,9.945,10.096,
     * 9.021,9.130,9.266,9.427,9.561,9.674,9.787,9.910,
     * 8.936,9.043,9.179,9.341,9.472,9.584,9.686,9.788,
     * 8.856,8.971,9.114,9.284,9.419,9.537,9.644,9.750,
     * 8.632,8.781,8.959,9.165,9.316,9.450,9.572,9.688,
     * 8.256,8.407,8.576,8.754,8.939,9.126,9.299,9.448,
     * 8.231,8.337,8.464,8.608,8.767,8.940,9.103,9.245,
     * 7.867,8.053,8.267,8.518,8.690,8.850,9.009,9.168,
     * 7.393,7.668,7.972,8.315,8.478,8.622,8.771,8.919/
C
      DATA ((EDEN(I,3,5,L,2),I=25,32),L=1,NZEN)/
     * 9.176,9.309,9.466,9.644,9.816,9.979,10.137,10.279,
     * 9.130,9.256,9.406,9.576,9.734,9.886,10.042,10.187,
     * 9.072,9.193,9.338,9.504,9.657,9.796,9.947,10.094,
     * 9.003,9.120,9.261,9.424,9.570,9.681,9.790,9.911,
     * 8.889,9.006,9.147,9.311,9.464,9.581,9.682,9.782,
     * 8.795,8.921,9.070,9.244,9.404,9.528,9.636,9.742,
     * 8.573,8.734,8.919,9.130,9.306,9.444,9.565,9.683,
     * 8.224,8.358,8.537,8.726,8.917,9.109,9.288,9.440,
     * 8.211,8.300,8.437,8.590,8.755,8.931,9.098,9.241,
     * 7.853,8.028,8.228,8.463,8.678,8.849,9.009,9.169,
     * 7.395,7.647,7.923,8.237,8.446,8.606,8.758,8.907/
C
      DATA ((EDEN(I,4,5,L,2),I=25,32),L=1,NZEN)/
     * 9.245,9.399,9.570,9.755,9.948,10.116,10.267,10.402,
     * 9.184,9.327,9.487,9.663,9.837,9.998,10.150,10.290,
     * 9.088,9.221,9.372,9.539,9.692,9.823,9.965,10.111,
     * 8.963,9.091,9.237,9.400,9.560,9.675,9.778,9.897,
     * 8.789,8.916,9.061,9.224,9.407,9.542,9.648,9.752,
     * 8.676,8.813,8.967,9.140,9.334,9.482,9.597,9.707,
     * 8.428,8.591,8.772,8.974,9.205,9.379,9.515,9.642,
     * 8.149,8.246,8.435,8.652,8.863,9.068,9.257,9.415,
     * 8.170,8.230,8.381,8.564,8.748,8.933,9.108,9.256,
     * 7.794,7.949,8.120,8.322,8.622,8.842,9.017,9.178,
     * 7.420,7.636,7.869,8.134,8.396,8.582,8.738,8.888/
C
      DATA ((EDEN(I,5,5,L,2),I=25,32),L=1,NZEN)/
     * 9.332,9.502,9.682,9.869,10.075,10.247,10.391,10.517,
     * 9.254,9.411,9.579,9.755,9.950,10.120,10.271,10.402,
     * 9.166,9.315,9.473,9.639,9.802,9.951,10.101,10.242,
     * 8.998,9.137,9.286,9.445,9.604,9.713,9.808,9.925,
     * 8.756,8.891,9.035,9.191,9.386,9.533,9.642,9.745,
     * 8.621,8.762,8.911,9.073,9.285,9.453,9.578,9.693,
     * 8.391,8.547,8.711,8.889,9.143,9.348,9.495,9.625,
     * 8.174,8.246,8.422,8.650,8.868,9.073,9.259,9.418,
     * 8.194,8.234,8.375,8.570,8.763,8.949,9.124,9.273,
     * 7.805,7.946,8.094,8.264,8.579,8.842,9.028,9.193,
     * 7.427,7.612,7.804,8.019,8.316,8.564,8.736,8.890/
C
      DATA ((EDEN(I,1,6,L,2),I=25,32),L=1,NZEN)/
     * 9.176,9.299,9.452,9.629,9.823,10.007,10.178,10.325,
     * 9.141,9.255,9.399,9.567,9.752,9.933,10.102,10.254,
     * 9.097,9.202,9.337,9.496,9.653,9.812,9.977,10.130,
     * 9.046,9.145,9.273,9.426,9.559,9.677,9.797,9.925,
     * 8.961,9.060,9.187,9.341,9.473,9.589,9.694,9.794,
     * 8.891,8.999,9.135,9.300,9.428,9.548,9.658,9.763,
     * 8.664,8.807,8.980,9.181,9.326,9.461,9.585,9.701,
     * 8.242,8.396,8.566,8.749,8.938,9.128,9.303,9.453,
     * 8.224,8.339,8.474,8.625,8.792,8.968,9.132,9.272,
     * 7.848,8.049,8.279,8.544,8.716,8.877,9.039,9.196,
     * 7.378,7.667,7.983,8.337,8.498,8.643,8.788,8.932/
C
      DATA ((EDEN(I,2,6,L,2),I=25,32),L=1,NZEN)/
     * 9.134,9.257,9.407,9.581,9.756,9.928,10.094,10.247,
     * 9.110,9.229,9.376,9.547,9.727,9.902,10.076,10.231,
     * 9.072,9.185,9.326,9.491,9.652,9.811,9.974,10.130,
     * 9.024,9.131,9.265,9.425,9.561,9.680,9.802,9.936,
     * 8.947,9.050,9.181,9.339,9.470,9.580,9.680,9.781,
     * 8.874,8.982,9.119,9.283,9.417,9.533,9.639,9.744,
     * 8.683,8.823,8.991,9.189,9.329,9.456,9.573,9.688,
     * 8.283,8.431,8.595,8.767,8.947,9.129,9.300,9.447,
     * 8.239,8.332,8.447,8.581,8.734,8.903,9.066,9.209,
     * 7.906,8.075,8.274,8.513,8.669,8.824,8.980,9.141,
     * 7.429,7.684,7.970,8.297,8.450,8.591,8.739,8.891/
C
      DATA ((EDEN(I,3,6,L,2),I=25,32),L=1,NZEN)/
     * 9.155,9.292,9.453,9.634,9.820,9.993,10.158,10.302,
     * 9.102,9.232,9.385,9.559,9.745,9.922,10.091,10.247,
     * 9.049,9.171,9.317,9.485,9.656,9.816,9.980,10.135,
     * 8.980,9.097,9.237,9.400,9.556,9.680,9.804,9.941,
     * 8.873,8.985,9.122,9.283,9.445,9.566,9.668,9.770,
     * 8.803,8.922,9.066,9.236,9.397,9.520,9.626,9.732,
     * 8.583,8.732,8.907,9.109,9.290,9.431,9.552,9.670,
     * 8.234,8.352,8.524,8.710,8.899,9.090,9.271,9.426,
     * 8.229,8.294,8.412,8.552,8.708,8.880,9.049,9.195,
     * 7.878,8.017,8.184,8.390,8.614,8.788,8.952,9.113,
     * 7.395,7.607,7.846,8.130,8.364,8.543,8.705,8.863/
C
      DATA ((EDEN(I,4,6,L,2),I=25,32),L=1,NZEN)/
     * 9.223,9.390,9.575,9.775,9.996,10.181,10.338,10.476,
     * 9.131,9.281,9.450,9.633,9.840,10.030,10.198,10.344,
     * 9.013,9.150,9.306,9.478,9.661,9.833,10.008,10.168,
     * 8.900,9.030,9.178,9.343,9.519,9.654,9.780,9.922,
     * 8.728,8.851,8.993,9.154,9.353,9.506,9.620,9.730,
     * 8.616,8.742,8.888,9.054,9.264,9.432,9.560,9.678,
     * 8.399,8.547,8.714,8.905,9.147,9.338,9.481,9.613,
     * 8.127,8.190,8.364,8.585,8.802,9.013,9.210,9.378,
     * 8.159,8.177,8.301,8.476,8.658,8.847,9.030,9.187,
     * 7.781,7.885,8.009,8.171,8.485,8.743,8.936,9.108,
     * 7.412,7.562,7.734,7.948,8.240,8.473,8.650,8.811/
C
      DATA ((EDEN(I,5,6,L,2),I=25,32),L=1,NZEN)/
     * 9.318,9.511,9.714,9.925,10.159,10.343,10.495,10.628,
     * 9.228,9.405,9.591,9.786,10.020,10.218,10.377,10.513,
     * 9.069,9.223,9.387,9.560,9.755,9.943,10.123,10.281,
     * 8.897,9.037,9.187,9.347,9.531,9.668,9.800,9.952,
     * 8.666,8.792,8.929,9.079,9.292,9.465,9.596,9.711,
     * 8.550,8.677,8.814,8.967,9.189,9.381,9.527,9.653,
     * 8.348,8.481,8.624,8.785,9.047,9.276,9.441,9.581,
     * 8.149,8.176,8.314,8.541,8.769,8.984,9.183,9.356,
     * 8.200,8.188,8.282,8.458,8.645,8.832,9.014,9.175,
     * 7.839,7.918,8.008,8.131,8.428,8.712,8.915,9.091,
     * 7.438,7.545,7.664,7.819,8.068,8.390,8.608,8.782/
C
      DATA ((EDEN(I,1,7,L,2),I=25,32),L=1,NZEN)/
     * 9.188,9.311,9.464,9.641,9.837,10.021,10.189,10.337,
     * 9.152,9.266,9.409,9.578,9.761,9.943,10.113,10.264,
     * 9.107,9.211,9.344,9.502,9.661,9.822,9.987,10.141,
     * 9.055,9.152,9.278,9.430,9.562,9.680,9.802,9.931,
     * 8.972,9.068,9.193,9.345,9.474,9.590,9.694,9.794,
     * 8.904,9.009,9.143,9.304,9.431,9.549,9.659,9.763,
     * 8.680,8.820,8.989,9.186,9.329,9.463,9.586,9.702,
     * 8.249,8.402,8.572,8.753,8.941,9.130,9.304,9.453,
     * 8.227,8.341,8.474,8.624,8.790,8.965,9.129,9.269,
     * 7.857,8.055,8.283,8.545,8.714,8.877,9.036,9.194,
     * 7.387,7.673,7.988,8.339,8.497,8.641,8.786,8.928/
C
      DATA ((EDEN(I,2,7,L,2),I=25,32),L=1,NZEN)/
     * 9.146,9.267,9.416,9.588,9.764,9.935,10.103,10.255,
     * 9.122,9.240,9.385,9.555,9.736,9.913,10.087,10.242,
     * 9.086,9.197,9.335,9.499,9.661,9.822,9.988,10.141,
     * 9.037,9.141,9.273,9.431,9.566,9.686,9.812,9.948,
     * 8.962,9.062,9.190,9.346,9.474,9.583,9.683,9.784,
     * 8.893,8.997,9.129,9.290,9.422,9.536,9.641,9.745,
     * 8.709,8.843,9.006,9.199,9.336,9.460,9.576,9.690,
     * 8.301,8.447,8.607,8.778,8.955,9.135,9.304,9.450,
     * 8.247,8.335,8.446,8.576,8.727,8.894,9.057,9.200,
     * 7.923,8.085,8.277,8.509,8.662,8.815,8.972,9.131,
     * 7.447,7.694,7.972,8.293,8.442,8.581,8.729,8.879/
C
      DATA ((EDEN(I,3,7,L,2),I=25,32),L=1,NZEN)/
     * 9.169,9.305,9.466,9.646,9.835,10.009,10.173,10.318,
     * 9.117,9.245,9.398,9.571,9.759,9.937,10.108,10.262,
     * 9.064,9.185,9.329,9.496,9.668,9.831,9.998,10.152,
     * 8.997,9.110,9.248,9.409,9.564,9.689,9.818,9.959,
     * 8.896,9.003,9.136,9.294,9.453,9.571,9.673,9.773,
     * 8.830,8.943,9.082,9.247,9.404,9.526,9.630,9.735,
     * 8.614,8.757,8.926,9.123,9.301,9.438,9.557,9.672,
     * 8.257,8.372,8.541,8.722,8.907,9.096,9.275,9.428,
     * 8.241,8.299,8.410,8.543,8.694,8.864,9.032,9.179,
     * 7.901,8.030,8.187,8.384,8.602,8.773,8.935,9.098,
     * 7.419,7.619,7.849,8.124,8.352,8.527,8.691,8.849/
C
      DATA ((EDEN(I,4,7,L,2),I=25,32),L=1,NZEN)/
     * 9.249,9.418,9.605,9.807,10.026,10.210,10.365,10.500,
     * 9.152,9.302,9.470,9.654,9.868,10.060,10.226,10.368,
     * 9.031,9.167,9.320,9.490,9.677,9.856,10.030,10.190,
     * 8.921,9.048,9.193,9.355,9.528,9.665,9.795,9.943,
     * 8.759,8.877,9.014,9.171,9.365,9.515,9.628,9.736,
     * 8.651,8.771,8.911,9.074,9.279,9.443,9.568,9.683,
     * 8.437,8.580,8.742,8.928,9.162,9.349,9.488,9.619,
     * 8.150,8.208,8.377,8.591,8.803,9.012,9.208,9.377,
     * 8.179,8.187,8.300,8.466,8.642,8.829,9.011,9.170,
     * 7.811,7.901,8.014,8.165,8.471,8.723,8.914,9.088,
     * 7.444,7.579,7.737,7.939,8.223,8.452,8.626,8.790/
C
      DATA ((EDEN(I,5,7,L,2),I=25,32),L=1,NZEN)/
     * 9.347,9.545,9.752,9.968,10.197,10.377,10.526,10.659,
     * 9.257,9.436,9.625,9.822,10.059,10.254,10.410,10.544,
     * 9.090,9.243,9.405,9.576,9.779,9.975,10.158,10.312,
     * 8.920,9.056,9.203,9.360,9.540,9.680,9.819,9.977,
     * 8.699,8.821,8.953,9.100,9.304,9.475,9.604,9.717,
     * 8.589,8.710,8.842,8.989,9.204,9.393,9.534,9.658,
     * 8.388,8.517,8.658,8.817,9.067,9.288,9.448,9.586,
     * 8.186,8.210,8.349,8.566,8.783,8.991,9.186,9.357,
     * 8.221,8.196,8.278,8.444,8.622,8.806,8.988,9.151,
     * 7.871,7.933,8.007,8.117,8.408,8.687,8.888,9.064,
     * 7.476,7.563,7.663,7.804,8.048,8.364,8.579,8.754/
C
      DATA ((EDEN(I,1,8,L,2),I=25,32),L=1,NZEN)/
     * 9.185,9.308,9.460,9.637,9.830,10.012,10.178,10.326,
     * 9.151,9.264,9.407,9.575,9.756,9.936,10.105,10.254,
     * 9.106,9.210,9.344,9.503,9.658,9.816,9.977,10.130,
     * 9.054,9.152,9.279,9.432,9.564,9.680,9.800,9.928,
     * 8.972,9.069,9.195,9.349,9.477,9.592,9.696,9.797,
     * 8.903,9.009,9.144,9.307,9.433,9.551,9.661,9.766,
     * 8.678,8.820,8.990,9.189,9.331,9.465,9.587,9.704,
     * 8.250,8.404,8.573,8.755,8.943,9.132,9.306,9.456,
     * 8.231,8.346,8.480,8.630,8.795,8.970,9.134,9.274,
     * 7.857,8.058,8.287,8.551,8.719,8.881,9.042,9.198,
     * 7.386,7.675,7.992,8.345,8.501,8.645,8.790,8.932/
C
      DATA ((EDEN(I,2,8,L,2),I=25,32),L=1,NZEN)/
     * 9.162,9.282,9.430,9.602,9.776,9.946,10.113,10.262,
     * 9.137,9.253,9.397,9.564,9.740,9.913,10.082,10.235,
     * 9.100,9.209,9.345,9.506,9.662,9.815,9.974,10.128,
     * 9.050,9.153,9.284,9.441,9.572,9.687,9.806,9.936,
     * 8.975,9.074,9.202,9.357,9.482,9.591,9.692,9.792,
     * 8.903,9.008,9.140,9.301,9.432,9.546,9.651,9.755,
     * 8.694,8.830,8.995,9.190,9.334,9.461,9.580,9.693,
     * 8.295,8.442,8.604,8.775,8.954,9.136,9.306,9.453,
     * 8.252,8.345,8.460,8.593,8.746,8.915,9.077,9.219,
     * 7.913,8.084,8.285,8.524,8.679,8.831,8.989,9.148,
     * 7.435,7.690,7.975,8.301,8.456,8.596,8.744,8.894/
C
      DATA ((EDEN(I,3,8,L,2),I=25,32),L=1,NZEN)/
     * 9.208,9.340,9.495,9.670,9.847,10.016,10.173,10.314,
     * 9.159,9.282,9.429,9.596,9.770,9.938,10.100,10.247,
     * 9.108,9.224,9.363,9.524,9.681,9.829,9.984,10.132,
     * 9.044,9.153,9.287,9.443,9.585,9.700,9.815,9.944,
     * 8.944,9.050,9.180,9.335,9.480,9.593,9.691,9.790,
     * 8.862,8.973,9.109,9.271,9.422,9.541,9.645,9.749,
     * 8.655,8.800,8.969,9.166,9.328,9.461,9.577,9.691,
     * 8.281,8.405,8.571,8.749,8.931,9.116,9.292,9.442,
     * 8.263,8.335,8.447,8.581,8.731,8.899,9.063,9.207,
     * 7.914,8.059,8.231,8.442,8.642,8.806,8.967,9.127,
     * 7.427,7.648,7.895,8.186,8.398,8.563,8.721,8.876/
C
      DATA ((EDEN(I,4,8,L,2),I=25,32),L=1,NZEN)/
     * 9.304,9.460,9.632,9.820,10.016,10.186,10.332,10.461,
     * 9.224,9.361,9.517,9.689,9.875,10.044,10.198,10.334,
     * 9.138,9.264,9.407,9.567,9.727,9.873,10.024,10.169,
     * 9.021,9.139,9.275,9.428,9.583,9.700,9.810,9.937,
     * 8.870,8.983,9.115,9.266,9.436,9.566,9.667,9.766,
     * 8.770,8.890,9.028,9.188,9.367,9.506,9.615,9.721,
     * 8.529,8.676,8.843,9.032,9.242,9.405,9.533,9.653,
     * 8.223,8.308,8.479,8.677,8.872,9.068,9.253,9.412,
     * 8.223,8.255,8.377,8.535,8.703,8.882,9.057,9.209,
     * 7.872,7.990,8.127,8.300,8.577,8.787,8.960,9.126,
     * 7.507,7.680,7.873,8.106,8.346,8.522,8.676,8.831/
C
      DATA ((EDEN(I,5,8,L,2),I=25,32),L=1,NZEN)/
     * 9.405,9.580,9.765,9.958,10.165,10.332,10.469,10.590,
     * 9.326,9.484,9.652,9.828,10.037,10.210,10.353,10.477,
     * 9.214,9.353,9.502,9.660,9.838,10.002,10.157,10.292,
     * 9.043,9.170,9.306,9.452,9.611,9.729,9.836,9.964,
     * 8.851,8.969,9.097,9.238,9.418,9.560,9.664,9.762,
     * 8.733,8.855,8.987,9.134,9.326,9.483,9.599,9.707,
     * 8.504,8.646,8.797,8.965,9.194,9.379,9.515,9.638,
     * 8.248,8.302,8.460,8.665,8.865,9.060,9.244,9.404,
     * 8.272,8.280,8.388,8.546,8.711,8.885,9.057,9.210,
     * 7.922,8.016,8.122,8.258,8.546,8.780,8.957,9.125,
     * 7.534,7.665,7.807,7.983,8.254,8.486,8.656,8.815/
C
      DATA ((EDEN(I,1,9,L,2),I=25,32),L=1,NZEN)/
     * 9.173,9.295,9.445,9.620,9.808,9.986,10.153,10.299,
     * 9.141,9.254,9.397,9.564,9.739,9.912,10.078,10.228,
     * 9.101,9.206,9.340,9.500,9.650,9.799,9.955,10.104,
     * 9.052,9.151,9.280,9.435,9.565,9.680,9.794,9.916,
     * 8.967,9.066,9.195,9.351,9.479,9.595,9.700,9.802,
     * 8.894,9.003,9.142,9.308,9.435,9.554,9.664,9.770,
     * 8.663,8.808,8.983,9.186,9.330,9.466,9.590,9.707,
     * 8.249,8.403,8.572,8.754,8.943,9.134,9.309,9.459,
     * 8.235,8.353,8.488,8.640,8.805,8.980,9.144,9.285,
     * 7.852,8.058,8.292,8.561,8.729,8.889,9.053,9.208,
     * 7.381,7.675,7.997,8.356,8.511,8.654,8.800,8.943/
C
      DATA ((EDEN(I,2,9,L,2),I=25,32),L=1,NZEN)/
     * 9.181,9.300,9.448,9.619,9.793,9.962,10.127,10.274,
     * 9.155,9.268,9.410,9.575,9.742,9.907,10.073,10.222,
     * 9.117,9.223,9.357,9.515,9.659,9.800,9.949,10.097,
     * 9.066,9.168,9.298,9.454,9.581,9.689,9.797,9.914,
     * 8.981,9.083,9.212,9.369,9.494,9.604,9.705,9.804,
     * 8.902,9.011,9.149,9.313,9.443,9.558,9.665,9.769,
     * 8.676,8.821,8.993,9.194,9.340,9.472,9.591,9.706,
     * 8.272,8.422,8.590,8.768,8.952,9.138,9.311,9.459,
     * 8.249,8.355,8.482,8.626,8.786,8.958,9.120,9.261,
     * 7.878,8.067,8.285,8.538,8.709,8.867,9.025,9.185,
     * 7.396,7.671,7.974,8.316,8.484,8.630,8.777,8.923/
C
      DATA ((EDEN(I,3,9,L,2),I=25,32),L=1,NZEN)/
     * 9.259,9.382,9.528,9.695,9.860,10.019,10.170,10.308,
     * 9.217,9.331,9.469,9.628,9.776,9.920,10.071,10.210,
     * 9.171,9.278,9.410,9.562,9.693,9.816,9.947,10.085,
     * 9.104,9.208,9.337,9.487,9.614,9.716,9.816,9.927,
     * 9.000,9.105,9.234,9.386,9.518,9.625,9.721,9.816,
     * 8.914,9.028,9.165,9.326,9.463,9.577,9.678,9.778,
     * 8.678,8.828,9.002,9.200,9.356,9.486,9.602,9.715,
     * 8.286,8.426,8.596,8.776,8.960,9.145,9.317,9.465,
     * 8.262,8.358,8.487,8.632,8.790,8.960,9.122,9.262,
     * 7.896,8.077,8.283,8.521,8.712,8.872,9.030,9.186,
     * 7.447,7.702,7.982,8.297,8.477,8.623,8.767,8.912/
C
      DATA ((EDEN(I,4,9,L,2),I=25,32),L=1,NZEN)/
     * 9.390,9.520,9.667,9.830,9.988,10.136,10.274,10.395,
     * 9.339,9.458,9.594,9.746,9.890,10.028,10.165,10.290,
     * 9.268,9.378,9.505,9.648,9.766,9.875,9.995,10.125,
     * 9.191,9.296,9.418,9.557,9.671,9.755,9.838,9.937,
     * 9.047,9.153,9.276,9.417,9.548,9.646,9.731,9.818,
     * 8.950,9.066,9.198,9.350,9.487,9.593,9.683,9.777,
     * 8.699,8.852,9.023,9.214,9.374,9.500,9.606,9.713,
     * 8.312,8.437,8.609,8.789,8.970,9.151,9.320,9.465,
     * 8.301,8.387,8.518,8.664,8.818,8.983,9.142,9.280,
     * 7.922,8.093,8.281,8.496,8.719,8.889,9.046,9.201,
     * 7.554,7.797,8.054,8.342,8.503,8.634,8.770,8.910/
C
      DATA ((EDEN(I,5,9,L,2),I=25,32),L=1,NZEN)/
     * 9.534,9.670,9.817,9.971,10.124,10.255,10.375,10.482,
     * 9.475,9.599,9.733,9.875,10.014,10.141,10.261,10.372,
     * 9.414,9.528,9.651,9.782,9.892,9.995,10.109,10.223,
     * 9.325,9.432,9.549,9.674,9.765,9.827,9.893,9.983,
     * 9.151,9.258,9.374,9.500,9.617,9.696,9.762,9.837,
     * 9.046,9.164,9.290,9.427,9.549,9.637,9.710,9.793,
     * 8.783,8.938,9.101,9.277,9.426,9.537,9.629,9.727,
     * 8.408,8.531,8.696,8.866,9.030,9.194,9.351,9.488,
     * 8.383,8.466,8.595,8.734,8.877,9.030,9.180,9.313,
     * 8.014,8.186,8.362,8.558,8.773,8.933,9.079,9.229,
     * 7.646,7.881,8.120,8.381,8.544,8.665,8.791,8.927/
C
      DATA ((EDEN(I,1,10,L,2),I=25,32),L=1,NZEN)/
     * 9.167,9.287,9.435,9.609,9.788,9.961,10.126,10.274,
     * 9.138,9.250,9.392,9.558,9.725,9.890,10.055,10.203,
     * 9.101,9.206,9.340,9.500,9.644,9.786,9.932,10.083,
     * 9.054,9.154,9.283,9.439,9.568,9.680,9.790,9.906,
     * 8.967,9.068,9.198,9.355,9.484,9.599,9.704,9.807,
     * 8.893,9.004,9.144,9.311,9.438,9.557,9.668,9.775,
     * 8.656,8.805,8.982,9.187,9.331,9.466,9.592,9.710,
     * 8.251,8.406,8.574,8.755,8.944,9.135,9.311,9.462,
     * 8.244,8.364,8.498,8.650,8.816,8.992,9.156,9.297,
     * 7.852,8.062,8.301,8.574,8.740,8.901,9.064,9.220,
     * 7.379,7.678,8.006,8.370,8.521,8.663,8.810,8.954/
C
      DATA ((EDEN(I,2,10,L,2),I=25,32),L=1,NZEN)/
     * 9.206,9.326,9.474,9.645,9.821,9.991,10.154,10.298,
     * 9.174,9.285,9.425,9.588,9.748,9.907,10.065,10.214,
     * 9.134,9.238,9.371,9.527,9.665,9.797,9.939,10.082,
     * 9.082,9.183,9.311,9.463,9.589,9.696,9.800,9.911,
     * 8.986,9.089,9.219,9.375,9.501,9.614,9.717,9.816,
     * 8.899,9.014,9.156,9.324,9.453,9.571,9.679,9.783,
     * 8.652,8.806,8.988,9.196,9.342,9.478,9.601,9.718,
     * 8.242,8.395,8.570,8.757,8.949,9.141,9.316,9.466,
     * 8.242,8.364,8.507,8.663,8.832,9.007,9.171,9.311,
     * 7.839,8.056,8.299,8.574,8.754,8.920,9.080,9.235,
     * 7.365,7.666,7.993,8.354,8.527,8.674,8.820,8.961/
C
      DATA ((EDEN(I,3,10,L,2),I=25,32),L=1,NZEN)/
     * 9.307,9.423,9.563,9.723,9.883,10.040,10.188,10.322,
     * 9.266,9.374,9.505,9.657,9.799,9.940,10.085,10.221,
     * 9.223,9.325,9.450,9.595,9.717,9.832,9.958,10.090,
     * 9.164,9.262,9.384,9.527,9.643,9.739,9.830,9.931,
     * 9.052,9.155,9.281,9.428,9.551,9.655,9.749,9.841,
     * 8.951,9.068,9.207,9.368,9.498,9.610,9.710,9.806,
     * 8.697,8.854,9.034,9.236,9.386,9.517,9.633,9.743,
     * 8.272,8.427,8.608,8.797,8.986,9.173,9.344,9.488,
     * 8.253,8.380,8.531,8.693,8.862,9.034,9.193,9.327,
     * 7.850,8.071,8.313,8.580,8.780,8.946,9.105,9.255,
     * 7.385,7.684,8.002,8.348,8.545,8.698,8.838,8.973/
C
      DATA ((EDEN(I,4,10,L,2),I=25,32),L=1,NZEN)/
     * 9.479,9.584,9.707,9.845,9.981,10.113,10.242,10.362,
     * 9.436,9.533,9.648,9.778,9.896,10.015,10.141,10.262,
     * 9.394,9.486,9.595,9.719,9.815,9.907,10.012,10.128,
     * 9.333,9.422,9.528,9.650,9.739,9.810,9.881,9.965,
     * 9.204,9.300,9.413,9.541,9.642,9.723,9.795,9.872,
     * 9.117,9.224,9.347,9.485,9.591,9.679,9.756,9.837,
     * 8.860,9.008,9.172,9.351,9.481,9.587,9.681,9.774,
     * 8.377,8.537,8.714,8.893,9.068,9.238,9.393,9.525,
     * 8.326,8.458,8.608,8.763,8.920,9.078,9.225,9.350,
     * 7.939,8.163,8.398,8.649,8.837,8.992,9.136,9.277,
     * 7.499,7.805,8.119,8.451,8.616,8.747,8.874,9.003/
C
      DATA ((EDEN(I,5,10,L,2),I=25,32),L=1,NZEN)/
     * 9.694,9.787,9.890,10.000,10.107,10.214,10.319,10.419,
     * 9.649,9.734,9.828,9.930,10.021,10.113,10.217,10.316,
     * 9.602,9.682,9.770,9.866,9.935,10.003,10.085,10.182,
     * 9.548,9.626,9.713,9.806,9.863,9.904,9.951,10.019,
     * 9.428,9.514,9.607,9.707,9.771,9.818,9.861,9.919,
     * 9.335,9.433,9.537,9.648,9.719,9.772,9.821,9.883,
     * 9.115,9.247,9.385,9.529,9.616,9.684,9.748,9.821,
     * 8.555,8.728,8.895,9.051,9.196,9.336,9.465,9.578,
     * 8.467,8.613,8.753,8.887,9.019,9.153,9.280,9.393,
     * 8.097,8.331,8.563,8.800,8.940,9.063,9.189,9.321,
     * 7.678,7.998,8.313,8.632,8.728,8.823,8.929,9.047/
C
      DATA ((EDEN(I,1,11,L,2),I=25,32),L=1,NZEN)/
     * 9.174,9.290,9.436,9.606,9.776,9.945,10.111,10.257,
     * 9.146,9.256,9.395,9.560,9.717,9.875,10.037,10.189,
     * 9.112,9.215,9.348,9.506,9.644,9.777,9.920,10.068,
     * 9.067,9.166,9.293,9.447,9.572,9.682,9.789,9.902,
     * 8.980,9.079,9.207,9.362,9.486,9.601,9.706,9.809,
     * 8.905,9.014,9.151,9.317,9.440,9.558,9.669,9.776,
     * 8.665,8.811,8.986,9.189,9.331,9.465,9.591,9.711,
     * 8.259,8.413,8.578,8.756,8.944,9.135,9.311,9.462,
     * 8.258,8.378,8.511,8.660,8.826,9.002,9.166,9.308,
     * 7.862,8.073,8.313,8.588,8.748,8.908,9.073,9.232,
     * 7.387,7.688,8.019,8.386,8.529,8.671,8.818,8.964/
C
      DATA ((EDEN(I,2,11,L,2),I=25,32),L=1,NZEN)/
     * 9.236,9.356,9.504,9.675,9.855,10.029,10.189,10.331,
     * 9.192,9.303,9.442,9.604,9.767,9.926,10.086,10.230,
     * 9.161,9.262,9.392,9.545,9.679,9.811,9.952,10.093,
     * 9.106,9.205,9.330,9.480,9.601,9.706,9.808,9.913,
     * 9.002,9.105,9.234,9.389,9.513,9.625,9.728,9.826,
     * 8.899,9.016,9.160,9.329,9.458,9.578,9.687,9.792,
     * 8.635,8.794,8.980,9.192,9.342,9.479,9.606,9.725,
     * 8.232,8.390,8.570,8.761,8.956,9.151,9.328,9.477,
     * 8.239,8.375,8.530,8.697,8.874,9.054,9.218,9.356,
     * 7.810,8.049,8.314,8.607,8.796,8.966,9.129,9.283,
     * 7.313,7.639,7.990,8.370,8.561,8.718,8.864,9.005/
C
      DATA ((EDEN(I,3,11,L,2),I=25,32),L=1,NZEN)/
     * 9.344,9.458,9.595,9.753,9.919,10.078,10.226,10.360,
     * 9.299,9.404,9.534,9.683,9.830,9.981,10.127,10.261,
     * 9.253,9.353,9.476,9.620,9.744,9.868,9.996,10.127,
     * 9.206,9.302,9.421,9.560,9.670,9.765,9.857,9.956,
     * 9.079,9.183,9.309,9.456,9.575,9.681,9.774,9.862,
     * 8.986,9.104,9.243,9.403,9.527,9.639,9.737,9.830,
     * 8.694,8.860,9.047,9.253,9.405,9.539,9.656,9.764,
     * 8.250,8.420,8.613,8.814,9.012,9.203,9.373,9.514,
     * 8.228,8.379,8.554,8.735,8.915,9.092,9.249,9.377,
     * 7.797,8.054,8.328,8.621,8.838,9.014,9.170,9.313,
     * 7.301,7.642,7.998,8.373,8.600,8.767,8.908,9.037/
C
      DATA ((EDEN(I,4,11,L,2),I=25,32),L=1,NZEN)/
     * 9.538,9.630,9.740,9.864,9.996,10.128,10.256,10.373,
     * 9.497,9.583,9.686,9.804,9.918,10.035,10.157,10.277,
     * 9.456,9.538,9.637,9.751,9.844,9.935,10.036,10.148,
     * 9.405,9.488,9.586,9.699,9.783,9.852,9.920,10.000,
     * 9.292,9.385,9.493,9.614,9.706,9.781,9.848,9.915,
     * 9.207,9.314,9.434,9.566,9.664,9.745,9.815,9.885,
     * 8.962,9.109,9.269,9.441,9.562,9.661,9.745,9.827,
     * 8.389,8.577,8.774,8.967,9.147,9.315,9.461,9.581,
     * 8.307,8.475,8.652,8.826,8.993,9.150,9.288,9.401,
     * 7.922,8.186,8.456,8.732,8.922,9.074,9.207,9.334,
     * 7.427,7.786,8.148,8.514,8.703,8.842,8.962,9.075/
C
      DATA ((EDEN(I,5,11,L,2),I=25,32),L=1,NZEN)/
     * 9.799,9.868,9.946,10.033,10.130,10.228,10.328,10.423,
     * 9.759,9.822,9.894,9.974,10.056,10.140,10.234,10.330,
     * 9.716,9.776,9.846,9.922,9.983,10.042,10.117,10.205,
     * 9.665,9.728,9.799,9.875,9.926,9.962,10.002,10.061,
     * 9.568,9.644,9.726,9.811,9.866,9.904,9.938,9.983,
     * 9.482,9.574,9.669,9.767,9.828,9.872,9.909,9.955,
     * 9.278,9.404,9.533,9.663,9.741,9.797,9.846,9.902,
     * 8.633,8.841,9.030,9.198,9.343,9.471,9.579,9.670,
     * 8.473,8.659,8.828,8.981,9.120,9.248,9.358,9.453,
     * 8.115,8.394,8.662,8.922,9.055,9.165,9.274,9.382,
     * 7.635,8.020,8.391,8.750,8.864,8.958,9.051,9.147/
C
      DATA ((EDEN(I,1,12,L,2),I=25,32),L=1,NZEN)/
     * 9.186,9.300,9.442,9.609,9.776,9.942,10.103,10.251,
     * 9.160,9.267,9.403,9.564,9.717,9.871,10.030,10.181,
     * 9.127,9.227,9.357,9.512,9.645,9.776,9.916,10.061,
     * 9.081,9.177,9.302,9.453,9.575,9.682,9.788,9.901,
     * 8.995,9.091,9.216,9.368,9.489,9.602,9.708,9.809,
     * 8.921,9.025,9.159,9.321,9.441,9.558,9.669,9.776,
     * 8.683,8.825,8.995,9.193,9.330,9.464,9.590,9.710,
     * 8.269,8.420,8.582,8.759,8.945,9.134,9.310,9.462,
     * 8.270,8.388,8.519,8.668,8.832,9.007,9.172,9.315,
     * 7.877,8.087,8.325,8.598,8.754,8.916,9.078,9.238,
     * 7.398,7.700,8.030,8.397,8.535,8.675,8.825,8.971/
C
      DATA ((EDEN(I,2,12,L,2),I=25,32),L=1,NZEN)/
     * 9.260,9.381,9.530,9.702,9.882,10.056,10.217,10.359,
     * 9.213,9.324,9.462,9.624,9.792,9.954,10.114,10.257,
     * 9.176,9.278,9.408,9.561,9.700,9.839,9.986,10.129,
     * 9.124,9.220,9.343,9.490,9.608,9.712,9.812,9.917,
     * 9.015,9.116,9.245,9.398,9.518,9.630,9.733,9.831,
     * 8.923,9.037,9.178,9.342,9.469,9.587,9.696,9.798,
     * 8.631,8.791,8.977,9.188,9.337,9.478,9.607,9.728,
     * 8.233,8.394,8.575,8.768,8.965,9.160,9.337,9.485,
     * 8.237,8.378,8.538,8.711,8.891,9.072,9.235,9.371,
     * 7.802,8.054,8.329,8.630,8.823,8.997,9.160,9.313,
     * 7.285,7.628,7.993,8.386,8.583,8.745,8.893,9.031/
C
      DATA ((EDEN(I,3,12,L,2),I=25,32),L=1,NZEN)/
     * 9.367,9.483,9.622,9.782,9.953,10.114,10.264,10.396,
     * 9.316,9.422,9.552,9.702,9.860,10.015,10.162,10.296,
     * 9.266,9.366,9.489,9.632,9.763,9.894,10.030,10.163,
     * 9.218,9.314,9.433,9.571,9.681,9.777,9.871,9.975,
     * 9.106,9.210,9.337,9.481,9.598,9.701,9.790,9.875,
     * 8.991,9.112,9.254,9.415,9.541,9.653,9.750,9.841,
     * 8.729,8.894,9.079,9.282,9.432,9.564,9.676,9.778,
     * 8.230,8.410,8.615,8.825,9.029,9.222,9.392,9.530,
     * 8.205,8.373,8.564,8.759,8.948,9.127,9.283,9.407,
     * 7.777,8.053,8.343,8.648,8.870,9.047,9.203,9.342,
     * 7.241,7.607,7.986,8.378,8.629,8.809,8.951,9.075/
C
      DATA ((EDEN(I,4,12,L,2),I=25,32),L=1,NZEN)/
     * 9.555,9.646,9.754,9.877,10.016,10.153,10.283,10.400,
     * 9.512,9.596,9.698,9.815,9.936,10.061,10.186,10.306,
     * 9.468,9.550,9.648,9.760,9.858,9.956,10.065,10.178,
     * 9.413,9.497,9.597,9.709,9.797,9.869,9.940,10.021,
     * 9.311,9.408,9.519,9.641,9.734,9.810,9.873,9.936,
     * 9.216,9.329,9.455,9.591,9.693,9.776,9.843,9.909,
     * 8.993,9.145,9.309,9.482,9.604,9.701,9.781,9.856,
     * 8.384,8.592,8.807,9.012,9.198,9.365,9.505,9.618,
     * 8.272,8.466,8.665,8.856,9.030,9.189,9.321,9.426,
     * 7.877,8.170,8.464,8.757,8.962,9.119,9.248,9.367,
     * 7.359,7.753,8.144,8.532,8.751,8.900,9.016,9.117/
C
      DATA ((EDEN(I,5,12,L,2),I=25,32),L=1,NZEN)/
     * 9.827,9.894,9.971,10.055,10.159,10.260,10.361,10.455,
     * 9.784,9.846,9.918,9.997,10.084,10.171,10.268,10.363,
     * 9.738,9.799,9.867,9.942,10.008,10.071,10.149,10.238,
     * 9.684,9.749,9.821,9.897,9.951,9.989,10.030,10.088,
     * 9.581,9.665,9.753,9.842,9.901,9.940,9.972,10.011,
     * 9.499,9.599,9.703,9.805,9.871,9.914,9.948,9.988,
     * 9.296,9.433,9.571,9.706,9.789,9.844,9.890,9.938,
     * 8.628,8.864,9.077,9.262,9.415,9.542,9.643,9.723,
     * 8.421,8.643,8.844,9.020,9.170,9.298,9.401,9.483,
     * 8.072,8.388,8.690,8.973,9.120,9.231,9.330,9.425,
     * 7.555,7.985,8.397,8.785,8.928,9.026,9.106,9.184/
C
      DATA ((EDEN(I,1,1,L,3),I=25,32),L=1,NZEN)/
     * 9.307,9.415,9.552,9.715,9.879,10.045,10.206,10.352,
     * 9.287,9.387,9.516,9.671,9.823,9.979,10.136,10.282,
     * 9.262,9.353,9.474,9.621,9.754,9.885,10.028,10.167,
     * 9.234,9.317,9.431,9.572,9.688,9.796,9.905,10.014,
     * 9.189,9.266,9.374,9.512,9.623,9.730,9.830,9.924,
     * 9.140,9.220,9.334,9.478,9.586,9.695,9.797,9.894,
     * 8.935,9.045,9.188,9.364,9.486,9.608,9.722,9.828,
     * 8.505,8.620,8.752,8.904,9.071,9.244,9.405,9.542,
     * 8.521,8.585,8.669,8.781,8.918,9.071,9.217,9.342,
     * 8.160,8.308,8.496,8.730,8.849,8.982,9.127,9.265,
     * 7.714,7.944,8.217,8.540,8.637,8.751,8.875,8.998/
C
      DATA ((EDEN(I,2,1,L,3),I=25,32),L=1,NZEN)/
     * 9.382,9.500,9.647,9.817,9.995,10.163,10.320,10.461,
     * 9.341,9.448,9.583,9.742,9.903,10.067,10.222,10.362,
     * 9.308,9.403,9.527,9.675,9.815,9.955,10.100,10.237,
     * 9.274,9.358,9.471,9.609,9.722,9.826,9.928,10.030,
     * 9.205,9.288,9.402,9.542,9.653,9.759,9.855,9.946,
     * 9.136,9.228,9.349,9.498,9.613,9.723,9.822,9.912,
     * 8.875,9.006,9.167,9.357,9.491,9.619,9.736,9.840,
     * 8.450,8.576,8.726,8.895,9.074,9.255,9.418,9.554,
     * 8.478,8.570,8.686,8.824,8.978,9.138,9.283,9.400,
     * 8.074,8.269,8.499,8.765,8.919,9.069,9.211,9.343,
     * 7.586,7.867,8.184,8.540,8.693,8.825,8.947,9.064/
C
      DATA ((EDEN(I,3,1,L,3),I=25,32),L=1,NZEN)/
     * 9.486,9.600,9.738,9.896,10.063,10.221,10.369,10.502,
     * 9.440,9.543,9.670,9.818,9.974,10.126,10.271,10.403,
     * 9.395,9.489,9.607,9.746,9.875,10.011,10.144,10.274,
     * 9.358,9.446,9.559,9.692,9.800,9.899,9.996,10.099,
     * 9.282,9.373,9.487,9.622,9.731,9.828,9.913,9.987,
     * 9.190,9.293,9.420,9.568,9.684,9.788,9.878,9.957,
     * 8.960,9.101,9.267,9.453,9.587,9.706,9.809,9.897,
     * 8.435,8.590,8.770,8.960,9.149,9.328,9.484,9.610,
     * 8.425,8.554,8.707,8.872,9.037,9.197,9.333,9.439,
     * 8.023,8.255,8.510,8.788,8.972,9.123,9.257,9.376,
     * 7.514,7.833,8.175,8.540,8.745,8.894,9.012,9.111/
C
      DATA ((EDEN(I,4,1,L,3),I=25,32),L=1,NZEN)/
     * 9.660,9.751,9.860,9.984,10.122,10.257,10.386,10.505,
     * 9.618,9.702,9.804,9.920,10.045,10.168,10.293,10.409,
     * 9.579,9.658,9.754,9.864,9.965,10.064,10.174,10.285,
     * 9.533,9.612,9.708,9.817,9.903,9.976,10.050,10.128,
     * 9.453,9.543,9.648,9.766,9.854,9.927,9.986,10.047,
     * 9.376,9.479,9.597,9.727,9.823,9.900,9.962,10.021,
     * 9.177,9.315,9.468,9.631,9.744,9.835,9.906,9.971,
     * 8.579,8.776,8.977,9.170,9.344,9.498,9.625,9.724,
     * 8.459,8.625,8.798,8.966,9.121,9.262,9.377,9.464,
     * 8.081,8.343,8.614,8.889,9.065,9.198,9.307,9.407,
     * 7.582,7.941,8.307,8.676,8.861,8.989,9.082,9.161/
C
      DATA ((EDEN(I,5,1,L,3),I=25,32),L=1,NZEN)/
     * 9.911,9.983,10.065,10.155,10.259,10.361,10.463,10.560,
     * 9.867,9.933,10.008,10.091,10.183,10.276,10.371,10.467,
     * 9.823,9.886,9.957,10.035,10.104,10.176,10.255,10.344,
     * 9.773,9.839,9.912,9.990,10.046,10.087,10.130,10.192,
     * 9.685,9.768,9.857,9.948,10.007,10.045,10.075,10.113,
     * 9.612,9.711,9.814,9.918,9.983,10.025,10.056,10.091,
     * 9.423,9.557,9.694,9.830,9.910,9.965,10.004,10.046,
     * 8.771,9.001,9.209,9.389,9.538,9.659,9.751,9.822,
     * 8.562,8.770,8.959,9.124,9.262,9.377,9.465,9.533,
     * 8.216,8.518,8.809,9.085,9.219,9.313,9.392,9.468,
     * 7.704,8.118,8.517,8.897,9.031,9.115,9.176,9.234/
C
      DATA ((EDEN(I,1,2,L,3),I=25,32),L=1,NZEN)/
     * 9.295,9.406,9.546,9.711,9.875,10.042,10.204,10.349,
     * 9.275,9.377,9.509,9.666,9.819,9.975,10.133,10.282,
     * 9.251,9.344,9.467,9.616,9.750,9.885,10.024,10.167,
     * 9.222,9.307,9.423,9.566,9.684,9.794,9.900,10.010,
     * 9.176,9.255,9.366,9.507,9.619,9.727,9.828,9.922,
     * 9.125,9.208,9.325,9.472,9.582,9.692,9.796,9.892,
     * 8.917,9.031,9.178,9.358,9.483,9.606,9.720,9.826,
     * 8.493,8.611,8.745,8.899,9.067,9.242,9.404,9.542,
     * 8.512,8.577,8.663,8.776,8.914,9.068,9.214,9.338,
     * 8.147,8.297,8.488,8.724,8.847,8.982,9.125,9.262,
     * 7.701,7.933,8.207,8.532,8.634,8.748,8.873,8.994/
C
      DATA ((EDEN(I,2,2,L,3),I=25,32),L=1,NZEN)/
     * 9.356,9.475,9.622,9.793,9.965,10.135,10.291,10.433,
     * 9.317,9.425,9.561,9.720,9.877,10.037,10.192,10.332,
     * 9.288,9.383,9.507,9.656,9.790,9.923,10.062,10.199,
     * 9.249,9.337,9.454,9.596,9.711,9.816,9.919,10.022,
     * 9.181,9.268,9.385,9.528,9.642,9.748,9.845,9.936,
     * 9.115,9.209,9.333,9.485,9.602,9.712,9.812,9.904,
     * 8.862,8.995,9.158,9.351,9.485,9.613,9.729,9.834,
     * 8.442,8.571,8.723,8.892,9.071,9.251,9.414,9.550,
     * 8.476,8.568,8.683,8.820,8.972,9.133,9.279,9.399,
     * 8.072,8.264,8.490,8.753,8.904,9.051,9.194,9.328,
     * 7.601,7.876,8.187,8.539,8.682,8.811,8.935,9.050/
C
      DATA ((EDEN(I,3,2,L,3),I=25,32),L=1,NZEN)/
     * 9.456,9.571,9.710,9.870,10.029,10.186,10.331,10.462,
     * 9.413,9.519,9.648,9.797,9.944,10.089,10.233,10.366,
     * 9.370,9.467,9.588,9.729,9.854,9.980,10.107,10.235,
     * 9.331,9.421,9.536,9.671,9.778,9.874,9.967,10.064,
     * 9.251,9.345,9.462,9.601,9.710,9.807,9.892,9.971,
     * 9.163,9.267,9.397,9.547,9.664,9.767,9.858,9.940,
     * 8.932,9.074,9.241,9.430,9.565,9.684,9.787,9.879,
     * 8.435,8.589,8.765,8.950,9.135,9.313,9.470,9.597,
     * 8.429,8.553,8.699,8.857,9.019,9.177,9.315,9.424,
     * 8.019,8.247,8.499,8.776,8.957,9.110,9.244,9.366,
     * 7.542,7.854,8.191,8.554,8.736,8.873,8.991,9.096/
C
      DATA ((EDEN(I,4,2,L,3),I=25,32),L=1,NZEN)/
     * 9.613,9.711,9.827,9.957,10.090,10.222,10.351,10.469,
     * 9.571,9.661,9.769,9.892,10.011,10.132,10.254,10.370,
     * 9.529,9.614,9.717,9.834,9.931,10.028,10.135,10.245,
     * 9.483,9.567,9.668,9.784,9.870,9.942,10.012,10.095,
     * 9.396,9.490,9.599,9.722,9.811,9.884,9.947,10.011,
     * 9.314,9.421,9.543,9.678,9.775,9.854,9.919,9.983,
     * 9.089,9.233,9.393,9.566,9.684,9.778,9.856,9.929,
     * 8.536,8.726,8.921,9.109,9.283,9.440,9.573,9.679,
     * 8.437,8.598,8.767,8.934,9.091,9.236,9.358,9.452,
     * 8.064,8.320,8.585,8.860,9.034,9.169,9.283,9.389,
     * 7.580,7.929,8.288,8.653,8.830,8.952,9.052,9.141/
C
      DATA ((EDEN(I,5,2,L,3),I=25,32),L=1,NZEN)/
     * 9.828,9.911,10.004,10.104,10.211,10.314,10.419,10.517,
     * 9.782,9.859,9.945,10.039,10.129,10.222,10.322,10.419,
     * 9.736,9.809,9.891,9.980,10.052,10.120,10.201,10.295,
     * 9.683,9.758,9.841,9.930,9.989,10.033,10.080,10.141,
     * 9.589,9.680,9.776,9.875,9.936,9.980,10.016,10.060,
     * 9.510,9.616,9.727,9.839,9.907,9.953,9.991,10.036,
     * 9.309,9.451,9.596,9.742,9.826,9.887,9.934,9.987,
     * 8.683,8.908,9.114,9.293,9.446,9.574,9.677,9.759,
     * 8.510,8.714,8.901,9.067,9.210,9.334,9.433,9.511,
     * 8.150,8.447,8.736,9.014,9.157,9.265,9.358,9.449,
     * 7.656,8.062,8.458,8.839,8.971,9.064,9.141,9.215/
C
      DATA ((EDEN(I,1,3,L,3),I=25,32),L=1,NZEN)/
     * 9.277,9.391,9.534,9.702,9.875,10.045,10.206,10.354,
     * 9.257,9.363,9.498,9.659,9.819,9.979,10.136,10.284,
     * 9.232,9.328,9.454,9.606,9.746,9.885,10.028,10.170,
     * 9.203,9.291,9.410,9.556,9.679,9.791,9.900,10.010,
     * 9.154,9.238,9.352,9.497,9.614,9.724,9.826,9.920,
     * 9.103,9.192,9.313,9.465,9.579,9.690,9.794,9.891,
     * 8.891,9.011,9.165,9.351,9.480,9.604,9.720,9.826,
     * 8.479,8.599,8.737,8.895,9.066,9.243,9.405,9.544,
     * 8.496,8.564,8.653,8.768,8.907,9.061,9.207,9.331,
     * 8.127,8.281,8.474,8.713,8.841,8.975,9.117,9.256,
     * 7.687,7.919,8.194,8.519,8.627,8.743,8.868,8.988/
C
      DATA ((EDEN(I,2,3,L,3),I=25,32),L=1,NZEN)/
     * 9.316,9.436,9.584,9.755,9.927,10.094,10.251,10.394,
     * 9.288,9.397,9.535,9.697,9.855,10.014,10.168,10.312,
     * 9.253,9.353,9.481,9.634,9.768,9.904,10.043,10.181,
     * 9.215,9.307,9.428,9.574,9.695,9.802,9.905,10.013,
     * 9.150,9.240,9.360,9.507,9.625,9.733,9.831,9.923,
     * 9.079,9.178,9.307,9.465,9.584,9.695,9.797,9.892,
     * 8.850,8.984,9.149,9.344,9.480,9.606,9.720,9.826,
     * 8.439,8.568,8.719,8.887,9.065,9.245,9.408,9.546,
     * 8.466,8.553,8.662,8.793,8.942,9.100,9.246,9.367,
     * 8.076,8.255,8.471,8.727,8.873,9.016,9.158,9.295,
     * 7.625,7.885,8.183,8.524,8.658,8.781,8.904,9.020/
C
      DATA ((EDEN(I,3,3,L,3),I=25,32),L=1,NZEN)/
     * 9.404,9.522,9.664,9.826,9.984,10.137,10.282,10.415,
     * 9.364,9.473,9.605,9.758,9.902,10.043,10.181,10.315,
     * 9.328,9.429,9.554,9.700,9.821,9.937,10.065,10.191,
     * 9.275,9.370,9.489,9.630,9.743,9.840,9.932,10.028,
     * 9.186,9.284,9.406,9.552,9.669,9.769,9.858,9.944,
     * 9.114,9.221,9.353,9.508,9.628,9.733,9.826,9.913,
     * 8.862,9.009,9.182,9.379,9.521,9.643,9.750,9.849,
     * 8.432,8.578,8.747,8.926,9.108,9.285,9.444,9.575,
     * 8.434,8.544,8.678,8.825,8.981,9.139,9.280,9.394,
     * 8.037,8.243,8.476,8.740,8.912,9.060,9.195,9.324,
     * 7.585,7.871,8.186,8.533,8.695,8.824,8.942,9.052/
C
      DATA ((EDEN(I,4,3,L,3),I=25,32),L=1,NZEN)/
     * 9.533,9.643,9.771,9.913,10.051,10.184,10.314,10.434,
     * 9.492,9.594,9.714,9.849,9.967,10.089,10.211,10.334,
     * 9.450,9.546,9.660,9.788,9.887,9.982,10.086,10.200,
     * 9.399,9.492,9.603,9.729,9.818,9.891,9.964,10.049,
     * 9.290,9.390,9.507,9.639,9.737,9.816,9.885,9.958,
     * 9.196,9.309,9.438,9.583,9.691,9.776,9.851,9.926,
     * 8.967,9.118,9.285,9.467,9.595,9.696,9.783,9.868,
     * 8.472,8.642,8.828,9.011,9.186,9.351,9.497,9.616,
     * 8.429,8.568,8.722,8.879,9.033,9.182,9.313,9.418,
     * 8.042,8.273,8.519,8.781,8.963,9.107,9.233,9.353,
     * 7.603,7.920,8.250,8.598,8.758,8.880,8.987,9.090/
C
      DATA ((EDEN(I,5,3,L,3),I=25,32),L=1,NZEN)/
     * 9.703,9.808,9.922,10.044,10.156,10.267,10.379,10.484,
     * 9.654,9.752,9.859,9.973,10.071,10.169,10.274,10.382,
     * 9.605,9.698,9.800,9.909,9.985,10.058,10.145,10.246,
     * 9.547,9.638,9.739,9.846,9.911,9.959,10.010,10.080,
     * 9.434,9.536,9.646,9.761,9.832,9.883,9.929,9.986,
     * 9.339,9.456,9.580,9.709,9.788,9.845,9.895,9.954,
     * 9.106,9.262,9.424,9.591,9.691,9.766,9.829,9.899,
     * 8.570,8.771,8.964,9.139,9.297,9.439,9.562,9.665,
     * 8.471,8.649,8.822,8.981,9.126,9.260,9.376,9.470,
     * 8.089,8.359,8.627,8.895,9.057,9.180,9.295,9.403,
     * 7.657,8.023,8.384,8.743,8.866,8.964,9.059,9.154/
C
      DATA ((EDEN(I,1,4,L,3),I=25,32),L=1,NZEN)/
     * 9.267,9.384,9.531,9.702,9.885,10.060,10.222,10.370,
     * 9.245,9.354,9.491,9.655,9.823,9.991,10.153,10.301,
     * 9.218,9.316,9.445,9.599,9.746,9.889,10.040,10.182,
     * 9.187,9.277,9.398,9.546,9.674,9.789,9.900,10.014,
     * 9.139,9.224,9.341,9.487,9.610,9.721,9.822,9.917,
     * 9.088,9.179,9.302,9.457,9.575,9.688,9.792,9.888,
     * 8.877,9.001,9.157,9.347,9.479,9.605,9.720,9.825,
     * 8.471,8.593,8.735,8.896,9.068,9.245,9.408,9.547,
     * 8.481,8.549,8.639,8.755,8.896,9.050,9.196,9.319,
     * 8.113,8.265,8.458,8.697,8.831,8.968,9.107,9.243,
     * 7.675,7.905,8.177,8.499,8.618,8.735,8.858,8.978/
C
      DATA ((EDEN(I,2,4,L,3),I=25,32),L=1,NZEN)/
     * 9.274,9.394,9.542,9.714,9.886,10.053,10.212,10.357,
     * 9.247,9.359,9.499,9.663,9.824,9.980,10.139,10.284,
     * 9.215,9.318,9.450,9.606,9.749,9.887,10.028,10.170,
     * 9.178,9.273,9.398,9.548,9.677,9.788,9.895,10.007,
     * 9.127,9.219,9.342,9.493,9.613,9.721,9.818,9.913,
     * 9.065,9.163,9.292,9.450,9.573,9.682,9.783,9.879,
     * 8.851,8.982,9.144,9.338,9.476,9.600,9.712,9.817,
     * 8.458,8.585,8.733,8.896,9.068,9.245,9.408,9.545,
     * 8.464,8.537,8.634,8.755,8.896,9.051,9.196,9.318,
     * 8.098,8.254,8.449,8.687,8.829,8.966,9.108,9.244,
     * 7.665,7.899,8.174,8.496,8.619,8.736,8.857,8.976/
C
      DATA ((EDEN(I,3,4,L,3),I=25,32),L=1,NZEN)/
     * 9.339,9.462,9.610,9.778,9.939,10.094,10.242,10.379,
     * 9.303,9.418,9.557,9.717,9.858,9.997,10.142,10.277,
     * 9.259,9.366,9.498,9.651,9.782,9.902,10.033,10.165,
     * 9.207,9.308,9.434,9.582,9.707,9.809,9.906,10.012,
     * 9.122,9.222,9.347,9.497,9.625,9.731,9.824,9.914,
     * 9.048,9.156,9.291,9.452,9.583,9.693,9.790,9.883,
     * 8.817,8.962,9.134,9.333,9.482,9.608,9.717,9.821,
     * 8.432,8.564,8.724,8.898,9.075,9.253,9.416,9.553,
     * 8.434,8.517,8.631,8.763,8.910,9.066,9.211,9.332,
     * 8.058,8.226,8.425,8.662,8.837,8.983,9.125,9.258,
     * 7.634,7.877,8.152,8.469,8.622,8.747,8.869,8.988/
C
      DATA ((EDEN(I,4,4,L,3),I=25,32),L=1,NZEN)/
     * 9.432,9.564,9.713,9.877,10.029,10.176,10.314,10.441,
     * 9.387,9.509,9.648,9.802,9.938,10.071,10.204,10.334,
     * 9.335,9.447,9.578,9.725,9.839,9.943,10.061,10.183,
     * 9.251,9.359,9.484,9.626,9.741,9.829,9.912,10.009,
     * 9.128,9.238,9.365,9.511,9.637,9.734,9.815,9.899,
     * 9.026,9.147,9.286,9.444,9.580,9.686,9.774,9.863,
     * 8.777,8.936,9.115,9.313,9.472,9.596,9.700,9.800,
     * 8.399,8.531,8.707,8.892,9.074,9.252,9.414,9.549,
     * 8.393,8.484,8.623,8.774,8.931,9.090,9.235,9.355,
     * 8.015,8.198,8.400,8.630,8.846,9.008,9.148,9.282,
     * 7.624,7.878,8.153,8.458,8.635,8.766,8.888,9.005/
C
      DATA ((EDEN(I,5,4,L,3),I=25,32),L=1,NZEN)/
     * 9.545,9.685,9.835,9.992,10.140,10.275,10.401,10.515,
     * 9.490,9.619,9.759,9.906,10.037,10.160,10.286,10.404,
     * 9.430,9.552,9.683,9.822,9.928,10.025,10.136,10.252,
     * 9.352,9.469,9.595,9.729,9.822,9.886,9.951,10.038,
     * 9.177,9.297,9.427,9.565,9.682,9.763,9.828,9.903,
     * 9.057,9.191,9.334,9.486,9.616,9.708,9.781,9.863,
     * 8.795,8.969,9.151,9.343,9.501,9.614,9.705,9.800,
     * 8.418,8.562,8.749,8.938,9.115,9.283,9.436,9.565,
     * 8.401,8.514,8.672,8.833,8.988,9.140,9.280,9.396,
     * 8.002,8.209,8.421,8.648,8.884,9.055,9.193,9.324,
     * 7.630,7.912,8.199,8.501,8.681,8.809,8.924,9.041/
C
      DATA ((EDEN(I,1,5,L,3),I=25,32),L=1,NZEN)/
     * 9.276,9.396,9.545,9.719,9.904,10.085,10.252,10.401,
     * 9.250,9.360,9.498,9.663,9.841,10.014,10.178,10.328,
     * 9.217,9.316,9.444,9.599,9.752,9.903,10.061,10.208,
     * 9.184,9.274,9.394,9.541,9.672,9.790,9.905,10.024,
     * 9.139,9.222,9.338,9.483,9.606,9.718,9.819,9.913,
     * 9.091,9.179,9.301,9.454,9.573,9.686,9.789,9.885,
     * 8.887,9.007,9.160,9.347,9.481,9.606,9.720,9.824,
     * 8.474,8.596,8.739,8.901,9.073,9.250,9.412,9.549,
     * 8.473,8.539,8.630,8.746,8.885,9.040,9.184,9.307,
     * 8.115,8.263,8.451,8.685,8.821,8.958,9.096,9.234,
     * 7.676,7.901,8.168,8.486,8.608,8.726,8.849,8.967/
C
      DATA ((EDEN(I,2,5,L,3),I=25,32),L=1,NZEN)/
     * 9.251,9.371,9.520,9.692,9.865,10.034,10.195,10.343,
     * 9.231,9.345,9.488,9.655,9.828,9.998,10.163,10.314,
     * 9.199,9.305,9.438,9.597,9.752,9.901,10.055,10.201,
     * 9.167,9.263,9.389,9.541,9.673,9.787,9.899,10.023,
     * 9.121,9.210,9.330,9.480,9.603,9.710,9.807,9.902,
     * 9.067,9.159,9.283,9.438,9.562,9.672,9.772,9.869,
     * 8.872,8.994,9.148,9.336,9.473,9.595,9.705,9.809,
     * 8.493,8.617,8.759,8.916,9.084,9.256,9.415,9.551,
     * 8.470,8.527,8.610,8.720,8.852,9.003,9.148,9.271,
     * 8.135,8.264,8.433,8.650,8.786,8.919,9.059,9.193,
     * 7.686,7.896,8.148,8.452,8.577,8.695,8.819,8.944/
C
      DATA ((EDEN(I,3,5,L,3),I=25,32),L=1,NZEN)/
     * 9.287,9.418,9.573,9.748,9.919,10.083,10.240,10.381,
     * 9.245,9.366,9.512,9.678,9.837,9.993,10.146,10.291,
     * 9.198,9.311,9.449,9.608,9.763,9.903,10.053,10.199,
     * 9.145,9.250,9.381,9.534,9.677,9.791,9.901,10.019,
     * 9.071,9.171,9.297,9.449,9.592,9.704,9.799,9.894,
     * 9.002,9.107,9.239,9.398,9.545,9.660,9.760,9.857,
     * 8.800,8.938,9.105,9.300,9.459,9.585,9.695,9.801,
     * 8.437,8.550,8.706,8.876,9.051,9.230,9.397,9.538,
     * 8.426,8.474,8.571,8.692,8.832,8.988,9.138,9.265,
     * 8.091,8.219,8.381,8.589,8.766,8.913,9.052,9.195,
     * 7.651,7.849,8.084,8.369,8.539,8.672,8.804,8.932/
C
      DATA ((EDEN(I,4,5,L,3),I=25,32),L=1,NZEN)/
     * 9.354,9.506,9.676,9.861,10.051,10.217,10.369,10.503,
     * 9.298,9.437,9.593,9.765,9.940,10.103,10.254,10.390,
     * 9.209,9.334,9.478,9.637,9.790,9.927,10.071,10.213,
     * 9.102,9.218,9.352,9.504,9.658,9.775,9.881,9.999,
     * 8.966,9.078,9.210,9.362,9.530,9.656,9.755,9.851,
     * 8.872,8.993,9.133,9.295,9.470,9.606,9.710,9.811,
     * 8.633,8.784,8.955,9.150,9.355,9.511,9.632,9.746,
     * 8.313,8.395,8.567,8.767,8.964,9.159,9.339,9.490,
     * 8.340,8.372,8.492,8.646,8.809,8.980,9.141,9.276,
     * 7.987,8.106,8.251,8.434,8.697,8.895,9.053,9.198,
     * 7.625,7.802,8.005,8.250,8.478,8.639,8.778,8.912/
C
      DATA ((EDEN(I,5,5,L,3),I=25,32),L=1,NZEN)/
     * 9.436,9.607,9.788,9.976,10.177,10.349,10.492,10.620,
     * 9.363,9.518,9.684,9.857,10.051,10.223,10.372,10.504,
     * 9.279,9.420,9.572,9.732,9.900,10.054,10.204,10.341,
     * 9.126,9.254,9.392,9.540,9.693,9.804,9.905,10.023,
     * 8.920,9.043,9.177,9.324,9.499,9.636,9.736,9.833,
     * 8.801,8.932,9.073,9.228,9.415,9.566,9.679,9.783,
     * 8.565,8.719,8.884,9.064,9.287,9.469,9.600,9.717,
     * 8.284,8.351,8.517,8.733,8.942,9.141,9.322,9.477,
     * 8.311,8.336,8.459,8.634,8.811,8.985,9.150,9.290,
     * 7.937,8.057,8.190,8.351,8.641,8.883,9.059,9.209,
     * 7.558,7.722,7.899,8.108,8.385,8.610,8.769,8.909/
C
      DATA ((EDEN(I,1,6,L,3),I=25,32),L=1,NZEN)/
     * 9.292,9.413,9.564,9.739,9.931,10.113,10.281,10.430,
     * 9.265,9.375,9.514,9.679,9.862,10.038,10.206,10.357,
     * 9.229,9.326,9.453,9.607,9.767,9.925,10.084,10.235,
     * 9.194,9.281,9.398,9.542,9.673,9.793,9.913,10.038,
     * 9.149,9.230,9.342,9.483,9.606,9.717,9.818,9.911,
     * 9.105,9.189,9.307,9.455,9.574,9.686,9.788,9.883,
     * 8.911,9.025,9.172,9.353,9.483,9.608,9.720,9.823,
     * 8.483,8.605,8.748,8.908,9.079,9.255,9.415,9.550,
     * 8.474,8.537,8.625,8.740,8.879,9.032,9.176,9.298,
     * 8.127,8.269,8.451,8.680,8.814,8.950,9.088,9.222,
     * 7.690,7.909,8.170,8.481,8.602,8.719,8.840,8.957/
C
      DATA ((EDEN(I,2,6,L,3),I=25,32),L=1,NZEN)/
     * 9.250,9.369,9.517,9.688,9.862,10.034,10.200,10.349,
     * 9.230,9.345,9.488,9.655,9.835,10.012,10.181,10.334,
     * 9.202,9.307,9.440,9.599,9.762,9.922,10.083,10.234,
     * 9.168,9.263,9.387,9.538,9.673,9.794,9.919,10.049,
     * 9.130,9.215,9.331,9.476,9.599,9.706,9.802,9.897,
     * 9.083,9.169,9.286,9.436,9.559,9.667,9.766,9.862,
     * 8.922,9.034,9.179,9.357,9.485,9.600,9.707,9.810,
     * 8.527,8.648,8.786,8.939,9.101,9.269,9.425,9.558,
     * 8.481,8.525,8.596,8.696,8.823,8.969,9.113,9.237,
     * 8.174,8.287,8.441,8.646,8.768,8.894,9.031,9.167,
     * 7.723,7.913,8.146,8.435,8.550,8.664,8.789,8.913/
C
      DATA ((EDEN(I,3,6,L,3),I=25,32),L=1,NZEN)/
     * 9.266,9.401,9.560,9.740,9.926,10.096,10.260,10.404,
     * 9.220,9.345,9.494,9.664,9.852,10.030,10.197,10.348,
     * 9.173,9.288,9.428,9.590,9.762,9.925,10.086,10.241,
     * 9.121,9.226,9.356,9.509,9.663,9.790,9.917,10.052,
     * 9.053,9.148,9.270,9.419,9.571,9.687,9.785,9.882,
     * 9.006,9.105,9.231,9.387,9.535,9.651,9.750,9.848,
     * 8.808,8.936,9.092,9.280,9.443,9.571,9.682,9.789,
     * 8.454,8.554,8.705,8.872,9.044,9.222,9.391,9.533,
     * 8.444,8.470,8.549,8.656,8.787,8.939,9.090,9.221,
     * 8.116,8.210,8.340,8.520,8.705,8.854,8.997,9.140,
     * 7.652,7.811,8.010,8.264,8.459,8.610,8.750,8.885/
C
      DATA ((EDEN(I,4,6,L,3),I=25,32),L=1,NZEN)/
     * 9.331,9.497,9.682,9.881,10.098,10.283,10.441,10.580,
     * 9.243,9.390,9.555,9.736,9.943,10.133,10.300,10.446,
     * 9.131,9.260,9.408,9.573,9.759,9.937,10.111,10.270,
     * 9.034,9.152,9.288,9.443,9.613,9.753,9.884,10.027,
     * 8.899,9.008,9.137,9.288,9.470,9.615,9.726,9.830,
     * 8.807,8.919,9.052,9.209,9.398,9.554,9.672,9.781,
     * 8.600,8.736,8.895,9.080,9.296,9.469,9.601,9.720,
     * 8.293,8.344,8.503,8.707,8.911,9.113,9.302,9.463,
     * 8.328,8.318,8.412,8.559,8.720,8.894,9.063,9.208,
     * 7.971,8.040,8.139,8.282,8.563,8.796,8.970,9.128,
     * 7.614,7.726,7.869,8.063,8.325,8.532,8.689,8.835/
C
      DATA ((EDEN(I,5,6,L,3),I=25,32),L=1,NZEN)/
     * 9.419,9.616,9.822,10.036,10.259,10.443,10.597,10.735,
     * 9.333,9.510,9.697,9.892,10.121,10.319,10.477,10.618,
     * 9.178,9.326,9.485,9.653,9.849,10.046,10.226,10.380,
     * 9.019,9.150,9.291,9.443,9.614,9.755,9.896,10.053,
     * 8.823,8.941,9.071,9.215,9.399,9.561,9.686,9.797,
     * 8.720,8.840,8.972,9.120,9.312,9.491,9.624,9.741,
     * 8.513,8.648,8.796,8.963,9.189,9.395,9.545,9.674,
     * 8.255,8.277,8.411,8.629,8.849,9.060,9.255,9.425,
     * 8.313,8.285,8.362,8.520,8.691,8.868,9.041,9.193,
     * 7.966,8.025,8.101,8.214,8.489,8.755,8.944,9.111,
     * 7.561,7.646,7.751,7.899,8.137,8.437,8.639,8.801/
C
      DATA ((EDEN(I,1,7,L,3),I=25,32),L=1,NZEN)/
     * 9.304,9.425,9.575,9.750,9.942,10.124,10.291,10.441,
     * 9.276,9.385,9.524,9.689,9.872,10.052,10.218,10.369,
     * 9.239,9.335,9.462,9.615,9.775,9.935,10.094,10.246,
     * 9.201,9.287,9.403,9.546,9.676,9.796,9.917,10.045,
     * 9.159,9.237,9.347,9.487,9.608,9.718,9.818,9.911,
     * 9.115,9.198,9.313,9.459,9.575,9.687,9.788,9.883,
     * 8.926,9.036,9.180,9.357,9.486,9.609,9.721,9.824,
     * 8.491,8.613,8.755,8.913,9.084,9.258,9.417,9.552,
     * 8.477,8.540,8.626,8.740,8.876,9.029,9.173,9.294,
     * 8.137,8.277,8.456,8.682,8.813,8.947,9.086,9.220,
     * 7.700,7.916,8.175,8.483,8.601,8.717,8.838,8.957/
C
      DATA ((EDEN(I,2,7,L,3),I=25,32),L=1,NZEN)/
     * 9.260,9.379,9.525,9.695,9.869,10.043,10.208,10.359,
     * 9.243,9.356,9.498,9.664,9.846,10.020,10.192,10.344,
     * 9.214,9.318,9.450,9.608,9.771,9.932,10.094,10.245,
     * 9.180,9.273,9.395,9.544,9.678,9.801,9.926,10.059,
     * 9.145,9.227,9.340,9.483,9.604,9.709,9.805,9.900,
     * 9.101,9.183,9.297,9.443,9.563,9.670,9.769,9.864,
     * 8.946,9.053,9.193,9.366,9.491,9.605,9.710,9.813,
     * 8.547,8.666,8.802,8.952,9.111,9.277,9.431,9.563,
     * 8.491,8.530,8.597,8.692,8.816,8.961,9.104,9.228,
     * 8.191,8.298,8.445,8.644,8.761,8.885,9.021,9.159,
     * 7.742,7.924,8.151,8.433,8.544,8.655,8.780,8.905/
C
      DATA ((EDEN(I,3,7,L,3),I=25,32),L=1,NZEN)/
     * 9.281,9.415,9.574,9.753,9.939,10.114,10.276,10.420,
     * 9.235,9.359,9.507,9.677,9.864,10.045,10.213,10.364,
     * 9.189,9.303,9.441,9.601,9.776,9.940,10.106,10.256,
     * 9.138,9.240,9.368,9.519,9.672,9.800,9.931,10.065,
     * 9.075,9.166,9.284,9.430,9.578,9.693,9.790,9.886,
     * 9.032,9.126,9.247,9.398,9.543,9.657,9.755,9.852,
     * 8.841,8.962,9.112,9.294,9.453,9.579,9.688,9.793,
     * 8.483,8.580,8.726,8.888,9.057,9.231,9.398,9.539,
     * 8.461,8.479,8.550,8.650,8.776,8.926,9.077,9.207,
     * 8.142,8.225,8.345,8.516,8.696,8.839,8.984,9.125,
     * 7.677,7.825,8.014,8.260,8.447,8.596,8.736,8.872/
C
      DATA ((EDEN(I,4,7,L,3),I=25,32),L=1,NZEN)/
     * 9.357,9.526,9.712,9.913,10.127,10.310,10.467,10.606,
     * 9.267,9.414,9.579,9.760,9.973,10.162,10.328,10.471,
     * 9.152,9.279,9.425,9.588,9.778,9.961,10.135,10.294,
     * 9.057,9.171,9.304,9.456,9.625,9.767,9.902,10.045,
     * 8.930,9.034,9.158,9.304,9.483,9.627,9.735,9.837,
     * 8.842,8.948,9.076,9.228,9.414,9.565,9.681,9.788,
     * 8.640,8.770,8.923,9.103,9.314,9.482,9.609,9.727,
     * 8.323,8.370,8.524,8.722,8.920,9.118,9.306,9.467,
     * 8.350,8.330,8.414,8.552,8.707,8.877,9.046,9.191,
     * 8.003,8.059,8.145,8.278,8.552,8.778,8.952,9.109,
     * 7.653,7.749,7.878,8.060,8.310,8.512,8.668,8.812/
C
      DATA ((EDEN(I,5,7,L,3),I=25,32),L=1,NZEN)/
     * 9.449,9.649,9.859,10.077,10.299,10.477,10.630,10.766,
     * 9.363,9.543,9.732,9.930,10.161,10.352,10.512,10.649,
     * 9.202,9.349,9.507,9.674,9.877,10.080,10.260,10.412,
     * 9.043,9.170,9.308,9.457,9.625,9.771,9.919,10.075,
     * 8.856,8.969,9.094,9.234,9.414,9.575,9.696,9.806,
     * 8.761,8.874,9.000,9.144,9.332,9.504,9.635,9.750,
     * 8.559,8.688,8.831,8.995,9.212,9.410,9.556,9.682,
     * 8.306,8.328,8.465,8.674,8.881,9.082,9.272,9.438,
     * 8.339,8.299,8.363,8.509,8.672,8.845,9.017,9.170,
     * 8.002,8.043,8.103,8.203,8.471,8.731,8.921,9.085,
     * 7.600,7.667,7.753,7.887,8.118,8.413,8.612,8.775/
C
      DATA ((EDEN(I,1,8,L,3),I=25,32),L=1,NZEN)/
     * 9.302,9.423,9.572,9.746,9.934,10.115,10.282,10.430,
     * 9.274,9.383,9.521,9.685,9.865,10.041,10.208,10.357,
     * 9.238,9.335,9.461,9.614,9.771,9.928,10.086,10.235,
     * 9.203,9.289,9.406,9.549,9.678,9.797,9.917,10.042,
     * 9.159,9.238,9.349,9.490,9.610,9.720,9.821,9.914,
     * 9.115,9.199,9.315,9.462,9.578,9.689,9.790,9.885,
     * 8.924,9.036,9.181,9.359,9.489,9.611,9.723,9.826,
     * 8.494,8.616,8.757,8.915,9.085,9.259,9.419,9.554,
     * 8.481,8.544,8.631,8.745,8.882,9.034,9.178,9.300,
     * 8.136,8.278,8.459,8.687,8.817,8.950,9.091,9.224,
     * 7.701,7.919,8.180,8.491,8.607,8.721,8.842,8.960/
C
      DATA ((EDEN(I,2,8,L,3),I=25,32),L=1,NZEN)/
     * 9.278,9.395,9.540,9.708,9.882,10.053,10.217,10.366,
     * 9.260,9.371,9.510,9.673,9.850,10.021,10.187,10.337,
     * 9.229,9.331,9.461,9.616,9.772,9.925,10.083,10.233,
     * 9.196,9.287,9.408,9.556,9.685,9.802,9.923,10.050,
     * 9.159,9.241,9.353,9.495,9.614,9.719,9.815,9.908,
     * 9.112,9.194,9.309,9.455,9.574,9.681,9.779,9.873,
     * 8.934,9.043,9.184,9.359,9.489,9.606,9.714,9.816,
     * 8.540,8.660,8.797,8.949,9.110,9.277,9.432,9.564,
     * 8.496,8.540,8.610,8.709,8.834,8.980,9.123,9.246,
     * 8.184,8.299,8.454,8.660,8.778,8.905,9.040,9.178,
     * 7.733,7.923,8.156,8.443,8.558,8.671,8.795,8.920/
C
      DATA ((EDEN(I,3,8,L,3),I=25,32),L=1,NZEN)/
     * 9.323,9.452,9.605,9.778,9.953,10.119,10.274,10.417,
     * 9.279,9.397,9.540,9.704,9.876,10.045,10.204,10.350,
     * 9.236,9.344,9.476,9.631,9.789,9.941,10.092,10.239,
     * 9.188,9.285,9.409,9.556,9.695,9.812,9.929,10.053,
     * 9.126,9.215,9.331,9.474,9.608,9.717,9.812,9.904,
     * 9.068,9.159,9.277,9.424,9.562,9.674,9.771,9.866,
     * 8.884,9.005,9.154,9.335,9.481,9.603,9.709,9.812,
     * 8.507,8.611,8.754,8.912,9.078,9.250,9.413,9.551,
     * 8.484,8.515,8.587,8.688,8.813,8.959,9.106,9.234,
     * 8.158,8.256,8.390,8.573,8.736,8.876,9.014,9.155,
     * 7.691,7.857,8.063,8.323,8.493,8.632,8.767,8.898/
C
      DATA ((EDEN(I,4,8,L,3),I=25,32),L=1,NZEN)/
     * 9.417,9.571,9.742,9.928,10.120,10.287,10.434,10.565,
     * 9.342,9.475,9.627,9.795,9.981,10.147,10.300,10.435,
     * 9.263,9.381,9.517,9.669,9.830,9.979,10.130,10.270,
     * 9.163,9.268,9.393,9.535,9.686,9.805,9.918,10.044,
     * 9.049,9.145,9.262,9.401,9.560,9.682,9.778,9.871,
     * 8.968,9.070,9.193,9.340,9.504,9.632,9.733,9.829,
     * 8.741,8.873,9.026,9.206,9.393,9.540,9.656,9.764,
     * 8.409,8.480,8.634,8.814,8.995,9.179,9.354,9.504,
     * 8.403,8.406,8.495,8.625,8.771,8.933,9.094,9.232,
     * 8.074,8.155,8.264,8.417,8.660,8.844,9.000,9.151,
     * 7.722,7.854,8.016,8.228,8.434,8.583,8.718,8.856/
C
      DATA ((EDEN(I,5,8,L,3),I=25,32),L=1,NZEN)/
     * 9.516,9.690,9.875,10.066,10.269,10.432,10.571,10.696,
     * 9.440,9.596,9.761,9.935,10.137,10.310,10.455,10.581,
     * 9.335,9.467,9.609,9.761,9.941,10.107,10.259,10.392,
     * 9.176,9.291,9.416,9.551,9.704,9.826,9.938,10.066,
     * 9.016,9.122,9.239,9.371,9.536,9.668,9.765,9.858,
     * 8.913,9.024,9.147,9.286,9.457,9.602,9.707,9.806,
     * 8.689,8.825,8.972,9.139,9.339,9.506,9.628,9.739,
     * 8.384,8.434,8.584,8.777,8.966,9.152,9.329,9.484,
     * 8.403,8.395,8.480,8.616,8.764,8.925,9.087,9.230,
     * 8.066,8.140,8.229,8.355,8.616,8.827,8.989,9.146,
     * 7.676,7.784,7.910,8.078,8.329,8.538,8.691,8.838/
C
      DATA ((EDEN(I,1,9,L,3),I=25,32),L=1,NZEN)/
     * 9.288,9.407,9.554,9.727,9.913,10.090,10.255,10.403,
     * 9.265,9.373,9.511,9.674,9.848,10.019,10.182,10.332,
     * 9.233,9.330,9.457,9.610,9.760,9.911,10.064,10.210,
     * 9.200,9.287,9.406,9.551,9.679,9.795,9.909,10.028,
     * 9.154,9.236,9.349,9.493,9.612,9.723,9.825,9.918,
     * 9.108,9.194,9.313,9.463,9.580,9.691,9.794,9.890,
     * 8.909,9.026,9.175,9.358,9.488,9.611,9.725,9.829,
     * 8.489,8.612,8.753,8.912,9.083,9.258,9.419,9.555,
     * 8.485,8.550,8.638,8.752,8.890,9.043,9.187,9.309,
     * 8.131,8.277,8.463,8.695,8.826,8.961,9.099,9.236,
     * 7.694,7.917,8.183,8.499,8.613,8.730,8.851,8.971/
C
      DATA ((EDEN(I,2,9,L,3),I=25,32),L=1,NZEN)/
     * 9.297,9.414,9.559,9.727,9.899,10.069,10.229,10.377,
     * 9.278,9.387,9.523,9.684,9.850,10.015,10.179,10.325,
     * 9.247,9.345,9.473,9.625,9.769,9.912,10.058,10.204,
     * 9.215,9.305,9.424,9.570,9.693,9.804,9.913,10.027,
     * 9.168,9.252,9.366,9.510,9.626,9.732,9.828,9.921,
     * 9.114,9.201,9.319,9.469,9.586,9.695,9.794,9.888,
     * 8.918,9.035,9.184,9.365,9.496,9.616,9.726,9.828,
     * 8.511,8.632,8.772,8.929,9.096,9.268,9.427,9.561,
     * 8.493,8.549,8.631,8.739,8.872,9.022,9.165,9.287,
     * 8.149,8.282,8.454,8.674,8.805,8.939,9.075,9.211,
     * 7.698,7.907,8.157,8.458,8.586,8.705,8.828,8.948/
C
      DATA ((EDEN(I,3,9,L,3),I=25,32),L=1,NZEN)/
     * 9.374,9.494,9.638,9.803,9.965,10.124,10.274,10.409,
     * 9.338,9.447,9.580,9.735,9.885,10.031,10.177,10.313,
     * 9.299,9.399,9.523,9.669,9.801,9.928,10.060,10.192,
     * 9.250,9.343,9.461,9.602,9.725,9.830,9.933,10.039,
     * 9.184,9.272,9.387,9.527,9.649,9.752,9.843,9.931,
     * 9.123,9.215,9.335,9.482,9.605,9.711,9.805,9.896,
     * 8.913,9.038,9.190,9.371,9.509,9.630,9.735,9.835,
     * 8.509,8.625,8.772,8.933,9.100,9.271,9.431,9.566,
     * 8.488,8.541,8.628,8.740,8.872,9.020,9.164,9.288,
     * 8.146,8.278,8.443,8.652,8.804,8.942,9.076,9.215,
     * 7.719,7.918,8.153,8.437,8.576,8.695,8.816,8.938/
C
      DATA ((EDEN(I,4,9,L,3),I=25,32),L=1,NZEN)/
     * 9.503,9.630,9.776,9.936,10.095,10.240,10.376,10.499,
     * 9.458,9.573,9.705,9.853,9.996,10.135,10.267,10.392,
     * 9.395,9.497,9.617,9.753,9.873,9.986,10.105,10.232,
     * 9.333,9.427,9.539,9.669,9.779,9.865,9.949,10.045,
     * 9.226,9.316,9.425,9.554,9.675,9.768,9.847,9.927,
     * 9.151,9.248,9.364,9.501,9.625,9.722,9.805,9.889,
     * 8.921,9.053,9.205,9.380,9.524,9.637,9.733,9.828,
     * 8.505,8.614,8.764,8.928,9.093,9.262,9.420,9.555,
     * 8.493,8.546,8.642,8.758,8.889,9.036,9.179,9.303,
     * 8.133,8.266,8.423,8.616,8.802,8.951,9.087,9.226,
     * 7.782,7.980,8.203,8.465,8.592,8.699,8.815,8.937/
C
      DATA ((EDEN(I,5,9,L,3),I=25,32),L=1,NZEN)/
     * 9.649,9.783,9.926,10.077,10.227,10.355,10.476,10.585,
     * 9.595,9.715,9.845,9.982,10.120,10.245,10.364,10.475,
     * 9.538,9.645,9.762,9.887,9.998,10.101,10.214,10.326,
     * 9.463,9.559,9.665,9.781,9.869,9.934,10.000,10.089,
     * 9.321,9.415,9.519,9.634,9.739,9.812,9.872,9.941,
     * 9.237,9.338,9.450,9.574,9.682,9.761,9.826,9.899,
     * 8.990,9.129,9.277,9.440,9.571,9.670,9.750,9.835,
     * 8.569,8.683,8.834,8.989,9.141,9.295,9.441,9.569,
     * 8.532,8.594,8.696,8.811,8.937,9.074,9.212,9.334,
     * 8.177,8.323,8.480,8.662,8.843,8.983,9.114,9.250,
     * 7.811,8.019,8.239,8.486,8.618,8.720,8.831,8.956/
C
      DATA ((EDEN(I,1,10,L,3),I=25,32),L=1,NZEN)/
     * 9.283,9.399,9.545,9.715,9.895,10.068,10.230,10.378,
     * 9.261,9.368,9.504,9.666,9.834,9.997,10.160,10.307,
     * 9.234,9.331,9.457,9.610,9.754,9.896,10.046,10.189,
     * 9.203,9.292,9.410,9.556,9.681,9.795,9.905,10.021,
     * 9.157,9.239,9.354,9.498,9.617,9.727,9.828,9.922,
     * 9.107,9.195,9.316,9.467,9.582,9.694,9.797,9.893,
     * 8.904,9.022,9.174,9.359,9.488,9.612,9.726,9.831,
     * 8.487,8.609,8.749,8.908,9.079,9.255,9.417,9.554,
     * 8.492,8.559,8.647,8.761,8.900,9.054,9.199,9.322,
     * 8.130,8.280,8.470,8.706,8.835,8.972,9.110,9.247,
     * 7.690,7.919,8.190,8.511,8.623,8.738,8.861,8.981/
C
      DATA ((EDEN(I,2,10,L,3),I=25,32),L=1,NZEN)/
     * 9.323,9.439,9.583,9.751,9.927,10.096,10.257,10.400,
     * 9.297,9.404,9.538,9.697,9.858,10.016,10.173,10.317,
     * 9.266,9.362,9.487,9.637,9.774,9.910,10.051,10.190,
     * 9.232,9.320,9.438,9.582,9.702,9.811,9.915,10.022,
     * 9.176,9.260,9.375,9.518,9.635,9.743,9.840,9.932,
     * 9.114,9.205,9.328,9.480,9.598,9.707,9.808,9.902,
     * 8.898,9.023,9.180,9.368,9.499,9.623,9.736,9.839,
     * 8.473,8.595,8.742,8.906,9.082,9.259,9.421,9.557,
     * 8.484,8.556,8.653,8.773,8.915,9.069,9.214,9.335,
     * 8.111,8.270,8.466,8.706,8.848,8.986,9.127,9.259,
     * 7.668,7.902,8.176,8.497,8.628,8.749,8.871,8.989/
C
      DATA ((EDEN(I,3,10,L,3),I=25,32),L=1,NZEN)/
     * 9.424,9.537,9.673,9.830,9.990,10.144,10.292,10.426,
     * 9.389,9.492,9.619,9.767,9.908,10.050,10.192,10.324,
     * 9.356,9.449,9.566,9.705,9.826,9.946,10.069,10.197,
     * 9.312,9.399,9.512,9.646,9.757,9.854,9.946,10.043,
     * 9.237,9.324,9.436,9.572,9.685,9.784,9.872,9.956,
     * 9.160,9.256,9.378,9.525,9.641,9.746,9.838,9.925,
     * 8.937,9.068,9.225,9.409,9.542,9.662,9.767,9.863,
     * 8.491,8.619,8.774,8.942,9.115,9.289,9.447,9.580,
     * 8.480,8.561,8.671,8.799,8.942,9.094,9.235,9.352,
     * 8.103,8.273,8.474,8.710,8.872,9.013,9.148,9.280,
     * 7.664,7.905,8.178,8.490,8.644,8.769,8.888,9.002/
C
      DATA ((EDEN(I,4,10,L,3),I=25,32),L=1,NZEN)/
     * 9.597,9.699,9.819,9.954,10.087,10.219,10.347,10.465,
     * 9.558,9.650,9.760,9.885,10.005,10.123,10.247,10.364,
     * 9.523,9.607,9.710,9.828,9.925,10.021,10.125,10.235,
     * 9.475,9.554,9.652,9.766,9.851,9.923,9.996,10.078,
     * 9.378,9.461,9.561,9.679,9.771,9.848,9.915,9.984,
     * 9.311,9.401,9.509,9.635,9.731,9.811,9.881,9.952,
     * 9.084,9.210,9.353,9.515,9.632,9.727,9.811,9.892,
     * 8.584,8.725,8.881,9.041,9.200,9.356,9.498,9.618,
     * 8.529,8.623,8.737,8.862,8.995,9.135,9.265,9.375,
     * 8.163,8.345,8.546,8.772,8.925,9.056,9.181,9.303,
     * 7.740,7.998,8.275,8.579,8.708,8.815,8.921,9.028/
C
      DATA ((EDEN(I,5,10,L,3),I=25,32),L=1,NZEN)/
     * 9.812,9.901,10.001,10.108,10.213,10.319,10.421,10.521,
     * 9.769,9.849,9.939,10.036,10.129,10.223,10.321,10.421,
     * 9.728,9.801,9.884,9.974,10.044,10.114,10.196,10.287,
     * 9.684,9.753,9.832,9.919,9.973,10.016,10.064,10.128,
     * 9.590,9.665,9.747,9.838,9.896,9.939,9.979,10.029,
     * 9.513,9.597,9.689,9.790,9.853,9.900,9.943,9.996,
     * 9.316,9.431,9.552,9.683,9.760,9.820,9.874,9.937,
     * 8.742,8.901,9.052,9.193,9.326,9.454,9.570,9.672,
     * 8.637,8.754,8.866,8.975,9.088,9.206,9.318,9.418,
     * 8.280,8.483,8.690,8.908,9.020,9.120,9.231,9.345,
     * 7.867,8.152,8.439,8.737,8.809,8.884,8.973,9.074/
C
      DATA ((EDEN(I,1,11,L,3),I=25,32),L=1,NZEN)/
     * 9.292,9.405,9.547,9.715,9.885,10.053,10.214,10.362,
     * 9.270,9.374,9.508,9.667,9.826,9.985,10.143,10.292,
     * 9.244,9.339,9.464,9.615,9.753,9.889,10.034,10.175,
     * 9.215,9.302,9.419,9.563,9.685,9.796,9.905,10.014,
     * 9.168,9.249,9.362,9.505,9.621,9.730,9.830,9.924,
     * 9.119,9.205,9.323,9.473,9.585,9.695,9.799,9.895,
     * 8.913,9.029,9.179,9.362,9.488,9.611,9.725,9.831,
     * 8.492,8.612,8.749,8.906,9.075,9.251,9.412,9.550,
     * 8.504,8.571,8.657,8.771,8.910,9.063,9.208,9.332,
     * 8.138,8.290,8.481,8.718,8.844,8.979,9.120,9.256,
     * 7.697,7.928,8.201,8.525,8.631,8.745,8.869,8.991/
C
      DATA ((EDEN(I,2,11,L,3),I=25,32),L=1,NZEN)/
     * 9.355,9.472,9.617,9.785,9.962,10.133,10.291,10.436,
     * 9.317,9.422,9.557,9.715,9.876,10.036,10.191,10.334,
     * 9.292,9.386,9.509,9.656,9.789,9.926,10.064,10.201,
     * 9.257,9.344,9.459,9.600,9.716,9.822,9.922,10.028,
     * 9.193,9.277,9.392,9.534,9.648,9.756,9.853,9.942,
     * 9.116,9.210,9.335,9.488,9.604,9.716,9.817,9.910,
     * 8.882,9.012,9.174,9.364,9.499,9.626,9.740,9.844,
     * 8.456,8.581,8.730,8.898,9.076,9.256,9.419,9.556,
     * 8.479,8.564,8.674,8.806,8.955,9.114,9.259,9.379,
     * 8.081,8.262,8.479,8.736,8.887,9.031,9.172,9.306,
     * 7.617,7.877,8.174,8.513,8.661,8.790,8.913,9.028/
C
      DATA ((EDEN(I,3,11,L,3),I=25,32),L=1,NZEN)/
     * 9.462,9.573,9.708,9.863,10.026,10.184,10.331,10.465,
     * 9.423,9.523,9.648,9.792,9.940,10.088,10.232,10.365,
     * 9.386,9.479,9.595,9.732,9.857,9.982,10.111,10.236,
     * 9.353,9.439,9.548,9.679,9.785,9.881,9.976,10.072,
     * 9.265,9.353,9.466,9.600,9.710,9.811,9.899,9.979,
     * 9.195,9.292,9.414,9.559,9.673,9.777,9.867,9.949,
     * 8.936,9.075,9.238,9.425,9.562,9.684,9.790,9.884,
     * 8.467,8.607,8.773,8.953,9.134,9.311,9.469,9.598,
     * 8.458,8.564,8.695,8.842,8.995,9.151,9.289,9.400,
     * 8.052,8.258,8.490,8.751,8.929,9.080,9.214,9.339,
     * 7.584,7.866,8.175,8.515,8.700,8.839,8.958,9.064/
C
      DATA ((EDEN(I,4,11,L,3),I=25,32),L=1,NZEN)/
     * 9.658,9.747,9.853,9.974,10.104,10.233,10.360,10.477,
     * 9.622,9.703,9.802,9.915,10.027,10.145,10.265,10.381,
     * 9.587,9.662,9.755,9.863,9.957,10.050,10.151,10.256,
     * 9.548,9.621,9.712,9.818,9.898,9.968,10.036,10.112,
     * 9.464,9.544,9.641,9.752,9.837,9.908,9.969,10.031,
     * 9.399,9.489,9.595,9.716,9.804,9.879,9.942,10.004,
     * 9.182,9.307,9.448,9.604,9.712,9.803,9.877,9.948,
     * 8.608,8.775,8.950,9.124,9.289,9.442,9.575,9.683,
     * 8.520,8.649,8.788,8.931,9.073,9.210,9.330,9.427,
     * 8.156,8.377,8.612,8.862,9.013,9.140,9.254,9.363,
     * 7.681,7.991,8.313,8.649,8.802,8.915,9.012,9.104/
C
      DATA ((EDEN(I,5,11,L,3),I=25,32),L=1,NZEN)/
     * 9.917,9.983,10.059,10.143,10.239,10.334,10.432,10.528,
     * 9.878,9.937,10.006,10.082,10.166,10.250,10.339,10.433,
     * 9.841,9.896,9.960,10.032,10.095,10.155,10.228,10.312,
     * 9.800,9.856,9.920,9.991,10.041,10.078,10.118,10.171,
     * 9.725,9.791,9.864,9.942,9.993,10.027,10.057,10.098,
     * 9.653,9.732,9.817,9.906,9.962,10.000,10.032,10.071,
     * 9.469,9.580,9.695,9.813,9.884,9.934,9.975,10.021,
     * 8.837,9.030,9.204,9.358,9.490,9.606,9.701,9.780,
     * 8.658,8.814,8.955,9.082,9.198,9.308,9.402,9.482,
     * 8.319,8.563,8.804,9.041,9.146,9.233,9.322,9.416,
     * 7.847,8.192,8.531,8.864,8.952,9.026,9.098,9.176/
C
      DATA ((EDEN(I,1,12,L,3),I=25,32),L=1,NZEN)/
     * 9.303,9.412,9.551,9.715,9.882,10.047,10.208,10.356,
     * 9.283,9.384,9.515,9.671,9.823,9.981,10.138,10.286,
     * 9.259,9.351,9.473,9.621,9.754,9.889,10.031,10.170,
     * 9.230,9.314,9.429,9.571,9.688,9.797,9.905,10.014,
     * 9.184,9.262,9.372,9.511,9.623,9.731,9.831,9.924,
     * 9.135,9.217,9.331,9.477,9.587,9.696,9.799,9.895,
     * 8.931,9.042,9.187,9.365,9.488,9.609,9.724,9.830,
     * 8.502,8.619,8.752,8.906,9.073,9.247,9.408,9.546,
     * 8.517,8.582,8.667,8.778,8.915,9.069,9.215,9.339,
     * 8.155,8.303,8.492,8.727,8.848,8.982,9.125,9.262,
     * 7.709,7.940,8.213,8.536,8.635,8.751,8.874,8.998/
C
      DATA ((EDEN(I,2,12,L,3),I=25,32),L=1,NZEN)/
     * 9.378,9.496,9.641,9.810,9.989,10.161,10.318,10.461,
     * 9.338,9.445,9.579,9.737,9.900,10.064,10.220,10.360,
     * 9.308,9.403,9.526,9.674,9.811,9.951,10.097,10.234,
     * 9.274,9.358,9.471,9.610,9.724,9.828,9.928,10.030,
     * 9.205,9.289,9.402,9.542,9.655,9.761,9.858,9.945,
     * 9.137,9.229,9.351,9.500,9.616,9.725,9.825,9.915,
     * 8.878,9.009,9.171,9.361,9.497,9.623,9.740,9.844,
     * 8.452,8.578,8.727,8.897,9.075,9.256,9.420,9.555,
     * 8.477,8.567,8.681,8.818,8.970,9.130,9.275,9.393,
     * 8.074,8.266,8.493,8.758,8.912,9.061,9.204,9.336,
     * 7.591,7.866,8.177,8.528,8.684,8.817,8.941,9.056/
C
      DATA ((EDEN(I,3,12,L,3),I=25,32),L=1,NZEN)/
     * 9.485,9.597,9.733,9.890,10.058,10.217,10.366,10.500,
     * 9.442,9.543,9.668,9.813,9.970,10.123,10.268,10.401,
     * 9.399,9.492,9.609,9.746,9.875,10.007,10.141,10.271,
     * 9.366,9.452,9.561,9.691,9.797,9.893,9.991,10.088,
     * 9.290,9.379,9.492,9.626,9.733,9.831,9.915,9.994,
     * 9.200,9.301,9.426,9.572,9.687,9.791,9.881,9.960,
     * 8.970,9.109,9.272,9.456,9.589,9.709,9.812,9.900,
     * 8.446,8.595,8.771,8.959,9.146,9.325,9.483,9.609,
     * 8.435,8.557,8.705,8.865,9.028,9.187,9.324,9.431,
     * 8.035,8.259,8.507,8.779,8.962,9.113,9.248,9.366,
     * 7.526,7.834,8.166,8.524,8.731,8.881,9.000,9.102/
C
      DATA ((EDEN(I,4,12,L,3),I=25,32),L=1,NZEN)/
     * 9.677,9.765,9.870,9.991,10.125,10.259,10.386,10.505,
     * 9.637,9.717,9.815,9.927,10.048,10.170,10.292,10.408,
     * 9.600,9.675,9.767,9.874,9.972,10.070,10.176,10.285,
     * 9.557,9.632,9.724,9.829,9.913,9.986,10.056,10.133,
     * 9.483,9.567,9.667,9.779,9.865,9.936,9.995,10.052,
     * 9.406,9.503,9.615,9.740,9.833,9.909,9.970,10.026,
     * 9.211,9.342,9.487,9.644,9.754,9.843,9.913,9.978,
     * 8.610,8.797,8.990,9.176,9.346,9.498,9.625,9.724,
     * 8.492,8.646,8.807,8.965,9.115,9.252,9.367,9.455,
     * 8.116,8.366,8.624,8.890,9.059,9.190,9.295,9.397,
     * 7.621,7.965,8.317,8.674,8.852,8.974,9.067,9.150/
C
      DATA ((EDEN(I,5,12,L,3),I=25,32),L=1,NZEN)/
     * 9.945,10.010,10.084,10.166,10.267,10.365,10.465,10.560,
     * 9.905,9.964,10.032,10.107,10.194,10.280,10.373,10.468,
     * 9.864,9.919,9.983,10.054,10.120,10.184,10.262,10.348,
     * 9.819,9.876,9.942,10.012,10.064,10.104,10.146,10.203,
     * 9.737,9.811,9.891,9.973,10.027,10.063,10.089,10.125,
     * 9.669,9.757,9.850,9.944,10.003,10.042,10.071,10.107,
     * 9.486,9.608,9.732,9.857,9.931,9.982,10.019,10.059,
     * 8.837,9.057,9.255,9.426,9.567,9.682,9.771,9.839,
     * 8.615,8.806,8.977,9.127,9.255,9.362,9.448,9.515,
     * 8.283,8.564,8.837,9.097,9.217,9.303,9.381,9.459,
     * 7.779,8.169,8.546,8.907,9.025,9.099,9.159,9.218/
C
      DATA ((EDEN(I,1,1,L,1),I=33,40),L=1,NZEN)/
     * 10.182,10.302,10.426,10.554,10.678,10.787,10.874,10.941,
     * 10.116,10.236,10.357,10.488,10.617,10.733,10.829,10.902,
     * 10.006,10.120,10.232,10.354,10.485,10.614,10.726,10.815,
     * 9.849,9.948,10.042,10.130,10.227,10.341,10.466,10.587,
     * 9.761,9.840,9.905,9.960,10.010,10.053,10.096,10.149,
     * 9.740,9.824,9.893,9.948,9.994,10.025,10.051,10.078,
     * 9.691,9.785,9.859,9.920,9.966,10.001,10.027,10.047,
     * 9.520,9.628,9.714,9.781,9.836,9.879,9.914,9.941,
     * 9.412,9.514,9.593,9.654,9.699,9.730,9.749,9.764,
     * 9.345,9.453,9.536,9.598,9.644,9.674,9.692,9.700,
     * 9.073,9.177,9.256,9.318,9.363,9.395,9.414,9.426/
C
      DATA ((EDEN(I,2,1,L,1),I=33,40),L=1,NZEN)/
     * 10.292,10.418,10.543,10.662,10.767,10.853,10.920,10.969,
     * 10.187,10.306,10.427,10.548,10.663,10.762,10.841,10.902,
     * 10.063,10.175,10.285,10.404,10.527,10.642,10.740,10.816,
     * 9.859,9.952,10.041,10.129,10.219,10.325,10.445,10.564,
     * 9.781,9.857,9.917,9.970,10.012,10.054,10.092,10.137,
     * 9.755,9.834,9.896,9.949,9.987,10.019,10.044,10.071,
     * 9.715,9.807,9.879,9.937,9.980,10.015,10.036,10.055,
     * 9.552,9.653,9.731,9.794,9.842,9.880,9.910,9.933,
     * 9.463,9.556,9.626,9.679,9.716,9.740,9.753,9.765,
     * 9.418,9.516,9.591,9.647,9.686,9.710,9.724,9.727,
     * 9.127,9.219,9.289,9.344,9.382,9.406,9.421,9.428/
C
      DATA ((EDEN(I,3,1,L,1),I=33,40),L=1,NZEN)/
     * 10.327,10.448,10.569,10.683,10.782,10.863,10.925,10.972,
     * 10.222,10.334,10.453,10.570,10.679,10.773,10.848,10.905,
     * 10.091,10.194,10.299,10.412,10.524,10.632,10.724,10.800,
     * 9.918,10.009,10.093,10.181,10.277,10.383,10.493,10.597,
     * 9.806,9.870,9.923,9.973,10.020,10.064,10.111,10.165,
     * 9.785,9.853,9.908,9.956,9.992,10.021,10.049,10.071,
     * 9.742,9.819,9.880,9.929,9.968,9.998,10.022,10.037,
     * 9.579,9.671,9.744,9.801,9.847,9.882,9.910,9.932,
     * 9.488,9.571,9.633,9.680,9.712,9.733,9.742,9.749,
     * 9.431,9.518,9.583,9.631,9.662,9.681,9.688,9.688,
     * 9.158,9.236,9.298,9.344,9.374,9.392,9.402,9.404/
C
      DATA ((EDEN(I,4,1,L,1),I=33,40),L=1,NZEN)/
     * 10.314,10.425,10.539,10.651,10.752,10.839,10.907,10.958,
     * 10.215,10.318,10.425,10.537,10.647,10.743,10.824,10.887,
     * 10.094,10.188,10.280,10.382,10.490,10.598,10.694,10.774,
     * 9.936,10.018,10.094,10.173,10.252,10.344,10.447,10.547,
     * 9.844,9.900,9.949,9.995,10.038,10.083,10.126,10.178,
     * 9.822,9.878,9.925,9.966,10.000,10.029,10.058,10.085,
     * 9.782,9.846,9.898,9.942,9.975,10.002,10.024,10.036,
     * 9.614,9.694,9.759,9.812,9.854,9.888,9.915,9.937,
     * 9.487,9.559,9.614,9.655,9.682,9.699,9.706,9.713,
     * 9.436,9.511,9.567,9.608,9.635,9.650,9.655,9.653,
     * 9.181,9.249,9.300,9.338,9.363,9.376,9.381,9.383/
C
      DATA ((EDEN(I,5,1,L,1),I=33,40),L=1,NZEN)/
     * 10.353,10.450,10.552,10.656,10.751,10.834,10.901,10.951,
     * 10.258,10.346,10.443,10.543,10.644,10.737,10.816,10.877,
     * 10.135,10.218,10.300,10.393,10.490,10.591,10.683,10.761,
     * 9.982,10.053,10.123,10.191,10.262,10.344,10.436,10.531,
     * 9.895,9.942,9.984,10.025,10.061,10.098,10.135,10.171,
     * 9.877,9.924,9.966,10.003,10.032,10.058,10.079,10.098,
     * 9.839,9.892,9.939,9.978,10.008,10.032,10.050,10.059,
     * 9.679,9.749,9.806,9.852,9.889,9.918,9.941,9.959,
     * 9.537,9.599,9.647,9.683,9.707,9.719,9.724,9.729,
     * 9.477,9.540,9.589,9.626,9.648,9.659,9.661,9.657,
     * 9.227,9.285,9.327,9.360,9.380,9.388,9.392,9.388/
C
      DATA ((EDEN(I,1,2,L,1),I=33,40),L=1,NZEN)/
     * 10.179,10.300,10.422,10.547,10.667,10.775,10.863,10.930,
     * 10.113,10.231,10.351,10.479,10.607,10.722,10.817,10.890,
     * 10.002,10.117,10.227,10.346,10.473,10.601,10.712,10.801,
     * 9.847,9.944,10.038,10.127,10.220,10.330,10.450,10.569,
     * 9.760,9.839,9.904,9.960,10.006,10.049,10.092,10.143,
     * 9.738,9.822,9.891,9.948,9.994,10.025,10.051,10.078,
     * 9.690,9.783,9.858,9.918,9.966,10.001,10.027,10.051,
     * 9.519,9.626,9.712,9.780,9.835,9.878,9.914,9.942,
     * 9.409,9.512,9.590,9.651,9.696,9.728,9.749,9.765,
     * 9.342,9.451,9.533,9.597,9.642,9.673,9.691,9.701,
     * 9.070,9.172,9.254,9.316,9.362,9.394,9.414,9.426/
C
      DATA ((EDEN(I,2,2,L,1),I=33,40),L=1,NZEN)/
     * 10.259,10.383,10.505,10.625,10.732,10.823,10.894,10.949,
     * 10.159,10.273,10.389,10.509,10.625,10.728,10.812,10.878,
     * 10.030,10.142,10.250,10.366,10.488,10.609,10.715,10.800,
     * 9.854,9.948,10.037,10.122,10.210,10.315,10.432,10.549,
     * 9.775,9.851,9.913,9.965,10.011,10.050,10.089,10.134,
     * 9.749,9.828,9.892,9.944,9.985,10.018,10.043,10.067,
     * 9.709,9.801,9.873,9.931,9.975,10.010,10.035,10.051,
     * 9.543,9.645,9.724,9.788,9.837,9.876,9.907,9.932,
     * 9.465,9.561,9.634,9.690,9.731,9.758,9.774,9.785,
     * 9.403,9.503,9.579,9.637,9.677,9.703,9.718,9.722,
     * 9.115,9.212,9.284,9.340,9.378,9.405,9.421,9.429/
C
      DATA ((EDEN(I,3,2,L,1),I=33,40),L=1,NZEN)/
     * 10.282,10.400,10.519,10.634,10.738,10.826,10.895,10.948,
     * 10.181,10.291,10.402,10.519,10.631,10.731,10.812,10.877,
     * 10.054,10.155,10.256,10.362,10.474,10.582,10.681,10.764,
     * 9.888,9.978,10.063,10.147,10.236,10.338,10.450,10.558,
     * 9.796,9.863,9.919,9.969,10.015,10.056,10.104,10.152,
     * 9.774,9.846,9.903,9.951,9.988,10.021,10.045,10.067,
     * 9.730,9.811,9.875,9.925,9.965,9.997,10.017,10.036,
     * 9.569,9.664,9.738,9.796,9.843,9.879,9.907,9.930,
     * 9.477,9.562,9.627,9.676,9.710,9.731,9.742,9.751,
     * 9.427,9.518,9.586,9.637,9.671,9.692,9.700,9.703,
     * 9.146,9.229,9.293,9.342,9.374,9.395,9.405,9.409/
C
      DATA ((EDEN(I,4,2,L,1),I=33,40),L=1,NZEN)/
     * 10.277,10.383,10.493,10.605,10.710,10.803,10.877,10.935,
     * 10.180,10.280,10.382,10.490,10.601,10.702,10.789,10.859,
     * 10.062,10.156,10.244,10.340,10.443,10.551,10.652,10.740,
     * 9.915,9.995,10.070,10.146,10.219,10.307,10.401,10.503,
     * 9.833,9.892,9.944,9.990,10.032,10.074,10.112,10.158,
     * 9.810,9.871,9.923,9.964,9.999,10.028,10.054,10.077,
     * 9.768,9.837,9.894,9.939,9.975,10.003,10.027,10.039,
     * 9.602,9.686,9.754,9.808,9.852,9.886,9.913,9.935,
     * 9.487,9.563,9.621,9.664,9.694,9.712,9.721,9.728,
     * 9.433,9.512,9.573,9.616,9.645,9.661,9.666,9.666,
     * 9.169,9.243,9.299,9.342,9.368,9.384,9.391,9.391/
C
      DATA ((EDEN(I,5,2,L,1),I=33,40),L=1,NZEN)/
     * 10.311,10.405,10.505,10.609,10.710,10.799,10.872,10.931,
     * 10.220,10.305,10.396,10.497,10.599,10.698,10.783,10.852,
     * 10.099,10.182,10.263,10.350,10.445,10.548,10.644,10.729,
     * 9.958,10.028,10.098,10.163,10.233,10.309,10.396,10.491,
     * 9.882,9.933,9.979,10.020,10.055,10.089,10.123,10.158,
     * 9.861,9.915,9.962,10.000,10.029,10.057,10.078,10.094,
     * 9.822,9.881,9.932,9.972,10.004,10.029,10.049,10.057,
     * 9.661,9.735,9.796,9.845,9.883,9.913,9.937,9.956,
     * 9.529,9.597,9.649,9.687,9.713,9.727,9.733,9.737,
     * 9.472,9.542,9.596,9.635,9.659,9.671,9.675,9.671,
     * 9.233,9.296,9.345,9.382,9.404,9.414,9.417,9.413/
C
      DATA ((EDEN(I,1,3,L,1),I=33,40),L=1,NZEN)/
     * 10.184,10.304,10.425,10.550,10.667,10.770,10.854,10.919,
     * 10.116,10.234,10.355,10.482,10.607,10.718,10.810,10.880,
     * 10.002,10.117,10.227,10.346,10.475,10.597,10.705,10.791,
     * 9.846,9.944,10.035,10.124,10.217,10.324,10.444,10.558,
     * 9.757,9.836,9.901,9.957,10.006,10.049,10.089,10.137,
     * 9.735,9.819,9.888,9.944,9.990,10.021,10.051,10.075,
     * 9.687,9.780,9.854,9.915,9.963,9.997,10.023,10.047,
     * 9.514,9.622,9.708,9.776,9.831,9.876,9.912,9.940,
     * 9.401,9.503,9.582,9.643,9.689,9.722,9.743,9.760,
     * 9.335,9.442,9.524,9.587,9.634,9.665,9.686,9.695,
     * 9.061,9.165,9.244,9.308,9.353,9.385,9.407,9.420/
C
      DATA ((EDEN(I,2,3,L,1),I=33,40),L=1,NZEN)/
     * 10.219,10.340,10.460,10.581,10.693,10.788,10.866,10.927,
     * 10.138,10.256,10.375,10.499,10.619,10.725,10.812,10.879,
     * 10.014,10.126,10.232,10.348,10.472,10.592,10.698,10.785,
     * 9.847,9.942,10.032,10.118,10.207,10.312,10.428,10.542,
     * 9.764,9.840,9.903,9.957,10.002,10.045,10.085,10.130,
     * 9.743,9.825,9.892,9.946,9.990,10.021,10.050,10.070,
     * 9.696,9.787,9.861,9.918,9.963,9.997,10.027,10.047,
     * 9.525,9.628,9.710,9.774,9.826,9.867,9.899,9.925,
     * 9.434,9.532,9.607,9.665,9.708,9.737,9.755,9.769,
     * 9.371,9.473,9.551,9.611,9.653,9.681,9.699,9.705,
     * 9.088,9.186,9.264,9.321,9.364,9.392,9.410,9.420/
C
      DATA ((EDEN(I,3,3,L,1),I=33,40),L=1,NZEN)/
     * 10.233,10.347,10.462,10.580,10.690,10.785,10.862,10.923,
     * 10.133,10.240,10.349,10.464,10.578,10.683,10.773,10.845,
     * 10.017,10.119,10.219,10.324,10.437,10.552,10.658,10.747,
     * 9.861,9.950,10.034,10.116,10.200,10.298,10.409,10.522,
     * 9.781,9.853,9.912,9.963,10.005,10.047,10.088,10.129,
     * 9.755,9.830,9.892,9.941,9.978,10.011,10.039,10.062,
     * 9.713,9.799,9.868,9.920,9.962,9.995,10.019,10.038,
     * 9.547,9.645,9.721,9.783,9.832,9.869,9.899,9.924,
     * 9.453,9.545,9.614,9.667,9.704,9.729,9.743,9.754,
     * 9.391,9.488,9.560,9.615,9.652,9.675,9.688,9.691,
     * 9.111,9.202,9.270,9.324,9.360,9.385,9.398,9.403/
C
      DATA ((EDEN(I,4,3,L,1),I=33,40),L=1,NZEN)/
     * 10.244,10.349,10.458,10.571,10.681,10.776,10.857,10.918,
     * 10.151,10.249,10.349,10.456,10.567,10.673,10.764,10.840,
     * 10.031,10.124,10.215,10.308,10.409,10.518,10.622,10.714,
     * 9.892,9.972,10.051,10.124,10.198,10.282,10.380,10.483,
     * 9.814,9.879,9.933,9.980,10.021,10.061,10.103,10.144,
     * 9.788,9.857,9.913,9.957,9.992,10.021,10.051,10.071,
     * 9.745,9.822,9.884,9.931,9.969,10.000,10.024,10.037,
     * 9.578,9.669,9.741,9.799,9.845,9.881,9.909,9.931,
     * 9.468,9.551,9.615,9.662,9.695,9.714,9.723,9.731,
     * 9.410,9.497,9.565,9.614,9.645,9.665,9.673,9.673,
     * 9.139,9.222,9.285,9.335,9.365,9.384,9.392,9.392/
C
      DATA ((EDEN(I,5,3,L,1),I=33,40),L=1,NZEN)/
     * 10.283,10.378,10.479,10.586,10.690,10.784,10.861,10.921,
     * 10.190,10.278,10.371,10.473,10.577,10.679,10.769,10.842,
     * 10.071,10.156,10.239,10.326,10.422,10.526,10.627,10.716,
     * 9.934,10.007,10.079,10.144,10.211,10.288,10.377,10.474,
     * 9.859,9.917,9.966,10.008,10.043,10.078,10.114,10.149,
     * 9.835,9.895,9.946,9.985,10.017,10.045,10.067,10.083,
     * 9.791,9.859,9.916,9.959,9.991,10.020,10.041,10.050,
     * 9.627,9.711,9.778,9.831,9.873,9.904,9.929,9.948,
     * 9.510,9.586,9.646,9.689,9.718,9.734,9.740,9.744,
     * 9.451,9.532,9.594,9.639,9.667,9.682,9.686,9.681,
     * 9.196,9.272,9.333,9.377,9.403,9.417,9.418,9.415/
C
      DATA ((EDEN(I,1,4,L,1),I=33,40),L=1,NZEN)/
     * 10.202,10.327,10.452,10.576,10.690,10.788,10.866,10.926,
     * 10.135,10.255,10.379,10.509,10.630,10.736,10.822,10.886,
     * 10.018,10.132,10.247,10.371,10.498,10.617,10.720,10.800,
     * 9.850,9.950,10.042,10.133,10.233,10.345,10.463,10.574,
     * 9.752,9.830,9.894,9.950,10.002,10.046,10.085,10.137,
     * 9.730,9.813,9.880,9.936,9.981,10.013,10.043,10.067,
     * 9.682,9.773,9.847,9.906,9.955,9.988,10.019,10.040,
     * 9.507,9.614,9.699,9.768,9.823,9.868,9.904,9.934,
     * 9.388,9.489,9.568,9.628,9.673,9.706,9.727,9.745,
     * 9.320,9.428,9.510,9.572,9.618,9.649,9.670,9.681,
     * 9.047,9.151,9.231,9.292,9.338,9.370,9.391,9.404/
C
      DATA ((EDEN(I,2,4,L,1),I=33,40),L=1,NZEN)/
     * 10.185,10.305,10.427,10.551,10.667,10.768,10.850,10.914,
     * 10.117,10.234,10.356,10.481,10.604,10.715,10.804,10.873,
     * 10.005,10.119,10.231,10.352,10.478,10.600,10.705,10.789,
     * 9.845,9.945,10.039,10.127,10.225,10.339,10.455,10.568,
     * 9.752,9.831,9.896,9.949,9.997,10.037,10.077,10.116,
     * 9.724,9.807,9.873,9.926,9.969,10.004,10.034,10.058,
     * 9.677,9.768,9.841,9.897,9.943,9.979,10.009,10.030,
     * 9.506,9.612,9.696,9.763,9.818,9.861,9.896,9.924,
     * 9.386,9.486,9.564,9.623,9.667,9.698,9.718,9.734,
     * 9.321,9.426,9.507,9.568,9.612,9.643,9.661,9.670,
     * 9.049,9.149,9.230,9.290,9.336,9.365,9.385,9.398/
C
      DATA ((EDEN(I,3,4,L,1),I=33,40),L=1,NZEN)/
     * 10.201,10.316,10.435,10.555,10.669,10.768,10.849,10.914,
     * 10.103,10.210,10.322,10.438,10.554,10.665,10.759,10.834,
     * 9.999,10.108,10.215,10.328,10.449,10.569,10.678,10.767,
     * 9.849,9.945,10.036,10.122,10.213,10.319,10.434,10.547,
     * 9.756,9.832,9.895,9.946,9.991,10.037,10.083,10.134,
     * 9.730,9.810,9.875,9.924,9.964,10.000,10.029,10.052,
     * 9.685,9.773,9.844,9.897,9.939,9.975,10.005,10.020,
     * 9.515,9.618,9.699,9.763,9.816,9.857,9.889,9.915,
     * 9.396,9.493,9.568,9.624,9.664,9.691,9.707,9.719,
     * 9.333,9.434,9.512,9.570,9.610,9.636,9.650,9.655,
     * 9.056,9.156,9.232,9.291,9.329,9.355,9.370,9.377/
C
      DATA ((EDEN(I,4,4,L,1),I=33,40),L=1,NZEN)/
     * 10.257,10.369,10.484,10.599,10.707,10.799,10.873,10.929,
     * 10.156,10.261,10.367,10.481,10.595,10.698,10.784,10.853,
     * 10.024,10.125,10.222,10.323,10.437,10.549,10.653,10.742,
     * 9.875,9.964,10.049,10.129,10.214,10.311,10.421,10.530,
     * 9.786,9.857,9.915,9.964,10.007,10.050,10.094,10.139,
     * 9.757,9.833,9.894,9.940,9.976,10.009,10.040,10.061,
     * 9.712,9.796,9.863,9.913,9.952,9.984,10.012,10.026,
     * 9.539,9.638,9.716,9.778,9.828,9.866,9.896,9.920,
     * 9.420,9.513,9.583,9.635,9.671,9.693,9.704,9.713,
     * 9.356,9.454,9.529,9.583,9.619,9.640,9.651,9.651,
     * 9.072,9.170,9.244,9.299,9.334,9.356,9.365,9.365/
C
      DATA ((EDEN(I,5,4,L,1),I=33,40),L=1,NZEN)/
     * 10.318,10.426,10.540,10.651,10.751,10.835,10.901,10.949,
     * 10.217,10.315,10.422,10.533,10.642,10.738,10.817,10.877,
     * 10.084,10.181,10.273,10.373,10.481,10.588,10.683,10.761,
     * 9.919,10.001,10.080,10.155,10.232,10.325,10.425,10.524,
     * 9.825,9.890,9.945,9.989,10.029,10.067,10.107,10.149,
     * 9.796,9.865,9.921,9.963,9.996,10.027,10.051,10.072,
     * 9.750,9.829,9.892,9.938,9.973,10.002,10.024,10.038,
     * 9.577,9.671,9.746,9.805,9.851,9.885,9.912,9.932,
     * 9.456,9.545,9.612,9.661,9.693,9.711,9.718,9.722,
     * 9.394,9.488,9.559,9.610,9.642,9.659,9.663,9.658,
     * 9.104,9.199,9.271,9.321,9.354,9.369,9.373,9.371/
C
      DATA ((EDEN(I,1,5,L,1),I=33,40),L=1,NZEN)/
     * 10.234,10.363,10.493,10.618,10.729,10.821,10.894,10.946,
     * 10.163,10.289,10.420,10.550,10.670,10.771,10.850,10.908,
     * 10.044,10.162,10.282,10.412,10.540,10.658,10.754,10.826,
     * 9.860,9.964,10.061,10.158,10.267,10.386,10.508,10.614,
     * 9.746,9.822,9.887,9.945,9.998,10.046,10.092,10.153,
     * 9.723,9.805,9.871,9.926,9.972,10.005,10.036,10.060,
     * 9.677,9.766,9.838,9.898,9.944,9.980,10.007,10.032,
     * 9.499,9.606,9.691,9.759,9.814,9.859,9.896,9.925,
     * 9.373,9.474,9.552,9.611,9.656,9.687,9.708,9.725,
     * 9.307,9.413,9.495,9.556,9.601,9.631,9.651,9.661,
     * 9.035,9.137,9.217,9.276,9.321,9.351,9.372,9.384/
C
      DATA ((EDEN(I,2,5,L,1),I=33,40),L=1,NZEN)/
     * 10.174,10.299,10.424,10.551,10.671,10.776,10.861,10.924,
     * 10.149,10.279,10.410,10.544,10.669,10.775,10.856,10.915,
     * 10.039,10.158,10.277,10.406,10.535,10.653,10.750,10.823,
     * 9.860,9.963,10.057,10.152,10.255,10.370,10.485,10.589,
     * 9.737,9.814,9.880,9.935,9.987,10.036,10.083,10.132,
     * 9.708,9.788,9.854,9.906,9.949,9.986,10.021,10.050,
     * 9.661,9.750,9.822,9.878,9.923,9.959,9.988,10.013,
     * 9.483,9.590,9.674,9.743,9.799,9.844,9.880,9.910,
     * 9.338,9.440,9.519,9.580,9.625,9.656,9.677,9.695,
     * 9.270,9.379,9.461,9.524,9.569,9.599,9.619,9.629,
     * 9.017,9.124,9.204,9.267,9.311,9.341,9.361,9.370/
C
      DATA ((EDEN(I,3,5,L,1),I=33,40),L=1,NZEN)/
     * 10.207,10.329,10.451,10.577,10.693,10.792,10.870,10.931,
     * 10.122,10.241,10.364,10.492,10.617,10.726,10.815,10.882,
     * 10.038,10.155,10.273,10.399,10.528,10.646,10.742,10.817,
     * 9.864,9.968,10.066,10.162,10.269,10.384,10.498,10.600,
     * 9.735,9.812,9.878,9.933,9.983,10.035,10.089,10.149,
     * 9.704,9.784,9.850,9.898,9.939,9.979,10.015,10.043,
     * 9.660,9.750,9.821,9.875,9.917,9.955,9.990,10.006,
     * 9.480,9.586,9.670,9.737,9.792,9.835,9.870,9.899,
     * 9.334,9.436,9.514,9.572,9.614,9.642,9.659,9.675,
     * 9.273,9.382,9.464,9.525,9.568,9.597,9.614,9.622,
     * 9.009,9.117,9.198,9.260,9.302,9.328,9.342,9.348/
C
      DATA ((EDEN(I,4,5,L,1),I=33,40),L=1,NZEN)/
     * 10.325,10.451,10.574,10.689,10.786,10.863,10.920,10.962,
     * 10.214,10.333,10.454,10.573,10.682,10.771,10.842,10.894,
     * 10.059,10.169,10.283,10.405,10.528,10.640,10.733,10.806,
     * 9.878,9.981,10.078,10.170,10.275,10.389,10.501,10.600,
     * 9.756,9.832,9.896,9.949,9.998,10.051,10.106,10.165,
     * 9.725,9.804,9.868,9.915,9.953,9.990,10.025,10.051,
     * 9.678,9.766,9.836,9.887,9.926,9.961,9.990,10.006,
     * 9.493,9.600,9.684,9.750,9.803,9.844,9.876,9.903,
     * 9.352,9.453,9.529,9.585,9.623,9.648,9.660,9.671,
     * 9.284,9.391,9.471,9.531,9.569,9.592,9.604,9.606,
     * 8.991,9.100,9.182,9.240,9.279,9.301,9.311,9.312/
C
      DATA ((EDEN(I,5,5,L,1),I=33,40),L=1,NZEN)/
     * 10.438,10.565,10.684,10.785,10.865,10.924,10.964,10.991,
     * 10.320,10.443,10.565,10.677,10.771,10.843,10.896,10.930,
     * 10.166,10.280,10.395,10.515,10.623,10.714,10.783,10.835,
     * 9.918,10.018,10.111,10.207,10.308,10.420,10.526,10.618,
     * 9.785,9.858,9.920,9.971,10.018,10.072,10.126,10.187,
     * 9.751,9.828,9.889,9.934,9.969,10.005,10.039,10.065,
     * 9.703,9.789,9.857,9.905,9.941,9.972,9.998,10.013,
     * 9.516,9.622,9.705,9.770,9.821,9.860,9.890,9.913,
     * 9.369,9.469,9.544,9.598,9.632,9.652,9.659,9.665,
     * 9.296,9.405,9.484,9.540,9.575,9.593,9.599,9.596,
     * 8.994,9.104,9.184,9.241,9.277,9.297,9.303,9.301/
C
      DATA ((EDEN(I,1,6,L,1),I=33,40),L=1,NZEN)/
     * 10.266,10.398,10.530,10.655,10.762,10.850,10.916,10.964,
     * 10.193,10.325,10.458,10.590,10.706,10.803,10.876,10.928,
     * 10.069,10.193,10.318,10.452,10.582,10.695,10.785,10.851,
     * 9.874,9.983,10.083,10.187,10.303,10.431,10.551,10.652,
     * 9.740,9.817,9.882,9.941,9.998,10.049,10.106,10.176,
     * 9.718,9.798,9.864,9.918,9.963,9.997,10.028,10.056,
     * 9.671,9.760,9.831,9.890,9.936,9.969,9.995,10.021,
     * 9.493,9.599,9.684,9.752,9.807,9.851,9.888,9.917,
     * 9.362,9.462,9.539,9.598,9.642,9.672,9.691,9.708,
     * 9.297,9.402,9.483,9.543,9.586,9.616,9.634,9.643,
     * 9.026,9.127,9.204,9.265,9.308,9.337,9.356,9.368/
C
      DATA ((EDEN(I,2,6,L,1),I=33,40),L=1,NZEN)/
     * 10.182,10.307,10.436,10.567,10.690,10.795,10.878,10.941,
     * 10.172,10.305,10.441,10.579,10.702,10.804,10.881,10.936,
     * 10.070,10.194,10.320,10.457,10.585,10.698,10.787,10.853,
     * 9.885,9.995,10.096,10.200,10.314,10.434,10.547,10.642,
     * 9.727,9.805,9.873,9.933,9.988,10.043,10.100,10.162,
     * 9.695,9.775,9.840,9.892,9.937,9.977,10.017,10.049,
     * 9.654,9.742,9.815,9.871,9.917,9.955,9.987,10.013,
     * 9.469,9.576,9.661,9.730,9.786,9.832,9.870,9.901,
     * 9.302,9.405,9.485,9.546,9.592,9.624,9.644,9.665,
     * 9.247,9.357,9.442,9.507,9.554,9.587,9.609,9.621,
     * 8.991,9.100,9.183,9.247,9.292,9.323,9.340,9.349/
C
      DATA ((EDEN(I,3,6,L,1),I=33,40),L=1,NZEN)/
     * 10.235,10.361,10.489,10.616,10.731,10.824,10.897,10.951,
     * 10.184,10.315,10.453,10.587,10.707,10.805,10.879,10.931,
     * 10.077,10.203,10.332,10.467,10.595,10.704,10.788,10.852,
     * 9.894,10.008,10.111,10.221,10.341,10.462,10.570,10.659,
     * 9.720,9.800,9.872,9.932,9.990,10.054,10.119,10.195,
     * 9.689,9.769,9.835,9.885,9.928,9.971,10.014,10.042,
     * 9.642,9.730,9.802,9.854,9.897,9.937,9.971,9.992,
     * 9.456,9.564,9.649,9.719,9.775,9.820,9.857,9.887,
     * 9.291,9.397,9.478,9.539,9.584,9.614,9.633,9.650,
     * 9.222,9.335,9.419,9.483,9.526,9.556,9.573,9.582,
     * 8.971,9.083,9.166,9.229,9.271,9.293,9.307,9.309/
C
      DATA ((EDEN(I,4,6,L,1),I=33,40),L=1,NZEN)/
     * 10.415,10.549,10.670,10.773,10.854,10.915,10.956,10.985,
     * 10.276,10.409,10.538,10.658,10.759,10.836,10.892,10.932,
     * 10.112,10.240,10.370,10.501,10.622,10.722,10.797,10.853,
     * 9.904,10.017,10.123,10.235,10.354,10.473,10.579,10.663,
     * 9.737,9.817,9.887,9.948,10.005,10.065,10.131,10.207,
     * 9.700,9.782,9.847,9.897,9.938,9.980,10.023,10.057,
     * 9.653,9.743,9.814,9.866,9.906,9.943,9.975,9.995,
     * 9.460,9.573,9.660,9.729,9.786,9.829,9.865,9.893,
     * 9.291,9.398,9.479,9.537,9.578,9.603,9.616,9.629,
     * 9.220,9.336,9.420,9.481,9.522,9.546,9.558,9.560,
     * 8.925,9.040,9.127,9.190,9.232,9.255,9.266,9.268/
C
      DATA ((EDEN(I,5,6,L,1),I=33,40),L=1,NZEN)/
     * 10.576,10.707,10.811,10.889,10.943,10.980,10.998,11.009,
     * 10.450,10.587,10.703,10.797,10.866,10.912,10.941,10.959,
     * 10.211,10.343,10.475,10.598,10.701,10.780,10.834,10.871,
     * 9.944,10.061,10.172,10.285,10.406,10.519,10.614,10.686,
     * 9.757,9.837,9.910,9.974,10.035,10.100,10.176,10.253,
     * 9.719,9.799,9.863,9.913,9.954,9.998,10.040,10.081,
     * 9.669,9.758,9.827,9.877,9.914,9.948,9.980,9.997,
     * 9.469,9.585,9.674,9.744,9.799,9.842,9.875,9.902,
     * 9.281,9.389,9.470,9.527,9.564,9.586,9.595,9.604,
     * 9.206,9.322,9.407,9.467,9.504,9.525,9.533,9.530,
     * 8.903,9.021,9.109,9.172,9.211,9.230,9.238,9.238/
C
      DATA ((EDEN(I,1,7,L,1),I=33,40),L=1,NZEN)/
     * 10.277,10.410,10.543,10.666,10.773,10.858,10.924,10.970,
     * 10.203,10.336,10.470,10.601,10.716,10.810,10.882,10.932,
     * 10.079,10.203,10.331,10.464,10.593,10.705,10.794,10.858,
     * 9.881,9.992,10.093,10.198,10.316,10.444,10.562,10.663,
     * 9.739,9.816,9.882,9.942,9.998,10.053,10.113,10.188,
     * 9.717,9.797,9.863,9.916,9.961,9.997,10.024,10.056,
     * 9.670,9.758,9.830,9.887,9.933,9.967,9.995,10.017,
     * 9.492,9.598,9.682,9.750,9.804,9.849,9.885,9.915,
     * 9.359,9.459,9.536,9.594,9.638,9.668,9.687,9.703,
     * 9.293,9.399,9.479,9.539,9.583,9.612,9.630,9.639,
     * 9.022,9.122,9.202,9.262,9.303,9.333,9.353,9.363/
C
      DATA ((EDEN(I,2,7,L,1),I=33,40),L=1,NZEN)/
     * 10.191,10.316,10.445,10.577,10.699,10.804,10.887,10.947,
     * 10.182,10.315,10.451,10.588,10.711,10.811,10.887,10.940,
     * 10.083,10.207,10.334,10.469,10.598,10.708,10.795,10.859,
     * 9.895,10.007,10.109,10.214,10.328,10.449,10.560,10.652,
     * 9.727,9.805,9.874,9.935,9.990,10.047,10.107,10.174,
     * 9.695,9.774,9.839,9.891,9.936,9.978,10.016,10.053,
     * 9.654,9.741,9.812,9.868,9.915,9.952,9.986,10.008,
     * 9.468,9.574,9.659,9.728,9.784,9.830,9.868,9.899,
     * 9.292,9.396,9.477,9.538,9.584,9.616,9.637,9.658,
     * 9.236,9.347,9.432,9.499,9.545,9.578,9.600,9.613,
     * 8.981,9.092,9.176,9.240,9.284,9.316,9.333,9.343/
C
      DATA ((EDEN(I,3,7,L,1),I=33,40),L=1,NZEN)/
     * 10.250,10.376,10.505,10.630,10.743,10.835,10.904,10.957,
     * 10.200,10.332,10.467,10.601,10.719,10.814,10.885,10.936,
     * 10.094,10.219,10.348,10.481,10.608,10.714,10.796,10.857,
     * 9.909,10.020,10.127,10.239,10.357,10.477,10.582,10.669,
     * 9.722,9.802,9.874,9.937,9.996,10.061,10.128,10.205,
     * 9.689,9.768,9.833,9.884,9.927,9.972,10.013,10.045,
     * 9.641,9.729,9.800,9.852,9.894,9.934,9.968,9.991,
     * 9.453,9.562,9.647,9.716,9.773,9.819,9.856,9.886,
     * 9.276,9.384,9.466,9.528,9.573,9.604,9.623,9.641,
     * 9.206,9.321,9.407,9.471,9.516,9.545,9.564,9.573,
     * 8.957,9.072,9.158,9.222,9.261,9.286,9.299,9.301/
C
      DATA ((EDEN(I,4,7,L,1),I=33,40),L=1,NZEN)/
     * 10.443,10.574,10.692,10.790,10.868,10.922,10.960,10.986,
     * 10.301,10.434,10.562,10.677,10.773,10.844,10.898,10.935,
     * 10.132,10.260,10.389,10.519,10.638,10.733,10.805,10.856,
     * 9.919,10.034,10.139,10.252,10.372,10.488,10.589,10.671,
     * 9.737,9.818,9.890,9.952,10.011,10.072,10.141,10.215,
     * 9.699,9.780,9.845,9.895,9.938,9.981,10.023,10.060,
     * 9.652,9.741,9.811,9.863,9.903,9.941,9.974,9.994,
     * 9.455,9.568,9.656,9.727,9.784,9.828,9.863,9.893,
     * 9.274,9.383,9.465,9.524,9.566,9.592,9.607,9.620,
     * 9.203,9.319,9.406,9.468,9.510,9.535,9.548,9.551,
     * 8.906,9.023,9.113,9.176,9.220,9.242,9.256,9.258/
C
      DATA ((EDEN(I,5,7,L,1),I=33,40),L=1,NZEN)/
     * 10.611,10.736,10.833,10.905,10.953,10.984,10.999,11.008,
     * 10.485,10.617,10.729,10.816,10.877,10.918,10.943,10.958,
     * 10.240,10.373,10.503,10.622,10.718,10.790,10.839,10.873,
     * 9.962,10.080,10.190,10.306,10.426,10.533,10.623,10.691,
     * 9.757,9.839,9.913,9.978,10.039,10.106,10.184,10.263,
     * 9.716,9.797,9.862,9.912,9.955,10.001,10.044,10.084,
     * 9.667,9.755,9.825,9.874,9.911,9.946,9.979,9.997,
     * 9.467,9.582,9.672,9.742,9.798,9.841,9.875,9.903,
     * 9.259,9.370,9.453,9.511,9.550,9.573,9.583,9.594,
     * 9.183,9.301,9.389,9.451,9.489,9.510,9.519,9.518,
     * 8.876,9.000,9.090,9.153,9.194,9.216,9.227,9.226/
C
      DATA ((EDEN(I,1,8,L,1),I=33,40),L=1,NZEN)/
     * 10.264,10.395,10.525,10.648,10.755,10.844,10.911,10.959,
     * 10.193,10.321,10.452,10.583,10.698,10.795,10.868,10.922,
     * 10.069,10.190,10.314,10.444,10.572,10.686,10.777,10.844,
     * 9.875,9.983,10.083,10.184,10.297,10.420,10.538,10.641,
     * 9.743,9.819,9.885,9.944,9.998,10.049,10.102,10.170,
     * 9.721,9.801,9.868,9.921,9.966,10.001,10.032,10.060,
     * 9.674,9.762,9.834,9.892,9.939,9.974,9.999,10.024,
     * 9.496,9.602,9.687,9.755,9.810,9.854,9.892,9.922,
     * 9.364,9.465,9.542,9.602,9.646,9.677,9.696,9.714,
     * 9.298,9.404,9.485,9.546,9.590,9.620,9.639,9.649,
     * 9.026,9.130,9.207,9.267,9.312,9.341,9.362,9.373/
C
      DATA ((EDEN(I,2,8,L,1),I=33,40),L=1,NZEN)/
     * 10.196,10.320,10.447,10.576,10.696,10.799,10.880,10.941,
     * 10.172,10.302,10.434,10.571,10.693,10.795,10.873,10.929,
     * 10.067,10.186,10.307,10.439,10.567,10.680,10.771,10.839,
     * 9.882,9.986,10.086,10.184,10.292,10.408,10.519,10.618,
     * 9.736,9.813,9.879,9.937,9.991,10.043,10.093,10.150,
     * 9.705,9.784,9.849,9.901,9.945,9.984,10.021,10.053,
     * 9.659,9.745,9.816,9.872,9.916,9.953,9.983,10.005,
     * 9.476,9.583,9.668,9.736,9.793,9.838,9.876,9.907,
     * 9.310,9.414,9.495,9.556,9.602,9.634,9.656,9.677,
     * 9.254,9.364,9.449,9.515,9.562,9.595,9.617,9.629,
     * 8.993,9.101,9.184,9.248,9.293,9.323,9.341,9.353/
C
      DATA ((EDEN(I,3,8,L,1),I=33,40),L=1,NZEN)/
     * 10.242,10.365,10.489,10.613,10.725,10.818,10.892,10.945,
     * 10.182,10.307,10.435,10.566,10.687,10.787,10.864,10.920,
     * 10.070,10.189,10.311,10.438,10.564,10.675,10.765,10.833,
     * 9.891,10.000,10.097,10.196,10.307,10.423,10.532,10.627,
     * 9.736,9.812,9.879,9.938,9.992,10.050,10.106,10.169,
     * 9.702,9.780,9.844,9.894,9.936,9.978,10.018,10.050,
     * 9.658,9.745,9.814,9.868,9.910,9.949,9.982,10.000,
     * 9.469,9.576,9.661,9.729,9.785,9.830,9.866,9.896,
     * 9.302,9.408,9.489,9.550,9.595,9.627,9.646,9.663,
     * 9.233,9.345,9.429,9.494,9.539,9.569,9.587,9.596,
     * 8.982,9.094,9.177,9.240,9.283,9.308,9.321,9.324/
C
      DATA ((EDEN(I,4,8,L,1),I=33,40),L=1,NZEN)/
     * 10.392,10.517,10.634,10.738,10.824,10.888,10.937,10.970,
     * 10.259,10.381,10.503,10.620,10.722,10.805,10.867,10.914,
     * 10.106,10.218,10.335,10.456,10.573,10.676,10.759,10.822,
     * 9.906,10.011,10.109,10.207,10.314,10.427,10.534,10.624,
     * 9.755,9.830,9.896,9.953,10.006,10.062,10.122,10.185,
     * 9.722,9.798,9.861,9.910,9.950,9.990,10.028,10.057,
     * 9.672,9.759,9.828,9.880,9.919,9.955,9.985,10.004,
     * 9.478,9.588,9.674,9.742,9.798,9.841,9.875,9.903,
     * 9.309,9.416,9.496,9.555,9.596,9.622,9.636,9.649,
     * 9.236,9.349,9.435,9.497,9.538,9.564,9.577,9.581,
     * 8.942,9.054,9.143,9.205,9.246,9.271,9.286,9.287/
C
      DATA ((EDEN(I,5,8,L,1),I=33,40),L=1,NZEN)/
     * 10.523,10.644,10.750,10.835,10.899,10.944,10.972,10.991,
     * 10.399,10.523,10.637,10.734,10.811,10.868,10.908,10.934,
     * 10.212,10.325,10.442,10.557,10.655,10.736,10.796,10.840,
     * 9.944,10.049,10.146,10.243,10.348,10.454,10.553,10.635,
     * 9.783,9.855,9.920,9.977,10.029,10.081,10.141,10.206,
     * 9.745,9.821,9.883,9.929,9.968,10.006,10.041,10.071,
     * 9.696,9.781,9.848,9.897,9.934,9.968,9.995,10.010,
     * 9.495,9.606,9.692,9.760,9.814,9.855,9.887,9.912,
     * 9.312,9.419,9.499,9.557,9.596,9.619,9.629,9.639,
     * 9.236,9.351,9.436,9.497,9.537,9.560,9.569,9.569,
     * 8.930,9.046,9.135,9.197,9.239,9.261,9.270,9.270/
C
      DATA ((EDEN(I,1,9,L,1),I=33,40),L=1,NZEN)/
     * 10.236,10.363,10.491,10.613,10.723,10.815,10.888,10.941,
     * 10.163,10.289,10.417,10.545,10.665,10.765,10.845,10.903,
     * 10.044,10.162,10.282,10.408,10.535,10.652,10.747,10.820,
     * 9.863,9.967,10.061,10.158,10.262,10.379,10.498,10.604,
     * 9.750,9.827,9.891,9.948,10.002,10.046,10.092,10.149,
     * 9.727,9.809,9.876,9.931,9.976,10.009,10.040,10.067,
     * 9.680,9.770,9.843,9.901,9.950,9.984,10.011,10.036,
     * 9.503,9.610,9.695,9.763,9.819,9.864,9.901,9.931,
     * 9.376,9.477,9.555,9.615,9.660,9.693,9.714,9.732,
     * 9.310,9.417,9.497,9.560,9.605,9.636,9.657,9.668,
     * 9.038,9.139,9.219,9.281,9.324,9.356,9.377,9.391/
C
      DATA ((EDEN(I,2,9,L,1),I=33,40),L=1,NZEN)/
     * 10.206,10.328,10.454,10.580,10.695,10.794,10.872,10.933,
     * 10.158,10.283,10.410,10.541,10.664,10.768,10.851,10.910,
     * 10.035,10.149,10.265,10.387,10.514,10.631,10.730,10.807,
     * 9.861,9.959,10.050,10.139,10.233,10.339,10.450,10.555,
     * 9.753,9.828,9.892,9.947,9.996,10.040,10.083,10.125,
     * 9.724,9.802,9.867,9.920,9.963,9.999,10.029,10.057,
     * 9.676,9.764,9.834,9.890,9.936,9.973,10.000,10.026,
     * 9.495,9.601,9.685,9.753,9.809,9.854,9.890,9.919,
     * 9.352,9.454,9.533,9.594,9.639,9.671,9.692,9.711,
     * 9.285,9.393,9.476,9.538,9.584,9.614,9.634,9.644,
     * 9.020,9.125,9.204,9.265,9.311,9.341,9.358,9.368/
C
      DATA ((EDEN(I,3,9,L,1),I=33,40),L=1,NZEN)/
     * 10.233,10.351,10.472,10.592,10.704,10.799,10.875,10.932,
     * 10.141,10.254,10.372,10.496,10.616,10.724,10.811,10.878,
     * 10.022,10.133,10.241,10.358,10.481,10.601,10.704,10.788,
     * 9.870,9.967,10.058,10.144,10.239,10.344,10.454,10.559,
     * 9.760,9.833,9.895,9.947,9.995,10.043,10.089,10.133,
     * 9.730,9.806,9.869,9.918,9.958,9.994,10.028,10.055,
     * 9.683,9.768,9.837,9.889,9.932,9.968,9.998,10.014,
     * 9.498,9.603,9.686,9.753,9.807,9.849,9.884,9.911,
     * 9.352,9.453,9.531,9.589,9.632,9.661,9.679,9.694,
     * 9.286,9.395,9.477,9.538,9.583,9.611,9.629,9.637,
     * 9.007,9.112,9.195,9.254,9.299,9.325,9.342,9.350/
C
      DATA ((EDEN(I,4,9,L,1),I=33,40),L=1,NZEN)/
     * 10.314,10.429,10.544,10.655,10.753,10.834,10.897,10.945,
     * 10.206,10.313,10.426,10.539,10.646,10.738,10.814,10.873,
     * 10.059,10.163,10.266,10.375,10.489,10.602,10.698,10.776,
     * 9.893,9.984,10.071,10.154,10.245,10.347,10.456,10.559,
     * 9.785,9.854,9.914,9.964,10.010,10.056,10.103,10.150,
     * 9.755,9.827,9.888,9.934,9.973,10.007,10.038,10.063,
     * 9.708,9.790,9.856,9.907,9.946,9.980,10.009,10.023,
     * 9.517,9.620,9.702,9.767,9.819,9.859,9.890,9.915,
     * 9.369,9.469,9.546,9.602,9.642,9.668,9.681,9.693,
     * 9.300,9.408,9.488,9.547,9.587,9.611,9.623,9.627,
     * 9.008,9.116,9.197,9.254,9.295,9.319,9.330,9.332/
C
      DATA ((EDEN(I,5,9,L,1),I=33,40),L=1,NZEN)/
     * 10.390,10.500,10.609,10.710,10.797,10.866,10.920,10.959,
     * 10.280,10.384,10.490,10.596,10.693,10.775,10.841,10.890,
     * 10.139,10.234,10.331,10.434,10.537,10.633,10.715,10.781,
     * 9.940,10.025,10.105,10.183,10.266,10.360,10.459,10.554,
     * 9.822,9.886,9.940,9.988,10.030,10.072,10.116,10.160,
     * 9.790,9.858,9.915,9.958,9.993,10.024,10.052,10.073,
     * 9.741,9.819,9.882,9.930,9.966,9.996,10.020,10.034,
     * 9.549,9.649,9.729,9.792,9.841,9.878,9.907,9.929,
     * 9.400,9.498,9.574,9.628,9.665,9.686,9.695,9.702,
     * 9.329,9.434,9.514,9.571,9.608,9.629,9.637,9.636,
     * 9.026,9.132,9.213,9.275,9.311,9.332,9.340,9.340/
C
      DATA ((EDEN(I,1,10,L,1),I=33,40),L=1,NZEN)/
     * 10.210,10.335,10.461,10.584,10.697,10.795,10.873,10.932,
     * 10.141,10.262,10.388,10.516,10.638,10.743,10.828,10.892,
     * 10.022,10.139,10.254,10.378,10.506,10.626,10.726,10.806,
     * 9.854,9.955,10.046,10.139,10.238,10.351,10.470,10.580,
     * 9.757,9.834,9.899,9.954,10.006,10.049,10.092,10.140,
     * 9.734,9.817,9.884,9.939,9.985,10.017,10.047,10.075,
     * 9.686,9.777,9.851,9.911,9.959,9.993,10.023,10.044,
     * 9.511,9.618,9.703,9.772,9.828,9.872,9.909,9.939,
     * 9.389,9.491,9.570,9.630,9.675,9.708,9.730,9.748,
     * 9.323,9.430,9.512,9.574,9.620,9.652,9.672,9.684,
     * 9.050,9.154,9.233,9.295,9.340,9.372,9.394,9.407/
C
      DATA ((EDEN(I,2,10,L,1),I=33,40),L=1,NZEN)/
     * 10.231,10.353,10.478,10.603,10.714,10.807,10.881,10.938,
     * 10.147,10.268,10.391,10.517,10.637,10.743,10.826,10.892,
     * 10.024,10.135,10.247,10.367,10.492,10.612,10.715,10.798,
     * 9.856,9.952,10.043,10.131,10.225,10.333,10.451,10.562,
     * 9.764,9.839,9.901,9.955,10.001,10.045,10.087,10.136,
     * 9.742,9.821,9.887,9.940,9.983,10.017,10.046,10.069,
     * 9.694,9.783,9.854,9.912,9.957,9.992,10.022,10.042,
     * 9.512,9.617,9.700,9.766,9.819,9.862,9.896,9.924,
     * 9.401,9.500,9.577,9.636,9.680,9.710,9.730,9.746,
     * 9.335,9.440,9.521,9.581,9.624,9.655,9.673,9.682,
     * 9.053,9.155,9.234,9.292,9.335,9.366,9.385,9.396/
C
      DATA ((EDEN(I,3,10,L,1),I=33,40),L=1,NZEN)/
     * 10.247,10.365,10.485,10.605,10.714,10.807,10.881,10.937,
     * 10.145,10.257,10.370,10.488,10.604,10.707,10.795,10.864,
     * 10.024,10.128,10.232,10.345,10.465,10.584,10.690,10.776,
     * 9.870,9.959,10.045,10.129,10.216,10.317,10.431,10.542,
     * 9.784,9.854,9.912,9.962,10.007,10.049,10.091,10.135,
     * 9.759,9.833,9.894,9.943,9.983,10.017,10.042,10.064,
     * 9.712,9.795,9.862,9.915,9.957,9.993,10.017,10.033,
     * 9.529,9.630,9.710,9.773,9.824,9.864,9.896,9.921,
     * 9.411,9.507,9.580,9.636,9.676,9.703,9.719,9.731,
     * 9.348,9.449,9.526,9.582,9.622,9.647,9.662,9.667,
     * 9.061,9.157,9.232,9.287,9.325,9.349,9.363,9.370/
C
      DATA ((EDEN(I,4,10,L,1),I=33,40),L=1,NZEN)/
     * 10.275,10.385,10.497,10.610,10.715,10.805,10.877,10.932,
     * 10.175,10.276,10.383,10.494,10.605,10.705,10.790,10.858,
     * 10.050,10.146,10.239,10.337,10.445,10.553,10.653,10.738,
     * 9.902,9.983,10.061,10.140,10.219,10.312,10.418,10.524,
     * 9.814,9.878,9.932,9.980,10.023,10.062,10.105,10.149,
     * 9.787,9.855,9.910,9.955,9.991,10.023,10.049,10.069,
     * 9.741,9.818,9.880,9.929,9.967,9.998,10.021,10.038,
     * 9.558,9.654,9.730,9.790,9.839,9.876,9.905,9.929,
     * 9.427,9.518,9.588,9.640,9.676,9.698,9.710,9.719,
     * 9.364,9.461,9.535,9.588,9.623,9.645,9.654,9.656,
     * 9.081,9.174,9.249,9.300,9.335,9.357,9.369,9.371/
C
      DATA ((EDEN(I,5,10,L,1),I=33,40),L=1,NZEN)/
     * 10.318,10.416,10.520,10.625,10.724,10.811,10.880,10.935,
     * 10.220,10.311,10.407,10.509,10.613,10.710,10.792,10.858,
     * 10.093,10.181,10.265,10.354,10.455,10.559,10.654,10.737,
     * 9.947,10.019,10.089,10.156,10.230,10.311,10.403,10.499,
     * 9.857,9.913,9.962,10.006,10.044,10.080,10.115,10.153,
     * 9.830,9.890,9.940,9.981,10.014,10.041,10.064,10.084,
     * 9.784,9.853,9.910,9.955,9.990,10.017,10.037,10.047,
     * 9.601,9.691,9.763,9.820,9.864,9.898,9.924,9.944,
     * 9.463,9.550,9.617,9.666,9.699,9.718,9.726,9.732,
     * 9.400,9.494,9.564,9.616,9.648,9.667,9.672,9.669,
     * 9.124,9.214,9.285,9.336,9.370,9.387,9.392,9.390/
C
      DATA ((EDEN(I,1,11,L,1),I=33,40),L=1,NZEN)/
     * 10.192,10.315,10.442,10.569,10.686,10.789,10.871,10.934,
     * 10.126,10.246,10.370,10.501,10.626,10.737,10.828,10.896,
     * 10.010,10.126,10.240,10.363,10.493,10.617,10.724,10.808,
     * 9.851,9.950,10.042,10.133,10.230,10.345,10.464,10.580,
     * 9.761,9.838,9.903,9.958,10.006,10.049,10.092,10.143,
     * 9.738,9.822,9.890,9.946,9.990,10.025,10.051,10.078,
     * 9.690,9.782,9.857,9.916,9.964,10.001,10.027,10.047,
     * 9.517,9.624,9.710,9.778,9.833,9.877,9.913,9.941,
     * 9.401,9.503,9.582,9.642,9.688,9.720,9.741,9.758,
     * 9.335,9.442,9.524,9.587,9.633,9.664,9.683,9.693,
     * 9.061,9.165,9.244,9.308,9.353,9.384,9.405,9.417/
C
      DATA ((EDEN(I,2,11,L,1),I=33,40),L=1,NZEN)/
     * 10.266,10.393,10.520,10.640,10.747,10.836,10.904,10.956,
     * 10.162,10.279,10.401,10.525,10.641,10.742,10.823,10.887,
     * 10.033,10.145,10.254,10.375,10.501,10.621,10.724,10.806,
     * 9.858,9.951,10.040,10.125,10.218,10.325,10.441,10.558,
     * 9.777,9.851,9.913,9.965,10.011,10.049,10.088,10.133,
     * 9.755,9.836,9.902,9.956,9.999,10.030,10.054,10.077,
     * 9.708,9.798,9.870,9.927,9.972,10.006,10.031,10.051,
     * 9.534,9.637,9.718,9.782,9.833,9.873,9.906,9.932,
     * 9.444,9.541,9.615,9.672,9.713,9.742,9.760,9.773,
     * 9.381,9.482,9.560,9.618,9.660,9.687,9.702,9.710,
     * 9.091,9.189,9.264,9.319,9.361,9.387,9.405,9.413/
C
      DATA ((EDEN(I,3,11,L,1),I=33,40),L=1,NZEN)/
     * 10.289,10.410,10.533,10.650,10.754,10.840,10.908,10.958,
     * 10.184,10.297,10.414,10.534,10.647,10.746,10.826,10.887,
     * 10.057,10.161,10.263,10.373,10.490,10.600,10.697,10.778,
     * 9.891,9.979,10.062,10.149,10.241,10.345,10.461,10.570,
     * 9.803,9.869,9.924,9.973,10.018,10.059,10.096,10.139,
     * 9.777,9.847,9.903,9.952,9.989,10.019,10.047,10.069,
     * 9.734,9.814,9.878,9.932,9.972,10.004,10.028,10.043,
     * 9.558,9.655,9.731,9.791,9.839,9.877,9.906,9.930,
     * 9.454,9.543,9.610,9.660,9.696,9.719,9.730,9.741,
     * 9.400,9.495,9.566,9.617,9.654,9.676,9.687,9.690,
     * 9.115,9.203,9.270,9.320,9.354,9.375,9.387,9.392/
C
      DATA ((EDEN(I,4,11,L,1),I=33,40),L=1,NZEN)/
     * 10.287,10.396,10.511,10.625,10.730,10.820,10.892,10.946,
     * 10.188,10.292,10.397,10.509,10.620,10.721,10.806,10.873,
     * 10.069,10.161,10.253,10.354,10.461,10.572,10.671,10.756,
     * 9.921,10.000,10.076,10.148,10.229,10.320,10.419,10.521,
     * 9.835,9.893,9.943,9.990,10.031,10.073,10.114,10.164,
     * 9.811,9.872,9.922,9.965,9.999,10.031,10.056,10.075,
     * 9.768,9.837,9.893,9.939,9.975,10.003,10.025,10.038,
     * 9.591,9.679,9.749,9.805,9.850,9.885,9.914,9.936,
     * 9.464,9.544,9.606,9.652,9.684,9.703,9.712,9.720,
     * 9.407,9.492,9.556,9.603,9.634,9.651,9.658,9.657,
     * 9.137,9.217,9.278,9.324,9.352,9.371,9.377,9.378/
C
      DATA ((EDEN(I,5,11,L,1),I=33,40),L=1,NZEN)/
     * 10.321,10.416,10.519,10.625,10.725,10.813,10.885,10.940,
     * 10.225,10.314,10.408,10.510,10.615,10.713,10.797,10.864,
     * 10.105,10.187,10.270,10.360,10.460,10.562,10.659,10.743,
     * 9.964,10.032,10.100,10.165,10.235,10.317,10.408,10.507,
     * 9.885,9.934,9.979,10.020,10.057,10.091,10.122,10.156,
     * 9.861,9.912,9.957,9.996,10.026,10.050,10.072,10.092,
     * 9.818,9.877,9.926,9.967,10.000,10.025,10.042,10.055,
     * 9.645,9.723,9.786,9.837,9.878,9.909,9.934,9.953,
     * 9.502,9.576,9.634,9.676,9.704,9.720,9.726,9.731,
     * 9.442,9.519,9.579,9.623,9.650,9.665,9.669,9.665,
     * 9.198,9.273,9.331,9.375,9.399,9.414,9.417,9.413/
C
      DATA ((EDEN(I,1,12,L,1),I=33,40),L=1,NZEN)/
     * 10.187,10.309,10.434,10.562,10.685,10.791,10.878,10.942,
     * 10.120,10.241,10.365,10.495,10.623,10.738,10.833,10.903,
     * 10.006,10.123,10.237,10.358,10.491,10.618,10.729,10.817,
     * 9.850,9.949,10.042,10.133,10.230,10.345,10.470,10.589,
     * 9.762,9.840,9.904,9.961,10.010,10.053,10.096,10.149,
     * 9.740,9.824,9.892,9.948,9.994,10.025,10.051,10.078,
     * 9.691,9.784,9.859,9.919,9.967,10.001,10.027,10.047,
     * 9.520,9.628,9.713,9.781,9.836,9.879,9.914,9.942,
     * 9.408,9.511,9.590,9.650,9.695,9.726,9.746,9.762,
     * 9.342,9.450,9.533,9.596,9.640,9.671,9.688,9.699,
     * 9.067,9.174,9.254,9.316,9.361,9.391,9.410,9.422/
C
      DATA ((EDEN(I,2,12,L,1),I=33,40),L=1,NZEN)/
     * 10.294,10.422,10.547,10.668,10.771,10.856,10.923,10.970,
     * 10.187,10.306,10.429,10.553,10.668,10.766,10.844,10.905,
     * 10.063,10.174,10.285,10.408,10.530,10.645,10.742,10.820,
     * 9.861,9.953,10.041,10.129,10.221,10.328,10.448,10.566,
     * 9.783,9.857,9.917,9.969,10.016,10.054,10.092,10.137,
     * 9.756,9.834,9.896,9.948,9.987,10.019,10.044,10.071,
     * 9.716,9.806,9.878,9.936,9.980,10.015,10.035,10.055,
     * 9.548,9.650,9.729,9.792,9.841,9.880,9.910,9.934,
     * 9.456,9.550,9.620,9.674,9.712,9.736,9.750,9.762,
     * 9.409,9.508,9.584,9.641,9.680,9.705,9.718,9.723,
     * 9.117,9.212,9.282,9.337,9.375,9.400,9.415,9.422/
C
      DATA ((EDEN(I,3,12,L,1),I=33,40),L=1,NZEN)/
     * 10.325,10.450,10.572,10.687,10.786,10.867,10.928,10.974,
     * 10.219,10.336,10.454,10.575,10.684,10.778,10.852,10.909,
     * 10.090,10.194,10.301,10.414,10.528,10.637,10.730,10.803,
     * 9.906,9.996,10.082,10.169,10.264,10.374,10.488,10.595,
     * 9.808,9.871,9.924,9.974,10.019,10.064,10.111,10.165,
     * 9.786,9.854,9.908,9.956,9.993,10.025,10.049,10.071,
     * 9.742,9.819,9.880,9.930,9.968,9.997,10.021,10.036,
     * 9.575,9.668,9.741,9.799,9.845,9.882,9.910,9.933,
     * 9.480,9.564,9.628,9.675,9.708,9.728,9.739,9.746,
     * 9.423,9.511,9.577,9.625,9.657,9.676,9.684,9.685,
     * 9.145,9.228,9.290,9.335,9.367,9.386,9.395,9.399/
C
      DATA ((EDEN(I,4,12,L,1),I=33,40),L=1,NZEN)/
     * 10.314,10.427,10.543,10.655,10.758,10.844,10.910,10.961,
     * 10.215,10.320,10.428,10.542,10.651,10.749,10.829,10.891,
     * 10.093,10.187,10.282,10.385,10.495,10.602,10.700,10.779,
     * 9.938,10.017,10.093,10.172,10.255,10.351,10.452,10.553,
     * 9.845,9.901,9.949,9.995,10.037,10.082,10.126,10.180,
     * 9.823,9.878,9.925,9.967,10.001,10.029,10.058,10.085,
     * 9.782,9.846,9.897,9.942,9.976,10.003,10.023,10.040,
     * 9.610,9.692,9.757,9.811,9.853,9.887,9.915,9.936,
     * 9.480,9.554,9.610,9.651,9.680,9.696,9.704,9.711,
     * 9.426,9.504,9.562,9.604,9.632,9.647,9.653,9.652,
     * 9.170,9.238,9.292,9.333,9.358,9.372,9.378,9.377/
C
      DATA ((EDEN(I,5,12,L,1),I=33,40),L=1,NZEN)/
     * 10.353,10.450,10.554,10.658,10.755,10.837,10.903,10.953,
     * 10.255,10.346,10.442,10.546,10.648,10.742,10.819,10.881,
     * 10.134,10.217,10.300,10.392,10.492,10.595,10.688,10.766,
     * 9.983,10.052,10.122,10.190,10.261,10.346,10.439,10.536,
     * 9.896,9.942,9.984,10.025,10.064,10.097,10.134,10.171,
     * 9.877,9.925,9.965,10.002,10.032,10.057,10.078,10.098,
     * 9.838,9.892,9.938,9.977,10.009,10.032,10.049,10.062,
     * 9.674,9.743,9.801,9.848,9.885,9.915,9.939,9.958,
     * 9.517,9.581,9.631,9.668,9.692,9.706,9.711,9.717,
     * 9.467,9.533,9.585,9.622,9.645,9.657,9.659,9.656,
     * 9.216,9.274,9.321,9.356,9.377,9.386,9.389,9.385/
C
      DATA ((EDEN(I,1,1,L,2),I=33,40),L=1,NZEN)/
     * 10.376,10.491,10.600,10.710,10.812,10.906,10.978,11.034,
     * 10.307,10.423,10.534,10.646,10.757,10.855,10.934,10.995,
     * 10.189,10.304,10.412,10.524,10.637,10.746,10.841,10.915,
     * 10.005,10.106,10.202,10.299,10.396,10.502,10.610,10.711,
     * 9.893,9.961,10.018,10.074,10.124,10.175,10.230,10.295,
     * 9.866,9.937,9.996,10.049,10.090,10.125,10.157,10.190,
     * 9.809,9.888,9.953,10.008,10.055,10.091,10.122,10.151,
     * 9.584,9.683,9.764,9.830,9.888,9.938,9.982,10.020,
     * 9.432,9.521,9.589,9.642,9.684,9.716,9.738,9.760,
     * 9.364,9.456,9.528,9.582,9.624,9.653,9.675,9.687,
     * 9.088,9.175,9.242,9.295,9.336,9.367,9.390,9.405/
C
      DATA ((EDEN(I,2,1,L,2),I=33,40),L=1,NZEN)/
     * 10.481,10.593,10.700,10.800,10.888,10.959,11.016,11.056,
     * 10.379,10.489,10.594,10.699,10.794,10.877,10.944,10.994,
     * 10.253,10.361,10.464,10.568,10.673,10.770,10.850,10.916,
     * 10.015,10.111,10.201,10.293,10.385,10.487,10.593,10.694,
     * 9.910,9.973,10.023,10.074,10.120,10.168,10.219,10.277,
     * 9.879,9.945,9.999,10.043,10.084,10.118,10.151,10.186,
     * 9.824,9.901,9.965,10.014,10.059,10.094,10.125,10.150,
     * 9.604,9.697,9.771,9.834,9.887,9.933,9.974,10.010,
     * 9.482,9.561,9.622,9.668,9.703,9.727,9.743,9.761,
     * 9.436,9.520,9.583,9.631,9.666,9.690,9.708,9.714,
     * 9.142,9.217,9.275,9.321,9.354,9.381,9.398,9.409/
C
      DATA ((EDEN(I,3,1,L,2),I=33,40),L=1,NZEN)/
     * 10.514,10.622,10.723,10.819,10.901,10.969,11.021,11.060,
     * 10.414,10.519,10.620,10.719,10.810,10.889,10.951,10.997,
     * 10.281,10.382,10.478,10.576,10.673,10.763,10.840,10.901,
     * 10.085,10.178,10.267,10.352,10.443,10.540,10.634,10.720,
     * 9.941,9.993,10.042,10.090,10.140,10.193,10.251,10.317,
     * 9.912,9.968,10.015,10.057,10.092,10.124,10.158,10.192,
     * 9.860,9.924,9.975,10.022,10.058,10.092,10.118,10.142,
     * 9.640,9.726,9.795,9.852,9.903,9.946,9.984,10.018,
     * 9.509,9.580,9.633,9.674,9.704,9.724,9.736,9.750,
     * 9.451,9.525,9.578,9.619,9.648,9.666,9.677,9.681,
     * 9.176,9.240,9.289,9.325,9.353,9.373,9.385,9.392/
C
      DATA ((EDEN(I,4,1,L,2),I=33,40),L=1,NZEN)/
     * 10.506,10.606,10.702,10.796,10.881,10.954,11.011,11.054,
     * 10.410,10.505,10.600,10.695,10.787,10.869,10.935,10.987,
     * 10.280,10.374,10.463,10.553,10.647,10.738,10.819,10.885,
     * 10.101,10.182,10.260,10.339,10.419,10.505,10.595,10.681,
     * 9.983,10.026,10.069,10.110,10.157,10.205,10.257,10.317,
     * 9.957,9.999,10.036,10.067,10.096,10.127,10.159,10.194,
     * 9.912,9.960,10.000,10.037,10.063,10.088,10.109,10.126,
     * 9.706,9.777,9.835,9.883,9.925,9.960,9.990,10.014,
     * 9.514,9.575,9.621,9.656,9.682,9.698,9.707,9.719,
     * 9.461,9.524,9.571,9.607,9.631,9.645,9.653,9.654,
     * 9.206,9.260,9.298,9.332,9.352,9.366,9.375,9.376/
C
      DATA ((EDEN(I,5,1,L,2),I=33,40),L=1,NZEN)/
     * 10.545,10.631,10.719,10.805,10.884,10.955,11.009,11.053,
     * 10.452,10.533,10.618,10.705,10.790,10.867,10.934,10.984,
     * 10.324,10.405,10.483,10.565,10.651,10.737,10.814,10.880,
     * 10.148,10.217,10.285,10.353,10.426,10.505,10.588,10.671,
     * 10.034,10.066,10.099,10.132,10.166,10.203,10.244,10.290,
     * 10.013,10.044,10.073,10.095,10.116,10.137,10.156,10.178,
     * 9.970,10.007,10.039,10.068,10.086,10.103,10.113,10.120,
     * 9.780,9.836,9.883,9.922,9.954,9.979,9.999,10.014,
     * 9.567,9.620,9.662,9.693,9.715,9.727,9.732,9.739,
     * 9.508,9.561,9.603,9.634,9.654,9.664,9.668,9.665,
     * 9.258,9.302,9.337,9.366,9.383,9.390,9.392,9.391/
C
      DATA ((EDEN(I,1,2,L,2),I=33,40),L=1,NZEN)/
     * 10.372,10.485,10.594,10.703,10.804,10.894,10.967,11.022,
     * 10.305,10.418,10.528,10.639,10.747,10.844,10.923,10.984,
     * 10.187,10.302,10.408,10.516,10.627,10.735,10.828,10.902,
     * 10.005,10.103,10.200,10.292,10.388,10.489,10.597,10.696,
     * 9.892,9.960,10.018,10.071,10.124,10.172,10.225,10.287,
     * 9.864,9.936,9.996,10.045,10.090,10.125,10.157,10.190,
     * 9.807,9.887,9.953,10.008,10.055,10.091,10.122,10.151,
     * 9.583,9.682,9.763,9.830,9.887,9.938,9.982,10.020,
     * 9.429,9.518,9.586,9.640,9.682,9.715,9.738,9.761,
     * 9.361,9.454,9.525,9.579,9.622,9.653,9.675,9.688,
     * 9.085,9.173,9.239,9.292,9.333,9.366,9.390,9.405/
C
      DATA ((EDEN(I,2,2,L,2),I=33,40),L=1,NZEN)/
     * 10.451,10.562,10.667,10.768,10.857,10.935,10.994,11.040,
     * 10.351,10.458,10.562,10.665,10.762,10.848,10.919,10.974,
     * 10.217,10.326,10.429,10.534,10.641,10.743,10.831,10.902,
     * 10.010,10.104,10.195,10.286,10.377,10.477,10.580,10.681,
     * 9.904,9.967,10.022,10.070,10.117,10.165,10.214,10.272,
     * 9.872,9.939,9.994,10.042,10.080,10.115,10.147,10.184,
     * 9.818,9.896,9.959,10.013,10.055,10.091,10.121,10.147,
     * 9.598,9.692,9.767,9.831,9.885,9.932,9.973,10.010,
     * 9.485,9.567,9.629,9.678,9.716,9.744,9.762,9.781,
     * 9.421,9.506,9.571,9.620,9.657,9.684,9.701,9.710,
     * 9.133,9.209,9.268,9.315,9.351,9.379,9.398,9.410/
C
      DATA ((EDEN(I,3,2,L,2),I=33,40),L=1,NZEN)/
     * 10.473,10.579,10.680,10.778,10.865,10.937,10.995,11.039,
     * 10.374,10.477,10.576,10.675,10.769,10.852,10.920,10.974,
     * 10.243,10.341,10.436,10.532,10.628,10.721,10.803,10.870,
     * 10.048,10.141,10.229,10.314,10.402,10.499,10.595,10.689,
     * 9.929,9.985,10.032,10.082,10.130,10.182,10.238,10.299,
     * 9.899,9.958,10.005,10.048,10.084,10.120,10.155,10.186,
     * 9.848,9.913,9.967,10.016,10.053,10.087,10.117,10.139,
     * 9.630,9.718,9.788,9.847,9.899,9.943,9.981,10.015,
     * 9.497,9.571,9.626,9.669,9.700,9.722,9.735,9.751,
     * 9.447,9.524,9.581,9.625,9.657,9.678,9.691,9.695,
     * 9.164,9.233,9.284,9.323,9.354,9.376,9.388,9.396/
C
      DATA ((EDEN(I,4,2,L,2),I=33,40),L=1,NZEN)/
     * 10.470,10.568,10.663,10.758,10.846,10.923,10.986,11.033,
     * 10.373,10.468,10.560,10.655,10.748,10.834,10.907,10.965,
     * 10.242,10.336,10.424,10.513,10.606,10.699,10.784,10.855,
     * 10.065,10.146,10.225,10.303,10.382,10.466,10.555,10.643,
     * 9.957,10.005,10.051,10.090,10.136,10.184,10.231,10.288,
     * 9.929,9.978,10.021,10.053,10.087,10.116,10.145,10.179,
     * 9.881,9.937,9.983,10.023,10.054,10.080,10.104,10.121,
     * 9.678,9.755,9.816,9.868,9.912,9.949,9.980,10.006,
     * 9.511,9.577,9.627,9.665,9.692,9.710,9.720,9.732,
     * 9.456,9.524,9.575,9.613,9.639,9.656,9.664,9.665,
     * 9.194,9.254,9.298,9.333,9.358,9.373,9.382,9.385/
C
      DATA ((EDEN(I,5,2,L,2),I=33,40),L=1,NZEN)/
     * 10.506,10.592,10.678,10.767,10.851,10.926,10.988,11.034,
     * 10.408,10.494,10.577,10.665,10.753,10.835,10.907,10.964,
     * 10.280,10.363,10.443,10.525,10.611,10.699,10.781,10.852,
     * 10.100,10.174,10.246,10.318,10.391,10.467,10.551,10.636,
     * 9.995,10.035,10.072,10.108,10.144,10.180,10.221,10.265,
     * 9.970,10.011,10.046,10.077,10.100,10.122,10.142,10.165,
     * 9.925,9.972,10.012,10.044,10.072,10.090,10.101,10.112,
     * 9.738,9.803,9.857,9.901,9.936,9.965,9.987,10.004,
     * 9.557,9.616,9.662,9.697,9.720,9.734,9.740,9.747,
     * 9.498,9.560,9.607,9.642,9.664,9.676,9.680,9.678,
     * 9.259,9.312,9.351,9.385,9.405,9.415,9.417,9.416/
C
      DATA ((EDEN(I,1,3,L,2),I=33,40),L=1,NZEN)/
     * 10.376,10.489,10.598,10.703,10.802,10.889,10.960,11.013,
     * 10.307,10.421,10.530,10.640,10.746,10.839,10.916,10.974,
     * 10.189,10.302,10.408,10.516,10.626,10.730,10.820,10.893,
     * 10.005,10.103,10.197,10.290,10.384,10.485,10.589,10.686,
     * 9.890,9.958,10.018,10.071,10.121,10.169,10.220,10.280,
     * 9.862,9.934,9.992,10.045,10.090,10.125,10.155,10.188,
     * 9.806,9.885,9.949,10.008,10.055,10.091,10.122,10.151,
     * 9.582,9.681,9.761,9.829,9.887,9.937,9.982,10.020,
     * 9.422,9.510,9.578,9.632,9.675,9.708,9.732,9.756,
     * 9.354,9.446,9.516,9.571,9.614,9.646,9.668,9.682,
     * 9.077,9.162,9.230,9.285,9.326,9.358,9.383,9.400/
C
      DATA ((EDEN(I,2,3,L,2),I=33,40),L=1,NZEN)/
     * 10.412,10.523,10.628,10.730,10.824,10.905,10.969,11.019,
     * 10.332,10.443,10.550,10.655,10.757,10.846,10.918,10.975,
     * 10.199,10.310,10.412,10.518,10.624,10.726,10.816,10.887,
     * 10.001,10.099,10.192,10.281,10.373,10.472,10.575,10.674,
     * 9.893,9.958,10.013,10.066,10.113,10.162,10.213,10.269,
     * 9.864,9.934,9.992,10.041,10.082,10.120,10.150,10.180,
     * 9.809,9.887,9.950,10.004,10.047,10.086,10.117,10.143,
     * 9.587,9.682,9.759,9.824,9.880,9.928,9.970,10.008,
     * 9.454,9.539,9.603,9.654,9.694,9.724,9.744,9.765,
     * 9.388,9.476,9.543,9.595,9.635,9.662,9.682,9.693,
     * 9.107,9.189,9.249,9.299,9.337,9.367,9.389,9.403/
C
      DATA ((EDEN(I,3,3,L,2),I=33,40),L=1,NZEN)/
     * 10.426,10.531,10.633,10.731,10.824,10.903,10.967,11.017,
     * 10.327,10.430,10.527,10.627,10.724,10.812,10.887,10.947,
     * 10.201,10.302,10.399,10.496,10.596,10.694,10.782,10.858,
     * 10.014,10.103,10.192,10.278,10.367,10.461,10.560,10.657,
     * 9.909,9.970,10.021,10.069,10.115,10.166,10.217,10.271,
     * 9.879,9.942,9.993,10.037,10.075,10.112,10.147,10.179,
     * 9.827,9.899,9.958,10.005,10.047,10.085,10.115,10.137,
     * 9.611,9.701,9.774,9.836,9.889,9.935,9.975,10.010,
     * 9.474,9.553,9.613,9.660,9.695,9.720,9.736,9.754,
     * 9.412,9.494,9.557,9.603,9.638,9.661,9.676,9.682,
     * 9.130,9.205,9.262,9.308,9.341,9.365,9.381,9.390/
C
      DATA ((EDEN(I,4,3,L,2),I=33,40),L=1,NZEN)/
     * 10.438,10.537,10.633,10.729,10.821,10.902,10.967,11.019,
     * 10.339,10.434,10.528,10.624,10.720,10.809,10.884,10.946,
     * 10.204,10.301,10.389,10.481,10.574,10.669,10.757,10.834,
     * 10.029,10.112,10.194,10.273,10.355,10.441,10.532,10.624,
     * 9.918,9.974,10.026,10.068,10.114,10.161,10.211,10.267,
     * 9.887,9.946,9.995,10.037,10.069,10.099,10.131,10.162,
     * 9.838,9.904,9.958,10.000,10.038,10.069,10.091,10.109,
     * 9.639,9.723,9.791,9.848,9.895,9.935,9.968,9.996,
     * 9.490,9.564,9.620,9.662,9.692,9.712,9.723,9.735,
     * 9.432,9.509,9.566,9.610,9.640,9.659,9.668,9.671,
     * 9.160,9.231,9.284,9.325,9.353,9.372,9.382,9.386/
C
      DATA ((EDEN(I,5,3,L,2),I=33,40),L=1,NZEN)/
     * 10.478,10.568,10.657,10.748,10.836,10.914,10.978,11.028,
     * 10.377,10.466,10.554,10.643,10.734,10.821,10.896,10.956,
     * 10.238,10.329,10.415,10.499,10.589,10.681,10.767,10.842,
     * 10.054,10.132,10.211,10.287,10.365,10.446,10.531,10.619,
     * 9.940,9.992,10.038,10.076,10.120,10.159,10.202,10.253,
     * 9.910,9.963,10.009,10.046,10.076,10.100,10.123,10.149,
     * 9.862,9.923,9.973,10.012,10.042,10.067,10.083,10.099,
     * 9.679,9.757,9.820,9.870,9.911,9.943,9.968,9.987,
     * 9.535,9.605,9.658,9.698,9.724,9.740,9.747,9.753,
     * 9.476,9.549,9.605,9.645,9.671,9.686,9.690,9.688,
     * 9.218,9.287,9.338,9.377,9.403,9.415,9.419,9.415/
C
      DATA ((EDEN(I,1,4,L,2),I=33,40),L=1,NZEN)/
     * 10.394,10.509,10.618,10.724,10.821,10.901,10.968,11.017,
     * 10.325,10.440,10.551,10.661,10.764,10.853,10.924,10.979,
     * 10.205,10.320,10.426,10.536,10.645,10.746,10.832,10.900,
     * 10.013,10.112,10.208,10.303,10.400,10.500,10.603,10.697,
     * 9.886,9.954,10.010,10.067,10.121,10.169,10.223,10.282,
     * 9.858,9.929,9.988,10.038,10.083,10.119,10.152,10.185,
     * 9.803,9.881,9.946,10.001,10.051,10.085,10.116,10.148,
     * 9.579,9.678,9.758,9.825,9.883,9.934,9.980,10.018,
     * 9.408,9.496,9.563,9.617,9.660,9.693,9.717,9.742,
     * 9.342,9.433,9.501,9.556,9.599,9.631,9.653,9.669,
     * 9.067,9.151,9.216,9.269,9.311,9.343,9.368,9.385/
C
      DATA ((EDEN(I,2,4,L,2),I=33,40),L=1,NZEN)/
     * 10.379,10.491,10.597,10.703,10.801,10.887,10.955,11.008,
     * 10.308,10.422,10.529,10.639,10.743,10.835,10.910,10.969,
     * 10.191,10.305,10.411,10.519,10.628,10.731,10.820,10.890,
     * 10.003,10.105,10.202,10.296,10.393,10.495,10.599,10.693,
     * 9.883,9.951,10.011,10.061,10.109,10.155,10.205,10.252,
     * 9.851,9.921,9.981,10.028,10.070,10.106,10.143,10.177,
     * 9.797,9.874,9.938,9.990,10.034,10.075,10.107,10.136,
     * 9.578,9.675,9.754,9.821,9.879,9.929,9.973,10.011,
     * 9.406,9.494,9.561,9.613,9.655,9.686,9.709,9.732,
     * 9.339,9.430,9.500,9.553,9.594,9.625,9.646,9.659,
     * 9.067,9.151,9.215,9.268,9.309,9.341,9.365,9.380/
C
      DATA ((EDEN(I,3,4,L,2),I=33,40),L=1,NZEN)/
     * 10.394,10.503,10.606,10.708,10.804,10.888,10.955,11.008,
     * 10.293,10.398,10.499,10.601,10.702,10.795,10.874,10.937,
     * 10.182,10.292,10.393,10.498,10.605,10.708,10.798,10.872,
     * 10.002,10.101,10.193,10.283,10.377,10.478,10.580,10.675,
     * 9.885,9.950,10.005,10.055,10.106,10.161,10.220,10.279,
     * 9.855,9.923,9.979,10.022,10.061,10.104,10.139,10.172,
     * 9.802,9.877,9.939,9.987,10.028,10.069,10.103,10.126,
     * 9.587,9.683,9.759,9.824,9.880,9.929,9.970,10.007,
     * 9.418,9.503,9.568,9.618,9.656,9.683,9.701,9.721,
     * 9.353,9.441,9.508,9.559,9.596,9.623,9.640,9.649,
     * 9.074,9.160,9.223,9.273,9.310,9.336,9.355,9.366/
C
      DATA ((EDEN(I,4,4,L,2),I=33,40),L=1,NZEN)/
     * 10.451,10.553,10.653,10.752,10.842,10.920,10.981,11.027,
     * 10.344,10.447,10.545,10.644,10.741,10.828,10.900,10.957,
     * 10.196,10.301,10.397,10.495,10.596,10.695,10.782,10.856,
     * 10.003,10.101,10.192,10.283,10.373,10.470,10.570,10.663,
     * 9.875,9.940,9.997,10.048,10.096,10.149,10.206,10.265,
     * 9.841,9.909,9.965,10.010,10.046,10.082,10.118,10.147,
     * 9.789,9.866,9.927,9.975,10.012,10.046,10.077,10.096,
     * 9.591,9.685,9.760,9.822,9.874,9.916,9.952,9.982,
     * 9.442,9.525,9.588,9.635,9.669,9.691,9.703,9.716,
     * 9.377,9.464,9.530,9.579,9.613,9.634,9.646,9.650,
     * 9.094,9.179,9.242,9.288,9.323,9.343,9.355,9.360/
C
      DATA ((EDEN(I,5,4,L,2),I=33,40),L=1,NZEN)/
     * 10.515,10.613,10.710,10.804,10.888,10.958,11.012,11.049,
     * 10.409,10.506,10.601,10.697,10.789,10.869,10.934,10.984,
     * 10.260,10.360,10.451,10.547,10.642,10.734,10.814,10.878,
     * 10.032,10.127,10.217,10.304,10.389,10.480,10.574,10.662,
     * 9.882,9.944,9.999,10.050,10.097,10.149,10.202,10.261,
     * 9.845,9.912,9.965,10.009,10.042,10.073,10.105,10.136,
     * 9.795,9.870,9.930,9.977,10.012,10.042,10.059,10.076,
     * 9.614,9.705,9.777,9.834,9.879,9.915,9.944,9.966,
     * 9.481,9.563,9.625,9.670,9.700,9.717,9.724,9.731,
     * 9.418,9.504,9.570,9.616,9.646,9.662,9.668,9.664,
     * 9.127,9.213,9.277,9.324,9.354,9.370,9.374,9.373/
C
      DATA ((EDEN(I,1,5,L,2),I=33,40),L=1,NZEN)/
     * 10.426,10.542,10.654,10.759,10.853,10.930,10.990,11.034,
     * 10.356,10.472,10.586,10.697,10.797,10.882,10.947,10.996,
     * 10.234,10.349,10.460,10.572,10.682,10.779,10.860,10.920,
     * 10.028,10.134,10.233,10.330,10.433,10.539,10.641,10.731,
     * 9.881,9.949,10.006,10.067,10.121,10.177,10.235,10.304,
     * 9.854,9.924,9.980,10.031,10.077,10.113,10.146,10.182,
     * 9.800,9.877,9.941,9.997,10.041,10.078,10.110,10.140,
     * 9.576,9.674,9.754,9.821,9.879,9.931,9.976,10.015,
     * 9.394,9.481,9.548,9.601,9.643,9.675,9.698,9.723,
     * 9.326,9.418,9.487,9.540,9.582,9.613,9.635,9.650,
     * 9.053,9.136,9.201,9.253,9.293,9.325,9.349,9.366/
C
      DATA ((EDEN(I,2,5,L,2),I=33,40),L=1,NZEN)/
     * 10.368,10.483,10.594,10.703,10.805,10.893,10.962,11.016,
     * 10.344,10.461,10.578,10.692,10.797,10.886,10.953,11.004,
     * 10.227,10.344,10.455,10.567,10.677,10.776,10.856,10.918,
     * 10.029,10.132,10.229,10.323,10.423,10.524,10.622,10.709,
     * 9.872,9.941,10.001,10.056,10.111,10.166,10.222,10.281,
     * 9.838,9.908,9.966,10.014,10.055,10.099,10.139,10.176,
     * 9.786,9.862,9.926,9.975,10.021,10.060,10.096,10.123,
     * 9.570,9.667,9.746,9.814,9.872,9.923,9.968,10.007,
     * 9.359,9.449,9.517,9.571,9.613,9.646,9.669,9.694,
     * 9.292,9.384,9.455,9.510,9.552,9.584,9.605,9.620,
     * 9.036,9.125,9.191,9.245,9.286,9.316,9.339,9.354/
C
      DATA ((EDEN(I,3,5,L,2),I=33,40),L=1,NZEN)/
     * 10.401,10.512,10.621,10.727,10.824,10.907,10.972,11.021,
     * 10.316,10.428,10.538,10.649,10.754,10.845,10.919,10.976,
     * 10.226,10.339,10.451,10.562,10.672,10.771,10.850,10.913,
     * 10.027,10.137,10.236,10.332,10.432,10.536,10.633,10.719,
     * 9.867,9.936,9.997,10.052,10.108,10.171,10.234,10.300,
     * 9.830,9.901,9.960,10.006,10.043,10.091,10.137,10.173,
     * 9.782,9.859,9.924,9.970,10.013,10.056,10.094,10.120,
     * 9.564,9.663,9.743,9.810,9.869,9.919,9.962,10.001,
     * 9.356,9.446,9.514,9.567,9.607,9.635,9.655,9.678,
     * 9.293,9.388,9.460,9.516,9.556,9.585,9.605,9.616,
     * 9.028,9.120,9.190,9.243,9.280,9.309,9.327,9.336/
C
      DATA ((EDEN(I,4,5,L,2),I=33,40),L=1,NZEN)/
     * 10.519,10.628,10.733,10.828,10.908,10.972,11.019,11.054,
     * 10.408,10.516,10.623,10.724,10.815,10.889,10.947,10.990,
     * 10.240,10.355,10.461,10.571,10.677,10.771,10.848,10.908,
     * 10.019,10.132,10.234,10.335,10.437,10.542,10.639,10.723,
     * 9.839,9.911,9.976,10.036,10.097,10.161,10.228,10.300,
     * 9.800,9.874,9.935,9.982,10.026,10.068,10.108,10.148,
     * 9.745,9.829,9.895,9.946,9.986,10.023,10.056,10.079,
     * 9.543,9.646,9.727,9.793,9.849,9.896,9.935,9.968,
     * 9.374,9.466,9.535,9.585,9.622,9.646,9.660,9.676,
     * 9.304,9.401,9.473,9.527,9.563,9.587,9.600,9.605,
     * 9.011,9.105,9.177,9.230,9.266,9.289,9.301,9.306/
C
      DATA ((EDEN(I,5,5,L,2),I=33,40),L=1,NZEN)/
     * 10.630,10.737,10.835,10.919,10.984,11.031,11.062,11.082,
     * 10.517,10.625,10.728,10.821,10.898,10.956,10.999,11.027,
     * 10.361,10.468,10.573,10.675,10.764,10.840,10.898,10.938,
     * 10.048,10.164,10.271,10.370,10.473,10.573,10.665,10.742,
     * 9.832,9.906,9.972,10.036,10.100,10.168,10.238,10.314,
     * 9.787,9.863,9.924,9.973,10.016,10.057,10.097,10.142,
     * 9.732,9.817,9.884,9.935,9.974,10.004,10.027,10.049,
     * 9.546,9.649,9.730,9.794,9.844,9.885,9.918,9.943,
     * 9.394,9.487,9.557,9.606,9.639,9.659,9.667,9.675,
     * 9.320,9.422,9.494,9.546,9.579,9.597,9.604,9.602,
     * 9.017,9.118,9.191,9.245,9.277,9.296,9.303,9.302/
C
      DATA ((EDEN(I,1,6,L,2),I=33,40),L=1,NZEN)/
     * 10.455,10.574,10.686,10.790,10.881,10.955,11.011,11.051,
     * 10.384,10.504,10.619,10.730,10.827,10.908,10.969,11.013,
     * 10.261,10.378,10.491,10.606,10.716,10.812,10.887,10.941,
     * 10.046,10.158,10.259,10.361,10.468,10.576,10.677,10.763,
     * 9.878,9.945,10.006,10.067,10.130,10.188,10.254,10.332,
     * 9.850,9.919,9.976,10.027,10.070,10.106,10.143,10.182,
     * 9.797,9.873,9.935,9.989,10.037,10.071,10.100,10.131,
     * 9.574,9.671,9.751,9.817,9.875,9.926,9.972,10.010,
     * 9.384,9.470,9.536,9.588,9.629,9.660,9.682,9.706,
     * 9.317,9.407,9.474,9.528,9.568,9.599,9.619,9.632,
     * 9.041,9.127,9.189,9.241,9.281,9.311,9.334,9.350/
C
      DATA ((EDEN(I,2,6,L,2),I=33,40),L=1,NZEN)/
     * 10.375,10.492,10.605,10.717,10.821,10.910,10.980,11.031,
     * 10.365,10.487,10.605,10.722,10.826,10.911,10.976,11.020,
     * 10.263,10.380,10.495,10.611,10.719,10.813,10.888,10.942,
     * 10.060,10.172,10.271,10.372,10.476,10.579,10.674,10.754,
     * 9.867,9.937,10.004,10.062,10.123,10.184,10.249,10.317,
     * 9.831,9.900,9.958,10.005,10.050,10.098,10.141,10.186,
     * 9.784,9.860,9.923,9.973,10.020,10.059,10.098,10.125,
     * 9.568,9.664,9.743,9.810,9.868,9.920,9.965,10.005,
     * 9.325,9.415,9.484,9.539,9.582,9.614,9.638,9.666,
     * 9.267,9.364,9.436,9.494,9.539,9.571,9.596,9.612,
     * 9.009,9.102,9.170,9.226,9.267,9.298,9.318,9.333/
C
      DATA ((EDEN(I,3,6,L,2),I=33,40),L=1,NZEN)/
     * 10.427,10.543,10.653,10.761,10.856,10.935,10.994,11.039,
     * 10.378,10.498,10.616,10.729,10.830,10.913,10.973,11.017,
     * 10.270,10.389,10.505,10.620,10.727,10.818,10.890,10.941,
     * 10.069,10.186,10.290,10.393,10.499,10.602,10.693,10.768,
     * 9.858,9.932,10.003,10.065,10.129,10.200,10.274,10.349,
     * 9.822,9.892,9.952,9.999,10.041,10.090,10.139,10.180,
     * 9.769,9.846,9.911,9.959,9.999,10.043,10.086,10.113,
     * 9.553,9.654,9.734,9.803,9.863,9.914,9.959,9.998,
     * 9.314,9.408,9.480,9.535,9.578,9.609,9.630,9.655,
     * 9.245,9.343,9.417,9.474,9.516,9.546,9.567,9.578,
     * 8.991,9.087,9.158,9.213,9.250,9.277,9.290,9.297/
C
      DATA ((EDEN(I,4,6,L,2),I=33,40),L=1,NZEN)/
     * 10.598,10.712,10.813,10.899,10.966,11.016,11.051,11.070,
     * 10.469,10.586,10.697,10.796,10.879,10.944,10.990,11.021,
     * 10.305,10.427,10.542,10.656,10.757,10.839,10.902,10.947,
     * 10.060,10.183,10.296,10.403,10.512,10.615,10.704,10.775,
     * 9.821,9.902,9.979,10.053,10.122,10.195,10.268,10.346,
     * 9.775,9.852,9.916,9.968,10.019,10.065,10.116,10.167,
     * 9.720,9.806,9.875,9.927,9.969,10.007,10.046,10.069,
     * 9.514,9.623,9.708,9.778,9.837,9.886,9.928,9.963,
     * 9.313,9.411,9.484,9.538,9.577,9.603,9.618,9.636,
     * 9.243,9.346,9.422,9.478,9.517,9.541,9.555,9.560,
     * 8.944,9.048,9.125,9.183,9.218,9.244,9.258,9.263/
C
      DATA ((EDEN(I,5,6,L,2),I=33,40),L=1,NZEN)/
     * 10.750,10.857,10.942,11.005,11.049,11.078,11.090,11.097,
     * 10.636,10.750,10.846,10.921,10.977,11.014,11.036,11.049,
     * 10.411,10.532,10.645,10.748,10.832,10.895,10.940,10.968,
     * 10.100,10.231,10.347,10.457,10.565,10.662,10.741,10.800,
     * 9.807,9.894,9.979,10.065,10.143,10.224,10.306,10.390,
     * 9.752,9.833,9.899,9.957,10.012,10.066,10.122,10.179,
     * 9.692,9.782,9.852,9.904,9.946,9.979,10.011,10.038,
     * 9.497,9.608,9.696,9.765,9.821,9.866,9.903,9.933,
     * 9.306,9.407,9.482,9.535,9.571,9.593,9.602,9.615,
     * 9.230,9.338,9.417,9.474,9.509,9.529,9.537,9.536,
     * 8.925,9.035,9.115,9.175,9.210,9.231,9.240,9.240/
C
      DATA ((EDEN(I,1,7,L,2),I=33,40),L=1,NZEN)/
     * 10.466,10.584,10.696,10.800,10.890,10.962,11.017,11.055,
     * 10.395,10.514,10.630,10.740,10.837,10.917,10.976,11.017,
     * 10.271,10.390,10.502,10.618,10.726,10.820,10.894,10.948,
     * 10.057,10.166,10.268,10.372,10.478,10.587,10.688,10.772,
     * 9.877,9.945,10.006,10.071,10.133,10.194,10.261,10.342,
     * 9.850,9.918,9.976,10.023,10.070,10.106,10.143,10.185,
     * 9.796,9.872,9.934,9.989,10.034,10.068,10.100,10.131,
     * 9.574,9.671,9.750,9.817,9.875,9.925,9.971,10.010,
     * 9.380,9.467,9.533,9.584,9.625,9.656,9.678,9.702,
     * 9.314,9.404,9.471,9.524,9.565,9.595,9.615,9.628,
     * 9.038,9.122,9.186,9.238,9.278,9.307,9.330,9.346/
C
      DATA ((EDEN(I,2,7,L,2),I=33,40),L=1,NZEN)/
     * 10.384,10.501,10.615,10.727,10.830,10.917,10.987,11.035,
     * 10.374,10.496,10.615,10.731,10.834,10.918,10.981,11.026,
     * 10.274,10.392,10.507,10.622,10.730,10.822,10.895,10.948,
     * 10.073,10.186,10.286,10.385,10.489,10.592,10.684,10.762,
     * 9.869,9.940,10.007,10.068,10.129,10.193,10.258,10.329,
     * 9.832,9.900,9.957,10.005,10.054,10.098,10.144,10.188,
     * 9.785,9.860,9.923,9.973,10.020,10.059,10.095,10.125,
     * 9.569,9.666,9.744,9.810,9.869,9.920,9.965,10.005,
     * 9.316,9.407,9.476,9.531,9.574,9.607,9.630,9.659,
     * 9.257,9.355,9.428,9.485,9.530,9.564,9.588,9.604,
     * 9.002,9.094,9.162,9.217,9.259,9.291,9.312,9.326/
C
      DATA ((EDEN(I,3,7,L,2),I=33,40),L=1,NZEN)/
     * 10.442,10.557,10.667,10.772,10.866,10.943,11.001,11.043,
     * 10.392,10.512,10.628,10.741,10.839,10.921,10.979,11.020,
     * 10.287,10.405,10.519,10.633,10.739,10.828,10.897,10.947,
     * 10.086,10.202,10.305,10.409,10.514,10.615,10.703,10.775,
     * 9.861,9.936,10.006,10.075,10.141,10.211,10.285,10.361,
     * 9.823,9.893,9.953,9.999,10.045,10.096,10.145,10.185,
     * 9.770,9.847,9.911,9.959,9.998,10.042,10.085,10.112,
     * 9.554,9.655,9.736,9.804,9.865,9.916,9.961,10.000,
     * 9.300,9.396,9.468,9.524,9.567,9.599,9.620,9.646,
     * 9.229,9.330,9.405,9.463,9.505,9.535,9.557,9.569,
     * 8.977,9.076,9.147,9.204,9.242,9.268,9.284,9.291/
C
      DATA ((EDEN(I,4,7,L,2),I=33,40),L=1,NZEN)/
     * 10.622,10.734,10.832,10.913,10.976,11.022,11.051,11.074,
     * 10.493,10.608,10.716,10.812,10.892,10.951,10.994,11.023,
     * 10.326,10.446,10.561,10.672,10.770,10.851,10.909,10.951,
     * 10.081,10.205,10.314,10.421,10.528,10.629,10.714,10.781,
     * 9.828,9.910,9.988,10.063,10.134,10.205,10.282,10.357,
     * 9.779,9.855,9.918,9.971,10.022,10.072,10.125,10.175,
     * 9.724,9.808,9.877,9.928,9.971,10.011,10.045,10.072,
     * 9.514,9.623,9.709,9.780,9.839,9.887,9.929,9.964,
     * 9.297,9.397,9.471,9.526,9.565,9.592,9.608,9.627,
     * 9.226,9.330,9.409,9.465,9.505,9.531,9.545,9.551,
     * 8.926,9.031,9.109,9.168,9.208,9.234,9.247,9.254/
C
      DATA ((EDEN(I,5,7,L,2),I=33,40),L=1,NZEN)/
     * 10.778,10.881,10.960,11.017,11.057,11.082,11.090,11.097,
     * 10.665,10.776,10.866,10.938,10.987,11.020,11.038,11.049,
     * 10.439,10.558,10.669,10.767,10.845,10.904,10.945,10.969,
     * 10.125,10.255,10.367,10.476,10.581,10.674,10.749,10.804,
     * 9.814,9.903,9.991,10.076,10.158,10.236,10.319,10.398,
     * 9.756,9.836,9.904,9.963,10.019,10.076,10.131,10.187,
     * 9.696,9.784,9.854,9.906,9.948,9.982,10.014,10.041,
     * 9.497,9.609,9.697,9.767,9.824,9.869,9.907,9.936,
     * 9.284,9.389,9.466,9.521,9.557,9.580,9.591,9.604,
     * 9.207,9.319,9.399,9.457,9.494,9.515,9.525,9.526,
     * 8.898,9.011,9.096,9.155,9.194,9.217,9.226,9.230/
C
      DATA ((EDEN(I,1,8,L,2),I=33,40),L=1,NZEN)/
     * 10.454,10.571,10.681,10.785,10.875,10.949,11.006,11.047,
     * 10.384,10.501,10.615,10.724,10.821,10.902,10.964,11.009,
     * 10.261,10.378,10.487,10.601,10.708,10.803,10.880,10.936,
     * 10.046,10.158,10.257,10.357,10.460,10.568,10.668,10.754,
     * 9.880,9.948,10.010,10.071,10.130,10.188,10.250,10.324,
     * 9.853,9.922,9.980,10.031,10.073,10.109,10.146,10.185,
     * 9.800,9.875,9.939,9.993,10.037,10.075,10.104,10.137,
     * 9.577,9.674,9.753,9.820,9.878,9.929,9.975,10.014,
     * 9.385,9.472,9.539,9.591,9.633,9.665,9.687,9.712,
     * 9.319,9.409,9.477,9.530,9.572,9.602,9.624,9.638,
     * 9.044,9.129,9.193,9.245,9.285,9.316,9.338,9.355/
C
      DATA ((EDEN(I,2,8,L,2),I=33,40),L=1,NZEN)/
     * 10.389,10.504,10.617,10.726,10.827,10.912,10.981,11.031,
     * 10.367,10.486,10.601,10.715,10.818,10.903,10.969,11.015,
     * 10.257,10.373,10.484,10.597,10.704,10.799,10.875,10.932,
     * 10.057,10.164,10.262,10.358,10.455,10.557,10.651,10.734,
     * 9.875,9.944,10.008,10.066,10.123,10.179,10.239,10.302,
     * 9.841,9.908,9.964,10.013,10.058,10.101,10.144,10.183,
     * 9.788,9.863,9.925,9.974,10.020,10.059,10.092,10.123,
     * 9.574,9.670,9.749,9.815,9.874,9.925,9.970,10.009,
     * 9.334,9.425,9.494,9.549,9.592,9.625,9.649,9.677,
     * 9.275,9.370,9.444,9.500,9.545,9.579,9.604,9.620,
     * 9.014,9.103,9.171,9.227,9.267,9.300,9.320,9.336/
C
      DATA ((EDEN(I,3,8,L,2),I=33,40),L=1,NZEN)/
     * 10.435,10.547,10.654,10.758,10.852,10.929,10.991,11.035,
     * 10.375,10.491,10.604,10.714,10.814,10.897,10.962,11.008,
     * 10.263,10.377,10.486,10.597,10.703,10.796,10.871,10.927,
     * 10.066,10.175,10.274,10.370,10.470,10.571,10.664,10.742,
     * 9.874,9.943,10.007,10.069,10.126,10.193,10.257,10.326,
     * 9.836,9.903,9.960,10.008,10.049,10.097,10.145,10.186,
     * 9.786,9.861,9.923,9.972,10.011,10.054,10.093,10.119,
     * 9.566,9.665,9.745,9.813,9.872,9.923,9.967,10.005,
     * 9.326,9.419,9.491,9.547,9.589,9.621,9.643,9.668,
     * 9.255,9.354,9.429,9.485,9.527,9.558,9.579,9.592,
     * 9.001,9.097,9.168,9.224,9.262,9.288,9.305,9.313/
C
      DATA ((EDEN(I,4,8,L,2),I=33,40),L=1,NZEN)/
     * 10.577,10.684,10.782,10.869,10.939,10.994,11.033,11.058,
     * 10.451,10.561,10.666,10.763,10.849,10.917,10.969,11.006,
     * 10.294,10.404,10.510,10.616,10.716,10.802,10.869,10.922,
     * 10.061,10.174,10.278,10.376,10.477,10.576,10.667,10.743,
     * 9.851,9.923,9.991,10.053,10.119,10.184,10.254,10.326,
     * 9.810,9.881,9.941,9.989,10.031,10.076,10.120,10.161,
     * 9.754,9.834,9.899,9.949,9.991,10.028,10.057,10.080,
     * 9.542,9.647,9.730,9.798,9.854,9.901,9.940,9.973,
     * 9.332,9.429,9.502,9.556,9.595,9.621,9.637,9.655,
     * 9.259,9.362,9.437,9.495,9.534,9.559,9.574,9.581,
     * 8.960,9.063,9.140,9.195,9.237,9.261,9.275,9.283/
C
      DATA ((EDEN(I,5,8,L,2),I=33,40),L=1,NZEN)/
     * 10.701,10.802,10.887,10.958,11.009,11.045,11.070,11.082,
     * 10.589,10.693,10.788,10.867,10.930,10.977,11.008,11.028,
     * 10.407,10.513,10.614,10.708,10.790,10.857,10.906,10.942,
     * 10.094,10.208,10.314,10.412,10.510,10.605,10.688,10.757,
     * 9.846,9.921,9.993,10.062,10.130,10.196,10.267,10.338,
     * 9.798,9.872,9.932,9.981,10.025,10.070,10.113,10.156,
     * 9.741,9.823,9.889,9.938,9.979,10.008,10.032,10.054,
     * 9.536,9.643,9.728,9.794,9.847,9.889,9.924,9.949,
     * 9.336,9.436,9.512,9.566,9.603,9.626,9.637,9.649,
     * 9.262,9.368,9.447,9.505,9.542,9.564,9.574,9.575,
     * 8.951,9.059,9.142,9.200,9.237,9.262,9.271,9.273/
C
      DATA ((EDEN(I,1,9,L,2),I=33,40),L=1,NZEN)/
     * 10.427,10.541,10.652,10.755,10.848,10.925,10.985,11.030,
     * 10.356,10.472,10.584,10.693,10.792,10.877,10.943,10.992,
     * 10.236,10.349,10.457,10.568,10.677,10.774,10.854,10.916,
     * 10.031,10.134,10.233,10.328,10.428,10.532,10.633,10.723,
     * 9.886,9.953,10.014,10.071,10.124,10.177,10.233,10.300,
     * 9.858,9.928,9.988,10.038,10.080,10.116,10.152,10.185,
     * 9.804,9.881,9.945,10.001,10.048,10.085,10.116,10.145,
     * 9.581,9.679,9.759,9.826,9.883,9.935,9.981,10.019,
     * 9.397,9.484,9.552,9.605,9.647,9.680,9.704,9.729,
     * 9.330,9.420,9.489,9.544,9.586,9.618,9.641,9.655,
     * 9.056,9.141,9.203,9.256,9.298,9.330,9.354,9.373/
C
      DATA ((EDEN(I,2,9,L,2),I=33,40),L=1,NZEN)/
     * 10.399,10.512,10.621,10.728,10.824,10.908,10.975,11.024,
     * 10.349,10.466,10.579,10.691,10.794,10.881,10.949,11.000,
     * 10.225,10.337,10.443,10.551,10.659,10.758,10.841,10.906,
     * 10.025,10.123,10.217,10.309,10.399,10.496,10.594,10.682,
     * 9.888,9.953,10.010,10.063,10.114,10.166,10.217,10.270,
     * 9.855,9.922,9.979,10.026,10.069,10.109,10.145,10.181,
     * 9.802,9.876,9.938,9.988,10.033,10.074,10.106,10.135,
     * 9.580,9.677,9.756,9.823,9.881,9.931,9.976,10.014,
     * 9.375,9.463,9.532,9.585,9.628,9.660,9.684,9.709,
     * 9.307,9.399,9.469,9.524,9.567,9.598,9.620,9.635,
     * 9.039,9.125,9.193,9.244,9.284,9.314,9.337,9.353/
C
      DATA ((EDEN(I,3,9,L,2),I=33,40),L=1,NZEN)/
     * 10.424,10.532,10.639,10.740,10.835,10.913,10.977,11.024,
     * 10.332,10.441,10.547,10.653,10.755,10.845,10.917,10.974,
     * 10.211,10.319,10.422,10.526,10.634,10.734,10.821,10.889,
     * 10.031,10.132,10.224,10.313,10.405,10.502,10.598,10.686,
     * 9.896,9.959,10.015,10.064,10.118,10.172,10.227,10.282,
     * 9.862,9.926,9.979,10.027,10.066,10.105,10.144,10.179,
     * 9.808,9.880,9.939,9.988,10.029,10.067,10.104,10.127,
     * 9.585,9.682,9.760,9.826,9.883,9.932,9.975,10.012,
     * 9.375,9.464,9.532,9.585,9.625,9.654,9.674,9.697,
     * 9.309,9.403,9.475,9.529,9.570,9.599,9.620,9.632,
     * 9.029,9.118,9.186,9.241,9.278,9.306,9.327,9.339/
C
      DATA ((EDEN(I,4,9,L,2),I=33,40),L=1,NZEN)/
     * 10.504,10.607,10.705,10.798,10.880,10.947,11.000,11.041,
     * 10.399,10.499,10.598,10.693,10.783,10.861,10.924,10.973,
     * 10.243,10.345,10.444,10.543,10.644,10.738,10.819,10.885,
     * 10.037,10.136,10.226,10.316,10.406,10.503,10.600,10.688,
     * 9.892,9.954,10.011,10.061,10.111,10.165,10.223,10.282,
     * 9.856,9.921,9.973,10.015,10.054,10.090,10.125,10.157,
     * 9.803,9.876,9.936,9.982,10.021,10.054,10.082,10.100,
     * 9.585,9.682,9.760,9.823,9.876,9.919,9.956,9.985,
     * 9.393,9.483,9.551,9.603,9.641,9.666,9.682,9.698,
     * 9.325,9.420,9.491,9.545,9.583,9.606,9.620,9.626,
     * 9.031,9.123,9.194,9.248,9.285,9.308,9.322,9.327/
C
      DATA ((EDEN(I,5,9,L,2),I=33,40),L=1,NZEN)/
     * 10.580,10.676,10.767,10.851,10.923,10.982,11.025,11.057,
     * 10.473,10.566,10.659,10.749,10.830,10.897,10.952,10.992,
     * 10.325,10.419,10.509,10.600,10.689,10.770,10.838,10.893,
     * 10.079,10.171,10.259,10.341,10.427,10.517,10.606,10.686,
     * 9.905,9.964,10.017,10.066,10.116,10.168,10.222,10.280,
     * 9.867,9.928,9.979,10.021,10.054,10.085,10.116,10.146,
     * 9.812,9.883,9.941,9.986,10.021,10.045,10.067,10.080,
     * 9.603,9.699,9.775,9.835,9.883,9.920,9.949,9.970,
     * 9.426,9.517,9.587,9.637,9.672,9.693,9.703,9.712,
     * 9.353,9.450,9.524,9.578,9.613,9.634,9.642,9.642,
     * 9.050,9.147,9.222,9.275,9.311,9.332,9.342,9.342/
C
      DATA ((EDEN(I,1,10,L,2),I=33,40),L=1,NZEN)/
     * 10.402,10.516,10.626,10.731,10.826,10.909,10.972,11.021,
     * 10.332,10.448,10.559,10.668,10.770,10.860,10.930,10.983,
     * 10.212,10.324,10.433,10.543,10.652,10.753,10.838,10.905,
     * 10.017,10.119,10.213,10.307,10.405,10.507,10.610,10.703,
     * 9.891,9.958,10.018,10.071,10.124,10.175,10.225,10.287,
     * 9.863,9.934,9.992,10.045,10.090,10.125,10.155,10.188,
     * 9.809,9.886,9.953,10.005,10.055,10.091,10.122,10.151,
     * 9.585,9.683,9.763,9.830,9.888,9.939,9.985,10.024,
     * 9.410,9.498,9.565,9.619,9.663,9.696,9.720,9.745,
     * 9.343,9.434,9.503,9.558,9.601,9.633,9.656,9.670,
     * 9.067,9.151,9.218,9.271,9.312,9.345,9.370,9.387/
C
      DATA ((EDEN(I,2,10,L,2),I=33,40),L=1,NZEN)/
     * 10.421,10.534,10.644,10.747,10.840,10.919,10.982,11.027,
     * 10.339,10.451,10.562,10.669,10.771,10.858,10.929,10.984,
     * 10.209,10.319,10.425,10.533,10.641,10.743,10.829,10.898,
     * 10.016,10.112,10.204,10.296,10.391,10.492,10.594,10.689,
     * 9.898,9.962,10.016,10.068,10.118,10.170,10.220,10.278,
     * 9.870,9.937,9.994,10.039,10.084,10.119,10.152,10.182,
     * 9.815,9.889,9.951,10.002,10.049,10.085,10.119,10.145,
     * 9.586,9.682,9.760,9.825,9.883,9.932,9.976,10.014,
     * 9.422,9.508,9.574,9.626,9.667,9.698,9.720,9.744,
     * 9.354,9.445,9.513,9.566,9.607,9.637,9.658,9.670,
     * 9.074,9.157,9.219,9.271,9.310,9.341,9.363,9.380/
C
      DATA ((EDEN(I,3,10,L,2),I=33,40),L=1,NZEN)/
     * 10.439,10.547,10.651,10.752,10.842,10.920,10.982,11.028,
     * 10.339,10.442,10.543,10.646,10.744,10.830,10.904,10.960,
     * 10.207,10.312,10.413,10.515,10.620,10.721,10.809,10.880,
     * 10.027,10.118,10.205,10.292,10.381,10.479,10.579,10.673,
     * 9.917,9.976,10.026,10.074,10.120,10.170,10.223,10.279,
     * 9.887,9.949,10.001,10.046,10.083,10.117,10.152,10.181,
     * 9.833,9.903,9.960,10.009,10.047,10.087,10.116,10.141,
     * 9.605,9.698,9.773,9.837,9.892,9.939,9.980,10.015,
     * 9.434,9.517,9.581,9.630,9.668,9.695,9.713,9.733,
     * 9.369,9.456,9.522,9.572,9.609,9.635,9.652,9.661,
     * 9.082,9.161,9.224,9.271,9.307,9.331,9.350,9.359/
C
      DATA ((EDEN(I,4,10,L,2),I=33,40),L=1,NZEN)/
     * 10.467,10.567,10.665,10.760,10.847,10.924,10.984,11.030,
     * 10.367,10.463,10.559,10.655,10.748,10.833,10.905,10.961,
     * 10.232,10.326,10.418,10.508,10.603,10.697,10.781,10.851,
     * 10.050,10.133,10.215,10.297,10.379,10.472,10.566,10.658,
     * 9.937,9.990,10.038,10.080,10.125,10.171,10.222,10.278,
     * 9.906,9.961,10.005,10.045,10.077,10.110,10.138,10.166,
     * 9.854,9.917,9.969,10.012,10.046,10.077,10.099,10.116,
     * 9.633,9.721,9.792,9.850,9.899,9.939,9.972,9.999,
     * 9.451,9.532,9.594,9.640,9.674,9.697,9.710,9.724,
     * 9.387,9.473,9.537,9.585,9.618,9.639,9.651,9.654,
     * 9.106,9.185,9.247,9.294,9.325,9.346,9.359,9.365/
C
      DATA ((EDEN(I,5,10,L,2),I=33,40),L=1,NZEN)/
     * 10.510,10.599,10.689,10.778,10.861,10.933,10.992,11.036,
     * 10.410,10.497,10.583,10.673,10.761,10.843,10.912,10.967,
     * 10.274,10.361,10.443,10.528,10.616,10.706,10.787,10.856,
     * 10.090,10.163,10.238,10.312,10.386,10.466,10.554,10.639,
     * 9.970,10.015,10.056,10.095,10.134,10.175,10.217,10.269,
     * 9.938,9.986,10.026,10.061,10.087,10.111,10.133,10.156,
     * 9.887,9.943,9.990,10.025,10.057,10.078,10.090,10.102,
     * 9.675,9.756,9.822,9.874,9.916,9.948,9.973,9.991,
     * 9.490,9.569,9.630,9.676,9.706,9.725,9.733,9.741,
     * 9.427,9.511,9.575,9.623,9.653,9.671,9.677,9.676,
     * 9.149,9.230,9.293,9.340,9.370,9.388,9.394,9.393/
C
      DATA ((EDEN(I,1,11,L,2),I=33,40),L=1,NZEN)/
     * 10.386,10.500,10.611,10.719,10.818,10.905,10.974,11.025,
     * 10.317,10.432,10.544,10.655,10.761,10.855,10.930,10.987,
     * 10.197,10.312,10.420,10.531,10.642,10.747,10.837,10.907,
     * 10.009,10.109,10.205,10.299,10.396,10.500,10.606,10.704,
     * 9.893,9.961,10.018,10.074,10.124,10.175,10.225,10.287,
     * 9.866,9.937,9.996,10.049,10.090,10.125,10.157,10.190,
     * 9.810,9.889,9.953,10.008,10.055,10.094,10.125,10.154,
     * 9.586,9.684,9.765,9.832,9.890,9.941,9.986,10.024,
     * 9.422,9.510,9.578,9.632,9.674,9.707,9.730,9.754,
     * 9.354,9.446,9.516,9.571,9.613,9.645,9.668,9.680,
     * 9.077,9.164,9.230,9.283,9.325,9.357,9.380,9.398/
C
      DATA ((EDEN(I,2,11,L,2),I=33,40),L=1,NZEN)/
     * 10.456,10.569,10.678,10.779,10.869,10.943,11.001,11.044,
     * 10.354,10.464,10.571,10.676,10.774,10.858,10.927,10.980,
     * 10.219,10.328,10.433,10.541,10.649,10.751,10.837,10.906,
     * 10.013,10.110,10.200,10.290,10.383,10.484,10.588,10.686,
     * 9.908,9.971,10.025,10.073,10.120,10.167,10.216,10.274,
     * 9.879,9.947,10.004,10.052,10.092,10.127,10.156,10.186,
     * 9.823,9.899,9.962,10.012,10.058,10.093,10.123,10.149,
     * 9.596,9.691,9.768,9.832,9.887,9.935,9.977,10.014,
     * 9.464,9.548,9.611,9.661,9.700,9.729,9.749,9.769,
     * 9.399,9.486,9.552,9.602,9.641,9.668,9.687,9.698,
     * 9.110,9.189,9.249,9.296,9.335,9.361,9.382,9.395/
C
      DATA ((EDEN(I,3,11,L,2),I=33,40),L=1,NZEN)/
     * 10.477,10.586,10.691,10.790,10.877,10.949,11.004,11.047,
     * 10.377,10.482,10.585,10.687,10.782,10.863,10.931,10.981,
     * 10.244,10.344,10.442,10.541,10.641,10.734,10.815,10.882,
     * 10.053,10.140,10.228,10.316,10.405,10.504,10.604,10.697,
     * 9.935,9.990,10.038,10.084,10.129,10.178,10.227,10.283,
     * 9.905,9.962,10.010,10.053,10.089,10.122,10.156,10.188,
     * 9.852,9.919,9.973,10.021,10.062,10.096,10.125,10.146,
     * 9.626,9.715,9.787,9.848,9.900,9.945,9.984,10.019,
     * 9.476,9.552,9.610,9.654,9.688,9.711,9.725,9.742,
     * 9.421,9.502,9.561,9.606,9.640,9.663,9.676,9.683,
     * 9.134,9.206,9.260,9.304,9.334,9.357,9.372,9.381/
C
      DATA ((EDEN(I,4,11,L,2),I=33,40),L=1,NZEN)/
     * 10.479,10.578,10.677,10.773,10.861,10.936,10.996,11.041,
     * 10.382,10.477,10.572,10.669,10.763,10.847,10.917,10.974,
     * 10.251,10.343,10.433,10.525,10.620,10.713,10.797,10.868,
     * 10.082,10.157,10.234,10.311,10.392,10.478,10.568,10.658,
     * 9.972,10.018,10.060,10.099,10.144,10.188,10.240,10.298,
     * 9.945,9.991,10.029,10.066,10.092,10.120,10.150,10.180,
     * 9.895,9.948,9.993,10.031,10.062,10.088,10.109,10.123,
     * 9.678,9.757,9.820,9.873,9.918,9.955,9.986,10.011,
     * 9.490,9.559,9.613,9.653,9.682,9.701,9.712,9.725,
     * 9.432,9.504,9.559,9.600,9.629,9.646,9.655,9.657,
     * 9.162,9.227,9.278,9.316,9.342,9.360,9.369,9.374/
C
      DATA ((EDEN(I,5,11,L,2),I=33,40),L=1,NZEN)/
     * 10.514,10.601,10.688,10.777,10.861,10.935,10.995,11.040,
     * 10.417,10.501,10.586,10.674,10.763,10.845,10.915,10.971,
     * 10.291,10.370,10.450,10.533,10.620,10.709,10.791,10.863,
     * 10.123,10.188,10.255,10.324,10.396,10.476,10.561,10.647,
     * 10.023,10.057,10.088,10.123,10.152,10.188,10.225,10.269,
     * 9.997,10.032,10.064,10.089,10.111,10.130,10.150,10.169,
     * 9.951,9.992,10.027,10.056,10.080,10.095,10.106,10.116,
     * 9.747,9.811,9.865,9.908,9.943,9.971,9.992,10.007,
     * 9.532,9.597,9.648,9.686,9.712,9.727,9.734,9.741,
     * 9.471,9.539,9.592,9.631,9.656,9.670,9.674,9.672,
     * 9.226,9.290,9.340,9.378,9.401,9.414,9.418,9.415/
C
      DATA ((EDEN(I,1,12,L,2),I=33,40),L=1,NZEN)/
     * 10.379,10.495,10.605,10.716,10.818,10.907,10.980,11.034,
     * 10.311,10.426,10.539,10.652,10.760,10.858,10.936,10.995,
     * 10.192,10.308,10.417,10.528,10.641,10.750,10.843,10.915,
     * 10.009,10.109,10.205,10.299,10.398,10.505,10.612,10.713,
     * 9.894,9.962,10.018,10.074,10.124,10.175,10.228,10.293,
     * 9.866,9.938,9.996,10.049,10.090,10.125,10.157,10.190,
     * 9.810,9.889,9.953,10.008,10.055,10.094,10.122,10.154,
     * 9.585,9.684,9.764,9.831,9.888,9.939,9.984,10.021,
     * 9.429,9.517,9.585,9.639,9.681,9.713,9.735,9.758,
     * 9.361,9.454,9.524,9.579,9.621,9.652,9.672,9.684,
     * 9.085,9.173,9.239,9.290,9.333,9.363,9.387,9.403/
C
      DATA ((EDEN(I,2,12,L,2),I=33,40),L=1,NZEN)/
     * 10.481,10.594,10.703,10.803,10.891,10.963,11.017,11.056,
     * 10.379,10.491,10.597,10.701,10.798,10.880,10.946,10.996,
     * 10.251,10.359,10.464,10.570,10.675,10.772,10.853,10.917,
     * 10.018,10.110,10.201,10.293,10.388,10.489,10.595,10.695,
     * 9.911,9.972,10.026,10.074,10.120,10.168,10.219,10.276,
     * 9.881,9.946,9.998,10.046,10.083,10.118,10.151,10.186,
     * 9.826,9.903,9.964,10.018,10.062,10.097,10.124,10.150,
     * 9.603,9.697,9.772,9.834,9.888,9.935,9.976,10.013,
     * 9.476,9.556,9.616,9.662,9.698,9.723,9.739,9.758,
     * 9.427,9.512,9.575,9.624,9.660,9.686,9.703,9.711,
     * 9.135,9.211,9.268,9.315,9.348,9.375,9.392,9.403/
C
      DATA ((EDEN(I,3,12,L,2),I=33,40),L=1,NZEN)/
     * 10.513,10.623,10.726,10.822,10.905,10.973,11.023,11.060,
     * 10.412,10.519,10.622,10.722,10.813,10.891,10.953,10.999,
     * 10.278,10.380,10.478,10.577,10.676,10.767,10.842,10.903,
     * 10.070,10.163,10.252,10.340,10.432,10.532,10.630,10.720,
     * 9.944,9.994,10.041,10.089,10.140,10.193,10.251,10.316,
     * 9.915,9.969,10.014,10.056,10.091,10.127,10.158,10.192,
     * 9.863,9.925,9.975,10.021,10.057,10.091,10.120,10.142,
     * 9.638,9.725,9.794,9.853,9.904,9.948,9.987,10.021,
     * 9.501,9.573,9.627,9.668,9.699,9.720,9.733,9.747,
     * 9.444,9.518,9.572,9.614,9.644,9.662,9.672,9.676,
     * 9.166,9.232,9.281,9.319,9.348,9.367,9.380,9.387/
C
      DATA ((EDEN(I,4,12,L,2),I=33,40),L=1,NZEN)/
     * 10.506,10.607,10.705,10.800,10.885,10.957,11.013,11.054,
     * 10.410,10.506,10.602,10.699,10.791,10.873,10.939,10.989,
     * 10.280,10.373,10.462,10.556,10.650,10.741,10.822,10.889,
     * 10.104,10.184,10.260,10.339,10.421,10.508,10.599,10.685,
     * 9.989,10.029,10.071,10.112,10.156,10.204,10.257,10.319,
     * 9.963,10.004,10.039,10.073,10.102,10.129,10.161,10.193,
     * 9.917,9.964,10.004,10.040,10.066,10.091,10.111,10.125,
     * 9.706,9.778,9.836,9.885,9.927,9.963,9.992,10.017,
     * 9.507,9.569,9.617,9.653,9.679,9.696,9.704,9.717,
     * 9.454,9.519,9.566,9.603,9.628,9.643,9.651,9.651,
     * 9.195,9.250,9.294,9.325,9.349,9.363,9.371,9.372/
C
      DATA ((EDEN(I,5,12,L,2),I=33,40),L=1,NZEN)/
     * 10.545,10.631,10.719,10.806,10.886,10.956,11.012,11.053,
     * 10.450,10.534,10.618,10.706,10.792,10.871,10.935,10.987,
     * 10.323,10.403,10.482,10.566,10.652,10.738,10.817,10.881,
     * 10.154,10.219,10.285,10.355,10.426,10.506,10.590,10.673,
     * 10.046,10.077,10.105,10.134,10.168,10.205,10.243,10.292,
     * 10.024,10.053,10.079,10.101,10.123,10.139,10.159,10.177,
     * 9.980,10.015,10.046,10.070,10.092,10.106,10.116,10.123,
     * 9.788,9.843,9.889,9.927,9.958,9.983,10.002,10.016,
     * 9.549,9.604,9.647,9.679,9.701,9.715,9.720,9.728,
     * 9.499,9.555,9.599,9.631,9.652,9.663,9.666,9.663,
     * 9.247,9.294,9.334,9.362,9.380,9.388,9.391,9.390/
C
      DATA ((EDEN(I,1,1,L,3),I=33,40),L=1,NZEN)/
     * 10.480,10.597,10.709,10.818,10.917,11.002,11.068,11.119,
     * 10.412,10.528,10.641,10.755,10.859,10.951,11.025,11.080,
     * 10.293,10.408,10.518,10.630,10.743,10.847,10.934,11.000,
     * 10.113,10.210,10.302,10.398,10.499,10.607,10.714,10.809,
     * 9.998,10.058,10.113,10.161,10.214,10.266,10.322,10.396,
     * 9.971,10.033,10.084,10.129,10.172,10.208,10.241,10.280,
     * 9.913,9.981,10.037,10.087,10.132,10.167,10.201,10.236,
     * 9.652,9.739,9.812,9.875,9.934,9.990,10.042,10.089,
     * 9.437,9.508,9.562,9.607,9.645,9.677,9.703,9.733,
     * 9.367,9.440,9.495,9.540,9.578,9.609,9.632,9.650,
     * 9.089,9.152,9.199,9.242,9.278,9.311,9.338,9.358/
C
      DATA ((EDEN(I,2,1,L,3),I=33,40),L=1,NZEN)/
     * 10.587,10.702,10.808,10.904,10.986,11.051,11.104,11.137,
     * 10.484,10.597,10.704,10.804,10.896,10.973,11.031,11.078,
     * 10.357,10.465,10.569,10.676,10.778,10.869,10.943,11.001,
     * 10.123,10.213,10.300,10.392,10.488,10.593,10.698,10.794,
     * 10.016,10.069,10.116,10.161,10.205,10.254,10.309,10.374,
     * 9.987,10.039,10.082,10.124,10.162,10.198,10.235,10.276,
     * 9.924,9.988,10.039,10.086,10.127,10.164,10.198,10.230,
     * 9.659,9.742,9.810,9.870,9.927,9.981,10.031,10.077,
     * 9.486,9.549,9.595,9.633,9.664,9.690,9.708,9.734,
     * 9.437,9.502,9.550,9.589,9.620,9.646,9.665,9.678,
     * 9.143,9.196,9.236,9.270,9.300,9.326,9.347,9.363/
C
      DATA ((EDEN(I,3,1,L,3),I=33,40),L=1,NZEN)/
     * 10.621,10.731,10.832,10.923,11.000,11.061,11.109,11.141,
     * 10.518,10.626,10.728,10.826,10.912,10.983,11.037,11.082,
     * 10.385,10.486,10.585,10.683,10.779,10.865,10.934,10.988,
     * 10.197,10.282,10.367,10.456,10.550,10.647,10.738,10.817,
     * 10.050,10.095,10.135,10.181,10.230,10.284,10.347,10.418,
     * 10.020,10.064,10.100,10.137,10.173,10.210,10.245,10.284,
     * 9.967,10.016,10.060,10.098,10.134,10.170,10.200,10.228,
     * 9.706,9.782,9.844,9.901,9.955,10.005,10.051,10.094,
     * 9.516,9.571,9.611,9.644,9.671,9.693,9.708,9.729,
     * 9.457,9.513,9.552,9.584,9.609,9.628,9.641,9.651,
     * 9.181,9.224,9.255,9.283,9.306,9.326,9.343,9.354/
C
      DATA ((EDEN(I,4,1,L,3),I=33,40),L=1,NZEN)/
     * 10.612,10.715,10.813,10.905,10.983,11.047,11.099,11.137,
     * 10.514,10.612,10.709,10.804,10.891,10.967,11.027,11.074,
     * 10.384,10.477,10.568,10.662,10.756,10.841,10.916,10.976,
     * 10.209,10.285,10.361,10.438,10.523,10.613,10.702,10.783,
     * 10.092,10.128,10.164,10.203,10.247,10.297,10.355,10.421,
     * 10.065,10.099,10.129,10.156,10.184,10.216,10.249,10.286,
     * 10.022,10.058,10.091,10.119,10.147,10.172,10.194,10.214,
     * 9.799,9.859,9.910,9.955,9.997,10.035,10.069,10.100,
     * 9.527,9.574,9.609,9.638,9.660,9.677,9.688,9.706,
     * 9.474,9.521,9.555,9.583,9.603,9.620,9.629,9.633,
     * 9.216,9.251,9.277,9.301,9.319,9.333,9.343,9.350/
C
      DATA ((EDEN(I,5,1,L,3),I=33,40),L=1,NZEN)/
     * 10.652,10.741,10.830,10.914,10.988,11.051,11.102,11.138,
     * 10.556,10.641,10.728,10.815,10.897,10.969,11.028,11.075,
     * 10.429,10.508,10.589,10.674,10.761,10.843,10.915,10.973,
     * 10.255,10.318,10.386,10.454,10.530,10.614,10.697,10.776,
     * 10.144,10.169,10.195,10.224,10.255,10.293,10.337,10.390,
     * 10.121,10.144,10.165,10.186,10.204,10.222,10.242,10.266,
     * 10.078,10.106,10.131,10.152,10.170,10.186,10.196,10.207,
     * 9.878,9.924,9.964,9.999,10.030,10.057,10.080,10.098,
     * 9.585,9.627,9.660,9.686,9.704,9.717,9.723,9.734,
     * 9.526,9.567,9.599,9.624,9.640,9.651,9.655,9.654,
     * 9.278,9.306,9.329,9.351,9.363,9.372,9.376,9.378/
C
      DATA ((EDEN(I,1,2,L,3),I=33,40),L=1,NZEN)/
     * 10.477,10.593,10.704,10.810,10.908,10.991,11.056,11.108,
     * 10.409,10.523,10.636,10.747,10.851,10.941,11.013,11.069,
     * 10.291,10.403,10.511,10.623,10.733,10.835,10.921,10.989,
     * 10.113,10.207,10.297,10.391,10.491,10.596,10.700,10.795,
     * 9.998,10.058,10.110,10.161,10.212,10.262,10.318,10.385,
     * 9.971,10.033,10.084,10.129,10.172,10.208,10.241,10.280,
     * 9.913,9.981,10.037,10.085,10.132,10.167,10.201,10.238,
     * 9.652,9.739,9.811,9.875,9.934,9.990,10.043,10.090,
     * 9.434,9.505,9.559,9.604,9.644,9.676,9.702,9.733,
     * 9.364,9.437,9.491,9.537,9.575,9.607,9.633,9.651,
     * 9.086,9.150,9.197,9.239,9.276,9.309,9.337,9.359/
C
      DATA ((EDEN(I,2,2,L,3),I=33,40),L=1,NZEN)/
     * 10.557,10.670,10.776,10.873,10.959,11.026,11.081,11.120,
     * 10.453,10.564,10.671,10.773,10.866,10.945,11.009,11.058,
     * 10.321,10.428,10.535,10.641,10.747,10.843,10.925,10.989,
     * 10.116,10.207,10.293,10.384,10.478,10.582,10.686,10.780,
     * 10.011,10.064,10.112,10.158,10.203,10.251,10.304,10.368,
     * 9.977,10.034,10.078,10.120,10.158,10.197,10.232,10.274,
     * 9.919,9.983,10.035,10.082,10.124,10.164,10.198,10.227,
     * 9.656,9.740,9.808,9.870,9.927,9.981,10.032,10.078,
     * 9.488,9.554,9.603,9.643,9.677,9.706,9.727,9.753,
     * 9.422,9.490,9.538,9.579,9.610,9.639,9.659,9.673,
     * 9.132,9.187,9.229,9.265,9.297,9.323,9.347,9.364/
C
      DATA ((EDEN(I,3,2,L,3),I=33,40),L=1,NZEN)/
     * 10.580,10.688,10.790,10.884,10.965,11.032,11.083,11.124,
     * 10.479,10.583,10.685,10.783,10.873,10.950,11.011,11.059,
     * 10.346,10.445,10.541,10.639,10.736,10.823,10.899,10.962,
     * 10.159,10.243,10.328,10.413,10.506,10.605,10.701,10.788,
     * 10.036,10.083,10.125,10.171,10.217,10.273,10.333,10.401,
     * 10.005,10.054,10.095,10.130,10.166,10.201,10.239,10.277,
     * 9.952,10.005,10.051,10.090,10.127,10.164,10.196,10.225,
     * 9.696,9.773,9.837,9.895,9.949,10.000,10.047,10.090,
     * 9.504,9.561,9.605,9.639,9.668,9.690,9.707,9.730,
     * 9.452,9.511,9.555,9.590,9.617,9.639,9.656,9.665,
     * 9.166,9.215,9.250,9.281,9.306,9.328,9.345,9.359/
C
      DATA ((EDEN(I,4,2,L,3),I=33,40),L=1,NZEN)/
     * 10.575,10.675,10.773,10.866,10.951,11.021,11.075,11.119,
     * 10.477,10.573,10.669,10.764,10.855,10.934,11.001,11.053,
     * 10.346,10.437,10.527,10.620,10.714,10.805,10.884,10.949,
     * 10.175,10.248,10.324,10.401,10.484,10.572,10.663,10.747,
     * 10.062,10.104,10.140,10.182,10.223,10.273,10.326,10.389,
     * 10.036,10.074,10.109,10.139,10.168,10.201,10.233,10.269,
     * 9.988,10.034,10.070,10.104,10.133,10.160,10.185,10.207,
     * 9.762,9.828,9.883,9.932,9.977,10.017,10.055,10.087,
     * 9.522,9.574,9.614,9.645,9.670,9.688,9.700,9.719,
     * 9.465,9.519,9.557,9.588,9.611,9.628,9.638,9.643,
     * 9.201,9.244,9.275,9.302,9.322,9.339,9.351,9.357/
C
      DATA ((EDEN(I,5,2,L,3),I=33,40),L=1,NZEN)/
     * 10.610,10.700,10.789,10.876,10.957,11.026,11.082,11.124,
     * 10.511,10.597,10.686,10.775,10.861,10.938,11.003,11.054,
     * 10.382,10.464,10.546,10.632,10.721,10.807,10.884,10.949,
     * 10.208,10.274,10.344,10.416,10.491,10.573,10.660,10.743,
     * 10.097,10.130,10.163,10.196,10.229,10.268,10.309,10.361,
     * 10.073,10.106,10.135,10.158,10.181,10.203,10.225,10.250,
     * 10.031,10.068,10.097,10.125,10.149,10.166,10.183,10.194,
     * 9.826,9.882,9.929,9.970,10.005,10.036,10.063,10.084,
     * 9.572,9.620,9.658,9.687,9.709,9.722,9.729,9.741,
     * 9.514,9.562,9.601,9.629,9.648,9.661,9.666,9.666,
     * 9.271,9.311,9.341,9.367,9.384,9.393,9.399,9.400/
C
      DATA ((EDEN(I,1,3,L,3),I=33,40),L=1,NZEN)/
     * 10.480,10.596,10.707,10.811,10.905,10.986,11.048,11.097,
     * 10.412,10.527,10.638,10.748,10.848,10.936,11.005,11.057,
     * 10.293,10.404,10.514,10.623,10.732,10.831,10.913,10.979,
     * 10.113,10.207,10.297,10.391,10.488,10.592,10.693,10.784,
     * 9.998,10.058,10.110,10.161,10.212,10.260,10.314,10.379,
     * 9.971,10.029,10.084,10.129,10.172,10.208,10.241,10.280,
     * 9.912,9.977,10.034,10.085,10.132,10.167,10.201,10.238,
     * 9.654,9.741,9.814,9.878,9.938,9.993,10.045,10.091,
     * 9.426,9.498,9.552,9.597,9.636,9.670,9.696,9.729,
     * 9.357,9.429,9.484,9.529,9.568,9.601,9.627,9.646,
     * 9.079,9.140,9.189,9.230,9.268,9.302,9.331,9.353/
C
      DATA ((EDEN(I,2,3,L,3),I=33,40),L=1,NZEN)/
     * 10.517,10.630,10.737,10.838,10.926,11.000,11.058,11.102,
     * 10.436,10.549,10.657,10.763,10.859,10.941,11.007,11.058,
     * 10.303,10.412,10.517,10.624,10.731,10.827,10.910,10.974,
     * 10.112,10.201,10.290,10.380,10.475,10.578,10.681,10.773,
     * 10.000,10.056,10.104,10.151,10.199,10.251,10.306,10.368,
     * 9.969,10.028,10.079,10.122,10.160,10.199,10.234,10.269,
     * 9.911,9.976,10.029,10.077,10.120,10.160,10.194,10.227,
     * 9.653,9.738,9.808,9.870,9.929,9.983,10.034,10.080,
     * 9.458,9.526,9.578,9.620,9.656,9.686,9.710,9.738,
     * 9.391,9.460,9.512,9.554,9.589,9.619,9.641,9.658,
     * 9.106,9.166,9.210,9.248,9.282,9.313,9.337,9.357/
C
      DATA ((EDEN(I,3,3,L,3),I=33,40),L=1,NZEN)/
     * 10.531,10.639,10.742,10.839,10.925,10.999,11.055,11.102,
     * 10.430,10.534,10.635,10.735,10.829,10.912,10.980,11.034,
     * 10.304,10.405,10.503,10.602,10.704,10.798,10.880,10.947,
     * 10.120,10.205,10.290,10.377,10.467,10.566,10.667,10.759,
     * 10.016,10.065,10.112,10.155,10.200,10.253,10.309,10.368,
     * 9.986,10.038,10.082,10.118,10.153,10.194,10.233,10.269,
     * 9.929,9.990,10.039,10.083,10.119,10.159,10.194,10.223,
     * 9.678,9.759,9.826,9.886,9.942,9.994,10.042,10.086,
     * 9.480,9.544,9.592,9.630,9.663,9.689,9.708,9.732,
     * 9.416,9.481,9.530,9.568,9.599,9.624,9.641,9.652,
     * 9.132,9.187,9.228,9.264,9.292,9.318,9.339,9.353/
C
      DATA ((EDEN(I,4,3,L,3),I=33,40),L=1,NZEN)/
     * 10.543,10.644,10.742,10.838,10.925,10.999,11.059,11.105,
     * 10.440,10.538,10.635,10.733,10.826,10.911,10.980,11.036,
     * 10.307,10.400,10.493,10.587,10.683,10.776,10.858,10.929,
     * 10.133,10.210,10.290,10.369,10.453,10.546,10.640,10.729,
     * 10.017,10.069,10.113,10.155,10.199,10.247,10.303,10.366,
     * 9.989,10.039,10.079,10.116,10.147,10.182,10.216,10.251,
     * 9.938,9.993,10.040,10.078,10.111,10.145,10.172,10.194,
     * 9.710,9.786,9.849,9.903,9.953,9.997,10.038,10.073,
     * 9.498,9.559,9.605,9.641,9.668,9.689,9.702,9.720,
     * 9.439,9.502,9.547,9.583,9.611,9.630,9.642,9.649,
     * 9.164,9.221,9.259,9.292,9.317,9.336,9.350,9.357/
C
      DATA ((EDEN(I,5,3,L,3),I=33,40),L=1,NZEN)/
     * 10.580,10.674,10.767,10.858,10.941,11.013,11.070,11.116,
     * 10.477,10.569,10.660,10.752,10.843,10.923,10.992,11.045,
     * 10.340,10.430,10.517,10.606,10.698,10.789,10.869,10.938,
     * 10.154,10.231,10.306,10.383,10.464,10.550,10.639,10.725,
     * 10.036,10.080,10.123,10.160,10.200,10.242,10.291,10.347,
     * 10.007,10.051,10.090,10.122,10.152,10.177,10.203,10.235,
     * 9.959,10.009,10.052,10.086,10.117,10.140,10.159,10.177,
     * 9.750,9.820,9.879,9.928,9.970,10.006,10.037,10.062,
     * 9.545,9.606,9.651,9.686,9.710,9.726,9.734,9.745,
     * 9.487,9.549,9.596,9.631,9.654,9.668,9.675,9.675,
     * 9.227,9.284,9.325,9.359,9.379,9.393,9.399,9.398/
C
      DATA ((EDEN(I,1,4,L,3),I=33,40),L=1,NZEN)/
     * 10.500,10.617,10.727,10.830,10.921,10.996,11.056,11.097,
     * 10.430,10.547,10.659,10.768,10.864,10.947,11.012,11.061,
     * 10.307,10.422,10.532,10.643,10.749,10.845,10.923,10.983,
     * 10.119,10.217,10.310,10.403,10.503,10.607,10.706,10.794,
     * 9.994,10.054,10.107,10.159,10.212,10.262,10.318,10.385,
     * 9.967,10.026,10.080,10.126,10.167,10.204,10.238,10.278,
     * 9.911,9.977,10.034,10.082,10.127,10.165,10.199,10.236,
     * 9.656,9.743,9.815,9.879,9.938,9.994,10.046,10.092,
     * 9.413,9.484,9.537,9.582,9.622,9.655,9.682,9.715,
     * 9.344,9.417,9.470,9.515,9.554,9.586,9.613,9.632,
     * 9.066,9.128,9.175,9.217,9.253,9.287,9.315,9.340/
C
      DATA ((EDEN(I,2,4,L,3),I=33,40),L=1,NZEN)/
     * 10.482,10.598,10.708,10.811,10.904,10.982,11.046,11.091,
     * 10.410,10.526,10.638,10.746,10.845,10.931,11.000,11.050,
     * 10.296,10.406,10.516,10.626,10.733,10.830,10.912,10.975,
     * 10.113,10.208,10.302,10.395,10.496,10.601,10.701,10.789,
     * 9.989,10.050,10.102,10.149,10.196,10.245,10.297,10.351,
     * 9.957,10.017,10.066,10.111,10.150,10.190,10.231,10.268,
     * 9.903,9.967,10.022,10.068,10.112,10.154,10.191,10.223,
     * 9.654,9.740,9.811,9.874,9.934,9.989,10.040,10.086,
     * 9.412,9.483,9.536,9.580,9.618,9.650,9.676,9.707,
     * 9.344,9.416,9.469,9.513,9.551,9.582,9.607,9.625,
     * 9.067,9.131,9.177,9.219,9.254,9.288,9.314,9.336/
C
      DATA ((EDEN(I,3,4,L,3),I=33,40),L=1,NZEN)/
     * 10.498,10.609,10.715,10.816,10.908,10.984,11.047,11.091,
     * 10.396,10.503,10.606,10.710,10.808,10.895,10.966,11.023,
     * 10.285,10.393,10.498,10.605,10.711,10.811,10.893,10.960,
     * 10.110,10.203,10.292,10.383,10.479,10.584,10.685,10.774,
     * 9.988,10.045,10.099,10.143,10.192,10.251,10.314,10.381,
     * 9.959,10.016,10.066,10.105,10.142,10.184,10.229,10.263,
     * 9.905,9.969,10.021,10.065,10.103,10.148,10.187,10.216,
     * 9.661,9.747,9.817,9.880,9.939,9.992,10.041,10.086,
     * 9.425,9.494,9.547,9.589,9.625,9.653,9.674,9.701,
     * 9.357,9.429,9.482,9.524,9.558,9.586,9.607,9.620,
     * 9.077,9.142,9.190,9.230,9.262,9.290,9.313,9.328/
C
      DATA ((EDEN(I,4,4,L,3),I=33,40),L=1,NZEN)/
     * 10.554,10.661,10.762,10.860,10.944,11.015,11.067,11.108,
     * 10.447,10.550,10.653,10.753,10.845,10.927,10.992,11.045,
     * 10.297,10.400,10.500,10.602,10.704,10.799,10.880,10.946,
     * 10.104,10.198,10.287,10.378,10.475,10.576,10.675,10.764,
     * 9.970,10.030,10.082,10.128,10.179,10.234,10.297,10.366,
     * 9.937,9.995,10.045,10.086,10.124,10.162,10.200,10.239,
     * 9.883,9.950,10.003,10.049,10.085,10.122,10.154,10.181,
     * 9.657,9.744,9.815,9.876,9.930,9.979,10.022,10.060,
     * 9.448,9.519,9.572,9.613,9.644,9.667,9.682,9.701,
     * 9.383,9.457,9.510,9.552,9.583,9.606,9.620,9.627,
     * 9.098,9.167,9.217,9.256,9.286,9.307,9.322,9.332/
C
      DATA ((EDEN(I,5,4,L,3),I=33,40),L=1,NZEN)/
     * 10.620,10.721,10.819,10.910,10.988,11.051,11.098,11.134,
     * 10.510,10.610,10.710,10.806,10.894,10.967,11.026,11.070,
     * 10.359,10.459,10.557,10.655,10.751,10.838,10.910,10.968,
     * 10.129,10.223,10.312,10.399,10.492,10.588,10.681,10.764,
     * 9.969,10.025,10.077,10.127,10.178,10.231,10.291,10.362,
     * 9.932,9.991,10.040,10.082,10.116,10.151,10.185,10.221,
     * 9.880,9.946,10.001,10.045,10.081,10.108,10.132,10.155,
     * 9.671,9.757,9.827,9.885,9.934,9.976,10.012,10.042,
     * 9.489,9.562,9.616,9.657,9.685,9.703,9.712,9.723,
     * 9.425,9.503,9.559,9.600,9.627,9.644,9.651,9.651,
     * 9.134,9.208,9.263,9.303,9.330,9.347,9.353,9.354/
C
      DATA ((EDEN(I,1,5,L,3),I=33,40),L=1,NZEN)/
     * 10.532,10.650,10.763,10.863,10.950,11.019,11.076,11.115,
     * 10.459,10.580,10.694,10.801,10.896,10.974,11.034,11.076,
     * 10.336,10.454,10.566,10.679,10.783,10.875,10.947,11.003,
     * 10.137,10.236,10.333,10.433,10.539,10.644,10.741,10.824,
     * 9.990,10.051,10.104,10.159,10.214,10.271,10.331,10.407,
     * 9.962,10.022,10.074,10.118,10.162,10.199,10.234,10.276,
     * 9.908,9.973,10.027,10.075,10.121,10.160,10.192,10.229,
     * 9.657,9.743,9.815,9.879,9.938,9.993,10.045,10.091,
     * 9.400,9.470,9.523,9.567,9.606,9.638,9.665,9.698,
     * 9.333,9.403,9.455,9.499,9.537,9.569,9.595,9.613,
     * 9.056,9.115,9.161,9.201,9.238,9.270,9.298,9.321/
C
      DATA ((EDEN(I,2,5,L,3),I=33,40),L=1,NZEN)/
     * 10.473,10.590,10.703,10.810,10.908,10.988,11.050,11.099,
     * 10.447,10.569,10.686,10.797,10.895,10.978,11.040,11.085,
     * 10.330,10.448,10.561,10.674,10.779,10.873,10.945,11.000,
     * 10.133,10.234,10.331,10.426,10.527,10.629,10.723,10.805,
     * 9.981,10.040,10.097,10.148,10.202,10.258,10.318,10.384,
     * 9.948,10.006,10.057,10.100,10.144,10.187,10.229,10.273,
     * 9.894,9.959,10.016,10.059,10.101,10.144,10.182,10.217,
     * 9.658,9.744,9.814,9.877,9.936,9.991,10.042,10.088,
     * 9.367,9.439,9.494,9.539,9.579,9.612,9.638,9.672,
     * 9.297,9.371,9.427,9.472,9.510,9.542,9.568,9.587,
     * 9.037,9.104,9.153,9.194,9.232,9.264,9.290,9.311/
C
      DATA ((EDEN(I,3,5,L,3),I=33,40),L=1,NZEN)/
     * 10.505,10.619,10.729,10.834,10.925,11.001,11.059,11.102,
     * 10.417,10.533,10.645,10.756,10.856,10.941,11.008,11.058,
     * 10.327,10.443,10.556,10.669,10.776,10.868,10.941,10.996,
     * 10.135,10.236,10.337,10.434,10.538,10.641,10.735,10.813,
     * 9.973,10.036,10.092,10.146,10.198,10.266,10.334,10.404,
     * 9.936,9.998,10.051,10.092,10.130,10.179,10.229,10.272,
     * 9.888,9.957,10.012,10.054,10.093,10.139,10.182,10.212,
     * 9.651,9.740,9.812,9.877,9.937,9.992,10.042,10.087,
     * 9.364,9.439,9.495,9.540,9.577,9.607,9.629,9.660,
     * 9.300,9.378,9.437,9.482,9.519,9.549,9.573,9.589,
     * 9.032,9.103,9.157,9.199,9.235,9.262,9.284,9.299/
C
      DATA ((EDEN(I,4,5,L,3),I=33,40),L=1,NZEN)/
     * 10.624,10.737,10.839,10.930,11.004,11.060,11.102,11.134,
     * 10.512,10.624,10.730,10.829,10.914,10.982,11.033,11.073,
     * 10.340,10.456,10.567,10.678,10.781,10.870,10.939,10.992,
     * 10.116,10.228,10.331,10.436,10.542,10.648,10.740,10.817,
     * 9.932,10.001,10.062,10.121,10.184,10.252,10.326,10.406,
     * 9.894,9.961,10.018,10.063,10.104,10.150,10.197,10.244,
     * 9.840,9.913,9.972,10.020,10.063,10.102,10.139,10.169,
     * 9.612,9.708,9.786,9.852,9.912,9.964,10.010,10.050,
     * 9.381,9.460,9.519,9.564,9.599,9.624,9.640,9.663,
     * 9.310,9.394,9.455,9.501,9.534,9.559,9.576,9.584,
     * 9.017,9.095,9.152,9.197,9.229,9.254,9.270,9.279/
C
      DATA ((EDEN(I,5,5,L,3),I=33,40),L=1,NZEN)/
     * 10.737,10.844,10.938,11.015,11.073,11.117,11.146,11.162,
     * 10.622,10.733,10.835,10.922,10.992,11.045,11.082,11.105,
     * 10.463,10.574,10.681,10.781,10.866,10.934,10.984,11.020,
     * 10.144,10.260,10.368,10.472,10.580,10.680,10.767,10.835,
     * 9.914,9.985,10.053,10.118,10.186,10.258,10.336,10.422,
     * 9.869,9.939,9.998,10.047,10.093,10.137,10.184,10.236,
     * 9.814,9.892,9.955,10.002,10.044,10.078,10.108,10.135,
     * 9.602,9.703,9.783,9.849,9.905,9.952,9.993,10.027,
     * 9.401,9.486,9.549,9.594,9.626,9.646,9.655,9.669,
     * 9.328,9.418,9.483,9.531,9.562,9.581,9.590,9.591,
     * 9.025,9.112,9.177,9.224,9.255,9.275,9.284,9.287/
C
      DATA ((EDEN(I,1,6,L,3),I=33,40),L=1,NZEN)/
     * 10.561,10.682,10.794,10.894,10.978,11.044,11.094,11.133,
     * 10.490,10.612,10.726,10.833,10.925,10.998,11.054,11.091,
     * 10.364,10.483,10.599,10.712,10.816,10.905,10.972,11.021,
     * 10.153,10.259,10.359,10.465,10.573,10.680,10.775,10.852,
     * 9.986,10.047,10.104,10.161,10.221,10.283,10.354,10.436,
     * 9.958,10.018,10.068,10.115,10.157,10.196,10.234,10.278,
     * 9.905,9.969,10.023,10.072,10.116,10.152,10.187,10.222,
     * 9.657,9.742,9.813,9.876,9.934,9.990,10.042,10.088,
     * 9.390,9.459,9.511,9.554,9.592,9.623,9.649,9.681,
     * 9.321,9.392,9.444,9.487,9.524,9.555,9.579,9.598,
     * 9.045,9.106,9.150,9.189,9.225,9.257,9.283,9.306/
C
      DATA ((EDEN(I,2,6,L,3),I=33,40),L=1,NZEN)/
     * 10.479,10.600,10.715,10.825,10.924,11.005,11.066,11.113,
     * 10.469,10.593,10.713,10.826,10.923,11.001,11.060,11.100,
     * 10.366,10.486,10.602,10.716,10.820,10.907,10.974,11.022,
     * 10.168,10.274,10.373,10.476,10.581,10.683,10.771,10.844,
     * 9.975,10.042,10.099,10.158,10.216,10.279,10.350,10.422,
     * 9.940,10.000,10.052,10.096,10.140,10.186,10.236,10.283,
     * 9.894,9.960,10.013,10.061,10.103,10.143,10.184,10.219,
     * 9.664,9.748,9.817,9.879,9.938,9.992,10.043,10.089,
     * 9.334,9.407,9.463,9.509,9.548,9.582,9.608,9.645,
     * 9.274,9.351,9.409,9.456,9.496,9.531,9.559,9.579,
     * 9.010,9.081,9.133,9.176,9.212,9.244,9.271,9.292/
C
      DATA ((EDEN(I,3,6,L,3),I=33,40),L=1,NZEN)/
     * 10.532,10.650,10.762,10.866,10.955,11.027,11.079,11.120,
     * 10.482,10.605,10.723,10.834,10.927,11.003,11.057,11.097,
     * 10.373,10.494,10.611,10.725,10.828,10.912,10.975,11.022,
     * 10.176,10.286,10.391,10.497,10.605,10.705,10.789,10.857,
     * 9.964,10.033,10.099,10.159,10.224,10.299,10.379,10.457,
     * 9.929,9.993,10.047,10.089,10.131,10.183,10.237,10.279,
     * 9.877,9.947,10.003,10.043,10.085,10.132,10.178,10.208,
     * 9.648,9.738,9.811,9.877,9.938,9.993,10.043,10.088,
     * 9.323,9.402,9.462,9.509,9.549,9.581,9.606,9.639,
     * 9.252,9.333,9.395,9.442,9.480,9.512,9.537,9.554,
     * 8.992,9.068,9.124,9.168,9.202,9.230,9.250,9.264/
C
      DATA ((EDEN(I,4,6,L,3),I=33,40),L=1,NZEN)/
     * 10.706,10.819,10.916,10.995,11.056,11.100,11.132,11.151,
     * 10.575,10.694,10.803,10.897,10.973,11.031,11.070,11.099,
     * 10.405,10.530,10.649,10.761,10.856,10.931,10.988,11.025,
     * 10.161,10.282,10.396,10.508,10.619,10.719,10.800,10.863,
     * 9.918,9.996,10.069,10.140,10.212,10.291,10.374,10.454,
     * 9.869,9.941,10.001,10.052,10.101,10.156,10.213,10.267,
     * 9.817,9.894,9.958,10.007,10.048,10.093,10.133,10.164,
     * 9.592,9.694,9.776,9.846,9.908,9.962,10.010,10.052,
     * 9.321,9.406,9.470,9.518,9.554,9.582,9.600,9.625,
     * 9.248,9.340,9.404,9.453,9.489,9.515,9.532,9.542,
     * 8.949,9.036,9.099,9.149,9.184,9.212,9.229,9.238/
C
      DATA ((EDEN(I,5,6,L,3),I=33,40),L=1,NZEN)/
     * 10.856,10.958,11.038,11.097,11.136,11.160,11.173,11.178,
     * 10.744,10.855,10.946,11.015,11.065,11.098,11.116,11.126,
     * 10.514,10.639,10.752,10.849,10.928,10.984,11.023,11.047,
     * 10.198,10.328,10.448,10.563,10.671,10.764,10.834,10.887,
     * 9.890,9.977,10.064,10.151,10.236,10.323,10.413,10.498,
     * 9.834,9.912,9.979,10.038,10.096,10.155,10.217,10.283,
     * 9.776,9.860,9.928,9.979,10.023,10.063,10.098,10.132,
     * 9.563,9.673,9.761,9.832,9.892,9.943,9.988,10.024,
     * 9.314,9.407,9.475,9.524,9.559,9.582,9.594,9.613,
     * 9.238,9.337,9.408,9.459,9.492,9.514,9.524,9.528,
     * 8.932,9.031,9.102,9.155,9.189,9.212,9.222,9.229/
C
      DATA ((EDEN(I,1,7,L,3),I=33,40),L=1,NZEN)/
     * 10.572,10.693,10.803,10.904,10.986,11.052,11.102,11.136,
     * 10.500,10.623,10.738,10.842,10.933,11.006,11.062,11.099,
     * 10.376,10.495,10.610,10.724,10.826,10.913,10.979,11.029,
     * 10.162,10.268,10.370,10.476,10.585,10.691,10.785,10.862,
     * 9.986,10.047,10.107,10.164,10.226,10.290,10.363,10.447,
     * 9.958,10.018,10.068,10.112,10.157,10.196,10.234,10.282,
     * 9.906,9.969,10.023,10.072,10.116,10.152,10.184,10.222,
     * 9.658,9.743,9.813,9.876,9.934,9.990,10.042,10.088,
     * 9.387,9.456,9.508,9.551,9.589,9.620,9.645,9.677,
     * 9.319,9.388,9.441,9.484,9.520,9.551,9.575,9.594,
     * 9.042,9.102,9.146,9.187,9.222,9.254,9.280,9.302/
C
      DATA ((EDEN(I,2,7,L,3),I=33,40),L=1,NZEN)/
     * 10.489,10.608,10.724,10.835,10.932,11.012,11.074,11.120,
     * 10.479,10.604,10.722,10.835,10.931,11.009,11.064,11.104,
     * 10.378,10.498,10.613,10.727,10.830,10.916,10.981,11.026,
     * 10.181,10.286,10.389,10.491,10.595,10.695,10.782,10.854,
     * 9.978,10.045,10.105,10.163,10.223,10.290,10.361,10.435,
     * 9.944,9.999,10.052,10.098,10.142,10.191,10.240,10.289,
     * 9.896,9.959,10.016,10.061,10.103,10.146,10.186,10.219,
     * 9.668,9.752,9.821,9.882,9.941,9.995,10.045,10.091,
     * 9.325,9.398,9.455,9.501,9.541,9.574,9.601,9.639,
     * 9.264,9.342,9.401,9.447,9.488,9.523,9.552,9.573,
     * 9.004,9.073,9.125,9.170,9.206,9.239,9.265,9.284/
C
      DATA ((EDEN(I,3,7,L,3),I=33,40),L=1,NZEN)/
     * 10.547,10.665,10.776,10.878,10.966,11.037,11.087,11.127,
     * 10.497,10.620,10.736,10.845,10.938,11.010,11.061,11.100,
     * 10.390,10.511,10.627,10.739,10.839,10.920,10.982,11.027,
     * 10.192,10.302,10.408,10.514,10.620,10.717,10.800,10.863,
     * 9.971,10.039,10.108,10.169,10.234,10.311,10.391,10.468,
     * 9.932,9.995,10.046,10.091,10.136,10.188,10.242,10.285,
     * 9.880,9.945,10.002,10.046,10.087,10.134,10.177,10.210,
     * 9.653,9.743,9.815,9.880,9.941,9.995,10.045,10.090,
     * 9.311,9.391,9.451,9.499,9.539,9.572,9.597,9.631,
     * 9.238,9.322,9.382,9.431,9.470,9.502,9.528,9.545,
     * 8.982,9.058,9.116,9.161,9.196,9.224,9.244,9.258/
C
      DATA ((EDEN(I,4,7,L,3),I=33,40),L=1,NZEN)/
     * 10.730,10.841,10.934,11.009,11.064,11.107,11.135,11.154,
     * 10.599,10.716,10.822,10.912,10.985,11.038,11.078,11.103,
     * 10.429,10.552,10.668,10.777,10.868,10.941,10.994,11.029,
     * 10.183,10.304,10.416,10.527,10.635,10.731,10.810,10.870,
     * 9.926,10.003,10.082,10.151,10.227,10.304,10.386,10.467,
     * 9.875,9.945,10.007,10.058,10.109,10.161,10.220,10.275,
     * 9.823,9.899,9.962,10.009,10.051,10.095,10.135,10.166,
     * 9.596,9.698,9.781,9.850,9.911,9.965,10.013,10.054,
     * 9.305,9.392,9.457,9.506,9.544,9.572,9.591,9.617,
     * 9.233,9.324,9.391,9.441,9.478,9.505,9.523,9.533,
     * 8.931,9.023,9.088,9.138,9.173,9.200,9.219,9.230/
C
      DATA ((EDEN(I,5,7,L,3),I=33,40),L=1,NZEN)/
     * 10.884,10.982,11.055,11.109,11.143,11.163,11.173,11.178,
     * 10.774,10.880,10.967,11.030,11.073,11.101,11.120,11.126,
     * 10.544,10.666,10.775,10.869,10.941,10.993,11.028,11.047,
     * 10.223,10.353,10.470,10.584,10.689,10.776,10.842,10.892,
     * 9.900,9.991,10.078,10.166,10.250,10.337,10.426,10.508,
     * 9.842,9.919,9.986,10.048,10.105,10.167,10.227,10.291,
     * 9.784,9.867,9.934,9.984,10.030,10.065,10.104,10.137,
     * 9.574,9.683,9.770,9.841,9.901,9.951,9.995,10.030,
     * 9.294,9.389,9.459,9.510,9.546,9.570,9.583,9.603,
     * 9.216,9.317,9.390,9.444,9.480,9.502,9.514,9.518,
     * 8.906,9.011,9.084,9.140,9.176,9.198,9.212,9.219/
C
      DATA ((EDEN(I,1,8,L,3),I=33,40),L=1,NZEN)/
     * 10.560,10.679,10.790,10.888,10.972,11.040,11.091,11.126,
     * 10.489,10.608,10.722,10.828,10.918,10.993,11.050,11.088,
     * 10.364,10.482,10.595,10.707,10.811,10.898,10.967,11.017,
     * 10.156,10.259,10.359,10.460,10.567,10.672,10.767,10.845,
     * 9.990,10.051,10.107,10.164,10.224,10.283,10.350,10.430,
     * 9.962,10.022,10.071,10.118,10.162,10.199,10.236,10.280,
     * 9.908,9.973,10.027,10.075,10.121,10.157,10.189,10.227,
     * 9.661,9.745,9.817,9.880,9.938,9.993,10.045,10.091,
     * 9.392,9.462,9.514,9.558,9.596,9.628,9.654,9.687,
     * 9.324,9.394,9.447,9.490,9.528,9.559,9.584,9.603,
     * 9.047,9.109,9.152,9.192,9.229,9.261,9.288,9.311/
C
      DATA ((EDEN(I,2,8,L,3),I=33,40),L=1,NZEN)/
     * 10.494,10.613,10.725,10.832,10.929,11.008,11.070,11.113,
     * 10.470,10.592,10.709,10.820,10.916,10.995,11.052,11.096,
     * 10.361,10.478,10.590,10.703,10.805,10.895,10.963,11.014,
     * 10.163,10.265,10.363,10.460,10.562,10.662,10.752,10.827,
     * 9.988,10.045,10.106,10.158,10.216,10.275,10.339,10.408,
     * 9.949,10.008,10.059,10.102,10.148,10.191,10.238,10.281,
     * 9.899,9.961,10.017,10.061,10.103,10.143,10.184,10.217,
     * 9.670,9.754,9.823,9.885,9.943,9.998,10.049,10.094,
     * 9.343,9.416,9.472,9.518,9.558,9.592,9.619,9.656,
     * 9.282,9.358,9.416,9.464,9.504,9.538,9.567,9.588,
     * 9.015,9.082,9.134,9.177,9.214,9.246,9.273,9.293/
C
      DATA ((EDEN(I,3,8,L,3),I=33,40),L=1,NZEN)/
     * 10.540,10.654,10.762,10.864,10.952,11.023,11.079,11.117,
     * 10.477,10.597,10.711,10.819,10.913,10.990,11.049,11.089,
     * 10.366,10.481,10.593,10.703,10.804,10.892,10.960,11.010,
     * 10.171,10.275,10.373,10.474,10.577,10.676,10.763,10.834,
     * 9.981,10.044,10.106,10.162,10.220,10.289,10.360,10.433,
     * 9.944,10.001,10.055,10.096,10.137,10.189,10.240,10.286,
     * 9.897,9.960,10.016,10.058,10.097,10.143,10.186,10.215,
     * 9.662,9.750,9.822,9.886,9.946,10.000,10.049,10.093,
     * 9.336,9.414,9.473,9.521,9.561,9.593,9.618,9.651,
     * 9.264,9.345,9.406,9.453,9.492,9.524,9.549,9.566,
     * 9.005,9.081,9.136,9.180,9.216,9.243,9.264,9.277/
C
      DATA ((EDEN(I,4,8,L,3),I=33,40),L=1,NZEN)/
     * 10.684,10.792,10.887,10.968,11.032,11.080,11.117,11.140,
     * 10.556,10.668,10.773,10.866,10.946,11.008,11.054,11.085,
     * 10.396,10.508,10.618,10.723,10.818,10.898,10.959,11.005,
     * 10.164,10.273,10.376,10.479,10.583,10.682,10.767,10.837,
     * 9.950,10.018,10.084,10.144,10.210,10.278,10.354,10.433,
     * 9.909,9.973,10.030,10.074,10.117,10.165,10.213,10.261,
     * 9.854,9.925,9.984,10.031,10.070,10.109,10.145,10.173,
     * 9.625,9.722,9.801,9.867,9.926,9.977,10.023,10.061,
     * 9.341,9.425,9.488,9.536,9.573,9.600,9.619,9.644,
     * 9.267,9.355,9.421,9.470,9.506,9.534,9.552,9.562,
     * 8.968,9.054,9.117,9.165,9.202,9.228,9.246,9.257/
C
      DATA ((EDEN(I,5,8,L,3),I=33,40),L=1,NZEN)/
     * 10.809,10.909,10.988,11.053,11.098,11.132,11.149,11.162,
     * 10.696,10.801,10.891,10.966,11.022,11.061,11.090,11.108,
     * 10.510,10.620,10.722,10.814,10.891,10.951,10.995,11.024,
     * 10.193,10.307,10.413,10.517,10.618,10.712,10.789,10.851,
     * 9.938,10.010,10.080,10.152,10.220,10.292,10.368,10.448,
     * 9.888,9.956,10.011,10.062,10.107,10.156,10.204,10.253,
     * 9.832,9.907,9.969,10.018,10.055,10.089,10.118,10.145,
     * 9.611,9.714,9.797,9.864,9.920,9.966,10.006,10.038,
     * 9.347,9.438,9.506,9.556,9.591,9.614,9.627,9.646,
     * 9.271,9.368,9.439,9.491,9.526,9.549,9.561,9.566,
     * 8.960,9.059,9.131,9.182,9.217,9.242,9.255,9.261/
C
      DATA ((EDEN(I,1,9,L,3),I=33,40),L=1,NZEN)/
     * 10.532,10.650,10.760,10.860,10.947,11.017,11.072,11.112,
     * 10.460,10.580,10.692,10.799,10.892,10.969,11.029,11.073,
     * 10.340,10.454,10.565,10.676,10.780,10.870,10.943,10.998,
     * 10.139,10.239,10.333,10.430,10.534,10.638,10.734,10.818,
     * 9.994,10.054,10.110,10.161,10.216,10.271,10.331,10.404,
     * 9.967,10.026,10.077,10.124,10.167,10.204,10.238,10.280,
     * 9.913,9.977,10.034,10.082,10.127,10.165,10.199,10.234,
     * 9.663,9.749,9.820,9.883,9.941,9.997,10.050,10.095,
     * 9.403,9.473,9.526,9.570,9.609,9.643,9.670,9.704,
     * 9.334,9.405,9.458,9.503,9.542,9.574,9.600,9.620,
     * 9.058,9.117,9.165,9.205,9.242,9.275,9.303,9.327/
C
      DATA ((EDEN(I,2,9,L,3),I=33,40),L=1,NZEN)/
     * 10.503,10.620,10.731,10.834,10.927,11.003,11.062,11.106,
     * 10.454,10.572,10.687,10.797,10.894,10.974,11.035,11.082,
     * 10.328,10.441,10.549,10.659,10.764,10.856,10.932,10.990,
     * 10.133,10.227,10.318,10.408,10.505,10.603,10.698,10.781,
     * 9.998,10.055,10.107,10.157,10.207,10.258,10.313,10.370,
     * 9.965,10.022,10.071,10.116,10.155,10.194,10.237,10.273,
     * 9.911,9.972,10.027,10.073,10.116,10.155,10.192,10.225,
     * 9.668,9.752,9.822,9.884,9.943,9.997,10.048,10.093,
     * 9.382,9.453,9.508,9.553,9.593,9.626,9.652,9.686,
     * 9.313,9.386,9.441,9.485,9.524,9.557,9.583,9.602,
     * 9.040,9.106,9.153,9.194,9.231,9.262,9.289,9.310/
C
      DATA ((EDEN(I,3,9,L,3),I=33,40),L=1,NZEN)/
     * 10.530,10.641,10.747,10.848,10.935,11.008,11.063,11.106,
     * 10.435,10.547,10.655,10.761,10.859,10.941,11.007,11.058,
     * 10.313,10.421,10.527,10.634,10.739,10.836,10.914,10.976,
     * 10.142,10.234,10.323,10.415,10.509,10.609,10.702,10.785,
     * 10.003,10.059,10.109,10.158,10.209,10.264,10.323,10.385,
     * 9.969,10.026,10.072,10.111,10.151,10.193,10.237,10.275,
     * 9.916,9.975,10.028,10.071,10.109,10.151,10.190,10.219,
     * 9.674,9.759,9.830,9.892,9.951,10.004,10.052,10.095,
     * 9.384,9.458,9.513,9.558,9.595,9.625,9.648,9.679,
     * 9.317,9.393,9.451,9.497,9.534,9.564,9.588,9.604,
     * 9.033,9.104,9.154,9.198,9.232,9.262,9.285,9.303/
C
      DATA ((EDEN(I,4,9,L,3),I=33,40),L=1,NZEN)/
     * 10.611,10.716,10.813,10.901,10.978,11.039,11.087,11.119,
     * 10.502,10.606,10.706,10.800,10.886,10.958,11.014,11.057,
     * 10.343,10.449,10.549,10.651,10.750,10.840,10.913,10.973,
     * 10.143,10.236,10.326,10.417,10.512,10.611,10.705,10.786,
     * 9.995,10.052,10.102,10.150,10.201,10.257,10.318,10.384,
     * 9.959,10.013,10.062,10.099,10.136,10.174,10.213,10.251,
     * 9.907,9.969,10.020,10.062,10.098,10.134,10.165,10.189,
     * 9.665,9.754,9.825,9.887,9.942,9.990,10.032,10.069,
     * 9.402,9.480,9.538,9.583,9.618,9.644,9.662,9.685,
     * 9.333,9.413,9.473,9.520,9.554,9.579,9.596,9.605,
     * 9.037,9.115,9.170,9.216,9.249,9.274,9.290,9.300/
C
      DATA ((EDEN(I,5,9,L,3),I=33,40),L=1,NZEN)/
     * 10.688,10.784,10.875,10.954,11.020,11.074,11.110,11.138,
     * 10.577,10.674,10.768,10.855,10.931,10.993,11.041,11.078,
     * 10.428,10.523,10.616,10.709,10.795,10.871,10.933,10.980,
     * 10.179,10.269,10.356,10.443,10.533,10.625,10.712,10.787,
     * 10.001,10.054,10.105,10.153,10.204,10.256,10.315,10.384,
     * 9.963,10.017,10.064,10.101,10.135,10.166,10.199,10.236,
     * 9.909,9.971,10.021,10.061,10.097,10.123,10.147,10.163,
     * 9.678,9.767,9.840,9.899,9.949,9.990,10.025,10.052,
     * 9.436,9.518,9.580,9.627,9.659,9.681,9.692,9.706,
     * 9.363,9.451,9.516,9.564,9.597,9.618,9.628,9.630,
     * 9.061,9.146,9.211,9.258,9.289,9.312,9.324,9.326/
C
      DATA ((EDEN(I,1,10,L,3),I=33,40),L=1,NZEN)/
     * 10.506,10.624,10.735,10.837,10.928,11.002,11.060,11.105,
     * 10.436,10.554,10.666,10.775,10.870,10.952,11.017,11.065,
     * 10.315,10.428,10.539,10.650,10.756,10.851,10.928,10.988,
     * 10.125,10.222,10.314,10.408,10.509,10.613,10.712,10.800,
     * 9.998,10.058,10.113,10.164,10.216,10.266,10.322,10.389,
     * 9.971,10.033,10.084,10.129,10.172,10.208,10.243,10.282,
     * 9.916,9.981,10.037,10.087,10.132,10.170,10.204,10.240,
     * 9.663,9.749,9.820,9.884,9.943,9.999,10.052,10.098,
     * 9.416,9.486,9.540,9.585,9.624,9.658,9.685,9.719,
     * 9.347,9.418,9.473,9.517,9.556,9.589,9.614,9.634,
     * 9.069,9.132,9.177,9.218,9.256,9.289,9.319,9.342/
C
      DATA ((EDEN(I,2,10,L,3),I=33,40),L=1,NZEN)/
     * 10.527,10.643,10.752,10.852,10.940,11.012,11.070,11.109,
     * 10.443,10.559,10.670,10.776,10.872,10.953,11.017,11.066,
     * 10.314,10.423,10.530,10.640,10.746,10.841,10.921,10.982,
     * 10.126,10.215,10.304,10.397,10.494,10.598,10.698,10.786,
     * 10.005,10.061,10.109,10.158,10.208,10.259,10.315,10.380,
     * 9.978,10.032,10.083,10.126,10.167,10.203,10.238,10.274,
     * 9.921,9.984,10.037,10.081,10.127,10.164,10.198,10.233,
     * 9.664,9.749,9.819,9.881,9.939,9.994,10.045,10.090,
     * 9.428,9.497,9.549,9.593,9.631,9.663,9.688,9.719,
     * 9.360,9.430,9.483,9.526,9.564,9.594,9.618,9.635,
     * 9.077,9.137,9.181,9.221,9.257,9.288,9.315,9.336/
C
      DATA ((EDEN(I,3,10,L,3),I=33,40),L=1,NZEN)/
     * 10.543,10.654,10.759,10.857,10.943,11.014,11.067,11.110,
     * 10.442,10.548,10.652,10.754,10.848,10.928,10.995,11.042,
     * 10.313,10.416,10.517,10.622,10.726,10.823,10.903,10.968,
     * 10.136,10.219,10.304,10.392,10.485,10.584,10.684,10.773,
     * 10.026,10.075,10.119,10.164,10.211,10.263,10.318,10.380,
     * 9.995,10.047,10.091,10.130,10.165,10.203,10.239,10.274,
     * 9.940,9.996,10.045,10.089,10.128,10.165,10.200,10.228,
     * 9.684,9.767,9.835,9.896,9.952,10.004,10.052,10.095,
     * 9.442,9.510,9.561,9.602,9.637,9.665,9.685,9.712,
     * 9.376,9.445,9.497,9.538,9.572,9.598,9.619,9.631,
     * 9.085,9.146,9.190,9.230,9.260,9.286,9.308,9.323/
C
      DATA ((EDEN(I,4,10,L,3),I=33,40),L=1,NZEN)/
     * 10.573,10.675,10.774,10.866,10.949,11.018,11.071,11.112,
     * 10.471,10.569,10.666,10.763,10.854,10.933,10.997,11.045,
     * 10.337,10.429,10.522,10.616,10.711,10.801,10.878,10.943,
     * 10.158,10.235,10.313,10.394,10.483,10.577,10.673,10.760,
     * 10.042,10.089,10.128,10.169,10.213,10.262,10.314,10.377,
     * 10.011,10.058,10.094,10.130,10.161,10.193,10.224,10.259,
     * 9.959,10.010,10.056,10.093,10.125,10.156,10.182,10.202,
     * 9.715,9.793,9.858,9.914,9.963,10.007,10.046,10.080,
     * 9.462,9.530,9.581,9.621,9.651,9.674,9.690,9.710,
     * 9.398,9.468,9.519,9.561,9.590,9.612,9.626,9.633,
     * 9.113,9.176,9.224,9.261,9.290,9.313,9.328,9.337/
C
      DATA ((EDEN(I,5,10,L,3),I=33,40),L=1,NZEN)/
     * 10.616,10.708,10.799,10.885,10.963,11.029,11.082,11.120,
     * 10.513,10.602,10.692,10.782,10.868,10.943,11.006,11.053,
     * 10.377,10.462,10.547,10.635,10.725,10.812,10.887,10.949,
     * 10.198,10.266,10.336,10.410,10.489,10.574,10.662,10.744,
     * 10.072,10.111,10.148,10.184,10.220,10.262,10.309,10.366,
     * 10.041,10.081,10.116,10.143,10.169,10.194,10.216,10.245,
     * 9.992,10.038,10.074,10.107,10.135,10.154,10.173,10.185,
     * 9.759,9.831,9.891,9.941,9.983,10.018,10.048,10.071,
     * 9.503,9.572,9.625,9.665,9.694,9.712,9.722,9.734,
     * 9.440,9.513,9.569,9.609,9.637,9.655,9.663,9.663,
     * 9.162,9.230,9.282,9.322,9.349,9.366,9.375,9.376/
C
      DATA ((EDEN(I,1,11,L,3),I=33,40),L=1,NZEN)/
     * 10.490,10.608,10.720,10.825,10.920,11.000,11.060,11.108,
     * 10.421,10.538,10.652,10.762,10.863,10.949,11.017,11.069,
     * 10.301,10.414,10.524,10.637,10.747,10.846,10.927,10.991,
     * 10.119,10.212,10.306,10.400,10.500,10.607,10.709,10.800,
     * 10.002,10.061,10.113,10.164,10.214,10.264,10.320,10.389,
     * 9.975,10.033,10.087,10.132,10.175,10.211,10.243,10.282,
     * 9.917,9.981,10.037,10.087,10.135,10.172,10.204,10.240,
     * 9.659,9.746,9.819,9.882,9.942,9.998,10.050,10.096,
     * 9.427,9.497,9.551,9.597,9.635,9.669,9.695,9.727,
     * 9.358,9.429,9.484,9.529,9.567,9.599,9.625,9.643,
     * 9.079,9.142,9.189,9.230,9.268,9.302,9.330,9.351/
C
      DATA ((EDEN(I,2,11,L,3),I=33,40),L=1,NZEN)/
     * 10.562,10.677,10.786,10.883,10.967,11.034,11.085,11.124,
     * 10.458,10.571,10.679,10.782,10.875,10.953,11.016,11.062,
     * 10.323,10.433,10.539,10.648,10.754,10.850,10.929,10.991,
     * 10.124,10.212,10.299,10.389,10.487,10.590,10.692,10.785,
     * 10.012,10.070,10.114,10.162,10.207,10.255,10.308,10.372,
     * 9.987,10.043,10.089,10.131,10.171,10.207,10.239,10.273,
     * 9.926,9.988,10.040,10.087,10.131,10.168,10.202,10.234,
     * 9.662,9.747,9.816,9.878,9.936,9.990,10.040,10.086,
     * 9.469,9.535,9.585,9.627,9.662,9.691,9.715,9.742,
     * 9.402,9.470,9.521,9.561,9.594,9.624,9.646,9.661,
     * 9.112,9.168,9.210,9.248,9.279,9.309,9.332,9.351/
C
      DATA ((EDEN(I,3,11,L,3),I=33,40),L=1,NZEN)/
     * 10.584,10.695,10.799,10.893,10.975,11.041,11.090,11.127,
     * 10.481,10.589,10.693,10.793,10.884,10.959,11.019,11.063,
     * 10.349,10.449,10.548,10.648,10.746,10.835,10.910,10.969,
     * 10.160,10.244,10.329,10.416,10.510,10.611,10.708,10.794,
     * 10.044,10.090,10.129,10.173,10.216,10.267,10.320,10.381,
     * 10.012,10.061,10.099,10.137,10.170,10.208,10.243,10.280,
     * 9.958,10.012,10.058,10.100,10.137,10.173,10.205,10.233,
     * 9.698,9.777,9.843,9.902,9.957,10.007,10.054,10.096,
     * 9.483,9.543,9.589,9.626,9.656,9.680,9.697,9.721,
     * 9.428,9.490,9.536,9.572,9.602,9.625,9.642,9.653,
     * 9.139,9.190,9.228,9.260,9.287,9.312,9.330,9.344/
C
      DATA ((EDEN(I,4,11,L,3),I=33,40),L=1,NZEN)/
     * 10.584,10.687,10.786,10.880,10.962,11.031,11.083,11.123,
     * 10.484,10.583,10.681,10.777,10.867,10.946,11.009,11.057,
     * 10.356,10.448,10.538,10.633,10.728,10.817,10.896,10.958,
     * 10.190,10.262,10.333,10.410,10.495,10.584,10.676,10.759,
     * 10.080,10.118,10.153,10.191,10.234,10.281,10.335,10.401,
     * 10.054,10.091,10.122,10.151,10.180,10.208,10.239,10.273,
     * 10.004,10.047,10.083,10.113,10.143,10.169,10.191,10.213,
     * 9.767,9.835,9.892,9.941,9.986,10.026,10.063,10.094,
     * 9.502,9.558,9.601,9.634,9.661,9.680,9.693,9.712,
     * 9.442,9.500,9.543,9.576,9.601,9.619,9.631,9.636,
     * 9.172,9.220,9.255,9.285,9.308,9.326,9.339,9.346/
C
      DATA ((EDEN(I,5,11,L,3),I=33,40),L=1,NZEN)/
     * 10.618,10.709,10.798,10.884,10.964,11.031,11.086,11.124,
     * 10.521,10.607,10.694,10.784,10.870,10.947,11.010,11.058,
     * 10.395,10.473,10.553,10.640,10.729,10.816,10.892,10.956,
     * 10.232,10.292,10.355,10.424,10.497,10.582,10.669,10.752,
     * 10.132,10.157,10.185,10.214,10.241,10.276,10.317,10.367,
     * 10.105,10.132,10.157,10.176,10.196,10.215,10.236,10.258,
     * 10.058,10.091,10.120,10.141,10.161,10.178,10.189,10.200,
     * 9.844,9.899,9.945,9.985,10.019,10.048,10.073,10.092,
     * 9.549,9.603,9.645,9.678,9.701,9.716,9.724,9.736,
     * 9.488,9.544,9.587,9.620,9.642,9.655,9.661,9.661,
     * 9.241,9.292,9.330,9.360,9.381,9.394,9.399,9.399/
C
      DATA ((EDEN(I,1,12,L,3),I=33,40),L=1,NZEN)/
     * 10.484,10.601,10.714,10.823,10.920,11.003,11.068,11.115,
     * 10.415,10.532,10.646,10.760,10.863,10.953,11.025,11.076,
     * 10.295,10.411,10.520,10.635,10.747,10.849,10.935,11.000,
     * 10.116,10.210,10.304,10.400,10.502,10.610,10.716,10.809,
     * 10.002,10.061,10.113,10.164,10.214,10.266,10.324,10.396,
     * 9.975,10.033,10.087,10.132,10.172,10.208,10.243,10.282,
     * 9.915,9.981,10.037,10.087,10.132,10.170,10.201,10.238,
     * 9.655,9.742,9.815,9.879,9.938,9.994,10.046,10.092,
     * 9.434,9.505,9.559,9.604,9.642,9.675,9.700,9.731,
     * 9.364,9.438,9.491,9.537,9.575,9.606,9.630,9.648,
     * 9.086,9.150,9.197,9.238,9.276,9.308,9.335,9.356/
C
      DATA ((EDEN(I,2,12,L,3),I=33,40),L=1,NZEN)/
     * 10.587,10.704,10.811,10.906,10.988,11.055,11.104,11.137,
     * 10.484,10.597,10.705,10.807,10.899,10.974,11.031,11.078,
     * 10.355,10.463,10.571,10.678,10.779,10.870,10.944,11.000,
     * 10.125,10.212,10.302,10.392,10.490,10.594,10.699,10.794,
     * 10.019,10.072,10.115,10.160,10.208,10.256,10.308,10.374,
     * 9.986,10.042,10.084,10.126,10.164,10.200,10.235,10.276,
     * 9.927,9.991,10.042,10.088,10.130,10.167,10.200,10.232,
     * 9.661,9.745,9.813,9.874,9.931,9.985,10.035,10.081,
     * 9.480,9.543,9.590,9.628,9.660,9.686,9.705,9.732,
     * 9.429,9.495,9.544,9.583,9.615,9.641,9.661,9.674,
     * 9.134,9.189,9.229,9.263,9.293,9.320,9.342,9.359/
C
      DATA ((EDEN(I,3,12,L,3),I=33,40),L=1,NZEN)/
     * 10.620,10.731,10.834,10.924,11.002,11.061,11.109,11.141,
     * 10.518,10.626,10.730,10.828,10.913,10.985,11.042,11.082,
     * 10.383,10.484,10.585,10.686,10.781,10.866,10.936,10.990,
     * 10.182,10.267,10.352,10.441,10.537,10.638,10.733,10.817,
     * 10.052,10.097,10.138,10.183,10.230,10.286,10.346,10.417,
     * 10.022,10.066,10.102,10.139,10.175,10.212,10.247,10.284,
     * 9.970,10.018,10.059,10.100,10.137,10.172,10.201,10.230,
     * 9.707,9.783,9.847,9.904,9.957,10.007,10.054,10.096,
     * 9.509,9.565,9.607,9.640,9.668,9.689,9.705,9.727,
     * 9.449,9.506,9.546,9.579,9.605,9.625,9.639,9.647,
     * 9.171,9.214,9.247,9.277,9.301,9.320,9.337,9.350/
C
      DATA ((EDEN(I,4,12,L,3),I=33,40),L=1,NZEN)/
     * 10.613,10.716,10.815,10.906,10.986,11.052,11.102,11.137,
     * 10.514,10.613,10.710,10.806,10.894,10.970,11.029,11.074,
     * 10.385,10.478,10.568,10.664,10.758,10.845,10.920,10.977,
     * 10.214,10.286,10.360,10.439,10.525,10.616,10.705,10.786,
     * 10.097,10.133,10.166,10.205,10.249,10.298,10.354,10.422,
     * 10.074,10.104,10.134,10.161,10.189,10.218,10.250,10.288,
     * 10.029,10.063,10.096,10.124,10.152,10.174,10.196,10.215,
     * 9.800,9.861,9.911,9.957,9.998,10.036,10.071,10.101,
     * 9.520,9.569,9.606,9.635,9.658,9.675,9.686,9.705,
     * 9.466,9.515,9.551,9.580,9.601,9.617,9.627,9.632,
     * 9.205,9.243,9.271,9.297,9.315,9.329,9.340,9.348/
C
      DATA ((EDEN(I,5,12,L,3),I=33,40),L=1,NZEN)/
     * 10.651,10.741,10.830,10.914,10.989,11.052,11.102,11.138,
     * 10.554,10.640,10.727,10.816,10.897,10.970,11.029,11.075,
     * 10.429,10.507,10.589,10.674,10.762,10.844,10.916,10.975,
     * 10.262,10.322,10.387,10.455,10.531,10.615,10.699,10.778,
     * 10.156,10.177,10.202,10.229,10.259,10.295,10.338,10.391,
     * 10.132,10.156,10.173,10.191,10.208,10.226,10.246,10.268,
     * 10.091,10.118,10.139,10.160,10.175,10.190,10.201,10.208,
     * 9.892,9.937,9.975,10.008,10.037,10.063,10.084,10.102,
     * 9.568,9.611,9.646,9.672,9.692,9.705,9.712,9.724,
     * 9.517,9.562,9.595,9.621,9.638,9.649,9.654,9.654,
     * 9.265,9.297,9.324,9.346,9.361,9.370,9.374,9.376/
C
      DATA ((EDEN(I,1,1,L,1),I=41,48),L=1,NZEN)/
     * 10.989,11.021,11.041,11.050,11.060,11.068,11.076,11.087,
     * 10.953,10.986,11.007,11.018,11.026,11.033,11.040,11.047,
     * 10.879,10.924,10.949,10.964,10.972,10.979,10.987,10.995,
     * 10.686,10.764,10.817,10.850,10.868,10.880,10.890,10.902,
     * 10.220,10.309,10.407,10.498,10.573,10.630,10.672,10.703,
     * 10.101,10.126,10.157,10.200,10.268,10.349,10.429,10.501,
     * 10.067,10.076,10.087,10.090,10.101,10.120,10.150,10.201,
     * 9.963,9.974,9.980,9.982,9.982,9.984,9.991,10.001,
     * 9.772,9.774,9.770,9.763,9.749,9.735,9.723,9.712,
     * 9.701,9.695,9.682,9.665,9.645,9.624,9.601,9.581,
     * 9.427,9.423,9.413,9.398,9.379,9.360,9.338,9.317/
C
      DATA ((EDEN(I,2,1,L,1),I=41,48),L=1,NZEN)/
     * 11.003,11.027,11.040,11.049,11.054,11.062,11.069,11.081,
     * 10.945,10.976,10.994,11.007,11.014,11.021,11.031,11.042,
     * 10.873,10.915,10.939,10.953,10.961,10.969,10.978,10.990,
     * 10.668,10.748,10.803,10.840,10.862,10.875,10.885,10.898,
     * 10.195,10.277,10.376,10.472,10.551,10.613,10.660,10.694,
     * 10.097,10.128,10.176,10.246,10.329,10.413,10.483,10.540,
     * 10.067,10.075,10.082,10.087,10.096,10.113,10.139,10.189,
     * 9.953,9.963,9.969,9.972,9.972,9.974,9.979,9.985,
     * 9.769,9.768,9.761,9.750,9.731,9.711,9.693,9.677,
     * 9.723,9.712,9.694,9.671,9.646,9.618,9.590,9.564,
     * 9.425,9.417,9.404,9.383,9.358,9.334,9.307,9.281/
C
      DATA ((EDEN(I,3,1,L,1),I=41,48),L=1,NZEN)/
     * 11.003,11.024,11.035,11.042,11.049,11.056,11.062,11.073,
     * 10.946,10.974,10.990,11.000,11.007,11.013,11.022,11.035,
     * 10.856,10.896,10.922,10.937,10.946,10.954,10.966,10.978,
     * 10.682,10.749,10.796,10.828,10.845,10.859,10.870,10.883,
     * 10.240,10.332,10.429,10.511,10.576,10.626,10.661,10.689,
     * 10.096,10.131,10.180,10.247,10.327,10.403,10.470,10.525,
     * 10.048,10.058,10.066,10.076,10.096,10.128,10.170,10.232,
     * 9.951,9.958,9.962,9.963,9.963,9.964,9.970,9.977,
     * 9.750,9.745,9.735,9.721,9.699,9.678,9.659,9.641,
     * 9.680,9.665,9.644,9.616,9.585,9.553,9.521,9.491,
     * 9.399,9.389,9.371,9.345,9.319,9.288,9.257,9.227/
C
      DATA ((EDEN(I,4,1,L,1),I=41,48),L=1,NZEN)/
     * 10.993,11.016,11.029,11.037,11.037,11.046,11.055,11.066,
     * 10.933,10.963,10.982,10.991,10.996,11.002,11.012,11.024,
     * 10.836,10.881,10.909,10.925,10.935,10.943,10.954,10.967,
     * 10.637,10.709,10.762,10.799,10.822,10.837,10.852,10.866,
     * 10.246,10.330,10.416,10.492,10.552,10.600,10.637,10.665,
     * 10.114,10.152,10.206,10.273,10.342,10.410,10.466,10.514,
     * 10.049,10.060,10.068,10.078,10.097,10.128,10.171,10.226,
     * 9.955,9.961,9.962,9.960,9.957,9.954,9.958,9.961,
     * 9.715,9.711,9.701,9.687,9.664,9.641,9.619,9.599,
     * 9.645,9.631,9.609,9.581,9.548,9.514,9.480,9.448,
     * 9.374,9.361,9.341,9.313,9.281,9.247,9.211,9.177/
C
      DATA ((EDEN(I,5,1,L,1),I=41,48),L=1,NZEN)/
     * 10.986,11.009,11.019,11.026,11.027,11.029,11.040,11.051,
     * 10.924,10.955,10.971,10.979,10.982,10.988,10.996,11.009,
     * 10.824,10.868,10.897,10.912,10.920,10.927,10.936,10.951,
     * 10.618,10.690,10.743,10.779,10.802,10.819,10.832,10.849,
     * 10.220,10.284,10.359,10.431,10.496,10.550,10.594,10.630,
     * 10.116,10.138,10.171,10.216,10.277,10.343,10.405,10.462,
     * 10.067,10.074,10.074,10.075,10.082,10.101,10.125,10.173,
     * 9.974,9.974,9.969,9.960,9.948,9.936,9.932,9.930,
     * 9.727,9.720,9.705,9.686,9.656,9.627,9.599,9.573,
     * 9.646,9.630,9.604,9.571,9.535,9.496,9.457,9.419,
     * 9.379,9.360,9.338,9.304,9.266,9.226,9.182,9.140/
C
      DATA ((EDEN(I,1,2,L,1),I=41,48),L=1,NZEN)/
     * 10.977,11.011,11.031,11.046,11.052,11.064,11.071,11.083,
     * 10.941,10.977,10.998,11.011,11.019,11.026,11.035,11.044,
     * 10.867,10.912,10.939,10.957,10.965,10.973,10.981,10.991,
     * 10.670,10.748,10.802,10.837,10.858,10.873,10.884,10.897,
     * 10.206,10.289,10.383,10.474,10.551,10.611,10.657,10.693,
     * 10.101,10.122,10.148,10.189,10.247,10.325,10.403,10.478,
     * 10.067,10.080,10.087,10.090,10.098,10.113,10.137,10.185,
     * 9.965,9.976,9.983,9.984,9.983,9.983,9.989,9.997,
     * 9.774,9.777,9.774,9.768,9.754,9.741,9.728,9.718,
     * 9.702,9.699,9.687,9.671,9.652,9.632,9.611,9.591,
     * 9.429,9.425,9.418,9.404,9.387,9.367,9.347,9.326/
C
      DATA ((EDEN(I,2,2,L,1),I=41,48),L=1,NZEN)/
     * 10.988,11.014,11.031,11.041,11.050,11.058,11.065,11.077,
     * 10.927,10.961,10.984,10.997,11.007,11.015,11.026,11.039,
     * 10.863,10.907,10.934,10.951,10.960,10.968,10.976,10.987,
     * 10.651,10.733,10.791,10.829,10.853,10.868,10.881,10.892,
     * 10.189,10.266,10.359,10.454,10.533,10.598,10.648,10.684,
     * 10.093,10.124,10.167,10.231,10.312,10.393,10.466,10.526,
     * 10.067,10.075,10.082,10.087,10.093,10.110,10.133,10.178,
     * 9.952,9.963,9.969,9.972,9.972,9.974,9.978,9.984,
     * 9.790,9.788,9.781,9.770,9.752,9.734,9.717,9.702,
     * 9.720,9.711,9.696,9.675,9.650,9.624,9.598,9.573,
     * 9.429,9.422,9.409,9.390,9.368,9.346,9.320,9.295/
C
      DATA ((EDEN(I,3,2,L,1),I=41,48),L=1,NZEN)/
     * 10.986,11.012,11.027,11.038,11.041,11.047,11.058,11.069,
     * 10.925,10.958,10.979,10.992,10.999,11.007,11.018,11.031,
     * 10.827,10.876,10.907,10.927,10.938,10.949,10.960,10.974,
     * 10.653,10.728,10.782,10.818,10.840,10.855,10.867,10.881,
     * 10.218,10.307,10.402,10.488,10.557,10.609,10.649,10.680,
     * 10.092,10.121,10.165,10.227,10.304,10.383,10.452,10.510,
     * 10.047,10.057,10.066,10.076,10.089,10.120,10.157,10.216,
     * 9.949,9.958,9.962,9.965,9.965,9.967,9.973,9.980,
     * 9.754,9.752,9.743,9.731,9.711,9.691,9.673,9.657,
     * 9.699,9.686,9.668,9.643,9.615,9.586,9.556,9.528,
     * 9.405,9.395,9.379,9.356,9.330,9.302,9.273,9.243/
C
      DATA ((EDEN(I,4,2,L,1),I=41,48),L=1,NZEN)/
     * 10.976,11.004,11.020,11.028,11.033,11.037,11.047,11.062,
     * 10.913,10.949,10.971,10.983,10.990,10.997,11.007,11.021,
     * 10.809,10.861,10.894,10.915,10.928,10.937,10.948,10.962,
     * 10.598,10.677,10.738,10.781,10.808,10.829,10.846,10.863,
     * 10.217,10.292,10.377,10.458,10.525,10.579,10.623,10.656,
     * 10.102,10.135,10.181,10.242,10.313,10.381,10.444,10.497,
     * 10.052,10.059,10.067,10.074,10.087,10.114,10.148,10.197,
     * 9.953,9.959,9.960,9.958,9.954,9.951,9.954,9.957,
     * 9.730,9.725,9.715,9.701,9.678,9.656,9.634,9.616,
     * 9.658,9.643,9.621,9.593,9.562,9.530,9.498,9.467,
     * 9.385,9.371,9.353,9.326,9.295,9.263,9.228,9.194/
C
      DATA ((EDEN(I,5,2,L,1),I=41,48),L=1,NZEN)/
     * 10.972,10.998,11.013,11.017,11.023,11.024,11.032,11.047,
     * 10.905,10.942,10.962,10.973,10.978,10.982,10.992,11.005,
     * 10.799,10.850,10.884,10.903,10.914,10.921,10.933,10.947,
     * 10.583,10.661,10.721,10.764,10.792,10.810,10.826,10.844,
     * 10.200,10.257,10.326,10.401,10.473,10.531,10.580,10.621,
     * 10.112,10.131,10.158,10.200,10.260,10.328,10.391,10.452,
     * 10.066,10.069,10.073,10.075,10.079,10.096,10.118,10.163,
     * 9.971,9.972,9.968,9.960,9.948,9.938,9.935,9.934,
     * 9.736,9.729,9.715,9.696,9.668,9.640,9.613,9.590,
     * 9.661,9.644,9.619,9.588,9.552,9.516,9.479,9.444,
     * 9.402,9.384,9.358,9.325,9.286,9.247,9.207,9.166/
C
      DATA ((EDEN(I,1,3,L,1),I=41,48),L=1,NZEN)/
     * 10.967,11.000,11.022,11.037,11.043,11.052,11.063,11.075,
     * 10.932,10.967,10.988,11.001,11.009,11.017,11.027,11.037,
     * 10.856,10.901,10.930,10.946,10.956,10.965,10.974,10.983,
     * 10.656,10.733,10.789,10.825,10.849,10.864,10.876,10.888,
     * 10.196,10.273,10.364,10.453,10.530,10.592,10.642,10.681,
     * 10.101,10.119,10.142,10.175,10.229,10.300,10.379,10.455,
     * 10.067,10.080,10.087,10.090,10.098,10.110,10.131,10.170,
     * 9.964,9.976,9.983,9.984,9.983,9.982,9.987,9.994,
     * 9.771,9.775,9.774,9.769,9.756,9.744,9.733,9.722,
     * 9.698,9.696,9.686,9.674,9.656,9.637,9.618,9.599,
     * 9.426,9.424,9.418,9.406,9.390,9.374,9.354,9.334/
C
      DATA ((EDEN(I,2,3,L,1),I=41,48),L=1,NZEN)/
     * 10.969,11.000,11.020,11.031,11.042,11.049,11.061,11.072,
     * 10.929,10.963,10.984,10.997,11.006,11.013,11.022,11.034,
     * 10.850,10.895,10.924,10.942,10.951,10.960,10.969,10.979,
     * 10.643,10.723,10.780,10.819,10.842,10.860,10.872,10.885,
     * 10.186,10.263,10.353,10.443,10.521,10.585,10.636,10.675,
     * 10.093,10.111,10.137,10.175,10.230,10.304,10.381,10.456,
     * 10.059,10.071,10.081,10.083,10.093,10.108,10.128,10.169,
     * 9.947,9.959,9.967,9.970,9.972,9.973,9.978,9.985,
     * 9.776,9.777,9.773,9.764,9.749,9.734,9.720,9.707,
     * 9.705,9.698,9.686,9.669,9.648,9.626,9.603,9.581,
     * 9.423,9.419,9.410,9.395,9.376,9.355,9.334,9.312/
C
      DATA ((EDEN(I,3,3,L,1),I=41,48),L=1,NZEN)/
     * 10.967,10.998,11.016,11.028,11.032,11.043,11.050,11.065,
     * 10.901,10.941,10.967,10.982,10.991,10.999,11.010,11.024,
     * 10.818,10.870,10.903,10.924,10.936,10.946,10.957,10.969,
     * 10.623,10.705,10.765,10.804,10.831,10.847,10.860,10.874,
     * 10.184,10.260,10.349,10.438,10.514,10.575,10.624,10.662,
     * 10.083,10.113,10.156,10.216,10.292,10.368,10.439,10.498,
     * 10.049,10.059,10.068,10.075,10.086,10.105,10.130,10.171,
     * 9.943,9.953,9.959,9.962,9.962,9.963,9.968,9.974,
     * 9.758,9.756,9.750,9.739,9.721,9.704,9.688,9.674,
     * 9.687,9.678,9.661,9.640,9.615,9.590,9.565,9.541,
     * 9.401,9.394,9.380,9.360,9.338,9.315,9.288,9.264/
C
      DATA ((EDEN(I,4,3,L,1),I=41,48),L=1,NZEN)/
     * 10.963,10.994,11.012,11.021,11.025,11.029,11.042,11.054,
     * 10.896,10.937,10.962,10.974,10.983,10.989,10.999,11.012,
     * 10.788,10.844,10.882,10.906,10.920,10.930,10.941,10.955,
     * 10.582,10.666,10.730,10.775,10.805,10.826,10.843,10.859,
     * 10.199,10.273,10.357,10.439,10.510,10.567,10.613,10.649,
     * 10.094,10.121,10.159,10.214,10.281,10.354,10.420,10.478,
     * 10.045,10.057,10.062,10.069,10.084,10.106,10.137,10.186,
     * 9.950,9.956,9.959,9.958,9.956,9.955,9.960,9.965,
     * 9.733,9.729,9.720,9.708,9.687,9.667,9.650,9.635,
     * 9.665,9.652,9.633,9.608,9.581,9.553,9.525,9.501,
     * 9.385,9.374,9.355,9.330,9.302,9.274,9.244,9.218/
C
      DATA ((EDEN(I,5,3,L,1),I=41,48),L=1,NZEN)/
     * 10.965,10.993,11.008,11.013,11.015,11.016,11.024,11.038,
     * 10.897,10.935,10.957,10.967,10.972,10.975,10.984,10.997,
     * 10.788,10.842,10.877,10.897,10.908,10.916,10.926,10.940,
     * 10.569,10.649,10.712,10.756,10.785,10.804,10.823,10.840,
     * 10.192,10.252,10.327,10.403,10.473,10.533,10.581,10.622,
     * 10.102,10.122,10.154,10.200,10.260,10.328,10.393,10.451,
     * 10.059,10.066,10.067,10.070,10.076,10.092,10.117,10.160,
     * 9.962,9.965,9.962,9.957,9.948,9.942,9.942,9.944,
     * 9.742,9.734,9.721,9.703,9.678,9.653,9.632,9.613,
     * 9.671,9.653,9.629,9.599,9.566,9.534,9.502,9.475,
     * 9.400,9.380,9.355,9.321,9.284,9.251,9.214,9.182/
C
      DATA ((EDEN(I,1,4,L,1),I=41,48),L=1,NZEN)/
     * 10.969,10.999,11.018,11.030,11.039,11.047,11.055,11.066,
     * 10.933,10.965,10.985,10.996,11.003,11.011,11.019,11.030,
     * 10.859,10.901,10.927,10.941,10.950,10.958,10.966,10.977,
     * 10.667,10.738,10.789,10.824,10.844,10.859,10.870,10.883,
     * 10.201,10.283,10.372,10.458,10.533,10.593,10.639,10.677,
     * 10.094,10.116,10.142,10.178,10.232,10.304,10.380,10.454,
     * 10.060,10.073,10.084,10.087,10.094,10.107,10.128,10.167,
     * 9.958,9.971,9.978,9.980,9.978,9.978,9.983,9.991,
     * 9.757,9.762,9.762,9.758,9.747,9.735,9.724,9.715,
     * 9.686,9.683,9.675,9.662,9.646,9.628,9.609,9.592,
     * 9.410,9.410,9.406,9.394,9.379,9.364,9.346,9.328/
C
      DATA ((EDEN(I,2,4,L,1),I=41,48),L=1,NZEN)/
     * 10.961,10.993,11.014,11.027,11.033,11.045,11.052,11.064,
     * 10.924,10.959,10.981,10.993,11.000,11.007,11.017,11.028,
     * 10.852,10.896,10.922,10.938,10.948,10.954,10.962,10.973,
     * 10.662,10.735,10.786,10.819,10.842,10.854,10.867,10.878,
     * 10.162,10.225,10.299,10.382,10.462,10.534,10.594,10.644,
     * 10.085,10.107,10.139,10.183,10.245,10.319,10.395,10.464,
     * 10.047,10.059,10.070,10.080,10.090,10.108,10.132,10.177,
     * 9.947,9.959,9.967,9.970,9.972,9.974,9.981,9.990,
     * 9.744,9.748,9.748,9.743,9.732,9.721,9.711,9.703,
     * 9.673,9.669,9.659,9.645,9.628,9.610,9.593,9.575,
     * 9.403,9.401,9.395,9.382,9.366,9.349,9.332,9.315/
C
      DATA ((EDEN(I,3,4,L,1),I=41,48),L=1,NZEN)/
     * 10.961,10.992,11.012,11.022,11.028,11.035,11.045,11.057,
     * 10.893,10.935,10.962,10.977,10.986,10.995,11.005,11.018,
     * 10.834,10.883,10.912,10.929,10.938,10.944,10.952,10.963,
     * 10.643,10.719,10.773,10.808,10.830,10.843,10.855,10.868,
     * 10.200,10.285,10.377,10.459,10.530,10.585,10.629,10.664,
     * 10.078,10.105,10.142,10.197,10.265,10.340,10.409,10.474,
     * 10.036,10.046,10.059,10.067,10.083,10.110,10.136,10.185,
     * 9.937,9.948,9.954,9.958,9.961,9.964,9.973,9.984,
     * 9.726,9.727,9.723,9.717,9.705,9.693,9.683,9.675,
     * 9.653,9.645,9.632,9.615,9.595,9.577,9.557,9.541,
     * 9.377,9.370,9.358,9.342,9.323,9.306,9.285,9.268/
C
      DATA ((EDEN(I,4,4,L,1),I=41,48),L=1,NZEN)/
     * 10.969,10.995,11.009,11.016,11.019,11.025,11.030,11.041,
     * 10.904,10.940,10.960,10.971,10.977,10.981,10.991,11.002,
     * 10.810,10.861,10.892,10.911,10.920,10.926,10.937,10.948,
     * 10.625,10.700,10.754,10.790,10.813,10.828,10.840,10.854,
     * 10.203,10.280,10.367,10.446,10.514,10.568,10.611,10.647,
     * 10.084,10.112,10.154,10.210,10.277,10.349,10.412,10.472,
     * 10.038,10.046,10.056,10.065,10.081,10.105,10.135,10.186,
     * 9.940,9.948,9.953,9.955,9.955,9.957,9.966,9.976,
     * 9.715,9.713,9.707,9.698,9.683,9.669,9.658,9.651,
     * 9.645,9.634,9.618,9.597,9.575,9.555,9.535,9.519,
     * 9.359,9.348,9.331,9.310,9.287,9.266,9.245,9.227/
C
      DATA ((EDEN(I,5,4,L,1),I=41,48),L=1,NZEN)/
     * 10.982,11.001,11.009,11.010,11.009,11.007,11.015,11.026,
     * 10.919,10.947,10.961,10.967,10.967,10.968,10.973,10.984,
     * 10.821,10.862,10.887,10.901,10.906,10.910,10.918,10.930,
     * 10.613,10.685,10.736,10.771,10.792,10.807,10.821,10.836,
     * 10.207,10.279,10.359,10.433,10.495,10.549,10.590,10.627,
     * 10.092,10.119,10.157,10.211,10.276,10.341,10.403,10.462,
     * 10.047,10.051,10.057,10.061,10.073,10.098,10.126,10.175,
     * 9.948,9.952,9.951,9.950,9.946,9.945,9.953,9.962,
     * 9.720,9.713,9.702,9.689,9.669,9.652,9.639,9.630,
     * 9.647,9.629,9.607,9.581,9.554,9.530,9.508,9.489,
     * 9.357,9.340,9.318,9.292,9.264,9.238,9.214,9.196/
C
      DATA ((EDEN(I,1,5,L,1),I=41,48),L=1,NZEN)/
     * 10.983,11.008,11.023,11.033,11.039,11.047,11.055,11.066,
     * 10.948,10.975,10.990,10.999,11.006,11.012,11.019,11.027,
     * 10.879,10.913,10.934,10.946,10.953,10.959,10.966,10.975,
     * 10.699,10.764,10.807,10.835,10.851,10.861,10.870,10.883,
     * 10.230,10.322,10.414,10.497,10.564,10.615,10.654,10.685,
     * 10.091,10.116,10.151,10.197,10.263,10.339,10.414,10.484,
     * 10.049,10.066,10.073,10.080,10.091,10.110,10.141,10.190,
     * 9.950,9.962,9.969,9.971,9.972,9.973,9.981,9.991,
     * 9.736,9.741,9.741,9.738,9.726,9.716,9.706,9.698,
     * 9.664,9.661,9.652,9.637,9.621,9.603,9.585,9.567,
     * 9.390,9.388,9.383,9.370,9.355,9.340,9.322,9.304/
C
      DATA ((EDEN(I,2,5,L,1),I=41,48),L=1,NZEN)/
     * 10.971,11.002,11.019,11.031,11.037,11.045,11.052,11.064,
     * 10.956,10.982,10.996,11.004,11.007,11.011,11.016,11.024,
     * 10.876,10.911,10.931,10.944,10.948,10.955,10.961,10.970,
     * 10.675,10.740,10.787,10.817,10.837,10.849,10.862,10.873,
     * 10.196,10.273,10.358,10.438,10.510,10.570,10.619,10.660,
     * 10.081,10.116,10.160,10.220,10.291,10.365,10.434,10.497,
     * 10.031,10.047,10.062,10.072,10.090,10.119,10.152,10.207,
     * 9.936,9.949,9.957,9.962,9.966,9.971,9.983,9.996,
     * 9.707,9.713,9.715,9.713,9.705,9.698,9.692,9.688,
     * 9.632,9.629,9.622,9.610,9.595,9.581,9.565,9.552,
     * 9.374,9.369,9.363,9.349,9.333,9.319,9.304,9.288/
C
      DATA ((EDEN(I,3,5,L,1),I=41,48),L=1,NZEN)/
     * 10.972,11.001,11.016,11.024,11.028,11.035,11.041,11.052,
     * 10.932,10.962,10.980,10.990,10.993,10.998,11.005,11.014,
     * 10.870,10.906,10.924,10.935,10.939,10.943,10.949,10.958,
     * 10.683,10.744,10.786,10.814,10.829,10.841,10.851,10.862,
     * 10.220,10.306,10.390,10.466,10.530,10.582,10.624,10.660,
     * 10.076,10.117,10.176,10.243,10.315,10.387,10.450,10.505,
     * 10.026,10.041,10.054,10.067,10.083,10.108,10.136,10.181,
     * 9.923,9.936,9.945,9.952,9.957,9.963,9.975,9.988,
     * 9.685,9.690,9.692,9.691,9.683,9.676,9.670,9.668,
     * 9.622,9.618,9.609,9.597,9.583,9.571,9.559,9.548,
     * 9.344,9.337,9.324,9.309,9.292,9.278,9.262,9.248/
C
      DATA ((EDEN(I,4,5,L,1),I=41,48),L=1,NZEN)/
     * 10.990,11.005,11.014,11.016,11.016,11.017,11.024,11.034,
     * 10.931,10.955,10.968,10.973,10.975,10.977,10.984,10.995,
     * 10.858,10.892,10.911,10.920,10.924,10.926,10.933,10.943,
     * 10.681,10.739,10.779,10.803,10.817,10.827,10.835,10.848,
     * 10.241,10.327,10.408,10.480,10.536,10.582,10.619,10.652,
     * 10.082,10.119,10.173,10.234,10.301,10.369,10.429,10.484,
     * 10.023,10.036,10.050,10.064,10.085,10.117,10.152,10.203,
     * 9.926,9.935,9.942,9.948,9.952,9.958,9.972,9.988,
     * 9.676,9.677,9.675,9.671,9.662,9.656,9.652,9.652,
     * 9.601,9.592,9.578,9.563,9.547,9.534,9.523,9.515,
     * 9.305,9.292,9.278,9.261,9.245,9.232,9.216,9.207/
C
      DATA ((EDEN(I,5,5,L,1),I=41,48),L=1,NZEN)/
     * 11.005,11.010,11.010,11.006,11.001,10.998,11.002,11.011,
     * 10.953,10.964,10.966,10.964,10.960,10.959,10.963,10.974,
     * 10.870,10.892,10.902,10.904,10.903,10.903,10.909,10.920,
     * 10.689,10.741,10.774,10.794,10.804,10.811,10.819,10.832,
     * 10.262,10.346,10.421,10.485,10.534,10.575,10.608,10.639,
     * 10.096,10.142,10.198,10.259,10.324,10.384,10.437,10.487,
     * 10.027,10.040,10.050,10.064,10.084,10.117,10.155,10.208,
     * 9.934,9.939,9.942,9.944,9.946,9.950,9.964,9.981,
     * 9.665,9.662,9.656,9.650,9.639,9.631,9.628,9.629,
     * 9.586,9.571,9.554,9.534,9.516,9.502,9.491,9.485,
     * 9.289,9.276,9.259,9.241,9.223,9.210,9.199,9.193/
C
      DATA ((EDEN(I,1,6,L,1),I=41,48),L=1,NZEN)/
     * 10.996,11.018,11.031,11.042,11.047,11.052,11.059,11.066,
     * 10.962,10.985,10.998,11.005,11.010,11.015,11.021,11.027,
     * 10.898,10.926,10.943,10.953,10.958,10.963,10.969,10.975,
     * 10.731,10.788,10.824,10.846,10.858,10.866,10.875,10.884,
     * 10.267,10.367,10.459,10.535,10.595,10.638,10.670,10.695,
     * 10.091,10.122,10.169,10.231,10.305,10.382,10.452,10.514,
     * 10.042,10.055,10.066,10.077,10.091,10.120,10.162,10.220,
     * 9.942,9.954,9.961,9.964,9.966,9.970,9.981,9.992,
     * 9.718,9.722,9.722,9.718,9.707,9.697,9.687,9.680,
     * 9.645,9.640,9.629,9.615,9.596,9.578,9.559,9.540,
     * 9.372,9.369,9.361,9.347,9.332,9.315,9.296,9.278/
C
      DATA ((EDEN(I,2,6,L,1),I=41,48),L=1,NZEN)/
     * 10.984,11.012,11.029,11.036,11.046,11.049,11.057,11.064,
     * 10.970,10.992,11.003,11.009,11.012,11.015,11.018,11.022,
     * 10.898,10.925,10.940,10.949,10.953,10.958,10.962,10.969,
     * 10.717,10.772,10.808,10.831,10.845,10.853,10.863,10.875,
     * 10.243,10.329,10.415,10.487,10.550,10.600,10.641,10.674,
     * 10.087,10.134,10.196,10.266,10.341,10.411,10.472,10.526,
     * 10.034,10.050,10.065,10.075,10.087,10.110,10.135,10.181,
     * 9.927,9.942,9.951,9.958,9.962,9.967,9.978,9.990,
     * 9.681,9.690,9.695,9.698,9.691,9.684,9.679,9.676,
     * 9.627,9.626,9.620,9.611,9.599,9.587,9.575,9.564,
     * 9.352,9.349,9.340,9.328,9.312,9.298,9.283,9.270/
C
      DATA ((EDEN(I,3,6,L,1),I=41,48),L=1,NZEN)/
     * 10.988,11.011,11.023,11.029,11.032,11.035,11.045,11.052,
     * 10.965,10.985,10.994,10.998,11.000,11.001,11.005,11.011,
     * 10.894,10.921,10.934,10.940,10.942,10.944,10.949,10.958,
     * 10.728,10.777,10.808,10.825,10.835,10.843,10.852,10.864,
     * 10.281,10.369,10.449,10.515,10.567,10.607,10.642,10.671,
     * 10.079,10.122,10.176,10.240,10.307,10.375,10.438,10.495,
     * 10.012,10.028,10.050,10.066,10.090,10.122,10.159,10.212,
     * 9.914,9.927,9.937,9.946,9.954,9.964,9.980,9.997,
     * 9.661,9.668,9.672,9.674,9.670,9.668,9.668,9.670,
     * 9.583,9.581,9.574,9.563,9.553,9.545,9.537,9.531,
     * 9.303,9.292,9.276,9.259,9.242,9.231,9.216,9.207/
C
      DATA ((EDEN(I,4,6,L,1),I=41,48),L=1,NZEN)/
     * 11.002,11.011,11.015,11.015,11.014,11.014,11.020,11.029,
     * 10.956,10.970,10.977,10.977,10.977,10.977,10.984,10.993,
     * 10.889,10.911,10.921,10.923,10.924,10.925,10.930,10.940,
     * 10.727,10.771,10.798,10.814,10.823,10.829,10.835,10.847,
     * 10.287,10.369,10.442,10.500,10.549,10.589,10.622,10.653,
     * 10.097,10.152,10.215,10.283,10.347,10.408,10.460,10.510,
     * 10.013,10.030,10.049,10.066,10.091,10.128,10.167,10.221,
     * 9.919,9.929,9.937,9.944,9.951,9.959,9.977,9.996,
     * 9.637,9.641,9.643,9.644,9.640,9.639,9.641,9.647,
     * 9.556,9.549,9.540,9.526,9.517,9.510,9.504,9.502,
     * 9.263,9.252,9.241,9.226,9.215,9.209,9.202,9.200/
C
      DATA ((EDEN(I,5,6,L,1),I=41,48),L=1,NZEN)/
     * 11.013,11.010,11.005,10.999,10.993,10.990,10.993,11.002,
     * 10.966,10.967,10.964,10.958,10.953,10.950,10.955,10.966,
     * 10.895,10.905,10.909,10.906,10.902,10.901,10.907,10.916,
     * 10.738,10.773,10.793,10.802,10.806,10.809,10.816,10.828,
     * 10.336,10.412,10.475,10.523,10.560,10.591,10.618,10.644,
     * 10.130,10.187,10.249,10.310,10.365,10.417,10.463,10.506,
     * 10.016,10.033,10.052,10.073,10.104,10.143,10.187,10.242,
     * 9.925,9.932,9.937,9.942,9.948,9.956,9.976,9.997,
     * 9.609,9.611,9.611,9.613,9.608,9.609,9.614,9.624,
     * 9.522,9.510,9.498,9.483,9.473,9.468,9.464,9.467,
     * 9.230,9.219,9.207,9.196,9.188,9.186,9.185,9.190/
C
      DATA ((EDEN(I,1,7,L,1),I=41,48),L=1,NZEN)/
     * 11.000,11.021,11.032,11.042,11.047,11.052,11.059,11.066,
     * 10.966,10.988,11.001,11.009,11.013,11.018,11.022,11.027,
     * 10.902,10.931,10.946,10.955,10.960,10.965,10.970,10.976,
     * 10.740,10.794,10.827,10.849,10.859,10.868,10.876,10.886,
     * 10.278,10.378,10.471,10.546,10.601,10.642,10.673,10.696,
     * 10.091,10.126,10.175,10.241,10.316,10.391,10.460,10.519,
     * 10.038,10.055,10.066,10.077,10.094,10.123,10.167,10.228,
     * 9.940,9.952,9.960,9.963,9.965,9.970,9.981,9.993,
     * 9.713,9.717,9.717,9.713,9.702,9.692,9.683,9.676,
     * 9.639,9.635,9.624,9.609,9.590,9.571,9.552,9.534,
     * 9.366,9.363,9.356,9.341,9.326,9.309,9.290,9.273/
C
      DATA ((EDEN(I,2,7,L,1),I=41,48),L=1,NZEN)/
     * 10.989,11.016,11.031,11.041,11.046,11.054,11.057,11.064,
     * 10.974,10.995,11.005,11.011,11.014,11.016,11.019,11.022,
     * 10.902,10.928,10.943,10.951,10.955,10.958,10.964,10.971,
     * 10.725,10.778,10.812,10.833,10.846,10.855,10.865,10.877,
     * 10.255,10.341,10.425,10.497,10.556,10.606,10.644,10.676,
     * 10.094,10.141,10.204,10.275,10.349,10.417,10.478,10.530,
     * 10.030,10.046,10.061,10.072,10.090,10.113,10.142,10.187,
     * 9.926,9.941,9.951,9.957,9.961,9.967,9.978,9.991,
     * 9.673,9.683,9.689,9.692,9.685,9.680,9.675,9.673,
     * 9.618,9.618,9.613,9.604,9.593,9.581,9.569,9.558,
     * 9.345,9.340,9.333,9.320,9.304,9.292,9.278,9.265/
C
      DATA ((EDEN(I,3,7,L,1),I=41,48),L=1,NZEN)/
     * 10.991,11.014,11.025,11.033,11.032,11.039,11.045,11.052,
     * 10.967,10.987,10.996,11.000,11.001,11.003,11.007,11.012,
     * 10.897,10.922,10.935,10.941,10.943,10.945,10.950,10.959,
     * 10.734,10.780,10.810,10.827,10.837,10.843,10.852,10.864,
     * 10.294,10.380,10.457,10.520,10.570,10.611,10.643,10.672,
     * 10.082,10.128,10.184,10.247,10.315,10.381,10.443,10.498,
     * 10.012,10.027,10.049,10.066,10.090,10.125,10.165,10.218,
     * 9.913,9.926,9.936,9.946,9.954,9.964,9.982,9.999,
     * 9.653,9.660,9.665,9.669,9.666,9.665,9.666,9.669,
     * 9.575,9.571,9.565,9.556,9.547,9.541,9.534,9.530,
     * 9.295,9.284,9.270,9.254,9.236,9.224,9.212,9.204/
C
      DATA ((EDEN(I,4,7,L,1),I=41,48),L=1,NZEN)/
     * 11.003,11.012,11.014,11.014,11.013,11.015,11.021,11.030,
     * 10.958,10.970,10.975,10.977,10.976,10.979,10.984,10.994,
     * 10.891,10.912,10.920,10.924,10.924,10.926,10.932,10.941,
     * 10.732,10.773,10.800,10.814,10.821,10.829,10.836,10.849,
     * 10.298,10.378,10.446,10.504,10.551,10.590,10.624,10.655,
     * 10.104,10.158,10.223,10.287,10.351,10.410,10.464,10.512,
     * 10.012,10.029,10.048,10.066,10.092,10.128,10.170,10.226,
     * 9.918,9.928,9.936,9.943,9.950,9.959,9.978,9.998,
     * 9.629,9.634,9.637,9.639,9.636,9.637,9.641,9.649,
     * 9.547,9.540,9.532,9.521,9.513,9.507,9.503,9.504,
     * 9.253,9.245,9.235,9.222,9.211,9.207,9.203,9.202/
C
      DATA ((EDEN(I,5,7,L,1),I=41,48),L=1,NZEN)/
     * 11.010,11.008,11.003,10.996,10.991,10.989,10.993,11.003,
     * 10.964,10.964,10.962,10.955,10.951,10.950,10.956,10.967,
     * 10.894,10.904,10.906,10.904,10.900,10.900,10.907,10.918,
     * 10.740,10.773,10.791,10.800,10.804,10.808,10.816,10.829,
     * 10.342,10.415,10.475,10.521,10.559,10.589,10.618,10.645,
     * 10.133,10.189,10.251,10.309,10.365,10.414,10.462,10.508,
     * 10.015,10.032,10.055,10.073,10.104,10.143,10.188,10.242,
     * 9.926,9.933,9.939,9.944,9.949,9.957,9.978,10.000,
     * 9.600,9.603,9.605,9.609,9.607,9.609,9.616,9.628,
     * 9.512,9.502,9.491,9.479,9.471,9.467,9.467,9.472,
     * 9.221,9.210,9.201,9.191,9.186,9.186,9.189,9.196/
C
      DATA ((EDEN(I,1,8,L,1),I=41,48),L=1,NZEN)/
     * 10.992,11.015,11.029,11.037,11.043,11.052,11.059,11.066,
     * 10.958,10.981,10.995,11.004,11.010,11.015,11.021,11.028,
     * 10.892,10.923,10.940,10.950,10.956,10.962,10.968,10.976,
     * 10.721,10.780,10.818,10.842,10.854,10.865,10.874,10.883,
     * 10.255,10.352,10.443,10.521,10.582,10.628,10.663,10.690,
     * 10.091,10.122,10.163,10.218,10.288,10.364,10.435,10.498,
     * 10.045,10.058,10.070,10.080,10.091,10.116,10.153,10.207,
     * 9.946,9.959,9.966,9.968,9.969,9.972,9.981,9.992,
     * 9.724,9.729,9.729,9.726,9.715,9.705,9.696,9.688,
     * 9.651,9.648,9.638,9.623,9.606,9.588,9.570,9.552,
     * 9.377,9.376,9.369,9.357,9.341,9.325,9.307,9.290/
C
      DATA ((EDEN(I,2,8,L,1),I=41,48),L=1,NZEN)/
     * 10.984,11.011,11.028,11.036,11.042,11.049,11.057,11.068,
     * 10.966,10.990,11.001,11.008,11.012,11.016,11.019,11.025,
     * 10.889,10.920,10.937,10.946,10.951,10.957,10.964,10.972,
     * 10.696,10.756,10.797,10.823,10.839,10.852,10.863,10.877,
     * 10.220,10.300,10.384,10.461,10.527,10.583,10.628,10.666,
     * 10.087,10.125,10.177,10.240,10.313,10.385,10.451,10.508,
     * 10.026,10.043,10.061,10.075,10.093,10.126,10.165,10.222,
     * 9.933,9.947,9.956,9.964,9.969,9.977,9.990,10.005,
     * 9.691,9.701,9.706,9.709,9.703,9.698,9.695,9.693,
     * 9.636,9.636,9.631,9.622,9.611,9.599,9.587,9.575,
     * 9.354,9.350,9.344,9.331,9.317,9.304,9.288,9.276/
C
      DATA ((EDEN(I,3,8,L,1),I=41,48),L=1,NZEN)/
     * 10.984,11.009,11.022,11.029,11.032,11.039,11.045,11.057,
     * 10.957,10.980,10.993,10.998,11.000,11.003,11.008,11.016,
     * 10.882,10.911,10.929,10.938,10.941,10.945,10.952,10.962,
     * 10.701,10.757,10.794,10.818,10.833,10.842,10.852,10.866,
     * 10.248,10.332,10.414,10.483,10.542,10.591,10.630,10.664,
     * 10.086,10.132,10.192,10.261,10.333,10.400,10.460,10.514,
     * 10.021,10.036,10.053,10.066,10.087,10.116,10.147,10.192,
     * 9.921,9.933,9.943,9.950,9.957,9.964,9.979,9.994,
     * 9.675,9.682,9.686,9.687,9.682,9.679,9.678,9.679,
     * 9.598,9.596,9.590,9.579,9.569,9.560,9.552,9.545,
     * 9.319,9.309,9.295,9.279,9.262,9.249,9.236,9.226/
C
      DATA ((EDEN(I,4,8,L,1),I=41,48),L=1,NZEN)/
     * 10.993,11.007,11.013,11.014,11.016,11.018,11.025,11.036,
     * 10.945,10.964,10.974,10.977,10.978,10.981,10.988,10.998,
     * 10.868,10.896,10.913,10.921,10.924,10.927,10.934,10.946,
     * 10.696,10.749,10.784,10.805,10.818,10.828,10.839,10.851,
     * 10.264,10.348,10.426,10.491,10.544,10.587,10.624,10.656,
     * 10.091,10.131,10.186,10.248,10.314,10.379,10.437,10.493,
     * 10.017,10.034,10.049,10.067,10.088,10.121,10.160,10.213,
     * 9.925,9.935,9.941,9.947,9.953,9.960,9.977,9.996,
     * 9.656,9.659,9.660,9.660,9.654,9.652,9.653,9.657,
     * 9.578,9.571,9.561,9.549,9.538,9.529,9.523,9.520,
     * 9.282,9.273,9.262,9.248,9.234,9.226,9.219,9.214/
C
      DATA ((EDEN(I,5,8,L,1),I=41,48),L=1,NZEN)/
     * 11.002,11.005,11.004,11.000,10.997,10.996,11.002,11.012,
     * 10.951,10.958,10.960,10.959,10.957,10.956,10.963,10.976,
     * 10.871,10.889,10.899,10.901,10.901,10.903,10.911,10.923,
     * 10.698,10.744,10.774,10.791,10.801,10.809,10.819,10.833,
     * 10.280,10.356,10.425,10.482,10.531,10.570,10.606,10.639,
     * 10.104,10.149,10.204,10.262,10.326,10.384,10.440,10.492,
     * 10.024,10.037,10.052,10.063,10.088,10.122,10.159,10.214,
     * 9.933,9.939,9.942,9.945,9.948,9.954,9.971,9.991,
     * 9.643,9.643,9.642,9.641,9.635,9.633,9.636,9.643,
     * 9.562,9.551,9.538,9.524,9.512,9.505,9.500,9.501,
     * 9.263,9.252,9.242,9.228,9.216,9.212,9.208,9.210/
C
      DATA ((EDEN(I,1,9,L,1),I=41,48),L=1,NZEN)/
     * 10.980,11.006,11.022,11.031,11.039,11.047,11.055,11.066,
     * 10.944,10.971,10.988,10.998,11.005,11.011,11.019,11.028,
     * 10.873,10.910,10.932,10.945,10.952,10.958,10.966,10.974,
     * 10.690,10.756,10.801,10.829,10.848,10.859,10.871,10.881,
     * 10.222,10.309,10.399,10.481,10.550,10.604,10.647,10.681,
     * 10.094,10.119,10.148,10.189,10.249,10.323,10.396,10.468,
     * 10.057,10.069,10.080,10.087,10.094,10.110,10.134,10.179,
     * 9.955,9.968,9.975,9.977,9.976,9.977,9.984,9.993,
     * 9.744,9.750,9.750,9.747,9.736,9.725,9.715,9.707,
     * 9.671,9.669,9.661,9.648,9.632,9.615,9.597,9.580,
     * 9.396,9.398,9.392,9.380,9.367,9.351,9.334,9.316/
C
      DATA ((EDEN(I,2,9,L,1),I=41,48),L=1,NZEN)/
     * 10.976,11.004,11.022,11.032,11.037,11.045,11.057,11.064,
     * 10.952,10.979,10.995,11.002,11.008,11.012,11.019,11.026,
     * 10.865,10.904,10.928,10.941,10.948,10.954,10.962,10.972,
     * 10.646,10.717,10.770,10.806,10.829,10.845,10.859,10.875,
     * 10.179,10.245,10.322,10.403,10.479,10.544,10.600,10.648,
     * 10.084,10.113,10.148,10.196,10.262,10.334,10.406,10.473,
     * 10.042,10.058,10.069,10.079,10.093,10.112,10.140,10.184,
     * 9.944,9.957,9.965,9.969,9.972,9.975,9.984,9.995,
     * 9.723,9.729,9.731,9.729,9.721,9.712,9.705,9.700,
     * 9.649,9.646,9.640,9.628,9.614,9.599,9.584,9.571,
     * 9.372,9.370,9.361,9.350,9.335,9.321,9.306,9.291/
C
      DATA ((EDEN(I,3,9,L,1),I=41,48),L=1,NZEN)/
     * 10.973,11.001,11.018,11.026,11.032,11.039,11.045,11.057,
     * 10.927,10.960,10.979,10.990,10.995,11.000,11.009,11.018,
     * 10.850,10.893,10.918,10.932,10.940,10.946,10.954,10.965,
     * 10.647,10.717,10.768,10.801,10.822,10.837,10.851,10.866,
     * 10.193,10.267,10.350,10.427,10.498,10.558,10.607,10.649,
     * 10.080,10.111,10.153,10.210,10.279,10.352,10.419,10.483,
     * 10.030,10.045,10.058,10.070,10.086,10.114,10.147,10.197,
     * 9.935,9.946,9.955,9.960,9.965,9.970,9.982,9.995,
     * 9.704,9.709,9.710,9.709,9.700,9.693,9.688,9.685,
     * 9.639,9.635,9.627,9.615,9.601,9.588,9.575,9.564,
     * 9.350,9.344,9.335,9.322,9.308,9.294,9.280,9.267/
C
      DATA ((EDEN(I,4,9,L,1),I=41,48),L=1,NZEN)/
     * 10.978,10.999,11.011,11.016,11.019,11.020,11.030,11.041,
     * 10.916,10.946,10.963,10.972,10.976,10.981,10.990,11.003,
     * 10.835,10.878,10.903,10.917,10.924,10.929,10.938,10.949,
     * 10.647,10.715,10.764,10.796,10.816,10.829,10.841,10.855,
     * 10.212,10.288,10.368,10.443,10.508,10.561,10.606,10.645,
     * 10.086,10.120,10.165,10.222,10.290,10.361,10.425,10.484,
     * 10.036,10.048,10.058,10.068,10.084,10.107,10.136,10.180,
     * 9.936,9.945,9.950,9.953,9.955,9.958,9.969,9.982,
     * 9.698,9.699,9.696,9.691,9.679,9.670,9.664,9.662,
     * 9.622,9.613,9.599,9.583,9.566,9.551,9.537,9.526,
     * 9.326,9.316,9.302,9.282,9.264,9.249,9.233,9.221/
C
      DATA ((EDEN(I,5,9,L,1),I=41,48),L=1,NZEN)/
     * 10.984,10.999,11.005,11.006,11.005,11.005,11.011,11.026,
     * 10.925,10.947,10.959,10.962,10.963,10.965,10.972,10.985,
     * 10.831,10.865,10.887,10.898,10.904,10.907,10.917,10.930,
     * 10.635,10.699,10.744,10.775,10.795,10.809,10.823,10.838,
     * 10.221,10.293,10.369,10.440,10.500,10.550,10.593,10.631,
     * 10.096,10.123,10.164,10.220,10.285,10.351,10.413,10.470,
     * 10.044,10.052,10.058,10.064,10.077,10.104,10.134,10.182,
     * 9.946,9.950,9.951,9.950,9.949,9.951,9.961,9.975,
     * 9.702,9.698,9.690,9.681,9.666,9.655,9.648,9.646,
     * 9.626,9.613,9.594,9.573,9.552,9.535,9.520,9.509,
     * 9.329,9.316,9.299,9.278,9.258,9.242,9.227,9.217/
C
      DATA ((EDEN(I,1,10,L,1),I=41,48),L=1,NZEN)/
     * 10.974,11.003,11.022,11.033,11.039,11.047,11.059,11.071,
     * 10.939,10.970,10.989,11.000,11.006,11.013,11.021,11.031,
     * 10.866,10.906,10.930,10.946,10.953,10.960,10.968,10.978,
     * 10.672,10.744,10.795,10.828,10.848,10.861,10.873,10.885,
     * 10.206,10.287,10.377,10.462,10.536,10.595,10.642,10.680,
     * 10.097,10.119,10.145,10.181,10.234,10.306,10.382,10.455,
     * 10.067,10.080,10.087,10.094,10.101,10.113,10.131,10.170,
     * 9.963,9.976,9.983,9.984,9.983,9.983,9.989,9.996,
     * 9.760,9.766,9.766,9.762,9.751,9.740,9.729,9.720,
     * 9.688,9.686,9.678,9.665,9.649,9.632,9.614,9.597,
     * 9.413,9.414,9.408,9.397,9.383,9.368,9.350,9.333/
C
      DATA ((EDEN(I,2,10,L,1),I=41,48),L=1,NZEN)/
     * 10.979,11.006,11.023,11.032,11.042,11.045,11.057,11.068,
     * 10.937,10.969,10.987,10.999,11.004,11.011,11.019,11.029,
     * 10.859,10.900,10.927,10.943,10.951,10.957,10.965,10.977,
     * 10.658,10.732,10.785,10.821,10.842,10.857,10.869,10.883,
     * 10.197,10.274,10.364,10.451,10.526,10.587,10.636,10.674,
     * 10.092,10.113,10.139,10.177,10.235,10.306,10.382,10.454,
     * 10.058,10.070,10.080,10.087,10.096,10.111,10.132,10.170,
     * 9.947,9.960,9.968,9.972,9.974,9.976,9.982,9.990,
     * 9.756,9.760,9.759,9.755,9.743,9.731,9.720,9.710,
     * 9.685,9.681,9.671,9.657,9.639,9.622,9.603,9.585,
     * 9.401,9.399,9.393,9.379,9.365,9.349,9.330,9.313/
C
      DATA ((EDEN(I,3,10,L,1),I=41,48),L=1,NZEN)/
     * 10.978,11.004,11.020,11.029,11.032,11.039,11.050,11.061,
     * 10.915,10.950,10.973,10.985,10.993,10.999,11.009,11.020,
     * 10.841,10.888,10.916,10.933,10.941,10.948,10.957,10.968,
     * 10.639,10.716,10.772,10.809,10.832,10.847,10.860,10.874,
     * 10.195,10.273,10.364,10.449,10.521,10.580,10.626,10.664,
     * 10.085,10.109,10.142,10.189,10.253,10.328,10.398,10.467,
     * 10.048,10.058,10.067,10.075,10.090,10.109,10.135,10.178,
     * 9.942,9.952,9.958,9.962,9.964,9.967,9.974,9.983,
     * 9.738,9.739,9.735,9.728,9.714,9.701,9.690,9.681,
     * 9.666,9.658,9.644,9.627,9.607,9.587,9.567,9.549,
     * 9.370,9.362,9.352,9.335,9.315,9.297,9.278,9.259/
C
      DATA ((EDEN(I,4,10,L,1),I=41,48),L=1,NZEN)/
     * 10.973,11.000,11.014,11.021,11.025,11.029,11.038,11.049,
     * 10.909,10.943,10.964,10.977,10.982,10.988,10.997,11.009,
     * 10.806,10.856,10.890,10.909,10.920,10.929,10.940,10.953,
     * 10.618,10.696,10.753,10.791,10.815,10.831,10.844,10.860,
     * 10.206,10.283,10.367,10.447,10.514,10.570,10.614,10.650,
     * 10.092,10.122,10.167,10.226,10.296,10.366,10.432,10.489,
     * 10.047,10.058,10.064,10.072,10.084,10.108,10.137,10.184,
     * 9.948,9.955,9.958,9.958,9.957,9.957,9.965,9.974,
     * 9.722,9.720,9.713,9.704,9.687,9.672,9.659,9.649,
     * 9.649,9.636,9.619,9.598,9.573,9.551,9.530,9.510,
     * 9.366,9.353,9.337,9.317,9.294,9.273,9.251,9.231/
C
      DATA ((EDEN(I,5,10,L,1),I=41,48),L=1,NZEN)/
     * 10.973,10.997,11.009,11.013,11.015,11.016,11.024,11.034,
     * 10.907,10.941,10.958,10.967,10.970,10.974,10.981,10.994,
     * 10.802,10.850,10.882,10.900,10.909,10.915,10.925,10.938,
     * 10.589,10.665,10.722,10.762,10.788,10.807,10.825,10.840,
     * 10.205,10.273,10.352,10.428,10.493,10.548,10.594,10.632,
     * 10.103,10.126,10.161,10.212,10.276,10.344,10.408,10.467,
     * 10.056,10.063,10.065,10.069,10.077,10.101,10.128,10.178,
     * 9.960,9.962,9.960,9.956,9.950,9.947,9.953,9.961,
     * 9.731,9.724,9.712,9.698,9.676,9.657,9.641,9.630,
     * 9.658,9.641,9.618,9.591,9.562,9.535,9.510,9.488,
     * 9.379,9.360,9.338,9.310,9.281,9.254,9.226,9.203/
C
      DATA ((EDEN(I,1,11,L,1),I=41,48),L=1,NZEN)/
     * 10.980,11.010,11.029,11.042,11.047,11.056,11.067,11.075,
     * 10.943,10.977,10.996,11.008,11.014,11.021,11.029,11.038,
     * 10.870,10.912,10.939,10.953,10.962,10.967,10.976,10.985,
     * 10.676,10.751,10.802,10.836,10.856,10.869,10.881,10.891,
     * 10.209,10.294,10.386,10.476,10.550,10.609,10.654,10.690,
     * 10.101,10.122,10.148,10.186,10.247,10.321,10.400,10.474,
     * 10.067,10.080,10.091,10.094,10.101,10.116,10.137,10.182,
     * 9.966,9.978,9.984,9.986,9.986,9.985,9.991,9.999,
     * 9.768,9.772,9.771,9.766,9.753,9.741,9.729,9.720,
     * 9.696,9.692,9.684,9.668,9.651,9.632,9.613,9.594,
     * 9.422,9.420,9.413,9.401,9.384,9.368,9.349,9.329/
C
      DATA ((EDEN(I,2,11,L,1),I=41,48),L=1,NZEN)/
     * 10.992,11.017,11.031,11.041,11.046,11.054,11.061,11.072,
     * 10.934,10.965,10.986,10.997,11.005,11.012,11.021,11.032,
     * 10.866,10.909,10.935,10.950,10.957,10.964,10.972,10.982,
     * 10.657,10.736,10.792,10.828,10.850,10.866,10.876,10.890,
     * 10.192,10.270,10.361,10.453,10.532,10.595,10.644,10.682,
     * 10.096,10.118,10.140,10.177,10.235,10.312,10.391,10.466,
     * 10.066,10.078,10.085,10.090,10.096,10.111,10.134,10.174,
     * 9.953,9.965,9.972,9.975,9.975,9.976,9.981,9.987,
     * 9.780,9.780,9.775,9.766,9.750,9.735,9.720,9.707,
     * 9.709,9.702,9.688,9.670,9.649,9.625,9.602,9.580,
     * 9.415,9.411,9.400,9.384,9.365,9.343,9.323,9.299/
C
      DATA ((EDEN(I,3,11,L,1),I=41,48),L=1,NZEN)/
     * 10.992,11.015,11.028,11.038,11.041,11.047,11.054,11.065,
     * 10.932,10.964,10.982,10.993,10.999,11.005,11.014,11.026,
     * 10.839,10.883,10.913,10.929,10.939,10.948,10.957,10.971,
     * 10.662,10.736,10.787,10.821,10.842,10.855,10.866,10.878,
     * 10.199,10.277,10.367,10.455,10.530,10.589,10.634,10.671,
     * 10.091,10.124,10.167,10.232,10.309,10.386,10.453,10.512,
     * 10.057,10.064,10.072,10.079,10.089,10.111,10.135,10.182,
     * 9.950,9.959,9.964,9.966,9.966,9.966,9.971,9.977,
     * 9.745,9.743,9.736,9.727,9.708,9.690,9.673,9.658,
     * 9.686,9.676,9.658,9.637,9.611,9.586,9.561,9.536,
     * 9.390,9.381,9.367,9.348,9.324,9.300,9.275,9.249/
C
      DATA ((EDEN(I,4,11,L,1),I=41,48),L=1,NZEN)/
     * 10.984,11.009,11.022,11.028,11.033,11.037,11.042,11.058,
     * 10.922,10.955,10.974,10.984,10.990,10.995,11.004,11.015,
     * 10.821,10.869,10.901,10.918,10.929,10.936,10.946,10.959,
     * 10.615,10.692,10.749,10.788,10.815,10.831,10.847,10.861,
     * 10.229,10.311,10.398,10.477,10.540,10.591,10.630,10.661,
     * 10.101,10.137,10.185,10.251,10.323,10.396,10.455,10.506,
     * 10.050,10.061,10.066,10.077,10.090,10.118,10.156,10.208,
     * 9.955,9.961,9.962,9.961,9.958,9.956,9.962,9.968,
     * 9.722,9.718,9.709,9.696,9.675,9.655,9.637,9.622,
     * 9.649,9.634,9.613,9.587,9.557,9.529,9.499,9.474,
     * 9.373,9.360,9.341,9.318,9.289,9.260,9.233,9.204/
C
      DATA ((EDEN(I,5,11,L,1),I=41,48),L=1,NZEN)/
     * 10.979,11.003,11.016,11.022,11.019,11.024,11.032,11.042,
     * 10.913,10.948,10.966,10.975,10.978,10.981,10.990,11.002,
     * 10.810,10.859,10.890,10.907,10.916,10.922,10.931,10.944,
     * 10.597,10.675,10.732,10.771,10.797,10.814,10.829,10.846,
     * 10.201,10.265,10.338,10.416,10.484,10.542,10.589,10.628,
     * 10.110,10.133,10.169,10.220,10.284,10.353,10.415,10.472,
     * 10.064,10.067,10.071,10.074,10.083,10.104,10.133,10.184,
     * 9.967,9.969,9.966,9.959,9.950,9.943,9.944,9.948,
     * 9.729,9.722,9.708,9.691,9.665,9.640,9.618,9.599,
     * 9.654,9.636,9.611,9.581,9.547,9.514,9.480,9.451,
     * 9.401,9.383,9.358,9.326,9.289,9.253,9.217,9.182/
C
      DATA ((EDEN(I,1,12,L,1),I=41,48),L=1,NZEN)/
     * 10.989,11.020,11.041,11.050,11.060,11.064,11.076,11.083,
     * 10.953,10.987,11.005,11.016,11.023,11.030,11.036,11.044,
     * 10.880,10.922,10.949,10.962,10.970,10.977,10.983,10.992,
     * 10.688,10.764,10.815,10.848,10.867,10.879,10.889,10.899,
     * 10.217,10.309,10.406,10.497,10.571,10.627,10.670,10.702,
     * 10.101,10.126,10.154,10.200,10.266,10.347,10.425,10.497,
     * 10.067,10.080,10.087,10.094,10.101,10.120,10.150,10.198,
     * 9.965,9.976,9.983,9.984,9.984,9.985,9.992,10.001,
     * 9.770,9.773,9.769,9.763,9.749,9.736,9.724,9.714,
     * 9.698,9.694,9.680,9.665,9.646,9.624,9.604,9.583,
     * 9.426,9.422,9.413,9.398,9.379,9.360,9.340,9.319/
C
      DATA ((EDEN(I,2,12,L,1),I=41,48),L=1,NZEN)/
     * 11.004,11.026,11.040,11.049,11.054,11.062,11.069,11.077,
     * 10.947,10.976,10.995,11.005,11.012,11.019,11.029,11.038,
     * 10.875,10.915,10.939,10.952,10.961,10.968,10.977,10.987,
     * 10.668,10.748,10.803,10.839,10.860,10.874,10.884,10.897,
     * 10.195,10.277,10.374,10.469,10.549,10.611,10.657,10.693,
     * 10.097,10.128,10.176,10.243,10.327,10.410,10.481,10.538,
     * 10.067,10.079,10.082,10.087,10.096,10.113,10.139,10.186,
     * 9.954,9.965,9.971,9.973,9.974,9.975,9.980,9.986,
     * 9.767,9.766,9.760,9.750,9.731,9.712,9.694,9.679,
     * 9.719,9.710,9.693,9.672,9.646,9.620,9.593,9.567,
     * 9.421,9.414,9.401,9.381,9.359,9.334,9.309,9.284/
C
      DATA ((EDEN(I,3,12,L,1),I=41,48),L=1,NZEN)/
     * 11.005,11.025,11.035,11.042,11.049,11.052,11.062,11.069,
     * 10.948,10.975,10.992,11.000,11.005,11.012,11.021,11.031,
     * 10.860,10.898,10.923,10.939,10.947,10.954,10.964,10.977,
     * 10.685,10.754,10.801,10.831,10.849,10.862,10.872,10.884,
     * 10.242,10.334,10.429,10.512,10.578,10.626,10.662,10.689,
     * 10.095,10.131,10.180,10.247,10.327,10.404,10.471,10.524,
     * 10.047,10.057,10.066,10.076,10.096,10.129,10.171,10.233,
     * 9.951,9.959,9.963,9.964,9.964,9.966,9.973,9.980,
     * 9.748,9.743,9.733,9.720,9.699,9.679,9.661,9.644,
     * 9.676,9.663,9.641,9.615,9.585,9.554,9.524,9.494,
     * 9.393,9.384,9.367,9.343,9.316,9.289,9.259,9.228/
C
      DATA ((EDEN(I,4,12,L,1),I=41,48),L=1,NZEN)/
     * 10.995,11.017,11.029,11.037,11.037,11.042,11.051,11.062,
     * 10.935,10.965,10.983,10.991,10.996,11.001,11.010,11.022,
     * 10.841,10.884,10.911,10.927,10.935,10.942,10.952,10.965,
     * 10.642,10.713,10.766,10.801,10.823,10.837,10.852,10.868,
     * 10.248,10.334,10.419,10.496,10.556,10.603,10.638,10.667,
     * 10.113,10.151,10.206,10.275,10.347,10.414,10.470,10.518,
     * 10.048,10.059,10.068,10.078,10.097,10.132,10.175,10.230,
     * 9.955,9.960,9.962,9.960,9.957,9.956,9.961,9.965,
     * 9.713,9.709,9.699,9.686,9.663,9.641,9.620,9.602,
     * 9.642,9.627,9.607,9.578,9.547,9.514,9.482,9.451,
     * 9.370,9.357,9.338,9.310,9.280,9.247,9.214,9.180/
C
      DATA ((EDEN(I,5,12,L,1),I=41,48),L=1,NZEN)/
     * 10.989,11.010,11.021,11.026,11.027,11.029,11.036,11.047,
     * 10.927,10.956,10.973,10.981,10.984,10.987,10.995,11.008,
     * 10.827,10.872,10.899,10.914,10.921,10.927,10.936,10.950,
     * 10.623,10.694,10.747,10.783,10.805,10.820,10.834,10.850,
     * 10.220,10.288,10.362,10.437,10.501,10.554,10.597,10.633,
     * 10.116,10.138,10.171,10.221,10.281,10.349,10.411,10.466,
     * 10.066,10.073,10.073,10.075,10.083,10.101,10.129,10.178,
     * 9.972,9.973,9.968,9.960,9.948,9.938,9.936,9.935,
     * 9.717,9.711,9.698,9.681,9.652,9.624,9.598,9.575,
     * 9.644,9.627,9.602,9.569,9.533,9.495,9.458,9.422,
     * 9.374,9.358,9.334,9.303,9.264,9.227,9.186,9.145/
C
      DATA ((EDEN(I,1,1,L,2),I=41,48),L=1,NZEN)/
     * 11.074,11.099,11.115,11.126,11.132,11.139,11.147,11.151,
     * 11.037,11.064,11.081,11.089,11.095,11.102,11.106,11.114,
     * 10.968,11.005,11.027,11.036,11.042,11.050,11.053,11.065,
     * 10.796,10.860,10.901,10.927,10.941,10.951,10.961,10.972,
     * 10.371,10.456,10.542,10.616,10.676,10.721,10.755,10.782,
     * 10.227,10.260,10.301,10.351,10.415,10.483,10.549,10.608,
     * 10.179,10.195,10.212,10.225,10.243,10.270,10.304,10.356,
     * 10.054,10.072,10.085,10.092,10.096,10.101,10.113,10.125,
     * 9.775,9.784,9.787,9.787,9.778,9.769,9.762,9.755,
     * 9.691,9.689,9.680,9.667,9.651,9.635,9.619,9.605,
     * 9.411,9.412,9.407,9.395,9.381,9.366,9.352,9.338/
C
      DATA ((EDEN(I,2,1,L,2),I=41,48),L=1,NZEN)/
     * 11.086,11.103,11.115,11.122,11.126,11.133,11.137,11.145,
     * 11.029,11.056,11.069,11.077,11.085,11.093,11.100,11.107,
     * 10.963,10.994,11.014,11.027,11.032,11.039,11.046,11.057,
     * 10.781,10.846,10.891,10.919,10.935,10.946,10.957,10.967,
     * 10.346,10.429,10.516,10.595,10.659,10.710,10.745,10.774,
     * 10.225,10.270,10.325,10.393,10.467,10.535,10.592,10.638,
     * 10.169,10.188,10.204,10.216,10.235,10.262,10.295,10.343,
     * 10.042,10.061,10.074,10.083,10.088,10.093,10.103,10.112,
     * 9.773,9.778,9.779,9.776,9.761,9.747,9.734,9.722,
     * 9.714,9.707,9.694,9.675,9.653,9.631,9.608,9.588,
     * 9.412,9.408,9.399,9.382,9.363,9.344,9.323,9.304/
C
      DATA ((EDEN(I,3,1,L,2),I=41,48),L=1,NZEN)/
     * 11.084,11.100,11.110,11.115,11.122,11.124,11.130,11.138,
     * 11.032,11.053,11.064,11.074,11.077,11.083,11.093,11.100,
     * 10.947,10.980,11.001,11.012,11.019,11.025,11.035,11.045,
     * 10.791,10.844,10.883,10.906,10.921,10.930,10.941,10.953,
     * 10.394,10.478,10.559,10.626,10.678,10.717,10.745,10.767,
     * 10.229,10.274,10.329,10.396,10.464,10.527,10.579,10.623,
     * 10.160,10.180,10.199,10.219,10.244,10.283,10.325,10.378,
     * 10.048,10.063,10.074,10.080,10.084,10.089,10.098,10.105,
     * 9.758,9.760,9.757,9.751,9.735,9.719,9.704,9.689,
     * 9.675,9.665,9.648,9.625,9.599,9.573,9.546,9.523,
     * 9.391,9.384,9.371,9.351,9.327,9.305,9.280,9.255/
C
      DATA ((EDEN(I,4,1,L,2),I=41,48),L=1,NZEN)/
     * 11.080,11.101,11.108,11.114,11.117,11.121,11.126,11.137,
     * 11.024,11.050,11.063,11.069,11.073,11.077,11.085,11.095,
     * 10.936,10.972,10.994,11.007,11.012,11.019,11.028,11.038,
     * 10.756,10.815,10.858,10.885,10.902,10.915,10.926,10.940,
     * 10.389,10.468,10.542,10.605,10.653,10.691,10.720,10.744,
     * 10.231,10.278,10.336,10.400,10.462,10.519,10.564,10.604,
     * 10.139,10.150,10.162,10.180,10.204,10.244,10.288,10.341,
     * 10.035,10.041,10.042,10.041,10.038,10.037,10.043,10.048,
     * 9.725,9.726,9.721,9.712,9.692,9.672,9.653,9.636,
     * 9.648,9.635,9.615,9.590,9.560,9.530,9.499,9.472,
     * 9.373,9.361,9.343,9.318,9.288,9.259,9.226,9.195/
C
      DATA ((EDEN(I,5,1,L,2),I=41,48),L=1,NZEN)/
     * 11.081,11.098,11.105,11.107,11.111,11.111,11.118,11.128,
     * 11.022,11.048,11.060,11.063,11.067,11.068,11.078,11.088,
     * 10.931,10.967,10.989,10.999,11.002,11.011,11.017,11.030,
     * 10.743,10.803,10.846,10.873,10.890,10.903,10.914,10.928,
     * 10.350,10.417,10.487,10.551,10.605,10.649,10.684,10.715,
     * 10.203,10.233,10.274,10.330,10.392,10.451,10.507,10.556,
     * 10.127,10.126,10.125,10.128,10.142,10.168,10.207,10.263,
     * 10.023,10.019,10.010,9.998,9.985,9.975,9.974,9.975,
     * 9.740,9.734,9.721,9.704,9.675,9.648,9.623,9.600,
     * 9.655,9.638,9.614,9.581,9.545,9.507,9.469,9.435,
     * 9.383,9.366,9.343,9.311,9.272,9.233,9.192,9.154/
C
      DATA ((EDEN(I,1,2,L,2),I=41,48),L=1,NZEN)/
     * 11.062,11.092,11.108,11.116,11.125,11.132,11.140,11.148,
     * 11.028,11.056,11.073,11.082,11.091,11.095,11.102,11.110,
     * 10.957,10.994,11.016,11.028,11.038,11.042,11.049,11.061,
     * 10.780,10.845,10.888,10.917,10.933,10.944,10.954,10.966,
     * 10.358,10.438,10.521,10.597,10.657,10.706,10.743,10.773,
     * 10.224,10.256,10.291,10.336,10.396,10.462,10.528,10.590,
     * 10.179,10.195,10.212,10.223,10.238,10.263,10.294,10.339,
     * 10.053,10.072,10.084,10.091,10.095,10.099,10.110,10.121,
     * 9.777,9.786,9.791,9.791,9.783,9.774,9.766,9.760,
     * 9.693,9.692,9.685,9.673,9.658,9.643,9.627,9.614,
     * 9.414,9.415,9.410,9.401,9.388,9.374,9.360,9.346/
C
      DATA ((EDEN(I,2,2,L,2),I=41,48),L=1,NZEN)/
     * 11.071,11.092,11.104,11.115,11.119,11.126,11.134,11.145,
     * 11.014,11.044,11.061,11.069,11.078,11.085,11.096,11.107,
     * 10.954,10.990,11.012,11.023,11.032,11.040,11.046,11.053,
     * 10.766,10.833,10.879,10.910,10.928,10.940,10.950,10.963,
     * 10.338,10.417,10.501,10.579,10.645,10.696,10.735,10.766,
     * 10.220,10.263,10.315,10.380,10.451,10.519,10.578,10.626,
     * 10.169,10.187,10.201,10.216,10.233,10.258,10.287,10.333,
     * 10.042,10.060,10.074,10.083,10.088,10.093,10.102,10.111,
     * 9.792,9.798,9.797,9.794,9.781,9.768,9.756,9.745,
     * 9.711,9.707,9.694,9.676,9.656,9.636,9.616,9.597,
     * 9.414,9.412,9.404,9.389,9.373,9.354,9.335,9.317/
C
      DATA ((EDEN(I,3,2,L,2),I=41,48),L=1,NZEN)/
     * 11.069,11.088,11.102,11.108,11.114,11.120,11.127,11.138,
     * 11.013,11.040,11.056,11.066,11.073,11.079,11.085,11.100,
     * 10.924,10.962,10.987,11.002,11.011,11.019,11.031,11.041,
     * 10.767,10.830,10.871,10.900,10.916,10.928,10.938,10.950,
     * 10.372,10.455,10.536,10.606,10.661,10.703,10.734,10.760,
     * 10.221,10.262,10.315,10.378,10.445,10.510,10.564,10.611,
     * 10.157,10.176,10.193,10.213,10.237,10.274,10.312,10.364,
     * 10.045,10.061,10.072,10.080,10.085,10.090,10.100,10.108,
     * 9.761,9.765,9.764,9.761,9.745,9.731,9.717,9.704,
     * 9.693,9.685,9.669,9.650,9.626,9.602,9.579,9.557,
     * 9.396,9.392,9.379,9.361,9.339,9.317,9.294,9.271/
C
      DATA ((EDEN(I,4,2,L,2),I=41,48),L=1,NZEN)/
     * 11.068,11.089,11.100,11.107,11.110,11.114,11.122,11.133,
     * 11.007,11.038,11.054,11.062,11.066,11.073,11.082,11.092,
     * 10.913,10.955,10.982,10.998,11.005,11.011,11.024,11.034,
     * 10.723,10.789,10.838,10.870,10.892,10.908,10.920,10.935,
     * 10.355,10.432,10.508,10.576,10.632,10.675,10.709,10.736,
     * 10.212,10.254,10.307,10.370,10.436,10.495,10.546,10.592,
     * 10.134,10.146,10.156,10.171,10.194,10.226,10.266,10.317,
     * 10.028,10.035,10.038,10.038,10.036,10.035,10.042,10.047,
     * 9.738,9.739,9.734,9.725,9.705,9.686,9.669,9.653,
     * 9.659,9.647,9.627,9.601,9.574,9.545,9.516,9.490,
     * 9.381,9.371,9.354,9.330,9.301,9.273,9.244,9.217/
C
      DATA ((EDEN(I,5,2,L,2),I=41,48),L=1,NZEN)/
     * 11.069,11.090,11.101,11.104,11.104,11.108,11.115,11.125,
     * 11.006,11.035,11.052,11.059,11.060,11.064,11.071,11.084,
     * 10.910,10.952,10.978,10.992,10.998,11.003,11.009,11.023,
     * 10.714,10.779,10.828,10.861,10.881,10.894,10.907,10.922,
     * 10.320,10.384,10.455,10.525,10.583,10.632,10.673,10.706,
     * 10.191,10.219,10.257,10.311,10.373,10.438,10.495,10.547,
     * 10.119,10.118,10.118,10.125,10.139,10.166,10.202,10.256,
     * 10.015,10.014,10.008,9.998,9.988,9.981,9.983,9.985,
     * 9.748,9.742,9.730,9.714,9.687,9.662,9.639,9.618,
     * 9.669,9.652,9.628,9.597,9.563,9.528,9.492,9.461,
     * 9.406,9.389,9.364,9.331,9.295,9.257,9.218,9.180/
C
      DATA ((EDEN(I,1,3,L,2),I=41,48),L=1,NZEN)/
     * 11.054,11.080,11.097,11.109,11.114,11.121,11.133,11.141,
     * 11.016,11.043,11.061,11.074,11.079,11.087,11.095,11.102,
     * 10.946,10.983,11.006,11.019,11.025,11.033,11.041,11.052,
     * 10.767,10.832,10.876,10.905,10.923,10.935,10.945,10.957,
     * 10.347,10.425,10.504,10.577,10.640,10.691,10.730,10.762,
     * 10.222,10.251,10.284,10.325,10.378,10.441,10.507,10.570,
     * 10.179,10.198,10.212,10.223,10.236,10.256,10.283,10.325,
     * 10.054,10.072,10.084,10.091,10.094,10.097,10.108,10.118,
     * 9.774,9.785,9.791,9.793,9.785,9.777,9.770,9.764,
     * 9.690,9.691,9.685,9.674,9.662,9.648,9.633,9.621,
     * 9.410,9.413,9.410,9.402,9.390,9.379,9.365,9.353/
C
      DATA ((EDEN(I,2,3,L,2),I=41,48),L=1,NZEN)/
     * 11.054,11.080,11.096,11.104,11.112,11.119,11.127,11.138,
     * 11.015,11.044,11.061,11.069,11.074,11.081,11.088,11.100,
     * 10.941,10.979,11.001,11.014,11.022,11.031,11.038,11.045,
     * 10.757,10.823,10.869,10.899,10.918,10.930,10.941,10.954,
     * 10.335,10.413,10.494,10.570,10.633,10.685,10.725,10.757,
     * 10.212,10.241,10.276,10.321,10.377,10.445,10.509,10.571,
     * 10.168,10.184,10.200,10.215,10.230,10.254,10.282,10.322,
     * 10.041,10.060,10.073,10.083,10.088,10.093,10.103,10.112,
     * 9.779,9.787,9.790,9.789,9.778,9.768,9.759,9.750,
     * 9.696,9.695,9.685,9.671,9.655,9.637,9.620,9.605,
     * 9.410,9.410,9.403,9.392,9.378,9.363,9.347,9.333/
C
      DATA ((EDEN(I,3,3,L,2),I=41,48),L=1,NZEN)/
     * 11.052,11.077,11.091,11.101,11.107,11.109,11.120,11.131,
     * 10.993,11.024,11.043,11.054,11.061,11.071,11.078,11.092,
     * 10.914,10.956,10.984,10.999,11.008,11.016,11.025,11.037,
     * 10.741,10.809,10.856,10.887,10.907,10.920,10.931,10.944,
     * 10.334,10.411,10.491,10.565,10.626,10.676,10.714,10.745,
     * 10.212,10.254,10.306,10.365,10.433,10.498,10.553,10.602,
     * 10.158,10.175,10.192,10.208,10.228,10.256,10.284,10.326,
     * 10.042,10.058,10.069,10.077,10.081,10.085,10.094,10.103,
     * 9.764,9.770,9.770,9.767,9.755,9.742,9.731,9.721,
     * 9.683,9.677,9.664,9.646,9.627,9.607,9.587,9.569,
     * 9.392,9.388,9.378,9.363,9.346,9.328,9.308,9.292/
C
      DATA ((EDEN(I,4,3,L,2),I=41,48),L=1,NZEN)/
     * 11.056,11.082,11.093,11.099,11.103,11.107,11.112,11.123,
     * 10.993,11.025,11.046,11.054,11.058,11.061,11.070,11.085,
     * 10.896,10.941,10.971,10.988,10.997,11.005,11.014,11.026,
     * 10.709,10.778,10.831,10.865,10.887,10.903,10.915,10.930,
     * 10.333,10.410,10.488,10.559,10.618,10.664,10.700,10.730,
     * 10.194,10.233,10.282,10.342,10.408,10.471,10.526,10.575,
     * 10.126,10.138,10.148,10.165,10.186,10.221,10.259,10.309,
     * 10.019,10.028,10.033,10.037,10.038,10.041,10.050,10.059,
     * 9.741,9.742,9.738,9.731,9.714,9.699,9.685,9.673,
     * 9.667,9.656,9.638,9.616,9.592,9.568,9.544,9.524,
     * 9.383,9.373,9.358,9.335,9.311,9.286,9.261,9.240/
C
      DATA ((EDEN(I,5,3,L,2),I=41,48),L=1,NZEN)/
     * 11.061,11.087,11.094,11.097,11.097,11.097,11.104,11.115,
     * 10.999,11.030,11.048,11.052,11.052,11.057,11.063,11.073,
     * 10.901,10.944,10.971,10.986,10.991,10.995,11.001,11.015,
     * 10.700,10.769,10.821,10.853,10.874,10.887,10.901,10.916,
     * 10.311,10.379,10.454,10.524,10.584,10.633,10.673,10.706,
     * 10.177,10.209,10.253,10.311,10.375,10.440,10.496,10.547,
     * 10.106,10.109,10.112,10.123,10.140,10.171,10.207,10.259,
     * 10.001,10.003,10.002,9.998,9.993,9.993,9.998,10.005,
     * 9.753,9.747,9.736,9.722,9.698,9.677,9.659,9.644,
     * 9.678,9.661,9.638,9.609,9.578,9.548,9.518,9.492,
     * 9.403,9.386,9.360,9.329,9.295,9.261,9.229,9.197/
C
      DATA ((EDEN(I,1,4,L,2),I=41,48),L=1,NZEN)/
     * 11.054,11.076,11.093,11.101,11.106,11.114,11.125,11.133,
     * 11.017,11.043,11.057,11.066,11.071,11.080,11.087,11.095,
     * 10.948,10.981,11.002,11.014,11.021,11.027,11.034,11.044,
     * 10.775,10.834,10.875,10.902,10.918,10.929,10.939,10.952,
     * 10.354,10.431,10.510,10.582,10.641,10.689,10.727,10.757,
     * 10.219,10.249,10.284,10.327,10.380,10.443,10.508,10.569,
     * 10.176,10.192,10.209,10.220,10.234,10.256,10.283,10.323,
     * 10.052,10.071,10.083,10.089,10.092,10.095,10.105,10.116,
     * 9.760,9.773,9.780,9.783,9.776,9.769,9.763,9.758,
     * 9.676,9.678,9.673,9.664,9.651,9.639,9.626,9.614,
     * 9.396,9.400,9.399,9.391,9.381,9.370,9.357,9.346/
C
      DATA ((EDEN(I,2,4,L,2),I=41,48),L=1,NZEN)/
     * 11.046,11.072,11.089,11.100,11.105,11.112,11.123,11.131,
     * 11.010,11.039,11.057,11.065,11.070,11.077,11.085,11.092,
     * 10.941,10.977,10.998,11.010,11.018,11.023,11.030,11.040,
     * 10.771,10.831,10.872,10.900,10.914,10.926,10.935,10.947,
     * 10.310,10.375,10.447,10.518,10.584,10.642,10.690,10.731,
     * 10.208,10.245,10.285,10.334,10.393,10.458,10.518,10.575,
     * 10.162,10.180,10.200,10.215,10.233,10.259,10.288,10.331,
     * 10.045,10.063,10.075,10.083,10.089,10.093,10.105,10.116,
     * 9.750,9.761,9.767,9.770,9.763,9.757,9.751,9.747,
     * 9.665,9.665,9.660,9.649,9.637,9.624,9.611,9.600,
     * 9.389,9.391,9.389,9.381,9.370,9.358,9.347,9.336/
C
      DATA ((EDEN(I,3,4,L,2),I=41,48),L=1,NZEN)/
     * 11.044,11.073,11.087,11.093,11.100,11.106,11.112,11.124,
     * 10.985,11.018,11.039,11.050,11.057,11.063,11.074,11.085,
     * 10.928,10.966,10.990,11.002,11.008,11.013,11.021,11.030,
     * 10.756,10.819,10.862,10.888,10.905,10.915,10.925,10.937,
     * 10.352,10.433,10.512,10.580,10.637,10.682,10.717,10.745,
     * 10.205,10.243,10.292,10.346,10.410,10.473,10.529,10.582,
     * 10.147,10.170,10.188,10.207,10.231,10.262,10.293,10.339,
     * 10.039,10.056,10.067,10.077,10.083,10.088,10.101,10.113,
     * 9.734,9.742,9.746,9.748,9.740,9.733,9.727,9.723,
     * 9.650,9.646,9.637,9.624,9.608,9.595,9.582,9.571,
     * 9.368,9.365,9.358,9.346,9.333,9.320,9.306,9.297/
C
      DATA ((EDEN(I,4,4,L,2),I=41,48),L=1,NZEN)/
     * 11.060,11.078,11.089,11.092,11.096,11.096,11.101,11.112,
     * 10.998,11.027,11.042,11.049,11.050,11.054,11.063,11.073,
     * 10.913,10.952,10.976,10.990,10.996,11.000,11.007,11.019,
     * 10.743,10.805,10.849,10.876,10.891,10.903,10.911,10.924,
     * 10.338,10.419,10.496,10.564,10.620,10.664,10.699,10.728,
     * 10.183,10.226,10.276,10.337,10.403,10.467,10.521,10.571,
     * 10.113,10.129,10.143,10.161,10.189,10.226,10.263,10.312,
     * 10.008,10.020,10.028,10.035,10.041,10.048,10.062,10.075,
     * 9.724,9.726,9.726,9.723,9.711,9.702,9.695,9.691,
     * 9.647,9.638,9.623,9.605,9.587,9.571,9.556,9.544,
     * 9.356,9.347,9.333,9.315,9.297,9.281,9.265,9.252/
C
      DATA ((EDEN(I,5,4,L,2),I=41,48),L=1,NZEN)/
     * 11.077,11.090,11.094,11.093,11.089,11.087,11.090,11.101,
     * 11.017,11.039,11.048,11.047,11.044,11.045,11.048,11.059,
     * 10.925,10.959,10.978,10.985,10.986,10.988,10.993,11.005,
     * 10.737,10.797,10.838,10.863,10.878,10.887,10.897,10.910,
     * 10.332,10.409,10.484,10.551,10.603,10.645,10.679,10.710,
     * 10.170,10.211,10.262,10.325,10.391,10.454,10.507,10.556,
     * 10.088,10.095,10.103,10.119,10.149,10.187,10.228,10.279,
     * 9.983,9.989,9.993,9.996,9.999,10.005,10.018,10.032,
     * 9.731,9.726,9.718,9.708,9.692,9.679,9.669,9.663,
     * 9.654,9.638,9.616,9.591,9.567,9.545,9.525,9.509,
     * 9.361,9.346,9.323,9.297,9.274,9.253,9.231,9.214/
C
      DATA ((EDEN(I,1,5,L,2),I=41,48),L=1,NZEN)/
     * 11.066,11.084,11.097,11.105,11.110,11.114,11.122,11.133,
     * 11.028,11.052,11.061,11.070,11.075,11.080,11.087,11.095,
     * 10.963,10.992,11.008,11.017,11.023,11.028,11.034,11.042,
     * 10.802,10.855,10.890,10.911,10.922,10.931,10.939,10.951,
     * 10.384,10.466,10.544,10.611,10.665,10.706,10.738,10.763,
     * 10.219,10.256,10.297,10.348,10.409,10.474,10.535,10.591,
     * 10.168,10.187,10.204,10.218,10.236,10.261,10.294,10.342,
     * 10.049,10.066,10.078,10.084,10.088,10.092,10.104,10.116,
     * 9.741,9.753,9.761,9.764,9.757,9.751,9.746,9.741,
     * 9.657,9.657,9.652,9.642,9.628,9.616,9.603,9.591,
     * 9.375,9.379,9.378,9.369,9.358,9.347,9.335,9.325/
C
      DATA ((EDEN(I,2,5,L,2),I=41,48),L=1,NZEN)/
     * 11.054,11.080,11.092,11.104,11.108,11.116,11.123,11.131,
     * 11.037,11.056,11.069,11.073,11.078,11.081,11.085,11.088,
     * 10.961,10.990,11.006,11.014,11.018,11.023,11.029,11.037,
     * 10.780,10.835,10.871,10.896,10.910,10.920,10.931,10.942,
     * 10.348,10.422,10.496,10.562,10.621,10.669,10.710,10.743,
     * 10.218,10.261,10.311,10.369,10.432,10.495,10.549,10.601,
     * 10.152,10.174,10.196,10.215,10.238,10.271,10.308,10.358,
     * 10.042,10.060,10.073,10.082,10.088,10.094,10.109,10.123,
     * 9.713,9.727,9.736,9.742,9.739,9.736,9.734,9.733,
     * 9.627,9.628,9.624,9.616,9.606,9.597,9.587,9.580,
     * 9.361,9.362,9.359,9.349,9.339,9.331,9.320,9.312/
C
      DATA ((EDEN(I,3,5,L,2),I=41,48),L=1,NZEN)/
     * 11.057,11.077,11.091,11.097,11.100,11.102,11.112,11.120,
     * 11.015,11.040,11.056,11.062,11.061,11.068,11.074,11.081,
     * 10.956,10.985,10.999,11.006,11.009,11.012,11.018,11.026,
     * 10.787,10.837,10.871,10.892,10.903,10.911,10.920,10.933,
     * 10.372,10.450,10.523,10.586,10.637,10.679,10.712,10.741,
     * 10.217,10.267,10.328,10.389,10.452,10.511,10.561,10.607,
     * 10.142,10.166,10.189,10.209,10.231,10.264,10.294,10.336,
     * 10.035,10.052,10.066,10.076,10.083,10.090,10.105,10.120,
     * 9.695,9.708,9.717,9.724,9.720,9.718,9.717,9.718,
     * 9.621,9.620,9.616,9.607,9.598,9.591,9.583,9.579,
     * 9.337,9.334,9.327,9.314,9.303,9.296,9.286,9.280/
C
      DATA ((EDEN(I,4,5,L,2),I=41,48),L=1,NZEN)/
     * 11.072,11.085,11.089,11.092,11.089,11.089,11.094,11.105,
     * 11.019,11.037,11.046,11.049,11.050,11.050,11.055,11.066,
     * 10.949,10.976,10.991,10.996,10.997,10.997,11.003,11.013,
     * 10.788,10.836,10.868,10.884,10.894,10.900,10.908,10.919,
     * 10.382,10.460,10.532,10.591,10.637,10.674,10.704,10.731,
     * 10.192,10.241,10.300,10.364,10.426,10.484,10.536,10.582,
     * 10.104,10.120,10.138,10.165,10.200,10.244,10.283,10.333,
     * 9.996,10.010,10.022,10.033,10.043,10.055,10.074,10.092,
     * 9.685,9.691,9.695,9.698,9.693,9.691,9.692,9.695,
     * 9.602,9.595,9.585,9.573,9.561,9.554,9.546,9.543,
     * 9.304,9.294,9.282,9.269,9.256,9.250,9.239,9.238/
C
      DATA ((EDEN(I,5,5,L,2),I=41,48),L=1,NZEN)/
     * 11.093,11.094,11.094,11.086,11.078,11.076,11.075,11.086,
     * 11.040,11.051,11.052,11.043,11.040,11.033,11.037,11.047,
     * 10.965,10.981,10.987,10.985,10.982,10.979,10.984,10.994,
     * 10.800,10.842,10.868,10.878,10.883,10.886,10.894,10.903,
     * 10.395,10.475,10.541,10.593,10.633,10.665,10.693,10.718,
     * 10.193,10.246,10.310,10.376,10.437,10.490,10.537,10.578,
     * 10.073,10.084,10.100,10.130,10.169,10.220,10.263,10.317,
     * 9.965,9.974,9.983,9.994,10.004,10.018,10.039,10.059,
     * 9.677,9.676,9.674,9.672,9.664,9.661,9.662,9.666,
     * 9.594,9.580,9.563,9.545,9.530,9.521,9.512,9.508,
     * 9.294,9.280,9.263,9.247,9.234,9.228,9.221,9.216/
C
      DATA ((EDEN(I,1,6,L,2),I=41,48),L=1,NZEN)/
     * 11.078,11.095,11.104,11.112,11.117,11.121,11.125,11.133,
     * 11.041,11.060,11.069,11.078,11.079,11.083,11.087,11.095,
     * 10.979,11.003,11.017,11.023,11.029,11.033,11.036,11.042,
     * 10.829,10.875,10.903,10.921,10.929,10.937,10.945,10.953,
     * 10.417,10.504,10.582,10.643,10.689,10.724,10.749,10.771,
     * 10.227,10.269,10.320,10.380,10.443,10.508,10.564,10.614,
     * 10.163,10.182,10.202,10.218,10.241,10.274,10.314,10.369,
     * 10.045,10.062,10.075,10.081,10.086,10.092,10.105,10.118,
     * 9.724,9.736,9.742,9.746,9.740,9.734,9.729,9.725,
     * 9.638,9.637,9.632,9.620,9.606,9.592,9.579,9.568,
     * 9.358,9.360,9.357,9.348,9.337,9.325,9.312,9.300/
C
      DATA ((EDEN(I,2,6,L,2),I=41,48),L=1,NZEN)/
     * 11.066,11.092,11.104,11.111,11.116,11.119,11.127,11.131,
     * 11.050,11.068,11.077,11.081,11.082,11.085,11.085,11.089,
     * 10.979,11.002,11.014,11.021,11.023,11.027,11.031,11.037,
     * 10.816,10.860,10.889,10.906,10.918,10.925,10.934,10.945,
     * 10.394,10.470,10.542,10.604,10.653,10.694,10.726,10.754,
     * 10.234,10.287,10.346,10.410,10.472,10.530,10.580,10.623,
     * 10.154,10.176,10.196,10.215,10.236,10.263,10.292,10.337,
     * 10.040,10.057,10.070,10.080,10.086,10.092,10.106,10.121,
     * 9.689,9.705,9.718,9.728,9.725,9.724,9.723,9.723,
     * 9.621,9.624,9.623,9.617,9.609,9.602,9.595,9.590,
     * 9.340,9.341,9.337,9.329,9.318,9.310,9.303,9.295/
C
      DATA ((EDEN(I,3,6,L,2),I=41,48),L=1,NZEN)/
     * 11.069,11.088,11.095,11.101,11.104,11.106,11.112,11.120,
     * 11.044,11.061,11.068,11.070,11.069,11.071,11.074,11.078,
     * 10.976,10.997,11.007,11.011,11.012,11.014,11.018,11.025,
     * 10.824,10.864,10.888,10.902,10.909,10.915,10.922,10.933,
     * 10.429,10.504,10.571,10.623,10.664,10.698,10.725,10.750,
     * 10.223,10.273,10.329,10.387,10.444,10.501,10.552,10.598,
     * 10.138,10.164,10.188,10.213,10.243,10.280,10.317,10.364,
     * 10.033,10.050,10.063,10.075,10.084,10.094,10.113,10.130,
     * 9.674,9.688,9.700,9.710,9.710,9.712,9.715,9.721,
     * 9.584,9.584,9.582,9.576,9.571,9.568,9.565,9.564,
     * 9.296,9.290,9.281,9.269,9.259,9.254,9.247,9.245/
C
      DATA ((EDEN(I,4,6,L,2),I=41,48),L=1,NZEN)/
     * 11.084,11.089,11.093,11.088,11.089,11.089,11.090,11.097,
     * 11.039,11.049,11.054,11.053,11.050,11.050,11.055,11.062,
     * 10.975,10.992,10.999,11.000,10.997,10.997,11.001,11.009,
     * 10.825,10.863,10.882,10.894,10.896,10.901,10.908,10.917,
     * 10.425,10.497,10.559,10.607,10.647,10.679,10.706,10.732,
     * 10.222,10.281,10.345,10.408,10.466,10.518,10.563,10.603,
     * 10.098,10.118,10.142,10.173,10.210,10.259,10.299,10.349,
     * 9.993,10.007,10.020,10.033,10.045,10.059,10.082,10.104,
     * 9.648,9.657,9.665,9.673,9.673,9.677,9.683,9.692,
     * 9.560,9.555,9.547,9.538,9.533,9.531,9.530,9.533,
     * 9.261,9.253,9.246,9.237,9.231,9.230,9.229,9.233/
C
      DATA ((EDEN(I,5,6,L,2),I=41,48),L=1,NZEN)/
     * 11.096,11.094,11.090,11.078,11.071,11.064,11.068,11.074,
     * 11.053,11.051,11.047,11.039,11.032,11.025,11.029,11.040,
     * 10.984,10.991,10.992,10.986,10.979,10.976,10.981,10.990,
     * 10.842,10.867,10.881,10.885,10.884,10.884,10.890,10.902,
     * 10.467,10.533,10.587,10.624,10.653,10.678,10.700,10.722,
     * 10.243,10.305,10.366,10.422,10.474,10.519,10.558,10.594,
     * 10.067,10.088,10.114,10.152,10.200,10.254,10.299,10.352,
     * 9.957,9.969,9.982,9.996,10.010,10.029,10.055,10.080,
     * 9.622,9.627,9.631,9.638,9.638,9.643,9.651,9.663,
     * 9.531,9.520,9.507,9.496,9.489,9.489,9.489,9.495,
     * 9.235,9.224,9.214,9.207,9.203,9.205,9.207,9.217/
C
      DATA ((EDEN(I,1,7,L,2),I=41,48),L=1,NZEN)/
     * 11.081,11.099,11.108,11.112,11.117,11.121,11.125,11.133,
     * 11.045,11.064,11.073,11.082,11.083,11.087,11.091,11.095,
     * 10.983,11.007,11.019,11.028,11.029,11.033,11.041,11.043,
     * 10.835,10.880,10.908,10.924,10.932,10.939,10.946,10.955,
     * 10.429,10.516,10.592,10.651,10.696,10.729,10.753,10.773,
     * 10.229,10.274,10.328,10.389,10.453,10.516,10.571,10.619,
     * 10.160,10.182,10.202,10.220,10.243,10.278,10.322,10.376,
     * 10.044,10.062,10.074,10.081,10.087,10.092,10.106,10.119,
     * 9.719,9.731,9.738,9.742,9.736,9.730,9.725,9.721,
     * 9.633,9.632,9.625,9.614,9.600,9.586,9.573,9.562,
     * 9.354,9.356,9.352,9.342,9.331,9.319,9.306,9.296/
C
      DATA ((EDEN(I,2,7,L,2),I=41,48),L=1,NZEN)/
     * 11.071,11.092,11.107,11.115,11.116,11.123,11.127,11.135,
     * 11.054,11.068,11.080,11.084,11.085,11.085,11.085,11.089,
     * 10.983,11.005,11.016,11.022,11.028,11.027,11.032,11.038,
     * 10.823,10.866,10.892,10.908,10.919,10.927,10.935,10.947,
     * 10.404,10.482,10.552,10.611,10.659,10.699,10.729,10.756,
     * 10.239,10.293,10.355,10.419,10.480,10.537,10.584,10.627,
     * 10.154,10.176,10.198,10.214,10.238,10.265,10.297,10.343,
     * 10.040,10.058,10.071,10.080,10.087,10.093,10.108,10.122,
     * 9.682,9.699,9.712,9.723,9.721,9.720,9.720,9.721,
     * 9.613,9.617,9.616,9.610,9.604,9.597,9.591,9.586,
     * 9.333,9.333,9.331,9.322,9.312,9.306,9.298,9.292/
C
      DATA ((EDEN(I,3,7,L,2),I=41,48),L=1,NZEN)/
     * 11.073,11.088,11.099,11.104,11.104,11.109,11.116,11.120,
     * 11.048,11.061,11.068,11.070,11.073,11.072,11.074,11.082,
     * 10.979,11.000,11.009,11.012,11.014,11.015,11.020,11.027,
     * 10.828,10.867,10.889,10.903,10.909,10.917,10.923,10.935,
     * 10.440,10.514,10.577,10.629,10.668,10.701,10.727,10.752,
     * 10.230,10.281,10.336,10.393,10.452,10.507,10.556,10.602,
     * 10.137,10.164,10.191,10.216,10.245,10.285,10.321,10.369,
     * 10.035,10.051,10.064,10.076,10.086,10.095,10.115,10.133,
     * 9.666,9.681,9.693,9.705,9.706,9.709,9.714,9.720,
     * 9.575,9.576,9.575,9.569,9.566,9.564,9.563,9.563,
     * 9.290,9.284,9.274,9.263,9.254,9.250,9.244,9.242/
C
      DATA ((EDEN(I,4,7,L,2),I=41,48),L=1,NZEN)/
     * 11.084,11.089,11.093,11.088,11.089,11.089,11.094,11.101,
     * 11.039,11.049,11.054,11.053,11.050,11.050,11.055,11.067,
     * 10.977,10.993,11.000,10.999,10.998,10.997,11.003,11.012,
     * 10.830,10.863,10.884,10.892,10.897,10.901,10.910,10.921,
     * 10.436,10.505,10.563,10.610,10.649,10.680,10.708,10.734,
     * 10.229,10.289,10.352,10.412,10.469,10.521,10.564,10.605,
     * 10.101,10.121,10.145,10.175,10.213,10.260,10.302,10.353,
     * 9.994,10.008,10.021,10.033,10.045,10.059,10.082,10.105,
     * 9.641,9.651,9.660,9.669,9.670,9.675,9.683,9.694,
     * 9.552,9.547,9.540,9.534,9.530,9.530,9.530,9.534,
     * 9.252,9.246,9.240,9.233,9.227,9.230,9.229,9.235/
C
      DATA ((EDEN(I,5,7,L,2),I=41,48),L=1,NZEN)/
     * 11.096,11.090,11.087,11.078,11.071,11.064,11.068,11.078,
     * 11.049,11.047,11.043,11.035,11.028,11.025,11.029,11.040,
     * 10.984,10.990,10.989,10.984,10.979,10.976,10.982,10.993,
     * 10.843,10.867,10.881,10.883,10.883,10.885,10.891,10.903,
     * 10.474,10.536,10.586,10.624,10.652,10.677,10.700,10.723,
     * 10.250,10.309,10.369,10.424,10.473,10.517,10.558,10.596,
     * 10.073,10.091,10.116,10.154,10.200,10.253,10.298,10.351,
     * 9.960,9.972,9.984,9.997,10.011,10.029,10.055,10.080,
     * 9.613,9.620,9.626,9.634,9.635,9.642,9.653,9.667,
     * 9.520,9.511,9.500,9.491,9.487,9.489,9.490,9.498,
     * 9.224,9.214,9.208,9.202,9.199,9.205,9.211,9.222/
C
      DATA ((EDEN(I,1,8,L,2),I=41,48),L=1,NZEN)/
     * 11.074,11.092,11.104,11.109,11.114,11.121,11.125,11.133,
     * 11.037,11.056,11.069,11.074,11.079,11.083,11.087,11.095,
     * 10.974,11.001,11.014,11.023,11.025,11.033,11.036,11.044,
     * 10.820,10.869,10.899,10.918,10.927,10.935,10.944,10.953,
     * 10.408,10.492,10.569,10.632,10.681,10.717,10.745,10.767,
     * 10.227,10.265,10.314,10.368,10.430,10.494,10.552,10.603,
     * 10.165,10.184,10.204,10.220,10.241,10.270,10.308,10.358,
     * 10.048,10.066,10.078,10.084,10.089,10.093,10.106,10.118,
     * 9.730,9.742,9.749,9.753,9.747,9.741,9.736,9.732,
     * 9.644,9.645,9.639,9.628,9.615,9.602,9.590,9.578,
     * 9.365,9.366,9.365,9.356,9.345,9.335,9.322,9.311/
C
      DATA ((EDEN(I,2,8,L,2),I=41,48),L=1,NZEN)/
     * 11.066,11.088,11.100,11.111,11.116,11.119,11.127,11.135,
     * 11.046,11.064,11.073,11.081,11.082,11.085,11.089,11.092,
     * 10.972,10.997,11.012,11.019,11.023,11.027,11.032,11.040,
     * 10.799,10.848,10.880,10.902,10.914,10.923,10.934,10.946,
     * 10.373,10.448,10.520,10.583,10.636,10.681,10.717,10.748,
     * 10.229,10.276,10.331,10.388,10.452,10.511,10.563,10.611,
     * 10.152,10.176,10.199,10.219,10.245,10.283,10.321,10.371,
     * 10.044,10.063,10.076,10.086,10.093,10.101,10.118,10.133,
     * 9.700,9.717,9.729,9.739,9.738,9.737,9.738,9.739,
     * 9.630,9.634,9.633,9.628,9.620,9.613,9.607,9.601,
     * 9.342,9.344,9.341,9.332,9.324,9.316,9.307,9.303/
C
      DATA ((EDEN(I,3,8,L,2),I=41,48),L=1,NZEN)/
     * 11.065,11.085,11.095,11.101,11.104,11.109,11.116,11.124,
     * 11.040,11.057,11.064,11.070,11.069,11.071,11.078,11.082,
     * 10.966,10.991,11.003,11.009,11.012,11.015,11.021,11.029,
     * 10.803,10.849,10.877,10.896,10.906,10.914,10.923,10.935,
     * 10.399,10.475,10.543,10.600,10.646,10.686,10.717,10.745,
     * 10.233,10.284,10.346,10.407,10.466,10.523,10.570,10.614,
     * 10.144,10.167,10.194,10.213,10.238,10.271,10.304,10.348,
     * 10.039,10.055,10.067,10.077,10.085,10.093,10.110,10.127,
     * 9.687,9.701,9.712,9.721,9.721,9.722,9.724,9.729,
     * 9.598,9.599,9.596,9.591,9.585,9.581,9.577,9.577,
     * 9.311,9.307,9.299,9.288,9.277,9.272,9.264,9.261/
C
      DATA ((EDEN(I,4,8,L,2),I=41,48),L=1,NZEN)/
     * 11.076,11.085,11.089,11.092,11.089,11.092,11.097,11.105,
     * 11.031,11.045,11.050,11.053,11.054,11.054,11.059,11.070,
     * 10.958,10.981,10.992,10.997,10.998,11.000,11.006,11.017,
     * 10.802,10.844,10.871,10.887,10.894,10.901,10.910,10.923,
     * 10.405,10.479,10.546,10.600,10.644,10.678,10.708,10.736,
     * 10.209,10.259,10.316,10.376,10.437,10.494,10.543,10.589,
     * 10.105,10.125,10.143,10.170,10.203,10.245,10.288,10.341,
     * 10.001,10.013,10.024,10.034,10.043,10.055,10.076,10.097,
     * 9.667,9.675,9.681,9.687,9.685,9.687,9.691,9.699,
     * 9.581,9.576,9.569,9.559,9.552,9.548,9.545,9.546,
     * 9.280,9.275,9.264,9.255,9.246,9.244,9.241,9.243/
C
      DATA ((EDEN(I,5,8,L,2),I=41,48),L=1,NZEN)/
     * 11.089,11.090,11.087,11.082,11.075,11.072,11.079,11.086,
     * 11.040,11.047,11.043,11.039,11.036,11.033,11.041,11.052,
     * 10.965,10.978,10.984,10.983,10.980,10.980,10.986,10.999,
     * 10.808,10.846,10.867,10.878,10.883,10.886,10.895,10.907,
     * 10.415,10.483,10.543,10.592,10.630,10.662,10.691,10.719,
     * 10.208,10.259,10.319,10.380,10.438,10.492,10.538,10.583,
     * 10.078,10.089,10.105,10.132,10.169,10.217,10.262,10.319,
     * 9.970,9.978,9.985,9.993,10.003,10.016,10.039,10.062,
     * 9.655,9.658,9.660,9.663,9.660,9.661,9.667,9.677,
     * 9.570,9.559,9.548,9.534,9.526,9.522,9.519,9.522,
     * 9.269,9.257,9.247,9.235,9.229,9.227,9.229,9.232/
C
      DATA ((EDEN(I,1,9,L,2),I=41,48),L=1,NZEN)/
     * 11.062,11.084,11.097,11.105,11.110,11.114,11.125,11.133,
     * 11.024,11.047,11.061,11.070,11.075,11.080,11.087,11.095,
     * 10.960,10.989,11.006,11.017,11.021,11.028,11.033,11.042,
     * 10.795,10.849,10.885,10.908,10.921,10.931,10.939,10.951,
     * 10.375,10.456,10.533,10.600,10.655,10.698,10.733,10.760,
     * 10.224,10.256,10.295,10.340,10.397,10.461,10.521,10.578,
     * 10.174,10.192,10.209,10.223,10.238,10.261,10.289,10.335,
     * 10.054,10.072,10.084,10.090,10.093,10.096,10.107,10.118,
     * 9.748,9.761,9.769,9.773,9.766,9.760,9.755,9.750,
     * 9.663,9.665,9.661,9.651,9.639,9.627,9.615,9.604,
     * 9.383,9.387,9.387,9.379,9.369,9.358,9.347,9.338/
C
      DATA ((EDEN(I,2,9,L,2),I=41,48),L=1,NZEN)/
     * 11.059,11.084,11.096,11.104,11.108,11.116,11.123,11.131,
     * 11.033,11.056,11.069,11.073,11.078,11.081,11.085,11.092,
     * 10.952,10.984,11.003,11.013,11.019,11.024,11.030,11.040,
     * 10.758,10.817,10.859,10.888,10.906,10.919,10.931,10.944,
     * 10.330,10.397,10.468,10.536,10.597,10.651,10.696,10.735,
     * 10.218,10.254,10.297,10.347,10.406,10.470,10.528,10.584,
     * 10.161,10.182,10.202,10.217,10.238,10.267,10.295,10.340,
     * 10.048,10.065,10.077,10.085,10.091,10.096,10.109,10.122,
     * 9.729,9.742,9.751,9.757,9.753,9.749,9.747,9.745,
     * 9.643,9.645,9.642,9.633,9.623,9.614,9.604,9.597,
     * 9.359,9.361,9.358,9.350,9.340,9.332,9.322,9.315/
C
      DATA ((EDEN(I,3,9,L,2),I=41,48),L=1,NZEN)/
     * 11.057,11.081,11.091,11.097,11.104,11.106,11.112,11.124,
     * 11.014,11.040,11.056,11.062,11.065,11.068,11.078,11.085,
     * 10.941,10.974,10.995,11.005,11.011,11.015,11.022,11.032,
     * 10.759,10.816,10.856,10.883,10.900,10.911,10.923,10.936,
     * 10.346,10.417,10.489,10.555,10.612,10.660,10.700,10.734,
     * 10.214,10.256,10.306,10.361,10.421,10.484,10.538,10.589,
     * 10.149,10.172,10.192,10.212,10.235,10.270,10.302,10.348,
     * 10.044,10.060,10.072,10.081,10.088,10.095,10.110,10.125,
     * 9.714,9.726,9.734,9.741,9.737,9.734,9.732,9.733,
     * 9.637,9.637,9.632,9.624,9.614,9.606,9.598,9.592,
     * 9.342,9.342,9.338,9.327,9.318,9.311,9.303,9.297/
C
      DATA ((EDEN(I,4,9,L,2),I=41,48),L=1,NZEN)/
     * 11.064,11.082,11.089,11.092,11.092,11.096,11.101,11.112,
     * 11.008,11.033,11.042,11.049,11.050,11.054,11.063,11.074,
     * 10.933,10.966,10.986,10.995,10.999,11.003,11.009,11.021,
     * 10.761,10.818,10.856,10.880,10.894,10.905,10.913,10.925,
     * 10.350,10.426,10.496,10.561,10.614,10.658,10.694,10.726,
     * 10.195,10.236,10.290,10.351,10.417,10.477,10.531,10.580,
     * 10.120,10.133,10.147,10.163,10.187,10.220,10.256,10.303,
     * 10.010,10.020,10.028,10.033,10.038,10.045,10.060,10.076,
     * 9.708,9.713,9.715,9.715,9.707,9.702,9.700,9.700,
     * 9.625,9.617,9.606,9.591,9.578,9.567,9.557,9.551,
     * 9.324,9.315,9.304,9.289,9.273,9.262,9.252,9.244/
C
      DATA ((EDEN(I,5,9,L,2),I=41,48),L=1,NZEN)/
     * 11.077,11.087,11.090,11.089,11.086,11.083,11.090,11.101,
     * 11.019,11.039,11.043,11.043,11.044,11.041,11.049,11.059,
     * 10.932,10.960,10.976,10.981,10.983,10.985,10.993,11.006,
     * 10.754,10.807,10.844,10.866,10.878,10.888,10.899,10.912,
     * 10.351,10.422,10.493,10.555,10.605,10.646,10.682,10.714,
     * 10.182,10.219,10.272,10.334,10.398,10.460,10.514,10.564,
     * 10.092,10.099,10.104,10.121,10.144,10.182,10.223,10.280,
     * 9.986,9.990,9.992,9.993,9.995,10.002,10.018,10.035,
     * 9.714,9.712,9.706,9.700,9.687,9.679,9.675,9.676,
     * 9.635,9.621,9.603,9.583,9.563,9.549,9.535,9.528,
     * 9.334,9.321,9.305,9.286,9.268,9.253,9.241,9.233/
C
      DATA ((EDEN(I,1,10,L,2),I=41,48),L=1,NZEN)/
     * 11.058,11.080,11.097,11.105,11.110,11.118,11.125,11.137,
     * 11.022,11.047,11.061,11.070,11.075,11.083,11.087,11.099,
     * 10.952,10.986,11.006,11.016,11.023,11.028,11.035,11.045,
     * 10.780,10.839,10.879,10.906,10.921,10.931,10.942,10.952,
     * 10.358,10.436,10.515,10.585,10.644,10.691,10.729,10.759,
     * 10.224,10.253,10.289,10.329,10.384,10.446,10.509,10.570,
     * 10.182,10.200,10.214,10.225,10.238,10.261,10.285,10.327,
     * 10.058,10.076,10.088,10.094,10.098,10.100,10.110,10.120,
     * 9.764,9.776,9.784,9.787,9.780,9.774,9.767,9.762,
     * 9.679,9.682,9.677,9.667,9.655,9.643,9.630,9.620,
     * 9.400,9.403,9.402,9.395,9.385,9.374,9.363,9.352/
C
      DATA ((EDEN(I,2,10,L,2),I=41,48),L=1,NZEN)/
     * 11.063,11.084,11.096,11.104,11.108,11.116,11.123,11.135,
     * 11.021,11.048,11.061,11.069,11.074,11.081,11.085,11.096,
     * 10.947,10.982,11.003,11.014,11.020,11.026,11.033,11.042,
     * 10.769,10.831,10.873,10.901,10.917,10.929,10.939,10.950,
     * 10.347,10.424,10.502,10.575,10.636,10.685,10.724,10.755,
     * 10.214,10.245,10.283,10.326,10.382,10.447,10.509,10.569,
     * 10.170,10.189,10.205,10.218,10.235,10.259,10.285,10.326,
     * 10.047,10.066,10.079,10.087,10.092,10.097,10.108,10.118,
     * 9.760,9.771,9.777,9.780,9.773,9.766,9.759,9.754,
     * 9.676,9.676,9.669,9.660,9.647,9.634,9.620,9.609,
     * 9.388,9.389,9.387,9.378,9.368,9.358,9.345,9.335/
C
      DATA ((EDEN(I,3,10,L,2),I=41,48),L=1,NZEN)/
     * 11.061,11.081,11.095,11.101,11.104,11.109,11.116,11.127,
     * 11.003,11.032,11.048,11.058,11.061,11.067,11.078,11.089,
     * 10.934,10.971,10.994,11.006,11.012,11.017,11.025,11.035,
     * 10.753,10.818,10.861,10.891,10.907,10.919,10.930,10.941,
     * 10.347,10.423,10.501,10.573,10.631,10.679,10.715,10.746,
     * 10.214,10.246,10.288,10.339,10.400,10.463,10.522,10.576,
     * 10.162,10.179,10.196,10.212,10.233,10.259,10.288,10.332,
     * 10.046,10.061,10.072,10.080,10.085,10.090,10.102,10.112,
     * 9.746,9.754,9.757,9.758,9.749,9.740,9.733,9.728,
     * 9.662,9.658,9.649,9.635,9.619,9.605,9.590,9.578,
     * 9.363,9.360,9.352,9.341,9.327,9.315,9.301,9.288/
C
      DATA ((EDEN(I,4,10,L,2),I=41,48),L=1,NZEN)/
     * 11.060,11.082,11.093,11.099,11.099,11.103,11.108,11.119,
     * 11.002,11.029,11.046,11.054,11.058,11.062,11.067,11.081,
     * 10.908,10.948,10.974,10.989,10.996,11.002,11.012,11.024,
     * 10.738,10.802,10.847,10.877,10.894,10.906,10.917,10.930,
     * 10.344,10.421,10.496,10.564,10.620,10.665,10.700,10.730,
     * 10.200,10.239,10.290,10.352,10.418,10.481,10.537,10.584,
     * 10.133,10.142,10.152,10.164,10.184,10.215,10.251,10.302,
     * 10.022,10.030,10.034,10.036,10.037,10.039,10.050,10.062,
     * 9.731,9.733,9.731,9.727,9.713,9.701,9.692,9.686,
     * 9.650,9.641,9.625,9.605,9.585,9.566,9.547,9.533,
     * 9.361,9.355,9.340,9.322,9.302,9.284,9.267,9.251/
C
      DATA ((EDEN(I,5,10,L,2),I=41,48),L=1,NZEN)/
     * 11.065,11.087,11.094,11.097,11.093,11.094,11.101,11.111,
     * 11.006,11.030,11.048,11.051,11.052,11.053,11.060,11.070,
     * 10.911,10.950,10.973,10.986,10.991,10.995,11.002,11.012,
     * 10.716,10.779,10.826,10.857,10.876,10.889,10.902,10.916,
     * 10.331,10.402,10.476,10.544,10.601,10.646,10.682,10.714,
     * 10.186,10.217,10.263,10.324,10.387,10.452,10.507,10.559,
     * 10.109,10.112,10.110,10.119,10.137,10.171,10.212,10.269,
     * 10.003,10.003,10.000,9.995,9.990,9.991,10.000,10.011,
     * 9.741,9.737,9.727,9.715,9.695,9.679,9.666,9.658,
     * 9.666,9.649,9.627,9.601,9.573,9.547,9.523,9.505,
     * 9.383,9.366,9.344,9.318,9.288,9.264,9.237,9.216/
C
      DATA ((EDEN(I,1,11,L,2),I=41,48),L=1,NZEN)/
     * 11.062,11.088,11.104,11.112,11.117,11.125,11.133,11.144,
     * 11.028,11.056,11.069,11.078,11.083,11.087,11.095,11.102,
     * 10.958,10.993,11.013,11.023,11.029,11.038,11.041,11.052,
     * 10.785,10.846,10.888,10.914,10.929,10.941,10.949,10.960,
     * 10.360,10.441,10.523,10.596,10.656,10.704,10.740,10.769,
     * 10.224,10.256,10.291,10.336,10.394,10.459,10.524,10.585,
     * 10.182,10.200,10.214,10.228,10.241,10.263,10.291,10.337,
     * 10.058,10.076,10.088,10.094,10.098,10.102,10.113,10.123,
     * 9.771,9.782,9.788,9.790,9.782,9.775,9.768,9.762,
     * 9.687,9.688,9.682,9.671,9.657,9.643,9.629,9.617,
     * 9.408,9.410,9.407,9.399,9.386,9.374,9.361,9.349/
C
      DATA ((EDEN(I,2,11,L,2),I=41,48),L=1,NZEN)/
     * 11.074,11.092,11.104,11.111,11.116,11.123,11.130,11.138,
     * 11.017,11.044,11.057,11.069,11.074,11.081,11.088,11.100,
     * 10.956,10.990,11.010,11.021,11.028,11.035,11.038,11.049,
     * 10.770,10.835,10.879,10.908,10.925,10.936,10.946,10.958,
     * 10.339,10.418,10.501,10.578,10.642,10.693,10.731,10.763,
     * 10.217,10.246,10.281,10.327,10.384,10.451,10.517,10.578,
     * 10.174,10.189,10.206,10.218,10.235,10.258,10.286,10.330,
     * 10.047,10.066,10.078,10.087,10.091,10.096,10.106,10.115,
     * 9.782,9.790,9.792,9.790,9.780,9.769,9.759,9.750,
     * 9.700,9.697,9.687,9.672,9.655,9.637,9.619,9.603,
     * 9.401,9.401,9.395,9.382,9.368,9.353,9.336,9.323/
C
      DATA ((EDEN(I,3,11,L,2),I=41,48),L=1,NZEN)/
     * 11.073,11.092,11.102,11.108,11.111,11.113,11.123,11.131,
     * 11.018,11.045,11.056,11.066,11.069,11.075,11.081,11.092,
     * 10.932,10.968,10.990,11.003,11.010,11.016,11.025,11.037,
     * 10.774,10.833,10.876,10.900,10.916,10.926,10.935,10.947,
     * 10.348,10.426,10.506,10.579,10.639,10.686,10.722,10.752,
     * 10.223,10.264,10.316,10.380,10.445,10.511,10.564,10.612,
     * 10.167,10.183,10.198,10.213,10.233,10.260,10.289,10.335,
     * 10.049,10.065,10.075,10.082,10.085,10.088,10.097,10.106,
     * 9.753,9.758,9.759,9.756,9.742,9.729,9.717,9.706,
     * 9.682,9.675,9.663,9.644,9.623,9.603,9.582,9.564,
     * 9.382,9.378,9.369,9.352,9.333,9.316,9.296,9.278/
C
      DATA ((EDEN(I,4,11,L,2),I=41,48),L=1,NZEN)/
     * 11.072,11.093,11.100,11.107,11.107,11.110,11.115,11.126,
     * 11.013,11.042,11.054,11.061,11.066,11.069,11.074,11.088,
     * 10.922,10.961,10.985,10.998,11.005,11.011,11.020,11.030,
     * 10.736,10.799,10.844,10.875,10.894,10.908,10.919,10.933,
     * 10.369,10.448,10.524,10.590,10.642,10.682,10.713,10.739,
     * 10.214,10.258,10.313,10.377,10.443,10.504,10.554,10.598,
     * 10.136,10.148,10.157,10.171,10.191,10.227,10.271,10.325,
     * 10.032,10.038,10.040,10.039,10.037,10.037,10.045,10.054,
     * 9.731,9.732,9.727,9.720,9.702,9.686,9.671,9.659,
     * 9.651,9.638,9.620,9.596,9.569,9.544,9.518,9.496,
     * 9.370,9.359,9.344,9.321,9.297,9.272,9.247,9.224/
C
      DATA ((EDEN(I,5,11,L,2),I=41,48),L=1,NZEN)/
     * 11.073,11.090,11.101,11.104,11.104,11.104,11.111,11.122,
     * 11.012,11.039,11.052,11.059,11.060,11.060,11.067,11.081,
     * 10.917,10.957,10.981,10.993,10.998,11.003,11.010,11.023,
     * 10.724,10.788,10.836,10.867,10.885,10.896,10.909,10.922,
     * 10.325,10.392,10.465,10.535,10.593,10.641,10.679,10.712,
     * 10.195,10.228,10.272,10.331,10.395,10.459,10.514,10.562,
     * 10.120,10.119,10.119,10.124,10.140,10.170,10.215,10.274,
     * 10.017,10.014,10.007,9.997,9.988,9.983,9.986,9.992,
     * 9.741,9.735,9.724,9.708,9.684,9.661,9.641,9.625,
     * 9.662,9.645,9.621,9.591,9.557,9.525,9.493,9.465,
     * 9.406,9.388,9.363,9.331,9.297,9.262,9.226,9.193/
C
      DATA ((EDEN(I,1,12,L,2),I=41,48),L=1,NZEN)/
     * 11.074,11.099,11.111,11.123,11.128,11.135,11.143,11.151,
     * 11.037,11.064,11.081,11.089,11.095,11.099,11.106,11.110,
     * 10.967,11.003,11.023,11.032,11.042,11.046,11.053,11.061,
     * 10.796,10.859,10.899,10.925,10.939,10.949,10.957,10.967,
     * 10.371,10.456,10.540,10.615,10.674,10.720,10.753,10.780,
     * 10.227,10.258,10.299,10.350,10.412,10.481,10.546,10.604,
     * 10.179,10.198,10.214,10.225,10.243,10.270,10.302,10.352,
     * 10.055,10.074,10.086,10.093,10.097,10.102,10.114,10.126,
     * 9.773,9.783,9.787,9.788,9.779,9.770,9.763,9.757,
     * 9.691,9.689,9.679,9.667,9.652,9.637,9.621,9.607,
     * 9.410,9.412,9.406,9.395,9.381,9.369,9.353,9.341/
C
      DATA ((EDEN(I,2,12,L,2),I=41,48),L=1,NZEN)/
     * 11.086,11.103,11.115,11.118,11.123,11.130,11.137,11.145,
     * 11.029,11.056,11.069,11.077,11.082,11.089,11.096,11.107,
     * 10.963,10.995,11.013,11.023,11.032,11.035,11.042,11.053,
     * 10.780,10.846,10.890,10.918,10.934,10.945,10.954,10.966,
     * 10.346,10.427,10.513,10.592,10.657,10.706,10.743,10.772,
     * 10.225,10.267,10.323,10.391,10.463,10.532,10.589,10.635,
     * 10.172,10.190,10.204,10.216,10.233,10.260,10.291,10.340,
     * 10.045,10.063,10.076,10.084,10.089,10.093,10.103,10.113,
     * 9.770,9.777,9.778,9.776,9.761,9.748,9.735,9.724,
     * 9.712,9.704,9.692,9.674,9.653,9.632,9.610,9.592,
     * 9.408,9.404,9.396,9.380,9.362,9.344,9.325,9.307/
C
      DATA ((EDEN(I,3,12,L,2),I=41,48),L=1,NZEN)/
     * 11.084,11.100,11.110,11.115,11.118,11.124,11.130,11.138,
     * 11.032,11.053,11.064,11.074,11.077,11.083,11.089,11.100,
     * 10.949,10.981,11.000,11.012,11.017,11.025,11.031,11.045,
     * 10.794,10.850,10.887,10.910,10.924,10.933,10.942,10.954,
     * 10.393,10.479,10.560,10.626,10.678,10.716,10.744,10.766,
     * 10.229,10.274,10.329,10.396,10.463,10.528,10.578,10.623,
     * 10.163,10.179,10.199,10.219,10.244,10.283,10.325,10.377,
     * 10.051,10.065,10.075,10.082,10.085,10.090,10.100,10.108,
     * 9.755,9.758,9.756,9.751,9.735,9.720,9.706,9.692,
     * 9.673,9.663,9.646,9.623,9.599,9.574,9.549,9.527,
     * 9.385,9.379,9.368,9.348,9.326,9.305,9.280,9.258/
C
      DATA ((EDEN(I,4,12,L,2),I=41,48),L=1,NZEN)/
     * 11.084,11.101,11.108,11.114,11.114,11.117,11.122,11.133,
     * 11.026,11.050,11.062,11.069,11.073,11.077,11.085,11.092,
     * 10.938,10.973,10.995,11.006,11.012,11.015,11.024,11.038,
     * 10.759,10.818,10.859,10.887,10.904,10.915,10.926,10.938,
     * 10.391,10.471,10.545,10.608,10.656,10.693,10.721,10.744,
     * 10.231,10.278,10.338,10.403,10.465,10.521,10.567,10.605,
     * 10.141,10.150,10.162,10.180,10.204,10.244,10.290,10.344,
     * 10.036,10.042,10.043,10.041,10.038,10.038,10.044,10.050,
     * 9.723,9.724,9.719,9.711,9.691,9.672,9.654,9.638,
     * 9.645,9.633,9.613,9.587,9.558,9.529,9.500,9.474,
     * 9.369,9.358,9.340,9.316,9.287,9.259,9.229,9.200/
C
      DATA ((EDEN(I,5,12,L,2),I=41,48),L=1,NZEN)/
     * 11.081,11.098,11.105,11.107,11.107,11.111,11.118,11.125,
     * 11.024,11.048,11.060,11.063,11.067,11.068,11.074,11.088,
     * 10.932,10.968,10.990,11.000,11.007,11.007,11.018,11.026,
     * 10.747,10.807,10.849,10.876,10.892,10.903,10.913,10.926,
     * 10.349,10.420,10.489,10.555,10.608,10.653,10.687,10.716,
     * 10.203,10.233,10.276,10.333,10.393,10.455,10.510,10.558,
     * 10.126,10.125,10.124,10.128,10.139,10.168,10.211,10.267,
     * 10.024,10.020,10.010,9.998,9.984,9.976,9.976,9.978,
     * 9.730,9.726,9.714,9.698,9.671,9.645,9.621,9.601,
     * 9.653,9.636,9.611,9.579,9.543,9.506,9.469,9.436,
     * 9.380,9.364,9.341,9.308,9.271,9.234,9.194,9.158/
C
      DATA ((EDEN(I,1,1,L,3),I=41,48),L=1,NZEN)/
     * 11.152,11.177,11.194,11.206,11.216,11.223,11.234,11.243,
     * 11.116,11.142,11.160,11.171,11.178,11.186,11.196,11.205,
     * 11.049,11.079,11.099,11.115,11.123,11.131,11.139,11.150,
     * 10.884,10.940,10.977,11.001,11.013,11.026,11.038,11.050,
     * 10.478,10.563,10.643,10.709,10.760,10.799,10.828,10.853,
     * 10.322,10.361,10.407,10.461,10.523,10.587,10.643,10.693,
     * 10.269,10.293,10.314,10.331,10.352,10.383,10.422,10.471,
     * 10.131,10.154,10.169,10.179,10.185,10.191,10.207,10.221,
     * 9.756,9.772,9.784,9.792,9.787,9.785,9.782,9.781,
     * 9.659,9.663,9.659,9.650,9.640,9.631,9.622,9.617,
     * 9.372,9.378,9.377,9.372,9.363,9.356,9.349,9.345/
C
      DATA ((EDEN(I,2,1,L,3),I=41,48),L=1,NZEN)/
     * 11.163,11.183,11.196,11.205,11.212,11.218,11.229,11.237,
     * 11.110,11.133,11.147,11.157,11.167,11.174,11.184,11.196,
     * 11.042,11.072,11.092,11.103,11.111,11.122,11.132,11.144,
     * 10.871,10.930,10.969,10.994,11.009,11.024,11.035,11.046,
     * 10.451,10.536,10.620,10.691,10.746,10.789,10.819,10.845,
     * 10.320,10.371,10.432,10.502,10.569,10.630,10.678,10.718,
     * 10.259,10.281,10.301,10.319,10.342,10.375,10.411,10.459,
     * 10.119,10.142,10.159,10.170,10.177,10.184,10.197,10.209,
     * 9.754,9.767,9.776,9.781,9.772,9.764,9.757,9.751,
     * 9.682,9.680,9.671,9.657,9.642,9.627,9.612,9.601,
     * 9.373,9.375,9.371,9.361,9.348,9.336,9.324,9.315/
C
      DATA ((EDEN(I,3,1,L,3),I=41,48),L=1,NZEN)/
     * 11.165,11.181,11.192,11.199,11.204,11.212,11.219,11.230,
     * 11.109,11.130,11.143,11.154,11.160,11.168,11.177,11.189,
     * 11.029,11.058,11.077,11.086,11.096,11.105,11.115,11.129,
     * 10.881,10.929,10.960,10.980,10.993,11.004,11.015,11.029,
     * 10.500,10.585,10.659,10.717,10.761,10.794,10.818,10.838,
     * 10.326,10.378,10.439,10.506,10.568,10.624,10.667,10.705,
     * 10.253,10.278,10.302,10.326,10.358,10.401,10.443,10.490,
     * 10.132,10.150,10.164,10.173,10.179,10.184,10.196,10.206,
     * 9.745,9.756,9.761,9.763,9.753,9.743,9.733,9.724,
     * 9.650,9.644,9.633,9.616,9.597,9.579,9.561,9.546,
     * 9.360,9.358,9.351,9.337,9.321,9.307,9.289,9.275/
C
      DATA ((EDEN(I,4,1,L,3),I=41,48),L=1,NZEN)/
     * 11.162,11.179,11.190,11.197,11.199,11.205,11.213,11.224,
     * 11.106,11.128,11.142,11.150,11.152,11.158,11.169,11.183,
     * 11.020,11.052,11.072,11.082,11.089,11.096,11.107,11.121,
     * 10.849,10.902,10.938,10.962,10.976,10.987,10.999,11.013,
     * 10.498,10.574,10.642,10.697,10.737,10.769,10.792,10.814,
     * 10.330,10.384,10.445,10.507,10.562,10.610,10.648,10.681,
     * 10.234,10.248,10.264,10.287,10.316,10.359,10.401,10.450,
     * 10.126,10.134,10.136,10.136,10.133,10.131,10.139,10.145,
     * 9.720,9.727,9.729,9.728,9.712,9.697,9.682,9.669,
     * 9.632,9.623,9.607,9.586,9.561,9.537,9.511,9.491,
     * 9.350,9.344,9.331,9.311,9.286,9.262,9.237,9.212/
C
      DATA ((EDEN(I,5,1,L,3),I=41,48),L=1,NZEN)/
     * 11.163,11.180,11.187,11.191,11.192,11.198,11.204,11.217,
     * 11.104,11.127,11.136,11.144,11.146,11.152,11.161,11.174,
     * 11.017,11.047,11.068,11.074,11.080,11.087,11.097,11.109,
     * 10.842,10.894,10.930,10.951,10.964,10.974,10.986,11.002,
     * 10.454,10.525,10.592,10.650,10.694,10.730,10.759,10.784,
     * 10.297,10.335,10.381,10.438,10.496,10.549,10.595,10.636,
     * 10.214,10.217,10.218,10.228,10.248,10.281,10.322,10.373,
     * 10.112,10.109,10.101,10.089,10.076,10.068,10.069,10.071,
     * 9.740,9.739,9.732,9.720,9.695,9.671,9.649,9.629,
     * 9.647,9.633,9.610,9.581,9.547,9.514,9.480,9.451,
     * 9.372,9.358,9.337,9.307,9.274,9.239,9.201,9.169/
C
      DATA ((EDEN(I,1,2,L,3),I=41,48),L=1,NZEN)/
     * 11.142,11.167,11.184,11.197,11.208,11.218,11.228,11.237,
     * 11.105,11.132,11.151,11.161,11.172,11.179,11.187,11.199,
     * 11.037,11.071,11.092,11.104,11.113,11.124,11.132,11.144,
     * 10.870,10.927,10.965,10.990,11.005,11.017,11.029,11.045,
     * 10.464,10.547,10.625,10.692,10.744,10.785,10.816,10.843,
     * 10.320,10.356,10.398,10.447,10.504,10.568,10.625,10.678,
     * 10.271,10.293,10.314,10.329,10.348,10.376,10.409,10.457,
     * 10.132,10.154,10.169,10.178,10.183,10.188,10.203,10.217,
     * 9.757,9.775,9.787,9.795,9.791,9.788,9.786,9.785,
     * 9.662,9.666,9.663,9.655,9.646,9.637,9.629,9.625,
     * 9.373,9.380,9.382,9.376,9.369,9.362,9.356,9.352/
C
      DATA ((EDEN(I,2,2,L,3),I=41,48),L=1,NZEN)/
     * 11.150,11.171,11.184,11.196,11.203,11.213,11.223,11.234,
     * 11.095,11.119,11.137,11.147,11.158,11.168,11.178,11.193,
     * 11.034,11.068,11.088,11.100,11.108,11.118,11.129,11.141,
     * 10.859,10.917,10.958,10.984,11.000,11.014,11.026,11.042,
     * 10.443,10.525,10.606,10.677,10.733,10.777,10.810,10.838,
     * 10.316,10.365,10.425,10.489,10.556,10.616,10.665,10.708,
     * 10.258,10.281,10.301,10.319,10.339,10.370,10.403,10.449,
     * 10.120,10.143,10.159,10.170,10.177,10.183,10.196,10.208,
     * 9.773,9.786,9.793,9.797,9.789,9.782,9.776,9.771,
     * 9.679,9.679,9.672,9.658,9.645,9.631,9.618,9.608,
     * 9.375,9.378,9.376,9.368,9.356,9.346,9.335,9.326/
C
      DATA ((EDEN(I,3,2,L,3),I=41,48),L=1,NZEN)/
     * 11.152,11.168,11.183,11.190,11.198,11.203,11.216,11.227,
     * 11.094,11.116,11.133,11.144,11.150,11.159,11.171,11.182,
     * 11.007,11.042,11.065,11.079,11.085,11.098,11.108,11.122,
     * 10.859,10.914,10.950,10.974,10.989,11.001,11.013,11.030,
     * 10.479,10.562,10.638,10.699,10.746,10.782,10.809,10.832,
     * 10.318,10.366,10.425,10.487,10.551,10.609,10.655,10.694,
     * 10.250,10.275,10.297,10.323,10.352,10.392,10.432,10.478,
     * 10.128,10.148,10.162,10.172,10.179,10.185,10.198,10.208,
     * 9.748,9.760,9.767,9.771,9.762,9.753,9.745,9.737,
     * 9.668,9.664,9.653,9.639,9.621,9.605,9.588,9.575,
     * 9.365,9.365,9.358,9.345,9.331,9.317,9.301,9.289/
C
      DATA ((EDEN(I,4,2,L,3),I=41,48),L=1,NZEN)/
     * 11.149,11.169,11.181,11.188,11.193,11.199,11.207,11.218,
     * 11.091,11.114,11.131,11.140,11.146,11.152,11.163,11.177,
     * 11.000,11.036,11.060,11.071,11.082,11.089,11.101,11.114,
     * 10.821,10.878,10.920,10.948,10.965,10.979,10.992,11.008,
     * 10.461,10.539,10.611,10.672,10.718,10.754,10.782,10.807,
     * 10.309,10.358,10.415,10.479,10.537,10.591,10.634,10.671,
     * 10.226,10.243,10.257,10.277,10.303,10.342,10.380,10.428,
     * 10.115,10.125,10.130,10.131,10.130,10.129,10.138,10.146,
     * 9.732,9.739,9.740,9.738,9.723,9.709,9.696,9.684,
     * 9.641,9.633,9.617,9.596,9.573,9.551,9.528,9.509,
     * 9.358,9.352,9.338,9.319,9.298,9.277,9.253,9.233/
C
      DATA ((EDEN(I,5,2,L,3),I=41,48),L=1,NZEN)/
     * 11.150,11.170,11.181,11.185,11.186,11.189,11.198,11.212,
     * 11.089,11.116,11.130,11.137,11.140,11.143,11.155,11.168,
     * 10.999,11.035,11.056,11.070,11.073,11.080,11.090,11.103,
     * 10.814,10.873,10.914,10.940,10.955,10.967,10.981,10.996,
     * 10.424,10.493,10.563,10.625,10.676,10.717,10.750,10.778,
     * 10.281,10.317,10.363,10.421,10.481,10.538,10.586,10.630,
     * 10.205,10.211,10.212,10.225,10.246,10.278,10.319,10.368,
     * 10.101,10.101,10.096,10.089,10.080,10.074,10.078,10.082,
     * 9.747,9.747,9.740,9.730,9.707,9.685,9.665,9.648,
     * 9.660,9.646,9.623,9.596,9.564,9.534,9.502,9.476,
     * 9.392,9.378,9.355,9.327,9.293,9.261,9.226,9.195/
C
      DATA ((EDEN(I,1,3,L,3),I=41,48),L=1,NZEN)/
     * 11.132,11.158,11.172,11.185,11.196,11.206,11.217,11.228,
     * 11.094,11.122,11.137,11.151,11.160,11.167,11.178,11.189,
     * 11.024,11.060,11.081,11.093,11.102,11.113,11.121,11.133,
     * 10.858,10.914,10.953,10.978,10.994,11.007,11.019,11.034,
     * 10.454,10.532,10.608,10.674,10.727,10.770,10.804,10.832,
     * 10.318,10.352,10.391,10.434,10.489,10.549,10.607,10.662,
     * 10.271,10.293,10.314,10.328,10.345,10.369,10.399,10.442,
     * 10.133,10.155,10.169,10.177,10.182,10.186,10.200,10.214,
     * 9.754,9.773,9.787,9.796,9.793,9.791,9.789,9.789,
     * 9.659,9.665,9.663,9.656,9.649,9.641,9.635,9.631,
     * 9.370,9.378,9.382,9.378,9.372,9.367,9.361,9.358/
C
      DATA ((EDEN(I,2,3,L,3),I=41,48),L=1,NZEN)/
     * 11.133,11.158,11.172,11.184,11.191,11.201,11.211,11.226,
     * 11.095,11.119,11.137,11.147,11.155,11.165,11.172,11.187,
     * 11.021,11.056,11.077,11.089,11.097,11.108,11.119,11.130,
     * 10.849,10.907,10.947,10.974,10.990,11.003,11.015,11.030,
     * 10.440,10.520,10.599,10.667,10.721,10.766,10.799,10.828,
     * 10.304,10.340,10.382,10.431,10.488,10.550,10.608,10.661,
     * 10.255,10.280,10.302,10.319,10.339,10.367,10.398,10.442,
     * 10.122,10.144,10.160,10.171,10.178,10.184,10.197,10.210,
     * 9.760,9.776,9.786,9.793,9.788,9.783,9.779,9.776,
     * 9.665,9.667,9.663,9.654,9.643,9.633,9.623,9.616,
     * 9.370,9.376,9.376,9.370,9.361,9.354,9.345,9.339/
C
      DATA ((EDEN(I,3,3,L,3),I=41,48),L=1,NZEN)/
     * 11.132,11.155,11.170,11.178,11.186,11.195,11.204,11.218,
     * 11.075,11.102,11.119,11.131,11.140,11.149,11.159,11.176,
     * 10.999,11.033,11.057,11.071,11.082,11.091,11.105,11.116,
     * 10.836,10.895,10.936,10.962,10.979,10.992,11.003,11.018,
     * 10.440,10.521,10.597,10.663,10.715,10.757,10.789,10.817,
     * 10.309,10.358,10.414,10.477,10.540,10.597,10.644,10.686,
     * 10.248,10.271,10.294,10.314,10.339,10.372,10.403,10.447,
     * 10.125,10.145,10.159,10.169,10.175,10.180,10.193,10.204,
     * 9.751,9.764,9.772,9.777,9.769,9.763,9.757,9.752,
     * 9.656,9.655,9.648,9.635,9.622,9.609,9.596,9.587,
     * 9.360,9.362,9.358,9.348,9.337,9.327,9.315,9.307/
C
      DATA ((EDEN(I,4,3,L,3),I=41,48),L=1,NZEN)/
     * 11.135,11.156,11.172,11.176,11.182,11.188,11.196,11.210,
     * 11.076,11.104,11.121,11.130,11.136,11.143,11.151,11.165,
     * 10.982,11.021,11.048,11.063,11.071,11.078,11.090,11.104,
     * 10.807,10.867,10.911,10.941,10.960,10.973,10.986,11.002,
     * 10.438,10.519,10.593,10.656,10.705,10.744,10.774,10.800,
     * 10.291,10.335,10.391,10.453,10.515,10.571,10.617,10.658,
     * 10.216,10.231,10.248,10.269,10.297,10.337,10.376,10.424,
     * 10.104,10.116,10.124,10.129,10.132,10.135,10.147,10.158,
     * 9.733,9.741,9.743,9.744,9.732,9.721,9.712,9.705,
     * 9.648,9.640,9.627,9.609,9.590,9.572,9.554,9.541,
     * 9.359,9.353,9.341,9.324,9.306,9.290,9.271,9.256/
C
      DATA ((EDEN(I,5,3,L,3),I=41,48),L=1,NZEN)/
     * 11.143,11.164,11.172,11.176,11.178,11.181,11.187,11.200,
     * 11.085,11.109,11.123,11.127,11.130,11.134,11.143,11.156,
     * 10.990,11.026,11.051,11.058,11.066,11.070,11.080,11.093,
     * 10.801,10.861,10.905,10.931,10.948,10.960,10.972,10.988,
     * 10.414,10.488,10.560,10.625,10.676,10.716,10.749,10.777,
     * 10.269,10.309,10.359,10.422,10.483,10.540,10.588,10.631,
     * 10.194,10.200,10.205,10.224,10.249,10.286,10.323,10.373,
     * 10.083,10.088,10.089,10.088,10.085,10.086,10.094,10.102,
     * 9.750,9.750,9.745,9.737,9.718,9.701,9.686,9.674,
     * 9.667,9.653,9.632,9.606,9.579,9.554,9.528,9.508,
     * 9.388,9.374,9.351,9.323,9.295,9.267,9.239,9.215/
C
      DATA ((EDEN(I,1,4,L,3),I=41,48),L=1,NZEN)/
     * 11.129,11.155,11.169,11.179,11.187,11.197,11.208,11.222,
     * 11.094,11.118,11.134,11.145,11.153,11.160,11.171,11.183,
     * 11.028,11.056,11.077,11.089,11.095,11.106,11.114,11.126,
     * 10.864,10.916,10.951,10.974,10.989,11.000,11.013,11.027,
     * 10.461,10.540,10.613,10.676,10.727,10.767,10.800,10.828,
     * 10.318,10.352,10.391,10.437,10.491,10.550,10.606,10.659,
     * 10.269,10.291,10.312,10.326,10.343,10.369,10.399,10.442,
     * 10.134,10.156,10.169,10.177,10.181,10.184,10.198,10.212,
     * 9.742,9.762,9.777,9.787,9.785,9.784,9.783,9.783,
     * 9.646,9.651,9.652,9.647,9.640,9.633,9.627,9.624,
     * 9.356,9.366,9.370,9.368,9.363,9.358,9.354,9.353/
C
      DATA ((EDEN(I,2,4,L,3),I=41,48),L=1,NZEN)/
     * 11.126,11.148,11.165,11.175,11.185,11.195,11.205,11.220,
     * 11.088,11.115,11.130,11.141,11.148,11.155,11.166,11.177,
     * 11.021,11.052,11.073,11.085,11.093,11.101,11.112,11.123,
     * 10.860,10.913,10.949,10.971,10.986,10.998,11.008,11.022,
     * 10.415,10.483,10.554,10.621,10.679,10.729,10.769,10.805,
     * 10.307,10.346,10.393,10.444,10.501,10.561,10.614,10.665,
     * 10.254,10.279,10.302,10.320,10.343,10.373,10.405,10.449,
     * 10.127,10.148,10.163,10.172,10.178,10.184,10.199,10.214,
     * 9.732,9.751,9.766,9.776,9.774,9.774,9.773,9.774,
     * 9.635,9.640,9.640,9.634,9.627,9.621,9.615,9.613,
     * 9.351,9.359,9.363,9.359,9.354,9.350,9.346,9.344/
C
      DATA ((EDEN(I,3,4,L,3),I=41,48),L=1,NZEN)/
     * 11.125,11.149,11.164,11.172,11.177,11.186,11.195,11.210,
     * 11.063,11.094,11.112,11.124,11.134,11.143,11.152,11.167,
     * 11.008,11.041,11.061,11.075,11.082,11.088,11.098,11.113,
     * 10.847,10.901,10.938,10.961,10.975,10.986,10.997,11.011,
     * 10.458,10.541,10.614,10.676,10.723,10.761,10.790,10.816,
     * 10.303,10.349,10.402,10.459,10.518,10.576,10.625,10.668,
     * 10.242,10.267,10.292,10.315,10.343,10.380,10.415,10.457,
     * 10.126,10.145,10.159,10.170,10.177,10.184,10.200,10.215,
     * 9.722,9.738,9.750,9.759,9.757,9.756,9.755,9.756,
     * 9.626,9.627,9.622,9.614,9.606,9.601,9.594,9.590,
     * 9.337,9.339,9.338,9.332,9.326,9.322,9.316,9.314/
C
      DATA ((EDEN(I,4,4,L,3),I=41,48),L=1,NZEN)/
     * 11.139,11.156,11.165,11.170,11.173,11.179,11.187,11.198,
     * 11.079,11.103,11.117,11.123,11.126,11.133,11.142,11.156,
     * 10.995,11.031,11.052,11.063,11.067,11.075,11.084,11.098,
     * 10.835,10.890,10.926,10.949,10.961,10.972,10.982,10.995,
     * 10.446,10.526,10.598,10.659,10.706,10.743,10.772,10.797,
     * 10.281,10.329,10.386,10.449,10.510,10.568,10.613,10.655,
     * 10.205,10.224,10.241,10.268,10.300,10.345,10.382,10.429,
     * 10.093,10.108,10.119,10.128,10.135,10.143,10.160,10.175,
     * 9.716,9.725,9.731,9.736,9.730,9.726,9.724,9.724,
     * 9.628,9.623,9.612,9.599,9.587,9.577,9.567,9.562,
     * 9.332,9.328,9.318,9.306,9.294,9.287,9.276,9.271/
C
      DATA ((EDEN(I,5,4,L,3),I=41,48),L=1,NZEN)/
     * 11.153,11.167,11.172,11.170,11.169,11.169,11.175,11.186,
     * 11.096,11.116,11.122,11.124,11.123,11.124,11.130,11.144,
     * 11.010,11.038,11.055,11.058,11.058,11.063,11.070,11.084,
     * 10.833,10.884,10.918,10.938,10.949,10.957,10.967,10.981,
     * 10.439,10.518,10.588,10.647,10.691,10.725,10.753,10.780,
     * 10.265,10.314,10.371,10.436,10.497,10.552,10.597,10.638,
     * 10.176,10.186,10.199,10.225,10.260,10.305,10.346,10.396,
     * 10.066,10.076,10.083,10.089,10.094,10.101,10.117,10.132,
     * 9.729,9.730,9.728,9.725,9.713,9.704,9.698,9.696,
     * 9.643,9.629,9.610,9.590,9.570,9.555,9.539,9.529,
     * 9.347,9.333,9.316,9.296,9.276,9.263,9.246,9.234/
C
      DATA ((EDEN(I,1,5,L,3),I=41,48),L=1,NZEN)/
     * 11.142,11.161,11.175,11.185,11.193,11.200,11.211,11.222,
     * 11.105,11.125,11.141,11.148,11.156,11.164,11.171,11.183,
     * 11.041,11.068,11.085,11.093,11.102,11.110,11.118,11.126,
     * 10.887,10.934,10.964,10.982,10.995,11.006,11.016,11.030,
     * 10.490,10.571,10.644,10.702,10.747,10.783,10.809,10.833,
     * 10.320,10.359,10.407,10.460,10.517,10.576,10.628,10.677,
     * 10.265,10.287,10.308,10.326,10.347,10.376,10.412,10.458,
     * 10.133,10.153,10.167,10.175,10.179,10.183,10.198,10.213,
     * 9.724,9.745,9.759,9.771,9.769,9.768,9.768,9.769,
     * 9.626,9.633,9.633,9.627,9.620,9.613,9.607,9.604,
     * 9.337,9.347,9.350,9.347,9.342,9.338,9.336,9.333/
C
      DATA ((EDEN(I,2,5,L,3),I=41,48),L=1,NZEN)/
     * 11.133,11.155,11.172,11.181,11.188,11.198,11.208,11.220,
     * 11.113,11.132,11.147,11.154,11.158,11.165,11.172,11.178,
     * 11.038,11.064,11.080,11.092,11.097,11.104,11.112,11.124,
     * 10.868,10.915,10.947,10.968,10.981,10.993,11.006,11.019,
     * 10.457,10.531,10.600,10.660,10.710,10.752,10.785,10.816,
     * 10.318,10.367,10.423,10.479,10.537,10.594,10.641,10.685,
     * 10.249,10.276,10.301,10.324,10.352,10.389,10.427,10.473,
     * 10.129,10.149,10.163,10.173,10.180,10.187,10.205,10.222,
     * 9.699,9.721,9.738,9.752,9.753,9.756,9.760,9.764,
     * 9.600,9.606,9.607,9.605,9.600,9.599,9.596,9.597,
     * 9.325,9.331,9.334,9.330,9.326,9.325,9.322,9.324/
C
      DATA ((EDEN(I,3,5,L,3),I=41,48),L=1,NZEN)/
     * 11.135,11.155,11.167,11.175,11.180,11.186,11.195,11.207,
     * 11.094,11.116,11.129,11.138,11.144,11.149,11.156,11.167,
     * 11.032,11.057,11.072,11.082,11.085,11.092,11.098,11.110,
     * 10.873,10.917,10.946,10.964,10.975,10.984,10.995,11.009,
     * 10.481,10.557,10.624,10.679,10.722,10.758,10.786,10.812,
     * 10.319,10.375,10.438,10.499,10.555,10.607,10.649,10.688,
     * 10.240,10.268,10.295,10.318,10.347,10.382,10.415,10.456,
     * 10.127,10.146,10.160,10.172,10.180,10.187,10.206,10.223,
     * 9.686,9.707,9.723,9.739,9.740,9.744,9.748,9.754,
     * 9.598,9.603,9.603,9.600,9.597,9.598,9.596,9.599,
     * 9.306,9.309,9.309,9.304,9.299,9.300,9.299,9.302/
C
      DATA ((EDEN(I,4,5,L,3),I=41,48),L=1,NZEN)/
     * 11.152,11.163,11.168,11.170,11.170,11.173,11.181,11.192,
     * 11.098,11.114,11.121,11.123,11.126,11.130,11.139,11.150,
     * 11.028,11.051,11.063,11.071,11.071,11.075,11.081,11.095,
     * 10.874,10.916,10.942,10.956,10.963,10.971,10.979,10.993,
     * 10.489,10.566,10.630,10.681,10.720,10.751,10.776,10.800,
     * 10.294,10.350,10.412,10.473,10.530,10.583,10.625,10.665,
     * 10.197,10.221,10.244,10.276,10.314,10.363,10.401,10.449,
     * 10.086,10.102,10.116,10.130,10.141,10.153,10.175,10.194,
     * 9.680,9.693,9.704,9.715,9.715,9.719,9.724,9.731,
     * 9.587,9.584,9.578,9.570,9.565,9.565,9.563,9.565,
     * 9.281,9.277,9.271,9.265,9.259,9.262,9.260,9.264/
C
      DATA ((EDEN(I,5,5,L,3),I=41,48),L=1,NZEN)/
     * 11.170,11.173,11.172,11.167,11.163,11.160,11.163,11.174,
     * 11.118,11.126,11.126,11.124,11.117,11.118,11.121,11.132,
     * 11.043,11.058,11.062,11.058,11.055,11.056,11.063,11.074,
     * 10.886,10.922,10.942,10.951,10.955,10.957,10.965,10.978,
     * 10.505,10.579,10.637,10.683,10.716,10.743,10.765,10.787,
     * 10.296,10.358,10.422,10.484,10.538,10.586,10.623,10.658,
     * 10.166,10.182,10.204,10.242,10.288,10.342,10.383,10.432,
     * 10.055,10.069,10.080,10.092,10.104,10.119,10.141,10.162,
     * 9.678,9.684,9.688,9.694,9.691,9.692,9.697,9.704,
     * 9.585,9.574,9.561,9.548,9.539,9.537,9.531,9.533,
     * 9.282,9.273,9.261,9.251,9.243,9.245,9.243,9.246/
C
      DATA ((EDEN(I,1,6,L,3),I=41,48),L=1,NZEN)/
     * 11.155,11.174,11.188,11.194,11.202,11.209,11.217,11.222,
     * 11.119,11.139,11.151,11.161,11.166,11.173,11.178,11.183,
     * 11.057,11.079,11.096,11.104,11.113,11.117,11.125,11.133,
     * 10.911,10.952,10.978,10.995,11.005,11.015,11.025,11.037,
     * 10.525,10.608,10.676,10.729,10.769,10.799,10.821,10.842,
     * 10.328,10.375,10.430,10.489,10.548,10.604,10.653,10.696,
     * 10.258,10.283,10.306,10.328,10.354,10.391,10.431,10.481,
     * 10.130,10.151,10.165,10.173,10.178,10.184,10.200,10.215,
     * 9.707,9.728,9.743,9.754,9.753,9.753,9.753,9.754,
     * 9.609,9.614,9.613,9.607,9.599,9.592,9.586,9.584,
     * 9.321,9.329,9.332,9.328,9.322,9.319,9.315,9.313/
C
      DATA ((EDEN(I,2,6,L,3),I=41,48),L=1,NZEN)/
     * 11.146,11.168,11.181,11.190,11.200,11.207,11.214,11.223,
     * 11.127,11.146,11.157,11.163,11.167,11.171,11.176,11.181,
     * 11.058,11.079,11.092,11.099,11.104,11.112,11.119,11.124,
     * 10.899,10.939,10.965,10.980,10.992,11.001,11.012,11.026,
     * 10.502,10.576,10.642,10.695,10.737,10.772,10.800,10.826,
     * 10.338,10.395,10.458,10.518,10.573,10.624,10.666,10.704,
     * 10.250,10.277,10.302,10.323,10.346,10.378,10.412,10.455,
     * 10.130,10.149,10.163,10.173,10.180,10.186,10.203,10.220,
     * 9.676,9.701,9.721,9.739,9.741,9.745,9.750,9.756,
     * 9.595,9.602,9.606,9.605,9.603,9.603,9.602,9.605,
     * 9.304,9.311,9.313,9.311,9.309,9.309,9.309,9.312/
C
      DATA ((EDEN(I,3,6,L,3),I=41,48),L=1,NZEN)/
     * 11.149,11.165,11.177,11.181,11.186,11.192,11.201,11.210,
     * 11.123,11.137,11.146,11.151,11.153,11.156,11.163,11.168,
     * 11.053,11.073,11.083,11.090,11.093,11.095,11.102,11.114,
     * 10.906,10.940,10.961,10.974,10.981,10.990,11.000,11.013,
     * 10.535,10.606,10.666,10.711,10.746,10.775,10.799,10.821,
     * 10.328,10.383,10.442,10.497,10.551,10.601,10.644,10.684,
     * 10.238,10.269,10.298,10.325,10.359,10.401,10.437,10.481,
     * 10.129,10.146,10.161,10.174,10.183,10.192,10.214,10.234,
     * 9.666,9.689,9.708,9.726,9.732,9.740,9.748,9.758,
     * 9.564,9.570,9.573,9.573,9.574,9.578,9.582,9.589,
     * 9.268,9.268,9.267,9.263,9.260,9.264,9.266,9.276/
C
      DATA ((EDEN(I,4,6,L,3),I=41,48),L=1,NZEN)/
     * 11.162,11.169,11.172,11.173,11.173,11.173,11.181,11.189,
     * 11.116,11.128,11.131,11.133,11.133,11.133,11.142,11.150,
     * 11.053,11.067,11.075,11.074,11.075,11.079,11.085,11.096,
     * 10.908,10.938,10.956,10.965,10.969,10.974,10.983,10.996,
     * 10.533,10.600,10.653,10.695,10.728,10.756,10.779,10.803,
     * 10.329,10.394,10.456,10.514,10.566,10.611,10.648,10.684,
     * 10.198,10.224,10.251,10.287,10.329,10.380,10.419,10.466,
     * 10.088,10.104,10.119,10.133,10.146,10.160,10.184,10.207,
     * 9.645,9.662,9.677,9.694,9.699,9.708,9.719,9.732,
     * 9.545,9.545,9.542,9.540,9.540,9.547,9.550,9.561,
     * 9.241,9.241,9.239,9.237,9.239,9.248,9.252,9.266/
C
      DATA ((EDEN(I,5,6,L,3),I=41,48),L=1,NZEN)/
     * 11.179,11.177,11.172,11.164,11.160,11.154,11.157,11.164,
     * 11.128,11.130,11.125,11.120,11.113,11.111,11.118,11.129,
     * 11.059,11.065,11.066,11.061,11.059,11.056,11.064,11.075,
     * 10.922,10.943,10.955,10.957,10.956,10.958,10.966,10.978,
     * 10.574,10.633,10.677,10.709,10.733,10.753,10.772,10.793,
     * 10.354,10.416,10.475,10.526,10.570,10.609,10.641,10.674,
     * 10.169,10.193,10.224,10.270,10.319,10.375,10.415,10.464,
     * 10.054,10.069,10.084,10.099,10.114,10.132,10.159,10.184,
     * 9.627,9.638,9.650,9.664,9.669,9.679,9.691,9.706,
     * 9.526,9.518,9.509,9.504,9.504,9.512,9.516,9.528,
     * 9.228,9.222,9.216,9.213,9.218,9.229,9.237,9.253/
C
      DATA ((EDEN(I,1,7,L,3),I=41,48),L=1,NZEN)/
     * 11.159,11.177,11.191,11.200,11.205,11.212,11.217,11.225,
     * 11.123,11.142,11.157,11.164,11.169,11.176,11.181,11.186,
     * 11.061,11.083,11.099,11.108,11.116,11.120,11.125,11.133,
     * 10.918,10.958,10.982,10.998,11.008,11.017,11.029,11.041,
     * 10.536,10.618,10.686,10.737,10.774,10.803,10.825,10.846,
     * 10.332,10.382,10.438,10.498,10.557,10.613,10.660,10.701,
     * 10.258,10.283,10.308,10.329,10.357,10.394,10.438,10.488,
     * 10.130,10.151,10.165,10.173,10.179,10.184,10.201,10.217,
     * 9.704,9.724,9.739,9.751,9.749,9.749,9.750,9.751,
     * 9.605,9.610,9.608,9.602,9.593,9.587,9.581,9.579,
     * 9.317,9.324,9.327,9.322,9.318,9.313,9.310,9.307/
C
      DATA ((EDEN(I,2,7,L,3),I=41,48),L=1,NZEN)/
     * 11.150,11.174,11.187,11.196,11.203,11.210,11.217,11.226,
     * 11.130,11.149,11.160,11.166,11.171,11.174,11.179,11.181,
     * 11.062,11.083,11.095,11.103,11.108,11.115,11.119,11.128,
     * 10.906,10.944,10.968,10.984,10.994,11.004,11.015,11.029,
     * 10.513,10.587,10.650,10.702,10.743,10.776,10.803,10.829,
     * 10.345,10.404,10.466,10.525,10.580,10.630,10.671,10.708,
     * 10.252,10.279,10.304,10.325,10.350,10.383,10.415,10.460,
     * 10.132,10.151,10.165,10.174,10.181,10.187,10.205,10.222,
     * 9.670,9.696,9.716,9.735,9.738,9.742,9.748,9.754,
     * 9.587,9.596,9.600,9.599,9.598,9.599,9.599,9.603,
     * 9.297,9.304,9.307,9.306,9.304,9.304,9.304,9.309/
C
      DATA ((EDEN(I,3,7,L,3),I=41,48),L=1,NZEN)/
     * 11.152,11.168,11.180,11.187,11.189,11.195,11.204,11.212,
     * 11.126,11.140,11.149,11.154,11.157,11.159,11.166,11.171,
     * 11.057,11.076,11.087,11.093,11.096,11.099,11.106,11.114,
     * 10.911,10.944,10.965,10.976,10.984,10.992,11.002,11.016,
     * 10.546,10.615,10.672,10.716,10.749,10.778,10.801,10.824,
     * 10.337,10.391,10.450,10.503,10.556,10.605,10.648,10.688,
     * 10.240,10.270,10.300,10.329,10.363,10.405,10.442,10.486,
     * 10.131,10.148,10.163,10.175,10.185,10.194,10.216,10.236,
     * 9.659,9.683,9.703,9.722,9.729,9.737,9.747,9.758,
     * 9.556,9.563,9.567,9.567,9.569,9.575,9.579,9.588,
     * 9.262,9.262,9.262,9.258,9.257,9.262,9.264,9.276/
C
      DATA ((EDEN(I,4,7,L,3),I=41,48),L=1,NZEN)/
     * 11.165,11.169,11.175,11.173,11.173,11.176,11.181,11.192,
     * 11.120,11.128,11.131,11.133,11.133,11.137,11.142,11.153,
     * 11.053,11.071,11.075,11.078,11.079,11.079,11.089,11.097,
     * 10.912,10.940,10.958,10.967,10.970,10.976,10.985,10.999,
     * 10.542,10.606,10.658,10.698,10.730,10.758,10.781,10.805,
     * 10.338,10.400,10.461,10.518,10.568,10.614,10.650,10.686,
     * 10.200,10.225,10.255,10.289,10.331,10.380,10.421,10.467,
     * 10.090,10.105,10.119,10.132,10.145,10.159,10.185,10.208,
     * 9.639,9.656,9.673,9.690,9.696,9.706,9.719,9.733,
     * 9.538,9.538,9.537,9.535,9.537,9.545,9.550,9.562,
     * 9.235,9.233,9.233,9.233,9.236,9.246,9.253,9.267/
C
      DATA ((EDEN(I,5,7,L,3),I=41,48),L=1,NZEN)/
     * 11.176,11.173,11.172,11.164,11.156,11.154,11.157,11.167,
     * 11.128,11.130,11.125,11.120,11.113,11.111,11.118,11.129,
     * 11.059,11.065,11.066,11.061,11.059,11.056,11.064,11.079,
     * 10.924,10.944,10.954,10.956,10.956,10.959,10.967,10.982,
     * 10.579,10.636,10.678,10.709,10.732,10.753,10.773,10.795,
     * 10.361,10.420,10.477,10.527,10.570,10.608,10.641,10.675,
     * 10.173,10.197,10.228,10.271,10.319,10.374,10.414,10.461,
     * 10.060,10.074,10.086,10.100,10.114,10.131,10.158,10.184,
     * 9.619,9.632,9.645,9.661,9.667,9.678,9.692,9.709,
     * 9.516,9.511,9.504,9.500,9.501,9.510,9.516,9.531,
     * 9.220,9.214,9.211,9.210,9.215,9.229,9.239,9.257/
C
      DATA ((EDEN(I,1,8,L,3),I=41,48),L=1,NZEN)/
     * 11.152,11.171,11.184,11.194,11.202,11.209,11.217,11.225,
     * 11.116,11.136,11.147,11.158,11.166,11.170,11.178,11.186,
     * 11.053,11.075,11.092,11.100,11.109,11.117,11.125,11.133,
     * 10.904,10.947,10.974,10.991,11.002,11.012,11.022,11.036,
     * 10.515,10.596,10.666,10.721,10.761,10.793,10.818,10.839,
     * 10.328,10.371,10.424,10.479,10.538,10.594,10.644,10.688,
     * 10.262,10.285,10.310,10.328,10.352,10.386,10.425,10.472,
     * 10.133,10.153,10.167,10.175,10.180,10.185,10.201,10.216,
     * 9.713,9.734,9.749,9.761,9.760,9.759,9.760,9.761,
     * 9.616,9.621,9.620,9.615,9.607,9.601,9.595,9.593,
     * 9.327,9.335,9.338,9.336,9.331,9.327,9.323,9.323/
C
      DATA ((EDEN(I,2,8,L,3),I=41,48),L=1,NZEN)/
     * 11.146,11.168,11.181,11.190,11.197,11.207,11.214,11.223,
     * 11.124,11.143,11.153,11.163,11.167,11.171,11.175,11.181,
     * 11.050,11.075,11.088,11.099,11.104,11.112,11.119,11.128,
     * 10.885,10.928,10.957,10.975,10.988,10.999,11.011,11.025,
     * 10.483,10.555,10.622,10.678,10.723,10.762,10.793,10.822,
     * 10.330,10.383,10.442,10.499,10.555,10.608,10.653,10.694,
     * 10.250,10.278,10.306,10.329,10.361,10.399,10.439,10.486,
     * 10.135,10.154,10.169,10.179,10.187,10.195,10.214,10.231,
     * 9.687,9.712,9.732,9.750,9.753,9.758,9.763,9.770,
     * 9.604,9.612,9.616,9.615,9.614,9.613,9.613,9.615,
     * 9.307,9.314,9.317,9.315,9.313,9.313,9.314,9.318/
C
      DATA ((EDEN(I,3,8,L,3),I=41,48),L=1,NZEN)/
     * 11.145,11.165,11.177,11.181,11.186,11.195,11.201,11.212,
     * 11.116,11.133,11.142,11.151,11.153,11.159,11.166,11.171,
     * 11.045,11.065,11.080,11.086,11.093,11.099,11.106,11.114,
     * 10.889,10.928,10.953,10.969,10.979,10.989,11.000,11.013,
     * 10.509,10.580,10.642,10.692,10.732,10.766,10.792,10.818,
     * 10.337,10.395,10.458,10.515,10.569,10.619,10.659,10.697,
     * 10.245,10.271,10.300,10.324,10.354,10.391,10.424,10.468,
     * 10.133,10.150,10.164,10.175,10.183,10.191,10.211,10.230,
     * 9.679,9.701,9.719,9.737,9.741,9.748,9.755,9.765,
     * 9.578,9.584,9.587,9.586,9.586,9.590,9.593,9.599,
     * 9.283,9.285,9.284,9.278,9.276,9.281,9.281,9.289/
C
      DATA ((EDEN(I,4,8,L,3),I=41,48),L=1,NZEN)/
     * 11.155,11.166,11.172,11.173,11.173,11.176,11.184,11.195,
     * 11.109,11.121,11.127,11.130,11.133,11.137,11.145,11.156,
     * 11.037,11.055,11.067,11.071,11.075,11.079,11.089,11.100,
     * 10.887,10.924,10.947,10.960,10.967,10.974,10.984,10.999,
     * 10.514,10.584,10.643,10.690,10.727,10.757,10.781,10.806,
     * 10.313,10.368,10.426,10.485,10.540,10.590,10.632,10.672,
     * 10.203,10.224,10.249,10.281,10.318,10.366,10.407,10.455,
     * 10.095,10.109,10.121,10.131,10.142,10.153,10.176,10.198,
     * 9.663,9.678,9.692,9.705,9.708,9.715,9.724,9.735,
     * 9.566,9.565,9.562,9.558,9.556,9.560,9.562,9.569,
     * 9.262,9.260,9.256,9.252,9.250,9.258,9.260,9.271/
C
      DATA ((EDEN(I,5,8,L,3),I=41,48),L=1,NZEN)/
     * 11.170,11.170,11.169,11.164,11.163,11.160,11.166,11.177,
     * 11.118,11.123,11.122,11.120,11.117,11.118,11.125,11.139,
     * 11.043,11.053,11.058,11.058,11.059,11.060,11.068,11.082,
     * 10.894,10.926,10.943,10.951,10.956,10.960,10.969,10.984,
     * 10.524,10.588,10.641,10.682,10.713,10.740,10.765,10.790,
     * 10.313,10.371,10.430,10.489,10.539,10.585,10.625,10.663,
     * 10.172,10.188,10.210,10.243,10.286,10.338,10.381,10.430,
     * 10.064,10.074,10.083,10.092,10.102,10.116,10.140,10.164,
     * 9.658,9.667,9.676,9.685,9.686,9.692,9.701,9.714,
     * 9.563,9.557,9.547,9.539,9.534,9.536,9.537,9.545,
     * 9.259,9.253,9.245,9.238,9.238,9.243,9.247,9.259/
C
      DATA ((EDEN(I,1,9,L,3),I=41,48),L=1,NZEN)/
     * 11.139,11.161,11.172,11.182,11.190,11.200,11.211,11.222,
     * 11.102,11.125,11.137,11.148,11.156,11.164,11.171,11.183,
     * 11.037,11.064,11.081,11.093,11.098,11.106,11.118,11.126,
     * 10.882,10.928,10.960,10.980,10.993,11.003,11.014,11.028,
     * 10.484,10.562,10.634,10.693,10.739,10.776,10.805,10.830,
     * 10.322,10.359,10.404,10.451,10.507,10.565,10.618,10.668,
     * 10.269,10.291,10.314,10.329,10.348,10.376,10.408,10.452,
     * 10.137,10.158,10.172,10.179,10.183,10.187,10.201,10.215,
     * 9.731,9.752,9.767,9.779,9.777,9.776,9.776,9.777,
     * 9.632,9.639,9.641,9.636,9.630,9.623,9.618,9.616,
     * 9.345,9.354,9.358,9.356,9.352,9.349,9.345,9.344/
C
      DATA ((EDEN(I,2,9,L,3),I=41,48),L=1,NZEN)/
     * 11.136,11.158,11.175,11.184,11.191,11.198,11.208,11.220,
     * 11.110,11.132,11.144,11.154,11.158,11.165,11.172,11.181,
     * 11.029,11.060,11.077,11.088,11.097,11.104,11.112,11.124,
     * 10.848,10.900,10.937,10.961,10.977,10.991,11.005,11.019,
     * 10.436,10.505,10.575,10.636,10.691,10.736,10.775,10.808,
     * 10.316,10.358,10.407,10.460,10.515,10.573,10.623,10.672,
     * 10.256,10.280,10.305,10.326,10.350,10.381,10.415,10.458,
     * 10.134,10.153,10.168,10.177,10.183,10.189,10.205,10.221,
     * 9.713,9.735,9.751,9.765,9.766,9.768,9.770,9.774,
     * 9.615,9.622,9.624,9.620,9.616,9.613,9.611,9.611,
     * 9.324,9.332,9.334,9.332,9.328,9.328,9.325,9.327/
C
      DATA ((EDEN(I,3,9,L,3),I=41,48),L=1,NZEN)/
     * 11.135,11.159,11.170,11.178,11.183,11.189,11.198,11.210,
     * 11.090,11.116,11.129,11.138,11.144,11.149,11.159,11.170,
     * 11.019,11.049,11.068,11.078,11.085,11.092,11.102,11.113,
     * 10.851,10.900,10.935,10.957,10.971,10.983,10.995,11.010,
     * 10.454,10.526,10.595,10.654,10.702,10.744,10.776,10.807,
     * 10.314,10.362,10.417,10.474,10.531,10.586,10.632,10.676,
     * 10.245,10.272,10.297,10.321,10.351,10.388,10.423,10.468,
     * 10.134,10.152,10.165,10.176,10.184,10.191,10.210,10.227,
     * 9.704,9.723,9.739,9.753,9.755,9.757,9.761,9.766,
     * 9.614,9.618,9.618,9.615,9.611,9.610,9.608,9.610,
     * 9.313,9.318,9.319,9.316,9.313,9.315,9.313,9.316/
C
      DATA ((EDEN(I,4,9,L,3),I=41,48),L=1,NZEN)/
     * 11.142,11.160,11.165,11.170,11.173,11.179,11.187,11.198,
     * 11.087,11.107,11.117,11.123,11.126,11.133,11.142,11.156,
     * 11.014,11.043,11.060,11.067,11.071,11.079,11.088,11.099,
     * 10.851,10.901,10.934,10.953,10.965,10.974,10.985,10.999,
     * 10.459,10.532,10.601,10.657,10.701,10.738,10.768,10.797,
     * 10.295,10.344,10.401,10.462,10.521,10.575,10.621,10.663,
     * 10.213,10.231,10.249,10.272,10.298,10.338,10.374,10.420,
     * 10.100,10.112,10.121,10.128,10.134,10.141,10.159,10.177,
     * 9.701,9.713,9.722,9.729,9.726,9.726,9.728,9.733,
     * 9.607,9.604,9.597,9.587,9.578,9.573,9.569,9.568,
     * 9.302,9.299,9.292,9.281,9.274,9.270,9.267,9.266/
C
      DATA ((EDEN(I,5,9,L,3),I=41,48),L=1,NZEN)/
     * 11.153,11.164,11.169,11.167,11.166,11.166,11.175,11.186,
     * 11.100,11.112,11.122,11.121,11.120,11.124,11.131,11.145,
     * 11.014,11.037,11.050,11.054,11.058,11.060,11.071,11.085,
     * 10.848,10.893,10.922,10.940,10.950,10.958,10.969,10.985,
     * 10.458,10.533,10.597,10.651,10.693,10.727,10.756,10.783,
     * 10.279,10.326,10.383,10.445,10.504,10.557,10.603,10.646,
     * 10.184,10.191,10.202,10.224,10.256,10.300,10.342,10.394,
     * 10.074,10.080,10.083,10.086,10.090,10.097,10.115,10.134,
     * 9.713,9.717,9.717,9.717,9.708,9.704,9.704,9.707,
     * 9.625,9.614,9.599,9.582,9.567,9.557,9.547,9.545,
     * 9.322,9.312,9.298,9.283,9.270,9.262,9.256,9.252/
C
      DATA ((EDEN(I,1,10,L,3),I=41,48),L=1,NZEN)/
     * 11.135,11.158,11.172,11.182,11.190,11.200,11.211,11.222,
     * 11.098,11.122,11.137,11.145,11.153,11.164,11.171,11.183,
     * 11.028,11.060,11.077,11.089,11.098,11.106,11.114,11.126,
     * 10.868,10.920,10.955,10.977,10.991,11.003,11.014,11.027,
     * 10.465,10.543,10.617,10.679,10.730,10.770,10.802,10.830,
     * 10.322,10.356,10.396,10.440,10.494,10.552,10.608,10.661,
     * 10.275,10.297,10.318,10.331,10.348,10.374,10.403,10.445,
     * 10.140,10.161,10.175,10.182,10.186,10.189,10.203,10.217,
     * 9.745,9.766,9.781,9.792,9.790,9.788,9.788,9.788,
     * 9.648,9.655,9.655,9.651,9.644,9.638,9.632,9.629,
     * 9.360,9.369,9.374,9.371,9.367,9.363,9.359,9.357/
C
      DATA ((EDEN(I,2,10,L,3),I=41,48),L=1,NZEN)/
     * 11.140,11.158,11.172,11.181,11.188,11.198,11.208,11.220,
     * 11.099,11.122,11.137,11.144,11.151,11.158,11.169,11.181,
     * 11.025,11.056,11.077,11.089,11.097,11.104,11.112,11.123,
     * 10.859,10.912,10.949,10.972,10.988,10.999,11.011,11.025,
     * 10.453,10.532,10.606,10.671,10.723,10.764,10.797,10.825,
     * 10.311,10.346,10.390,10.437,10.493,10.553,10.608,10.659,
     * 10.263,10.285,10.307,10.324,10.344,10.373,10.403,10.444,
     * 10.132,10.153,10.168,10.178,10.184,10.189,10.203,10.217,
     * 9.743,9.762,9.775,9.786,9.783,9.782,9.781,9.781,
     * 9.647,9.651,9.650,9.645,9.637,9.631,9.625,9.621,
     * 9.350,9.359,9.361,9.358,9.353,9.350,9.346,9.343/
C
      DATA ((EDEN(I,3,10,L,3),I=41,48),L=1,NZEN)/
     * 11.139,11.159,11.170,11.178,11.183,11.192,11.201,11.212,
     * 11.079,11.106,11.122,11.131,11.137,11.146,11.156,11.170,
     * 11.015,11.050,11.069,11.078,11.085,11.095,11.101,11.116,
     * 10.846,10.901,10.939,10.963,10.978,10.989,11.000,11.014,
     * 10.454,10.532,10.606,10.669,10.719,10.759,10.790,10.817,
     * 10.309,10.349,10.397,10.452,10.511,10.568,10.618,10.666,
     * 10.253,10.276,10.299,10.319,10.345,10.378,10.409,10.452,
     * 10.133,10.151,10.165,10.174,10.180,10.186,10.201,10.214,
     * 9.733,9.749,9.760,9.769,9.765,9.763,9.761,9.760,
     * 9.638,9.639,9.634,9.625,9.616,9.609,9.601,9.597,
     * 9.332,9.335,9.334,9.327,9.320,9.316,9.309,9.308/
C
      DATA ((EDEN(I,4,10,L,3),I=41,48),L=1,NZEN)/
     * 11.142,11.160,11.168,11.176,11.179,11.182,11.193,11.204,
     * 11.083,11.107,11.121,11.127,11.133,11.136,11.148,11.162,
     * 10.992,11.027,11.048,11.059,11.067,11.075,11.087,11.101,
     * 10.832,10.887,10.926,10.950,10.964,10.975,10.986,11.001,
     * 10.451,10.529,10.600,10.660,10.706,10.744,10.773,10.799,
     * 10.297,10.343,10.399,10.462,10.522,10.578,10.623,10.665,
     * 10.224,10.238,10.251,10.271,10.296,10.332,10.369,10.418,
     * 10.109,10.119,10.125,10.129,10.131,10.134,10.148,10.161,
     * 9.724,9.733,9.738,9.740,9.731,9.724,9.719,9.717,
     * 9.633,9.627,9.615,9.599,9.583,9.571,9.558,9.550,
     * 9.340,9.335,9.326,9.312,9.299,9.288,9.276,9.268/
C
      DATA ((EDEN(I,5,10,L,3),I=41,48),L=1,NZEN)/
     * 11.147,11.164,11.172,11.173,11.175,11.175,11.184,11.194,
     * 11.089,11.109,11.122,11.127,11.127,11.130,11.140,11.150,
     * 10.996,11.029,11.047,11.058,11.062,11.067,11.077,11.091,
     * 10.813,10.868,10.908,10.933,10.947,10.958,10.970,10.986,
     * 10.437,10.510,10.581,10.641,10.688,10.725,10.755,10.782,
     * 10.280,10.319,10.372,10.433,10.493,10.549,10.596,10.639,
     * 10.198,10.203,10.208,10.221,10.245,10.285,10.328,10.381,
     * 10.089,10.091,10.089,10.086,10.083,10.084,10.096,10.108,
     * 9.740,9.741,9.736,9.731,9.715,9.702,9.692,9.687,
     * 9.656,9.641,9.622,9.598,9.574,9.553,9.534,9.519,
     * 9.368,9.356,9.335,9.314,9.288,9.267,9.248,9.233/
C
      DATA ((EDEN(I,1,11,L,3),I=41,48),L=1,NZEN)/
     * 11.142,11.164,11.178,11.191,11.199,11.209,11.217,11.228,
     * 11.105,11.129,11.144,11.155,11.163,11.170,11.181,11.189,
     * 11.037,11.068,11.088,11.100,11.106,11.117,11.125,11.137,
     * 10.873,10.926,10.962,10.986,11.000,11.011,11.022,11.036,
     * 10.467,10.549,10.625,10.690,10.741,10.781,10.812,10.839,
     * 10.322,10.357,10.399,10.447,10.503,10.565,10.621,10.673,
     * 10.275,10.297,10.318,10.331,10.350,10.376,10.409,10.454,
     * 10.138,10.160,10.174,10.182,10.187,10.191,10.206,10.220,
     * 9.752,9.771,9.785,9.794,9.791,9.789,9.788,9.788,
     * 9.656,9.662,9.661,9.654,9.646,9.638,9.631,9.628,
     * 9.368,9.375,9.378,9.374,9.368,9.363,9.358,9.356/
C
      DATA ((EDEN(I,2,11,L,3),I=41,48),L=1,NZEN)/
     * 11.150,11.171,11.181,11.190,11.197,11.204,11.214,11.226,
     * 11.095,11.119,11.134,11.144,11.151,11.162,11.172,11.184,
     * 11.034,11.064,11.085,11.096,11.104,11.111,11.122,11.134,
     * 10.860,10.917,10.955,10.980,10.996,11.007,11.019,11.033,
     * 10.446,10.527,10.606,10.674,10.729,10.772,10.805,10.832,
     * 10.310,10.343,10.386,10.434,10.494,10.557,10.615,10.668,
     * 10.264,10.286,10.306,10.323,10.342,10.371,10.403,10.446,
     * 10.127,10.149,10.165,10.175,10.181,10.186,10.200,10.213,
     * 9.764,9.779,9.789,9.795,9.789,9.784,9.780,9.776,
     * 9.669,9.671,9.666,9.655,9.645,9.633,9.623,9.616,
     * 9.363,9.369,9.368,9.361,9.353,9.346,9.337,9.331/
C
      DATA ((EDEN(I,3,11,L,3),I=41,48),L=1,NZEN)/
     * 11.152,11.168,11.180,11.187,11.192,11.198,11.207,11.218,
     * 11.098,11.120,11.133,11.141,11.147,11.152,11.165,11.176,
     * 11.013,11.042,11.065,11.075,11.085,11.091,11.101,11.116,
     * 10.864,10.916,10.951,10.973,10.986,10.997,11.008,11.021,
     * 10.455,10.535,10.611,10.675,10.726,10.767,10.797,10.823,
     * 10.319,10.368,10.426,10.490,10.551,10.608,10.654,10.694,
     * 10.257,10.280,10.300,10.320,10.345,10.377,10.409,10.455,
     * 10.134,10.152,10.165,10.174,10.179,10.184,10.196,10.207,
     * 9.740,9.753,9.762,9.767,9.759,9.752,9.745,9.740,
     * 9.656,9.654,9.647,9.634,9.619,9.607,9.592,9.583,
     * 9.352,9.353,9.348,9.339,9.327,9.317,9.305,9.297/
C
      DATA ((EDEN(I,4,11,L,3),I=41,48),L=1,NZEN)/
     * 11.152,11.169,11.178,11.185,11.188,11.194,11.202,11.212,
     * 11.095,11.118,11.131,11.137,11.143,11.146,11.157,11.168,
     * 11.005,11.040,11.060,11.071,11.078,11.085,11.094,11.108,
     * 10.830,10.885,10.924,10.949,10.965,10.977,10.988,11.003,
     * 10.476,10.554,10.625,10.683,10.725,10.758,10.784,10.806,
     * 10.312,10.362,10.422,10.485,10.545,10.596,10.638,10.675,
     * 10.229,10.243,10.258,10.278,10.305,10.344,10.386,10.435,
     * 10.121,10.129,10.133,10.133,10.131,10.131,10.142,10.151,
     * 9.725,9.732,9.734,9.734,9.720,9.709,9.699,9.690,
     * 9.634,9.625,9.610,9.590,9.569,9.549,9.529,9.515,
     * 9.347,9.342,9.330,9.313,9.293,9.275,9.256,9.240/
C
      DATA ((EDEN(I,5,11,L,3),I=41,48),L=1,NZEN)/
     * 11.153,11.170,11.178,11.182,11.184,11.186,11.193,11.203,
     * 11.093,11.116,11.129,11.134,11.137,11.140,11.149,11.162,
     * 11.004,11.038,11.055,11.066,11.073,11.077,11.087,11.100,
     * 10.822,10.877,10.917,10.942,10.956,10.966,10.977,10.991,
     * 10.431,10.502,10.571,10.632,10.682,10.721,10.752,10.780,
     * 10.290,10.328,10.379,10.440,10.500,10.554,10.600,10.641,
     * 10.210,10.212,10.215,10.227,10.249,10.286,10.330,10.383,
     * 10.106,10.104,10.098,10.089,10.080,10.075,10.082,10.088,
     * 9.741,9.740,9.734,9.725,9.703,9.684,9.667,9.654,
     * 9.654,9.639,9.616,9.590,9.559,9.530,9.503,9.481,
     * 9.392,9.376,9.355,9.326,9.294,9.265,9.235,9.208/
C
      DATA ((EDEN(I,1,12,L,3),I=41,48),L=1,NZEN)/
     * 11.152,11.177,11.191,11.203,11.211,11.220,11.228,11.240,
     * 11.116,11.139,11.157,11.167,11.175,11.182,11.190,11.199,
     * 11.045,11.079,11.099,11.111,11.120,11.127,11.135,11.147,
     * 10.884,10.938,10.974,10.998,11.012,11.021,11.034,11.045,
     * 10.477,10.562,10.642,10.707,10.757,10.796,10.825,10.850,
     * 10.322,10.361,10.407,10.459,10.521,10.583,10.640,10.691,
     * 10.271,10.293,10.316,10.331,10.352,10.383,10.420,10.468,
     * 10.134,10.156,10.171,10.180,10.186,10.192,10.207,10.222,
     * 9.754,9.772,9.783,9.792,9.788,9.785,9.783,9.783,
     * 9.658,9.662,9.658,9.650,9.641,9.632,9.624,9.619,
     * 9.370,9.376,9.377,9.372,9.365,9.357,9.351,9.348/
C
      DATA ((EDEN(I,2,12,L,3),I=41,48),L=1,NZEN)/
     * 11.163,11.180,11.193,11.202,11.209,11.215,11.223,11.234,
     * 11.110,11.133,11.147,11.157,11.164,11.171,11.181,11.193,
     * 11.042,11.072,11.088,11.100,11.108,11.118,11.125,11.140,
     * 10.870,10.929,10.967,10.991,11.007,11.019,11.031,11.046,
     * 10.450,10.536,10.618,10.688,10.743,10.786,10.818,10.844,
     * 10.320,10.371,10.432,10.500,10.567,10.627,10.675,10.715,
     * 10.260,10.283,10.303,10.321,10.342,10.374,10.410,10.458,
     * 10.123,10.146,10.161,10.172,10.179,10.185,10.199,10.211,
     * 9.752,9.766,9.775,9.781,9.773,9.766,9.759,9.754,
     * 9.679,9.677,9.670,9.657,9.642,9.628,9.614,9.604,
     * 9.368,9.372,9.368,9.359,9.348,9.338,9.326,9.318/
C
      DATA ((EDEN(I,3,12,L,3),I=41,48),L=1,NZEN)/
     * 11.165,11.181,11.192,11.196,11.204,11.209,11.216,11.224,
     * 11.109,11.130,11.143,11.151,11.156,11.165,11.174,11.186,
     * 11.029,11.058,11.077,11.086,11.093,11.102,11.111,11.126,
     * 10.882,10.931,10.964,10.984,10.996,11.006,11.019,11.030,
     * 10.501,10.585,10.659,10.717,10.760,10.792,10.816,10.837,
     * 10.326,10.378,10.439,10.504,10.567,10.622,10.666,10.704,
     * 10.255,10.279,10.301,10.326,10.359,10.402,10.442,10.490,
     * 10.134,10.152,10.165,10.175,10.180,10.185,10.198,10.208,
     * 9.743,9.754,9.759,9.763,9.753,9.744,9.735,9.727,
     * 9.648,9.642,9.632,9.615,9.597,9.581,9.564,9.550,
     * 9.356,9.354,9.348,9.335,9.321,9.307,9.291,9.278/
C
      DATA ((EDEN(I,4,12,L,3),I=41,48),L=1,NZEN)/
     * 11.162,11.179,11.190,11.194,11.196,11.202,11.210,11.221,
     * 11.106,11.128,11.142,11.147,11.152,11.158,11.166,11.177,
     * 11.021,11.052,11.068,11.079,11.086,11.092,11.104,11.118,
     * 10.852,10.904,10.939,10.961,10.975,10.985,10.997,11.012,
     * 10.499,10.576,10.645,10.699,10.738,10.769,10.792,10.813,
     * 10.332,10.385,10.446,10.509,10.564,10.612,10.650,10.682,
     * 10.234,10.250,10.264,10.287,10.318,10.359,10.404,10.453,
     * 10.127,10.134,10.136,10.136,10.133,10.133,10.141,10.148,
     * 9.718,9.726,9.728,9.727,9.711,9.697,9.684,9.672,
     * 9.629,9.621,9.605,9.583,9.559,9.537,9.513,9.495,
     * 9.347,9.341,9.328,9.308,9.285,9.262,9.238,9.217/
C
      DATA ((EDEN(I,5,12,L,3),I=41,48),L=1,NZEN)/
     * 11.163,11.180,11.187,11.191,11.192,11.195,11.201,11.212,
     * 11.104,11.127,11.136,11.144,11.146,11.149,11.158,11.171,
     * 11.018,11.047,11.067,11.074,11.080,11.084,11.093,11.106,
     * 10.843,10.895,10.930,10.952,10.964,10.973,10.984,10.998,
     * 10.457,10.527,10.595,10.652,10.697,10.731,10.760,10.785,
     * 10.297,10.334,10.383,10.441,10.499,10.552,10.597,10.637,
     * 10.216,10.217,10.220,10.228,10.246,10.280,10.324,10.377,
     * 10.115,10.111,10.102,10.090,10.077,10.067,10.070,10.073,
     * 9.731,9.732,9.726,9.715,9.691,9.669,9.648,9.630,
     * 9.646,9.631,9.608,9.578,9.546,9.513,9.480,9.453,
     * 9.369,9.355,9.335,9.305,9.271,9.238,9.204,9.172/
C
      DATA ((EDEN(I,1,1,L,1),I=49,56),L=1,NZEN)/
     * 11.098,11.105,11.112,11.109,11.104,11.098,11.089,11.082,
     * 11.057,11.065,11.068,11.067,11.062,11.056,11.050,11.042,
     * 11.006,11.014,11.018,11.018,11.016,11.011,11.006,11.001,
     * 10.915,10.928,10.937,10.943,10.944,10.944,10.942,10.940,
     * 10.727,10.747,10.761,10.770,10.777,10.781,10.786,10.790,
     * 10.558,10.604,10.635,10.658,10.670,10.678,10.684,10.690,
     * 10.256,10.320,10.380,10.427,10.467,10.491,10.510,10.523,
     * 10.010,10.016,10.016,10.013,10.006,10.000,9.999,10.003,
     * 9.702,9.690,9.674,9.653,9.628,9.600,9.571,9.542,
     * 9.558,9.531,9.501,9.464,9.425,9.385,9.345,9.307,
     * 9.293,9.265,9.232,9.193,9.149,9.108,9.066,9.025/
C
      DATA ((EDEN(I,2,1,L,1),I=49,56),L=1,NZEN)/
     * 11.091,11.099,11.105,11.105,11.102,11.095,11.089,11.081,
     * 11.055,11.066,11.074,11.075,11.073,11.068,11.063,11.057,
     * 11.002,11.015,11.022,11.025,11.023,11.019,11.015,11.009,
     * 10.914,10.928,10.936,10.943,10.945,10.945,10.944,10.943,
     * 10.720,10.740,10.754,10.764,10.771,10.776,10.780,10.785,
     * 10.583,10.615,10.637,10.651,10.658,10.663,10.667,10.672,
     * 10.242,10.305,10.364,10.413,10.449,10.474,10.492,10.504,
     * 9.990,9.993,9.991,9.987,9.982,9.980,9.983,9.992,
     * 9.661,9.645,9.624,9.600,9.572,9.543,9.513,9.484,
     * 9.533,9.501,9.466,9.426,9.383,9.340,9.297,9.259,
     * 9.252,9.217,9.181,9.138,9.093,9.047,9.004,8.962/
C
      DATA ((EDEN(I,3,1,L,1),I=49,56),L=1,NZEN)/
     * 11.084,11.097,11.104,11.104,11.102,11.098,11.091,11.086,
     * 11.048,11.062,11.069,11.072,11.072,11.069,11.064,11.060,
     * 10.995,11.010,11.021,11.027,11.029,11.026,11.024,11.020,
     * 10.900,10.917,10.929,10.939,10.941,10.944,10.944,10.943,
     * 10.710,10.728,10.741,10.750,10.757,10.761,10.766,10.769,
     * 10.567,10.598,10.619,10.633,10.641,10.646,10.651,10.656,
     * 10.285,10.339,10.386,10.422,10.447,10.463,10.474,10.483,
     * 9.981,9.981,9.975,9.967,9.958,9.953,9.953,9.959,
     * 9.623,9.604,9.580,9.552,9.522,9.490,9.458,9.429,
     * 9.458,9.422,9.385,9.343,9.300,9.257,9.216,9.178,
     * 9.191,9.154,9.115,9.070,9.023,8.976,8.933,8.891/
C
      DATA ((EDEN(I,4,1,L,1),I=49,56),L=1,NZEN)/
     * 11.077,11.091,11.099,11.106,11.104,11.101,11.097,11.091,
     * 11.040,11.056,11.066,11.071,11.072,11.070,11.066,11.062,
     * 10.984,11.001,11.014,11.023,11.027,11.027,11.024,11.022,
     * 10.886,10.905,10.921,10.932,10.939,10.944,10.944,10.945,
     * 10.690,10.709,10.724,10.735,10.743,10.749,10.753,10.759,
     * 10.551,10.580,10.601,10.615,10.622,10.628,10.633,10.637,
     * 10.275,10.326,10.369,10.402,10.427,10.441,10.452,10.459,
     * 9.963,9.961,9.954,9.945,9.936,9.931,9.930,9.936,
     * 9.579,9.557,9.532,9.503,9.471,9.438,9.406,9.378,
     * 9.413,9.376,9.338,9.294,9.249,9.204,9.161,9.122,
     * 9.137,9.097,9.053,9.003,8.955,8.905,8.857,8.814/
C
      DATA ((EDEN(I,5,1,L,1),I=49,56),L=1,NZEN)/
     * 11.067,11.085,11.096,11.100,11.103,11.102,11.099,11.094,
     * 11.028,11.046,11.059,11.067,11.070,11.070,11.068,11.064,
     * 10.970,10.990,11.005,11.017,11.022,11.024,11.023,11.022,
     * 10.870,10.891,10.908,10.922,10.931,10.936,10.940,10.942,
     * 10.660,10.685,10.705,10.718,10.729,10.736,10.742,10.748,
     * 10.507,10.546,10.573,10.591,10.603,10.611,10.616,10.622,
     * 10.219,10.273,10.323,10.362,10.392,10.411,10.426,10.435,
     * 9.928,9.924,9.915,9.906,9.896,9.889,9.888,9.892,
     * 9.549,9.524,9.496,9.465,9.431,9.396,9.363,9.335,
     * 9.380,9.340,9.297,9.250,9.201,9.152,9.106,9.064,
     * 9.097,9.050,9.000,8.947,8.893,8.841,8.789,8.741/
C
      DATA ((EDEN(I,1,2,L,1),I=49,56),L=1,NZEN)/
     * 11.094,11.105,11.112,11.112,11.108,11.102,11.095,11.088,
     * 11.055,11.065,11.069,11.069,11.066,11.060,11.055,11.048,
     * 11.002,11.014,11.018,11.021,11.020,11.015,11.010,11.006,
     * 10.911,10.926,10.936,10.941,10.945,10.945,10.944,10.944,
     * 10.720,10.741,10.757,10.767,10.774,10.780,10.785,10.790,
     * 10.540,10.589,10.626,10.650,10.665,10.675,10.682,10.688,
     * 10.233,10.295,10.358,10.411,10.451,10.482,10.502,10.517,
     * 10.005,10.011,10.012,10.009,10.003,9.997,9.994,9.997,
     * 9.708,9.696,9.680,9.660,9.635,9.607,9.578,9.550,
     * 9.568,9.542,9.512,9.476,9.437,9.396,9.355,9.318,
     * 9.302,9.275,9.243,9.205,9.162,9.117,9.076,9.033/
C
      DATA ((EDEN(I,2,2,L,1),I=49,56),L=1,NZEN)/
     * 11.091,11.103,11.110,11.110,11.108,11.102,11.096,11.089,
     * 11.054,11.067,11.075,11.079,11.078,11.073,11.068,11.064,
     * 11.000,11.012,11.019,11.023,11.022,11.018,11.014,11.010,
     * 10.908,10.923,10.935,10.941,10.945,10.946,10.945,10.945,
     * 10.713,10.735,10.750,10.761,10.768,10.774,10.778,10.783,
     * 10.572,10.607,10.631,10.646,10.655,10.661,10.666,10.671,
     * 10.223,10.286,10.346,10.398,10.438,10.466,10.487,10.501,
     * 9.989,9.992,9.990,9.987,9.982,9.979,9.981,9.989,
     * 9.688,9.672,9.652,9.629,9.602,9.572,9.542,9.512,
     * 9.545,9.514,9.480,9.440,9.399,9.356,9.314,9.275,
     * 9.267,9.234,9.198,9.157,9.111,9.067,9.025,8.981/
C
      DATA ((EDEN(I,3,2,L,1),I=49,56),L=1,NZEN)/
     * 11.084,11.097,11.104,11.109,11.108,11.103,11.098,11.093,
     * 11.047,11.061,11.071,11.077,11.076,11.074,11.070,11.066,
     * 10.993,11.009,11.021,11.028,11.032,11.032,11.028,11.027,
     * 10.898,10.914,10.927,10.935,10.940,10.943,10.943,10.943,
     * 10.705,10.724,10.738,10.747,10.755,10.760,10.764,10.770,
     * 10.554,10.590,10.614,10.630,10.639,10.646,10.651,10.655,
     * 10.267,10.323,10.373,10.410,10.439,10.459,10.471,10.482,
     * 9.985,9.986,9.981,9.974,9.965,9.959,9.958,9.963,
     * 9.641,9.623,9.601,9.575,9.546,9.515,9.484,9.455,
     * 9.498,9.465,9.429,9.389,9.345,9.302,9.261,9.223,
     * 9.211,9.177,9.136,9.093,9.049,9.003,8.960,8.916/
C
      DATA ((EDEN(I,4,2,L,1),I=49,56),L=1,NZEN)/
     * 11.077,11.091,11.103,11.106,11.107,11.106,11.101,11.098,
     * 11.038,11.054,11.066,11.073,11.075,11.074,11.072,11.069,
     * 10.981,10.999,11.014,11.023,11.029,11.030,11.028,11.028,
     * 10.882,10.902,10.918,10.929,10.939,10.943,10.947,10.948,
     * 10.684,10.706,10.722,10.734,10.742,10.749,10.754,10.760,
     * 10.539,10.573,10.596,10.613,10.622,10.628,10.634,10.639,
     * 10.244,10.298,10.346,10.385,10.415,10.434,10.449,10.460,
     * 9.959,9.959,9.953,9.946,9.939,9.934,9.934,9.939,
     * 9.597,9.578,9.555,9.528,9.498,9.467,9.436,9.409,
     * 9.433,9.399,9.363,9.322,9.278,9.235,9.193,9.155,
     * 9.159,9.120,9.080,9.033,8.986,8.938,8.892,8.851/
C
      DATA ((EDEN(I,5,2,L,1),I=49,56),L=1,NZEN)/
     * 11.067,11.081,11.096,11.100,11.105,11.105,11.102,11.099,
     * 11.024,11.043,11.057,11.067,11.071,11.072,11.070,11.069,
     * 10.966,10.987,11.003,11.016,11.021,11.025,11.025,11.026,
     * 10.866,10.887,10.904,10.919,10.930,10.937,10.941,10.944,
     * 10.654,10.683,10.703,10.718,10.728,10.736,10.744,10.750,
     * 10.501,10.542,10.571,10.592,10.605,10.612,10.619,10.625,
     * 10.206,10.264,10.313,10.356,10.389,10.411,10.427,10.440,
     * 9.933,9.931,9.924,9.916,9.907,9.900,9.898,9.902,
     * 9.568,9.546,9.521,9.492,9.460,9.428,9.396,9.368,
     * 9.407,9.370,9.330,9.287,9.240,9.193,9.149,9.108,
     * 9.125,9.081,9.035,8.983,8.931,8.879,8.830,8.787/
C
      DATA ((EDEN(I,1,3,L,1),I=49,56),L=1,NZEN)/
     * 11.090,11.101,11.107,11.111,11.109,11.105,11.099,11.093,
     * 11.049,11.061,11.067,11.069,11.067,11.064,11.059,11.052,
     * 10.996,11.009,11.016,11.019,11.019,11.015,11.012,11.009,
     * 10.904,10.918,10.930,10.939,10.942,10.944,10.944,10.944,
     * 10.710,10.734,10.750,10.761,10.769,10.775,10.780,10.786,
     * 10.519,10.573,10.613,10.641,10.658,10.670,10.678,10.684,
     * 10.212,10.271,10.335,10.391,10.436,10.471,10.495,10.512,
     * 10.002,10.007,10.008,10.006,10.001,9.994,9.990,9.991,
     * 9.712,9.701,9.685,9.665,9.642,9.614,9.586,9.557,
     * 9.577,9.552,9.523,9.487,9.448,9.407,9.367,9.328,
     * 9.312,9.286,9.252,9.215,9.173,9.129,9.086,9.044/
C
      DATA ((EDEN(I,2,3,L,1),I=49,56),L=1,NZEN)/
     * 11.087,11.099,11.110,11.112,11.110,11.106,11.100,11.096,
     * 11.047,11.059,11.068,11.070,11.069,11.065,11.061,11.057,
     * 10.994,11.007,11.014,11.020,11.020,11.019,11.014,11.013,
     * 10.900,10.916,10.927,10.937,10.941,10.943,10.944,10.944,
     * 10.704,10.728,10.745,10.756,10.763,10.769,10.775,10.781,
     * 10.517,10.569,10.608,10.634,10.651,10.662,10.669,10.675,
     * 10.209,10.270,10.331,10.384,10.427,10.459,10.483,10.500,
     * 9.991,9.995,9.994,9.992,9.988,9.985,9.986,9.992,
     * 9.695,9.682,9.664,9.643,9.618,9.590,9.561,9.533,
     * 9.557,9.529,9.499,9.461,9.422,9.380,9.339,9.302,
     * 9.286,9.256,9.223,9.184,9.142,9.097,9.052,9.010/
C
      DATA ((EDEN(I,3,3,L,1),I=49,56),L=1,NZEN)/
     * 11.080,11.092,11.104,11.109,11.109,11.107,11.102,11.098,
     * 11.041,11.057,11.068,11.076,11.078,11.076,11.073,11.071,
     * 10.987,11.003,11.015,11.023,11.025,11.026,11.025,11.023,
     * 10.890,10.906,10.920,10.931,10.936,10.940,10.943,10.944,
     * 10.692,10.715,10.733,10.745,10.753,10.759,10.765,10.772,
     * 10.545,10.584,10.610,10.628,10.639,10.647,10.652,10.658,
     * 10.216,10.272,10.329,10.379,10.418,10.447,10.469,10.484,
     * 9.979,9.983,9.981,9.977,9.973,9.969,9.969,9.974,
     * 9.661,9.647,9.629,9.608,9.583,9.555,9.526,9.499,
     * 9.516,9.487,9.456,9.420,9.381,9.340,9.300,9.263,
     * 9.237,9.208,9.174,9.133,9.090,9.049,9.005,8.963/
C
      DATA ((EDEN(I,4,3,L,1),I=49,56),L=1,NZEN)/
     * 11.069,11.086,11.095,11.102,11.105,11.105,11.102,11.098,
     * 11.030,11.047,11.060,11.068,11.073,11.074,11.072,11.071,
     * 10.974,10.993,11.008,11.018,11.024,11.027,11.028,11.029,
     * 10.878,10.895,10.913,10.925,10.933,10.939,10.943,10.946,
     * 10.679,10.702,10.720,10.732,10.741,10.748,10.754,10.761,
     * 10.525,10.563,10.593,10.612,10.625,10.632,10.639,10.645,
     * 10.233,10.288,10.340,10.382,10.416,10.438,10.456,10.470,
     * 9.970,9.973,9.970,9.965,9.959,9.954,9.952,9.956,
     * 9.622,9.607,9.588,9.566,9.540,9.513,9.485,9.458,
     * 9.473,9.445,9.413,9.376,9.338,9.298,9.259,9.221,
     * 9.188,9.155,9.121,9.080,9.038,8.995,8.950,8.911/
C
      DATA ((EDEN(I,5,3,L,1),I=49,56),L=1,NZEN)/
     * 11.054,11.073,11.087,11.096,11.098,11.100,11.099,11.097,
     * 11.015,11.035,11.049,11.059,11.065,11.068,11.068,11.068,
     * 10.958,10.979,10.995,11.008,11.016,11.021,11.023,11.024,
     * 10.861,10.881,10.899,10.914,10.925,10.933,10.938,10.943,
     * 10.655,10.683,10.704,10.719,10.730,10.739,10.745,10.752,
     * 10.501,10.543,10.574,10.596,10.610,10.620,10.626,10.633,
     * 10.204,10.258,10.311,10.354,10.391,10.419,10.437,10.451,
     * 9.947,9.949,9.946,9.941,9.935,9.930,9.928,9.932,
     * 9.597,9.581,9.561,9.538,9.512,9.483,9.455,9.428,
     * 9.444,9.413,9.382,9.344,9.303,9.261,9.220,9.182,
     * 9.148,9.109,9.070,9.028,8.983,8.937,8.894,8.854/
C
      DATA ((EDEN(I,1,4,L,1),I=49,56),L=1,NZEN)/
     * 11.081,11.097,11.103,11.105,11.103,11.099,11.093,11.088,
     * 11.043,11.054,11.060,11.062,11.060,11.056,11.052,11.048,
     * 10.989,11.001,11.008,11.012,11.012,11.010,11.006,11.003,
     * 10.898,10.912,10.923,10.931,10.935,10.937,10.938,10.938,
     * 10.706,10.729,10.745,10.756,10.763,10.769,10.774,10.780,
     * 10.518,10.571,10.610,10.638,10.654,10.665,10.672,10.678,
     * 10.212,10.271,10.333,10.389,10.432,10.466,10.490,10.508,
     * 9.999,10.005,10.006,10.004,9.999,9.992,9.988,9.989,
     * 9.706,9.696,9.680,9.661,9.637,9.610,9.582,9.554,
     * 9.570,9.546,9.518,9.483,9.444,9.403,9.363,9.325,
     * 9.306,9.279,9.248,9.211,9.169,9.124,9.081,9.038/
C
      DATA ((EDEN(I,2,4,L,1),I=49,56),L=1,NZEN)/
     * 11.083,11.095,11.101,11.106,11.106,11.101,11.097,11.090,
     * 11.041,11.053,11.061,11.064,11.063,11.059,11.055,11.051,
     * 10.987,10.999,11.008,11.012,11.013,11.011,11.009,11.006,
     * 10.893,10.907,10.920,10.927,10.932,10.935,10.937,10.937,
     * 10.684,10.716,10.738,10.753,10.763,10.771,10.778,10.784,
     * 10.523,10.573,10.608,10.634,10.651,10.661,10.668,10.674,
     * 10.221,10.279,10.340,10.392,10.433,10.463,10.486,10.504,
     * 9.999,10.005,10.006,10.004,9.999,9.993,9.989,9.990,
     * 9.694,9.685,9.670,9.652,9.629,9.603,9.576,9.549,
     * 9.555,9.532,9.504,9.471,9.434,9.395,9.357,9.318,
     * 9.294,9.268,9.237,9.201,9.160,9.118,9.076,9.036/
C
      DATA ((EDEN(I,3,4,L,1),I=49,56),L=1,NZEN)/
     * 11.072,11.088,11.095,11.102,11.102,11.100,11.096,11.092,
     * 11.034,11.050,11.061,11.069,11.071,11.071,11.068,11.064,
     * 10.978,10.993,11.002,11.010,11.012,11.014,11.012,11.010,
     * 10.885,10.900,10.912,10.922,10.928,10.933,10.935,10.937,
     * 10.693,10.715,10.732,10.743,10.751,10.757,10.764,10.770,
     * 10.526,10.571,10.603,10.627,10.642,10.652,10.659,10.666,
     * 10.230,10.288,10.343,10.391,10.427,10.458,10.479,10.496,
     * 9.993,10.000,10.001,9.999,9.995,9.990,9.988,9.991,
     * 9.668,9.659,9.646,9.629,9.608,9.584,9.559,9.534,
     * 9.522,9.500,9.474,9.443,9.409,9.372,9.337,9.302,
     * 9.248,9.223,9.196,9.162,9.125,9.085,9.046,9.009/
C
      DATA ((EDEN(I,4,4,L,1),I=49,56),L=1,NZEN)/
     * 11.056,11.074,11.085,11.092,11.094,11.094,11.092,11.089,
     * 11.020,11.036,11.050,11.058,11.063,11.063,11.064,11.062,
     * 10.966,10.982,10.996,11.007,11.013,11.016,11.017,11.018,
     * 10.871,10.887,10.903,10.914,10.922,10.928,10.933,10.937,
     * 10.677,10.702,10.719,10.733,10.742,10.749,10.756,10.763,
     * 10.521,10.562,10.594,10.616,10.632,10.641,10.649,10.656,
     * 10.233,10.287,10.341,10.385,10.422,10.450,10.470,10.485,
     * 9.986,9.993,9.994,9.993,9.990,9.987,9.986,9.990,
     * 9.644,9.637,9.625,9.609,9.589,9.566,9.543,9.519,
     * 9.500,9.479,9.456,9.427,9.394,9.360,9.324,9.290,
     * 9.206,9.181,9.156,9.125,9.088,9.053,9.016,8.979/
C
      DATA ((EDEN(I,5,4,L,1),I=49,56),L=1,NZEN)/
     * 11.042,11.056,11.070,11.078,11.083,11.085,11.085,11.085,
     * 11.001,11.019,11.034,11.045,11.052,11.055,11.056,11.057,
     * 10.948,10.966,10.982,10.996,11.004,11.010,11.014,11.015,
     * 10.854,10.873,10.891,10.906,10.917,10.925,10.931,10.937,
     * 10.659,10.686,10.706,10.721,10.733,10.741,10.749,10.757,
     * 10.508,10.550,10.582,10.604,10.621,10.631,10.640,10.647,
     * 10.222,10.277,10.330,10.374,10.411,10.438,10.460,10.475,
     * 9.971,9.979,9.981,9.981,9.979,9.977,9.978,9.982,
     * 9.623,9.616,9.604,9.589,9.570,9.549,9.526,9.502,
     * 9.469,9.450,9.427,9.399,9.367,9.334,9.300,9.267,
     * 9.172,9.149,9.123,9.092,9.059,9.023,8.989,8.953/
C
      DATA ((EDEN(I,1,5,L,1),I=49,56),L=1,NZEN)/
     * 11.077,11.088,11.093,11.093,11.090,11.085,11.079,11.073,
     * 11.038,11.048,11.051,11.051,11.048,11.044,11.037,11.032,
     * 10.986,10.996,11.001,11.003,11.000,10.997,10.993,10.990,
     * 10.896,10.910,10.918,10.925,10.928,10.928,10.928,10.928,
     * 10.710,10.729,10.743,10.753,10.759,10.765,10.770,10.775,
     * 10.540,10.584,10.618,10.640,10.654,10.663,10.669,10.675,
     * 10.238,10.302,10.360,10.410,10.446,10.474,10.495,10.508,
     * 10.000,10.007,10.007,10.004,9.998,9.992,9.989,9.992,
     * 9.690,9.679,9.664,9.644,9.620,9.594,9.565,9.538,
     * 9.546,9.521,9.493,9.459,9.420,9.381,9.342,9.304,
     * 9.281,9.256,9.226,9.187,9.147,9.103,9.061,9.023/
C
      DATA ((EDEN(I,2,5,L,1),I=49,56),L=1,NZEN)/
     * 11.079,11.090,11.095,11.096,11.095,11.089,11.083,11.078,
     * 11.033,11.040,11.044,11.044,11.040,11.037,11.032,11.027,
     * 10.981,10.992,10.998,11.002,11.000,10.999,10.997,10.994,
     * 10.889,10.905,10.917,10.926,10.931,10.934,10.936,10.938,
     * 10.693,10.720,10.739,10.753,10.762,10.770,10.776,10.783,
     * 10.548,10.588,10.619,10.641,10.654,10.663,10.669,10.675,
     * 10.257,10.316,10.372,10.418,10.452,10.478,10.496,10.512,
     * 10.007,10.016,10.017,10.016,10.011,10.006,10.005,10.008,
     * 9.683,9.676,9.664,9.648,9.627,9.603,9.578,9.554,
     * 9.535,9.516,9.492,9.461,9.427,9.392,9.355,9.321,
     * 9.267,9.246,9.217,9.186,9.149,9.111,9.072,9.033/
C
      DATA ((EDEN(I,3,5,L,1),I=49,56),L=1,NZEN)/
     * 11.067,11.080,11.089,11.092,11.091,11.088,11.084,11.079,
     * 11.028,11.040,11.048,11.052,11.052,11.050,11.046,11.042,
     * 10.970,10.983,10.992,10.998,11.000,10.998,10.999,10.998,
     * 10.879,10.894,10.908,10.917,10.924,10.929,10.932,10.935,
     * 10.689,10.714,10.731,10.745,10.754,10.762,10.769,10.776,
     * 10.551,10.588,10.616,10.636,10.649,10.657,10.664,10.671,
     * 10.226,10.284,10.340,10.392,10.433,10.467,10.491,10.509,
     * 10.000,10.010,10.014,10.017,10.017,10.017,10.019,10.025,
     * 9.666,9.662,9.654,9.642,9.625,9.605,9.584,9.563,
     * 9.535,9.520,9.500,9.473,9.443,9.410,9.376,9.342,
     * 9.233,9.214,9.192,9.163,9.131,9.100,9.066,9.031/
C
      DATA ((EDEN(I,4,5,L,1),I=49,56),L=1,NZEN)/
     * 11.048,11.062,11.072,11.079,11.081,11.079,11.077,11.074,
     * 11.012,11.027,11.040,11.048,11.050,11.052,11.051,11.049,
     * 10.958,10.974,10.986,10.995,10.999,11.002,11.003,11.003,
     * 10.864,10.882,10.897,10.908,10.917,10.923,10.928,10.932,
     * 10.679,10.703,10.722,10.736,10.747,10.754,10.762,10.769,
     * 10.531,10.572,10.604,10.626,10.642,10.653,10.662,10.669,
     * 10.252,10.308,10.359,10.405,10.441,10.470,10.490,10.506,
     * 10.002,10.013,10.019,10.022,10.024,10.025,10.029,10.035,
     * 9.653,9.652,9.647,9.637,9.623,9.605,9.585,9.565,
     * 9.505,9.493,9.477,9.455,9.429,9.399,9.368,9.339,
     * 9.195,9.180,9.163,9.138,9.111,9.083,9.054,9.021/
C
      DATA ((EDEN(I,5,5,L,1),I=49,56),L=1,NZEN)/
     * 11.026,11.042,11.054,11.062,11.066,11.068,11.068,11.068,
     * 10.990,11.007,11.022,11.031,11.038,11.041,11.042,11.043,
     * 10.938,10.957,10.973,10.985,10.993,11.000,11.004,11.006,
     * 10.849,10.868,10.885,10.899,10.910,10.918,10.925,10.930,
     * 10.667,10.691,10.711,10.726,10.739,10.748,10.756,10.765,
     * 10.531,10.567,10.597,10.620,10.636,10.648,10.657,10.663,
     * 10.257,10.311,10.363,10.404,10.441,10.468,10.489,10.504,
     * 9.996,10.010,10.018,10.023,10.026,10.029,10.034,10.041,
     * 9.632,9.634,9.632,9.625,9.613,9.598,9.580,9.563,
     * 9.476,9.468,9.456,9.438,9.416,9.390,9.361,9.332,
     * 9.186,9.175,9.163,9.143,9.120,9.095,9.067,9.037/
C
      DATA ((EDEN(I,1,6,L,1),I=49,56),L=1,NZEN)/
     * 11.077,11.081,11.084,11.083,11.078,11.072,11.064,11.057,
     * 11.035,11.042,11.044,11.041,11.037,11.029,11.023,11.016,
     * 10.984,10.991,10.995,10.993,10.990,10.985,10.980,10.975,
     * 10.898,10.910,10.917,10.921,10.921,10.921,10.919,10.917,
     * 10.714,10.730,10.743,10.752,10.759,10.764,10.768,10.772,
     * 10.561,10.599,10.625,10.643,10.654,10.662,10.667,10.673,
     * 10.275,10.335,10.388,10.431,10.462,10.484,10.498,10.508,
     * 10.002,10.009,10.008,10.004,9.998,9.993,9.992,9.998,
     * 9.672,9.662,9.646,9.626,9.602,9.575,9.547,9.520,
     * 9.518,9.494,9.466,9.433,9.394,9.356,9.317,9.281,
     * 9.256,9.230,9.200,9.164,9.122,9.081,9.040,9.003/
C
      DATA ((EDEN(I,2,6,L,1),I=49,56),L=1,NZEN)/
     * 11.074,11.084,11.087,11.088,11.083,11.076,11.070,11.063,
     * 11.028,11.033,11.035,11.032,11.027,11.022,11.016,11.011,
     * 10.979,10.987,10.991,10.993,10.991,10.987,10.983,10.980,
     * 10.890,10.904,10.915,10.923,10.927,10.929,10.929,10.930,
     * 10.701,10.724,10.741,10.753,10.763,10.770,10.776,10.783,
     * 10.569,10.603,10.628,10.645,10.656,10.664,10.670,10.676,
     * 10.229,10.289,10.351,10.403,10.444,10.477,10.498,10.517,
     * 10.002,10.011,10.015,10.017,10.017,10.017,10.020,10.027,
     * 9.672,9.667,9.658,9.643,9.625,9.604,9.582,9.560,
     * 9.548,9.530,9.508,9.478,9.444,9.409,9.372,9.337,
     * 9.254,9.231,9.208,9.176,9.142,9.106,9.069,9.033/
C
      DATA ((EDEN(I,3,6,L,1),I=49,56),L=1,NZEN)/
     * 11.063,11.075,11.081,11.083,11.081,11.075,11.071,11.066,
     * 11.020,11.029,11.033,11.035,11.033,11.027,11.024,11.020,
     * 10.969,10.979,10.986,10.990,10.991,10.989,10.988,10.986,
     * 10.879,10.894,10.906,10.915,10.920,10.924,10.927,10.929,
     * 10.696,10.717,10.735,10.747,10.757,10.765,10.773,10.779,
     * 10.544,10.585,10.618,10.640,10.656,10.667,10.675,10.682,
     * 10.264,10.322,10.376,10.421,10.458,10.486,10.506,10.521,
     * 10.012,10.024,10.031,10.035,10.037,10.039,10.044,10.052,
     * 9.672,9.672,9.667,9.657,9.642,9.623,9.604,9.585,
     * 9.522,9.510,9.494,9.472,9.445,9.414,9.384,9.352,
     * 9.191,9.177,9.158,9.133,9.106,9.078,9.049,9.020/
C
      DATA ((EDEN(I,4,6,L,1),I=49,56),L=1,NZEN)/
     * 11.043,11.055,11.065,11.069,11.070,11.068,11.065,11.062,
     * 11.007,11.021,11.031,11.036,11.038,11.038,11.035,11.033,
     * 10.954,10.968,10.979,10.988,10.991,10.993,10.993,10.994,
     * 10.864,10.881,10.896,10.907,10.915,10.921,10.926,10.930,
     * 10.681,10.706,10.726,10.741,10.753,10.764,10.772,10.780,
     * 10.551,10.587,10.616,10.637,10.653,10.664,10.672,10.679,
     * 10.271,10.328,10.380,10.425,10.461,10.490,10.511,10.524,
     * 10.014,10.029,10.038,10.045,10.049,10.053,10.059,10.067,
     * 9.653,9.658,9.657,9.652,9.642,9.628,9.612,9.596,
     * 9.498,9.493,9.483,9.467,9.444,9.420,9.394,9.365,
     * 9.194,9.189,9.177,9.159,9.138,9.113,9.089,9.061/
C
      DATA ((EDEN(I,5,6,L,1),I=49,56),L=1,NZEN)/
     * 11.015,11.031,11.043,11.052,11.055,11.056,11.057,11.057,
     * 10.981,10.999,11.013,11.023,11.029,11.032,11.032,11.034,
     * 10.934,10.953,10.967,10.979,10.987,10.993,10.995,10.997,
     * 10.846,10.865,10.884,10.898,10.909,10.918,10.924,10.929,
     * 10.671,10.696,10.716,10.733,10.747,10.757,10.767,10.776,
     * 10.546,10.582,10.611,10.633,10.649,10.661,10.671,10.680,
     * 10.293,10.346,10.394,10.437,10.470,10.496,10.515,10.529,
     * 10.017,10.035,10.047,10.056,10.063,10.069,10.077,10.086,
     * 9.634,9.644,9.647,9.646,9.639,9.628,9.616,9.604,
     * 9.468,9.466,9.463,9.450,9.433,9.413,9.389,9.362,
     * 9.192,9.193,9.190,9.178,9.164,9.144,9.122,9.095/
C
      DATA ((EDEN(I,1,7,L,1),I=49,56),L=1,NZEN)/
     * 11.073,11.081,11.083,11.081,11.075,11.068,11.061,11.054,
     * 11.034,11.040,11.042,11.039,11.034,11.027,11.020,11.014,
     * 10.984,10.990,10.992,10.993,10.989,10.983,10.978,10.973,
     * 10.899,10.910,10.917,10.920,10.921,10.920,10.917,10.916,
     * 10.715,10.731,10.743,10.752,10.759,10.764,10.768,10.772,
     * 10.565,10.601,10.626,10.643,10.654,10.661,10.667,10.673,
     * 10.285,10.341,10.394,10.433,10.462,10.482,10.498,10.506,
     * 10.003,10.010,10.009,10.004,9.998,9.993,9.993,9.998,
     * 9.668,9.658,9.642,9.622,9.597,9.570,9.543,9.517,
     * 9.512,9.488,9.460,9.426,9.389,9.352,9.314,9.277,
     * 9.249,9.224,9.192,9.157,9.117,9.076,9.038,8.997/
C
      DATA ((EDEN(I,2,7,L,1),I=49,56),L=1,NZEN)/
     * 11.074,11.084,11.086,11.085,11.080,11.073,11.066,11.058,
     * 11.028,11.033,11.033,11.031,11.026,11.020,11.013,11.009,
     * 10.979,10.987,10.991,10.992,10.990,10.985,10.982,10.979,
     * 10.890,10.904,10.916,10.923,10.927,10.928,10.929,10.930,
     * 10.703,10.725,10.742,10.755,10.764,10.772,10.778,10.785,
     * 10.571,10.604,10.630,10.646,10.657,10.665,10.672,10.679,
     * 10.235,10.297,10.356,10.408,10.447,10.478,10.502,10.517,
     * 10.003,10.013,10.018,10.020,10.019,10.020,10.023,10.031,
     * 9.670,9.666,9.656,9.643,9.625,9.604,9.583,9.561,
     * 9.545,9.527,9.505,9.476,9.444,9.409,9.372,9.338,
     * 9.249,9.228,9.204,9.175,9.141,9.105,9.072,9.035/
C
      DATA ((EDEN(I,3,7,L,1),I=49,56),L=1,NZEN)/
     * 11.063,11.075,11.080,11.082,11.079,11.074,11.068,11.062,
     * 11.021,11.029,11.033,11.033,11.031,11.028,11.022,11.019,
     * 10.970,10.980,10.988,10.990,10.991,10.989,10.988,10.986,
     * 10.881,10.895,10.908,10.917,10.922,10.926,10.928,10.931,
     * 10.697,10.719,10.736,10.750,10.760,10.769,10.776,10.783,
     * 10.548,10.588,10.619,10.643,10.658,10.670,10.679,10.686,
     * 10.270,10.328,10.380,10.427,10.462,10.489,10.511,10.525,
     * 10.015,10.028,10.035,10.040,10.042,10.045,10.050,10.059,
     * 9.673,9.674,9.669,9.660,9.646,9.628,9.610,9.592,
     * 9.521,9.511,9.496,9.475,9.448,9.419,9.389,9.357,
     * 9.191,9.175,9.160,9.136,9.109,9.084,9.056,9.027/
C
      DATA ((EDEN(I,4,7,L,1),I=49,56),L=1,NZEN)/
     * 11.043,11.056,11.065,11.071,11.071,11.069,11.065,11.061,
     * 11.008,11.021,11.033,11.038,11.040,11.040,11.036,11.034,
     * 10.955,10.970,10.983,10.989,10.992,10.994,10.995,10.995,
     * 10.866,10.884,10.899,10.911,10.919,10.925,10.929,10.933,
     * 10.683,10.708,10.729,10.745,10.757,10.768,10.777,10.786,
     * 10.554,10.591,10.619,10.642,10.656,10.668,10.678,10.686,
     * 10.276,10.333,10.385,10.430,10.467,10.496,10.516,10.531,
     * 10.017,10.033,10.044,10.051,10.057,10.061,10.068,10.076,
     * 9.657,9.663,9.664,9.659,9.650,9.637,9.622,9.607,
     * 9.502,9.498,9.490,9.474,9.454,9.431,9.403,9.375,
     * 9.198,9.195,9.186,9.170,9.148,9.127,9.101,9.073/
C
      DATA ((EDEN(I,5,7,L,1),I=49,56),L=1,NZEN)/
     * 11.018,11.033,11.045,11.054,11.058,11.059,11.059,11.058,
     * 10.985,11.001,11.016,11.026,11.032,11.035,11.035,11.036,
     * 10.937,10.955,10.971,10.983,10.990,10.995,10.999,11.000,
     * 10.848,10.869,10.887,10.903,10.914,10.923,10.929,10.934,
     * 10.673,10.699,10.720,10.738,10.752,10.764,10.774,10.783,
     * 10.547,10.585,10.616,10.638,10.656,10.668,10.679,10.688,
     * 10.294,10.349,10.399,10.441,10.475,10.504,10.523,10.538,
     * 10.022,10.041,10.054,10.065,10.073,10.080,10.088,10.098,
     * 9.641,9.652,9.658,9.658,9.652,9.643,9.633,9.622,
     * 9.475,9.477,9.475,9.464,9.449,9.429,9.405,9.380,
     * 9.201,9.204,9.205,9.196,9.181,9.163,9.142,9.116/
C
      DATA ((EDEN(I,1,8,L,1),I=49,56),L=1,NZEN)/
     * 11.077,11.086,11.090,11.089,11.085,11.079,11.072,11.067,
     * 11.037,11.045,11.048,11.047,11.043,11.038,11.033,11.026,
     * 10.986,10.995,10.999,11.000,10.998,10.993,10.988,10.985,
     * 10.897,10.910,10.918,10.924,10.926,10.925,10.925,10.924,
     * 10.711,10.729,10.742,10.752,10.759,10.765,10.770,10.775,
     * 10.550,10.590,10.620,10.640,10.653,10.662,10.667,10.674,
     * 10.258,10.318,10.372,10.417,10.451,10.476,10.493,10.506,
     * 10.002,10.008,10.008,10.005,9.998,9.992,9.990,9.994,
     * 9.680,9.670,9.655,9.635,9.612,9.585,9.558,9.532,
     * 9.532,9.507,9.479,9.446,9.409,9.371,9.333,9.296,
     * 9.269,9.243,9.212,9.174,9.136,9.096,9.054,9.014/
C
      DATA ((EDEN(I,2,8,L,1),I=49,56),L=1,NZEN)/
     * 11.079,11.086,11.092,11.092,11.089,11.082,11.076,11.070,
     * 11.033,11.039,11.040,11.041,11.037,11.031,11.027,11.021,
     * 10.982,10.992,10.998,11.000,10.999,10.995,10.994,10.991,
     * 10.891,10.907,10.919,10.928,10.933,10.936,10.938,10.938,
     * 10.696,10.722,10.741,10.755,10.765,10.773,10.780,10.787,
     * 10.556,10.594,10.623,10.642,10.657,10.665,10.672,10.679,
     * 10.274,10.331,10.384,10.426,10.461,10.484,10.501,10.515,
     * 10.018,10.027,10.029,10.028,10.024,10.019,10.018,10.022,
     * 9.691,9.686,9.675,9.659,9.639,9.616,9.592,9.569,
     * 9.561,9.544,9.521,9.491,9.457,9.421,9.383,9.349,
     * 9.259,9.240,9.214,9.183,9.149,9.114,9.078,9.042/
C
      DATA ((EDEN(I,3,8,L,1),I=49,56),L=1,NZEN)/
     * 11.067,11.080,11.087,11.090,11.088,11.084,11.078,11.074,
     * 11.026,11.036,11.041,11.043,11.042,11.038,11.034,11.031,
     * 10.974,10.986,10.995,11.000,11.001,11.000,10.999,10.998,
     * 10.882,10.899,10.912,10.922,10.928,10.933,10.936,10.939,
     * 10.693,10.717,10.736,10.750,10.761,10.771,10.778,10.785,
     * 10.558,10.594,10.622,10.643,10.656,10.666,10.674,10.680,
     * 10.240,10.298,10.355,10.405,10.446,10.478,10.501,10.519,
     * 10.008,10.020,10.026,10.030,10.031,10.033,10.036,10.043,
     * 9.681,9.680,9.674,9.663,9.648,9.630,9.610,9.590,
     * 9.536,9.524,9.507,9.483,9.455,9.424,9.393,9.360,
     * 9.213,9.197,9.179,9.155,9.126,9.099,9.068,9.036/
C
      DATA ((EDEN(I,4,8,L,1),I=49,56),L=1,NZEN)/
     * 11.051,11.065,11.075,11.081,11.082,11.080,11.077,11.074,
     * 11.015,11.029,11.040,11.046,11.050,11.050,11.047,11.046,
     * 10.962,10.978,10.990,11.000,11.005,11.005,11.007,11.007,
     * 10.868,10.887,10.903,10.916,10.925,10.931,10.936,10.941,
     * 10.685,10.710,10.730,10.745,10.756,10.767,10.775,10.783,
     * 10.540,10.580,10.613,10.637,10.654,10.666,10.675,10.684,
     * 10.265,10.322,10.374,10.421,10.457,10.486,10.508,10.523,
     * 10.013,10.027,10.036,10.041,10.044,10.046,10.050,10.057,
     * 9.663,9.666,9.664,9.658,9.646,9.631,9.614,9.597,
     * 9.515,9.507,9.496,9.476,9.454,9.428,9.400,9.370,
     * 9.206,9.198,9.186,9.165,9.142,9.116,9.090,9.059/
C
      DATA ((EDEN(I,5,8,L,1),I=49,56),L=1,NZEN)/
     * 11.028,11.045,11.059,11.068,11.072,11.074,11.073,11.072,
     * 10.993,11.012,11.027,11.038,11.045,11.047,11.048,11.049,
     * 10.943,10.963,10.981,10.993,11.002,11.008,11.011,11.012,
     * 10.853,10.874,10.893,10.909,10.921,10.929,10.936,10.942,
     * 10.669,10.698,10.720,10.739,10.753,10.765,10.775,10.785,
     * 10.537,10.578,10.609,10.634,10.651,10.665,10.675,10.685,
     * 10.264,10.321,10.375,10.421,10.460,10.488,10.512,10.527,
     * 10.010,10.028,10.039,10.048,10.053,10.059,10.065,10.073,
     * 9.651,9.659,9.661,9.658,9.650,9.638,9.624,9.608,
     * 9.499,9.496,9.489,9.476,9.456,9.433,9.406,9.379,
     * 9.210,9.208,9.202,9.188,9.170,9.148,9.123,9.092/
C
      DATA ((EDEN(I,1,9,L,1),I=49,56),L=1,NZEN)/
     * 11.081,11.093,11.098,11.101,11.098,11.094,11.089,11.085,
     * 11.040,11.050,11.056,11.058,11.055,11.052,11.048,11.043,
     * 10.987,10.998,11.004,11.008,11.007,11.004,11.002,11.000,
     * 10.896,10.910,10.920,10.927,10.931,10.933,10.934,10.935,
     * 10.707,10.728,10.742,10.753,10.760,10.766,10.771,10.776,
     * 10.526,10.575,10.611,10.637,10.653,10.663,10.670,10.676,
     * 10.222,10.283,10.342,10.393,10.434,10.466,10.488,10.506,
     * 10.001,10.008,10.008,10.006,10.000,9.993,9.989,9.991,
     * 9.699,9.689,9.674,9.656,9.633,9.606,9.580,9.553,
     * 9.560,9.537,9.509,9.475,9.437,9.399,9.360,9.321,
     * 9.296,9.270,9.239,9.203,9.162,9.120,9.079,9.038/
C
      DATA ((EDEN(I,2,9,L,1),I=49,56),L=1,NZEN)/
     * 11.079,11.090,11.098,11.100,11.098,11.095,11.090,11.084,
     * 11.036,11.045,11.051,11.052,11.050,11.046,11.041,11.039,
     * 10.985,10.997,11.005,11.009,11.010,11.009,11.007,11.006,
     * 10.892,10.907,10.922,10.931,10.938,10.942,10.946,10.948,
     * 10.686,10.717,10.739,10.755,10.765,10.773,10.780,10.788,
     * 10.529,10.576,10.613,10.637,10.654,10.664,10.672,10.678,
     * 10.231,10.290,10.348,10.398,10.438,10.469,10.492,10.509,
     * 10.005,10.013,10.015,10.014,10.010,10.004,10.001,10.003,
     * 9.695,9.688,9.675,9.659,9.638,9.614,9.589,9.564,
     * 9.554,9.533,9.510,9.477,9.443,9.408,9.369,9.334,
     * 9.272,9.251,9.222,9.190,9.154,9.116,9.077,9.042/
C
      DATA ((EDEN(I,3,9,L,1),I=49,56),L=1,NZEN)/
     * 11.072,11.084,11.093,11.097,11.097,11.095,11.092,11.088,
     * 11.033,11.046,11.054,11.060,11.060,11.057,11.055,11.053,
     * 10.979,10.992,11.003,11.009,11.011,11.012,11.011,11.009,
     * 10.884,10.900,10.915,10.926,10.934,10.939,10.943,10.946,
     * 10.684,10.713,10.734,10.748,10.759,10.768,10.775,10.784,
     * 10.534,10.577,10.611,10.634,10.650,10.660,10.669,10.675,
     * 10.243,10.301,10.357,10.403,10.441,10.470,10.492,10.508,
     * 10.007,10.017,10.020,10.020,10.018,10.015,10.014,10.017,
     * 9.682,9.677,9.668,9.654,9.636,9.614,9.591,9.568,
     * 9.550,9.533,9.512,9.484,9.452,9.417,9.383,9.348,
     * 9.252,9.233,9.212,9.183,9.149,9.114,9.080,9.044/
C
      DATA ((EDEN(I,4,9,L,1),I=49,56),L=1,NZEN)/
     * 11.056,11.074,11.084,11.091,11.093,11.093,11.091,11.089,
     * 11.020,11.038,11.050,11.058,11.063,11.065,11.063,11.063,
     * 10.966,10.983,10.997,11.007,11.012,11.014,11.015,11.015,
     * 10.872,10.891,10.907,10.919,10.928,10.934,10.939,10.943,
     * 10.677,10.705,10.725,10.741,10.752,10.762,10.770,10.778,
     * 10.532,10.574,10.606,10.629,10.645,10.656,10.664,10.671,
     * 10.225,10.283,10.340,10.388,10.432,10.463,10.487,10.505,
     * 9.994,10.005,10.010,10.013,10.014,10.015,10.018,10.023,
     * 9.661,9.658,9.651,9.641,9.625,9.606,9.586,9.565,
     * 9.514,9.500,9.483,9.458,9.431,9.400,9.367,9.336,
     * 9.205,9.188,9.168,9.142,9.112,9.080,9.048,9.015/
C
      DATA ((EDEN(I,5,9,L,1),I=49,56),L=1,NZEN)/
     * 11.042,11.056,11.071,11.081,11.086,11.088,11.088,11.086,
     * 11.003,11.022,11.037,11.049,11.056,11.059,11.060,11.061,
     * 10.950,10.971,10.988,11.001,11.010,11.016,11.020,11.022,
     * 10.857,10.879,10.898,10.912,10.924,10.933,10.940,10.945,
     * 10.664,10.694,10.716,10.734,10.746,10.758,10.766,10.776,
     * 10.521,10.564,10.598,10.622,10.639,10.652,10.661,10.670,
     * 10.230,10.288,10.345,10.392,10.432,10.463,10.486,10.503,
     * 9.988,10.001,10.008,10.012,10.014,10.016,10.019,10.025,
     * 9.646,9.645,9.640,9.630,9.616,9.599,9.579,9.559,
     * 9.498,9.486,9.469,9.448,9.422,9.393,9.361,9.331,
     * 9.205,9.190,9.174,9.151,9.122,9.091,9.062,9.029/
C
      DATA ((EDEN(I,1,10,L,1),I=49,56),L=1,NZEN)/
     * 11.085,11.097,11.103,11.106,11.105,11.101,11.097,11.092,
     * 11.044,11.054,11.061,11.064,11.063,11.061,11.056,11.052,
     * 10.991,11.001,11.009,11.014,11.014,11.013,11.009,11.007,
     * 10.899,10.913,10.924,10.932,10.936,10.938,10.940,10.940,
     * 10.709,10.730,10.747,10.757,10.765,10.770,10.775,10.780,
     * 10.518,10.571,10.611,10.638,10.656,10.667,10.675,10.681,
     * 10.212,10.271,10.333,10.387,10.432,10.466,10.490,10.509,
     * 10.004,10.010,10.011,10.009,10.004,9.997,9.992,9.992,
     * 9.711,9.701,9.686,9.667,9.644,9.617,9.590,9.563,
     * 9.576,9.553,9.525,9.490,9.452,9.412,9.373,9.334,
     * 9.312,9.286,9.256,9.217,9.177,9.133,9.091,9.049/
C
      DATA ((EDEN(I,2,10,L,1),I=49,56),L=1,NZEN)/
     * 11.083,11.095,11.101,11.105,11.104,11.101,11.096,11.091,
     * 11.042,11.053,11.061,11.065,11.063,11.060,11.056,11.052,
     * 10.989,11.002,11.009,11.014,11.016,11.014,11.013,11.010,
     * 10.897,10.912,10.922,10.932,10.936,10.940,10.941,10.942,
     * 10.704,10.728,10.744,10.756,10.763,10.769,10.774,10.780,
     * 10.517,10.568,10.608,10.635,10.653,10.664,10.671,10.678,
     * 10.212,10.269,10.330,10.384,10.430,10.462,10.487,10.505,
     * 9.997,10.003,10.004,10.002,10.000,9.997,9.998,10.004,
     * 9.701,9.691,9.676,9.657,9.634,9.608,9.581,9.554,
     * 9.564,9.541,9.513,9.478,9.440,9.401,9.362,9.324,
     * 9.293,9.267,9.235,9.199,9.160,9.117,9.076,9.035/
C
      DATA ((EDEN(I,3,10,L,1),I=49,56),L=1,NZEN)/
     * 11.076,11.088,11.095,11.103,11.102,11.101,11.097,11.094,
     * 11.037,11.052,11.063,11.069,11.071,11.071,11.069,11.066,
     * 10.983,10.997,11.008,11.015,11.018,11.017,11.017,11.015,
     * 10.890,10.905,10.917,10.927,10.934,10.938,10.941,10.944,
     * 10.694,10.719,10.736,10.748,10.757,10.762,10.769,10.775,
     * 10.522,10.569,10.605,10.628,10.645,10.656,10.663,10.669,
     * 10.221,10.279,10.337,10.386,10.426,10.458,10.481,10.497,
     * 9.991,9.996,9.997,9.996,9.993,9.991,9.992,9.998,
     * 9.672,9.663,9.648,9.630,9.609,9.584,9.558,9.533,
     * 9.528,9.505,9.479,9.446,9.410,9.373,9.336,9.301,
     * 9.239,9.215,9.186,9.151,9.116,9.077,9.038,9.000/
C
      DATA ((EDEN(I,4,10,L,1),I=49,56),L=1,NZEN)/
     * 11.065,11.078,11.091,11.097,11.100,11.100,11.097,11.096,
     * 11.026,11.042,11.056,11.065,11.068,11.070,11.068,11.068,
     * 10.970,10.989,11.004,11.015,11.021,11.026,11.026,11.028,
     * 10.877,10.894,10.910,10.922,10.930,10.936,10.941,10.944,
     * 10.681,10.705,10.724,10.738,10.747,10.754,10.761,10.769,
     * 10.536,10.575,10.603,10.624,10.636,10.645,10.651,10.658,
     * 10.231,10.287,10.342,10.388,10.425,10.451,10.471,10.488,
     * 9.982,9.988,9.989,9.988,9.985,9.983,9.984,9.990,
     * 9.641,9.632,9.618,9.601,9.580,9.556,9.531,9.508,
     * 9.490,9.467,9.442,9.411,9.376,9.340,9.304,9.271,
     * 9.210,9.185,9.156,9.121,9.086,9.048,9.010,8.973/
C
      DATA ((EDEN(I,5,10,L,1),I=49,56),L=1,NZEN)/
     * 11.050,11.069,11.083,11.091,11.095,11.096,11.097,11.094,
     * 11.012,11.031,11.046,11.057,11.063,11.065,11.066,11.067,
     * 10.957,10.977,10.994,11.006,11.015,11.021,11.023,11.025,
     * 10.861,10.882,10.900,10.916,10.927,10.935,10.941,10.947,
     * 10.665,10.692,10.712,10.728,10.739,10.748,10.756,10.763,
     * 10.516,10.556,10.589,10.610,10.625,10.636,10.644,10.651,
     * 10.226,10.282,10.337,10.382,10.417,10.444,10.464,10.477,
     * 9.969,9.976,9.977,9.976,9.973,9.971,9.972,9.977,
     * 9.621,9.611,9.598,9.580,9.559,9.536,9.511,9.487,
     * 9.466,9.443,9.417,9.387,9.353,9.317,9.280,9.247,
     * 9.176,9.150,9.120,9.085,9.048,9.010,8.973,8.933/
C
      DATA ((EDEN(I,1,11,L,1),I=49,56),L=1,NZEN)/
     * 11.090,11.101,11.107,11.108,11.105,11.101,11.095,11.090,
     * 11.049,11.059,11.064,11.066,11.063,11.059,11.053,11.049,
     * 10.997,11.007,11.013,11.015,11.015,11.011,11.009,11.006,
     * 10.905,10.918,10.929,10.936,10.939,10.940,10.940,10.941,
     * 10.717,10.738,10.753,10.763,10.769,10.774,10.780,10.784,
     * 10.534,10.586,10.622,10.648,10.663,10.673,10.679,10.685,
     * 10.228,10.290,10.352,10.406,10.448,10.477,10.501,10.517,
     * 10.007,10.014,10.015,10.012,10.007,10.000,9.996,9.998,
     * 9.710,9.699,9.684,9.664,9.640,9.613,9.585,9.557,
     * 9.572,9.547,9.519,9.483,9.444,9.404,9.364,9.326,
     * 9.307,9.281,9.250,9.211,9.169,9.127,9.084,9.041/
C
      DATA ((EDEN(I,2,11,L,1),I=49,56),L=1,NZEN)/
     * 11.083,11.095,11.101,11.104,11.103,11.099,11.093,11.088,
     * 11.048,11.060,11.068,11.073,11.073,11.070,11.065,11.062,
     * 10.996,11.007,11.015,11.017,11.017,11.016,11.013,11.009,
     * 10.903,10.918,10.929,10.936,10.940,10.942,10.943,10.943,
     * 10.711,10.733,10.749,10.759,10.766,10.771,10.776,10.781,
     * 10.528,10.578,10.615,10.641,10.656,10.665,10.672,10.676,
     * 10.219,10.279,10.342,10.394,10.436,10.468,10.489,10.505,
     * 9.993,9.997,9.997,9.994,9.991,9.988,9.989,9.996,
     * 9.695,9.681,9.663,9.641,9.616,9.587,9.558,9.529,
     * 9.554,9.526,9.495,9.457,9.417,9.375,9.334,9.294,
     * 9.274,9.245,9.210,9.173,9.129,9.086,9.042,9.003/
C
      DATA ((EDEN(I,3,11,L,1),I=49,56),L=1,NZEN)/
     * 11.080,11.092,11.100,11.103,11.102,11.098,11.094,11.091,
     * 11.041,11.054,11.065,11.070,11.071,11.069,11.067,11.064,
     * 10.986,11.002,11.014,11.022,11.025,11.025,11.024,11.023,
     * 10.894,10.910,10.921,10.930,10.936,10.938,10.940,10.940,
     * 10.700,10.722,10.738,10.748,10.756,10.762,10.768,10.773,
     * 10.557,10.592,10.618,10.634,10.644,10.650,10.654,10.659,
     * 10.227,10.287,10.344,10.391,10.429,10.455,10.475,10.489,
     * 9.982,9.985,9.983,9.979,9.974,9.971,9.971,9.977,
     * 9.644,9.630,9.611,9.589,9.563,9.535,9.507,9.480,
     * 9.508,9.479,9.447,9.409,9.369,9.328,9.289,9.250,
     * 9.222,9.191,9.157,9.117,9.075,9.030,8.988,8.948/
C
      DATA ((EDEN(I,4,11,L,1),I=49,56),L=1,NZEN)/
     * 11.073,11.086,11.095,11.101,11.102,11.100,11.097,11.094,
     * 11.032,11.048,11.060,11.067,11.069,11.069,11.068,11.065,
     * 10.977,10.994,11.008,11.018,11.023,11.024,11.024,11.024,
     * 10.880,10.899,10.913,10.927,10.935,10.940,10.943,10.946,
     * 10.687,10.708,10.724,10.735,10.743,10.750,10.755,10.760,
     * 10.548,10.581,10.604,10.619,10.628,10.635,10.640,10.644,
     * 10.259,10.311,10.360,10.398,10.427,10.448,10.462,10.473,
     * 9.973,9.975,9.971,9.965,9.959,9.953,9.953,9.957,
     * 9.608,9.592,9.572,9.548,9.522,9.493,9.465,9.439,
     * 9.444,9.413,9.382,9.343,9.304,9.263,9.226,9.189,
     * 9.174,9.139,9.103,9.062,9.018,8.974,8.934,8.891/
C
      DATA ((EDEN(I,5,11,L,1),I=49,56),L=1,NZEN)/
     * 11.058,11.073,11.087,11.096,11.099,11.099,11.097,11.096,
     * 11.020,11.037,11.052,11.061,11.065,11.067,11.066,11.065,
     * 10.963,10.983,10.998,11.010,11.017,11.021,11.022,11.022,
     * 10.865,10.885,10.902,10.917,10.926,10.934,10.939,10.942,
     * 10.661,10.688,10.708,10.723,10.733,10.741,10.748,10.754,
     * 10.518,10.555,10.583,10.602,10.614,10.622,10.628,10.632,
     * 10.230,10.285,10.337,10.375,10.407,10.429,10.445,10.455,
     * 9.951,9.952,9.948,9.942,9.935,9.930,9.929,9.934,
     * 9.581,9.564,9.542,9.518,9.490,9.460,9.432,9.406,
     * 9.419,9.387,9.352,9.313,9.273,9.230,9.188,9.151,
     * 9.145,9.107,9.066,9.020,8.975,8.927,8.881,8.837/
C
      DATA ((EDEN(I,1,12,L,1),I=49,56),L=1,NZEN)/
     * 11.094,11.105,11.107,11.108,11.103,11.097,11.090,11.084,
     * 11.055,11.063,11.067,11.066,11.062,11.057,11.048,11.044,
     * 11.003,11.012,11.017,11.018,11.015,11.010,11.006,11.001,
     * 10.913,10.926,10.934,10.940,10.942,10.942,10.940,10.940,
     * 10.727,10.745,10.759,10.768,10.774,10.780,10.784,10.787,
     * 10.554,10.601,10.634,10.656,10.669,10.677,10.683,10.688,
     * 10.251,10.316,10.376,10.426,10.463,10.490,10.508,10.521,
     * 10.010,10.017,10.017,10.014,10.007,10.001,9.999,10.003,
     * 9.704,9.693,9.676,9.656,9.631,9.603,9.574,9.546,
     * 9.561,9.534,9.505,9.469,9.429,9.389,9.349,9.312,
     * 9.296,9.268,9.236,9.197,9.156,9.113,9.069,9.028/
C
      DATA ((EDEN(I,2,12,L,1),I=49,56),L=1,NZEN)/
     * 11.087,11.099,11.101,11.104,11.101,11.095,11.089,11.082,
     * 11.052,11.064,11.071,11.073,11.071,11.067,11.062,11.056,
     * 11.000,11.012,11.019,11.022,11.021,11.019,11.014,11.009,
     * 10.910,10.924,10.934,10.940,10.944,10.944,10.943,10.942,
     * 10.719,10.740,10.753,10.763,10.769,10.775,10.779,10.783,
     * 10.583,10.615,10.637,10.651,10.658,10.663,10.668,10.671,
     * 10.238,10.302,10.361,10.412,10.448,10.474,10.492,10.506,
     * 9.992,9.995,9.993,9.990,9.985,9.983,9.986,9.994,
     * 9.664,9.649,9.629,9.606,9.579,9.550,9.520,9.491,
     * 9.538,9.507,9.472,9.432,9.392,9.348,9.307,9.268,
     * 9.255,9.221,9.186,9.146,9.100,9.056,9.014,8.973/
C
      DATA ((EDEN(I,3,12,L,1),I=49,56),L=1,NZEN)/
     * 11.084,11.092,11.100,11.102,11.100,11.095,11.090,11.086,
     * 11.045,11.059,11.067,11.070,11.070,11.066,11.062,11.057,
     * 10.992,11.008,11.017,11.024,11.027,11.025,11.023,11.019,
     * 10.900,10.915,10.928,10.935,10.938,10.940,10.939,10.939,
     * 10.710,10.728,10.741,10.750,10.756,10.760,10.765,10.769,
     * 10.566,10.598,10.620,10.634,10.642,10.648,10.652,10.657,
     * 10.286,10.342,10.388,10.424,10.448,10.466,10.477,10.486,
     * 9.985,9.986,9.980,9.972,9.964,9.958,9.958,9.964,
     * 9.627,9.609,9.586,9.560,9.530,9.498,9.468,9.439,
     * 9.462,9.429,9.392,9.352,9.310,9.268,9.226,9.190,
     * 9.197,9.160,9.124,9.080,9.034,8.988,8.943,8.903/
C
      DATA ((EDEN(I,4,12,L,1),I=49,56),L=1,NZEN)/
     * 11.077,11.091,11.099,11.102,11.102,11.099,11.094,11.091,
     * 11.038,11.053,11.063,11.069,11.071,11.068,11.064,11.062,
     * 10.983,10.999,11.012,11.021,11.024,11.024,11.023,11.022,
     * 10.885,10.902,10.918,10.930,10.937,10.941,10.943,10.944,
     * 10.691,10.710,10.724,10.735,10.742,10.748,10.753,10.758,
     * 10.554,10.582,10.602,10.616,10.624,10.629,10.633,10.638,
     * 10.281,10.331,10.374,10.407,10.429,10.446,10.455,10.465,
     * 9.968,9.967,9.960,9.952,9.944,9.939,9.939,9.945,
     * 9.583,9.563,9.538,9.511,9.480,9.448,9.418,9.391,
     * 9.417,9.383,9.345,9.303,9.260,9.215,9.175,9.136,
     * 9.143,9.102,9.063,9.017,8.967,8.919,8.875,8.832/
C
      DATA ((EDEN(I,5,12,L,1),I=49,56),L=1,NZEN)/
     * 11.067,11.081,11.092,11.100,11.101,11.101,11.097,11.093,
     * 11.026,11.042,11.056,11.064,11.068,11.067,11.066,11.063,
     * 10.968,10.988,11.003,11.014,11.019,11.022,11.021,11.021,
     * 10.869,10.888,10.905,10.920,10.929,10.935,10.938,10.941,
     * 10.662,10.687,10.706,10.719,10.729,10.737,10.743,10.750,
     * 10.512,10.550,10.577,10.595,10.606,10.613,10.618,10.624,
     * 10.226,10.280,10.329,10.369,10.398,10.418,10.431,10.442,
     * 9.934,9.932,9.924,9.916,9.908,9.902,9.901,9.906,
     * 9.553,9.530,9.504,9.475,9.443,9.410,9.379,9.352,
     * 9.384,9.345,9.305,9.260,9.214,9.167,9.122,9.081,
     * 9.103,9.059,9.013,8.962,8.910,8.857,8.808,8.762/
C
      DATA ((EDEN(I,1,1,L,2),I=49,56),L=1,NZEN)/
     * 11.162,11.171,11.175,11.173,11.170,11.161,11.156,11.150,
     * 11.125,11.129,11.131,11.132,11.127,11.121,11.115,11.111,
     * 11.071,11.077,11.083,11.083,11.080,11.075,11.073,11.070,
     * 10.983,10.994,11.003,11.008,11.009,11.009,11.009,11.009,
     * 10.801,10.817,10.829,10.838,10.845,10.851,10.857,10.863,
     * 10.652,10.688,10.714,10.731,10.743,10.751,10.757,10.764,
     * 10.396,10.446,10.491,10.528,10.557,10.577,10.592,10.605,
     * 10.135,10.141,10.140,10.136,10.130,10.126,10.125,10.130,
     * 9.749,9.740,9.726,9.709,9.688,9.665,9.641,9.620,
     * 9.586,9.564,9.540,9.511,9.479,9.445,9.412,9.381,
     * 9.319,9.299,9.272,9.241,9.206,9.171,9.137,9.104/
C
      DATA ((EDEN(I,2,1,L,2),I=49,56),L=1,NZEN)/
     * 11.156,11.165,11.169,11.168,11.164,11.160,11.151,11.149,
     * 11.122,11.130,11.137,11.137,11.137,11.131,11.128,11.123,
     * 11.067,11.078,11.085,11.088,11.087,11.084,11.081,11.077,
     * 10.981,10.994,11.002,11.008,11.010,11.010,11.011,11.011,
     * 10.794,10.811,10.823,10.832,10.839,10.845,10.850,10.855,
     * 10.670,10.695,10.712,10.723,10.730,10.735,10.741,10.746,
     * 10.384,10.431,10.475,10.513,10.542,10.560,10.575,10.585,
     * 10.118,10.121,10.117,10.113,10.109,10.106,10.109,10.116,
     * 9.710,9.696,9.678,9.657,9.634,9.608,9.584,9.561,
     * 9.562,9.535,9.506,9.472,9.435,9.399,9.364,9.332,
     * 9.280,9.252,9.222,9.188,9.149,9.112,9.075,9.043/
C
      DATA ((EDEN(I,3,1,L,2),I=49,56),L=1,NZEN)/
     * 11.149,11.159,11.167,11.167,11.164,11.160,11.157,11.152,
     * 11.114,11.127,11.134,11.136,11.136,11.131,11.128,11.125,
     * 11.059,11.075,11.085,11.091,11.092,11.091,11.090,11.088,
     * 10.968,10.982,10.994,11.002,11.006,11.009,11.009,11.010,
     * 10.783,10.798,10.809,10.818,10.824,10.830,10.834,10.839,
     * 10.654,10.678,10.695,10.705,10.713,10.718,10.723,10.729,
     * 10.415,10.456,10.490,10.516,10.534,10.546,10.555,10.563,
     * 10.109,10.108,10.100,10.092,10.083,10.077,10.076,10.081,
     * 9.674,9.657,9.635,9.610,9.583,9.555,9.529,9.506,
     * 9.492,9.462,9.430,9.394,9.356,9.319,9.283,9.252,
     * 9.225,9.193,9.159,9.121,9.081,9.041,9.002,8.966/
C
      DATA ((EDEN(I,4,1,L,2),I=49,56),L=1,NZEN)/
     * 11.149,11.163,11.169,11.174,11.173,11.171,11.168,11.163,
     * 11.110,11.123,11.136,11.139,11.140,11.141,11.137,11.135,
     * 11.056,11.072,11.083,11.092,11.096,11.097,11.096,11.094,
     * 10.957,10.976,10.990,11.001,11.009,11.012,11.015,11.018,
     * 10.764,10.782,10.795,10.805,10.814,10.819,10.825,10.832,
     * 10.634,10.659,10.676,10.688,10.695,10.700,10.706,10.713,
     * 10.385,10.427,10.462,10.490,10.508,10.522,10.531,10.539,
     * 10.050,10.049,10.043,10.036,10.030,10.027,10.029,10.036,
     * 9.619,9.600,9.576,9.550,9.521,9.493,9.466,9.444,
     * 9.439,9.406,9.372,9.333,9.293,9.251,9.213,9.180,
     * 9.161,9.125,9.086,9.042,8.998,8.955,8.912,8.876/
C
      DATA ((EDEN(I,5,1,L,2),I=49,56),L=1,NZEN)/
     * 11.144,11.160,11.168,11.175,11.179,11.179,11.174,11.170,
     * 11.106,11.121,11.135,11.143,11.146,11.144,11.142,11.140,
     * 11.049,11.065,11.082,11.093,11.098,11.098,11.099,11.098,
     * 10.947,10.966,10.983,10.997,11.006,11.012,11.016,11.018,
     * 10.741,10.763,10.780,10.794,10.804,10.812,10.819,10.826,
     * 10.596,10.629,10.652,10.669,10.679,10.688,10.693,10.700,
     * 10.315,10.369,10.415,10.451,10.476,10.494,10.506,10.516,
     * 9.976,9.975,9.970,9.965,9.960,9.960,9.964,9.974,
     * 9.579,9.556,9.531,9.502,9.471,9.439,9.410,9.385,
     * 9.396,9.359,9.320,9.277,9.231,9.187,9.144,9.105,
     * 9.111,9.068,9.023,8.973,8.924,8.874,8.827,8.785/
C
      DATA ((EDEN(I,1,2,L,2),I=49,56),L=1,NZEN)/
     * 11.162,11.171,11.175,11.177,11.173,11.169,11.161,11.157,
     * 11.121,11.129,11.131,11.132,11.131,11.126,11.121,11.116,
     * 11.066,11.077,11.083,11.084,11.084,11.080,11.077,11.074,
     * 10.978,10.991,11.001,11.006,11.010,11.012,11.012,11.013,
     * 10.794,10.812,10.825,10.835,10.843,10.849,10.854,10.861,
     * 10.636,10.676,10.705,10.726,10.739,10.748,10.755,10.763,
     * 10.378,10.426,10.473,10.514,10.547,10.569,10.587,10.600,
     * 10.130,10.136,10.136,10.133,10.128,10.122,10.120,10.124,
     * 9.754,9.745,9.732,9.715,9.694,9.671,9.648,9.627,
     * 9.596,9.575,9.551,9.522,9.488,9.455,9.421,9.391,
     * 9.329,9.308,9.282,9.251,9.216,9.182,9.146,9.112/
C
      DATA ((EDEN(I,2,2,L,2),I=49,56),L=1,NZEN)/
     * 11.156,11.165,11.173,11.175,11.172,11.168,11.160,11.157,
     * 11.118,11.130,11.141,11.142,11.141,11.140,11.135,11.130,
     * 11.067,11.078,11.083,11.086,11.085,11.083,11.079,11.078,
     * 10.975,10.989,10.999,11.007,11.010,11.011,11.012,11.012,
     * 10.788,10.806,10.819,10.829,10.836,10.842,10.848,10.854,
     * 10.660,10.688,10.706,10.719,10.727,10.733,10.738,10.744,
     * 10.368,10.417,10.461,10.501,10.531,10.555,10.571,10.584,
     * 10.117,10.120,10.117,10.113,10.109,10.107,10.108,10.114,
     * 9.734,9.721,9.704,9.684,9.661,9.635,9.610,9.587,
     * 9.573,9.547,9.519,9.487,9.451,9.414,9.380,9.349,
     * 9.294,9.269,9.238,9.205,9.168,9.131,9.094,9.059/
C
      DATA ((EDEN(I,3,2,L,2),I=49,56),L=1,NZEN)/
     * 11.149,11.162,11.167,11.170,11.172,11.168,11.165,11.159,
     * 11.111,11.127,11.134,11.140,11.140,11.140,11.135,11.133,
     * 11.060,11.075,11.085,11.092,11.095,11.096,11.095,11.094,
     * 10.965,10.980,10.991,11.000,11.004,11.008,11.009,11.010,
     * 10.778,10.795,10.807,10.816,10.823,10.828,10.833,10.839,
     * 10.645,10.672,10.690,10.703,10.711,10.717,10.722,10.728,
     * 10.401,10.443,10.479,10.507,10.529,10.543,10.554,10.563,
     * 10.112,10.113,10.106,10.099,10.090,10.084,10.082,10.086,
     * 9.690,9.675,9.655,9.632,9.606,9.580,9.554,9.531,
     * 9.529,9.501,9.471,9.436,9.398,9.361,9.325,9.294,
     * 9.244,9.213,9.181,9.143,9.105,9.065,9.028,8.993/
C
      DATA ((EDEN(I,4,2,L,2),I=49,56),L=1,NZEN)/
     * 11.149,11.159,11.169,11.174,11.177,11.175,11.173,11.167,
     * 11.107,11.124,11.136,11.143,11.145,11.145,11.142,11.140,
     * 11.052,11.068,11.084,11.093,11.096,11.099,11.099,11.099,
     * 10.953,10.972,10.987,11.000,11.008,11.013,11.017,11.021,
     * 10.759,10.778,10.793,10.804,10.813,10.819,10.826,10.833,
     * 10.625,10.653,10.672,10.686,10.695,10.701,10.707,10.714,
     * 10.358,10.404,10.445,10.478,10.500,10.517,10.530,10.541,
     * 10.050,10.051,10.045,10.040,10.034,10.032,10.034,10.041,
     * 9.637,9.620,9.599,9.575,9.549,9.521,9.495,9.474,
     * 9.460,9.429,9.397,9.360,9.321,9.282,9.246,9.212,
     * 9.184,9.148,9.112,9.073,9.030,8.988,8.947,8.910/
C
      DATA ((EDEN(I,5,2,L,2),I=49,56),L=1,NZEN)/
     * 11.141,11.156,11.168,11.175,11.179,11.179,11.178,11.174,
     * 11.099,11.117,11.131,11.140,11.147,11.145,11.147,11.145,
     * 11.045,11.062,11.079,11.090,11.095,11.099,11.100,11.101,
     * 10.942,10.963,10.981,10.995,11.004,11.012,11.016,11.021,
     * 10.736,10.760,10.779,10.793,10.803,10.812,10.819,10.827,
     * 10.590,10.625,10.651,10.670,10.681,10.689,10.696,10.702,
     * 10.306,10.361,10.408,10.446,10.476,10.495,10.509,10.520,
     * 9.987,9.988,9.984,9.979,9.975,9.973,9.977,9.986,
     * 9.600,9.580,9.557,9.531,9.502,9.472,9.444,9.419,
     * 9.425,9.392,9.355,9.314,9.271,9.229,9.188,9.151,
     * 9.142,9.100,9.057,9.011,8.964,8.914,8.870,8.830/
C
      DATA ((EDEN(I,1,3,L,2),I=49,56),L=1,NZEN)/
     * 11.155,11.167,11.171,11.173,11.173,11.169,11.165,11.161,
     * 11.117,11.125,11.131,11.132,11.131,11.128,11.124,11.120,
     * 11.062,11.073,11.080,11.083,11.083,11.080,11.079,11.077,
     * 10.971,10.984,10.995,11.003,11.006,11.009,11.010,11.013,
     * 10.785,10.804,10.818,10.829,10.837,10.844,10.849,10.856,
     * 10.620,10.663,10.695,10.717,10.732,10.743,10.750,10.757,
     * 10.359,10.405,10.455,10.497,10.533,10.560,10.579,10.595,
     * 10.127,10.133,10.133,10.130,10.125,10.120,10.117,10.118,
     * 9.758,9.750,9.737,9.721,9.701,9.678,9.656,9.634,
     * 9.604,9.584,9.561,9.531,9.498,9.465,9.432,9.401,
     * 9.337,9.318,9.292,9.260,9.228,9.190,9.155,9.121/
C
      DATA ((EDEN(I,2,3,L,2),I=49,56),L=1,NZEN)/
     * 11.153,11.165,11.169,11.175,11.172,11.172,11.168,11.162,
     * 11.110,11.122,11.129,11.134,11.133,11.131,11.127,11.124,
     * 11.059,11.071,11.079,11.083,11.084,11.083,11.081,11.080,
     * 10.967,10.981,10.992,11.000,11.005,11.008,11.010,11.011,
     * 10.780,10.799,10.814,10.823,10.831,10.837,10.844,10.851,
     * 10.618,10.658,10.689,10.710,10.724,10.734,10.741,10.749,
     * 10.357,10.403,10.450,10.490,10.524,10.549,10.568,10.583,
     * 10.120,10.124,10.122,10.119,10.116,10.113,10.113,10.118,
     * 9.742,9.732,9.717,9.699,9.677,9.654,9.630,9.608,
     * 9.585,9.562,9.537,9.507,9.474,9.438,9.404,9.374,
     * 9.313,9.290,9.262,9.230,9.195,9.158,9.123,9.090/
C
      DATA ((EDEN(I,3,3,L,2),I=49,56),L=1,NZEN)/
     * 11.146,11.159,11.167,11.170,11.172,11.172,11.169,11.167,
     * 11.107,11.119,11.131,11.141,11.141,11.140,11.138,11.137,
     * 11.052,11.067,11.078,11.086,11.089,11.090,11.090,11.090,
     * 10.958,10.973,10.984,10.994,11.001,11.004,11.008,11.010,
     * 10.768,10.788,10.801,10.813,10.820,10.827,10.833,10.841,
     * 10.637,10.666,10.688,10.701,10.711,10.717,10.724,10.729,
     * 10.360,10.403,10.446,10.484,10.514,10.536,10.553,10.567,
     * 10.109,10.112,10.109,10.105,10.100,10.096,10.096,10.100,
     * 9.711,9.699,9.683,9.664,9.642,9.619,9.595,9.574,
     * 9.548,9.524,9.498,9.466,9.433,9.398,9.364,9.335,
     * 9.269,9.244,9.215,9.182,9.146,9.109,9.072,9.040/
C
      DATA ((EDEN(I,4,3,L,2),I=49,56),L=1,NZEN)/
     * 11.138,11.156,11.166,11.170,11.173,11.175,11.173,11.171,
     * 11.100,11.117,11.129,11.136,11.142,11.142,11.143,11.142,
     * 11.045,11.061,11.077,11.086,11.092,11.097,11.098,11.099,
     * 10.947,10.966,10.980,10.993,11.002,11.008,11.013,11.017,
     * 10.754,10.775,10.790,10.802,10.811,10.818,10.824,10.832,
     * 10.614,10.645,10.670,10.686,10.697,10.705,10.712,10.718,
     * 10.352,10.398,10.441,10.476,10.503,10.522,10.537,10.549,
     * 10.065,10.069,10.066,10.062,10.057,10.054,10.054,10.060,
     * 9.663,9.650,9.634,9.614,9.591,9.567,9.544,9.523,
     * 9.498,9.474,9.447,9.415,9.381,9.346,9.310,9.280,
     * 9.212,9.185,9.155,9.119,9.082,9.044,9.007,8.972/
C
      DATA ((EDEN(I,5,3,L,2),I=49,56),L=1,NZEN)/
     * 11.131,11.146,11.157,11.167,11.172,11.175,11.174,11.174,
     * 11.089,11.107,11.121,11.134,11.137,11.142,11.140,11.142,
     * 11.034,11.052,11.069,11.080,11.090,11.095,11.097,11.100,
     * 10.935,10.955,10.972,10.987,10.999,11.007,11.013,11.018,
     * 10.735,10.760,10.778,10.793,10.803,10.812,10.819,10.827,
     * 10.590,10.626,10.653,10.672,10.685,10.694,10.701,10.709,
     * 10.306,10.358,10.406,10.447,10.478,10.502,10.519,10.533,
     * 10.010,10.014,10.012,10.010,10.007,10.007,10.010,10.018,
     * 9.630,9.616,9.599,9.578,9.554,9.528,9.503,9.480,
     * 9.465,9.436,9.406,9.372,9.336,9.297,9.261,9.226,
     * 9.165,9.133,9.097,9.058,9.017,8.975,8.935,8.900/
C
      DATA ((EDEN(I,1,4,L,2),I=49,56),L=1,NZEN)/
     * 11.148,11.160,11.164,11.169,11.165,11.165,11.161,11.156,
     * 11.109,11.117,11.123,11.128,11.127,11.122,11.118,11.116,
     * 11.055,11.065,11.072,11.076,11.076,11.075,11.074,11.072,
     * 10.965,10.977,10.987,10.996,11.001,11.002,11.004,11.007,
     * 10.781,10.799,10.813,10.823,10.831,10.837,10.843,10.850,
     * 10.617,10.659,10.691,10.712,10.727,10.738,10.745,10.752,
     * 10.357,10.405,10.452,10.496,10.530,10.556,10.575,10.590,
     * 10.125,10.131,10.131,10.129,10.124,10.118,10.115,10.117,
     * 9.752,9.745,9.733,9.717,9.697,9.675,9.653,9.632,
     * 9.597,9.579,9.556,9.527,9.495,9.463,9.429,9.398,
     * 9.331,9.311,9.287,9.256,9.222,9.188,9.151,9.119/
C
      DATA ((EDEN(I,2,4,L,2),I=49,56),L=1,NZEN)/
     * 11.146,11.158,11.165,11.171,11.168,11.168,11.164,11.158,
     * 11.107,11.118,11.125,11.126,11.129,11.125,11.122,11.119,
     * 11.052,11.063,11.071,11.075,11.077,11.076,11.074,11.075,
     * 10.959,10.973,10.984,10.992,10.997,11.000,11.003,11.005,
     * 10.764,10.789,10.808,10.822,10.831,10.839,10.847,10.855,
     * 10.621,10.660,10.689,10.710,10.723,10.733,10.740,10.748,
     * 10.366,10.410,10.457,10.496,10.529,10.554,10.571,10.586,
     * 10.126,10.132,10.132,10.129,10.124,10.119,10.116,10.119,
     * 9.742,9.735,9.724,9.708,9.689,9.668,9.647,9.628,
     * 9.585,9.566,9.545,9.517,9.486,9.455,9.423,9.393,
     * 9.320,9.302,9.278,9.248,9.216,9.181,9.145,9.113/
C
      DATA ((EDEN(I,3,4,L,2),I=49,56),L=1,NZEN)/
     * 11.139,11.151,11.160,11.167,11.168,11.164,11.161,11.160,
     * 11.100,11.116,11.124,11.133,11.134,11.137,11.134,11.133,
     * 11.044,11.057,11.067,11.073,11.077,11.078,11.077,11.079,
     * 10.950,10.964,10.977,10.986,10.992,10.998,11.001,11.006,
     * 10.767,10.786,10.800,10.810,10.818,10.825,10.831,10.838,
     * 10.622,10.657,10.682,10.701,10.714,10.723,10.730,10.738,
     * 10.374,10.417,10.460,10.494,10.524,10.546,10.563,10.577,
     * 10.123,10.129,10.128,10.126,10.122,10.118,10.116,10.118,
     * 9.719,9.712,9.701,9.687,9.669,9.650,9.630,9.613,
     * 9.555,9.538,9.517,9.492,9.463,9.433,9.404,9.376,
     * 9.281,9.261,9.238,9.211,9.181,9.148,9.117,9.086/
C
      DATA ((EDEN(I,4,4,L,2),I=49,56),L=1,NZEN)/
     * 11.127,11.141,11.151,11.159,11.161,11.163,11.160,11.163,
     * 11.089,11.106,11.119,11.126,11.131,11.131,11.133,11.134,
     * 11.035,11.050,11.064,11.075,11.082,11.085,11.087,11.089,
     * 10.940,10.956,10.971,10.982,10.990,10.996,11.001,11.007,
     * 10.752,10.773,10.789,10.802,10.811,10.818,10.826,10.833,
     * 10.611,10.645,10.672,10.690,10.703,10.713,10.721,10.728,
     * 10.353,10.401,10.445,10.481,10.510,10.533,10.551,10.565,
     * 10.086,10.093,10.095,10.094,10.092,10.091,10.092,10.097,
     * 9.687,9.682,9.672,9.658,9.641,9.622,9.603,9.586,
     * 9.527,9.511,9.492,9.467,9.438,9.410,9.380,9.351,
     * 9.233,9.214,9.194,9.165,9.136,9.106,9.075,9.044/
C
      DATA ((EDEN(I,5,4,L,2),I=49,56),L=1,NZEN)/
     * 11.113,11.128,11.143,11.152,11.156,11.159,11.158,11.161,
     * 11.074,11.093,11.107,11.119,11.122,11.128,11.130,11.132,
     * 11.021,11.040,11.056,11.069,11.077,11.084,11.088,11.091,
     * 10.927,10.946,10.964,10.978,10.988,10.997,11.005,11.012,
     * 10.736,10.760,10.779,10.793,10.804,10.813,10.822,10.831,
     * 10.596,10.632,10.660,10.681,10.694,10.705,10.713,10.721,
     * 10.327,10.378,10.427,10.466,10.497,10.521,10.541,10.554,
     * 10.043,10.052,10.055,10.057,10.057,10.059,10.063,10.070,
     * 9.659,9.653,9.644,9.631,9.614,9.595,9.575,9.556,
     * 9.493,9.475,9.455,9.431,9.403,9.373,9.344,9.315,
     * 9.194,9.175,9.153,9.125,9.095,9.064,9.033,8.999/
C
      DATA ((EDEN(I,1,5,L,2),I=49,56),L=1,NZEN)/
     * 11.144,11.153,11.156,11.158,11.153,11.153,11.145,11.141,
     * 11.105,11.113,11.115,11.115,11.113,11.109,11.104,11.100,
     * 11.051,11.060,11.065,11.066,11.066,11.062,11.060,11.057,
     * 10.962,10.975,10.984,10.990,10.993,10.993,10.995,10.996,
     * 10.783,10.799,10.812,10.821,10.828,10.834,10.839,10.845,
     * 10.633,10.669,10.697,10.714,10.726,10.735,10.742,10.749,
     * 10.381,10.428,10.473,10.511,10.540,10.563,10.578,10.590,
     * 10.126,10.132,10.132,10.129,10.124,10.119,10.117,10.120,
     * 9.736,9.729,9.717,9.701,9.681,9.659,9.637,9.618,
     * 9.575,9.556,9.534,9.505,9.474,9.441,9.409,9.379,
     * 9.310,9.290,9.265,9.236,9.202,9.169,9.135,9.102/
C
      DATA ((EDEN(I,2,5,L,2),I=49,56),L=1,NZEN)/
     * 11.142,11.154,11.158,11.160,11.160,11.156,11.150,11.146,
     * 11.099,11.103,11.109,11.108,11.105,11.102,11.098,11.095,
     * 11.047,11.056,11.062,11.066,11.065,11.064,11.063,11.063,
     * 10.957,10.971,10.982,10.990,10.996,11.000,11.002,11.007,
     * 10.770,10.792,10.809,10.821,10.830,10.838,10.846,10.853,
     * 10.639,10.673,10.697,10.715,10.726,10.735,10.741,10.748,
     * 10.397,10.442,10.483,10.518,10.545,10.566,10.582,10.592,
     * 10.135,10.143,10.144,10.142,10.138,10.134,10.133,10.136,
     * 9.732,9.728,9.719,9.706,9.689,9.671,9.653,9.637,
     * 9.567,9.552,9.535,9.510,9.483,9.454,9.423,9.395,
     * 9.299,9.281,9.262,9.236,9.205,9.176,9.145,9.114/
C
      DATA ((EDEN(I,3,5,L,2),I=49,56),L=1,NZEN)/
     * 11.135,11.144,11.152,11.155,11.156,11.152,11.148,11.147,
     * 11.092,11.105,11.112,11.118,11.116,11.115,11.112,11.111,
     * 11.037,11.048,11.057,11.061,11.063,11.065,11.066,11.064,
     * 10.946,10.960,10.972,10.982,10.988,10.994,10.998,11.003,
     * 10.764,10.785,10.801,10.813,10.822,10.830,10.837,10.846,
     * 10.641,10.670,10.693,10.709,10.720,10.729,10.736,10.743,
     * 10.371,10.415,10.460,10.499,10.532,10.557,10.576,10.591,
     * 10.132,10.141,10.145,10.147,10.146,10.146,10.148,10.152,
     * 9.719,9.717,9.711,9.702,9.689,9.674,9.659,9.647,
     * 9.569,9.557,9.543,9.522,9.497,9.470,9.442,9.414,
     * 9.268,9.255,9.238,9.215,9.190,9.163,9.136,9.111/
C
      DATA ((EDEN(I,4,5,L,2),I=49,56),L=1,NZEN)/
     * 11.116,11.130,11.139,11.147,11.149,11.146,11.148,11.146,
     * 11.081,11.095,11.108,11.115,11.120,11.121,11.121,11.122,
     * 11.028,11.041,11.054,11.063,11.068,11.071,11.073,11.075,
     * 10.934,10.950,10.964,10.975,10.983,10.991,10.997,11.003,
     * 10.754,10.775,10.791,10.804,10.815,10.823,10.832,10.841,
     * 10.620,10.654,10.681,10.701,10.715,10.725,10.734,10.742,
     * 10.374,10.422,10.464,10.501,10.530,10.553,10.573,10.587,
     * 10.107,10.118,10.124,10.127,10.130,10.133,10.137,10.145,
     * 9.698,9.700,9.696,9.689,9.677,9.663,9.649,9.636,
     * 9.535,9.526,9.515,9.497,9.476,9.452,9.426,9.401,
     * 9.228,9.218,9.203,9.184,9.162,9.138,9.113,9.088/
C
      DATA ((EDEN(I,5,5,L,2),I=49,56),L=1,NZEN)/
     * 11.098,11.113,11.124,11.133,11.137,11.139,11.141,11.143,
     * 11.063,11.078,11.092,11.101,11.107,11.114,11.115,11.117,
     * 11.011,11.029,11.045,11.057,11.066,11.073,11.076,11.080,
     * 10.920,10.939,10.956,10.970,10.981,10.989,10.997,11.004,
     * 10.742,10.764,10.783,10.798,10.810,10.820,10.829,10.838,
     * 10.615,10.648,10.673,10.694,10.709,10.720,10.729,10.738,
     * 10.365,10.413,10.457,10.496,10.528,10.550,10.570,10.584,
     * 10.076,10.090,10.099,10.106,10.111,10.116,10.124,10.133,
     * 9.671,9.675,9.674,9.669,9.659,9.647,9.633,9.620,
     * 9.502,9.496,9.487,9.472,9.452,9.430,9.408,9.382,
     * 9.212,9.204,9.194,9.179,9.159,9.137,9.114,9.087/
C
      DATA ((EDEN(I,1,6,L,2),I=49,56),L=1,NZEN)/
     * 11.141,11.145,11.148,11.146,11.141,11.135,11.129,11.123,
     * 11.101,11.104,11.106,11.105,11.100,11.095,11.089,11.084,
     * 11.049,11.055,11.059,11.057,11.054,11.050,11.047,11.043,
     * 10.964,10.974,10.983,10.986,10.987,10.987,10.986,10.986,
     * 10.787,10.801,10.812,10.821,10.828,10.832,10.838,10.844,
     * 10.651,10.681,10.703,10.717,10.727,10.734,10.740,10.748,
     * 10.410,10.454,10.494,10.527,10.551,10.568,10.580,10.590,
     * 10.129,10.135,10.133,10.130,10.125,10.120,10.120,10.125,
     * 9.720,9.713,9.700,9.684,9.664,9.642,9.621,9.602,
     * 9.551,9.532,9.509,9.481,9.451,9.419,9.388,9.358,
     * 9.286,9.265,9.241,9.213,9.181,9.146,9.113,9.084/
C
      DATA ((EDEN(I,2,6,L,2),I=49,56),L=1,NZEN)/
     * 11.142,11.151,11.150,11.152,11.148,11.143,11.136,11.131,
     * 11.095,11.099,11.096,11.095,11.092,11.087,11.082,11.079,
     * 11.044,11.052,11.055,11.056,11.055,11.051,11.050,11.049,
     * 10.958,10.971,10.981,10.988,10.992,10.994,10.998,10.999,
     * 10.776,10.796,10.811,10.823,10.832,10.839,10.847,10.855,
     * 10.656,10.684,10.705,10.720,10.729,10.738,10.744,10.751,
     * 10.374,10.421,10.467,10.509,10.542,10.567,10.585,10.601,
     * 10.133,10.142,10.146,10.147,10.147,10.147,10.149,10.155,
     * 9.724,9.722,9.715,9.704,9.690,9.675,9.660,9.648,
     * 9.580,9.567,9.549,9.527,9.499,9.470,9.440,9.411,
     * 9.284,9.270,9.252,9.228,9.200,9.171,9.142,9.116/
C
      DATA ((EDEN(I,3,6,L,2),I=49,56),L=1,NZEN)/
     * 11.131,11.140,11.145,11.147,11.148,11.144,11.137,11.133,
     * 11.085,11.093,11.096,11.098,11.097,11.094,11.090,11.087,
     * 11.035,11.044,11.051,11.055,11.056,11.054,11.053,11.055,
     * 10.947,10.960,10.971,10.979,10.987,10.991,10.994,10.997,
     * 10.770,10.789,10.804,10.816,10.826,10.834,10.842,10.850,
     * 10.637,10.671,10.697,10.716,10.729,10.740,10.749,10.757,
     * 10.403,10.448,10.489,10.524,10.553,10.574,10.591,10.604,
     * 10.145,10.157,10.162,10.165,10.167,10.168,10.172,10.178,
     * 9.726,9.728,9.725,9.717,9.707,9.694,9.682,9.672,
     * 9.559,9.552,9.540,9.524,9.502,9.478,9.452,9.427,
     * 9.234,9.224,9.212,9.192,9.171,9.149,9.127,9.104/
C
      DATA ((EDEN(I,4,6,L,2),I=49,56),L=1,NZEN)/
     * 11.113,11.123,11.132,11.135,11.137,11.138,11.135,11.134,
     * 11.078,11.087,11.096,11.103,11.106,11.107,11.106,11.106,
     * 11.023,11.037,11.047,11.054,11.058,11.062,11.062,11.066,
     * 10.933,10.949,10.963,10.976,10.984,10.989,10.995,11.001,
     * 10.756,10.778,10.796,10.811,10.823,10.833,10.843,10.852,
     * 10.638,10.668,10.693,10.712,10.725,10.735,10.745,10.753,
     * 10.393,10.440,10.484,10.521,10.550,10.574,10.593,10.606,
     * 10.122,10.137,10.146,10.152,10.158,10.163,10.169,10.178,
     * 9.701,9.708,9.709,9.706,9.699,9.689,9.679,9.672,
     * 9.532,9.530,9.524,9.512,9.495,9.476,9.452,9.430,
     * 9.231,9.228,9.221,9.208,9.192,9.172,9.152,9.129/
C
      DATA ((EDEN(I,5,6,L,2),I=49,56),L=1,NZEN)/
     * 11.086,11.101,11.116,11.121,11.124,11.130,11.129,11.130,
     * 11.056,11.070,11.085,11.093,11.101,11.104,11.107,11.108,
     * 11.007,11.024,11.039,11.051,11.059,11.064,11.069,11.072,
     * 10.919,10.938,10.955,10.970,10.981,10.989,10.998,11.004,
     * 10.746,10.768,10.788,10.804,10.818,10.829,10.840,10.850,
     * 10.629,10.660,10.687,10.707,10.722,10.735,10.745,10.755,
     * 10.399,10.447,10.489,10.525,10.554,10.577,10.595,10.608,
     * 10.100,10.118,10.131,10.141,10.150,10.159,10.169,10.179,
     * 9.676,9.686,9.692,9.692,9.687,9.681,9.674,9.668,
     * 9.497,9.499,9.496,9.488,9.475,9.457,9.437,9.416,
     * 9.222,9.226,9.225,9.218,9.207,9.189,9.170,9.148/
C
      DATA ((EDEN(I,1,7,L,2),I=49,56),L=1,NZEN)/
     * 11.141,11.145,11.144,11.146,11.141,11.133,11.126,11.120,
     * 11.097,11.104,11.106,11.103,11.098,11.091,11.086,11.082,
     * 11.050,11.055,11.057,11.055,11.053,11.048,11.044,11.041,
     * 10.966,10.975,10.981,10.986,10.986,10.986,10.985,10.984,
     * 10.787,10.801,10.813,10.822,10.828,10.834,10.839,10.844,
     * 10.654,10.683,10.704,10.717,10.727,10.734,10.740,10.748,
     * 10.416,10.460,10.499,10.530,10.552,10.568,10.580,10.590,
     * 10.130,10.136,10.134,10.130,10.125,10.121,10.121,10.126,
     * 9.717,9.709,9.696,9.680,9.660,9.639,9.618,9.600,
     * 9.545,9.526,9.504,9.476,9.446,9.415,9.384,9.355,
     * 9.281,9.260,9.238,9.209,9.175,9.144,9.111,9.082/
C
      DATA ((EDEN(I,2,7,L,2),I=49,56),L=1,NZEN)/
     * 11.142,11.147,11.150,11.148,11.144,11.138,11.132,11.126,
     * 11.095,11.094,11.097,11.094,11.089,11.083,11.079,11.075,
     * 11.045,11.051,11.055,11.054,11.053,11.051,11.047,11.046,
     * 10.959,10.971,10.982,10.989,10.993,10.995,10.996,10.999,
     * 10.778,10.797,10.812,10.824,10.834,10.842,10.849,10.857,
     * 10.659,10.686,10.706,10.721,10.731,10.739,10.746,10.754,
     * 10.380,10.427,10.473,10.513,10.545,10.569,10.588,10.604,
     * 10.135,10.145,10.148,10.149,10.149,10.150,10.153,10.159,
     * 9.722,9.721,9.714,9.704,9.691,9.676,9.663,9.652,
     * 9.576,9.564,9.548,9.525,9.498,9.471,9.441,9.413,
     * 9.281,9.268,9.250,9.226,9.200,9.174,9.145,9.116/
C
      DATA ((EDEN(I,3,7,L,2),I=49,56),L=1,NZEN)/
     * 11.131,11.140,11.145,11.147,11.144,11.139,11.135,11.130,
     * 11.085,11.093,11.096,11.097,11.096,11.092,11.089,11.086,
     * 11.037,11.045,11.052,11.055,11.055,11.055,11.054,11.053,
     * 10.949,10.962,10.973,10.983,10.989,10.991,10.995,10.998,
     * 10.772,10.791,10.807,10.819,10.830,10.838,10.846,10.855,
     * 10.641,10.673,10.699,10.719,10.733,10.743,10.752,10.761,
     * 10.409,10.453,10.494,10.529,10.557,10.578,10.595,10.608,
     * 10.148,10.161,10.166,10.170,10.172,10.175,10.179,10.185,
     * 9.726,9.730,9.728,9.721,9.711,9.700,9.689,9.681,
     * 9.559,9.552,9.544,9.527,9.507,9.483,9.459,9.434,
     * 9.234,9.225,9.214,9.195,9.175,9.153,9.134,9.112/
C
      DATA ((EDEN(I,4,7,L,2),I=49,56),L=1,NZEN)/
     * 11.113,11.123,11.132,11.139,11.137,11.138,11.135,11.133,
     * 11.078,11.092,11.100,11.107,11.108,11.108,11.107,11.106,
     * 11.025,11.038,11.049,11.058,11.062,11.063,11.065,11.065,
     * 10.936,10.952,10.966,10.979,10.988,10.993,11.000,11.004,
     * 10.758,10.780,10.800,10.815,10.828,10.839,10.849,10.859,
     * 10.641,10.671,10.697,10.715,10.730,10.742,10.750,10.760,
     * 10.397,10.445,10.489,10.526,10.557,10.580,10.598,10.612,
     * 10.124,10.140,10.151,10.158,10.164,10.170,10.178,10.187,
     * 9.704,9.712,9.715,9.713,9.707,9.698,9.690,9.683,
     * 9.535,9.535,9.530,9.519,9.503,9.485,9.463,9.440,
     * 9.236,9.234,9.229,9.217,9.203,9.186,9.165,9.142/
C
      DATA ((EDEN(I,5,7,L,2),I=49,56),L=1,NZEN)/
     * 11.090,11.105,11.116,11.125,11.128,11.130,11.132,11.132,
     * 11.056,11.075,11.089,11.098,11.104,11.108,11.109,11.111,
     * 11.010,11.028,11.044,11.055,11.064,11.069,11.073,11.076,
     * 10.922,10.941,10.960,10.975,10.986,10.996,11.002,11.009,
     * 10.748,10.772,10.793,10.810,10.825,10.837,10.849,10.859,
     * 10.631,10.664,10.691,10.712,10.729,10.743,10.753,10.764,
     * 10.400,10.447,10.493,10.532,10.561,10.585,10.604,10.618,
     * 10.103,10.123,10.137,10.149,10.159,10.168,10.179,10.190,
     * 9.681,9.694,9.701,9.703,9.701,9.695,9.690,9.686,
     * 9.504,9.508,9.508,9.502,9.489,9.474,9.452,9.432,
     * 9.231,9.237,9.238,9.234,9.221,9.208,9.190,9.166/
C
      DATA ((EDEN(I,1,8,L,2),I=49,56),L=1,NZEN)/
     * 11.141,11.149,11.152,11.154,11.149,11.144,11.139,11.133,
     * 11.101,11.109,11.111,11.111,11.108,11.102,11.098,11.094,
     * 11.052,11.059,11.063,11.063,11.060,11.058,11.055,11.053,
     * 10.965,10.975,10.983,10.988,10.991,10.991,10.992,10.993,
     * 10.785,10.800,10.812,10.821,10.828,10.834,10.840,10.846,
     * 10.643,10.675,10.699,10.715,10.726,10.734,10.741,10.749,
     * 10.397,10.441,10.484,10.518,10.544,10.563,10.578,10.589,
     * 10.128,10.134,10.134,10.130,10.125,10.120,10.119,10.123,
     * 9.728,9.721,9.709,9.693,9.674,9.652,9.632,9.613,
     * 9.561,9.543,9.521,9.494,9.464,9.432,9.401,9.372,
     * 9.295,9.277,9.254,9.226,9.192,9.160,9.128,9.097/
C
      DATA ((EDEN(I,2,8,L,2),I=49,56),L=1,NZEN)/
     * 11.142,11.151,11.154,11.156,11.152,11.147,11.142,11.137,
     * 11.099,11.103,11.105,11.103,11.100,11.096,11.091,11.088,
     * 11.049,11.056,11.061,11.063,11.063,11.061,11.059,11.058,
     * 10.959,10.974,10.985,10.993,10.998,11.001,11.004,11.007,
     * 10.774,10.795,10.812,10.824,10.834,10.842,10.851,10.858,
     * 10.647,10.679,10.702,10.718,10.730,10.739,10.747,10.754,
     * 10.411,10.454,10.493,10.527,10.553,10.573,10.587,10.598,
     * 10.146,10.154,10.156,10.155,10.151,10.148,10.147,10.151,
     * 9.739,9.737,9.729,9.717,9.702,9.685,9.669,9.655,
     * 9.592,9.578,9.561,9.538,9.510,9.481,9.451,9.422,
     * 9.290,9.276,9.258,9.234,9.208,9.180,9.150,9.121/
C
      DATA ((EDEN(I,3,8,L,2),I=49,56),L=1,NZEN)/
     * 11.135,11.144,11.152,11.155,11.152,11.148,11.145,11.141,
     * 11.092,11.101,11.104,11.105,11.106,11.103,11.100,11.098,
     * 11.040,11.051,11.059,11.064,11.064,11.065,11.064,11.065,
     * 10.950,10.965,10.977,10.987,10.994,10.998,11.003,11.007,
     * 10.769,10.790,10.806,10.819,10.830,10.839,10.848,10.856,
     * 10.648,10.677,10.700,10.717,10.728,10.738,10.746,10.753,
     * 10.386,10.430,10.472,10.512,10.543,10.568,10.588,10.603,
     * 10.141,10.152,10.157,10.160,10.161,10.163,10.165,10.171,
     * 9.733,9.735,9.731,9.724,9.712,9.699,9.686,9.676,
     * 9.571,9.563,9.552,9.533,9.511,9.486,9.460,9.434,
     * 9.251,9.242,9.230,9.211,9.189,9.165,9.143,9.118/
C
      DATA ((EDEN(I,4,8,L,2),I=49,56),L=1,NZEN)/
     * 11.120,11.134,11.143,11.147,11.149,11.150,11.148,11.145,
     * 11.082,11.099,11.108,11.115,11.116,11.118,11.118,11.117,
     * 11.032,11.047,11.060,11.068,11.073,11.076,11.077,11.077,
     * 10.939,10.957,10.972,10.983,10.994,11.000,11.005,11.011,
     * 10.760,10.782,10.800,10.815,10.826,10.837,10.847,10.856,
     * 10.628,10.663,10.691,10.712,10.728,10.739,10.749,10.758,
     * 10.385,10.432,10.478,10.516,10.547,10.571,10.589,10.605,
     * 10.114,10.129,10.137,10.143,10.147,10.151,10.157,10.165,
     * 9.707,9.712,9.713,9.709,9.700,9.689,9.678,9.668,
     * 9.544,9.541,9.532,9.519,9.501,9.479,9.455,9.432,
     * 9.238,9.235,9.227,9.211,9.193,9.172,9.150,9.126/
C
      DATA ((EDEN(I,5,8,L,2),I=49,56),L=1,NZEN)/
     * 11.102,11.117,11.132,11.141,11.145,11.147,11.146,11.148,
     * 11.068,11.087,11.101,11.110,11.117,11.123,11.123,11.124,
     * 11.017,11.037,11.053,11.066,11.075,11.081,11.085,11.089,
     * 10.927,10.946,10.965,10.981,10.993,11.002,11.011,11.017,
     * 10.746,10.772,10.794,10.812,10.826,10.839,10.850,10.861,
     * 10.622,10.658,10.687,10.709,10.726,10.739,10.750,10.760,
     * 10.369,10.422,10.471,10.512,10.546,10.572,10.592,10.607,
     * 10.083,10.101,10.114,10.125,10.133,10.142,10.151,10.163,
     * 9.689,9.698,9.702,9.701,9.695,9.686,9.676,9.666,
     * 9.523,9.523,9.519,9.508,9.492,9.472,9.451,9.426,
     * 9.234,9.235,9.231,9.222,9.207,9.190,9.167,9.142/
C
      DATA ((EDEN(I,1,9,L,2),I=49,56),L=1,NZEN)/
     * 11.144,11.156,11.160,11.165,11.161,11.157,11.156,11.151,
     * 11.105,11.113,11.119,11.123,11.120,11.117,11.113,11.112,
     * 11.053,11.062,11.069,11.071,11.071,11.069,11.068,11.067,
     * 10.963,10.975,10.985,10.992,10.996,10.998,11.000,11.003,
     * 10.781,10.798,10.811,10.821,10.828,10.835,10.841,10.848,
     * 10.625,10.662,10.692,10.712,10.726,10.735,10.742,10.750,
     * 10.368,10.415,10.460,10.500,10.531,10.556,10.574,10.589,
     * 10.128,10.134,10.134,10.131,10.126,10.120,10.118,10.119,
     * 9.746,9.739,9.728,9.712,9.693,9.672,9.651,9.632,
     * 9.589,9.570,9.549,9.521,9.491,9.458,9.425,9.395,
     * 9.322,9.303,9.280,9.251,9.218,9.184,9.151,9.116/
C
      DATA ((EDEN(I,2,9,L,2),I=49,56),L=1,NZEN)/
     * 11.146,11.154,11.162,11.164,11.164,11.160,11.155,11.153,
     * 11.103,11.111,11.113,11.114,11.113,11.111,11.107,11.106,
     * 11.051,11.062,11.068,11.073,11.074,11.075,11.073,11.073,
     * 10.960,10.974,10.986,10.996,11.003,11.009,11.012,11.015,
     * 10.765,10.791,10.809,10.824,10.833,10.841,10.849,10.858,
     * 10.627,10.664,10.693,10.713,10.727,10.736,10.744,10.752,
     * 10.377,10.422,10.466,10.504,10.535,10.559,10.577,10.592,
     * 10.133,10.141,10.142,10.140,10.136,10.132,10.130,10.132,
     * 9.743,9.739,9.729,9.716,9.700,9.681,9.662,9.645,
     * 9.584,9.568,9.551,9.525,9.497,9.468,9.437,9.408,
     * 9.303,9.287,9.266,9.242,9.211,9.181,9.150,9.122/
C
      DATA ((EDEN(I,3,9,L,2),I=49,56),L=1,NZEN)/
     * 11.139,11.148,11.156,11.163,11.160,11.160,11.157,11.155,
     * 11.100,11.109,11.120,11.122,11.126,11.123,11.122,11.120,
     * 11.045,11.058,11.067,11.073,11.076,11.076,11.077,11.076,
     * 10.951,10.966,10.979,10.991,10.999,11.005,11.009,11.014,
     * 10.762,10.785,10.804,10.818,10.827,10.836,10.844,10.853,
     * 10.629,10.664,10.690,10.710,10.723,10.733,10.741,10.748,
     * 10.386,10.428,10.471,10.507,10.537,10.558,10.577,10.591,
     * 10.136,10.145,10.148,10.147,10.145,10.142,10.141,10.144,
     * 9.733,9.731,9.723,9.712,9.697,9.680,9.664,9.649,
     * 9.581,9.569,9.553,9.532,9.506,9.477,9.448,9.419,
     * 9.286,9.272,9.255,9.233,9.206,9.180,9.151,9.122/
C
      DATA ((EDEN(I,4,9,L,2),I=49,56),L=1,NZEN)/
     * 11.127,11.141,11.151,11.159,11.161,11.163,11.160,11.158,
     * 11.089,11.107,11.119,11.127,11.132,11.132,11.135,11.135,
     * 11.036,11.052,11.065,11.075,11.081,11.084,11.086,11.088,
     * 10.942,10.960,10.975,10.987,10.996,11.003,11.009,11.014,
     * 10.753,10.778,10.797,10.811,10.822,10.832,10.841,10.851,
     * 10.621,10.656,10.684,10.703,10.717,10.728,10.737,10.745,
     * 10.347,10.396,10.445,10.487,10.522,10.549,10.570,10.586,
     * 10.090,10.101,10.107,10.111,10.113,10.116,10.120,10.128,
     * 9.702,9.702,9.698,9.689,9.677,9.662,9.646,9.632,
     * 9.541,9.530,9.516,9.497,9.474,9.448,9.422,9.394,
     * 9.233,9.220,9.205,9.184,9.159,9.135,9.107,9.079/
C
      DATA ((EDEN(I,5,9,L,2),I=49,56),L=1,NZEN)/
     * 11.116,11.132,11.146,11.156,11.160,11.163,11.162,11.166,
     * 11.079,11.097,11.112,11.121,11.128,11.134,11.136,11.138,
     * 11.024,11.044,11.062,11.075,11.085,11.092,11.095,11.098,
     * 10.932,10.951,10.970,10.987,10.998,11.007,11.014,11.021,
     * 10.742,10.769,10.790,10.807,10.820,10.831,10.841,10.851,
     * 10.608,10.645,10.676,10.698,10.714,10.727,10.736,10.745,
     * 10.332,10.387,10.440,10.483,10.518,10.547,10.568,10.584,
     * 10.052,10.066,10.075,10.081,10.086,10.091,10.098,10.108,
     * 9.679,9.681,9.677,9.670,9.658,9.644,9.627,9.611,
     * 9.518,9.509,9.497,9.478,9.455,9.429,9.402,9.373,
     * 9.226,9.213,9.199,9.179,9.155,9.130,9.103,9.073/
C
      DATA ((EDEN(I,1,10,L,2),I=49,56),L=1,NZEN)/
     * 11.148,11.160,11.167,11.169,11.170,11.165,11.165,11.160,
     * 11.109,11.117,11.123,11.128,11.127,11.125,11.122,11.119,
     * 11.055,11.065,11.072,11.077,11.078,11.077,11.076,11.075,
     * 10.965,10.977,10.987,10.996,11.000,11.003,11.005,11.009,
     * 10.782,10.801,10.815,10.824,10.832,10.838,10.844,10.850,
     * 10.618,10.660,10.692,10.714,10.729,10.739,10.747,10.753,
     * 10.359,10.405,10.452,10.496,10.530,10.556,10.575,10.592,
     * 10.130,10.136,10.137,10.134,10.130,10.124,10.120,10.122,
     * 9.757,9.750,9.738,9.723,9.704,9.682,9.661,9.640,
     * 9.604,9.585,9.563,9.535,9.504,9.471,9.439,9.406,
     * 9.337,9.318,9.294,9.265,9.232,9.196,9.162,9.128/
C
      DATA ((EDEN(I,2,10,L,2),I=49,56),L=1,NZEN)/
     * 11.146,11.158,11.165,11.168,11.168,11.164,11.164,11.158,
     * 11.107,11.118,11.125,11.126,11.129,11.125,11.123,11.121,
     * 11.054,11.065,11.073,11.078,11.079,11.078,11.078,11.077,
     * 10.963,10.977,10.987,10.996,11.000,11.003,11.006,11.011,
     * 10.779,10.798,10.812,10.822,10.830,10.836,10.842,10.850,
     * 10.617,10.658,10.690,10.711,10.726,10.735,10.743,10.751,
     * 10.358,10.404,10.451,10.492,10.526,10.552,10.572,10.588,
     * 10.127,10.132,10.132,10.131,10.128,10.126,10.127,10.131,
     * 9.748,9.741,9.729,9.713,9.694,9.673,9.651,9.631,
     * 9.593,9.575,9.552,9.524,9.493,9.460,9.428,9.398,
     * 9.319,9.300,9.276,9.248,9.215,9.180,9.147,9.114/
C
      DATA ((EDEN(I,3,10,L,2),I=49,56),L=1,NZEN)/
     * 11.139,11.151,11.160,11.167,11.168,11.164,11.161,11.161,
     * 11.104,11.116,11.127,11.133,11.133,11.137,11.134,11.133,
     * 11.048,11.062,11.071,11.077,11.081,11.082,11.081,11.082,
     * 10.955,10.969,10.981,10.991,10.997,11.003,11.007,11.010,
     * 10.770,10.789,10.804,10.815,10.822,10.830,10.836,10.843,
     * 10.620,10.656,10.684,10.704,10.717,10.726,10.733,10.740,
     * 10.367,10.410,10.454,10.493,10.522,10.546,10.565,10.579,
     * 10.121,10.127,10.126,10.124,10.121,10.119,10.119,10.123,
     * 9.722,9.715,9.703,9.688,9.670,9.649,9.628,9.610,
     * 9.560,9.542,9.521,9.494,9.464,9.432,9.401,9.372,
     * 9.271,9.253,9.229,9.203,9.172,9.140,9.108,9.077/
C
      DATA ((EDEN(I,4,10,L,2),I=49,56),L=1,NZEN)/
     * 11.135,11.149,11.158,11.166,11.169,11.171,11.168,11.167,
     * 11.096,11.114,11.122,11.133,11.139,11.139,11.140,11.140,
     * 11.041,11.058,11.073,11.084,11.090,11.094,11.097,11.099,
     * 10.947,10.964,10.979,10.991,10.999,11.005,11.010,11.014,
     * 10.756,10.777,10.794,10.806,10.816,10.824,10.832,10.840,
     * 10.623,10.655,10.679,10.696,10.708,10.716,10.724,10.731,
     * 10.347,10.396,10.443,10.482,10.512,10.535,10.552,10.566,
     * 10.072,10.080,10.081,10.081,10.080,10.080,10.084,10.091,
     * 9.681,9.674,9.663,9.649,9.631,9.611,9.591,9.573,
     * 9.514,9.496,9.475,9.449,9.419,9.388,9.358,9.329,
     * 9.233,9.212,9.191,9.161,9.131,9.097,9.064,9.032/
C
      DATA ((EDEN(I,5,10,L,2),I=49,56),L=1,NZEN)/
     * 11.127,11.142,11.154,11.164,11.168,11.171,11.170,11.170,
     * 11.090,11.104,11.118,11.131,11.138,11.140,11.142,11.143,
     * 11.031,11.049,11.067,11.083,11.090,11.095,11.099,11.102,
     * 10.936,10.955,10.974,10.988,11.001,11.010,11.016,11.022,
     * 10.742,10.767,10.786,10.801,10.812,10.821,10.830,10.838,
     * 10.601,10.638,10.666,10.686,10.700,10.710,10.718,10.726,
     * 10.322,10.378,10.429,10.470,10.502,10.526,10.543,10.558,
     * 10.022,10.031,10.035,10.037,10.038,10.040,10.047,10.056,
     * 9.651,9.644,9.633,9.618,9.600,9.579,9.558,9.538,
     * 9.485,9.464,9.441,9.413,9.383,9.352,9.318,9.288,
     * 9.194,9.170,9.145,9.113,9.079,9.046,9.011,8.976/
C
      DATA ((EDEN(I,1,11,L,2),I=49,56),L=1,NZEN)/
     * 11.155,11.164,11.171,11.173,11.170,11.169,11.161,11.158,
     * 11.113,11.125,11.127,11.132,11.127,11.125,11.121,11.117,
     * 11.062,11.071,11.077,11.080,11.079,11.077,11.076,11.074,
     * 10.971,10.984,10.994,11.000,11.004,11.005,11.008,11.008,
     * 10.790,10.808,10.821,10.830,10.836,10.842,10.848,10.854,
     * 10.632,10.672,10.702,10.722,10.736,10.744,10.751,10.757,
     * 10.372,10.421,10.468,10.509,10.541,10.567,10.584,10.598,
     * 10.133,10.140,10.140,10.137,10.131,10.126,10.123,10.125,
     * 9.757,9.749,9.736,9.720,9.700,9.678,9.655,9.634,
     * 9.600,9.580,9.557,9.528,9.496,9.463,9.431,9.399,
     * 9.332,9.313,9.287,9.258,9.224,9.188,9.153,9.121/
C
      DATA ((EDEN(I,2,11,L,2),I=49,56),L=1,NZEN)/
     * 11.149,11.158,11.165,11.168,11.168,11.164,11.160,11.155,
     * 11.110,11.126,11.133,11.138,11.137,11.136,11.132,11.129,
     * 11.059,11.071,11.078,11.081,11.082,11.080,11.078,11.077,
     * 10.970,10.982,10.994,11.000,11.006,11.008,11.009,11.011,
     * 10.785,10.804,10.817,10.826,10.833,10.839,10.844,10.850,
     * 10.625,10.665,10.695,10.715,10.727,10.736,10.743,10.749,
     * 10.364,10.412,10.459,10.500,10.531,10.556,10.573,10.587,
     * 10.122,10.126,10.125,10.122,10.118,10.116,10.116,10.122,
     * 9.741,9.731,9.715,9.697,9.675,9.651,9.627,9.605,
     * 9.583,9.560,9.534,9.502,9.468,9.434,9.399,9.369,
     * 9.303,9.279,9.250,9.219,9.185,9.149,9.114,9.080/
C
      DATA ((EDEN(I,3,11,L,2),I=49,56),L=1,NZEN)/
     * 11.142,11.155,11.164,11.167,11.164,11.164,11.161,11.158,
     * 11.107,11.119,11.127,11.132,11.132,11.136,11.132,11.130,
     * 11.052,11.066,11.078,11.086,11.089,11.090,11.091,11.091,
     * 10.960,10.975,10.986,10.995,11.001,11.003,11.005,11.009,
     * 10.774,10.792,10.806,10.815,10.822,10.828,10.835,10.841,
     * 10.646,10.673,10.693,10.706,10.714,10.719,10.724,10.731,
     * 10.371,10.414,10.457,10.493,10.521,10.541,10.558,10.570,
     * 10.112,10.114,10.111,10.106,10.101,10.097,10.097,10.101,
     * 9.695,9.683,9.666,9.647,9.624,9.600,9.577,9.556,
     * 9.540,9.515,9.489,9.457,9.422,9.388,9.352,9.322,
     * 9.255,9.229,9.200,9.167,9.131,9.093,9.058,9.025/
C
      DATA ((EDEN(I,4,11,L,2),I=49,56),L=1,NZEN)/
     * 11.142,11.156,11.162,11.170,11.173,11.171,11.168,11.167,
     * 11.103,11.116,11.129,11.136,11.137,11.138,11.139,11.138,
     * 11.049,11.065,11.076,11.086,11.092,11.094,11.096,11.097,
     * 10.950,10.968,10.982,10.995,11.004,11.010,11.015,11.018,
     * 10.760,10.779,10.793,10.804,10.812,10.819,10.825,10.832,
     * 10.630,10.658,10.678,10.691,10.700,10.706,10.711,10.717,
     * 10.368,10.414,10.455,10.487,10.510,10.527,10.541,10.551,
     * 10.060,10.063,10.059,10.055,10.050,10.048,10.049,10.056,
     * 9.647,9.633,9.616,9.595,9.572,9.547,9.524,9.504,
     * 9.470,9.443,9.415,9.382,9.347,9.312,9.278,9.247,
     * 9.197,9.167,9.138,9.100,9.063,9.024,8.988,8.952/
C
      DATA ((EDEN(I,5,11,L,2),I=49,56),L=1,NZEN)/
     * 11.134,11.149,11.165,11.171,11.172,11.175,11.174,11.174,
     * 11.096,11.114,11.125,11.137,11.140,11.142,11.144,11.142,
     * 11.038,11.059,11.072,11.083,11.093,11.097,11.099,11.101,
     * 10.940,10.960,10.977,10.991,11.002,11.008,11.014,11.019,
     * 10.740,10.764,10.783,10.797,10.807,10.815,10.822,10.830,
     * 10.602,10.636,10.660,10.678,10.689,10.697,10.702,10.709,
     * 10.325,10.379,10.425,10.462,10.489,10.509,10.522,10.534,
     * 9.998,10.001,10.000,9.998,9.996,9.997,10.002,10.012,
     * 9.610,9.595,9.576,9.554,9.529,9.502,9.477,9.455,
     * 9.435,9.406,9.376,9.339,9.302,9.262,9.226,9.191,
     * 9.160,9.124,9.089,9.048,9.005,8.962,8.918,8.880/
C
      DATA ((EDEN(I,1,12,L,2),I=49,56),L=1,NZEN)/
     * 11.159,11.167,11.171,11.173,11.170,11.161,11.156,11.152,
     * 11.121,11.129,11.131,11.132,11.127,11.122,11.116,11.112,
     * 11.066,11.076,11.080,11.082,11.079,11.076,11.072,11.071,
     * 10.980,10.991,11.000,11.005,11.007,11.008,11.007,11.009,
     * 10.800,10.816,10.827,10.836,10.843,10.848,10.853,10.859,
     * 10.649,10.685,10.712,10.730,10.741,10.749,10.755,10.762,
     * 10.394,10.443,10.488,10.525,10.555,10.576,10.592,10.604,
     * 10.136,10.142,10.141,10.137,10.132,10.127,10.126,10.130,
     * 9.750,9.742,9.729,9.712,9.691,9.668,9.645,9.623,
     * 9.589,9.568,9.544,9.515,9.482,9.450,9.416,9.385,
     * 9.322,9.302,9.275,9.245,9.210,9.175,9.142,9.109/
C
      DATA ((EDEN(I,2,12,L,2),I=49,56),L=1,NZEN)/
     * 11.153,11.162,11.165,11.168,11.164,11.160,11.155,11.149,
     * 11.118,11.126,11.133,11.137,11.137,11.131,11.128,11.124,
     * 11.067,11.078,11.083,11.086,11.086,11.083,11.081,11.078,
     * 10.978,10.990,11.000,11.005,11.009,11.010,11.010,11.010,
     * 10.793,10.810,10.822,10.831,10.837,10.843,10.848,10.853,
     * 10.669,10.694,10.711,10.723,10.729,10.734,10.739,10.745,
     * 10.379,10.428,10.473,10.512,10.540,10.560,10.575,10.587,
     * 10.119,10.123,10.120,10.116,10.111,10.109,10.111,10.118,
     * 9.713,9.700,9.683,9.663,9.640,9.615,9.591,9.569,
     * 9.567,9.541,9.513,9.479,9.443,9.407,9.372,9.340,
     * 9.283,9.257,9.227,9.193,9.157,9.119,9.084,9.051/
C
      DATA ((EDEN(I,3,12,L,2),I=49,56),L=1,NZEN)/
     * 11.146,11.155,11.164,11.167,11.164,11.160,11.157,11.151,
     * 11.111,11.123,11.130,11.132,11.132,11.131,11.128,11.125,
     * 11.059,11.071,11.081,11.088,11.090,11.090,11.088,11.087,
     * 10.967,10.980,10.991,10.999,11.003,11.004,11.005,11.007,
     * 10.783,10.798,10.809,10.817,10.823,10.828,10.833,10.838,
     * 10.654,10.678,10.696,10.707,10.713,10.718,10.723,10.729,
     * 10.415,10.456,10.491,10.517,10.535,10.549,10.557,10.566,
     * 10.112,10.112,10.105,10.097,10.089,10.083,10.082,10.086,
     * 9.678,9.662,9.642,9.618,9.592,9.565,9.539,9.517,
     * 9.498,9.469,9.438,9.404,9.366,9.331,9.296,9.264,
     * 9.230,9.199,9.167,9.130,9.091,9.052,9.014,8.982/
C
      DATA ((EDEN(I,4,12,L,2),I=49,56),L=1,NZEN)/
     * 11.145,11.159,11.166,11.170,11.173,11.171,11.164,11.163,
     * 11.107,11.123,11.132,11.139,11.141,11.137,11.137,11.134,
     * 11.052,11.068,11.079,11.089,11.093,11.095,11.094,11.094,
     * 10.955,10.972,10.987,10.999,11.007,11.011,11.013,11.017,
     * 10.764,10.782,10.795,10.805,10.813,10.818,10.824,10.831,
     * 10.636,10.660,10.677,10.689,10.696,10.701,10.707,10.713,
     * 10.387,10.431,10.467,10.493,10.512,10.525,10.535,10.542,
     * 10.054,10.054,10.048,10.042,10.036,10.034,10.037,10.043,
     * 9.623,9.605,9.583,9.558,9.531,9.503,9.477,9.456,
     * 9.443,9.412,9.379,9.342,9.302,9.263,9.227,9.194,
     * 9.167,9.131,9.096,9.053,9.011,8.969,8.930,8.892/
C
      DATA ((EDEN(I,5,12,L,2),I=49,56),L=1,NZEN)/
     * 11.141,11.156,11.168,11.175,11.175,11.175,11.174,11.170,
     * 11.103,11.117,11.131,11.140,11.143,11.145,11.142,11.141,
     * 11.045,11.066,11.078,11.089,11.094,11.097,11.098,11.097,
     * 10.945,10.964,10.982,10.994,11.003,11.010,11.013,11.017,
     * 10.742,10.764,10.781,10.795,10.804,10.812,10.818,10.825,
     * 10.598,10.632,10.656,10.672,10.682,10.690,10.695,10.701,
     * 10.320,10.374,10.421,10.456,10.482,10.499,10.512,10.521,
     * 9.980,9.981,9.977,9.973,9.971,9.971,9.976,9.986,
     * 9.582,9.561,9.538,9.511,9.482,9.452,9.425,9.402,
     * 9.400,9.365,9.328,9.285,9.243,9.199,9.158,9.122,
     * 9.117,9.077,9.034,8.988,8.938,8.892,8.847,8.805/
C
      DATA ((EDEN(I,1,1,L,3),I=49,56),L=1,NZEN)/
     * 11.254,11.264,11.267,11.269,11.268,11.264,11.263,11.259,
     * 11.214,11.223,11.225,11.228,11.226,11.224,11.222,11.221,
     * 11.162,11.170,11.174,11.177,11.177,11.177,11.177,11.174,
     * 11.064,11.080,11.090,11.095,11.101,11.104,11.108,11.112,
     * 10.871,10.888,10.902,10.915,10.925,10.934,10.944,10.954,
     * 10.730,10.761,10.784,10.802,10.815,10.826,10.836,10.847,
     * 10.505,10.546,10.583,10.614,10.637,10.657,10.671,10.685,
     * 10.233,10.241,10.241,10.240,10.237,10.236,10.238,10.246,
     * 9.780,9.777,9.769,9.757,9.743,9.728,9.713,9.703,
     * 9.602,9.589,9.573,9.552,9.529,9.504,9.480,9.460,
     * 9.334,9.320,9.303,9.282,9.259,9.235,9.212,9.189/
C
      DATA ((EDEN(I,2,1,L,3),I=49,56),L=1,NZEN)/
     * 11.249,11.256,11.262,11.264,11.263,11.262,11.258,11.255,
     * 11.211,11.220,11.229,11.232,11.234,11.232,11.231,11.230,
     * 11.155,11.167,11.174,11.180,11.185,11.181,11.181,11.183,
     * 11.060,11.076,11.090,11.098,11.101,11.104,11.108,11.112,
     * 10.865,10.881,10.896,10.907,10.918,10.927,10.936,10.946,
     * 10.745,10.767,10.783,10.795,10.803,10.812,10.821,10.831,
     * 10.491,10.532,10.568,10.598,10.621,10.638,10.653,10.666,
     * 10.218,10.223,10.220,10.218,10.215,10.215,10.219,10.228,
     * 9.745,9.736,9.723,9.708,9.691,9.672,9.656,9.643,
     * 9.581,9.560,9.538,9.512,9.485,9.458,9.430,9.409,
     * 9.297,9.277,9.256,9.231,9.203,9.176,9.150,9.127/
C
      DATA ((EDEN(I,3,1,L,3),I=49,56),L=1,NZEN)/
     * 11.242,11.253,11.260,11.262,11.262,11.262,11.259,11.259,
     * 11.203,11.216,11.225,11.230,11.232,11.231,11.230,11.229,
     * 11.147,11.159,11.171,11.181,11.185,11.186,11.187,11.189,
     * 11.047,11.063,11.077,11.090,11.097,11.099,11.105,11.109,
     * 10.855,10.869,10.881,10.892,10.903,10.910,10.919,10.928,
     * 10.729,10.749,10.765,10.777,10.786,10.794,10.802,10.812,
     * 10.519,10.550,10.576,10.597,10.612,10.622,10.632,10.642,
     * 10.210,10.211,10.204,10.197,10.190,10.186,10.187,10.192,
     * 9.714,9.701,9.684,9.664,9.642,9.621,9.602,9.589,
     * 9.521,9.496,9.471,9.441,9.412,9.381,9.355,9.332,
     * 9.249,9.226,9.198,9.168,9.137,9.105,9.077,9.053/
C
      DATA ((EDEN(I,4,1,L,3),I=49,56),L=1,NZEN)/
     * 11.239,11.251,11.262,11.266,11.268,11.269,11.267,11.269,
     * 11.198,11.212,11.225,11.234,11.234,11.237,11.237,11.238,
     * 11.139,11.155,11.170,11.181,11.187,11.188,11.194,11.196,
     * 11.035,11.054,11.069,11.082,11.094,11.100,11.108,11.113,
     * 10.832,10.850,10.865,10.878,10.887,10.897,10.906,10.916,
     * 10.706,10.727,10.743,10.755,10.764,10.772,10.781,10.791,
     * 10.483,10.517,10.544,10.566,10.582,10.593,10.602,10.612,
     * 10.148,10.148,10.142,10.135,10.130,10.128,10.130,10.137,
     * 9.655,9.639,9.619,9.597,9.573,9.551,9.533,9.523,
     * 9.463,9.435,9.405,9.372,9.339,9.305,9.273,9.245,
     * 9.182,9.152,9.120,9.084,9.047,9.010,8.976,8.947/
C
      DATA ((EDEN(I,5,1,L,3),I=49,56),L=1,NZEN)/
     * 11.233,11.249,11.262,11.268,11.271,11.274,11.274,11.273,
     * 11.192,11.210,11.222,11.232,11.237,11.242,11.243,11.245,
     * 11.130,11.151,11.167,11.179,11.186,11.192,11.195,11.198,
     * 11.023,11.043,11.063,11.080,11.088,11.100,11.108,11.112,
     * 10.808,10.829,10.848,10.863,10.876,10.885,10.896,10.907,
     * 10.669,10.696,10.718,10.734,10.746,10.755,10.763,10.774,
     * 10.415,10.460,10.497,10.525,10.547,10.562,10.573,10.584,
     * 10.072,10.070,10.064,10.058,10.054,10.053,10.057,10.065,
     * 9.610,9.590,9.566,9.540,9.513,9.487,9.465,9.449,
     * 9.416,9.382,9.345,9.306,9.266,9.226,9.187,9.154,
     * 9.129,9.091,9.050,9.005,8.961,8.914,8.874,8.838/
C
      DATA ((EDEN(I,1,2,L,3),I=49,56),L=1,NZEN)/
     * 11.252,11.262,11.267,11.271,11.271,11.270,11.266,11.263,
     * 11.210,11.220,11.225,11.228,11.230,11.227,11.226,11.224,
     * 11.155,11.167,11.174,11.177,11.177,11.177,11.177,11.178,
     * 11.060,11.076,11.086,11.095,11.100,11.104,11.109,11.114,
     * 10.864,10.882,10.897,10.909,10.920,10.930,10.940,10.951,
     * 10.717,10.750,10.776,10.796,10.809,10.821,10.832,10.844,
     * 10.489,10.529,10.568,10.602,10.628,10.649,10.665,10.681,
     * 10.229,10.237,10.238,10.237,10.234,10.232,10.234,10.240,
     * 9.785,9.782,9.774,9.763,9.749,9.733,9.719,9.708,
     * 9.612,9.598,9.582,9.561,9.537,9.512,9.489,9.468,
     * 9.341,9.329,9.313,9.293,9.268,9.243,9.218,9.196/
C
      DATA ((EDEN(I,2,2,L,3),I=49,56),L=1,NZEN)/
     * 11.246,11.256,11.265,11.267,11.270,11.269,11.265,11.265,
     * 11.208,11.220,11.229,11.236,11.238,11.239,11.238,11.237,
     * 11.155,11.164,11.175,11.177,11.181,11.181,11.182,11.183,
     * 11.056,11.072,11.082,11.094,11.100,11.104,11.108,11.113,
     * 10.858,10.876,10.890,10.902,10.913,10.922,10.932,10.943,
     * 10.736,10.759,10.777,10.790,10.800,10.809,10.818,10.828,
     * 10.480,10.519,10.556,10.588,10.614,10.634,10.649,10.663,
     * 10.217,10.222,10.220,10.218,10.216,10.216,10.219,10.227,
     * 9.766,9.758,9.746,9.732,9.714,9.696,9.679,9.665,
     * 9.590,9.571,9.551,9.526,9.500,9.472,9.446,9.424,
     * 9.309,9.291,9.271,9.248,9.220,9.192,9.167,9.144/
C
      DATA ((EDEN(I,3,2,L,3),I=49,56),L=1,NZEN)/
     * 11.239,11.253,11.260,11.265,11.269,11.268,11.265,11.266,
     * 11.200,11.213,11.226,11.230,11.236,11.235,11.237,11.237,
     * 11.140,11.160,11.171,11.181,11.186,11.190,11.191,11.198,
     * 11.044,11.059,11.074,11.086,11.093,11.098,11.104,11.108,
     * 10.849,10.864,10.877,10.889,10.898,10.908,10.916,10.926,
     * 10.721,10.744,10.761,10.773,10.783,10.791,10.800,10.810,
     * 10.507,10.540,10.568,10.590,10.607,10.620,10.631,10.641,
     * 10.214,10.215,10.210,10.203,10.197,10.193,10.193,10.198,
     * 9.728,9.717,9.701,9.683,9.663,9.643,9.625,9.612,
     * 9.552,9.530,9.506,9.478,9.449,9.419,9.391,9.368,
     * 9.266,9.243,9.217,9.190,9.160,9.128,9.100,9.077/
C
      DATA ((EDEN(I,4,2,L,3),I=49,56),L=1,NZEN)/
     * 11.236,11.251,11.262,11.269,11.271,11.272,11.274,11.272,
     * 11.195,11.209,11.222,11.231,11.238,11.241,11.241,11.242,
     * 11.135,11.152,11.168,11.179,11.188,11.193,11.195,11.198,
     * 11.027,11.047,11.066,11.080,11.091,11.098,11.106,11.115,
     * 10.827,10.845,10.861,10.874,10.885,10.895,10.905,10.915,
     * 10.698,10.721,10.740,10.753,10.764,10.772,10.781,10.791,
     * 10.463,10.499,10.531,10.556,10.576,10.591,10.602,10.614,
     * 10.150,10.150,10.145,10.140,10.136,10.134,10.136,10.143,
     * 9.672,9.659,9.641,9.621,9.599,9.578,9.560,9.548,
     * 9.482,9.456,9.429,9.398,9.366,9.334,9.303,9.278,
     * 9.203,9.177,9.147,9.112,9.076,9.042,9.008,8.979/
C
      DATA ((EDEN(I,5,2,L,3),I=49,56),L=1,NZEN)/
     * 11.227,11.246,11.259,11.268,11.274,11.277,11.277,11.277,
     * 11.186,11.204,11.219,11.232,11.238,11.242,11.243,11.245,
     * 11.124,11.145,11.165,11.177,11.184,11.194,11.196,11.200,
     * 11.016,11.036,11.060,11.074,11.087,11.099,11.106,11.111,
     * 10.803,10.826,10.846,10.861,10.875,10.885,10.896,10.907,
     * 10.665,10.694,10.717,10.734,10.747,10.756,10.765,10.775,
     * 10.411,10.455,10.493,10.523,10.546,10.564,10.577,10.588,
     * 10.084,10.084,10.079,10.074,10.069,10.068,10.071,10.078,
     * 9.631,9.613,9.592,9.569,9.543,9.518,9.497,9.481,
     * 9.444,9.413,9.380,9.343,9.305,9.266,9.230,9.199,
     * 9.158,9.122,9.083,9.041,8.999,8.957,8.916,8.880/
C
      DATA ((EDEN(I,1,3,L,3),I=49,56),L=1,NZEN)/
     * 11.243,11.256,11.264,11.269,11.271,11.270,11.269,11.266,
     * 11.201,11.213,11.222,11.225,11.226,11.227,11.229,11.228,
     * 11.148,11.160,11.167,11.173,11.177,11.177,11.181,11.182,
     * 11.049,11.065,11.078,11.088,11.095,11.100,11.106,11.111,
     * 10.854,10.873,10.888,10.901,10.911,10.922,10.932,10.943,
     * 10.702,10.738,10.766,10.787,10.801,10.814,10.825,10.836,
     * 10.473,10.511,10.551,10.587,10.616,10.639,10.657,10.673,
     * 10.225,10.233,10.235,10.234,10.232,10.229,10.230,10.235,
     * 9.788,9.786,9.778,9.768,9.755,9.740,9.726,9.715,
     * 9.619,9.606,9.591,9.571,9.547,9.522,9.499,9.476,
     * 9.350,9.337,9.322,9.301,9.277,9.251,9.226,9.205/
C
      DATA ((EDEN(I,2,3,L,3),I=49,56),L=1,NZEN)/
     * 11.240,11.253,11.262,11.270,11.270,11.269,11.268,11.269,
     * 11.198,11.211,11.219,11.226,11.228,11.229,11.227,11.230,
     * 11.145,11.157,11.168,11.174,11.178,11.178,11.183,11.184,
     * 11.045,11.061,11.075,11.085,11.093,11.099,11.104,11.110,
     * 10.848,10.867,10.883,10.895,10.905,10.915,10.926,10.936,
     * 10.700,10.734,10.760,10.779,10.794,10.805,10.815,10.827,
     * 10.471,10.508,10.547,10.580,10.607,10.629,10.645,10.661,
     * 10.220,10.226,10.226,10.225,10.223,10.223,10.226,10.233,
     * 9.773,9.769,9.759,9.746,9.731,9.715,9.699,9.687,
     * 9.602,9.586,9.568,9.545,9.521,9.495,9.470,9.448,
     * 9.327,9.311,9.294,9.271,9.246,9.220,9.193,9.172/
C
      DATA ((EDEN(I,3,3,L,3),I=49,56),L=1,NZEN)/
     * 11.233,11.247,11.257,11.265,11.269,11.268,11.269,11.270,
     * 11.191,11.208,11.220,11.231,11.233,11.239,11.238,11.241,
     * 11.134,11.150,11.165,11.175,11.180,11.185,11.186,11.192,
     * 11.034,11.051,11.066,11.077,11.086,11.093,11.100,11.106,
     * 10.837,10.855,10.870,10.883,10.893,10.903,10.913,10.924,
     * 10.714,10.738,10.757,10.771,10.780,10.789,10.797,10.808,
     * 10.474,10.508,10.543,10.573,10.595,10.614,10.629,10.643,
     * 10.212,10.216,10.214,10.211,10.207,10.205,10.206,10.212,
     * 9.747,9.740,9.728,9.714,9.698,9.680,9.664,9.653,
     * 9.569,9.551,9.532,9.507,9.482,9.456,9.431,9.408,
     * 9.290,9.271,9.251,9.227,9.200,9.172,9.146,9.123/
C
      DATA ((EDEN(I,4,3,L,3),I=49,56),L=1,NZEN)/
     * 11.225,11.243,11.254,11.263,11.268,11.272,11.274,11.276,
     * 11.183,11.203,11.217,11.226,11.233,11.239,11.239,11.243,
     * 11.123,11.143,11.159,11.174,11.183,11.188,11.194,11.200,
     * 11.020,11.040,11.058,11.073,11.084,11.093,11.101,11.110,
     * 10.821,10.840,10.856,10.869,10.880,10.891,10.901,10.912,
     * 10.689,10.715,10.736,10.751,10.763,10.773,10.782,10.792,
     * 10.458,10.495,10.529,10.558,10.578,10.595,10.608,10.621,
     * 10.165,10.169,10.166,10.162,10.158,10.156,10.158,10.164,
     * 9.698,9.689,9.675,9.659,9.641,9.622,9.606,9.595,
     * 9.519,9.499,9.477,9.451,9.423,9.394,9.366,9.343,
     * 9.234,9.211,9.187,9.158,9.129,9.098,9.070,9.042/
C
      DATA ((EDEN(I,5,3,L,3),I=49,56),L=1,NZEN)/
     * 11.216,11.235,11.251,11.259,11.266,11.271,11.274,11.277,
     * 11.175,11.193,11.211,11.221,11.230,11.237,11.238,11.243,
     * 11.115,11.136,11.153,11.169,11.180,11.186,11.193,11.200,
     * 11.007,11.029,11.048,11.067,11.080,11.092,11.100,11.110,
     * 10.802,10.824,10.843,10.859,10.872,10.883,10.894,10.905,
     * 10.665,10.695,10.719,10.737,10.749,10.760,10.769,10.780,
     * 10.411,10.456,10.494,10.526,10.552,10.570,10.586,10.600,
     * 10.108,10.111,10.109,10.106,10.103,10.102,10.105,10.112,
     * 9.663,9.651,9.635,9.616,9.595,9.574,9.554,9.539,
     * 9.482,9.458,9.432,9.402,9.370,9.335,9.304,9.275,
     * 9.185,9.155,9.125,9.090,9.055,9.017,8.984,8.953/
C
      DATA ((EDEN(I,1,4,L,3),I=49,56),L=1,NZEN)/
     * 11.234,11.247,11.258,11.262,11.265,11.264,11.263,11.263,
     * 11.194,11.207,11.215,11.218,11.223,11.220,11.222,11.221,
     * 11.140,11.153,11.160,11.165,11.169,11.173,11.173,11.174,
     * 11.042,11.058,11.070,11.080,11.088,11.094,11.098,11.104,
     * 10.848,10.866,10.882,10.894,10.905,10.915,10.925,10.936,
     * 10.698,10.733,10.761,10.782,10.797,10.808,10.819,10.830,
     * 10.472,10.510,10.550,10.584,10.612,10.635,10.652,10.668,
     * 10.224,10.232,10.234,10.233,10.231,10.228,10.228,10.233,
     * 9.783,9.781,9.774,9.764,9.751,9.736,9.724,9.714,
     * 9.613,9.601,9.587,9.567,9.543,9.519,9.495,9.474,
     * 9.344,9.332,9.317,9.298,9.273,9.249,9.224,9.202/
C
      DATA ((EDEN(I,2,4,L,3),I=49,56),L=1,NZEN)/
     * 11.234,11.248,11.256,11.264,11.267,11.266,11.265,11.265,
     * 11.192,11.205,11.213,11.220,11.222,11.223,11.224,11.224,
     * 11.138,11.150,11.158,11.167,11.168,11.172,11.176,11.177,
     * 11.037,11.053,11.066,11.076,11.084,11.090,11.097,11.103,
     * 10.832,10.856,10.875,10.890,10.902,10.913,10.925,10.937,
     * 10.701,10.734,10.759,10.778,10.792,10.804,10.814,10.825,
     * 10.479,10.517,10.552,10.585,10.611,10.632,10.649,10.664,
     * 10.225,10.233,10.235,10.234,10.231,10.229,10.229,10.234,
     * 9.775,9.773,9.767,9.757,9.745,9.731,9.719,9.711,
     * 9.602,9.591,9.577,9.558,9.536,9.513,9.490,9.470,
     * 9.336,9.325,9.309,9.290,9.267,9.243,9.220,9.197/
C
      DATA ((EDEN(I,3,4,L,3),I=49,56),L=1,NZEN)/
     * 11.224,11.241,11.251,11.259,11.262,11.265,11.265,11.266,
     * 11.185,11.202,11.214,11.222,11.228,11.233,11.236,11.235,
     * 11.128,11.141,11.152,11.162,11.167,11.172,11.176,11.178,
     * 11.026,11.042,11.057,11.068,11.077,11.084,11.093,11.100,
     * 10.835,10.852,10.867,10.879,10.889,10.899,10.910,10.921,
     * 10.701,10.730,10.752,10.769,10.782,10.793,10.802,10.813,
     * 10.486,10.521,10.553,10.583,10.606,10.625,10.640,10.653,
     * 10.226,10.233,10.234,10.233,10.230,10.228,10.228,10.232,
     * 9.756,9.755,9.748,9.738,9.726,9.714,9.703,9.696,
     * 9.578,9.567,9.554,9.536,9.516,9.493,9.471,9.452,
     * 9.302,9.290,9.276,9.257,9.236,9.213,9.191,9.169/
C
      DATA ((EDEN(I,4,4,L,3),I=49,56),L=1,NZEN)/
     * 11.213,11.231,11.245,11.254,11.259,11.260,11.264,11.266,
     * 11.174,11.191,11.205,11.217,11.224,11.231,11.234,11.238,
     * 11.113,11.134,11.151,11.162,11.172,11.177,11.182,11.189,
     * 11.012,11.031,11.047,11.061,11.073,11.082,11.091,11.101,
     * 10.818,10.838,10.854,10.868,10.879,10.889,10.901,10.913,
     * 10.687,10.715,10.738,10.756,10.769,10.780,10.789,10.800,
     * 10.462,10.500,10.535,10.564,10.588,10.607,10.623,10.636,
     * 10.186,10.194,10.196,10.196,10.195,10.195,10.197,10.203,
     * 9.723,9.721,9.714,9.705,9.692,9.679,9.667,9.659,
     * 9.549,9.538,9.523,9.504,9.483,9.459,9.437,9.415,
     * 9.256,9.244,9.228,9.208,9.186,9.162,9.138,9.116/
C
      DATA ((EDEN(I,5,4,L,3),I=49,56),L=1,NZEN)/
     * 11.202,11.217,11.233,11.244,11.250,11.255,11.261,11.263,
     * 11.160,11.178,11.197,11.207,11.215,11.223,11.231,11.235,
     * 11.103,11.121,11.139,11.155,11.166,11.176,11.183,11.190,
     * 11.000,11.020,11.040,11.057,11.070,11.082,11.093,11.103,
     * 10.803,10.825,10.843,10.858,10.872,10.884,10.896,10.907,
     * 10.671,10.701,10.725,10.743,10.759,10.769,10.780,10.791,
     * 10.434,10.476,10.514,10.546,10.571,10.592,10.608,10.622,
     * 10.143,10.151,10.154,10.155,10.156,10.157,10.161,10.168,
     * 9.693,9.690,9.682,9.672,9.658,9.642,9.628,9.617,
     * 9.514,9.500,9.484,9.463,9.440,9.415,9.390,9.365,
     * 9.218,9.203,9.184,9.162,9.136,9.111,9.084,9.057/
C
      DATA ((EDEN(I,1,5,L,3),I=49,56),L=1,NZEN)/
     * 11.234,11.244,11.249,11.253,11.255,11.254,11.249,11.249,
     * 11.191,11.204,11.208,11.211,11.213,11.210,11.211,11.209,
     * 11.137,11.149,11.156,11.161,11.162,11.161,11.165,11.166,
     * 11.043,11.058,11.068,11.076,11.083,11.088,11.092,11.097,
     * 10.851,10.868,10.882,10.895,10.905,10.914,10.925,10.935,
     * 10.712,10.743,10.766,10.783,10.797,10.807,10.818,10.829,
     * 10.492,10.530,10.565,10.596,10.621,10.640,10.656,10.669,
     * 10.225,10.233,10.234,10.233,10.231,10.229,10.230,10.236,
     * 9.769,9.768,9.761,9.750,9.738,9.724,9.712,9.704,
     * 9.593,9.581,9.567,9.547,9.525,9.501,9.478,9.459,
     * 9.324,9.313,9.298,9.279,9.255,9.233,9.209,9.189/
C
      DATA ((EDEN(I,2,5,L,3),I=49,56),L=1,NZEN)/
     * 11.231,11.245,11.253,11.257,11.257,11.256,11.255,11.255,
     * 11.189,11.195,11.200,11.203,11.205,11.201,11.203,11.205,
     * 11.135,11.144,11.151,11.157,11.161,11.161,11.165,11.166,
     * 11.035,11.051,11.065,11.076,11.085,11.091,11.098,11.106,
     * 10.839,10.860,10.877,10.892,10.903,10.915,10.927,10.939,
     * 10.717,10.745,10.767,10.783,10.796,10.807,10.818,10.829,
     * 10.506,10.541,10.575,10.603,10.626,10.644,10.659,10.672,
     * 10.236,10.245,10.247,10.248,10.246,10.245,10.247,10.253,
     * 9.768,9.769,9.765,9.758,9.748,9.738,9.731,9.729,
     * 9.589,9.581,9.571,9.555,9.536,9.515,9.495,9.476,
     * 9.317,9.309,9.298,9.281,9.262,9.241,9.219,9.201/
C
      DATA ((EDEN(I,3,5,L,3),I=49,56),L=1,NZEN)/
     * 11.221,11.235,11.245,11.250,11.253,11.255,11.255,11.256,
     * 11.182,11.192,11.205,11.209,11.215,11.213,11.215,11.218,
     * 11.121,11.134,11.146,11.153,11.158,11.162,11.163,11.168,
     * 11.023,11.039,11.054,11.066,11.075,11.083,11.092,11.100,
     * 10.833,10.852,10.869,10.883,10.895,10.906,10.917,10.929,
     * 10.717,10.742,10.762,10.778,10.790,10.800,10.811,10.822,
     * 10.485,10.522,10.557,10.588,10.615,10.636,10.653,10.668,
     * 10.237,10.247,10.252,10.254,10.255,10.257,10.260,10.266,
     * 9.759,9.763,9.761,9.756,9.749,9.742,9.738,9.738,
     * 9.594,9.588,9.580,9.567,9.550,9.530,9.511,9.492,
     * 9.295,9.289,9.280,9.266,9.250,9.233,9.215,9.196/
C
      DATA ((EDEN(I,4,5,L,3),I=49,56),L=1,NZEN)/
     * 11.207,11.222,11.233,11.242,11.246,11.250,11.250,11.255,
     * 11.168,11.183,11.196,11.208,11.215,11.218,11.225,11.225,
     * 11.111,11.129,11.142,11.153,11.160,11.165,11.170,11.177,
     * 11.009,11.026,11.043,11.057,11.069,11.078,11.088,11.097,
     * 10.820,10.841,10.858,10.872,10.885,10.897,10.908,10.921,
     * 10.696,10.725,10.748,10.767,10.782,10.793,10.804,10.816,
     * 10.483,10.520,10.555,10.585,10.609,10.629,10.645,10.660,
     * 10.210,10.222,10.228,10.232,10.235,10.239,10.244,10.252,
     * 9.738,9.743,9.742,9.739,9.732,9.724,9.718,9.716,
     * 9.561,9.558,9.551,9.539,9.524,9.507,9.488,9.469,
     * 9.258,9.255,9.247,9.234,9.219,9.202,9.184,9.165/
C
      DATA ((EDEN(I,5,5,L,3),I=49,56),L=1,NZEN)/
     * 11.189,11.205,11.218,11.228,11.234,11.239,11.244,11.250,
     * 11.151,11.166,11.185,11.195,11.203,11.211,11.219,11.223,
     * 11.094,11.113,11.131,11.148,11.159,11.165,11.176,11.183,
     * 10.997,11.016,11.036,11.051,11.065,11.077,11.088,11.099,
     * 10.808,10.830,10.849,10.865,10.879,10.892,10.905,10.917,
     * 10.688,10.716,10.740,10.759,10.774,10.786,10.799,10.810,
     * 10.469,10.508,10.546,10.576,10.601,10.622,10.638,10.653,
     * 10.179,10.192,10.200,10.206,10.211,10.217,10.224,10.232,
     * 9.711,9.717,9.717,9.714,9.708,9.700,9.693,9.689,
     * 9.530,9.527,9.521,9.509,9.495,9.477,9.458,9.439,
     * 9.243,9.240,9.234,9.222,9.208,9.191,9.171,9.151/
C
      DATA ((EDEN(I,1,6,L,3),I=49,56),L=1,NZEN)/
     * 11.231,11.241,11.243,11.243,11.242,11.240,11.235,11.234,
     * 11.191,11.197,11.202,11.201,11.202,11.198,11.196,11.193,
     * 11.140,11.149,11.152,11.154,11.154,11.153,11.153,11.151,
     * 11.049,11.061,11.070,11.076,11.080,11.083,11.086,11.089,
     * 10.857,10.874,10.887,10.899,10.909,10.918,10.927,10.938,
     * 10.727,10.753,10.774,10.788,10.801,10.811,10.821,10.832,
     * 10.515,10.551,10.584,10.610,10.630,10.645,10.659,10.672,
     * 10.228,10.235,10.236,10.234,10.231,10.230,10.233,10.241,
     * 9.755,9.753,9.746,9.735,9.722,9.709,9.698,9.693,
     * 9.571,9.559,9.545,9.526,9.504,9.481,9.461,9.440,
     * 9.304,9.290,9.276,9.259,9.238,9.215,9.192,9.173/
C
      DATA ((EDEN(I,2,6,L,3),I=49,56),L=1,NZEN)/
     * 11.234,11.241,11.247,11.248,11.247,11.246,11.241,11.241,
     * 11.186,11.192,11.193,11.193,11.194,11.191,11.188,11.189,
     * 11.135,11.144,11.148,11.150,11.154,11.153,11.153,11.154,
     * 11.040,11.056,11.067,11.077,11.084,11.089,11.094,11.101,
     * 10.846,10.866,10.883,10.897,10.910,10.921,10.933,10.945,
     * 10.732,10.756,10.776,10.791,10.802,10.814,10.824,10.837,
     * 10.487,10.525,10.563,10.598,10.624,10.647,10.665,10.681,
     * 10.235,10.246,10.250,10.253,10.255,10.257,10.262,10.271,
     * 9.761,9.764,9.763,9.758,9.752,9.746,9.743,9.746,
     * 9.600,9.594,9.585,9.570,9.551,9.530,9.510,9.491,
     * 9.306,9.301,9.291,9.275,9.258,9.239,9.222,9.204/
C
      DATA ((EDEN(I,3,6,L,3),I=49,56),L=1,NZEN)/
     * 11.221,11.232,11.241,11.243,11.246,11.245,11.241,11.242,
     * 11.176,11.186,11.192,11.196,11.194,11.195,11.197,11.196,
     * 11.122,11.135,11.144,11.146,11.151,11.156,11.156,11.162,
     * 11.027,11.043,11.057,11.067,11.076,11.083,11.090,11.097,
     * 10.840,10.859,10.875,10.890,10.903,10.914,10.926,10.938,
     * 10.715,10.744,10.768,10.786,10.801,10.813,10.825,10.837,
     * 10.514,10.549,10.583,10.611,10.634,10.653,10.669,10.684,
     * 10.250,10.263,10.269,10.273,10.276,10.280,10.285,10.293,
     * 9.767,9.774,9.775,9.773,9.769,9.766,9.765,9.769,
     * 9.588,9.587,9.582,9.572,9.559,9.542,9.525,9.507,
     * 9.271,9.268,9.262,9.251,9.240,9.224,9.210,9.196/
C
      DATA ((EDEN(I,4,6,L,3),I=49,56),L=1,NZEN)/
     * 11.204,11.215,11.226,11.232,11.236,11.240,11.240,11.241,
     * 11.165,11.180,11.190,11.199,11.206,11.209,11.212,11.211,
     * 11.108,11.126,11.136,11.148,11.154,11.159,11.165,11.167,
     * 11.012,11.030,11.046,11.060,11.071,11.081,11.089,11.098,
     * 10.825,10.846,10.865,10.882,10.896,10.910,10.923,10.936,
     * 10.712,10.739,10.761,10.780,10.794,10.807,10.819,10.831,
     * 10.501,10.540,10.575,10.606,10.630,10.651,10.667,10.682,
     * 10.226,10.242,10.251,10.258,10.265,10.271,10.278,10.286,
     * 9.744,9.754,9.758,9.759,9.757,9.754,9.754,9.759,
     * 9.563,9.565,9.566,9.558,9.547,9.533,9.518,9.503,
     * 9.267,9.269,9.268,9.262,9.253,9.241,9.226,9.210/
C
      DATA ((EDEN(I,5,6,L,3),I=49,56),L=1,NZEN)/
     * 11.180,11.196,11.208,11.219,11.224,11.229,11.234,11.239,
     * 11.145,11.160,11.176,11.189,11.198,11.205,11.209,11.214,
     * 11.091,11.111,11.129,11.143,11.154,11.161,11.167,11.175,
     * 10.997,11.018,11.038,11.054,11.067,11.080,11.091,11.101,
     * 10.814,10.836,10.857,10.875,10.892,10.906,10.920,10.933,
     * 10.703,10.730,10.754,10.774,10.791,10.805,10.818,10.831,
     * 10.499,10.539,10.575,10.605,10.630,10.650,10.667,10.680,
     * 10.204,10.222,10.234,10.245,10.253,10.262,10.271,10.280,
     * 9.720,9.733,9.740,9.743,9.742,9.741,9.743,9.749,
     * 9.531,9.537,9.537,9.534,9.525,9.512,9.495,9.478,
     * 9.259,9.265,9.270,9.266,9.259,9.248,9.233,9.215/
C
      DATA ((EDEN(I,1,7,L,3),I=49,56),L=1,NZEN)/
     * 11.231,11.238,11.243,11.243,11.239,11.237,11.231,11.230,
     * 11.191,11.197,11.202,11.201,11.198,11.195,11.192,11.189,
     * 11.140,11.149,11.152,11.150,11.150,11.149,11.148,11.149,
     * 11.052,11.063,11.071,11.077,11.080,11.083,11.085,11.089,
     * 10.859,10.876,10.889,10.901,10.911,10.920,10.929,10.939,
     * 10.731,10.755,10.775,10.790,10.802,10.812,10.823,10.835,
     * 10.521,10.556,10.588,10.613,10.632,10.648,10.660,10.673,
     * 10.229,10.236,10.237,10.235,10.233,10.232,10.235,10.243,
     * 9.752,9.750,9.743,9.732,9.720,9.707,9.697,9.691,
     * 9.566,9.555,9.541,9.522,9.500,9.479,9.457,9.437,
     * 9.299,9.287,9.273,9.255,9.232,9.210,9.190,9.170/
C
      DATA ((EDEN(I,2,7,L,3),I=49,56),L=1,NZEN)/
     * 11.234,11.241,11.247,11.248,11.244,11.242,11.237,11.237,
     * 11.186,11.192,11.193,11.193,11.190,11.187,11.188,11.186,
     * 11.135,11.144,11.148,11.150,11.150,11.150,11.154,11.153,
     * 11.043,11.058,11.070,11.078,11.085,11.091,11.096,11.101,
     * 10.849,10.869,10.885,10.900,10.913,10.925,10.936,10.948,
     * 10.735,10.758,10.778,10.793,10.806,10.817,10.828,10.840,
     * 10.492,10.531,10.569,10.602,10.629,10.650,10.668,10.684,
     * 10.237,10.248,10.253,10.257,10.259,10.262,10.267,10.275,
     * 9.760,9.764,9.763,9.759,9.754,9.749,9.748,9.751,
     * 9.597,9.592,9.583,9.569,9.550,9.532,9.512,9.493,
     * 9.303,9.297,9.289,9.275,9.259,9.242,9.223,9.207/
C
      DATA ((EDEN(I,3,7,L,3),I=49,56),L=1,NZEN)/
     * 11.224,11.235,11.241,11.243,11.243,11.242,11.241,11.238,
     * 11.179,11.186,11.192,11.196,11.194,11.196,11.193,11.196,
     * 11.126,11.136,11.144,11.147,11.152,11.156,11.157,11.158,
     * 11.030,11.047,11.059,11.071,11.080,11.086,11.092,11.100,
     * 10.843,10.863,10.879,10.894,10.908,10.920,10.931,10.944,
     * 10.719,10.747,10.771,10.790,10.806,10.818,10.830,10.843,
     * 10.518,10.554,10.587,10.616,10.639,10.659,10.675,10.689,
     * 10.253,10.266,10.273,10.279,10.282,10.286,10.292,10.300,
     * 9.768,9.776,9.778,9.778,9.775,9.773,9.774,9.780,
     * 9.589,9.588,9.585,9.576,9.562,9.548,9.532,9.515,
     * 9.272,9.270,9.265,9.255,9.244,9.231,9.217,9.203/
C
      DATA ((EDEN(I,4,7,L,3),I=49,56),L=1,NZEN)/
     * 11.204,11.219,11.229,11.235,11.240,11.240,11.240,11.241,
     * 11.169,11.183,11.194,11.203,11.206,11.209,11.212,11.212,
     * 11.113,11.127,11.140,11.148,11.155,11.160,11.166,11.172,
     * 11.017,11.034,11.051,11.065,11.076,11.085,11.094,11.102,
     * 10.828,10.851,10.870,10.887,10.903,10.917,10.931,10.944,
     * 10.715,10.743,10.766,10.785,10.801,10.814,10.826,10.839,
     * 10.503,10.543,10.579,10.610,10.635,10.656,10.674,10.689,
     * 10.228,10.245,10.255,10.264,10.271,10.278,10.286,10.295,
     * 9.747,9.758,9.764,9.766,9.766,9.764,9.766,9.772,
     * 9.566,9.570,9.572,9.566,9.557,9.543,9.528,9.513,
     * 9.272,9.275,9.276,9.271,9.263,9.253,9.239,9.223/
C
      DATA ((EDEN(I,5,7,L,3),I=49,56),L=1,NZEN)/
     * 11.183,11.199,11.211,11.222,11.228,11.233,11.237,11.239,
     * 11.148,11.164,11.179,11.193,11.202,11.209,11.213,11.218,
     * 11.096,11.115,11.134,11.147,11.159,11.166,11.172,11.180,
     * 11.001,11.023,11.043,11.061,11.074,11.087,11.097,11.106,
     * 10.818,10.841,10.863,10.883,10.899,10.915,10.929,10.943,
     * 10.705,10.735,10.760,10.781,10.799,10.814,10.828,10.841,
     * 10.500,10.542,10.579,10.611,10.637,10.658,10.675,10.690,
     * 10.206,10.225,10.239,10.251,10.261,10.271,10.281,10.291,
     * 9.725,9.740,9.749,9.753,9.755,9.756,9.760,9.767,
     * 9.538,9.545,9.548,9.547,9.538,9.526,9.511,9.493,
     * 9.266,9.276,9.281,9.281,9.276,9.265,9.253,9.236/
C
      DATA ((EDEN(I,1,8,L,3),I=49,56),L=1,NZEN)/
     * 11.234,11.244,11.249,11.250,11.249,11.247,11.242,11.241,
     * 11.194,11.200,11.205,11.208,11.205,11.202,11.203,11.201,
     * 11.140,11.149,11.156,11.158,11.158,11.157,11.157,11.157,
     * 11.048,11.061,11.071,11.077,11.083,11.087,11.090,11.095,
     * 10.856,10.872,10.886,10.898,10.908,10.918,10.928,10.938,
     * 10.720,10.749,10.770,10.787,10.800,10.811,10.821,10.833,
     * 10.505,10.541,10.575,10.603,10.625,10.643,10.657,10.670,
     * 10.228,10.236,10.237,10.235,10.233,10.231,10.233,10.239,
     * 9.762,9.761,9.754,9.744,9.732,9.719,9.708,9.702,
     * 9.582,9.571,9.556,9.538,9.516,9.495,9.473,9.453,
     * 9.314,9.302,9.288,9.270,9.247,9.225,9.203,9.182/
C
      DATA ((EDEN(I,2,8,L,3),I=49,56),L=1,NZEN)/
     * 11.234,11.245,11.250,11.251,11.250,11.249,11.248,11.244,
     * 11.189,11.195,11.200,11.200,11.198,11.198,11.195,11.197,
     * 11.139,11.148,11.155,11.158,11.158,11.161,11.162,11.163,
     * 11.041,11.057,11.070,11.081,11.088,11.094,11.100,11.107,
     * 10.844,10.865,10.883,10.898,10.911,10.923,10.935,10.948,
     * 10.725,10.752,10.773,10.790,10.803,10.814,10.825,10.837,
     * 10.518,10.554,10.585,10.613,10.634,10.652,10.666,10.680,
     * 10.246,10.256,10.259,10.260,10.259,10.259,10.261,10.267,
     * 9.775,9.778,9.775,9.769,9.761,9.754,9.749,9.750,
     * 9.611,9.604,9.595,9.580,9.560,9.540,9.519,9.500,
     * 9.312,9.306,9.297,9.283,9.266,9.248,9.229,9.210/
C
      DATA ((EDEN(I,3,8,L,3),I=49,56),L=1,NZEN)/
     * 11.224,11.235,11.245,11.250,11.250,11.249,11.248,11.249,
     * 11.183,11.193,11.199,11.203,11.205,11.203,11.205,11.207,
     * 11.129,11.139,11.147,11.154,11.159,11.163,11.164,11.170,
     * 11.030,11.046,11.061,11.074,11.082,11.090,11.097,11.105,
     * 10.839,10.859,10.877,10.892,10.906,10.918,10.930,10.942,
     * 10.725,10.750,10.771,10.788,10.800,10.812,10.824,10.835,
     * 10.498,10.534,10.570,10.601,10.628,10.649,10.667,10.682,
     * 10.245,10.258,10.264,10.268,10.271,10.274,10.278,10.286,
     * 9.773,9.779,9.780,9.778,9.774,9.769,9.767,9.770,
     * 9.597,9.595,9.591,9.579,9.565,9.548,9.531,9.512,
     * 9.284,9.282,9.276,9.266,9.252,9.237,9.224,9.208/
C
      DATA ((EDEN(I,4,8,L,3),I=49,56),L=1,NZEN)/
     * 11.210,11.225,11.236,11.245,11.249,11.250,11.250,11.252,
     * 11.172,11.186,11.200,11.209,11.213,11.219,11.219,11.222,
     * 11.116,11.134,11.147,11.159,11.165,11.171,11.176,11.183,
     * 11.017,11.036,11.053,11.068,11.080,11.091,11.100,11.108,
     * 10.828,10.850,10.869,10.885,10.900,10.913,10.926,10.939,
     * 10.704,10.734,10.760,10.780,10.797,10.810,10.823,10.836,
     * 10.491,10.531,10.568,10.600,10.625,10.647,10.664,10.679,
     * 10.217,10.232,10.241,10.247,10.252,10.257,10.264,10.273,
     * 9.746,9.755,9.759,9.759,9.756,9.751,9.749,9.751,
     * 9.570,9.571,9.568,9.560,9.550,9.533,9.517,9.500,
     * 9.271,9.272,9.269,9.262,9.250,9.236,9.220,9.204/
C
      DATA ((EDEN(I,5,8,L,3),I=49,56),L=1,NZEN)/
     * 11.193,11.211,11.224,11.235,11.241,11.246,11.251,11.253,
     * 11.155,11.174,11.192,11.203,11.215,11.219,11.223,11.228,
     * 11.102,11.122,11.140,11.157,11.169,11.176,11.183,11.190,
     * 11.003,11.026,11.047,11.064,11.080,11.091,11.102,11.113,
     * 10.815,10.840,10.863,10.882,10.899,10.914,10.929,10.942,
     * 10.696,10.728,10.755,10.776,10.793,10.809,10.822,10.835,
     * 10.474,10.517,10.559,10.592,10.621,10.644,10.663,10.679,
     * 10.185,10.203,10.215,10.225,10.234,10.242,10.251,10.262,
     * 9.727,9.738,9.744,9.745,9.743,9.740,9.738,9.738,
     * 9.548,9.552,9.551,9.545,9.533,9.518,9.501,9.482,
     * 9.264,9.269,9.269,9.264,9.254,9.240,9.225,9.204/
C
      DATA ((EDEN(I,1,9,L,3),I=49,56),L=1,NZEN)/
     * 11.234,11.247,11.252,11.259,11.258,11.260,11.259,11.259,
     * 11.194,11.204,11.212,11.215,11.216,11.217,11.218,11.217,
     * 11.137,11.149,11.160,11.161,11.165,11.169,11.169,11.170,
     * 11.042,11.057,11.068,11.077,11.085,11.091,11.096,11.102,
     * 10.849,10.866,10.881,10.893,10.904,10.913,10.924,10.935,
     * 10.705,10.737,10.762,10.782,10.796,10.807,10.817,10.829,
     * 10.482,10.520,10.556,10.589,10.615,10.635,10.652,10.668,
     * 10.227,10.236,10.237,10.236,10.234,10.231,10.232,10.236,
     * 9.778,9.777,9.771,9.761,9.749,9.736,9.724,9.717,
     * 9.606,9.595,9.580,9.562,9.539,9.517,9.494,9.474,
     * 9.336,9.326,9.313,9.293,9.271,9.247,9.224,9.202/
C
      DATA ((EDEN(I,2,9,L,3),I=49,56),L=1,NZEN)/
     * 11.231,11.245,11.253,11.257,11.260,11.259,11.258,11.258,
     * 11.192,11.198,11.207,11.210,11.212,11.209,11.210,11.213,
     * 11.135,11.147,11.158,11.164,11.169,11.168,11.173,11.178,
     * 11.036,11.054,11.068,11.080,11.089,11.097,11.105,11.113,
     * 10.835,10.858,10.878,10.892,10.905,10.917,10.929,10.942,
     * 10.707,10.738,10.763,10.782,10.797,10.808,10.819,10.831,
     * 10.488,10.525,10.561,10.593,10.618,10.639,10.656,10.671,
     * 10.234,10.243,10.246,10.246,10.245,10.243,10.244,10.249,
     * 9.777,9.778,9.774,9.767,9.757,9.746,9.738,9.733,
     * 9.603,9.595,9.584,9.567,9.548,9.526,9.506,9.487,
     * 9.321,9.313,9.301,9.287,9.267,9.246,9.227,9.207/
C
      DATA ((EDEN(I,3,9,L,3),I=49,56),L=1,NZEN)/
     * 11.224,11.238,11.248,11.256,11.259,11.259,11.259,11.259,
     * 11.186,11.199,11.208,11.215,11.218,11.220,11.222,11.225,
     * 11.128,11.142,11.153,11.164,11.169,11.173,11.174,11.180,
     * 11.027,11.044,11.060,11.073,11.083,11.092,11.100,11.109,
     * 10.831,10.853,10.871,10.886,10.899,10.911,10.922,10.935,
     * 10.708,10.737,10.760,10.778,10.792,10.803,10.813,10.825,
     * 10.498,10.533,10.566,10.594,10.619,10.638,10.654,10.668,
     * 10.240,10.250,10.253,10.254,10.253,10.253,10.254,10.259,
     * 9.771,9.773,9.770,9.764,9.755,9.746,9.739,9.737,
     * 9.604,9.598,9.588,9.574,9.556,9.535,9.514,9.495,
     * 9.310,9.304,9.295,9.282,9.263,9.245,9.225,9.207/
C
      DATA ((EDEN(I,4,9,L,3),I=49,56),L=1,NZEN)/
     * 11.213,11.231,11.245,11.254,11.259,11.260,11.264,11.266,
     * 11.175,11.192,11.206,11.218,11.225,11.232,11.235,11.236,
     * 11.118,11.136,11.149,11.164,11.170,11.179,11.181,11.188,
     * 11.017,11.036,11.053,11.068,11.080,11.090,11.099,11.107,
     * 10.821,10.843,10.862,10.878,10.892,10.904,10.917,10.929,
     * 10.696,10.726,10.750,10.769,10.784,10.795,10.807,10.819,
     * 10.456,10.498,10.538,10.572,10.601,10.623,10.642,10.658,
     * 10.192,10.203,10.209,10.214,10.217,10.220,10.225,10.233,
     * 9.738,9.741,9.740,9.735,9.728,9.719,9.711,9.707,
     * 9.563,9.557,9.549,9.535,9.518,9.498,9.479,9.458,
     * 9.259,9.252,9.243,9.230,9.211,9.193,9.174,9.153/
C
      DATA ((EDEN(I,5,9,L,3),I=49,56),L=1,NZEN)/
     * 11.205,11.220,11.236,11.247,11.256,11.262,11.264,11.267,
     * 11.164,11.182,11.201,11.215,11.223,11.231,11.236,11.241,
     * 11.104,11.127,11.145,11.162,11.173,11.183,11.190,11.198,
     * 11.005,11.028,11.049,11.066,11.080,11.093,11.104,11.114,
     * 10.809,10.834,10.855,10.873,10.889,10.902,10.916,10.929,
     * 10.682,10.714,10.741,10.762,10.778,10.792,10.804,10.816,
     * 10.438,10.484,10.527,10.564,10.592,10.616,10.635,10.650,
     * 10.150,10.164,10.172,10.178,10.183,10.188,10.196,10.205,
     * 9.712,9.715,9.715,9.710,9.701,9.691,9.681,9.672,
     * 9.538,9.532,9.523,9.508,9.491,9.469,9.446,9.425,
     * 9.246,9.239,9.230,9.216,9.196,9.175,9.155,9.131/
C
      DATA ((EDEN(I,1,10,L,3),I=49,56),L=1,NZEN)/
     * 11.237,11.250,11.258,11.262,11.265,11.267,11.266,11.266,
     * 11.194,11.207,11.215,11.221,11.223,11.224,11.226,11.224,
     * 11.140,11.153,11.160,11.165,11.169,11.173,11.177,11.178,
     * 11.042,11.057,11.069,11.080,11.087,11.093,11.099,11.105,
     * 10.850,10.867,10.882,10.894,10.904,10.913,10.924,10.935,
     * 10.699,10.734,10.762,10.782,10.797,10.808,10.819,10.830,
     * 10.474,10.511,10.550,10.585,10.612,10.635,10.653,10.669,
     * 10.229,10.237,10.239,10.239,10.236,10.233,10.233,10.237,
     * 9.789,9.787,9.781,9.771,9.758,9.744,9.732,9.723,
     * 9.620,9.608,9.593,9.575,9.552,9.528,9.505,9.484,
     * 9.350,9.340,9.324,9.304,9.282,9.258,9.234,9.211/
C
      DATA ((EDEN(I,2,10,L,3),I=49,56),L=1,NZEN)/
     * 11.234,11.245,11.256,11.261,11.263,11.266,11.265,11.265,
     * 11.192,11.205,11.213,11.220,11.222,11.223,11.224,11.227,
     * 11.138,11.150,11.161,11.167,11.172,11.175,11.176,11.181,
     * 11.040,11.055,11.068,11.078,11.087,11.094,11.099,11.107,
     * 10.846,10.864,10.879,10.892,10.902,10.912,10.922,10.933,
     * 10.698,10.732,10.759,10.779,10.793,10.805,10.815,10.826,
     * 10.474,10.510,10.549,10.582,10.610,10.631,10.649,10.665,
     * 10.228,10.235,10.237,10.237,10.236,10.236,10.239,10.246,
     * 9.781,9.779,9.772,9.762,9.749,9.735,9.722,9.713,
     * 9.610,9.598,9.583,9.564,9.541,9.518,9.495,9.473,
     * 9.335,9.323,9.309,9.291,9.267,9.244,9.220,9.198/
C
      DATA ((EDEN(I,3,10,L,3),I=49,56),L=1,NZEN)/
     * 11.227,11.241,11.251,11.259,11.262,11.262,11.265,11.266,
     * 11.185,11.201,11.214,11.221,11.227,11.233,11.235,11.239,
     * 11.131,11.144,11.156,11.166,11.171,11.175,11.180,11.182,
     * 11.031,11.046,11.061,11.073,11.083,11.090,11.097,11.105,
     * 10.837,10.855,10.870,10.883,10.894,10.903,10.914,10.924,
     * 10.699,10.729,10.753,10.771,10.784,10.795,10.805,10.816,
     * 10.480,10.516,10.550,10.580,10.605,10.624,10.640,10.654,
     * 10.225,10.231,10.232,10.231,10.229,10.229,10.231,10.236,
     * 9.760,9.757,9.749,9.739,9.726,9.712,9.700,9.692,
     * 9.583,9.570,9.557,9.536,9.514,9.491,9.469,9.447,
     * 9.296,9.283,9.269,9.251,9.229,9.205,9.184,9.162/
C
      DATA ((EDEN(I,4,10,L,3),I=49,56),L=1,NZEN)/
     * 11.219,11.237,11.248,11.257,11.265,11.266,11.270,11.272,
     * 11.177,11.198,11.211,11.220,11.230,11.233,11.237,11.241,
     * 11.120,11.137,11.154,11.169,11.178,11.184,11.193,11.196,
     * 11.018,11.037,11.054,11.068,11.080,11.090,11.098,11.107,
     * 10.821,10.841,10.858,10.872,10.884,10.894,10.906,10.918,
     * 10.695,10.723,10.743,10.759,10.771,10.782,10.792,10.802,
     * 10.454,10.494,10.532,10.562,10.587,10.607,10.623,10.637,
     * 10.172,10.179,10.181,10.181,10.181,10.182,10.187,10.194,
     * 9.715,9.712,9.704,9.693,9.680,9.665,9.653,9.645,
     * 9.535,9.521,9.505,9.485,9.462,9.438,9.414,9.391,
     * 9.254,9.240,9.221,9.200,9.176,9.151,9.127,9.103/
C
      DATA ((EDEN(I,5,10,L,3),I=49,56),L=1,NZEN)/
     * 11.213,11.229,11.245,11.256,11.262,11.268,11.271,11.277,
     * 11.172,11.191,11.206,11.219,11.228,11.236,11.240,11.245,
     * 11.110,11.131,11.153,11.165,11.176,11.186,11.193,11.200,
     * 11.006,11.028,11.049,11.066,11.080,11.092,11.102,11.113,
     * 10.807,10.830,10.849,10.865,10.878,10.890,10.902,10.914,
     * 10.674,10.705,10.729,10.748,10.762,10.773,10.783,10.795,
     * 10.427,10.472,10.513,10.546,10.573,10.593,10.608,10.622,
     * 10.120,10.128,10.130,10.132,10.133,10.136,10.141,10.150,
     * 9.682,9.677,9.668,9.655,9.640,9.623,9.608,9.596,
     * 9.502,9.484,9.466,9.442,9.417,9.389,9.361,9.337,
     * 9.212,9.193,9.171,9.146,9.118,9.088,9.059,9.029/
C
      DATA ((EDEN(I,1,11,L,3),I=49,56),L=1,NZEN)/
     * 11.243,11.256,11.261,11.265,11.268,11.267,11.266,11.266,
     * 11.201,11.213,11.218,11.225,11.226,11.224,11.226,11.224,
     * 11.148,11.160,11.167,11.173,11.173,11.177,11.177,11.178,
     * 11.049,11.064,11.077,11.085,11.092,11.098,11.102,11.108,
     * 10.857,10.875,10.889,10.901,10.911,10.920,10.930,10.941,
     * 10.712,10.745,10.770,10.790,10.804,10.815,10.825,10.836,
     * 10.486,10.525,10.563,10.596,10.623,10.644,10.660,10.675,
     * 10.232,10.240,10.241,10.240,10.238,10.235,10.236,10.241,
     * 9.788,9.785,9.778,9.768,9.754,9.740,9.727,9.717,
     * 9.616,9.603,9.588,9.568,9.545,9.522,9.498,9.477,
     * 9.347,9.335,9.319,9.299,9.275,9.251,9.228,9.205/
C
      DATA ((EDEN(I,2,11,L,3),I=49,56),L=1,NZEN)/
     * 11.237,11.250,11.256,11.264,11.263,11.262,11.262,11.262,
     * 11.198,11.211,11.222,11.229,11.231,11.233,11.234,11.238,
     * 11.145,11.157,11.168,11.170,11.174,11.178,11.178,11.180,
     * 11.048,11.063,11.075,11.085,11.092,11.098,11.103,11.108,
     * 10.852,10.871,10.885,10.897,10.906,10.916,10.925,10.936,
     * 10.706,10.739,10.764,10.783,10.796,10.806,10.816,10.827,
     * 10.476,10.516,10.554,10.586,10.611,10.633,10.648,10.664,
     * 10.223,10.229,10.228,10.226,10.225,10.224,10.227,10.234,
     * 9.773,9.768,9.757,9.745,9.729,9.712,9.696,9.683,
     * 9.599,9.583,9.566,9.542,9.516,9.491,9.466,9.444,
     * 9.319,9.302,9.285,9.263,9.238,9.212,9.188,9.164/
C
      DATA ((EDEN(I,3,11,L,3),I=49,56),L=1,NZEN)/
     * 11.230,11.244,11.254,11.259,11.262,11.262,11.262,11.263,
     * 11.191,11.204,11.216,11.224,11.226,11.232,11.231,11.234,
     * 11.134,11.150,11.161,11.171,11.180,11.184,11.189,11.191,
     * 11.036,11.053,11.066,11.077,11.085,11.092,11.098,11.104,
     * 10.841,10.859,10.873,10.885,10.895,10.904,10.914,10.924,
     * 10.720,10.743,10.760,10.773,10.782,10.790,10.798,10.808,
     * 10.482,10.517,10.550,10.579,10.601,10.618,10.632,10.645,
     * 10.214,10.218,10.215,10.212,10.208,10.206,10.207,10.212,
     * 9.734,9.726,9.714,9.699,9.682,9.664,9.649,9.638,
     * 9.563,9.545,9.525,9.499,9.474,9.446,9.420,9.397,
     * 9.277,9.258,9.238,9.215,9.186,9.159,9.132,9.110/
C
      DATA ((EDEN(I,4,11,L,3),I=49,56),L=1,NZEN)/
     * 11.228,11.243,11.254,11.260,11.265,11.269,11.270,11.272,
     * 11.186,11.203,11.217,11.225,11.232,11.235,11.239,11.239,
     * 11.126,11.146,11.159,11.173,11.179,11.184,11.189,11.196,
     * 11.021,11.041,11.059,11.073,11.085,11.094,11.102,11.109,
     * 10.825,10.843,10.858,10.870,10.881,10.891,10.901,10.912,
     * 10.701,10.724,10.741,10.754,10.764,10.773,10.781,10.791,
     * 10.470,10.506,10.539,10.563,10.584,10.597,10.610,10.621,
     * 10.158,10.161,10.158,10.155,10.151,10.149,10.151,10.158,
     * 9.682,9.672,9.658,9.641,9.622,9.603,9.588,9.578,
     * 9.492,9.470,9.448,9.420,9.392,9.363,9.335,9.311,
     * 9.218,9.194,9.169,9.139,9.109,9.078,9.048,9.022/
C
      DATA ((EDEN(I,5,11,L,3),I=49,56),L=1,NZEN)/
     * 11.222,11.238,11.251,11.262,11.269,11.271,11.274,11.277,
     * 11.181,11.199,11.214,11.224,11.233,11.237,11.241,11.243,
     * 11.118,11.139,11.156,11.172,11.179,11.185,11.192,11.199,
     * 11.011,11.035,11.051,11.070,11.083,11.094,11.103,11.110,
     * 10.804,10.827,10.845,10.860,10.873,10.884,10.894,10.906,
     * 10.673,10.701,10.722,10.738,10.750,10.759,10.768,10.778,
     * 10.425,10.469,10.506,10.536,10.557,10.574,10.587,10.599,
     * 10.094,10.097,10.095,10.092,10.090,10.091,10.095,10.104,
     * 9.642,9.628,9.611,9.591,9.570,9.548,9.530,9.516,
     * 9.453,9.427,9.400,9.368,9.335,9.301,9.268,9.239,
     * 9.177,9.146,9.113,9.075,9.039,9.000,8.965,8.933/
C
      DATA ((EDEN(I,1,12,L,3),I=49,56),L=1,NZEN)/
     * 11.252,11.262,11.267,11.269,11.268,11.267,11.263,11.263,
     * 11.210,11.220,11.225,11.225,11.226,11.224,11.222,11.221,
     * 11.155,11.167,11.171,11.177,11.177,11.177,11.177,11.178,
     * 11.060,11.076,11.086,11.092,11.098,11.102,11.106,11.109,
     * 10.868,10.884,10.898,10.910,10.920,10.929,10.939,10.949,
     * 10.727,10.758,10.782,10.799,10.812,10.823,10.832,10.844,
     * 10.503,10.544,10.579,10.610,10.636,10.654,10.669,10.683,
     * 10.234,10.242,10.243,10.241,10.239,10.237,10.238,10.245,
     * 9.782,9.779,9.771,9.760,9.746,9.731,9.717,9.706,
     * 9.606,9.593,9.577,9.556,9.532,9.509,9.485,9.464,
     * 9.337,9.325,9.308,9.288,9.262,9.239,9.216,9.193/
C
      DATA ((EDEN(I,2,12,L,3),I=49,56),L=1,NZEN)/
     * 11.243,11.253,11.259,11.264,11.263,11.262,11.258,11.258,
     * 11.204,11.217,11.226,11.229,11.231,11.232,11.231,11.230,
     * 11.151,11.163,11.175,11.177,11.181,11.181,11.182,11.183,
     * 11.056,11.072,11.086,11.094,11.099,11.102,11.106,11.111,
     * 10.862,10.880,10.893,10.905,10.914,10.923,10.932,10.943,
     * 10.742,10.764,10.781,10.792,10.802,10.810,10.818,10.828,
     * 10.491,10.529,10.566,10.596,10.620,10.637,10.652,10.665,
     * 10.220,10.225,10.223,10.221,10.219,10.218,10.222,10.230,
     * 9.748,9.741,9.728,9.714,9.697,9.679,9.663,9.650,
     * 9.584,9.566,9.545,9.520,9.492,9.465,9.439,9.417,
     * 9.300,9.283,9.263,9.238,9.211,9.185,9.160,9.136/
C
      DATA ((EDEN(I,3,12,L,3),I=49,56),L=1,NZEN)/
     * 11.236,11.247,11.257,11.259,11.259,11.259,11.259,11.256,
     * 11.197,11.210,11.219,11.226,11.229,11.231,11.230,11.229,
     * 11.140,11.156,11.167,11.177,11.182,11.186,11.187,11.189,
     * 11.048,11.063,11.073,11.082,11.091,11.096,11.099,11.104,
     * 10.852,10.867,10.881,10.891,10.899,10.907,10.916,10.925,
     * 10.728,10.749,10.764,10.776,10.785,10.792,10.800,10.810,
     * 10.519,10.551,10.576,10.597,10.612,10.624,10.633,10.643,
     * 10.214,10.215,10.209,10.202,10.196,10.192,10.192,10.198,
     * 9.718,9.707,9.690,9.671,9.651,9.630,9.612,9.599,
     * 9.526,9.503,9.479,9.451,9.421,9.393,9.366,9.344,
     * 9.255,9.233,9.208,9.177,9.149,9.118,9.089,9.066/
C
      DATA ((EDEN(I,4,12,L,3),I=49,56),L=1,NZEN)/
     * 11.234,11.248,11.257,11.263,11.265,11.266,11.267,11.269,
     * 11.195,11.209,11.222,11.228,11.231,11.234,11.238,11.238,
     * 11.135,11.152,11.167,11.178,11.184,11.189,11.191,11.193,
     * 11.031,11.047,11.066,11.079,11.091,11.097,11.104,11.111,
     * 10.831,10.847,10.863,10.874,10.885,10.894,10.903,10.913,
     * 10.706,10.726,10.743,10.755,10.764,10.771,10.780,10.789,
     * 10.486,10.519,10.547,10.568,10.583,10.595,10.604,10.614,
     * 10.152,10.152,10.147,10.141,10.137,10.135,10.137,10.144,
     * 9.659,9.645,9.626,9.605,9.582,9.561,9.544,9.534,
     * 9.467,9.440,9.413,9.381,9.349,9.316,9.285,9.260,
     * 9.189,9.160,9.130,9.095,9.059,9.024,8.991,8.962/
C
      DATA ((EDEN(I,5,12,L,3),I=49,56),L=1,NZEN)/
     * 11.230,11.244,11.257,11.265,11.271,11.271,11.274,11.273,
     * 11.186,11.204,11.219,11.229,11.234,11.239,11.240,11.241,
     * 11.127,11.148,11.165,11.177,11.183,11.189,11.192,11.196,
     * 11.020,11.040,11.060,11.073,11.086,11.097,11.101,11.110,
     * 10.807,10.829,10.847,10.862,10.874,10.884,10.894,10.905,
     * 10.670,10.698,10.719,10.734,10.745,10.755,10.764,10.773,
     * 10.421,10.464,10.500,10.529,10.549,10.564,10.576,10.587,
     * 10.076,10.076,10.072,10.067,10.064,10.064,10.068,10.076,
     * 9.613,9.595,9.574,9.550,9.525,9.501,9.481,9.468,
     * 9.420,9.386,9.354,9.316,9.278,9.240,9.201,9.171,
     * 9.137,9.100,9.061,9.019,8.976,8.934,8.895,8.860/
C
      DATA ((EDEN(I,1,1,L,1),I=57,64),L=1,NZEN)/
     * 11.075,11.070,11.064,11.059,11.054,11.050,11.048,11.047,
     * 11.037,11.032,11.028,11.024,11.020,11.017,11.014,11.014,
     * 10.997,10.994,10.991,10.988,10.985,10.983,10.982,10.983,
     * 10.939,10.939,10.938,10.937,10.936,10.934,10.933,10.934,
     * 10.795,10.798,10.801,10.804,10.805,10.806,10.806,10.809,
     * 10.694,10.700,10.704,10.708,10.711,10.713,10.716,10.719,
     * 10.531,10.537,10.543,10.546,10.550,10.552,10.557,10.561,
     * 10.014,10.030,10.047,10.062,10.075,10.084,10.090,10.112,
     * 9.515,9.490,9.467,9.446,9.429,9.414,9.403,9.396,
     * 9.271,9.234,9.200,9.168,9.136,9.108,9.081,9.057,
     * 8.985,8.946,8.912,8.879,8.846,8.818,8.794,8.771/
C
      DATA ((EDEN(I,2,1,L,1),I=57,64),L=1,NZEN)/
     * 11.076,11.070,11.065,11.060,11.055,11.051,11.049,11.048,
     * 11.052,11.046,11.042,11.037,11.032,11.028,11.025,11.025,
     * 11.006,11.002,10.998,10.995,10.991,10.988,10.987,10.986,
     * 10.941,10.941,10.940,10.938,10.937,10.936,10.936,10.937,
     * 10.788,10.792,10.797,10.799,10.800,10.802,10.805,10.808,
     * 10.677,10.682,10.686,10.691,10.695,10.699,10.703,10.707,
     * 10.514,10.520,10.526,10.532,10.537,10.543,10.548,10.556,
     * 10.006,10.022,10.038,10.052,10.063,10.071,10.079,10.102,
     * 9.458,9.434,9.415,9.398,9.386,9.378,9.375,9.375,
     * 9.222,9.186,9.155,9.128,9.100,9.075,9.053,9.033,
     * 8.923,8.887,8.858,8.828,8.802,8.779,8.757,8.738/
C
      DATA ((EDEN(I,3,1,L,1),I=57,64),L=1,NZEN)/
     * 11.081,11.075,11.071,11.066,11.062,11.058,11.057,11.056,
     * 11.055,11.050,11.046,11.042,11.038,11.034,11.033,11.033,
     * 11.017,11.014,11.011,11.008,11.005,11.002,11.001,11.001,
     * 10.942,10.941,10.940,10.939,10.938,10.936,10.937,10.939,
     * 10.773,10.778,10.782,10.786,10.788,10.791,10.795,10.800,
     * 10.661,10.668,10.674,10.679,10.686,10.690,10.697,10.704,
     * 10.489,10.497,10.504,10.511,10.518,10.525,10.533,10.543,
     * 9.971,9.986,10.002,10.017,10.031,10.043,10.053,10.080,
     * 9.403,9.382,9.366,9.356,9.350,9.349,9.351,9.356,
     * 9.143,9.111,9.085,9.061,9.039,9.022,9.005,8.992,
     * 8.852,8.819,8.790,8.763,8.744,8.725,8.710,8.697/
C
      DATA ((EDEN(I,4,1,L,1),I=57,64),L=1,NZEN)/
     * 11.086,11.081,11.078,11.075,11.072,11.068,11.068,11.069,
     * 11.060,11.055,11.051,11.048,11.046,11.044,11.043,11.044,
     * 11.019,11.017,11.015,11.012,11.010,11.009,11.009,11.010,
     * 10.946,10.946,10.946,10.945,10.945,10.945,10.947,10.949,
     * 10.763,10.769,10.773,10.777,10.781,10.785,10.791,10.797,
     * 10.643,10.650,10.657,10.664,10.671,10.677,10.685,10.693,
     * 10.466,10.475,10.483,10.490,10.500,10.509,10.519,10.531,
     * 9.946,9.960,9.975,9.990,10.004,10.017,10.029,10.059,
     * 9.356,9.341,9.332,9.329,9.330,9.335,9.340,9.348,
     * 9.087,9.055,9.028,9.009,8.990,8.974,8.961,8.952,
     * 8.776,8.744,8.716,8.694,8.678,8.667,8.656,8.647/
C
      DATA ((EDEN(I,5,1,L,1),I=57,64),L=1,NZEN)/
     * 11.091,11.088,11.084,11.081,11.079,11.077,11.077,11.079,
     * 11.061,11.059,11.056,11.053,11.051,11.050,11.050,11.052,
     * 11.019,11.018,11.017,11.015,11.014,11.014,11.015,11.017,
     * 10.942,10.944,10.944,10.946,10.946,10.947,10.950,10.955,
     * 10.754,10.761,10.766,10.772,10.777,10.784,10.790,10.797,
     * 10.628,10.635,10.643,10.651,10.660,10.668,10.677,10.688,
     * 10.444,10.452,10.461,10.471,10.482,10.493,10.505,10.519,
     * 9.903,9.917,9.934,9.952,9.969,9.985,10.000,10.032,
     * 9.312,9.297,9.289,9.287,9.290,9.295,9.303,9.313,
     * 9.027,8.996,8.970,8.950,8.934,8.921,8.912,8.906,
     * 8.701,8.668,8.643,8.623,8.610,8.602,8.596,8.595/
C
      DATA ((EDEN(I,1,2,L,1),I=57,64),L=1,NZEN)/
     * 11.082,11.077,11.071,11.066,11.061,11.056,11.054,11.054,
     * 11.044,11.039,11.035,11.031,11.027,11.023,11.021,11.021,
     * 11.003,11.000,10.998,10.995,10.991,10.989,10.988,10.988,
     * 10.943,10.943,10.943,10.941,10.940,10.939,10.939,10.939,
     * 10.795,10.799,10.803,10.805,10.806,10.808,10.808,10.811,
     * 10.693,10.699,10.704,10.708,10.711,10.713,10.716,10.719,
     * 10.528,10.535,10.541,10.546,10.548,10.551,10.555,10.560,
     * 10.007,10.021,10.038,10.055,10.068,10.078,10.086,10.107,
     * 9.523,9.497,9.474,9.452,9.434,9.419,9.407,9.399,
     * 9.280,9.243,9.209,9.176,9.145,9.114,9.088,9.062,
     * 8.994,8.955,8.918,8.886,8.853,8.825,8.798,8.774/
C
      DATA ((EDEN(I,2,2,L,1),I=57,64),L=1,NZEN)/
     * 11.084,11.080,11.074,11.069,11.064,11.059,11.057,11.056,
     * 11.058,11.055,11.049,11.045,11.040,11.036,11.034,11.033,
     * 11.007,11.004,11.002,10.998,10.996,10.993,10.993,10.993,
     * 10.945,10.944,10.944,10.943,10.942,10.941,10.941,10.943,
     * 10.788,10.794,10.797,10.800,10.803,10.804,10.807,10.811,
     * 10.676,10.682,10.686,10.692,10.696,10.699,10.703,10.708,
     * 10.512,10.519,10.527,10.531,10.536,10.542,10.548,10.554,
     * 10.002,10.018,10.034,10.048,10.060,10.070,10.077,10.101,
     * 9.485,9.459,9.437,9.417,9.401,9.389,9.381,9.377,
     * 9.239,9.204,9.172,9.142,9.114,9.087,9.064,9.044,
     * 8.943,8.905,8.872,8.843,8.816,8.789,8.765,8.750/
C
      DATA ((EDEN(I,3,2,L,1),I=57,64),L=1,NZEN)/
     * 11.088,11.084,11.079,11.075,11.071,11.067,11.065,11.065,
     * 11.062,11.058,11.054,11.051,11.046,11.043,11.041,11.042,
     * 11.024,11.021,11.019,11.016,11.012,11.009,11.008,11.009,
     * 10.943,10.943,10.943,10.943,10.942,10.941,10.943,10.944,
     * 10.775,10.780,10.785,10.788,10.791,10.795,10.799,10.803,
     * 10.662,10.669,10.676,10.681,10.686,10.692,10.698,10.706,
     * 10.491,10.497,10.505,10.511,10.519,10.526,10.535,10.542,
     * 9.973,9.988,10.004,10.020,10.033,10.045,10.055,10.081,
     * 9.430,9.408,9.391,9.378,9.371,9.367,9.366,9.369,
     * 9.186,9.152,9.122,9.096,9.072,9.050,9.031,9.014,
     * 8.878,8.845,8.813,8.788,8.764,8.744,8.724,8.713/
C
      DATA ((EDEN(I,4,2,L,1),I=57,64),L=1,NZEN)/
     * 11.094,11.089,11.086,11.082,11.079,11.077,11.076,11.077,
     * 11.065,11.061,11.059,11.056,11.053,11.051,11.050,11.052,
     * 11.026,11.024,11.021,11.020,11.018,11.016,11.016,11.018,
     * 10.950,10.951,10.952,10.952,10.951,10.952,10.953,10.956,
     * 10.765,10.771,10.776,10.781,10.785,10.790,10.794,10.801,
     * 10.646,10.652,10.660,10.666,10.674,10.679,10.687,10.697,
     * 10.470,10.479,10.487,10.495,10.505,10.512,10.524,10.535,
     * 9.950,9.964,9.979,9.994,10.009,10.022,10.033,10.061,
     * 9.385,9.368,9.356,9.349,9.347,9.348,9.351,9.356,
     * 9.120,9.087,9.061,9.038,9.016,8.998,8.983,8.972,
     * 8.810,8.779,8.750,8.727,8.705,8.688,8.675,8.667/
C
      DATA ((EDEN(I,5,2,L,1),I=57,64),L=1,NZEN)/
     * 11.096,11.094,11.090,11.088,11.086,11.085,11.085,11.087,
     * 11.067,11.066,11.062,11.060,11.059,11.058,11.059,11.061,
     * 11.024,11.024,11.023,11.022,11.021,11.021,11.022,11.025,
     * 10.946,10.948,10.950,10.951,10.952,10.954,10.957,10.961,
     * 10.757,10.764,10.770,10.777,10.783,10.789,10.795,10.802,
     * 10.631,10.639,10.646,10.654,10.663,10.671,10.681,10.689,
     * 10.449,10.458,10.466,10.476,10.487,10.497,10.507,10.521,
     * 9.912,9.926,9.942,9.960,9.976,9.991,10.006,10.036,
     * 9.345,9.328,9.317,9.312,9.312,9.315,9.320,9.327,
     * 9.070,9.039,9.011,8.988,8.970,8.955,8.941,8.932,
     * 8.744,8.711,8.682,8.664,8.648,8.633,8.624,8.619/
C
      DATA ((EDEN(I,1,3,L,1),I=57,64),L=1,NZEN)/
     * 11.088,11.083,11.077,11.072,11.067,11.063,11.060,11.060,
     * 11.048,11.045,11.040,11.036,11.032,11.029,11.027,11.027,
     * 11.006,11.004,11.002,10.999,10.997,10.994,10.993,10.993,
     * 10.945,10.945,10.945,10.944,10.943,10.942,10.941,10.942,
     * 10.791,10.797,10.800,10.803,10.805,10.807,10.808,10.811,
     * 10.689,10.695,10.700,10.704,10.707,10.710,10.712,10.716,
     * 10.522,10.532,10.538,10.541,10.545,10.546,10.551,10.555,
     * 9.999,10.013,10.029,10.046,10.060,10.071,10.080,10.102,
     * 9.530,9.504,9.481,9.459,9.441,9.425,9.412,9.403,
     * 9.290,9.253,9.217,9.184,9.151,9.121,9.093,9.067,
     * 9.002,8.961,8.925,8.893,8.860,8.829,8.801,8.774/
C
      DATA ((EDEN(I,2,3,L,1),I=57,64),L=1,NZEN)/
     * 11.090,11.085,11.081,11.076,11.071,11.067,11.065,11.064,
     * 11.053,11.049,11.045,11.041,11.038,11.034,11.032,11.032,
     * 11.010,11.008,11.006,11.003,11.001,10.999,10.998,10.998,
     * 10.945,10.946,10.946,10.946,10.945,10.944,10.945,10.946,
     * 10.785,10.792,10.796,10.800,10.802,10.805,10.808,10.811,
     * 10.681,10.687,10.693,10.697,10.702,10.706,10.710,10.714,
     * 10.512,10.522,10.527,10.533,10.538,10.541,10.545,10.553,
     * 10.004,10.019,10.035,10.049,10.061,10.070,10.077,10.100,
     * 9.506,9.481,9.458,9.439,9.422,9.408,9.398,9.392,
     * 9.263,9.228,9.196,9.164,9.132,9.104,9.080,9.056,
     * 8.970,8.932,8.899,8.866,8.834,8.805,8.783,8.759/
C
      DATA ((EDEN(I,3,3,L,1),I=57,64),L=1,NZEN)/
     * 11.093,11.089,11.086,11.082,11.078,11.075,11.073,11.073,
     * 11.066,11.064,11.060,11.056,11.053,11.050,11.048,11.049,
     * 11.021,11.019,11.016,11.014,11.012,11.010,11.009,11.010,
     * 10.946,10.948,10.948,10.949,10.950,10.950,10.951,10.954,
     * 10.778,10.784,10.789,10.795,10.799,10.802,10.806,10.811,
     * 10.663,10.669,10.676,10.682,10.688,10.693,10.698,10.703,
     * 10.497,10.504,10.513,10.521,10.526,10.531,10.537,10.546,
     * 9.985,9.999,10.015,10.029,10.042,10.053,10.061,10.085,
     * 9.473,9.451,9.432,9.416,9.404,9.395,9.389,9.386,
     * 9.227,9.192,9.161,9.133,9.105,9.080,9.056,9.036,
     * 8.927,8.891,8.858,8.827,8.800,8.773,8.753,8.734/
C
      DATA ((EDEN(I,4,3,L,1),I=57,64),L=1,NZEN)/
     * 11.096,11.093,11.090,11.087,11.084,11.083,11.083,11.084,
     * 11.068,11.066,11.063,11.061,11.059,11.057,11.057,11.058,
     * 11.027,11.027,11.025,11.024,11.023,11.021,11.022,11.023,
     * 10.948,10.950,10.951,10.952,10.953,10.953,10.955,10.957,
     * 10.767,10.773,10.780,10.785,10.790,10.794,10.800,10.805,
     * 10.651,10.658,10.665,10.672,10.679,10.686,10.692,10.700,
     * 10.480,10.489,10.495,10.503,10.511,10.518,10.525,10.534,
     * 9.965,9.978,9.992,10.007,10.020,10.032,10.042,10.067,
     * 9.435,9.416,9.401,9.390,9.383,9.380,9.378,9.378,
     * 9.187,9.154,9.122,9.095,9.070,9.046,9.025,9.006,
     * 8.872,8.840,8.809,8.779,8.756,8.733,8.715,8.698/
C
      DATA ((EDEN(I,5,3,L,1),I=57,64),L=1,NZEN)/
     * 11.097,11.095,11.093,11.092,11.091,11.090,11.091,11.094,
     * 11.067,11.066,11.065,11.065,11.064,11.064,11.064,11.067,
     * 11.025,11.026,11.026,11.026,11.026,11.027,11.028,11.031,
     * 10.947,10.950,10.953,10.955,10.957,10.960,10.962,10.966,
     * 10.759,10.766,10.773,10.779,10.785,10.791,10.798,10.805,
     * 10.640,10.646,10.654,10.662,10.669,10.677,10.685,10.694,
     * 10.463,10.472,10.482,10.491,10.499,10.507,10.516,10.526,
     * 9.941,9.954,9.969,9.984,9.998,10.011,10.022,10.049,
     * 9.405,9.386,9.372,9.362,9.357,9.354,9.354,9.355,
     * 9.146,9.113,9.083,9.056,9.033,9.011,8.991,8.974,
     * 8.815,8.781,8.752,8.728,8.706,8.688,8.670,8.658/
C
      DATA ((EDEN(I,1,4,L,1),I=57,64),L=1,NZEN)/
     * 11.082,11.077,11.072,11.067,11.062,11.058,11.055,11.054,
     * 11.042,11.038,11.035,11.031,11.027,11.024,11.021,11.021,
     * 11.001,10.998,10.996,10.994,10.991,10.989,10.987,10.988,
     * 10.939,10.940,10.939,10.938,10.937,10.935,10.936,10.937,
     * 10.785,10.790,10.794,10.797,10.799,10.800,10.802,10.804,
     * 10.684,10.689,10.694,10.698,10.701,10.703,10.705,10.710,
     * 10.521,10.528,10.532,10.538,10.539,10.541,10.546,10.550,
     * 9.996,10.010,10.026,10.042,10.056,10.067,10.075,10.098,
     * 9.527,9.502,9.479,9.458,9.440,9.425,9.413,9.403,
     * 9.287,9.250,9.213,9.180,9.147,9.117,9.088,9.062,
     * 8.999,8.958,8.922,8.886,8.856,8.825,8.798,8.771/
C
      DATA ((EDEN(I,2,4,L,1),I=57,64),L=1,NZEN)/
     * 11.085,11.082,11.077,11.072,11.067,11.063,11.061,11.061,
     * 11.047,11.044,11.040,11.037,11.033,11.030,11.028,11.029,
     * 11.004,11.003,11.001,10.999,10.996,10.994,10.994,10.994,
     * 10.939,10.940,10.941,10.941,10.941,10.940,10.940,10.941,
     * 10.790,10.797,10.802,10.806,10.808,10.810,10.812,10.816,
     * 10.680,10.686,10.691,10.695,10.699,10.702,10.706,10.710,
     * 10.515,10.524,10.530,10.534,10.537,10.541,10.543,10.548,
     * 9.998,10.011,10.026,10.042,10.055,10.066,10.074,10.096,
     * 9.524,9.500,9.479,9.460,9.443,9.429,9.418,9.409,
     * 9.281,9.246,9.212,9.179,9.147,9.117,9.088,9.061,
     * 8.996,8.958,8.920,8.885,8.854,8.823,8.795,8.771/
C
      DATA ((EDEN(I,3,4,L,1),I=57,64),L=1,NZEN)/
     * 11.090,11.084,11.082,11.078,11.074,11.071,11.069,11.070,
     * 11.063,11.060,11.056,11.053,11.049,11.047,11.045,11.045,
     * 11.010,11.009,11.008,11.007,11.006,11.005,11.005,11.006,
     * 10.940,10.942,10.944,10.945,10.945,10.946,10.947,10.950,
     * 10.776,10.782,10.788,10.793,10.796,10.799,10.802,10.807,
     * 10.672,10.678,10.684,10.689,10.693,10.697,10.702,10.707,
     * 10.506,10.514,10.522,10.526,10.531,10.534,10.539,10.544,
     * 9.998,10.010,10.024,10.038,10.050,10.060,10.068,10.090,
     * 9.511,9.490,9.471,9.455,9.442,9.430,9.421,9.414,
     * 9.267,9.234,9.201,9.169,9.137,9.108,9.081,9.054,
     * 8.973,8.936,8.900,8.869,8.837,8.805,8.780,8.755/
C
      DATA ((EDEN(I,4,4,L,1),I=57,64),L=1,NZEN)/
     * 11.089,11.086,11.084,11.081,11.079,11.078,11.077,11.079,
     * 11.060,11.060,11.058,11.056,11.054,11.052,11.052,11.053,
     * 11.017,11.018,11.017,11.017,11.015,11.015,11.015,11.017,
     * 10.941,10.944,10.947,10.949,10.951,10.952,10.954,10.957,
     * 10.769,10.777,10.783,10.788,10.793,10.797,10.801,10.806,
     * 10.662,10.667,10.675,10.681,10.685,10.690,10.695,10.702,
     * 10.496,10.507,10.512,10.518,10.522,10.527,10.531,10.537,
     * 9.998,10.009,10.021,10.033,10.044,10.052,10.059,10.080,
     * 9.498,9.478,9.461,9.448,9.436,9.427,9.419,9.412,
     * 9.257,9.223,9.191,9.157,9.127,9.095,9.067,9.039,
     * 8.944,8.910,8.875,8.843,8.813,8.783,8.757,8.731/
C
      DATA ((EDEN(I,5,4,L,1),I=57,64),L=1,NZEN)/
     * 11.085,11.085,11.084,11.084,11.084,11.084,11.086,11.088,
     * 11.057,11.058,11.058,11.058,11.059,11.058,11.060,11.062,
     * 11.018,11.020,11.021,11.022,11.023,11.023,11.024,11.027,
     * 10.942,10.946,10.949,10.952,10.953,10.955,10.958,10.962,
     * 10.764,10.771,10.778,10.784,10.790,10.795,10.801,10.807,
     * 10.654,10.660,10.667,10.673,10.678,10.684,10.691,10.697,
     * 10.487,10.497,10.504,10.511,10.516,10.519,10.526,10.530,
     * 9.990,10.001,10.012,10.023,10.033,10.041,10.047,10.068,
     * 9.481,9.462,9.445,9.432,9.422,9.413,9.405,9.398,
     * 9.235,9.200,9.166,9.135,9.103,9.075,9.046,9.017,
     * 8.919,8.883,8.849,8.818,8.787,8.759,8.731,8.707/
C
      DATA ((EDEN(I,1,5,L,1),I=57,64),L=1,NZEN)/
     * 11.067,11.062,11.057,11.052,11.047,11.042,11.039,11.039,
     * 11.028,11.024,11.020,11.016,11.012,11.009,11.007,11.006,
     * 10.987,10.984,10.982,10.979,10.977,10.974,10.974,10.973,
     * 10.927,10.928,10.927,10.927,10.925,10.924,10.923,10.924,
     * 10.780,10.785,10.788,10.790,10.791,10.792,10.793,10.796,
     * 10.680,10.686,10.690,10.694,10.696,10.698,10.701,10.704,
     * 10.517,10.525,10.529,10.533,10.536,10.538,10.541,10.545,
     * 10.002,10.016,10.033,10.048,10.060,10.069,10.076,10.098,
     * 9.512,9.488,9.467,9.448,9.432,9.418,9.408,9.400,
     * 9.266,9.232,9.198,9.163,9.132,9.101,9.074,9.049,
     * 8.982,8.943,8.909,8.876,8.842,8.814,8.786,8.763/
C
      DATA ((EDEN(I,2,5,L,1),I=57,64),L=1,NZEN)/
     * 11.073,11.068,11.062,11.058,11.053,11.049,11.047,11.047,
     * 11.024,11.021,11.019,11.016,11.014,11.012,11.012,11.012,
     * 10.993,10.991,10.991,10.989,10.988,10.986,10.985,10.986,
     * 10.939,10.941,10.942,10.941,10.941,10.940,10.939,10.940,
     * 10.789,10.795,10.799,10.802,10.804,10.806,10.807,10.809,
     * 10.681,10.686,10.691,10.696,10.698,10.701,10.704,10.707,
     * 10.521,10.527,10.534,10.536,10.538,10.541,10.545,10.548,
     * 10.017,10.031,10.045,10.058,10.069,10.076,10.082,10.102,
     * 9.531,9.510,9.491,9.474,9.458,9.445,9.433,9.423,
     * 9.285,9.249,9.214,9.180,9.148,9.115,9.085,9.059,
     * 8.998,8.960,8.924,8.889,8.856,8.825,8.798,8.771/
C
      DATA ((EDEN(I,3,5,L,1),I=57,64),L=1,NZEN)/
     * 11.075,11.071,11.067,11.063,11.059,11.057,11.056,11.056,
     * 11.039,11.037,11.034,11.031,11.029,11.027,11.026,11.027,
     * 10.997,10.997,10.997,10.997,10.996,10.996,10.996,10.998,
     * 10.938,10.941,10.943,10.944,10.944,10.944,10.945,10.946,
     * 10.783,10.789,10.794,10.799,10.802,10.804,10.805,10.809,
     * 10.676,10.683,10.688,10.692,10.695,10.698,10.701,10.705,
     * 10.521,10.531,10.536,10.539,10.542,10.543,10.546,10.550,
     * 10.035,10.047,10.059,10.069,10.077,10.082,10.085,10.103,
     * 9.543,9.525,9.509,9.493,9.479,9.466,9.454,9.442,
     * 9.308,9.272,9.236,9.200,9.164,9.129,9.097,9.066,
     * 8.995,8.961,8.926,8.891,8.858,8.827,8.797,8.766/
C
      DATA ((EDEN(I,4,5,L,1),I=57,64),L=1,NZEN)/
     * 11.073,11.071,11.068,11.066,11.064,11.063,11.062,11.064,
     * 11.049,11.047,11.045,11.043,11.040,11.039,11.039,11.040,
     * 11.003,11.004,11.004,11.004,11.003,11.003,11.003,11.006,
     * 10.937,10.941,10.943,10.946,10.948,10.948,10.951,10.954,
     * 10.776,10.783,10.789,10.794,10.797,10.800,10.804,10.807,
     * 10.676,10.682,10.688,10.692,10.695,10.698,10.701,10.705,
     * 10.518,10.526,10.533,10.535,10.537,10.539,10.540,10.543,
     * 10.045,10.055,10.065,10.072,10.078,10.080,10.081,10.098,
     * 9.547,9.529,9.513,9.499,9.485,9.471,9.458,9.445,
     * 9.304,9.272,9.235,9.198,9.162,9.125,9.091,9.059,
     * 8.989,8.956,8.920,8.885,8.852,8.818,8.784,8.753/
C
      DATA ((EDEN(I,5,5,L,1),I=57,64),L=1,NZEN)/
     * 11.068,11.068,11.069,11.068,11.068,11.069,11.070,11.073,
     * 11.044,11.045,11.045,11.045,11.045,11.045,11.046,11.048,
     * 11.008,11.010,11.010,11.011,11.011,11.012,11.012,11.015,
     * 10.935,10.939,10.943,10.946,10.948,10.950,10.952,10.956,
     * 10.772,10.779,10.785,10.791,10.794,10.799,10.802,10.805,
     * 10.670,10.676,10.681,10.687,10.690,10.692,10.696,10.700,
     * 10.517,10.526,10.530,10.533,10.535,10.535,10.539,10.540,
     * 10.050,10.059,10.067,10.073,10.076,10.077,10.077,10.092,
     * 9.545,9.529,9.515,9.501,9.487,9.474,9.459,9.444,
     * 9.302,9.267,9.229,9.191,9.154,9.117,9.079,9.043,
     * 9.004,8.969,8.931,8.892,8.856,8.818,8.780,8.746/
C
      DATA ((EDEN(I,1,6,L,1),I=57,64),L=1,NZEN)/
     * 11.051,11.045,11.040,11.035,11.031,11.027,11.024,11.023,
     * 11.012,11.007,11.004,11.000,10.997,10.993,10.991,10.991,
     * 10.972,10.970,10.967,10.965,10.962,10.960,10.958,10.959,
     * 10.917,10.916,10.916,10.915,10.913,10.912,10.911,10.912,
     * 10.776,10.780,10.782,10.785,10.786,10.787,10.788,10.790,
     * 10.678,10.684,10.689,10.692,10.695,10.697,10.698,10.702,
     * 10.516,10.522,10.526,10.532,10.534,10.537,10.540,10.545,
     * 10.009,10.024,10.040,10.054,10.064,10.072,10.078,10.099,
     * 9.495,9.473,9.454,9.437,9.423,9.412,9.403,9.397,
     * 9.245,9.211,9.177,9.146,9.116,9.087,9.061,9.036,
     * 8.965,8.927,8.892,8.862,8.831,8.803,8.779,8.755/
C
      DATA ((EDEN(I,2,6,L,1),I=57,64),L=1,NZEN)/
     * 11.057,11.051,11.046,11.042,11.038,11.034,11.032,11.032,
     * 11.008,11.005,11.003,11.001,10.998,10.997,10.996,10.997,
     * 10.979,10.977,10.977,10.975,10.973,10.972,10.972,10.972,
     * 10.931,10.931,10.931,10.931,10.931,10.929,10.928,10.929,
     * 10.790,10.795,10.798,10.801,10.802,10.802,10.802,10.804,
     * 10.682,10.688,10.693,10.696,10.698,10.700,10.701,10.705,
     * 10.529,10.537,10.541,10.544,10.545,10.548,10.550,10.552,
     * 10.039,10.053,10.066,10.076,10.084,10.089,10.091,10.110,
     * 9.539,9.521,9.503,9.488,9.473,9.459,9.446,9.435,
     * 9.299,9.262,9.226,9.191,9.155,9.120,9.089,9.061,
     * 8.997,8.962,8.927,8.894,8.859,8.830,8.800,8.771/
C
      DATA ((EDEN(I,3,6,L,1),I=57,64),L=1,NZEN)/
     * 11.061,11.056,11.052,11.048,11.045,11.041,11.040,11.041,
     * 11.018,11.016,11.014,11.012,11.011,11.009,11.009,11.011,
     * 10.986,10.986,10.985,10.985,10.983,10.983,10.983,10.985,
     * 10.931,10.933,10.934,10.935,10.936,10.935,10.936,10.937,
     * 10.786,10.792,10.797,10.799,10.802,10.802,10.803,10.805,
     * 10.690,10.695,10.699,10.703,10.706,10.707,10.709,10.710,
     * 10.532,10.541,10.545,10.548,10.548,10.547,10.549,10.551,
     * 10.063,10.074,10.084,10.092,10.096,10.097,10.096,10.112,
     * 9.567,9.549,9.533,9.516,9.500,9.484,9.468,9.453,
     * 9.318,9.283,9.246,9.209,9.172,9.135,9.100,9.067,
     * 8.988,8.956,8.924,8.892,8.859,8.828,8.797,8.767/
C
      DATA ((EDEN(I,4,6,L,1),I=57,64),L=1,NZEN)/
     * 11.060,11.057,11.054,11.052,11.049,11.048,11.047,11.049,
     * 11.031,11.030,11.028,11.026,11.023,11.022,11.022,11.024,
     * 10.993,10.994,10.994,10.994,10.994,10.994,10.995,10.997,
     * 10.933,10.936,10.938,10.939,10.940,10.941,10.942,10.945,
     * 10.787,10.794,10.799,10.803,10.805,10.806,10.807,10.808,
     * 10.686,10.692,10.696,10.699,10.701,10.703,10.705,10.705,
     * 10.536,10.545,10.547,10.550,10.550,10.549,10.550,10.550,
     * 10.077,10.087,10.095,10.100,10.102,10.102,10.099,10.112,
     * 9.580,9.566,9.552,9.537,9.521,9.504,9.486,9.468,
     * 9.332,9.299,9.262,9.222,9.183,9.142,9.105,9.067,
     * 9.029,8.995,8.960,8.921,8.883,8.843,8.807,8.775/
C
      DATA ((EDEN(I,5,6,L,1),I=57,64),L=1,NZEN)/
     * 11.055,11.055,11.054,11.054,11.053,11.053,11.054,11.056,
     * 11.033,11.034,11.033,11.032,11.031,11.031,11.031,11.033,
     * 10.999,10.999,11.000,11.000,11.000,11.001,11.002,11.003,
     * 10.934,10.938,10.941,10.943,10.944,10.946,10.947,10.949,
     * 10.784,10.790,10.795,10.799,10.801,10.803,10.803,10.806,
     * 10.687,10.693,10.697,10.700,10.701,10.702,10.703,10.705,
     * 10.540,10.546,10.549,10.551,10.551,10.550,10.549,10.547,
     * 10.095,10.103,10.108,10.110,10.109,10.105,10.099,10.110,
     * 9.593,9.582,9.570,9.556,9.540,9.521,9.500,9.477,
     * 9.333,9.298,9.260,9.217,9.175,9.134,9.092,9.051,
     * 9.065,9.029,8.987,8.947,8.900,8.859,8.813,8.771/
C
      DATA ((EDEN(I,1,7,L,1),I=57,64),L=1,NZEN)/
     * 11.048,11.042,11.037,11.033,11.028,11.024,11.022,11.022,
     * 11.009,11.005,11.001,10.998,10.994,10.991,10.990,10.990,
     * 10.970,10.967,10.965,10.963,10.961,10.959,10.958,10.958,
     * 10.915,10.914,10.914,10.913,10.912,10.911,10.911,10.912,
     * 10.776,10.780,10.783,10.785,10.787,10.788,10.789,10.790,
     * 10.678,10.684,10.690,10.693,10.695,10.697,10.701,10.703,
     * 10.514,10.522,10.526,10.530,10.534,10.538,10.543,10.547,
     * 10.010,10.025,10.040,10.054,10.065,10.073,10.078,10.100,
     * 9.493,9.471,9.453,9.437,9.424,9.413,9.405,9.399,
     * 9.241,9.207,9.175,9.144,9.116,9.087,9.061,9.036,
     * 8.962,8.924,8.892,8.859,8.831,8.803,8.779,8.755/
C
      DATA ((EDEN(I,2,7,L,1),I=57,64),L=1,NZEN)/
     * 11.053,11.048,11.044,11.039,11.035,11.031,11.030,11.030,
     * 11.005,11.002,11.000,10.998,10.996,10.995,10.994,10.995,
     * 10.977,10.976,10.975,10.974,10.972,10.971,10.970,10.971,
     * 10.931,10.932,10.932,10.932,10.931,10.929,10.928,10.929,
     * 10.790,10.797,10.800,10.802,10.804,10.804,10.805,10.806,
     * 10.685,10.691,10.696,10.700,10.702,10.703,10.705,10.707,
     * 10.529,10.537,10.543,10.546,10.548,10.551,10.552,10.556,
     * 10.043,10.056,10.069,10.080,10.087,10.091,10.094,10.112,
     * 9.542,9.524,9.508,9.493,9.478,9.464,9.451,9.440,
     * 9.300,9.265,9.228,9.192,9.156,9.123,9.092,9.060,
     * 9.000,8.964,8.929,8.895,8.863,8.830,8.803,8.774/
C
      DATA ((EDEN(I,3,7,L,1),I=57,64),L=1,NZEN)/
     * 11.057,11.054,11.049,11.046,11.042,11.039,11.038,11.038,
     * 11.016,11.014,11.012,11.010,11.009,11.007,11.007,11.009,
     * 10.985,10.985,10.984,10.983,10.982,10.981,10.982,10.983,
     * 10.933,10.934,10.935,10.936,10.936,10.935,10.936,10.937,
     * 10.790,10.796,10.800,10.803,10.804,10.806,10.805,10.807,
     * 10.693,10.700,10.704,10.707,10.710,10.710,10.712,10.713,
     * 10.536,10.544,10.549,10.550,10.550,10.553,10.554,10.555,
     * 10.069,10.081,10.090,10.097,10.101,10.102,10.101,10.116,
     * 9.574,9.558,9.541,9.525,9.509,9.492,9.475,9.459,
     * 9.323,9.290,9.252,9.215,9.177,9.139,9.105,9.071,
     * 8.998,8.966,8.934,8.901,8.867,8.832,8.804,8.772/
C
      DATA ((EDEN(I,4,7,L,1),I=57,64),L=1,NZEN)/
     * 11.059,11.056,11.053,11.050,11.047,11.045,11.045,11.045,
     * 11.032,11.029,11.027,11.025,11.022,11.021,11.021,11.021,
     * 10.995,10.995,10.995,10.994,10.994,10.993,10.994,10.996,
     * 10.936,10.938,10.941,10.941,10.941,10.941,10.942,10.944,
     * 10.793,10.799,10.805,10.807,10.809,10.810,10.811,10.812,
     * 10.692,10.697,10.703,10.705,10.707,10.708,10.708,10.709,
     * 10.543,10.549,10.553,10.556,10.554,10.556,10.554,10.556,
     * 10.087,10.097,10.104,10.109,10.111,10.109,10.105,10.117,
     * 9.592,9.578,9.564,9.548,9.532,9.514,9.494,9.474,
     * 9.344,9.308,9.271,9.231,9.191,9.149,9.110,9.071,
     * 9.041,9.007,8.971,8.934,8.891,8.854,8.816,8.778/
C
      DATA ((EDEN(I,5,7,L,1),I=57,64),L=1,NZEN)/
     * 11.055,11.055,11.054,11.053,11.052,11.051,11.052,11.054,
     * 11.035,11.035,11.034,11.032,11.030,11.030,11.029,11.031,
     * 11.001,11.002,11.001,11.001,11.000,11.000,11.000,11.002,
     * 10.939,10.942,10.944,10.946,10.946,10.947,10.947,10.950,
     * 10.791,10.798,10.803,10.805,10.807,10.807,10.809,10.808,
     * 10.694,10.701,10.706,10.707,10.708,10.708,10.708,10.708,
     * 10.548,10.554,10.559,10.559,10.558,10.557,10.552,10.554,
     * 10.108,10.116,10.121,10.122,10.120,10.115,10.107,10.117,
     * 9.611,9.600,9.588,9.572,9.554,9.533,9.509,9.484,
     * 9.349,9.314,9.274,9.230,9.187,9.144,9.099,9.057,
     * 9.083,9.046,9.007,8.962,8.917,8.870,8.826,8.781/
C
      DATA ((EDEN(I,1,8,L,1),I=57,64),L=1,NZEN)/
     * 11.061,11.057,11.052,11.047,11.043,11.040,11.037,11.036,
     * 11.023,11.019,11.015,11.012,11.009,11.006,11.004,11.004,
     * 10.983,10.980,10.979,10.976,10.975,10.972,10.972,10.972,
     * 10.924,10.925,10.925,10.925,10.924,10.923,10.923,10.924,
     * 10.779,10.785,10.788,10.791,10.793,10.794,10.795,10.798,
     * 10.680,10.686,10.691,10.695,10.699,10.702,10.704,10.708,
     * 10.516,10.523,10.529,10.532,10.536,10.540,10.544,10.549,
     * 10.004,10.018,10.034,10.049,10.061,10.070,10.077,10.099,
     * 9.507,9.485,9.465,9.448,9.434,9.422,9.412,9.405,
     * 9.261,9.224,9.192,9.161,9.129,9.101,9.074,9.049,
     * 8.976,8.940,8.906,8.873,8.842,8.814,8.786,8.763/
C
      DATA ((EDEN(I,2,8,L,1),I=57,64),L=1,NZEN)/
     * 11.065,11.061,11.057,11.052,11.048,11.045,11.043,11.043,
     * 11.019,11.016,11.014,11.012,11.010,11.009,11.008,11.009,
     * 10.990,10.988,10.988,10.986,10.985,10.984,10.984,10.984,
     * 10.940,10.941,10.942,10.942,10.941,10.940,10.939,10.940,
     * 10.793,10.800,10.804,10.807,10.808,10.810,10.811,10.812,
     * 10.685,10.692,10.697,10.701,10.705,10.706,10.708,10.711,
     * 10.525,10.532,10.536,10.542,10.543,10.546,10.549,10.554,
     * 10.031,10.044,10.057,10.069,10.079,10.085,10.089,10.108,
     * 9.547,9.527,9.509,9.492,9.476,9.462,9.449,9.438,
     * 9.311,9.274,9.237,9.201,9.166,9.132,9.100,9.070,
     * 9.007,8.969,8.935,8.898,8.867,8.835,8.806,8.778/
C
      DATA ((EDEN(I,3,8,L,1),I=57,64),L=1,NZEN)/
     * 11.070,11.066,11.062,11.058,11.055,11.051,11.050,11.051,
     * 11.029,11.027,11.025,11.023,11.022,11.021,11.020,11.022,
     * 10.997,10.997,10.996,10.995,10.995,10.993,10.994,10.995,
     * 10.942,10.944,10.945,10.945,10.946,10.945,10.946,10.947,
     * 10.792,10.799,10.804,10.807,10.809,10.810,10.811,10.813,
     * 10.687,10.693,10.698,10.702,10.705,10.706,10.709,10.712,
     * 10.532,10.540,10.545,10.551,10.552,10.553,10.553,10.558,
     * 10.053,10.065,10.076,10.085,10.091,10.095,10.096,10.112,
     * 9.571,9.552,9.535,9.517,9.500,9.484,9.468,9.453,
     * 9.324,9.289,9.253,9.217,9.179,9.144,9.107,9.075,
     * 9.005,8.974,8.940,8.906,8.873,8.836,8.806,8.776/
C
      DATA ((EDEN(I,4,8,L,1),I=57,64),L=1,NZEN)/
     * 11.072,11.069,11.066,11.063,11.060,11.058,11.057,11.059,
     * 11.043,11.041,11.039,11.036,11.034,11.032,11.033,11.034,
     * 11.006,11.005,11.005,11.004,11.002,11.001,11.002,11.003,
     * 10.944,10.947,10.949,10.950,10.950,10.950,10.951,10.953,
     * 10.790,10.797,10.802,10.805,10.808,10.809,10.811,10.814,
     * 10.692,10.698,10.703,10.705,10.709,10.710,10.712,10.714,
     * 10.535,10.542,10.547,10.550,10.550,10.550,10.551,10.553,
     * 10.066,10.077,10.086,10.093,10.097,10.098,10.097,10.111,
     * 9.580,9.563,9.547,9.531,9.515,9.498,9.481,9.463,
     * 9.338,9.301,9.264,9.224,9.186,9.146,9.108,9.071,
     * 9.030,8.993,8.958,8.920,8.880,8.845,8.809,8.774/
C
      DATA ((EDEN(I,5,8,L,1),I=57,64),L=1,NZEN)/
     * 11.070,11.071,11.067,11.067,11.065,11.064,11.065,11.067,
     * 11.048,11.047,11.046,11.045,11.043,11.042,11.042,11.044,
     * 11.013,11.014,11.013,11.012,11.011,11.010,11.011,11.011,
     * 10.946,10.949,10.953,10.953,10.954,10.955,10.957,10.958,
     * 10.793,10.800,10.806,10.809,10.811,10.811,10.813,10.815,
     * 10.692,10.697,10.701,10.705,10.706,10.708,10.709,10.711,
     * 10.539,10.547,10.551,10.553,10.555,10.553,10.553,10.552,
     * 10.083,10.092,10.099,10.103,10.104,10.102,10.098,10.110,
     * 9.593,9.578,9.562,9.545,9.527,9.507,9.486,9.465,
     * 9.346,9.308,9.269,9.228,9.184,9.142,9.098,9.059,
     * 9.060,9.024,8.983,8.940,8.899,8.855,8.811,8.770/
C
      DATA ((EDEN(I,1,9,L,1),I=57,64),L=1,NZEN)/
     * 11.079,11.075,11.071,11.067,11.062,11.058,11.056,11.056,
     * 11.040,11.037,11.033,11.030,11.027,11.024,11.023,11.023,
     * 10.998,10.996,10.995,10.993,10.991,10.989,10.988,10.989,
     * 10.936,10.937,10.937,10.938,10.937,10.936,10.936,10.938,
     * 10.783,10.788,10.793,10.796,10.799,10.801,10.802,10.806,
     * 10.681,10.688,10.693,10.697,10.701,10.703,10.707,10.710,
     * 10.517,10.526,10.532,10.536,10.539,10.543,10.546,10.550,
     * 9.998,10.011,10.027,10.043,10.057,10.068,10.076,10.098,
     * 9.527,9.504,9.482,9.463,9.446,9.432,9.420,9.412,
     * 9.285,9.250,9.215,9.182,9.149,9.119,9.090,9.067,
     * 8.999,8.961,8.925,8.889,8.856,8.825,8.801,8.774/
C
      DATA ((EDEN(I,2,9,L,1),I=57,64),L=1,NZEN)/
     * 11.081,11.077,11.073,11.069,11.065,11.062,11.059,11.059,
     * 11.036,11.033,11.032,11.030,11.027,11.026,11.025,11.026,
     * 11.005,11.004,11.004,11.003,11.001,10.999,10.998,11.000,
     * 10.950,10.952,10.953,10.953,10.952,10.952,10.951,10.953,
     * 10.794,10.800,10.805,10.809,10.811,10.813,10.815,10.817,
     * 10.684,10.691,10.696,10.700,10.703,10.706,10.709,10.713,
     * 10.521,10.529,10.536,10.540,10.543,10.545,10.547,10.552,
     * 10.010,10.023,10.037,10.052,10.064,10.073,10.080,10.101,
     * 9.540,9.519,9.498,9.480,9.464,9.450,9.437,9.427,
     * 9.299,9.262,9.227,9.193,9.160,9.128,9.097,9.072,
     * 9.004,8.968,8.932,8.898,8.862,8.833,8.802,8.776/
C
      DATA ((EDEN(I,3,9,L,1),I=57,64),L=1,NZEN)/
     * 11.084,11.080,11.077,11.074,11.071,11.068,11.066,11.067,
     * 11.050,11.047,11.045,11.042,11.040,11.037,11.037,11.038,
     * 11.009,11.008,11.008,11.006,11.005,11.003,11.003,11.005,
     * 10.949,10.952,10.954,10.955,10.955,10.956,10.956,10.958,
     * 10.790,10.798,10.803,10.807,10.811,10.812,10.814,10.818,
     * 10.682,10.688,10.694,10.698,10.701,10.705,10.708,10.712,
     * 10.519,10.528,10.533,10.538,10.541,10.544,10.547,10.550,
     * 10.025,10.036,10.049,10.060,10.070,10.077,10.082,10.101,
     * 9.547,9.526,9.508,9.491,9.476,9.462,9.449,9.438,
     * 9.313,9.277,9.239,9.205,9.168,9.134,9.102,9.071,
     * 9.009,8.970,8.936,8.899,8.864,8.831,8.798,8.773/
C
      DATA ((EDEN(I,4,9,L,1),I=57,64),L=1,NZEN)/
     * 11.086,11.083,11.081,11.079,11.076,11.075,11.074,11.075,
     * 11.061,11.059,11.057,11.055,11.052,11.050,11.050,11.050,
     * 11.016,11.016,11.016,11.015,11.014,11.013,11.014,11.016,
     * 10.946,10.949,10.951,10.953,10.953,10.954,10.955,10.957,
     * 10.786,10.793,10.800,10.804,10.807,10.810,10.813,10.817,
     * 10.679,10.685,10.691,10.696,10.699,10.703,10.705,10.710,
     * 10.518,10.528,10.534,10.539,10.543,10.544,10.547,10.551,
     * 10.033,10.044,10.055,10.065,10.073,10.077,10.080,10.097,
     * 9.545,9.526,9.508,9.492,9.476,9.462,9.449,9.436,
     * 9.302,9.267,9.231,9.195,9.160,9.124,9.090,9.057,
     * 8.983,8.948,8.913,8.879,8.844,8.812,8.783,8.754/
C
      DATA ((EDEN(I,5,9,L,1),I=57,64),L=1,NZEN)/
     * 11.086,11.085,11.084,11.083,11.081,11.081,11.082,11.084,
     * 11.060,11.061,11.059,11.058,11.057,11.056,11.057,11.059,
     * 11.024,11.025,11.025,11.024,11.024,11.023,11.024,11.025,
     * 10.950,10.953,10.955,10.957,10.958,10.959,10.960,10.962,
     * 10.784,10.790,10.797,10.802,10.806,10.809,10.812,10.816,
     * 10.676,10.684,10.689,10.694,10.697,10.701,10.703,10.707,
     * 10.516,10.526,10.530,10.536,10.540,10.542,10.545,10.547,
     * 10.033,10.043,10.054,10.062,10.069,10.073,10.074,10.090,
     * 9.539,9.519,9.501,9.484,9.469,9.454,9.439,9.425,
     * 9.295,9.260,9.223,9.187,9.149,9.112,9.076,9.040,
     * 8.993,8.956,8.918,8.882,8.844,8.808,8.772,8.740/
C
      DATA ((EDEN(I,1,10,L,1),I=57,64),L=1,NZEN)/
     * 11.088,11.084,11.080,11.075,11.071,11.067,11.064,11.064,
     * 11.048,11.045,11.042,11.039,11.035,11.032,11.031,11.031,
     * 11.006,11.004,11.003,11.001,10.999,10.997,10.996,10.997,
     * 10.941,10.943,10.944,10.944,10.943,10.943,10.943,10.944,
     * 10.786,10.791,10.796,10.799,10.801,10.803,10.805,10.808,
     * 10.686,10.691,10.696,10.700,10.702,10.705,10.709,10.712,
     * 10.522,10.531,10.535,10.540,10.544,10.546,10.547,10.552,
     * 9.999,10.012,10.027,10.043,10.058,10.069,10.078,10.101,
     * 9.536,9.512,9.489,9.468,9.451,9.435,9.422,9.413,
     * 9.296,9.259,9.224,9.192,9.157,9.128,9.099,9.071,
     * 9.008,8.970,8.931,8.896,8.863,8.832,8.805,8.778/
C
      DATA ((EDEN(I,2,10,L,1),I=57,64),L=1,NZEN)/
     * 11.087,11.085,11.080,11.076,11.072,11.069,11.067,11.066,
     * 11.050,11.047,11.045,11.041,11.038,11.036,11.034,11.034,
     * 11.009,11.008,11.006,11.004,11.002,11.001,11.000,11.001,
     * 10.944,10.946,10.947,10.946,10.946,10.945,10.946,10.948,
     * 10.786,10.791,10.796,10.799,10.802,10.805,10.806,10.809,
     * 10.684,10.688,10.694,10.698,10.701,10.705,10.707,10.713,
     * 10.518,10.527,10.533,10.537,10.540,10.542,10.546,10.550,
     * 10.014,10.028,10.043,10.057,10.068,10.076,10.082,10.104,
     * 9.528,9.504,9.482,9.462,9.445,9.431,9.420,9.412,
     * 9.286,9.252,9.217,9.184,9.152,9.122,9.094,9.069,
     * 8.996,8.958,8.921,8.886,8.853,8.823,8.795,8.772/
C
      DATA ((EDEN(I,3,10,L,1),I=57,64),L=1,NZEN)/
     * 11.090,11.087,11.084,11.081,11.078,11.075,11.073,11.074,
     * 11.064,11.062,11.059,11.056,11.053,11.051,11.049,11.049,
     * 11.015,11.014,11.013,11.011,11.010,11.009,11.009,11.010,
     * 10.946,10.948,10.950,10.950,10.951,10.951,10.952,10.954,
     * 10.781,10.787,10.793,10.798,10.801,10.804,10.807,10.811,
     * 10.675,10.682,10.688,10.693,10.697,10.701,10.705,10.711,
     * 10.510,10.518,10.526,10.531,10.535,10.538,10.541,10.549,
     * 10.008,10.021,10.035,10.048,10.059,10.067,10.073,10.095,
     * 9.509,9.487,9.467,9.451,9.437,9.426,9.417,9.410,
     * 9.264,9.230,9.198,9.164,9.133,9.103,9.077,9.052,
     * 8.961,8.927,8.892,8.859,8.828,8.801,8.773,8.749/
C
      DATA ((EDEN(I,4,10,L,1),I=57,64),L=1,NZEN)/
     * 11.094,11.091,11.089,11.086,11.084,11.083,11.082,11.083,
     * 11.065,11.064,11.063,11.061,11.059,11.057,11.056,11.058,
     * 11.026,11.027,11.026,11.025,11.024,11.022,11.022,11.023,
     * 10.946,10.949,10.951,10.952,10.953,10.954,10.955,10.958,
     * 10.775,10.783,10.789,10.794,10.799,10.802,10.806,10.811,
     * 10.664,10.670,10.676,10.682,10.688,10.692,10.696,10.702,
     * 10.499,10.509,10.517,10.523,10.526,10.531,10.536,10.543,
     * 9.999,10.012,10.025,10.037,10.047,10.056,10.062,10.083,
     * 9.486,9.467,9.450,9.437,9.427,9.418,9.410,9.404,
     * 9.236,9.203,9.171,9.139,9.109,9.080,9.054,9.028,
     * 8.935,8.899,8.866,8.835,8.803,8.775,8.750,8.726/
C
      DATA ((EDEN(I,5,10,L,1),I=57,64),L=1,NZEN)/
     * 11.094,11.093,11.092,11.090,11.089,11.089,11.090,11.092,
     * 11.066,11.066,11.065,11.064,11.064,11.063,11.064,11.065,
     * 11.027,11.027,11.028,11.028,11.027,11.028,11.028,11.030,
     * 10.951,10.954,10.957,10.959,10.960,10.962,10.963,10.966,
     * 10.770,10.778,10.784,10.790,10.795,10.800,10.804,10.809,
     * 10.657,10.664,10.671,10.677,10.684,10.689,10.694,10.701,
     * 10.490,10.499,10.505,10.511,10.518,10.523,10.528,10.536,
     * 9.985,9.997,10.010,10.021,10.032,10.040,10.047,10.069,
     * 9.466,9.446,9.430,9.417,9.407,9.398,9.391,9.385,
     * 9.212,9.177,9.144,9.112,9.084,9.055,9.028,9.001,
     * 8.899,8.863,8.827,8.798,8.769,8.743,8.717,8.696/
C
      DATA ((EDEN(I,1,11,L,1),I=57,64),L=1,NZEN)/
     * 11.085,11.080,11.075,11.071,11.066,11.062,11.059,11.058,
     * 11.045,11.042,11.038,11.034,11.031,11.027,11.026,11.025,
     * 11.003,11.001,11.000,10.997,10.995,10.993,10.991,10.992,
     * 10.941,10.942,10.942,10.941,10.941,10.939,10.940,10.941,
     * 10.788,10.794,10.797,10.800,10.802,10.804,10.805,10.808,
     * 10.689,10.695,10.699,10.702,10.704,10.708,10.710,10.713,
     * 10.526,10.535,10.540,10.543,10.545,10.548,10.549,10.553,
     * 10.006,10.020,10.036,10.052,10.066,10.076,10.084,10.106,
     * 9.530,9.505,9.482,9.461,9.443,9.428,9.415,9.407,
     * 9.288,9.252,9.217,9.184,9.151,9.121,9.093,9.067,
     * 9.002,8.961,8.925,8.893,8.860,8.829,8.801,8.774/
C
      DATA ((EDEN(I,2,11,L,1),I=57,64),L=1,NZEN)/
     * 11.084,11.078,11.075,11.071,11.066,11.063,11.060,11.059,
     * 11.058,11.054,11.050,11.047,11.042,11.039,11.036,11.035,
     * 11.007,11.005,11.002,11.000,10.998,10.995,10.995,10.995,
     * 10.943,10.944,10.944,10.944,10.943,10.942,10.942,10.943,
     * 10.786,10.791,10.795,10.798,10.801,10.802,10.804,10.809,
     * 10.682,10.688,10.692,10.696,10.700,10.703,10.707,10.711,
     * 10.516,10.525,10.530,10.533,10.538,10.542,10.544,10.551,
     * 10.008,10.023,10.039,10.053,10.065,10.073,10.080,10.103,
     * 9.502,9.477,9.454,9.435,9.418,9.406,9.396,9.390,
     * 9.259,9.222,9.190,9.158,9.129,9.101,9.075,9.052,
     * 8.963,8.924,8.892,8.858,8.830,8.802,8.777,8.756/
C
      DATA ((EDEN(I,3,11,L,1),I=57,64),L=1,NZEN)/
     * 11.087,11.083,11.079,11.076,11.072,11.069,11.067,11.067,
     * 11.059,11.057,11.054,11.050,11.047,11.044,11.042,11.043,
     * 11.022,11.020,11.018,11.015,11.013,11.010,11.009,11.010,
     * 10.941,10.942,10.943,10.943,10.943,10.942,10.942,10.944,
     * 10.778,10.784,10.789,10.793,10.797,10.800,10.804,10.807,
     * 10.664,10.671,10.677,10.682,10.687,10.691,10.696,10.702,
     * 10.500,10.509,10.514,10.521,10.527,10.533,10.538,10.545,
     * 9.989,10.003,10.018,10.033,10.045,10.055,10.063,10.087,
     * 9.455,9.434,9.417,9.403,9.394,9.388,9.385,9.385,
     * 9.214,9.180,9.149,9.120,9.092,9.068,9.046,9.024,
     * 8.911,8.875,8.842,8.815,8.789,8.763,8.744,8.726/
C
      DATA ((EDEN(I,4,11,L,1),I=57,64),L=1,NZEN)/
     * 11.091,11.088,11.085,11.083,11.079,11.077,11.077,11.078,
     * 11.062,11.061,11.058,11.056,11.053,11.052,11.052,11.052,
     * 11.023,11.023,11.021,11.019,11.018,11.017,11.016,11.017,
     * 10.948,10.950,10.951,10.951,10.951,10.952,10.953,10.955,
     * 10.766,10.772,10.778,10.783,10.788,10.792,10.797,10.802,
     * 10.649,10.657,10.663,10.669,10.676,10.682,10.688,10.696,
     * 10.481,10.488,10.496,10.503,10.512,10.517,10.527,10.535,
     * 9.967,9.980,9.995,10.009,10.022,10.033,10.043,10.069,
     * 9.417,9.399,9.387,9.379,9.375,9.373,9.373,9.374,
     * 9.155,9.123,9.095,9.067,9.045,9.024,9.003,8.986,
     * 8.854,8.819,8.788,8.762,8.737,8.720,8.699,8.683/
C
      DATA ((EDEN(I,5,11,L,1),I=57,64),L=1,NZEN)/
     * 11.094,11.091,11.089,11.087,11.086,11.085,11.085,11.087,
     * 11.063,11.062,11.061,11.060,11.059,11.058,11.059,11.061,
     * 11.022,11.023,11.022,11.022,11.022,11.021,11.023,11.026,
     * 10.945,10.948,10.950,10.952,10.953,10.955,10.957,10.961,
     * 10.761,10.767,10.774,10.780,10.785,10.790,10.796,10.802,
     * 10.639,10.646,10.653,10.660,10.667,10.675,10.683,10.691,
     * 10.465,10.474,10.481,10.490,10.498,10.506,10.515,10.524,
     * 9.943,9.956,9.971,9.985,9.999,10.012,10.023,10.050,
     * 9.384,9.367,9.355,9.348,9.345,9.345,9.346,9.348,
     * 9.114,9.080,9.052,9.027,9.005,8.983,8.967,8.951,
     * 8.798,8.764,8.736,8.711,8.690,8.670,8.655,8.645/
C
      DATA ((EDEN(I,1,12,L,1),I=57,64),L=1,NZEN)/
     * 11.077,11.072,11.067,11.062,11.057,11.053,11.050,11.050,
     * 11.038,11.034,11.030,11.026,11.022,11.019,11.017,11.017,
     * 10.998,10.995,10.993,10.990,10.988,10.985,10.984,10.984,
     * 10.939,10.939,10.938,10.937,10.936,10.934,10.934,10.935,
     * 10.792,10.797,10.799,10.802,10.803,10.804,10.804,10.807,
     * 10.692,10.698,10.702,10.705,10.709,10.710,10.712,10.715,
     * 10.531,10.537,10.543,10.546,10.548,10.551,10.554,10.558,
     * 10.014,10.029,10.046,10.061,10.073,10.082,10.089,10.111,
     * 9.519,9.494,9.471,9.450,9.432,9.418,9.406,9.399,
     * 9.275,9.237,9.205,9.172,9.140,9.110,9.083,9.059,
     * 8.988,8.952,8.915,8.883,8.849,8.822,8.794,8.771/
C
      DATA ((EDEN(I,2,12,L,1),I=57,64),L=1,NZEN)/
     * 11.077,11.071,11.066,11.062,11.057,11.053,11.050,11.049,
     * 11.052,11.047,11.042,11.038,11.034,11.030,11.027,11.027,
     * 11.006,11.003,10.999,10.996,10.993,10.990,10.988,10.988,
     * 10.941,10.941,10.940,10.939,10.938,10.936,10.936,10.938,
     * 10.788,10.791,10.795,10.798,10.800,10.801,10.804,10.806,
     * 10.676,10.681,10.685,10.690,10.693,10.697,10.701,10.706,
     * 10.515,10.521,10.528,10.532,10.537,10.541,10.547,10.553,
     * 10.008,10.024,10.039,10.053,10.064,10.073,10.080,10.103,
     * 9.465,9.442,9.421,9.405,9.393,9.385,9.380,9.379,
     * 9.229,9.194,9.163,9.134,9.108,9.080,9.058,9.038,
     * 8.931,8.896,8.863,8.833,8.806,8.783,8.763,8.745/
C
      DATA ((EDEN(I,3,12,L,1),I=57,64),L=1,NZEN)/
     * 11.079,11.076,11.071,11.067,11.064,11.060,11.058,11.058,
     * 11.054,11.050,11.046,11.043,11.039,11.036,11.034,11.034,
     * 11.017,11.014,11.011,11.009,11.006,11.003,11.002,11.002,
     * 10.939,10.938,10.938,10.937,10.937,10.935,10.936,10.938,
     * 10.773,10.778,10.781,10.785,10.788,10.791,10.794,10.799,
     * 10.662,10.668,10.674,10.679,10.685,10.691,10.696,10.702,
     * 10.494,10.500,10.507,10.512,10.519,10.526,10.532,10.541,
     * 9.976,9.991,10.007,10.022,10.035,10.046,10.056,10.082,
     * 9.414,9.393,9.377,9.366,9.359,9.357,9.358,9.362,
     * 9.154,9.124,9.097,9.070,9.048,9.028,9.010,8.996,
     * 8.866,8.832,8.800,8.776,8.752,8.732,8.716,8.702/
C
      DATA ((EDEN(I,4,12,L,1),I=57,64),L=1,NZEN)/
     * 11.087,11.082,11.078,11.075,11.072,11.070,11.069,11.070,
     * 11.059,11.055,11.052,11.049,11.046,11.044,11.043,11.045,
     * 11.019,11.017,11.015,11.012,11.010,11.009,11.009,11.011,
     * 10.945,10.946,10.946,10.946,10.946,10.946,10.947,10.949,
     * 10.763,10.768,10.773,10.777,10.782,10.786,10.791,10.796,
     * 10.644,10.650,10.658,10.665,10.672,10.679,10.685,10.693,
     * 10.471,10.477,10.485,10.493,10.502,10.509,10.519,10.530,
     * 9.955,9.968,9.983,9.997,10.010,10.022,10.033,10.061,
     * 9.369,9.353,9.344,9.340,9.341,9.344,9.349,9.355,
     * 9.100,9.069,9.043,9.020,9.000,8.983,8.969,8.958,
     * 8.794,8.759,8.731,8.711,8.690,8.677,8.665,8.655/
C
      DATA ((EDEN(I,5,12,L,1),I=57,64),L=1,NZEN)/
     * 11.089,11.087,11.084,11.081,11.079,11.078,11.078,11.080,
     * 11.061,11.057,11.055,11.054,11.052,11.051,11.051,11.053,
     * 11.020,11.018,11.017,11.016,11.014,11.014,11.016,11.018,
     * 10.942,10.943,10.945,10.946,10.946,10.948,10.950,10.955,
     * 10.755,10.761,10.767,10.773,10.779,10.784,10.790,10.798,
     * 10.631,10.637,10.645,10.653,10.661,10.670,10.679,10.686,
     * 10.449,10.457,10.465,10.476,10.484,10.494,10.506,10.519,
     * 9.916,9.929,9.945,9.962,9.977,9.992,10.005,10.036,
     * 9.331,9.317,9.309,9.307,9.309,9.314,9.320,9.327,
     * 9.044,9.013,8.986,8.965,8.947,8.933,8.922,8.914,
     * 8.724,8.688,8.663,8.642,8.626,8.617,8.608,8.605/
C
      DATA ((EDEN(I,1,1,L,2),I=57,64),L=1,NZEN)/
     * 11.146,11.141,11.137,11.134,11.130,11.128,11.127,11.128,
     * 11.107,11.105,11.102,11.099,11.098,11.096,11.096,11.097,
     * 11.068,11.066,11.066,11.065,11.064,11.064,11.064,11.066,
     * 11.011,11.012,11.013,11.014,11.016,11.016,11.017,11.021,
     * 10.868,10.875,10.880,10.885,10.889,10.893,10.897,10.901,
     * 10.771,10.779,10.786,10.792,10.799,10.803,10.809,10.814,
     * 10.613,10.622,10.629,10.636,10.641,10.647,10.654,10.661,
     * 10.140,10.154,10.169,10.184,10.196,10.205,10.213,10.239,
     * 9.601,9.585,9.573,9.565,9.560,9.559,9.560,9.564,
     * 9.352,9.324,9.298,9.272,9.250,9.226,9.205,9.187,
     * 9.073,9.045,9.016,8.990,8.967,8.945,8.926,8.911/
C
      DATA ((EDEN(I,2,1,L,2),I=57,64),L=1,NZEN)/
     * 11.144,11.141,11.137,11.134,11.130,11.128,11.127,11.128,
     * 11.120,11.118,11.114,11.111,11.109,11.106,11.106,11.106,
     * 11.075,11.075,11.073,11.071,11.069,11.068,11.068,11.070,
     * 11.011,11.013,11.014,11.015,11.016,11.016,11.019,11.022,
     * 10.862,10.868,10.874,10.879,10.884,10.887,10.893,10.899,
     * 10.753,10.760,10.768,10.775,10.782,10.787,10.795,10.802,
     * 10.595,10.604,10.613,10.620,10.629,10.636,10.644,10.653,
     * 10.127,10.141,10.156,10.169,10.181,10.191,10.200,10.227,
     * 9.543,9.528,9.519,9.515,9.516,9.521,9.530,9.542,
     * 9.303,9.275,9.252,9.230,9.209,9.191,9.174,9.160,
     * 9.012,8.985,8.960,8.940,8.921,8.903,8.889,8.875/
C
      DATA ((EDEN(I,3,1,L,2),I=57,64),L=1,NZEN)/
     * 11.148,11.144,11.142,11.138,11.137,11.134,11.134,11.135,
     * 11.123,11.120,11.118,11.116,11.113,11.111,11.111,11.113,
     * 11.085,11.085,11.082,11.082,11.081,11.080,11.081,11.083,
     * 11.011,11.013,11.014,11.015,11.016,11.017,11.019,11.022,
     * 10.845,10.852,10.858,10.863,10.870,10.875,10.881,10.888,
     * 10.736,10.745,10.752,10.761,10.769,10.777,10.785,10.795,
     * 10.571,10.579,10.587,10.597,10.607,10.615,10.627,10.638,
     * 10.090,10.103,10.118,10.132,10.146,10.158,10.170,10.200,
     * 9.488,9.477,9.473,9.475,9.482,9.493,9.506,9.521,
     * 9.224,9.200,9.180,9.162,9.147,9.134,9.122,9.113,
     * 8.938,8.911,8.889,8.872,8.858,8.844,8.833,8.827/
C
      DATA ((EDEN(I,4,1,L,2),I=57,64),L=1,NZEN)/
     * 11.159,11.156,11.154,11.152,11.149,11.149,11.149,11.151,
     * 11.132,11.130,11.127,11.127,11.125,11.124,11.125,11.127,
     * 11.094,11.092,11.090,11.091,11.091,11.090,11.092,11.095,
     * 11.019,11.021,11.024,11.025,11.027,11.028,11.032,11.036,
     * 10.838,10.845,10.851,10.858,10.865,10.871,10.879,10.886,
     * 10.719,10.728,10.737,10.747,10.756,10.765,10.775,10.785,
     * 10.546,10.556,10.566,10.576,10.587,10.599,10.612,10.625,
     * 10.047,10.061,10.076,10.092,10.107,10.121,10.135,10.167,
     * 9.430,9.423,9.424,9.429,9.439,9.451,9.463,9.476,
     * 9.150,9.125,9.104,9.087,9.075,9.063,9.053,9.047,
     * 8.843,8.816,8.796,8.778,8.766,8.757,8.753,8.750/
C
      DATA ((EDEN(I,5,1,L,2),I=57,64),L=1,NZEN)/
     * 11.167,11.165,11.164,11.161,11.160,11.160,11.161,11.164,
     * 11.139,11.138,11.136,11.134,11.134,11.134,11.135,11.139,
     * 11.097,11.098,11.096,11.096,11.097,11.098,11.101,11.105,
     * 11.020,11.024,11.025,11.027,11.030,11.033,11.037,11.042,
     * 10.833,10.840,10.848,10.855,10.863,10.870,10.878,10.888,
     * 10.707,10.716,10.726,10.735,10.745,10.756,10.766,10.778,
     * 10.525,10.535,10.545,10.556,10.568,10.582,10.595,10.610,
     * 9.988,10.005,10.024,10.043,10.061,10.078,10.095,10.130,
     * 9.367,9.357,9.354,9.356,9.363,9.373,9.383,9.396,
     * 9.071,9.045,9.022,9.007,8.993,8.985,8.980,8.976,
     * 8.750,8.719,8.697,8.682,8.674,8.667,8.665,8.668/
C
      DATA ((EDEN(I,1,2,L,2),I=57,64),L=1,NZEN)/
     * 11.152,11.148,11.144,11.140,11.137,11.135,11.134,11.134,
     * 11.113,11.110,11.109,11.106,11.104,11.102,11.102,11.104,
     * 11.073,11.073,11.072,11.071,11.070,11.070,11.070,11.073,
     * 11.014,11.016,11.018,11.019,11.020,11.020,11.022,11.025,
     * 10.868,10.875,10.881,10.886,10.890,10.894,10.898,10.903,
     * 10.769,10.777,10.785,10.792,10.797,10.803,10.808,10.814,
     * 10.611,10.620,10.627,10.634,10.639,10.645,10.651,10.660,
     * 10.133,10.147,10.162,10.177,10.190,10.200,10.209,10.235,
     * 9.608,9.592,9.580,9.571,9.565,9.562,9.562,9.565,
     * 9.361,9.334,9.305,9.281,9.255,9.232,9.212,9.191,
     * 9.080,9.050,9.024,8.997,8.973,8.952,8.930,8.911/
C
      DATA ((EDEN(I,2,2,L,2),I=57,64),L=1,NZEN)/
     * 11.151,11.148,11.145,11.141,11.139,11.136,11.135,11.136,
     * 11.127,11.125,11.123,11.119,11.117,11.115,11.113,11.115,
     * 11.076,11.075,11.075,11.074,11.073,11.072,11.073,11.076,
     * 11.014,11.017,11.018,11.020,11.021,11.022,11.024,11.028,
     * 10.861,10.868,10.874,10.880,10.885,10.890,10.896,10.901,
     * 10.752,10.759,10.767,10.775,10.782,10.789,10.796,10.802,
     * 10.594,10.603,10.612,10.620,10.627,10.635,10.643,10.651,
     * 10.125,10.139,10.153,10.167,10.179,10.189,10.198,10.225,
     * 9.566,9.550,9.537,9.529,9.525,9.526,9.530,9.537,
     * 9.319,9.290,9.266,9.244,9.222,9.202,9.185,9.170,
     * 9.029,9.001,8.976,8.954,8.934,8.913,8.898,8.887/
C
      DATA ((EDEN(I,3,2,L,2),I=57,64),L=1,NZEN)/
     * 11.156,11.154,11.150,11.148,11.145,11.143,11.143,11.144,
     * 11.130,11.127,11.126,11.124,11.122,11.120,11.120,11.122,
     * 11.093,11.093,11.090,11.090,11.089,11.088,11.089,11.090,
     * 11.012,11.014,11.016,11.018,11.019,11.021,11.024,11.028,
     * 10.845,10.853,10.860,10.866,10.872,10.878,10.883,10.891,
     * 10.736,10.744,10.753,10.762,10.770,10.778,10.787,10.796,
     * 10.571,10.580,10.589,10.598,10.607,10.616,10.625,10.638,
     * 10.095,10.107,10.121,10.136,10.149,10.161,10.172,10.201,
     * 9.513,9.501,9.495,9.495,9.500,9.509,9.520,9.532,
     * 9.263,9.238,9.215,9.194,9.176,9.160,9.145,9.133,
     * 8.963,8.936,8.911,8.893,8.874,8.862,8.847,8.839/
C
      DATA ((EDEN(I,4,2,L,2),I=57,64),L=1,NZEN)/
     * 11.165,11.163,11.161,11.159,11.158,11.156,11.157,11.160,
     * 11.138,11.136,11.135,11.135,11.132,11.132,11.133,11.135,
     * 11.098,11.097,11.099,11.099,11.097,11.098,11.099,11.102,
     * 11.024,11.026,11.028,11.031,11.032,11.034,11.037,11.042,
     * 10.839,10.847,10.854,10.862,10.868,10.875,10.882,10.890,
     * 10.720,10.730,10.739,10.748,10.757,10.767,10.777,10.787,
     * 10.550,10.560,10.570,10.581,10.591,10.603,10.613,10.627,
     * 10.052,10.066,10.082,10.098,10.113,10.127,10.140,10.171,
     * 9.458,9.448,9.446,9.448,9.455,9.464,9.474,9.484,
     * 9.182,9.156,9.135,9.117,9.102,9.088,9.077,9.069,
     * 8.877,8.850,8.826,8.810,8.796,8.782,8.775,8.770/
C
      DATA ((EDEN(I,5,2,L,2),I=57,64),L=1,NZEN)/
     * 11.170,11.172,11.169,11.169,11.166,11.167,11.169,11.172,
     * 11.144,11.142,11.141,11.141,11.141,11.141,11.143,11.146,
     * 11.101,11.102,11.102,11.102,11.104,11.105,11.108,11.112,
     * 11.023,11.026,11.029,11.033,11.035,11.039,11.043,11.049,
     * 10.835,10.843,10.851,10.858,10.866,10.874,10.882,10.891,
     * 10.710,10.718,10.727,10.737,10.748,10.757,10.769,10.780,
     * 10.530,10.540,10.550,10.560,10.573,10.585,10.597,10.612,
     * 9.999,10.016,10.034,10.052,10.070,10.086,10.101,10.135,
     * 9.401,9.389,9.383,9.384,9.388,9.395,9.403,9.413,
     * 9.117,9.087,9.066,9.046,9.031,9.018,9.010,9.001,
     * 8.793,8.762,8.739,8.722,8.711,8.701,8.696,8.692/
C
      DATA ((EDEN(I,1,3,L,2),I=57,64),L=1,NZEN)/
     * 11.156,11.153,11.151,11.148,11.144,11.141,11.140,11.141,
     * 11.118,11.115,11.115,11.112,11.110,11.108,11.108,11.110,
     * 11.076,11.077,11.076,11.075,11.075,11.075,11.075,11.078,
     * 11.015,11.017,11.020,11.021,11.023,11.024,11.025,11.029,
     * 10.863,10.871,10.878,10.883,10.888,10.892,10.897,10.901,
     * 10.765,10.773,10.780,10.787,10.792,10.798,10.804,10.810,
     * 10.606,10.616,10.624,10.630,10.636,10.640,10.647,10.655,
     * 10.126,10.139,10.154,10.169,10.183,10.194,10.203,10.229,
     * 9.615,9.600,9.587,9.578,9.572,9.568,9.568,9.569,
     * 9.370,9.340,9.312,9.288,9.261,9.238,9.216,9.196,
     * 9.088,9.059,9.030,9.003,8.976,8.955,8.933,8.915/
C
      DATA ((EDEN(I,2,3,L,2),I=57,64),L=1,NZEN)/
     * 11.158,11.157,11.152,11.150,11.148,11.145,11.144,11.145,
     * 11.122,11.119,11.118,11.116,11.114,11.113,11.113,11.114,
     * 11.080,11.079,11.080,11.079,11.079,11.079,11.080,11.082,
     * 11.015,11.018,11.020,11.023,11.024,11.026,11.028,11.031,
     * 10.858,10.866,10.873,10.879,10.885,10.889,10.894,10.900,
     * 10.756,10.764,10.772,10.779,10.786,10.792,10.799,10.808,
     * 10.594,10.604,10.613,10.621,10.626,10.634,10.641,10.650,
     * 10.128,10.142,10.156,10.169,10.181,10.191,10.199,10.224,
     * 9.589,9.573,9.562,9.554,9.550,9.549,9.551,9.555,
     * 9.344,9.315,9.289,9.264,9.242,9.221,9.200,9.182,
     * 9.058,9.028,9.000,8.976,8.951,8.932,8.912,8.899/
C
      DATA ((EDEN(I,3,3,L,2),I=57,64),L=1,NZEN)/
     * 11.161,11.160,11.158,11.155,11.153,11.151,11.151,11.153,
     * 11.135,11.134,11.132,11.130,11.130,11.128,11.128,11.130,
     * 11.089,11.090,11.090,11.089,11.089,11.089,11.091,11.093,
     * 11.015,11.018,11.021,11.024,11.027,11.030,11.033,11.038,
     * 10.848,10.856,10.864,10.872,10.878,10.884,10.891,10.898,
     * 10.737,10.745,10.753,10.762,10.769,10.778,10.785,10.794,
     * 10.577,10.588,10.596,10.605,10.612,10.621,10.630,10.640,
     * 10.108,10.120,10.133,10.146,10.159,10.169,10.179,10.206,
     * 9.556,9.543,9.535,9.531,9.531,9.535,9.540,9.547,
     * 9.305,9.278,9.253,9.230,9.208,9.189,9.172,9.155,
     * 9.008,8.981,8.955,8.931,8.910,8.891,8.875,8.860/
C
      DATA ((EDEN(I,4,3,L,2),I=57,64),L=1,NZEN)/
     * 11.169,11.167,11.165,11.165,11.163,11.163,11.164,11.167,
     * 11.140,11.140,11.139,11.138,11.138,11.138,11.139,11.142,
     * 11.100,11.101,11.102,11.102,11.103,11.104,11.106,11.108,
     * 11.021,11.025,11.028,11.031,11.034,11.036,11.040,11.044,
     * 10.840,10.848,10.856,10.864,10.871,10.878,10.885,10.894,
     * 10.726,10.734,10.744,10.753,10.762,10.771,10.780,10.790,
     * 10.559,10.568,10.578,10.587,10.595,10.604,10.615,10.625,
     * 10.069,10.083,10.097,10.112,10.125,10.138,10.149,10.177,
     * 9.507,9.496,9.490,9.490,9.492,9.497,9.502,9.509,
     * 9.249,9.223,9.199,9.177,9.156,9.137,9.120,9.105,
     * 8.941,8.914,8.888,8.865,8.846,8.830,8.811,8.800/
C
      DATA ((EDEN(I,5,3,L,2),I=57,64),L=1,NZEN)/
     * 11.170,11.172,11.172,11.171,11.173,11.173,11.175,11.179,
     * 11.143,11.144,11.145,11.146,11.145,11.147,11.149,11.153,
     * 11.101,11.104,11.106,11.106,11.109,11.111,11.114,11.118,
     * 11.024,11.028,11.032,11.036,11.040,11.044,11.048,11.053,
     * 10.835,10.844,10.853,10.861,10.869,10.877,10.884,10.893,
     * 10.716,10.725,10.734,10.743,10.753,10.763,10.772,10.782,
     * 10.543,10.553,10.563,10.573,10.583,10.593,10.603,10.615,
     * 10.030,10.045,10.061,10.077,10.092,10.105,10.118,10.147,
     * 9.462,9.448,9.440,9.436,9.436,9.439,9.442,9.446,
     * 9.194,9.164,9.139,9.117,9.095,9.078,9.060,9.047,
     * 8.867,8.835,8.811,8.790,8.772,8.755,8.743,8.731/
C
      DATA ((EDEN(I,1,4,L,2),I=57,64),L=1,NZEN)/
     * 11.152,11.149,11.146,11.143,11.139,11.137,11.136,11.137,
     * 11.113,11.111,11.110,11.108,11.106,11.104,11.104,11.105,
     * 11.072,11.072,11.071,11.071,11.070,11.070,11.071,11.073,
     * 11.009,11.012,11.014,11.016,11.017,11.018,11.020,11.023,
     * 10.857,10.865,10.871,10.877,10.882,10.886,10.891,10.895,
     * 10.759,10.767,10.774,10.780,10.787,10.792,10.798,10.804,
     * 10.602,10.612,10.618,10.625,10.630,10.636,10.641,10.648,
     * 10.124,10.136,10.151,10.166,10.179,10.190,10.200,10.225,
     * 9.614,9.599,9.588,9.580,9.575,9.573,9.571,9.573,
     * 9.367,9.339,9.310,9.285,9.259,9.234,9.214,9.193,
     * 9.086,9.056,9.027,9.000,8.976,8.952,8.930,8.911/
C
      DATA ((EDEN(I,2,4,L,2),I=57,64),L=1,NZEN)/
     * 11.155,11.153,11.150,11.148,11.145,11.142,11.142,11.143,
     * 11.118,11.116,11.116,11.113,11.112,11.111,11.110,11.113,
     * 11.073,11.073,11.075,11.075,11.075,11.075,11.076,11.080,
     * 11.009,11.012,11.015,11.018,11.020,11.022,11.024,11.028,
     * 10.863,10.871,10.878,10.885,10.890,10.894,10.900,10.905,
     * 10.754,10.763,10.770,10.778,10.784,10.790,10.796,10.803,
     * 10.598,10.608,10.614,10.622,10.627,10.631,10.639,10.646,
     * 10.126,10.137,10.151,10.165,10.177,10.188,10.196,10.222,
     * 9.612,9.599,9.589,9.583,9.580,9.578,9.578,9.578,
     * 9.364,9.335,9.307,9.283,9.257,9.233,9.211,9.190,
     * 9.081,9.051,9.024,8.996,8.972,8.946,8.927,8.908/
C
      DATA ((EDEN(I,3,4,L,2),I=57,64),L=1,NZEN)/
     * 11.157,11.156,11.154,11.152,11.150,11.149,11.150,11.151,
     * 11.132,11.130,11.129,11.129,11.127,11.126,11.126,11.128,
     * 11.078,11.080,11.082,11.083,11.083,11.085,11.086,11.090,
     * 11.009,11.013,11.017,11.021,11.024,11.026,11.030,11.034,
     * 10.846,10.855,10.862,10.870,10.876,10.882,10.887,10.895,
     * 10.745,10.753,10.760,10.769,10.776,10.783,10.789,10.797,
     * 10.588,10.597,10.605,10.612,10.618,10.624,10.630,10.638,
     * 10.125,10.135,10.147,10.159,10.170,10.179,10.186,10.211,
     * 9.599,9.589,9.582,9.579,9.578,9.578,9.579,9.580,
     * 9.347,9.319,9.295,9.269,9.244,9.219,9.197,9.175,
     * 9.057,9.027,9.001,8.973,8.947,8.923,8.902,8.885/
C
      DATA ((EDEN(I,4,4,L,2),I=57,64),L=1,NZEN)/
     * 11.161,11.160,11.160,11.159,11.160,11.159,11.160,11.163,
     * 11.135,11.135,11.135,11.135,11.134,11.135,11.136,11.139,
     * 11.090,11.093,11.095,11.096,11.097,11.098,11.100,11.104,
     * 11.013,11.018,11.023,11.027,11.031,11.035,11.039,11.044,
     * 10.842,10.851,10.859,10.867,10.874,10.881,10.887,10.894,
     * 10.736,10.744,10.752,10.760,10.768,10.775,10.783,10.791,
     * 10.577,10.586,10.595,10.601,10.607,10.613,10.620,10.630,
     * 10.106,10.117,10.129,10.141,10.151,10.159,10.167,10.191,
     * 9.571,9.561,9.554,9.550,9.549,9.548,9.547,9.547,
     * 9.322,9.293,9.267,9.240,9.213,9.187,9.161,9.140,
     * 9.013,8.985,8.958,8.929,8.904,8.880,8.856,8.836/
C
      DATA ((EDEN(I,5,4,L,2),I=57,64),L=1,NZEN)/
     * 11.162,11.163,11.165,11.166,11.167,11.168,11.171,11.175,
     * 11.134,11.137,11.137,11.140,11.141,11.143,11.146,11.149,
     * 11.094,11.098,11.101,11.104,11.106,11.108,11.111,11.115,
     * 11.017,11.023,11.028,11.033,11.037,11.041,11.045,11.050,
     * 10.839,10.848,10.857,10.865,10.873,10.880,10.887,10.894,
     * 10.729,10.738,10.746,10.754,10.762,10.769,10.776,10.785,
     * 10.566,10.576,10.585,10.592,10.598,10.606,10.612,10.620,
     * 10.081,10.093,10.106,10.118,10.128,10.137,10.143,10.167,
     * 9.540,9.527,9.517,9.511,9.506,9.503,9.500,9.496,
     * 9.284,9.253,9.226,9.198,9.169,9.143,9.116,9.092,
     * 8.970,8.940,8.909,8.880,8.853,8.826,8.803,8.783/
C
      DATA ((EDEN(I,1,5,L,2),I=57,64),L=1,NZEN)/
     * 11.136,11.133,11.130,11.126,11.124,11.121,11.121,11.121,
     * 11.098,11.097,11.094,11.092,11.090,11.089,11.088,11.090,
     * 11.057,11.057,11.057,11.057,11.056,11.056,11.056,11.059,
     * 10.998,11.000,11.002,11.004,11.005,11.006,11.008,11.011,
     * 10.852,10.859,10.866,10.871,10.876,10.879,10.882,10.888,
     * 10.755,10.763,10.771,10.777,10.782,10.788,10.793,10.799,
     * 10.599,10.609,10.614,10.622,10.628,10.632,10.638,10.646,
     * 10.128,10.141,10.156,10.170,10.182,10.192,10.200,10.225,
     * 9.601,9.589,9.580,9.575,9.572,9.571,9.571,9.573,
     * 9.350,9.321,9.295,9.270,9.246,9.222,9.201,9.180,
     * 9.070,9.042,9.016,8.987,8.964,8.942,8.923,8.904/
C
      DATA ((EDEN(I,2,5,L,2),I=57,64),L=1,NZEN)/
     * 11.142,11.139,11.136,11.132,11.130,11.128,11.128,11.129,
     * 11.093,11.094,11.092,11.092,11.092,11.092,11.093,11.095,
     * 11.062,11.063,11.064,11.065,11.066,11.066,11.067,11.070,
     * 11.010,11.013,11.017,11.018,11.020,11.021,11.023,11.026,
     * 10.862,10.869,10.876,10.882,10.887,10.891,10.895,10.900,
     * 10.756,10.764,10.771,10.778,10.784,10.789,10.795,10.800,
     * 10.603,10.612,10.619,10.624,10.629,10.634,10.639,10.647,
     * 10.144,10.156,10.168,10.180,10.190,10.198,10.204,10.228,
     * 9.625,9.616,9.611,9.607,9.604,9.602,9.600,9.599,
     * 9.367,9.340,9.313,9.286,9.259,9.235,9.210,9.188,
     * 9.085,9.058,9.029,9.001,8.978,8.953,8.931,8.908/
C
      DATA ((EDEN(I,3,5,L,2),I=57,64),L=1,NZEN)/
     * 11.144,11.142,11.141,11.139,11.137,11.135,11.136,11.138,
     * 11.109,11.109,11.109,11.108,11.107,11.107,11.108,11.110,
     * 11.067,11.070,11.071,11.073,11.074,11.076,11.078,11.081,
     * 11.007,11.012,11.016,11.020,11.022,11.024,11.028,11.032,
     * 10.854,10.863,10.870,10.876,10.882,10.887,10.891,10.896,
     * 10.751,10.759,10.765,10.772,10.778,10.783,10.789,10.795,
     * 10.603,10.613,10.620,10.625,10.629,10.633,10.637,10.644,
     * 10.160,10.170,10.179,10.188,10.195,10.200,10.203,10.224,
     * 9.638,9.632,9.629,9.626,9.625,9.622,9.618,9.614,
     * 9.387,9.357,9.330,9.299,9.270,9.241,9.214,9.188,
     * 9.081,9.055,9.028,9.000,8.972,8.946,8.920,8.897/
C
      DATA ((EDEN(I,4,5,L,2),I=57,64),L=1,NZEN)/
     * 11.146,11.146,11.146,11.145,11.144,11.144,11.146,11.149,
     * 11.121,11.123,11.121,11.121,11.122,11.122,11.123,11.125,
     * 11.077,11.078,11.081,11.082,11.085,11.086,11.089,11.093,
     * 11.009,11.015,11.020,11.024,11.028,11.032,11.035,11.040,
     * 10.849,10.857,10.865,10.873,10.879,10.884,10.889,10.895,
     * 10.750,10.759,10.766,10.771,10.778,10.783,10.788,10.794,
     * 10.597,10.607,10.613,10.618,10.622,10.625,10.630,10.635,
     * 10.154,10.163,10.172,10.179,10.184,10.188,10.189,10.209,
     * 9.626,9.619,9.613,9.609,9.605,9.600,9.593,9.586,
     * 9.372,9.345,9.315,9.282,9.250,9.219,9.188,9.160,
     * 9.061,9.033,9.004,8.974,8.944,8.916,8.888,8.860/
C
      DATA ((EDEN(I,5,5,L,2),I=57,64),L=1,NZEN)/
     * 11.144,11.147,11.148,11.149,11.151,11.153,11.156,11.159,
     * 11.121,11.123,11.125,11.125,11.128,11.130,11.132,11.135,
     * 11.085,11.087,11.090,11.092,11.095,11.096,11.099,11.102,
     * 11.011,11.017,11.022,11.026,11.031,11.034,11.038,11.043,
     * 10.847,10.856,10.864,10.871,10.877,10.882,10.887,10.893,
     * 10.746,10.753,10.761,10.767,10.772,10.778,10.783,10.788,
     * 10.596,10.604,10.610,10.615,10.619,10.622,10.626,10.630,
     * 10.143,10.153,10.161,10.167,10.171,10.173,10.173,10.191,
     * 9.610,9.601,9.594,9.587,9.580,9.572,9.562,9.550,
     * 9.354,9.323,9.292,9.257,9.223,9.188,9.153,9.122,
     * 9.059,9.027,8.994,8.961,8.926,8.890,8.856,8.826/
C
      DATA ((EDEN(I,1,6,L,2),I=57,64),L=1,NZEN)/
     * 11.119,11.115,11.114,11.110,11.107,11.104,11.103,11.104,
     * 11.081,11.079,11.077,11.075,11.074,11.072,11.072,11.074,
     * 11.042,11.041,11.041,11.041,11.040,11.040,11.041,11.043,
     * 10.987,10.989,10.991,10.992,10.993,10.994,10.996,10.998,
     * 10.850,10.856,10.861,10.867,10.870,10.874,10.878,10.881,
     * 10.754,10.763,10.770,10.776,10.782,10.787,10.792,10.798,
     * 10.599,10.607,10.613,10.620,10.626,10.633,10.638,10.646,
     * 10.135,10.148,10.162,10.175,10.186,10.194,10.201,10.227,
     * 9.588,9.578,9.571,9.568,9.568,9.568,9.570,9.573,
     * 9.331,9.304,9.280,9.255,9.231,9.209,9.188,9.171,
     * 9.054,9.025,9.001,8.975,8.954,8.932,8.912,8.897/
C
      DATA ((EDEN(I,2,6,L,2),I=57,64),L=1,NZEN)/
     * 11.126,11.122,11.120,11.117,11.114,11.112,11.111,11.113,
     * 11.078,11.076,11.076,11.076,11.075,11.075,11.076,11.079,
     * 11.048,11.049,11.050,11.050,11.051,11.052,11.053,11.055,
     * 11.002,11.004,11.007,11.009,11.010,11.011,11.012,11.015,
     * 10.863,10.870,10.876,10.881,10.885,10.889,10.892,10.896,
     * 10.759,10.767,10.774,10.780,10.785,10.791,10.794,10.800,
     * 10.611,10.620,10.628,10.633,10.638,10.642,10.646,10.652,
     * 10.165,10.176,10.188,10.198,10.205,10.210,10.214,10.236,
     * 9.640,9.635,9.631,9.628,9.625,9.622,9.619,9.615,
     * 9.382,9.355,9.325,9.295,9.267,9.239,9.214,9.191,
     * 9.089,9.061,9.034,9.007,8.979,8.955,8.930,8.910/
C
      DATA ((EDEN(I,3,6,L,2),I=57,64),L=1,NZEN)/
     * 11.131,11.128,11.125,11.123,11.121,11.120,11.120,11.122,
     * 11.088,11.086,11.087,11.087,11.088,11.088,11.090,11.093,
     * 11.056,11.057,11.059,11.060,11.061,11.063,11.064,11.067,
     * 11.001,11.005,11.009,11.011,11.014,11.016,11.018,11.022,
     * 10.858,10.866,10.873,10.879,10.883,10.886,10.890,10.894,
     * 10.764,10.772,10.780,10.785,10.789,10.793,10.797,10.802,
     * 10.614,10.624,10.628,10.633,10.637,10.641,10.644,10.649,
     * 10.187,10.196,10.203,10.209,10.213,10.215,10.216,10.234,
     * 9.666,9.662,9.659,9.655,9.651,9.644,9.636,9.628,
     * 9.401,9.372,9.343,9.312,9.281,9.251,9.221,9.193,
     * 9.079,9.053,9.027,9.001,8.976,8.946,8.922,8.899/
C
      DATA ((EDEN(I,4,6,L,2),I=57,64),L=1,NZEN)/
     * 11.132,11.130,11.130,11.129,11.129,11.128,11.130,11.132,
     * 11.105,11.105,11.105,11.104,11.105,11.104,11.106,11.109,
     * 11.066,11.069,11.071,11.073,11.075,11.077,11.079,11.082,
     * 11.006,11.010,11.015,11.018,11.021,11.024,11.026,11.031,
     * 10.861,10.870,10.877,10.882,10.887,10.891,10.894,10.898,
     * 10.762,10.769,10.776,10.781,10.785,10.789,10.792,10.796,
     * 10.618,10.625,10.631,10.635,10.637,10.639,10.641,10.644,
     * 10.187,10.196,10.204,10.208,10.210,10.210,10.208,10.224,
     * 9.666,9.663,9.660,9.655,9.648,9.639,9.627,9.613,
     * 9.403,9.375,9.342,9.309,9.275,9.239,9.203,9.172,
     * 9.105,9.075,9.045,9.013,8.979,8.946,8.912,8.882/
C
      DATA ((EDEN(I,5,6,L,2),I=57,64),L=1,NZEN)/
     * 11.131,11.132,11.133,11.134,11.135,11.136,11.139,11.142,
     * 11.109,11.112,11.113,11.113,11.114,11.115,11.117,11.120,
     * 11.074,11.077,11.080,11.082,11.084,11.085,11.087,11.092,
     * 11.010,11.016,11.020,11.024,11.028,11.030,11.033,11.037,
     * 10.859,10.867,10.875,10.880,10.884,10.888,10.891,10.894,
     * 10.763,10.771,10.777,10.781,10.785,10.788,10.791,10.793,
     * 10.620,10.626,10.631,10.635,10.636,10.636,10.637,10.638,
     * 10.190,10.198,10.203,10.205,10.205,10.201,10.196,10.210,
     * 9.665,9.661,9.657,9.650,9.639,9.624,9.607,9.587,
     * 9.389,9.357,9.322,9.285,9.246,9.207,9.167,9.131,
     * 9.121,9.090,9.054,9.013,8.975,8.931,8.890,8.852/
C
      DATA ((EDEN(I,1,7,L,2),I=57,64),L=1,NZEN)/
     * 11.115,11.113,11.108,11.106,11.104,11.101,11.100,11.101,
     * 11.078,11.076,11.074,11.072,11.071,11.069,11.069,11.071,
     * 11.039,11.038,11.039,11.038,11.038,11.038,11.039,11.040,
     * 10.985,10.988,10.989,10.991,10.992,10.992,10.994,10.997,
     * 10.850,10.857,10.862,10.867,10.871,10.874,10.878,10.882,
     * 10.755,10.764,10.771,10.778,10.783,10.788,10.793,10.799,
     * 10.599,10.607,10.614,10.622,10.628,10.633,10.641,10.649,
     * 10.137,10.150,10.164,10.176,10.187,10.196,10.202,10.227,
     * 9.586,9.577,9.571,9.569,9.568,9.570,9.572,9.576,
     * 9.328,9.302,9.276,9.253,9.229,9.207,9.188,9.171,
     * 9.051,9.025,8.998,8.975,8.954,8.932,8.912,8.897/
C
      DATA ((EDEN(I,2,7,L,2),I=57,64),L=1,NZEN)/
     * 11.123,11.119,11.116,11.113,11.111,11.108,11.108,11.110,
     * 11.073,11.073,11.073,11.072,11.072,11.072,11.073,11.076,
     * 11.046,11.046,11.048,11.048,11.049,11.049,11.051,11.053,
     * 11.001,11.003,11.006,11.008,11.009,11.009,11.011,11.014,
     * 10.865,10.872,10.878,10.884,10.887,10.891,10.893,10.897,
     * 10.762,10.770,10.777,10.783,10.789,10.793,10.798,10.802,
     * 10.614,10.623,10.630,10.636,10.640,10.644,10.650,10.656,
     * 10.169,10.181,10.192,10.201,10.208,10.213,10.217,10.238,
     * 9.645,9.640,9.637,9.634,9.632,9.628,9.624,9.620,
     * 9.383,9.354,9.325,9.297,9.269,9.242,9.217,9.191,
     * 9.091,9.063,9.036,9.009,8.983,8.958,8.934,8.912/
C
      DATA ((EDEN(I,3,7,L,2),I=57,64),L=1,NZEN)/
     * 11.126,11.125,11.121,11.120,11.118,11.116,11.116,11.119,
     * 11.084,11.085,11.085,11.085,11.085,11.085,11.087,11.090,
     * 11.054,11.056,11.057,11.058,11.059,11.060,11.062,11.066,
     * 11.003,11.006,11.009,11.012,11.014,11.016,11.017,11.021,
     * 10.863,10.871,10.877,10.882,10.886,10.889,10.893,10.896,
     * 10.769,10.777,10.784,10.790,10.795,10.798,10.801,10.805,
     * 10.619,10.627,10.633,10.638,10.641,10.644,10.647,10.652,
     * 10.194,10.203,10.210,10.215,10.219,10.220,10.220,10.237,
     * 9.676,9.673,9.670,9.665,9.660,9.652,9.643,9.634,
     * 9.407,9.378,9.348,9.318,9.286,9.255,9.224,9.195,
     * 9.086,9.063,9.037,9.010,8.981,8.953,8.929,8.904/
C
      DATA ((EDEN(I,4,7,L,2),I=57,64),L=1,NZEN)/
     * 11.131,11.130,11.128,11.127,11.126,11.125,11.126,11.129,
     * 11.104,11.104,11.103,11.103,11.103,11.102,11.103,11.106,
     * 11.068,11.069,11.071,11.073,11.074,11.075,11.077,11.081,
     * 11.009,11.014,11.017,11.020,11.022,11.024,11.027,11.030,
     * 10.868,10.876,10.883,10.888,10.891,10.894,10.897,10.900,
     * 10.769,10.776,10.782,10.788,10.791,10.794,10.797,10.800,
     * 10.623,10.633,10.637,10.641,10.642,10.644,10.646,10.648,
     * 10.197,10.206,10.212,10.217,10.218,10.217,10.214,10.229,
     * 9.679,9.676,9.672,9.667,9.659,9.647,9.634,9.618,
     * 9.413,9.385,9.353,9.317,9.281,9.246,9.209,9.175,
     * 9.118,9.088,9.056,9.023,8.988,8.953,8.917,8.885/
C
      DATA ((EDEN(I,5,7,L,2),I=57,64),L=1,NZEN)/
     * 11.132,11.132,11.132,11.132,11.133,11.134,11.136,11.139,
     * 11.111,11.113,11.112,11.113,11.113,11.113,11.114,11.118,
     * 11.077,11.081,11.081,11.082,11.083,11.085,11.087,11.089,
     * 11.015,11.020,11.024,11.026,11.029,11.031,11.033,11.037,
     * 10.868,10.875,10.882,10.887,10.890,10.893,10.895,10.898,
     * 10.773,10.779,10.786,10.790,10.793,10.795,10.797,10.799,
     * 10.628,10.635,10.640,10.642,10.644,10.645,10.644,10.644,
     * 10.201,10.210,10.216,10.217,10.216,10.211,10.205,10.217,
     * 9.683,9.680,9.674,9.665,9.652,9.634,9.614,9.592,
     * 9.403,9.373,9.337,9.298,9.255,9.215,9.175,9.134,
     * 9.139,9.108,9.071,9.028,8.988,8.942,8.902,8.858/
C
      DATA ((EDEN(I,1,8,L,2),I=57,64),L=1,NZEN)/
     * 11.130,11.127,11.123,11.121,11.118,11.116,11.116,11.117,
     * 11.092,11.089,11.088,11.087,11.086,11.084,11.084,11.086,
     * 11.051,11.051,11.052,11.052,11.052,11.052,11.052,11.055,
     * 10.995,10.997,11.000,11.002,11.003,11.004,11.006,11.009,
     * 10.853,10.860,10.866,10.873,10.876,10.881,10.884,10.890,
     * 10.756,10.765,10.773,10.780,10.786,10.791,10.797,10.803,
     * 10.598,10.607,10.614,10.622,10.629,10.635,10.642,10.651,
     * 10.131,10.144,10.158,10.172,10.184,10.194,10.202,10.227,
     * 9.599,9.588,9.581,9.578,9.576,9.576,9.578,9.580,
     * 9.344,9.317,9.291,9.268,9.244,9.222,9.201,9.182,
     * 9.067,9.036,9.013,8.987,8.964,8.942,8.923,8.904/
C
      DATA ((EDEN(I,2,8,L,2),I=57,64),L=1,NZEN)/
     * 11.133,11.132,11.128,11.125,11.124,11.122,11.122,11.123,
     * 11.087,11.086,11.087,11.086,11.086,11.086,11.088,11.090,
     * 11.059,11.059,11.061,11.061,11.062,11.062,11.064,11.066,
     * 11.010,11.013,11.016,11.017,11.019,11.020,11.022,11.025,
     * 10.867,10.875,10.882,10.887,10.891,10.894,10.898,10.903,
     * 10.762,10.771,10.778,10.784,10.790,10.795,10.800,10.805,
     * 10.609,10.618,10.623,10.630,10.634,10.640,10.646,10.653,
     * 10.159,10.170,10.181,10.192,10.200,10.207,10.212,10.234,
     * 9.645,9.638,9.633,9.629,9.626,9.622,9.619,9.615,
     * 9.393,9.364,9.335,9.306,9.277,9.250,9.223,9.199,
     * 9.094,9.067,9.040,9.014,8.986,8.960,8.939,8.915/
C
      DATA ((EDEN(I,3,8,L,2),I=57,64),L=1,NZEN)/
     * 11.138,11.137,11.133,11.132,11.130,11.129,11.129,11.131,
     * 11.097,11.097,11.097,11.097,11.098,11.098,11.099,11.103,
     * 11.065,11.067,11.068,11.070,11.071,11.072,11.073,11.077,
     * 11.011,11.015,11.018,11.021,11.023,11.024,11.027,11.030,
     * 10.864,10.872,10.880,10.885,10.890,10.893,10.897,10.901,
     * 10.762,10.770,10.777,10.783,10.788,10.793,10.797,10.803,
     * 10.614,10.625,10.631,10.636,10.639,10.642,10.647,10.652,
     * 10.179,10.189,10.198,10.205,10.210,10.213,10.214,10.233,
     * 9.668,9.662,9.658,9.653,9.647,9.641,9.633,9.625,
     * 9.406,9.377,9.348,9.317,9.285,9.256,9.224,9.197,
     * 9.094,9.069,9.041,9.012,8.985,8.960,8.930,8.907/
C
      DATA ((EDEN(I,4,8,L,2),I=57,64),L=1,NZEN)/
     * 11.145,11.142,11.142,11.141,11.139,11.139,11.139,11.142,
     * 11.116,11.116,11.116,11.116,11.114,11.115,11.116,11.118,
     * 11.080,11.081,11.082,11.083,11.083,11.084,11.085,11.089,
     * 11.017,11.021,11.025,11.028,11.031,11.033,11.036,11.039,
     * 10.864,10.872,10.880,10.885,10.890,10.894,10.898,10.902,
     * 10.767,10.775,10.781,10.787,10.792,10.796,10.800,10.804,
     * 10.615,10.624,10.630,10.635,10.637,10.640,10.643,10.647,
     * 10.175,10.185,10.194,10.200,10.204,10.205,10.205,10.222,
     * 9.660,9.654,9.648,9.642,9.635,9.625,9.614,9.602,
     * 9.404,9.373,9.341,9.309,9.274,9.238,9.203,9.170,
     * 9.099,9.070,9.040,9.009,8.975,8.940,8.910,8.877/
C
      DATA ((EDEN(I,5,8,L,2),I=57,64),L=1,NZEN)/
     * 11.147,11.148,11.146,11.146,11.147,11.148,11.149,11.153,
     * 11.125,11.125,11.125,11.126,11.126,11.126,11.127,11.130,
     * 11.090,11.091,11.093,11.094,11.094,11.095,11.097,11.099,
     * 11.023,11.028,11.032,11.035,11.038,11.040,11.043,11.046,
     * 10.869,10.878,10.885,10.891,10.894,10.898,10.901,10.904,
     * 10.769,10.777,10.783,10.788,10.791,10.794,10.797,10.800,
     * 10.619,10.628,10.633,10.637,10.639,10.640,10.642,10.643,
     * 10.174,10.185,10.193,10.197,10.199,10.198,10.195,10.209,
     * 9.658,9.649,9.640,9.630,9.617,9.602,9.585,9.566,
     * 9.396,9.364,9.328,9.290,9.251,9.211,9.170,9.134,
     * 9.112,9.082,9.043,9.006,8.966,8.923,8.883,8.847/
C
      DATA ((EDEN(I,1,9,L,2),I=57,64),L=1,NZEN)/
     * 11.148,11.146,11.143,11.140,11.139,11.137,11.136,11.137,
     * 11.110,11.108,11.108,11.106,11.105,11.103,11.104,11.106,
     * 11.068,11.068,11.069,11.069,11.069,11.070,11.071,11.073,
     * 11.006,11.009,11.013,11.015,11.017,11.018,11.021,11.024,
     * 10.856,10.863,10.871,10.877,10.882,10.886,10.891,10.897,
     * 10.757,10.766,10.774,10.780,10.786,10.792,10.799,10.805,
     * 10.601,10.611,10.618,10.625,10.630,10.635,10.642,10.649,
     * 10.127,10.139,10.153,10.167,10.180,10.191,10.200,10.226,
     * 9.616,9.603,9.594,9.588,9.585,9.583,9.583,9.584,
     * 9.367,9.339,9.312,9.287,9.261,9.238,9.216,9.198,
     * 9.086,9.056,9.030,9.003,8.980,8.955,8.933,8.915/
C
      DATA ((EDEN(I,2,9,L,2),I=57,64),L=1,NZEN)/
     * 11.149,11.148,11.145,11.143,11.142,11.140,11.139,11.141,
     * 11.104,11.104,11.104,11.105,11.104,11.104,11.106,11.108,
     * 11.075,11.076,11.077,11.078,11.078,11.079,11.080,11.083,
     * 11.020,11.024,11.027,11.030,11.032,11.033,11.035,11.038,
     * 10.866,10.875,10.882,10.889,10.893,10.898,10.902,10.907,
     * 10.759,10.769,10.776,10.782,10.788,10.794,10.800,10.806,
     * 10.603,10.614,10.620,10.627,10.632,10.637,10.644,10.649,
     * 10.139,10.150,10.162,10.175,10.186,10.196,10.203,10.228,
     * 9.632,9.622,9.615,9.609,9.606,9.604,9.601,9.600,
     * 9.380,9.352,9.324,9.297,9.272,9.247,9.222,9.200,
     * 9.093,9.064,9.036,9.009,8.984,8.960,8.938,8.917/
C
      DATA ((EDEN(I,3,9,L,2),I=57,64),L=1,NZEN)/
     * 11.153,11.152,11.150,11.149,11.146,11.146,11.146,11.148,
     * 11.119,11.119,11.119,11.117,11.117,11.117,11.118,11.121,
     * 11.079,11.079,11.081,11.082,11.083,11.084,11.085,11.088,
     * 11.019,11.024,11.028,11.030,11.033,11.035,11.038,11.042,
     * 10.861,10.870,10.878,10.884,10.890,10.895,10.899,10.905,
     * 10.756,10.765,10.772,10.778,10.784,10.790,10.796,10.802,
     * 10.601,10.611,10.617,10.624,10.629,10.633,10.638,10.645,
     * 10.151,10.160,10.171,10.181,10.189,10.195,10.200,10.222,
     * 9.637,9.629,9.624,9.620,9.616,9.613,9.610,9.606,
     * 9.391,9.362,9.332,9.303,9.275,9.245,9.218,9.193,
     * 9.093,9.065,9.034,9.007,8.977,8.950,8.925,8.903/
C
      DATA ((EDEN(I,4,9,L,2),I=57,64),L=1,NZEN)/
     * 11.160,11.159,11.159,11.158,11.157,11.156,11.158,11.160,
     * 11.134,11.134,11.135,11.135,11.134,11.134,11.134,11.137,
     * 11.090,11.091,11.094,11.094,11.096,11.097,11.099,11.103,
     * 11.019,11.024,11.028,11.032,11.035,11.037,11.041,11.045,
     * 10.860,10.869,10.876,10.884,10.889,10.894,10.900,10.905,
     * 10.753,10.762,10.769,10.776,10.783,10.788,10.793,10.800,
     * 10.599,10.609,10.616,10.623,10.628,10.632,10.638,10.642,
     * 10.138,10.150,10.161,10.171,10.179,10.184,10.188,10.208,
     * 9.619,9.609,9.601,9.595,9.589,9.583,9.576,9.568,
     * 9.366,9.336,9.307,9.277,9.245,9.214,9.185,9.158,
     * 9.051,9.024,8.993,8.964,8.936,8.906,8.879,8.856/
C
      DATA ((EDEN(I,5,9,L,2),I=57,64),L=1,NZEN)/
     * 11.163,11.164,11.164,11.165,11.164,11.165,11.168,11.171,
     * 11.138,11.140,11.140,11.141,11.141,11.142,11.144,11.147,
     * 11.101,11.104,11.106,11.107,11.108,11.109,11.111,11.113,
     * 11.026,11.032,11.036,11.040,11.042,11.045,11.048,11.052,
     * 10.860,10.869,10.877,10.884,10.890,10.895,10.900,10.905,
     * 10.754,10.762,10.769,10.775,10.781,10.786,10.791,10.797,
     * 10.595,10.606,10.613,10.619,10.623,10.627,10.631,10.637,
     * 10.121,10.134,10.146,10.155,10.163,10.167,10.170,10.189,
     * 9.596,9.583,9.570,9.560,9.550,9.539,9.529,9.518,
     * 9.343,9.313,9.281,9.246,9.211,9.179,9.145,9.114,
     * 9.045,9.010,8.978,8.943,8.910,8.876,8.844,8.813/
C
      DATA ((EDEN(I,1,10,L,2),I=57,64),L=1,NZEN)/
     * 11.158,11.156,11.153,11.152,11.148,11.146,11.146,11.148,
     * 11.119,11.118,11.117,11.115,11.114,11.113,11.113,11.116,
     * 11.076,11.076,11.077,11.078,11.078,11.079,11.079,11.082,
     * 11.012,11.015,11.018,11.022,11.024,11.025,11.027,11.031,
     * 10.858,10.865,10.873,10.879,10.884,10.889,10.894,10.899,
     * 10.760,10.768,10.775,10.782,10.788,10.793,10.800,10.806,
     * 10.604,10.613,10.621,10.627,10.633,10.637,10.643,10.651,
     * 10.128,10.139,10.153,10.168,10.181,10.192,10.202,10.228,
     * 9.623,9.609,9.598,9.591,9.586,9.583,9.582,9.583,
     * 9.377,9.348,9.322,9.296,9.269,9.246,9.222,9.202,
     * 9.096,9.067,9.036,9.012,8.986,8.962,8.940,8.922/
C
      DATA ((EDEN(I,2,10,L,2),I=57,64),L=1,NZEN)/
     * 11.157,11.155,11.153,11.152,11.149,11.148,11.147,11.149,
     * 11.120,11.120,11.119,11.117,11.117,11.116,11.116,11.118,
     * 11.079,11.080,11.080,11.081,11.081,11.082,11.083,11.085,
     * 11.014,11.017,11.021,11.024,11.026,11.028,11.030,11.034,
     * 10.857,10.865,10.872,10.879,10.884,10.889,10.894,10.900,
     * 10.758,10.765,10.773,10.779,10.785,10.792,10.797,10.805,
     * 10.600,10.610,10.617,10.624,10.630,10.634,10.640,10.648,
     * 10.140,10.153,10.165,10.178,10.188,10.197,10.204,10.229,
     * 9.614,9.600,9.590,9.583,9.579,9.578,9.578,9.579,
     * 9.368,9.340,9.311,9.286,9.261,9.238,9.216,9.196,
     * 9.083,9.053,9.024,9.000,8.973,8.949,8.927,8.908/
C
      DATA ((EDEN(I,3,10,L,2),I=57,64),L=1,NZEN)/
     * 11.159,11.158,11.157,11.156,11.155,11.153,11.153,11.156,
     * 11.133,11.132,11.132,11.131,11.130,11.130,11.130,11.132,
     * 11.083,11.085,11.087,11.087,11.088,11.089,11.091,11.094,
     * 11.014,11.019,11.023,11.026,11.028,11.031,11.034,11.039,
     * 10.851,10.859,10.867,10.874,10.880,10.886,10.891,10.898,
     * 10.748,10.757,10.764,10.772,10.778,10.785,10.792,10.800,
     * 10.589,10.599,10.606,10.615,10.621,10.627,10.633,10.642,
     * 10.131,10.142,10.154,10.166,10.176,10.184,10.191,10.215,
     * 9.594,9.583,9.576,9.572,9.571,9.572,9.573,9.575,
     * 9.343,9.316,9.289,9.264,9.240,9.216,9.193,9.173,
     * 9.048,9.021,8.993,8.969,8.944,8.922,8.899,8.879/
C
      DATA ((EDEN(I,4,10,L,2),I=57,64),L=1,NZEN)/
     * 11.167,11.167,11.166,11.165,11.165,11.165,11.166,11.168,
     * 11.139,11.139,11.141,11.139,11.141,11.140,11.142,11.144,
     * 11.101,11.102,11.104,11.104,11.105,11.106,11.107,11.111,
     * 11.020,11.024,11.028,11.032,11.035,11.038,11.042,11.046,
     * 10.849,10.857,10.866,10.873,10.880,10.886,10.893,10.900,
     * 10.738,10.746,10.754,10.762,10.770,10.777,10.784,10.792,
     * 10.579,10.588,10.596,10.605,10.610,10.619,10.625,10.634,
     * 10.102,10.115,10.128,10.141,10.151,10.160,10.168,10.192,
     * 9.558,9.547,9.540,9.536,9.534,9.533,9.533,9.533,
     * 9.300,9.272,9.244,9.218,9.193,9.169,9.147,9.126,
     * 9.001,8.973,8.944,8.917,8.893,8.872,8.846,8.828/
C
      DATA ((EDEN(I,5,10,L,2),I=57,64),L=1,NZEN)/
     * 11.170,11.173,11.173,11.174,11.173,11.176,11.177,11.180,
     * 11.144,11.146,11.146,11.148,11.149,11.149,11.152,11.155,
     * 11.104,11.106,11.108,11.110,11.112,11.114,11.116,11.120,
     * 11.027,11.033,11.038,11.041,11.044,11.047,11.051,11.055,
     * 10.846,10.856,10.864,10.871,10.879,10.885,10.892,10.899,
     * 10.734,10.742,10.751,10.759,10.767,10.775,10.782,10.790,
     * 10.570,10.578,10.587,10.594,10.602,10.608,10.615,10.624,
     * 10.069,10.084,10.099,10.112,10.124,10.134,10.142,10.167,
     * 9.520,9.506,9.496,9.487,9.482,9.477,9.474,9.471,
     * 9.258,9.227,9.198,9.172,9.144,9.119,9.094,9.072,
     * 8.944,8.914,8.884,8.856,8.832,8.807,8.786,8.769/
C
      DATA ((EDEN(I,1,11,L,2),I=57,64),L=1,NZEN)/
     * 11.155,11.151,11.149,11.148,11.144,11.142,11.141,11.143,
     * 11.115,11.115,11.112,11.111,11.110,11.109,11.109,11.111,
     * 11.073,11.073,11.075,11.074,11.075,11.075,11.075,11.078,
     * 11.012,11.015,11.018,11.020,11.022,11.023,11.025,11.029,
     * 10.861,10.868,10.875,10.881,10.886,10.890,10.895,10.900,
     * 10.764,10.771,10.778,10.785,10.790,10.796,10.802,10.808,
     * 10.610,10.617,10.625,10.631,10.636,10.640,10.646,10.653,
     * 10.133,10.146,10.160,10.175,10.188,10.198,10.207,10.233,
     * 9.616,9.601,9.590,9.582,9.577,9.574,9.574,9.576,
     * 9.370,9.340,9.314,9.288,9.263,9.240,9.218,9.198,
     * 9.088,9.059,9.030,9.006,8.980,8.955,8.933,8.918/
C
      DATA ((EDEN(I,2,11,L,2),I=57,64),L=1,NZEN)/
     * 11.153,11.151,11.148,11.147,11.143,11.142,11.141,11.142,
     * 11.127,11.126,11.125,11.123,11.121,11.119,11.119,11.120,
     * 11.076,11.077,11.077,11.077,11.077,11.077,11.078,11.081,
     * 11.013,11.017,11.019,11.021,11.023,11.024,11.026,11.031,
     * 10.857,10.865,10.872,10.878,10.883,10.888,10.893,10.898,
     * 10.756,10.764,10.771,10.779,10.784,10.791,10.797,10.805,
     * 10.597,10.606,10.614,10.621,10.626,10.633,10.640,10.647,
     * 10.132,10.145,10.159,10.172,10.183,10.192,10.201,10.226,
     * 9.585,9.569,9.557,9.550,9.546,9.546,9.549,9.554,
     * 9.338,9.311,9.285,9.259,9.237,9.216,9.198,9.180,
     * 9.050,9.020,8.995,8.972,8.948,8.929,8.909,8.893/
C
      DATA ((EDEN(I,3,11,L,2),I=57,64),L=1,NZEN)/
     * 11.155,11.153,11.152,11.151,11.148,11.147,11.147,11.149,
     * 11.129,11.127,11.127,11.126,11.124,11.124,11.124,11.126,
     * 11.091,11.092,11.092,11.091,11.091,11.091,11.092,11.094,
     * 11.010,11.014,11.016,11.019,11.021,11.023,11.026,11.030,
     * 10.848,10.856,10.864,10.870,10.876,10.882,10.888,10.896,
     * 10.737,10.745,10.753,10.761,10.768,10.777,10.784,10.793,
     * 10.580,10.589,10.597,10.605,10.614,10.621,10.628,10.638,
     * 10.110,10.122,10.136,10.149,10.160,10.170,10.180,10.207,
     * 9.540,9.529,9.524,9.523,9.527,9.534,9.542,9.551,
     * 9.293,9.265,9.241,9.219,9.198,9.179,9.161,9.146,
     * 8.995,8.966,8.943,8.919,8.902,8.884,8.866,8.855/
C
      DATA ((EDEN(I,4,11,L,2),I=57,64),L=1,NZEN)/
     * 11.164,11.164,11.162,11.161,11.160,11.160,11.161,11.164,
     * 11.137,11.136,11.135,11.135,11.135,11.135,11.136,11.139,
     * 11.098,11.098,11.099,11.098,11.100,11.101,11.103,11.106,
     * 11.022,11.026,11.028,11.032,11.033,11.036,11.040,11.044,
     * 10.840,10.848,10.855,10.863,10.870,10.877,10.884,10.892,
     * 10.725,10.733,10.742,10.750,10.759,10.768,10.776,10.785,
     * 10.560,10.570,10.578,10.587,10.597,10.605,10.617,10.627,
     * 10.067,10.081,10.096,10.111,10.124,10.137,10.148,10.177,
     * 9.489,9.480,9.476,9.477,9.482,9.488,9.495,9.502,
     * 9.217,9.192,9.169,9.146,9.128,9.111,9.097,9.083,
     * 8.921,8.890,8.868,8.846,8.828,8.813,8.799,8.789/
C
      DATA ((EDEN(I,5,11,L,2),I=57,64),L=1,NZEN)/
     * 11.170,11.171,11.171,11.170,11.171,11.171,11.172,11.176,
     * 11.143,11.143,11.144,11.143,11.142,11.144,11.146,11.150,
     * 11.101,11.102,11.103,11.105,11.106,11.108,11.111,11.115,
     * 11.023,11.027,11.031,11.035,11.038,11.041,11.046,11.050,
     * 10.838,10.846,10.854,10.861,10.869,10.877,10.883,10.892,
     * 10.717,10.724,10.734,10.743,10.752,10.761,10.770,10.781,
     * 10.543,10.553,10.562,10.573,10.582,10.592,10.604,10.616,
     * 10.026,10.042,10.059,10.075,10.090,10.104,10.116,10.146,
     * 9.438,9.426,9.419,9.417,9.418,9.422,9.426,9.431,
     * 9.159,9.130,9.105,9.083,9.066,9.049,9.032,9.020,
     * 8.847,8.815,8.790,8.769,8.753,8.738,8.724,8.718/
C
      DATA ((EDEN(I,1,12,L,2),I=57,64),L=1,NZEN)/
     * 11.147,11.144,11.141,11.138,11.134,11.132,11.131,11.132,
     * 11.109,11.107,11.104,11.103,11.101,11.100,11.099,11.101,
     * 11.068,11.068,11.068,11.067,11.066,11.066,11.067,11.070,
     * 11.010,11.012,11.014,11.015,11.016,11.018,11.019,11.023,
     * 10.865,10.872,10.878,10.883,10.887,10.891,10.895,10.900,
     * 10.769,10.776,10.783,10.790,10.795,10.800,10.806,10.812,
     * 10.613,10.621,10.629,10.635,10.639,10.645,10.650,10.657,
     * 10.140,10.153,10.168,10.182,10.194,10.204,10.212,10.238,
     * 9.605,9.590,9.578,9.570,9.566,9.564,9.565,9.568,
     * 9.356,9.327,9.302,9.278,9.252,9.228,9.207,9.189,
     * 9.078,9.048,9.021,8.997,8.970,8.949,8.930,8.911/
C
      DATA ((EDEN(I,2,12,L,2),I=57,64),L=1,NZEN)/
     * 11.146,11.142,11.139,11.136,11.133,11.131,11.130,11.131,
     * 11.121,11.118,11.116,11.115,11.111,11.109,11.108,11.109,
     * 11.075,11.075,11.074,11.073,11.071,11.070,11.071,11.073,
     * 11.012,11.013,11.015,11.016,11.017,11.018,11.021,11.024,
     * 10.859,10.867,10.873,10.878,10.883,10.887,10.892,10.898,
     * 10.752,10.758,10.767,10.774,10.779,10.786,10.793,10.801,
     * 10.597,10.605,10.613,10.620,10.627,10.633,10.642,10.651,
     * 10.129,10.143,10.157,10.170,10.182,10.191,10.200,10.226,
     * 9.550,9.536,9.527,9.523,9.524,9.529,9.537,9.547,
     * 9.310,9.283,9.259,9.236,9.216,9.196,9.179,9.164,
     * 9.020,8.993,8.967,8.945,8.928,8.910,8.892,8.882/
C
      DATA ((EDEN(I,3,12,L,2),I=57,64),L=1,NZEN)/
     * 11.149,11.146,11.142,11.140,11.139,11.137,11.136,11.138,
     * 11.123,11.121,11.118,11.116,11.115,11.114,11.114,11.115,
     * 11.085,11.085,11.085,11.083,11.083,11.082,11.083,11.085,
     * 11.008,11.009,11.011,11.013,11.015,11.017,11.019,11.023,
     * 10.844,10.851,10.857,10.863,10.869,10.875,10.880,10.887,
     * 10.736,10.743,10.752,10.760,10.768,10.776,10.784,10.793,
     * 10.572,10.581,10.589,10.597,10.607,10.616,10.625,10.637,
     * 10.095,10.108,10.122,10.136,10.149,10.161,10.172,10.201,
     * 9.499,9.488,9.484,9.486,9.492,9.503,9.515,9.528,
     * 9.236,9.212,9.190,9.172,9.156,9.142,9.129,9.120,
     * 8.951,8.924,8.902,8.884,8.866,8.854,8.843,8.832/
C
      DATA ((EDEN(I,4,12,L,2),I=57,64),L=1,NZEN)/
     * 11.159,11.157,11.155,11.153,11.151,11.151,11.152,11.154,
     * 11.132,11.130,11.130,11.127,11.127,11.126,11.127,11.130,
     * 11.092,11.092,11.092,11.092,11.092,11.093,11.094,11.097,
     * 11.019,11.021,11.023,11.026,11.028,11.030,11.033,11.037,
     * 10.837,10.844,10.851,10.858,10.865,10.871,10.878,10.886,
     * 10.720,10.728,10.737,10.747,10.756,10.765,10.774,10.785,
     * 10.550,10.558,10.567,10.578,10.588,10.598,10.611,10.624,
     * 10.055,10.068,10.083,10.098,10.113,10.126,10.138,10.170,
     * 9.442,9.435,9.435,9.440,9.449,9.460,9.471,9.482,
     * 9.165,9.137,9.118,9.100,9.085,9.072,9.061,9.053,
     * 8.859,8.832,8.811,8.792,8.781,8.771,8.762,8.757/
C
      DATA ((EDEN(I,5,12,L,2),I=57,64),L=1,NZEN)/
     * 11.168,11.167,11.165,11.163,11.162,11.162,11.164,11.167,
     * 11.139,11.138,11.137,11.135,11.136,11.136,11.138,11.141,
     * 11.097,11.097,11.098,11.097,11.099,11.100,11.103,11.106,
     * 11.020,11.023,11.025,11.028,11.031,11.034,11.038,11.043,
     * 10.833,10.840,10.848,10.856,10.863,10.871,10.878,10.887,
     * 10.708,10.718,10.727,10.736,10.746,10.756,10.766,10.778,
     * 10.529,10.539,10.549,10.559,10.571,10.583,10.595,10.611,
     * 9.999,10.016,10.033,10.051,10.068,10.084,10.099,10.134,
     * 9.385,9.376,9.373,9.375,9.381,9.389,9.398,9.408,
     * 9.088,9.060,9.038,9.022,9.007,8.997,8.990,8.984,
     * 8.770,8.740,8.716,8.700,8.690,8.681,8.677,8.678/
C
      DATA ((EDEN(I,1,1,L,3),I=57,64),L=1,NZEN)/
     * 11.257,11.258,11.256,11.256,11.256,11.257,11.259,11.262,
     * 11.221,11.221,11.221,11.224,11.225,11.226,11.229,11.234,
     * 11.177,11.181,11.184,11.187,11.190,11.193,11.197,11.203,
     * 11.116,11.123,11.127,11.134,11.139,11.143,11.149,11.155,
     * 10.965,10.976,10.987,10.997,11.006,11.015,11.024,11.033,
     * 10.860,10.873,10.886,10.898,10.910,10.920,10.930,10.942,
     * 10.698,10.710,10.723,10.734,10.745,10.757,10.770,10.783,
     * 10.258,10.274,10.291,10.308,10.323,10.336,10.348,10.377,
     * 9.696,9.695,9.698,9.706,9.716,9.729,9.742,9.756,
     * 9.441,9.424,9.406,9.392,9.377,9.363,9.351,9.340,
     * 9.170,9.153,9.137,9.122,9.111,9.098,9.087,9.079/
C
      DATA ((EDEN(I,2,1,L,3),I=57,64),L=1,NZEN)/
     * 11.256,11.254,11.255,11.256,11.256,11.256,11.257,11.260,
     * 11.230,11.231,11.231,11.236,11.234,11.235,11.237,11.240,
     * 11.186,11.189,11.190,11.191,11.194,11.197,11.200,11.205,
     * 11.116,11.122,11.126,11.133,11.137,11.142,11.148,11.155,
     * 10.957,10.967,10.979,10.989,10.998,11.008,11.018,11.028,
     * 10.843,10.855,10.868,10.881,10.894,10.905,10.917,10.930,
     * 10.679,10.692,10.705,10.718,10.731,10.744,10.758,10.772,
     * 10.241,10.256,10.273,10.289,10.304,10.318,10.332,10.363,
     * 9.635,9.634,9.640,9.652,9.669,9.689,9.710,9.732,
     * 9.388,9.371,9.356,9.345,9.335,9.325,9.316,9.311,
     * 9.108,9.094,9.079,9.069,9.061,9.052,9.049,9.045/
C
      DATA ((EDEN(I,3,1,L,3),I=57,64),L=1,NZEN)/
     * 11.257,11.256,11.258,11.259,11.259,11.260,11.262,11.266,
     * 11.230,11.231,11.232,11.233,11.236,11.237,11.240,11.244,
     * 11.192,11.196,11.196,11.201,11.203,11.206,11.209,11.215,
     * 11.114,11.119,11.125,11.130,11.135,11.140,11.146,11.153,
     * 10.939,10.949,10.960,10.971,10.982,10.991,11.002,11.013,
     * 10.824,10.836,10.850,10.863,10.877,10.889,10.903,10.916,
     * 10.653,10.665,10.679,10.693,10.707,10.722,10.737,10.753,
     * 10.203,10.217,10.233,10.250,10.266,10.282,10.297,10.331,
     * 9.583,9.586,9.596,9.613,9.635,9.659,9.683,9.706,
     * 9.312,9.297,9.285,9.276,9.269,9.263,9.258,9.258,
     * 9.031,9.016,9.003,8.994,8.988,8.984,8.984,8.983/
C
      DATA ((EDEN(I,4,1,L,3),I=57,64),L=1,NZEN)/
     * 11.268,11.268,11.271,11.269,11.271,11.272,11.277,11.280,
     * 11.240,11.242,11.240,11.245,11.247,11.248,11.252,11.257,
     * 11.196,11.201,11.202,11.207,11.209,11.214,11.218,11.224,
     * 11.119,11.124,11.131,11.136,11.142,11.148,11.154,11.162,
     * 10.927,10.937,10.949,10.960,10.972,10.982,10.994,11.006,
     * 10.802,10.815,10.829,10.843,10.857,10.872,10.885,10.900,
     * 10.624,10.636,10.649,10.664,10.680,10.697,10.713,10.731,
     * 10.148,10.162,10.179,10.196,10.214,10.231,10.247,10.283,
     * 9.521,9.527,9.541,9.560,9.580,9.601,9.622,9.641,
     * 9.223,9.205,9.191,9.182,9.174,9.170,9.166,9.166,
     * 8.920,8.903,8.889,8.881,8.877,8.874,8.875,8.879/
C
      DATA ((EDEN(I,5,1,L,3),I=57,64),L=1,NZEN)/
     * 11.274,11.274,11.278,11.278,11.282,11.285,11.288,11.292,
     * 11.243,11.247,11.250,11.249,11.251,11.258,11.260,11.267,
     * 11.199,11.204,11.206,11.212,11.214,11.217,11.224,11.231,
     * 11.117,11.123,11.130,11.135,11.141,11.148,11.154,11.164,
     * 10.917,10.928,10.940,10.952,10.963,10.974,10.986,11.000,
     * 10.784,10.797,10.810,10.825,10.839,10.854,10.868,10.884,
     * 10.595,10.608,10.621,10.637,10.653,10.669,10.687,10.707,
     * 10.079,10.095,10.113,10.133,10.152,10.171,10.189,10.227,
     * 9.441,9.442,9.450,9.463,9.479,9.497,9.514,9.531,
     * 9.125,9.104,9.088,9.077,9.069,9.064,9.062,9.061,
     * 8.807,8.783,8.768,8.756,8.754,8.754,8.754,8.759/
C
      DATA ((EDEN(I,1,2,L,3),I=57,64),L=1,NZEN)/
     * 11.264,11.262,11.264,11.264,11.260,11.264,11.265,11.268,
     * 11.224,11.225,11.229,11.229,11.230,11.232,11.235,11.238,
     * 11.181,11.184,11.188,11.193,11.195,11.199,11.201,11.208,
     * 11.120,11.125,11.130,11.137,11.141,11.148,11.152,11.159,
     * 10.962,10.974,10.986,10.996,11.006,11.014,11.023,11.033,
     * 10.856,10.870,10.882,10.895,10.906,10.918,10.929,10.940,
     * 10.693,10.707,10.719,10.731,10.742,10.754,10.765,10.778,
     * 10.251,10.267,10.284,10.301,10.317,10.331,10.343,10.373,
     * 9.701,9.700,9.703,9.710,9.719,9.731,9.743,9.756,
     * 9.448,9.430,9.413,9.397,9.382,9.369,9.355,9.344,
     * 9.178,9.158,9.143,9.128,9.114,9.101,9.090,9.082/
C
      DATA ((EDEN(I,2,2,L,3),I=57,64),L=1,NZEN)/
     * 11.263,11.262,11.263,11.264,11.264,11.264,11.266,11.269,
     * 11.238,11.239,11.240,11.240,11.239,11.242,11.244,11.247,
     * 11.186,11.189,11.191,11.194,11.196,11.199,11.203,11.208,
     * 11.118,11.124,11.131,11.135,11.140,11.145,11.152,11.159,
     * 10.954,10.966,10.978,10.989,10.998,11.008,11.018,11.028,
     * 10.840,10.853,10.867,10.879,10.892,10.903,10.915,10.928,
     * 10.676,10.690,10.702,10.715,10.728,10.741,10.755,10.769,
     * 10.240,10.255,10.272,10.288,10.303,10.317,10.330,10.361,
     * 9.655,9.651,9.653,9.660,9.672,9.688,9.705,9.724,
     * 9.403,9.386,9.371,9.358,9.346,9.336,9.326,9.318,
     * 9.124,9.107,9.092,9.082,9.071,9.061,9.057,9.052/
C
      DATA ((EDEN(I,3,2,L,3),I=57,64),L=1,NZEN)/
     * 11.265,11.267,11.265,11.267,11.268,11.268,11.270,11.274,
     * 11.238,11.239,11.241,11.241,11.241,11.245,11.249,11.252,
     * 11.197,11.201,11.205,11.208,11.209,11.213,11.217,11.222,
     * 11.113,11.121,11.127,11.131,11.137,11.144,11.150,11.157,
     * 10.937,10.949,10.960,10.971,10.982,10.992,11.003,11.015,
     * 10.822,10.835,10.848,10.862,10.875,10.888,10.901,10.915,
     * 10.653,10.665,10.678,10.692,10.705,10.720,10.735,10.750,
     * 10.208,10.221,10.237,10.253,10.269,10.284,10.298,10.331,
     * 9.606,9.608,9.617,9.632,9.651,9.672,9.694,9.715,
     * 9.347,9.329,9.315,9.304,9.294,9.286,9.278,9.273,
     * 9.055,9.038,9.024,9.014,9.007,9.002,8.997,8.995/
C
      DATA ((EDEN(I,4,2,L,3),I=57,64),L=1,NZEN)/
     * 11.272,11.275,11.274,11.277,11.279,11.281,11.283,11.287,
     * 11.244,11.246,11.248,11.250,11.251,11.257,11.259,11.264,
     * 11.202,11.206,11.207,11.212,11.216,11.220,11.225,11.231,
     * 11.121,11.127,11.134,11.140,11.146,11.152,11.159,11.167,
     * 10.927,10.938,10.950,10.961,10.973,10.984,10.996,11.008,
     * 10.802,10.815,10.829,10.844,10.857,10.871,10.885,10.900,
     * 10.626,10.639,10.652,10.667,10.682,10.698,10.714,10.732,
     * 10.155,10.169,10.185,10.202,10.219,10.236,10.252,10.286,
     * 9.544,9.547,9.558,9.574,9.592,9.611,9.630,9.647,
     * 9.254,9.236,9.221,9.209,9.200,9.194,9.189,9.185,
     * 8.954,8.937,8.922,8.912,8.904,8.898,8.896,8.898/
C
      DATA ((EDEN(I,5,2,L,3),I=57,64),L=1,NZEN)/
     * 11.277,11.281,11.282,11.286,11.286,11.289,11.292,11.298,
     * 11.248,11.251,11.254,11.253,11.260,11.263,11.266,11.272,
     * 11.205,11.206,11.212,11.213,11.219,11.225,11.230,11.237,
     * 11.120,11.127,11.133,11.140,11.147,11.153,11.161,11.169,
     * 10.918,10.931,10.943,10.954,10.966,10.978,10.989,11.002,
     * 10.787,10.799,10.812,10.827,10.841,10.855,10.870,10.885,
     * 10.600,10.612,10.626,10.641,10.656,10.672,10.689,10.706,
     * 10.090,10.106,10.124,10.143,10.161,10.179,10.197,10.233,
     * 9.472,9.472,9.478,9.489,9.503,9.518,9.533,9.548,
     * 9.170,9.146,9.129,9.115,9.105,9.099,9.092,9.088,
     * 8.852,8.826,8.809,8.796,8.788,8.784,8.784,8.785/
C
      DATA ((EDEN(I,1,3,L,3),I=57,64),L=1,NZEN)/
     * 11.268,11.266,11.268,11.269,11.269,11.268,11.273,11.274,
     * 11.228,11.229,11.233,11.233,11.235,11.238,11.241,11.246,
     * 11.185,11.188,11.193,11.196,11.199,11.203,11.207,11.213,
     * 11.119,11.125,11.131,11.138,11.143,11.149,11.154,11.161,
     * 10.955,10.968,10.980,10.991,11.000,11.010,11.019,11.029,
     * 10.848,10.862,10.874,10.887,10.899,10.910,10.921,10.933,
     * 10.687,10.700,10.712,10.723,10.734,10.746,10.757,10.770,
     * 10.245,10.260,10.277,10.294,10.309,10.324,10.337,10.366,
     * 9.709,9.708,9.711,9.717,9.726,9.737,9.748,9.760,
     * 9.457,9.437,9.421,9.405,9.389,9.375,9.361,9.348,
     * 9.185,9.166,9.148,9.133,9.120,9.104,9.093,9.085/
C
      DATA ((EDEN(I,2,3,L,3),I=57,64),L=1,NZEN)/
     * 11.271,11.269,11.271,11.272,11.273,11.273,11.274,11.277,
     * 11.231,11.231,11.236,11.236,11.239,11.241,11.244,11.249,
     * 11.187,11.190,11.195,11.199,11.201,11.205,11.209,11.215,
     * 11.117,11.125,11.130,11.137,11.143,11.149,11.155,11.162,
     * 10.949,10.962,10.974,10.985,10.995,11.005,11.015,11.026,
     * 10.839,10.852,10.865,10.879,10.891,10.903,10.916,10.928,
     * 10.674,10.688,10.701,10.712,10.725,10.737,10.749,10.763,
     * 10.245,10.259,10.275,10.290,10.304,10.317,10.330,10.359,
     * 9.680,9.678,9.681,9.689,9.700,9.713,9.727,9.742,
     * 9.429,9.410,9.395,9.380,9.367,9.354,9.341,9.332,
     * 9.150,9.133,9.118,9.104,9.091,9.080,9.071,9.061/
C
      DATA ((EDEN(I,3,3,L,3),I=57,64),L=1,NZEN)/
     * 11.272,11.271,11.273,11.275,11.276,11.277,11.281,11.284,
     * 11.243,11.244,11.249,11.250,11.251,11.253,11.257,11.261,
     * 11.195,11.199,11.203,11.207,11.211,11.214,11.218,11.224,
     * 11.114,11.121,11.130,11.137,11.142,11.150,11.156,11.165,
     * 10.936,10.949,10.962,10.974,10.985,10.996,11.007,11.019,
     * 10.819,10.832,10.846,10.859,10.872,10.885,10.898,10.911,
     * 10.657,10.669,10.682,10.695,10.707,10.719,10.734,10.748,
     * 10.221,10.235,10.249,10.264,10.278,10.291,10.304,10.334,
     * 9.647,9.647,9.654,9.665,9.680,9.696,9.712,9.728,
     * 9.387,9.369,9.354,9.341,9.327,9.316,9.304,9.295,
     * 9.100,9.082,9.067,9.052,9.042,9.031,9.022,9.016/
C
      DATA ((EDEN(I,4,3,L,3),I=57,64),L=1,NZEN)/
     * 11.275,11.278,11.282,11.281,11.287,11.289,11.291,11.297,
     * 11.245,11.251,11.253,11.255,11.261,11.262,11.266,11.272,
     * 11.205,11.209,11.214,11.215,11.221,11.226,11.231,11.238,
     * 11.118,11.126,11.133,11.141,11.147,11.153,11.161,11.170,
     * 10.925,10.937,10.950,10.962,10.973,10.985,10.997,11.009,
     * 10.804,10.817,10.831,10.844,10.858,10.872,10.885,10.900,
     * 10.632,10.644,10.658,10.671,10.684,10.698,10.712,10.726,
     * 10.174,10.187,10.202,10.217,10.233,10.247,10.260,10.291,
     * 9.590,9.592,9.600,9.613,9.628,9.643,9.658,9.671,
     * 9.320,9.300,9.283,9.267,9.255,9.242,9.232,9.222,
     * 9.020,8.999,8.983,8.967,8.956,8.946,8.935,8.931/
C
      DATA ((EDEN(I,5,3,L,3),I=57,64),L=1,NZEN)/
     * 11.277,11.281,11.286,11.290,11.293,11.297,11.301,11.305,
     * 11.250,11.253,11.256,11.259,11.266,11.268,11.275,11.280,
     * 11.205,11.210,11.215,11.221,11.226,11.231,11.236,11.244,
     * 11.118,11.126,11.136,11.143,11.150,11.158,11.166,11.174,
     * 10.917,10.930,10.943,10.955,10.967,10.979,10.991,11.004,
     * 10.791,10.804,10.817,10.831,10.844,10.858,10.872,10.886,
     * 10.612,10.624,10.638,10.651,10.663,10.678,10.693,10.707,
     * 10.124,10.138,10.153,10.170,10.185,10.200,10.213,10.245,
     * 9.530,9.528,9.531,9.539,9.549,9.561,9.571,9.582,
     * 9.249,9.224,9.203,9.187,9.171,9.157,9.144,9.132,
     * 8.926,8.901,8.882,8.866,8.852,8.841,8.831,8.824/
C
      DATA ((EDEN(I,1,4,L,3),I=57,64),L=1,NZEN)/
     * 11.264,11.262,11.264,11.264,11.264,11.268,11.268,11.272,
     * 11.224,11.225,11.229,11.229,11.233,11.233,11.237,11.243,
     * 11.181,11.184,11.188,11.192,11.195,11.199,11.204,11.209,
     * 11.112,11.118,11.127,11.132,11.137,11.144,11.150,11.157,
     * 10.949,10.961,10.973,10.984,10.994,11.004,11.013,11.023,
     * 10.842,10.855,10.868,10.880,10.892,10.903,10.915,10.926,
     * 10.682,10.695,10.708,10.719,10.729,10.739,10.750,10.763,
     * 10.243,10.257,10.274,10.290,10.306,10.320,10.332,10.361,
     * 9.709,9.710,9.714,9.722,9.732,9.743,9.753,9.764,
     * 9.454,9.436,9.417,9.403,9.385,9.371,9.359,9.346,
     * 9.182,9.163,9.148,9.131,9.117,9.104,9.093,9.082/
C
      DATA ((EDEN(I,2,4,L,3),I=57,64),L=1,NZEN)/
     * 11.267,11.269,11.267,11.268,11.268,11.273,11.274,11.277,
     * 11.227,11.228,11.233,11.232,11.237,11.239,11.243,11.247,
     * 11.180,11.188,11.192,11.195,11.199,11.204,11.208,11.213,
     * 11.110,11.118,11.126,11.133,11.140,11.147,11.152,11.160,
     * 10.950,10.963,10.976,10.988,10.998,11.008,11.019,11.029,
     * 10.837,10.850,10.863,10.876,10.888,10.900,10.911,10.923,
     * 10.677,10.690,10.703,10.714,10.725,10.735,10.746,10.759,
     * 10.243,10.257,10.273,10.288,10.303,10.316,10.328,10.356,
     * 9.708,9.710,9.717,9.726,9.737,9.748,9.759,9.769,
     * 9.450,9.432,9.415,9.399,9.384,9.368,9.354,9.340,
     * 9.176,9.158,9.142,9.126,9.111,9.095,9.083,9.074/
C
      DATA ((EDEN(I,3,4,L,3),I=57,64),L=1,NZEN)/
     * 11.268,11.271,11.269,11.271,11.272,11.277,11.280,11.284,
     * 11.240,11.241,11.247,11.247,11.252,11.253,11.256,11.261,
     * 11.185,11.189,11.196,11.200,11.205,11.210,11.217,11.223,
     * 11.108,11.117,11.125,11.132,11.140,11.147,11.155,11.163,
     * 10.934,10.946,10.959,10.971,10.982,10.993,11.004,11.015,
     * 10.825,10.838,10.851,10.863,10.876,10.887,10.899,10.912,
     * 10.665,10.678,10.690,10.701,10.712,10.721,10.733,10.745,
     * 10.241,10.252,10.265,10.278,10.290,10.301,10.311,10.339,
     * 9.695,9.700,9.708,9.720,9.732,9.744,9.755,9.764,
     * 9.433,9.414,9.397,9.380,9.363,9.348,9.331,9.318,
     * 9.149,9.131,9.113,9.097,9.079,9.066,9.052,9.040/
C
      DATA ((EDEN(I,4,4,L,3),I=57,64),L=1,NZEN)/
     * 11.268,11.275,11.274,11.277,11.283,11.285,11.291,11.295,
     * 11.240,11.245,11.247,11.253,11.258,11.263,11.265,11.270,
     * 11.197,11.201,11.205,11.214,11.217,11.222,11.227,11.234,
     * 11.110,11.120,11.129,11.137,11.145,11.153,11.161,11.170,
     * 10.925,10.938,10.951,10.964,10.975,10.986,10.998,11.009,
     * 10.812,10.825,10.837,10.850,10.863,10.875,10.887,10.899,
     * 10.649,10.662,10.672,10.684,10.693,10.705,10.716,10.728,
     * 10.212,10.224,10.236,10.248,10.259,10.269,10.278,10.305,
     * 9.656,9.659,9.666,9.676,9.686,9.696,9.704,9.711,
     * 9.394,9.374,9.352,9.334,9.313,9.294,9.275,9.259,
     * 9.095,9.073,9.053,9.034,9.017,8.999,8.980,8.968/
C
      DATA ((EDEN(I,5,4,L,3),I=57,64),L=1,NZEN)/
     * 11.270,11.274,11.278,11.282,11.290,11.293,11.296,11.305,
     * 11.241,11.244,11.250,11.257,11.263,11.265,11.272,11.278,
     * 11.198,11.203,11.212,11.217,11.222,11.229,11.235,11.243,
     * 11.113,11.123,11.133,11.140,11.149,11.156,11.164,11.173,
     * 10.920,10.934,10.946,10.958,10.970,10.981,10.993,11.004,
     * 10.802,10.815,10.828,10.840,10.852,10.863,10.876,10.888,
     * 10.635,10.646,10.658,10.668,10.677,10.688,10.699,10.711,
     * 10.178,10.189,10.201,10.212,10.222,10.231,10.239,10.265,
     * 9.611,9.609,9.611,9.616,9.623,9.628,9.632,9.634,
     * 9.340,9.317,9.293,9.269,9.246,9.224,9.203,9.182,
     * 9.032,9.009,8.985,8.959,8.939,8.919,8.898,8.880/
C
      DATA ((EDEN(I,1,5,L,3),I=57,64),L=1,NZEN)/
     * 11.250,11.251,11.252,11.252,11.252,11.251,11.254,11.256,
     * 11.209,11.213,11.212,11.216,11.217,11.220,11.223,11.228,
     * 11.168,11.171,11.176,11.178,11.183,11.185,11.191,11.196,
     * 11.103,11.109,11.117,11.122,11.129,11.134,11.139,11.146,
     * 10.947,10.959,10.970,10.981,10.991,11.000,11.009,11.019,
     * 10.842,10.854,10.868,10.880,10.892,10.902,10.914,10.926,
     * 10.682,10.695,10.706,10.718,10.729,10.739,10.751,10.763,
     * 10.247,10.262,10.278,10.294,10.309,10.322,10.333,10.362,
     * 9.701,9.704,9.712,9.722,9.733,9.745,9.756,9.768,
     * 9.438,9.422,9.405,9.390,9.375,9.360,9.347,9.335,
     * 9.168,9.151,9.135,9.122,9.108,9.095,9.084,9.075/
C
      DATA ((EDEN(I,2,5,L,3),I=57,64),L=1,NZEN)/
     * 11.252,11.254,11.255,11.256,11.256,11.259,11.261,11.266,
     * 11.204,11.208,11.212,11.215,11.219,11.222,11.226,11.231,
     * 11.173,11.176,11.182,11.186,11.190,11.194,11.199,11.205,
     * 11.113,11.120,11.128,11.135,11.140,11.147,11.152,11.158,
     * 10.952,10.965,10.977,10.988,10.999,11.007,11.017,11.026,
     * 10.841,10.854,10.868,10.880,10.891,10.902,10.913,10.924,
     * 10.685,10.697,10.708,10.719,10.729,10.739,10.750,10.762,
     * 10.263,10.276,10.290,10.304,10.316,10.327,10.337,10.363,
     * 9.731,9.738,9.748,9.759,9.770,9.779,9.787,9.793,
     * 9.458,9.439,9.423,9.406,9.389,9.373,9.358,9.342,
     * 9.183,9.167,9.149,9.133,9.119,9.102,9.091,9.078/
C
      DATA ((EDEN(I,3,5,L,3),I=57,64),L=1,NZEN)/
     * 11.257,11.256,11.258,11.259,11.264,11.264,11.267,11.273,
     * 11.218,11.223,11.227,11.227,11.233,11.236,11.241,11.245,
     * 11.175,11.179,11.186,11.192,11.197,11.203,11.207,11.214,
     * 11.108,11.118,11.125,11.133,11.141,11.147,11.153,11.161,
     * 10.942,10.956,10.968,10.980,10.990,11.000,11.009,11.019,
     * 10.833,10.846,10.858,10.871,10.881,10.892,10.902,10.914,
     * 10.682,10.694,10.705,10.715,10.725,10.732,10.742,10.753,
     * 10.276,10.286,10.297,10.307,10.316,10.323,10.329,10.353,
     * 9.743,9.753,9.765,9.776,9.786,9.793,9.797,9.800,
     * 9.472,9.453,9.432,9.412,9.391,9.370,9.351,9.331,
     * 9.178,9.160,9.143,9.125,9.106,9.087,9.071,9.053/
C
      DATA ((EDEN(I,4,5,L,3),I=57,64),L=1,NZEN)/
     * 11.257,11.260,11.263,11.265,11.271,11.272,11.277,11.283,
     * 11.230,11.235,11.237,11.242,11.247,11.249,11.254,11.259,
     * 11.184,11.188,11.195,11.200,11.205,11.211,11.218,11.224,
     * 11.107,11.117,11.126,11.136,11.142,11.151,11.159,11.166,
     * 10.934,10.947,10.960,10.971,10.982,10.992,11.002,11.012,
     * 10.828,10.840,10.852,10.864,10.874,10.884,10.894,10.905,
     * 10.672,10.683,10.693,10.702,10.712,10.719,10.727,10.736,
     * 10.261,10.271,10.280,10.288,10.294,10.299,10.303,10.324,
     * 9.719,9.726,9.735,9.743,9.750,9.754,9.755,9.753,
     * 9.449,9.427,9.405,9.380,9.355,9.332,9.305,9.281,
     * 9.147,9.126,9.107,9.083,9.062,9.036,9.016,8.996/
C
      DATA ((EDEN(I,5,5,L,3),I=57,64),L=1,NZEN)/
     * 11.253,11.260,11.263,11.271,11.274,11.281,11.283,11.292,
     * 11.229,11.235,11.241,11.247,11.249,11.254,11.260,11.267,
     * 11.187,11.196,11.204,11.208,11.215,11.219,11.226,11.232,
     * 11.109,11.119,11.127,11.136,11.145,11.153,11.159,11.167,
     * 10.930,10.943,10.955,10.967,10.977,10.987,10.996,11.006,
     * 10.821,10.833,10.844,10.855,10.865,10.875,10.884,10.894,
     * 10.666,10.676,10.685,10.694,10.700,10.708,10.716,10.725,
     * 10.241,10.250,10.257,10.264,10.268,10.271,10.272,10.292,
     * 9.690,9.694,9.700,9.705,9.707,9.706,9.702,9.694,
     * 9.417,9.390,9.364,9.334,9.306,9.276,9.244,9.216,
     * 9.127,9.102,9.073,9.044,9.016,8.986,8.954,8.926/
C
      DATA ((EDEN(I,1,6,L,3),I=57,64),L=1,NZEN)/
     * 11.234,11.231,11.231,11.231,11.233,11.234,11.235,11.239,
     * 11.196,11.196,11.198,11.199,11.202,11.202,11.205,11.210,
     * 11.154,11.157,11.160,11.164,11.168,11.171,11.175,11.180,
     * 11.095,11.101,11.107,11.113,11.118,11.122,11.128,11.134,
     * 10.948,10.960,10.970,10.980,10.990,10.998,11.006,11.015,
     * 10.846,10.859,10.872,10.884,10.895,10.906,10.916,10.927,
     * 10.684,10.697,10.709,10.721,10.733,10.744,10.756,10.769,
     * 10.253,10.268,10.284,10.300,10.314,10.326,10.337,10.366,
     * 9.692,9.698,9.708,9.720,9.733,9.746,9.759,9.771,
     * 9.422,9.407,9.391,9.376,9.362,9.350,9.337,9.327,
     * 9.153,9.138,9.123,9.111,9.099,9.085,9.077,9.069/
C
      DATA ((EDEN(I,2,6,L,3),I=57,64),L=1,NZEN)/
     * 11.238,11.238,11.239,11.239,11.241,11.242,11.245,11.247,
     * 11.188,11.192,11.196,11.200,11.202,11.205,11.209,11.215,
     * 11.160,11.163,11.168,11.173,11.177,11.180,11.184,11.191,
     * 11.108,11.114,11.120,11.127,11.132,11.137,11.142,11.148,
     * 10.957,10.970,10.981,10.991,11.001,11.009,11.017,11.025,
     * 10.849,10.862,10.875,10.887,10.898,10.908,10.917,10.928,
     * 10.695,10.708,10.719,10.729,10.740,10.749,10.759,10.771,
     * 10.282,10.295,10.308,10.320,10.330,10.339,10.347,10.373,
     * 9.753,9.763,9.775,9.785,9.794,9.801,9.807,9.811,
     * 9.471,9.453,9.434,9.414,9.395,9.378,9.358,9.342,
     * 9.189,9.171,9.154,9.140,9.123,9.108,9.091,9.080/
C
      DATA ((EDEN(I,3,6,L,3),I=57,64),L=1,NZEN)/
     * 11.243,11.244,11.246,11.247,11.247,11.248,11.251,11.257,
     * 11.199,11.203,11.206,11.210,11.213,11.218,11.222,11.228,
     * 11.164,11.169,11.176,11.180,11.185,11.191,11.196,11.202,
     * 11.105,11.114,11.120,11.128,11.134,11.140,11.146,11.153,
     * 10.950,10.963,10.975,10.986,10.994,11.003,11.012,11.020,
     * 10.850,10.862,10.874,10.885,10.895,10.905,10.914,10.923,
     * 10.697,10.708,10.717,10.727,10.735,10.743,10.752,10.762,
     * 10.302,10.312,10.321,10.329,10.335,10.340,10.344,10.365,
     * 9.778,9.789,9.800,9.809,9.815,9.817,9.817,9.815,
     * 9.490,9.471,9.451,9.428,9.406,9.383,9.360,9.339,
     * 9.182,9.165,9.150,9.132,9.112,9.095,9.075,9.060/
C
      DATA ((EDEN(I,4,6,L,3),I=57,64),L=1,NZEN)/
     * 11.243,11.245,11.247,11.253,11.254,11.258,11.261,11.267,
     * 11.216,11.221,11.222,11.226,11.229,11.234,11.237,11.244,
     * 11.174,11.181,11.186,11.191,11.196,11.202,11.209,11.216,
     * 11.106,11.115,11.124,11.131,11.139,11.146,11.152,11.159,
     * 10.949,10.962,10.974,10.984,10.993,11.002,11.010,11.017,
     * 10.843,10.855,10.867,10.877,10.886,10.894,10.903,10.911,
     * 10.695,10.705,10.714,10.722,10.729,10.736,10.743,10.750,
     * 10.296,10.305,10.312,10.318,10.321,10.323,10.324,10.342,
     * 9.768,9.779,9.789,9.796,9.799,9.797,9.792,9.783,
     * 9.483,9.460,9.436,9.410,9.383,9.352,9.325,9.296,
     * 9.192,9.172,9.148,9.123,9.097,9.070,9.040,9.017/
C
      DATA ((EDEN(I,5,6,L,3),I=57,64),L=1,NZEN)/
     * 11.242,11.249,11.252,11.255,11.257,11.264,11.268,11.276,
     * 11.219,11.225,11.231,11.232,11.237,11.242,11.247,11.252,
     * 11.182,11.186,11.193,11.199,11.205,11.209,11.214,11.221,
     * 11.110,11.120,11.127,11.135,11.142,11.150,11.156,11.163,
     * 10.946,10.959,10.970,10.979,10.988,10.995,11.003,11.011,
     * 10.843,10.854,10.865,10.875,10.882,10.890,10.896,10.903,
     * 10.693,10.703,10.710,10.717,10.723,10.728,10.733,10.739,
     * 10.289,10.297,10.302,10.304,10.303,10.301,10.298,10.313,
     * 9.759,9.769,9.776,9.778,9.774,9.765,9.752,9.736,
     * 9.457,9.430,9.402,9.370,9.335,9.300,9.264,9.229,
     * 9.195,9.169,9.137,9.103,9.068,9.031,8.992,8.956/
C
      DATA ((EDEN(I,1,7,L,3),I=57,64),L=1,NZEN)/
     * 11.231,11.227,11.227,11.229,11.230,11.230,11.232,11.236,
     * 11.192,11.191,11.195,11.196,11.198,11.199,11.202,11.207,
     * 11.151,11.154,11.158,11.162,11.163,11.168,11.171,11.177,
     * 11.093,11.098,11.104,11.110,11.116,11.121,11.126,11.132,
     * 10.950,10.961,10.972,10.981,10.990,10.999,11.007,11.017,
     * 10.847,10.862,10.874,10.887,10.898,10.909,10.919,10.930,
     * 10.686,10.698,10.711,10.723,10.735,10.747,10.761,10.774,
     * 10.255,10.270,10.286,10.302,10.316,10.328,10.340,10.369,
     * 9.692,9.698,9.708,9.721,9.735,9.749,9.762,9.774,
     * 9.421,9.403,9.388,9.376,9.362,9.348,9.337,9.327,
     * 9.153,9.135,9.121,9.111,9.099,9.085,9.077,9.069/
C
      DATA ((EDEN(I,2,7,L,3),I=57,64),L=1,NZEN)/
     * 11.234,11.234,11.235,11.235,11.236,11.237,11.240,11.244,
     * 11.184,11.188,11.191,11.196,11.198,11.202,11.206,11.211,
     * 11.157,11.160,11.165,11.169,11.175,11.177,11.181,11.187,
     * 11.107,11.113,11.120,11.125,11.130,11.136,11.141,11.146,
     * 10.961,10.973,10.984,10.994,11.003,11.011,11.019,11.027,
     * 10.853,10.867,10.880,10.891,10.902,10.912,10.922,10.931,
     * 10.698,10.711,10.723,10.733,10.744,10.753,10.764,10.775,
     * 10.287,10.300,10.313,10.325,10.335,10.343,10.351,10.376,
     * 9.759,9.770,9.782,9.793,9.802,9.808,9.813,9.816,
     * 9.473,9.455,9.436,9.418,9.399,9.379,9.361,9.344,
     * 9.191,9.174,9.159,9.142,9.124,9.112,9.094,9.083/
C
      DATA ((EDEN(I,3,7,L,3),I=57,64),L=1,NZEN)/
     * 11.239,11.240,11.242,11.242,11.243,11.244,11.248,11.252,
     * 11.195,11.199,11.204,11.207,11.211,11.214,11.219,11.224,
     * 11.164,11.169,11.174,11.178,11.183,11.188,11.193,11.198,
     * 11.106,11.115,11.121,11.128,11.133,11.139,11.145,11.151,
     * 10.956,10.969,10.980,10.990,10.999,11.006,11.014,11.023,
     * 10.856,10.869,10.881,10.892,10.901,10.910,10.918,10.927,
     * 10.701,10.713,10.724,10.733,10.741,10.749,10.757,10.767,
     * 10.310,10.320,10.329,10.336,10.342,10.346,10.349,10.370,
     * 9.790,9.802,9.813,9.821,9.825,9.826,9.825,9.822,
     * 9.498,9.479,9.457,9.434,9.411,9.387,9.363,9.341,
     * 9.190,9.175,9.157,9.139,9.120,9.100,9.081,9.062/
C
      DATA ((EDEN(I,4,7,L,3),I=57,64),L=1,NZEN)/
     * 11.243,11.245,11.247,11.249,11.250,11.254,11.257,11.262,
     * 11.216,11.221,11.222,11.226,11.227,11.230,11.235,11.239,
     * 11.175,11.181,11.186,11.191,11.196,11.201,11.206,11.213,
     * 11.110,11.120,11.127,11.134,11.140,11.145,11.151,11.159,
     * 10.957,10.970,10.981,10.991,10.999,11.007,11.014,11.021,
     * 10.852,10.864,10.875,10.885,10.894,10.901,10.909,10.917,
     * 10.701,10.713,10.722,10.730,10.737,10.743,10.749,10.756,
     * 10.305,10.315,10.322,10.327,10.330,10.331,10.331,10.348,
     * 9.782,9.793,9.803,9.809,9.810,9.806,9.799,9.789,
     * 9.493,9.470,9.446,9.418,9.389,9.358,9.328,9.300,
     * 9.208,9.185,9.160,9.134,9.107,9.077,9.050,9.021/
C
      DATA ((EDEN(I,5,7,L,3),I=57,64),L=1,NZEN)/
     * 11.242,11.245,11.248,11.255,11.257,11.260,11.264,11.271,
     * 11.219,11.225,11.231,11.232,11.236,11.240,11.243,11.248,
     * 11.184,11.192,11.195,11.200,11.204,11.209,11.214,11.218,
     * 11.115,11.124,11.132,11.138,11.145,11.150,11.156,11.162,
     * 10.956,10.968,10.979,10.987,10.995,11.002,11.009,11.015,
     * 10.854,10.866,10.875,10.884,10.891,10.897,10.904,10.910,
     * 10.703,10.713,10.721,10.727,10.731,10.736,10.741,10.746,
     * 10.301,10.309,10.313,10.315,10.314,10.311,10.307,10.321,
     * 9.778,9.788,9.793,9.793,9.786,9.775,9.759,9.740,
     * 9.471,9.444,9.415,9.381,9.343,9.307,9.270,9.233,
     * 9.214,9.184,9.154,9.118,9.078,9.042,9.001,8.963/
C
      DATA ((EDEN(I,1,8,L,3),I=57,64),L=1,NZEN)/
     * 11.242,11.243,11.244,11.244,11.243,11.245,11.247,11.250,
     * 11.205,11.204,11.207,11.209,11.211,11.212,11.216,11.220,
     * 11.162,11.165,11.169,11.174,11.177,11.181,11.186,11.190,
     * 11.102,11.107,11.113,11.121,11.126,11.131,11.136,11.143,
     * 10.950,10.962,10.973,10.984,10.994,11.002,11.011,11.021,
     * 10.846,10.859,10.872,10.885,10.897,10.908,10.920,10.931,
     * 10.683,10.697,10.710,10.721,10.734,10.745,10.757,10.771,
     * 10.251,10.266,10.282,10.298,10.312,10.325,10.337,10.366,
     * 9.702,9.707,9.716,9.727,9.740,9.753,9.765,9.776,
     * 9.434,9.418,9.403,9.388,9.375,9.360,9.349,9.337,
     * 9.165,9.148,9.135,9.119,9.108,9.095,9.087,9.079/
C
      DATA ((EDEN(I,2,8,L,3),I=57,64),L=1,NZEN)/
     * 11.245,11.246,11.247,11.248,11.247,11.249,11.252,11.257,
     * 11.196,11.200,11.205,11.207,11.211,11.215,11.218,11.223,
     * 11.167,11.172,11.177,11.182,11.186,11.189,11.195,11.200,
     * 11.115,11.122,11.128,11.134,11.139,11.144,11.150,11.156,
     * 10.960,10.973,10.985,10.995,11.004,11.013,11.022,11.031,
     * 10.851,10.864,10.877,10.889,10.901,10.911,10.921,10.932,
     * 10.693,10.705,10.717,10.728,10.739,10.749,10.760,10.771,
     * 10.278,10.290,10.304,10.316,10.328,10.337,10.346,10.372,
     * 9.755,9.764,9.774,9.784,9.793,9.800,9.806,9.810,
     * 9.480,9.461,9.441,9.422,9.404,9.385,9.367,9.352,
     * 9.193,9.176,9.163,9.146,9.130,9.112,9.099,9.085/
C
      DATA ((EDEN(I,3,8,L,3),I=57,64),L=1,NZEN)/
     * 11.250,11.252,11.254,11.255,11.255,11.255,11.260,11.265,
     * 11.207,11.211,11.215,11.217,11.222,11.225,11.231,11.235,
     * 11.172,11.179,11.184,11.188,11.192,11.197,11.203,11.208,
     * 11.112,11.120,11.128,11.136,11.140,11.148,11.153,11.160,
     * 10.955,10.968,10.980,10.991,11.000,11.008,11.017,11.025,
     * 10.848,10.861,10.873,10.884,10.895,10.904,10.914,10.924,
     * 10.696,10.708,10.719,10.729,10.738,10.747,10.754,10.764,
     * 10.295,10.306,10.316,10.325,10.332,10.338,10.342,10.365,
     * 9.777,9.786,9.796,9.804,9.809,9.812,9.812,9.811,
     * 9.494,9.474,9.453,9.432,9.407,9.385,9.363,9.341,
     * 9.192,9.176,9.159,9.142,9.123,9.101,9.085,9.064/
C
      DATA ((EDEN(I,4,8,L,3),I=57,64),L=1,NZEN)/
     * 11.254,11.257,11.259,11.261,11.263,11.264,11.269,11.274,
     * 11.227,11.229,11.234,11.234,11.238,11.242,11.245,11.252,
     * 11.186,11.190,11.195,11.201,11.205,11.209,11.213,11.219,
     * 11.116,11.125,11.133,11.139,11.147,11.153,11.159,11.166,
     * 10.952,10.965,10.976,10.987,10.996,11.004,11.012,11.021,
     * 10.848,10.861,10.872,10.883,10.892,10.901,10.909,10.919,
     * 10.692,10.703,10.713,10.722,10.730,10.736,10.743,10.752,
     * 10.283,10.293,10.302,10.309,10.315,10.318,10.320,10.340,
     * 9.757,9.765,9.773,9.779,9.782,9.781,9.776,9.770,
     * 9.479,9.456,9.433,9.406,9.377,9.349,9.321,9.293,
     * 9.186,9.164,9.143,9.115,9.090,9.064,9.035,9.013/
C
      DATA ((EDEN(I,5,8,L,3),I=57,64),L=1,NZEN)/
     * 11.256,11.260,11.263,11.267,11.269,11.272,11.278,11.284,
     * 11.234,11.236,11.242,11.244,11.250,11.252,11.255,11.261,
     * 11.194,11.203,11.207,11.210,11.213,11.218,11.223,11.229,
     * 11.122,11.130,11.139,11.145,11.151,11.156,11.163,11.170,
     * 10.956,10.968,10.979,10.989,10.997,11.005,11.012,11.019,
     * 10.847,10.859,10.869,10.879,10.887,10.895,10.901,10.909,
     * 10.691,10.702,10.711,10.718,10.724,10.730,10.736,10.742,
     * 10.273,10.282,10.289,10.293,10.296,10.295,10.294,10.310,
     * 9.742,9.746,9.749,9.749,9.745,9.736,9.724,9.709,
     * 9.457,9.431,9.400,9.366,9.331,9.296,9.261,9.227,
     * 9.181,9.155,9.123,9.089,9.052,9.016,8.982,8.947/
C
      DATA ((EDEN(I,1,9,L,3),I=57,64),L=1,NZEN)/
     * 11.260,11.258,11.264,11.260,11.264,11.264,11.268,11.272,
     * 11.221,11.221,11.225,11.229,11.230,11.233,11.237,11.241,
     * 11.177,11.181,11.186,11.189,11.194,11.198,11.202,11.209,
     * 11.109,11.116,11.124,11.132,11.138,11.144,11.150,11.157,
     * 10.947,10.960,10.972,10.984,10.994,11.005,11