"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cdk = require("@aws-cdk/core");
const net = require("net");
const ec2_generated_1 = require("./ec2.generated");
/**
 * The VPN connection type.
 */
var VpnConnectionType;
(function (VpnConnectionType) {
    /**
     * The IPsec 1 VPN connection type.
     */
    VpnConnectionType["IPSEC_1"] = "ipsec.1";
    /**
     * Dummy member
     * TODO: remove once https://github.com/aws/jsii/issues/231 is fixed
     */
    VpnConnectionType["DUMMY"] = "dummy";
})(VpnConnectionType = exports.VpnConnectionType || (exports.VpnConnectionType = {}));
/**
 * The VPN Gateway that shall be added to the VPC
 *
 * @resource AWS::EC2::VPNGateway
 */
class VpnGateway extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        // This is 'Default' instead of 'Resource', because using 'Default' will generate
        // a logical ID for a VpnGateway which is exactly the same as the logical ID that used
        // to be created for the CfnVPNGateway (and 'Resource' would not do that).
        const vpnGW = new ec2_generated_1.CfnVPNGateway(this, 'Default', props);
        this.gatewayId = vpnGW.ref;
    }
}
exports.VpnGateway = VpnGateway;
/**
 * Define a VPN Connection
 *
 * @resource AWS::EC2::VPNConnection
 */
class VpnConnection extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.vpc.vpnGatewayId) {
            props.vpc.enableVpnGateway({
                type: 'ipsec.1',
                amazonSideAsn: props.asn,
            });
        }
        if (!net.isIPv4(props.ip)) {
            throw new Error(`The \`ip\` ${props.ip} is not a valid IPv4 address.`);
        }
        const type = VpnConnectionType.IPSEC_1;
        const bgpAsn = props.asn || 65000;
        const customerGateway = new ec2_generated_1.CfnCustomerGateway(this, 'CustomerGateway', {
            bgpAsn,
            ipAddress: props.ip,
            type,
        });
        this.customerGatewayId = customerGateway.ref;
        this.customerGatewayAsn = bgpAsn;
        this.customerGatewayIp = props.ip;
        // Validate tunnel options
        if (props.tunnelOptions) {
            if (props.tunnelOptions.length > 2) {
                throw new Error('Cannot specify more than two `tunnelOptions`');
            }
            if (props.tunnelOptions.length === 2 && props.tunnelOptions[0].tunnelInsideCidr === props.tunnelOptions[1].tunnelInsideCidr) {
                throw new Error(`Same ${props.tunnelOptions[0].tunnelInsideCidr} \`tunnelInsideCidr\` cannot be used for both tunnels.`);
            }
            props.tunnelOptions.forEach((options, index) => {
                if (options.preSharedKey && !/^[a-zA-Z1-9._][a-zA-Z\d._]{7,63}$/.test(options.preSharedKey)) {
                    // tslint:disable:max-line-length
                    throw new Error(`The \`preSharedKey\` ${options.preSharedKey} for tunnel ${index + 1} is invalid. Allowed characters are alphanumeric characters and ._. Must be between 8 and 64 characters in length and cannot start with zero (0).`);
                    // tslint:enable:max-line-length
                }
                if (options.tunnelInsideCidr) {
                    if (exports.RESERVED_TUNNEL_INSIDE_CIDR.includes(options.tunnelInsideCidr)) {
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is a reserved inside CIDR.`);
                    }
                    if (!/^169\.254\.\d{1,3}\.\d{1,3}\/30$/.test(options.tunnelInsideCidr)) {
                        // tslint:disable:max-line-length
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is not a size /30 CIDR block from the 169.254.0.0/16 range.`);
                        // tslint:enable:max-line-length
                    }
                }
            });
        }
        const vpnConnection = new ec2_generated_1.CfnVPNConnection(this, 'Resource', {
            type,
            customerGatewayId: customerGateway.ref,
            staticRoutesOnly: props.staticRoutes ? true : false,
            vpnGatewayId: props.vpc.vpnGatewayId,
            vpnTunnelOptionsSpecifications: props.tunnelOptions,
        });
        this.vpnId = vpnConnection.ref;
        if (props.staticRoutes) {
            props.staticRoutes.forEach(route => {
                new ec2_generated_1.CfnVPNConnectionRoute(this, `Route${route.replace(/[^\d]/g, '')}`, {
                    destinationCidrBlock: route,
                    vpnConnectionId: this.vpnId,
                });
            });
        }
    }
    /**
     * Return the given named metric for all VPN connections in the account/region.
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/VPN',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the tunnel state of all VPN connections in the account/region.
     *
     * @default average over 5 minutes
     */
    static metricAllTunnelState(props) {
        return this.metricAll('TunnelState', { statistic: 'avg', ...props });
    }
    /**
     * Metric for the tunnel data in of all VPN connections in the account/region.
     *
     * @default sum over 5 minutes
     */
    static metricAllTunnelDataIn(props) {
        return this.metricAll('TunnelDataIn', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the tunnel data out of all VPN connections.
     *
     * @default sum over 5 minutes
     */
    static metricAllTunnelDataOut(props) {
        return this.metricAll('TunnelDataOut', { statistic: 'sum', ...props });
    }
}
exports.VpnConnection = VpnConnection;
exports.RESERVED_TUNNEL_INSIDE_CIDR = [
    '169.254.0.0/30',
    '169.254.1.0/30',
    '169.254.2.0/30',
    '169.254.3.0/30',
    '169.254.4.0/30',
    '169.254.5.0/30',
    '169.254.169.252/30',
];
//# sourceMappingURL=data:application/json;base64,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