"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Describes a block device mapping for an EC2 instance or Auto Scaling group.
 */
class BlockDeviceVolume {
    /**
     * @param ebsDevice EBS device info
     * @param virtualName Virtual device name
     */
    constructor(ebsDevice, virtualName) {
        this.ebsDevice = ebsDevice;
        this.virtualName = virtualName;
    }
    /**
     * Creates a new Elastic Block Storage device
     *
     * @param volumeSize The volume size, in Gibibytes (GiB)
     * @param options additional device options
     */
    static ebs(volumeSize, options = {}) {
        return new this({ ...options, volumeSize });
    }
    /**
     * Creates a new Elastic Block Storage device from an existing snapshot
     *
     * @param snapshotId The snapshot ID of the volume to use
     * @param options additional device options
     */
    static ebsFromSnapshot(snapshotId, options = {}) {
        return new this({ ...options, snapshotId });
    }
    /**
     * Creates a virtual, ephemeral device.
     * The name will be in the form ephemeral{volumeIndex}.
     *
     * @param volumeIndex the volume index. Must be equal or greater than 0
     */
    static ephemeral(volumeIndex) {
        if (volumeIndex < 0) {
            throw new Error(`volumeIndex must be a number starting from 0, got "${volumeIndex}"`);
        }
        return new this(undefined, `ephemeral${volumeIndex}`);
    }
}
exports.BlockDeviceVolume = BlockDeviceVolume;
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== EbsDeviceVolumeType.IO1) {
                construct.node.addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName, ebs, virtualName,
            noDevice: mappingEnabled === false ? {} : undefined,
        };
    });
}
exports.synthesizeBlockDeviceMappings = synthesizeBlockDeviceMappings;
/**
 * Supported EBS volume types for blockDevices
 */
var EbsDeviceVolumeType;
(function (EbsDeviceVolumeType) {
    /**
     * Magnetic
     */
    EbsDeviceVolumeType["STANDARD"] = "standard";
    /**
     *  Provisioned IOPS SSD
     */
    EbsDeviceVolumeType["IO1"] = "io1";
    /**
     * General Purpose SSD
     */
    EbsDeviceVolumeType["GP2"] = "gp2";
    /**
     * Throughput Optimized HDD
     */
    EbsDeviceVolumeType["ST1"] = "st1";
    /**
     * Cold HDD
     */
    EbsDeviceVolumeType["SC1"] = "sc1";
})(EbsDeviceVolumeType = exports.EbsDeviceVolumeType || (exports.EbsDeviceVolumeType = {}));
//# sourceMappingURL=data:application/json;base64,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