"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * A NetworkAclBase that is not created in this template
 *
 * @experimental
 */
class NetworkAclBase extends core_1.Resource {
    /**
     * Add a new entry to the ACL
     */
    addEntry(id, options) {
        return new NetworkAclEntry(this, id, {
            networkAcl: this,
            ...options,
        });
    }
}
/**
 * Define a new custom network ACL
 *
 * By default, will deny all inbound and outbound traffic unless entries are
 * added explicitly allowing it.
 *
 * @experimental
 */
class NetworkAcl extends NetworkAclBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.networkAclName,
        });
        this.vpc = props.vpc;
        this.networkAcl = new ec2_generated_1.CfnNetworkAcl(this, 'Resource', {
            vpcId: props.vpc.vpcId,
        });
        this.networkAclId = this.networkAcl.ref;
        this.networkAclVpcId = this.networkAcl.vpcId;
        if (props.subnetSelection !== undefined) {
            this.associateWithSubnet('DefaultAssociation', props.subnetSelection);
        }
    }
    /**
     * Import an existing NetworkAcl into this app.
     */
    static fromNetworkAclId(scope, id, networkAclId) {
        class Import extends NetworkAclBase {
            constructor() {
                super(...arguments);
                this.networkAclId = networkAclId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Associate the ACL with a given set of subnets
     */
    associateWithSubnet(id, selection) {
        const subnets = this.vpc.selectSubnets(selection);
        for (const subnet of subnets.subnets) {
            subnet.associateNetworkAcl(id, this);
        }
    }
}
exports.NetworkAcl = NetworkAcl;
/**
 * What action to apply to traffic matching the ACL
 *
 * @experimental
 */
var Action;
(function (Action) {
    /**
     * Allow the traffic
     */
    Action["ALLOW"] = "allow";
    /**
     * Deny the traffic
     */
    Action["DENY"] = "deny";
})(Action = exports.Action || (exports.Action = {}));
/**
 * Base class for NetworkAclEntries
 *
 * @experimental
 */
class NetworkAclEntryBase extends core_1.Resource {
}
/**
 * Direction of traffic the AclEntry applies to
 *
 * @experimental
 */
var TrafficDirection;
(function (TrafficDirection) {
    /**
     * Traffic leaving the subnet
     */
    TrafficDirection[TrafficDirection["EGRESS"] = 0] = "EGRESS";
    /**
     * Traffic entering the subnet
     */
    TrafficDirection[TrafficDirection["INGRESS"] = 1] = "INGRESS";
})(TrafficDirection = exports.TrafficDirection || (exports.TrafficDirection = {}));
/**
 * Define an entry in a Network ACL table
 *
 * @experimental
 */
class NetworkAclEntry extends NetworkAclEntryBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.networkAclEntryName,
        });
        this.networkAcl = props.networkAcl;
        new ec2_generated_1.CfnNetworkAclEntry(this, 'Resource', {
            networkAclId: this.networkAcl.networkAclId,
            ruleNumber: props.ruleNumber,
            ruleAction: props.ruleAction !== undefined ? props.ruleAction : Action.ALLOW,
            egress: props.direction !== undefined ? props.direction === TrafficDirection.EGRESS : undefined,
            ...props.traffic.toTrafficConfig(),
            ...props.cidr.toCidrConfig(),
        });
    }
}
exports.NetworkAclEntry = NetworkAclEntry;
/**
 * Associate a network ACL with a subnet
 *
 * @experimental
 */
class SubnetNetworkAclAssociationBase extends core_1.Resource {
}
class SubnetNetworkAclAssociation extends SubnetNetworkAclAssociationBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.subnetNetworkAclAssociationName,
        });
        this.association = new ec2_generated_1.CfnSubnetNetworkAclAssociation(this, 'Resource', {
            networkAclId: props.networkAcl.networkAclId,
            subnetId: props.subnet.subnetId,
        });
        this.networkAcl = props.networkAcl;
        this.subnet = props.subnet;
        this.subnetNetworkAclAssociationAssociationId = this.association.attrAssociationId;
    }
    static fromSubnetNetworkAclAssociationAssociationId(scope, id, subnetNetworkAclAssociationAssociationId) {
        class Import extends SubnetNetworkAclAssociationBase {
            constructor() {
                super(...arguments);
                this.subnetNetworkAclAssociationAssociationId = subnetNetworkAclAssociationAssociationId;
            }
        }
        return new Import(scope, id);
    }
}
exports.SubnetNetworkAclAssociation = SubnetNetworkAclAssociation;
//# sourceMappingURL=data:application/json;base64,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