# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/utils.ipynb.

# %% ../nbs/utils.ipynb 1
# Contains code from:
# fastai - Apache License 2.0 - Copyright (c) 2023 fast.ai
# mish-cuda - MIT License - Copyright (c) 2019 thomasbrandon https://github.com/thomasbrandon/mish-cuda

# %% ../nbs/utils.ipynb 3
from __future__ import annotations

import torch, random, gc
import numpy as np
import PIL.Image as Image

from fastcore.foundation import contextmanager

from fastai.learner import Learner
from fastai.data.core import DataLoaders
from fastai.callback.core import set_random_states, get_random_states

# %% auto 0
__all__ = ['free_gpu_memory', 'less_random', 'scale_time', 'pil_to_numpy']

# %% ../nbs/utils.ipynb 4
def free_gpu_memory(learn:Learner, dls:DataLoaders=None):
    "Frees GPU memory using `gc.collect` and `torch.cuda.empty_cache`"
    learn.dls, learn, dls = None, None, None
    gc.collect()
    torch.cuda.empty_cache()

# %% ../nbs/utils.ipynb 5
@contextmanager
def less_random(
    seed:int=42, # Seed for `random`, `torch`, and `numpy`
    deterministic:bool|None=None, # Set `torch.backends.cudnn.deterministic` if not None
    benchmark:bool|None=None # Set `torch.backends.cudnn.benchmark` if not None
):
    """
    Stores and retrieves state of random number generators. Sets random seed for `random`, `torch`, and `numpy`.

    Does not set `torch.backends.cudnn.benchmark` or `torch.backends.cudnn.deterministic` by default.
    """
    states = get_random_states()

    try: torch.manual_seed(seed)
    except NameError: pass
    try: torch.cuda.manual_seed_all(seed)
    except NameError: pass
    try: np.random.seed(seed%(2**32-1))
    except NameError: pass

    random.seed(seed)
    if deterministic is not None:
        torch.backends.cudnn.deterministic = deterministic
    if benchmark is not None:
        torch.backends.cudnn.benchmark = benchmark

    try:
        yield # we are managing global variables
    finally:
        set_random_states(**states)

# %% ../nbs/utils.ipynb 7
# modified from https://github.com/thomasbrandon/mish-cuda/blob/master/test/perftest.py
def scale_time(val:float, spec:str="#0.4G"):
    "Scale fractional second `time` values and return formatted to `spec`"
    if val == 0:
        return '-'
    PREFIXES = np.array([c for c in u"yzafpnµm kMGTPEZY"])
    exp = np.int8(np.log10(np.abs(val)) // 3 * 3 * np.sign(val))
    val /= 10.**exp
    prefix = PREFIXES[exp//3 + len(PREFIXES)//2]
    return f"{val:{spec}}{prefix}s"

# %% ../nbs/utils.ipynb 8
# From https://uploadcare.com/blog/fast-import-of-pillow-images-to-numpy-opencv-arrays/
# Up to 2.5 times faster with the same functionality and a smaller number of allocations than numpy.asarray(img)
def pil_to_numpy(img:Image.Image) -> np.ndarray:
    "Fast conversion of Pillow `Image` to NumPy NDArray"
    img.load()
    # unpack data
    enc = Image._getencoder(img.mode, 'raw', img.mode)
    enc.setimage(img.im)

    # NumPy buffer for the result
    shape, typestr = Image._conv_type_shape(img)
    data = np.empty(shape, dtype=np.dtype(typestr))
    mem = data.data.cast('B', (data.data.nbytes,))

    bufsize, s, offset = 65536, 0, 0
    while not s:
        l, s, d = enc.encode(bufsize)
        mem[offset:offset + len(d)] = d
        offset += len(d)
    if s < 0:
        raise RuntimeError("encoder error %d in tobytes" % s)
    return data
