# AUTOGENERATED! DO NOT EDIT! File to edit: ../../nbs/callback.compiler.ipynb.

# %% ../../nbs/callback.compiler.ipynb 1
# Contains code from:
# fastai - Apache License 2.0 - Copyright (c) 2023 fast.ai

# %% ../../nbs/callback.compiler.ipynb 4
from __future__ import annotations
from typing import Dict

from enum import Enum
from pathlib import Path
import pickle
import warnings

from packaging.version import parse

import torch._dynamo as dynamo
from torch.serialization import FILE_LIKE

from fastai.learner import Learner, save_model, join_path_file
from fastai.callback.core import Callback, TrainEvalCallback
from fastai.callback import schedule

from ..imports import *

# %% auto 0
__all__ = ['CompileMode', 'MatMulPrecision', 'CompilerCallback', 'load_learner']

# %% ../../nbs/callback.compiler.ipynb 6
_torch_version = parse(torch.__version__)
_torch_20  = parse('2.0')
_torch_21  = parse('2.1')

if _torch_version < _torch_20:
    warn('Imported `fastxtend.callback.compiler`, which requires a minimum of PyTorch 2.0 to work.')

# %% ../../nbs/callback.compiler.ipynb 7
class CompileMode(Enum):
    "All valid `torch.compile` modes for tab-completion and typo-proofing"
    default        = 'default'
    reduceoverhead = 'reduce-overhead'
    maxautotune    = 'max-autotune'

# %% ../../nbs/callback.compiler.ipynb 9
class MatMulPrecision(Enum):
    "All valid `matmul_precision` modes for tab-completion and typo-proofing"
    highest = 'highest'
    high    = 'high'
    medium  = 'medium'

# %% ../../nbs/callback.compiler.ipynb 10
class CompilerCallback(Callback):
    "An experimental callback for `torch.compile` (beta) and fastai"
    order = TrainEvalCallback.order + 1 # Compiling needs to occur on the GPU, but before distributed training starts

    def __init__(self,
        fullgraph:bool=False, # Prevent breaking model into subgraphs
        dynamic:bool=False, # Use dynamic shape tracing
        backend:str|Callable='inductor', # `torch.compile` backend to use
        mode:str|CompileMode|None=None, # `torch.compile` mode to use
        options:Dict[str, Union[str,int,bool]]|None=None, # Extra options to pass to compile backend
        matmul_precision:str|MatMulPrecision='high', # Set Ampere and newer TF32 matmul precision
        recompile:bool=False, # Force a compiled model to recompile. Use when freezing/unfreezing a compiled model.
        verbose:bool=True, # Verbose output
    ):
        if isinstance(mode, CompileMode):
            mode = mode.value
        if isinstance(mode, MatMulPrecision):
            matmul_precision = matmul_precision.value
        if mode is not None and options is not None:
            raise ValueError(f"Both {mode=} or {options=} cannot be set at the same time.")
        store_attr(but='recompile')
        self._recompile = recompile

    def before_fit(self):
        if _torch_version < _torch_20:
            self.run = False
            warn("Attempting to use `CompilerCallback` without PyTorch 2.0 or greater. Disabling.")

        if torch.cuda.get_device_capability() >= (8, 0) and torch.get_float32_matmul_precision() != self.matmul_precision:
            if self.verbose and self.matmul_precision!='highest':
                print(f"Your GPU has modern tensor cores, automatically enabling by setting `torch.set_float32_matmul_precision('{self.matmul_precision}')`")
            torch.set_float32_matmul_precision(self.matmul_precision)

        if hasattr(self.learn, 'lr_finder'):
            self.run = False
            return

        if hasattr(self.learn, 'progressive_resize') and _torch_version < _torch_21:
            warn("Using `ProgressiveResize` and `torch.compile` at the same time will result in a new compile every size change.")
        msg = ""
        if self.dynamic:
            msg += "Using `torch.compile` with dynamic shapes is under active development and might fail\n"
        if self.mode == 'max-autotune':
            msg += "Using `torch.compile` with `mode='max-autotune'` is under active development and might fail\n"
        if msg != "":
            msg += "See https://pytorch.org/docs/master/compile/index.html#troubleshooting-and-gotchas for more details"
            warn(msg)
        if self.mode == 'reduce-overhead':
            warn("Using `torch.compile` & fastai with `mode='reduce-overhead'` currently doesn't appear to train.")
            
        if self._recompile and isinstance(self.learn.model, dynamo.OptimizedModule):
            if self.verbose: 
                print("Recompiling model")
            dynamo.reset()
            self.learn.model = self.learn.model._orig_mod
        self._recompile = False
        
        if not isinstance(self.learn.model, dynamo.OptimizedModule):
            self.learn.model = torch.compile(self.learn.model, fullgraph=self.fullgraph, 
                                             dynamic=self.dynamic, backend=self.backend, 
                                             mode=self.mode, options=self.options)

# %% ../../nbs/callback.compiler.ipynb 13
@patch
def compile(self:Learner,
    fullgraph:bool=False, # Prevent breaking model into subgraphs
    backend:str|Callable='inductor', # `torch.compile` backend to use
    mode:str|CompileMode|None=None, # `torch.compile` mode to use
    options:Dict[str, Union[str,int,bool]]|None=None, # Extra options to pass to compile backend
    matmul_precision:str|MatMulPrecision='high', # Set Ampere and newer TF32 matmul precision
    recompile:bool=False, # Force a compiled model to recompile. Use when freezing/unfreezing a compiled model.
    verbose:bool=True, # Verbose output
):
    "Set `Learner` to compile model using `torch.compile`."
    return self.add_cb(CompilerCallback(fullgraph=fullgraph, backend=backend,
                                        mode=mode, options=options, 
                                        matmul_precision=matmul_precision, 
                                        recompile=recompile, verbose=verbose))

# %% ../../nbs/callback.compiler.ipynb 17
@patch
@delegates(save_model)
def save(self:Learner, 
    file:FILE_LIKE, # Save file name, path, bytes, or IO
    save_compiled:bool=False, # Save compiled model
    **kwargs
):
    "Save model and optimizer state (if `with_opt`) to `self.path/self.model_dir/file`"
    file = join_path_file(file, self.path/self.model_dir, ext='.pth')
    if _torch_version >= _torch_20 and isinstance(self.model, dynamo.OptimizedModule) and not save_compiled:
        save_model(file, self.model._orig_mod, getattr(self,'opt',None), **kwargs)
    else:
        save_model(file, self.model, getattr(self,'opt',None), **kwargs)
    return file

# %% ../../nbs/callback.compiler.ipynb 19
@patch
def export(self:Learner, 
    fname:FILE_LIKE='export.pkl', # Learner export file name, path, bytes, or IO
    pickle_module:Any=pickle, # Module used for pickling metadata and objects
    pickle_protocol:int=2 # Pickle protocol used
):
    "Export the content of `self` without the items and the optimizer state for inference"
    if rank_distrib(): return # don't export if child proc
    self._end_cleanup()
    old_dbunch = self.dls
    self.dls = self.dls.new_empty()
    state = self.opt.state_dict() if self.opt is not None else None
    self.opt = None
    # torch.compiled models currently cannot be pickled.
    if _torch_version >= _torch_20 and isinstance(self.model, dynamo.OptimizedModule):
        self.model = self.model._orig_mod
    with warnings.catch_warnings():
        # To avoid the warning that come from PyTorch about model not being checked
        warnings.simplefilter("ignore")
        torch.save(self, self.path/fname, pickle_module=pickle_module, pickle_protocol=pickle_protocol)
    self.create_opt()
    if state is not None: self.opt.load_state_dict(state)
    self.dls = old_dbunch

# %% ../../nbs/callback.compiler.ipynb 21
def load_learner(
    fname:FILE_LIKE, # File name, path, bytes, or IO
    cpu:bool=True, # Load model to CPU
    pickle_module=pickle # Module used for unpickling metadata and objects
):
    "Load a `Learner` object in `fname`, by default putting it on the `cpu`"
    distrib_barrier()
    map_loc = 'cpu' if cpu else default_device()
    try: res = torch.load(fname, map_location=map_loc, pickle_module=pickle_module)
    except AttributeError as e: 
        e.args = [f"Custom classes or functions exported with your `Learner` not available in namespace.\Re-declare/import before loading:\n\t{e.args[0]}"]
        raise
    if cpu: 
        res.dls.cpu()
        if hasattr(res, 'channels_last'): res = res.to_contiguous(to_fp32=True)
        elif hasattr(res, 'mixed_precision'): res = res.to_fp32()
        elif hasattr(res, 'non_native_mixed_precision'): res = res.to_non_native_fp32()
        if hasattr(res, 'compiler'): res = res.remove_cb(CompilerCallback)
    return res

# %% ../../nbs/callback.compiler.ipynb 24
@patch
def freeze_to(self:Learner, n:int):
    "Freeze parameter groups up to `n`"
    if self.opt is None: 
        self.create_opt()
    self.opt.freeze_to(n)
    self.opt.clear_state()
    if _torch_version >= _torch_20 and isinstance(self.model, dynamo.OptimizedModule):
        if hasattr(self, 'compiler'):
            self.compiler._recompile = True
        else:
            warn("Freezing or unfreezing a compiled model isn't supported."\
                 "\nThe model must be recompiled to take effect."\
                 "\nPass `CompilerCallback(..., recompile=True)` to `Learner.cbs`"\
                 "\nor call `torch._dynamo.reset() and recompile model.")

# %% ../../nbs/callback.compiler.ipynb 27
@patch
@delegates(Learner.fit_one_cycle)
def fine_tune(self:Learner, 
    epochs:int, # Number of unfrozen epochs to train
    base_lr:float=2e-3, # Base learning rate, model head unfrozen learning rate
    freeze_epochs:int=1, # Number of frozen epochs to train
    lr_mult:Numeric=100, # Model stem unfrozen learning rate: `base_lr/lr_mult``
    pct_start:float=0.3, # Start unfrozen learning rate cosine annealing
    div:Numeric=5.0, # Initial unfrozen learning rate: `base_lr/div`
    freeze_compile:bool=False, # pct_start for unfrozen fit_one_cycle
    **kwargs
):
    "Fine tune with `Learner.freeze` for `freeze_epochs`, then with `Learner.unfreeze` for `epochs`, using discriminative LR."
    self.freeze()
    if _torch_version >= _torch_20 and hasattr(self, 'compiler') and not freeze_compile:
            self.compiler.run = isinstance(self.model, dynamo.OptimizedModule)
    self.fit_one_cycle(freeze_epochs, slice(base_lr), pct_start=0.99, **kwargs)
    base_lr /= 2
    self.unfreeze()
    self.fit_one_cycle(epochs, slice(base_lr/lr_mult, base_lr), pct_start=pct_start, div=div, **kwargs)
