"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var index_1 = require("../index");
var concat_1 = require("./concat");
function interpolate(y, from, to) {
    return (from * (1 - y) + to * y);
}
function flip(fn) {
    return function (x) { return 1 - fn(1 - x); };
}
function createEasing(fn) {
    var fnFlipped = flip(fn);
    return {
        easeIn: function (x, from, to) {
            return interpolate(fn(x), from, to);
        },
        easeOut: function (x, from, to) {
            return interpolate(fnFlipped(x), from, to);
        },
        easeInOut: function (x, from, to) {
            var y = (x < 0.5) ?
                (fn(2 * x) * 0.5) :
                (0.5 + fnFlipped(2 * (x - 0.5)) * 0.5);
            return interpolate(y, from, to);
        }
    };
}
var easingPower2 = createEasing(function (x) { return x * x; });
var easingPower3 = createEasing(function (x) { return x * x * x; });
var easingPower4 = createEasing(function (x) {
    var xx = x * x;
    return xx * xx;
});
var EXP_WEIGHT = 6;
var EXP_MAX = Math.exp(EXP_WEIGHT) - 1;
function expFn(x) {
    return (Math.exp(x * EXP_WEIGHT) - 1) / EXP_MAX;
}
var easingExponential = createEasing(expFn);
var OVERSHOOT = 1.70158;
var easingBack = createEasing(function (x) { return x * x * ((OVERSHOOT + 1) * x - OVERSHOOT); });
var PARAM1 = 7.5625;
var PARAM2 = 2.75;
function easeOutFn(x) {
    var z = x;
    if (z < 1 / PARAM2) {
        return (PARAM1 * z * z);
    }
    else if (z < 2 / PARAM2) {
        return (PARAM1 * (z -= 1.5 / PARAM2) * z + 0.75);
    }
    else if (z < 2.5 / PARAM2) {
        return (PARAM1 * (z -= 2.25 / PARAM2) * z + 0.9375);
    }
    else {
        return (PARAM1 * (z -= 2.625 / PARAM2) * z + 0.984375);
    }
}
var easingBounce = createEasing(function (x) { return 1 - easeOutFn(1 - x); });
var easingCirc = createEasing(function (x) { return -(Math.sqrt(1 - x * x) - 1); });
var PERIOD = 0.3;
var OVERSHOOT_ELASTIC = PERIOD / 4;
var AMPLITUDE = 1;
function elasticIn(x) {
    var z = x;
    if (z <= 0) {
        return 0;
    }
    else if (z >= 1) {
        return 1;
    }
    else {
        z -= 1;
        return -(AMPLITUDE * Math.pow(2, 10 * z))
            * Math.sin((z - OVERSHOOT_ELASTIC) * (2 * Math.PI) / PERIOD);
    }
}
var easingElastic = createEasing(elasticIn);
var HALF_PI = Math.PI * 0.5;
var easingSine = createEasing(function (x) { return 1 - Math.cos(x * HALF_PI); });
var DEFAULT_INTERVAL = 15;
/**
 * Creates a stream of numbers emitted in a quick burst, following a numeric
 * function like sine or elastic or quadratic. tween() is meant for creating
 * streams for animations.
 *
 * Example:
 *
 * ```js
 * import tween from 'xstream/extra/tween'
 *
 * const stream = tween({
 *   from: 20,
 *   to: 100,
 *   ease: tween.exponential.easeIn,
 *   duration: 1000, // milliseconds
 * })
 *
 * stream.addListener({
 *   next: (x) => console.log(x),
 *   error: (err) => console.error(err),
 *   complete: () => console.log('concat completed'),
 * })
 * ```
 *
 * The stream would behave like the plot below:
 *
 * ```text
 * 100                  #
 * |
 * |
 * |
 * |
 * 80                  #
 * |
 * |
 * |
 * |                  #
 * 60
 * |
 * |                 #
 * |
 * |                #
 * 40
 * |               #
 * |              #
 * |            ##
 * |         ###
 * 20########
 * +---------------------> time
 * ```
 *
 * Provide a configuration object with **from**, **to**, **duration**, **ease**,
 * **interval** (optional), and this factory function will return a stream of
 * numbers following that pattern. The first number emitted will be `from`, and
 * the last number will be `to`. The numbers in between follow the easing
 * function you specify in `ease`, and the stream emission will last in total
 * `duration` milliseconds.
 *
 * The easing functions are attached to `tween` too, such as
 * `tween.linear.ease`, `tween.power2.easeIn`, `tween.exponential.easeOut`, etc.
 * Here is a list of all the available easing options:
 *
 * - `tween.linear` with ease
 * - `tween.power2` with easeIn, easeOut, easeInOut
 * - `tween.power3` with easeIn, easeOut, easeInOut
 * - `tween.power4` with easeIn, easeOut, easeInOut
 * - `tween.exponential` with easeIn, easeOut, easeInOut
 * - `tween.back` with easeIn, easeOut, easeInOut
 * - `tween.bounce` with easeIn, easeOut, easeInOut
 * - `tween.circular` with easeIn, easeOut, easeInOut
 * - `tween.elastic` with easeIn, easeOut, easeInOut
 * - `tween.sine` with easeIn, easeOut, easeInOut
 *
 * @factory true
 * @param {TweenConfig} config An object with properties `from: number`,
 * `to: number`, `duration: number`, `ease: function` (optional, defaults to
 * linear), `interval: number` (optional, defaults to 15).
 * @return {Stream}
 */
function tween(_a) {
    var from = _a.from, to = _a.to, duration = _a.duration, _b = _a.ease, ease = _b === void 0 ? tweenFactory.linear.ease : _b, _c = _a.interval, interval = _c === void 0 ? DEFAULT_INTERVAL : _c;
    var totalTicks = Math.round(duration / interval);
    return index_1.Stream.periodic(interval)
        .take(totalTicks)
        .map(function (tick) { return ease(tick / totalTicks, from, to); })
        .compose(function (s) { return concat_1.default(s, index_1.Stream.of(to)); });
}
var tweenFactory = tween;
tweenFactory.linear = { ease: interpolate };
tweenFactory.power2 = easingPower2;
tweenFactory.power3 = easingPower3;
tweenFactory.power4 = easingPower4;
tweenFactory.exponential = easingExponential;
tweenFactory.back = easingBack;
tweenFactory.bounce = easingBounce;
tweenFactory.circular = easingCirc;
tweenFactory.elastic = easingElastic;
tweenFactory.sine = easingSine;
exports.default = tweenFactory;
//# sourceMappingURL=data:application/json;base64,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