#!/usr/bin/env python3

import os
import sys
import subprocess

argv = sys.argv[1:]
assert len(argv) == 2, 'Must provide: name, path'
name, path = argv

os.mkdir(os.path.join(path, 'script'))

templates = {
	"bootstrap": '''#!/bin/bash
set -e
set -x

which python3 || which python
#which virtualenv
which poetry

poetry install''',
	
	"console": '''#!/bin/bash
set -e
set -x

poetry run python manage.py shell''',

	"server": '''#!/bin/bash
set -e
set -x

poetry run python manage.py runserver 0.0.0.0:8000''',

	"setup": '''#!/bin/bash
set -e
set -x

script/bootstrap

# Create .env if not present
MAIN_FOLDER=main
if [ ! -e $MAIN_FOLDER/.env ]; then
	# Copy template
	cp $MAIN_FOLDER/.env.example $MAIN_FOLDER/.env
	# Create secret key (TODO: find a crypto-safer way like /dev/random)
	SECRET=$(mcookie)
	sed -i 's|DJANGO_SECRET_KEY=""|DJANGO_SECRET_KEY="'$SECRET'"|g' $MAIN_FOLDER/.env
	# Show
	cat $MAIN_FOLDER/.env
	echo ".env has been autogenerated, to edit: $ nano $MAIN_FOLDER/.env"
fi
# Clear SQLite database if present
if [ -e db.sqlite3 ]; then
        mv db.sqlite3 db.sqlite3~
fi
# Run migrations
poetry run python manage.py migrate''',

	"test": '''#!/bin/bash
set -e
set -x

make lint
poetry run pytest''',

	"update": '''#!/bin/bash
set -e
set -x

script/bootstrap
poetry run python manage.py migrate''',

	"cibuild": '''#!/bin/bash
set -e
set -x

export CI=true
script/test''',
}

import shlex

for key in templates:
	print(f'Writing script/{key}...')
	with open(os.path.join(path, 'script', key), 'w') as f:
		f.write(templates[key])
	subprocess.run(f"chmod +x {shlex.quote(os.path.join(path, 'script', key))}", shell=True)
		       
symlinks = {
	'shell': 'console',
	'run': 'server',
}

for key in symlinks:
	fp = os.path.join(path, 'script')
	print(f'Symlink: {key} -> {symlinks[key]}')
	subprocess.run(f'cd {shlex.quote(fp)}; ln -s {symlinks[key]} {key}', shell=True)
