"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const index_1 = require("../lib/index");
const core_1 = require("@aws-cdk/core");
require("@aws-cdk/assert/jest");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
test('Create the Runner', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    new index_1.GitlabContainerRunner(stack, 'testing', { gitlabtoken: 'GITLAB_TOKEN' });
    expect(stack).toHaveResource('AWS::EC2::Instance');
    expect(stack).toHaveResource('AWS::IAM::Role');
    expect(stack).toHaveResource('AWS::EC2::SecurityGroup', {
        SecurityGroupEgress: [
            {
                CidrIp: '0.0.0.0/0',
                Description: 'Allow all outbound traffic by default',
                IpProtocol: '-1',
            },
        ],
    });
});
test('Testing runner tag change ', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    new index_1.GitlabContainerRunner(stack, 'testing-have-type-tag', {
        gitlabtoken: 'GITLAB_TOKEN',
        tag1: 'aa',
        tag2: 'bb',
        tag3: 'cc',
    });
    expect(stack).toHaveResource('AWS::EC2::Instance');
    expect(stack).toHaveResource('AWS::EC2::SecurityGroup', {
        SecurityGroupEgress: [
            {
                CidrIp: '0.0.0.0/0',
                Description: 'Allow all outbound traffic by default',
                IpProtocol: '-1',
            },
        ],
    });
});
test('Testing Runner Instance Type Change ', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    new index_1.GitlabContainerRunner(stack, 'testing', {
        gitlabtoken: 'GITLAB_TOKEN',
        ec2type: 't2.micro',
    });
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        InstanceType: 't2.micro',
    });
    expect(stack).toHaveResource('AWS::EC2::SecurityGroup', {
        SecurityGroupEgress: [
            {
                CidrIp: '0.0.0.0/0',
                Description: 'Allow all outbound traffic by default',
                IpProtocol: '-1',
            },
        ],
    });
});
test('Runner Can Add Ingress ', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    const runner = new index_1.GitlabContainerRunner(stack, 'testing', {
        gitlabtoken: 'GITLAB_TOKEN',
        ec2type: 't2.micro',
        tag1: 'aa',
        tag2: 'bb',
        tag3: 'cc',
    });
    runner.runnerEc2.connections.allowFrom(aws_ec2_1.Peer.ipv4('1.2.3.4/8'), aws_ec2_1.Port.tcp(80));
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        InstanceType: 't2.micro',
    });
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        CidrBlock: '10.0.0.0/16',
    });
    expect(stack).toHaveResource('AWS::EC2::SecurityGroup', {
        SecurityGroupIngress: [
            {
                CidrIp: '0.0.0.0/0',
                Description: 'from 0.0.0.0/0:22',
                FromPort: 22,
                IpProtocol: 'tcp',
                ToPort: 22,
            },
            {
                CidrIp: '1.2.3.4/8',
                Description: 'from 1.2.3.4/8:80',
                FromPort: 80,
                IpProtocol: 'tcp',
                ToPort: 80,
            },
        ],
    });
});
test('Runner Can Use Self VPC ', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    const newvpc = new aws_ec2_1.Vpc(stack, 'NEWVPC', {
        cidr: '10.1.0.0/16',
        maxAzs: 2,
        subnetConfiguration: [
            {
                cidrMask: 26,
                name: 'RunnerVPC',
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            },
        ],
        natGateways: 0,
    });
    new index_1.GitlabContainerRunner(stack, 'testing', {
        gitlabtoken: 'GITLAB_TOKEN',
        ec2type: 't2.micro',
        selfvpc: newvpc,
    });
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        InstanceType: 't2.micro',
    });
    expect(stack).not.toHaveResource('AWS::S3::Bucket');
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        CidrBlock: '10.1.0.0/16',
    });
});
test('Runner Can Use Self Role ', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    const role = new aws_iam_1.Role(stack, 'runner-role', {
        assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
        description: 'For Gitlab EC2 Runner Test Role',
        roleName: 'TestRole',
    });
    new index_1.GitlabContainerRunner(stack, 'testing', {
        gitlabtoken: 'GITLAB_TOKEN',
        ec2type: 't2.micro',
        ec2iamrole: role,
    });
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        InstanceType: 't2.micro',
    });
    expect(stack).toHaveResource('AWS::IAM::Role', {
        RoleName: 'TestRole',
    });
});
test('Can Use Coustom Gitlab Url', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    const role = new aws_iam_1.Role(stack, 'runner-role', {
        assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
        description: 'For Gitlab EC2 Runner Test Role',
        roleName: 'TestRole',
    });
    new index_1.GitlabContainerRunner(stack, 'testing', {
        gitlabtoken: 'GITLAB_TOKEN',
        ec2type: 't2.micro',
        ec2iamrole: role,
        gitlaburl: 'https://gitlab.my.com/',
    });
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        InstanceType: 't2.micro',
    });
    expect(stack).toHaveResource('AWS::EC2::Instance');
});
test('Can Use Coustom EBS Size', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    new index_1.GitlabContainerRunner(stack, 'testing', {
        gitlabtoken: 'GITLAB_TOKEN',
        ebsSize: 50,
    });
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        BlockDeviceMappings: [
            {
                DeviceName: '/dev/xvda',
                Ebs: {
                    VolumeSize: 50,
                },
            },
        ],
    });
});
test('Can Use Spotfleet Runner', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-spotfleet');
    const testspot = new index_1.GitlabContainerRunner(stack, 'testing', {
        gitlabtoken: 'GITLAB_TOKEN',
        spotFleet: true,
        instanceInterruptionBehavior: index_1.InstanceInterruptionBehavior.HIBERNATE,
        ebsSize: 100,
        blockDuration: index_1.BlockDuration.SIXTEEN_HOURS,
        vpcSubnet: {
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        },
    });
    testspot.expireAfter(core_1.Duration.hours(6));
    expect(stack).toHaveResource('AWS::EC2::SpotFleet');
});
test('Can Use Spotfleet Runner None ', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-spotfleet');
    const testspot = new index_1.GitlabContainerRunner(stack, 'testing', {
        gitlabtoken: 'GITLAB_TOKEN',
        spotFleet: true,
    });
    testspot.expireAfter(core_1.Duration.hours(6));
    expect(stack).toHaveResource('AWS::EC2::SpotFleet');
});
//# sourceMappingURL=data:application/json;base64,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