# cloudofficeprint-python
This project provides a Python interface for Cloud Office Print.

# Installation
To install the Cloud Office Print Python package, you can type in your terminal:  
`pip install cloudofficeprint`

# Usage
1. Create a template (docx, xlsx, pptx, HTML, md, txt, csv), for the possible tags, click [here](http://www.cloudofficeprint.com/docs/#templates).
2. Create the input data with this Python SDK
3. Send template and data to a Cloud Office Print server and save the response to a file with this Python SDK

To see the JSON-data that is sent to the Cloud Office Print server, you can turn on verbose-mode by setting the argument `cop_verbose` to `True` when creating a `PrintJob`.

# Quickstart: Hello World example
## Template (docx)
<img src="https://raw.githubusercontent.com/United-Codes/cloudofficeprint-python/master/imgs/hello_world_template.png" width="600" />

## Data
The data can be generated by the Python SDK:
```python
import cloudofficeprint as cop

# Main object that holds the data
collection = cop.elements.ElementCollection()

# Create the title element and add it to the element collection
title = cop.elements.Property(
    name='title',
    value='Hello World!'
)
collection.add(title)

# Create the text element and add it to the element collection
text = cop.elements.Property(
    name='text',
    value='This is an example created with the Cloud Office Print Python SDK'
)
collection.add(text)
...
```

## Cloud Office Print server
The template and the data need to be sent to a Cloud Office Print server that merges both. This can be done by setting up the configuration for a Cloud Office Print server and passing it to the print job instance. You can get your API key by signing up at https://www.cloudofficeprint.com.
```python
...
SERVER_URL = "https://api.cloudofficeprint.com/"
API_KEY = "YOUR_API_KEY"  # Replace by your own API key

# Add server
server = cop.config.Server(
    SERVER_URL,
    cop.config.ServerConfig(api_key=API_KEY)
)

# Create print job
printjob = cop.PrintJob(
    data=collection,
    server=server,
    template=cop.Resource.from_local_file('PATH_TO_TEMPLATE_FILE'),
)

# Execute print job and save response to file
response = printjob.execute()
response.to_file('PATH_OF_OUTPUT_FILE')
```
where `PATH_TO_TEMPLATE_FILE` is the (relative) path to your template file (e.g. `./data/template.docx`) and `PATH_OF_OUTPUT_FILE` is the (relative) path where you want the output file to be stored (e.g. `./output/output_file`).

## Result
<img src="https://raw.githubusercontent.com/United-Codes/cloudofficeprint-python/master/imgs/hello_world_output.png" width="600" />

# Other examples
Going through the other examples is also recommended as this gives you a good idea on how to use the SDK. The current examples are:
- order_confirmation_example.py
- pdfsignature_example.py
- solar_system_example.py
- spacex_example.py (the most extensive example)
- multiple_request_merge_example.py

The examples can be found in the parent directory of the project on [Github](https://github.com/United-Codes/cloudofficeprint-python). Each example also has its own folder containing the used templates, the generated output files and a markdown file with explanation.

# Documentation

The documentation for this SDK can be found inside the `docs/cloudofficeprint/` folder on [Github](https://github.com/United-Codes/cloudofficeprint-python). Open the index.html file in a browser.

The documentation for Cloud Office Print can be found at the [Cloud Office Print docs](https://www.cloudofficeprint.com/docs/).


# Development

## Documentation
To generate the documentation, you can run the following command (after installing pdoc: `pip install pdoc3`) in the project directory:
```bash
pdoc --html --force --output-dir docs/ cloudofficeprint/
```
[pdoc](https://pdoc3.github.io/pdoc/) is used for documentation generation.
Things to keep in mind when writing docs (some of these are non-standard):
- Docstrings are inherited from `super()`.
- Instance variables (attributes) can have docstrings, start the docstring on the line *underneath* the attribute
- For `@property` properties, the setter's documentation is ignored. Make sure everything is in the getter.
- You can use markdown in the docstrings, along with the generated google-style docs.
  - Doing something like \``ClassName`\` (with backticks, which are generally for inline code) makes pdoc look for the reference and try to hyperlink it in the generated docs.</br>
  This works with any object, useful methods or instance variables of a class too.

## Tests
There are tests for all classes and methods. The tests check if the JSON that needs to be sent to the server is as expected. To run the tests:
1. Open a terminal in the parent directory of this project, which can be found on [Github](https://github.com/United-Codes/cloudofficeprint-python).
2. Type in the terminal:
    ```bash
    python test.py
    ```
3. The tests succeeded if nothing is printed to the terminal

## Useful Visual Studio Code extensions
- `njpwerner.autodocstring`: Python docstring generator, uses Google-style docs by default.

## Publishing
To publish this project to [PyPI](https://pypi.org/), follow these steps:
1. Create an account on PyPI.
2. Install the [Twine utility](https://twine.readthedocs.io/en/latest/) by typing in your terminal:
    ```bash
    pip install twine
    ```
3. In the parent directory, type in your terminal:
    ```bash
    python setup.py sdist bdist_wheel
    ```
4. Check if the building succeeded (PASSED):
    ```bash
    twine check dist/*
    ```
5. Upload to PyPI:
    ```bash
    twine upload dist/*
    ```
More information can be found [here](https://realpython.com/pypi-publish-python-package).