"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const certificatemanager = require("@aws-cdk/aws-certificatemanager");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const nodeunit_shim_1 = require("nodeunit-shim");
const lib_1 = require("../lib");
/* eslint-disable quote-props */
nodeunit_shim_1.nodeunitShim({
    'distribution with custom origin adds custom origin'(test) {
        const stack = new cdk.Stack();
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    originHeaders: {
                        'X-Custom-Header': 'somevalue',
                    },
                    customOriginSource: {
                        domainName: 'myorigin.com',
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                        },
                    ],
                },
            ],
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'AnAmazingWebsiteProbablyCFDistribution47E3983B': {
                    'Type': 'AWS::CloudFront::Distribution',
                    'Properties': {
                        'DistributionConfig': {
                            'DefaultCacheBehavior': {
                                'AllowedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'CachedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'ForwardedValues': {
                                    'Cookies': {
                                        'Forward': 'none',
                                    },
                                    'QueryString': false,
                                },
                                'TargetOriginId': 'origin1',
                                'ViewerProtocolPolicy': 'redirect-to-https',
                                'Compress': true,
                            },
                            'DefaultRootObject': 'index.html',
                            'Enabled': true,
                            'HttpVersion': 'http2',
                            'IPV6Enabled': true,
                            'Origins': [
                                {
                                    'CustomOriginConfig': {
                                        'HTTPPort': 80,
                                        'HTTPSPort': 443,
                                        'OriginKeepaliveTimeout': 5,
                                        'OriginProtocolPolicy': 'https-only',
                                        'OriginReadTimeout': 30,
                                        'OriginSSLProtocols': [
                                            'TLSv1.2',
                                        ],
                                    },
                                    'ConnectionAttempts': 3,
                                    'ConnectionTimeout': 10,
                                    'DomainName': 'myorigin.com',
                                    'Id': 'origin1',
                                    'OriginCustomHeaders': [
                                        {
                                            'HeaderName': 'X-Custom-Header',
                                            'HeaderValue': 'somevalue',
                                        },
                                    ],
                                },
                            ],
                            'PriceClass': 'PriceClass_100',
                            'ViewerCertificate': {
                                'CloudFrontDefaultCertificate': true,
                            },
                        },
                    },
                },
            },
        });
        test.done();
    },
    'most basic distribution'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket,
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                        },
                    ],
                },
            ],
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'Bucket83908E77': {
                    'Type': 'AWS::S3::Bucket',
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
                'AnAmazingWebsiteProbablyCFDistribution47E3983B': {
                    'Type': 'AWS::CloudFront::Distribution',
                    'Properties': {
                        'DistributionConfig': {
                            'DefaultRootObject': 'index.html',
                            'Origins': [
                                {
                                    'ConnectionAttempts': 3,
                                    'ConnectionTimeout': 10,
                                    'DomainName': {
                                        'Fn::GetAtt': [
                                            'Bucket83908E77',
                                            'RegionalDomainName',
                                        ],
                                    },
                                    'Id': 'origin1',
                                    'S3OriginConfig': {},
                                },
                            ],
                            'ViewerCertificate': {
                                'CloudFrontDefaultCertificate': true,
                            },
                            'PriceClass': 'PriceClass_100',
                            'DefaultCacheBehavior': {
                                'AllowedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'CachedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'TargetOriginId': 'origin1',
                                'ViewerProtocolPolicy': 'redirect-to-https',
                                'ForwardedValues': {
                                    'QueryString': false,
                                    'Cookies': { 'Forward': 'none' },
                                },
                                'Compress': true,
                            },
                            'Enabled': true,
                            'IPV6Enabled': true,
                            'HttpVersion': 'http2',
                        },
                    },
                },
            },
        });
        test.done();
    },
    'distribution with trusted signers on default distribution'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket,
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                            trustedSigners: ['1234'],
                        },
                    ],
                },
            ],
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'Bucket83908E77': {
                    'Type': 'AWS::S3::Bucket',
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
                'AnAmazingWebsiteProbablyCFDistribution47E3983B': {
                    'Type': 'AWS::CloudFront::Distribution',
                    'Properties': {
                        'DistributionConfig': {
                            'DefaultRootObject': 'index.html',
                            'Origins': [
                                {
                                    'ConnectionAttempts': 3,
                                    'ConnectionTimeout': 10,
                                    'DomainName': {
                                        'Fn::GetAtt': [
                                            'Bucket83908E77',
                                            'RegionalDomainName',
                                        ],
                                    },
                                    'Id': 'origin1',
                                    'S3OriginConfig': {},
                                },
                            ],
                            'ViewerCertificate': {
                                'CloudFrontDefaultCertificate': true,
                            },
                            'PriceClass': 'PriceClass_100',
                            'DefaultCacheBehavior': {
                                'AllowedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'CachedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'TargetOriginId': 'origin1',
                                'ViewerProtocolPolicy': 'redirect-to-https',
                                'ForwardedValues': {
                                    'QueryString': false,
                                    'Cookies': { 'Forward': 'none' },
                                },
                                'TrustedSigners': [
                                    '1234',
                                ],
                                'Compress': true,
                            },
                            'Enabled': true,
                            'IPV6Enabled': true,
                            'HttpVersion': 'http2',
                        },
                    },
                },
            },
        });
        test.done();
    },
    'distribution with ViewerProtocolPolicy set to a non-default value'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            viewerProtocolPolicy: lib_1.ViewerProtocolPolicy.ALLOW_ALL,
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket,
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                        },
                    ],
                },
            ],
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'Bucket83908E77': {
                    'Type': 'AWS::S3::Bucket',
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
                'AnAmazingWebsiteProbablyCFDistribution47E3983B': {
                    'Type': 'AWS::CloudFront::Distribution',
                    'Properties': {
                        'DistributionConfig': {
                            'DefaultRootObject': 'index.html',
                            'Origins': [
                                {
                                    'ConnectionAttempts': 3,
                                    'ConnectionTimeout': 10,
                                    'DomainName': {
                                        'Fn::GetAtt': [
                                            'Bucket83908E77',
                                            'RegionalDomainName',
                                        ],
                                    },
                                    'Id': 'origin1',
                                    'S3OriginConfig': {},
                                },
                            ],
                            'ViewerCertificate': {
                                'CloudFrontDefaultCertificate': true,
                            },
                            'PriceClass': 'PriceClass_100',
                            'DefaultCacheBehavior': {
                                'AllowedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'CachedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'TargetOriginId': 'origin1',
                                'ViewerProtocolPolicy': 'allow-all',
                                'ForwardedValues': {
                                    'QueryString': false,
                                    'Cookies': { 'Forward': 'none' },
                                },
                                'Compress': true,
                            },
                            'Enabled': true,
                            'IPV6Enabled': true,
                            'HttpVersion': 'http2',
                        },
                    },
                },
            },
        });
        test.done();
    },
    'distribution with disabled compression'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket,
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                            compress: false,
                        },
                    ],
                },
            ],
        });
        assert_1.expect(stack).toMatch({
            'Resources': {
                'Bucket83908E77': {
                    'Type': 'AWS::S3::Bucket',
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
                'AnAmazingWebsiteProbablyCFDistribution47E3983B': {
                    'Type': 'AWS::CloudFront::Distribution',
                    'Properties': {
                        'DistributionConfig': {
                            'DefaultRootObject': 'index.html',
                            'Origins': [
                                {
                                    'ConnectionAttempts': 3,
                                    'ConnectionTimeout': 10,
                                    'DomainName': {
                                        'Fn::GetAtt': [
                                            'Bucket83908E77',
                                            'RegionalDomainName',
                                        ],
                                    },
                                    'Id': 'origin1',
                                    'S3OriginConfig': {},
                                },
                            ],
                            'ViewerCertificate': {
                                'CloudFrontDefaultCertificate': true,
                            },
                            'PriceClass': 'PriceClass_100',
                            'DefaultCacheBehavior': {
                                'AllowedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'CachedMethods': [
                                    'GET',
                                    'HEAD',
                                ],
                                'TargetOriginId': 'origin1',
                                'ViewerProtocolPolicy': 'redirect-to-https',
                                'ForwardedValues': {
                                    'QueryString': false,
                                    'Cookies': { 'Forward': 'none' },
                                },
                                'Compress': false,
                            },
                            'Enabled': true,
                            'IPV6Enabled': true,
                            'HttpVersion': 'http2',
                        },
                    },
                },
            },
        });
        test.done();
    },
    'distribution with resolvable lambda-association'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        const lambdaFunction = new lambda.SingletonFunction(stack, 'Lambda', {
            uuid: 'xxxx-xxxx-xxxx-xxxx',
            code: lambda.Code.inline('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket,
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                            lambdaFunctionAssociations: [{
                                    eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                                    lambdaFunction: lambdaFunction.latestVersion,
                                }],
                        },
                    ],
                },
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
            'DistributionConfig': {
                'DefaultCacheBehavior': {
                    'LambdaFunctionAssociations': [
                        {
                            'EventType': 'origin-request',
                            'LambdaFunctionARN': {
                                'Fn::Join': [
                                    '',
                                    [
                                        { 'Fn::GetAtt': ['SingletonLambdaxxxxxxxxxxxxxxxx69D4268A', 'Arn'] },
                                        ':$LATEST',
                                    ],
                                ],
                            },
                        },
                    ],
                },
            },
        }));
        test.done();
    },
    'distribution has a defaultChild'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        const distribution = new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket,
                    },
                    behaviors: [{ isDefaultBehavior: true }],
                },
            ],
        });
        test.ok(distribution.node.defaultChild instanceof lib_1.CfnDistribution);
        test.done();
    },
    'allows multiple aliasConfiguration CloudFrontWebDistribution per stack'(test) {
        const stack = new cdk.Stack();
        const s3BucketSource = new s3.Bucket(stack, 'Bucket');
        const originConfigs = [{
                s3OriginSource: { s3BucketSource },
                behaviors: [{ isDefaultBehavior: true }],
            }];
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs,
            aliasConfiguration: { acmCertRef: 'acm_ref', names: ['www.example.com'] },
        });
        new lib_1.CloudFrontWebDistribution(stack, 'AnotherAmazingWebsiteProbably', {
            originConfigs,
            aliasConfiguration: { acmCertRef: 'another_acm_ref', names: ['ftp.example.com'] },
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
            'DistributionConfig': {
                'Aliases': ['www.example.com'],
                'ViewerCertificate': {
                    'AcmCertificateArn': 'acm_ref',
                    'SslSupportMethod': 'sni-only',
                },
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
            'DistributionConfig': {
                'Aliases': ['ftp.example.com'],
                'ViewerCertificate': {
                    'AcmCertificateArn': 'another_acm_ref',
                    'SslSupportMethod': 'sni-only',
                },
            },
        }));
        test.done();
    },
    'viewerCertificate': {
        'acmCertificate': {
            'base usage'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                const certificate = new certificatemanager.Certificate(stack, 'cert', {
                    domainName: 'example.com',
                });
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromAcmCertificate(certificate),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    'DistributionConfig': {
                        'Aliases': [],
                        'ViewerCertificate': {
                            'AcmCertificateArn': {
                                'Ref': 'cert56CA94EB',
                            },
                            'SslSupportMethod': 'sni-only',
                        },
                    },
                }));
                test.done();
            },
            'imported certificate fromCertificateArn'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                const certificate = certificatemanager.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromAcmCertificate(certificate),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    'DistributionConfig': {
                        'Aliases': [],
                        'ViewerCertificate': {
                            'AcmCertificateArn': 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d',
                            'SslSupportMethod': 'sni-only',
                        },
                    },
                }));
                test.done();
            },
            'advanced usage'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                const certificate = new certificatemanager.Certificate(stack, 'cert', {
                    domainName: 'example.com',
                });
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromAcmCertificate(certificate, {
                        securityPolicy: lib_1.SecurityPolicyProtocol.SSL_V3,
                        sslMethod: lib_1.SSLMethod.VIP,
                        aliases: ['example.com', 'www.example.com'],
                    }),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    'DistributionConfig': {
                        'Aliases': ['example.com', 'www.example.com'],
                        'ViewerCertificate': {
                            'AcmCertificateArn': {
                                'Ref': 'cert56CA94EB',
                            },
                            'MinimumProtocolVersion': 'SSLv3',
                            'SslSupportMethod': 'vip',
                        },
                    },
                }));
                test.done();
            },
        },
        'iamCertificate': {
            'base usage'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromIamCertificate('test'),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    'DistributionConfig': {
                        'Aliases': [],
                        'ViewerCertificate': {
                            'IamCertificateId': 'test',
                            'SslSupportMethod': 'sni-only',
                        },
                    },
                }));
                test.done();
            },
            'advanced usage'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromIamCertificate('test', {
                        securityPolicy: lib_1.SecurityPolicyProtocol.TLS_V1,
                        sslMethod: lib_1.SSLMethod.VIP,
                        aliases: ['example.com'],
                    }),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    'DistributionConfig': {
                        'Aliases': ['example.com'],
                        'ViewerCertificate': {
                            'IamCertificateId': 'test',
                            'MinimumProtocolVersion': 'TLSv1',
                            'SslSupportMethod': 'vip',
                        },
                    },
                }));
                test.done();
            },
        },
        'cloudFrontDefaultCertificate': {
            'base usage'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromCloudFrontDefaultCertificate(),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    'DistributionConfig': {
                        'Aliases': [],
                        'ViewerCertificate': {
                            'CloudFrontDefaultCertificate': true,
                        },
                    },
                }));
                test.done();
            },
            'aliases are set'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromCloudFrontDefaultCertificate('example.com', 'www.example.com'),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    'DistributionConfig': {
                        'Aliases': ['example.com', 'www.example.com'],
                        'ViewerCertificate': {
                            'CloudFrontDefaultCertificate': true,
                        },
                    },
                }));
                test.done();
            },
        },
        'errors': {
            'throws if both deprecated aliasConfiguration and viewerCertificate'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                test.throws(() => {
                    new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                        originConfigs: [{
                                s3OriginSource: { s3BucketSource: sourceBucket },
                                behaviors: [{ isDefaultBehavior: true }],
                            }],
                        aliasConfiguration: { acmCertRef: 'test', names: ['ftp.example.com'] },
                        viewerCertificate: lib_1.ViewerCertificate.fromCloudFrontDefaultCertificate('example.com', 'www.example.com'),
                    });
                }, /You cannot set both aliasConfiguration and viewerCertificate properties/);
                test.done();
            },
            'throws if invalid security policy for SSL method'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                test.throws(() => {
                    new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                        originConfigs: [{
                                s3OriginSource: { s3BucketSource: sourceBucket },
                                behaviors: [{ isDefaultBehavior: true }],
                            }],
                        viewerCertificate: lib_1.ViewerCertificate.fromIamCertificate('test', {
                            securityPolicy: lib_1.SecurityPolicyProtocol.TLS_V1_1_2016,
                            sslMethod: lib_1.SSLMethod.VIP,
                        }),
                    });
                }, /TLSv1.1_2016 is not compabtible with sslMethod vip./);
                test.done();
            },
            // FIXME https://github.com/aws/aws-cdk/issues/4724
            'does not throw if acmCertificate explicitly not in us-east-1'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                const certificate = certificatemanager.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:eu-west-3:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    viewerCertificate: lib_1.ViewerCertificate.fromAcmCertificate(certificate),
                });
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
                    'DistributionConfig': {
                        'Aliases': [],
                        'ViewerCertificate': {
                            'AcmCertificateArn': 'arn:aws:acm:eu-west-3:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d',
                            'SslSupportMethod': 'sni-only',
                        },
                    },
                }));
                test.done();
            },
        },
    },
    'edgelambda.amazonaws.com is added to the trust policy of lambda'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        const fn = new lambda.Function(stack, 'Lambda', {
            code: lambda.Code.fromInline('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const lambdaVersion = new lambda.Version(stack, 'LambdaVersion', { lambda: fn });
        // WHEN
        new lib_1.CloudFrontWebDistribution(stack, 'MyDistribution', {
            originConfigs: [
                {
                    s3OriginSource: { s3BucketSource: sourceBucket },
                    behaviors: [
                        {
                            isDefaultBehavior: true, lambdaFunctionAssociations: [
                                {
                                    eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                                    lambdaFunction: lambdaVersion,
                                },
                            ],
                        },
                    ],
                },
            ],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                'Statement': [
                    {
                        'Action': 'sts:AssumeRole',
                        'Effect': 'Allow',
                        'Principal': {
                            'Service': 'lambda.amazonaws.com',
                        },
                    },
                    {
                        'Action': 'sts:AssumeRole',
                        'Effect': 'Allow',
                        'Principal': {
                            'Service': 'edgelambda.amazonaws.com',
                        },
                    },
                ],
                'Version': '2012-10-17',
            },
        }));
        test.done();
    },
    'edgelambda.amazonaws.com is not added to lambda role for imported functions'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        const lambdaVersion = lambda.Version.fromVersionArn(stack, 'Version', 'arn:my-version');
        // WHEN
        new lib_1.CloudFrontWebDistribution(stack, 'MyDistribution', {
            originConfigs: [
                {
                    s3OriginSource: { s3BucketSource: sourceBucket },
                    behaviors: [
                        {
                            isDefaultBehavior: true, lambdaFunctionAssociations: [
                                {
                                    eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                                    lambdaFunction: lambdaVersion,
                                },
                            ],
                        },
                    ],
                },
            ],
        });
        assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Role'));
        test.done();
    },
    'geo restriction': {
        'success': {
            'whitelist'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    geoRestriction: lib_1.GeoRestriction.whitelist('US', 'UK'),
                });
                assert_1.expect(stack).toMatch({
                    'Resources': {
                        'Bucket83908E77': {
                            'Type': 'AWS::S3::Bucket',
                            'DeletionPolicy': 'Retain',
                            'UpdateReplacePolicy': 'Retain',
                        },
                        'AnAmazingWebsiteProbablyCFDistribution47E3983B': {
                            'Type': 'AWS::CloudFront::Distribution',
                            'Properties': {
                                'DistributionConfig': {
                                    'DefaultRootObject': 'index.html',
                                    'Origins': [
                                        {
                                            'ConnectionAttempts': 3,
                                            'ConnectionTimeout': 10,
                                            'DomainName': {
                                                'Fn::GetAtt': [
                                                    'Bucket83908E77',
                                                    'RegionalDomainName',
                                                ],
                                            },
                                            'Id': 'origin1',
                                            'S3OriginConfig': {},
                                        },
                                    ],
                                    'ViewerCertificate': {
                                        'CloudFrontDefaultCertificate': true,
                                    },
                                    'PriceClass': 'PriceClass_100',
                                    'DefaultCacheBehavior': {
                                        'AllowedMethods': [
                                            'GET',
                                            'HEAD',
                                        ],
                                        'CachedMethods': [
                                            'GET',
                                            'HEAD',
                                        ],
                                        'TargetOriginId': 'origin1',
                                        'ViewerProtocolPolicy': 'redirect-to-https',
                                        'ForwardedValues': {
                                            'QueryString': false,
                                            'Cookies': { 'Forward': 'none' },
                                        },
                                        'Compress': true,
                                    },
                                    'Enabled': true,
                                    'IPV6Enabled': true,
                                    'HttpVersion': 'http2',
                                    'Restrictions': {
                                        'GeoRestriction': {
                                            'Locations': ['US', 'UK'],
                                            'RestrictionType': 'whitelist',
                                        },
                                    },
                                },
                            },
                        },
                    },
                });
                test.done();
            },
            'blacklist'(test) {
                const stack = new cdk.Stack();
                const sourceBucket = new s3.Bucket(stack, 'Bucket');
                new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
                    originConfigs: [{
                            s3OriginSource: { s3BucketSource: sourceBucket },
                            behaviors: [{ isDefaultBehavior: true }],
                        }],
                    geoRestriction: lib_1.GeoRestriction.blacklist('US'),
                });
                assert_1.expect(stack).toMatch({
                    'Resources': {
                        'Bucket83908E77': {
                            'Type': 'AWS::S3::Bucket',
                            'DeletionPolicy': 'Retain',
                            'UpdateReplacePolicy': 'Retain',
                        },
                        'AnAmazingWebsiteProbablyCFDistribution47E3983B': {
                            'Type': 'AWS::CloudFront::Distribution',
                            'Properties': {
                                'DistributionConfig': {
                                    'DefaultRootObject': 'index.html',
                                    'Origins': [
                                        {
                                            'ConnectionAttempts': 3,
                                            'ConnectionTimeout': 10,
                                            'DomainName': {
                                                'Fn::GetAtt': [
                                                    'Bucket83908E77',
                                                    'RegionalDomainName',
                                                ],
                                            },
                                            'Id': 'origin1',
                                            'S3OriginConfig': {},
                                        },
                                    ],
                                    'ViewerCertificate': {
                                        'CloudFrontDefaultCertificate': true,
                                    },
                                    'PriceClass': 'PriceClass_100',
                                    'DefaultCacheBehavior': {
                                        'AllowedMethods': [
                                            'GET',
                                            'HEAD',
                                        ],
                                        'CachedMethods': [
                                            'GET',
                                            'HEAD',
                                        ],
                                        'TargetOriginId': 'origin1',
                                        'ViewerProtocolPolicy': 'redirect-to-https',
                                        'ForwardedValues': {
                                            'QueryString': false,
                                            'Cookies': { 'Forward': 'none' },
                                        },
                                        'Compress': true,
                                    },
                                    'Enabled': true,
                                    'IPV6Enabled': true,
                                    'HttpVersion': 'http2',
                                    'Restrictions': {
                                        'GeoRestriction': {
                                            'Locations': ['US'],
                                            'RestrictionType': 'blacklist',
                                        },
                                    },
                                },
                            },
                        },
                    },
                });
                test.done();
            },
        },
        'error': {
            'throws if locations is empty array'(test) {
                test.throws(() => {
                    lib_1.GeoRestriction.whitelist();
                }, /Should provide at least 1 location/);
                test.throws(() => {
                    lib_1.GeoRestriction.blacklist();
                }, /Should provide at least 1 location/);
                test.done();
            },
            'throws if locations format is wrong'(test) {
                test.throws(() => {
                    lib_1.GeoRestriction.whitelist('us');
                }, /Invalid location format for location: us, location should be two-letter and uppercase country ISO 3166-1-alpha-2 code/);
                test.throws(() => {
                    lib_1.GeoRestriction.blacklist('us');
                }, /Invalid location format for location: us, location should be two-letter and uppercase country ISO 3166-1-alpha-2 code/);
                test.done();
            },
        },
    },
    'Connection behaviors between CloudFront and your origin': {
        'success': {
            'connectionAttempts = 1'(test) {
                const stack = new cdk.Stack();
                test.doesNotThrow(() => {
                    new lib_1.CloudFrontWebDistribution(stack, 'Distribution', {
                        originConfigs: [{
                                behaviors: [{ isDefaultBehavior: true }],
                                connectionAttempts: 1,
                                customOriginSource: { domainName: 'myorigin.com' },
                            }],
                    });
                }, /connectionAttempts: You can specify 1, 2, or 3 as the number of attempts./);
                test.done();
            },
            '3 = connectionAttempts'(test) {
                const stack = new cdk.Stack();
                test.doesNotThrow(() => {
                    new lib_1.CloudFrontWebDistribution(stack, 'Distribution', {
                        originConfigs: [{
                                behaviors: [{ isDefaultBehavior: true }],
                                connectionAttempts: 3,
                                customOriginSource: { domainName: 'myorigin.com' },
                            }],
                    });
                }, /connectionAttempts: You can specify 1, 2, or 3 as the number of attempts./);
                test.done();
            },
            'connectionTimeout = 1'(test) {
                const stack = new cdk.Stack();
                test.doesNotThrow(() => {
                    new lib_1.CloudFrontWebDistribution(stack, 'Distribution', {
                        originConfigs: [{
                                behaviors: [{ isDefaultBehavior: true }],
                                connectionTimeout: cdk.Duration.seconds(1),
                                customOriginSource: { domainName: 'myorigin.com' },
                            }],
                    });
                }, /connectionTimeout: You can specify a number of seconds between 1 and 10 (inclusive)./);
                test.done();
            },
            '10 = connectionTimeout'(test) {
                const stack = new cdk.Stack();
                test.doesNotThrow(() => {
                    new lib_1.CloudFrontWebDistribution(stack, 'Distribution', {
                        originConfigs: [{
                                behaviors: [{ isDefaultBehavior: true }],
                                connectionTimeout: cdk.Duration.seconds(10),
                                customOriginSource: { domainName: 'myorigin.com' },
                            }],
                    });
                }, /connectionTimeout: You can specify a number of seconds between 1 and 10 (inclusive)./);
                test.done();
            },
        },
        'errors': {
            'connectionAttempts = 1.1'(test) {
                const stack = new cdk.Stack();
                test.throws(() => {
                    new lib_1.CloudFrontWebDistribution(stack, 'Distribution', {
                        originConfigs: [{
                                behaviors: [{ isDefaultBehavior: true }],
                                connectionAttempts: 1.1,
                                customOriginSource: { domainName: 'myorigin.com' },
                            }],
                    });
                }, /connectionAttempts: You can specify 1, 2, or 3 as the number of attempts./);
                test.done();
            },
            'connectionAttempts = -1'(test) {
                const stack = new cdk.Stack();
                test.throws(() => {
                    new lib_1.CloudFrontWebDistribution(stack, 'Distribution', {
                        originConfigs: [{
                                behaviors: [{ isDefaultBehavior: true }],
                                connectionAttempts: -1,
                                customOriginSource: { domainName: 'myorigin.com' },
                            }],
                    });
                }, /connectionAttempts: You can specify 1, 2, or 3 as the number of attempts./);
                test.done();
            },
            'connectionAttempts < 1'(test) {
                const stack = new cdk.Stack();
                test.throws(() => {
                    new lib_1.CloudFrontWebDistribution(stack, 'Distribution', {
                        originConfigs: [{
                                behaviors: [{ isDefaultBehavior: true }],
                                connectionAttempts: 0,
                                customOriginSource: { domainName: 'myorigin.com' },
                            }],
                    });
                }, /connectionAttempts: You can specify 1, 2, or 3 as the number of attempts./);
                test.done();
            },
            '3 < connectionAttempts'(test) {
                const stack = new cdk.Stack();
                test.throws(() => {
                    new lib_1.CloudFrontWebDistribution(stack, 'Distribution', {
                        originConfigs: [{
                                behaviors: [{ isDefaultBehavior: true }],
                                connectionAttempts: 4,
                                customOriginSource: { domainName: 'myorigin.com' },
                            }],
                    });
                }, /connectionAttempts: You can specify 1, 2, or 3 as the number of attempts./);
                test.done();
            },
            'connectionTimeout = 1.1'(test) {
                const stack = new cdk.Stack();
                test.throws(() => {
                    new lib_1.CloudFrontWebDistribution(stack, 'Distribution', {
                        originConfigs: [{
                                behaviors: [{ isDefaultBehavior: true }],
                                connectionTimeout: cdk.Duration.seconds(1.1),
                                customOriginSource: { domainName: 'myorigin.com' },
                            }],
                    });
                }, /connectionTimeout: You can specify a number of seconds between 1 and 10 \(inclusive\)./);
                test.done();
            },
            'connectionTimeout < 1'(test) {
                const stack = new cdk.Stack();
                test.throws(() => {
                    new lib_1.CloudFrontWebDistribution(stack, 'Distribution', {
                        originConfigs: [{
                                behaviors: [{ isDefaultBehavior: true }],
                                connectionTimeout: cdk.Duration.seconds(0),
                                customOriginSource: { domainName: 'myorigin.com' },
                            }],
                    });
                }, /connectionTimeout: You can specify a number of seconds between 1 and 10 \(inclusive\)./);
                test.done();
            },
            '10 < connectionTimeout'(test) {
                const stack = new cdk.Stack();
                test.throws(() => {
                    new lib_1.CloudFrontWebDistribution(stack, 'Distribution', {
                        originConfigs: [{
                                behaviors: [{ isDefaultBehavior: true }],
                                connectionTimeout: cdk.Duration.seconds(11),
                                customOriginSource: { domainName: 'myorigin.com' },
                            }],
                    });
                }, /connectionTimeout: You can specify a number of seconds between 1 and 10 \(inclusive\)./);
                test.done();
            },
        },
    },
});
//# sourceMappingURL=data:application/json;base64,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