__version__ = '0.2'

import warnings

import gmxapi as gmx
import mdtraj
import os
from gromorg.swisparam import SwissParams
from gromorg.utils import extract_energy, extract_forces
import numpy as np
from gromorg.capture import captured_stdout
import shutil
from gromorg.data_structure import DataStructure


class GromOrg:
    def __init__(self, structure,
                 params,
                 box=(10, 10, 10),
                 angles=(90, 90, 90),
                 supercell=(1, 1, 1),
                 solvent=None,
                 solvent_scale=0.57,
                 maxwarn=0,
                 omp_num_threads=1,
                 silent=False,
                 delete_scratch=True):

        self._structure = structure
        self._filename = 'test'
        self._box = np.array(box)/10.0  # from Angs to nm
        self._supercell = supercell
        self._angles = angles
        self._silent = silent
        self._delete_scratch = delete_scratch
        self._solvent = solvent
        self._solvent_scale = solvent_scale
        self._maxwarn = maxwarn

        os.putenv('GMX_MAXBACKUP', '-1')
        os.putenv('OMP_NUM_THREADS', '{}'.format(omp_num_threads))

        self._work_dir = 'gromorg_{}/'.format(os.getpid())
        # os.mkdir(self._work_dir)
        try:
            os.mkdir(self._work_dir)
        except FileExistsError:
            pass

        self._filename_dir = self._work_dir + self._filename

        # Default parameters
        self._params = {# Run paramters
                        'integrator': 'md-vv',     # Verlet integrator
                        'nsteps': 5000,            # 0.001 * 5000 = 50 ps
                        'dt': 0.001,               # ps
                        # Output control
                        'nstxout': 1,              # save coordinates every 0.001 ps
                        'nstvout': 1,              # save velocities every 0.001 ps
                        'nstenergy': 1,            # save energies every 0.001 ps
                        'nstlog': 100,             # update log file every 0.1 ps
                        # Bond parameters
                        'continuation': 'no',       # first dynamics run
                        'cutoff-scheme': 'Verlet',  # Buffered neighbor searching
                        'verlet-buffer-tolerance': 3.3e-03,
                        # 'ns_type': 'grid',          # search neighboring grid cells
                        'nstlist': 10,              # 20 fs, largely irrelevant with Verlet
                        'rcoulomb': 1.0,            # short-range electrostatic cutoff (in nm)
                        'rvdw': 1.0,                # short-range van der Waals cutoff (in nm)
                        'DispCorr': 'EnerPres',     # account for cut-off vdW scheme
                        # Electrostatics
                        'coulombtype': 'PME',       # Particle Mesh Ewald for long-range electrostatics
                        'pme_order': 4,             # cubic interpolation
                        'fourierspacing': 0.16,     # grid spacing for FFT
                        # Temperature coupling is on
                        'tcoupl': 'nose-hoover',    # Nose-Hoover thermostat
                        'tc-grps': 'system',        # one coupling group
                        'tau_t': 0.3,               # time constant, in ps
                        'ref_t': 100,               # reference temperature, one for each group, in K
                        # Pressure coupling is off
                        'pcoupl': 'no',             # no pressure coupling in NVT
                        # Periodic boundary conditions
                        'pbc': 'xyz',               # 3-D PBC
                        # Velocity generation
                        'gen_vel': 'yes',           # assign velocities from Maxwell distributio
                        'gen_temp': 10,             # temperature for Maxwell distribution
                        'gen_seed': -1,             # generate a random seed
                        }

        if params is not None:
            self._params.update(params)

    def get_mdp(self):
        file = ';Autogenerated MDP\n'
        for keys, values in self._params.items():
            file += '{:30} = {}\n'.format(keys, values)

        return file

    def get_topology(self):

        num_mol = np.prod(self._supercell)

        topology_data = {'':['; Autogenerated Topology',
                             '#include "charmm27.ff/forcefield.itp"',
                             '#include "{}.itp"'.format(self._filename)],
                         'system': ['molecular system name'],
                         'molecules': ['{} {}\n'.format(self._filename, num_mol)]}

        return DataStructure(topology_data)

    def get_tpr(self):

        # write mdp file on disk
        with open('{}.mdp'.format(self._filename_dir), 'w') as f:
            f.write(self.get_mdp())

        # get parameters for main system
        sw = SwissParams(self._structure, silent=self._silent)

        with open('{}.pdb'.format(self._filename_dir), 'w') as f:
            f.write(sw.get_pdb_data())

        # define unit cell and create gro file
        if self._angles is None:
            grompp = gmx.commandline_operation('gmx', 'editconf',
                                               input_files={'-f': self._filename_dir + '.pdb',
                                                            '-box': ['{}'.format(self._box[0]),
                                                                     '{}'.format(self._box[1]),
                                                                     '{}'.format(self._box[2])]},

                                               output_files={'-o': self._filename_dir + '.gro'})
        else:
            grompp = gmx.commandline_operation('gmx',
                                               arguments=['editconf',
                                                          #'-noc'
                                                          ],
                                               input_files={'-f': self._filename_dir + '.pdb',
                                                            '-box': ['{}'.format(self._box[0]),
                                                                     '{}'.format(self._box[1]),
                                                                     '{}'.format(self._box[2])],
                                                            '-bt': 'triclinic',
                                                            '-angles': ['{}'.format(self._angles[0]),
                                                                        '{}'.format(self._angles[1]),
                                                                        '{}'.format(self._angles[2])]
                                                            },
                                               output_files={'-o': self._filename_dir + '.gro'})

        grompp.run()

        if grompp.output.returncode.result() != 0:
            print(grompp.output.erroroutput.result())

        # create supercell from unitcell
        grompp = gmx.commandline_operation('gmx', 'genconf',
                                           input_files={'-f': self._filename_dir + '.gro',
                                                        '-nbox': ['{}'.format(self._supercell[0]),
                                                                  '{}'.format(self._supercell[1]),
                                                                  '{}'.format(self._supercell[2])],
                                                        },
                                           output_files={'-o': self._filename_dir + '.gro'})
        grompp.run()

        if grompp.output.returncode.result() != 0:
            print(grompp.output.erroroutput.result())

        # get itp and topology data
        itp = DataStructure(sw.get_itp_data())
        top = self.get_topology()

        # Add solvent to itp and top files
        if self._solvent is not None:
            top, itp = self._add_solvent(top, itp)

        # write topology and itp files on disk
        with open('{}.top'.format(self._filename_dir), 'w') as f:
            f.write(top.get_txt())

        with open('{}.itp'.format(self._filename_dir), 'w') as f:
            f.write(itp.get_txt())

        grompp = gmx.commandline_operation('gmx', 'grompp',
                                           input_files={'-f': self._filename_dir + '.mdp',
                                                        '-c': self._filename_dir + '.gro',
                                                        '-p': self._filename_dir + '.top',
                                                        '-po': self._filename_dir + '_log.mdp',
                                                        '-maxwarn': '{}'.format(self._maxwarn)},
                                           output_files={'-o': self._filename_dir + '.tpr'})
        grompp.run()

        if grompp.output.returncode.result() != 0:
            print(grompp.output.erroroutput.result())

        tpr_data = gmx.read_tpr(self._filename_dir + '.tpr')

        return tpr_data

    def _add_solvent(self, top, itp):

        sw_sol = SwissParams(self._solvent, silent=self._silent)

        # solvent pdb
        pdb_solvent = sw_sol.get_pdb_data().replace('LIG', 'SOL')

        # get box that fits solvent
        coords = []
        for line in pdb_solvent.split('\n'):
            if line.strip().startswith('ATOM'):
                coords.append(line.split()[5:8])

        s_box = np.max(np.array(coords, dtype=float), axis=0) - np.min(np.array(coords, dtype=float), axis=0)
        s_box = np.maximum([1.0, 1.0, 1.0], s_box)

        with open('{}_sol.pdb'.format(self._filename_dir), 'w') as f:
            f.write(pdb_solvent)

        # pdb to gro + box
        grompp = gmx.commandline_operation('gmx', 'editconf',
                                           input_files={'-f': self._filename_dir + '_sol.pdb',
                                                        '-box': ['{}'.format(s_box[0]),
                                                                 '{}'.format(s_box[1]),
                                                                 '{}'.format(s_box[2])],
                                                        },

                                           output_files={'-o': self._filename_dir + '_sol.gro'})

        grompp.run()

        if grompp.output.returncode.result() != 0:
            print(grompp.output.erroroutput.result())

        # get solvent itp
        itp_solvent = DataStructure(sw_sol.get_itp_data().replace('LIG', 'SOL').replace('test', 'test_sol'))
        itp.append_data('atomtypes', itp_solvent.get_data('atomtypes'))
        itp_solvent.remove_data('')
        itp_solvent.remove_data('atomtypes')
        itp_solvent.remove_data('pairtypes')

        # store solvent itp file
        with open('{}_sol.itp'.format(self._filename_dir), 'w') as f:
            f.write(itp_solvent.get_txt())

        # append itp file reference to topology
        top.append_data('', ['#include "{}_sol.itp"'.format(self._filename)])

        open(self._filename_dir + '_sol.top', 'w').close()  # temp file

        # solvate system with solvent
        grompp = gmx.commandline_operation('gmx', 'solvate',
                                           input_files={'-cp': self._filename_dir + '.gro',
                                                        '-cs': self._filename_dir + '_sol.gro',
                                                        '-scale': '{}'.format(self._solvent_scale),
                                                        '-p': self._filename_dir + '_sol.top'},
                                           output_files={'-o': self._filename_dir + '.gro'})
        grompp.run()

        if grompp.output.returncode.result() != 0:
            print(grompp.output.erroroutput.result())

        with open(self._filename_dir + '_sol.top', 'rb') as f:
            line_sol = f.read().decode('utf-8')

        os.remove(self._filename_dir + '_sol.top') # delete temp file

        # append solvent molecule info to topology
        try:
            n_solvent_mol = int(line_sol.split()[1])
            top.append_data('molecules', ['test_sol {}'.format(n_solvent_mol)])

        except IndexError:
            warnings.warn('Solvent molecules do not fit in the supercell. '
                          'Decrease the solvent size or increase solvent density using solvent_scale')

        return top, itp

    def run_md(self, whole=True):

        md = gmx.mdrun(input=self.get_tpr())

        if self._silent:
            with captured_stdout(self._filename_dir + '.log'):
                md.run()
        else:
            md.run()

        trajectory_file = md.output.trajectory.result()
        md_data_dir = md.output._work_dir.result()

        if whole:
            grompp = gmx.commandline_operation('gmx', 'trjconv',
                                               stdin='0',
                                               input_files={'-f': trajectory_file,
                                                            '-s': self._filename_dir + '.tpr',
                                                            '-pbc': 'whole'},
                                               output_files={'-o': md_data_dir + '/{}.trr'.format(self._filename)})

            if grompp.output.returncode.result() != 0:
                print(grompp.output.erroroutput.result())

            trajectory_file = md_data_dir + '/{}.trr'.format(self._filename)

        trajectory = mdtraj.load_trr(trajectory_file, top=md_data_dir + '/confout.gro')
        energy = extract_energy(md_data_dir + '/ener.edr', initial=0)

        if self._delete_scratch:
            import shutil
            shutil.rmtree(md.output._work_dir.result())
            # shutil.rmtree(self._work_dir)

        return trajectory, energy

    def get_forces(self):

        mod_input = gmx.modify_input(input=self.get_tpr(), parameters={'nsteps': 1, 'nstfout': 1})
        md = gmx.mdrun(input=mod_input)

        if self._silent:
            with captured_stdout(self._filename_dir + '.log'):
                md.run()
        else:
            md.run()

        trajectory_file = md.output.trajectory.result()
        md_data_dir = md.output._work_dir.result()

        self._trajectory = mdtraj.load_trr(trajectory_file, top=md_data_dir + '/confout.gro')
        forces = extract_forces(trajectory_file, self._filename_dir + '.tpr', step=0)

        if self._delete_scratch:
            import shutil
            shutil.rmtree(md.output._work_dir.result())
            # shutil.rmtree(self._work_dir)

        return forces

    def __del__(self):
        if os.path.isdir(self._work_dir) and self._delete_scratch:
            shutil.rmtree(self._work_dir)

if __name__ == '__main__':
    pass
