# AUTOGENERATED! DO NOT EDIT! File to edit: ../notebooks/retraitement_erfs-fpr/modules/ToolBase.ipynb.

# %% auto 0
__all__ = [
    "tc",
    "config",
    "convert_erfr_to_openfisca",
    "create_simulation",
    "individus_to_foyers_fiscaux",
    "prepare_ff_to_concat",
    "foyers_fiscaux_to_individus",
    "compute_var_in_ff",
]

# %% ../notebooks/retraitement_erfs-fpr/modules/ToolBase.ipynb 3
import unittest

import pandas as pd

# %% ../notebooks/retraitement_erfs-fpr/modules/ToolBase.ipynb 4
from leximpact_socio_fisca_simu_etat.config import Configuration
from openfisca_core.simulation_builder import SimulationBuilder

# import numpy as np


tc = unittest.TestCase()
config = Configuration(project_folder="leximpact-prepare-data")

# %% ../notebooks/retraitement_erfs-fpr/modules/ToolBase.ipynb 6
def convert_erfr_to_openfisca(data):
    """
    # Conversion des noms ERFS en noms  OpenFisca
    Traduction des roles attribués au format openfisca
    ::data:: Le dataset à convertir
    """

    # Traduction des roles attribués au format openfisca
    data["quimenof"] = "enfant"
    data.loc[data["quifoy"] == 1, "quimenof"] = "conjoint"
    data.loc[data["quifoy"] == 0, "quimenof"] = "personne_de_reference"

    data["quifoyof"] = "personne_a_charge"
    data.loc[data["quifoy"] == 1, "quifoyof"] = "conjoint"
    data.loc[data["quifoy"] == 0, "quifoyof"] = "declarant_principal"

    data["quifamof"] = "enfant"
    data.loc[data["quifam"] == 1, "quifamof"] = "conjoint"
    data.loc[data["quifam"] == 0, "quifamof"] = "demandeur"
    return data


# %% ../notebooks/retraitement_erfs-fpr/modules/ToolBase.ipynb 8
def create_simulation(period, data, tbs):
    sb = SimulationBuilder()
    sb.create_entities(tbs)
    sb.declare_person_entity("individu", data.index)

    if "quifoyof" not in data.columns or "quifamof" not in data.columns:
        data = convert_erfr_to_openfisca(data)

    # Creates openfisca entities and generates grouped
    listentities = {"foy": "foyer_fiscal", "men": "menage", "fam": "famille"}

    instances = {}
    dictionnaire_datagrouped = {"individu": data}

    for ent, ofent in listentities.items():
        persons_ent = data["id" + ent].values
        persons_ent_roles = data["qui" + ent + "of"].values
        ent_ids = data["id" + ent].unique()
        instances[ofent] = sb.declare_entity(ofent, ent_ids)
        sb.join_with_persons(instances[ofent], persons_ent, roles=persons_ent_roles)

        # print('instances',instances[ofent])
        # The following assumes data defined for an entity are the same for all rows in
        # the same entity. Or at least that the first non null value found for an
        # entity will always be the total value for an entity (which is the case for
        # f4ba). These checks are performed in the checkdata function defined below.

        # TO DO: Trouver cette checkdata function et l'utiliser ici ! Maybe compare_input_data or test_h5_input?
        dictionnaire_datagrouped[ofent] = (
            data.groupby("id" + ent, as_index=False).first().sort_values(by="id" + ent)
        )
        # print("dict", dictionnaire_datagrouped)

    # These variables should not be attributed to any OpenFisca Entity
    columns_not_OF_variables = {
        "idmen",
        "idfoy",
        "idfam",
        "noindiv",
        "level_0",
        "quifam",
        "quifoy",
        "quimen",
        "idmen_x",
        "idmen_y",
        "wprm",
        "index",
        "idmen_original",
        "idfoy_original",
        "idfam_original",
        "quifamof",
        "quifoyof",
        "quimenof",
        "fake_id",
        "assiette_csg_plus_values",
        # "csg_deductible_chomage",
        # "csg_imposable_chomage",
        # "csg_chomage",
        # "csg_deductible_retraite",
        # "csg_imposable_retraite",
        # "csg_retraite",
        "pote_rev_capital",
        "pote_rente_viagere",
        "pote_rev_categ_foncier",
        "pote_plus_values",
        "pote_chomage",
        "pote_retraite",
        "pote_pre-retraite",
    }

    simulation = sb.build(tbs)

    # Attribution des variables à la bonne entité OpenFisca
    for colonne in data.columns:
        # if colonne=='rfr':
        #    print("data", data['rfr'])
        # On entre les valeurs sur les 3 dernieres annees (par exemple pour calculer la retraite)
        for PERIOD in range(int(period) - 9, int(period) + 1):  # test sur 10 ans
            if colonne not in columns_not_OF_variables:
                try:
                    # if colonne == "rfr":
                    # print("colonne", colonne, PERIOD)
                    # print("dict ", dictionnaire_datagrouped)
                    # print("value ", dictionnaire_datagrouped[tbs.get_variable(colonne).entity.key][colonne],)
                    simulation.set_input(
                        colonne,
                        str(PERIOD),
                        dictionnaire_datagrouped[tbs.get_variable(colonne).entity.key][
                            colonne
                        ],
                    )
                except AttributeError as e:
                    print(
                        f"AttributeError, are you sure {colonne} is known by your OpenFisca TaxBenefitSystem ? {e}"
                    )
                    raise e

    return simulation, dictionnaire_datagrouped


# %% ../notebooks/retraitement_erfs-fpr/modules/ToolBase.ipynb 10
def individus_to_foyers_fiscaux(sample_individus):
    """
    Regroupe un échantillon par individus en échantillon par foyers fiscaux
    """
    print(
        "On a une base initiale de ",
        len(sample_individus),
        " individus pour ",
        sample_individus["idfoy"].nunique(),
        " foyers fiscaux",
    )

    # On regroupe les individus de sample_pop_individus en foyers fiscaux par leur idfoy (et leurs 'wprm' pour ne pas sommer les poids)
    if "fake_id" in sample_individus.columns:
        sample_foyers_fiscaux = sample_individus.groupby(
            ["idfoy", "fake_id", "wprm"], as_index=False
        ).sum()
    else:
        sample_foyers_fiscaux = sample_individus.groupby(
            ["idfoy", "wprm"], as_index=False
        ).sum()

    # print("nb de foyers apres groupby", sample_foyers_fiscaux["idfoy"].nunique())

    # On supprime les colonnes qui n'ont plus de sens au niveau foyer
    for col in [
        "quifoy",
        "quifam",
        "age",
        "categorie_salarie",
        "contrat_de_travail",
        "statut_marital",
    ]:
        if col in sample_foyers_fiscaux.columns:
            sample_foyers_fiscaux.drop(col, axis=1, inplace=True)

    # On vérifie qu'on a bien le même nombre de foyers fiscaux distincts avant et après
    tc.assertEqual(
        sample_individus["idfoy"].nunique(), sample_foyers_fiscaux["idfoy"].nunique()
    )
    return sample_foyers_fiscaux


# %% ../notebooks/retraitement_erfs-fpr/modules/ToolBase.ipynb 12
def prepare_ff_to_concat(sample_ind_origin, sample_ff_to_merge, to_update):

    # On sélectionne les foyers qui ont été ajoutés (notamment dans la calibration)
    list_avant = list(sample_ind_origin["idfoy"].unique())
    list_ind_to_merge = list(sample_ff_to_merge["idfoy"].unique())
    list_ff_added = list(set(list_ind_to_merge) - set(list_avant))

    sample_to_concat = sample_ff_to_merge[
        sample_ff_to_merge["idfoy"].isin(list_ff_added)
    ].copy(deep=True)

    tc.assertAlmostEqual(len(list_ind_to_merge), len(list_ff_added) + len(list_avant))

    # Ils sont tous déclarants principaux
    sample_to_concat.loc[:, "quifoy"] = 0
    sample_to_concat.loc[:, "quifam"] = 0
    sample_to_concat.loc[:, "quimen"] = 0
    sample_to_concat.loc[:, "quifoyof"] = "declarant_principal"
    sample_to_concat.loc[:, "quimenof"] = "personne_de_reference"
    sample_to_concat.loc[:, "quifamof"] = "demandeur"

    # Pour les autres colonnes, on attribue la valeur du déclarant principal du foyer
    to_keep = to_update + [
        "fake_id",
        "wprm",
        "idfoy",
        "idfam",
        "idmen",
        "quifoy",
        "quifam",
        "quimen",
        "quifoyof",
        "quimenof",
        "quifamof",
    ]
    for col in sample_ind_origin.columns:
        # On ne touche pas aux colonnes d'intéret
        if col not in to_keep:

            # sample_to_concat.loc[:, col] = np.nan
            # La valeur la plus fréquente:
            sample_to_concat.loc[:, col] = (
                sample_ind_origin[col].value_counts().idxmax()
            )
            # Mais cela ne serait pas représentatif d'une situation réelle, donc on évite

    return sample_to_concat


# %% ../notebooks/retraitement_erfs-fpr/modules/ToolBase.ipynb 13
# On modifie la fonction foyer_fiscaux_to_individus pour prendre en compte ces gens 'ajoutés'


def foyers_fiscaux_to_individus(
    sample_ind_origin, sample_ff_to_merge, to_update, cols_declarant_principal
):
    """
    Ajoute les données d'un échantillon de foyers fiscaux dans un échantillon par individus
    sample_ind_origin: l' échantillon par individus originel
    sample_ff_to_merge : l'échantillon de foyers fiscaux
    to_update : les colonnes de sample_ff_to_merge qui ont bougé et qu'on veut merger
    cols_declarant_principal : les colonnes que l'on ne donne qu'au déclarant principal de chaque foyer
    new_ppl=True : on peut le mettre à False si est sûrs qu'on n'a ajouté personne entre-temps (par exemple en calib)
    et qu'on veut gagner du temps de calcul
    """

    sample_individus_new = sample_ind_origin.copy(deep=True)

    print(
        "Il s'agit ici de fusionner un échantillon de ",
        sample_ff_to_merge["idfoy"].nunique(),
        " foyers fiscaux, dans notre base qui en contient déjà ",
        sample_ind_origin["idfoy"].nunique(),
    )

    # On ne garde que ce qui sera updaté
    sample_ff_to_merge = sample_ff_to_merge[["idfoy"] + to_update]

    #  On enlève de l'échantillon individus les colonnes que l'on va ajouter
    for col in to_update:
        if col in sample_individus_new:
            sample_individus_new = sample_individus_new.drop(col, axis=1)

    # On update les valeurs dans l'échantillon par individus
    sample_individus_new = pd.merge(
        sample_individus_new,
        sample_ff_to_merge,
        left_on="idfoy",
        right_on="idfoy",
        indicator=False,
    )

    # Contrôles
    tc.assertEqual(
        sample_individus_new["idfoy"].nunique(),
        sample_ind_origin["idfoy"].nunique(),
    )

    # On supprime les valeurs qui ne s'appliquent qu'au déclarant principal pour les autres déclarants
    for col in cols_declarant_principal:
        sample_individus_new.loc[sample_individus_new["quifoy"] != 0, col] = 0

    return sample_individus_new


# %% ../notebooks/retraitement_erfs-fpr/modules/ToolBase.ipynb 16
def compute_var_in_ff(
    my_simu,
    annee_de_calcul,
    sample_individus,
    var_list,
    cols_declarant_principal,
):
    """
    Calcule une variable qui ne se calcule que par foyers fiscaux,
    en partant et en revenant sur une base de données en individus
    """
    # Si on n'a pas enore enlarged
    if "fake_id" not in sample_individus.columns:
        sample_individus["fake_id"] = 1

    # On regroupe les individus de sample_individus en foyers fiscaux par leur idfoy
    sample = sample_individus[["idfoy", "fake_id"]]
    sample_foyers_fiscaux = sample.groupby(["idfoy", "fake_id"], as_index=False).sum()

    # On calcule nos variables d'intérêt
    for var in var_list:
        to_calc = my_simu.calculate(var, annee_de_calcul)
        sample_foyers_fiscaux[var] = to_calc

    sample_individus_new = foyers_fiscaux_to_individus(
        sample_individus,
        sample_foyers_fiscaux,
        var_list,
        cols_declarant_principal,
    )

    return sample_individus_new
