"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = exports.TLSSecurityPolicy = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const url_1 = require("url");
const acm = require("@aws-cdk/aws-certificatemanager");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const route53 = require("@aws-cdk/aws-route53");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const log_group_resource_policy_1 = require("./log-group-resource-policy");
const opensearch_access_policy_1 = require("./opensearch-access-policy");
const opensearchservice_generated_1 = require("./opensearchservice.generated");
const perms = require("./perms");
/**
 * The minimum TLS version required for traffic to the domain.
 */
var TLSSecurityPolicy;
(function (TLSSecurityPolicy) {
    /** Cipher suite TLS 1.0 */
    TLSSecurityPolicy["TLS_1_0"] = "Policy-Min-TLS-1-0-2019-07";
    /** Cipher suite TLS 1.2 */
    TLSSecurityPolicy["TLS_1_2"] = "Policy-Min-TLS-1-2-2019-07";
})(TLSSecurityPolicy = exports.TLSSecurityPolicy || (exports.TLSSecurityPolicy = {}));
/**
 * A new or imported domain.
 */
class DomainBase extends cdk.Resource {
    /**
     * Grant read permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantRead(identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant write permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantWrite(identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant read/write permissions for this domain and its contents to an IAM
     * principal (Role/Group/User).
     *
     * @param identity The principal
     */
    grantReadWrite(identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * Grant read permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexRead(index, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant write permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexWrite(index, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant read/write permissions for an index in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param index The index to grant permissions for
     * @param identity The principal
     */
    grantIndexReadWrite(index, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * Grant read permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathRead(path, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Grant write permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathWrite(path, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Grant read/write permissions for a specific path in this domain to an IAM
     * principal (Role/Group/User).
     *
     * @param path The path to grant permissions for
     * @param identity The principal
     */
    grantPathReadWrite(path, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * Return the given named metric for this domain.
     */
    metric(metricName, props) {
        return new aws_cloudwatch_1.Metric({
            namespace: 'AWS/ES',
            metricName,
            dimensionsMap: {
                DomainName: this.domainName,
                ClientId: this.env.account,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the time the cluster status is red.
     *
     * @default maximum over 5 minutes
     */
    metricClusterStatusRed(props) {
        return this.metric('ClusterStatus.red', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for the time the cluster status is yellow.
     *
     * @default maximum over 5 minutes
     */
    metricClusterStatusYellow(props) {
        return this.metric('ClusterStatus.yellow', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for the storage space of nodes in the cluster.
     *
     * @default minimum over 5 minutes
     */
    metricFreeStorageSpace(props) {
        return this.metric('FreeStorageSpace', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...props,
        });
    }
    /**
     * Metric for the cluster blocking index writes.
     *
     * @default maximum over 1 minute
     */
    metricClusterIndexWritesBlocked(props) {
        return this.metric('ClusterIndexWritesBlocked', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            period: cdk.Duration.minutes(1),
            ...props,
        });
    }
    /**
     * Metric for the number of nodes.
     *
     * @default minimum over 1 hour
     */
    metricNodes(props) {
        return this.metric('Nodes', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            period: cdk.Duration.hours(1),
            ...props,
        });
    }
    /**
     * Metric for automated snapshot failures.
     *
     * @default maximum over 5 minutes
     */
    metricAutomatedSnapshotFailure(props) {
        return this.metric('AutomatedSnapshotFailure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for CPU utilization.
     *
     * @default maximum over 5 minutes
     */
    metricCPUUtilization(props) {
        return this.metric('CPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for JVM memory pressure.
     *
     * @default maximum over 5 minutes
     */
    metricJVMMemoryPressure(props) {
        return this.metric('JVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for master CPU utilization.
     *
     * @default maximum over 5 minutes
     */
    metricMasterCPUUtilization(props) {
        return this.metric('MasterCPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for master JVM memory pressure.
     *
     * @default maximum over 5 minutes
     */
    metricMasterJVMMemoryPressure(props) {
        return this.metric('MasterJVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for KMS key errors.
     *
     * @default maximum over 5 minutes
     */
    metricKMSKeyError(props) {
        return this.metric('KMSKeyError', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for KMS key being inaccessible.
     *
     * @default maximum over 5 minutes
     */
    metricKMSKeyInaccessible(props) {
        return this.metric('KMSKeyInaccessible', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for number of searchable documents.
     *
     * @default maximum over 5 minutes
     */
    metricSearchableDocuments(props) {
        return this.metric('SearchableDocuments', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * Metric for search latency.
     *
     * @default p99 over 5 minutes
     */
    metricSearchLatency(props) {
        return this.metric('SearchLatency', { statistic: 'p99', ...props });
    }
    /**
     * Metric for indexing latency.
     *
     * @default p99 over 5 minutes
     */
    metricIndexingLatency(props) {
        return this.metric('IndexingLatency', { statistic: 'p99', ...props });
    }
    grant(grantee, domainActions, resourceArn, ...otherResourceArns) {
        const resourceArns = [resourceArn, ...otherResourceArns];
        const grant = iam.Grant.addToPrincipal({
            grantee,
            actions: domainActions,
            resourceArns,
            scope: this,
        });
        return grant;
    }
}
/**
 * Provides an Amazon OpenSearch Service domain.
 */
class Domain extends DomainBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.domainName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_opensearchservice_DomainProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Domain);
            }
            throw error;
        }
        const defaultInstanceType = 'r5.large.search';
        const warmDefaultInstanceType = 'ultrawarm1.medium.search';
        const dedicatedMasterType = initializeInstanceType(defaultInstanceType, props.capacity?.masterNodeInstanceType);
        const dedicatedMasterCount = props.capacity?.masterNodes ?? 0;
        const dedicatedMasterEnabled = cdk.Token.isUnresolved(dedicatedMasterCount) ? true : dedicatedMasterCount > 0;
        const instanceType = initializeInstanceType(defaultInstanceType, props.capacity?.dataNodeInstanceType);
        const instanceCount = props.capacity?.dataNodes ?? 1;
        const warmType = initializeInstanceType(warmDefaultInstanceType, props.capacity?.warmInstanceType);
        const warmCount = props.capacity?.warmNodes ?? 0;
        const warmEnabled = cdk.Token.isUnresolved(warmCount) ? true : warmCount > 0;
        const availabilityZoneCount = props.zoneAwareness?.availabilityZoneCount ?? 2;
        if (![2, 3].includes(availabilityZoneCount)) {
            throw new Error('Invalid zone awareness configuration; availabilityZoneCount must be 2 or 3');
        }
        const zoneAwarenessEnabled = props.zoneAwareness?.enabled ??
            props.zoneAwareness?.availabilityZoneCount != null;
        let securityGroups;
        let subnets;
        if (props.vpc) {
            subnets = selectSubnets(props.vpc, props.vpcSubnets ?? [{ subnetType: ec2.SubnetType.PRIVATE_WITH_NAT }]);
            securityGroups = props.securityGroups ?? [new ec2.SecurityGroup(this, 'SecurityGroup', {
                    vpc: props.vpc,
                    description: `Security group for domain ${this.node.id}`,
                })];
            this._connections = new ec2.Connections({ securityGroups });
        }
        // If VPC options are supplied ensure that the number of subnets matches the number AZ
        if (subnets && zoneAwarenessEnabled && new Set(subnets.map((subnet) => subnet.availabilityZone)).size < availabilityZoneCount) {
            throw new Error('When providing vpc options you need to provide a subnet for each AZ you are using');
        }
        if ([dedicatedMasterType, instanceType, warmType].some(t => (!cdk.Token.isUnresolved(t) && !t.endsWith('.search')))) {
            throw new Error('Master, data and UltraWarm node instance types must end with ".search".');
        }
        if (!cdk.Token.isUnresolved(warmType) && !warmType.startsWith('ultrawarm')) {
            throw new Error('UltraWarm node instance type must start with "ultrawarm".');
        }
        const unsignedBasicAuthEnabled = props.useUnsignedBasicAuth ?? false;
        if (unsignedBasicAuthEnabled) {
            if (props.enforceHttps == false) {
                throw new Error('You cannot disable HTTPS and use unsigned basic auth');
            }
            if (props.nodeToNodeEncryption == false) {
                throw new Error('You cannot disable node to node encryption and use unsigned basic auth');
            }
            if (props.encryptionAtRest?.enabled == false) {
                throw new Error('You cannot disable encryption at rest and use unsigned basic auth');
            }
        }
        const unsignedAccessPolicy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['es:ESHttp*'],
            principals: [new iam.AnyPrincipal()],
            resources: [cdk.Lazy.string({ produce: () => `${this.domainArn}/*` })],
        });
        const masterUserArn = props.fineGrainedAccessControl?.masterUserArn;
        const masterUserNameProps = props.fineGrainedAccessControl?.masterUserName;
        // If basic auth is enabled set the user name to admin if no other user info is supplied.
        const masterUserName = unsignedBasicAuthEnabled
            ? (masterUserArn == null ? (masterUserNameProps ?? 'admin') : undefined)
            : masterUserNameProps;
        if (masterUserArn != null && masterUserName != null) {
            throw new Error('Invalid fine grained access control settings. Only provide one of master user ARN or master user name. Not both.');
        }
        const advancedSecurityEnabled = (masterUserArn ?? masterUserName) != null;
        const internalUserDatabaseEnabled = masterUserName != null;
        const masterUserPasswordProp = props.fineGrainedAccessControl?.masterUserPassword;
        const createMasterUserPassword = () => {
            return new secretsmanager.Secret(this, 'MasterUser', {
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({
                        username: masterUserName,
                    }),
                    generateStringKey: 'password',
                    excludeCharacters: "{}'\\*[]()`",
                },
            })
                .secretValueFromJson('password');
        };
        this.masterUserPassword = internalUserDatabaseEnabled ?
            (masterUserPasswordProp ?? createMasterUserPassword())
            : undefined;
        const encryptionAtRestEnabled = props.encryptionAtRest?.enabled ?? (props.encryptionAtRest?.kmsKey != null || unsignedBasicAuthEnabled);
        const nodeToNodeEncryptionEnabled = props.nodeToNodeEncryption ?? unsignedBasicAuthEnabled;
        const volumeSize = props.ebs?.volumeSize ?? 10;
        const volumeType = props.ebs?.volumeType ?? ec2.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
        const ebsEnabled = props.ebs?.enabled ?? true;
        const enforceHttps = props.enforceHttps ?? unsignedBasicAuthEnabled;
        function isInstanceType(t) {
            return dedicatedMasterType.startsWith(t) || instanceType.startsWith(t);
        }
        ;
        function isSomeInstanceType(...instanceTypes) {
            return instanceTypes.some(isInstanceType);
        }
        ;
        function isEveryDatanodeInstanceType(...instanceTypes) {
            return instanceTypes.some(t => instanceType.startsWith(t));
        }
        ;
        // Validate feature support for the given Elasticsearch/OpenSearch version, per
        // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/features-by-version.html
        const { versionNum: versionNum, isElasticsearchVersion } = parseVersion(props.version);
        if (isElasticsearchVersion) {
            if (versionNum <= 7.7 &&
                ![
                    1.5, 2.3, 5.1, 5.3, 5.5, 5.6, 6.0,
                    6.2, 6.3, 6.4, 6.5, 6.7, 6.8, 7.1, 7.4,
                    7.7,
                ].includes(versionNum)) {
                throw new Error(`Unknown Elasticsearch version: ${versionNum}`);
            }
            if (versionNum < 5.1) {
                if (props.logging?.appLogEnabled) {
                    throw new Error('Error logs publishing requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
                if (props.encryptionAtRest?.enabled) {
                    throw new Error('Encryption of data at rest requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
                if (props.cognitoDashboardsAuth != null) {
                    throw new Error('Cognito authentication for OpenSearch Dashboards requires Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
                if (isSomeInstanceType('c5', 'i3', 'm5', 'r5')) {
                    throw new Error('C5, I3, M5, and R5 instance types require Elasticsearch version 5.1 or later or OpenSearch version 1.0 or later.');
                }
            }
            if (versionNum < 6.0) {
                if (props.nodeToNodeEncryption) {
                    throw new Error('Node-to-node encryption requires Elasticsearch version 6.0 or later or OpenSearch version 1.0 or later.');
                }
            }
            if (versionNum < 6.7) {
                if (unsignedBasicAuthEnabled) {
                    throw new Error('Using unsigned basic auth requires Elasticsearch version 6.7 or later or OpenSearch version 1.0 or later.');
                }
                if (advancedSecurityEnabled) {
                    throw new Error('Fine-grained access control requires Elasticsearch version 6.7 or later or OpenSearch version 1.0 or later.');
                }
            }
            if (versionNum < 6.8 && warmEnabled) {
                throw new Error('UltraWarm requires Elasticsearch version 6.8 or later or OpenSearch version 1.0 or later.');
            }
        }
        // Validate against instance type restrictions, per
        // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/supported-instance-types.html
        if (isSomeInstanceType('i3', 'r6gd') && ebsEnabled) {
            throw new Error('I3 and R6GD instance types do not support EBS storage volumes.');
        }
        if (isSomeInstanceType('m3', 'r3', 't2') && encryptionAtRestEnabled) {
            throw new Error('M3, R3, and T2 instance types do not support encryption of data at rest.');
        }
        if (isInstanceType('t2.micro') && !(isElasticsearchVersion && versionNum <= 2.3)) {
            throw new Error('The t2.micro.search instance type supports only Elasticsearch versions 1.5 and 2.3.');
        }
        if (isSomeInstanceType('t2', 't3') && warmEnabled) {
            throw new Error('T2 and T3 instance types do not support UltraWarm storage.');
        }
        // Only R3, I3 and r6gd support instance storage, per
        // https://aws.amazon.com/opensearch-service/pricing/
        if (!ebsEnabled && !isEveryDatanodeInstanceType('r3', 'i3', 'r6gd')) {
            throw new Error('EBS volumes are required when using instance types other than r3, i3 or r6gd.');
        }
        // Fine-grained access control requires node-to-node encryption, encryption at rest,
        // and enforced HTTPS.
        if (advancedSecurityEnabled) {
            if (!nodeToNodeEncryptionEnabled) {
                throw new Error('Node-to-node encryption is required when fine-grained access control is enabled.');
            }
            if (!encryptionAtRestEnabled) {
                throw new Error('Encryption-at-rest is required when fine-grained access control is enabled.');
            }
            if (!enforceHttps) {
                throw new Error('Enforce HTTPS is required when fine-grained access control is enabled.');
            }
        }
        // Validate fine grained access control enabled for audit logs, per
        // https://aws.amazon.com/about-aws/whats-new/2020/09/elasticsearch-audit-logs-now-available-on-amazon-elasticsearch-service/
        if (props.logging?.auditLogEnabled && !advancedSecurityEnabled) {
            throw new Error('Fine-grained access control is required when audit logs publishing is enabled.');
        }
        // Validate UltraWarm requirement for dedicated master nodes, per
        // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/ultrawarm.html
        if (warmEnabled && !dedicatedMasterEnabled) {
            throw new Error('Dedicated master node is required when UltraWarm storage is enabled.');
        }
        let cfnVpcOptions;
        if (securityGroups && subnets) {
            cfnVpcOptions = {
                securityGroupIds: securityGroups.map((sg) => sg.securityGroupId),
                subnetIds: subnets.map((subnet) => subnet.subnetId),
            };
        }
        // Setup logging
        const logGroups = [];
        if (props.logging?.slowSearchLogEnabled) {
            this.slowSearchLogGroup = props.logging.slowSearchLogGroup ??
                new logs.LogGroup(this, 'SlowSearchLogs', {
                    retention: logs.RetentionDays.ONE_MONTH,
                });
            logGroups.push(this.slowSearchLogGroup);
        }
        ;
        if (props.logging?.slowIndexLogEnabled) {
            this.slowIndexLogGroup = props.logging.slowIndexLogGroup ??
                new logs.LogGroup(this, 'SlowIndexLogs', {
                    retention: logs.RetentionDays.ONE_MONTH,
                });
            logGroups.push(this.slowIndexLogGroup);
        }
        ;
        if (props.logging?.appLogEnabled) {
            this.appLogGroup = props.logging.appLogGroup ??
                new logs.LogGroup(this, 'AppLogs', {
                    retention: logs.RetentionDays.ONE_MONTH,
                });
            logGroups.push(this.appLogGroup);
        }
        ;
        if (props.logging?.auditLogEnabled) {
            this.auditLogGroup = props.logging.auditLogGroup ??
                new logs.LogGroup(this, 'AuditLogs', {
                    retention: logs.RetentionDays.ONE_MONTH,
                });
            logGroups.push(this.auditLogGroup);
        }
        ;
        let logGroupResourcePolicy = null;
        if (logGroups.length > 0) {
            const logPolicyStatement = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: logGroups.map((lg) => lg.logGroupArn),
                principals: [new iam.ServicePrincipal('es.amazonaws.com')],
            });
            // Use a custom resource to set the log group resource policy since it is not supported by CDK and cfn.
            // https://github.com/aws/aws-cdk/issues/5343
            logGroupResourcePolicy = new log_group_resource_policy_1.LogGroupResourcePolicy(this, `ESLogGroupPolicy${this.node.addr}`, {
                // create a cloudwatch logs resource policy name that is unique to this domain instance
                policyName: `ESLogPolicy${this.node.addr}`,
                policyStatements: [logPolicyStatement],
            });
        }
        const logPublishing = {};
        if (this.appLogGroup) {
            logPublishing.ES_APPLICATION_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.appLogGroup.logGroupArn,
            };
        }
        if (this.slowSearchLogGroup) {
            logPublishing.SEARCH_SLOW_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.slowSearchLogGroup.logGroupArn,
            };
        }
        if (this.slowIndexLogGroup) {
            logPublishing.INDEX_SLOW_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.slowIndexLogGroup.logGroupArn,
            };
        }
        if (this.auditLogGroup) {
            logPublishing.AUDIT_LOGS = {
                enabled: this.auditLogGroup != null,
                cloudWatchLogsLogGroupArn: this.auditLogGroup?.logGroupArn,
            };
        }
        let customEndpointCertificate;
        if (props.customEndpoint) {
            if (props.customEndpoint.certificate) {
                customEndpointCertificate = props.customEndpoint.certificate;
            }
            else {
                customEndpointCertificate = new acm.Certificate(this, 'CustomEndpointCertificate', {
                    domainName: props.customEndpoint.domainName,
                    validation: props.customEndpoint.hostedZone ? acm.CertificateValidation.fromDns(props.customEndpoint.hostedZone) : undefined,
                });
            }
        }
        // Create the domain
        this.domain = new opensearchservice_generated_1.CfnDomain(this, 'Resource', {
            domainName: this.physicalName,
            engineVersion: props.version.version,
            clusterConfig: {
                dedicatedMasterEnabled,
                dedicatedMasterCount: dedicatedMasterEnabled
                    ? dedicatedMasterCount
                    : undefined,
                dedicatedMasterType: dedicatedMasterEnabled
                    ? dedicatedMasterType
                    : undefined,
                instanceCount,
                instanceType,
                warmEnabled: warmEnabled
                    ? warmEnabled
                    : undefined,
                warmCount: warmEnabled
                    ? warmCount
                    : undefined,
                warmType: warmEnabled
                    ? warmType
                    : undefined,
                zoneAwarenessEnabled,
                zoneAwarenessConfig: zoneAwarenessEnabled
                    ? { availabilityZoneCount }
                    : undefined,
            },
            ebsOptions: {
                ebsEnabled,
                volumeSize: ebsEnabled ? volumeSize : undefined,
                volumeType: ebsEnabled ? volumeType : undefined,
                iops: ebsEnabled ? props.ebs?.iops : undefined,
            },
            encryptionAtRestOptions: {
                enabled: encryptionAtRestEnabled,
                kmsKeyId: encryptionAtRestEnabled
                    ? props.encryptionAtRest?.kmsKey?.keyId
                    : undefined,
            },
            nodeToNodeEncryptionOptions: { enabled: nodeToNodeEncryptionEnabled },
            logPublishingOptions: logPublishing,
            cognitoOptions: {
                enabled: props.cognitoDashboardsAuth != null,
                identityPoolId: props.cognitoDashboardsAuth?.identityPoolId,
                roleArn: props.cognitoDashboardsAuth?.role.roleArn,
                userPoolId: props.cognitoDashboardsAuth?.userPoolId,
            },
            vpcOptions: cfnVpcOptions,
            snapshotOptions: props.automatedSnapshotStartHour
                ? { automatedSnapshotStartHour: props.automatedSnapshotStartHour }
                : undefined,
            domainEndpointOptions: {
                enforceHttps,
                tlsSecurityPolicy: props.tlsSecurityPolicy ?? TLSSecurityPolicy.TLS_1_0,
                ...props.customEndpoint && {
                    customEndpointEnabled: true,
                    customEndpoint: props.customEndpoint.domainName,
                    customEndpointCertificateArn: customEndpointCertificate.certificateArn,
                },
            },
            advancedSecurityOptions: advancedSecurityEnabled
                ? {
                    enabled: true,
                    internalUserDatabaseEnabled,
                    masterUserOptions: {
                        masterUserArn: masterUserArn,
                        masterUserName: masterUserName,
                        masterUserPassword: this.masterUserPassword?.unsafeUnwrap(),
                    },
                }
                : undefined,
            advancedOptions: props.advancedOptions,
        });
        this.domain.applyRemovalPolicy(props.removalPolicy);
        if (props.enableVersionUpgrade) {
            this.domain.cfnOptions.updatePolicy = {
                ...this.domain.cfnOptions.updatePolicy,
                enableVersionUpgrade: props.enableVersionUpgrade,
            };
        }
        if (logGroupResourcePolicy) {
            this.domain.node.addDependency(logGroupResourcePolicy);
        }
        if (props.domainName) {
            if (!cdk.Token.isUnresolved(props.domainName)) {
                // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/configuration-api.html#configuration-api-datatypes-domainname
                if (!props.domainName.match(/^[a-z0-9\-]+$/)) {
                    throw new Error(`Invalid domainName '${props.domainName}'. Valid characters are a-z (lowercase only), 0-9, and – (hyphen).`);
                }
                if (props.domainName.length < 3 || props.domainName.length > 28) {
                    throw new Error(`Invalid domainName '${props.domainName}'. It must be between 3 and 28 characters`);
                }
                if (props.domainName[0] < 'a' || props.domainName[0] > 'z') {
                    throw new Error(`Invalid domainName '${props.domainName}'. It must start with a lowercase letter`);
                }
            }
            this.node.addMetadata('aws:cdk:hasPhysicalName', props.domainName);
        }
        this.domainName = this.getResourceNameAttribute(this.domain.ref);
        this.domainId = this.domain.getAtt('Id').toString();
        this.domainEndpoint = this.domain.getAtt('DomainEndpoint').toString();
        this.domainArn = this.getResourceArnAttribute(this.domain.attrArn, {
            service: 'es',
            resource: 'domain',
            resourceName: this.physicalName,
        });
        if (props.customEndpoint?.hostedZone) {
            new route53.CnameRecord(this, 'CnameRecord', {
                recordName: props.customEndpoint.domainName,
                zone: props.customEndpoint.hostedZone,
                domainName: this.domainEndpoint,
            });
        }
        this.encryptionAtRestOptions = props.encryptionAtRest;
        if (props.accessPolicies) {
            this.addAccessPolicies(...props.accessPolicies);
        }
        if (unsignedBasicAuthEnabled) {
            this.addAccessPolicies(unsignedAccessPolicy);
        }
    }
    /**
     * Creates a domain construct that represents an external domain via domain endpoint.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param domainEndpoint The domain's endpoint.
     */
    static fromDomainEndpoint(scope, id, domainEndpoint) {
        const stack = cdk.Stack.of(scope);
        const domainName = extractNameFromEndpoint(domainEndpoint);
        const domainArn = stack.formatArn({
            service: 'es',
            resource: 'domain',
            resourceName: domainName,
        });
        return Domain.fromDomainAttributes(scope, id, {
            domainArn,
            domainEndpoint,
        });
    }
    /**
     * Creates a domain construct that represents an external domain.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `DomainAttributes` object.
     */
    static fromDomainAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_opensearchservice_DomainAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDomainAttributes);
            }
            throw error;
        }
        const { domainArn, domainEndpoint } = attrs;
        const domainName = cdk.Stack.of(scope).splitArn(domainArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName
            ?? extractNameFromEndpoint(domainEndpoint);
        return new class extends DomainBase {
            constructor() {
                super(scope, id);
                this.domainArn = domainArn;
                this.domainName = domainName;
                this.domainId = domainName;
                this.domainEndpoint = domainEndpoint.replace(/^https?:\/\//, '');
            }
        };
    }
    /**
     * Manages network connections to the domain. This will throw an error in case the domain
     * is not placed inside a VPC.
     */
    get connections() {
        if (!this._connections) {
            throw new Error("Connections are only available on VPC enabled domains. Use the 'vpc' property to place a domain inside a VPC");
        }
        return this._connections;
    }
    /**
     * Add policy statements to the domain access policy
     */
    addAccessPolicies(...accessPolicyStatements) {
        if (accessPolicyStatements.length > 0) {
            if (!this.accessPolicy) {
                // Only create the custom resource after there are statements to set.
                this.accessPolicy = new opensearch_access_policy_1.OpenSearchAccessPolicy(this, 'AccessPolicy', {
                    domainName: this.domainName,
                    domainArn: this.domainArn,
                    accessPolicies: accessPolicyStatements,
                });
                if (this.encryptionAtRestOptions?.kmsKey) {
                    // https://docs.aws.amazon.com/opensearch-service/latest/developerguide/encryption-at-rest.html
                    // these permissions are documented as required during domain creation.
                    // while not strictly documented for updates as well, it stands to reason that an update
                    // operation might require these in case the cluster uses a kms key.
                    // empircal evidence shows this is indeed required: https://github.com/aws/aws-cdk/issues/11412
                    this.accessPolicy.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                        actions: ['kms:List*', 'kms:Describe*', 'kms:CreateGrant'],
                        resources: [this.encryptionAtRestOptions.kmsKey.keyArn],
                        effect: iam.Effect.ALLOW,
                    }));
                }
            }
            else {
                this.accessPolicy.addAccessPolicies(...accessPolicyStatements);
            }
        }
    }
}
exports.Domain = Domain;
_a = JSII_RTTI_SYMBOL_1;
Domain[_a] = { fqn: "@aws-cdk/aws-opensearchservice.Domain", version: "1.181.0" };
/**
 * Given an Amazon OpenSearch Service domain endpoint, returns a CloudFormation expression that
 * extracts the domain name.
 *
 * Domain endpoints look like this:
 *
 *   https://example-domain-jcjotrt6f7otem4sqcwbch3c4u.us-east-1.es.amazonaws.com
 *   https://<domain-name>-<suffix>.<region>.es.amazonaws.com
 *
 * ..which means that in order to extract the domain name from the endpoint, we can
 * split the endpoint using "-<suffix>" and select the component in index 0.
 *
 * @param domainEndpoint The Amazon OpenSearch Service domain endpoint
 */
function extractNameFromEndpoint(domainEndpoint) {
    const { hostname } = new url_1.URL(domainEndpoint);
    const domain = hostname.split('.')[0];
    const suffix = '-' + domain.split('-').slice(-1)[0];
    return domain.split(suffix)[0];
}
/**
 * Converts an engine version into a into a decimal number with major and minor version i.e x.y.
 *
 * @param version The engine version object
 */
function parseVersion(version) {
    const elasticsearchPrefix = 'Elasticsearch_';
    const openSearchPrefix = 'OpenSearch_';
    const isElasticsearchVersion = version.version.startsWith(elasticsearchPrefix);
    const versionStr = isElasticsearchVersion
        ? version.version.substring(elasticsearchPrefix.length)
        : version.version.substring(openSearchPrefix.length);
    const firstDot = versionStr.indexOf('.');
    if (firstDot < 1) {
        throw new Error(`Invalid engine version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
    const secondDot = versionStr.indexOf('.', firstDot + 1);
    try {
        if (secondDot == -1) {
            return { versionNum: parseFloat(versionStr), isElasticsearchVersion };
        }
        else {
            return { versionNum: parseFloat(versionStr.substring(0, secondDot)), isElasticsearchVersion };
        }
    }
    catch (error) {
        throw new Error(`Invalid engine version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
}
function selectSubnets(vpc, vpcSubnets) {
    const selected = [];
    for (const selection of vpcSubnets) {
        selected.push(...vpc.selectSubnets(selection).subnets);
    }
    return selected;
}
/**
 * Initializes an instance type.
 *
 * @param defaultInstanceType Default instance type which is used if no instance type is provided
 * @param instanceType Instance type
 * @returns Instance type in lowercase (if provided) or default instance type
 */
function initializeInstanceType(defaultInstanceType, instanceType) {
    if (instanceType) {
        return cdk.Token.isUnresolved(instanceType) ? instanceType : instanceType.toLowerCase();
    }
    else {
        return defaultInstanceType;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZG9tYWluLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiZG9tYWluLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7OztBQUFBLDZCQUEwQjtBQUUxQix1REFBdUQ7QUFDdkQsNERBQTJFO0FBQzNFLHdDQUF3QztBQUN4Qyx3Q0FBd0M7QUFFeEMsMENBQTBDO0FBQzFDLGdEQUFnRDtBQUNoRCw4REFBOEQ7QUFDOUQscUNBQXFDO0FBR3JDLDJFQUFxRTtBQUNyRSx5RUFBb0U7QUFDcEUsK0VBQTBEO0FBQzFELGlDQUFpQztBQW9QakM7O0dBRUc7QUFDSCxJQUFZLGlCQUtYO0FBTEQsV0FBWSxpQkFBaUI7SUFDM0IsMkJBQTJCO0lBQzNCLDJEQUFzQyxDQUFBO0lBQ3RDLDJCQUEyQjtJQUMzQiwyREFBc0MsQ0FBQTtBQUN4QyxDQUFDLEVBTFcsaUJBQWlCLEdBQWpCLHlCQUFpQixLQUFqQix5QkFBaUIsUUFLNUI7QUE2Y0Q7O0dBRUc7QUFDSCxNQUFlLFVBQVcsU0FBUSxHQUFHLENBQUMsUUFBUTtJQU01Qzs7Ozs7T0FLRztJQUNILFNBQVMsQ0FBQyxRQUF3QjtRQUNoQyxPQUFPLElBQUksQ0FBQyxLQUFLLENBQ2YsUUFBUSxFQUNSLEtBQUssQ0FBQyxlQUFlLEVBQ3JCLElBQUksQ0FBQyxTQUFTLEVBQ2QsR0FBRyxJQUFJLENBQUMsU0FBUyxJQUFJLENBQ3RCLENBQUM7S0FDSDtJQUVEOzs7OztPQUtHO0lBQ0gsVUFBVSxDQUFDLFFBQXdCO1FBQ2pDLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FDZixRQUFRLEVBQ1IsS0FBSyxDQUFDLGdCQUFnQixFQUN0QixJQUFJLENBQUMsU0FBUyxFQUNkLEdBQUcsSUFBSSxDQUFDLFNBQVMsSUFBSSxDQUN0QixDQUFDO0tBQ0g7SUFFRDs7Ozs7T0FLRztJQUNILGNBQWMsQ0FBQyxRQUF3QjtRQUNyQyxPQUFPLElBQUksQ0FBQyxLQUFLLENBQ2YsUUFBUSxFQUNSLEtBQUssQ0FBQyxxQkFBcUIsRUFDM0IsSUFBSSxDQUFDLFNBQVMsRUFDZCxHQUFHLElBQUksQ0FBQyxTQUFTLElBQUksQ0FDdEIsQ0FBQztLQUNIO0lBRUQ7Ozs7OztPQU1HO0lBQ0gsY0FBYyxDQUFDLEtBQWEsRUFBRSxRQUF3QjtRQUNwRCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQ2YsUUFBUSxFQUNSLEtBQUssQ0FBQyxlQUFlLEVBQ3JCLEdBQUcsSUFBSSxDQUFDLFNBQVMsSUFBSSxLQUFLLEVBQUUsRUFDNUIsR0FBRyxJQUFJLENBQUMsU0FBUyxJQUFJLEtBQUssSUFBSSxDQUMvQixDQUFDO0tBQ0g7SUFFRDs7Ozs7O09BTUc7SUFDSCxlQUFlLENBQUMsS0FBYSxFQUFFLFFBQXdCO1FBQ3JELE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FDZixRQUFRLEVBQ1IsS0FBSyxDQUFDLGdCQUFnQixFQUN0QixHQUFHLElBQUksQ0FBQyxTQUFTLElBQUksS0FBSyxFQUFFLEVBQzVCLEdBQUcsSUFBSSxDQUFDLFNBQVMsSUFBSSxLQUFLLElBQUksQ0FDL0IsQ0FBQztLQUNIO0lBRUQ7Ozs7OztPQU1HO0lBQ0gsbUJBQW1CLENBQUMsS0FBYSxFQUFFLFFBQXdCO1FBQ3pELE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FDZixRQUFRLEVBQ1IsS0FBSyxDQUFDLHFCQUFxQixFQUMzQixHQUFHLElBQUksQ0FBQyxTQUFTLElBQUksS0FBSyxFQUFFLEVBQzVCLEdBQUcsSUFBSSxDQUFDLFNBQVMsSUFBSSxLQUFLLElBQUksQ0FDL0IsQ0FBQztLQUNIO0lBRUQ7Ozs7OztPQU1HO0lBQ0gsYUFBYSxDQUFDLElBQVksRUFBRSxRQUF3QjtRQUNsRCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQ2YsUUFBUSxFQUNSLEtBQUssQ0FBQyxlQUFlLEVBQ3JCLEdBQUcsSUFBSSxDQUFDLFNBQVMsSUFBSSxJQUFJLEVBQUUsQ0FDNUIsQ0FBQztLQUNIO0lBRUQ7Ozs7OztPQU1HO0lBQ0gsY0FBYyxDQUFDLElBQVksRUFBRSxRQUF3QjtRQUNuRCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQ2YsUUFBUSxFQUNSLEtBQUssQ0FBQyxnQkFBZ0IsRUFDdEIsR0FBRyxJQUFJLENBQUMsU0FBUyxJQUFJLElBQUksRUFBRSxDQUM1QixDQUFDO0tBQ0g7SUFFRDs7Ozs7O09BTUc7SUFDSCxrQkFBa0IsQ0FBQyxJQUFZLEVBQUUsUUFBd0I7UUFDdkQsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUNmLFFBQVEsRUFDUixLQUFLLENBQUMscUJBQXFCLEVBQzNCLEdBQUcsSUFBSSxDQUFDLFNBQVMsSUFBSSxJQUFJLEVBQUUsQ0FDNUIsQ0FBQztLQUNIO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsVUFBa0IsRUFBRSxLQUFxQjtRQUNyRCxPQUFPLElBQUksdUJBQU0sQ0FBQztZQUNoQixTQUFTLEVBQUUsUUFBUTtZQUNuQixVQUFVO1lBQ1YsYUFBYSxFQUFFO2dCQUNiLFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVTtnQkFDM0IsUUFBUSxFQUFFLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBTzthQUMzQjtZQUNELEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7S0FDbkI7SUFFRDs7OztPQUlHO0lBQ0ksc0JBQXNCLENBQUMsS0FBcUI7UUFDakQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLG1CQUFtQixFQUFFO1lBQ3RDLFNBQVMsRUFBRSwwQkFBUyxDQUFDLE9BQU87WUFDNUIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0tBQ0o7SUFFRDs7OztPQUlHO0lBQ0kseUJBQXlCLENBQUMsS0FBcUI7UUFDcEQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLHNCQUFzQixFQUFFO1lBQ3pDLFNBQVMsRUFBRSwwQkFBUyxDQUFDLE9BQU87WUFDNUIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0tBQ0o7SUFFRDs7OztPQUlHO0lBQ0ksc0JBQXNCLENBQUMsS0FBcUI7UUFDakQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLGtCQUFrQixFQUFFO1lBQ3JDLFNBQVMsRUFBRSwwQkFBUyxDQUFDLE9BQU87WUFDNUIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0tBQ0o7SUFFRDs7OztPQUlHO0lBQ0ksK0JBQStCLENBQUMsS0FBcUI7UUFDMUQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLDJCQUEyQixFQUFFO1lBQzlDLFNBQVMsRUFBRSwwQkFBUyxDQUFDLE9BQU87WUFDNUIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztZQUMvQixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7S0FDSjtJQUVEOzs7O09BSUc7SUFDSSxXQUFXLENBQUMsS0FBcUI7UUFDdEMsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU8sRUFBRTtZQUMxQixTQUFTLEVBQUUsMEJBQVMsQ0FBQyxPQUFPO1lBQzVCLE1BQU0sRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7WUFDN0IsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0tBQ0o7SUFFRDs7OztPQUlHO0lBQ0ksOEJBQThCLENBQUMsS0FBcUI7UUFDekQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLDBCQUEwQixFQUFFO1lBQzdDLFNBQVMsRUFBRSwwQkFBUyxDQUFDLE9BQU87WUFDNUIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0tBQ0o7SUFFRDs7OztPQUlHO0lBQ0ksb0JBQW9CLENBQUMsS0FBcUI7UUFDL0MsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLGdCQUFnQixFQUFFO1lBQ25DLFNBQVMsRUFBRSwwQkFBUyxDQUFDLE9BQU87WUFDNUIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0tBQ0o7SUFFRDs7OztPQUlHO0lBQ0ksdUJBQXVCLENBQUMsS0FBcUI7UUFDbEQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLG1CQUFtQixFQUFFO1lBQ3RDLFNBQVMsRUFBRSwwQkFBUyxDQUFDLE9BQU87WUFDNUIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0tBQ0o7SUFFRDs7OztPQUlHO0lBQ0ksMEJBQTBCLENBQUMsS0FBcUI7UUFDckQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLHNCQUFzQixFQUFFO1lBQ3pDLFNBQVMsRUFBRSwwQkFBUyxDQUFDLE9BQU87WUFDNUIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0tBQ0o7SUFFRDs7OztPQUlHO0lBQ0ksNkJBQTZCLENBQUMsS0FBcUI7UUFDeEQsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLHlCQUF5QixFQUFFO1lBQzVDLFNBQVMsRUFBRSwwQkFBUyxDQUFDLE9BQU87WUFDNUIsR0FBRyxLQUFLO1NBQ1QsQ0FBQyxDQUFDO0tBQ0o7SUFFRDs7OztPQUlHO0lBQ0ksaUJBQWlCLENBQUMsS0FBcUI7UUFDNUMsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLGFBQWEsRUFBRTtZQUNoQyxTQUFTLEVBQUUsMEJBQVMsQ0FBQyxPQUFPO1lBQzVCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztLQUNKO0lBRUQ7Ozs7T0FJRztJQUNJLHdCQUF3QixDQUFDLEtBQXFCO1FBQ25ELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxvQkFBb0IsRUFBRTtZQUN2QyxTQUFTLEVBQUUsMEJBQVMsQ0FBQyxPQUFPO1lBQzVCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztLQUNKO0lBRUQ7Ozs7T0FJRztJQUNJLHlCQUF5QixDQUFDLEtBQXFCO1FBQ3BELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxxQkFBcUIsRUFBRTtZQUN4QyxTQUFTLEVBQUUsMEJBQVMsQ0FBQyxPQUFPO1lBQzVCLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztLQUNKO0lBRUQ7Ozs7T0FJRztJQUNJLG1CQUFtQixDQUFDLEtBQXFCO1FBQzlDLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxlQUFlLEVBQUUsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLEdBQUcsS0FBSyxFQUFFLENBQUMsQ0FBQztLQUNyRTtJQUVEOzs7O09BSUc7SUFDSSxxQkFBcUIsQ0FBQyxLQUFxQjtRQUNoRCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsaUJBQWlCLEVBQUUsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLEdBQUcsS0FBSyxFQUFFLENBQUMsQ0FBQztLQUN2RTtJQUVPLEtBQUssQ0FDWCxPQUF1QixFQUN2QixhQUF1QixFQUN2QixXQUFtQixFQUNuQixHQUFHLGlCQUEyQjtRQUU5QixNQUFNLFlBQVksR0FBRyxDQUFDLFdBQVcsRUFBRSxHQUFHLGlCQUFpQixDQUFDLENBQUM7UUFFekQsTUFBTSxLQUFLLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUM7WUFDckMsT0FBTztZQUNQLE9BQU8sRUFBRSxhQUFhO1lBQ3RCLFlBQVk7WUFDWixLQUFLLEVBQUUsSUFBSTtTQUNaLENBQUMsQ0FBQztRQUVILE9BQU8sS0FBSyxDQUFDO0tBQ2Q7Q0FFRjtBQW1CRDs7R0FFRztBQUNILE1BQWEsTUFBTyxTQUFRLFVBQVU7SUFnR3BDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBa0I7UUFDMUQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixZQUFZLEVBQUUsS0FBSyxDQUFDLFVBQVU7U0FDL0IsQ0FBQyxDQUFDOzs7Ozs7K0NBbkdNLE1BQU07Ozs7UUFxR2YsTUFBTSxtQkFBbUIsR0FBRyxpQkFBaUIsQ0FBQztRQUM5QyxNQUFNLHVCQUF1QixHQUFHLDBCQUEwQixDQUFDO1FBRTNELE1BQU0sbUJBQW1CLEdBQUcsc0JBQXNCLENBQUMsbUJBQW1CLEVBQUUsS0FBSyxDQUFDLFFBQVEsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2hILE1BQU0sb0JBQW9CLEdBQUcsS0FBSyxDQUFDLFFBQVEsRUFBRSxXQUFXLElBQUksQ0FBQyxDQUFDO1FBQzlELE1BQU0sc0JBQXNCLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsb0JBQW9CLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxvQkFBb0IsR0FBRyxDQUFDLENBQUM7UUFFOUcsTUFBTSxZQUFZLEdBQUcsc0JBQXNCLENBQUMsbUJBQW1CLEVBQUUsS0FBSyxDQUFDLFFBQVEsRUFBRSxvQkFBb0IsQ0FBQyxDQUFDO1FBQ3ZHLE1BQU0sYUFBYSxHQUFHLEtBQUssQ0FBQyxRQUFRLEVBQUUsU0FBUyxJQUFJLENBQUMsQ0FBQztRQUVyRCxNQUFNLFFBQVEsR0FBRyxzQkFBc0IsQ0FBQyx1QkFBdUIsRUFBRSxLQUFLLENBQUMsUUFBUSxFQUFFLGdCQUFnQixDQUFDLENBQUM7UUFDbkcsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDLFFBQVEsRUFBRSxTQUFTLElBQUksQ0FBQyxDQUFDO1FBQ2pELE1BQU0sV0FBVyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLFNBQVMsR0FBRyxDQUFDLENBQUM7UUFFN0UsTUFBTSxxQkFBcUIsR0FDekIsS0FBSyxDQUFDLGFBQWEsRUFBRSxxQkFBcUIsSUFBSSxDQUFDLENBQUM7UUFFbEQsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxFQUFFO1lBQzNDLE1BQU0sSUFBSSxLQUFLLENBQUMsNEVBQTRFLENBQUMsQ0FBQztTQUMvRjtRQUVELE1BQU0sb0JBQW9CLEdBQ3hCLEtBQUssQ0FBQyxhQUFhLEVBQUUsT0FBTztZQUM1QixLQUFLLENBQUMsYUFBYSxFQUFFLHFCQUFxQixJQUFJLElBQUksQ0FBQztRQUdyRCxJQUFJLGNBQWdELENBQUM7UUFDckQsSUFBSSxPQUFrQyxDQUFDO1FBRXZDLElBQUksS0FBSyxDQUFDLEdBQUcsRUFBRTtZQUNiLE9BQU8sR0FBRyxhQUFhLENBQUMsS0FBSyxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsVUFBVSxJQUFJLENBQUMsRUFBRSxVQUFVLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDLENBQUMsQ0FBQztZQUMxRyxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFO29CQUNyRixHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7b0JBQ2QsV0FBVyxFQUFFLDZCQUE2QixJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsRUFBRTtpQkFDekQsQ0FBQyxDQUFDLENBQUM7WUFDSixJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksR0FBRyxDQUFDLFdBQVcsQ0FBQyxFQUFFLGNBQWMsRUFBRSxDQUFDLENBQUM7U0FDN0Q7UUFFRCxzRkFBc0Y7UUFDdEYsSUFBSSxPQUFPLElBQUksb0JBQW9CLElBQUksSUFBSSxHQUFHLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsTUFBTSxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxJQUFJLEdBQUcscUJBQXFCLEVBQUU7WUFDN0gsTUFBTSxJQUFJLEtBQUssQ0FBQyxtRkFBbUYsQ0FBQyxDQUFDO1NBQ3RHO1FBRUQsSUFBSSxDQUFDLG1CQUFtQixFQUFFLFlBQVksRUFBRSxRQUFRLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUNuSCxNQUFNLElBQUksS0FBSyxDQUFDLHlFQUF5RSxDQUFDLENBQUM7U0FDNUY7UUFFRCxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FBQyxFQUFFO1lBQzFFLE1BQU0sSUFBSSxLQUFLLENBQUMsMkRBQTJELENBQUMsQ0FBQztTQUM5RTtRQUVELE1BQU0sd0JBQXdCLEdBQUcsS0FBSyxDQUFDLG9CQUFvQixJQUFJLEtBQUssQ0FBQztRQUVyRSxJQUFJLHdCQUF3QixFQUFFO1lBQzVCLElBQUksS0FBSyxDQUFDLFlBQVksSUFBSSxLQUFLLEVBQUU7Z0JBQy9CLE1BQU0sSUFBSSxLQUFLLENBQUMsc0RBQXNELENBQUMsQ0FBQzthQUN6RTtZQUNELElBQUksS0FBSyxDQUFDLG9CQUFvQixJQUFJLEtBQUssRUFBRTtnQkFDdkMsTUFBTSxJQUFJLEtBQUssQ0FBQyx3RUFBd0UsQ0FBQyxDQUFDO2FBQzNGO1lBQ0QsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUUsT0FBTyxJQUFJLEtBQUssRUFBRTtnQkFDNUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxtRUFBbUUsQ0FBQyxDQUFDO2FBQ3RGO1NBQ0Y7UUFFRCxNQUFNLG9CQUFvQixHQUFHLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUNuRCxNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLE9BQU8sRUFBRSxDQUFDLFlBQVksQ0FBQztZQUN2QixVQUFVLEVBQUUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxZQUFZLEVBQUUsQ0FBQztZQUNwQyxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxHQUFHLElBQUksQ0FBQyxTQUFTLElBQUksRUFBRSxDQUFDLENBQUM7U0FDdkUsQ0FBQyxDQUFDO1FBRUgsTUFBTSxhQUFhLEdBQUcsS0FBSyxDQUFDLHdCQUF3QixFQUFFLGFBQWEsQ0FBQztRQUNwRSxNQUFNLG1CQUFtQixHQUFHLEtBQUssQ0FBQyx3QkFBd0IsRUFBRSxjQUFjLENBQUM7UUFDM0UseUZBQXlGO1FBQ3pGLE1BQU0sY0FBYyxHQUFHLHdCQUF3QjtZQUM3QyxDQUFDLENBQUMsQ0FBQyxhQUFhLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLG1CQUFtQixJQUFJLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7WUFDeEUsQ0FBQyxDQUFDLG1CQUFtQixDQUFDO1FBRXhCLElBQUksYUFBYSxJQUFJLElBQUksSUFBSSxjQUFjLElBQUksSUFBSSxFQUFFO1lBQ25ELE1BQU0sSUFBSSxLQUFLLENBQUMsa0hBQWtILENBQUMsQ0FBQztTQUNySTtRQUVELE1BQU0sdUJBQXVCLEdBQUcsQ0FBQyxhQUFhLElBQUksY0FBYyxDQUFDLElBQUksSUFBSSxDQUFDO1FBQzFFLE1BQU0sMkJBQTJCLEdBQUcsY0FBYyxJQUFJLElBQUksQ0FBQztRQUMzRCxNQUFNLHNCQUFzQixHQUFHLEtBQUssQ0FBQyx3QkFBd0IsRUFBRSxrQkFBa0IsQ0FBQztRQUNsRixNQUFNLHdCQUF3QixHQUFHLEdBQW9CLEVBQUU7WUFDckQsT0FBTyxJQUFJLGNBQWMsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRTtnQkFDbkQsb0JBQW9CLEVBQUU7b0JBQ3BCLG9CQUFvQixFQUFFLElBQUksQ0FBQyxTQUFTLENBQUM7d0JBQ25DLFFBQVEsRUFBRSxjQUFjO3FCQUN6QixDQUFDO29CQUNGLGlCQUFpQixFQUFFLFVBQVU7b0JBQzdCLGlCQUFpQixFQUFFLGFBQWE7aUJBQ2pDO2FBQ0YsQ0FBQztpQkFDQyxtQkFBbUIsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUNyQyxDQUFDLENBQUM7UUFDRixJQUFJLENBQUMsa0JBQWtCLEdBQUcsMkJBQTJCLENBQUMsQ0FBQztZQUNyRCxDQUFDLHNCQUFzQixJQUFJLHdCQUF3QixFQUFFLENBQUM7WUFDdEQsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUVkLE1BQU0sdUJBQXVCLEdBQzNCLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRSxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLEVBQUUsTUFBTSxJQUFJLElBQUksSUFBSSx3QkFBd0IsQ0FBQyxDQUFDO1FBQzFHLE1BQU0sMkJBQTJCLEdBQUcsS0FBSyxDQUFDLG9CQUFvQixJQUFJLHdCQUF3QixDQUFDO1FBQzNGLE1BQU0sVUFBVSxHQUFHLEtBQUssQ0FBQyxHQUFHLEVBQUUsVUFBVSxJQUFJLEVBQUUsQ0FBQztRQUMvQyxNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsR0FBRyxFQUFFLFVBQVUsSUFBSSxHQUFHLENBQUMsbUJBQW1CLENBQUMsbUJBQW1CLENBQUM7UUFDeEYsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLEdBQUcsRUFBRSxPQUFPLElBQUksSUFBSSxDQUFDO1FBQzlDLE1BQU0sWUFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLElBQUksd0JBQXdCLENBQUM7UUFFcEUsU0FBUyxjQUFjLENBQUMsQ0FBUztZQUMvQixPQUFPLG1CQUFtQixDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsSUFBSSxZQUFZLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3pFLENBQUM7UUFBQSxDQUFDO1FBRUYsU0FBUyxrQkFBa0IsQ0FBQyxHQUFHLGFBQXVCO1lBQ3BELE9BQU8sYUFBYSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUM1QyxDQUFDO1FBQUEsQ0FBQztRQUVGLFNBQVMsMkJBQTJCLENBQUMsR0FBRyxhQUF1QjtZQUM3RCxPQUFPLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxZQUFZLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDN0QsQ0FBQztRQUFBLENBQUM7UUFFRiwrRUFBK0U7UUFDL0UsZ0dBQWdHO1FBQ2hHLE1BQU0sRUFBRSxVQUFVLEVBQUUsVUFBVSxFQUFFLHNCQUFzQixFQUFFLEdBQUcsWUFBWSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUN2RixJQUFJLHNCQUFzQixFQUFFO1lBRTFCLElBQ0UsVUFBVSxJQUFJLEdBQUc7Z0JBQ2pCLENBQUM7b0JBQ0MsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRztvQkFDakMsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUc7b0JBQ3RDLEdBQUc7aUJBQ0osQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLEVBQ3RCO2dCQUNBLE1BQU0sSUFBSSxLQUFLLENBQUMsa0NBQWtDLFVBQVUsRUFBRSxDQUFDLENBQUM7YUFDakU7WUFFRCxJQUFJLFVBQVUsR0FBRyxHQUFHLEVBQUU7Z0JBQ3BCLElBQUksS0FBSyxDQUFDLE9BQU8sRUFBRSxhQUFhLEVBQUU7b0JBQ2hDLE1BQU0sSUFBSSxLQUFLLENBQUMsdUdBQXVHLENBQUMsQ0FBQztpQkFDMUg7Z0JBQ0QsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUUsT0FBTyxFQUFFO29CQUNuQyxNQUFNLElBQUksS0FBSyxDQUFDLDRHQUE0RyxDQUFDLENBQUM7aUJBQy9IO2dCQUNELElBQUksS0FBSyxDQUFDLHFCQUFxQixJQUFJLElBQUksRUFBRTtvQkFDdkMsTUFBTSxJQUFJLEtBQUssQ0FBQyxrSUFBa0ksQ0FBQyxDQUFDO2lCQUNySjtnQkFDRCxJQUFJLGtCQUFrQixDQUFDLElBQUksRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLElBQUksQ0FBQyxFQUFFO29CQUM5QyxNQUFNLElBQUksS0FBSyxDQUFDLGtIQUFrSCxDQUFDLENBQUM7aUJBQ3JJO2FBQ0Y7WUFFRCxJQUFJLFVBQVUsR0FBRyxHQUFHLEVBQUU7Z0JBQ3BCLElBQUksS0FBSyxDQUFDLG9CQUFvQixFQUFFO29CQUM5QixNQUFNLElBQUksS0FBSyxDQUFDLHlHQUF5RyxDQUFDLENBQUM7aUJBQzVIO2FBQ0Y7WUFFRCxJQUFJLFVBQVUsR0FBRyxHQUFHLEVBQUU7Z0JBQ3BCLElBQUksd0JBQXdCLEVBQUU7b0JBQzVCLE1BQU0sSUFBSSxLQUFLLENBQUMsMkdBQTJHLENBQUMsQ0FBQztpQkFDOUg7Z0JBQ0QsSUFBSSx1QkFBdUIsRUFBRTtvQkFDM0IsTUFBTSxJQUFJLEtBQUssQ0FBQyw2R0FBNkcsQ0FBQyxDQUFDO2lCQUNoSTthQUNGO1lBRUQsSUFBSSxVQUFVLEdBQUcsR0FBRyxJQUFJLFdBQVcsRUFBRTtnQkFDbkMsTUFBTSxJQUFJLEtBQUssQ0FBQywyRkFBMkYsQ0FBQyxDQUFDO2FBQzlHO1NBQ0Y7UUFFRCxtREFBbUQ7UUFDbkQscUdBQXFHO1FBQ3JHLElBQUksa0JBQWtCLENBQUMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLFVBQVUsRUFBRTtZQUNsRCxNQUFNLElBQUksS0FBSyxDQUFDLGdFQUFnRSxDQUFDLENBQUM7U0FDbkY7UUFFRCxJQUFJLGtCQUFrQixDQUFDLElBQUksRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUksdUJBQXVCLEVBQUU7WUFDbkUsTUFBTSxJQUFJLEtBQUssQ0FBQywwRUFBMEUsQ0FBQyxDQUFDO1NBQzdGO1FBRUQsSUFBSSxjQUFjLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxDQUFDLHNCQUFzQixJQUFJLFVBQVUsSUFBSSxHQUFHLENBQUMsRUFBRTtZQUNoRixNQUFNLElBQUksS0FBSyxDQUFDLHFGQUFxRixDQUFDLENBQUM7U0FDeEc7UUFFRCxJQUFJLGtCQUFrQixDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSSxXQUFXLEVBQUU7WUFDakQsTUFBTSxJQUFJLEtBQUssQ0FBQyw0REFBNEQsQ0FBQyxDQUFDO1NBQy9FO1FBRUQscURBQXFEO1FBQ3JELHFEQUFxRDtRQUNyRCxJQUFJLENBQUMsVUFBVSxJQUFJLENBQUMsMkJBQTJCLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxNQUFNLENBQUMsRUFBRTtZQUNuRSxNQUFNLElBQUksS0FBSyxDQUFDLCtFQUErRSxDQUFDLENBQUM7U0FDbEc7UUFFRCxvRkFBb0Y7UUFDcEYsc0JBQXNCO1FBQ3RCLElBQUksdUJBQXVCLEVBQUU7WUFDM0IsSUFBSSxDQUFDLDJCQUEyQixFQUFFO2dCQUNoQyxNQUFNLElBQUksS0FBSyxDQUFDLGtGQUFrRixDQUFDLENBQUM7YUFDckc7WUFDRCxJQUFJLENBQUMsdUJBQXVCLEVBQUU7Z0JBQzVCLE1BQU0sSUFBSSxLQUFLLENBQUMsNkVBQTZFLENBQUMsQ0FBQzthQUNoRztZQUNELElBQUksQ0FBQyxZQUFZLEVBQUU7Z0JBQ2pCLE1BQU0sSUFBSSxLQUFLLENBQUMsd0VBQXdFLENBQUMsQ0FBQzthQUMzRjtTQUNGO1FBRUQsbUVBQW1FO1FBQ25FLDZIQUE2SDtRQUM3SCxJQUFJLEtBQUssQ0FBQyxPQUFPLEVBQUUsZUFBZSxJQUFJLENBQUMsdUJBQXVCLEVBQUU7WUFDOUQsTUFBTSxJQUFJLEtBQUssQ0FBQyxnRkFBZ0YsQ0FBQyxDQUFDO1NBQ25HO1FBRUQsaUVBQWlFO1FBQ2pFLHNGQUFzRjtRQUN0RixJQUFJLFdBQVcsSUFBSSxDQUFDLHNCQUFzQixFQUFFO1lBQzFDLE1BQU0sSUFBSSxLQUFLLENBQUMsc0VBQXNFLENBQUMsQ0FBQztTQUN6RjtRQUVELElBQUksYUFBdUQsQ0FBQztRQUU1RCxJQUFJLGNBQWMsSUFBSSxPQUFPLEVBQUU7WUFDN0IsYUFBYSxHQUFHO2dCQUNkLGdCQUFnQixFQUFFLGNBQWMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUM7Z0JBQ2hFLFNBQVMsRUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDO2FBQ3BELENBQUM7U0FDSDtRQUVELGdCQUFnQjtRQUNoQixNQUFNLFNBQVMsR0FBcUIsRUFBRSxDQUFDO1FBRXZDLElBQUksS0FBSyxDQUFDLE9BQU8sRUFBRSxvQkFBb0IsRUFBRTtZQUN2QyxJQUFJLENBQUMsa0JBQWtCLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxrQkFBa0I7Z0JBQ3hELElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUU7b0JBQ3hDLFNBQVMsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLFNBQVM7aUJBQ3hDLENBQUMsQ0FBQztZQUVMLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLENBQUM7U0FDekM7UUFBQSxDQUFDO1FBRUYsSUFBSSxLQUFLLENBQUMsT0FBTyxFQUFFLG1CQUFtQixFQUFFO1lBQ3RDLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLGlCQUFpQjtnQkFDdEQsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxlQUFlLEVBQUU7b0JBQ3ZDLFNBQVMsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLFNBQVM7aUJBQ3hDLENBQUMsQ0FBQztZQUVMLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLENBQUM7U0FDeEM7UUFBQSxDQUFDO1FBRUYsSUFBSSxLQUFLLENBQUMsT0FBTyxFQUFFLGFBQWEsRUFBRTtZQUNoQyxJQUFJLENBQUMsV0FBVyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsV0FBVztnQkFDMUMsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7b0JBQ2pDLFNBQVMsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLFNBQVM7aUJBQ3hDLENBQUMsQ0FBQztZQUVMLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1NBQ2xDO1FBQUEsQ0FBQztRQUVGLElBQUksS0FBSyxDQUFDLE9BQU8sRUFBRSxlQUFlLEVBQUU7WUFDbEMsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLGFBQWE7Z0JBQzlDLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsV0FBVyxFQUFFO29CQUNuQyxTQUFTLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxTQUFTO2lCQUN4QyxDQUFDLENBQUM7WUFFTCxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztTQUNwQztRQUFBLENBQUM7UUFFRixJQUFJLHNCQUFzQixHQUFrQyxJQUFJLENBQUM7UUFDakUsSUFBSSxTQUFTLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUN4QixNQUFNLGtCQUFrQixHQUFHLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDakQsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztnQkFDeEIsT0FBTyxFQUFFLENBQUMsbUJBQW1CLEVBQUUsc0JBQXNCLENBQUM7Z0JBQ3RELFNBQVMsRUFBRSxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUMsV0FBVyxDQUFDO2dCQUNoRCxVQUFVLEVBQUUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO2FBQzNELENBQUMsQ0FBQztZQUVILHVHQUF1RztZQUN2Ryw2Q0FBNkM7WUFDN0Msc0JBQXNCLEdBQUcsSUFBSSxrREFBc0IsQ0FBQyxJQUFJLEVBQUUsbUJBQW1CLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLEVBQUU7Z0JBQzdGLHVGQUF1RjtnQkFDdkYsVUFBVSxFQUFFLGNBQWMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUU7Z0JBQzFDLGdCQUFnQixFQUFFLENBQUMsa0JBQWtCLENBQUM7YUFDdkMsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxNQUFNLGFBQWEsR0FBd0IsRUFBRSxDQUFDO1FBRTlDLElBQUksSUFBSSxDQUFDLFdBQVcsRUFBRTtZQUNwQixhQUFhLENBQUMsbUJBQW1CLEdBQUc7Z0JBQ2xDLE9BQU8sRUFBRSxJQUFJO2dCQUNiLHlCQUF5QixFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsV0FBVzthQUN4RCxDQUFDO1NBQ0g7UUFFRCxJQUFJLElBQUksQ0FBQyxrQkFBa0IsRUFBRTtZQUMzQixhQUFhLENBQUMsZ0JBQWdCLEdBQUc7Z0JBQy9CLE9BQU8sRUFBRSxJQUFJO2dCQUNiLHlCQUF5QixFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxXQUFXO2FBQy9ELENBQUM7U0FDSDtRQUVELElBQUksSUFBSSxDQUFDLGlCQUFpQixFQUFFO1lBQzFCLGFBQWEsQ0FBQyxlQUFlLEdBQUc7Z0JBQzlCLE9BQU8sRUFBRSxJQUFJO2dCQUNiLHlCQUF5QixFQUFFLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxXQUFXO2FBQzlELENBQUM7U0FDSDtRQUVELElBQUksSUFBSSxDQUFDLGFBQWEsRUFBRTtZQUN0QixhQUFhLENBQUMsVUFBVSxHQUFHO2dCQUN6QixPQUFPLEVBQUUsSUFBSSxDQUFDLGFBQWEsSUFBSSxJQUFJO2dCQUNuQyx5QkFBeUIsRUFBRSxJQUFJLENBQUMsYUFBYSxFQUFFLFdBQVc7YUFDM0QsQ0FBQztTQUNIO1FBRUQsSUFBSSx5QkFBdUQsQ0FBQztRQUM1RCxJQUFJLEtBQUssQ0FBQyxjQUFjLEVBQUU7WUFDeEIsSUFBSSxLQUFLLENBQUMsY0FBYyxDQUFDLFdBQVcsRUFBRTtnQkFDcEMseUJBQXlCLEdBQUcsS0FBSyxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUM7YUFDOUQ7aUJBQU07Z0JBQ0wseUJBQXlCLEdBQUcsSUFBSSxHQUFHLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSwyQkFBMkIsRUFBRTtvQkFDakYsVUFBVSxFQUFFLEtBQUssQ0FBQyxjQUFjLENBQUMsVUFBVTtvQkFDM0MsVUFBVSxFQUFFLEtBQUssQ0FBQyxjQUFjLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMscUJBQXFCLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7aUJBQzdILENBQUMsQ0FBQzthQUNKO1NBQ0Y7UUFFRCxvQkFBb0I7UUFDcEIsSUFBSSxDQUFDLE1BQU0sR0FBRyxJQUFJLHVDQUFTLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUM1QyxVQUFVLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDN0IsYUFBYSxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTztZQUNwQyxhQUFhLEVBQUU7Z0JBQ2Isc0JBQXNCO2dCQUN0QixvQkFBb0IsRUFBRSxzQkFBc0I7b0JBQzFDLENBQUMsQ0FBQyxvQkFBb0I7b0JBQ3RCLENBQUMsQ0FBQyxTQUFTO2dCQUNiLG1CQUFtQixFQUFFLHNCQUFzQjtvQkFDekMsQ0FBQyxDQUFDLG1CQUFtQjtvQkFDckIsQ0FBQyxDQUFDLFNBQVM7Z0JBQ2IsYUFBYTtnQkFDYixZQUFZO2dCQUNaLFdBQVcsRUFBRSxXQUFXO29CQUN0QixDQUFDLENBQUMsV0FBVztvQkFDYixDQUFDLENBQUMsU0FBUztnQkFDYixTQUFTLEVBQUUsV0FBVztvQkFDcEIsQ0FBQyxDQUFDLFNBQVM7b0JBQ1gsQ0FBQyxDQUFDLFNBQVM7Z0JBQ2IsUUFBUSxFQUFFLFdBQVc7b0JBQ25CLENBQUMsQ0FBQyxRQUFRO29CQUNWLENBQUMsQ0FBQyxTQUFTO2dCQUNiLG9CQUFvQjtnQkFDcEIsbUJBQW1CLEVBQUUsb0JBQW9CO29CQUN2QyxDQUFDLENBQUMsRUFBRSxxQkFBcUIsRUFBRTtvQkFDM0IsQ0FBQyxDQUFDLFNBQVM7YUFDZDtZQUNELFVBQVUsRUFBRTtnQkFDVixVQUFVO2dCQUNWLFVBQVUsRUFBRSxVQUFVLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsU0FBUztnQkFDL0MsVUFBVSxFQUFFLFVBQVUsQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUMvQyxJQUFJLEVBQUUsVUFBVSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUMsU0FBUzthQUMvQztZQUNELHVCQUF1QixFQUFFO2dCQUN2QixPQUFPLEVBQUUsdUJBQXVCO2dCQUNoQyxRQUFRLEVBQUUsdUJBQXVCO29CQUMvQixDQUFDLENBQUMsS0FBSyxDQUFDLGdCQUFnQixFQUFFLE1BQU0sRUFBRSxLQUFLO29CQUN2QyxDQUFDLENBQUMsU0FBUzthQUNkO1lBQ0QsMkJBQTJCLEVBQUUsRUFBRSxPQUFPLEVBQUUsMkJBQTJCLEVBQUU7WUFDckUsb0JBQW9CLEVBQUUsYUFBYTtZQUNuQyxjQUFjLEVBQUU7Z0JBQ2QsT0FBTyxFQUFFLEtBQUssQ0FBQyxxQkFBcUIsSUFBSSxJQUFJO2dCQUM1QyxjQUFjLEVBQUUsS0FBSyxDQUFDLHFCQUFxQixFQUFFLGNBQWM7Z0JBQzNELE9BQU8sRUFBRSxLQUFLLENBQUMscUJBQXFCLEVBQUUsSUFBSSxDQUFDLE9BQU87Z0JBQ2xELFVBQVUsRUFBRSxLQUFLLENBQUMscUJBQXFCLEVBQUUsVUFBVTthQUNwRDtZQUNELFVBQVUsRUFBRSxhQUFhO1lBQ3pCLGVBQWUsRUFBRSxLQUFLLENBQUMsMEJBQTBCO2dCQUMvQyxDQUFDLENBQUMsRUFBRSwwQkFBMEIsRUFBRSxLQUFLLENBQUMsMEJBQTBCLEVBQUU7Z0JBQ2xFLENBQUMsQ0FBQyxTQUFTO1lBQ2IscUJBQXFCLEVBQUU7Z0JBQ3JCLFlBQVk7Z0JBQ1osaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGlCQUFpQixJQUFJLGlCQUFpQixDQUFDLE9BQU87Z0JBQ3ZFLEdBQUcsS0FBSyxDQUFDLGNBQWMsSUFBSTtvQkFDekIscUJBQXFCLEVBQUUsSUFBSTtvQkFDM0IsY0FBYyxFQUFFLEtBQUssQ0FBQyxjQUFjLENBQUMsVUFBVTtvQkFDL0MsNEJBQTRCLEVBQUUseUJBQTBCLENBQUMsY0FBYztpQkFDeEU7YUFDRjtZQUNELHVCQUF1QixFQUFFLHVCQUF1QjtnQkFDOUMsQ0FBQyxDQUFDO29CQUNBLE9BQU8sRUFBRSxJQUFJO29CQUNiLDJCQUEyQjtvQkFDM0IsaUJBQWlCLEVBQUU7d0JBQ2pCLGFBQWEsRUFBRSxhQUFhO3dCQUM1QixjQUFjLEVBQUUsY0FBYzt3QkFDOUIsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixFQUFFLFlBQVksRUFBRTtxQkFDNUQ7aUJBQ0Y7Z0JBQ0QsQ0FBQyxDQUFDLFNBQVM7WUFDYixlQUFlLEVBQUUsS0FBSyxDQUFDLGVBQWU7U0FDdkMsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFcEQsSUFBSSxLQUFLLENBQUMsb0JBQW9CLEVBQUU7WUFDOUIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsWUFBWSxHQUFHO2dCQUNwQyxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLFlBQVk7Z0JBQ3RDLG9CQUFvQixFQUFFLEtBQUssQ0FBQyxvQkFBb0I7YUFDakQsQ0FBQztTQUNIO1FBRUQsSUFBSSxzQkFBc0IsRUFBRTtZQUFFLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1NBQUU7UUFFdkYsSUFBSSxLQUFLLENBQUMsVUFBVSxFQUFFO1lBQ3BCLElBQUksQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLEVBQUU7Z0JBQzdDLHFJQUFxSTtnQkFDckksSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxFQUFFO29CQUM1QyxNQUFNLElBQUksS0FBSyxDQUFDLHVCQUF1QixLQUFLLENBQUMsVUFBVSxvRUFBb0UsQ0FBQyxDQUFDO2lCQUM5SDtnQkFDRCxJQUFJLEtBQUssQ0FBQyxVQUFVLENBQUMsTUFBTSxHQUFHLENBQUMsSUFBSSxLQUFLLENBQUMsVUFBVSxDQUFDLE1BQU0sR0FBRyxFQUFFLEVBQUU7b0JBQy9ELE1BQU0sSUFBSSxLQUFLLENBQUMsdUJBQXVCLEtBQUssQ0FBQyxVQUFVLDJDQUEyQyxDQUFDLENBQUM7aUJBQ3JHO2dCQUNELElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsR0FBRyxHQUFHLElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsR0FBRyxHQUFHLEVBQUU7b0JBQzFELE1BQU0sSUFBSSxLQUFLLENBQUMsdUJBQXVCLEtBQUssQ0FBQyxVQUFVLDBDQUEwQyxDQUFDLENBQUM7aUJBQ3BHO2FBQ0Y7WUFDRCxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyx5QkFBeUIsRUFBRSxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUM7U0FDcEU7UUFFRCxJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRWpFLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUM7UUFFcEQsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBRXRFLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDLHVCQUF1QixDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsT0FBTyxFQUFFO1lBQ2pFLE9BQU8sRUFBRSxJQUFJO1lBQ2IsUUFBUSxFQUFFLFFBQVE7WUFDbEIsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO1NBQ2hDLENBQUMsQ0FBQztRQUVILElBQUksS0FBSyxDQUFDLGNBQWMsRUFBRSxVQUFVLEVBQUU7WUFDcEMsSUFBSSxPQUFPLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7Z0JBQzNDLFVBQVUsRUFBRSxLQUFLLENBQUMsY0FBYyxDQUFDLFVBQVU7Z0JBQzNDLElBQUksRUFBRSxLQUFLLENBQUMsY0FBYyxDQUFDLFVBQVU7Z0JBQ3JDLFVBQVUsRUFBRSxJQUFJLENBQUMsY0FBYzthQUNoQyxDQUFDLENBQUM7U0FDSjtRQUVELElBQUksQ0FBQyx1QkFBdUIsR0FBRyxLQUFLLENBQUMsZ0JBQWdCLENBQUM7UUFDdEQsSUFBSSxLQUFLLENBQUMsY0FBYyxFQUFFO1lBQ3hCLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxHQUFHLEtBQUssQ0FBQyxjQUFjLENBQUMsQ0FBQztTQUNqRDtRQUNELElBQUksd0JBQXdCLEVBQUU7WUFDNUIsSUFBSSxDQUFDLGlCQUFpQixDQUFDLG9CQUFvQixDQUFDLENBQUM7U0FDOUM7S0FDRjtJQS9pQkQ7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLGtCQUFrQixDQUM5QixLQUFnQixFQUNoQixFQUFVLEVBQ1YsY0FBc0I7UUFFdEIsTUFBTSxLQUFLLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDbEMsTUFBTSxVQUFVLEdBQUcsdUJBQXVCLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDM0QsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztZQUNoQyxPQUFPLEVBQUUsSUFBSTtZQUNiLFFBQVEsRUFBRSxRQUFRO1lBQ2xCLFlBQVksRUFBRSxVQUFVO1NBQ3pCLENBQUMsQ0FBQztRQUVILE9BQU8sTUFBTSxDQUFDLG9CQUFvQixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDNUMsU0FBUztZQUNULGNBQWM7U0FDZixDQUFDLENBQUM7S0FDSjtJQUVEOzs7Ozs7T0FNRztJQUNJLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF1Qjs7Ozs7Ozs7OztRQUN0RixNQUFNLEVBQUUsU0FBUyxFQUFFLGNBQWMsRUFBRSxHQUFHLEtBQUssQ0FBQztRQUM1QyxNQUFNLFVBQVUsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFLEdBQUcsQ0FBQyxTQUFTLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxZQUFZO2VBQ3JHLHVCQUF1QixDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBRTdDLE9BQU8sSUFBSSxLQUFNLFNBQVEsVUFBVTtZQU1qQztnQkFBZ0IsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztnQkFMakIsY0FBUyxHQUFHLFNBQVMsQ0FBQztnQkFDdEIsZUFBVSxHQUFHLFVBQVUsQ0FBQztnQkFDeEIsYUFBUSxHQUFHLFVBQVUsQ0FBQztnQkFDdEIsbUJBQWMsR0FBRyxjQUFjLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxFQUFFLENBQUMsQ0FBQztZQUUxQyxDQUFDO1NBQ3BDLENBQUM7S0FDSDtJQW1nQkQ7OztPQUdHO0lBQ0gsSUFBVyxXQUFXO1FBQ3BCLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxFQUFFO1lBQ3RCLE1BQU0sSUFBSSxLQUFLLENBQUMsOEdBQThHLENBQUMsQ0FBQztTQUNqSTtRQUNELE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQztLQUMxQjtJQUdEOztPQUVHO0lBQ0ksaUJBQWlCLENBQUMsR0FBRyxzQkFBNkM7UUFDdkUsSUFBSSxzQkFBc0IsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ3JDLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxFQUFFO2dCQUN0QixxRUFBcUU7Z0JBQ3JFLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxpREFBc0IsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO29CQUNuRSxVQUFVLEVBQUUsSUFBSSxDQUFDLFVBQVU7b0JBQzNCLFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUztvQkFDekIsY0FBYyxFQUFFLHNCQUFzQjtpQkFDdkMsQ0FBQyxDQUFDO2dCQUVILElBQUksSUFBSSxDQUFDLHVCQUF1QixFQUFFLE1BQU0sRUFBRTtvQkFDeEMsK0ZBQStGO29CQUUvRix1RUFBdUU7b0JBQ3ZFLHdGQUF3RjtvQkFDeEYsb0VBQW9FO29CQUNwRSwrRkFBK0Y7b0JBQy9GLElBQUksQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDLG9CQUFvQixDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQzt3QkFDNUUsT0FBTyxFQUFFLENBQUMsV0FBVyxFQUFFLGVBQWUsRUFBRSxpQkFBaUIsQ0FBQzt3QkFDMUQsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLHVCQUF1QixDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUM7d0JBQ3ZELE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7cUJBQ3pCLENBQUMsQ0FBQyxDQUFDO2lCQUNMO2FBQ0Y7aUJBQU07Z0JBQ0wsSUFBSSxDQUFDLFlBQVksQ0FBQyxpQkFBaUIsQ0FBQyxHQUFHLHNCQUFzQixDQUFDLENBQUM7YUFDaEU7U0FDRjtLQUNGOztBQTVsQkgsd0JBNmxCQzs7O0FBRUQ7Ozs7Ozs7Ozs7Ozs7R0FhRztBQUNILFNBQVMsdUJBQXVCLENBQUMsY0FBc0I7SUFDckQsTUFBTSxFQUFFLFFBQVEsRUFBRSxHQUFHLElBQUksU0FBRyxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQzdDLE1BQU0sTUFBTSxHQUFHLFFBQVEsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDdEMsTUFBTSxNQUFNLEdBQUcsR0FBRyxHQUFHLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDcEQsT0FBTyxNQUFNLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0FBQ2pDLENBQUM7QUFFRDs7OztHQUlHO0FBQ0gsU0FBUyxZQUFZLENBQUMsT0FBc0I7SUFDMUMsTUFBTSxtQkFBbUIsR0FBRyxnQkFBZ0IsQ0FBQztJQUM3QyxNQUFNLGdCQUFnQixHQUFHLGFBQWEsQ0FBQztJQUN2QyxNQUFNLHNCQUFzQixHQUFHLE9BQU8sQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDL0UsTUFBTSxVQUFVLEdBQUcsc0JBQXNCO1FBQ3ZDLENBQUMsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxtQkFBbUIsQ0FBQyxNQUFNLENBQUM7UUFDdkQsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQ3ZELE1BQU0sUUFBUSxHQUFHLFVBQVUsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7SUFFekMsSUFBSSxRQUFRLEdBQUcsQ0FBQyxFQUFFO1FBQ2hCLE1BQU0sSUFBSSxLQUFLLENBQUMsMkJBQTJCLFVBQVUscUVBQXFFLENBQUMsQ0FBQztLQUM3SDtJQUVELE1BQU0sU0FBUyxHQUFHLFVBQVUsQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLFFBQVEsR0FBRyxDQUFDLENBQUMsQ0FBQztJQUV4RCxJQUFJO1FBQ0YsSUFBSSxTQUFTLElBQUksQ0FBQyxDQUFDLEVBQUU7WUFDbkIsT0FBTyxFQUFFLFVBQVUsRUFBRSxVQUFVLENBQUMsVUFBVSxDQUFDLEVBQUUsc0JBQXNCLEVBQUUsQ0FBQztTQUN2RTthQUFNO1lBQ0wsT0FBTyxFQUFFLFVBQVUsRUFBRSxVQUFVLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxDQUFDLEVBQUUsU0FBUyxDQUFDLENBQUMsRUFBRSxzQkFBc0IsRUFBRSxDQUFDO1NBQy9GO0tBQ0Y7SUFBQyxPQUFPLEtBQUssRUFBRTtRQUNkLE1BQU0sSUFBSSxLQUFLLENBQUMsMkJBQTJCLFVBQVUscUVBQXFFLENBQUMsQ0FBQztLQUM3SDtBQUNILENBQUM7QUFFRCxTQUFTLGFBQWEsQ0FBQyxHQUFhLEVBQUUsVUFBaUM7SUFDckUsTUFBTSxRQUFRLEdBQUcsRUFBRSxDQUFDO0lBQ3BCLEtBQUssTUFBTSxTQUFTLElBQUksVUFBVSxFQUFFO1FBQ2xDLFFBQVEsQ0FBQyxJQUFJLENBQUMsR0FBRyxHQUFHLENBQUMsYUFBYSxDQUFDLFNBQVMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0tBQ3hEO0lBQ0QsT0FBTyxRQUFRLENBQUM7QUFDbEIsQ0FBQztBQUVEOzs7Ozs7R0FNRztBQUNILFNBQVMsc0JBQXNCLENBQUMsbUJBQTJCLEVBQUUsWUFBcUI7SUFDaEYsSUFBSSxZQUFZLEVBQUU7UUFDaEIsT0FBTyxHQUFHLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsV0FBVyxFQUFFLENBQUM7S0FDekY7U0FBTTtRQUNMLE9BQU8sbUJBQW1CLENBQUM7S0FDNUI7QUFDSCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgVVJMIH0gZnJvbSAndXJsJztcblxuaW1wb3J0ICogYXMgYWNtIGZyb20gJ0Bhd3MtY2RrL2F3cy1jZXJ0aWZpY2F0ZW1hbmFnZXInO1xuaW1wb3J0IHsgTWV0cmljLCBNZXRyaWNPcHRpb25zLCBTdGF0aXN0aWMgfSBmcm9tICdAYXdzLWNkay9hd3MtY2xvdWR3YXRjaCc7XG5pbXBvcnQgKiBhcyBlYzIgZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBrbXMgZnJvbSAnQGF3cy1jZGsvYXdzLWttcyc7XG5pbXBvcnQgKiBhcyBsb2dzIGZyb20gJ0Bhd3MtY2RrL2F3cy1sb2dzJztcbmltcG9ydCAqIGFzIHJvdXRlNTMgZnJvbSAnQGF3cy1jZGsvYXdzLXJvdXRlNTMnO1xuaW1wb3J0ICogYXMgc2VjcmV0c21hbmFnZXIgZnJvbSAnQGF3cy1jZGsvYXdzLXNlY3JldHNtYW5hZ2VyJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuXG5pbXBvcnQgeyBMb2dHcm91cFJlc291cmNlUG9saWN5IH0gZnJvbSAnLi9sb2ctZ3JvdXAtcmVzb3VyY2UtcG9saWN5JztcbmltcG9ydCB7IE9wZW5TZWFyY2hBY2Nlc3NQb2xpY3kgfSBmcm9tICcuL29wZW5zZWFyY2gtYWNjZXNzLXBvbGljeSc7XG5pbXBvcnQgeyBDZm5Eb21haW4gfSBmcm9tICcuL29wZW5zZWFyY2hzZXJ2aWNlLmdlbmVyYXRlZCc7XG5pbXBvcnQgKiBhcyBwZXJtcyBmcm9tICcuL3Blcm1zJztcbmltcG9ydCB7IEVuZ2luZVZlcnNpb24gfSBmcm9tICcuL3ZlcnNpb24nO1xuXG4vKipcbiAqIENvbmZpZ3VyZXMgdGhlIGNhcGFjaXR5IG9mIHRoZSBjbHVzdGVyIHN1Y2ggYXMgdGhlIGluc3RhbmNlIHR5cGUgYW5kIHRoZVxuICogbnVtYmVyIG9mIGluc3RhbmNlcy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDYXBhY2l0eUNvbmZpZyB7XG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIGluc3RhbmNlcyB0byB1c2UgZm9yIHRoZSBtYXN0ZXIgbm9kZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBkZWRpY2F0ZWQgbWFzdGVyIG5vZGVzXG4gICAqL1xuICByZWFkb25seSBtYXN0ZXJOb2Rlcz86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIGhhcmR3YXJlIGNvbmZpZ3VyYXRpb24gb2YgdGhlIGNvbXB1dGVyIHRoYXQgaG9zdHMgdGhlIGRlZGljYXRlZCBtYXN0ZXJcbiAgICogbm9kZSwgc3VjaCBhcyBgbTMubWVkaXVtLnNlYXJjaGAuIEZvciB2YWxpZCB2YWx1ZXMsIHNlZSBbU3VwcG9ydGVkXG4gICAqIEluc3RhbmNlIFR5cGVzXVxuICAgKiAoaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wZW5zZWFyY2gtc2VydmljZS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc3VwcG9ydGVkLWluc3RhbmNlLXR5cGVzLmh0bWwpXG4gICAqIGluIHRoZSBBbWF6b24gT3BlblNlYXJjaCBTZXJ2aWNlIERldmVsb3BlciBHdWlkZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSByNS5sYXJnZS5zZWFyY2hcbiAgICovXG4gIHJlYWRvbmx5IG1hc3Rlck5vZGVJbnN0YW5jZVR5cGU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgZGF0YSBub2RlcyAoaW5zdGFuY2VzKSB0byB1c2UgaW4gdGhlIEFtYXpvbiBPcGVuU2VhcmNoIFNlcnZpY2UgZG9tYWluLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIDFcbiAgICovXG4gIHJlYWRvbmx5IGRhdGFOb2Rlcz86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIGluc3RhbmNlIHR5cGUgZm9yIHlvdXIgZGF0YSBub2Rlcywgc3VjaCBhc1xuICAgKiBgbTMubWVkaXVtLnNlYXJjaGAuIEZvciB2YWxpZCB2YWx1ZXMsIHNlZSBbU3VwcG9ydGVkIEluc3RhbmNlXG4gICAqIFR5cGVzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3BlbnNlYXJjaC1zZXJ2aWNlL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zdXBwb3J0ZWQtaW5zdGFuY2UtdHlwZXMuaHRtbClcbiAgICogaW4gdGhlIEFtYXpvbiBPcGVuU2VhcmNoIFNlcnZpY2UgRGV2ZWxvcGVyIEd1aWRlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHI1LmxhcmdlLnNlYXJjaFxuICAgKi9cbiAgcmVhZG9ubHkgZGF0YU5vZGVJbnN0YW5jZVR5cGU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgVWx0cmFXYXJtIG5vZGVzIChpbnN0YW5jZXMpIHRvIHVzZSBpbiB0aGUgQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gVWx0cmFXYXJtIG5vZGVzXG4gICAqL1xuICByZWFkb25seSB3YXJtTm9kZXM/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBpbnN0YW5jZSB0eXBlIGZvciB5b3VyIFVsdHJhV2FybSBub2RlLCBzdWNoIGFzIGB1bHRyYXdhcm0xLm1lZGl1bS5zZWFyY2hgLlxuICAgKiBGb3IgdmFsaWQgdmFsdWVzLCBzZWUgW1VsdHJhV2FybSBTdG9yYWdlIExpbWl0c11cbiAgICogKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcGVuc2VhcmNoLXNlcnZpY2UvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2xpbWl0cy5odG1sI2xpbWl0cy11bHRyYXdhcm0pXG4gICAqIGluIHRoZSBBbWF6b24gT3BlblNlYXJjaCBTZXJ2aWNlIERldmVsb3BlciBHdWlkZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB1bHRyYXdhcm0xLm1lZGl1bS5zZWFyY2hcbiAgICovXG4gIHJlYWRvbmx5IHdhcm1JbnN0YW5jZVR5cGU/OiBzdHJpbmc7XG5cbn1cblxuLyoqXG4gKiBTcGVjaWZpZXMgem9uZSBhd2FyZW5lc3MgY29uZmlndXJhdGlvbiBvcHRpb25zLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFpvbmVBd2FyZW5lc3NDb25maWcge1xuICAvKipcbiAgICogSW5kaWNhdGVzIHdoZXRoZXIgdG8gZW5hYmxlIHpvbmUgYXdhcmVuZXNzIGZvciB0aGUgQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4uXG4gICAqIFdoZW4geW91IGVuYWJsZSB6b25lIGF3YXJlbmVzcywgQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSBhbGxvY2F0ZXMgdGhlIG5vZGVzIGFuZCByZXBsaWNhXG4gICAqIGluZGV4IHNoYXJkcyB0aGF0IGJlbG9uZyB0byBhIGNsdXN0ZXIgYWNyb3NzIHR3byBBdmFpbGFiaWxpdHkgWm9uZXMgKEFacylcbiAgICogaW4gdGhlIHNhbWUgcmVnaW9uIHRvIHByZXZlbnQgZGF0YSBsb3NzIGFuZCBtaW5pbWl6ZSBkb3dudGltZSBpbiB0aGUgZXZlbnRcbiAgICogb2Ygbm9kZSBvciBkYXRhIGNlbnRlciBmYWlsdXJlLiBEb24ndCBlbmFibGUgem9uZSBhd2FyZW5lc3MgaWYgeW91ciBjbHVzdGVyXG4gICAqIGhhcyBubyByZXBsaWNhIGluZGV4IHNoYXJkcyBvciBpcyBhIHNpbmdsZS1ub2RlIGNsdXN0ZXIuIEZvciBtb3JlIGluZm9ybWF0aW9uLFxuICAgKiBzZWUgW0NvbmZpZ3VyaW5nIGEgTXVsdGktQVogRG9tYWluXVxuICAgKiAoaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wZW5zZWFyY2gtc2VydmljZS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvbWFuYWdlZG9tYWlucy1tdWx0aWF6Lmh0bWwpXG4gICAqIGluIHRoZSBBbWF6b24gT3BlblNlYXJjaCBTZXJ2aWNlIERldmVsb3BlciBHdWlkZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgZW5hYmxlZD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIElmIHlvdSBlbmFibGVkIG11bHRpcGxlIEF2YWlsYWJpbGl0eSBab25lcyAoQVpzKSwgdGhlIG51bWJlciBvZiBBWnMgdGhhdCB5b3VcbiAgICogd2FudCB0aGUgZG9tYWluIHRvIHVzZS4gVmFsaWQgdmFsdWVzIGFyZSAyIGFuZCAzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIDIgaWYgem9uZSBhd2FyZW5lc3MgaXMgZW5hYmxlZC5cbiAgICovXG4gIHJlYWRvbmx5IGF2YWlsYWJpbGl0eVpvbmVDb3VudD86IG51bWJlcjtcbn1cblxuLyoqXG4gKiBUaGUgY29uZmlndXJhdGlvbnMgb2YgQW1hem9uIEVsYXN0aWMgQmxvY2sgU3RvcmUgKEFtYXpvbiBFQlMpIHZvbHVtZXMgdGhhdFxuICogYXJlIGF0dGFjaGVkIHRvIGRhdGEgbm9kZXMgaW4gdGhlIEFtYXpvbiBPcGVuU2VhcmNoIFNlcnZpY2UgZG9tYWluLiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlXG4gKiBbQW1hem9uIEVCU11cbiAqIChodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTRUMyL2xhdGVzdC9Vc2VyR3VpZGUvQW1hem9uRUJTLmh0bWwpXG4gKiBpbiB0aGUgQW1hem9uIEVsYXN0aWMgQ29tcHV0ZSBDbG91ZCBEZXZlbG9wZXIgR3VpZGUuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRWJzT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgd2hldGhlciBBbWF6b24gRUJTIHZvbHVtZXMgYXJlIGF0dGFjaGVkIHRvIGRhdGEgbm9kZXMgaW4gdGhlXG4gICAqIEFtYXpvbiBPcGVuU2VhcmNoIFNlcnZpY2UgZG9tYWluLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGVuYWJsZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIEkvTyBvcGVyYXRpb25zIHBlciBzZWNvbmQgKElPUFMpIHRoYXQgdGhlIHZvbHVtZVxuICAgKiBzdXBwb3J0cy4gVGhpcyBwcm9wZXJ0eSBhcHBsaWVzIG9ubHkgdG8gdGhlIFByb3Zpc2lvbmVkIElPUFMgKFNTRCkgRUJTXG4gICAqIHZvbHVtZSB0eXBlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGlvcHMgYXJlIG5vdCBzZXQuXG4gICAqL1xuICByZWFkb25seSBpb3BzPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgc2l6ZSAoaW4gR2lCKSBvZiB0aGUgRUJTIHZvbHVtZSBmb3IgZWFjaCBkYXRhIG5vZGUuIFRoZSBtaW5pbXVtIGFuZFxuICAgKiBtYXhpbXVtIHNpemUgb2YgYW4gRUJTIHZvbHVtZSBkZXBlbmRzIG9uIHRoZSBFQlMgdm9sdW1lIHR5cGUgYW5kIHRoZVxuICAgKiBpbnN0YW5jZSB0eXBlIHRvIHdoaWNoIGl0IGlzIGF0dGFjaGVkLiAgRm9yICB2YWxpZCB2YWx1ZXMsIHNlZVxuICAgKiBbRUJTIHZvbHVtZSBzaXplIGxpbWl0c11cbiAgICogKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcGVuc2VhcmNoLXNlcnZpY2UvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2xpbWl0cy5odG1sI2Vic3Jlc291cmNlKVxuICAgKiBpbiB0aGUgQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSBEZXZlbG9wZXIgR3VpZGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IDEwXG4gICAqL1xuICByZWFkb25seSB2b2x1bWVTaXplPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgRUJTIHZvbHVtZSB0eXBlIHRvIHVzZSB3aXRoIHRoZSBBbWF6b24gT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbiwgc3VjaCBhcyBzdGFuZGFyZCwgZ3AyLCBpbzEuXG4gICAqXG4gICAqIEBkZWZhdWx0IGdwMlxuICAgKi9cbiAgcmVhZG9ubHkgdm9sdW1lVHlwZT86IGVjMi5FYnNEZXZpY2VWb2x1bWVUeXBlO1xufVxuXG4vKipcbiAqIENvbmZpZ3VyZXMgbG9nIHNldHRpbmdzIGZvciB0aGUgZG9tYWluLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIExvZ2dpbmdPcHRpb25zIHtcbiAgLyoqXG4gICAqIFNwZWNpZnkgaWYgc2xvdyBzZWFyY2ggbG9nZ2luZyBzaG91bGQgYmUgc2V0IHVwLlxuICAgKiBSZXF1aXJlcyBFbGFzdGljc2VhcmNoIHZlcnNpb24gNS4xIG9yIGxhdGVyIG9yIE9wZW5TZWFyY2ggdmVyc2lvbiAxLjAgb3IgbGF0ZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHNsb3dTZWFyY2hMb2dFbmFibGVkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogTG9nIHNsb3cgc2VhcmNoZXMgdG8gdGhpcyBsb2cgZ3JvdXAuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gYSBuZXcgbG9nIGdyb3VwIGlzIGNyZWF0ZWQgaWYgc2xvdyBzZWFyY2ggbG9nZ2luZyBpcyBlbmFibGVkXG4gICAqL1xuICByZWFkb25seSBzbG93U2VhcmNoTG9nR3JvdXA/OiBsb2dzLklMb2dHcm91cDtcblxuICAvKipcbiAgICogU3BlY2lmeSBpZiBzbG93IGluZGV4IGxvZ2dpbmcgc2hvdWxkIGJlIHNldCB1cC5cbiAgICogUmVxdWlyZXMgRWxhc3RpY3NlYXJjaCB2ZXJzaW9uIDUuMSBvciBsYXRlciBvciBPcGVuU2VhcmNoIHZlcnNpb24gMS4wIG9yIGxhdGVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGZhbHNlXG4gICAqL1xuICByZWFkb25seSBzbG93SW5kZXhMb2dFbmFibGVkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogTG9nIHNsb3cgaW5kaWNlcyB0byB0aGlzIGxvZyBncm91cC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBhIG5ldyBsb2cgZ3JvdXAgaXMgY3JlYXRlZCBpZiBzbG93IGluZGV4IGxvZ2dpbmcgaXMgZW5hYmxlZFxuICAgKi9cbiAgcmVhZG9ubHkgc2xvd0luZGV4TG9nR3JvdXA/OiBsb2dzLklMb2dHcm91cDtcblxuICAvKipcbiAgICogU3BlY2lmeSBpZiBBbWF6b24gT3BlblNlYXJjaCBTZXJ2aWNlIGFwcGxpY2F0aW9uIGxvZ2dpbmcgc2hvdWxkIGJlIHNldCB1cC5cbiAgICogUmVxdWlyZXMgRWxhc3RpY3NlYXJjaCB2ZXJzaW9uIDUuMSBvciBsYXRlciBvciBPcGVuU2VhcmNoIHZlcnNpb24gMS4wIG9yIGxhdGVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGZhbHNlXG4gICAqL1xuICByZWFkb25seSBhcHBMb2dFbmFibGVkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogTG9nIEFtYXpvbiBPcGVuU2VhcmNoIFNlcnZpY2UgYXBwbGljYXRpb24gbG9ncyB0byB0aGlzIGxvZyBncm91cC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBhIG5ldyBsb2cgZ3JvdXAgaXMgY3JlYXRlZCBpZiBhcHAgbG9nZ2luZyBpcyBlbmFibGVkXG4gICAqL1xuICByZWFkb25seSBhcHBMb2dHcm91cD86IGxvZ3MuSUxvZ0dyb3VwO1xuXG4gIC8qKlxuICAgKiBTcGVjaWZ5IGlmIEFtYXpvbiBPcGVuU2VhcmNoIFNlcnZpY2UgYXVkaXQgbG9nZ2luZyBzaG91bGQgYmUgc2V0IHVwLlxuICAgKiBSZXF1aXJlcyBFbGFzdGljc2VhcmNoIHZlcnNpb24gNi43IG9yIGxhdGVyIG9yIE9wZW5TZWFyY2ggdmVyc2lvbiAxLjAgb3IgbGF0ZXIgYW5kIGZpbmUgZ3JhaW5lZCBhY2Nlc3MgY29udHJvbCB0byBiZSBlbmFibGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGZhbHNlXG4gICAqL1xuICByZWFkb25seSBhdWRpdExvZ0VuYWJsZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBMb2cgQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSBhdWRpdCBsb2dzIHRvIHRoaXMgbG9nIGdyb3VwLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGEgbmV3IGxvZyBncm91cCBpcyBjcmVhdGVkIGlmIGF1ZGl0IGxvZ2dpbmcgaXMgZW5hYmxlZFxuICAgKi9cbiAgcmVhZG9ubHkgYXVkaXRMb2dHcm91cD86IGxvZ3MuSUxvZ0dyb3VwO1xufVxuXG4vKipcbiAqIFdoZXRoZXIgdGhlIGRvbWFpbiBzaG91bGQgZW5jcnlwdCBkYXRhIGF0IHJlc3QsIGFuZCBpZiBzbywgdGhlIEFXUyBLZXlcbiAqIE1hbmFnZW1lbnQgU2VydmljZSAoS01TKSBrZXkgdG8gdXNlLiBDYW4gb25seSBiZSB1c2VkIHRvIGNyZWF0ZSBhIG5ldyBkb21haW4sXG4gKiBub3QgdXBkYXRlIGFuIGV4aXN0aW5nIG9uZS4gUmVxdWlyZXMgRWxhc3RpY3NlYXJjaCB2ZXJzaW9uIDUuMSBvciBsYXRlciBvciBPcGVuU2VhcmNoIHZlcnNpb24gMS4wIG9yIGxhdGVyLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEVuY3J5cHRpb25BdFJlc3RPcHRpb25zIHtcbiAgLyoqXG4gICAqIFNwZWNpZnkgdHJ1ZSB0byBlbmFibGUgZW5jcnlwdGlvbiBhdCByZXN0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGVuY3J5cHRpb24gYXQgcmVzdCBpcyBkaXNhYmxlZC5cbiAgICovXG4gIHJlYWRvbmx5IGVuYWJsZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBTdXBwbHkgaWYgdXNpbmcgS01TIGtleSBmb3IgZW5jcnlwdGlvbiBhdCByZXN0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHVzZXMgZGVmYXVsdCBhd3MvZXMgS01TIGtleS5cbiAgICovXG4gIHJlYWRvbmx5IGttc0tleT86IGttcy5JS2V5O1xufVxuXG4vKipcbiAqIENvbmZpZ3VyZXMgQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSB0byB1c2UgQW1hem9uIENvZ25pdG8gYXV0aGVudGljYXRpb24gZm9yIE9wZW5TZWFyY2ggRGFzaGJvYXJkcy5cbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wZW5zZWFyY2gtc2VydmljZS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvY29nbml0by1hdXRoLmh0bWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDb2duaXRvT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgQW1hem9uIENvZ25pdG8gaWRlbnRpdHkgcG9vbCBJRCB0aGF0IHlvdSB3YW50IEFtYXpvbiBPcGVuU2VhcmNoIFNlcnZpY2UgdG8gdXNlIGZvciBPcGVuU2VhcmNoIERhc2hib2FyZHMgYXV0aGVudGljYXRpb24uXG4gICAqL1xuICByZWFkb25seSBpZGVudGl0eVBvb2xJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBIHJvbGUgdGhhdCBhbGxvd3MgQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSB0byBjb25maWd1cmUgeW91ciB1c2VyIHBvb2wgYW5kIGlkZW50aXR5IHBvb2wuIEl0IG11c3QgaGF2ZSB0aGUgYEFtYXpvbkVTQ29nbml0b0FjY2Vzc2AgcG9saWN5IGF0dGFjaGVkIHRvIGl0LlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcGVuc2VhcmNoLXNlcnZpY2UvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2NvZ25pdG8tYXV0aC5odG1sI2NvZ25pdG8tYXV0aC1wcmVyZXFcbiAgICovXG4gIHJlYWRvbmx5IHJvbGU6IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogVGhlIEFtYXpvbiBDb2duaXRvIHVzZXIgcG9vbCBJRCB0aGF0IHlvdSB3YW50IEFtYXpvbiBPcGVuU2VhcmNoIFNlcnZpY2UgdG8gdXNlIGZvciBPcGVuU2VhcmNoIERhc2hib2FyZHMgYXV0aGVudGljYXRpb24uXG4gICAqL1xuICByZWFkb25seSB1c2VyUG9vbElkOiBzdHJpbmc7XG59XG5cbi8qKlxuICogVGhlIG1pbmltdW0gVExTIHZlcnNpb24gcmVxdWlyZWQgZm9yIHRyYWZmaWMgdG8gdGhlIGRvbWFpbi5cbiAqL1xuZXhwb3J0IGVudW0gVExTU2VjdXJpdHlQb2xpY3kge1xuICAvKiogQ2lwaGVyIHN1aXRlIFRMUyAxLjAgKi9cbiAgVExTXzFfMCA9ICdQb2xpY3ktTWluLVRMUy0xLTAtMjAxOS0wNycsXG4gIC8qKiBDaXBoZXIgc3VpdGUgVExTIDEuMiAqL1xuICBUTFNfMV8yID0gJ1BvbGljeS1NaW4tVExTLTEtMi0yMDE5LTA3J1xufVxuXG4vKipcbiAqIFNwZWNpZmllcyBvcHRpb25zIGZvciBmaW5lLWdyYWluZWQgYWNjZXNzIGNvbnRyb2wuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQWR2YW5jZWRTZWN1cml0eU9wdGlvbnMge1xuICAvKipcbiAgICogQVJOIGZvciB0aGUgbWFzdGVyIHVzZXIuIE9ubHkgc3BlY2lmeSB0aGlzIG9yIG1hc3RlclVzZXJOYW1lLCBidXQgbm90IGJvdGguXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZmluZS1ncmFpbmVkIGFjY2VzcyBjb250cm9sIGlzIGRpc2FibGVkXG4gICAqL1xuICByZWFkb25seSBtYXN0ZXJVc2VyQXJuPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBVc2VybmFtZSBmb3IgdGhlIG1hc3RlciB1c2VyLiBPbmx5IHNwZWNpZnkgdGhpcyBvciBtYXN0ZXJVc2VyQXJuLCBidXQgbm90IGJvdGguXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZmluZS1ncmFpbmVkIGFjY2VzcyBjb250cm9sIGlzIGRpc2FibGVkXG4gICAqL1xuICByZWFkb25seSBtYXN0ZXJVc2VyTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogUGFzc3dvcmQgZm9yIHRoZSBtYXN0ZXIgdXNlci5cbiAgICpcbiAgICogWW91IGNhbiB1c2UgYFNlY3JldFZhbHVlLnVuc2FmZVBsYWluVGV4dGAgdG8gc3BlY2lmeSBhIHBhc3N3b3JkIGluIHBsYWluIHRleHQgb3JcbiAgICogdXNlIGBzZWNyZXRzbWFuYWdlci5TZWNyZXQuZnJvbVNlY3JldEF0dHJpYnV0ZXNgIHRvIHJlZmVyZW5jZSBhIHNlY3JldCBpblxuICAgKiBTZWNyZXRzIE1hbmFnZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQSBTZWNyZXRzIE1hbmFnZXIgZ2VuZXJhdGVkIHBhc3N3b3JkXG4gICAqL1xuICByZWFkb25seSBtYXN0ZXJVc2VyUGFzc3dvcmQ/OiBjZGsuU2VjcmV0VmFsdWU7XG59XG5cbi8qKlxuICogQ29uZmlndXJlcyBhIGN1c3RvbSBkb21haW4gZW5kcG9pbnQgZm9yIHRoZSBBbWF6b24gT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpblxuICovXG5leHBvcnQgaW50ZXJmYWNlIEN1c3RvbUVuZHBvaW50T3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgY3VzdG9tIGRvbWFpbiBuYW1lIHRvIGFzc2lnblxuICAgKi9cbiAgcmVhZG9ubHkgZG9tYWluTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgY2VydGlmaWNhdGUgdG8gdXNlXG4gICAqIEBkZWZhdWx0IC0gY3JlYXRlIGEgbmV3IG9uZVxuICAgKi9cbiAgcmVhZG9ubHkgY2VydGlmaWNhdGU/OiBhY20uSUNlcnRpZmljYXRlO1xuXG4gIC8qKlxuICAgKiBUaGUgaG9zdGVkIHpvbmUgaW4gUm91dGU1MyB0byBjcmVhdGUgdGhlIENOQU1FIHJlY29yZCBpblxuICAgKiBAZGVmYXVsdCAtIGRvIG5vdCBjcmVhdGUgYSBDTkFNRVxuICAgKi9cbiAgcmVhZG9ubHkgaG9zdGVkWm9uZT86IHJvdXRlNTMuSUhvc3RlZFpvbmU7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYW4gQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRG9tYWluUHJvcHMge1xuICAvKipcbiAgICogRG9tYWluIGFjY2VzcyBwb2xpY2llcy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBhY2Nlc3MgcG9saWNpZXMuXG4gICAqL1xuICByZWFkb25seSBhY2Nlc3NQb2xpY2llcz86IGlhbS5Qb2xpY3lTdGF0ZW1lbnRbXTtcblxuICAvKipcbiAgICogQWRkaXRpb25hbCBvcHRpb25zIHRvIHNwZWNpZnkgZm9yIHRoZSBBbWF6b24gT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbi5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3BlbnNlYXJjaC1zZXJ2aWNlL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9jcmVhdGV1cGRhdGVkb21haW5zLmh0bWwjY3JlYXRlZG9tYWluLWNvbmZpZ3VyZS1hZHZhbmNlZC1vcHRpb25zXG4gICAqIEBkZWZhdWx0IC0gbm8gYWR2YW5jZWQgb3B0aW9ucyBhcmUgc3BlY2lmaWVkXG4gICAqL1xuICByZWFkb25seSBhZHZhbmNlZE9wdGlvbnM/OiB7IFtrZXk6IHN0cmluZ106IChzdHJpbmcpIH07XG5cbiAgLyoqXG4gICAqIENvbmZpZ3VyZXMgQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSB0byB1c2UgQW1hem9uIENvZ25pdG8gYXV0aGVudGljYXRpb24gZm9yIE9wZW5TZWFyY2ggRGFzaGJvYXJkcy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBDb2duaXRvIG5vdCB1c2VkIGZvciBhdXRoZW50aWNhdGlvbiB0byBPcGVuU2VhcmNoIERhc2hib2FyZHMuXG4gICAqL1xuICByZWFkb25seSBjb2duaXRvRGFzaGJvYXJkc0F1dGg/OiBDb2duaXRvT3B0aW9ucztcblxuICAvKipcbiAgICogRW5mb3JjZXMgYSBwYXJ0aWN1bGFyIHBoeXNpY2FsIGRvbWFpbiBuYW1lLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEEgbmFtZSB3aWxsIGJlIGF1dG8tZ2VuZXJhdGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgZG9tYWluTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGNvbmZpZ3VyYXRpb25zIG9mIEFtYXpvbiBFbGFzdGljIEJsb2NrIFN0b3JlIChBbWF6b24gRUJTKSB2b2x1bWVzIHRoYXRcbiAgICogYXJlIGF0dGFjaGVkIHRvIGRhdGEgbm9kZXMgaW4gdGhlIEFtYXpvbiBPcGVuU2VhcmNoIFNlcnZpY2UgZG9tYWluLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIDEwIEdpQiBHZW5lcmFsIFB1cnBvc2UgKFNTRCkgdm9sdW1lcyBwZXIgbm9kZS5cbiAgICovXG4gIHJlYWRvbmx5IGVicz86IEVic09wdGlvbnM7XG5cbiAgLyoqXG4gICAqIFRoZSBjbHVzdGVyIGNhcGFjaXR5IGNvbmZpZ3VyYXRpb24gZm9yIHRoZSBBbWF6b24gT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSAxIHI1LmxhcmdlLnNlYXJjaCBkYXRhIG5vZGU7IG5vIGRlZGljYXRlZCBtYXN0ZXIgbm9kZXMuXG4gICAqL1xuICByZWFkb25seSBjYXBhY2l0eT86IENhcGFjaXR5Q29uZmlnO1xuXG4gIC8qKlxuICAgKiBUaGUgY2x1c3RlciB6b25lIGF3YXJlbmVzcyBjb25maWd1cmF0aW9uIGZvciB0aGUgQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gem9uZSBhd2FyZW5lc3MgKDEgQVopXG4gICAqL1xuICByZWFkb25seSB6b25lQXdhcmVuZXNzPzogWm9uZUF3YXJlbmVzc0NvbmZpZztcblxuICAvKipcbiAgICogVGhlIEVsYXN0aWNzZWFyY2gvT3BlblNlYXJjaCB2ZXJzaW9uIHRoYXQgeW91ciBkb21haW4gd2lsbCBsZXZlcmFnZS5cbiAgICovXG4gIHJlYWRvbmx5IHZlcnNpb246IEVuZ2luZVZlcnNpb247XG5cbiAgLyoqXG4gICAqIEVuY3J5cHRpb24gYXQgcmVzdCBvcHRpb25zIGZvciB0aGUgY2x1c3Rlci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBlbmNyeXB0aW9uIGF0IHJlc3RcbiAgICovXG4gIHJlYWRvbmx5IGVuY3J5cHRpb25BdFJlc3Q/OiBFbmNyeXB0aW9uQXRSZXN0T3B0aW9ucztcblxuICAvKipcbiAgICogQ29uZmlndXJhdGlvbiBsb2cgcHVibGlzaGluZyBjb25maWd1cmF0aW9uIG9wdGlvbnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gbG9ncyBhcmUgcHVibGlzaGVkXG4gICAqL1xuICByZWFkb25seSBsb2dnaW5nPzogTG9nZ2luZ09wdGlvbnM7XG5cbiAgLyoqXG4gICAqIFNwZWNpZnkgdHJ1ZSB0byBlbmFibGUgbm9kZSB0byBub2RlIGVuY3J5cHRpb24uXG4gICAqIFJlcXVpcmVzIEVsYXN0aWNzZWFyY2ggdmVyc2lvbiA2LjAgb3IgbGF0ZXIgb3IgT3BlblNlYXJjaCB2ZXJzaW9uIDEuMCBvciBsYXRlci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb2RlIHRvIG5vZGUgZW5jcnlwdGlvbiBpcyBub3QgZW5hYmxlZC5cbiAgICovXG4gIHJlYWRvbmx5IG5vZGVUb05vZGVFbmNyeXB0aW9uPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIGhvdXIgaW4gVVRDIGR1cmluZyB3aGljaCB0aGUgc2VydmljZSB0YWtlcyBhbiBhdXRvbWF0ZWQgZGFpbHkgc25hcHNob3RcbiAgICogb2YgdGhlIGluZGljZXMgaW4gdGhlIEFtYXpvbiBPcGVuU2VhcmNoIFNlcnZpY2UgZG9tYWluLiBPbmx5IGFwcGxpZXMgZm9yIEVsYXN0aWNzZWFyY2ggdmVyc2lvbnNcbiAgICogYmVsb3cgNS4zLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEhvdXJseSBhdXRvbWF0ZWQgc25hcHNob3RzIG5vdCB1c2VkXG4gICAqL1xuICByZWFkb25seSBhdXRvbWF0ZWRTbmFwc2hvdFN0YXJ0SG91cj86IG51bWJlcjtcblxuICAvKipcbiAgICogUGxhY2UgdGhlIGRvbWFpbiBpbnNpZGUgdGhpcyBWUEMuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wZW5zZWFyY2gtc2VydmljZS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdnBjLmh0bWxcbiAgICogQGRlZmF1bHQgLSBEb21haW4gaXMgbm90IHBsYWNlZCBpbiBhIFZQQy5cbiAgICovXG4gIHJlYWRvbmx5IHZwYz86IGVjMi5JVnBjO1xuXG4gIC8qKlxuICAgKiBUaGUgbGlzdCBvZiBzZWN1cml0eSBncm91cHMgdGhhdCBhcmUgYXNzb2NpYXRlZCB3aXRoIHRoZSBWUEMgZW5kcG9pbnRzXG4gICAqIGZvciB0aGUgZG9tYWluLlxuICAgKlxuICAgKiBPbmx5IHVzZWQgaWYgYHZwY2AgaXMgc3BlY2lmaWVkLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS92cGMvbGF0ZXN0L3VzZXJndWlkZS9WUENfU2VjdXJpdHlHcm91cHMuaHRtbFxuICAgKiBAZGVmYXVsdCAtIE9uZSBuZXcgc2VjdXJpdHkgZ3JvdXAgaXMgY3JlYXRlZC5cbiAgICovXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXBzPzogZWMyLklTZWN1cml0eUdyb3VwW107XG5cbiAgLyoqXG4gICAqIFRoZSBzcGVjaWZpYyB2cGMgc3VibmV0cyB0aGUgZG9tYWluIHdpbGwgYmUgcGxhY2VkIGluLiBZb3UgbXVzdCBwcm92aWRlIG9uZSBzdWJuZXQgZm9yIGVhY2ggQXZhaWxhYmlsaXR5IFpvbmVcbiAgICogdGhhdCB5b3VyIGRvbWFpbiB1c2VzLiBGb3IgZXhhbXBsZSwgeW91IG11c3Qgc3BlY2lmeSB0aHJlZSBzdWJuZXQgSURzIGZvciBhIHRocmVlIEF2YWlsYWJpbGl0eSBab25lXG4gICAqIGRvbWFpbi5cbiAgICpcbiAgICogT25seSB1c2VkIGlmIGB2cGNgIGlzIHNwZWNpZmllZC5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdnBjL2xhdGVzdC91c2VyZ3VpZGUvVlBDX1N1Ym5ldHMuaHRtbFxuICAgKiBAZGVmYXVsdCAtIEFsbCBwcml2YXRlIHN1Ym5ldHMuXG4gICAqL1xuICByZWFkb25seSB2cGNTdWJuZXRzPzogZWMyLlN1Ym5ldFNlbGVjdGlvbltdO1xuXG4gIC8qKlxuICAgKiBUcnVlIHRvIHJlcXVpcmUgdGhhdCBhbGwgdHJhZmZpYyB0byB0aGUgZG9tYWluIGFycml2ZSBvdmVyIEhUVFBTLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGZhbHNlXG4gICAqL1xuICByZWFkb25seSBlbmZvcmNlSHR0cHM/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgbWluaW11bSBUTFMgdmVyc2lvbiByZXF1aXJlZCBmb3IgdHJhZmZpYyB0byB0aGUgZG9tYWluLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFRMU1NlY3VyaXR5UG9saWN5LlRMU18xXzBcbiAgICovXG4gIHJlYWRvbmx5IHRsc1NlY3VyaXR5UG9saWN5PzogVExTU2VjdXJpdHlQb2xpY3k7XG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyBvcHRpb25zIGZvciBmaW5lLWdyYWluZWQgYWNjZXNzIGNvbnRyb2wuXG4gICAqIFJlcXVpcmVzIEVsYXN0aWNzZWFyY2ggdmVyc2lvbiA2Ljcgb3IgbGF0ZXIgb3IgT3BlblNlYXJjaCB2ZXJzaW9uIDEuMCBvciBsYXRlci4gRW5hYmxpbmcgZmluZS1ncmFpbmVkIGFjY2VzcyBjb250cm9sXG4gICAqIGFsc28gcmVxdWlyZXMgZW5jcnlwdGlvbiBvZiBkYXRhIGF0IHJlc3QgYW5kIG5vZGUtdG8tbm9kZSBlbmNyeXB0aW9uLCBhbG9uZyB3aXRoXG4gICAqIGVuZm9yY2VkIEhUVFBTLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGZpbmUtZ3JhaW5lZCBhY2Nlc3MgY29udHJvbCBpcyBkaXNhYmxlZFxuICAgKi9cbiAgcmVhZG9ubHkgZmluZUdyYWluZWRBY2Nlc3NDb250cm9sPzogQWR2YW5jZWRTZWN1cml0eU9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIENvbmZpZ3VyZXMgdGhlIGRvbWFpbiBzbyB0aGF0IHVuc2lnbmVkIGJhc2ljIGF1dGggaXMgZW5hYmxlZC4gSWYgbm8gbWFzdGVyIHVzZXIgaXMgcHJvdmlkZWQgYSBkZWZhdWx0IG1hc3RlciB1c2VyXG4gICAqIHdpdGggdXNlcm5hbWUgYGFkbWluYCBhbmQgYSBkeW5hbWljYWxseSBnZW5lcmF0ZWQgcGFzc3dvcmQgc3RvcmVkIGluIEtNUyBpcyBjcmVhdGVkLiBUaGUgcGFzc3dvcmQgY2FuIGJlIHJldHJpZXZlZFxuICAgKiBieSBnZXR0aW5nIGBtYXN0ZXJVc2VyUGFzc3dvcmRgIGZyb20gdGhlIGRvbWFpbiBpbnN0YW5jZS5cbiAgICpcbiAgICogU2V0dGluZyB0aGlzIHRvIHRydWUgd2lsbCBhbHNvIGFkZCBhbiBhY2Nlc3MgcG9saWN5IHRoYXQgYWxsb3dzIHVuc2lnbmVkXG4gICAqIGFjY2VzcywgZW5hYmxlIG5vZGUgdG8gbm9kZSBlbmNyeXB0aW9uLCBlbmNyeXB0aW9uIGF0IHJlc3QuIElmIGNvbmZsaWN0aW5nXG4gICAqIHNldHRpbmdzIGFyZSBlbmNvdW50ZXJlZCAobGlrZSBkaXNhYmxpbmcgZW5jcnlwdGlvbiBhdCByZXN0KSBlbmFibGluZyB0aGlzXG4gICAqIHNldHRpbmcgd2lsbCBjYXVzZSBhIGZhaWx1cmUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHVzZVVuc2lnbmVkQmFzaWNBdXRoPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVG8gdXBncmFkZSBhbiBBbWF6b24gT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbiB0byBhIG5ldyB2ZXJzaW9uLCByYXRoZXIgdGhhbiByZXBsYWNpbmcgdGhlIGVudGlyZVxuICAgKiBkb21haW4gcmVzb3VyY2UsIHVzZSB0aGUgRW5hYmxlVmVyc2lvblVwZ3JhZGUgdXBkYXRlIHBvbGljeS5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS9hd3MtYXR0cmlidXRlLXVwZGF0ZXBvbGljeS5odG1sI2Nmbi1hdHRyaWJ1dGVzLXVwZGF0ZXBvbGljeS11cGdyYWRlb3BlbnNlYXJjaGRvbWFpblxuICAgKiBAZGVmYXVsdCAtIGZhbHNlXG4gICAqL1xuICByZWFkb25seSBlbmFibGVWZXJzaW9uVXBncmFkZT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFBvbGljeSB0byBhcHBseSB3aGVuIHRoZSBkb21haW4gaXMgcmVtb3ZlZCBmcm9tIHRoZSBzdGFja1xuICAgKlxuICAgKiBAZGVmYXVsdCBSZW1vdmFsUG9saWN5LlJFVEFJTlxuICAgKi9cbiAgcmVhZG9ubHkgcmVtb3ZhbFBvbGljeT86IGNkay5SZW1vdmFsUG9saWN5O1xuXG4gIC8qKlxuICAgKiBUbyBjb25maWd1cmUgYSBjdXN0b20gZG9tYWluIGNvbmZpZ3VyZSB0aGVzZSBvcHRpb25zXG4gICAqXG4gICAqIElmIHlvdSBzcGVjaWZ5IGEgUm91dGU1MyBob3N0ZWQgem9uZSBpdCB3aWxsIGNyZWF0ZSBhIENOQU1FIHJlY29yZCBhbmQgdXNlIEROUyB2YWxpZGF0aW9uIGZvciB0aGUgY2VydGlmaWNhdGVcbiAgICogQGRlZmF1bHQgLSBubyBjdXN0b20gZG9tYWluIGVuZHBvaW50IHdpbGwgYmUgY29uZmlndXJlZFxuICAgKi9cbiAgcmVhZG9ubHkgY3VzdG9tRW5kcG9pbnQ/OiBDdXN0b21FbmRwb2ludE9wdGlvbnM7XG59XG5cbi8qKlxuICogQW4gaW50ZXJmYWNlIHRoYXQgcmVwcmVzZW50cyBhbiBBbWF6b24gT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbiAtIGVpdGhlciBjcmVhdGVkIHdpdGggdGhlIENESywgb3IgYW4gZXhpc3Rpbmcgb25lLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElEb21haW4gZXh0ZW5kcyBjZGsuSVJlc291cmNlIHtcbiAgLyoqXG4gICAqIEFybiBvZiB0aGUgQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4uXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGRvbWFpbkFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBEb21haW4gbmFtZSBvZiB0aGUgQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4uXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGRvbWFpbk5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogSWRlbnRpZmllciBvZiB0aGUgQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4uXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGRvbWFpbklkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEVuZHBvaW50IG9mIHRoZSBBbWF6b24gT3BlblNlYXJjaCBTZXJ2aWNlIGRvbWFpbi5cbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgZG9tYWluRW5kcG9pbnQ6IHN0cmluZztcblxuICAvKipcbiAgICogR3JhbnQgcmVhZCBwZXJtaXNzaW9ucyBmb3IgdGhpcyBkb21haW4gYW5kIGl0cyBjb250ZW50cyB0byBhbiBJQU1cbiAgICogcHJpbmNpcGFsIChSb2xlL0dyb3VwL1VzZXIpLlxuICAgKlxuICAgKiBAcGFyYW0gaWRlbnRpdHkgVGhlIHByaW5jaXBhbFxuICAgKi9cbiAgZ3JhbnRSZWFkKGlkZW50aXR5OiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudDtcblxuICAvKipcbiAgICogR3JhbnQgd3JpdGUgcGVybWlzc2lvbnMgZm9yIHRoaXMgZG9tYWluIGFuZCBpdHMgY29udGVudHMgdG8gYW4gSUFNXG4gICAqIHByaW5jaXBhbCAoUm9sZS9Hcm91cC9Vc2VyKS5cbiAgICpcbiAgICogQHBhcmFtIGlkZW50aXR5IFRoZSBwcmluY2lwYWxcbiAgICovXG4gIGdyYW50V3JpdGUoaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50O1xuXG4gIC8qKlxuICAgKiBHcmFudCByZWFkL3dyaXRlIHBlcm1pc3Npb25zIGZvciB0aGlzIGRvbWFpbiBhbmQgaXRzIGNvbnRlbnRzIHRvIGFuIElBTVxuICAgKiBwcmluY2lwYWwgKFJvbGUvR3JvdXAvVXNlcikuXG4gICAqXG4gICAqIEBwYXJhbSBpZGVudGl0eSBUaGUgcHJpbmNpcGFsXG4gICAqL1xuICBncmFudFJlYWRXcml0ZShpZGVudGl0eTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQ7XG5cbiAgLyoqXG4gICAqIEdyYW50IHJlYWQgcGVybWlzc2lvbnMgZm9yIGFuIGluZGV4IGluIHRoaXMgZG9tYWluIHRvIGFuIElBTVxuICAgKiBwcmluY2lwYWwgKFJvbGUvR3JvdXAvVXNlcikuXG4gICAqXG4gICAqIEBwYXJhbSBpbmRleCBUaGUgaW5kZXggdG8gZ3JhbnQgcGVybWlzc2lvbnMgZm9yXG4gICAqIEBwYXJhbSBpZGVudGl0eSBUaGUgcHJpbmNpcGFsXG4gICAqL1xuICBncmFudEluZGV4UmVhZChpbmRleDogc3RyaW5nLCBpZGVudGl0eTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQ7XG5cbiAgLyoqXG4gICAqIEdyYW50IHdyaXRlIHBlcm1pc3Npb25zIGZvciBhbiBpbmRleCBpbiB0aGlzIGRvbWFpbiB0byBhbiBJQU1cbiAgICogcHJpbmNpcGFsIChSb2xlL0dyb3VwL1VzZXIpLlxuICAgKlxuICAgKiBAcGFyYW0gaW5kZXggVGhlIGluZGV4IHRvIGdyYW50IHBlcm1pc3Npb25zIGZvclxuICAgKiBAcGFyYW0gaWRlbnRpdHkgVGhlIHByaW5jaXBhbFxuICAgKi9cbiAgZ3JhbnRJbmRleFdyaXRlKGluZGV4OiBzdHJpbmcsIGlkZW50aXR5OiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudDtcblxuICAvKipcbiAgICogR3JhbnQgcmVhZC93cml0ZSBwZXJtaXNzaW9ucyBmb3IgYW4gaW5kZXggaW4gdGhpcyBkb21haW4gdG8gYW4gSUFNXG4gICAqIHByaW5jaXBhbCAoUm9sZS9Hcm91cC9Vc2VyKS5cbiAgICpcbiAgICogQHBhcmFtIGluZGV4IFRoZSBpbmRleCB0byBncmFudCBwZXJtaXNzaW9ucyBmb3JcbiAgICogQHBhcmFtIGlkZW50aXR5IFRoZSBwcmluY2lwYWxcbiAgICovXG4gIGdyYW50SW5kZXhSZWFkV3JpdGUoaW5kZXg6IHN0cmluZywgaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50O1xuXG4gIC8qKlxuICAgKiBHcmFudCByZWFkIHBlcm1pc3Npb25zIGZvciBhIHNwZWNpZmljIHBhdGggaW4gdGhpcyBkb21haW4gdG8gYW4gSUFNXG4gICAqIHByaW5jaXBhbCAoUm9sZS9Hcm91cC9Vc2VyKS5cbiAgICpcbiAgICogQHBhcmFtIHBhdGggVGhlIHBhdGggdG8gZ3JhbnQgcGVybWlzc2lvbnMgZm9yXG4gICAqIEBwYXJhbSBpZGVudGl0eSBUaGUgcHJpbmNpcGFsXG4gICAqL1xuICBncmFudFBhdGhSZWFkKHBhdGg6IHN0cmluZywgaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50O1xuXG4gIC8qKlxuICAgKiBHcmFudCB3cml0ZSBwZXJtaXNzaW9ucyBmb3IgYSBzcGVjaWZpYyBwYXRoIGluIHRoaXMgZG9tYWluIHRvIGFuIElBTVxuICAgKiBwcmluY2lwYWwgKFJvbGUvR3JvdXAvVXNlcikuXG4gICAqXG4gICAqIEBwYXJhbSBwYXRoIFRoZSBwYXRoIHRvIGdyYW50IHBlcm1pc3Npb25zIGZvclxuICAgKiBAcGFyYW0gaWRlbnRpdHkgVGhlIHByaW5jaXBhbFxuICAgKi9cbiAgZ3JhbnRQYXRoV3JpdGUocGF0aDogc3RyaW5nLCBpZGVudGl0eTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQ7XG5cbiAgLyoqXG4gICAqIEdyYW50IHJlYWQvd3JpdGUgcGVybWlzc2lvbnMgZm9yIGEgc3BlY2lmaWMgcGF0aCBpbiB0aGlzIGRvbWFpbiB0byBhbiBJQU1cbiAgICogcHJpbmNpcGFsIChSb2xlL0dyb3VwL1VzZXIpLlxuICAgKlxuICAgKiBAcGFyYW0gcGF0aCBUaGUgcGF0aCB0byBncmFudCBwZXJtaXNzaW9ucyBmb3JcbiAgICogQHBhcmFtIGlkZW50aXR5IFRoZSBwcmluY2lwYWxcbiAgICovXG4gIGdyYW50UGF0aFJlYWRXcml0ZShwYXRoOiBzdHJpbmcsIGlkZW50aXR5OiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudDtcblxuICAvKipcbiAgICogUmV0dXJuIHRoZSBnaXZlbiBuYW1lZCBtZXRyaWMgZm9yIHRoaXMgZG9tYWluLlxuICAgKi9cbiAgbWV0cmljKG1ldHJpY05hbWU6IHN0cmluZywgcHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljO1xuXG4gIC8qKlxuICAgKiBNZXRyaWMgZm9yIHRoZSB0aW1lIHRoZSBjbHVzdGVyIHN0YXR1cyBpcyByZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IG1heGltdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIG1ldHJpY0NsdXN0ZXJTdGF0dXNSZWQocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljO1xuXG4gIC8qKlxuICAgKiBNZXRyaWMgZm9yIHRoZSB0aW1lIHRoZSBjbHVzdGVyIHN0YXR1cyBpcyB5ZWxsb3cuXG4gICAqXG4gICAqIEBkZWZhdWx0IG1heGltdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIG1ldHJpY0NsdXN0ZXJTdGF0dXNZZWxsb3cocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljO1xuXG4gIC8qKlxuICAgKiBNZXRyaWMgZm9yIHRoZSBzdG9yYWdlIHNwYWNlIG9mIG5vZGVzIGluIHRoZSBjbHVzdGVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCBtaW5pbXVtIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBtZXRyaWNGcmVlU3RvcmFnZVNwYWNlKHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYztcblxuICAvKipcbiAgICogTWV0cmljIGZvciB0aGUgY2x1c3RlciBibG9ja2luZyBpbmRleCB3cml0ZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IG1heGltdW0gb3ZlciAxIG1pbnV0ZVxuICAgKi9cbiAgbWV0cmljQ2x1c3RlckluZGV4V3JpdGVzQmxvY2tlZChwcm9wcz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWM7XG5cbiAgLyoqXG4gICAqIE1ldHJpYyBmb3IgdGhlIG51bWJlciBvZiBub2Rlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgbWluaW11bSBvdmVyIDEgaG91clxuICAgKi9cbiAgbWV0cmljTm9kZXMocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljO1xuXG4gIC8qKlxuICAgKiBNZXRyaWMgZm9yIGF1dG9tYXRlZCBzbmFwc2hvdCBmYWlsdXJlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgbWF4aW11bSBvdmVyIDUgbWludXRlc1xuICAgKi9cbiAgbWV0cmljQXV0b21hdGVkU25hcHNob3RGYWlsdXJlKHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYztcblxuICAvKipcbiAgICogTWV0cmljIGZvciBDUFUgdXRpbGl6YXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IG1heGltdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIG1ldHJpY0NQVVV0aWxpemF0aW9uKHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYztcblxuICAvKipcbiAgICogTWV0cmljIGZvciBKVk0gbWVtb3J5IHByZXNzdXJlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBtYXhpbXVtIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBtZXRyaWNKVk1NZW1vcnlQcmVzc3VyZShwcm9wcz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWM7XG5cbiAgLyoqXG4gICAqIE1ldHJpYyBmb3IgbWFzdGVyIENQVSB1dGlsaXphdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgbWF4aW11bSBvdmVyIDUgbWludXRlc1xuICAgKi9cbiAgbWV0cmljTWFzdGVyQ1BVVXRpbGl6YXRpb24ocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljO1xuXG4gIC8qKlxuICAgKiBNZXRyaWMgZm9yIG1hc3RlciBKVk0gbWVtb3J5IHByZXNzdXJlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBtYXhpbXVtIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBtZXRyaWNNYXN0ZXJKVk1NZW1vcnlQcmVzc3VyZShwcm9wcz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWM7XG5cbiAgLyoqXG4gICAqIE1ldHJpYyBmb3IgS01TIGtleSBlcnJvcnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IG1heGltdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIG1ldHJpY0tNU0tleUVycm9yKHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYztcblxuICAvKipcbiAgICogTWV0cmljIGZvciBLTVMga2V5IGJlaW5nIGluYWNjZXNzaWJsZS5cbiAgICpcbiAgICogQGRlZmF1bHQgbWF4aW11bSBvdmVyIDUgbWludXRlc1xuICAgKi9cbiAgbWV0cmljS01TS2V5SW5hY2Nlc3NpYmxlKHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYztcblxuICAvKipcbiAgICogTWV0cmljIGZvciBudW1iZXIgb2Ygc2VhcmNoYWJsZSBkb2N1bWVudHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IG1heGltdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIG1ldHJpY1NlYXJjaGFibGVEb2N1bWVudHMocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljO1xuXG4gIC8qKlxuICAgKiBNZXRyaWMgZm9yIHNlYXJjaCBsYXRlbmN5LlxuICAgKlxuICAgKiBAZGVmYXVsdCBwOTkgb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIG1ldHJpY1NlYXJjaExhdGVuY3kocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljO1xuXG4gIC8qKlxuICAgKiBNZXRyaWMgZm9yIGluZGV4aW5nIGxhdGVuY3kuXG4gICAqXG4gICAqIEBkZWZhdWx0IHA5OSBvdmVyIDUgbWludXRlc1xuICAgKi9cbiAgbWV0cmljSW5kZXhpbmdMYXRlbmN5KHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYztcbn1cblxuXG4vKipcbiAqIEEgbmV3IG9yIGltcG9ydGVkIGRvbWFpbi5cbiAqL1xuYWJzdHJhY3QgY2xhc3MgRG9tYWluQmFzZSBleHRlbmRzIGNkay5SZXNvdXJjZSBpbXBsZW1lbnRzIElEb21haW4ge1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgZG9tYWluQXJuOiBzdHJpbmc7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBkb21haW5OYW1lOiBzdHJpbmc7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBkb21haW5JZDogc3RyaW5nO1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgZG9tYWluRW5kcG9pbnQ6IHN0cmluZztcblxuICAvKipcbiAgICogR3JhbnQgcmVhZCBwZXJtaXNzaW9ucyBmb3IgdGhpcyBkb21haW4gYW5kIGl0cyBjb250ZW50cyB0byBhbiBJQU1cbiAgICogcHJpbmNpcGFsIChSb2xlL0dyb3VwL1VzZXIpLlxuICAgKlxuICAgKiBAcGFyYW0gaWRlbnRpdHkgVGhlIHByaW5jaXBhbFxuICAgKi9cbiAgZ3JhbnRSZWFkKGlkZW50aXR5OiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudCB7XG4gICAgcmV0dXJuIHRoaXMuZ3JhbnQoXG4gICAgICBpZGVudGl0eSxcbiAgICAgIHBlcm1zLkVTX1JFQURfQUNUSU9OUyxcbiAgICAgIHRoaXMuZG9tYWluQXJuLFxuICAgICAgYCR7dGhpcy5kb21haW5Bcm59LypgLFxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnQgd3JpdGUgcGVybWlzc2lvbnMgZm9yIHRoaXMgZG9tYWluIGFuZCBpdHMgY29udGVudHMgdG8gYW4gSUFNXG4gICAqIHByaW5jaXBhbCAoUm9sZS9Hcm91cC9Vc2VyKS5cbiAgICpcbiAgICogQHBhcmFtIGlkZW50aXR5IFRoZSBwcmluY2lwYWxcbiAgICovXG4gIGdyYW50V3JpdGUoaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICByZXR1cm4gdGhpcy5ncmFudChcbiAgICAgIGlkZW50aXR5LFxuICAgICAgcGVybXMuRVNfV1JJVEVfQUNUSU9OUyxcbiAgICAgIHRoaXMuZG9tYWluQXJuLFxuICAgICAgYCR7dGhpcy5kb21haW5Bcm59LypgLFxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnQgcmVhZC93cml0ZSBwZXJtaXNzaW9ucyBmb3IgdGhpcyBkb21haW4gYW5kIGl0cyBjb250ZW50cyB0byBhbiBJQU1cbiAgICogcHJpbmNpcGFsIChSb2xlL0dyb3VwL1VzZXIpLlxuICAgKlxuICAgKiBAcGFyYW0gaWRlbnRpdHkgVGhlIHByaW5jaXBhbFxuICAgKi9cbiAgZ3JhbnRSZWFkV3JpdGUoaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICByZXR1cm4gdGhpcy5ncmFudChcbiAgICAgIGlkZW50aXR5LFxuICAgICAgcGVybXMuRVNfUkVBRF9XUklURV9BQ1RJT05TLFxuICAgICAgdGhpcy5kb21haW5Bcm4sXG4gICAgICBgJHt0aGlzLmRvbWFpbkFybn0vKmAsXG4gICAgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudCByZWFkIHBlcm1pc3Npb25zIGZvciBhbiBpbmRleCBpbiB0aGlzIGRvbWFpbiB0byBhbiBJQU1cbiAgICogcHJpbmNpcGFsIChSb2xlL0dyb3VwL1VzZXIpLlxuICAgKlxuICAgKiBAcGFyYW0gaW5kZXggVGhlIGluZGV4IHRvIGdyYW50IHBlcm1pc3Npb25zIGZvclxuICAgKiBAcGFyYW0gaWRlbnRpdHkgVGhlIHByaW5jaXBhbFxuICAgKi9cbiAgZ3JhbnRJbmRleFJlYWQoaW5kZXg6IHN0cmluZywgaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICByZXR1cm4gdGhpcy5ncmFudChcbiAgICAgIGlkZW50aXR5LFxuICAgICAgcGVybXMuRVNfUkVBRF9BQ1RJT05TLFxuICAgICAgYCR7dGhpcy5kb21haW5Bcm59LyR7aW5kZXh9YCxcbiAgICAgIGAke3RoaXMuZG9tYWluQXJufS8ke2luZGV4fS8qYCxcbiAgICApO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50IHdyaXRlIHBlcm1pc3Npb25zIGZvciBhbiBpbmRleCBpbiB0aGlzIGRvbWFpbiB0byBhbiBJQU1cbiAgICogcHJpbmNpcGFsIChSb2xlL0dyb3VwL1VzZXIpLlxuICAgKlxuICAgKiBAcGFyYW0gaW5kZXggVGhlIGluZGV4IHRvIGdyYW50IHBlcm1pc3Npb25zIGZvclxuICAgKiBAcGFyYW0gaWRlbnRpdHkgVGhlIHByaW5jaXBhbFxuICAgKi9cbiAgZ3JhbnRJbmRleFdyaXRlKGluZGV4OiBzdHJpbmcsIGlkZW50aXR5OiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudCB7XG4gICAgcmV0dXJuIHRoaXMuZ3JhbnQoXG4gICAgICBpZGVudGl0eSxcbiAgICAgIHBlcm1zLkVTX1dSSVRFX0FDVElPTlMsXG4gICAgICBgJHt0aGlzLmRvbWFpbkFybn0vJHtpbmRleH1gLFxuICAgICAgYCR7dGhpcy5kb21haW5Bcm59LyR7aW5kZXh9LypgLFxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnQgcmVhZC93cml0ZSBwZXJtaXNzaW9ucyBmb3IgYW4gaW5kZXggaW4gdGhpcyBkb21haW4gdG8gYW4gSUFNXG4gICAqIHByaW5jaXBhbCAoUm9sZS9Hcm91cC9Vc2VyKS5cbiAgICpcbiAgICogQHBhcmFtIGluZGV4IFRoZSBpbmRleCB0byBncmFudCBwZXJtaXNzaW9ucyBmb3JcbiAgICogQHBhcmFtIGlkZW50aXR5IFRoZSBwcmluY2lwYWxcbiAgICovXG4gIGdyYW50SW5kZXhSZWFkV3JpdGUoaW5kZXg6IHN0cmluZywgaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICByZXR1cm4gdGhpcy5ncmFudChcbiAgICAgIGlkZW50aXR5LFxuICAgICAgcGVybXMuRVNfUkVBRF9XUklURV9BQ1RJT05TLFxuICAgICAgYCR7dGhpcy5kb21haW5Bcm59LyR7aW5kZXh9YCxcbiAgICAgIGAke3RoaXMuZG9tYWluQXJufS8ke2luZGV4fS8qYCxcbiAgICApO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50IHJlYWQgcGVybWlzc2lvbnMgZm9yIGEgc3BlY2lmaWMgcGF0aCBpbiB0aGlzIGRvbWFpbiB0byBhbiBJQU1cbiAgICogcHJpbmNpcGFsIChSb2xlL0dyb3VwL1VzZXIpLlxuICAgKlxuICAgKiBAcGFyYW0gcGF0aCBUaGUgcGF0aCB0byBncmFudCBwZXJtaXNzaW9ucyBmb3JcbiAgICogQHBhcmFtIGlkZW50aXR5IFRoZSBwcmluY2lwYWxcbiAgICovXG4gIGdyYW50UGF0aFJlYWQocGF0aDogc3RyaW5nLCBpZGVudGl0eTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQge1xuICAgIHJldHVybiB0aGlzLmdyYW50KFxuICAgICAgaWRlbnRpdHksXG4gICAgICBwZXJtcy5FU19SRUFEX0FDVElPTlMsXG4gICAgICBgJHt0aGlzLmRvbWFpbkFybn0vJHtwYXRofWAsXG4gICAgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudCB3cml0ZSBwZXJtaXNzaW9ucyBmb3IgYSBzcGVjaWZpYyBwYXRoIGluIHRoaXMgZG9tYWluIHRvIGFuIElBTVxuICAgKiBwcmluY2lwYWwgKFJvbGUvR3JvdXAvVXNlcikuXG4gICAqXG4gICAqIEBwYXJhbSBwYXRoIFRoZSBwYXRoIHRvIGdyYW50IHBlcm1pc3Npb25zIGZvclxuICAgKiBAcGFyYW0gaWRlbnRpdHkgVGhlIHByaW5jaXBhbFxuICAgKi9cbiAgZ3JhbnRQYXRoV3JpdGUocGF0aDogc3RyaW5nLCBpZGVudGl0eTogaWFtLklHcmFudGFibGUpOiBpYW0uR3JhbnQge1xuICAgIHJldHVybiB0aGlzLmdyYW50KFxuICAgICAgaWRlbnRpdHksXG4gICAgICBwZXJtcy5FU19XUklURV9BQ1RJT05TLFxuICAgICAgYCR7dGhpcy5kb21haW5Bcm59LyR7cGF0aH1gLFxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnQgcmVhZC93cml0ZSBwZXJtaXNzaW9ucyBmb3IgYSBzcGVjaWZpYyBwYXRoIGluIHRoaXMgZG9tYWluIHRvIGFuIElBTVxuICAgKiBwcmluY2lwYWwgKFJvbGUvR3JvdXAvVXNlcikuXG4gICAqXG4gICAqIEBwYXJhbSBwYXRoIFRoZSBwYXRoIHRvIGdyYW50IHBlcm1pc3Npb25zIGZvclxuICAgKiBAcGFyYW0gaWRlbnRpdHkgVGhlIHByaW5jaXBhbFxuICAgKi9cbiAgZ3JhbnRQYXRoUmVhZFdyaXRlKHBhdGg6IHN0cmluZywgaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICByZXR1cm4gdGhpcy5ncmFudChcbiAgICAgIGlkZW50aXR5LFxuICAgICAgcGVybXMuRVNfUkVBRF9XUklURV9BQ1RJT05TLFxuICAgICAgYCR7dGhpcy5kb21haW5Bcm59LyR7cGF0aH1gLFxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJuIHRoZSBnaXZlbiBuYW1lZCBtZXRyaWMgZm9yIHRoaXMgZG9tYWluLlxuICAgKi9cbiAgcHVibGljIG1ldHJpYyhtZXRyaWNOYW1lOiBzdHJpbmcsIHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIG5ldyBNZXRyaWMoe1xuICAgICAgbmFtZXNwYWNlOiAnQVdTL0VTJyxcbiAgICAgIG1ldHJpY05hbWUsXG4gICAgICBkaW1lbnNpb25zTWFwOiB7XG4gICAgICAgIERvbWFpbk5hbWU6IHRoaXMuZG9tYWluTmFtZSxcbiAgICAgICAgQ2xpZW50SWQ6IHRoaXMuZW52LmFjY291bnQsXG4gICAgICB9LFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSkuYXR0YWNoVG8odGhpcyk7XG4gIH1cblxuICAvKipcbiAgICogTWV0cmljIGZvciB0aGUgdGltZSB0aGUgY2x1c3RlciBzdGF0dXMgaXMgcmVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBtYXhpbXVtIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBwdWJsaWMgbWV0cmljQ2x1c3RlclN0YXR1c1JlZChwcm9wcz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnQ2x1c3RlclN0YXR1cy5yZWQnLCB7XG4gICAgICBzdGF0aXN0aWM6IFN0YXRpc3RpYy5NQVhJTVVNLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogTWV0cmljIGZvciB0aGUgdGltZSB0aGUgY2x1c3RlciBzdGF0dXMgaXMgeWVsbG93LlxuICAgKlxuICAgKiBAZGVmYXVsdCBtYXhpbXVtIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBwdWJsaWMgbWV0cmljQ2x1c3RlclN0YXR1c1llbGxvdyhwcm9wcz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnQ2x1c3RlclN0YXR1cy55ZWxsb3cnLCB7XG4gICAgICBzdGF0aXN0aWM6IFN0YXRpc3RpYy5NQVhJTVVNLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogTWV0cmljIGZvciB0aGUgc3RvcmFnZSBzcGFjZSBvZiBub2RlcyBpbiB0aGUgY2x1c3Rlci5cbiAgICpcbiAgICogQGRlZmF1bHQgbWluaW11bSBvdmVyIDUgbWludXRlc1xuICAgKi9cbiAgcHVibGljIG1ldHJpY0ZyZWVTdG9yYWdlU3BhY2UocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ0ZyZWVTdG9yYWdlU3BhY2UnLCB7XG4gICAgICBzdGF0aXN0aWM6IFN0YXRpc3RpYy5NSU5JTVVNLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogTWV0cmljIGZvciB0aGUgY2x1c3RlciBibG9ja2luZyBpbmRleCB3cml0ZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IG1heGltdW0gb3ZlciAxIG1pbnV0ZVxuICAgKi9cbiAgcHVibGljIG1ldHJpY0NsdXN0ZXJJbmRleFdyaXRlc0Jsb2NrZWQocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ0NsdXN0ZXJJbmRleFdyaXRlc0Jsb2NrZWQnLCB7XG4gICAgICBzdGF0aXN0aWM6IFN0YXRpc3RpYy5NQVhJTVVNLFxuICAgICAgcGVyaW9kOiBjZGsuRHVyYXRpb24ubWludXRlcygxKSxcbiAgICAgIC4uLnByb3BzLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIE1ldHJpYyBmb3IgdGhlIG51bWJlciBvZiBub2Rlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgbWluaW11bSBvdmVyIDEgaG91clxuICAgKi9cbiAgcHVibGljIG1ldHJpY05vZGVzKHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdOb2RlcycsIHtcbiAgICAgIHN0YXRpc3RpYzogU3RhdGlzdGljLk1JTklNVU0sXG4gICAgICBwZXJpb2Q6IGNkay5EdXJhdGlvbi5ob3VycygxKSxcbiAgICAgIC4uLnByb3BzLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIE1ldHJpYyBmb3IgYXV0b21hdGVkIHNuYXBzaG90IGZhaWx1cmVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBtYXhpbXVtIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBwdWJsaWMgbWV0cmljQXV0b21hdGVkU25hcHNob3RGYWlsdXJlKHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdBdXRvbWF0ZWRTbmFwc2hvdEZhaWx1cmUnLCB7XG4gICAgICBzdGF0aXN0aWM6IFN0YXRpc3RpYy5NQVhJTVVNLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogTWV0cmljIGZvciBDUFUgdXRpbGl6YXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IG1heGltdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIHB1YmxpYyBtZXRyaWNDUFVVdGlsaXphdGlvbihwcm9wcz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnQ1BVVXRpbGl6YXRpb24nLCB7XG4gICAgICBzdGF0aXN0aWM6IFN0YXRpc3RpYy5NQVhJTVVNLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogTWV0cmljIGZvciBKVk0gbWVtb3J5IHByZXNzdXJlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBtYXhpbXVtIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBwdWJsaWMgbWV0cmljSlZNTWVtb3J5UHJlc3N1cmUocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ0pWTU1lbW9yeVByZXNzdXJlJywge1xuICAgICAgc3RhdGlzdGljOiBTdGF0aXN0aWMuTUFYSU1VTSxcbiAgICAgIC4uLnByb3BzLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIE1ldHJpYyBmb3IgbWFzdGVyIENQVSB1dGlsaXphdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgbWF4aW11bSBvdmVyIDUgbWludXRlc1xuICAgKi9cbiAgcHVibGljIG1ldHJpY01hc3RlckNQVVV0aWxpemF0aW9uKHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdNYXN0ZXJDUFVVdGlsaXphdGlvbicsIHtcbiAgICAgIHN0YXRpc3RpYzogU3RhdGlzdGljLk1BWElNVU0sXG4gICAgICAuLi5wcm9wcyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNZXRyaWMgZm9yIG1hc3RlciBKVk0gbWVtb3J5IHByZXNzdXJlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBtYXhpbXVtIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBwdWJsaWMgbWV0cmljTWFzdGVySlZNTWVtb3J5UHJlc3N1cmUocHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWMoJ01hc3RlckpWTU1lbW9yeVByZXNzdXJlJywge1xuICAgICAgc3RhdGlzdGljOiBTdGF0aXN0aWMuTUFYSU1VTSxcbiAgICAgIC4uLnByb3BzLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIE1ldHJpYyBmb3IgS01TIGtleSBlcnJvcnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IG1heGltdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIHB1YmxpYyBtZXRyaWNLTVNLZXlFcnJvcihwcm9wcz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnS01TS2V5RXJyb3InLCB7XG4gICAgICBzdGF0aXN0aWM6IFN0YXRpc3RpYy5NQVhJTVVNLFxuICAgICAgLi4ucHJvcHMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogTWV0cmljIGZvciBLTVMga2V5IGJlaW5nIGluYWNjZXNzaWJsZS5cbiAgICpcbiAgICogQGRlZmF1bHQgbWF4aW11bSBvdmVyIDUgbWludXRlc1xuICAgKi9cbiAgcHVibGljIG1ldHJpY0tNU0tleUluYWNjZXNzaWJsZShwcm9wcz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnS01TS2V5SW5hY2Nlc3NpYmxlJywge1xuICAgICAgc3RhdGlzdGljOiBTdGF0aXN0aWMuTUFYSU1VTSxcbiAgICAgIC4uLnByb3BzLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIE1ldHJpYyBmb3IgbnVtYmVyIG9mIHNlYXJjaGFibGUgZG9jdW1lbnRzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBtYXhpbXVtIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBwdWJsaWMgbWV0cmljU2VhcmNoYWJsZURvY3VtZW50cyhwcm9wcz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnU2VhcmNoYWJsZURvY3VtZW50cycsIHtcbiAgICAgIHN0YXRpc3RpYzogU3RhdGlzdGljLk1BWElNVU0sXG4gICAgICAuLi5wcm9wcyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNZXRyaWMgZm9yIHNlYXJjaCBsYXRlbmN5LlxuICAgKlxuICAgKiBAZGVmYXVsdCBwOTkgb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIHB1YmxpYyBtZXRyaWNTZWFyY2hMYXRlbmN5KHByb3BzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIHRoaXMubWV0cmljKCdTZWFyY2hMYXRlbmN5JywgeyBzdGF0aXN0aWM6ICdwOTknLCAuLi5wcm9wcyB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNZXRyaWMgZm9yIGluZGV4aW5nIGxhdGVuY3kuXG4gICAqXG4gICAqIEBkZWZhdWx0IHA5OSBvdmVyIDUgbWludXRlc1xuICAgKi9cbiAgcHVibGljIG1ldHJpY0luZGV4aW5nTGF0ZW5jeShwcm9wcz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiB0aGlzLm1ldHJpYygnSW5kZXhpbmdMYXRlbmN5JywgeyBzdGF0aXN0aWM6ICdwOTknLCAuLi5wcm9wcyB9KTtcbiAgfVxuXG4gIHByaXZhdGUgZ3JhbnQoXG4gICAgZ3JhbnRlZTogaWFtLklHcmFudGFibGUsXG4gICAgZG9tYWluQWN0aW9uczogc3RyaW5nW10sXG4gICAgcmVzb3VyY2VBcm46IHN0cmluZyxcbiAgICAuLi5vdGhlclJlc291cmNlQXJuczogc3RyaW5nW11cbiAgKTogaWFtLkdyYW50IHtcbiAgICBjb25zdCByZXNvdXJjZUFybnMgPSBbcmVzb3VyY2VBcm4sIC4uLm90aGVyUmVzb3VyY2VBcm5zXTtcblxuICAgIGNvbnN0IGdyYW50ID0gaWFtLkdyYW50LmFkZFRvUHJpbmNpcGFsKHtcbiAgICAgIGdyYW50ZWUsXG4gICAgICBhY3Rpb25zOiBkb21haW5BY3Rpb25zLFxuICAgICAgcmVzb3VyY2VBcm5zLFxuICAgICAgc2NvcGU6IHRoaXMsXG4gICAgfSk7XG5cbiAgICByZXR1cm4gZ3JhbnQ7XG4gIH1cblxufVxuXG5cbi8qKlxuICogUmVmZXJlbmNlIHRvIGFuIEFtYXpvbiBPcGVuU2VhcmNoIFNlcnZpY2UgZG9tYWluLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIERvbWFpbkF0dHJpYnV0ZXMge1xuICAvKipcbiAgICogVGhlIEFSTiBvZiB0aGUgQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4uXG4gICAqL1xuICByZWFkb25seSBkb21haW5Bcm46IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGRvbWFpbiBlbmRwb2ludCBvZiB0aGUgQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4uXG4gICAqL1xuICByZWFkb25seSBkb21haW5FbmRwb2ludDogc3RyaW5nO1xufVxuXG5cbi8qKlxuICogUHJvdmlkZXMgYW4gQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4uXG4gKi9cbmV4cG9ydCBjbGFzcyBEb21haW4gZXh0ZW5kcyBEb21haW5CYXNlIGltcGxlbWVudHMgSURvbWFpbiwgZWMyLklDb25uZWN0YWJsZSB7XG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgZG9tYWluIGNvbnN0cnVjdCB0aGF0IHJlcHJlc2VudHMgYW4gZXh0ZXJuYWwgZG9tYWluIHZpYSBkb21haW4gZW5kcG9pbnQuXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZSBUaGUgcGFyZW50IGNyZWF0aW5nIGNvbnN0cnVjdCAodXN1YWxseSBgdGhpc2ApLlxuICAgKiBAcGFyYW0gaWQgVGhlIGNvbnN0cnVjdCdzIG5hbWUuXG4gICAqIEBwYXJhbSBkb21haW5FbmRwb2ludCBUaGUgZG9tYWluJ3MgZW5kcG9pbnQuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21Eb21haW5FbmRwb2ludChcbiAgICBzY29wZTogQ29uc3RydWN0LFxuICAgIGlkOiBzdHJpbmcsXG4gICAgZG9tYWluRW5kcG9pbnQ6IHN0cmluZyxcbiAgKTogSURvbWFpbiB7XG4gICAgY29uc3Qgc3RhY2sgPSBjZGsuU3RhY2sub2Yoc2NvcGUpO1xuICAgIGNvbnN0IGRvbWFpbk5hbWUgPSBleHRyYWN0TmFtZUZyb21FbmRwb2ludChkb21haW5FbmRwb2ludCk7XG4gICAgY29uc3QgZG9tYWluQXJuID0gc3RhY2suZm9ybWF0QXJuKHtcbiAgICAgIHNlcnZpY2U6ICdlcycsXG4gICAgICByZXNvdXJjZTogJ2RvbWFpbicsXG4gICAgICByZXNvdXJjZU5hbWU6IGRvbWFpbk5hbWUsXG4gICAgfSk7XG5cbiAgICByZXR1cm4gRG9tYWluLmZyb21Eb21haW5BdHRyaWJ1dGVzKHNjb3BlLCBpZCwge1xuICAgICAgZG9tYWluQXJuLFxuICAgICAgZG9tYWluRW5kcG9pbnQsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIGRvbWFpbiBjb25zdHJ1Y3QgdGhhdCByZXByZXNlbnRzIGFuIGV4dGVybmFsIGRvbWFpbi5cbiAgICpcbiAgICogQHBhcmFtIHNjb3BlIFRoZSBwYXJlbnQgY3JlYXRpbmcgY29uc3RydWN0ICh1c3VhbGx5IGB0aGlzYCkuXG4gICAqIEBwYXJhbSBpZCBUaGUgY29uc3RydWN0J3MgbmFtZS5cbiAgICogQHBhcmFtIGF0dHJzIEEgYERvbWFpbkF0dHJpYnV0ZXNgIG9iamVjdC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbURvbWFpbkF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IERvbWFpbkF0dHJpYnV0ZXMpOiBJRG9tYWluIHtcbiAgICBjb25zdCB7IGRvbWFpbkFybiwgZG9tYWluRW5kcG9pbnQgfSA9IGF0dHJzO1xuICAgIGNvbnN0IGRvbWFpbk5hbWUgPSBjZGsuU3RhY2sub2Yoc2NvcGUpLnNwbGl0QXJuKGRvbWFpbkFybiwgY2RrLkFybkZvcm1hdC5TTEFTSF9SRVNPVVJDRV9OQU1FKS5yZXNvdXJjZU5hbWVcbiAgICAgID8/IGV4dHJhY3ROYW1lRnJvbUVuZHBvaW50KGRvbWFpbkVuZHBvaW50KTtcblxuICAgIHJldHVybiBuZXcgY2xhc3MgZXh0ZW5kcyBEb21haW5CYXNlIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBkb21haW5Bcm4gPSBkb21haW5Bcm47XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZG9tYWluTmFtZSA9IGRvbWFpbk5hbWU7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgZG9tYWluSWQgPSBkb21haW5OYW1lO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGRvbWFpbkVuZHBvaW50ID0gZG9tYWluRW5kcG9pbnQucmVwbGFjZSgvXmh0dHBzPzpcXC9cXC8vLCAnJyk7XG5cbiAgICAgIGNvbnN0cnVjdG9yKCkgeyBzdXBlcihzY29wZSwgaWQpOyB9XG4gICAgfTtcbiAgfVxuXG4gIHB1YmxpYyByZWFkb25seSBkb21haW5Bcm46IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGRvbWFpbk5hbWU6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGRvbWFpbklkOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBkb21haW5FbmRwb2ludDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBMb2cgZ3JvdXAgdGhhdCBzbG93IHNlYXJjaGVzIGFyZSBsb2dnZWQgdG8uXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzbG93U2VhcmNoTG9nR3JvdXA/OiBsb2dzLklMb2dHcm91cDtcblxuICAvKipcbiAgICogTG9nIGdyb3VwIHRoYXQgc2xvdyBpbmRpY2VzIGFyZSBsb2dnZWQgdG8uXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzbG93SW5kZXhMb2dHcm91cD86IGxvZ3MuSUxvZ0dyb3VwO1xuXG4gIC8qKlxuICAgKiBMb2cgZ3JvdXAgdGhhdCBhcHBsaWNhdGlvbiBsb2dzIGFyZSBsb2dnZWQgdG8uXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhcHBMb2dHcm91cD86IGxvZ3MuSUxvZ0dyb3VwO1xuXG4gIC8qKlxuICAgKiBMb2cgZ3JvdXAgdGhhdCBhdWRpdCBsb2dzIGFyZSBsb2dnZWQgdG8uXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhdWRpdExvZ0dyb3VwPzogbG9ncy5JTG9nR3JvdXA7XG5cbiAgLyoqXG4gICAqIE1hc3RlciB1c2VyIHBhc3N3b3JkIGlmIGZpbmUgZ3JhaW5lZCBhY2Nlc3MgY29udHJvbCBpcyBjb25maWd1cmVkLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IG1hc3RlclVzZXJQYXNzd29yZD86IGNkay5TZWNyZXRWYWx1ZTtcblxuXG4gIHByaXZhdGUgcmVhZG9ubHkgZG9tYWluOiBDZm5Eb21haW47XG5cbiAgcHJpdmF0ZSBhY2Nlc3NQb2xpY3k/OiBPcGVuU2VhcmNoQWNjZXNzUG9saWN5XG5cbiAgcHJpdmF0ZSBlbmNyeXB0aW9uQXRSZXN0T3B0aW9ucz86IEVuY3J5cHRpb25BdFJlc3RPcHRpb25zXG5cbiAgcHJpdmF0ZSByZWFkb25seSBfY29ubmVjdGlvbnM6IGVjMi5Db25uZWN0aW9ucyB8IHVuZGVmaW5lZDtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRG9tYWluUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMuZG9tYWluTmFtZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGRlZmF1bHRJbnN0YW5jZVR5cGUgPSAncjUubGFyZ2Uuc2VhcmNoJztcbiAgICBjb25zdCB3YXJtRGVmYXVsdEluc3RhbmNlVHlwZSA9ICd1bHRyYXdhcm0xLm1lZGl1bS5zZWFyY2gnO1xuXG4gICAgY29uc3QgZGVkaWNhdGVkTWFzdGVyVHlwZSA9IGluaXRpYWxpemVJbnN0YW5jZVR5cGUoZGVmYXVsdEluc3RhbmNlVHlwZSwgcHJvcHMuY2FwYWNpdHk/Lm1hc3Rlck5vZGVJbnN0YW5jZVR5cGUpO1xuICAgIGNvbnN0IGRlZGljYXRlZE1hc3RlckNvdW50ID0gcHJvcHMuY2FwYWNpdHk/Lm1hc3Rlck5vZGVzID8/IDA7XG4gICAgY29uc3QgZGVkaWNhdGVkTWFzdGVyRW5hYmxlZCA9IGNkay5Ub2tlbi5pc1VucmVzb2x2ZWQoZGVkaWNhdGVkTWFzdGVyQ291bnQpID8gdHJ1ZSA6IGRlZGljYXRlZE1hc3RlckNvdW50ID4gMDtcblxuICAgIGNvbnN0IGluc3RhbmNlVHlwZSA9IGluaXRpYWxpemVJbnN0YW5jZVR5cGUoZGVmYXVsdEluc3RhbmNlVHlwZSwgcHJvcHMuY2FwYWNpdHk/LmRhdGFOb2RlSW5zdGFuY2VUeXBlKTtcbiAgICBjb25zdCBpbnN0YW5jZUNvdW50ID0gcHJvcHMuY2FwYWNpdHk/LmRhdGFOb2RlcyA/PyAxO1xuXG4gICAgY29uc3Qgd2FybVR5cGUgPSBpbml0aWFsaXplSW5zdGFuY2VUeXBlKHdhcm1EZWZhdWx0SW5zdGFuY2VUeXBlLCBwcm9wcy5jYXBhY2l0eT8ud2FybUluc3RhbmNlVHlwZSk7XG4gICAgY29uc3Qgd2FybUNvdW50ID0gcHJvcHMuY2FwYWNpdHk/Lndhcm1Ob2RlcyA/PyAwO1xuICAgIGNvbnN0IHdhcm1FbmFibGVkID0gY2RrLlRva2VuLmlzVW5yZXNvbHZlZCh3YXJtQ291bnQpID8gdHJ1ZSA6IHdhcm1Db3VudCA+IDA7XG5cbiAgICBjb25zdCBhdmFpbGFiaWxpdHlab25lQ291bnQgPVxuICAgICAgcHJvcHMuem9uZUF3YXJlbmVzcz8uYXZhaWxhYmlsaXR5Wm9uZUNvdW50ID8/IDI7XG5cbiAgICBpZiAoIVsyLCAzXS5pbmNsdWRlcyhhdmFpbGFiaWxpdHlab25lQ291bnQpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0ludmFsaWQgem9uZSBhd2FyZW5lc3MgY29uZmlndXJhdGlvbjsgYXZhaWxhYmlsaXR5Wm9uZUNvdW50IG11c3QgYmUgMiBvciAzJyk7XG4gICAgfVxuXG4gICAgY29uc3Qgem9uZUF3YXJlbmVzc0VuYWJsZWQgPVxuICAgICAgcHJvcHMuem9uZUF3YXJlbmVzcz8uZW5hYmxlZCA/P1xuICAgICAgcHJvcHMuem9uZUF3YXJlbmVzcz8uYXZhaWxhYmlsaXR5Wm9uZUNvdW50ICE9IG51bGw7XG5cblxuICAgIGxldCBzZWN1cml0eUdyb3VwczogZWMyLklTZWN1cml0eUdyb3VwW10gfCB1bmRlZmluZWQ7XG4gICAgbGV0IHN1Ym5ldHM6IGVjMi5JU3VibmV0W10gfCB1bmRlZmluZWQ7XG5cbiAgICBpZiAocHJvcHMudnBjKSB7XG4gICAgICBzdWJuZXRzID0gc2VsZWN0U3VibmV0cyhwcm9wcy52cGMsIHByb3BzLnZwY1N1Ym5ldHMgPz8gW3sgc3VibmV0VHlwZTogZWMyLlN1Ym5ldFR5cGUuUFJJVkFURV9XSVRIX05BVCB9XSk7XG4gICAgICBzZWN1cml0eUdyb3VwcyA9IHByb3BzLnNlY3VyaXR5R3JvdXBzID8/IFtuZXcgZWMyLlNlY3VyaXR5R3JvdXAodGhpcywgJ1NlY3VyaXR5R3JvdXAnLCB7XG4gICAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgICAgICBkZXNjcmlwdGlvbjogYFNlY3VyaXR5IGdyb3VwIGZvciBkb21haW4gJHt0aGlzLm5vZGUuaWR9YCxcbiAgICAgIH0pXTtcbiAgICAgIHRoaXMuX2Nvbm5lY3Rpb25zID0gbmV3IGVjMi5Db25uZWN0aW9ucyh7IHNlY3VyaXR5R3JvdXBzIH0pO1xuICAgIH1cblxuICAgIC8vIElmIFZQQyBvcHRpb25zIGFyZSBzdXBwbGllZCBlbnN1cmUgdGhhdCB0aGUgbnVtYmVyIG9mIHN1Ym5ldHMgbWF0Y2hlcyB0aGUgbnVtYmVyIEFaXG4gICAgaWYgKHN1Ym5ldHMgJiYgem9uZUF3YXJlbmVzc0VuYWJsZWQgJiYgbmV3IFNldChzdWJuZXRzLm1hcCgoc3VibmV0KSA9PiBzdWJuZXQuYXZhaWxhYmlsaXR5Wm9uZSkpLnNpemUgPCBhdmFpbGFiaWxpdHlab25lQ291bnQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignV2hlbiBwcm92aWRpbmcgdnBjIG9wdGlvbnMgeW91IG5lZWQgdG8gcHJvdmlkZSBhIHN1Ym5ldCBmb3IgZWFjaCBBWiB5b3UgYXJlIHVzaW5nJyk7XG4gICAgfVxuXG4gICAgaWYgKFtkZWRpY2F0ZWRNYXN0ZXJUeXBlLCBpbnN0YW5jZVR5cGUsIHdhcm1UeXBlXS5zb21lKHQgPT4gKCFjZGsuVG9rZW4uaXNVbnJlc29sdmVkKHQpICYmICF0LmVuZHNXaXRoKCcuc2VhcmNoJykpKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdNYXN0ZXIsIGRhdGEgYW5kIFVsdHJhV2FybSBub2RlIGluc3RhbmNlIHR5cGVzIG11c3QgZW5kIHdpdGggXCIuc2VhcmNoXCIuJyk7XG4gICAgfVxuXG4gICAgaWYgKCFjZGsuVG9rZW4uaXNVbnJlc29sdmVkKHdhcm1UeXBlKSAmJiAhd2FybVR5cGUuc3RhcnRzV2l0aCgndWx0cmF3YXJtJykpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignVWx0cmFXYXJtIG5vZGUgaW5zdGFuY2UgdHlwZSBtdXN0IHN0YXJ0IHdpdGggXCJ1bHRyYXdhcm1cIi4nKTtcbiAgICB9XG5cbiAgICBjb25zdCB1bnNpZ25lZEJhc2ljQXV0aEVuYWJsZWQgPSBwcm9wcy51c2VVbnNpZ25lZEJhc2ljQXV0aCA/PyBmYWxzZTtcblxuICAgIGlmICh1bnNpZ25lZEJhc2ljQXV0aEVuYWJsZWQpIHtcbiAgICAgIGlmIChwcm9wcy5lbmZvcmNlSHR0cHMgPT0gZmFsc2UpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdZb3UgY2Fubm90IGRpc2FibGUgSFRUUFMgYW5kIHVzZSB1bnNpZ25lZCBiYXNpYyBhdXRoJyk7XG4gICAgICB9XG4gICAgICBpZiAocHJvcHMubm9kZVRvTm9kZUVuY3J5cHRpb24gPT0gZmFsc2UpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdZb3UgY2Fubm90IGRpc2FibGUgbm9kZSB0byBub2RlIGVuY3J5cHRpb24gYW5kIHVzZSB1bnNpZ25lZCBiYXNpYyBhdXRoJyk7XG4gICAgICB9XG4gICAgICBpZiAocHJvcHMuZW5jcnlwdGlvbkF0UmVzdD8uZW5hYmxlZCA9PSBmYWxzZSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1lvdSBjYW5ub3QgZGlzYWJsZSBlbmNyeXB0aW9uIGF0IHJlc3QgYW5kIHVzZSB1bnNpZ25lZCBiYXNpYyBhdXRoJyk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3QgdW5zaWduZWRBY2Nlc3NQb2xpY3kgPSBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICBhY3Rpb25zOiBbJ2VzOkVTSHR0cConXSxcbiAgICAgIHByaW5jaXBhbHM6IFtuZXcgaWFtLkFueVByaW5jaXBhbCgpXSxcbiAgICAgIHJlc291cmNlczogW2Nkay5MYXp5LnN0cmluZyh7IHByb2R1Y2U6ICgpID0+IGAke3RoaXMuZG9tYWluQXJufS8qYCB9KV0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBtYXN0ZXJVc2VyQXJuID0gcHJvcHMuZmluZUdyYWluZWRBY2Nlc3NDb250cm9sPy5tYXN0ZXJVc2VyQXJuO1xuICAgIGNvbnN0IG1hc3RlclVzZXJOYW1lUHJvcHMgPSBwcm9wcy5maW5lR3JhaW5lZEFjY2Vzc0NvbnRyb2w/Lm1hc3RlclVzZXJOYW1lO1xuICAgIC8vIElmIGJhc2ljIGF1dGggaXMgZW5hYmxlZCBzZXQgdGhlIHVzZXIgbmFtZSB0byBhZG1pbiBpZiBubyBvdGhlciB1c2VyIGluZm8gaXMgc3VwcGxpZWQuXG4gICAgY29uc3QgbWFzdGVyVXNlck5hbWUgPSB1bnNpZ25lZEJhc2ljQXV0aEVuYWJsZWRcbiAgICAgID8gKG1hc3RlclVzZXJBcm4gPT0gbnVsbCA/IChtYXN0ZXJVc2VyTmFtZVByb3BzID8/ICdhZG1pbicpIDogdW5kZWZpbmVkKVxuICAgICAgOiBtYXN0ZXJVc2VyTmFtZVByb3BzO1xuXG4gICAgaWYgKG1hc3RlclVzZXJBcm4gIT0gbnVsbCAmJiBtYXN0ZXJVc2VyTmFtZSAhPSBudWxsKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0ludmFsaWQgZmluZSBncmFpbmVkIGFjY2VzcyBjb250cm9sIHNldHRpbmdzLiBPbmx5IHByb3ZpZGUgb25lIG9mIG1hc3RlciB1c2VyIEFSTiBvciBtYXN0ZXIgdXNlciBuYW1lLiBOb3QgYm90aC4nKTtcbiAgICB9XG5cbiAgICBjb25zdCBhZHZhbmNlZFNlY3VyaXR5RW5hYmxlZCA9IChtYXN0ZXJVc2VyQXJuID8/IG1hc3RlclVzZXJOYW1lKSAhPSBudWxsO1xuICAgIGNvbnN0IGludGVybmFsVXNlckRhdGFiYXNlRW5hYmxlZCA9IG1hc3RlclVzZXJOYW1lICE9IG51bGw7XG4gICAgY29uc3QgbWFzdGVyVXNlclBhc3N3b3JkUHJvcCA9IHByb3BzLmZpbmVHcmFpbmVkQWNjZXNzQ29udHJvbD8ubWFzdGVyVXNlclBhc3N3b3JkO1xuICAgIGNvbnN0IGNyZWF0ZU1hc3RlclVzZXJQYXNzd29yZCA9ICgpOiBjZGsuU2VjcmV0VmFsdWUgPT4ge1xuICAgICAgcmV0dXJuIG5ldyBzZWNyZXRzbWFuYWdlci5TZWNyZXQodGhpcywgJ01hc3RlclVzZXInLCB7XG4gICAgICAgIGdlbmVyYXRlU2VjcmV0U3RyaW5nOiB7XG4gICAgICAgICAgc2VjcmV0U3RyaW5nVGVtcGxhdGU6IEpTT04uc3RyaW5naWZ5KHtcbiAgICAgICAgICAgIHVzZXJuYW1lOiBtYXN0ZXJVc2VyTmFtZSxcbiAgICAgICAgICB9KSxcbiAgICAgICAgICBnZW5lcmF0ZVN0cmluZ0tleTogJ3Bhc3N3b3JkJyxcbiAgICAgICAgICBleGNsdWRlQ2hhcmFjdGVyczogXCJ7fSdcXFxcKltdKClgXCIsXG4gICAgICAgIH0sXG4gICAgICB9KVxuICAgICAgICAuc2VjcmV0VmFsdWVGcm9tSnNvbigncGFzc3dvcmQnKTtcbiAgICB9O1xuICAgIHRoaXMubWFzdGVyVXNlclBhc3N3b3JkID0gaW50ZXJuYWxVc2VyRGF0YWJhc2VFbmFibGVkID9cbiAgICAgIChtYXN0ZXJVc2VyUGFzc3dvcmRQcm9wID8/IGNyZWF0ZU1hc3RlclVzZXJQYXNzd29yZCgpKVxuICAgICAgOiB1bmRlZmluZWQ7XG5cbiAgICBjb25zdCBlbmNyeXB0aW9uQXRSZXN0RW5hYmxlZCA9XG4gICAgICBwcm9wcy5lbmNyeXB0aW9uQXRSZXN0Py5lbmFibGVkID8/IChwcm9wcy5lbmNyeXB0aW9uQXRSZXN0Py5rbXNLZXkgIT0gbnVsbCB8fCB1bnNpZ25lZEJhc2ljQXV0aEVuYWJsZWQpO1xuICAgIGNvbnN0IG5vZGVUb05vZGVFbmNyeXB0aW9uRW5hYmxlZCA9IHByb3BzLm5vZGVUb05vZGVFbmNyeXB0aW9uID8/IHVuc2lnbmVkQmFzaWNBdXRoRW5hYmxlZDtcbiAgICBjb25zdCB2b2x1bWVTaXplID0gcHJvcHMuZWJzPy52b2x1bWVTaXplID8/IDEwO1xuICAgIGNvbnN0IHZvbHVtZVR5cGUgPSBwcm9wcy5lYnM/LnZvbHVtZVR5cGUgPz8gZWMyLkVic0RldmljZVZvbHVtZVR5cGUuR0VORVJBTF9QVVJQT1NFX1NTRDtcbiAgICBjb25zdCBlYnNFbmFibGVkID0gcHJvcHMuZWJzPy5lbmFibGVkID8/IHRydWU7XG4gICAgY29uc3QgZW5mb3JjZUh0dHBzID0gcHJvcHMuZW5mb3JjZUh0dHBzID8/IHVuc2lnbmVkQmFzaWNBdXRoRW5hYmxlZDtcblxuICAgIGZ1bmN0aW9uIGlzSW5zdGFuY2VUeXBlKHQ6IHN0cmluZyk6IEJvb2xlYW4ge1xuICAgICAgcmV0dXJuIGRlZGljYXRlZE1hc3RlclR5cGUuc3RhcnRzV2l0aCh0KSB8fCBpbnN0YW5jZVR5cGUuc3RhcnRzV2l0aCh0KTtcbiAgICB9O1xuXG4gICAgZnVuY3Rpb24gaXNTb21lSW5zdGFuY2VUeXBlKC4uLmluc3RhbmNlVHlwZXM6IHN0cmluZ1tdKTogQm9vbGVhbiB7XG4gICAgICByZXR1cm4gaW5zdGFuY2VUeXBlcy5zb21lKGlzSW5zdGFuY2VUeXBlKTtcbiAgICB9O1xuXG4gICAgZnVuY3Rpb24gaXNFdmVyeURhdGFub2RlSW5zdGFuY2VUeXBlKC4uLmluc3RhbmNlVHlwZXM6IHN0cmluZ1tdKTogQm9vbGVhbiB7XG4gICAgICByZXR1cm4gaW5zdGFuY2VUeXBlcy5zb21lKHQgPT4gaW5zdGFuY2VUeXBlLnN0YXJ0c1dpdGgodCkpO1xuICAgIH07XG5cbiAgICAvLyBWYWxpZGF0ZSBmZWF0dXJlIHN1cHBvcnQgZm9yIHRoZSBnaXZlbiBFbGFzdGljc2VhcmNoL09wZW5TZWFyY2ggdmVyc2lvbiwgcGVyXG4gICAgLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wZW5zZWFyY2gtc2VydmljZS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvZmVhdHVyZXMtYnktdmVyc2lvbi5odG1sXG4gICAgY29uc3QgeyB2ZXJzaW9uTnVtOiB2ZXJzaW9uTnVtLCBpc0VsYXN0aWNzZWFyY2hWZXJzaW9uIH0gPSBwYXJzZVZlcnNpb24ocHJvcHMudmVyc2lvbik7XG4gICAgaWYgKGlzRWxhc3RpY3NlYXJjaFZlcnNpb24pIHtcblxuICAgICAgaWYgKFxuICAgICAgICB2ZXJzaW9uTnVtIDw9IDcuNyAmJlxuICAgICAgICAhW1xuICAgICAgICAgIDEuNSwgMi4zLCA1LjEsIDUuMywgNS41LCA1LjYsIDYuMCxcbiAgICAgICAgICA2LjIsIDYuMywgNi40LCA2LjUsIDYuNywgNi44LCA3LjEsIDcuNCxcbiAgICAgICAgICA3LjcsXG4gICAgICAgIF0uaW5jbHVkZXModmVyc2lvbk51bSlcbiAgICAgICkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFVua25vd24gRWxhc3RpY3NlYXJjaCB2ZXJzaW9uOiAke3ZlcnNpb25OdW19YCk7XG4gICAgICB9XG5cbiAgICAgIGlmICh2ZXJzaW9uTnVtIDwgNS4xKSB7XG4gICAgICAgIGlmIChwcm9wcy5sb2dnaW5nPy5hcHBMb2dFbmFibGVkKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdFcnJvciBsb2dzIHB1Ymxpc2hpbmcgcmVxdWlyZXMgRWxhc3RpY3NlYXJjaCB2ZXJzaW9uIDUuMSBvciBsYXRlciBvciBPcGVuU2VhcmNoIHZlcnNpb24gMS4wIG9yIGxhdGVyLicpO1xuICAgICAgICB9XG4gICAgICAgIGlmIChwcm9wcy5lbmNyeXB0aW9uQXRSZXN0Py5lbmFibGVkKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdFbmNyeXB0aW9uIG9mIGRhdGEgYXQgcmVzdCByZXF1aXJlcyBFbGFzdGljc2VhcmNoIHZlcnNpb24gNS4xIG9yIGxhdGVyIG9yIE9wZW5TZWFyY2ggdmVyc2lvbiAxLjAgb3IgbGF0ZXIuJyk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHByb3BzLmNvZ25pdG9EYXNoYm9hcmRzQXV0aCAhPSBudWxsKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdDb2duaXRvIGF1dGhlbnRpY2F0aW9uIGZvciBPcGVuU2VhcmNoIERhc2hib2FyZHMgcmVxdWlyZXMgRWxhc3RpY3NlYXJjaCB2ZXJzaW9uIDUuMSBvciBsYXRlciBvciBPcGVuU2VhcmNoIHZlcnNpb24gMS4wIG9yIGxhdGVyLicpO1xuICAgICAgICB9XG4gICAgICAgIGlmIChpc1NvbWVJbnN0YW5jZVR5cGUoJ2M1JywgJ2kzJywgJ201JywgJ3I1JykpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0M1LCBJMywgTTUsIGFuZCBSNSBpbnN0YW5jZSB0eXBlcyByZXF1aXJlIEVsYXN0aWNzZWFyY2ggdmVyc2lvbiA1LjEgb3IgbGF0ZXIgb3IgT3BlblNlYXJjaCB2ZXJzaW9uIDEuMCBvciBsYXRlci4nKTtcbiAgICAgICAgfVxuICAgICAgfVxuXG4gICAgICBpZiAodmVyc2lvbk51bSA8IDYuMCkge1xuICAgICAgICBpZiAocHJvcHMubm9kZVRvTm9kZUVuY3J5cHRpb24pIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ05vZGUtdG8tbm9kZSBlbmNyeXB0aW9uIHJlcXVpcmVzIEVsYXN0aWNzZWFyY2ggdmVyc2lvbiA2LjAgb3IgbGF0ZXIgb3IgT3BlblNlYXJjaCB2ZXJzaW9uIDEuMCBvciBsYXRlci4nKTtcbiAgICAgICAgfVxuICAgICAgfVxuXG4gICAgICBpZiAodmVyc2lvbk51bSA8IDYuNykge1xuICAgICAgICBpZiAodW5zaWduZWRCYXNpY0F1dGhFbmFibGVkKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdVc2luZyB1bnNpZ25lZCBiYXNpYyBhdXRoIHJlcXVpcmVzIEVsYXN0aWNzZWFyY2ggdmVyc2lvbiA2Ljcgb3IgbGF0ZXIgb3IgT3BlblNlYXJjaCB2ZXJzaW9uIDEuMCBvciBsYXRlci4nKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAoYWR2YW5jZWRTZWN1cml0eUVuYWJsZWQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0ZpbmUtZ3JhaW5lZCBhY2Nlc3MgY29udHJvbCByZXF1aXJlcyBFbGFzdGljc2VhcmNoIHZlcnNpb24gNi43IG9yIGxhdGVyIG9yIE9wZW5TZWFyY2ggdmVyc2lvbiAxLjAgb3IgbGF0ZXIuJyk7XG4gICAgICAgIH1cbiAgICAgIH1cblxuICAgICAgaWYgKHZlcnNpb25OdW0gPCA2LjggJiYgd2FybUVuYWJsZWQpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdVbHRyYVdhcm0gcmVxdWlyZXMgRWxhc3RpY3NlYXJjaCB2ZXJzaW9uIDYuOCBvciBsYXRlciBvciBPcGVuU2VhcmNoIHZlcnNpb24gMS4wIG9yIGxhdGVyLicpO1xuICAgICAgfVxuICAgIH1cblxuICAgIC8vIFZhbGlkYXRlIGFnYWluc3QgaW5zdGFuY2UgdHlwZSByZXN0cmljdGlvbnMsIHBlclxuICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcGVuc2VhcmNoLXNlcnZpY2UvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3N1cHBvcnRlZC1pbnN0YW5jZS10eXBlcy5odG1sXG4gICAgaWYgKGlzU29tZUluc3RhbmNlVHlwZSgnaTMnLCAncjZnZCcpICYmIGVic0VuYWJsZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignSTMgYW5kIFI2R0QgaW5zdGFuY2UgdHlwZXMgZG8gbm90IHN1cHBvcnQgRUJTIHN0b3JhZ2Ugdm9sdW1lcy4nKTtcbiAgICB9XG5cbiAgICBpZiAoaXNTb21lSW5zdGFuY2VUeXBlKCdtMycsICdyMycsICd0MicpICYmIGVuY3J5cHRpb25BdFJlc3RFbmFibGVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ00zLCBSMywgYW5kIFQyIGluc3RhbmNlIHR5cGVzIGRvIG5vdCBzdXBwb3J0IGVuY3J5cHRpb24gb2YgZGF0YSBhdCByZXN0LicpO1xuICAgIH1cblxuICAgIGlmIChpc0luc3RhbmNlVHlwZSgndDIubWljcm8nKSAmJiAhKGlzRWxhc3RpY3NlYXJjaFZlcnNpb24gJiYgdmVyc2lvbk51bSA8PSAyLjMpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1RoZSB0Mi5taWNyby5zZWFyY2ggaW5zdGFuY2UgdHlwZSBzdXBwb3J0cyBvbmx5IEVsYXN0aWNzZWFyY2ggdmVyc2lvbnMgMS41IGFuZCAyLjMuJyk7XG4gICAgfVxuXG4gICAgaWYgKGlzU29tZUluc3RhbmNlVHlwZSgndDInLCAndDMnKSAmJiB3YXJtRW5hYmxlZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdUMiBhbmQgVDMgaW5zdGFuY2UgdHlwZXMgZG8gbm90IHN1cHBvcnQgVWx0cmFXYXJtIHN0b3JhZ2UuJyk7XG4gICAgfVxuXG4gICAgLy8gT25seSBSMywgSTMgYW5kIHI2Z2Qgc3VwcG9ydCBpbnN0YW5jZSBzdG9yYWdlLCBwZXJcbiAgICAvLyBodHRwczovL2F3cy5hbWF6b24uY29tL29wZW5zZWFyY2gtc2VydmljZS9wcmljaW5nL1xuICAgIGlmICghZWJzRW5hYmxlZCAmJiAhaXNFdmVyeURhdGFub2RlSW5zdGFuY2VUeXBlKCdyMycsICdpMycsICdyNmdkJykpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignRUJTIHZvbHVtZXMgYXJlIHJlcXVpcmVkIHdoZW4gdXNpbmcgaW5zdGFuY2UgdHlwZXMgb3RoZXIgdGhhbiByMywgaTMgb3IgcjZnZC4nKTtcbiAgICB9XG5cbiAgICAvLyBGaW5lLWdyYWluZWQgYWNjZXNzIGNvbnRyb2wgcmVxdWlyZXMgbm9kZS10by1ub2RlIGVuY3J5cHRpb24sIGVuY3J5cHRpb24gYXQgcmVzdCxcbiAgICAvLyBhbmQgZW5mb3JjZWQgSFRUUFMuXG4gICAgaWYgKGFkdmFuY2VkU2VjdXJpdHlFbmFibGVkKSB7XG4gICAgICBpZiAoIW5vZGVUb05vZGVFbmNyeXB0aW9uRW5hYmxlZCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ05vZGUtdG8tbm9kZSBlbmNyeXB0aW9uIGlzIHJlcXVpcmVkIHdoZW4gZmluZS1ncmFpbmVkIGFjY2VzcyBjb250cm9sIGlzIGVuYWJsZWQuJyk7XG4gICAgICB9XG4gICAgICBpZiAoIWVuY3J5cHRpb25BdFJlc3RFbmFibGVkKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignRW5jcnlwdGlvbi1hdC1yZXN0IGlzIHJlcXVpcmVkIHdoZW4gZmluZS1ncmFpbmVkIGFjY2VzcyBjb250cm9sIGlzIGVuYWJsZWQuJyk7XG4gICAgICB9XG4gICAgICBpZiAoIWVuZm9yY2VIdHRwcykge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0VuZm9yY2UgSFRUUFMgaXMgcmVxdWlyZWQgd2hlbiBmaW5lLWdyYWluZWQgYWNjZXNzIGNvbnRyb2wgaXMgZW5hYmxlZC4nKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICAvLyBWYWxpZGF0ZSBmaW5lIGdyYWluZWQgYWNjZXNzIGNvbnRyb2wgZW5hYmxlZCBmb3IgYXVkaXQgbG9ncywgcGVyXG4gICAgLy8gaHR0cHM6Ly9hd3MuYW1hem9uLmNvbS9hYm91dC1hd3Mvd2hhdHMtbmV3LzIwMjAvMDkvZWxhc3RpY3NlYXJjaC1hdWRpdC1sb2dzLW5vdy1hdmFpbGFibGUtb24tYW1hem9uLWVsYXN0aWNzZWFyY2gtc2VydmljZS9cbiAgICBpZiAocHJvcHMubG9nZ2luZz8uYXVkaXRMb2dFbmFibGVkICYmICFhZHZhbmNlZFNlY3VyaXR5RW5hYmxlZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdGaW5lLWdyYWluZWQgYWNjZXNzIGNvbnRyb2wgaXMgcmVxdWlyZWQgd2hlbiBhdWRpdCBsb2dzIHB1Ymxpc2hpbmcgaXMgZW5hYmxlZC4nKTtcbiAgICB9XG5cbiAgICAvLyBWYWxpZGF0ZSBVbHRyYVdhcm0gcmVxdWlyZW1lbnQgZm9yIGRlZGljYXRlZCBtYXN0ZXIgbm9kZXMsIHBlclxuICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9vcGVuc2VhcmNoLXNlcnZpY2UvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3VsdHJhd2FybS5odG1sXG4gICAgaWYgKHdhcm1FbmFibGVkICYmICFkZWRpY2F0ZWRNYXN0ZXJFbmFibGVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0RlZGljYXRlZCBtYXN0ZXIgbm9kZSBpcyByZXF1aXJlZCB3aGVuIFVsdHJhV2FybSBzdG9yYWdlIGlzIGVuYWJsZWQuJyk7XG4gICAgfVxuXG4gICAgbGV0IGNmblZwY09wdGlvbnM6IENmbkRvbWFpbi5WUENPcHRpb25zUHJvcGVydHkgfCB1bmRlZmluZWQ7XG5cbiAgICBpZiAoc2VjdXJpdHlHcm91cHMgJiYgc3VibmV0cykge1xuICAgICAgY2ZuVnBjT3B0aW9ucyA9IHtcbiAgICAgICAgc2VjdXJpdHlHcm91cElkczogc2VjdXJpdHlHcm91cHMubWFwKChzZykgPT4gc2cuc2VjdXJpdHlHcm91cElkKSxcbiAgICAgICAgc3VibmV0SWRzOiBzdWJuZXRzLm1hcCgoc3VibmV0KSA9PiBzdWJuZXQuc3VibmV0SWQpLFxuICAgICAgfTtcbiAgICB9XG5cbiAgICAvLyBTZXR1cCBsb2dnaW5nXG4gICAgY29uc3QgbG9nR3JvdXBzOiBsb2dzLklMb2dHcm91cFtdID0gW107XG5cbiAgICBpZiAocHJvcHMubG9nZ2luZz8uc2xvd1NlYXJjaExvZ0VuYWJsZWQpIHtcbiAgICAgIHRoaXMuc2xvd1NlYXJjaExvZ0dyb3VwID0gcHJvcHMubG9nZ2luZy5zbG93U2VhcmNoTG9nR3JvdXAgPz9cbiAgICAgICAgbmV3IGxvZ3MuTG9nR3JvdXAodGhpcywgJ1Nsb3dTZWFyY2hMb2dzJywge1xuICAgICAgICAgIHJldGVudGlvbjogbG9ncy5SZXRlbnRpb25EYXlzLk9ORV9NT05USCxcbiAgICAgICAgfSk7XG5cbiAgICAgIGxvZ0dyb3Vwcy5wdXNoKHRoaXMuc2xvd1NlYXJjaExvZ0dyb3VwKTtcbiAgICB9O1xuXG4gICAgaWYgKHByb3BzLmxvZ2dpbmc/LnNsb3dJbmRleExvZ0VuYWJsZWQpIHtcbiAgICAgIHRoaXMuc2xvd0luZGV4TG9nR3JvdXAgPSBwcm9wcy5sb2dnaW5nLnNsb3dJbmRleExvZ0dyb3VwID8/XG4gICAgICAgIG5ldyBsb2dzLkxvZ0dyb3VwKHRoaXMsICdTbG93SW5kZXhMb2dzJywge1xuICAgICAgICAgIHJldGVudGlvbjogbG9ncy5SZXRlbnRpb25EYXlzLk9ORV9NT05USCxcbiAgICAgICAgfSk7XG5cbiAgICAgIGxvZ0dyb3Vwcy5wdXNoKHRoaXMuc2xvd0luZGV4TG9nR3JvdXApO1xuICAgIH07XG5cbiAgICBpZiAocHJvcHMubG9nZ2luZz8uYXBwTG9nRW5hYmxlZCkge1xuICAgICAgdGhpcy5hcHBMb2dHcm91cCA9IHByb3BzLmxvZ2dpbmcuYXBwTG9nR3JvdXAgPz9cbiAgICAgICAgbmV3IGxvZ3MuTG9nR3JvdXAodGhpcywgJ0FwcExvZ3MnLCB7XG4gICAgICAgICAgcmV0ZW50aW9uOiBsb2dzLlJldGVudGlvbkRheXMuT05FX01PTlRILFxuICAgICAgICB9KTtcblxuICAgICAgbG9nR3JvdXBzLnB1c2godGhpcy5hcHBMb2dHcm91cCk7XG4gICAgfTtcblxuICAgIGlmIChwcm9wcy5sb2dnaW5nPy5hdWRpdExvZ0VuYWJsZWQpIHtcbiAgICAgIHRoaXMuYXVkaXRMb2dHcm91cCA9IHByb3BzLmxvZ2dpbmcuYXVkaXRMb2dHcm91cCA/P1xuICAgICAgICBuZXcgbG9ncy5Mb2dHcm91cCh0aGlzLCAnQXVkaXRMb2dzJywge1xuICAgICAgICAgIHJldGVudGlvbjogbG9ncy5SZXRlbnRpb25EYXlzLk9ORV9NT05USCxcbiAgICAgICAgfSk7XG5cbiAgICAgIGxvZ0dyb3Vwcy5wdXNoKHRoaXMuYXVkaXRMb2dHcm91cCk7XG4gICAgfTtcblxuICAgIGxldCBsb2dHcm91cFJlc291cmNlUG9saWN5OiBMb2dHcm91cFJlc291cmNlUG9saWN5IHwgbnVsbCA9IG51bGw7XG4gICAgaWYgKGxvZ0dyb3Vwcy5sZW5ndGggPiAwKSB7XG4gICAgICBjb25zdCBsb2dQb2xpY3lTdGF0ZW1lbnQgPSBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgYWN0aW9uczogWydsb2dzOlB1dExvZ0V2ZW50cycsICdsb2dzOkNyZWF0ZUxvZ1N0cmVhbSddLFxuICAgICAgICByZXNvdXJjZXM6IGxvZ0dyb3Vwcy5tYXAoKGxnKSA9PiBsZy5sb2dHcm91cEFybiksXG4gICAgICAgIHByaW5jaXBhbHM6IFtuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2VzLmFtYXpvbmF3cy5jb20nKV0sXG4gICAgICB9KTtcblxuICAgICAgLy8gVXNlIGEgY3VzdG9tIHJlc291cmNlIHRvIHNldCB0aGUgbG9nIGdyb3VwIHJlc291cmNlIHBvbGljeSBzaW5jZSBpdCBpcyBub3Qgc3VwcG9ydGVkIGJ5IENESyBhbmQgY2ZuLlxuICAgICAgLy8gaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtY2RrL2lzc3Vlcy81MzQzXG4gICAgICBsb2dHcm91cFJlc291cmNlUG9saWN5ID0gbmV3IExvZ0dyb3VwUmVzb3VyY2VQb2xpY3kodGhpcywgYEVTTG9nR3JvdXBQb2xpY3kke3RoaXMubm9kZS5hZGRyfWAsIHtcbiAgICAgICAgLy8gY3JlYXRlIGEgY2xvdWR3YXRjaCBsb2dzIHJlc291cmNlIHBvbGljeSBuYW1lIHRoYXQgaXMgdW5pcXVlIHRvIHRoaXMgZG9tYWluIGluc3RhbmNlXG4gICAgICAgIHBvbGljeU5hbWU6IGBFU0xvZ1BvbGljeSR7dGhpcy5ub2RlLmFkZHJ9YCxcbiAgICAgICAgcG9saWN5U3RhdGVtZW50czogW2xvZ1BvbGljeVN0YXRlbWVudF0sXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBjb25zdCBsb2dQdWJsaXNoaW5nOiBSZWNvcmQ8c3RyaW5nLCBhbnk+ID0ge307XG5cbiAgICBpZiAodGhpcy5hcHBMb2dHcm91cCkge1xuICAgICAgbG9nUHVibGlzaGluZy5FU19BUFBMSUNBVElPTl9MT0dTID0ge1xuICAgICAgICBlbmFibGVkOiB0cnVlLFxuICAgICAgICBjbG91ZFdhdGNoTG9nc0xvZ0dyb3VwQXJuOiB0aGlzLmFwcExvZ0dyb3VwLmxvZ0dyb3VwQXJuLFxuICAgICAgfTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5zbG93U2VhcmNoTG9nR3JvdXApIHtcbiAgICAgIGxvZ1B1Ymxpc2hpbmcuU0VBUkNIX1NMT1dfTE9HUyA9IHtcbiAgICAgICAgZW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgY2xvdWRXYXRjaExvZ3NMb2dHcm91cEFybjogdGhpcy5zbG93U2VhcmNoTG9nR3JvdXAubG9nR3JvdXBBcm4sXG4gICAgICB9O1xuICAgIH1cblxuICAgIGlmICh0aGlzLnNsb3dJbmRleExvZ0dyb3VwKSB7XG4gICAgICBsb2dQdWJsaXNoaW5nLklOREVYX1NMT1dfTE9HUyA9IHtcbiAgICAgICAgZW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgY2xvdWRXYXRjaExvZ3NMb2dHcm91cEFybjogdGhpcy5zbG93SW5kZXhMb2dHcm91cC5sb2dHcm91cEFybixcbiAgICAgIH07XG4gICAgfVxuXG4gICAgaWYgKHRoaXMuYXVkaXRMb2dHcm91cCkge1xuICAgICAgbG9nUHVibGlzaGluZy5BVURJVF9MT0dTID0ge1xuICAgICAgICBlbmFibGVkOiB0aGlzLmF1ZGl0TG9nR3JvdXAgIT0gbnVsbCxcbiAgICAgICAgY2xvdWRXYXRjaExvZ3NMb2dHcm91cEFybjogdGhpcy5hdWRpdExvZ0dyb3VwPy5sb2dHcm91cEFybixcbiAgICAgIH07XG4gICAgfVxuXG4gICAgbGV0IGN1c3RvbUVuZHBvaW50Q2VydGlmaWNhdGU6IGFjbS5JQ2VydGlmaWNhdGUgfCB1bmRlZmluZWQ7XG4gICAgaWYgKHByb3BzLmN1c3RvbUVuZHBvaW50KSB7XG4gICAgICBpZiAocHJvcHMuY3VzdG9tRW5kcG9pbnQuY2VydGlmaWNhdGUpIHtcbiAgICAgICAgY3VzdG9tRW5kcG9pbnRDZXJ0aWZpY2F0ZSA9IHByb3BzLmN1c3RvbUVuZHBvaW50LmNlcnRpZmljYXRlO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgY3VzdG9tRW5kcG9pbnRDZXJ0aWZpY2F0ZSA9IG5ldyBhY20uQ2VydGlmaWNhdGUodGhpcywgJ0N1c3RvbUVuZHBvaW50Q2VydGlmaWNhdGUnLCB7XG4gICAgICAgICAgZG9tYWluTmFtZTogcHJvcHMuY3VzdG9tRW5kcG9pbnQuZG9tYWluTmFtZSxcbiAgICAgICAgICB2YWxpZGF0aW9uOiBwcm9wcy5jdXN0b21FbmRwb2ludC5ob3N0ZWRab25lID8gYWNtLkNlcnRpZmljYXRlVmFsaWRhdGlvbi5mcm9tRG5zKHByb3BzLmN1c3RvbUVuZHBvaW50Lmhvc3RlZFpvbmUpIDogdW5kZWZpbmVkLFxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICAvLyBDcmVhdGUgdGhlIGRvbWFpblxuICAgIHRoaXMuZG9tYWluID0gbmV3IENmbkRvbWFpbih0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBkb21haW5OYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICAgIGVuZ2luZVZlcnNpb246IHByb3BzLnZlcnNpb24udmVyc2lvbixcbiAgICAgIGNsdXN0ZXJDb25maWc6IHtcbiAgICAgICAgZGVkaWNhdGVkTWFzdGVyRW5hYmxlZCxcbiAgICAgICAgZGVkaWNhdGVkTWFzdGVyQ291bnQ6IGRlZGljYXRlZE1hc3RlckVuYWJsZWRcbiAgICAgICAgICA/IGRlZGljYXRlZE1hc3RlckNvdW50XG4gICAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICAgIGRlZGljYXRlZE1hc3RlclR5cGU6IGRlZGljYXRlZE1hc3RlckVuYWJsZWRcbiAgICAgICAgICA/IGRlZGljYXRlZE1hc3RlclR5cGVcbiAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgICAgaW5zdGFuY2VDb3VudCxcbiAgICAgICAgaW5zdGFuY2VUeXBlLFxuICAgICAgICB3YXJtRW5hYmxlZDogd2FybUVuYWJsZWRcbiAgICAgICAgICA/IHdhcm1FbmFibGVkXG4gICAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICAgIHdhcm1Db3VudDogd2FybUVuYWJsZWRcbiAgICAgICAgICA/IHdhcm1Db3VudFxuICAgICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgICB3YXJtVHlwZTogd2FybUVuYWJsZWRcbiAgICAgICAgICA/IHdhcm1UeXBlXG4gICAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICAgIHpvbmVBd2FyZW5lc3NFbmFibGVkLFxuICAgICAgICB6b25lQXdhcmVuZXNzQ29uZmlnOiB6b25lQXdhcmVuZXNzRW5hYmxlZFxuICAgICAgICAgID8geyBhdmFpbGFiaWxpdHlab25lQ291bnQgfVxuICAgICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgfSxcbiAgICAgIGVic09wdGlvbnM6IHtcbiAgICAgICAgZWJzRW5hYmxlZCxcbiAgICAgICAgdm9sdW1lU2l6ZTogZWJzRW5hYmxlZCA/IHZvbHVtZVNpemUgOiB1bmRlZmluZWQsXG4gICAgICAgIHZvbHVtZVR5cGU6IGVic0VuYWJsZWQgPyB2b2x1bWVUeXBlIDogdW5kZWZpbmVkLFxuICAgICAgICBpb3BzOiBlYnNFbmFibGVkID8gcHJvcHMuZWJzPy5pb3BzIDogdW5kZWZpbmVkLFxuICAgICAgfSxcbiAgICAgIGVuY3J5cHRpb25BdFJlc3RPcHRpb25zOiB7XG4gICAgICAgIGVuYWJsZWQ6IGVuY3J5cHRpb25BdFJlc3RFbmFibGVkLFxuICAgICAgICBrbXNLZXlJZDogZW5jcnlwdGlvbkF0UmVzdEVuYWJsZWRcbiAgICAgICAgICA/IHByb3BzLmVuY3J5cHRpb25BdFJlc3Q/Lmttc0tleT8ua2V5SWRcbiAgICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgIH0sXG4gICAgICBub2RlVG9Ob2RlRW5jcnlwdGlvbk9wdGlvbnM6IHsgZW5hYmxlZDogbm9kZVRvTm9kZUVuY3J5cHRpb25FbmFibGVkIH0sXG4gICAgICBsb2dQdWJsaXNoaW5nT3B0aW9uczogbG9nUHVibGlzaGluZyxcbiAgICAgIGNvZ25pdG9PcHRpb25zOiB7XG4gICAgICAgIGVuYWJsZWQ6IHByb3BzLmNvZ25pdG9EYXNoYm9hcmRzQXV0aCAhPSBudWxsLFxuICAgICAgICBpZGVudGl0eVBvb2xJZDogcHJvcHMuY29nbml0b0Rhc2hib2FyZHNBdXRoPy5pZGVudGl0eVBvb2xJZCxcbiAgICAgICAgcm9sZUFybjogcHJvcHMuY29nbml0b0Rhc2hib2FyZHNBdXRoPy5yb2xlLnJvbGVBcm4sXG4gICAgICAgIHVzZXJQb29sSWQ6IHByb3BzLmNvZ25pdG9EYXNoYm9hcmRzQXV0aD8udXNlclBvb2xJZCxcbiAgICAgIH0sXG4gICAgICB2cGNPcHRpb25zOiBjZm5WcGNPcHRpb25zLFxuICAgICAgc25hcHNob3RPcHRpb25zOiBwcm9wcy5hdXRvbWF0ZWRTbmFwc2hvdFN0YXJ0SG91clxuICAgICAgICA/IHsgYXV0b21hdGVkU25hcHNob3RTdGFydEhvdXI6IHByb3BzLmF1dG9tYXRlZFNuYXBzaG90U3RhcnRIb3VyIH1cbiAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICBkb21haW5FbmRwb2ludE9wdGlvbnM6IHtcbiAgICAgICAgZW5mb3JjZUh0dHBzLFxuICAgICAgICB0bHNTZWN1cml0eVBvbGljeTogcHJvcHMudGxzU2VjdXJpdHlQb2xpY3kgPz8gVExTU2VjdXJpdHlQb2xpY3kuVExTXzFfMCxcbiAgICAgICAgLi4ucHJvcHMuY3VzdG9tRW5kcG9pbnQgJiYge1xuICAgICAgICAgIGN1c3RvbUVuZHBvaW50RW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgICBjdXN0b21FbmRwb2ludDogcHJvcHMuY3VzdG9tRW5kcG9pbnQuZG9tYWluTmFtZSxcbiAgICAgICAgICBjdXN0b21FbmRwb2ludENlcnRpZmljYXRlQXJuOiBjdXN0b21FbmRwb2ludENlcnRpZmljYXRlIS5jZXJ0aWZpY2F0ZUFybixcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICBhZHZhbmNlZFNlY3VyaXR5T3B0aW9uczogYWR2YW5jZWRTZWN1cml0eUVuYWJsZWRcbiAgICAgICAgPyB7XG4gICAgICAgICAgZW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgICBpbnRlcm5hbFVzZXJEYXRhYmFzZUVuYWJsZWQsXG4gICAgICAgICAgbWFzdGVyVXNlck9wdGlvbnM6IHtcbiAgICAgICAgICAgIG1hc3RlclVzZXJBcm46IG1hc3RlclVzZXJBcm4sXG4gICAgICAgICAgICBtYXN0ZXJVc2VyTmFtZTogbWFzdGVyVXNlck5hbWUsXG4gICAgICAgICAgICBtYXN0ZXJVc2VyUGFzc3dvcmQ6IHRoaXMubWFzdGVyVXNlclBhc3N3b3JkPy51bnNhZmVVbndyYXAoKSwgLy8gU2FmZSB1c2FnZVxuICAgICAgICAgIH0sXG4gICAgICAgIH1cbiAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICBhZHZhbmNlZE9wdGlvbnM6IHByb3BzLmFkdmFuY2VkT3B0aW9ucyxcbiAgICB9KTtcbiAgICB0aGlzLmRvbWFpbi5hcHBseVJlbW92YWxQb2xpY3kocHJvcHMucmVtb3ZhbFBvbGljeSk7XG5cbiAgICBpZiAocHJvcHMuZW5hYmxlVmVyc2lvblVwZ3JhZGUpIHtcbiAgICAgIHRoaXMuZG9tYWluLmNmbk9wdGlvbnMudXBkYXRlUG9saWN5ID0ge1xuICAgICAgICAuLi50aGlzLmRvbWFpbi5jZm5PcHRpb25zLnVwZGF0ZVBvbGljeSxcbiAgICAgICAgZW5hYmxlVmVyc2lvblVwZ3JhZGU6IHByb3BzLmVuYWJsZVZlcnNpb25VcGdyYWRlLFxuICAgICAgfTtcbiAgICB9XG5cbiAgICBpZiAobG9nR3JvdXBSZXNvdXJjZVBvbGljeSkgeyB0aGlzLmRvbWFpbi5ub2RlLmFkZERlcGVuZGVuY3kobG9nR3JvdXBSZXNvdXJjZVBvbGljeSk7IH1cblxuICAgIGlmIChwcm9wcy5kb21haW5OYW1lKSB7XG4gICAgICBpZiAoIWNkay5Ub2tlbi5pc1VucmVzb2x2ZWQocHJvcHMuZG9tYWluTmFtZSkpIHtcbiAgICAgICAgLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL29wZW5zZWFyY2gtc2VydmljZS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvY29uZmlndXJhdGlvbi1hcGkuaHRtbCNjb25maWd1cmF0aW9uLWFwaS1kYXRhdHlwZXMtZG9tYWlubmFtZVxuICAgICAgICBpZiAoIXByb3BzLmRvbWFpbk5hbWUubWF0Y2goL15bYS16MC05XFwtXSskLykpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEludmFsaWQgZG9tYWluTmFtZSAnJHtwcm9wcy5kb21haW5OYW1lfScuIFZhbGlkIGNoYXJhY3RlcnMgYXJlIGEteiAobG93ZXJjYXNlIG9ubHkpLCAwLTksIGFuZCDigJMgKGh5cGhlbikuYCk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHByb3BzLmRvbWFpbk5hbWUubGVuZ3RoIDwgMyB8fCBwcm9wcy5kb21haW5OYW1lLmxlbmd0aCA+IDI4KSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBJbnZhbGlkIGRvbWFpbk5hbWUgJyR7cHJvcHMuZG9tYWluTmFtZX0nLiBJdCBtdXN0IGJlIGJldHdlZW4gMyBhbmQgMjggY2hhcmFjdGVyc2ApO1xuICAgICAgICB9XG4gICAgICAgIGlmIChwcm9wcy5kb21haW5OYW1lWzBdIDwgJ2EnIHx8IHByb3BzLmRvbWFpbk5hbWVbMF0gPiAneicpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEludmFsaWQgZG9tYWluTmFtZSAnJHtwcm9wcy5kb21haW5OYW1lfScuIEl0IG11c3Qgc3RhcnQgd2l0aCBhIGxvd2VyY2FzZSBsZXR0ZXJgKTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgICAgdGhpcy5ub2RlLmFkZE1ldGFkYXRhKCdhd3M6Y2RrOmhhc1BoeXNpY2FsTmFtZScsIHByb3BzLmRvbWFpbk5hbWUpO1xuICAgIH1cblxuICAgIHRoaXMuZG9tYWluTmFtZSA9IHRoaXMuZ2V0UmVzb3VyY2VOYW1lQXR0cmlidXRlKHRoaXMuZG9tYWluLnJlZik7XG5cbiAgICB0aGlzLmRvbWFpbklkID0gdGhpcy5kb21haW4uZ2V0QXR0KCdJZCcpLnRvU3RyaW5nKCk7XG5cbiAgICB0aGlzLmRvbWFpbkVuZHBvaW50ID0gdGhpcy5kb21haW4uZ2V0QXR0KCdEb21haW5FbmRwb2ludCcpLnRvU3RyaW5nKCk7XG5cbiAgICB0aGlzLmRvbWFpbkFybiA9IHRoaXMuZ2V0UmVzb3VyY2VBcm5BdHRyaWJ1dGUodGhpcy5kb21haW4uYXR0ckFybiwge1xuICAgICAgc2VydmljZTogJ2VzJyxcbiAgICAgIHJlc291cmNlOiAnZG9tYWluJyxcbiAgICAgIHJlc291cmNlTmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgfSk7XG5cbiAgICBpZiAocHJvcHMuY3VzdG9tRW5kcG9pbnQ/Lmhvc3RlZFpvbmUpIHtcbiAgICAgIG5ldyByb3V0ZTUzLkNuYW1lUmVjb3JkKHRoaXMsICdDbmFtZVJlY29yZCcsIHtcbiAgICAgICAgcmVjb3JkTmFtZTogcHJvcHMuY3VzdG9tRW5kcG9pbnQuZG9tYWluTmFtZSxcbiAgICAgICAgem9uZTogcHJvcHMuY3VzdG9tRW5kcG9pbnQuaG9zdGVkWm9uZSxcbiAgICAgICAgZG9tYWluTmFtZTogdGhpcy5kb21haW5FbmRwb2ludCxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHRoaXMuZW5jcnlwdGlvbkF0UmVzdE9wdGlvbnMgPSBwcm9wcy5lbmNyeXB0aW9uQXRSZXN0O1xuICAgIGlmIChwcm9wcy5hY2Nlc3NQb2xpY2llcykge1xuICAgICAgdGhpcy5hZGRBY2Nlc3NQb2xpY2llcyguLi5wcm9wcy5hY2Nlc3NQb2xpY2llcyk7XG4gICAgfVxuICAgIGlmICh1bnNpZ25lZEJhc2ljQXV0aEVuYWJsZWQpIHtcbiAgICAgIHRoaXMuYWRkQWNjZXNzUG9saWNpZXModW5zaWduZWRBY2Nlc3NQb2xpY3kpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBNYW5hZ2VzIG5ldHdvcmsgY29ubmVjdGlvbnMgdG8gdGhlIGRvbWFpbi4gVGhpcyB3aWxsIHRocm93IGFuIGVycm9yIGluIGNhc2UgdGhlIGRvbWFpblxuICAgKiBpcyBub3QgcGxhY2VkIGluc2lkZSBhIFZQQy5cbiAgICovXG4gIHB1YmxpYyBnZXQgY29ubmVjdGlvbnMoKTogZWMyLkNvbm5lY3Rpb25zIHtcbiAgICBpZiAoIXRoaXMuX2Nvbm5lY3Rpb25zKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXCJDb25uZWN0aW9ucyBhcmUgb25seSBhdmFpbGFibGUgb24gVlBDIGVuYWJsZWQgZG9tYWlucy4gVXNlIHRoZSAndnBjJyBwcm9wZXJ0eSB0byBwbGFjZSBhIGRvbWFpbiBpbnNpZGUgYSBWUENcIik7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLl9jb25uZWN0aW9ucztcbiAgfVxuXG5cbiAgLyoqXG4gICAqIEFkZCBwb2xpY3kgc3RhdGVtZW50cyB0byB0aGUgZG9tYWluIGFjY2VzcyBwb2xpY3lcbiAgICovXG4gIHB1YmxpYyBhZGRBY2Nlc3NQb2xpY2llcyguLi5hY2Nlc3NQb2xpY3lTdGF0ZW1lbnRzOiBpYW0uUG9saWN5U3RhdGVtZW50W10pIHtcbiAgICBpZiAoYWNjZXNzUG9saWN5U3RhdGVtZW50cy5sZW5ndGggPiAwKSB7XG4gICAgICBpZiAoIXRoaXMuYWNjZXNzUG9saWN5KSB7XG4gICAgICAgIC8vIE9ubHkgY3JlYXRlIHRoZSBjdXN0b20gcmVzb3VyY2UgYWZ0ZXIgdGhlcmUgYXJlIHN0YXRlbWVudHMgdG8gc2V0LlxuICAgICAgICB0aGlzLmFjY2Vzc1BvbGljeSA9IG5ldyBPcGVuU2VhcmNoQWNjZXNzUG9saWN5KHRoaXMsICdBY2Nlc3NQb2xpY3knLCB7XG4gICAgICAgICAgZG9tYWluTmFtZTogdGhpcy5kb21haW5OYW1lLFxuICAgICAgICAgIGRvbWFpbkFybjogdGhpcy5kb21haW5Bcm4sXG4gICAgICAgICAgYWNjZXNzUG9saWNpZXM6IGFjY2Vzc1BvbGljeVN0YXRlbWVudHMsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIGlmICh0aGlzLmVuY3J5cHRpb25BdFJlc3RPcHRpb25zPy5rbXNLZXkpIHtcbiAgICAgICAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vb3BlbnNlYXJjaC1zZXJ2aWNlL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9lbmNyeXB0aW9uLWF0LXJlc3QuaHRtbFxuXG4gICAgICAgICAgLy8gdGhlc2UgcGVybWlzc2lvbnMgYXJlIGRvY3VtZW50ZWQgYXMgcmVxdWlyZWQgZHVyaW5nIGRvbWFpbiBjcmVhdGlvbi5cbiAgICAgICAgICAvLyB3aGlsZSBub3Qgc3RyaWN0bHkgZG9jdW1lbnRlZCBmb3IgdXBkYXRlcyBhcyB3ZWxsLCBpdCBzdGFuZHMgdG8gcmVhc29uIHRoYXQgYW4gdXBkYXRlXG4gICAgICAgICAgLy8gb3BlcmF0aW9uIG1pZ2h0IHJlcXVpcmUgdGhlc2UgaW4gY2FzZSB0aGUgY2x1c3RlciB1c2VzIGEga21zIGtleS5cbiAgICAgICAgICAvLyBlbXBpcmNhbCBldmlkZW5jZSBzaG93cyB0aGlzIGlzIGluZGVlZCByZXF1aXJlZDogaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtY2RrL2lzc3Vlcy8xMTQxMlxuICAgICAgICAgIHRoaXMuYWNjZXNzUG9saWN5LmdyYW50UHJpbmNpcGFsLmFkZFRvUHJpbmNpcGFsUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgIGFjdGlvbnM6IFsna21zOkxpc3QqJywgJ2ttczpEZXNjcmliZSonLCAna21zOkNyZWF0ZUdyYW50J10sXG4gICAgICAgICAgICByZXNvdXJjZXM6IFt0aGlzLmVuY3J5cHRpb25BdFJlc3RPcHRpb25zLmttc0tleS5rZXlBcm5dLFxuICAgICAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICAgIH0pKTtcbiAgICAgICAgfVxuICAgICAgfSBlbHNlIHtcbiAgICAgICAgdGhpcy5hY2Nlc3NQb2xpY3kuYWRkQWNjZXNzUG9saWNpZXMoLi4uYWNjZXNzUG9saWN5U3RhdGVtZW50cyk7XG4gICAgICB9XG4gICAgfVxuICB9XG59XG5cbi8qKlxuICogR2l2ZW4gYW4gQW1hem9uIE9wZW5TZWFyY2ggU2VydmljZSBkb21haW4gZW5kcG9pbnQsIHJldHVybnMgYSBDbG91ZEZvcm1hdGlvbiBleHByZXNzaW9uIHRoYXRcbiAqIGV4dHJhY3RzIHRoZSBkb21haW4gbmFtZS5cbiAqXG4gKiBEb21haW4gZW5kcG9pbnRzIGxvb2sgbGlrZSB0aGlzOlxuICpcbiAqICAgaHR0cHM6Ly9leGFtcGxlLWRvbWFpbi1qY2pvdHJ0NmY3b3RlbTRzcWN3YmNoM2M0dS51cy1lYXN0LTEuZXMuYW1hem9uYXdzLmNvbVxuICogICBodHRwczovLzxkb21haW4tbmFtZT4tPHN1ZmZpeD4uPHJlZ2lvbj4uZXMuYW1hem9uYXdzLmNvbVxuICpcbiAqIC4ud2hpY2ggbWVhbnMgdGhhdCBpbiBvcmRlciB0byBleHRyYWN0IHRoZSBkb21haW4gbmFtZSBmcm9tIHRoZSBlbmRwb2ludCwgd2UgY2FuXG4gKiBzcGxpdCB0aGUgZW5kcG9pbnQgdXNpbmcgXCItPHN1ZmZpeD5cIiBhbmQgc2VsZWN0IHRoZSBjb21wb25lbnQgaW4gaW5kZXggMC5cbiAqXG4gKiBAcGFyYW0gZG9tYWluRW5kcG9pbnQgVGhlIEFtYXpvbiBPcGVuU2VhcmNoIFNlcnZpY2UgZG9tYWluIGVuZHBvaW50XG4gKi9cbmZ1bmN0aW9uIGV4dHJhY3ROYW1lRnJvbUVuZHBvaW50KGRvbWFpbkVuZHBvaW50OiBzdHJpbmcpIHtcbiAgY29uc3QgeyBob3N0bmFtZSB9ID0gbmV3IFVSTChkb21haW5FbmRwb2ludCk7XG4gIGNvbnN0IGRvbWFpbiA9IGhvc3RuYW1lLnNwbGl0KCcuJylbMF07XG4gIGNvbnN0IHN1ZmZpeCA9ICctJyArIGRvbWFpbi5zcGxpdCgnLScpLnNsaWNlKC0xKVswXTtcbiAgcmV0dXJuIGRvbWFpbi5zcGxpdChzdWZmaXgpWzBdO1xufVxuXG4vKipcbiAqIENvbnZlcnRzIGFuIGVuZ2luZSB2ZXJzaW9uIGludG8gYSBpbnRvIGEgZGVjaW1hbCBudW1iZXIgd2l0aCBtYWpvciBhbmQgbWlub3IgdmVyc2lvbiBpLmUgeC55LlxuICpcbiAqIEBwYXJhbSB2ZXJzaW9uIFRoZSBlbmdpbmUgdmVyc2lvbiBvYmplY3RcbiAqL1xuZnVuY3Rpb24gcGFyc2VWZXJzaW9uKHZlcnNpb246IEVuZ2luZVZlcnNpb24pOiB7IHZlcnNpb25OdW06IG51bWJlciwgaXNFbGFzdGljc2VhcmNoVmVyc2lvbjogYm9vbGVhbiB9IHtcbiAgY29uc3QgZWxhc3RpY3NlYXJjaFByZWZpeCA9ICdFbGFzdGljc2VhcmNoXyc7XG4gIGNvbnN0IG9wZW5TZWFyY2hQcmVmaXggPSAnT3BlblNlYXJjaF8nO1xuICBjb25zdCBpc0VsYXN0aWNzZWFyY2hWZXJzaW9uID0gdmVyc2lvbi52ZXJzaW9uLnN0YXJ0c1dpdGgoZWxhc3RpY3NlYXJjaFByZWZpeCk7XG4gIGNvbnN0IHZlcnNpb25TdHIgPSBpc0VsYXN0aWNzZWFyY2hWZXJzaW9uXG4gICAgPyB2ZXJzaW9uLnZlcnNpb24uc3Vic3RyaW5nKGVsYXN0aWNzZWFyY2hQcmVmaXgubGVuZ3RoKVxuICAgIDogdmVyc2lvbi52ZXJzaW9uLnN1YnN0cmluZyhvcGVuU2VhcmNoUHJlZml4Lmxlbmd0aCk7XG4gIGNvbnN0IGZpcnN0RG90ID0gdmVyc2lvblN0ci5pbmRleE9mKCcuJyk7XG5cbiAgaWYgKGZpcnN0RG90IDwgMSkge1xuICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBlbmdpbmUgdmVyc2lvbjogJHt2ZXJzaW9uU3RyfS4gVmVyc2lvbiBzdHJpbmcgbmVlZHMgdG8gc3RhcnQgd2l0aCBtYWpvciBhbmQgbWlub3IgdmVyc2lvbiAoeC55KS5gKTtcbiAgfVxuXG4gIGNvbnN0IHNlY29uZERvdCA9IHZlcnNpb25TdHIuaW5kZXhPZignLicsIGZpcnN0RG90ICsgMSk7XG5cbiAgdHJ5IHtcbiAgICBpZiAoc2Vjb25kRG90ID09IC0xKSB7XG4gICAgICByZXR1cm4geyB2ZXJzaW9uTnVtOiBwYXJzZUZsb2F0KHZlcnNpb25TdHIpLCBpc0VsYXN0aWNzZWFyY2hWZXJzaW9uIH07XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiB7IHZlcnNpb25OdW06IHBhcnNlRmxvYXQodmVyc2lvblN0ci5zdWJzdHJpbmcoMCwgc2Vjb25kRG90KSksIGlzRWxhc3RpY3NlYXJjaFZlcnNpb24gfTtcbiAgICB9XG4gIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGBJbnZhbGlkIGVuZ2luZSB2ZXJzaW9uOiAke3ZlcnNpb25TdHJ9LiBWZXJzaW9uIHN0cmluZyBuZWVkcyB0byBzdGFydCB3aXRoIG1ham9yIGFuZCBtaW5vciB2ZXJzaW9uICh4LnkpLmApO1xuICB9XG59XG5cbmZ1bmN0aW9uIHNlbGVjdFN1Ym5ldHModnBjOiBlYzIuSVZwYywgdnBjU3VibmV0czogZWMyLlN1Ym5ldFNlbGVjdGlvbltdKTogZWMyLklTdWJuZXRbXSB7XG4gIGNvbnN0IHNlbGVjdGVkID0gW107XG4gIGZvciAoY29uc3Qgc2VsZWN0aW9uIG9mIHZwY1N1Ym5ldHMpIHtcbiAgICBzZWxlY3RlZC5wdXNoKC4uLnZwYy5zZWxlY3RTdWJuZXRzKHNlbGVjdGlvbikuc3VibmV0cyk7XG4gIH1cbiAgcmV0dXJuIHNlbGVjdGVkO1xufVxuXG4vKipcbiAqIEluaXRpYWxpemVzIGFuIGluc3RhbmNlIHR5cGUuXG4gKlxuICogQHBhcmFtIGRlZmF1bHRJbnN0YW5jZVR5cGUgRGVmYXVsdCBpbnN0YW5jZSB0eXBlIHdoaWNoIGlzIHVzZWQgaWYgbm8gaW5zdGFuY2UgdHlwZSBpcyBwcm92aWRlZFxuICogQHBhcmFtIGluc3RhbmNlVHlwZSBJbnN0YW5jZSB0eXBlXG4gKiBAcmV0dXJucyBJbnN0YW5jZSB0eXBlIGluIGxvd2VyY2FzZSAoaWYgcHJvdmlkZWQpIG9yIGRlZmF1bHQgaW5zdGFuY2UgdHlwZVxuICovXG5mdW5jdGlvbiBpbml0aWFsaXplSW5zdGFuY2VUeXBlKGRlZmF1bHRJbnN0YW5jZVR5cGU6IHN0cmluZywgaW5zdGFuY2VUeXBlPzogc3RyaW5nKTogc3RyaW5nIHtcbiAgaWYgKGluc3RhbmNlVHlwZSkge1xuICAgIHJldHVybiBjZGsuVG9rZW4uaXNVbnJlc29sdmVkKGluc3RhbmNlVHlwZSkgPyBpbnN0YW5jZVR5cGUgOiBpbnN0YW5jZVR5cGUudG9Mb3dlckNhc2UoKTtcbiAgfSBlbHNlIHtcbiAgICByZXR1cm4gZGVmYXVsdEluc3RhbmNlVHlwZTtcbiAgfVxufVxuIl19