import re
from typing import Type

import aiohttp
from marshmallow import ValidationError

from .config import ConfigSchema
from .consts import Joined
from .exceptions import ConfigurationException, NoAuthenticationMethod, UnexpectedSchema
from .resource import QueryBuilder, Resource, Schema, select


def load_config(config_data):
    return ConfigSchema().load(config_data)


class Application:
    """Snow Application

    The Application class serves a number of purposes:
        - Config validation and transformation
        - Resource factory
        - ClientSession factory

    Args:
        config_data: Config dictionary

    Attributes:
        config (ConfigSchema): Application configuration object
    """

    def __init__(self, config_data):
        try:
            self.config = load_config(config_data)
        except ValidationError as e:
            raise ConfigurationException(e)

    @property
    def _auth(self):
        """Get authentication object built using config

        Returns:
            aiohttp-compatible authentication object
        """

        if self.config.basic_auth:
            return aiohttp.BasicAuth(*self.config.basic_auth)
        else:
            raise NoAuthenticationMethod("No known authentication methods was provided")

    def get_session(self):
        """New client session

        Returns:
            aiohttp.ClientSession:  HTTP client session

        Raises:
            NoAuthenticationMethod
        """

        connector_args = {}

        if self.config.use_ssl:
            connector_args["verify_ssl"] = self.config.verify_ssl

        return aiohttp.ClientSession(
            auth=self._auth, connector=aiohttp.TCPConnector(**connector_args),
        )

    def resource(self, schema: Type[Schema]) -> Resource:
        """Snow Resource factory

        Args:
            schema (Schema): Resource Schema

        Returns:
            Resource: Resource object

        Raises:
            UnexpectedSchema
        """

        if not issubclass(schema, Schema):
            raise UnexpectedSchema(
                f"Invalid schema class: {schema}, must be of type {Schema}"
            )
        if not re.match(r"^/.*", str(schema.__location__)):
            raise UnexpectedSchema(
                f"Unexpected path in {schema.__name__}.__location__: {schema.__location__}"
            )

        return Resource(schema, self)
