"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseFargateService = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const events = require("@aws-cdk/aws-events");
const event_targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const common_functions_1 = require("./common/common-functions");
/**
 * @stability stable
 */
class BaseFargateService extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e;
        super(scope, id);
        /**
         * @stability stable
         */
        this.zoneName = '';
        /**
         * @stability stable
         */
        this.hasExternalLoadBalancer = false;
        /**
         * @stability stable
         */
        this.hasInternalLoadBalancer = false;
        /**
         * @stability stable
         */
        this.vpcSubnets = { subnetType: ec2.SubnetType.PRIVATE };
        this.hasSpotCapacity = false;
        /**
         * determine if vpcSubnets are all public ones
         */
        this.isPublicSubnets = false;
        this.enableLoadBalancerAlias = ((_b = props.route53Ops) === null || _b === void 0 ? void 0 : _b.enableLoadBalancerAlias) != false;
        this.vpc = (_c = props.vpc) !== null && _c !== void 0 ? _c : common_functions_1.getOrCreateVpc(this),
            this.service = [];
        if (props.vpcSubnets) {
            this.vpcSubnets = props.vpcSubnets;
            this.validateSubnets(this.vpc, this.vpcSubnets);
        }
        // determine whether we need the external LB
        props.tasks.forEach(t => {
            // determine the accessibility
            if (t.external) {
                this.hasExternalLoadBalancer = true;
            }
            if (t.internal) {
                this.hasInternalLoadBalancer = true;
            }
        });
        const cluster = new ecs.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            enableFargateCapacityProviders: true,
        });
        const spotOnlyStrategy = [
            {
                capacityProvider: 'FARGATE_SPOT',
                base: 0,
                weight: 1,
            },
            {
                capacityProvider: 'FARGATE',
                base: 0,
                weight: 0,
            },
        ];
        if (props.spot == true)
            this.hasSpotCapacity = true;
        props.tasks.forEach(t => {
            var _b, _c, _d, _e;
            const defaultContainerName = (_b = t.task.defaultContainer) === null || _b === void 0 ? void 0 : _b.containerName;
            const svc = new ecs.FargateService(this, `${defaultContainerName}Service`, {
                taskDefinition: t.task,
                cluster,
                capacityProviderStrategies: (_c = t.capacityProviderStrategy) !== null && _c !== void 0 ? _c : (props.spot ? spotOnlyStrategy : undefined),
                desiredCount: t.desiredCount,
                enableExecuteCommand: (_d = props.enableExecuteCommand) !== null && _d !== void 0 ? _d : false,
                vpcSubnets: this.vpcSubnets,
                assignPublicIp: this.isPublicSubnets,
                circuitBreaker: props.circuitBreaker != false ? {
                    rollback: true,
                } : undefined,
            });
            this.service.push(svc);
            /**
             * determine if we have spot capacity in this cluster
             * scenario 1: FARGATE_SPOT with weight > 0
             * scenario 2: FARGATE_SPOT with base > 0
             * scenario 3: props.spot = true
             */
            (_e = t.capacityProviderStrategy) === null || _e === void 0 ? void 0 : _e.forEach(s => {
                if (s.capacityProvider == 'FARGATE_SPOT' && ((s.weight && s.weight > 0)
                    || (s.base && s.base > 0))) {
                    this.hasSpotCapacity = true;
                }
            });
        });
        // ensure the dependency
        const cp = this.node.tryFindChild('Cluster');
        this.service.forEach(s => {
            s.node.addDependency(cp);
        });
        // Route53
        this.zoneName = (_e = (_d = props.route53Ops) === null || _d === void 0 ? void 0 : _d.zoneName) !== null && _e !== void 0 ? _e : 'svc.local';
        // spot termination handler by default
        if (this.hasSpotCapacity && props.spotTerminationHandler !== false) {
            this.createSpotTerminationHandler(cluster);
        }
    }
    createSpotTerminationHandler(cluster) {
        // create the handler
        const handler = new lambda.DockerImageFunction(this, 'SpotTermHandler', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/spot-term-handler')),
            timeout: cdk.Duration.seconds(20),
        });
        // create event rule
        const rule = new events.Rule(this, 'OnTaskStateChangeEvent', {
            eventPattern: {
                source: ['aws.ecs'],
                detailType: ['ECS Task State Change'],
                detail: {
                    clusterArn: [cluster.clusterArn],
                },
            },
        });
        rule.addTarget(new event_targets.LambdaFunction(handler));
        handler.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:DescribeServices',
                'elasticloadbalancing:DeregisterTargets',
                'ec2:DescribeSubnets',
            ],
            resources: ['*'],
        }));
    }
    validateSubnets(vpc, vpcSubnets) {
        const subnets = vpc.selectSubnets(vpcSubnets);
        // get all subnets in the VPC
        const allsubnetIds = vpc.publicSubnets.concat(vpc.privateSubnets).concat(vpc.isolatedSubnets).map(x => x.subnetId);
        // validate the given subnets
        subnets.subnetIds.forEach(s => {
            if (!allsubnetIds.includes(s)) {
                throw new Error(`${s} does not exist in the VPC`);
            }
            if (vpc.isolatedSubnets.map(i => i.subnetId).includes(s)) {
                throw new Error(`Isolated subnet ${s} is not allowed`);
            }
        });
        const hasPublic = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
        const hasPrivate = subnets.subnetIds.some(s => new Set(vpc.privateSubnets.map(x => x.subnetId)).has(s));
        if (hasPublic && hasPrivate) {
            throw new Error('You should provide either all public or all private subnets, not both.');
        }
        this.isPublicSubnets = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
    }
}
exports.BaseFargateService = BaseFargateService;
_a = JSII_RTTI_SYMBOL_1;
BaseFargateService[_a] = { fqn: "cdk-fargate-patterns.BaseFargateService", version: "0.3.2" };
//# sourceMappingURL=data:application/json;base64,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