"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.clusterArnComponents = exports.ClusterResource = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * A low-level CFN resource Amazon EKS cluster implemented through a custom
 * resource.
 *
 * Implements EKS create/update/delete through a CloudFormation custom resource
 * in order to allow us to control the IAM role which creates the cluster. This
 * is required in order to be able to allow CloudFormation to interact with the
 * cluster via `kubectl` to enable Kubernetes management capabilities like apply
 * manifest and IAM role/user RBAC mapping.
 */
class ClusterResource extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.roleArn) {
            throw new Error('"roleArn" is required');
        }
        this.adminRole = this.createAdminRole(props);
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this, {
            adminRole: this.adminRole,
            subnets: props.subnets,
            vpc: props.vpc,
            environment: props.environment,
            onEventLayer: props.onEventLayer,
            securityGroup: props.clusterHandlerSecurityGroup,
        });
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: consts_1.CLUSTER_RESOURCE_TYPE,
            serviceToken: provider.serviceToken,
            properties: {
                // the structure of config needs to be that of 'aws.EKS.CreateClusterRequest' since its passed as is
                // to the eks.createCluster sdk invocation.
                Config: {
                    name: props.name,
                    version: props.version,
                    roleArn: props.roleArn,
                    encryptionConfig: props.encryptionConfig,
                    kubernetesNetworkConfig: props.kubernetesNetworkConfig,
                    resourcesVpcConfig: {
                        subnetIds: props.resourcesVpcConfig.subnetIds,
                        securityGroupIds: props.resourcesVpcConfig.securityGroupIds,
                        endpointPublicAccess: props.endpointPublicAccess,
                        endpointPrivateAccess: props.endpointPrivateAccess,
                        publicAccessCidrs: props.publicAccessCidrs,
                    },
                    tags: props.tags,
                },
                AssumeRoleArn: this.adminRole.roleArn,
                // IMPORTANT: increment this number when you add new attributes to the
                // resource. Otherwise, CloudFormation will error with "Vendor response
                // doesn't contain XXX key in object" (see #8276) by incrementing this
                // number, you will effectively cause a "no-op update" to the cluster
                // which will return the new set of attribute.
                AttributesRevision: 2,
            },
        });
        resource.node.addDependency(this.adminRole);
        this.ref = resource.ref;
        this.attrEndpoint = core_1.Token.asString(resource.getAtt('Endpoint'));
        this.attrArn = core_1.Token.asString(resource.getAtt('Arn'));
        this.attrCertificateAuthorityData = core_1.Token.asString(resource.getAtt('CertificateAuthorityData'));
        this.attrClusterSecurityGroupId = core_1.Token.asString(resource.getAtt('ClusterSecurityGroupId'));
        this.attrEncryptionConfigKeyArn = core_1.Token.asString(resource.getAtt('EncryptionConfigKeyArn'));
        this.attrOpenIdConnectIssuerUrl = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuerUrl'));
        this.attrOpenIdConnectIssuer = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuer'));
    }
    createAdminRole(props) {
        const stack = core_1.Stack.of(this);
        // the role used to create the cluster. this becomes the administrator role
        // of the cluster.
        const creationRole = new iam.Role(this, 'CreationRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        // the CreateCluster API will allow the cluster to assume this role, so we
        // need to allow the lambda execution role to pass it.
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [props.roleArn],
        }));
        // if we know the cluster name, restrict the policy to only allow
        // interacting with this specific cluster otherwise, we will have to grant
        // this role to manage all clusters in the account. this must be lazy since
        // `props.name` may contain a lazy value that conditionally resolves to a
        // physical name.
        const resourceArns = core_1.Lazy.list({
            produce: () => {
                const arn = stack.formatArn(clusterArnComponents(stack.resolve(props.name)));
                return stack.resolve(props.name)
                    ? [arn, `${arn}/*`] // see https://github.com/aws/aws-cdk/issues/6060
                    : ['*'];
            },
        });
        const fargateProfileResourceArn = core_1.Lazy.string({
            produce: () => stack.resolve(props.name)
                ? stack.formatArn({ service: 'eks', resource: 'fargateprofile', resourceName: stack.resolve(props.name) + '/*' })
                : '*',
        });
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'eks:CreateCluster',
                'eks:DescribeCluster',
                'eks:DescribeUpdate',
                'eks:DeleteCluster',
                'eks:UpdateClusterVersion',
                'eks:UpdateClusterConfig',
                'eks:CreateFargateProfile',
                'eks:TagResource',
                'eks:UntagResource',
            ],
            resources: resourceArns,
        }));
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeFargateProfile', 'eks:DeleteFargateProfile'],
            resources: [fargateProfileResourceArn],
        }));
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:GetRole', 'iam:listAttachedRolePolicies'],
            resources: ['*'],
        }));
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:CreateServiceLinkedRole'],
            resources: ['*'],
        }));
        // see https://github.com/aws/aws-cdk/issues/9027
        // these actions are the combined 'ec2:Describe*' actions taken from the EKS SLR policies.
        // (AWSServiceRoleForAmazonEKS, AWSServiceRoleForAmazonEKSForFargate, AWSServiceRoleForAmazonEKSNodegroup)
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'ec2:DescribeInstances',
                'ec2:DescribeNetworkInterfaces',
                'ec2:DescribeSecurityGroups',
                'ec2:DescribeSubnets',
                'ec2:DescribeRouteTables',
                'ec2:DescribeDhcpOptions',
                'ec2:DescribeVpcs',
            ],
            resources: ['*'],
        }));
        // grant cluster creation role sufficient permission to access the specified key
        // see https://docs.aws.amazon.com/eks/latest/userguide/create-cluster.html
        if (props.secretsEncryptionKey) {
            creationRole.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'kms:Encrypt',
                    'kms:Decrypt',
                    'kms:DescribeKey',
                    'kms:CreateGrant',
                ],
                resources: [props.secretsEncryptionKey.keyArn],
            }));
        }
        return creationRole;
    }
}
exports.ClusterResource = ClusterResource;
function clusterArnComponents(clusterName) {
    return {
        service: 'eks',
        resource: 'cluster',
        resourceName: clusterName,
    };
}
exports.clusterArnComponents = clusterArnComponents;
//# sourceMappingURL=data:application/json;base64,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