# Automatic Project Updates (1M-315)

Automatically post project status updates when tickets are transitioned, providing real-time epic/project health monitoring.

## Overview

The Auto Project Updates feature automatically generates and posts comprehensive project status summaries to Linear (or other supported platforms) whenever a ticket transitions to a new state. This provides:

- **Real-time visibility**: Instant updates when work progresses
- **Epic health tracking**: Automatic health assessment (on_track, at_risk, off_track)
- **Actionable insights**: Next steps, blockers, and recommendations
- **Zero overhead**: No manual status updates required

## How It Works

### Integration Points

1. **Ticket Transition Hook**: Triggers when `ticket_transition` is called
2. **StatusAnalyzer Reuse**: Leverages existing project analysis from 1M-316
3. **Project Update Posting**: Uses adapter's `create_project_update` method

### Workflow

```
ticket_transition called
  → Check if auto_project_updates enabled
  → Check if ticket has parent_epic
  → Fetch epic and all tickets
  → Analyze using StatusAnalyzer
  → Format as concise markdown
  → Post to project updates
  → Return to caller (transition continues regardless)
```

### Error Handling

- **Non-blocking**: Update failures don't block ticket transitions
- **Graceful degradation**: Skips update if adapter unsupported
- **Comprehensive logging**: All errors logged for debugging

## Configuration

Add the following to your `.mcp-ticketer/config.json`:

```json
{
  "auto_project_updates": {
    "enabled": true,
    "update_frequency": "on_transition",
    "health_tracking": true
  }
}
```

### Configuration Options

- **`enabled`** (boolean, default: `false`)
  - Set to `true` to enable automatic project updates

- **`update_frequency`** (string, default: `"on_transition"`)
  - `"on_transition"`: Post update on every ticket transition
  - `"on_completion"`: Only post when tickets are marked DONE or CLOSED
  - `"daily"`: Reserved for future daily summary feature

- **`health_tracking`** (boolean, default: `true`)
  - Include health status (on_track/at_risk/off_track) in updates
  - Set to `false` to omit health indicators

## Update Format

Updates are posted in this format:

```markdown
## Progress Update (Automated)

**Ticket Transitioned**: TICKET-123 "Fix authentication bug" → **DONE**

**Epic Status** (Authentication Epic):
- Completed: 5/10 tickets (50%)
- In Progress: 3 tickets
- Blocked: 1 ticket

**Recent Completions**:
- TICKET-123: Fix authentication bug
- TICKET-120: Add session validation

**Next Up**:
- TICKET-124: Implement refresh token (Priority: HIGH)
- TICKET-125: Add rate limiting (Priority: MEDIUM)

**Health**: ✅ On Track

**Blockers**:
- TICKET-130: API dependency unavailable (blocks 2 tickets)

---
*Auto-generated by mcp-ticketer on 2025-11-28 13:30 UTC*
```

## Usage Examples

### Example 1: Enable for MCP Ticketer Epic

```json
{
  "default_adapter": "linear",
  "auto_project_updates": {
    "enabled": true,
    "update_frequency": "on_transition",
    "health_tracking": true
  },
  "default_epic": "cbeff74a-edd7-4125-ac73-f64161cf91b3"
}
```

When you transition any ticket in the MCP Ticketer epic:

```python
# This will trigger an automatic project update
ticket_transition(
    ticket_id="1M-315",
    to_state="done",
    comment="Implemented automatic project updates"
)
```

**Result**: A project update is automatically posted showing:
- "1M-315 → DONE"
- Current epic completion (e.g., 45/60 tickets = 75%)
- Next recommended tickets
- Current health status

### Example 2: Only Update on Completions

```json
{
  "auto_project_updates": {
    "enabled": true,
    "update_frequency": "on_completion",
    "health_tracking": true
  }
}
```

Updates only posted when tickets reach terminal states (DONE or CLOSED), reducing noise while still tracking major milestones.

### Example 3: Disable Health Tracking

```json
{
  "auto_project_updates": {
    "enabled": true,
    "update_frequency": "on_transition",
    "health_tracking": false
  }
}
```

Updates posted without health indicators (useful if you prefer manual health assessment).

## Platform Support

### Supported Platforms

- ✅ **Linear**: Full support with native ProjectUpdate API
- ✅ **GitHub V2**: Supported via ProjectV2StatusUpdate
- ✅ **Asana**: Supported via Project Status Updates
- ⚠️ **JIRA**: Limited support (uses comments as workaround)

### Platform-Specific Notes

**Linear**:
- Updates appear in project's "Updates" tab
- Health status fully supported (on_track, at_risk, off_track)
- Automatic diff generation available

**GitHub V2**:
- Updates appear in project status updates
- Health maps to status options (complete, inactive, etc.)
- Requires Projects V2 enabled

**Asana**:
- Updates appear as Project Status Updates
- Health maps to color-coded status (green, yellow, red)
- Visible on project overview page

**JIRA**:
- Uses formatted comments on project/epic
- Health indicated via emoji in comment
- Less integrated than other platforms

## Architecture

### Component Overview

```
ticket_transition (user_ticket_tools.py)
  ↓
AutoProjectUpdateManager (automation/project_updates.py)
  ↓
StatusAnalyzer (analysis/project_status.py)  [REUSED from 1M-316]
  ↓
adapter.create_project_update (adapters/*/adapter.py)
```

### Key Classes

**`AutoProjectUpdateManager`** (`src/mcp_ticketer/automation/project_updates.py`)
- Main manager class for automatic updates
- Methods:
  - `is_enabled()`: Check if feature enabled
  - `create_transition_update()`: Generate and post update
  - `_format_markdown_summary()`: Format analysis as markdown

**Reused Components** (from 1M-316):
- `StatusAnalyzer`: Performs comprehensive project analysis
- `HealthAssessor`: Calculates project health metrics
- `DependencyGraph`: Identifies blockers and critical path

### Configuration Schema

Configuration is validated as part of `TicketerConfig`:

```python
@dataclass
class TicketerConfig:
    # ... other fields ...

    auto_project_updates: dict[str, Any] | None = None
    # Schema:
    # {
    #   "enabled": bool,
    #   "update_frequency": "on_transition" | "on_completion" | "daily",
    #   "health_tracking": bool
    # }
```

## Testing

### Unit Tests

Run the comprehensive test suite:

```bash
pytest tests/automation/test_auto_project_updates.py -v
```

**Test Coverage**:
- Configuration validation
- Update frequency settings
- Health tracking settings
- Epic data fetching
- Ticket fetching with fallbacks
- Markdown formatting
- Error handling (non-blocking failures)
- Adapter compatibility checks

### Integration Testing

1. **Enable feature in config**:
   ```json
   {
     "auto_project_updates": {
       "enabled": true,
       "update_frequency": "on_transition"
     }
   }
   ```

2. **Transition a ticket in your epic**:
   ```bash
   # Via MCP tool
   ticket_transition ticket_id="YOUR-TICKET" to_state="done"
   ```

3. **Verify update**:
   - Check Linear project's "Updates" tab
   - Verify update contains all expected sections
   - Confirm health status is accurate

## Troubleshooting

### Updates Not Appearing

**Problem**: Updates not being posted when tickets transition.

**Solutions**:
1. Check config: Ensure `auto_project_updates.enabled` is `true`
2. Verify parent epic: Only tickets with `parent_epic` trigger updates
3. Check adapter support: Run `hasattr(adapter, 'create_project_update')`
4. Review logs: Look for warnings/errors in application logs

### Incorrect Health Status

**Problem**: Health status doesn't match project reality.

**Solutions**:
1. Review StatusAnalyzer thresholds (see 1M-316 documentation)
2. Check ticket states are correctly classified
3. Verify blocked tickets are properly marked
4. Ensure completion rates align with expectations

### Too Many Updates

**Problem**: Receiving too many project updates.

**Solutions**:
1. Change `update_frequency` to `"on_completion"`
2. Disable for specific epics by removing `parent_epic`
3. Use manual `project_status` tool instead (1M-316)

### Missing Sections in Updates

**Problem**: Updates missing recommendations or blockers.

**Solutions**:
1. Check if epic has enough tickets for analysis
2. Verify tickets have proper dependencies set
3. Ensure priority levels are assigned
4. Review StatusAnalyzer configuration

## Performance Considerations

### Update Generation Overhead

- **Fetch epic data**: ~100-200ms (cached by adapter)
- **Fetch tickets**: ~200-500ms (depends on epic size)
- **Analysis**: ~50-100ms (StatusAnalyzer is efficient)
- **Post update**: ~200-400ms (API call)

**Total**: ~550-1200ms per transition (non-blocking)

### Optimization Strategies

1. **Frequency Control**: Use `"on_completion"` for large epics
2. **Caching**: Adapters cache epic data automatically
3. **Async Execution**: Updates posted asynchronously (don't block transition)
4. **Fail Fast**: Quick skip if adapter unsupported or epic not found

## Future Enhancements

### Planned Features

- [ ] **Daily Summaries**: Scheduled daily rollup updates
- [ ] **Custom Templates**: User-defined update formats
- [ ] **Update Filters**: Skip updates for specific ticket types
- [ ] **Aggregated Updates**: Batch multiple transitions into single update
- [ ] **Notification Integration**: Slack/Teams notifications on updates

### Enhancement Ideas

- Smart update throttling (e.g., max 1 update per hour)
- Customizable health thresholds
- Integration with Linear's project milestones
- Trend analysis (velocity, burndown charts)
- Predictive completion dates

## Related Features

- **1M-316**: `project_status` tool - Manual project analysis
- **1M-238**: `project_update_create` - Manual update posting
- **ISS-0002**: Semantic priority matching
- **Ticket State Machine**: Workflow state transitions

## Migration Guide

### From Manual Updates (1M-238)

**Before** (Manual):
```python
# After every ticket transition
ticket_transition(ticket_id="ABC-123", to_state="done")

# Manually generate status
status = project_status(project_id="epic-abc")

# Manually post update
project_update_create(
    project_id="epic-abc",
    body="Manual summary of status...",
    health="on_track"
)
```

**After** (Automatic):
```json
// One-time configuration
{
  "auto_project_updates": {
    "enabled": true
  }
}
```

```python
# Just transition - update posted automatically
ticket_transition(ticket_id="ABC-123", to_state="done")
# ✨ Update automatically posted!
```

### Gradual Adoption

1. **Phase 1**: Enable for one epic, monitor quality
2. **Phase 2**: Adjust update_frequency if needed
3. **Phase 3**: Enable globally for all projects
4. **Phase 4**: Deprecate manual update scripts

## Contributing

To enhance automatic project updates:

1. Add new update frequency options in `AutoProjectUpdateManager`
2. Create custom formatters in `_format_markdown_summary()`
3. Add adapter-specific enhancements for better platform integration
4. Contribute tests for edge cases

**Related Files**:
- `src/mcp_ticketer/automation/project_updates.py`
- `src/mcp_ticketer/mcp/server/tools/user_ticket_tools.py`
- `tests/automation/test_auto_project_updates.py`

## Ticket References

- **1M-315**: Implement automatic project_update tool
- **1M-316**: project_status tool with StatusAnalyzer
- **1M-238**: Add project updates support
