from __future__ import annotations

import hashlib
import json
import logging
import re
from datetime import date
from enum import StrEnum
from importlib.resources import files
from itertools import groupby
from operator import attrgetter
from pathlib import Path, PurePosixPath
from typing import Annotated, Any, Self

from pydantic import (
    AfterValidator,
    ConfigDict,
    Field,
    StringConstraints,
    ValidationError,
    field_validator,
    model_validator,
)
from pydantic.json_schema import GenerateJsonSchema

from ...common import StrictBaseModel
from ...mii.consent import Consent, ProvisionType
from .versioning import Version

SCHEMA_URL_PATTERN = r"https://raw\.githubusercontent\.com/BfArM-MVH/MVGenomseq/refs/tags/v([0-9]+)\.([0-9]+)(?:\.([0-9]+))?/GRZ/grz-schema\.json"
REDACTED_TAN = "0" * 64

log = logging.getLogger(__name__)


Tan = Annotated[str, StringConstraints(pattern=r"^[A-Fa-f0-9]{64}$")]
SubmitterId = Annotated[str, StringConstraints(pattern=r"^[0-9]{9}$")]
GenomicDataCenterId = Annotated[str, StringConstraints(pattern=r"^GRZ[A-Z0-9]{3}[0-9]{3}$")]
ClinicalDataNodeId = Annotated[str, StringConstraints(pattern=r"^KDK[A-Z0-9]{3}[0-9]{3}$")]


def is_supported_version(version: str) -> bool:
    return Version("1.1.1") <= Version(version) <= Version("1.3")


class ResearchConsentCodes(StrEnum):
    PATDAT_ERHEBEN_SPEICHERN_NUTZEN = "2.16.840.1.113883.3.1937.777.24.5.3.1"
    MDAT_WISSENSCHAFTLICH_NUTZEN_EU_DSGVO_NIVEAU = "2.16.840.1.113883.3.1937.777.24.5.3.8"


class SubmissionType(StrEnum):
    """
    The options are: 'initial' for first submission, 'followup' is for followup submissions, 'addition' for additional submission, 'correction' for correction, 'test' for test submissions
    """

    initial = "initial"
    followup = "followup"
    addition = "addition"
    correction = "correction"
    test = "test"


class GenomicStudyType(StrEnum):
    """
    whether additional persons are tested as well
    """

    single = "single"
    duo = "duo"
    trio = "trio"


class GenomicStudySubtype(StrEnum):
    """
    whether tumor and/or germ-line are tested
    """

    tumor_only = "tumor-only"
    tumor_germline = "tumor+germline"
    germline_only = "germline-only"


class CoverageType(StrEnum):
    """
    Health insurance providers
    """

    # Gesetzliche Krankenversicherung
    GKV = "GKV"
    # Private Krankenversicherung
    PKV = "PKV"
    # Berufsgenossenschaft
    BG = "BG"
    # Selbstzahler
    SEL = "SEL"
    # Sozialamt
    SOZ = "SOZ"
    # Gesetzliche Pflegeversicherung
    GPV = "GPV"
    # Private Pflegeversicherung
    PPV = "PPV"
    # Beihilfe
    BEI = "BEI"
    # Sonstige Kostenträger
    SKT = "SKT"
    # Unklar/Unbekannt
    UNK = "UNK"


class DiseaseType(StrEnum):
    """
    Type of the disease
    """

    oncological = "oncological"
    rare = "rare"
    hereditary = "hereditary"


class Submission(StrictBaseModel):
    submission_date: date
    """
    Date of submission in ISO 8601 format YYYY-MM-DD
    """

    submission_type: SubmissionType
    """
    The options are: 'initial' for first submission, 'followup' is for followup submissions, 'addition' for additional submission, 'correction' for correction
    """

    tan_g: Tan
    """
    The VNg of the genomic data of the index patient that will be reimbursed --> a unique 32-length byte code represented in a hex string of length 64.
    Randomly generated by an LE and collision-free within an LE.
    """

    local_case_id: str
    """
    A local case identifier for synchronizing locally
    """

    coverage_type: CoverageType
    """
    Health insurance provider
    """

    submitter_id: SubmitterId
    """
    Institutional ID of the submitter according to §293 SGB V.
    """

    genomic_data_center_id: GenomicDataCenterId
    """
    ID of the genomic data center in the format GRZXXXnnn.
    """

    clinical_data_node_id: ClinicalDataNodeId
    """
    ID of the clinical data node in the format KDKXXXnnn.
    """

    disease_type: DiseaseType
    """
    Type of the disease
    """

    genomic_study_type: GenomicStudyType
    """
    whether additional persons are tested as well
    """

    genomic_study_subtype: GenomicStudySubtype
    """
    whether tumor and/or germ-line are tested
    """

    lab_name: str
    """
    Name of the sequencing lab.
    """


class Gender(StrEnum):
    """
    Gender of the donor.
    """

    male = "male"
    female = "female"
    other = "other"
    unknown = "unknown"


class Relation(StrEnum):
    """
    Relationship of the donor in respect to the index patient, e.g. 'index', 'brother', 'mother', etc.
    """

    mother = "mother"
    father = "father"
    brother = "brother"
    sister = "sister"
    child = "child"
    index_ = "index"
    other = "other"


class MvConsentScopeType(StrEnum):
    """
    Consent or refusal to participate and consent, must be indicated for each option listed in the scope of consent.
    """

    permit = "permit"
    deny = "deny"


class MvConsentScopeDomain(StrEnum):
    """
    Scope of consent or revocation.
    """

    mv_sequencing = "mvSequencing"
    re_identification = "reIdentification"
    case_identification = "caseIdentification"


class MvConsentScope(StrictBaseModel):
    """
    The scope of the Modellvorhaben consent given by the donor.
    """

    type_: Annotated[MvConsentScopeType, Field(alias="type")]
    """
    Consent or refusal to participate and consent, must be indicated for each option listed in the scope of consent.
    """

    date: date
    """
    Date of signature of the pilot projects consent; in ISO 8601 format YYYY-MM-DD.
    """

    domain: MvConsentScopeDomain
    """
    Scope of consent or revocation.
    """


class MvConsent(StrictBaseModel):
    presentation_date: date | None = None
    """
    Date of delivery. Date (in ISO 8601 format YYYY-MM-DD) on which the Model Project Declaration of Participation 
    was presented to the patient, unless identical to the date of signature
    """

    version: str
    """
    Version of the declaration of participation. 
    Name and version of the declaration of participation in the MV GenomSeq, e.g.: 
    'Patient Info TE Consent MVGenomSeq vers01'
    """

    scope: list[MvConsentScope]
    """
    Modules of the consent to MV: must have at least a permit of mvSequencing
    """


class ResearchConsentSchemaVersion(StrEnum):
    v_2025_0_1 = "2025.0.1"


class ResearchConsentNoScopeJustification(StrEnum):
    UNABLE = "patient unable to consent"
    REFUSED = "patient refuses to sign consent"
    NO_RETURN = "patient did not return consent documents"
    OTHER = "other patient-related reason"
    LE_TECH = "consent information cannot be submitted by LE due to technical reason"
    LE_ORG = "consent is not implemented at LE due to organizational issues"


class ResearchConsent(StrictBaseModel):
    """
    Research consents. Multiple declarations of consent are possible! Must be assigned to the respective data sets.
    """

    schema_version: ResearchConsentSchemaVersion | None = None
    """
    Schema version of de.medizininformatikinitiative.kerndatensatz.consent
    """

    presentation_date: date | None = None
    """
    Date of the delivery of the research consent in ISO 8601 format (YYYY-MM-DD)
    """

    # try Consent model first, falling back to dict
    scope: Annotated[Consent | dict | None, Field(union_mode="left_to_right")] = None
    """
    Scope of the research consent in JSON format following the MII IG Consent v2025 FHIR schema. 
    See 'https://www.medizininformatik-initiative.de/Kerndatensatz/KDS_Consent_V2025/MII-IG-Modul-Consent.html' and 
    'https://packages2.fhir.org/packages/de.medizininformatikinitiative.kerndatensatz.consent'.
    """

    no_scope_justification: ResearchConsentNoScopeJustification | None = None

    @model_validator(mode="after")
    def ensure_scope_xor_justification(self):
        if (self.scope is None) != (self.no_scope_justification is None):
            return self
        raise ValueError("Either scope or noScopeJustification must be provided, but not both.")

    @model_validator(mode="after")
    def ensure_top_level_provision_deny(self):
        if isinstance(self.scope, Consent):
            if (top_level_provision := self.scope.provision) and (top_level_provision.type != ProvisionType.DENY):
                raise ValueError(
                    f"The root provision type must be deny, not {self.scope.provision.type}, "
                    "since the profile follows an opt-in consent scheme. "
                    "Explicit opt-in consents must be made via nested provisions."
                )
        elif self.scope:
            log.warning(
                "Could not recognize the object passed to researchConsents[].scope as a FHIR Consent profile. "
                "GRZs cannot confirm non-MVH research consent unless a valid FHIR Consent profile is provided. "
                "This is not currently required for submission. "
                "See below for errors encountered during validation."
            )
            try:
                Consent(**self.scope)
                raise RuntimeError(
                    "Pydantic correctly parsed research consent scope but failed to assign it to the correct type during model validation"
                )
            except ValidationError as e:
                for error in e.errors():
                    log.warning(
                        f"Validation error of type '{error['type']}' at location '{'.'.join(str(l) for l in error['loc'])}': {error['msg']}"
                    )

        return self

    def consent_by_code(self, date: date) -> dict[str, bool]:
        code2consent: dict[str, bool] = {}
        if isinstance(self.scope, Consent) and (self.scope.provision is not None):
            provisions = self.scope.provision.provision
            for provision in provisions:
                if provision.period.start <= date <= provision.period.end:
                    for codeable_concept in provision.code:
                        for coding in codeable_concept.coding:
                            if provision.type == ProvisionType.PERMIT:
                                code2consent[coding.code] = code2consent.get(coding.code, True)  # propagate prior deny
                            else:
                                # explicit deny overrides any prior/later permits for code
                                code2consent[coding.code] = False
        return code2consent

    @staticmethod
    def merged_consent_by_code(consents: list[ResearchConsent], date: date) -> dict[str, bool]:
        code2consent: dict[str, bool] = {}
        for consent in consents:
            for code, consented in consent.consent_by_code(date).items():
                if code in code2consent:
                    # keep permit only if both permitted
                    code2consent[code] = code2consent[code] and consented
                else:
                    code2consent[code] = consented
        return code2consent

    @staticmethod
    def consents_to_research(consents: list[ResearchConsent], date: date) -> bool:
        all_consented = None
        code2consent = ResearchConsent.merged_consent_by_code(consents, date)

        for code, consented in code2consent.items():
            if code in ResearchConsentCodes:
                all_consented = consented if all_consented is None else consented and all_consented

        return False if all_consented is None else all_consented


class TissueOntology(StrictBaseModel):
    name: str
    """
    Name of the tissue ontology
    """

    version: str
    """
    Version of the tissue ontology
    """


class SampleConservation(StrEnum):
    """
    Sample conservation
    """

    fresh_tissue = "fresh-tissue"
    cryo_frozen = "cryo-frozen"
    ffpe = "ffpe"
    other = "other"
    unknown = "unknown"


class SequenceType(StrEnum):
    """
    Type of sequence (DNA or RNA)
    """

    dna = "dna"
    rna = "rna"


class SequenceSubtype(StrEnum):
    """
    Subtype of sequence (germline, somatic, etc.)
    """

    germline = "germline"
    somatic = "somatic"
    other = "other"
    unknown = "unknown"


class FragmentationMethod(StrEnum):
    """
    Fragmentation method
    """

    sonication = "sonication"
    enzymatic = "enzymatic"
    none = "none"
    other = "other"
    unknown = "unknown"


class LibraryType(StrEnum):
    """
    Library type
    """

    panel = "panel"
    panel_lr = "panel_lr"
    wes = "wes"
    wes_lr = "wes_lr"
    wgs = "wgs"
    wgs_lr = "wgs_lr"
    wxs = "wxs"
    wxs_lr = "wxs_lr"
    other = "other"
    unknown = "unknown"


class EnrichmentKitManufacturer(StrEnum):
    """
    Manufacturer of the enrichment kit
    """

    illumina = "Illumina"
    agilent = "Agilent"
    twist = "Twist"
    neb = "NEB"
    other = "other"
    unknown = "unknown"
    none = "none"


class SequencingLayout(StrEnum):
    """
    The sequencing layout, aka the end type of sequencing.
    """

    single_end = "single-end"
    paired_end = "paired-end"
    reverse = "reverse"
    other = "other"


class TumorCellCountMethod(StrEnum):
    """
    Method used to determine cell count.
    """

    pathology = "pathology"
    bioinformatics = "bioinformatics"
    other = "other"
    unknown = "unknown"


class TumorCellCount(StrictBaseModel):
    """
    Tuple of tumor cell counts and how they were determined.
    """

    count: Annotated[float, Field(ge=0.0, le=100.0)]
    """
    Tumor cell count in %
    """

    method: TumorCellCountMethod
    """
    Method used to determine cell count.
    """


class CallerUsedItem(StrictBaseModel):
    name: str
    """
    Name of the caller used
    """

    version: str
    """
    Version of the caller used
    """


class FileType(StrEnum):
    """
    Type of the file; if BED file is submitted, only 1 file is allowed.
    """

    bam = "bam"
    vcf = "vcf"
    bed = "bed"
    fastq = "fastq"


class ChecksumType(StrEnum):
    """
    Type of checksum algorithm used
    """

    sha256 = "sha256"


class ReadOrder(StrEnum):
    """
    Indicates the read order for paired-end reads.
    """

    r1 = "R1"
    r2 = "R2"


def _path_is_relative_and_normalized(path_str: str) -> str:
    path = PurePosixPath(path_str)
    if path.is_absolute():
        raise ValueError(
            "File paths must be relative to files/ under the submission root, "
            "e.g.: patient_001/patient_001_dna.fastq.gz; "
            "symlinks are allowed."
        )
    # pathlib.Path does not normalize '..' so we must explicitly check
    if (".." in path.parts) or (str(path) != path_str):
        raise ValueError("File paths must be normalized. '..' or '.' in paths is not allowed.")

    return path_str


class File(StrictBaseModel):
    file_path: Annotated[str, AfterValidator(_path_is_relative_and_normalized)]
    """
    Path relative to files/ under the submission root, for example 'patient_001/patient_001_dna.bam'
    for a submission layout similar to:

    my_submission/
      files/
        patient_001/…
      metadata/
    """

    file_type: FileType
    """
    Type of the file; if BED file is submitted, only 1 file is allowed.
    """

    read_length: Annotated[int, Field(strict=True, ge=0)] | None = None
    """
    The read length; in the case of long-read sequencing it is the rounded average read length.
    """

    checksum_type: ChecksumType | None = ChecksumType.sha256
    """
    Type of checksum algorithm used
    """

    file_checksum: Annotated[str, StringConstraints(pattern=r"^[a-fA-F0-9]{64}$")]
    """
    checksum of the file
    """

    file_size_in_bytes: Annotated[int, Field(strict=True, ge=0)]
    """
    Size of the file in bytes
    """

    read_order: ReadOrder | None = None
    """
    Indicates the read order for paired-end reads.
    """

    flowcell_id: str | None = None
    """
    Indicates the flow cell.
    """

    lane_id: str | None = None
    """
    Indicates the lane
    """

    @model_validator(mode="after")
    def ensure_read_length_is_present_for_bam_and_fastq(self):
        if self.file_type in {FileType.bam, FileType.fastq} and self.read_length is None:
            raise ValueError(f"Read length missing for file '{self.file_path}' of type '{self.file_type}'.")
        return self

    @model_validator(mode="after")
    def ensure_extension_matches_type(self):
        """
        The QC pipeline will reject file names that don't satify the criteria below.
        """
        file_path = Path(self.file_path)
        match self.file_type:
            case FileType.bam:
                if re.fullmatch(r"\S+\.bam", file_path.name) is None:
                    raise ValueError("BAM files must have no spaces in the file name and a .bam extension")
            case FileType.bed:
                if re.fullmatch(r"\S+\.bed(?:\.gz)?", file_path.name) is None:
                    raise ValueError("BED files must have no spaces in the file name and a .bed or .bed.gz extension")
            case FileType.fastq:
                if re.fullmatch(r"\S+\.f(?:ast)?q\.gz", file_path.name) is None:
                    raise ValueError(
                        "FASTQ files must have no spaces in the file name and have a .fastq.gz or .fq.gz extension"
                    )
        return self

    def encrypted_file_path(self):
        return self.file_path + ".c4gh"


class PercentBasesAboveQualityThreshold(StrictBaseModel):
    """Percentage of bases with a specified minimum quality threshold"""

    minimum_quality: Annotated[float, Field(strict=True, ge=0.0)]
    """The minimum quality score threshold"""

    percent: Annotated[float, Field(strict=True, ge=0.0, le=100.0)]
    """
    Percentage of bases with a specified minimum quality threshold, according to https://www.bfarm.de/SharedDocs/Downloads/DE/Forschung/modellvorhaben-genomsequenzierung/Qs-durch-GRZ.pdf?__blob=publicationFile
    """


class ReferenceGenome(StrEnum):
    """
    Reference genome used according to the Genome Reference Consortium (https://www.ncbi.nlm.nih.gov/grc)
    """

    GRCh37 = "GRCh37"
    GRCh38 = "GRCh38"


class SequenceData(StrictBaseModel):
    bioinformatics_pipeline_name: str
    """
    Name of the bioinformatics pipeline used
    """

    bioinformatics_pipeline_version: str
    """
    Version or commit hash of the bioinformatics pipeline
    """

    reference_genome: ReferenceGenome
    """
    Reference genome used according to the Genome Reference Consortium (https://www.ncbi.nlm.nih.gov/grc)
    """

    percent_bases_above_quality_threshold: PercentBasesAboveQualityThreshold
    """
    Percentage of bases with a specified minimum quality threshold
    """

    mean_depth_of_coverage: Annotated[float, Field(strict=True, ge=0.0)]
    """
    Mean depth of coverage
    """

    min_coverage: Annotated[float, Field(strict=True, ge=0.0)]
    """
    Minimum coverage
    """

    targeted_regions_above_min_coverage: Annotated[float, Field(strict=True, ge=0.0, le=1.0)]
    """
    Fraction of targeted regions that are above minimum coverage
    """

    non_coding_variants: bool
    """
    The analysis includes non-coding variants -> true or false
    """

    caller_used: list[CallerUsedItem]
    """
    Caller that is used in the pipeline
    """

    files: list[File]
    """
    List of files generated and required in this analysis.
    """

    def contains_files(self, file_type: FileType) -> bool:
        return any(f.file_type == file_type for f in self.files)

    def list_files(self, file_type: FileType) -> list[File]:
        return [f for f in self.files if f.file_type == file_type]


class LabDatum(StrictBaseModel):
    lab_data_name: str
    """
    Name/ID of the biospecimen e.g. 'Blut DNA normal'
    """

    tissue_ontology: TissueOntology

    tissue_type_id: str
    """
    Tissue ID according to the ontology in use.
    """

    tissue_type_name: str
    """
    Tissue name according to the ontology in use.
    """

    sample_date: date
    """
    Date of sample in ISO 8601 format YYYY-MM-DD
    """

    sample_conservation: SampleConservation
    """
    Sample conservation
    """

    sequence_type: SequenceType
    """
    Type of sequence (DNA or RNA)
    """

    sequence_subtype: SequenceSubtype
    """
    Subtype of sequence (germline, somatic, etc.)
    """

    fragmentation_method: FragmentationMethod
    """
    Fragmentation method
    """

    library_type: LibraryType
    """
    Library type
    """

    library_prep_kit: str
    """
    Name/version of the library prepkit
    """

    library_prep_kit_manufacturer: str
    """
    Library prep kit manufacturer
    """

    sequencer_model: str
    """
    Name/version of the sequencer model
    """

    sequencer_manufacturer: str
    """
    Sequencer manufacturer
    """

    kit_name: str
    """
    Name/version of the sequencing kit
    """

    kit_manufacturer: str
    """
    Sequencing kit manufacturer
    """

    enrichment_kit_manufacturer: EnrichmentKitManufacturer
    """
    Manufacturer of the enrichment kit
    """

    enrichment_kit_description: str
    """
    Name/version of the enrichment kit
    """

    barcode: str
    """
    The barcode used or 'na'
    """

    sequencing_layout: SequencingLayout
    """
    The sequencing layout, aka the end type of sequencing.
    """

    tumor_cell_count: list[TumorCellCount] | None = None
    """
    Tuple of tumor cell counts and how they were determined.
    """

    sequence_data: SequenceData | None = None
    """
    Sequence data generated from the wet lab experiment.
    """

    @model_validator(mode="after")
    def ensure_sequence_data_is_not_empty(self) -> Self:
        if not self.sequence_data:
            raise ValueError(
                f"No sequence data for '{self.lab_data_name}'. "
                "If this truly is a submission without sequence data, "
                "only submission of relevant data to a clinical data node / KDK is necessary."
            )
        return self

    @model_validator(mode="after")
    def validate_sequencing_setup(self) -> Self:
        if self.library_type in {LibraryType.wxs, LibraryType.wxs_lr} and self.sequence_type != SequenceType.rna:
            raise ValueError(
                f"Error in lab datum '{self.lab_data_name}': "
                f"WXS requires RNA sequencing, but got '{self.sequence_type}'."
            )
        return self

    @model_validator(mode="after")
    def ensure_unpaired_long_read_data(self) -> Self:
        if self.library_type.endswith("_lr") and (self.sequencing_layout == SequencingLayout.paired_end):
            raise ValueError("Long read libraries can't be paired-end.")
        return self

    @model_validator(mode="after")
    def validate_sequencing_file_exists(self):  # noqa: C901
        """
        Check if there is a FASTQ or BAM file (depending on library type)
        """
        if self.sequence_data is None:
            # Skip if no sequence data is present
            return self

        fastq_files = self.sequence_data.list_files(FileType.fastq)
        bam_files = self.sequence_data.list_files(FileType.bam)

        if self.library_type.endswith("_lr"):
            if len(fastq_files) + len(bam_files) == 0:
                raise ValueError("Long-read datasets must contain at least one BAM or FASTQ file!")
        else:
            if len(bam_files) > 0:
                raise ValueError("Short-read datasets cannot contain BAM files!")
            if len(fastq_files) == 0:
                raise ValueError("Short-read datasets must contain at least one FASTQ file!")

        if self.sequencing_layout == SequencingLayout.paired_end:
            # check if read order is specified
            for i in fastq_files:
                if i.read_order is None:
                    raise ValueError(
                        f"Error in lab datum '{self.lab_data_name}': "
                        f"No read order specified for FASTQ file '{i.file_path}'!"
                    )

            key = lambda f: (f.flowcell_id, f.lane_id)
            fastq_files.sort(key=key)
            for (flowcell_id, lane_id), group in groupby(fastq_files, key):
                files = list(group)

                # separate R1 and R2 files
                fastq_r1_files = [f for f in files if f.read_order == ReadOrder.r1]
                fastq_r2_files = [f for f in files if f.read_order == ReadOrder.r2]

                # check that there are exactly one R1 and on R2 file present
                if (len(fastq_r1_files) != 1) or (len(fastq_r2_files) != 1):
                    raise ValueError(
                        f"Error in lab datum '{self.lab_data_name}': "
                        f"Paired end sequencing layout but not there is not exactly one R1 and one R2 file for flowcell id '{flowcell_id}', lane id '{lane_id}'!"
                    )

        return self

    @model_validator(mode="after")
    def ensure_unique_run_id(self) -> Self:
        if not self.sequence_data:
            return self

        read_files = filter(lambda f: f.file_type in {FileType.fastq, FileType.bam}, self.sequence_data.files)
        read_files_sorted = sorted(read_files, key=attrgetter("flowcell_id", "lane_id", "read_order"))
        for (flowcell_id, lane_id, read_order), group in groupby(
            read_files_sorted, key=attrgetter("flowcell_id", "lane_id", "read_order")
        ):
            group_t = tuple(group)
            if len(group_t) > 1:
                raise ValueError(
                    "Each read file in a lab datum must have a unique combination of flowcell_id, lane_id, and read_order. "
                    f"Found {len(group_t)} files with flowcell_id '{flowcell_id}', lane_id '{lane_id}', and read_order '{read_order}': {', '.join(f.file_path for f in group_t)}"
                )
        return self


class Donor(StrictBaseModel):
    donor_pseudonym: str
    """
    A unique identifier given by the Leistungserbringer for each donor of a single, duo or trio sequencing; 
    the donorPseudonym needs to be identifiable by the Leistungserbringer 
    in case of changes to the consents by one of the donors. 
    For Index patient use TanG.
    """

    gender: Gender
    """
    Gender of the donor.
    """

    relation: Relation
    """
    Relationship of the donor in respect to the index patient, e.g. 'index', 'brother', 'mother', etc.
    """

    mv_consent: MvConsent

    research_consents: list[ResearchConsent]
    """
    Research consents. Multiple declarations of consent are possible! Must be assigned to the respective data sets.
    """

    lab_data: list[LabDatum]
    """
    Lab data related to the donor.
    """

    @model_validator(mode="after")
    def warn_empty_sequence_data(self):
        for lab_datum in self.lab_data:
            if lab_datum.sequence_data is None:
                log.warning(
                    f"No sequence data found for lab datum '{lab_datum.lab_data_name}' in donor '{self.donor_pseudonym}'. "
                    "Is this a submission without sequence data?"
                )
        return self

    @model_validator(mode="after")
    def validate_target_bed_files_exist(self):
        """
        Check if the submission has the required bed files for panel sequencing.
        """
        lib_types = {
            LibraryType.panel,
            LibraryType.wes,
            LibraryType.wxs,
            LibraryType.panel_lr,
            LibraryType.wes_lr,
            LibraryType.wxs_lr,
        }

        for lab_datum in self.lab_data:
            if (
                lab_datum.sequence_data is not None
                and lab_datum.library_type in lib_types
                and not lab_datum.sequence_data.contains_files(FileType.bed)
            ):
                raise ValueError(
                    f"BED file missing for lab datum '{lab_datum.lab_data_name}' in donor '{self.donor_pseudonym}'."
                )

        return self

    @model_validator(mode="after")
    def validate_vcf_file_exists(self):
        """
        Check if there is a VCF file
        """
        for lab_datum in self.lab_data:
            if lab_datum.sequence_data is not None and not lab_datum.sequence_data.contains_files(FileType.vcf):
                log.warning(
                    f"VCF file missing for lab datum '{lab_datum.lab_data_name}' in donor '{self.donor_pseudonym}'."
                    "VCF files are recommended, but not required."
                )

        return self

    def consents_to_research(self, date: date) -> bool:
        return ResearchConsent.consents_to_research(self.research_consents, date)

    def consents_to_mv(self) -> bool:
        if self.mv_consent.scope:
            return any(
                scope.domain == MvConsentScopeDomain.mv_sequencing and scope.type_ == MvConsentScopeType.permit
                for scope in self.mv_consent.scope
            )
        else:
            if self.relation not in {Relation.mother, Relation.father}:
                log.warning("Donors must have at least a permit of mvSequencing. Exemptions only apply to parents.")
                return False
            if not self.research_consents:
                log.warning(
                    "Neither mvConsent nor researchConsent provided. Cannot confirm donor consented to participation."
                )
                return False

            mv_consent_exempt = False
            for research_consent in self.research_consents:
                presented_before_cutoff = (research_consent.presentation_date is not None) and (
                    research_consent.presentation_date <= date(year=2025, month=6, day=15)
                )
                consents_to_research = ResearchConsent.consents_to_research([research_consent], date.today())
                if presented_before_cutoff and consents_to_research:
                    mv_consent_exempt = True

            if not mv_consent_exempt:
                log.warning(
                    "mvConsent not provided and researchConsent does not specify broad consent presented before 2025-06-15. Cannot confirm donor consented to participation."
                )

            return mv_consent_exempt

        return False

    @model_validator(mode="after")
    def ensure_index_has_dna_data(self):
        # index donors must have a known DNA library type submittable in a Prüfbericht
        pruefbericht_library_types = {"panel", "wes", "wgs", "wgs_lr"}
        donor_library_types = {datum.library_type for datum in self.lab_data}
        if self.relation == Relation.index_ and not (donor_library_types & pruefbericht_library_types):
            raise ValueError(
                "Index donor must have at least one lab datum with one of the following library types: "
                f"{', '.join(pruefbericht_library_types)}."
            )

        return self


class GrzSubmissionMetadata(StrictBaseModel):
    """
    General metadata schema for submissions to the GRZ
    """

    schema_: Annotated[str, Field(alias="$schema", pattern=SCHEMA_URL_PATTERN)]
    model_config = ConfigDict(json_schema_extra={"$schema": GenerateJsonSchema.schema_dialect})

    submission: Submission

    donors: list[Donor]
    """
    List of donors including the index patient.
    """

    @property
    def submission_id(self) -> str:
        """
        A deterministic metadata-derived submission identifier for long-term use within GRZs.

        Uses the first 8 characters of the SHA256 hash of the tanG to virtually prevent collisions.
        """
        if self.submission.tan_g == REDACTED_TAN:
            raise ValueError("Cannot compute submission ID from metadata with a redacted TAN.")

        return "_".join(
            (
                self.submission.submitter_id,
                self.submission.submission_date.isoformat(),
                hashlib.sha256(self.submission.tan_g.encode("utf-8")).hexdigest()[:8],
            )
        )

    def consents_to_research(self, date: date) -> bool:
        return all(donor.consents_to_research(date) for donor in self.donors)

    @field_validator("donors", mode="after")
    @classmethod
    def ensure_single_index_patient(cls, value: list[Donor]) -> list[Donor]:
        num_index_patients = sum(donor.relation == Relation.index_ for donor in value)
        if num_index_patients == 0:
            raise ValueError("No index donor found! Exactly one index donor required.")
        elif num_index_patients > 1:
            raise ValueError("Multiple index donors found! Exactly one index donor required.")
        return value

    @property
    def index_donor(self) -> Donor:
        """The index patient of the submission."""
        # note: this is valid because the above field_validator ensures only one index patient
        return next(donor for donor in self.donors if donor.relation == Relation.index_)

    @field_validator("donors", mode="after")
    @classmethod
    def ensure_unique_donor_pseudonyms(cls, value: list[Donor]) -> list[Donor]:
        donor_pseudonyms = set()
        for donor in value:
            if donor.donor_pseudonym in donor_pseudonyms:
                raise ValueError(f"Encountered duplicate donor pseudonym: '{donor.donor_pseudonym}'")
            donor_pseudonyms.add(donor.donor_pseudonym)
        return value

    def get_schema_version(self) -> str:
        if match := re.fullmatch(SCHEMA_URL_PATTERN, self.schema_):
            schema_version = ".".join(match.groups(default="0"))
        else:
            raise ValueError("Schema URL should have matched pattern but didn't")
        return schema_version

    @model_validator(mode="after")
    def validate_research_consent_after_minor_version_3(self):  # noqa: C901
        if Version(self.get_schema_version()) >= Version("1.3"):
            for donor in self.donors:
                if len(donor.research_consents) < 1:
                    raise ValueError("Donors must have research consent as of metadata schema v1.3")
                for consent in donor.research_consents:
                    if consent.presentation_date is None:
                        raise ValueError("Research consents must have presentationDate as of metadata schema v1.3")
                    if consent.no_scope_justification is None and not consent.scope:
                        raise ValueError("Either a non-empty scope must be provided or a noScopeJustification")
                    if (consent.scope is not None) and (not isinstance(consent.scope, Consent)):
                        raise ValueError("scope must be a valid MII Broad Consent as of metadata v1.3")
        else:
            for donor in self.donors:
                for consent in donor.research_consents:
                    if consent.schema_version is None:
                        raise ValueError("schemaVersion is required in researchConsent before metadata schema v1.3")
        return self

    @model_validator(mode="after")
    def ensure_supported_schema_version(self):
        schema_version = self.get_schema_version()
        if not is_supported_version(schema_version):
            raise ValueError(f"Unsupported metadata schema version: {schema_version}")
        return self

    @model_validator(mode="after")
    def validate_donor_count(self):
        """
        Check whether the submission has the required number of donors based on the study type.
        """
        study_type = self.submission.genomic_study_type

        match study_type:
            case GenomicStudyType.single:
                # Check if the submission has at least one donor
                if not self.donors:
                    raise ValueError("At least one donor is required for a single study.")
            case GenomicStudyType.duo:
                # Check if the submission has at least two donors
                if len(self.donors) < 2:
                    raise ValueError("At least two donors are required for a duo study.")
            case GenomicStudyType.trio:
                # Check if the submission has at least three donors
                if len(self.donors) < 3:
                    raise ValueError("At least three donors are required for a trio study.")

        return self

    @model_validator(mode="after")
    def ensure_all_donors_mv_consented_if_initial(self):
        all_donors_mv_consented = all(donor.consents_to_mv() for donor in self.donors)
        if (self.submission.submission_type == SubmissionType.initial) and not all_donors_mv_consented:
            raise ValueError("All donors must consent to model project participation for initial submissions.")
        return self

    @model_validator(mode="after")
    def check_for_tumor_cell_count(self):
        """
        Check if oncology samples have tumor cell counts.
        """
        for donor in self.donors:
            pseudonym = donor.donor_pseudonym
            for lab_datum in donor.lab_data:
                if lab_datum.sequence_subtype == SequenceSubtype.somatic and lab_datum.tumor_cell_count is None:
                    raise ValueError(
                        f"Missing tumor cell count for donor '{pseudonym}', lab datum '{lab_datum.lab_data_name}'!"
                    )

        return self

    @model_validator(mode="after")
    def check_duplicate_lab_data_names(self):
        """
        Check if the submission contains lab data with the same name within one donor.
        """
        for donor in self.donors:
            pseudonym = donor.donor_pseudonym
            lab_data_names = set()
            for lab_datum in donor.lab_data:
                if lab_datum.lab_data_name in lab_data_names:
                    raise ValueError(f"Duplicate lab datum '{lab_datum.lab_data_name}' in donor '{pseudonym}'")
                else:
                    lab_data_names.add(lab_datum.lab_data_name)

        return self

    @model_validator(mode="after")
    def check_duplicate_file_checksums(self):
        checksums = set()
        for donor in self.donors:
            for lab_datum in donor.lab_data:
                if sequence_data := lab_datum.sequence_data:
                    for file in sequence_data.files:
                        checksum = file.file_checksum
                        if not checksum in checksums:
                            checksums.add(checksum)
                        else:
                            log.warning(
                                f"Encountered duplicate file checksum '{checksum}' "
                                f"in '{lab_datum.lab_data_name}' "
                                f"in donor '{donor.donor_pseudonym}'. "
                                "This is highly unlikely, "
                                "please ensure that the submission does not contain duplicate files."
                            )

        return self

    @model_validator(mode="after")
    def check_duplicate_file_paths(self):
        file_paths = set()
        for donor in self.donors:
            for lab_datum in donor.lab_data:
                if sequence_data := lab_datum.sequence_data:
                    for file in sequence_data.files:
                        file_path = file.file_path
                        if not file_path in file_paths:
                            file_paths.add(file_path)
                        else:
                            log.warning(
                                f"Encountered duplicate file path '{file_path}' "
                                f"in '{lab_datum.lab_data_name}' "
                                f"in donor '{donor.donor_pseudonym}'. "
                                "Are you sure you want to add the same file path "
                                "multiple times? "
                            )

        return self

    @model_validator(mode="after")
    def validate_thresholds(self):
        """
        Check if the submission meets the minimum mean coverage requirements.
        """
        threshold_definitions = load_thresholds()

        for donor in self.donors:
            for lab_datum in donor.lab_data:
                key = (
                    self.submission.genomic_study_subtype,
                    lab_datum.library_type,
                    lab_datum.sequence_subtype,
                )
                thresholds = threshold_definitions.get(key)
                if thresholds is None:
                    allowed_combinations = sorted(list(threshold_definitions.keys()))
                    allowed_combinations = "\n".join([f"  - {combination}" for combination in allowed_combinations])
                    names = (
                        "submission.genomicStudySubtype",
                        "labData.libraryType",
                        "labData.sequenceSubtype",
                    )
                    info = dict(zip(names, key, strict=True))
                    log.warning(
                        f"No thresholds for the specified combination {info} found (donor {donor.donor_pseudonym})!\n"
                        f"Valid combinations:\n{allowed_combinations}.\n"
                        f"See https://www.bfarm.de/SharedDocs/Downloads/DE/Forschung/modellvorhaben-genomsequenzierung/Qs-durch-GRZ.pdf?__blob=publicationFile for more details.\n"
                        f"Skipping threshold validation."
                    )
                    continue

                _check_thresholds(donor, lab_datum, thresholds)

        return self

    @model_validator(mode="after")
    def validate_reference_genome_compatibility(self):
        reference_genomes = {
            (donor.donor_pseudonym, lab_datum.lab_data_name): lab_datum.sequence_data.reference_genome
            for donor in self.donors
            for lab_datum in donor.lab_data
            if lab_datum.sequence_data is not None
        }
        unique_reference_genomes = set(reference_genomes.values())
        if len(unique_reference_genomes) > 1:
            raise ValueError(
                f"Incompatible reference genomes found: {unique_reference_genomes}.\n"
                f"Reference genomes must be consistent within a submission.\n"
                f"Reference genomes: {reference_genomes}"
            )

        return self


def _check_thresholds(donor: Donor, lab_datum: LabDatum, thresholds: dict[str, Any]):  # noqa: C901
    def raise_if_index(message: str) -> None:
        if donor.relation == Relation.index_:
            raise ValueError(message)
        else:
            log.warning(message)

    if lab_datum.sequence_data is None:
        # Skip if no sequence data is present; warning issues in the validator `warn_empty_sequence_data` of `Donor`.
        return
    pseudonym = donor.donor_pseudonym
    lab_data_name = lab_datum.lab_data_name
    sequence_data = lab_datum.sequence_data

    mean_depth_of_coverage_t = thresholds.get("meanDepthOfCoverage")
    mean_depth_of_coverage_v = sequence_data.mean_depth_of_coverage
    if mean_depth_of_coverage_t and mean_depth_of_coverage_v < mean_depth_of_coverage_t:
        raise_if_index(
            f"Mean depth of coverage for donor '{pseudonym}', lab datum '{lab_data_name}' "
            f"below threshold: {mean_depth_of_coverage_v} < {mean_depth_of_coverage_t}"
        )

    if percent_bases_above_quality_threshold_t := thresholds.get("percentBasesAboveQualityThreshold"):
        minimum_quality_t = percent_bases_above_quality_threshold_t.get("qualityThreshold")
        minimum_quality_v = sequence_data.percent_bases_above_quality_threshold.minimum_quality
        if minimum_quality_t and (minimum_quality_t != minimum_quality_v):
            # TODO also print out genomic study subtype because that determines thresholds, but is defined at submission level
            # this should error regardless of relation because it should match the official threshold
            raise ValueError(
                f"Expected minimumQuality '{minimum_quality_t}' for library type '{lab_datum.library_type}'"
                f"and sequence subtype '{lab_datum.sequence_subtype}'. Got '{minimum_quality_v}' instead."
            )
        percent_t = percent_bases_above_quality_threshold_t.get("percentBasesAbove")
        percent_v = sequence_data.percent_bases_above_quality_threshold.percent
        if percent_t and (percent_v < percent_t):
            raise_if_index(
                f"Percentage of bases above quality threshold are below threshold: {percent_v} < {percent_t}"
            )

    if t := thresholds.get("targetedRegionsAboveMinCoverage"):
        min_coverage_t = t.get("minCoverage")
        min_coverage_v = sequence_data.min_coverage

        fraction_above_t = t.get("fractionAbove")
        fraction_above_v = sequence_data.targeted_regions_above_min_coverage

        if min_coverage_t and min_coverage_v != min_coverage_t:
            # this should error regardless of relation because it should match the official threshold
            raise ValueError(
                f"Minimum coverage for donor '{pseudonym}', lab datum '{lab_data_name}' "
                f"not equal to expected threshold: {min_coverage_v} != {min_coverage_t}"
            )
        if fraction_above_t and fraction_above_v < fraction_above_t:
            raise_if_index(
                f"Fraction of targeted regions above minimum coverage for donor '{pseudonym}', "
                f"lab datum '{lab_data_name}' below threshold: "
                f"{fraction_above_v} < {fraction_above_t}"
            )


type Thresholds = dict[tuple[str, str, str], dict[str, Any]]


def load_thresholds() -> Thresholds:
    threshold_definitions = json.load(
        files("grz_pydantic_models").joinpath("resources", "thresholds.json").open("r", encoding="utf-8")
    )
    threshold_definitions = {
        (d["genomicStudySubtype"], d["libraryType"], d["sequenceSubtype"]): d["thresholds"]
        for d in threshold_definitions
    }
    return threshold_definitions
