# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['coleo']

package_data = \
{'': ['*']}

install_requires = \
['ptera>=0.3.3,<0.4.0']

extras_require = \
{'toml': ['toml>=0.10.0,<0.11.0'], 'yaml': ['pyyaml>=5.3,<6.0']}

setup_kwargs = {
    'name': 'coleo',
    'version': '0.2.2',
    'description': 'The nicest way to develop a command-line interface',
    'long_description': '\n# Coleo\n\nColeo is a minimum-effort way to create a command-line interface in Python.\n\n* Declare options where they are used.\n* Scale easily to extensive CLIs with dozens of subcommands and options.\n\n\n## Basic usage\n\nFirst, define a command line interface as follows:\n\n```python\nfrom coleo import Option, auto_cli, default\n\n@auto_cli\ndef main():\n    # The greeting\n    greeting: Option = default("Hello")\n\n    # The name to greet\n    name: Option = default("you")\n\n    return f"{greeting}, {name}!"\n```\n\nThen you may run it like this on the command line:\n\n```bash\n$ python hello.py\nHello, you!\n$ python hello.py --name Luke\nHello, Luke!\n$ python hello.py --name Luke --greeting "Happy birthday"\nHappy birthday, Luke!\n$ python hello.py -h\nusage: hello.py [-h] [--greeting VALUE] [--name VALUE]\n\noptional arguments:\n  -h, --help        show this help message and exit\n  --greeting VALUE  The greeting\n  --name VALUE      The name to greet\n```\n\n* Any variable annotated with `Option` will become an option.\n* You can provide a default value with `default(value)`, although you don\'t have to, if the argument is required.\n* If there is a comment above the variable, it will be used as documentation for the option.\n\n\n## Option types\n\nBy default, all arguments are interpreted as strings, but you can easily give a different type to an argument:\n\n```python\n@auto_cli\ndef main():\n    # This argument will be converted to an int\n    x: Option & int\n    # This argument will be converted to a float\n    y: Option & float\n    return x + y\n```\n\n**Boolean flags**\n\nIf the type is bool, the option will take no argument, for example:\n\n```python\n@auto_cli\ndef main():\n    flag: Option & bool = default(False)\n    return "yes!" if flag else "no!"\n```\n\nUse it like this:\n\n```bash\n$ python script.py --flag\nyes!\n$ python script.py\nno!\n```\n\nYou can also *negate* the flag, meaning that you want to provide an option that will store False in the variable instead of True. For example:\n\n```python\n@auto_cli\ndef main():\n    # [negate]\n    flag: Option & bool = default(True)\n    return "yes!" if flag else "no!"\n```\n\nBy default, the above will create a flag called `--no-<optname>`:\n\n```bash\n$ python script.py\nyes!\n$ python script.py --no-flag\nno!\n```\n\nYou can write `[negate: --xyz -n]` if you want the option to be `--xyz` or `-n`. This overrides the default `--no-flag` option.\n\nNote that using `[negate]` will remove `--flag`, because we assume that it is True by default and there is therefore no need for this option.\n\nIf you wish, you can have both options that set the flag to True and others that set the flag to False, using `[false-options]`. You can optionally document these options with `[false-options-doc]` (if not provided, Coleo will use a sensible default):\n\n```python\n@auto_cli\ndef main():\n    # Set the flag to True\n    # [options: -y]\n    # [false-options: -n]\n    # [false-options-doc: Set the flag to False]\n    flag: Option & bool = default(None)\n    return flag\n```\n\n```bash\n$ python script.py\nNone\n$ python script.py -y\nTrue\n$ python script.py -n\nFalse\n```\n\n\n**Files**\n\nUse `coleo.FileType` (or `argparse.FileType`, it\'s the same thing) to open a file to read from or to write to:\n\n```python\n@auto_cli\ndef main():\n    grocery_list: Option & coleo.FileType("r")\n    with grocery_list as f:\n        for food in f.readlines():\n            print(f"Gotta buy some {food}")\n```\n\n**Config**\n\nYou can manipulate configuration files with `coleo.config` or `coleo.ConfigFile`:\n\n```python\n@auto_cli\ndef main():\n    # ConfigFile lets you read or write a configuration file\n    book: Option & ConfigFile\n    contents = book.read()\n    contents["xyz"] = "abc"\n    book.write(contents)\n\n    # config will read the file for you or parse the argument as JSON\n    magazine: Option & config\n    print(magazine)\n```\n\nUse it simply like this:\n\n```bash\n$ python librarian.py --book alice.json --magazine vogue.json\n$ python librarian.py --book history.yaml --magazine gamez.toml\n$ python librarian.py --book physics.json --magazine \'{"a": 1, "b": 2}\'\n# etc\n```\n\nSupported extensions are `json`, `yaml` and `toml` (the latter two require installing the `pyyaml` or `toml` packages).\n\n\n**Other**\n\nAny function can be used as a "type" for an argument. So for example, if you want to be able to provide lists and dictionaries on the command line you can simply use `json.loads` (although `coleo.config` is usually better, because it can also read files, in various formats):\n\n```python\n@auto_cli\ndef main():\n    obj: Option & json.loads\n    return type(obj).__name__\n```\n\n```bash\n$ python json.py --obj 1\nint\n$ python json.py --obj \'"hello"\'\nstr\n$ python json.py --obj \'{"a": 1, "b": 2}\'\ndict\n```\n\nIf you\'re feeling super feisty and care nothing about safety, you can even use `eval`:\n\n```python\n@auto_cli\ndef main():\n    obj: Option & eval\n    return type(obj).__name__\n```\n\n```bash\n$ python eval.py --obj "1 + 2"\nint\n$ python eval.py --obj "lambda x: x + 1"\nfunction\n```\n\n\n## Customization\n\nUsing comments of the form `# [<instruction>: <args ...>]` you can customize the option parser:\n\n```python\n@auto_cli\ndef main():\n    # This argument can be given as either --greeting or -g\n    # [alias: -g]\n    greeting: Option = default("Hello")\n\n    # This argument is positional\n    # [positional]\n    name: Option = default("you")\n\n    # This argument can only be given as -n\n    # [options: -n]\n    ntimes: Option & int = default(1)\n\n    for i in range(ntimes):\n        print(f"{greeting}, {name}!")\n```\n\nThe above would be used like this:\n\n```bash\n$ python hello.py Alice -g Greetings -n 2\nGreetings, Alice!\nGreetings, Alice!\n```\n\nThe following customizations are available:\n\n* `[alias: ...]` defines one or several options that are aliases for the main one. Options are separated by spaces, commas or semicolons.\n* `[options: ...]` defines one or several options for this argument, which *override* the default one. Options are separated by spaces, commas or semicolons.\n* `[positional]` defines one positional argument.\n  * `[positional: n]`: n positional arguments (a list is returned).\n  * `[positional: ?]`: one optional positional argument\n  * `[positional: *]`: zero or more positional arguments\n  * `[positional: +]`: one or more positional arguments\n* `[remainder]` represents all arguments that are not matched by the argument parser\n* `[nargs: n]` declares that the option takes n arguments\n  * `[nargs: ?]`: one optional argument\n  * `[nargs: *]`: zero or more arguments\n  * `[nargs: +]`: one or more arguments\n* `[action: <action>]` customizes the action to perform\n  * `[action: append]` lets you use an option multiple times, accumulating the results in a list (e.g. `python app.py -a 1 -a 2 -a 3`, would put `[1, 2, 3]` in `a`)\n* `[metavar: varname]` changes the variable name right after the option in the help string, e.g. `--opt METAVAR`\n* `[group: groupname]` puts the option in a named group. Options in the same group will appear together in the help.\n* For **bool** options only:\n    * `[negate: ...]` changes the option so that it sets the variable to False instead of True when they are given. Space/comma aliases may be provided for the option, otherwise the flag will be named `--no-<optname>`.\n    * `[false-options: ]` provide a list of options that set the flag to False.\n    * `[false-options-doc: ]` provide a documentation for the options given using the previous statement.\n\n\n## Subcommands\n\nYou can create an interface with a hierarchy of subcommands by decorating a class with `auto_cli`:\n\n```python\n@auto_cli\nclass main:\n    class calc:\n        def add():\n            x: Option & int\n            y: Option & int\n            return x + y\n\n        def mul():\n            x: Option & int\n            y: Option & int\n            return x * y\n\n        def pow():\n            base: Option & int\n            exponent: Option & int\n            return base ** exponent\n\n    def greet():\n        greeting: Option = default("Hello")\n        name: Option = default("you")\n        return f"{greeting}, {name}!"\n```\n\nThe class only holds structure and will never be instantiated, so don\'t add `self` to the argument lists for these functions.\n\nThen you may use it like this:\n\n```bash\n$ python multi.py greet --name Alice --greeting Hi\nHi, Alice!\n$ python multi.py calc add --x=3 --y=8\n11\n```\n\n\n## Sharing arguments\n\nIt is possible to share behavior and arguments between subcommands, or to split complex functionality into multiple pieces. For example, maybe multiple subcommands in your application require an API key, which can either be given on the command line or can be read from a file. This is how you would share this behavior across all subcommands:\n\n```python\nfrom coleo import Option, auto_cli, config, default, tooled\n\n@tooled\ndef apikey():\n    # The API key to use\n    key: Option = default(None)\n    if key is None:\n        # If no key parameter is given on the command line, try to read it from\n        # some standard location.\n        key = config("~/.config/myapp/config.json")["key"]\n    return key\n\n@auto_cli\nclass main:\n    def search():\n        interface = Application(apikey())\n        query: Option\n        return interface.search(query)\n\n    def install():\n        interface = Application(apikey())\n        package: Option\n        return interface.install(package)\n```\n\nIf a function is decorated with `@tooled` and is called from one of the main functions (or from another tooled function), Coleo will search for arguments in that function too. Thus any subcommand that calls `apikey()` will gain a `--key` option.\n\nIn addition to this, you can "share" arguments by defining the same argument with the same type in multiple functions. Coleo will set all of them to the same value.\n\nFor example, in the example above you could easily let the user specify the path to the file that contains the key, simply by replacing\n\n```python\nkey = config("~/.config/myapp/config.json")["key"]\n\n# ==>\n\nconfig_path: Option = default("~/.config/myapp/config.json")\nkey = config(config_path)["key"]\n```\n\nAnd that `config_path` argument could, of course, be declared in any other function that needs to read some configuration value.\n\n\n## run_cli\n\n```python\nfrom coleo import Option, auto_cli\n\n@auto_cli\ndef main():\n    x: Option\n    return x\n```\n\nIs equivalent to:\n\n```python\nfrom coleo import Option, run_cli, tooled\n\n@tooled\ndef main():\n    x: Option\n    return x\n\nresult = run_cli(main)\nif result is not None:\n    print(result)\n```\n\n\n## Non-CLI usage\n\nIt is possible to set arguments without `auto_cli` using `setvars`:\n\n```python\nfrom coleo import Option, setvars, tooled\n\n@tooled\ndef greet():\n    greeting: Option = default("Hello")\n    name: Option = default("you")\n    return f"{greeting} {name}!"\n\nwith setvars(greeting="Hi", name="Bob"):\n    assert greet() == "Hi bob!"\n```\n\nNote:\n\n* With `setvars`, you *must* decorate the function with `@tooled` (this is something `auto_cli` does on your behalf).\n* `setvars` entirely bypasses the option parsing and the type annotations will not be used to wrap these values. In other words, if a variable is annotated `Option & int` and you provide the value "1", it will remain a string.\n\n\n### Using with Ptera\n\nColeo is based on [Ptera](https://github.com/mila-iqia/ptera) and all of Ptera\'s functionality is de facto available on functions marked as `@tooled`. For example, using the example above:\n\n```python\n# Set the variables in the greet function -- it\'s a bit like making an object\nhibob = greet.new(greeting="Hi", name="Bob")\nassert hibob() == "Hi Bob!"\n\n# Same as above but this would also change greeting/name in any other function\n# that is called by greet, and so on recursively (a bit like dynamic scoping)\nhibob = greet.tweaking({"greeting": "Hi", "name": "Bob"})\nassert hibob() == "Hi Bob!"\n\n# More complex behavior\nfrom ptera import overlay\nwith overlay.tweaking({\n    "greet(greeting=\'Bonjour\') > name": "Toto"\n}):\n    assert greet() == "Hello you!"\n    assert greet.new(greeting="Hi")() == "Hi you!"\n    assert greet.new(greeting="Bonjour")() == "Bonjour toto!"\n```\n\nRead the documentation for [Ptera](https://github.com/mila-iqia/ptera) for more information. Note that Ptera is not limited to variables tagged `Option`, it can manipulate *any* variable in a tooled function.\n',
    'author': 'Olivier Breuleux',
    'author_email': 'breuleux@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/breuleux/coleo',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
