"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaRunner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
const utils_1 = require("../utils");
const common_1 = require("./common");
const codebuild_1 = require("./image-builders/codebuild");
/**
 * GitHub Actions runner provider using Lambda to execute the actions.
 *
 * Creates a Docker-based function that gets executed for each job.
 *
 * This construct is not meant to be used by itself. It should be passed in the providers property for GitHubRunners.
 */
class LambdaRunner extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.label = props.label || 'lambda';
        this.vpc = props.vpc;
        this.securityGroup = props.securityGroup;
        const imageBuilder = props.imageBuilder ?? new codebuild_1.CodeBuildImageBuilder(this, 'Image Builder', {
            dockerfilePath: LambdaRunner.LINUX_X64_DOCKERFILE_PATH,
        });
        const image = this.image = imageBuilder.bind();
        let architecture;
        if (image.os.is(common_1.Os.LINUX)) {
            if (image.architecture.is(common_1.Architecture.X86_64)) {
                architecture = aws_cdk_lib_1.aws_lambda.Architecture.X86_64;
            }
            if (image.architecture.is(common_1.Architecture.ARM64)) {
                architecture = aws_cdk_lib_1.aws_lambda.Architecture.ARM_64;
            }
        }
        if (!architecture) {
            throw new Error(`Unable to find support Lambda architecture for ${image.os.name}/${image.architecture.name}`);
        }
        // get image digest and make sure to get it every time the lambda function might be updated
        // pass all variables that may change and cause a function update
        // if we don't get the latest digest, the update may fail as a new image was already built outside the stack on a schedule
        // we automatically delete old images, so we must always get the latest digest
        const imageDigest = this.imageDigest(image, {
            version: 1,
            label: this.label,
            architecture: architecture.name,
            vpc: this.vpc?.vpcId,
            securityGroups: this.securityGroup?.securityGroupId,
            vpcSubnets: props.subnetSelection?.subnets?.map(s => s.subnetId),
            timeout: props.timeout?.toSeconds(),
            memorySize: props.memorySize,
            ephemeralStorageSize: props.ephemeralStorageSize?.toKibibytes(),
            logRetention: props.logRetention?.toFixed(),
        });
        this.function = new aws_cdk_lib_1.aws_lambda.DockerImageFunction(this, 'Function', {
            description: `GitHub Actions runner for "${this.label}" label`,
            // CDK requires "sha256:" literal prefix -- https://github.com/aws/aws-cdk/blob/ba91ca45ad759ab5db6da17a62333e2bc11e1075/packages/%40aws-cdk/aws-ecr/lib/repository.ts#L184
            code: aws_cdk_lib_1.aws_lambda.DockerImageCode.fromEcr(image.imageRepository, { tagOrDigest: `sha256:${imageDigest}` }),
            architecture,
            vpc: this.vpc,
            securityGroups: this.securityGroup && [this.securityGroup],
            vpcSubnets: props.subnetSelection,
            timeout: props.timeout || cdk.Duration.minutes(15),
            memorySize: props.memorySize || 2048,
            ephemeralStorageSize: props.ephemeralStorageSize || cdk.Size.gibibytes(10),
            logRetention: props.logRetention || aws_logs_1.RetentionDays.ONE_MONTH,
        });
        this.grantPrincipal = this.function.grantPrincipal;
        this.addImageUpdater(image);
    }
    /**
     * The network connections associated with this resource.
     */
    get connections() {
        return this.function.connections;
    }
    /**
     * Generate step function task(s) to start a new runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters workflow job details
     */
    getStepFunctionTask(parameters) {
        return new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, this.label, {
            lambdaFunction: this.function,
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                token: parameters.runnerTokenPath,
                runnerName: parameters.runnerNamePath,
                label: this.label,
                githubDomain: parameters.githubDomainPath,
                owner: parameters.ownerPath,
                repo: parameters.repoPath,
            }),
        });
    }
    addImageUpdater(image) {
        // Lambda needs to be pointing to a specific image digest and not just a tag.
        // Whenever we update the tag to a new digest, we need to update the lambda.
        const updater = utils_1.BundledNodejsFunction.singleton(this, 'update-lambda', {
            description: 'Function that updates a GitHub Actions runner function with the latest image digest after the image has been rebuilt',
            timeout: cdk.Duration.minutes(15),
        });
        updater.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['lambda:UpdateFunctionCode'],
            resources: [this.function.functionArn],
        }));
        let lambdaTarget = new aws_cdk_lib_1.aws_events_targets.LambdaFunction(updater, {
            event: aws_cdk_lib_1.aws_events.RuleTargetInput.fromObject({
                lambdaName: this.function.functionName,
                repositoryUri: image.imageRepository.repositoryUri,
                repositoryTag: image.imageTag,
            }),
        });
        const rule = image.imageRepository.onEvent('Push rule', {
            description: 'Update GitHub Actions runner Lambda on ECR image push',
            eventPattern: {
                detailType: ['ECR Image Action'],
                detail: {
                    'action-type': ['PUSH'],
                    'repository-name': [image.imageRepository.repositoryName],
                    'image-tag': [image.imageTag],
                    'result': ['SUCCESS'],
                },
            },
            target: lambdaTarget,
        });
        // the event never triggers without this - not sure why
        rule.node.defaultChild.addDeletionOverride('Properties.EventPattern.resources');
    }
    imageDigest(image, variableSettings) {
        // describe ECR image to get its digest
        // the physical id is random so the resource always runs and always gets the latest digest, even if a scheduled build replaced the stack image
        const reader = new aws_cdk_lib_1.custom_resources.AwsCustomResource(this, 'Image Digest Reader', {
            onCreate: {
                service: 'ECR',
                action: 'describeImages',
                parameters: {
                    repositoryName: image.imageRepository.repositoryName,
                    imageIds: [
                        {
                            imageTag: image.imageTag,
                        },
                    ],
                },
                physicalResourceId: aws_cdk_lib_1.custom_resources.PhysicalResourceId.of('ImageDigest'),
            },
            onUpdate: {
                service: 'ECR',
                action: 'describeImages',
                parameters: {
                    repositoryName: image.imageRepository.repositoryName,
                    imageIds: [
                        {
                            imageTag: image.imageTag,
                        },
                    ],
                },
                physicalResourceId: aws_cdk_lib_1.custom_resources.PhysicalResourceId.of('ImageDigest'),
            },
            onDelete: {
                // this will NOT be called thanks to RemovalPolicy.RETAIN below
                // we only use this to force the custom resource to be called again and get a new digest
                service: 'fake',
                action: 'fake',
                parameters: variableSettings,
            },
            policy: aws_cdk_lib_1.custom_resources.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [image.imageRepository.repositoryArn],
            }),
            resourceType: 'Custom::EcrImageDigest',
            installLatestAwsSdk: false,
            logRetention: aws_logs_1.RetentionDays.ONE_MONTH,
        });
        const res = reader.node.tryFindChild('Resource');
        if (res) {
            // don't actually call the fake onDelete above
            res.applyRemovalPolicy(cdk.RemovalPolicy.RETAIN);
        }
        else {
            throw new Error('Resource not found in AwsCustomResource. Report this bug at https://github.com/CloudSnorkel/cdk-github-runners/issues.');
        }
        // return only the digest because CDK expects 'sha256:' literal above
        return cdk.Fn.split(':', reader.getResponseField('imageDetails.0.imageDigest'), 2)[1];
    }
}
exports.LambdaRunner = LambdaRunner;
_a = JSII_RTTI_SYMBOL_1;
LambdaRunner[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.LambdaRunner", version: "0.5.2" };
/**
 * Path to Dockerfile for Linux x64 with all the requirement for Lambda runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
 *
 * Available build arguments that can be set in the image builder:
 * * `BASE_IMAGE` sets the `FROM` line. This should be similar to public.ecr.aws/lambda/nodejs:14.
 * * `EXTRA_PACKAGES` can be used to install additional packages.
 */
LambdaRunner.LINUX_X64_DOCKERFILE_PATH = path.join(__dirname, 'docker-images', 'lambda', 'linux-x64');
/**
 * Path to Dockerfile for Linux ARM64 with all the requirement for Lambda runner. Use this Dockerfile unless you need to customize it further than allowed by hooks.
 *
 * Available build arguments that can be set in the image builder:
 * * `BASE_IMAGE` sets the `FROM` line. This should be similar to public.ecr.aws/lambda/nodejs:14.
 * * `EXTRA_PACKAGES` can be used to install additional packages.
 */
LambdaRunner.LINUX_ARM64_DOCKERFILE_PATH = path.join(__dirname, 'docker-images', 'lambda', 'linux-arm64');
//# sourceMappingURL=data:application/json;base64,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