"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CacheQueryStringBehavior = exports.CacheHeaderBehavior = exports.CacheCookieBehavior = exports.CachePolicy = void 0;
const core_1 = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
/**
 * A Cache Policy configuration.
 *
 * @stability stable
 * @resource AWS::CloudFront::CachePolicy
 */
class CachePolicy extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d;
        super(scope, id, {
            physicalName: props.cachePolicyName,
        });
        const cachePolicyName = (_a = props.cachePolicyName) !== null && _a !== void 0 ? _a : core_1.Names.uniqueId(this);
        if (!core_1.Token.isUnresolved(cachePolicyName) && !cachePolicyName.match(/^[\w-]+$/i)) {
            throw new Error(`'cachePolicyName' can only include '-', '_', and alphanumeric characters, got: '${props.cachePolicyName}'`);
        }
        const minTtl = ((_b = props.minTtl) !== null && _b !== void 0 ? _b : core_1.Duration.seconds(0)).toSeconds();
        const defaultTtl = Math.max(((_c = props.defaultTtl) !== null && _c !== void 0 ? _c : core_1.Duration.days(1)).toSeconds(), minTtl);
        const maxTtl = Math.max(((_d = props.maxTtl) !== null && _d !== void 0 ? _d : core_1.Duration.days(365)).toSeconds(), defaultTtl);
        const resource = new cloudfront_generated_1.CfnCachePolicy(this, 'Resource', {
            cachePolicyConfig: {
                name: cachePolicyName,
                comment: props.comment,
                minTtl,
                maxTtl,
                defaultTtl,
                parametersInCacheKeyAndForwardedToOrigin: this.renderCacheKey(props),
            },
        });
        this.cachePolicyId = resource.ref;
    }
    /**
     * Imports a Cache Policy from its id.
     *
     * @stability stable
     */
    static fromCachePolicyId(scope, id, cachePolicyId) {
        return new class extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.cachePolicyId = cachePolicyId;
            }
        }(scope, id);
    }
    /** Use an existing managed cache policy. */
    static fromManagedCachePolicy(managedCachePolicyId) {
        return new class {
            constructor() {
                this.cachePolicyId = managedCachePolicyId;
            }
        }();
    }
    renderCacheKey(props) {
        var _a, _b, _c, _d, _e;
        const cookies = (_a = props.cookieBehavior) !== null && _a !== void 0 ? _a : CacheCookieBehavior.none();
        const headers = (_b = props.headerBehavior) !== null && _b !== void 0 ? _b : CacheHeaderBehavior.none();
        const queryStrings = (_c = props.queryStringBehavior) !== null && _c !== void 0 ? _c : CacheQueryStringBehavior.none();
        return {
            cookiesConfig: {
                cookieBehavior: cookies.behavior,
                cookies: cookies.cookies,
            },
            headersConfig: {
                headerBehavior: headers.behavior,
                headers: headers.headers,
            },
            enableAcceptEncodingGzip: (_d = props.enableAcceptEncodingGzip) !== null && _d !== void 0 ? _d : false,
            enableAcceptEncodingBrotli: (_e = props.enableAcceptEncodingBrotli) !== null && _e !== void 0 ? _e : false,
            queryStringsConfig: {
                queryStringBehavior: queryStrings.behavior,
                queryStrings: queryStrings.queryStrings,
            },
        };
    }
}
exports.CachePolicy = CachePolicy;
/**
 * Optimize cache efficiency by minimizing the values that CloudFront includes in the cache key.
 *
 * Query strings and cookies are not included in the cache key, and only the normalized 'Accept-Encoding' header is included.
 *
 * @stability stable
 */
CachePolicy.CACHING_OPTIMIZED = CachePolicy.fromManagedCachePolicy('658327ea-f89d-4fab-a63d-7e88639e58f6');
/**
 * Optimize cache efficiency by minimizing the values that CloudFront includes in the cache key.
 *
 * Query strings and cookies are not included in the cache key, and only the normalized 'Accept-Encoding' header is included.
 * Disables cache compression.
 *
 * @stability stable
 */
CachePolicy.CACHING_OPTIMIZED_FOR_UNCOMPRESSED_OBJECTS = CachePolicy.fromManagedCachePolicy('b2884449-e4de-46a7-ac36-70bc7f1ddd6d');
/**
 * Disables caching.
 *
 * This policy is useful for dynamic content and for requests that are not cacheable.
 *
 * @stability stable
 */
CachePolicy.CACHING_DISABLED = CachePolicy.fromManagedCachePolicy('4135ea2d-6df8-44a3-9df3-4b5a84be39ad');
/**
 * Designed for use with an origin that is an AWS Elemental MediaPackage endpoint.
 *
 * @stability stable
 */
CachePolicy.ELEMENTAL_MEDIA_PACKAGE = CachePolicy.fromManagedCachePolicy('08627262-05a9-4f76-9ded-b50ca2e3a84f');
/**
 * Determines whether any cookies in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin.
 *
 * @stability stable
 */
class CacheCookieBehavior {
    constructor(behavior, cookies) {
        this.behavior = behavior;
        this.cookies = cookies;
    }
    /**
     * Cookies in viewer requests are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static none() { return new CacheCookieBehavior('none'); }
    /**
     * All cookies in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static all() { return new CacheCookieBehavior('all'); }
    /**
     * Only the provided `cookies` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static allowList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to allow must be provided');
        }
        return new CacheCookieBehavior('whitelist', cookies);
    }
    /**
     * All cookies except the provided `cookies` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static denyList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to deny must be provided');
        }
        return new CacheCookieBehavior('allExcept', cookies);
    }
}
exports.CacheCookieBehavior = CacheCookieBehavior;
/**
 * Determines whether any HTTP headers are included in the cache key and automatically included in requests that CloudFront sends to the origin.
 *
 * @stability stable
 */
class CacheHeaderBehavior {
    constructor(behavior, headers) {
        this.behavior = behavior;
        this.headers = headers;
    }
    /**
     * HTTP headers are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static none() { return new CacheHeaderBehavior('none'); }
    /**
     * Listed headers are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static allowList(...headers) {
        if (headers.length === 0) {
            throw new Error('At least one header to allow must be provided');
        }
        return new CacheHeaderBehavior('whitelist', headers);
    }
}
exports.CacheHeaderBehavior = CacheHeaderBehavior;
/**
 * Determines whether any URL query strings in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin.
 *
 * @stability stable
 */
class CacheQueryStringBehavior {
    constructor(behavior, queryStrings) {
        this.behavior = behavior;
        this.queryStrings = queryStrings;
    }
    /**
     * Query strings in viewer requests are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static none() { return new CacheQueryStringBehavior('none'); }
    /**
     * All query strings in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static all() { return new CacheQueryStringBehavior('all'); }
    /**
     * Only the provided `queryStrings` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static allowList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to allow must be provided');
        }
        return new CacheQueryStringBehavior('whitelist', queryStrings);
    }
    /**
     * All query strings except the provided `queryStrings` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static denyList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to deny must be provided');
        }
        return new CacheQueryStringBehavior('allExcept', queryStrings);
    }
}
exports.CacheQueryStringBehavior = CacheQueryStringBehavior;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2FjaGUtcG9saWN5LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY2FjaGUtcG9saWN5LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHdDQUFpRTtBQUVqRSxpRUFBd0Q7Ozs7Ozs7QUFzRnhELE1BQWEsV0FBWSxTQUFRLGVBQVE7Ozs7SUFrQ3ZDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBMEIsRUFBRTs7UUFDcEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixZQUFZLEVBQUUsS0FBSyxDQUFDLGVBQWU7U0FDcEMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxlQUFlLFNBQUcsS0FBSyxDQUFDLGVBQWUsbUNBQUksWUFBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN0RSxJQUFJLENBQUMsWUFBSyxDQUFDLFlBQVksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLEVBQUU7WUFDL0UsTUFBTSxJQUFJLEtBQUssQ0FBQyxtRkFBbUYsS0FBSyxDQUFDLGVBQWUsR0FBRyxDQUFDLENBQUM7U0FDOUg7UUFFRCxNQUFNLE1BQU0sR0FBRyxPQUFDLEtBQUssQ0FBQyxNQUFNLG1DQUFJLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLEVBQUUsQ0FBQztRQUNqRSxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLE9BQUMsS0FBSyxDQUFDLFVBQVUsbUNBQUksZUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsRUFBRSxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3hGLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBQyxLQUFLLENBQUMsTUFBTSxtQ0FBSSxlQUFRLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsU0FBUyxFQUFFLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFFdEYsTUFBTSxRQUFRLEdBQUcsSUFBSSxxQ0FBYyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDcEQsaUJBQWlCLEVBQUU7Z0JBQ2pCLElBQUksRUFBRSxlQUFlO2dCQUNyQixPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87Z0JBQ3RCLE1BQU07Z0JBQ04sTUFBTTtnQkFDTixVQUFVO2dCQUNWLHdDQUF3QyxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDO2FBQ3JFO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGFBQWEsR0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDO0lBQ3BDLENBQUM7Ozs7OztJQXpDTSxNQUFNLENBQUMsaUJBQWlCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsYUFBcUI7UUFDakYsT0FBTyxJQUFJLEtBQU0sU0FBUSxlQUFRO1lBQXRCOztnQkFDTyxrQkFBYSxHQUFHLGFBQWEsQ0FBQztZQUNoRCxDQUFDO1NBQUEsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDZixDQUFDO0lBRUQsNENBQTRDO0lBQ3BDLE1BQU0sQ0FBQyxzQkFBc0IsQ0FBQyxvQkFBNEI7UUFDaEUsT0FBTyxJQUFJO1lBQUE7Z0JBQ08sa0JBQWEsR0FBRyxvQkFBb0IsQ0FBQztZQUN2RCxDQUFDO1NBQUEsRUFBRSxDQUFDO0lBQ04sQ0FBQztJQWdDTyxjQUFjLENBQUMsS0FBdUI7O1FBQzVDLE1BQU0sT0FBTyxTQUFHLEtBQUssQ0FBQyxjQUFjLG1DQUFJLG1CQUFtQixDQUFDLElBQUksRUFBRSxDQUFDO1FBQ25FLE1BQU0sT0FBTyxTQUFHLEtBQUssQ0FBQyxjQUFjLG1DQUFJLG1CQUFtQixDQUFDLElBQUksRUFBRSxDQUFDO1FBQ25FLE1BQU0sWUFBWSxTQUFHLEtBQUssQ0FBQyxtQkFBbUIsbUNBQUksd0JBQXdCLENBQUMsSUFBSSxFQUFFLENBQUM7UUFFbEYsT0FBTztZQUNMLGFBQWEsRUFBRTtnQkFDYixjQUFjLEVBQUUsT0FBTyxDQUFDLFFBQVE7Z0JBQ2hDLE9BQU8sRUFBRSxPQUFPLENBQUMsT0FBTzthQUN6QjtZQUNELGFBQWEsRUFBRTtnQkFDYixjQUFjLEVBQUUsT0FBTyxDQUFDLFFBQVE7Z0JBQ2hDLE9BQU8sRUFBRSxPQUFPLENBQUMsT0FBTzthQUN6QjtZQUNELHdCQUF3QixRQUFFLEtBQUssQ0FBQyx3QkFBd0IsbUNBQUksS0FBSztZQUNqRSwwQkFBMEIsUUFBRSxLQUFLLENBQUMsMEJBQTBCLG1DQUFJLEtBQUs7WUFDckUsa0JBQWtCLEVBQUU7Z0JBQ2xCLG1CQUFtQixFQUFFLFlBQVksQ0FBQyxRQUFRO2dCQUMxQyxZQUFZLEVBQUUsWUFBWSxDQUFDLFlBQVk7YUFDeEM7U0FDRixDQUFDO0lBQ0osQ0FBQzs7QUFuRkgsa0NBb0ZDOzs7Ozs7OztBQTlFd0IsNkJBQWlCLEdBQUcsV0FBVyxDQUFDLHNCQUFzQixDQUFDLHNDQUFzQyxDQUFDLENBQUM7Ozs7Ozs7OztBQU0vRixzREFBMEMsR0FBRyxXQUFXLENBQUMsc0JBQXNCLENBQUMsc0NBQXNDLENBQUMsQ0FBQzs7Ozs7Ozs7QUFFeEgsNEJBQWdCLEdBQUcsV0FBVyxDQUFDLHNCQUFzQixDQUFDLHNDQUFzQyxDQUFDLENBQUM7Ozs7OztBQUU5RixtQ0FBdUIsR0FBRyxXQUFXLENBQUMsc0JBQXNCLENBQUMsc0NBQXNDLENBQUMsQ0FBQzs7Ozs7O0FBMEU5SCxNQUFhLG1CQUFtQjtJQXNDOUIsWUFBb0IsUUFBZ0IsRUFBRSxPQUFrQjtRQUN0RCxJQUFJLENBQUMsUUFBUSxHQUFHLFFBQVEsQ0FBQztRQUN6QixJQUFJLENBQUMsT0FBTyxHQUFHLE9BQU8sQ0FBQztJQUN6QixDQUFDOzs7Ozs7SUFwQ00sTUFBTSxDQUFDLElBQUksS0FBSyxPQUFPLElBQUksbUJBQW1CLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDOzs7Ozs7SUFLekQsTUFBTSxDQUFDLEdBQUcsS0FBSyxPQUFPLElBQUksbUJBQW1CLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDOzs7Ozs7SUFLdkQsTUFBTSxDQUFDLFNBQVMsQ0FBQyxHQUFHLE9BQWlCO1FBQzFDLElBQUksT0FBTyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDeEIsTUFBTSxJQUFJLEtBQUssQ0FBQywrQ0FBK0MsQ0FBQyxDQUFDO1NBQ2xFO1FBQ0QsT0FBTyxJQUFJLG1CQUFtQixDQUFDLFdBQVcsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUN2RCxDQUFDOzs7Ozs7SUFNTSxNQUFNLENBQUMsUUFBUSxDQUFDLEdBQUcsT0FBaUI7UUFDekMsSUFBSSxPQUFPLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUN4QixNQUFNLElBQUksS0FBSyxDQUFDLDhDQUE4QyxDQUFDLENBQUM7U0FDakU7UUFDRCxPQUFPLElBQUksbUJBQW1CLENBQUMsV0FBVyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQ3ZELENBQUM7Q0FXRjtBQTFDRCxrREEwQ0M7Ozs7OztBQUtELE1BQWEsbUJBQW1CO0lBZ0I5QixZQUFvQixRQUFnQixFQUFFLE9BQWtCO1FBQ3RELElBQUksQ0FBQyxRQUFRLEdBQUcsUUFBUSxDQUFDO1FBQ3pCLElBQUksQ0FBQyxPQUFPLEdBQUcsT0FBTyxDQUFDO0lBQ3pCLENBQUM7Ozs7OztJQWpCTSxNQUFNLENBQUMsSUFBSSxLQUFLLE9BQU8sSUFBSSxtQkFBbUIsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7Ozs7OztJQUV6RCxNQUFNLENBQUMsU0FBUyxDQUFDLEdBQUcsT0FBaUI7UUFDMUMsSUFBSSxPQUFPLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUN4QixNQUFNLElBQUksS0FBSyxDQUFDLCtDQUErQyxDQUFDLENBQUM7U0FDbEU7UUFDRCxPQUFPLElBQUksbUJBQW1CLENBQUMsV0FBVyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQ3ZELENBQUM7Q0FXRjtBQXBCRCxrREFvQkM7Ozs7OztBQU1ELE1BQWEsd0JBQXdCO0lBc0NuQyxZQUFvQixRQUFnQixFQUFFLFlBQXVCO1FBQzNELElBQUksQ0FBQyxRQUFRLEdBQUcsUUFBUSxDQUFDO1FBQ3pCLElBQUksQ0FBQyxZQUFZLEdBQUcsWUFBWSxDQUFDO0lBQ25DLENBQUM7Ozs7OztJQXBDTSxNQUFNLENBQUMsSUFBSSxLQUFLLE9BQU8sSUFBSSx3QkFBd0IsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7Ozs7OztJQUs5RCxNQUFNLENBQUMsR0FBRyxLQUFLLE9BQU8sSUFBSSx3QkFBd0IsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7Ozs7OztJQUs1RCxNQUFNLENBQUMsU0FBUyxDQUFDLEdBQUcsWUFBc0I7UUFDL0MsSUFBSSxZQUFZLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUM3QixNQUFNLElBQUksS0FBSyxDQUFDLHFEQUFxRCxDQUFDLENBQUM7U0FDeEU7UUFDRCxPQUFPLElBQUksd0JBQXdCLENBQUMsV0FBVyxFQUFFLFlBQVksQ0FBQyxDQUFDO0lBQ2pFLENBQUM7Ozs7OztJQU1NLE1BQU0sQ0FBQyxRQUFRLENBQUMsR0FBRyxZQUFzQjtRQUM5QyxJQUFJLFlBQVksQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzdCLE1BQU0sSUFBSSxLQUFLLENBQUMsb0RBQW9ELENBQUMsQ0FBQztTQUN2RTtRQUNELE9BQU8sSUFBSSx3QkFBd0IsQ0FBQyxXQUFXLEVBQUUsWUFBWSxDQUFDLENBQUM7SUFDakUsQ0FBQztDQVdGO0FBMUNELDREQTBDQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IER1cmF0aW9uLCBOYW1lcywgUmVzb3VyY2UsIFRva2VuIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IENmbkNhY2hlUG9saWN5IH0gZnJvbSAnLi9jbG91ZGZyb250LmdlbmVyYXRlZCc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBJQ2FjaGVQb2xpY3kge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY2FjaGVQb2xpY3lJZDogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIENhY2hlUG9saWN5UHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY2FjaGVQb2xpY3lOYW1lPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjb21tZW50Pzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRlZmF1bHRUdGw/OiBEdXJhdGlvbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbWluVHRsPzogRHVyYXRpb247XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG1heFR0bD86IER1cmF0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY29va2llQmVoYXZpb3I/OiBDYWNoZUNvb2tpZUJlaGF2aW9yO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGhlYWRlckJlaGF2aW9yPzogQ2FjaGVIZWFkZXJCZWhhdmlvcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBxdWVyeVN0cmluZ0JlaGF2aW9yPzogQ2FjaGVRdWVyeVN0cmluZ0JlaGF2aW9yO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGVuYWJsZUFjY2VwdEVuY29kaW5nR3ppcD86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGVuYWJsZUFjY2VwdEVuY29kaW5nQnJvdGxpPzogYm9vbGVhbjtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIENhY2hlUG9saWN5IGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJQ2FjaGVQb2xpY3kge1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQ0FDSElOR19PUFRJTUlaRUQgPSBDYWNoZVBvbGljeS5mcm9tTWFuYWdlZENhY2hlUG9saWN5KCc2NTgzMjdlYS1mODlkLTRmYWItYTYzZC03ZTg4NjM5ZTU4ZjYnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBDQUNISU5HX09QVElNSVpFRF9GT1JfVU5DT01QUkVTU0VEX09CSkVDVFMgPSBDYWNoZVBvbGljeS5mcm9tTWFuYWdlZENhY2hlUG9saWN5KCdiMjg4NDQ0OS1lNGRlLTQ2YTctYWMzNi03MGJjN2YxZGRkNmQnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBDQUNISU5HX0RJU0FCTEVEID0gQ2FjaGVQb2xpY3kuZnJvbU1hbmFnZWRDYWNoZVBvbGljeSgnNDEzNWVhMmQtNmRmOC00NGEzLTlkZjMtNGI1YTg0YmUzOWFkJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgRUxFTUVOVEFMX01FRElBX1BBQ0tBR0UgPSBDYWNoZVBvbGljeS5mcm9tTWFuYWdlZENhY2hlUG9saWN5KCcwODYyNzI2Mi0wNWE5LTRmNzYtOWRlZC1iNTBjYTJlM2E4NGYnKTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBmcm9tQ2FjaGVQb2xpY3lJZChzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBjYWNoZVBvbGljeUlkOiBzdHJpbmcpOiBJQ2FjaGVQb2xpY3kge1xuICAgIHJldHVybiBuZXcgY2xhc3MgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElDYWNoZVBvbGljeSB7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgY2FjaGVQb2xpY3lJZCA9IGNhY2hlUG9saWN5SWQ7XG4gICAgfShzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqIFVzZSBhbiBleGlzdGluZyBtYW5hZ2VkIGNhY2hlIHBvbGljeS4gKi9cbiAgcHJpdmF0ZSBzdGF0aWMgZnJvbU1hbmFnZWRDYWNoZVBvbGljeShtYW5hZ2VkQ2FjaGVQb2xpY3lJZDogc3RyaW5nKTogSUNhY2hlUG9saWN5IHtcbiAgICByZXR1cm4gbmV3IGNsYXNzIGltcGxlbWVudHMgSUNhY2hlUG9saWN5IHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBjYWNoZVBvbGljeUlkID0gbWFuYWdlZENhY2hlUG9saWN5SWQ7XG4gICAgfSgpO1xuICB9XG5cbiAgcHVibGljIHJlYWRvbmx5IGNhY2hlUG9saWN5SWQ6IHN0cmluZztcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQ2FjaGVQb2xpY3lQcm9wcyA9IHt9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICBwaHlzaWNhbE5hbWU6IHByb3BzLmNhY2hlUG9saWN5TmFtZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGNhY2hlUG9saWN5TmFtZSA9IHByb3BzLmNhY2hlUG9saWN5TmFtZSA/PyBOYW1lcy51bmlxdWVJZCh0aGlzKTtcbiAgICBpZiAoIVRva2VuLmlzVW5yZXNvbHZlZChjYWNoZVBvbGljeU5hbWUpICYmICFjYWNoZVBvbGljeU5hbWUubWF0Y2goL15bXFx3LV0rJC9pKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGAnY2FjaGVQb2xpY3lOYW1lJyBjYW4gb25seSBpbmNsdWRlICctJywgJ18nLCBhbmQgYWxwaGFudW1lcmljIGNoYXJhY3RlcnMsIGdvdDogJyR7cHJvcHMuY2FjaGVQb2xpY3lOYW1lfSdgKTtcbiAgICB9XG5cbiAgICBjb25zdCBtaW5UdGwgPSAocHJvcHMubWluVHRsID8/IER1cmF0aW9uLnNlY29uZHMoMCkpLnRvU2Vjb25kcygpO1xuICAgIGNvbnN0IGRlZmF1bHRUdGwgPSBNYXRoLm1heCgocHJvcHMuZGVmYXVsdFR0bCA/PyBEdXJhdGlvbi5kYXlzKDEpKS50b1NlY29uZHMoKSwgbWluVHRsKTtcbiAgICBjb25zdCBtYXhUdGwgPSBNYXRoLm1heCgocHJvcHMubWF4VHRsID8/IER1cmF0aW9uLmRheXMoMzY1KSkudG9TZWNvbmRzKCksIGRlZmF1bHRUdGwpO1xuXG4gICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ2ZuQ2FjaGVQb2xpY3kodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgY2FjaGVQb2xpY3lDb25maWc6IHtcbiAgICAgICAgbmFtZTogY2FjaGVQb2xpY3lOYW1lLFxuICAgICAgICBjb21tZW50OiBwcm9wcy5jb21tZW50LFxuICAgICAgICBtaW5UdGwsXG4gICAgICAgIG1heFR0bCxcbiAgICAgICAgZGVmYXVsdFR0bCxcbiAgICAgICAgcGFyYW1ldGVyc0luQ2FjaGVLZXlBbmRGb3J3YXJkZWRUb09yaWdpbjogdGhpcy5yZW5kZXJDYWNoZUtleShwcm9wcyksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGhpcy5jYWNoZVBvbGljeUlkID0gcmVzb3VyY2UucmVmO1xuICB9XG5cbiAgcHJpdmF0ZSByZW5kZXJDYWNoZUtleShwcm9wczogQ2FjaGVQb2xpY3lQcm9wcyk6IENmbkNhY2hlUG9saWN5LlBhcmFtZXRlcnNJbkNhY2hlS2V5QW5kRm9yd2FyZGVkVG9PcmlnaW5Qcm9wZXJ0eSB7XG4gICAgY29uc3QgY29va2llcyA9IHByb3BzLmNvb2tpZUJlaGF2aW9yID8/IENhY2hlQ29va2llQmVoYXZpb3Iubm9uZSgpO1xuICAgIGNvbnN0IGhlYWRlcnMgPSBwcm9wcy5oZWFkZXJCZWhhdmlvciA/PyBDYWNoZUhlYWRlckJlaGF2aW9yLm5vbmUoKTtcbiAgICBjb25zdCBxdWVyeVN0cmluZ3MgPSBwcm9wcy5xdWVyeVN0cmluZ0JlaGF2aW9yID8/IENhY2hlUXVlcnlTdHJpbmdCZWhhdmlvci5ub25lKCk7XG5cbiAgICByZXR1cm4ge1xuICAgICAgY29va2llc0NvbmZpZzoge1xuICAgICAgICBjb29raWVCZWhhdmlvcjogY29va2llcy5iZWhhdmlvcixcbiAgICAgICAgY29va2llczogY29va2llcy5jb29raWVzLFxuICAgICAgfSxcbiAgICAgIGhlYWRlcnNDb25maWc6IHtcbiAgICAgICAgaGVhZGVyQmVoYXZpb3I6IGhlYWRlcnMuYmVoYXZpb3IsXG4gICAgICAgIGhlYWRlcnM6IGhlYWRlcnMuaGVhZGVycyxcbiAgICAgIH0sXG4gICAgICBlbmFibGVBY2NlcHRFbmNvZGluZ0d6aXA6IHByb3BzLmVuYWJsZUFjY2VwdEVuY29kaW5nR3ppcCA/PyBmYWxzZSxcbiAgICAgIGVuYWJsZUFjY2VwdEVuY29kaW5nQnJvdGxpOiBwcm9wcy5lbmFibGVBY2NlcHRFbmNvZGluZ0Jyb3RsaSA/PyBmYWxzZSxcbiAgICAgIHF1ZXJ5U3RyaW5nc0NvbmZpZzoge1xuICAgICAgICBxdWVyeVN0cmluZ0JlaGF2aW9yOiBxdWVyeVN0cmluZ3MuYmVoYXZpb3IsXG4gICAgICAgIHF1ZXJ5U3RyaW5nczogcXVlcnlTdHJpbmdzLnF1ZXJ5U3RyaW5ncyxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgQ2FjaGVDb29raWVCZWhhdmlvciB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgbm9uZSgpIHsgcmV0dXJuIG5ldyBDYWNoZUNvb2tpZUJlaGF2aW9yKCdub25lJyk7IH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGFsbCgpIHsgcmV0dXJuIG5ldyBDYWNoZUNvb2tpZUJlaGF2aW9yKCdhbGwnKTsgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBhbGxvd0xpc3QoLi4uY29va2llczogc3RyaW5nW10pIHtcbiAgICBpZiAoY29va2llcy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQXQgbGVhc3Qgb25lIGNvb2tpZSB0byBhbGxvdyBtdXN0IGJlIHByb3ZpZGVkJyk7XG4gICAgfVxuICAgIHJldHVybiBuZXcgQ2FjaGVDb29raWVCZWhhdmlvcignd2hpdGVsaXN0JywgY29va2llcyk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGRlbnlMaXN0KC4uLmNvb2tpZXM6IHN0cmluZ1tdKSB7XG4gICAgaWYgKGNvb2tpZXMubGVuZ3RoID09PSAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0F0IGxlYXN0IG9uZSBjb29raWUgdG8gZGVueSBtdXN0IGJlIHByb3ZpZGVkJyk7XG4gICAgfVxuICAgIHJldHVybiBuZXcgQ2FjaGVDb29raWVCZWhhdmlvcignYWxsRXhjZXB0JywgY29va2llcyk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBiZWhhdmlvcjogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGNvb2tpZXM/OiBzdHJpbmdbXTtcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKGJlaGF2aW9yOiBzdHJpbmcsIGNvb2tpZXM/OiBzdHJpbmdbXSkge1xuICAgIHRoaXMuYmVoYXZpb3IgPSBiZWhhdmlvcjtcbiAgICB0aGlzLmNvb2tpZXMgPSBjb29raWVzO1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgQ2FjaGVIZWFkZXJCZWhhdmlvciB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgbm9uZSgpIHsgcmV0dXJuIG5ldyBDYWNoZUhlYWRlckJlaGF2aW9yKCdub25lJyk7IH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBhbGxvd0xpc3QoLi4uaGVhZGVyczogc3RyaW5nW10pIHtcbiAgICBpZiAoaGVhZGVycy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQXQgbGVhc3Qgb25lIGhlYWRlciB0byBhbGxvdyBtdXN0IGJlIHByb3ZpZGVkJyk7XG4gICAgfVxuICAgIHJldHVybiBuZXcgQ2FjaGVIZWFkZXJCZWhhdmlvcignd2hpdGVsaXN0JywgaGVhZGVycyk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGJlaGF2aW9yOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGhlYWRlcnM/OiBzdHJpbmdbXTtcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKGJlaGF2aW9yOiBzdHJpbmcsIGhlYWRlcnM/OiBzdHJpbmdbXSkge1xuICAgIHRoaXMuYmVoYXZpb3IgPSBiZWhhdmlvcjtcbiAgICB0aGlzLmhlYWRlcnMgPSBoZWFkZXJzO1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgQ2FjaGVRdWVyeVN0cmluZ0JlaGF2aW9yIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBub25lKCkgeyByZXR1cm4gbmV3IENhY2hlUXVlcnlTdHJpbmdCZWhhdmlvcignbm9uZScpOyB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBhbGwoKSB7IHJldHVybiBuZXcgQ2FjaGVRdWVyeVN0cmluZ0JlaGF2aW9yKCdhbGwnKTsgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGFsbG93TGlzdCguLi5xdWVyeVN0cmluZ3M6IHN0cmluZ1tdKSB7XG4gICAgaWYgKHF1ZXJ5U3RyaW5ncy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQXQgbGVhc3Qgb25lIHF1ZXJ5IHN0cmluZyB0byBhbGxvdyBtdXN0IGJlIHByb3ZpZGVkJyk7XG4gICAgfVxuICAgIHJldHVybiBuZXcgQ2FjaGVRdWVyeVN0cmluZ0JlaGF2aW9yKCd3aGl0ZWxpc3QnLCBxdWVyeVN0cmluZ3MpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGRlbnlMaXN0KC4uLnF1ZXJ5U3RyaW5nczogc3RyaW5nW10pIHtcbiAgICBpZiAocXVlcnlTdHJpbmdzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBdCBsZWFzdCBvbmUgcXVlcnkgc3RyaW5nIHRvIGRlbnkgbXVzdCBiZSBwcm92aWRlZCcpO1xuICAgIH1cbiAgICByZXR1cm4gbmV3IENhY2hlUXVlcnlTdHJpbmdCZWhhdmlvcignYWxsRXhjZXB0JywgcXVlcnlTdHJpbmdzKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBiZWhhdmlvcjogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHF1ZXJ5U3RyaW5ncz86IHN0cmluZ1tdO1xuXG4gIHByaXZhdGUgY29uc3RydWN0b3IoYmVoYXZpb3I6IHN0cmluZywgcXVlcnlTdHJpbmdzPzogc3RyaW5nW10pIHtcbiAgICB0aGlzLmJlaGF2aW9yID0gYmVoYXZpb3I7XG4gICAgdGhpcy5xdWVyeVN0cmluZ3MgPSBxdWVyeVN0cmluZ3M7XG4gIH1cbn1cbiJdfQ==