from pytholic.bible._exceptions import ChapterDoesntExistError, VerseDoesntExistError, StartVerseGreaterThanEndOneError, ChapterAndVerseMustBeIntError, EndVerseEqualToStartOne
from typing import Union


def get_text(chapter_list, chapter, verse_start, verse_end, maxc):
    """Receives the parameters passed to the Bible class

    :param chapter_list: List that contains all chapters of each of the
    gospels
    :type chapter_list: dict

    :param chapter: Dictionary that contains all verses of a specified
    chapter
    :type chapter: dict

    :param verse_start: It can be the verse or the start verse if a end
    verse is passed
    :type verse_start: int

    :param verse_end: The end verse if a start verse is passed
    :type verse_end: int

    :param maxc: The last chapter of a book, so we can prevent an empty
    list or string
    :type maxc: int

    :return: The verse if only one verse is passed or a list of verses if
    a start verse and an end verse is passed
    :rtype: str or list

    :raises ChapterAndVerseMustBeIntError: If either the chapter or the
    verse isn't an int
    :raises ChapterDoesntExistErro: If the chapter passed as a parameter
    isn't in the chapter list
    :raises StartVerseGreaterThanEndOneError: If the start verse if greater
    than the end verse
    :raises VerseDoesntExistError: If the verse passed as a parameter isn't
    in the verse dictionary
    :raises EndVerseEqualToStartOne: If the end verse and the start verse
    have the same value
    """
    if type(chapter) != int or type(verse_start) != int:
        raise ChapterAndVerseMustBeIntError

    if chapter > maxc:
        raise ChapterDoesntExistError

    if verse_end != None and type(verse_end) != int:
        raise ChapterAndVerseMustBeIntError

    if verse_end != None:
        if verse_start > verse_end:
            raise StartVerseGreaterThanEndOneError
        if verse_start == verse_end:
            raise EndVerseEqualToStartOne

    if verse_end == None:
        for chapter_num, chapter_content in chapter_list.items():
            if chapter_num == chapter:
                for verse_num, verse_content in chapter_content.items():
                    if verse_num == verse_start:
                        return verse_content
                raise VerseDoesntExistError

    verse_index = verse_start
    verses = []
    for i in range(verse_start, verse_end + 1):
        for chapter_num, chapter_content in chapter_list.items():
            if chapter_num == chapter:
                try:
                    verse_formatted = f"{verse_index}. " + \
                        chapter_content[i]
                    verses.append(verse_formatted)
                    verse_index += 1
                except KeyError as e:
                    raise VerseDoesntExistError
    return verses


class BibleBooks:
    """Contains the 4 gospels of the Bible"""

    def matthew(self, chapter: int, verse_start: int, verse_end: int = None) -> Union[str, list]:
        """
        :param chapter: The chapter number
        :type chapter: int

        :param verse_start: The first verse
        :type verse_start: int

        :param verse_end: Alternatively, an end verse can be passed or let it
        as None to return only one verse
        :type verse_end: int or None
        """
        chapter_1 = {
            1: "Roll of the genealogy of Jesus Christ, son of David, son of Abraham:",
            2: "Abraham fathered Isaac, Isaac fathered Jacob, Jacob fathered Judah and his brothers,",
            3: "Judah fathered Perez and Zerah, whose mother was Tamar, Perez fathered Hezron, Hezron fathered Ram,",
            4: "Ram fathered Amminadab, Amminadab fathered Nahshon, Nahshon fathered Salmon,",
            5: "Salmon fathered Boaz, whose mother was Rahab, Boaz fathered Obed, whose mother was Ruth, Obed fathered Jesse;",
            6: "and Jesse fathered King David. David fathered Solomon, whose mother had been Uriah's wife,",
            7: "Solomon fathered Rehoboam, Rehoboam fathered Abijah, Abijah fathered Asa,",
            8: "Asa fathered Jehoshaphat, Jehoshaphat fathered Joram, Joram fathered Uzziah,",
            9: "Uzziah fathered Jotham, Jotham fathered Ahaz, Ahaz fathered Hezekiah,",
            10: "Hezekiah fathered Manasseh, Manasseh fathered Amon, Amon fathered Josiah;",
            11: "and Josiah fathered Jechoniah and his brothers. Then the deportation to Babylon took place.",
            12: "After the deportation to Babylon: Jechoniah fathered Shealtiel, Shealtiel fathered Zerubbabel,",
            13: "Zerubbabel fathered Abiud, Abiud fathered Eliakim, Eliakim fathered Azor,",
            14: "Azor fathered Zadok, Zadok fathered Achim, Achim fathered Eliud,",
            15: "Eliud fathered Eleazar, Eleazar fathered Matthan, Matthan fathered Jacob;",
            16: "and Jacob fathered Joseph the husband of Mary; of her was born Jesus who is called Christ.",
            17: "The sum of generations is therefore: fourteen from Abraham to David; fourteen from David to the Babylonian deportation; and fourteen from the Babylonian deportation to Christ.",
            18: "This is how Jesus Christ came to be born. His mother Mary was betrothed to Joseph; but before they came to live together she was found to be with child through the Holy Spirit.",
            19: "Her husband Joseph, being an upright man and wanting to spare her disgrace, decided to divorce her informally.",
            20: "He had made up his mind to do this when suddenly the angel of the Lord appeared to him in a dream and said, 'Joseph son of David, do not be afraid to take Mary home as your wife, because she has conceived what is in her by the Holy Spirit.",
            21: "She will give birth to a son and you must name him Jesus, because he is the one who is to save his people from their sins.'",
            22: "Now all this took place to fulfil what the Lord had spoken through the prophet:",
            23: "Look! the virgin is with child and will give birth to a son whom they will call Immanuel, a name which means 'God-is-with-us'.",
            24: "When Joseph woke up he did what the angel of the Lord had told him to do: he took his wife to his home;",
            25: "he had not had intercourse with her when she gave birth to a son; and he named him Jesus."
        }

        chapter_2 = {
            1: "After Jesus had been born at Bethlehem in Judaea during the reign of King Herod, suddenly some wise men came to Jerusalem from the east",
            2: "asking, 'Where is the infant king of the Jews? We saw his star as it rose and have come to do him homage.'",
            3: "When King Herod heard this he was perturbed, and so was the whole of Jerusalem.",
            4: "He called together all the chief priests and the scribes of the people, and enquired of them where the Christ was to be born.",
            5: "They told him, 'At Bethlehem in Judaea, for this is what the prophet wrote:",
            6: "And you, Bethlehem, in the land of Judah, you are by no means the least among the leaders of Judah, for from you will come a leader who will shepherd my people Israel.'",
            7: "Then Herod summoned the wise men to see him privately. He asked them the exact date on which the star had appeared",
            8: "and sent them on to Bethlehem with the words, 'Go and find out all about the child, and when you have found him, let me know, so that I too may go and do him homage.'",
            9: "Having listened to what the king had to say, they set out. And suddenly the star they had seen rising went forward and halted over the place where the child was.",
            10: "The sight of the star filled them with delight,",
            11: "and going into the house they saw the child with his mother Mary, and falling to their knees they did him homage. Then, opening their treasures, they offered him gifts of gold and frankincense and myrrh.",
            12: "But they were given a warning in a dream not to go back to Herod, and returned to their own country by a different way.",
            13: "After they had left, suddenly the angel of the Lord appeared to Joseph in a dream and said, 'Get up, take the child and his mother with you, and escape into Egypt, and stay there until I tell you, because Herod intends to search for the child and do away with him.'",
            14: "So Joseph got up and, taking the child and his mother with him, left that night for Egypt,",
            15: "where he stayed until Herod was dead. This was to fulfil what the Lord had spoken through the prophet: I called my son out of Egypt.",
            16: "Herod was furious on realising that he had been fooled by the wise men, and in Bethlehem and its surrounding district he had all the male children killed who were two years old or less, reckoning by the date he had been careful to ask the wise men.",
            17: "Then were fulfilled the words spoken through the prophet Jeremiah:",
            18: "A voice is heard in Ramah, lamenting and weeping bitterly: it is Rachel weeping for her children, refusing to be comforted because they are no more.",
            19: "After Herod's death, suddenly the angel of the Lord appeared in a dream to Joseph in Egypt",
            20: "and said, 'Get up, take the child and his mother with you and go back to the land of Israel, for those who wanted to kill the child are dead.'",
            21: "So Joseph got up and, taking the child and his mother with him, went back to the land of Israel.",
            22: "But when he learnt that Archelaus had succeeded his father Herod as ruler of Judaea he was afraid to go there, and being warned in a dream he withdrew to the region of Galilee.",
            23: "There he settled in a town called Nazareth. In this way the words spoken through the prophets were to be fulfilled: He will be called a Nazarene."
        }

        chapter_3 = {
            1: "In due course John the Baptist appeared; he proclaimed this message in the desert of Judaea,",
            2: "'Repent, for the kingdom of Heaven is close at hand.'",
            3: "This was the man spoken of by the prophet Isaiah when he said: A voice of one that cries in the desert, 'Prepare a way for the Lord, make his paths straight.'",
            4: "This man John wore a garment made of camel-hair with a leather loin-cloth round his waist, and his food was locusts and wild honey.",
            5: "Then Jerusalem and all Judaea and the whole Jordan district made their way to him,",
            6: "and as they were baptised by him in the river Jordan they confessed their sins.",
            7: "But when he saw a number of Pharisees and Sadducees coming for baptism he said to them, 'Brood of vipers, who warned you to flee from the coming retribution?",
            8: "Produce fruit in keeping with repentance,",
            9: 'and do not presume to tell yourselves, "We have Abraham as our father," because, I tell you, God can raise children for Abraham from these stones.',
            10: "Even now the axe is being laid to the root of the trees, so that any tree failing to produce good fruit will be cut down and thrown on the fire.",
            11: "I baptise you in water for repentance, but the one who comes after me is more powerful than I, and I am not fit to carry his sandals; he will baptise you with the Holy Spirit and fire.",
            12: "His winnowing-fan is in his hand; he will clear his threshing-floor and gather his wheat into his barn; but the chaff he will burn in a fire that will never go out.'",
            13: "Then Jesus appeared: he came from Galilee to the Jordan to be baptised by John.",
            14: "John tried to dissuade him, with the words, 'It is I who need baptism from you, and yet you come to me!'",
            15: "But Jesus replied, 'Leave it like this for the time being; it is fitting that we should, in this way, do all that uprightness demands.' Then John gave in to him.",
            16: "And when Jesus had been baptised he at once came up from the water, and suddenly the heavens opened and he saw the Spirit of God descending like a dove and coming down on him.",
            17: "And suddenly there was a voice from heaven, 'This is my Son, the Beloved; my favour rests on him.'"
        }

        chapter_4 = {
            1: "Then Jesus was led by the Spirit out into the desert to be put to the test by the devil.",
            2: "He fasted for forty days and forty nights, after which he was hungry,",
            3: "and the tester came and said to him, 'If you are Son of God, tell these stones to turn into loaves.'",
            4: "But he replied, 'Scripture says: Human beings live not on bread alone but on every word that comes from the mouth of God.'",
            5: "The devil then took him to the holy city and set him on the parapet of the Temple.",
            6: "'If you are Son of God,' he said, 'throw yourself down; for scripture says: He has given his angels orders about you, and they will carry you in their arms in case you trip over a stone.'",
            7: "Jesus said to him, 'Scripture also says: Do not put the Lord your God to the test.'",
            8: "Next, taking him to a very high mountain, the devil showed him all the kingdoms of the world and their splendour.",
            9: "And he said to him, 'I will give you all these, if you fall at my feet and do me homage.'",
            10: "Then Jesus replied, 'Away with you, Satan! For scripture says: The Lord your God is the one to whom you must do homage, him alone you must serve.'",
            11: "Then the devil left him, and suddenly angels appeared and looked after him.",
            12: "Hearing that John had been arrested he withdrew to Galilee,",
            13: "and leaving Nazara he went and settled in Capernaum, beside the lake, on the borders of Zebulun and Naphtali.",
            14: "This was to fulfil what was spoken by the prophet Isaiah:",
            15: "Land of Zebulun! Land of Naphtali! Way of the sea beyond Jordan. Galilee of the nations!",
            16: "The people that lived in darkness have seen a great light; on those who lived in a country of shadow dark as death a light has dawned.",
            17: "From then onwards Jesus began his proclamation with the message, 'Repent, for the kingdom of Heaven is close at hand.'",
            18: "As he was walking by the Lake of Galilee he saw two brothers, Simon, who was called Peter, and his brother Andrew; they were making a cast into the lake with their net, for they were fishermen.",
            19: "And he said to them, 'Come after me and I will make you fishers of people.'",
            20: "And at once they left their nets and followed him.",
            21: "Going on from there he saw another pair of brothers, James son of Zebedee and his brother John; they were in their boat with their father Zebedee, mending their nets, and he called them.",
            22: "And at once, leaving the boat and their father, they followed him.",
            23: "He went round the whole of Galilee teaching in their synagogues, proclaiming the good news of the kingdom and curing all kinds of disease and illness among the people.",
            24: "His fame spread throughout Syria, and those who were suffering from diseases and painful complaints of one kind or another, the possessed, epileptics, the paralysed, were all brought to him, and he cured them.",
            25: "Large crowds followed him, coming from Galilee, the Decapolis, Jerusalem, Judaea and Transjordan."
        }

        chapter_5 = {
            1: "Seeing the crowds, he went onto the mountain. And when he was seated his disciples came to him.",
            2: "Then he began to speak. This is what he taught them:",
            3: "How blessed are the poor in spirit: the kingdom of Heaven is theirs.",
            4: "Blessed are the gentle: they shall have the earth as inheritance.",
            5: "Blessed are those who mourn: they shall be comforted.",
            6: "Blessed are those who hunger and thirst for uprightness: they shall have their fill.",
            7: "Blessed are the merciful: they shall have mercy shown them.",
            8: "Blessed are the pure in heart: they shall see God.",
            9: "Blessed are the peacemakers: they shall be recognised as children of God.",
            10: "Blessed are those who are persecuted in the cause of uprightness: the kingdom of Heaven is theirs.",
            11: "'Blessed are you when people abuse you and persecute you and speak all kinds of calumny against you falsely on my account.",
            12: "Rejoice and be glad, for your reward will be great in heaven; this is how they persecuted the prophets before you.",
            13: "'You are salt for the earth. But if salt loses its taste, what can make it salty again? It is good for nothing, and can only be thrown out to be trampled under people's feet.",
            14: "'You are light for the world. A city built on a hill-top cannot be hidden.",
            15: "No one lights a lamp to put it under a tub; they put it on the lamp-stand where it shines for everyone in the house.",
            16: "In the same way your light must shine in people's sight, so that, seeing your good works, they may give praise to your Father in heaven.",
            17: "'Do not imagine that I have come to abolish the Law or the Prophets. I have come not to abolish but to complete them.",
            18: "In truth I tell you, till heaven and earth disappear, not one dot, not one little stroke, is to disappear from the Law until all its purpose is achieved",
            19: "Therefore, anyone who infringes even one of the least of these commandments and teaches others to do the same will be considered the least in the kingdom of Heaven; but the person who keeps them and teaches them will be considered great in the kingdom of Heaven.",
            20: "'For I tell you, if your uprightness does not surpass that of the scribes and Pharisees, you will never get into the kingdom of Heaven.",
            21: "'You have heard how it was said to our ancestors, You shall not kill; and if anyone does kill he must answer for it before the court.",
            22: 'But I say this to you, anyone who is angry with a brother will answer for it before the court; anyone who calls a brother "Fool" will answer for it before the Sanhedrin; and anyone who calls him "Traitor" will answer for it in hell fire.',
            23: "So then, if you are bringing your offering to the altar and there remember that your brother has something against you,",
            24: "leave your offering there before the altar, go and be reconciled with your brother first, and then come back and present your offering.",
            25: "Come to terms with your opponent in good time while you are still on the way to the court with him, or he may hand you over to the judge and the judge to the officer, and you will be thrown into prison.",
            26: "In truth I tell you, you will not get out till you have paid the last penny.",
            27: "'You have heard how it was said, You shall not commit adultery.",
            28: "But I say this to you, if a man looks at a woman lustfully, he has already committed adultery with her in his heart.",
            29: "If your right eye should be your downfall, tear it out and throw it away; for it will do you less harm to lose one part of yourself than to have your whole body thrown into hell.",
            30: "And if your right hand should be your downfall, cut it off and throw it away; for it will do you less harm to lose one part of yourself than to have your whole body go to hell.",
            31: "'It has also been said, Anyone who divorces his wife must give her a writ of dismissal.",
            32: "But I say this to you, everyone who divorces his wife, except for the case of an illicit marriage, makes her an adulteress; and anyone who marries a divorced woman commits adultery.",
            33: "'Again, you have heard how it was said to our ancestors, You must not break your oath, but must fulfil your oaths to the Lord.",
            34: "But I say this to you, do not swear at all, either by heaven, since that is God's throne;",
            35: "or by earth, since that is his footstool; or by Jerusalem, since that is the city of the great King.",
            36: "Do not swear by your own head either, since you cannot turn a single hair white or black.",
            37: 'All you need say is "Yes" if you mean yes, "No" if you mean no; anything more than this comes from the Evil One.',
            38: "'You have heard how it was said: Eye for eye and tooth for tooth.",
            39: "But I say this to you: offer no resistance to the wicked. On the contrary, if anyone hits you on the right cheek, offer him the other as well;",
            40: "if someone wishes to go to law with you to get your tunic, let him have your cloak as well.",
            41: "And if anyone requires you to go one mile, go two miles with him.",
            42: "Give to anyone who asks you, and if anyone wants to borrow, do not turn away.",
            43: "'You have heard how it was said, You will love your neighbour and hate your enemy.",
            44: "But I say this to you, love your enemies and pray for those who persecute you;",
            45: "so that you may be children of your Father in heaven, for he causes his sun to rise on the bad as well as the good, and sends down rain to fall on the upright and the wicked alike.",
            46: "For if you love those who love you, what reward will you get? Do not even the tax collectors do as much?",
            47: "And if you save your greetings for your brothers, are you doing anything exceptional?",
            48: "Do not even the gentiles do as much? You must therefore be perfect, just as your heavenly Father is perfect.'"
        }

        chapter_6 = {
            1: "Be careful not to parade your uprightness in public to attract attention; otherwise you will lose all reward from your Father in heaven.",
            2: "So when you give alms, do not have it trumpeted before you; this is what the hypocrites do in the synagogues and in the streets to win human admiration. In truth I tell you, they have had their reward.",
            3: "But when you give alms, your left hand must not know what your right is doing;",
            4: "your almsgiving must be secret, and your Father who sees all that is done in secret will reward you.",
            5: "'And when you pray, do not imitate the hypocrites: they love to say their prayers standing up in the synagogues and at the street corners for people to see them. In truth I tell you, they have had their reward.",
            6: "But when you pray, go to your private room, shut yourself in, and so pray to your Father who is in that secret place, and your Father who sees all that is done in secret will reward you.",
            7: "'In your prayers do not babble as the gentiles do, for they think that by using many words they will make themselves heard.",
            8: "Do not be like them; your Father knows what you need before you ask him.",
            9: "So you should pray like this: Our Father in heaven, may your name be held holy,",
            10: "your kingdom come, your will be done, on earth as in heaven.",
            11: "Give us today our daily bread.",
            12: "And forgive us our debts, as we have forgiven those who are in debt to us.",
            13: "And do not put us to the test, but save us from the Evil One.",
            14: "'Yes, if you forgive others their failings, your heavenly Father will forgive you yours;",
            15: "but if you do not forgive others, your Father will not forgive your failings either.",
            16: "'When you are fasting, do not put on a gloomy look as the hypocrites do: they go about looking unsightly to let people know they are fasting. In truth I tell you, they have had their reward.",
            17: "But when you fast, put scent on your head and wash your face,",
            18: "so that no one will know you are fasting except your Father who sees all that is done in secret; and your Father who sees all that is done in secret will reward you.",
            19: "'Do not store up treasures for yourselves on earth, where moth and woodworm destroy them and thieves can break in and steal.",
            20: "But store up treasures for yourselves in heaven, where neither moth nor woodworm destroys them and thieves cannot break in and steal.",
            21: "For wherever your treasure is, there will your heart be too.",
            22: "'The lamp of the body is the eye. It follows that if your eye is clear, your whole body will be filled with light.",
            23: "But if your eye is diseased, your whole body will be darkness. If then, the light inside you is darkened, what darkness that will be!",
            24: "'No one can be the slave of two masters: he will either hate the first and love the second, or be attached to the first and despise the second. You cannot be the slave both of God and of money.",
            25: "'That is why I am telling you not to worry about your life and what you are to eat, nor about your body and what you are to wear. Surely life is more than food, and the body more than clothing!",
            26: "Look at the birds in the sky. They do not sow or reap or gather into barns; yet your heavenly Father feeds them. Are you not worth much more than they are?",
            27: "Can any of you, however much you worry, add one single cubit to your span of life?",
            28: "And why worry about clothing? Think of the flowers growing in the fields; they never have to work or spin;",
            29: "yet I assure you that not even Solomon in all his royal robes was clothed like one of these.",
            30: "Now if that is how God clothes the wild flowers growing in the field which are there today and thrown into the furnace tomorrow, will he not much more look after you, you who have so little faith?",
            31: 'So do not worry; do not say, "What are we to eat? What are we to drink? What are we to wear?"',
            32: "It is the gentiles who set their hearts on all these things. Your heavenly Father knows you need them all.",
            33: "Set your hearts on his kingdom first, and on God's saving justice, and all these other things will be given you as well.",
            34: "So do not worry about tomorrow: tomorrow will take care of itself. Each day has enough trouble of its own.'"
        }

        chapter_7 = {
            1: "'Do not judge, and you will not be judged;",
            2: "because the judgements you give are the judgements you will get, and the standard you use will be the standard used for you.",
            3: "Why do you observe the splinter in your brother's eye and never notice the great log in your own?",
            4: 'And how dare you say to your brother, "Let me take that splinter out of your eye," when, look, there is a great log in your own?',
            5: "Hypocrite! Take the log out of your own eye first, and then you will see clearly enough to take the splinter out of your brother's eye.",
            6: "'Do not give dogs what is holy; and do not throw your pearls in front of pigs, or they may trample them and then turn on you and tear you to pieces.",
            7: "'Ask, and it will be given to you; search, and you will find; knock, and the door will be opened to you.",
            8: "Everyone who asks receives; everyone who searches finds; everyone who knocks will have the door opened.",
            9: "Is there anyone among you who would hand his son a stone when he asked for bread?",
            10: "Or would hand him a snake when he asked for a fish?",
            11: "If you, then, evil as you are, know how to give your children what is good, how much more will your Father in heaven give good things to those who ask him!",
            12: "'So always treat others as you would like them to treat you; that is the Law and the Prophets.",
            13: "'Enter by the narrow gate, since the road that leads to destruction is wide and spacious, and many take it;",
            14: "but it is a narrow gate and a hard road that leads to life, and only a few find it.",
            15: "'Beware of false prophets who come to you disguised as sheep but underneath are ravenous wolves.",
            16: "You will be able to tell them by their fruits. Can people pick grapes from thorns, or figs from thistles?",
            17: "In the same way, a sound tree produces good fruit but a rotten tree bad fruit.",
            18: "A sound tree cannot bear bad fruit, nor a rotten tree bear good fruit.",
            19: "Any tree that does not produce good fruit is cut down and thrown on the fire.",
            20: "I repeat, you will be able to tell them by their fruits.",
            21: """'It is not anyone who says to me, "Lord, Lord," who will enter the kingdom of Heaven, but the person who does the will of my Father in heaven.""",
            22: 'When the day comes many will say to me, "Lord, Lord, did we not prophesy in your name, drive out demons in your name, work many miracles in your name?"',
            23: "Then I shall tell them to their faces: I have never known you; away from me, all evil doers!",
            24: "'Therefore, everyone who listens to these words of mine and acts on them will be like a sensible man who built his house on rock.",
            25: "Rain came down, floods rose, gales blew and hurled themselves against that house, and it did not fall: it was founded on rock.",
            26: "But everyone who listens to these words of mine and does not act on them will be like a stupid man who built his house on sand.",
            27: "Rain came down, floods rose, gales blew and struck that house, and it fell; and what a fall it had!'",
            28: "Jesus had now finished what he wanted to say, and his teaching made a deep impression on the people",
            29: "because he taught them with authority, unlike their own scribes."
        }

        chapter_8 = {
            1: "After he had come down from the mountain large crowds followed him.",
            2: "Suddenly a man with a virulent skin-disease came up and bowed low in front of him, saying, 'Lord, if you are willing, you can cleanse me.'",
            3: "Jesus stretched out his hand and touched him saying, 'I am willing. Be cleansed.' And his skin-disease was cleansed at once.",
            4: "Then Jesus said to him, 'Mind you tell no one, but go and show yourself to the priest and make the offering prescribed by Moses, as evidence to them.'",
            5: "When he went into Capernaum a centurion came up and pleaded with him.",
            6: "'Sir,' he said, 'my servant is lying at home paralysed and in great pain.'",
            7: "Jesus said to him, 'I will come myself and cure him.'",
            8: "The centurion replied, 'Sir, I am not worthy to have you under my roof; just give the word and my servant will be cured.",
            9: """For I am under authority myself and have soldiers under me; and I say to one man, "Go," and he goes; to another, "Come here," and he comes; to my servant, "Do this," and he does it.'""",
            10: "When Jesus heard this he was astonished and said to those following him, 'In truth I tell you, in no one in Israel have I found faith as great as this.",
            11: "And I tell you that many will come from east and west and sit down with Abraham and Isaac and Jacob at the feast in the kingdom of Heaven;",
            12: "but the children of the kingdom will be thrown out into the darkness outside, where there will be weeping and grinding of teeth.'",
            13: "And to the centurion Jesus said, 'Go back, then; let this be done for you, as your faith demands.' And the servant was cured at that moment.",
            14: "And going into Peter's house Jesus found Peter's mother-in-law in bed and feverish.",
            15: "He touched her hand and the fever left her, and she got up and began to serve him.",
            16: "That evening they brought him many who were possessed by devils. He drove out the spirits with a command and cured all who were sick.",
            17: "This was to fulfil what was spoken by the prophet Isaiah: He himself bore our sicknesses away and carried our diseases.",
            18: "When Jesus saw the crowd all about him he gave orders to leave for the other side.",
            19: "One of the scribes then came up and said to him, 'Master, I will follow you wherever you go.'",
            20: "Jesus said, 'Foxes have holes and the birds of the air have nests, but the Son of man has nowhere to lay his head.'",
            21: "Another man, one of the disciples, said to him, 'Lord, let me go and bury my father first.'",
            22: "But Jesus said, 'Follow me, and leave the dead to bury their dead.'",
            23: "Then he got into the boat followed by his disciples.",
            24: "Suddenly a storm broke over the lake, so violent that the boat was being swamped by the waves. But he was asleep.",
            25: "So they went to him and woke him saying, 'Save us, Lord, we are lost!'",
            26: "And he said to them, 'Why are you so frightened, you who have so little faith?' And then he stood up and rebuked the winds and the sea; and there was a great calm.",
            27: "They were astounded and said, 'Whatever kind of man is this, that even the winds and the sea obey him?'",
            28: "When he reached the territory of the Gadarenes on the other side, two demoniacs came towards him out of the tombs -- they were so dangerously violent that nobody could use that path.",
            29: "Suddenly they shouted, 'What do you want with us, Son of God? Have you come here to torture us before the time?'",
            30: "Now some distance away there was a large herd of pigs feeding,",
            31: "and the devils pleaded with Jesus, 'If you drive us out, send us into the herd of pigs.'",
            32: "And he said to them, 'Go then,' and they came out and made for the pigs; and at that the whole herd charged down the cliff into the lake and perished in the water.",
            33: "The herdsmen ran off and made for the city, where they told the whole story, including what had happened to the demoniacs.",
            34: "Suddenly the whole city set out to meet Jesus; and as soon as they saw him they implored him to leave their neighbourhood."
        }

        chapter_9 = {
            1: "He got back in the boat, crossed the water and came to his home town.",
            2: "And suddenly some people brought him a paralytic stretched out on a bed. Seeing their faith, Jesus said to the paralytic, 'Take comfort, my child, your sins are forgiven.'",
            3: "And now some scribes said to themselves, 'This man is being blasphemous.'",
            4: "Knowing what was in their minds Jesus said, 'Why do you have such wicked thoughts in your hearts?",
            5: 'Now, which of these is easier: to say, "Your sins are forgiven," or to say, "Get up and walk"?',
            6: "But to prove to you that the Son of man has authority on earth to forgive sins,' -- then he said to the paralytic-'get up, pick up your bed and go off home.'",
            7: "And the man got up and went home.",
            8: "A feeling of awe came over the crowd when they saw this, and they praised God for having given such authority to human beings.",
            9: "As Jesus was walking on from there he saw a man named Matthew sitting at the tax office, and he said to him, 'Follow me.' And he got up and followed him.",
            10: "Now while he was at table in the house it happened that a number of tax collectors and sinners came to sit at the table with Jesus and his disciples.",
            11: "When the Pharisees saw this, they said to his disciples, 'Why does your master eat with tax collectors and sinners?'",
            12: "When he heard this he replied, 'It is not the healthy who need the doctor, but the sick.",
            13: "Go and learn the meaning of the words: Mercy is what pleases me, not sacrifice. And indeed I came to call not the upright, but sinners.'",
            14: "Then John's disciples came to him and said, 'Why is it that we and the Pharisees fast, but your disciples do not?'",
            15: "Jesus replied, 'Surely the bridegroom's attendants cannot mourn as long as the bridegroom is still with them? But the time will come when the bridegroom is taken away from them, and then they will fast.",
            16: "No one puts a piece of unshrunken cloth onto an old cloak, because the patch pulls away from the cloak and the tear gets worse.",
            17: "Nor do people put new wine into old wineskins; otherwise, the skins burst, the wine runs out, and the skins are lost. No; they put new wine in fresh skins and both are preserved.'",
            18: "While he was speaking to them, suddenly one of the officials came up, who bowed low in front of him and said, 'My daughter has just died, but come and lay your hand on her and her life will be saved.'",
            19: "Jesus rose and, with his disciples, followed him.",
            20: "Then suddenly from behind him came a woman, who had been suffering from a haemorrhage for twelve years, and she touched the fringe of his cloak,",
            21: "for she was thinking, 'If only I can touch his cloak I shall be saved.'",
            22: "Jesus turned round and saw her; and he said to her, 'Courage, my daughter, your faith has saved you.' And from that moment the woman was saved.",
            23: "When Jesus reached the official's house and saw the flute-players, with the crowd making a commotion, he said,",
            24: "'Get out of here; the little girl is not dead; she is asleep.' And they ridiculed him.",
            25: "But when the people had been turned out he went inside and took her by the hand; and she stood up.",
            26: "And the news of this spread all round the countryside.",
            27: "As Jesus went on his way two blind men followed him shouting, 'Take pity on us, son of David.'",
            28: "And when Jesus reached the house the blind men came up to him and he said to them, 'Do you believe I can do this?' They said, 'Lord, we do.'",
            29: "Then he touched their eyes saying, 'According to your faith, let it be done to you.'",
            30: "And their sight returned. Then Jesus sternly warned them, 'Take care that no one learns about this.'",
            31: "But when they had gone away, they talked about him all over the countryside.",
            32: "They had only just left when suddenly a man was brought to him, a dumb demoniac.",
            33: "And when the devil was driven out, the dumb man spoke and the people were amazed and said, 'Nothing like this has ever been seen in Israel.'",
            34: "But the Pharisees said, 'It is through the prince of devils that he drives out devils.'",
            35: "Jesus made a tour through all the towns and villages, teaching in their synagogues, proclaiming the good news of the kingdom and curing all kinds of disease and all kinds of illness.",
            36: "And when he saw the crowds he felt sorry for them because they were harassed and dejected, like sheep without a shepherd.",
            37: "Then he said to his disciples, 'The harvest is rich but the labourers are few, so ask the Lord of the harvest to send out labourers to his harvest.'"
        }

        chapter_10 = {
            1: "He summoned his twelve disciples and gave them authority over unclean spirits with power to drive them out and to cure all kinds of disease and all kinds of illness.",
            2: "These are the names of the twelve apostles: first, Simon who is known as Peter, and his brother Andrew; James the son of Zebedee, and his brother John;",
            3: "Philip and Bartholomew; Thomas, and Matthew the tax collector; James the son of Alphaeus, and Thaddaeus;",
            4: "Simon the Zealot and Judas Iscariot, who was also his betrayer.",
            5: "These twelve Jesus sent out, instructing them as follows: 'Do not make your way to gentile territory, and do not enter any Samaritan town;",
            6: "go instead to the lost sheep of the House of Israel.",
            7: "And as you go, proclaim that the kingdom of Heaven is close at hand.",
            8: "Cure the sick, raise the dead, cleanse those suffering from virulent skin-diseases, drive out devils. You received without charge, give without charge.",
            9: "Provide yourselves with no gold or silver, not even with coppers for your purses,",
            10: "with no haversack for the journey or spare tunic or footwear or a staff, for the labourer deserves his keep.",
            11: "'Whatever town or village you go into, seek out someone worthy and stay with him until you leave.",
            12: "As you enter his house, salute it,",
            13: "and if the house deserves it, may your peace come upon it; if it does not, may your peace come back to you.",
            14: "And if anyone does not welcome you or listen to what you have to say, as you walk out of the house or town shake the dust from your feet.",
            15: "In truth I tell you, on the Day of Judgement it will be more bearable for Sodom and Gomorrah than for that town.",
            16: "Look, I am sending you out like sheep among wolves; so be cunning as snakes and yet innocent as doves.",
            17: "'Be prepared for people to hand you over to sanhedrins and scourge you in their synagogues.",
            18: "You will be brought before governors and kings for my sake, as evidence to them and to the gentiles.",
            19: "But when you are handed over, do not worry about how to speak or what to say; what you are to say will be given to you when the time comes,",
            20: "because it is not you who will be speaking; the Spirit of your Father will be speaking in you.",
            21: "'Brother will betray brother to death, and a father his child; children will come forward against their parents and have them put to death.",
            22: "You will be universally hated on account of my name; but anyone who stands firm to the end will be saved.",
            23: "If they persecute you in one town, take refuge in the next; and if they persecute you in that, take refuge in another. In truth I tell you, you will not have gone the round of the towns of Israel before the Son of man comes.",
            24: "'Disciple is not superior to teacher, nor slave to master.",
            25: 'It is enough for disciple to grow to be like teacher, and slave like master. If they have called the master of the house "Beelzebul", how much more the members of his household?',
            26: "'So do not be afraid of them. Everything now covered up will be uncovered, and everything now hidden will be made clear.",
            27: "What I say to you in the dark, tell in the daylight; what you hear in whispers, proclaim from the housetops.",
            28: "'Do not be afraid of those who kill the body but cannot kill the soul; fear him rather who can destroy both body and soul in hell.",
            29: "Can you not buy two sparrows for a penny? And yet not one falls to the ground without your Father knowing.",
            30: "Why, every hair on your head has been counted.",
            31: "So there is no need to be afraid; you are worth more than many sparrows.",
            32: "'So if anyone declares himself for me in the presence of human beings, I will declare myself for him in the presence of my Father in heaven.",
            33: "But the one who disowns me in the presence of human beings, I will disown in the presence of my Father in heaven.",
            34: "'Do not suppose that I have come to bring peace to the earth: it is not peace I have come to bring, but a sword.",
            35: "For I have come to set son against father, daughter against mother, daughter-in-law against mother-in-law;",
            36: "a person's enemies will be the members of his own household.",
            37: "'No one who prefers father or mother to me is worthy of me. No one who prefers son or daughter to me is worthy of me.",
            38: "Anyone who does not take his cross and follow in my footsteps is not worthy of me.",
            39: "Anyone who finds his life will lose it; anyone who loses his life for my sake will find it.",
            40: "'Anyone who welcomes you welcomes me; and anyone who welcomes me welcomes the one who sent me.",
            41: "'Anyone who welcomes a prophet because he is a prophet will have a prophet's reward; and anyone who welcomes an upright person because he is upright will have the reward of an upright person.",
            42: "'If anyone gives so much as a cup of cold water to one of these little ones because he is a disciple, then in truth I tell you, he will most certainly not go without his reward.'"
        }

        chapter_11 = {
            1: "When Jesus had finished instructing his twelve disciples he moved on from there to teach and preach in their towns.",
            2: "Now John had heard in prison what Christ was doing and he sent his disciples to ask him,",
            3: "'Are you the one who is to come, or are we to expect someone else?'",
            4: "Jesus answered, 'Go back and tell John what you hear and see;",
            5: "the blind see again, and the lame walk, those suffering from virulent skin-diseases are cleansed, and the deaf hear, the dead are raised to life and the good news is proclaimed to the poor;",
            6: "and blessed is anyone who does not find me a cause of falling.'",
            7: "As the men were leaving, Jesus began to talk to the people about John, 'What did you go out into the desert to see? A reed swaying in the breeze? No?",
            8: "Then what did you go out to see? A man wearing fine clothes? Look, those who wear fine clothes are to be found in palaces.",
            9: "Then what did you go out for? To see a prophet? Yes, I tell you, and much more than a prophet:",
            10: "he is the one of whom scripture says: Look, I am going to send my messenger in front of you to prepare your way before you.",
            11: "'In truth I tell you, of all the children born to women, there has never been anyone greater than John the Baptist; yet the least in the kingdom of Heaven is greater than he.",
            12: "Since John the Baptist came, up to this present time, the kingdom of Heaven has been subjected to violence and the violent are taking it by storm.",
            13: "Because it was towards John that all the prophecies of the prophets and of the Law were leading;",
            14: "and he, if you will believe me, is the Elijah who was to return.",
            15: "Anyone who has ears should listen!",
            16: "'What comparison can I find for this generation? It is like children shouting to each other as they sit in the market place:",
            17: "We played the pipes for you, and you wouldn't dance; we sang dirges, and you wouldn't be mourners.",
            18: "'For John came, neither eating nor drinking, and they say, \"He is possessed.\"",
            19: "The Son of man came, eating and drinking, and they say, \"Look, a glutton and a drunkard, a friend of tax collectors and sinners.\" Yet wisdom is justified by her deeds.'",
            20: "Then he began to reproach the towns in which most of his miracles had been worked, because they refused to repent.",
            21: "'Alas for you, Chorazin! Alas for you, Bethsaida! For if the miracles done in you had been done in Tyre and Sidon, they would have repented long ago in sackcloth and ashes.",
            22: "Still, I tell you that it will be more bearable for Tyre and Sidon on Judgement Day than for you.",
            23: "And as for you, Capernaum, did you want to be raised as high as heaven? You shall be flung down to hell. For if the miracles done in you had been done in Sodom, it would have been standing yet.",
            24: "Still, I tell you that it will be more bearable for Sodom on Judgement Day than for you.'",
            25: "At that time Jesus exclaimed, 'I bless you, Father, Lord of heaven and of earth, for hiding these things from the learned and the clever and revealing them to little children.",
            26: "Yes, Father, for that is what it pleased you to do.",
            27: "Everything has been entrusted to me by my Father; and no one knows the Son except the Father, just as no one knows the Father except the Son and those to whom the Son chooses to reveal him.",
            28: "'Come to me, all you who labour and are overburdened, and I will give you rest.",
            29: "Shoulder my yoke and learn from me, for I am gentle and humble in heart, and you will find rest for your souls.",
            30: "Yes, my yoke is easy and my burden light.'"
        }

        chapter_12 = {
            1: "At that time Jesus went through the cornfields one Sabbath day. His disciples were hungry and began to pick ears of corn and eat them.",
            2: "The Pharisees noticed it and said to him, 'Look, your disciples are doing something that is forbidden on the Sabbath.'",
            3: "But he said to them, 'Have you not read what David did when he and his followers were hungry-",
            4: "how he went into the house of God and they ate the loaves of the offering although neither he nor his followers were permitted to eat them, but only the priests?",
            5: "Or again, have you not read in the Law that on the Sabbath day the Temple priests break the Sabbath without committing any fault?",
            6: "Now here, I tell you, is something greater than the Temple.",
            7: "And if you had understood the meaning of the words: Mercy is what pleases me, not sacrifice, you would not have condemned the blameless.",
            8: "For the Son of man is master of the Sabbath.'",
            9: "He moved on from there and went to their synagogue;",
            10: "now a man was there with a withered hand. They asked him, 'Is it permitted to cure somebody on the Sabbath day?' hoping for something to charge him with.",
            11: "But he said to them, 'If any one of you here had only one sheep and it fell down a hole on the Sabbath day, would he not get hold of it and lift it out?",
            12: "Now a man is far more important than a sheep, so it follows that it is permitted on the Sabbath day to do good.'",
            13: "Then he said to the man, 'Stretch out your hand.' He stretched it out and his hand was restored, as sound as the other one.",
            14: "At this the Pharisees went out and began to plot against him, discussing how to destroy him.",
            15: "Jesus knew this and withdrew from the district. Many followed him and he cured them all",
            16: "but warned them not to make him known.",
            17: "This was to fulfil what was spoken by the prophet Isaiah:",
            18: "Look! My servant whom I have chosen, my beloved, in whom my soul delights, I will send my Spirit upon him, and he will present judgement to the nations;",
            19: "he will not brawl or cry out, his voice is not heard in the streets,",
            20: "he will not break the crushed reed, or snuff the faltering wick,",
            21: "until he has made judgement victorious; in him the nations will put their hope.",
            22: "Then they brought to him a blind and dumb demoniac; and he cured him, so that the dumb man could speak and see.",
            23: "All the people were astounded and said, 'Can this be the son of David?'",
            24: "But when the Pharisees heard this they said, 'The man drives out devils only through Beelzebul, the chief of the devils.'",
            25: "Knowing what was in their minds he said to them, 'Every kingdom divided against itself is heading for ruin; and no town, no household divided against itself can last.",
            26: "Now if Satan drives out Satan, he is divided against himself; so how can his kingdom last?",
            27: "And if it is through Beelzebul that I drive devils out, through whom do your own experts drive them out? They shall be your judges, then.",
            28: "But if it is through the Spirit of God that I drive out devils, then be sure that the kingdom of God has caught you unawares.",
            29: "'Or again, how can anyone make his way into a strong man's house and plunder his property unless he has first tied up the strong man? Only then can he plunder his house.",
            30: "'Anyone who is not with me is against me, and anyone who does not gather in with me throws away.",
            31: "And so I tell you, every human sin and blasphemy will be forgiven, but blasphemy against the Spirit will not be forgiven.",
            32: "And anyone who says a word against the Son of man will be forgiven; but no one who speaks against the Holy Spirit will be forgiven either in this world or in the next.",
            33: "'Make a tree sound and its fruit will be sound; make a tree rotten and its fruit will be rotten. For the tree can be told by its fruit.",
            34: "You brood of vipers, how can your speech be good when you are evil? For words flow out of what fills the heart.",
            35: "Good people draw good things from their store of goodness; bad people draw bad things from their store of badness.",
            36: "So I tell you this, that for every unfounded word people utter they will answer on Judgement Day,",
            37: "since it is by your words you will be justified, and by your words condemned.'",
            38: "Then some of the scribes and Pharisees spoke up. 'Master,' they said, 'we should like to see a sign from you.'",
            39: "He replied, 'It is an evil and unfaithful generation that asks for a sign! The only sign it will be given is the sign of the prophet Jonah.",
            40: "For as Jonah remained in the belly of the sea-monster for three days and three nights, so will the Son of man be in the heart of the earth for three days and three nights.",
            41: "On Judgement Day the men of Nineveh will appear against this generation and they will be its condemnation, because when Jonah preached they repented; and look, there is something greater than Jonah here.",
            42: "On Judgement Day the Queen of the South will appear against this generation and be its condemnation, because she came from the ends of the earth to hear the wisdom of Solomon; and look, there is something greater than Solomon here.",
            43: "'When an unclean spirit goes out of someone it wanders through waterless country looking for a place to rest, and cannot find one.",
            44: 'Then it says, "I will return to the home I came from." But on arrival, finding it unoccupied, swept and tidied,',
            45: "it then goes off and collects seven other spirits more wicked than itself, and they go in and set up house there, and so that person ends up worse off than before. That is what will happen to this wicked generation.'",
            46: "He was still speaking to the crowds when suddenly his mother and his brothers were standing outside and were anxious to have a word with him.",
            47: "still speaking to the crowds when suddenly his mother and his brothers were standing outside and were anxious to have a word with him.",
            48: "But to the man who told him this Jesus replied, 'Who is my mother? Who are my brothers?'",
            49: "And stretching out his hand towards his disciples he said, 'Here are my mother and my brothers.",
            50: "Anyone who does the will of my Father in heaven is my brother and sister and mother.'"
        }

        chapter_13 = {
            1: "That same day, Jesus left the house and sat by the lakeside,",
            2: "but such large crowds gathered round him that he got into a boat and sat there. The people all stood on the shore,",
            3: "and he told them many things in parables. He said, 'Listen, a sower went out to sow.",
            4: "As he sowed, some seeds fell on the edge of the path, and the birds came and ate them up.",
            5: "Others fell on patches of rock where they found little soil and sprang up at once, because there was no depth of earth;",
            6: "but as soon as the sun came up they were scorched and, not having any roots, they withered away.",
            7: "Others fell among thorns, and the thorns grew up and choked them.",
            8: "Others fell on rich soil and produced their crop, some a hundredfold, some sixty, some thirty.",
            9: "Anyone who has ears should listen!'",
            10: "Then the disciples went up to him and asked, 'Why do you talk to them in parables?'",
            11: "In answer, he said, 'Because to you is granted to understand the mysteries of the kingdom of Heaven, but to them it is not granted.",
            12: "Anyone who has will be given more and will have more than enough; but anyone who has not will be deprived even of what he has.",
            13: "The reason I talk to them in parables is that they look without seeing and listen without hearing or understanding.",
            14: "So in their case what was spoken by the prophet Isaiah is being fulfilled: Listen and listen, but never understand! Look and look, but never perceive!",
            15: "This people's heart has grown coarse, their ears dulled, they have shut their eyes tight to avoid using their eyes to see, their ears to hear, their heart to understand, changing their ways and being healed by me.",
            16: "'But blessed are your eyes because they see, your ears because they hear!",
            17: "In truth I tell you, many prophets and upright people longed to see what you see, and never saw it; to hear what you hear, and never heard it.",
            18: "'So pay attention to the parable of the sower.",
            19: "When anyone hears the word of the kingdom without understanding, the Evil One comes and carries off what was sown in his heart: this is the seed sown on the edge of the path.",
            20: "The seed sown on patches of rock is someone who hears the word and welcomes it at once with joy.",
            21: "But such a person has no root deep down and does not last; should some trial come, or some persecution on account of the word, at once he falls away.",
            22: "The seed sown in thorns is someone who hears the word, but the worry of the world and the lure of riches choke the word and so it produces nothing.",
            23: "And the seed sown in rich soil is someone who hears the word and understands it; this is the one who yields a harvest and produces now a hundredfold, now sixty, now thirty.'",
            24: "He put another parable before them, 'The kingdom of Heaven may be compared to a man who sowed good seed in his field.",
            25: "While everybody was asleep his enemy came, sowed darnel all among the wheat, and made off.",
            26: "When the new wheat sprouted and ripened, then the darnel appeared as well.",
            27: '''The owner's labourers went to him and said, "Sir, was it not good seed that you sowed in your field? If so, where does the darnel come from?"''',
            28: 'He said to them, "Some enemy has done this." And the labourers said, "Do you want us to go and weed it out?"',
            29: 'But he said, "No, because when you weed out the darnel you might pull up the wheat with it.',
            30: """Let them both grow till the harvest; and at harvest time I shall say to the reapers: First collect the darnel and tie it in bundles to be burnt, then gather the wheat into my barn." '""",
            31: "He put another parable before them, 'The kingdom of Heaven is like a mustard seed which a man took and sowed in his field.",
            32: "It is the smallest of all the seeds, but when it has grown it is the biggest of shrubs and becomes a tree, so that the birds of the air can come and shelter in its branches.'",
            33: "He told them another parable, 'The kingdom of Heaven is like the yeast a woman took and mixed in with three measures of flour till it was leavened all through.'",
            34: "In all this Jesus spoke to the crowds in parables; indeed, he would never speak to them except in parables.",
            35: "This was to fulfil what was spoken by the prophet: I will speak to you in parables, unfold what has been hidden since the foundation of the world.",
            36: "Then, leaving the crowds, he went to the house; and his disciples came to him and said, 'Explain to us the parable about the darnel in the field.'",
            37: "He said in reply, 'The sower of the good seed is the Son of man.",
            38: "The field is the world; the good seed is the subjects of the kingdom; the darnel, the subjects of the Evil One;",
            39: "the enemy who sowed it, the devil; the harvest is the end of the world; the reapers are the angels.",
            40: "Well then, just as the darnel is gathered up and burnt in the fire, so it will be at the end of time.",
            41: "The Son of man will send his angels and they will gather out of his kingdom all causes of falling and all who do evil,",
            42: "and throw them into the blazing furnace, where there will be weeping and grinding of teeth.",
            43: "Then the upright will shine like the sun in the kingdom of their Father. Anyone who has ears should listen!",
            44: "'The kingdom of Heaven is like treasure hidden in a field which someone has found; he hides it again, goes off in his joy, sells everything he owns and buys the field.",
            45: "'Again, the kingdom of Heaven is like a merchant looking for fine pearls;",
            46: "when he finds one of great value he goes and sells everything he owns and buys it.",
            47: "'Again, the kingdom of Heaven is like a dragnet that is cast in the sea and brings in a haul of all kinds of fish.",
            48: "When it is full, the fishermen bring it ashore; then, sitting down, they collect the good ones in baskets and throw away those that are no use.",
            49: "This is how it will be at the end of time: the angels will appear and separate the wicked from the upright,",
            50: "to throw them into the blazing furnace, where there will be weeping and grinding of teeth.",
            51: "'Have you understood all these?' They said, 'Yes.'",
            52: "And he said to them, 'Well then, every scribe who becomes a disciple of the kingdom of Heaven is like a householder who brings out from his storeroom new things as well as old.'",
            53: "When Jesus had finished these parables he left the district;",
            54: "and, coming to his home town, he taught the people in their synagogue in such a way that they were astonished and said, 'Where did the man get this wisdom and these miraculous powers?",
            55: "This is the carpenter's son, surely? Is not his mother the woman called Mary, and his brothers James and Joseph and Simon and Jude?",
            56: "His sisters, too, are they not all here with us? So where did the man get it all?'",
            57: "And they would not accept him. But Jesus said to them, 'A prophet is despised only in his own country and in his own house,'",
            58: "and he did not work many miracles there because of their lack of faith."
        }

        chapter_14 = {
            1: "At that time Herod the tetrarch heard about the reputation of Jesus",
            2: "and said to his court, 'This is John the Baptist himself; he has risen from the dead, and that is why miraculous powers are at work in him.'",
            3: "Now it was Herod who had arrested John, chained him up and put him in prison because of Herodias, his brother Philip's wife.",
            4: "For John had told him, 'It is against the Law for you to have her.'",
            5: "He had wanted to kill him but was afraid of the people, who regarded John as a prophet.",
            6: "Then, during the celebrations for Herod's birthday, the daughter of Herodias danced before the company and so delighted Herod",
            7: "that he promised on oath to give her anything she asked.",
            8: "Prompted by her mother she said, 'Give me John the Baptist's head, here, on a dish.'",
            9: "The king was distressed but, thinking of the oaths he had sworn and of his guests, he ordered it to be given her,",
            10: "and sent and had John beheaded in the prison.",
            11: "The head was brought in on a dish and given to the girl, who took it to her mother.",
            12: "John's disciples came and took the body and buried it; then they went off to tell Jesus.",
            13: "When Jesus received this news he withdrew by boat to a lonely place where they could be by themselves. But the crowds heard of this and, leaving the towns, went after him on foot.",
            14: "So as he stepped ashore he saw a large crowd; and he took pity on them and healed their sick.",
            15: "When evening came, the disciples went to him and said, 'This is a lonely place, and time has slipped by; so send the people away, and they can go to the villages to buy themselves some food.'",
            16: "Jesus replied, 'There is no need for them to go: give them something to eat yourselves.'",
            17: "But they answered, 'All we have with us is five loaves and two fish.'",
            18: "So he said, 'Bring them here to me.'",
            19: "He gave orders that the people were to sit down on the grass; then he took the five loaves and the two fish, raised his eyes to heaven and said the blessing. And breaking the loaves he handed them to his disciples, who gave them to the crowds.",
            20: "They all ate as much as they wanted, and they collected the scraps left over, twelve baskets full.",
            21: "Now about five thousand men had eaten, to say nothing of women and children.",
            22: "And at once he made the disciples get into the boat and go on ahead to the other side while he sent the crowds away.",
            23: "After sending the crowds away he went up into the hills by himself to pray. When evening came, he was there alone,",
            24: "while the boat, by now some furlongs from land, was hard pressed by rough waves, for there was a head-wind.",
            25: "In the fourth watch of the night he came towards them, walking on the sea,",
            26: "and when the disciples saw him walking on the sea they were terrified. 'It is a ghost,' they said, and cried out in fear.",
            27: "But at once Jesus called out to them, saying, 'Courage! It's me! Don't be afraid.'",
            28: "It was Peter who answered. 'Lord,' he said, 'if it is you, tell me to come to you across the water.'",
            29: "Jesus said, 'Come.' Then Peter got out of the boat and started walking towards Jesus across the water,",
            30: "but then noticing the wind, he took fright and began to sink. 'Lord,' he cried, 'save me!'",
            31: "Jesus put out his hand at once and held him. 'You have so little faith,' he said, 'why did you doubt?'",
            32: "And as they got into the boat the wind dropped.",
            33: "The men in the boat bowed down before him and said, 'Truly, you are the Son of God.'",
            34: "Having made the crossing, they came to land at Gennesaret.",
            35: "When the local people recognised him they spread the news through the whole neighbourhood and took all that were sick to him,",
            36: "begging him just to let them touch the fringe of his cloak. And all those who touched it were saved."
        }

        chapter_15 = {
            1: "Then Pharisees and scribes from Jerusalem came to Jesus and said,",
            2: "'Why do your disciples break away from the tradition of the elders? They eat without washing their hands.'",
            3: "He answered, 'And why do you break away from the commandment of God for the sake of your tradition?",
            4: 'For God said, "Honour your father and your mother" and "Anyone who curses his father or mother will be put to death."',
            5: 'But you say, "If anyone says to his father or mother: Anything I might have used to help you is dedicated to God,',
            6: '''he is rid of his duty to father or mother." In this way you have made God's word ineffective by means of your tradition.''',
            7: "Hypocrites! How rightly Isaiah prophesied about you when he said:",
            8: "This people honours me only with lip-service, while their hearts are far from me.",
            9: "Their reverence of me is worthless; the lessons they teach are nothing but human commandments.'",
            10: "He called the people to him and said, 'Listen, and understand.",
            11: "What goes into the mouth does not make anyone unclean; it is what comes out of the mouth that makes someone unclean.'",
            12: "Then the disciples came to him and said, 'Do you know that the Pharisees were shocked when they heard what you said?'",
            13: "He replied, 'Any plant my heavenly Father has not planted will be pulled up by the roots.",
            14: "Leave them alone. They are blind leaders of the blind; and if one blind person leads another, both will fall into a pit.'",
            15: "At this, Peter said to him, 'Explain the parable for us.'",
            16: "Jesus replied, 'Even you -- don't you yet understand?",
            17: "Can't you see that whatever goes into the mouth passes through the stomach and is discharged into the sewer?",
            18: "But whatever comes out of the mouth comes from the heart, and it is this that makes someone unclean.",
            19: "For from the heart come evil intentions: murder, adultery, fornication, theft, perjury, slander.",
            20: "These are the things that make a person unclean. But eating with unwashed hands does not make anyone unclean.'",
            21: "Jesus left that place and withdrew to the region of Tyre and Sidon.",
            22: "And suddenly out came a Canaanite woman from that district and started shouting, 'Lord, Son of David, take pity on me. My daughter is tormented by a devil.'",
            23: "But he said not a word in answer to her. And his disciples went and pleaded with him, saying, 'Give her what she wants, because she keeps shouting after us.'",
            24: "He said in reply, 'I was sent only to the lost sheep of the House of Israel.'",
            25: "But the woman had come up and was bowing low before him. 'Lord,' she said, 'help me.'",
            26: "He replied, 'It is not fair to take the children's food and throw it to little dogs.'",
            27: "She retorted, 'Ah yes, Lord; but even little dogs eat the scraps that fall from their masters' table.'",
            28: "Then Jesus answered her, 'Woman, you have great faith. Let your desire be granted.' And from that moment her daughter was well again.",
            29: "Jesus went on from there and reached the shores of the Lake of Galilee, and he went up onto the mountain. He took his seat,",
            30: "and large crowds came to him bringing the lame, the crippled, the blind, the dumb and many others; these they put down at his feet, and he cured them.",
            31: "The crowds were astonished to see the dumb speaking, the cripples whole again, the lame walking and the blind with their sight, and they praised the God of Israel.",
            32: "But Jesus called his disciples to him and said, 'I feel sorry for all these people; they have been with me for three days now and have nothing to eat. I do not want to send them off hungry, or they might collapse on the way.'",
            33: "The disciples said to him, 'Where in a deserted place could we get sufficient bread for such a large crowd to have enough to eat?'",
            34: "Jesus said to them, 'How many loaves have you?' They said, 'Seven, and a few small fish.'",
            35: "Then he instructed the crowd to sit down on the ground,",
            36: "and he took the seven loaves and the fish, and after giving thanks he broke them and began handing them to the disciples, who gave them to the crowds.",
            37: "They all ate as much as they wanted, and they collected what was left of the scraps, seven baskets full.",
            38: "Now four thousand men had eaten, to say nothing of women and children.",
            39: "And when he had sent the crowds away he got into the boat and went to the territory of Magadan."
        }

        chapter_16 = {
            1: "The Pharisees and Sadducees came, and to put him to the test they asked if he would show them a sign from heaven.",
            2: '''He replied, 'In the evening you say, "It will be fine; there's a red sky,"''',
            3: 'and in the morning, "Stormy weather today; the sky is red and overcast." You know how to read the face of the sky, but you cannot read the signs of the times.',
            4: "It is an evil and unfaithful generation asking for a sign, and the only sign it will be given is the sign of Jonah.' And he left them and went off.",
            5: "The disciples, having crossed to the other side, had forgotten to take any food.",
            6: "Jesus said to them, 'Keep your eyes open, and be on your guard against the yeast of the Pharisees and Sadducees.'",
            7: "And they said among themselves, 'It is because we have not brought any bread.'",
            8: "Jesus knew it, and he said, 'You have so little faith, why are you talking among yourselves about having no bread?",
            9: "Do you still not understand? Do you not remember the five loaves for the five thousand and the number of baskets you collected?",
            10: "Or the seven loaves for the four thousand and the number of baskets you collected?",
            11: "How could you fail to understand that I was not talking about bread? What I said was: Beware of the yeast of the Pharisees and Sadducees.'",
            12: "Then they understood that he was telling them to be on their guard, not against yeast for making bread, but against the teaching of the Pharisees and Sadducees.",
            13: "When Jesus came to the region of Caesarea Philippi he put this question to his disciples, 'Who do people say the Son of man is?'",
            14: "And they said, 'Some say John the Baptist, some Elijah, and others Jeremiah or one of the prophets.'",
            15: "'But you,' he said, 'who do you say I am?'",
            16: "Then Simon Peter spoke up and said, 'You are the Christ, the Son of the living God.'",
            17: "Jesus replied, 'Simon son of Jonah, you are a blessed man! Because it was no human agency that revealed this to you but my Father in heaven.",
            18: "So I now say to you: You are Peter and on this rock I will build my community. And the gates of the underworld can never overpower it.",
            19: "I will give you the keys of the kingdom of Heaven: whatever you bind on earth will be bound in heaven; whatever you loose on earth will be loosed in heaven.'",
            20: "Then he gave the disciples strict orders not to say to anyone that he was the Christ.",
            21: "From then onwards Jesus began to make it clear to his disciples that he was destined to go to Jerusalem and suffer grievously at the hands of the elders and chief priests and scribes and to be put to death and to be raised up on the third day.",
            22: "Then, taking him aside, Peter started to rebuke him. 'Heaven preserve you, Lord,' he said, 'this must not happen to you.'",
            23: "But he turned and said to Peter, 'Get behind me, Satan! You are an obstacle in my path, because you are thinking not as God thinks but as human beings do.'",
            24: "Then Jesus said to his disciples, 'If anyone wants to be a follower of mine, let him renounce himself and take up his cross and follow me.",
            25: "Anyone who wants to save his life will lose it; but anyone who loses his life for my sake will find it.",
            26: "What, then, will anyone gain by winning the whole world and forfeiting his life? Or what can anyone offer in exchange for his life?",
            27: "'For the Son of man is going to come in the glory of his Father with his angels, and then he will reward each one according to his behaviour.",
            28: "In truth I tell you, there are some standing here who will not taste death before they see the Son of man coming with his kingdom.'"
        }

        chapter_17 = {
            1: "Six days later, Jesus took with him Peter and James and his brother John and led them up a high mountain by themselves.",
            2: "There in their presence he was transfigured: his face shone like the sun and his clothes became as dazzling as light.",
            3: "And suddenly Moses and Elijah appeared to them; they were talking with him.",
            4: "Then Peter spoke to Jesus. 'Lord,' he said, 'it is wonderful for us to be here; if you want me to, I will make three shelters here, one for you, one for Moses and one for Elijah.'",
            5: "He was still speaking when suddenly a bright cloud covered them with shadow, and suddenly from the cloud there came a voice which said, 'This is my Son, the Beloved; he enjoys my favour. Listen to him.'",
            6: "When they heard this, the disciples fell on their faces, overcome with fear.",
            7: "But Jesus came up and touched them, saying, 'Stand up, do not be afraid.'",
            8: "And when they raised their eyes they saw no one but Jesus.",
            9: "As they came down from the mountain Jesus gave them this order, 'Tell no one about this vision until the Son of man has risen from the dead.'",
            10: "And the disciples put this question to him, 'Why then do the scribes say that Elijah must come first?'",
            11: "He replied, 'Elijah is indeed coming, and he will set everything right again;",
            12: "however, I tell you that Elijah has come already and they did not recognise him but treated him as they pleased; and the Son of man will suffer similarly at their hands.'",
            13: "Then the disciples understood that he was speaking of John the Baptist.",
            14: "As they were rejoining the crowd a man came up to him and went down on his knees before him.",
            15: "'Lord,' he said, 'take pity on my son: he is demented and in a wretched state; he is always falling into fire and into water.",
            16: "I took him to your disciples and they were unable to cure him.'",
            17: "In reply, Jesus said, 'Faithless and perverse generation! How much longer must I be with you? How much longer must I put up with you? Bring him here to me.'",
            18: "And when Jesus rebuked it the devil came out of the boy, who was cured from that moment.",
            19: "Then the disciples came privately to Jesus. 'Why were we unable to drive it out?' they asked.",
            20: """He answered, 'Because you have so little faith. In truth I tell you, if your faith is the size of a mustard seed you will say to this mountain, "Move from here to there," and it will move; nothing will be impossible for you.'""",
            21: "",
            22: "When they were together in Galilee, Jesus said to them, 'The Son of man is going to be delivered into the power of men;",
            23: "they will put him to death, and on the third day he will be raised up again.' And a great sadness came over them.",
            24: "When they reached Capernaum, the collectors of the half-shekel came to Peter and said, 'Does your master not pay the half-shekel?'",
            25: "'Yes,' he replied, and went into the house. But before he could speak, Jesus said, 'Simon, what is your opinion? From whom do earthly kings take toll or tribute? From their sons or from foreigners?'",
            26: "And when he replied, 'From foreigners,' Jesus said, 'Well then, the sons are exempt.",
            27: "However, so that we shall not be the downfall of others, go to the lake and cast a hook; take the first fish that rises, open its mouth and there you will find a shekel; take it and give it to them for me and for yourself.'"
        }

        chapter_18 = {
            1: "At this time the disciples came to Jesus and said, 'Who is the greatest in the kingdom of Heaven?'",
            2: "So he called a little child to him whom he set among them.",
            3: "Then he said, 'In truth I tell you, unless you change and become like little children you will never enter the kingdom of Heaven.",
            4: "And so, the one who makes himself as little as this little child is the greatest in the kingdom of Heaven.",
            5: "'Anyone who welcomes one little child like this in my name welcomes me.",
            6: "But anyone who is the downfall of one of these little ones who have faith in me would be better drowned in the depths of the sea with a great millstone round his neck.",
            7: "Alas for the world that there should be such causes of falling! Causes of falling indeed there must be, but alas for anyone who provides them!",
            8: "'If your hand or your foot should be your downfall, cut it off and throw it away: it is better for you to enter into life crippled or lame, than to have two hands or two feet and be thrown into eternal fire. ",
            9: "And if your eye should be your downfall, tear it out and throw it away: it is better for you to enter into life with one eye, than to have two eyes and be thrown into the hell of fire. ",
            10: "'See that you never despise any of these little ones, for I tell you that their angels in heaven are continually in the presence of my Father in heaven. ",
            11: "",
            12: "'Tell me. Suppose a man has a hundred sheep and one of them strays; will he not leave the ninety-nine on the hillside and go in search of the stray? ",
            13: "In truth I tell you, if he finds it, it gives him more joy than do the ninety-nine that did not stray at all. ",
            14: "Similarly, it is never the will of your Father in heaven that one of these little ones should be lost. ",
            15: "'If your brother does something wrong, go and have it out with him alone, between your two selves. If he listens to you, you have won back your brother. ",
            16: "If he does not listen, take one or two others along with you: whatever the misdemeanour, the evidence of two or three witnesses is required to sustain the charge. ",
            17: "But if he refuses to listen to these, report it to the community; and if he refuses to listen to the community, treat him like a gentile or a tax collector. ",
            18: "'In truth I tell you, whatever you bind on earth will be bound in heaven; whatever you loose on earth will be loosed in heaven. ",
            19: "'In truth I tell you once again, if two of you on earth agree to ask anything at all, it will be granted to you by my Father in heaven. ",
            20: "For where two or three meet in my name, I am there among them.' ",
            21: "Then Peter went up to him and said, 'Lord, how often must I forgive my brother if he wrongs me? As often as seven times?' ",
            22: "Jesus answered, 'Not seven, I tell you, but seventy-seven times. ",
            23: "'And so the kingdom of Heaven may be compared to a king who decided to settle his accounts with his servants. ",
            24: "When the reckoning began, they brought him a man who owed ten thousand talents; ",
            25: "he had no means of paying, so his master gave orders that he should be sold, together with his wife and children and all his possessions, to meet the debt. ",
            26: '''At this, the servant threw himself down at his master's feet, with the words, "Be patient with me and I will pay the whole sum." ''',
            27: "And the servant's master felt so sorry for him that he let him go and cancelled the debt. ",
            28: 'Now as this servant went out, he happened to meet a fellow-servant who owed him one hundred denarii; and he seized him by the throat and began to throttle him, saying, "Pay what you owe me." ',
            29: 'His fellow-servant fell at his feet and appealed to him, saying, "Be patient with me and I will pay you." ',
            30: "But the other would not agree; on the contrary, he had him thrown into prison till he should pay the debt. ",
            31: "His fellow-servants were deeply distressed when they saw what had happened, and they went to their master and reported the whole affair to him. ",
            32: 'Then the master sent for the man and said to him, "You wicked servant, I cancelled all that debt of yours when you appealed to me. ',
            33: 'Were you not bound, then, to have pity on your fellow-servant just as I had pity on you?" ',
            34: "And in his anger the master handed him over to the torturers till he should pay all his debt. ",
            35: "And that is how my heavenly Father will deal with you unless you each forgive your brother from your heart.'"
        }

        chapter_19 = {
            1: "Jesus had now finished what he wanted to say, and he left Galilee and came into the territory of Judaea on the far side of the Jordan. ",
            2: "Large crowds followed him and he healed them there. ",
            3: "Some Pharisees approached him, and to put him to the test they said, 'Is it against the Law for a man to divorce his wife on any pretext whatever?' ",
            4: "He answered, 'Have you not read that the Creator from the beginning made them male and female ",
            5: "and that he said: This is why a man leaves his father and mother and becomes attached to his wife, and the two become one flesh? ",
            6: "They are no longer two, therefore, but one flesh. So then, what God has united, human beings must not divide.' ",
            7: "They said to him, 'Then why did Moses command that a writ of dismissal should be given in cases of divorce?' ",
            8: "He said to them, 'It was because you were so hard-hearted, that Moses allowed you to divorce your wives, but it was not like this from the beginning. ",
            9: "Now I say this to you: anyone who divorces his wife -- I am not speaking of an illicit marriage -- and marries another, is guilty of adultery.' ",
            10: "The disciples said to him, 'If that is how things are between husband and wife, it is advisable not to marry.' ",
            11: "But he replied, 'It is not everyone who can accept what I have said, but only those to whom it is granted. ",
            12: "There are eunuchs born so from their mother's womb, there are eunuchs made so by human agency and there are eunuchs who have made themselves so for the sake of the kingdom of Heaven. Let anyone accept this who can.' ",
            13: "Then people brought little children to him, for him to lay his hands on them and pray. The disciples scolded them, ",
            14: "but Jesus said, 'Let the little children alone, and do not stop them from coming to me; for it is to such as these that the kingdom of Heaven belongs.' ",
            15: "Then he laid his hands on them and went on his way. ",
            16: "And now a man came to him and asked, 'Master, what good deed must I do to possess eternal life?' ",
            17: "Jesus said to him, 'Why do you ask me about what is good? There is one alone who is good. But if you wish to enter into life, keep the commandments.' ",
            18: "He said, 'Which ones?' Jesus replied, 'These: You shall not kill. You shall not commit adultery. You shall not steal. You shall not give false witness. ",
            19: "Honour your father and your mother. You shall love your neighbour as yourself.' ",
            20: "The young man said to him, 'I have kept all these. What more do I need to do?' ",
            21: "Jesus said, 'If you wish to be perfect, go and sell your possessions and give the money to the poor, and you will have treasure in heaven; then come, follow me.' ",
            22: "But when the young man heard these words he went away sad, for he was a man of great wealth. ",
            23: "Then Jesus said to his disciples, 'In truth I tell you, it is hard for someone rich to enter the kingdom of Heaven. ",
            24: "Yes, I tell you again, it is easier for a camel to pass through the eye of a needle than for someone rich to enter the kingdom of Heaven.' ",
            25: "When the disciples heard this they were astonished. 'Who can be saved, then?' they said. ",
            26: "Jesus gazed at them. 'By human resources', he told them, 'this is impossible; for God everything is possible.' ",
            27: "Then Peter answered and said, 'Look, we have left everything and followed you. What are we to have, then?' ",
            28: "Jesus said to them, 'In truth I tell you, when everything is made new again and the Son of man is seated on his throne of glory, you yourselves will sit on twelve thrones to judge the twelve tribes of Israel. ",
            29: "And everyone who has left houses, brothers, sisters, father, mother, children or land for the sake of my name will receive a hundred times as much, and also inherit eternal life. ",
            30: "'Many who are first will be last, and the last, first.'"
        }

        chapter_20 = {
            1: "'Now the kingdom of Heaven is like a landowner going out at daybreak to hire workers for his vineyard. ",
            2: "He made an agreement with the workers for one denarius a day and sent them to his vineyard. ",
            3: "Going out at about the third hour he saw others standing idle in the market place ",
            4: 'and said to them, "You go to my vineyard too and I will give you a fair wage." ',
            5: "So they went. At about the sixth hour and again at about the ninth hour, he went out and did the same. ",
            6: 'Then at about the eleventh hour he went out and found more men standing around, and he said to them, "Why have you been standing here idle all day?" ',
            7: '"Because no one has hired us," they answered. He said to them, "You go into my vineyard too." ',
            8: 'In the evening, the owner of the vineyard said to his bailiff, "Call the workers and pay them their wages, starting with the last arrivals and ending with the first." ',
            9: "So those who were hired at about the eleventh hour came forward and received one denarius each. ",
            10: "When the first came, they expected to get more, but they too received one denarius each. ",
            11: "They took it, but grumbled at the landowner saying, ",
            12: '''"The men who came last have done only one hour, and you have treated them the same as us, though we have done a heavy day's work in all the heat." ''',
            13: 'He answered one of them and said, "My friend, I am not being unjust to you; did we not agree on one denarius? ',
            14: "Take your earnings and go. I choose to pay the lastcomer as much as I pay you. ",
            15: 'Have I no right to do what I like with my own? Why should you be envious because I am generous?" ',
            16: "Thus the last will be first, and the first, last.' ",
            17: "Jesus was going up to Jerusalem, and on the road he took the Twelve aside by themselves and said to them, ",
            18: "'Look, we are going up to Jerusalem, and the Son of man is about to be handed over to the chief priests and scribes. They will condemn him to death ",
            19: "and will hand him over to the gentiles to be mocked and scourged and crucified; and on the third day he will be raised up again.' ",
            20: "Then the mother of Zebedee's sons came with her sons to make a request of him, and bowed low; ",
            21: "and he said to her, 'What is it you want?' She said to him, 'Promise that these two sons of mine may sit one at your right hand and the other at your left in your kingdom.' ",
            22: "Jesus answered, 'You do not know what you are asking. Can you drink the cup that I am going to drink?' They replied, 'We can.' ",
            23: "He said to them, 'Very well; you shall drink my cup, but as for seats at my right hand and my left, these are not mine to grant; they belong to those to whom they have been allotted by my Father.' ",
            24: "When the other ten heard this they were indignant with the two brothers. ",
            25: "But Jesus called them to him and said, 'You know that among the gentiles the rulers lord it over them, and great men make their authority felt. ",
            26: "Among you this is not to happen. No; anyone who wants to become great among you must be your servant, ",
            27: "and anyone who wants to be first among you must be your slave, ",
            28: "just as the Son of man came not to be served but to serve, and to give his life as a ransom for many.' ",
            29: "As they left Jericho a large crowd followed him. ",
            30: "And now there were two blind men sitting at the side of the road. When they heard that it was Jesus who was passing by, they shouted, 'Lord! Have pity on us, son of David.' ",
            31: "And the crowd scolded them and told them to keep quiet, but they only shouted the louder, 'Lord! Have pity on us, son of David.' ",
            32: "Jesus stopped, called them over and said, 'What do you want me to do for you?' ",
            33: "They said to him, 'Lord, let us have our sight back.' ",
            34: "Jesus felt pity for them and touched their eyes, and at once their sight returned and they followed him.",
        }

        chapter_21 = {
            1: "When they were near Jerusalem and had come to Bethphage on the Mount of Olives, then Jesus sent two disciples, ",
            2: "saying to them, 'Go to the village facing you, and you will at once find a tethered donkey and a colt with her. Untie them and bring them to me. ",
            3: """If anyone says anything to you, you are to say, "The Master needs them and will send them back at once." ' """,
            4: "This was to fulfil what was spoken by the prophet: ",
            5: "Say to the daughter of Zion: Look, your king is approaching, humble and riding on a donkey and on a colt, the foal of a beast of burden. ",
            6: "So the disciples went and did as Jesus had told them. ",
            7: "They brought the donkey and the colt, then they laid their cloaks on their backs and he took his seat on them. ",
            8: "Great crowds of people spread their cloaks on the road, while others were cutting branches from the trees and spreading them in his path. ",
            9: "The crowds who went in front of him and those who followed were all shouting: Hosanna to the son of David! Blessed is he who is coming in the name of the Lord! Hosanna in the highest heavens! ",
            10: "And when he entered Jerusalem, the whole city was in turmoil as people asked, 'Who is this?' ",
            11: "and the crowds answered, 'This is the prophet Jesus from Nazareth in Galilee.' ",
            12: "Jesus then went into the Temple and drove out all those who were selling and buying there; he upset the tables of the money-changers and the seats of the dove-sellers. ",
            13: "He said to them, 'According to scripture, my house will be called a house of prayer; but you are turning it into a bandits' den.' ",
            14: "There were also blind and lame people who came to him in the Temple, and he cured them. ",
            15: "At the sight of the wonderful things he did and of the children shouting, 'Hosanna to the son of David' in the Temple, the chief priests and the scribes were indignant and said to him, ",
            16: "Do you hear what they are saying?' Jesus answered, 'Yes. Have you never read this: By the mouths of children, babes in arms, you have made sure of praise?' ",
            17: "With that he left them and went out of the city to Bethany, where he spent the night. ",
            18: "As he was returning to the city in the early morning, he felt hungry. ",
            19: "Seeing a fig tree by the road, he went up to it and found nothing on it but leaves. And he said to it, 'May you never bear fruit again,' and instantly the fig tree withered. ",
            20: "The disciples were amazed when they saw it and said, 'How is it that the fig tree withered instantly?' ",
            21: """Jesus answered, 'In truth I tell you, if you have faith and do not doubt at all, not only will you do what I have done to the fig tree, but even if you say to this mountain, "Be pulled up and thrown into the sea," it will be done. """,
            22: "And if you have faith, everything you ask for in prayer, you will receive.' ",
            23: "He had gone into the Temple and was teaching, when the chief priests and the elders of the people came to him and said, 'What authority have you for acting like this? And who gave you this authority?' ",
            24: "In reply Jesus said to them, 'And I will ask you a question, just one; if you tell me the answer to it, then I will tell you my authority for acting like this. ",
            25: """John's baptism: what was its origin, heavenly or human?' And they argued this way among themselves, 'If we say heavenly, he will retort to us, "Then why did you refuse to believe him?"; """,
            26: "but if we say human, we have the people to fear, for they all hold that John was a prophet.' ",
            27: "So their reply to Jesus was, 'We do not know.' And he retorted to them, 'Nor will I tell you my authority for acting like this.' ",
            28: """'What is your opinion? A man had two sons. He went and said to the first, "My boy, go and work in the vineyard today." """,
            29: 'He answered, "I will not go," but afterwards thought better of it and went. ',
            30: 'The man then went and said the same thing to the second who answered, "Certainly, sir," but did not go. ',
            31: "Which of the two did the father's will?' They said, 'The first.' Jesus said to them, 'In truth I tell you, tax collectors and prostitutes are making their way into the kingdom of God before you. ",
            32: "For John came to you, showing the way of uprightness, but you did not believe him, and yet the tax collectors and prostitutes did. Even after seeing that, you refused to think better of it and believe in him. ",
            33: "'Listen to another parable. There was a man, a landowner, who planted a vineyard; he fenced it round, dug a winepress in it and built a tower; then he leased it to tenants and went abroad. ",
            34: "When vintage time drew near he sent his servants to the tenants to collect his produce. ",
            35: "But the tenants seized his servants, thrashed one, killed another and stoned a third.",
            36: "Next he sent some more servants, this time a larger number, and they dealt with them in the same way. ",
            37: 'Finally he sent his son to them thinking, "They will respect my son." ',
            38: 'But when the tenants saw the son, they said to each other, "This is the heir. Come on, let us kill him and take over his inheritance." ',
            39: "So they seized him and threw him out of the vineyard and killed him. ",
            40: "Now when the owner of the vineyard comes, what will he do to those tenants?' ",
            41: "They answered, 'He will bring those wretches to a wretched end and lease the vineyard to other tenants who will deliver the produce to him at the proper time.' ",
            42: "Jesus said to them, 'Have you never read in the scriptures: The stone which the builders rejected has become the cornerstone; this is the Lord's doing and we marvel at it? ",
            43: "'I tell you, then, that the kingdom of God will be taken from you and given to a people who will produce its fruit.' ",
            44: "",
            45: "When they heard his parables, the chief priests and the scribes realised he was speaking about them, ",
            46: "but though they would have liked to arrest him they were afraid of the crowds, who looked on him as a prophet.",
        }

        chapter_22 = {
            1: "Jesus began to speak to them in parables once again, ",
            2: "'The kingdom of Heaven may be compared to a king who gave a feast for his son's wedding. ",
            3: "He sent his servants to call those who had been invited, but they would not come. ",
            4: 'Next he sent some more servants with the words, "Tell those who have been invited: Look, my banquet is all prepared, my oxen and fattened cattle have been slaughtered, everything is ready. Come to the wedding." ',
            5: "But they were not interested: one went off to his farm, another to his business, ",
            6: "and the rest seized his servants, maltreated them and killed them. ",
            7: "The king was furious. He despatched his troops, destroyed those murderers and burnt their town. ",
            8: 'Then he said to his servants, "The wedding is ready; but as those who were invited proved to be unworthy, ',
            9: 'go to the main crossroads and invite everyone you can find to come to the wedding." ',
            10: "So these servants went out onto the roads and collected together everyone they could find, bad and good alike; and the wedding hall was filled with guests. ",
            11: "When the king came in to look at the guests he noticed one man who was not wearing a wedding garment, ",
            12: 'and said to him, "How did you get in here, my friend, without a wedding garment?" And the man was silent. ',
            13: 'Then the king said to the attendants, "Bind him hand and foot and throw him into the darkness outside, where there will be weeping and grinding of teeth." ',
            14: "For many are invited but not all are chosen.' ",
            15: "Then the Pharisees went away to work out between them how to trap him in what he said. ",
            16: "And they sent their disciples to him, together with some Herodians, to say, 'Master, we know that you are an honest man and teach the way of God in all honesty, and that you are not afraid of anyone, because human rank means nothing to you. ",
            17: "Give us your opinion, then. Is it permissible to pay taxes to Caesar or not?' ",
            18: "But Jesus was aware of their malice and replied, 'You hypocrites! Why are you putting me to the test? ",
            19: "Show me the money you pay the tax with.' They handed him a denarius, ",
            20: "and he said, 'Whose portrait is this? Whose title?' ",
            21: "They replied, 'Caesar's.' Then he said to them, 'Very well, pay Caesar what belongs to Caesar -- and God what belongs to God.' ",
            22: "When they heard this they were amazed; they left him alone and went away. ",
            23: "That day some Sadducees -- who deny that there is a resurrection -- approached him and they put this question to him, ",
            24: "'Master, Moses said that if a man dies childless, his brother is to marry the widow, his sister-in-law, to raise children for his brother. ",
            25: "Now we had a case involving seven brothers; the first married and then died without children, leaving his wife to his brother; ",
            26: "the same thing happened with the second and third and so on to the seventh, ",
            27: "and then last of all the woman herself died. ",
            28: "Now at the resurrection, whose wife among the seven will she be, since she had been married to them all?' ",
            29: "Jesus answered them, 'You are wrong, because you understand neither the scriptures nor the power of God. ",
            30: "For at the resurrection men and women do not marry; no, they are like the angels in heaven. ",
            31: "And as for the resurrection of the dead, have you never read what God himself said to you: ",
            32: "I am the God of Abraham, the God of Isaac and the God of Jacob? He is God, not of the dead, but of the living.'",
            33: "And his teaching made a deep impression on the people who heard it. ",
            34: "But when the Pharisees heard that he had silenced the Sadducees they got together ",
            35: "and, to put him to the test, one of them put a further question, ",
            36: "'Master, which is the greatest commandment of the Law?' ",
            37: "Jesus said to him, 'You must love the Lord your God with all your heart, with all your soul, and with all your mind. ",
            38: "This is the greatest and the first commandment. ",
            39: "The second resembles it: You must love your neighbour as yourself. ",
            40: "On these two commandments hang the whole Law, and the Prophets too.' ",
            41: "While the Pharisees were gathered round, Jesus put to them this question, ",
            42: "'What is your opinion about the Christ? Whose son is he?' They told him, 'David's.' ",
            43: "He said to them, 'Then how is it that David, moved by the Spirit, calls him Lord, where he says: ",
            44: "The Lord declared to my Lord, take your seat at my right hand, till I have made your enemies your footstool? ",
            45: "'If David calls him Lord, how then can he be his son?' ",
            46: "No one could think of anything to say in reply, and from that day no one dared to ask him any further questions."
        }

        chapter_23 = {
            1: "Then addressing the crowds and his disciples Jesus said, ",
            2: "'The scribes and the Pharisees occupy the chair of Moses. ",
            3: "You must therefore do and observe what they tell you; but do not be guided by what they do, since they do not practise what they preach. ",
            4: "They tie up heavy burdens and lay them on people's shoulders, but will they lift a finger to move them? Not they! ",
            5: "Everything they do is done to attract attention, like wearing broader headbands and longer tassels, ",
            6: "like wanting to take the place of honour at banquets and the front seats in the synagogues, ",
            7: "being greeted respectfully in the market squares and having people call them Rabbi. ",
            8: "'You, however, must not allow yourselves to be called Rabbi, since you have only one Master, and you are all brothers. ",
            9: "You must call no one on earth your father, since you have only one Father, and he is in heaven. ",
            10: "Nor must you allow yourselves to be called teachers, for you have only one Teacher, the Christ. ",
            11: "The greatest among you must be your servant. ",
            12: "Anyone who raises himself up will be humbled, and anyone who humbles himself will be raised up. ",
            13: "'Alas for you, scribes and Pharisees, you hypocrites! You shut up the kingdom of Heaven in people's faces, neither going in yourselves nor allowing others to go who want to. ",
            14: "",
            15: "'Alas for you, scribes and Pharisees, you hypocrites! You travel over sea and land to make a single proselyte, and anyone who becomes one you make twice as fit for hell as you are. ",
            16: """'Alas for you, blind guides! You say, "If anyone swears by the Temple, it has no force; but anyone who swears by the gold of the Temple is bound." """,
            17: "Fools and blind! For which is of greater value, the gold or the Temple that makes the gold sacred? ",
            18: 'Again, "If anyone swears by the altar it has no force; but anyone who swears by the offering on the altar, is bound." ',
            19: "You blind men! For which is of greater worth, the offering or the altar that makes the offering sacred? ",
            20: "Therefore, someone who swears by the altar is swearing by that and by everything on it. ",
            21: "And someone who swears by the Temple is swearing by that and by the One who dwells in it. ",
            22: "And someone who swears by heaven is swearing by the throne of God and by the One who is seated there. ",
            23: "'Alas for you, scribes and Pharisees, you hypocrites! You pay your tithe of mint and dill and cummin and have neglected the weightier matters of the Law-justice, mercy, good faith! These you should have practised, those not neglected. ",
            24: "You blind guides, straining out gnats and swallowing camels! ",
            25: "'Alas for you, scribes and Pharisees, you hypocrites! You clean the outside of cup and dish and leave the inside full of extortion and intemperance. ",
            26: "Blind Pharisee! Clean the inside of cup and dish first so that it and the outside are both clean. ",
            27: "'Alas for you, scribes and Pharisees, you hypocrites! You are like whitewashed tombs that look handsome on the outside, but inside are full of the bones of the dead and every kind of corruption. ",
            28: "In just the same way, from the outside you look upright, but inside you are full of hypocrisy and lawlessness. ",
            29: "'Alas for you, scribes and Pharisees, you hypocrites! You build the sepulchres of the prophets and decorate the tombs of the upright, ",
            30: '''saying, "We would never have joined in shedding the blood of the prophets, had we lived in our ancestors' day." ''',
            31: "So! Your own evidence tells against you! You are the children of those who murdered the prophets! ",
            32: "Very well then, finish off the work that your ancestors began. ",
            33: "'You serpents, brood of vipers, how can you escape being condemned to hell? ",
            34: "This is why -- look -- I am sending you prophets and wise men and scribes; some you will slaughter and crucify, some you will scourge in your synagogues and hunt from town to town; ",
            35: "and so you will draw down on yourselves the blood of every upright person that has been shed on earth, from the blood of Abel the holy to the blood of Zechariah son of Barachiah whom you murdered between the sanctuary and the altar. ",
            36: "In truth I tell you, it will all recoil on this generation. ",
            37: "'Jerusalem, Jerusalem, you that kill the prophets and stone those who are sent to you! How often have I longed to gather your children together, as a hen gathers her chicks under her wings, and you refused! ",
            38: "Look! Your house will be deserted, ",
            39: "for, I promise, you shall not see me any more until you are saying: Blessed is he who is coming in the name of the Lord!'"
        }

        chapter_24 = {
            1: "Jesus left the Temple, and as he was going away his disciples came up to draw his attention to the Temple buildings. ",
            2: "He said to them in reply, 'You see all these? In truth I tell you, not a single stone here will be left on another: everything will be pulled down.' ",
            3: "And while he was sitting on the Mount of Olives the disciples came and asked him when they were by themselves, 'Tell us, when is this going to happen, and what sign will there be of your coming and of the end of the world?' ",
            4: "And Jesus answered them, 'Take care that no one deceives you, ",
            5: 'because many will come using my name and saying, "I am the Christ," and they will deceive many. ',
            6: "You will hear of wars and rumours of wars; see that you are not alarmed, for this is something that must happen, but the end will not be yet. ",
            7: "For nation will fight against nation, and kingdom against kingdom. There will be famines and earthquakes in various places. ",
            8: "All this is only the beginning of the birthpangs. ",
            9: "'Then you will be handed over to be tortured and put to death; and you will be hated by all nations on account of my name. ",
            10: "And then many will fall away; people will betray one another and hate one another. ",
            11: "Many false prophets will arise; they will deceive many, ",
            12: "and with the increase of lawlessness, love in most people will grow cold; ",
            13: "but anyone who stands firm to the end will be saved. ",
            14: "'This good news of the kingdom will be proclaimed to the whole world as evidence to the nations. And then the end will come. ",
            15: "'So when you see the appalling abomination, of which the prophet Daniel spoke, set up in the holy place (let the reader understand), ",
            16: "then those in Judaea must escape to the mountains; ",
            17: "if anyone is on the housetop, he must not come down to collect his belongings from the house; ",
            18: "if anyone is in the fields, he must not turn back to fetch his cloak. ",
            19: "Alas for those with child, or with babies at the breast, when those days come! ",
            20: "Pray that you will not have to make your escape in winter or on a Sabbath. ",
            21: "For then there will be great distress, unparalleled since the world began, and such as will never be again. ",
            22: "And if that time had not been shortened, no human being would have survived; but shortened that time shall be, for the sake of those who are chosen. ",
            23: """'If anyone says to you then, "Look, here is the Christ," or "Over here," do not believe it; """,
            24: "for false Christs and false prophets will arise and provide great signs and portents, enough to deceive even the elect, if that were possible. ",
            25: "Look! I have given you warning. ",
            26: """'If, then, they say to you, "Look, he is in the desert," do not go there; "Look, he is in some hiding place," do not believe it; """,
            27: "because the coming of the Son of man will be like lightning striking in the east and flashing far into the west. ",
            28: "Wherever the corpse is, that is where the vultures will gather. ",
            29: "'Immediately after the distress of those days the sun will be darkened, the moon will not give its light, the stars will fall from the sky and the powers of the heavens will be shaken. ",
            30: "And then the sign of the Son of man will appear in heaven; then, too, all the peoples of the earth will beat their breasts; and they will see the Son of man coming on the clouds of heaven with power and great glory. ",
            31: "And he will send his angels with a loud trumpet to gather his elect from the four winds, from one end of heaven to the other. ",
            32: "'Take the fig tree as a parable: as soon as its twigs grow supple and its leaves come out, you know that summer is near. ",
            33: "So with you when you see all these things: know that he is near, right at the gates. ",
            34: "In truth I tell you, before this generation has passed away, all these things will have taken place. ",
            35: "Sky and earth will pass away, but my words will never pass away. ",
            36: "But as for that day and hour, nobody knows it, neither the angels of heaven, nor the Son, no one but the Father alone. ",
            37: "'As it was in Noah's day, so will it be when the Son of man comes. ",
            38: "For in those days before the Flood people were eating, drinking, taking wives, taking husbands, right up to the day Noah went into the ark, ",
            39: "and they suspected nothing till the Flood came and swept them all away. This is what it will be like when the Son of man comes. ",
            40: "Then of two men in the fields, one is taken, one left; ",
            41: "of two women grinding at the mill, one is taken, one left. ",
            42: "'So stay awake, because you do not know the day when your master is coming. ",
            43: "You may be quite sure of this, that if the householder had known at what time of the night the burglar would come, he would have stayed awake and would not have allowed anyone to break through the wall of his house. ",
            44: "Therefore, you too must stand ready because the Son of man is coming at an hour you do not expect. ",
            45: "'Who, then, is the wise and trustworthy servant whom the master placed over his household to give them their food at the proper time? ",
            46: "Blessed that servant if his master's arrival finds him doing exactly that. ",
            47: "In truth I tell you, he will put him in charge of everything he owns. ",
            48: 'But if the servant is dishonest and says to himself, "My master is taking his time," ',
            49: "and sets about beating his fellow-servants and eating and drinking with drunkards, ",
            50: "his master will come on a day he does not expect and at an hour he does not know. ",
            51: "The master will cut him off and send him to the same fate as the hypocrites, where there will be weeping and grinding of teeth.'"
        }

        chapter_25 = {
            1: "'Then the kingdom of Heaven will be like this: Ten wedding attendants took their lamps and went to meet the bridegroom. ",
            2: "Five of them were foolish and five were sensible: ",
            3: "the foolish ones, though they took their lamps, took no oil with them, ",
            4: "whereas the sensible ones took flasks of oil as well as their lamps. ",
            5: "The bridegroom was late, and they all grew drowsy and fell asleep. ",
            6: 'But at midnight there was a cry, "Look! The bridegroom! Go out and meet him." ',
            7: "Then all those wedding attendants woke up and trimmed their lamps, ",
            8: 'and the foolish ones said to the sensible ones, "Give us some of your oil: our lamps are going out." ',
            9: 'But they replied, "There may not be enough for us and for you; you had better go to those who sell it and buy some for yourselves." ',
            10: "They had gone off to buy it when the bridegroom arrived. Those who were ready went in with him to the wedding hall and the door was closed.",
            11: 'The other attendants arrived later. "Lord, Lord," they said, "open the door for us." ',
            12: 'But he replied, "In truth I tell you, I do not know you." ',
            13: "So stay awake, because you do not know either the day or the hour. ",
            14: "'It is like a man about to go abroad who summoned his servants and entrusted his property to them. ",
            15: "To one he gave five talents, to another two, to a third one, each in proportion to his ability. Then he set out on his journey. ",
            16: "The man who had received the five talents promptly went and traded with them and made five more. ",
            17: "The man who had received two made two more in the same way. ",
            18: "But the man who had received one went off and dug a hole in the ground and hid his master's money. ",
            19: "Now a long time afterwards, the master of those servants came back and went through his accounts with them. ",
            20: 'The man who had received the five talents came forward bringing five more. "Sir," he said, "you entrusted me with five talents; here are five more that I have made." ',
            21: '''His master said to him, "Well done, good and trustworthy servant; you have shown you are trustworthy in small things; I will trust you with greater; come and join in your master's happiness." ''',
            22: 'Next the man with the two talents came forward. "Sir," he said, "you entrusted me with two talents; here are two more that I have made." ',
            23: '''His master said to him, "Well done, good and trustworthy servant; you have shown you are trustworthy in small things; I will trust you with greater; come and join in your master's happiness." ''',
            24: 'Last came forward the man who had the single talent. "Sir," said he, "I had heard you were a hard man, reaping where you had not sown and gathering where you had not scattered; ',
            25: 'so I was afraid, and I went off and hid your talent in the ground. Here it is; it was yours, you have it back." ',
            26: 'But his master answered him, "You wicked and lazy servant! So you knew that I reap where I have not sown and gather where I have not scattered? ',
            27: "Well then, you should have deposited my money with the bankers, and on my return I would have got my money back with interest. ",
            28: "So now, take the talent from him and give it to the man who has the ten talents. ",
            29: "For to everyone who has will be given more, and he will have more than enough; but anyone who has not, will be deprived even of what he has. ",
            30: 'As for this good-for-nothing servant, throw him into the darkness outside, where there will be weeping and grinding of teeth." ',
            31: "'When the Son of man comes in his glory, escorted by all the angels, then he will take his seat on his throne of glory. ",
            32: "All nations will be assembled before him and he will separate people one from another as the shepherd separates sheep from goats. ",
            33: "He will place the sheep on his right hand and the goats on his left. ",
            34: 'Then the King will say to those on his right hand, "Come, you whom my Father has blessed, take as your heritage the kingdom prepared for you since the foundation of the world. ',
            35: "For I was hungry and you gave me food, I was thirsty and you gave me drink, I was a stranger and you made me welcome, ",
            36: 'lacking clothes and you clothed me, sick and you visited me, in prison and you came to see me." ',
            37: 'Then the upright will say to him in reply, "Lord, when did we see you hungry and feed you, or thirsty and give you drink? ',
            38: "When did we see you a stranger and make you welcome, lacking clothes and clothe you? ",
            39: 'When did we find you sick or in prison and go to see you?" ',
            40: 'And the King will answer, "In truth I tell you, in so far as you did this to one of the least of these brothers of mine, you did it to me." ',
            41: 'Then he will say to those on his left hand, "Go away from me, with your curse upon you, to the eternal fire prepared for the devil and his angels. ',
            42: "For I was hungry and you never gave me food, I was thirsty and you never gave me anything to drink, ",
            43: 'I was a stranger and you never made me welcome, lacking clothes and you never clothed me, sick and in prison and you never visited me."',
            44: 'Then it will be their turn to ask, "Lord, when did we see you hungry or thirsty, a stranger or lacking clothes, sick or in prison, and did not come to your help?" ',
            45: 'Then he will answer, "In truth I tell you, in so far as you neglected to do this to one of the least of these, you neglected to do it to me." ',
            46: "And they will go away to eternal punishment, and the upright to eternal life.'"
        }

        chapter_26 = {
            1: "Jesus had now finished all he wanted to say, and he told his disciples, ",
            2: "'It will be Passover, as you know, in two days' time, and the Son of man will be handed over to be crucified.' ",
            3: "Then the chief priests and the elders of the people assembled in the palace of the high priest, whose name was Caiaphas, ",
            4: "and made plans to arrest Jesus by some trick and have him put to death. ",
            5: "They said, however, 'It must not be during the festivities; there must be no disturbance among the people.' ",
            6: "Jesus was at Bethany in the house of Simon, a man who had suffered from a virulent skin-disease, when ",
            7: "a woman came to him with an alabaster jar of very expensive ointment, and poured it on his head as he was at table. ",
            8: "When they saw this, the disciples said indignantly, 'Why this waste? ",
            9: "This could have been sold for a high price and the money given the poor.' ",
            10: "But Jesus noticed this and said, 'Why are you upsetting the woman? What she has done for me is indeed a good work! ",
            11: "You have the poor with you always, but you will not always have me. ",
            12: "When she poured this ointment on my body, she did it to prepare me for burial. ",
            13: "In truth I tell you, wherever in all the world this gospel is proclaimed, what she has done will be told as well, in remembrance of her.' ",
            14: "Then one of the Twelve, the man called Judas Iscariot, went to the chief priests ",
            15: "and said, 'What are you prepared to give me if I hand him over to you?' They paid him thirty silver pieces, ",
            16: "and from then onwards he began to look for an opportunity to betray him. ",
            17: "Now on the first day of Unleavened Bread the disciples came to Jesus to say, 'Where do you want us to make the preparations for you to eat the Passover?' ",
            18: """He said, 'Go to a certain man in the city and say to him, "The Master says: My time is near. It is at your house that I am keeping Passover with my disciples." ' """,
            19: "The disciples did what Jesus told them and prepared the Passover. ",
            20: "When evening came he was at table with the Twelve. ",
            21: "And while they were eating he said, 'In truth I tell you, one of you is about to betray me.' ",
            22: "They were greatly distressed and started asking him in turn, 'Not me, Lord, surely?' ",
            23: "He answered, 'Someone who has dipped his hand into the dish with me will betray me. ",
            24: "The Son of man is going to his fate, as the scriptures say he will, but alas for that man by whom the Son of man is betrayed! Better for that man if he had never been born!' ",
            25: "Judas, who was to betray him, asked in his turn, 'Not me, Rabbi, surely?' Jesus answered, 'It is you who say it.' ",
            26: "Now as they were eating, Jesus took bread, and when he had said the blessing he broke it and gave it to the disciples. 'Take it and eat,' he said, 'this is my body.' ",
            27: "Then he took a cup, and when he had given thanks he handed it to them saying, 'Drink from this, all of you, ",
            28: "for this is my blood, the blood of the covenant, poured out for many for the forgiveness of sins. ",
            29: "From now on, I tell you, I shall never again drink wine until the day I drink the new wine with you in the kingdom of my Father.' ",
            30: "After the psalms had been sung they left for the Mount of Olives. ",
            31: "Then Jesus said to them, 'You will all fall away from me tonight, for the scripture says: I shall strike the shepherd and the sheep of the flock will be scattered, ",
            32: "but after my resurrection I shall go ahead of you to Galilee.' ",
            33: "At this, Peter said to him, 'Even if all fall away from you, I will never fall away.' ",
            34: "Jesus answered him, 'In truth I tell you, this very night, before the cock crows, you will have disowned me three times.' ",
            35: "Peter said to him, 'Even if I have to die with you, I will never disown you.' And all the disciples said the same. ",
            36: "Then Jesus came with them to a plot of land called Gethsemane; and he said to his disciples, 'Stay here while I go over there to pray.' ",
            37: "He took Peter and the two sons of Zebedee with him. And he began to feel sadness and anguish. ",
            38: "Then he said to them, 'My soul is sorrowful to the point of death. Wait here and stay awake with me.' ",
            39: "And going on a little further he fell on his face and prayed. 'My Father,' he said, 'if it is possible, let this cup pass me by. Nevertheless, let it be as you, not I, would have it.' ",
            40: "He came back to the disciples and found them sleeping, and he said to Peter, 'So you had not the strength to stay awake with me for one hour? ",
            41: "Stay awake, and pray not to be put to the test. The spirit is willing enough, but human nature is weak.' ",
            42: "Again, a second time, he went away and prayed: 'My Father,' he said, 'if this cup cannot pass by, but I must drink it, your will be done!' ",
            43: "And he came back again and found them sleeping, their eyes were so heavy. ",
            44: "Leaving them there, he went away again and prayed for the third time, repeating the same words. ",
            45: "Then he came back to the disciples and said to them, 'You can sleep on now and have your rest. Look, the hour has come when the Son of man is to be betrayed into the hands of sinners. ",
            46: "Get up! Let us go! Look, my betrayer is not far away.' ",
            47: "And suddenly while he was still speaking, Judas, one of the Twelve, appeared, and with him a large number of men armed with swords and clubs, sent by the chief priests and elders of the people. ",
            48: "Now the traitor had arranged a sign with them saying, 'The one I kiss, he is the man. Arrest him.' ",
            49: "So he went up to Jesus at once and said, 'Greetings, Rabbi,' and kissed him. ",
            50: "Jesus said to him, 'My friend, do what you are here for.' Then they came forward, seized Jesus and arrested him. ",
            51: "And suddenly, one of the followers of Jesus grasped his sword and drew it; he struck the high priest's servant and cut off his ear. ",
            52: "Jesus then said, 'Put your sword back, for all who draw the sword will die by the sword. ",
            53: "Or do you think that I cannot appeal to my Father, who would promptly send more than twelve legions of angels to my defence? ",
            54: "But then, how would the scriptures be fulfilled that say this is the way it must be?' ",
            55: "It was at this time that Jesus said to the crowds, 'Am I a bandit, that you had to set out to capture me with swords and clubs? I sat teaching in the Temple day after day and you never laid a hand on me.' ",
            56: "Now all this happened to fulfil the prophecies in scripture. Then all the disciples deserted him and ran away. ",
            57: "The men who had arrested Jesus led him off to the house of Caiaphas the high priest, where the scribes and the elders were assembled. ",
            58: "Peter followed him at a distance right to the high priest's palace, and he went in and sat down with the attendants to see what the end would be. ",
            59: "The chief priests and the whole Sanhedrin were looking for evidence against Jesus, however false, on which they might have him executed. ",
            60: "But they could not find any, though several lying witnesses came forward. Eventually two came forward ",
            61: """and made a statement, 'This man said, "I have power to destroy the Temple of God and in three days build it up." ' """,
            62: "The high priest then rose and said to him, 'Have you no answer to that? What is this evidence these men are bringing against you?' ",
            63: "But Jesus was silent. And the high priest said to him, 'I put you on oath by the living God to tell us if you are the Christ, the Son of God.' ",
            64: "Jesus answered him, 'It is you who say it. But, I tell you that from this time onward you will see the Son of man seated at the right hand of the Power and coming on the clouds of heaven.' ",
            65: "Then the high priest tore his clothes and said, 'He has blasphemed. What need of witnesses have we now? There! You have just heard the blasphemy. ",
            66: "What is your opinion?' They answered, 'He deserves to die.' ",
            67: "Then they spat in his face and hit him with their fists; others said as they struck him, ",
            68: "'Prophesy to us, Christ! Who hit you then?' ",
            69: "Meanwhile Peter was sitting outside in the courtyard, and a servant-girl came up to him saying, 'You, too, were with Jesus the Galilean.' ",
            70: "But he denied it in front of them all. 'I do not know what you are talking about,' he said. ",
            71: "When he went out to the gateway another servant-girl saw him and said to the people there, 'This man was with Jesus the Nazarene.' ",
            72: "And again, with an oath, he denied it, 'I do not know the man.' ",
            73: "A little later the bystanders came up and said to Peter, 'You are certainly one of them too! Why, your accent gives you away.' ",
            74: "Then he started cursing and swearing, 'I do not know the man.' And at once the cock crowed, ",
            75: "and Peter remembered what Jesus had said, 'Before the cock crows you will have disowned me three times.' And he went outside and wept bitterly."
        }

        chapter_27 = {
            1: "When morning came, all the chief priests and the elders of the people met in council to bring about the death of Jesus. ",
            2: "They had him bound and led him away to hand him over to Pilate, the governor. ",
            3: "When he found that Jesus had been condemned, then Judas, his betrayer, was filled with remorse and took the thirty silver pieces back to the chief priests and elders ",
            4: "saying, 'I have sinned. I have betrayed innocent blood.' They replied, 'What is that to us? That is your concern.' ",
            5: "And flinging down the silver pieces in the sanctuary he made off, and went and hanged himself. ",
            6: "The chief priests picked up the silver pieces and said, 'It is against the Law to put this into the treasury; it is blood-money.' ",
            7: "So they discussed the matter and with it bought the potter's field as a graveyard for foreigners, ",
            8: "and this is why the field is still called the Field of Blood. ",
            9: "The word spoken through the prophet Jeremiah was then fulfilled: And they took the thirty silver pieces, the sum at which the precious One was priced by the children of Israel, ",
            10: "and they gave them for the potter's field, just as the Lord directed me. ",
            11: "Jesus, then, was brought before the governor, and the governor put to him this question, 'Are you the king of the Jews?' Jesus replied, 'It is you who say it.' ",
            12: "But when he was accused by the chief priests and the elders he refused to answer at all. ",
            13: "Pilate then said to him, 'Do you not hear how many charges they have made against you?' ",
            14: "But to the governor's amazement, he offered not a word in answer to any of the charges. ",
            15: "At festival time it was the governor's practice to release a prisoner for the people, anyone they chose. ",
            16: "Now there was then a notorious prisoner whose name was Barabbas. ",
            17: "So when the crowd gathered, Pilate said to them, 'Which do you want me to release for you: Barabbas, or Jesus who is called Christ?' ",
            18: "For Pilate knew it was out of jealousy that they had handed him over. ",
            19: "Now as he was seated in the chair of judgement, his wife sent him a message, 'Have nothing to do with that upright man; I have been extremely upset today by a dream that I had about him.' ",
            20: "The chief priests and the elders, however, had persuaded the crowd to demand the release of Barabbas and the execution of Jesus. ",
            21: "So when the governor spoke and asked them, 'Which of the two do you want me to release for you?' they said, 'Barabbas.' ",
            22: "Pilate said to them, 'But in that case, what am I to do with Jesus who is called Christ?' They all said, 'Let him be crucified!' ",
            23: "He asked, 'But what harm has he done?' But they shouted all the louder, 'Let him be crucified!' ",
            24: "Then Pilate saw that he was making no impression, that in fact a riot was imminent. So he took some water, washed his hands in front of the crowd and said, 'I am innocent of this man's blood. It is your concern.' ",
            25: "And the people, every one of them, shouted back, 'Let his blood be on us and on our children!' ",
            26: "Then he released Barabbas for them. After having Jesus scourged he handed him over to be crucified. ",
            27: "Then the governor's soldiers took Jesus with them into the Praetorium and collected the whole cohort round him. ",
            28: "And they stripped him and put a scarlet cloak round him, ",
            29: "and having twisted some thorns into a crown they put this on his head and placed a reed in his right hand. To make fun of him they knelt to him saying, 'Hail, king of the Jews!' ",
            30: "And they spat on him and took the reed and struck him on the head with it. ",
            31: "And when they had finished making fun of him, they took off the cloak and dressed him in his own clothes and led him away to crucifixion. ",
            32: "On their way out, they came across a man from Cyrene, called Simon, and enlisted him to carry his cross. ",
            33: "When they had reached a place called Golgotha, that is, the place of the skull, ",
            34: "they gave him wine to drink mixed with gall, which he tasted but refused to drink. ",
            35: "When they had finished crucifying him they shared out his clothing by casting lots, ",
            36: "and then sat down and stayed there keeping guard over him. ",
            37: "Above his head was placed the charge against him; it read: 'This is Jesus, the King of the Jews.' ",
            38: "Then two bandits were crucified with him, one on the right and one on the left. ",
            39: "The passers-by jeered at him; they shook their heads ",
            40: "and said, 'So you would destroy the Temple and in three days rebuild it! Then save yourself if you are God's son and come down from the cross!' ",
            41: "The chief priests with the scribes and elders mocked him in the same way, ",
            42: "with the words, 'He saved others; he cannot save himself. He is the king of Israel; let him come down from the cross now, and we will believe in him. ",
            43: """He has put his trust in God; now let God rescue him if he wants him. For he did say, "I am God's son." ' """,
            44: "Even the bandits who were crucified with him taunted him in the same way. ",
            45: "From the sixth hour there was darkness over all the land until the ninth hour. ",
            46: "And about the ninth hour, Jesus cried out in a loud voice, 'Eli, eli, lama sabachthani?' that is, 'My God, my God, why have you forsaken me?' ",
            47: "When some of those who stood there heard this, they said, 'The man is calling on Elijah,' ",
            48: "and one of them quickly ran to get a sponge which he filled with vinegar and, putting it on a reed, gave it him to drink. ",
            49: "But the rest of them said, 'Wait! And see if Elijah will come to save him.' ",
            50: "But Jesus, again crying out in a loud voice, yielded up his spirit. ",
            51: "And suddenly, the veil of the Sanctuary was torn in two from top to bottom, the earth quaked, the rocks were split, ",
            52: "the tombs opened and the bodies of many holy people rose from the dead, ",
            53: "and these, after his resurrection, came out of the tombs, entered the holy city and appeared to a number of people. ",
            54: "The centurion, together with the others guarding Jesus, had seen the earthquake and all that was taking place, and they were terrified and said, 'In truth this man was son of God.' ",
            55: "And many women were there, watching from a distance, the same women who had followed Jesus from Galilee and looked after him. ",
            56: "Among them were Mary of Magdala, Mary the mother of James and Joseph, and the mother of Zebedee's sons. ",
            57: "When it was evening, there came a rich man of Arimathaea, called Joseph, who had himself become a disciple of Jesus. ",
            58: "This man went to Pilate and asked for the body of Jesus. Then Pilate ordered it to be handed over. ",
            59: "So Joseph took the body, wrapped it in a clean shroud ",
            60: "and put it in his own new tomb which he had hewn out of the rock. He then rolled a large stone across the entrance of the tomb and went away. ",
            61: "Now Mary of Magdala and the other Mary were there, sitting opposite the sepulchre. ",
            62: "Next day, that is, when Preparation Day was over, the chief priests and the Pharisees went in a body to Pilate ",
            63: '''and said to him, 'Your Excellency, we recall that this impostor said, while he was still alive, "After three days I shall rise again." ''',
            64: """Therefore give the order to have the sepulchre kept secure until the third day, for fear his disciples come and steal him away and tell the people, "He has risen from the dead." This last piece of fraud would be worse than what went before.' """,
            65: "Pilate said to them, 'You may have your guard; go and make all as secure as you know how.' ",
            66: "So they went and made the sepulchre secure, putting seals on the stone and mounting a guard."
        }

        chapter_28 = {
            1: "After the Sabbath, and towards dawn on the first day of the week, Mary of Magdala and the other Mary went to visit the sepulchre. ",
            2: "And suddenly there was a violent earthquake, for an angel of the Lord, descending from heaven, came and rolled away the stone and sat on it. ",
            3: "His face was like lightning, his robe white as snow. ",
            4: "The guards were so shaken by fear of him that they were like dead men. ",
            5: "But the angel spoke; and he said to the women, 'There is no need for you to be afraid. I know you are looking for Jesus, who was crucified. ",
            6: "He is not here, for he has risen, as he said he would. Come and see the place where he lay, ",
            7: """then go quickly and tell his disciples, "He has risen from the dead and now he is going ahead of you to Galilee; that is where you will see him." Look! I have told you.' """,
            8: "Filled with awe and great joy the women came quickly away from the tomb and ran to tell his disciples. ",
            9: "And suddenly, coming to meet them, was Jesus. 'Greetings,' he said. And the women came up to him and, clasping his feet, they did him homage. ",
            10: "Then Jesus said to them, 'Do not be afraid; go and tell my brothers that they must leave for Galilee; there they will see me.' ",
            11: "Now while they were on their way, some of the guards went off into the city to tell the chief priests all that had happened. ",
            12: "These held a meeting with the elders and, after some discussion, handed a considerable sum of money to the soldiers ",
            13: '''with these instructions, 'This is what you must say, "His disciples came during the night and stole him away while we were asleep." ''',
            14: "And should the governor come to hear of this, we undertake to put things right with him ourselves and to see that you do not get into trouble.' ",
            15: "So they took the money and carried out their instructions, and to this day that is the story among the Jews. ",
            16: "Meanwhile the eleven disciples set out for Galilee, to the mountain where Jesus had arranged to meet them. ",
            17: "When they saw him they fell down before him, though some hesitated. ",
            18: "Jesus came up and spoke to them. He said, 'All authority in heaven and on earth has been given to me. ",
            19: "Go, therefore, make disciples of all nations; baptise them in the name of the Father and of the Son and of the Holy Spirit, ",
            20: "and teach them to observe all the commands I gave you. And look, I am with you always; yes, to the end of time.'"
        }

        chapters = {
            1: chapter_1,
            2: chapter_2,
            3: chapter_3,
            4: chapter_4,
            5: chapter_5,
            6: chapter_6,
            7: chapter_7,
            8: chapter_8,
            9: chapter_9,
            10: chapter_10,
            11: chapter_11,
            12: chapter_12,
            13: chapter_13,
            14: chapter_14,
            15: chapter_15,
            16: chapter_16,
            17: chapter_17,
            18: chapter_18,
            19: chapter_19,
            20: chapter_20,
            21: chapter_21,
            22: chapter_22,
            23: chapter_23,
            24: chapter_24,
            25: chapter_25,
            26: chapter_26,
            27: chapter_27,
            28: chapter_28
        }

        return get_text(chapters, chapter, verse_start, verse_end, 28)

    def mark(self, chapter: int, verse_start: int, verse_end: int = None) -> Union[str, list]:
        """
        :param chapter: The chapter number
        :type chapter: int

        :param verse_start: The first verse
        :type verse_start: int

        :param verse_end: Alternatively, an end verse can be passed or let it
        as None to return only one verse
        :type verse_end: int or None
        """
        chapter_1 = {
            1: "The beginning of the gospel about Jesus Christ, the Son of God. ",
            2: "It is written in the prophet Isaiah: Look, I am going to send my messenger in front of you to prepare your way before you. ",
            3: "A voice of one that cries in the desert: Prepare a way for the Lord, make his paths straight. ",
            4: "John the Baptist was in the desert, proclaiming a baptism of repentance for the forgiveness of sins. ",
            5: "All Judaea and all the people of Jerusalem made their way to him, and as they were baptised by him in the river Jordan they confessed their sins. ",
            6: "John wore a garment of camel-skin, and he lived on locusts and wild honey. ",
            7: "In the course of his preaching he said, 'After me is coming someone who is more powerful than me, and I am not fit to kneel down and undo the strap of his sandals. ",
            8: "I have baptised you with water, but he will baptise you with the Holy Spirit.'",
            9: "It was at this time that Jesus came from Nazareth in Galilee and was baptised in the Jordan by John. ",
            10: "And at once, as he was coming up out of the water, he saw the heavens torn apart and the Spirit, like a dove, descending on him. ",
            11: "And a voice came from heaven, 'You are my Son, the Beloved; my favour rests on you.' ",
            12: "And at once the Spirit drove him into the desert ",
            13: "and he remained there for forty days, and was put to the test by Satan. He was with the wild animals, and the angels looked after him. ",
            14: "After John had been arrested, Jesus went into Galilee. There he proclaimed the gospel from God saying, ",
            15: "'The time is fulfilled, and the kingdom of God is close at hand. Repent, and believe the gospel.' ",
            16: "As he was walking along by the Lake of Galilee he saw Simon and Simon's brother Andrew casting a net in the lake -- for they were fishermen. ",
            17: "And Jesus said to them, 'Come after me and I will make you into fishers of people.' ",
            18: "And at once they left their nets and followed him. ",
            19: "Going on a little further, he saw James son of Zebedee and his brother John; they too were in their boat, mending the nets. ",
            20: "At once he called them and, leaving their father Zebedee in the boat with the men he employed, they went after him. ",
            21: "They went as far as Capernaum, and at once on the Sabbath he went into the synagogue and began to teach. ",
            22: "And his teaching made a deep impression on them because, unlike the scribes, he taught them with authority. ",
            23: "And at once in their synagogue there was a man with an unclean spirit, and he shouted, ",
            24: "'What do you want with us, Jesus of Nazareth? Have you come to destroy us? I know who you are: the Holy One of God.' ",
            25: "But Jesus rebuked it saying, 'Be quiet! Come out of him!' ",
            26: "And the unclean spirit threw the man into convulsions and with a loud cry went out of him. ",
            27: "The people were so astonished that they started asking one another what it all meant, saying, 'Here is a teaching that is new, and with authority behind it: he gives orders even to unclean spirits and they obey him.' ",
            28: "And his reputation at once spread everywhere, through all the surrounding Galilean countryside. ",
            29: "And at once on leaving the synagogue, he went with James and John straight to the house of Simon and Andrew. ",
            30: "Now Simon's mother-in-law was in bed and feverish, and at once they told him about her. ",
            31: "He went in to her, took her by the hand and helped her up. And the fever left her and she began to serve them. ",
            32: "That evening, after sunset, they brought to him all who were sick and those who were possessed by devils. ",
            33: "The whole town came crowding round the door, ",
            34: "and he cured many who were sick with diseases of one kind or another; he also drove out many devils, but he would not allow them to speak, because they knew who he was. ",
            35: "In the morning, long before dawn, he got up and left the house and went off to a lonely place and prayed there. ",
            36: "Simon and his companions set out in search of him, ",
            37: "and when they found him they said, 'Everybody is looking for you.' ",
            38: "He answered, 'Let us go elsewhere, to the neighbouring country towns, so that I can proclaim the message there too, because that is why I came.' ",
            39: "And he went all through Galilee, preaching in their synagogues and driving out devils. ",
            40: "A man suffering from a virulent skin-disease came to him and pleaded on his knees saying, 'If you are willing, you can cleanse me.' ",
            41: "Feeling sorry for him, Jesus stretched out his hand, touched him and said to him, 'I am willing. Be cleansed.' ",
            42: "And at once the skin-disease left him and he was cleansed. ",
            43: "And at once Jesus sternly sent him away and said to him, ",
            44: "'Mind you tell no one anything, but go and show yourself to the priest, and make the offering for your cleansing prescribed by Moses as evidence to them.' ",
            45: "The man went away, but then started freely proclaiming and telling the story everywhere, so that Jesus could no longer go openly into any town, but stayed outside in deserted places. Even so, people from all around kept coming to him."
        }

        chapter_2 = {
            1: "When he returned to Capernaum, some time later word went round that he was in the house; ",
            2: "and so many people collected that there was no room left, even in front of the door. He was preaching the word to them ",
            3: "when some people came bringing him a paralytic carried by four men, ",
            4: "but as they could not get the man to him through the crowd, they stripped the roof over the place where Jesus was; and when they had made an opening, they lowered the stretcher on which the paralytic lay. ",
            5: "Seeing their faith, Jesus said to the paralytic, 'My child, your sins are forgiven.' ",
            6: "Now some scribes were sitting there, and they thought to themselves, ",
            7: "'How can this man talk like that? He is being blasphemous. Who but God can forgive sins?' ",
            8: "And at once, Jesus, inwardly aware that this is what they were thinking, said to them, 'Why do you have these thoughts in your hearts? ",
            9: 'Which of these is easier: to say to the paralytic, "Your sins are forgiven" or to say, "Get up, pick up your stretcher and walk"? ',
            10: "But to prove to you that the Son of man has authority to forgive sins on earth' -- ",
            11: "he said to the paralytic-'I order you: get up, pick up your stretcher, and go off home.' ",
            12: "And the man got up, and at once picked up his stretcher and walked out in front of everyone, so that they were all astonished and praised God saying, 'We have never seen anything like this.' ",
            13: "He went out again to the shore of the lake; and all the people came to him, and he taught them. ",
            14: "As he was walking along he saw Levi the son of Alphaeus sitting at the tax office, and he said to him, 'Follow me.' And he got up and followed him. ",
            15: "When Jesus was at dinner in his house, a number of tax collectors and sinners were also sitting at table with Jesus and his disciples; for there were many of them among his followers. ",
            16: "When the scribes of the Pharisee party saw him eating with sinners and tax collectors, they said to his disciples, 'Why does he eat with tax collectors and sinners?' ",
            17: "When Jesus heard this he said to them, 'It is not the healthy who need the doctor, but the sick. I came to call not the upright, but sinners.' ",
            18: "John's disciples and the Pharisees were keeping a fast, when some people came to him and said to him, 'Why is it that John's disciples and the disciples of the Pharisees fast, but your disciples do not?' ",
            19: "Jesus replied, 'Surely the bridegroom's attendants cannot fast while the bridegroom is still with them? As long as they have the bridegroom with them, they cannot fast. ",
            20: "But the time will come when the bridegroom is taken away from them, and then, on that day, they will fast. ",
            21: "No one sews a piece of unshrunken cloth on an old cloak; otherwise, the patch pulls away from it, the new from the old, and the tear gets worse. ",
            22: "And nobody puts new wine into old wineskins; otherwise, the wine will burst the skins, and the wine is lost and the skins too. No! New wine into fresh skins!' ",
            23: "It happened that one Sabbath day he was taking a walk through the cornfields, and his disciples began to make a path by plucking ears of corn. ",
            24: "And the Pharisees said to him, 'Look, why are they doing something on the Sabbath day that is forbidden?' ",
            25: "And he replied, 'Have you never read what David did in his time of need when he and his followers were hungry- ",
            26: "how he went into the house of God when Abiathar was high priest, and ate the loaves of the offering which only the priests are allowed to eat, and how he also gave some to the men with him?' ",
            27: "And he said to them, 'The Sabbath was made for man, not man for the Sabbath; ",
            28: "so the Son of man is master even of the Sabbath.'"
        }

        chapter_3 = {
            1: "Another time he went into the synagogue, and there was a man present whose hand was withered. ",
            2: "And they were watching him to see if he would cure him on the Sabbath day, hoping for something to charge him with. ",
            3: "He said to the man with the withered hand, 'Get up and stand in the middle!'",
            4: "Then he said to them, 'Is it permitted on the Sabbath day to do good, or to do evil; to save life, or to kill?' But they said nothing. ",
            5: "Then he looked angrily round at them, grieved to find them so obstinate, and said to the man, 'Stretch out your hand.' He stretched it out and his hand was restored. ",
            6: "The Pharisees went out and began at once to plot with the Herodians against him, discussing how to destroy him. ",
            7: "Jesus withdrew with his disciples to the lakeside, and great crowds from Galilee followed him. From Judaea, ",
            8: "and from Jerusalem, and from Idumaea and Transjordan and the region of Tyre and Sidon, great numbers who had heard of all he was doing came to him. ",
            9: "And he asked his disciples to have a boat ready for him because of the crowd, to keep him from being crushed. ",
            10: "For he had cured so many that all who were afflicted in any way were crowding forward to touch him. ",
            11: "And the unclean spirits, whenever they saw him, would fall down before him and shout, 'You are the Son of God!' ",
            12: "But he warned them strongly not to make him known. ",
            13: "He now went up onto the mountain and summoned those he wanted. So they came to him ",
            14: "and he appointed twelve; they were to be his companions and to be sent out to proclaim the message, ",
            15: "with power to drive out devils. ",
            16: "And so he appointed the Twelve, Simon to whom he gave the name Peter, ",
            17: "James the son of Zebedee and John the brother of James, to whom he gave the name Boanerges or 'Sons of Thunder'; ",
            18: "Andrew, Philip, Bartholomew, Matthew, Thomas, James the son of Alphaeus, Thaddaeus, Simon the Zealot ",
            19: "and Judas Iscariot, the man who was to betray him. ",
            20: "He went home again, and once more such a crowd collected that they could not even have a meal. ",
            21: "When his relations heard of this, they set out to take charge of him; they said, 'He is out of his mind.' ",
            22: "The scribes who had come down from Jerusalem were saying, 'Beelzebul is in him,' and, 'It is through the prince of devils that he drives devils out.' ",
            23: "So he called them to him and spoke to them in parables, ",
            24: "'How can Satan drive out Satan? If a kingdom is divided against itself, that kingdom cannot last. ",
            25: "And if a household is divided against itself, that household can never last. ",
            26: "Now if Satan has rebelled against himself and is divided, he cannot last either -- it is the end of him. ",
            27: "But no one can make his way into a strong man's house and plunder his property unless he has first tied up the strong man. Only then can he plunder his house. ",
            28: "'In truth I tell you, all human sins will be forgiven, and all the blasphemies ever uttered; ",
            29: "but anyone who blasphemes against the Holy Spirit will never be forgiven, but is guilty of an eternal sin.' ",
            30: "This was because they were saying, 'There is an unclean spirit in him.' ",
            31: "Now his mother and his brothers arrived and, standing outside, sent in a message asking for him. ",
            32: "A crowd was sitting round him at the time the message was passed to him, 'Look, your mother and brothers and sisters are outside asking for you.' ",
            33: "He replied, 'Who are my mother and my brothers?' ",
            34: "And looking at those sitting in a circle round him, he said, 'Here are my mother and my brothers. ",
            35: "Anyone who does the will of God, that person is my brother and sister and mother.'"
        }

        chapter_4 = {
            1: "Again he began to teach them by the lakeside, but such a huge crowd gathered round him that he got into a boat on the water and sat there. The whole crowd were at the lakeside on land. ",
            2: "He taught them many things in parables, and in the course of his teaching he said to them, ",
            3: "'Listen! Imagine a sower going out to sow. ",
            4: "Now it happened that, as he sowed, some of the seed fell on the edge of the path, and the birds came and ate it up. ",
            5: "Some seed fell on rocky ground where it found little soil and at once sprang up, because there was no depth of earth; ",
            6: "and when the sun came up it was scorched and, not having any roots, it withered away. ",
            7: "Some seed fell into thorns, and the thorns grew up and choked it, and it produced no crop. ",
            8: "And some seeds fell into rich soil, grew tall and strong, and produced a good crop; the yield was thirty, sixty, even a hundredfold.' ",
            9: "And he said, 'Anyone who has ears for listening should listen!' ",
            10: "When he was alone, the Twelve, together with the others who formed his company, asked what the parables meant. ",
            11: "He told them, 'To you is granted the secret of the kingdom of God, but to those who are outside everything comes in parables, ",
            12: "so that they may look and look, but never perceive; listen and listen, but never understand; to avoid changing their ways and being healed.' ",
            13: "He said to them, 'Do you not understand this parable? Then how will you understand any of the parables? ",
            14: "What the sower is sowing is the word. ",
            15: "Those on the edge of the path where the word is sown are people who have no sooner heard it than Satan at once comes and carries away the word that was sown in them. ",
            16: "Similarly, those who are sown on patches of rock are people who, when first they hear the word, welcome it at once with joy. ",
            17: "But they have no root deep down and do not last; should some trial come, or some persecution on account of the word, at once they fall away. ",
            18: "Then there are others who are sown in thorns. These have heard the word, ",
            19: "but the worries of the world, the lure of riches and all the other passions come in to choke the word, and so it produces nothing. ",
            20: "And there are those who have been sown in rich soil; they hear the word and accept it and yield a harvest, thirty and sixty and a hundredfold.' ",
            21: "He also said to them, 'Is a lamp brought in to be put under a tub or under the bed? Surely to be put on the lamp-stand? ",
            22: "For there is nothing hidden, but it must be disclosed, nothing kept secret except to be brought to light. ",
            23: "Anyone who has ears for listening should listen!' ",
            24: "He also said to them, 'Take notice of what you are hearing. The standard you use will be used for you -- and you will receive more besides; ",
            25: "anyone who has, will be given more; anyone who has not, will be deprived even of what he has.' ",
            26: "He also said, 'This is what the kingdom of God is like. A man scatters seed on the land. ",
            27: "Night and day, while he sleeps, when he is awake, the seed is sprouting and growing; how, he does not know. ",
            28: "Of its own accord the land produces first the shoot, then the ear, then the full grain in the ear. ",
            29: "And when the crop is ready, at once he starts to reap because the harvest has come.' ",
            30: "He also said, 'What can we say that the kingdom is like? What parable can we find for it? ",
            31: "It is like a mustard seed which, at the time of its sowing, is the smallest of all the seeds on earth. ",
            32: "Yet once it is sown it grows into the biggest shrub of them all and puts out big branches so that the birds of the air can shelter in its shade.' ",
            33: "Using many parables like these, he spoke the word to them, so far as they were capable of understanding it. ",
            34: "He would not speak to them except in parables, but he explained everything to his disciples when they were by themselves. ",
            35: "With the coming of evening that same day, he said to them, 'Let us cross over to the other side.' ",
            36: "And leaving the crowd behind they took him, just as he was, in the boat; and there were other boats with him. ",
            37: "Then it began to blow a great gale and the waves were breaking into the boat so that it was almost swamped. ",
            38: "But he was in the stern, his head on the cushion, asleep. ",
            39: "They woke him and said to him, 'Master, do you not care? We are lost!' And he woke up and rebuked the wind and said to the sea, 'Quiet now! Be calm!' And the wind dropped, and there followed a great calm. ",
            40: "Then he said to them, 'Why are you so frightened? Have you still no faith?' ",
            41: "They were overcome with awe and said to one another, 'Who can this be? Even the wind and the sea obey him.'"
        }

        chapter_5 = {
            1: "They reached the territory of the Gerasenes on the other side of the lake, ",
            2: "and when he disembarked, a man with an unclean spirit at once came out from the tombs towards him. ",
            3: "The man lived in the tombs and no one could secure him any more, even with a chain, ",
            4: "because he had often been secured with fetters and chains but had snapped the chains and broken the fetters, and no one had the strength to control him. ",
            5: "All night and all day, among the tombs and in the mountains, he would howl and gash himself with stones. ",
            6: "Catching sight of Jesus from a distance, he ran up and fell at his feet ",
            7: "and shouted at the top of his voice, 'What do you want with me, Jesus, son of the Most High God? In God's name do not torture me!' ",
            8: "For Jesus had been saying to him, 'Come out of the man, unclean spirit.' ",
            9: "Then he asked, 'What is your name?' He answered, 'My name is Legion, for there are many of us.' ",
            10: "And he begged him earnestly not to send them out of the district. ",
            11: "Now on the mountainside there was a great herd of pigs feeding, ",
            12: "and the unclean spirits begged him, 'Send us to the pigs, let us go into them.' ",
            13: "So he gave them leave. With that, the unclean spirits came out and went into the pigs, and the herd of about two thousand pigs charged down the cliff into the lake, and there they were drowned. ",
            14: "The men looking after them ran off and told their story in the city and in the country round about; and the people came to see what had really happened. ",
            15: "They came to Jesus and saw the demoniac sitting there -- the man who had had the legion in him -- properly dressed and in his full senses, and they were afraid. ",
            16: "And those who had witnessed it reported what had happened to the demoniac and what had become of the pigs. ",
            17: "Then they began to implore Jesus to leave their neighbourhood. ",
            18: "As he was getting into the boat, the man who had been possessed begged to be allowed to stay with him. ",
            19: "Jesus would not let him but said to him, 'Go home to your people and tell them all that the Lord in his mercy has done for you.' ",
            20: "So the man went off and proceeded to proclaim in the Decapolis all that Jesus had done for him. And everyone was amazed. ",
            21: "When Jesus had crossed again in the boat to the other side, a large crowd gathered round him and he stayed by the lake. ",
            22: "Then the president of the synagogue came up, named Jairus, and seeing him, fell at his feet ",
            23: "and begged him earnestly, saying, 'My little daughter is desperately sick. Do come and lay your hands on her that she may be saved and may live.' ",
            24: "Jesus went with him and a large crowd followed him; they were pressing all round him. ",
            25: "Now there was a woman who had suffered from a haemorrhage for twelve years; ",
            26: "after long and painful treatment under various doctors, she had spent all she had without being any the better for it; in fact, she was getting worse. ",
            27: "She had heard about Jesus, and she came up through the crowd and touched his cloak from behind, thinking, ",
            28: "'If I can just touch his clothes, I shall be saved.' ",
            29: "And at once the source of the bleeding dried up, and she felt in herself that she was cured of her complaint. ",
            30: "And at once aware of the power that had gone out from him, Jesus turned round in the crowd and said, 'Who touched my clothes?' ",
            31: """His disciples said to him, 'You see how the crowd is pressing round you; how can you ask, "Who touched me?" ' """,
            32: "But he continued to look all round to see who had done it. ",
            33: "Then the woman came forward, frightened and trembling because she knew what had happened to her, and she fell at his feet and told him the whole truth. ",
            34: "'My daughter,' he said, 'your faith has restored you to health; go in peace and be free of your complaint.' ",
            35: "While he was still speaking some people arrived from the house of the president of the synagogue to say, 'Your daughter is dead; why put the Master to any further trouble?' ",
            36: "But Jesus overheard what they said and he said to the president of the synagogue, 'Do not be afraid; only have faith.' ",
            37: "And he allowed no one to go with him except Peter and James and John the brother of James. ",
            38: "So they came to the house of the president of the synagogue, and Jesus noticed all the commotion, with people weeping and wailing unrestrainedly. ",
            39: "He went in and said to them, 'Why all this commotion and crying? The child is not dead, but asleep.' ",
            40: "But they ridiculed him. So he turned them all out and, taking with him the child's father and mother and his own companions, he went into the place where the child lay. ",
            41: "And taking the child by the hand he said to her, 'Talitha kum!' which means, 'Little girl, I tell you to get up.' ",
            42: "The little girl got up at once and began to walk about, for she was twelve years old. At once they were overcome with astonishment, ",
            43: "and he gave them strict orders not to let anyone know about it, and told them to give her something to eat."
        }

        chapter_6 = {
            1: "Leaving that district, he went to his home town, and his disciples accompanied him. ",
            2: "With the coming of the Sabbath he began teaching in the synagogue, and most of them were astonished when they heard him. They said, 'Where did the man get all this? What is this wisdom that has been granted him, and these miracles that are worked through him? ",
            3: "This is the carpenter, surely, the son of Mary, the brother of James and Joset and Jude and Simon? His sisters, too, are they not here with us?' And they would not accept him. ",
            4: "And Jesus said to them, 'A prophet is despised only in his own country, among his own relations and in his own house'; ",
            5: "and he could work no miracle there, except that he cured a few sick people by laying his hands on them. ",
            6: "He was amazed at their lack of faith. He made a tour round the villages, teaching. ",
            7: "Then he summoned the Twelve and began to send them out in pairs, giving them authority over unclean spirits. ",
            8: "And he instructed them to take nothing for the journey except a staff -- no bread, no haversack, no coppers for their purses. ",
            9: "They were to wear sandals but, he added, 'Don't take a spare tunic.' ",
            10: "And he said to them, 'If you enter a house anywhere, stay there until you leave the district. ",
            11: "And if any place does not welcome you and people refuse to listen to you, as you walk away shake off the dust under your feet as evidence to them.' ",
            12: "So they set off to proclaim repentance; ",
            13: "and they cast out many devils, and anointed many sick people with oil and cured them. ",
            14: "King Herod had heard about him, since by now his name was well known. Some were saying, 'John the Baptist has risen from the dead, and that is why miraculous powers are at work in him.' ",
            15: "Others said, 'He is Elijah,' others again, 'He is a prophet, like the prophets we used to have.' ",
            16: "But when Herod heard this he said, 'It is John whose head I cut off; he has risen from the dead.' ",
            17: "Now it was this same Herod who had sent to have John arrested, and had had him chained up in prison because of Herodias, his brother Philip's wife whom he had married. ",
            18: "For John had told Herod, 'It is against the law for you to have your brother's wife.' ",
            19: "As for Herodias, she was furious with him and wanted to kill him, but she was not able to do so, ",
            20: "because Herod was in awe of John, knowing him to be a good and upright man, and gave him his protection. When he had heard him speak he was greatly perplexed, and yet he liked to listen to him. ",
            21: "An opportunity came on Herod's birthday when he gave a banquet for the nobles of his court, for his army officers and for the leading figures in Galilee. ",
            22: "When the daughter of this same Herodias came in and danced, she delighted Herod and his guests; so the king said to the girl, 'Ask me anything you like and I will give it you.' ",
            23: "And he swore her an oath, 'I will give you anything you ask, even half my kingdom.' ",
            24: "She went out and said to her mother, 'What shall I ask for?' She replied, 'The head of John the Baptist.' ",
            25: "The girl at once rushed back to the king and made her request, 'I want you to give me John the Baptist's head, immediately, on a dish.' ",
            26: "The king was deeply distressed but, thinking of the oaths he had sworn and of his guests, he was reluctant to break his word to her. ",
            27: "At once the king sent one of the bodyguard with orders to bring John's head. ",
            28: "The man went off and beheaded him in the prison; then he brought the head on a dish and gave it to the girl, and the girl gave it to her mother. ",
            29: "When John's disciples heard about this, they came and took his body and laid it in a tomb. ",
            30: "The apostles rejoined Jesus and told him all they had done and taught. ",
            31: "And he said to them, 'Come away to some lonely place all by yourselves and rest for a while'; for there were so many coming and going that there was no time for them even to eat. ",
            32: "So they went off in the boat to a lonely place where they could be by themselves.",
            33: "But people saw them going, and many recognised them; and from every town they all hurried to the place on foot and reached it before them. ",
            34: "So as he stepped ashore he saw a large crowd; and he took pity on them because they were like sheep without a shepherd, and he set himself to teach them at some length. ",
            35: "By now it was getting very late, and his disciples came up to him and said, 'This is a lonely place and it is getting very late, ",
            36: "so send them away, and they can go to the farms and villages round about, to buy themselves something to eat.' ",
            37: "He replied, 'Give them something to eat yourselves.' They answered, 'Are we to go and spend two hundred denarii on bread for them to eat?' ",
            38: "He asked, 'How many loaves have you? Go and see.' And when they had found out they said, 'Five, and two fish.' ",
            39: "Then he ordered them to get all the people to sit down in groups on the green grass, ",
            40: "and they sat down on the ground in squares of hundreds and fifties. ",
            41: "Then he took the five loaves and the two fish, raised his eyes to heaven and said the blessing; then he broke the loaves and began handing them to his disciples to distribute among the people. He also shared out the two fish among them all. ",
            42: "They all ate as much as they wanted. ",
            43: "They collected twelve basketfuls of scraps of bread and pieces of fish. ",
            44: "Those who had eaten the loaves numbered five thousand men. ",
            45: "And at once he made his disciples get into the boat and go on ahead to the other side near Bethsaida, while he himself sent the crowd away. ",
            46: "After saying goodbye to them he went off into the hills to pray. ",
            47: "When evening came, the boat was far out on the sea, and he was alone on the land. ",
            48: "He could see that they were hard pressed in their rowing, for the wind was against them; and about the fourth watch of the night he came towards them, walking on the sea. He was going to pass them by, ",
            49: "but when they saw him walking on the sea they thought it was a ghost and cried out; ",
            50: "for they had all seen him and were terrified. But at once he spoke to them and said, 'Courage! It's me! Don't be afraid.' ",
            51: "Then he got into the boat with them and the wind dropped. They were utterly and completely dumbfounded, ",
            52: "because they had not seen what the miracle of the loaves meant; their minds were closed. ",
            53: "Having made the crossing, they came to land at Gennesaret and moored there. ",
            54: "When they disembarked people at once recognised him, ",
            55: "and started hurrying all through the countryside and brought the sick on stretchers to wherever they heard he was. ",
            56: "And wherever he went, to village or town or farm, they laid down the sick in the open spaces, begging him to let them touch even the fringe of his cloak. And all those who touched him were saved."
        }

        chapter_7 = {
            1: "The Pharisees and some of the scribes who had come from Jerusalem gathered round him, ",
            2: "and they noticed that some of his disciples were eating with unclean hands, that is, without washing them. ",
            3: "For the Pharisees, and all the Jews, keep the tradition of the elders and never eat without washing their arms as far as the elbow; ",
            4: "and on returning from the market place they never eat without first sprinkling themselves. There are also many other observances which have been handed down to them to keep, concerning the washing of cups and pots and bronze dishes. ",
            5: "So the Pharisees and scribes asked him, 'Why do your disciples not respect the tradition of the elders but eat their food with unclean hands?' ",
            6: "He answered, 'How rightly Isaiah prophesied about you hypocrites in the passage of scripture: This people honours me only with lip-service, while their hearts are far from me. ",
            7: "Their reverence of me is worthless; the lessons they teach are nothing but human commandments. ",
            8: "You put aside the commandment of God to observe human traditions.' ",
            9: "And he said to them, 'How ingeniously you get round the commandment of God in order to preserve your own tradition! ",
            10: "For Moses said: Honour your father and your mother, and, Anyone who curses father or mother must be put to death. ",
            11: 'But you say, "If a man says to his father or mother: Anything I have that I might have used to help you is Korban (that is, dedicated to God)," ',
            12: "then he is forbidden from that moment to do anything for his father or mother. ",
            13: "In this way you make God's word ineffective for the sake of your tradition which you have handed down. And you do many other things like this.' ",
            14: "He called the people to him again and said, 'Listen to me, all of you, and understand. ",
            15: "Nothing that goes into someone from outside can make that person unclean; it is the things that come out of someone that make that person unclean. ",
            16: "Anyone who has ears for listening should listen!' ",
            17: "When he had gone into the house, away from the crowd, his disciples questioned him about the parable. ",
            18: "He said to them, 'Even you -- don't you understand? Can't you see that nothing that goes into someone from outside can make that person unclean, ",
            19: "because it goes not into the heart but into the stomach and passes into the sewer? ",
            20: "And he went on, 'It is what comes out of someone that makes that person unclean. ",
            21: "For it is from within, from the heart, that evil intentions emerge: fornication, theft, murder, ",
            22: "adultery, avarice, malice, deceit, indecency, envy, slander, pride, folly. ",
            23: "All these evil things come from within and make a person unclean.' ",
            24: "He left that place and set out for the territory of Tyre. There he went into a house and did not want anyone to know he was there; but he could not pass unrecognised. ",
            25: "At once a woman whose little daughter had an unclean spirit heard about him and came and fell at his feet. ",
            26: "Now this woman was a gentile, by birth a Syro-Phoenician, and she begged him to drive the devil out of her daughter. ",
            27: "And he said to her, 'The children should be fed first, because it is not fair to take the children's food and throw it to little dogs.' ",
            28: "But she spoke up, 'Ah yes, sir,' she replied, 'but little dogs under the table eat the scraps from the children.' ",
            29: "And he said to her, 'For saying this you may go home happy; the devil has gone out of your daughter.' ",
            30: "So she went off home and found the child lying on the bed and the devil gone. ",
            31: "Returning from the territory of Tyre, he went by way of Sidon towards the Lake of Galilee, right through the Decapolis territory. ",
            32: "And they brought him a deaf man who had an impediment in his speech; and they asked him to lay his hand on him. ",
            33: "He took him aside to be by themselves, away from the crowd, put his fingers into the man's ears and touched his tongue with spittle. ",
            34: "Then looking up to heaven he sighed; and he said to him, 'Ephphatha,' that is, 'Be opened.' ",
            35: "And his ears were opened, and at once the impediment of his tongue was loosened and he spoke clearly. ",
            36: "And Jesus ordered them to tell no one about it, but the more he insisted, the more widely they proclaimed it. ",
            37: "Their admiration was unbounded, and they said, 'Everything he does is good, he makes the deaf hear and the dumb speak.'"
        }

        chapter_8 = {
            1: "And now once again a great crowd had gathered, and they had nothing to eat. So he called his disciples to him and said to them, ",
            2: "'I feel sorry for all these people; they have been with me for three days now and have nothing to eat. ",
            3: "If I send them off home hungry they will collapse on the way; some have come a great distance.' ",
            4: "His disciples replied, 'Where could anyone get these people enough bread to eat in a deserted place?' ",
            5: "He asked them, 'How many loaves have you?' And they said to him, 'Seven.' ",
            6: "Then he instructed the crowd to sit down on the ground, and he took the seven loaves, and after giving thanks he broke them and began handing them to his disciples to distribute; and they distributed them among the crowd. ",
            7: "They had a few small fishes as well, and over these he said a blessing and ordered them to be distributed too. ",
            8: "They ate as much as they wanted, and they collected seven basketfuls of the scraps left over. ",
            9: "Now there had been about four thousand people. He sent them away ",
            10: "and at once, getting into the boat with his disciples, went to the region of Dalmanutha. ",
            11: "The Pharisees came up and started a discussion with him; they demanded of him a sign from heaven, to put him to the test. ",
            12: "And with a profound sigh he said, 'Why does this generation demand a sign? In truth I tell you, no sign shall be given to this generation.' ",
            13: "And, leaving them again, he re-embarked and went away to the other side. ",
            14: "The disciples had forgotten to take any bread and they had only one loaf with them in the boat. ",
            15: "Then he gave them this warning, 'Keep your eyes open; look out for the yeast of the Pharisees and the yeast of Herod.' ",
            16: "And they said to one another, 'It is because we have no bread.' ",
            17: "And Jesus knew it, and he said to them, 'Why are you talking about having no bread? Do you still not understand, still not realise? Are your minds closed? ",
            18: "Have you eyes and do not see, ears and do not hear? Or do you not remember? ",
            19: "When I broke the five loaves for the five thousand, how many baskets full of scraps did you collect?' They answered, 'Twelve.' ",
            20: "'And when I broke the seven loaves for the four thousand, how many baskets full of scraps did you collect?' And they answered, 'Seven.' ",
            21: "Then he said to them, 'Do you still not realise?' ",
            22: "They came to Bethsaida, and some people brought to him a blind man whom they begged him to touch. ",
            23: "He took the blind man by the hand and led him outside the village. Then, putting spittle on his eyes and laying his hands on him, he asked, 'Can you see anything?' ",
            24: "The man, who was beginning to see, replied, 'I can see people; they look like trees as they walk around.' ",
            25: "Then he laid his hands on the man's eyes again and he saw clearly; he was cured, and he could see everything plainly and distinctly. ",
            26: "And Jesus sent him home, saying, 'Do not even go into the village.' ",
            27: "Jesus and his disciples left for the villages round Caesarea Philippi. On the way he put this question to his disciples, 'Who do people say I am?' ",
            28: "And they told him, 'John the Baptist, others Elijah, others again, one of the prophets.' ",
            29: "'But you,' he asked them, 'who do you say I am?' Peter spoke up and said to him, 'You are the Christ.' ",
            30: "And he gave them strict orders not to tell anyone about him. ",
            31: "Then he began to teach them that the Son of man was destined to suffer grievously, and to be rejected by the elders and the chief priests and the scribes, and to be put to death, and after three days to rise again; ",
            32: "and he said all this quite openly. Then, taking him aside, Peter tried to rebuke him. ",
            33: "But, turning and seeing his disciples, he rebuked Peter and said to him, 'Get behind me, Satan! You are thinking not as God thinks, but as human beings do.' ",
            34: "He called the people and his disciples to him and said, 'If anyone wants to be a follower of mine, let him renounce himself and take up his cross and follow me. ",
            35: "Anyone who wants to save his life will lose it; but anyone who loses his life for my sake, and for the sake of the gospel, will save it. ",
            36: "What gain, then, is it for anyone to win the whole world and forfeit his life? ",
            37: "And indeed what can anyone offer in exchange for his life? ",
            38: "For if anyone in this sinful and adulterous generation is ashamed of me and of my words, the Son of man will also be ashamed of him when he comes in the glory of his Father with the holy angels.'"
        }

        chapter_9 = {
            1: "And he said to them, 'In truth I tell you, there are some standing here who will not taste death before they see the kingdom of God come with power.' ",
            2: "Six days later, Jesus took with him Peter and James and John and led them up a high mountain on their own by themselves. There in their presence he was transfigured: ",
            3: "his clothes became brilliantly white, whiter than any earthly bleacher could make them. ",
            4: "Elijah appeared to them with Moses; and they were talking to Jesus. ",
            5: "Then Peter spoke to Jesus, 'Rabbi,' he said, 'it is wonderful for us to be here; so let us make three shelters, one for you, one for Moses and one for Elijah.' ",
            6: "He did not know what to say; they were so frightened. ",
            7: "And a cloud came, covering them in shadow; and from the cloud there came a voice, 'This is my Son, the Beloved. Listen to him.' ",
            8: "Then suddenly, when they looked round, they saw no one with them any more but only Jesus. ",
            9: "As they were coming down from the mountain he warned them to tell no one what they had seen, until after the Son of man had risen from the dead. ",
            10: "They observed the warning faithfully, though among themselves they discussed what 'rising from the dead' could mean. ",
            11: "And they put this question to him, 'Why do the scribes say that Elijah must come first?' ",
            12: "He said to them, 'Elijah is indeed first coming to set everything right again; yet how is it that the scriptures say about the Son of man that he must suffer grievously and be treated with contempt? ",
            13: "But I tell you that Elijah has come and they have treated him as they pleased, just as the scriptures say about him.' ",
            14: "As they were rejoining the disciples they saw a large crowd round them and some scribes arguing with them. ",
            15: "At once, when they saw him, the whole crowd were struck with amazement and ran to greet him. ",
            16: "And he asked them, 'What are you arguing about with them?' ",
            17: "A man answered him from the crowd, 'Master, I have brought my son to you; there is a spirit of dumbness in him, ",
            18: "and when it takes hold of him it throws him to the ground, and he foams at the mouth and grinds his teeth and goes rigid. And I asked your disciples to drive it out and they were unable to.' ",
            19: "In reply he said to them, 'Faithless generation, how much longer must I be among you? How much longer must I put up with you? Bring him to me.' ",
            20: "They brought the boy to him, and at once the spirit of dumbness threw the boy into convulsions, and he fell to the ground and lay writhing there, foaming at the mouth. ",
            21: "Jesus asked the father, 'How long has this been happening to him?' 'From childhood,' he said, ",
            22: "'and it has often thrown him into fire and into water, in order to destroy him. ",
            23: "But if you can do anything, have pity on us and help us.' ",
            24: "'If you can?' retorted Jesus. 'Everything is possible for one who has faith.' At once the father of the boy cried out, 'I have faith. Help my lack of faith!' ",
            25: "And when Jesus saw that a crowd was gathering, he rebuked the unclean spirit. 'Deaf and dumb spirit,' he said, 'I command you: come out of him and never enter him again.' ",
            26: "Then it threw the boy into violent convulsions and came out shouting, and the boy lay there so like a corpse that most of them said, 'He is dead.' ",
            27: "But Jesus took him by the hand and helped him up, and he was able to stand. ",
            28: "When he had gone indoors, his disciples asked him when they were by themselves, 'Why were we unable to drive it out?' ",
            29: "He answered, 'This is the kind that can be driven out only by prayer.' ",
            30: "After leaving that place they made their way through Galilee; and he did not want anyone to know, ",
            31: "because he was instructing his disciples; he was telling them, 'The Son of man will be delivered into the power of men; they will put him to death; and three days after he has been put to death he will rise again.' ",
            32: "But they did not understand what he said and were afraid to ask him. ",
            33: "They came to Capernaum, and when he got into the house he asked them, 'What were you arguing about on the road?' ",
            34: "They said nothing, because on the road they had been arguing which of them was the greatest. ",
            35: "So he sat down, called the Twelve to him and said, 'If anyone wants to be first, he must make himself last of all and servant of all.' ",
            36: "He then took a little child whom he set among them and embraced, and he said to them, ",
            37: "'Anyone who welcomes a little child such as this in my name, welcomes me; and anyone who welcomes me, welcomes not me but the one who sent me.' ",
            38: "John said to him, 'Master, we saw someone who is not one of us driving out devils in your name, and because he was not one of us we tried to stop him.' ",
            39: "But Jesus said, 'You must not stop him; no one who works a miracle in my name could soon afterwards speak evil of me. ",
            40: "Anyone who is not against us is for us. ",
            41: "'If anyone gives you a cup of water to drink because you belong to Christ, then in truth I tell you, he will most certainly not lose his reward. ",
            42: "'But anyone who is the downfall of one of these little ones who have faith, would be better thrown into the sea with a great millstone hung round his neck. ",
            43: "And if your hand should be your downfall, cut it off; it is better for you to enter into life crippled, than to have two hands and go to hell, into the fire that can never be put out. ",
            44: "",
            45: "And if your foot should be your downfall, cut it off; it is better for you enter into life lame, than to have two feet and be thrown into hell. ",
            46: "",
            47: "And if your eye should be your downfall, tear it out; it is better for you to enter into the kingdom of God with one eye, than to have two eyes and be thrown into hell ",
            48: "where their worm will never die nor their fire be put out. ",
            49: "For everyone will be salted with fire. ",
            50: "Salt is a good thing, but if salt has become insipid, how can you make it salty again? Have salt in yourselves and be at peace with one another.'"
        }

        chapter_10 = {
            1: "After leaving there, he came into the territory of Judaea and Transjordan. And again crowds gathered round him, and again he taught them, as his custom was. ",
            2: "Some Pharisees approached him and asked, 'Is it lawful for a man to divorce his wife?' They were putting him to the test. ",
            3: "He answered them, 'What did Moses command you?' ",
            4: "They replied, 'Moses allowed us to draw up a writ of dismissal in cases of divorce.' ",
            5: "Then Jesus said to them, 'It was because you were so hard hearted that he wrote this commandment for you. ",
            6: "But from the beginning of creation he made them male and female. ",
            7: "This is why a man leaves his father and mother, ",
            8: "and the two become one flesh. They are no longer two, therefore, but one flesh. ",
            9: "So then, what God has united, human beings must not divide.' ",
            10: "Back in the house the disciples questioned him again about this, ",
            11: "and he said to them, 'Whoever divorces his wife and marries another is guilty of adultery against her. ",
            12: "And if a woman divorces her husband and marries another she is guilty of adultery too.' ",
            13: "People were bringing little children to him, for him to touch them. The disciples scolded them, ",
            14: "but when Jesus saw this he was indignant and said to them, 'Let the little children come to me; do not stop them; for it is to such as these that the kingdom of God belongs. ",
            15: "In truth I tell you, anyone who does not welcome the kingdom of God like a little child will never enter it.' ",
            16: "Then he embraced them, laid his hands on them and gave them his blessing. ",
            17: "He was setting out on a journey when a man ran up, knelt before him and put this question to him, 'Good master, what must I do to inherit eternal life?' ",
            18: "Jesus said to him, 'Why do you call me good? No one is good but God alone. ",
            19: "You know the commandments: You shall not kill; You shall not commit adultery; You shall not steal; You shall not give false witness; You shall not defraud; Honour your father and mother.' ",
            20: "And he said to him, 'Master, I have kept all these since my earliest days.' ",
            21: "Jesus looked steadily at him and he was filled with love for him, and he said, 'You need to do one thing more. Go and sell what you own and give the money to the poor, and you will have treasure in heaven; then come, follow me.' ",
            22: "But his face fell at these words and he went away sad, for he was a man of great wealth. ",
            23: "Jesus looked round and said to his disciples, 'How hard it is for those who have riches to enter the kingdom of God!' ",
            24: "The disciples were astounded by these words, but Jesus insisted, 'My children,' he said to them, 'how hard it is to enter the kingdom of God! ",
            25: "It is easier for a camel to pass through the eye of a needle than for someone rich to enter the kingdom of God.' ",
            26: "They were more astonished than ever, saying to one another, 'In that case, who can be saved?' ",
            27: "Jesus gazed at them and said, 'By human resources it is impossible, but not for God: because for God everything is possible.' ",
            28: "Peter took this up. 'Look,' he said to him, 'we have left everything and followed you.' ",
            29: "Jesus said, 'In truth I tell you, there is no one who has left house, brothers, sisters, mother, father, children or land for my sake and for the sake of the gospel ",
            30: "who will not receive a hundred times as much, houses, brothers, sisters, mothers, children and land -- and persecutions too -- now in this present time and, in the world to come, eternal life. ",
            31: "Many who are first will be last, and the last, first.' ",
            32: "They were on the road, going up to Jerusalem; Jesus was walking on ahead of them; they were in a daze, and those who followed were apprehensive. Once more taking the Twelve aside he began to tell them what was going to happen to him, ",
            33: "'Now we are going up to Jerusalem, and the Son of man is about to be handed over to the chief priests and the scribes. They will condemn him to death and will hand him over to the gentiles, ",
            34: "who will mock him and spit at him and scourge him and put him to death; and after three days he will rise again.' ",
            35: "James and John, the sons of Zebedee, approached him. 'Master,' they said to him, 'We want you to do us a favour.' ",
            36: "He said to them, 'What is it you want me to do for you?' ",
            37: "They said to him, 'Allow us to sit one at your right hand and the other at your left in your glory.' ",
            38: "But Jesus said to them, 'You do not know what you are asking. Can you drink the cup that I shall drink, or be baptised with the baptism with which I shall be baptised?' ",
            39: "They replied, 'We can.' Jesus said to them, 'The cup that I shall drink you shall drink, and with the baptism with which I shall be baptised you shall be baptised, ",
            40: "but as for seats at my right hand or my left, these are not mine to grant; they belong to those to whom they have been allotted.' ",
            41: "When the other ten heard this they began to feel indignant with James and John, ",
            42: "so Jesus called them to him and said to them, 'You know that among the gentiles those they call their rulers lord it over them, and their great men make their authority felt. ",
            43: "Among you this is not to happen. No; anyone who wants to become great among you must be your servant, ",
            44: "and anyone who wants to be first among you must be slave to all. ",
            45: "For the Son of man himself came not to be served but to serve, and to give his life as a ransom for many.' ",
            46: "They reached Jericho; and as he left Jericho with his disciples and a great crowd, Bartimaeus -- that is, the son of Timaeus -- a blind beggar, was sitting at the side of the road. ",
            47: "When he heard that it was Jesus of Nazareth, he began to shout and cry out, 'Son of David, Jesus, have pity on me.' ",
            48: "And many of them scolded him and told him to keep quiet, but he only shouted all the louder, 'Son of David, have pity on me.' ",
            49: "Jesus stopped and said, 'Call him here.' So they called the blind man over. 'Courage,' they said, 'get up; he is calling you.' ",
            50: "So throwing off his cloak, he jumped up and went to Jesus. ",
            51: "Then Jesus spoke, 'What do you want me to do for you?' The blind man said to him, 'Rabbuni, let me see again.' ",
            52: "Jesus said to him, 'Go; your faith has saved you.' And at once his sight returned and he followed him along the road."
        }

        chapter_11 = {
            1: "When they were approaching Jerusalem, at Bethphage and Bethany, close by the Mount of Olives, he sent two of his disciples ",
            2: "and said to them, 'Go to the village facing you, and as you enter it you will at once find a tethered colt that no one has yet ridden. Untie it and bring it here. ",
            3: """If anyone says to you, "What are you doing?" say, "The Master needs it and will send it back here at once." ' """,
            4: "They went off and found a colt tethered near a door in the open street. As they untied it, ",
            5: "some men standing there said, 'What are you doing, untying that colt?' ",
            6: "They gave the answer Jesus had told them, and the men let them go. ",
            7: "Then they took the colt to Jesus and threw their cloaks on its back, and he mounted it. ",
            8: "Many people spread their cloaks on the road, and others greenery which they had cut in the fields. ",
            9: "And those who went in front and those who followed were all shouting, 'Hosanna! Blessed is he who is coming in the name of the Lord! ",
            10: "Blessed is the coming kingdom of David our father! Hosanna in the highest heavens!' ",
            11: "He entered Jerusalem and went into the Temple; and when he had surveyed it all, as it was late by now, he went out to Bethany with the Twelve. ",
            12: "Next day as they were leaving Bethany, he felt hungry. ",
            13: "Seeing a fig tree in leaf some distance away, he went to see if he could find any fruit on it, but when he came up to it he found nothing but leaves; for it was not the season for figs. ",
            14: "And he addressed the fig tree, 'May no one ever eat fruit from you again.' And his disciples heard him say this. ",
            15: "So they reached Jerusalem and he went into the Temple and began driving out the men selling and buying there; he upset the tables of the money changers and the seats of the dove sellers. ",
            16: "Nor would he allow anyone to carry anything through the Temple. ",
            17: "And he taught them and said, 'Does not scripture say: My house will be called a house of prayer for all peoples? But you have turned it into a bandits' den.' ",
            18: "This came to the ears of the chief priests and the scribes, and they tried to find some way of doing away with him; they were afraid of him because the people were carried away by his teaching. ",
            19: "And when evening came he went out of the city. ",
            20: "Next morning, as they passed by, they saw the fig tree withered to the roots. ",
            21: "Peter remembered. 'Look, Rabbi,' he said to Jesus, 'the fig tree that you cursed has withered away.' ",
            22: "Jesus answered, 'Have faith in God. ",
            23: 'In truth I tell you, if anyone says to this mountain, "Be pulled up and thrown into the sea," with no doubt in his heart, but believing that what he says will happen, it will be done for him. ',
            24: "I tell you, therefore, everything you ask and pray for, believe that you have it already, and it will be yours.",
            25: "And when you stand in prayer, forgive whatever you have against anybody, soyour Father in heaven may forgive your failings too.' ",
            26: "",
            27: "They came to Jerusalem again, and as Jesus was walking in the Temple, the chief priests and the scribes and the elders came to him, ",
            28: "and they said to him, 'What authority have you for acting like this? Or who gave you authority to act like this?' ",
            29: "Jesus said to them, 'And I will ask you a question, just one; answer me and I will tell you my authority for acting like this. ",
            30: "John's baptism, what was its origin, heavenly or human? Answer me that.' ",
            31: '''And they argued this way among themselves, 'If we say heavenly, he will say, "Then why did you refuse to believe him?" ''',
            32: "But dare we say human?' -- they had the people to fear, for everyone held that John had been a real prophet. ",
            33: "So their reply to Jesus was, 'We do not know.' And Jesus said to them, 'Nor will I tell you my authority for acting like this.'",
        }

        chapter_12 = {
            1: "He went on to speak to them in parables, 'A man planted a vineyard; he fenced it round, dug out a trough for the winepress and built a tower; then he leased it to tenants and went abroad. ",
            2: "When the time came, he sent a servant to the tenants to collect from them his share of the produce of the vineyard. ",
            3: "But they seized the man, thrashed him and sent him away empty handed. ",
            4: "Next he sent another servant to them; him they beat about the head and treated shamefully. ",
            5: "And he sent another and him they killed; then a number of others, and they thrashed some and killed the rest. ",
            6: 'He had still someone left: his beloved son. He sent him to them last of all, thinking, "They will respect my son." ',
            7: 'But those tenants said to each other, "This is the heir. Come on, let us kill him, and the inheritance will be ours." ',
            8: "So they seized him and killed him and threw him out of the vineyard. ",
            9: "Now what will the owner of the vineyard do? He will come and make an end of the tenants and give the vineyard to others. ",
            10: "Have you not read this text of scripture: The stone which the builders rejected has become the cornerstone; ",
            11: "this is the Lord's doing, and we marvel at it ?' ",
            12: "And they would have liked to arrest him, because they realised that the parable was aimed at them, but they were afraid of the crowds. So they left him alone and went away. ",
            13: "Next they sent to him some Pharisees and some Herodians to catch him out in what he said. ",
            14: "These came and said to him, 'Master, we know that you are an honest man, that you are not afraid of anyone, because human rank means nothing to you, and that you teach the way of God in all honesty. Is it permissible to pay taxes to Caesar or not? Should we pay or not?' ",
            15: "Recognising their hypocrisy he said to them, 'Why are you putting me to the test? Hand me a denarius and let me see it.' ",
            16: "They handed him one and he said to them, 'Whose portrait is this? Whose title?' They said to him, 'Caesar's.' ",
            17: "Jesus said to them, 'Pay Caesar what belongs to Caesar -- and God what belongs to God.' And they were amazed at him. ",
            18: "Then some Sadducees -- who deny that there is a resurrection -- came to him and they put this question to him, ",
            19: "'Master, Moses prescribed for us that if a man's brother dies leaving a wife but no child, the man must marry the widow to raise up children for his brother. ",
            20: "Now there were seven brothers; the first married a wife and then died leaving no children. ",
            21: "The second married the widow, and he too died leaving no children; with the third it was the same, ",
            22: "and none of the seven left any children. Last of all the woman herself died. ",
            23: "Now at the resurrection, when they rise again, whose wife will she be, since she had been married to all seven?' ",
            24: "Jesus said to them, 'Surely the reason why you are wrong is that you understand neither the scriptures nor the power of God. ",
            25: "For when they rise from the dead, men and women do not marry; no, they are like the angels in heaven. ",
            26: "Now about the dead rising again, have you never read in the Book of Moses, in the passage about the bush, how God spoke to him and said: I am the God of Abraham, the God of Isaac and the God of Jacob? ",
            27: "He is God, not of the dead, but of the living. You are very much mistaken.' ",
            28: "One of the scribes who had listened to them debating appreciated that Jesus had given a good answer and put a further question to him, 'Which is the first of all the commandments?' ",
            29: "Jesus replied, 'This is the first: Listen, Israel, the Lord our God is the one, only Lord, ",
            30: "and you must love the Lord your God with all your heart, with all your soul, with all your mind and with all your strength. ",
            31: "The second is this: You must love your neighbour as yourself. There is no commandment greater than these.' ",
            32: "The scribe said to him, 'Well spoken, Master; what you have said is true, that he is one and there is no other. ",
            33: "To love him with all your heart, with all your understanding and strength, and to love your neighbour as yourself, this is far more important than any burnt offering or sacrifice.' ",
            34: "Jesus, seeing how wisely he had spoken, said, 'You are not far from the kingdom of God.' And after that no one dared to question him any more. ",
            35: "While teaching in the Temple, Jesus said, 'How can the scribes maintain that the Christ is the son of David? ",
            36: "David himself, moved by the Holy Spirit, said: The Lord declared to my Lord, take your seat at my right hand till I have made your enemies your footstool. ",
            37: "David himself calls him Lord; in what way then can he be his son?' And the great crowd listened to him with delight. ",
            38: "In his teaching he said, 'Beware of the scribes who like to walk about in long robes, to be greeted respectfully in the market squares, ",
            39: "to take the front seats in the synagogues and the places of honour at banquets; ",
            40: "these are the men who devour the property of widows and for show offer long prayers. The more severe will be the sentence they receive.' ",
            41: "He sat down opposite the treasury and watched the people putting money into the treasury, and many of the rich put in a great deal. ",
            42: "A poor widow came and put in two small coins, the equivalent of a penny. ",
            43: "Then he called his disciples and said to them, 'In truth I tell you, this poor widow has put more in than all who have contributed to the treasury; ",
            44: "for they have all put in money they could spare, but she in her poverty has put in everything she possessed, all she had to live on.' ",
        }

        chapter_13 = {
            1: "As he was leaving the Temple one of his disciples said to him, 'Master, look at the size of those stones! Look at the size of those buildings!' ",
            2: "And Jesus said to him, 'You see these great buildings? Not a single stone will be left on another; everything will be pulled down.' ",
            3: "And while he was sitting on the Mount of Olives, facing the Temple, Peter, James, John and Andrew questioned him when they were by themselves, ",
            4: "'Tell us, when is this going to happen, and what sign will there be that it is all about to take place?' ",
            5: "Then Jesus began to tell them, 'Take care that no one deceives you. ",
            6: 'Many will come using my name and saying, "I am he," and they will deceive many. ',
            7: "When you hear of wars and rumours of wars, do not be alarmed; this is something that must happen, but the end will not be yet. ",
            8: "For nation will fight against nation, and kingdom against kingdom. There will be earthquakes in various places; there will be famines. This is the beginning of the birth-pangs. ",
            9: "'Be on your guard: you will be handed over to sanhedrins; you will be beaten in synagogues; and you will be brought before governors and kings for my sake, as evidence to them, ",
            10: "since the gospel must first be proclaimed to all nations. ",
            11: "'And when you are taken to be handed over, do not worry beforehand about what to say; no, say whatever is given to you when the time comes, because it is not you who will be speaking; it is the Holy Spirit. ",
            12: "Brother will betray brother to death, and a father his child; children will come forward against their parents and have them put to death. ",
            13: "You will be universally hated on account of my name; but anyone who stands firm to the end will be saved. ",
            14: "'When you see the appalling abomination set up where it ought not to be (let the reader understand), then those in Judaea must escape to the mountains; ",
            15: "if a man is on the housetop, he must not come down or go inside to collect anything from his house; ",
            16: "if a man is in the fields, he must not turn back to fetch his cloak. ",
            17: "Alas for those with child, or with babies at the breast, when those days come! ",
            18: "Pray that this may not be in winter. ",
            19: "For in those days there will be great distress, unparalleled since God created the world, and such as will never be again. ",
            20: "And if the Lord had not shortened that time, no human being would have survived; but he did shorten the time, for the sake of the elect whom he chose. ",
            21: """'And if anyone says to you then, "Look, here is the Christ" or, "Look, he is there," do not believe it; """,
            22: "for false Christs and false prophets will arise and produce signs and portents to deceive the elect, if that were possible. ",
            23: "You, therefore, must be on your guard. I have given you full warning. ",
            24: "'But in those days, after that time of distress, the sun will be darkened, the moon will not give its light, ",
            25: "the stars will come falling out of the sky and the powers in the heavens will be shaken. ",
            26: "And then they will see the Son of man coming in the clouds with great power and glory. ",
            27: "And then he will send the angels to gather his elect from the four winds, from the ends of the world to the ends of the sky. ",
            28: "'Take the fig tree as a parable: as soon as its twigs grow supple and its leaves come out, you know that summer is near. ",
            29: "So with you when you see these things happening: know that he is near, right at the gates. ",
            30: "In truth I tell you, before this generation has passed away all these things will have taken place. ",
            31: "Sky and earth will pass away, but my words will not pass away. ",
            32: "'But as for that day or hour, nobody knows it, neither the angels in heaven, nor the Son; no one but the Father. ",
            33: "'Be on your guard, stay awake, because you never know when the time will come. ",
            34: "It is like a man travelling abroad: he has gone from his home, and left his servants in charge, each with his own work to do; and he has told the doorkeeper to stay awake. ",
            35: "So stay awake, because you do not know when the master of the house is coming, evening, midnight, cockcrow or dawn; ",
            36: "if he comes unexpectedly, he must not find you asleep. ",
            37: "And what I am saying to you I say to all: Stay awake!'"
        }

        chapter_14 = {
            1: "It was two days before the Passover and the feast of Unleavened Bread, and the chief priests and the scribes were looking for a way to arrest Jesus by some trick and have him put to death. ",
            2: "For they said, 'It must not be during the festivities, or there will be a disturbance among the people.' ",
            3: "He was at Bethany in the house of Simon, a man who had suffered from a virulent skin-disease; he was at table when a woman came in with an alabaster jar of very costly ointment, pure nard. She broke the jar and poured the ointment on his head. ",
            4: "Some who were there said to one another indignantly, 'Why this waste of ointment? ",
            5: "Ointment like this could have been sold for over three hundred denarii and the money given to the poor'; and they were angry with her. ",
            6: "But Jesus said, 'Leave her alone. Why are you upsetting her? What she has done for me is a good work. ",
            7: "You have the poor with you always, and you can be kind to them whenever you wish, but you will not always have me. ",
            8: "She has done what she could: she has anointed my body beforehand for its burial. ",
            9: "In truth I tell you, wherever throughout all the world the gospel is proclaimed, what she has done will be told as well, in remembrance of her.' ",
            10: "Judas Iscariot, one of the Twelve, approached the chief priests with an offer to hand Jesus over to them. ",
            11: "They were delighted to hear it, and promised to give him money; and he began to look for a way of betraying him when the opportunity should occur. ",
            12: "On the first day of Unleavened Bread, when the Passover lamb was sacrificed, his disciples said to him, 'Where do you want us to go and make the preparations for you to eat the Passover?' ",
            13: "So he sent two of his disciples, saying to them, 'Go into the city and you will meet a man carrying a pitcher of water. Follow him, ",
            14: 'and say to the owner of the house which he enters, "The Master says: Where is the room for me to eat the Passover with my disciples?" ',
            15: "He will show you a large upper room furnished with couches, all prepared. Make the preparations for us there.' ",
            16: "The disciples set out and went to the city and found everything as he had told them, and prepared the Passover. ",
            17: "When evening came he arrived with the Twelve. ",
            18: "And while they were at table eating, Jesus said, 'In truth I tell you, one of you is about to betray me, one of you eating with me.' ",
            19: "They were distressed and said to him, one after another, 'Not me, surely?' ",
            20: "He said to them, 'It is one of the Twelve, one who is dipping into the same dish with me. ",
            21: "Yes, the Son of man is going to his fate, as the scriptures say he will, but alas for that man by whom the Son of man is betrayed! Better for that man if he had never been born.' ",
            22: "And as they were eating he took bread, and when he had said the blessing he broke it and gave it to them. 'Take it,' he said, 'this is my body.' ",
            23: "Then he took a cup, and when he had given thanks he handed it to them, and all drank from it, ",
            24: "and he said to them, 'This is my blood, the blood of the covenant, poured out for many. ",
            25: "In truth I tell you, I shall never drink wine any more until the day I drink the new wine in the kingdom of God.' ",
            26: "After the psalms had been sung they left for the Mount of Olives. ",
            27: "And Jesus said to them, 'You will all fall away, for the scripture says: I shall strike the shepherd and the sheep will be scattered; ",
            28: "however, after my resurrection I shall go before you into Galilee.' ",
            29: "Peter said, 'Even if all fall away, I will not.' ",
            30: "And Jesus said to him, 'In truth I tell you, this day, this very night, before the cock crows twice, you will have disowned me three times.' ",
            31: "But he repeated still more earnestly, 'If I have to die with you, I will never disown you.' And they all said the same. ",
            32: "They came to a plot of land called Gethsemane, and he said to his disciples, 'Stay here while I pray.' ",
            33: "Then he took Peter and James and John with him. ",
            34: "And he began to feel terror and anguish. And he said to them, 'My soul is sorrowful to the point of death. Wait here, and stay awake.' ",
            35: "And going on a little further he threw himself on the ground and prayed that, if it were possible, this hour might pass him by. ",
            36: "'Abba, Father!' he said, 'For you everything is possible. Take this cup away from me. But let it be as you, not I, would have it.' ",
            37: "He came back and found them sleeping, and he said to Peter, 'Simon, are you asleep? Had you not the strength to stay awake one hour? ",
            38: "Stay awake and pray not to be put to the test. The spirit is willing enough, but human nature is weak.' ",
            39: "Again he went away and prayed, saying the same words. ",
            40: "And once more he came back and found them sleeping, their eyes were so heavy; and they could find no answer for him. ",
            41: "He came back a third time and said to them, 'You can sleep on now and have your rest. It is all over. The hour has come. Now the Son of man is to be betrayed into the hands of sinners. ",
            42: "Get up! Let us go! My betrayer is not far away.' ",
            43: "And at once, while he was still speaking, Judas, one of the Twelve, came up and with him a number of men armed with swords and clubs, sent by the chief priests and the scribes and the elders. ",
            44: "Now the traitor had arranged a signal with them saying, 'The one I kiss, he is the man. Arrest him, and see he is well guarded when you lead him away.' ",
            45: "So when the traitor came, he went up to Jesus at once and said, 'Rabbi!' and kissed him. ",
            46: "The others seized him and arrested him. ",
            47: "Then one of the bystanders drew his sword and struck out at the high priest's servant and cut off his ear. ",
            48: "Then Jesus spoke. 'Am I a bandit,' he said, 'that you had to set out to capture me with swords and clubs? ",
            49: "I was among you teaching in the Temple day after day and you never laid a hand on me. But this is to fulfil the scriptures.'",
            50: "And they all deserted him and ran away. ",
            51: "A young man followed with nothing on but a linen cloth. They caught hold of him, ",
            52: "but he left the cloth in their hands and ran away naked. ",
            53: "They led Jesus off to the high priest; and all the chief priests and the elders and the scribes assembled there. ",
            54: "Peter had followed him at a distance, right into the high priest's palace, and was sitting with the attendants warming himself at the fire. ",
            55: "The chief priests and the whole Sanhedrin were looking for evidence against Jesus in order to have him executed. But they could not find any. ",
            56: "Several, indeed, brought false witness against him, but their evidence was conflicting. ",
            57: "Some stood up and submitted this false evidence against him, ",
            58: """'We heard him say, "I am going to destroy this Temple made by human hands, and in three days build another, not made by human hands." ' """,
            59: "But even on this point their evidence was conflicting. ",
            60: "The high priest then rose before the whole assembly and put this question to Jesus, 'Have you no answer to that? What is this evidence these men are bringing against you?' ",
            61: "But he was silent and made no answer at all. The high priest put a second question to him saying, 'Are you the Christ, the Son of the Blessed One?' ",
            62: "'I am,' said Jesus, 'and you will see the Son of man seated at the right hand of the Power and coming with the clouds of heaven.' ",
            63: "The high priest tore his robes and said, 'What need of witnesses have we now? ",
            64: "You heard the blasphemy. What is your finding?' Their verdict was unanimous: he deserved to die. ",
            65: "Some of them started spitting at his face, hitting him and saying, 'Play the prophet!' And the attendants struck him too. ",
            66: "While Peter was down below in the courtyard, one of the high priest's servant-girls came up. ",
            67: "She saw Peter warming himself there, looked closely at him and said, 'You too were with Jesus, the man from Nazareth.' ",
            68: "But he denied it. 'I do not know, I do not understand what you are talking about,' he said. And he went out into the forecourt, and a cock crowed. ",
            69: "The servant-girl saw him and again started telling the bystanders, 'This man is one of them.' ",
            70: "But again he denied it. A little later the bystanders themselves said to Peter, 'You are certainly one of them! Why, you are a Galilean.' ",
            71: "But he started cursing and swearing, 'I do not know the man you speak of.' ",
            72: "And at once the cock crowed for the second time, and Peter recalled what Jesus had said to him, 'Before the cock crows twice, you will have disowned me three times.' And he burst into tears.",
        }

        chapter_15 = {
            1: "First thing in the morning, the chief priests, together with the elders and scribes and the rest of the Sanhedrin, had their plan ready. They had Jesus bound and took him away and handed him over to Pilate. ",
            2: "Pilate put to him this question, 'Are you the king of the Jews?' He replied, 'It is you who say it.' ",
            3: "And the chief priests brought many accusations against him. ",
            4: "Pilate questioned him again, 'Have you no reply at all? See how many accusations they are bringing against you!' ",
            5: "But, to Pilate's surprise, Jesus made no further reply. ",
            6: "At festival time Pilate used to release a prisoner for them, any one they asked for. ",
            7: "Now a man called Barabbas was then in prison with the rebels who had committed murder during the uprising. ",
            8: "When the crowd went up and began to ask Pilate the customary favour, ",
            9: "Pilate answered them, 'Do you want me to release for you the king of the Jews?' ",
            10: "For he realised it was out of jealousy that the chief priests had handed Jesus over. ",
            11: "The chief priests, however, had incited the crowd to demand that he should release Barabbas for them instead. ",
            12: "Then Pilate spoke again, 'But in that case, what am I to do with the man you call king of the Jews?' ",
            13: "They shouted back, 'Crucify him!' ",
            14: "Pilate asked them, 'What harm has he done?' But they shouted all the louder, 'Crucify him!' ",
            15: "So Pilate, anxious to placate the crowd, released Barabbas for them and, after having Jesus scourged, he handed him over to be crucified. ",
            16: "The soldiers led him away to the inner part of the palace, that is, the Praetorium, and called the whole cohort together. ",
            17: "They dressed him up in purple, twisted some thorns into a crown and put it on him. ",
            18: "And they began saluting him, 'Hail, king of the Jews!' ",
            19: "They struck his head with a reed and spat on him; and they went down on their knees to do him homage. ",
            20: "And when they had finished making fun of him, they took off the purple and dressed him in his own clothes. They led him out to crucify him. ",
            21: "They enlisted a passer-by, Simon of Cyrene, father of Alexander and Rufus, who was coming in from the country, to carry his cross. ",
            22: "They brought Jesus to the place called Golgotha, which means the place of the skull. ",
            23: "They offered him wine mixed with myrrh, but he refused it. ",
            24: "Then they crucified him, and shared out his clothing, casting lots to decide what each should get. ",
            25: "It was the third hour when they crucified him. ",
            26: "The inscription giving the charge against him read, 'The King of the Jews'. ",
            27: "And they crucified two bandits with him, one on his right and one on his left. ",
            28: "",
            29: "The passers-by jeered at him; they shook their heads and said, 'Aha! So you would destroy the Temple and rebuild it in three days! ",
            30: "Then save yourself; come down from the cross!' ",
            31: "The chief priests and the scribes mocked him among themselves in the same way with the words, 'He saved others, he cannot save himself. ",
            32: "Let the Christ, the king of Israel, come down from the cross now, for us to see it and believe.' Even those who were crucified with him taunted him. ",
            33: "When the sixth hour came there was darkness over the whole land until the ninth hour. ",
            34: "And at the ninth hour Jesus cried out in a loud voice, 'Eloi, eloi, lama sabachthani?' which means, 'My God, my God, why have you forsaken me?' ",
            35: "When some of those who stood by heard this, they said, 'Listen, he is calling on Elijah.' ",
            36: "Someone ran and soaked a sponge in vinegar and, putting it on a reed, gave it to him to drink saying, 'Wait! And see if Elijah will come to take him down.' ",
            37: "But Jesus gave a loud cry and breathed his last. ",
            38: "And the veil of the Sanctuary was torn in two from top to bottom. ",
            39: "The centurion, who was standing in front of him, had seen how he had died, and he said, 'In truth this man was Son of God.' ",
            40: "There were some women watching from a distance. Among them were Mary of Magdala, Mary who was the mother of James the younger and Joset, and Salome. ",
            41: "These used to follow him and look after him when he was in Galilee. And many other women were there who had come up to Jerusalem with him. ",
            42: "It was now evening, and since it was Preparation Day -- that is, the day before the Sabbath- ",
            43: "there came Joseph of Arimathaea, a prominent member of the Council, who himself lived in the hope of seeing the kingdom of God, and he boldly went to Pilate and asked for the body of Jesus. ",
            44: "Pilate, astonished that he should have died so soon, summoned the centurion and enquired if he had been dead for some time. ",
            45: "Having been assured of this by the centurion, he granted the corpse to Joseph ",
            46: "who bought a shroud, took Jesus down from the cross, wrapped him in the shroud and laid him in a tomb which had been hewn out of the rock. He then rolled a stone against the entrance to the tomb. ",
            47: "Mary of Magdala and Mary the mother of Joset took note of where he was laid."
        }

        chapter_16 = {
            1: "When the Sabbath was over, Mary of Magdala, Mary the mother of James, and Salome, bought spices with which to go and anoint him. ",
            2: "And very early in the morning on the first day of the week they went to the tomb when the sun had risen. ",
            3: "They had been saying to one another, 'Who will roll away the stone for us from the entrance to the tomb?' ",
            4: "But when they looked they saw that the stone -- which was very big -- had already been rolled back. ",
            5: "On entering the tomb they saw a young man in a white robe seated on the right-hand side, and they were struck with amazement. ",
            6: "But he said to them, 'There is no need to be so amazed. You are looking for Jesus of Nazareth, who was crucified: he has risen, he is not here. See, here is the place where they laid him. ",
            7: """But you must go and tell his disciples and Peter, "He is going ahead of you to Galilee; that is where you will see him, just as he told you." ' """,
            8: "And the women came out and ran away from the tomb because they were frightened out of their wits; and they said nothing to anyone, for they were afraid. ",
            9: "Having risen in the morning on the first day of the week, he appeared first to Mary of Magdala from whom he had cast out seven devils. ",
            10: "She then went to those who had been his companions, and who were mourning and in tears, and told them. ",
            11: "But they did not believe her when they heard her say that he was alive and that she had seen him. ",
            12: "After this, he showed himself under another form to two of them as they were on their way into the country. ",
            13: "These went back and told the others, who did not believe them either. ",
            14: "Lastly, he showed himself to the Eleven themselves while they were at table. He reproached them for their incredulity and obstinacy, because they had refused to believe those who had seen him after he had risen. ",
            15: "And he said to them, 'Go out to the whole world; proclaim the gospel to all creation. ",
            16: "Whoever believes and is baptised will be saved; whoever does not believe will be condemned. ",
            17: "These are the signs that will be associated with believers: in my name they will cast out devils; they will have the gift of tongues; ",
            18: "they will pick up snakes in their hands and be unharmed should they drink deadly poison; they will lay their hands on the sick, who will recover.' ",
            19: "And so the Lord Jesus, after he had spoken to them, was taken up into heaven; there at the right hand of God he took his place, ",
            20: "while they, going out, preached everywhere, the Lord working with them and confirming the word by the signs that accompanied it."
        }

        chapters = {
            1: chapter_1,
            2: chapter_2,
            3: chapter_3,
            4: chapter_4,
            5: chapter_5,
            6: chapter_6,
            7: chapter_7,
            8: chapter_8,
            9: chapter_9,
            10: chapter_10,
            11: chapter_11,
            12: chapter_12,
            13: chapter_13,
            14: chapter_14,
            15: chapter_15,
            16: chapter_16,
        }

        return get_text(chapters, chapter, verse_start, verse_end, 16)

    def luke(self, chapter: int, verse_start: int, verse_end: int = None) -> Union[str, list]:
        """
        :param chapter: The chapter number
        :type chapter: int

        :param verse_start: The first verse
        :type verse_start: int

        :param verse_end: Alternatively, an end verse can be passed or let it
        as None to return only one verse
        :type verse_end: int or None
        """
        chapter_1 = {
            1: "Seeing that many others have undertaken to draw up accounts of the events that have reached their fulfilment among us, ",
            2: "as these were handed down to us by those who from the outset were eyewitnesses and ministers of the word, ",
            3: "I in my turn, after carefully going over the whole story from the beginning, have decided to write an ordered account for you, Theophilus,",
            4: "so that your Excellency may learn how well founded the teaching is that you have received. ",
            5: "In the days of King Herod of Judaea there lived a priest called Zechariah who belonged to the Abijah section of the priesthood, and he had a wife, Elizabeth by name, who was a descendant of Aaron. ",
            6: "Both were upright in the sight of God and impeccably carried out all the commandments and observances of the Lord. ",
            7: "But they were childless: Elizabeth was barren and they were both advanced in years. ",
            8: "Now it happened that it was the turn of his section to serve, and he was exercising his priestly office before God ",
            9: "when it fell to him by lot, as the priestly custom was, to enter the Lord's sanctuary and burn incense there. ",
            10: "And at the hour of incense all the people were outside, praying. ",
            11: "Then there appeared to him the angel of the Lord, standing on the right of the altar of incense. ",
            12: "The sight disturbed Zechariah and he was overcome with fear. ",
            13: "But the angel said to him, 'Zechariah, do not be afraid, for your prayer has been heard. Your wife Elizabeth is to bear you a son and you shall name him John. ",
            14: "He will be your joy and delight and many will rejoice at his birth, ",
            15: "for he will be great in the sight of the Lord; he must drink no wine, no strong drink; even from his mother's womb he will be filled with the Holy Spirit, ",
            16: "and he will bring back many of the Israelites to the Lord their God. ",
            17: "With the spirit and power of Elijah, he will go before him to reconcile fathers to their children and the disobedient to the good sense of the upright, preparing for the Lord a people fit for him.' ",
            18: "Zechariah said to the angel, 'How can I know this? I am an old man and my wife is getting on in years.' ",
            19: "The angel replied, 'I am Gabriel, who stand in God's presence, and I have been sent to speak to you and bring you this good news. ",
            20: "Look! Since you did not believe my words, which will come true at their appointed time, you will be silenced and have no power of speech until this has happened.' ",
            21: "Meanwhile the people were waiting for Zechariah and were surprised that he stayed in the sanctuary so long. ",
            22: "When he came out he could not speak to them, and they realised that he had seen a vision in the sanctuary. But he could only make signs to them and remained dumb. ",
            23: "When his time of service came to an end he returned home. ",
            24: "Some time later his wife Elizabeth conceived and for five months she kept to herself, saying, ",
            25: "'The Lord has done this for me, now that it has pleased him to take away the humiliation I suffered in public.' ",
            26: "In the sixth month the angel Gabriel was sent by God to a town in Galilee called Nazareth, ",
            27: "to a virgin betrothed to a man named Joseph, of the House of David; and the virgin's name was Mary. ",
            28: "He went in and said to her, 'Rejoice, you who enjoy God's favour! The Lord is with you.' ",
            29: "She was deeply disturbed by these words and asked herself what this greeting could mean, ",
            30: "but the angel said to her, 'Mary, do not be afraid; you have won God's favour. ",
            31: "Look! You are to conceive in your womb and bear a son, and you must name him Jesus. ",
            32: "He will be great and will be called Son of the Most High. The Lord God will give him the throne of his ancestor David; ",
            33: "he will rule over the House of Jacob for ever and his reign will have no end.' ",
            34: "Mary said to the angel, 'But how can this come about, since I have no knowledge of man?' ",
            35: "The angel answered, 'The Holy Spirit will come upon you, and the power of the Most High will cover you with its shadow. And so the child will be holy and will be called Son of God. ",
            36: "And I tell you this too: your cousin Elizabeth also, in her old age, has conceived a son, and she whom people called barren is now in her sixth month, ",
            37: "for nothing is impossible to God.' ",
            38: "Mary said, 'You see before you the Lord's servant, let it happen to me as you have said.' And the angel left her. ",
            39: "Mary set out at that time and went as quickly as she could into the hill country to a town in Judah. ",
            40: "She went into Zechariah's house and greeted Elizabeth. ",
            41: "Now it happened that as soon as Elizabeth heard Mary's greeting, the child leapt in her womb and Elizabeth was filled with the Holy Spirit. ",
            42: "She gave a loud cry and said, 'Of all women you are the most blessed, and blessed is the fruit of your womb. ",
            43: "Why should I be honoured with a visit from the mother of my Lord? ",
            44: "Look, the moment your greeting reached my ears, the child in my womb leapt for joy. ",
            45: "Yes, blessed is she who believed that the promise made her by the Lord would be fulfilled.' ",
            46: "And Mary said: My soul proclaims the greatness of the Lord ",
            47: "and my spirit rejoices in God my Saviour; ",
            48: "because he has looked upon the humiliation of his servant. Yes, from now onwards all generations will call me blessed, ",
            49: "for the Almighty has done great things for me. Holy is his name, ",
            50: "and his faithful love extends age after age to those who fear him. ",
            51: "He has used the power of his arm, he has routed the arrogant of heart. ",
            52: "He has pulled down princes from their thrones and raised high the lowly. ",
            53: "He has filled the starving with good things, sent the rich away empty. ",
            54: "He has come to the help of Israel his servant, mindful of his faithful love ",
            55: "-according to the promise he made to our ancestors -- of his mercy to Abraham and to his descendants for ever. ",
            56: "Mary stayed with her some three months and then went home. ",
            57: "The time came for Elizabeth to have her child, and she gave birth to a son; ",
            58: "and when her neighbours and relations heard that the Lord had lavished on her his faithful love, they shared her joy. ",
            59: "Now it happened that on the eighth day they came to circumcise the child; they were going to call him Zechariah after his father, ",
            60: "but his mother spoke up. 'No,' she said, 'he is to be called John.' ",
            61: "They said to her, 'But no one in your family has that name,' ",
            62: "and made signs to his father to find out what he wanted him called. ",
            63: "The father asked for a writing-tablet and wrote, 'His name is John.' And they were all astonished. ",
            64: "At that instant his power of speech returned and he spoke and praised God. ",
            65: "All their neighbours were filled with awe and the whole affair was talked about throughout the hill country of Judaea. ",
            66: "All those who heard of it treasured it in their hearts. 'What will this child turn out to be?' they wondered. And indeed the hand of the Lord was with him. ",
            67: "His father Zechariah was filled with the Holy Spirit and spoke this prophecy: ",
            68: "Blessed be the Lord, the God of Israel, for he has visited his people, he has set them free, ",
            69: "and he has established for us a saving power in the House of his servant David, ",
            70: "just as he proclaimed, by the mouth of his holy prophets from ancient times, ",
            71: "that he would save us from our enemies and from the hands of all those who hate us, ",
            72: "and show faithful love to our ancestors, and so keep in mind his holy covenant. ",
            73: "This was the oath he swore to our father Abraham, ",
            74: "that he would grant us, free from fear, to be delivered from the hands of our enemies, ",
            75: "to serve him in holiness and uprightness in his presence, all our days. ",
            76: "And you, little child, you shall be called Prophet of the Most High, for you will go before the Lord to prepare a way for him, ",
            77: "to give his people knowledge of salvation through the forgiveness of their sins, ",
            78: "because of the faithful love of our God in which the rising Sun has come from on high to visit us, ",
            79: "to give light to those who live in darkness and the shadow dark as death, and to guide our feet into the way of peace. ",
            80: "Meanwhile the child grew up and his spirit grew strong. And he lived in the desert until the day he appeared openly to Israel.",
        }

        chapter_2 = {
            1: "Now it happened that at this time Caesar Augustus issued a decree that a census should be made of the whole inhabited world. ",
            2: "This census -- the first -- took place while Quirinius was governor of Syria, ",
            3: "and everyone went to be registered, each to his own town. ",
            4: "So Joseph set out from the town of Nazareth in Galilee for Judaea, to David's town called Bethlehem, since he was of David's House and line, ",
            5: "in order to be registered together with Mary, his betrothed, who was with child. ",
            6: "Now it happened that, while they were there, the time came for her to have her child, ",
            7: "and she gave birth to a son, her first-born. She wrapped him in swaddling clothes and laid him in a manger because there was no room for them in the living-space. ",
            8: "In the countryside close by there were shepherds out in the fields keeping guard over their sheep during the watches of the night. ",
            9: "An angel of the Lord stood over them and the glory of the Lord shone round them. They were terrified, ",
            10: "but the angel said, 'Do not be afraid. Look, I bring you news of great joy, a joy to be shared by the whole people. ",
            11: "Today in the town of David a Saviour has been born to you; he is Christ the Lord. ",
            12: "And here is a sign for you: you will find a baby wrapped in swaddling clothes and lying in a manger.' ",
            13: "And all at once with the angel there was a great throng of the hosts of heaven, praising God with the words: ",
            14: "Glory to God in the highest heaven, and on earth peace for those he favours. ",
            15: "Now it happened that when the angels had gone from them into heaven, the shepherds said to one another, 'Let us go to Bethlehem and see this event which the Lord has made known to us.' ",
            16: "So they hurried away and found Mary and Joseph, and the baby lying in the manger. ",
            17: "When they saw the child they repeated what they had been told about him, ",
            18: "and everyone who heard it was astonished at what the shepherds said to them. ",
            19: "As for Mary, she treasured all these things and pondered them in her heart. ",
            20: "And the shepherds went back glorifying and praising God for all they had heard and seen, just as they had been told. ",
            21: "When the eighth day came and the child was to be circumcised, they gave him the name Jesus, the name the angel had given him before his conception. ",
            22: "And when the day came for them to be purified in keeping with the Law of Moses, they took him up to Jerusalem to present him to the Lord- ",
            23: "observing what is written in the Law of the Lord: Every first-born male must be consecrated to the Lord- ",
            24: "and also to offer in sacrifice, in accordance with what is prescribed in the Law of the Lord, a pair of turtledoves or two young pigeons. ",
            25: "Now in Jerusalem there was a man named Simeon. He was an upright and devout man; he looked forward to the restoration of Israel and the Holy Spirit rested on him. ",
            26: "It had been revealed to him by the Holy Spirit that he would not see death until he had set eyes on the Christ of the Lord. ",
            27: "Prompted by the Spirit he came to the Temple; and when the parents brought in the child Jesus to do for him what the Law required, ",
            28: "he took him into his arms and blessed God; and he said: ",
            29: "Now, Master, you are letting your servant go in peace as you promised; ",
            30: "for my eyes have seen the salvation ",
            31: "which you have made ready in the sight of the nations; ",
            32: "a light of revelation for the gentiles and glory for your people Israel. ",
            33: "As the child's father and mother were wondering at the things that were being said about him, ",
            34: "Simeon blessed them and said to Mary his mother, 'Look, he is destined for the fall and for the rise of many in Israel, destined to be a sign that is opposed- ",
            35: "and a sword will pierce your soul too -- so that the secret thoughts of many may be laid bare.' ",
            36: "There was a prophetess, too, Anna the daughter of Phanuel, of the tribe of Asher. She was well on in years. Her days of girlhood over, she had been married for seven years ",
            37: "before becoming a widow. She was now eighty-four years old and never left the Temple, serving God night and day with fasting and prayer. ",
            38: "She came up just at that moment and began to praise God; and she spoke of the child to all who looked forward to the deliverance of Jerusalem. ",
            39: "When they had done everything the Law of the Lord required, they went back to Galilee, to their own town of Nazareth. ",
            40: "And as the child grew to maturity, he was filled with wisdom; and God's favour was with him. ",
            41: "Every year his parents used to go to Jerusalem for the feast of the Passover. ",
            42: "When he was twelve years old, they went up for the feast as usual. ",
            43: "When the days of the feast were over and they set off home, the boy Jesus stayed behind in Jerusalem without his parents knowing it. ",
            44: "They assumed he was somewhere in the party, and it was only after a day's journey that they went to look for him among their relations and acquaintances. ",
            45: "When they failed to find him they went back to Jerusalem looking for him everywhere. ",
            46: "It happened that, three days later, they found him in the Temple, sitting among the teachers, listening to them, and asking them questions; ",
            47: "and all those who heard him were astounded at his intelligence and his replies. ",
            48: "They were overcome when they saw him, and his mother said to him, 'My child, why have you done this to us? See how worried your father and I have been, looking for you.' ",
            49: "He replied, 'Why were you looking for me? Did you not know that I must be in my Father's house?' ",
            50: "But they did not understand what he meant. ",
            51: "He went down with them then and came to Nazareth and lived under their authority. His mother stored up all these things in her heart. ",
            52: "And Jesus increased in wisdom, in stature, and in favour with God and with people."
        }

        chapter_3 = {
            1: "In the fifteenth year of Tiberius Caesar's reign, when Pontius Pilate was governor of Judaea, Herod tetrarch of Galilee, his brother Philip tetrarch of the territories of Ituraea and Trachonitis, Lysanias tetrarch of Abilene, ",
            2: "and while the high-priesthood was held by Annas and Caiaphas, the word of God came to John the son of Zechariah, in the desert. ",
            3: "He went through the whole Jordan area proclaiming a baptism of repentance for the forgiveness of sins, ",
            4: "as it is written in the book of the sayings of Isaiah the prophet: A voice of one that cries in the desert: Prepare a way for the Lord, make his paths straight! ",
            5: "Let every valley be filled in, every mountain and hill be levelled, winding ways be straightened and rough roads made smooth, ",
            6: "and all humanity will see the salvation of God. ",
            7: "He said, therefore, to the crowds who came to be baptised by him, 'Brood of vipers, who warned you to flee from the coming retribution? ",
            8: 'Produce fruit in keeping with repentance, and do not start telling yourselves, "We have Abraham as our father," because, I tell you, God can raise children for Abraham from these stones. ',
            9: "Yes, even now the axe is being laid to the root of the trees, so that any tree failing to produce good fruit will be cut down and thrown on the fire.' ",
            10: "When all the people asked him, 'What must we do, then?' ",
            11: "he answered, 'Anyone who has two tunics must share with the one who has none, and anyone with something to eat must do the same.' ",
            12: "There were tax collectors, too, who came for baptism, and these said to him, 'Master, what must we do?' ",
            13: "He said to them, 'Exact no more than the appointed rate.' ",
            14: "Some soldiers asked him in their turn, 'What about us? What must we do?' He said to them, 'No intimidation! No extortion! Be content with your pay!' ",
            15: "A feeling of expectancy had grown among the people, who were beginning to wonder whether John might be the Christ, ",
            16: "so John declared before them all, 'I baptise you with water, but someone is coming, who is more powerful than me, and I am not fit to undo the strap of his sandals; he will baptise you with the Holy Spirit and fire. ",
            17: "His winnowing-fan is in his hand, to clear his threshing-floor and to gather the wheat into his barn; but the chaff he will burn in a fire that will never go out.' ",
            18: "And he proclaimed the good news to the people with many other exhortations too. ",
            19: "But Herod the tetrarch, censured by John for his relations with his brother's wife Herodias and for all the other crimes he had committed, ",
            20: "added a further crime to all the rest by shutting John up in prison. ",
            21: "Now it happened that when all the people had been baptised and while Jesus after his own baptism was at prayer, heaven opened ",
            22: "and the Holy Spirit descended on him in a physical form, like a dove. And a voice came from heaven, 'You are my Son; today have I fathered you.' ",
            23: "When he began, Jesus was about thirty years old, being the son, as it was thought, of Joseph son of Heli, ",
            24: "son of Matthat, son of Levi, son of Melchi, son of Jannai, son of Joseph, ",
            25: "son of Mattathias, son of Amos, son of Nahum, son of Esli, son of Naggai, ",
            26: "son of Maath, son of Mattathias, son of Semein, son of Josech, son of Joda, ",
            27: "son of Joanan, son of Rhesa, son of Zerubbabel, son of Shealtiel, son of Neri, ",
            28: "son of Melchi, son of Addi, son of Cosam, son of Elmadam, son of Er, ",
            29: "son of Jesus, son of Eliezer, son of Jorim, son of Matthat, son of Levi, ",
            30: "son of Symeon, son of Judah, son of Joseph, son of Jonam, son of Eliakim, ",
            31: "son of Melea, son of Menna, son of Mattatha, son of Nathan, son of David, ",
            32: "son of Jesse, son of Obed, son of Boaz, son of Sala, son of Nahshon, ",
            33: "son of Amminadab, son of Admin, son of Arni, son of Hezron, son of Perez, son of Judah, ",
            34: "son of Jacob, son of Isaac, son of Abraham, son of Terah, son of Nahor, ",
            35: "son of Serug, son of Reu, son of Peleg, son of Eber, son of Shelah, ",
            36: "son of Cainan, son of Arphaxad, son of Shem, son of Noah, son of Lamech, ",
            37: "son of Methuselah, son of Enoch, son of Jared, son of Mahalaleel, son of Cainan, ",
            38: "son of Enos, son of Seth, son of Adam, son of God.",
        }

        chapter_4 = {
            1: "Filled with the Holy Spirit, Jesus left the Jordan and was led by the Spirit into the desert, ",
            2: "for forty days being put to the test by the devil. During that time he ate nothing and at the end he was hungry. ",
            3: "Then the devil said to him, 'If you are Son of God, tell this stone to turn into a loaf.' ",
            4: "But Jesus replied, 'Scripture says: Human beings live not on bread alone.' ",
            5: "Then leading him to a height, the devil showed him in a moment of time all the kingdoms of the world ",
            6: "and said to him, 'I will give you all this power and their splendour, for it has been handed over to me, for me to give it to anyone I choose. ",
            7: "Do homage, then, to me, and it shall all be yours.' ",
            8: "But Jesus answered him, 'Scripture says: You must do homage to the Lord your God, him alone you must serve.' ",
            9: "Then he led him to Jerusalem and set him on the parapet of the Temple. 'If you are Son of God,' he said to him, 'throw yourself down from here, ",
            10: "for scripture says: He has given his angels orders about you, to guard you, and again: ",
            11: "They will carry you in their arms in case you trip over a stone.' ",
            12: "But Jesus answered him, 'Scripture says: Do not put the Lord your God to the test.' ",
            13: "Having exhausted every way of putting him to the test, the devil left him, until the opportune moment. ",
            14: "Jesus, with the power of the Spirit in him, returned to Galilee; and his reputation spread throughout the countryside. ",
            15: "He taught in their synagogues and everyone glorified him. ",
            16: "He came to Nazara, where he had been brought up, and went into the synagogue on the Sabbath day as he usually did. He stood up to read, ",
            17: "and they handed him the scroll of the prophet Isaiah. Unrolling the scroll he found the place where it is written: ",
            18: "The spirit of the Lord is on me, for he has anointed me to bring the good news to the afflicted. He has sent me to proclaim liberty to captives, sight to the blind, to let the oppressed go free, ",
            19: "to proclaim a year of favour from the Lord. ",
            20: "He then rolled up the scroll, gave it back to the assistant and sat down. And all eyes in the synagogue were fixed on him. ",
            21: "Then he began to speak to them, 'This text is being fulfilled today even while you are listening.' ",
            22: "And he won the approval of all, and they were astonished by the gracious words that came from his lips. They said, 'This is Joseph's son, surely?' ",
            23: """But he replied, 'No doubt you will quote me the saying, "Physician, heal yourself," and tell me, "We have heard all that happened in Capernaum, do the same here in your own country." ' """,
            24: "And he went on, 'In truth I tell you, no prophet is ever accepted in his own country. ",
            25: "'There were many widows in Israel, I can assure you, in Elijah's day, when heaven remained shut for three years and six months and a great famine raged throughout the land, ",
            26: "but Elijah was not sent to any one of these: he was sent to a widow at Zarephath, a town in Sidonia. ",
            27: "And in the prophet Elisha's time there were many suffering from virulent skin-diseases in Israel, but none of these was cured -- only Naaman the Syrian.' ",
            28: "When they heard this everyone in the synagogue was enraged. ",
            29: "They sprang to their feet and hustled him out of the town; and they took him up to the brow of the hill their town was built on, intending to throw him off the cliff, ",
            30: "but he passed straight through the crowd and walked away. ",
            31: "He went down to Capernaum, a town in Galilee, and taught them on the Sabbath. ",
            32: "And his teaching made a deep impression on them because his word carried authority. ",
            33: "In the synagogue there was a man possessed by the spirit of an unclean devil, and he shouted at the top of his voice, ",
            34: "'Ha! What do you want with us, Jesus of Nazareth? Have you come to destroy us? I know who you are: the Holy One of God.' ",
            35: "But Jesus rebuked it, saying, 'Be quiet! Come out of him!' And the devil, throwing the man into the middle, went out of him without hurting him at all. ",
            36: "Astonishment seized them and they were all saying to one another, 'What is it in his words? He gives orders to unclean spirits with authority and power and they come out.' ",
            37: "And the news of him travelled all through the surrounding countryside. ",
            38: "Leaving the synagogue he went to Simon's house. Now Simon's mother-in-law was in the grip of a high fever and they asked him to do something for her. ",
            39: "Standing over her he rebuked the fever and it left her. And she immediately got up and began to serve them. ",
            40: "At sunset all those who had friends suffering from diseases of one kind or another brought them to him, and laying his hands on each he cured them. ",
            41: "Devils too came out of many people, shouting, 'You are the Son of God.' But he warned them and would not allow them to speak because they knew that he was the Christ. ",
            42: "When daylight came he left the house and made his way to a lonely place. The crowds went to look for him, and when they had caught up with him they wanted to prevent him leaving them, ",
            43: "but he answered, 'I must proclaim the good news of the kingdom of God to the other towns too, because that is what I was sent to do.' ",
            44: "And he continued his proclamation in the synagogues of Judaea.",
        }

        chapter_5 = {
            1: "Now it happened that he was standing one day by the Lake of Gennesaret, with the crowd pressing round him listening to the word of God, ",
            2: "when he caught sight of two boats at the water's edge. The fishermen had got out of them and were washing their nets. ",
            3: "He got into one of the boats -- it was Simon's -- and asked him to put out a little from the shore. Then he sat down and taught the crowds from the boat. ",
            4: "When he had finished speaking he said to Simon, 'Put out into deep water and pay out your nets for a catch.' ",
            5: "Simon replied, 'Master, we worked hard all night long and caught nothing, but if you say so, I will pay out the nets.' ",
            6: "And when they had done this they netted such a huge number of fish that their nets began to tear, ",
            7: "so they signalled to their companions in the other boat to come and help them; when these came, they filled both boats to sinking point. ",
            8: "When Simon Peter saw this he fell at the knees of Jesus saying, 'Leave me, Lord; I am a sinful man.' ",
            9: "For he and all his companions were completely awestruck at the catch they had made; ",
            10: "so also were James and John, sons of Zebedee, who were Simon's partners. But Jesus said to Simon, 'Do not be afraid; from now on it is people you will be catching.' ",
            11: "Then, bringing their boats back to land they left everything and followed him. ",
            12: "Now it happened that Jesus was in one of the towns when suddenly a man appeared, covered with a skin-disease. Seeing Jesus he fell on his face and implored him saying, 'Sir, if you are willing you can cleanse me.' ",
            13: "He stretched out his hand, and touched him saying, 'I am willing. Be cleansed.' At once the skin-disease left him. ",
            14: "He ordered him to tell no one, 'But go and show yourself to the priest and make the offering for your cleansing just as Moses prescribed, as evidence to them.' ",
            15: "But the news of him kept spreading, and large crowds would gather to hear him and to have their illnesses cured, ",
            16: "but he would go off to some deserted place and pray. ",
            17: "Now it happened that he was teaching one day, and Pharisees and teachers of the Law, who had come from every village in Galilee, from Judaea and from Jerusalem, were sitting there. And the power of the Lord was there so that he should heal. ",
            18: "And now some men appeared, bringing on a bed a paralysed man whom they were trying to bring in and lay down in front of him. ",
            19: "But as they could find no way of getting the man through the crowd, they went up onto the top of the house and lowered him and his stretcher down through the tiles into the middle of the gathering, in front of Jesus. ",
            20: "Seeing their faith he said, 'My friend, your sins are forgiven you.' ",
            21: "The scribes and the Pharisees began to think this over. 'Who is this man, talking blasphemy? Who but God alone can forgive sins?' ",
            22: "But Jesus, aware of their thoughts, made them this reply, 'What are these thoughts you have in your hearts? ",
            23: """Which of these is easier: to say, "Your sins are forgiven you," or to say, "Get up and walk"? """,
            24: "But to prove to you that the Son of man has authority on earth to forgive sins,' -- he said to the paralysed man-'I order you: get up, and pick up your stretcher and go home.' ",
            25: "And immediately before their very eyes he got up, picked up what he had been lying on and went home praising God. ",
            26: "They were all astounded and praised God and were filled with awe, saying, 'We have seen strange things today.' ",
            27: "When he went out after this, he noticed a tax collector, Levi by name, sitting at the tax office, and said to him, 'Follow me.' ",
            28: "And leaving everything Levi got up and followed him. ",
            29: "In his honour Levi held a great reception in his house, and with them at table was a large gathering of tax collectors and others. ",
            30: "The Pharisees and their scribes complained to his disciples and said, 'Why do you eat and drink with tax collectors and sinners?' ",
            31: "Jesus said to them in reply, 'It is not those that are well who need the doctor, but the sick. ",
            32: "I have come to call not the upright but sinners to repentance.'",
            33: "They then said to him, 'John's disciples are always fasting and saying prayers, and the disciples of the Pharisees, too, but yours go on eating and drinking.' ",
            34: "Jesus replied, 'Surely you cannot make the bridegroom's attendants fast while the bridegroom is still with them? ",
            35: "But the time will come when the bridegroom is taken away from them; then, in those days, they will fast.' ",
            36: "He also told them a parable, 'No one tears a piece from a new cloak to put it on an old cloak; otherwise, not only will the new one be torn, but the piece taken from the new will not match the old. ",
            37: "'And nobody puts new wine in old wineskins; otherwise, the new wine will burst the skins and run to waste, and the skins will be ruined. ",
            38: "No; new wine must be put in fresh skins. ",
            39: """And nobody who has been drinking old wine wants new. "The old is good," he says.'""",
        }

        chapter_6 = {
            1: "It happened that one Sabbath he was walking through the cornfields, and his disciples were picking ears of corn, rubbing them in their hands and eating them. ",
            2: "Some of the Pharisees said, 'Why are you doing something that is forbidden on the Sabbath day?' ",
            3: "Jesus answered them, 'So you have not read what David did when he and his followers were hungry- ",
            4: "how he went into the house of God and took the loaves of the offering and ate them and gave them to his followers, loaves which the priests alone are allowed to eat?' ",
            5: "And he said to them, 'The Son of man is master of the Sabbath.' ",
            6: "Now on another Sabbath he went into the synagogue and began to teach, and a man was present, and his right hand was withered. ",
            7: "The scribes and the Pharisees were watching him to see if he would cure somebody on the Sabbath, hoping to find something to charge him with. ",
            8: "But he knew their thoughts; and he said to the man with the withered hand, 'Get up and stand out in the middle!' And he came forward and stood there. ",
            9: "Then Jesus said to them, 'I put it to you: is it permitted on the Sabbath to do good, or to do evil; to save life, or to destroy it?' ",
            10: "Then he looked round at them all and said to the man, 'Stretch out your hand.' He did so, and his hand was restored. ",
            11: "But they were furious and began to discuss the best way of dealing with Jesus. ",
            12: "Now it happened in those days that he went onto the mountain to pray; and he spent the whole night in prayer to God. ",
            13: "When day came he summoned his disciples and picked out twelve of them; he called them 'apostles': ",
            14: "Simon whom he called Peter, and his brother Andrew, James, John, Philip, Bartholomew, ",
            15: "Matthew, Thomas, James son of Alphaeus, Simon called the Zealot, ",
            16: "Judas son of James, and Judas Iscariot who became a traitor. ",
            17: "He then came down with them and stopped at a piece of level ground where there was a large gathering of his disciples, with a great crowd of people from all parts of Judaea and Jerusalem and the coastal region of Tyre and Sidon ",
            18: "who had come to hear him and to be cured of their diseases. People tormented by unclean spirits were also cured, ",
            19: "and everyone in the crowd was trying to touch him because power came out of him that cured them all. ",
            20: "Then fixing his eyes on his disciples he said: How blessed are you who are poor: the kingdom of God is yours. ",
            21: "Blessed are you who are hungry now: you shall have your fill. Blessed are you who are weeping now: you shall laugh. ",
            22: "'Blessed are you when people hate you, drive you out, abuse you, denounce your name as criminal, on account of the Son of man. ",
            23: "Rejoice when that day comes and dance for joy, look!-your reward will be great in heaven. This was the way their ancestors treated the prophets. ",
            24: "But alas for you who are rich: you are having your consolation now. ",
            25: "Alas for you who have plenty to eat now: you shall go hungry. Alas for you who are laughing now: you shall mourn and weep. ",
            26: "'Alas for you when everyone speaks well of you! This was the way their ancestors treated the false prophets. ",
            27: "'But I say this to you who are listening: Love your enemies, do good to those who hate you, ",
            28: "bless those who curse you, pray for those who treat you badly. ",
            29: "To anyone who slaps you on one cheek, present the other cheek as well; to anyone who takes your cloak from you, do not refuse your tunic. ",
            30: "Give to everyone who asks you, and do not ask for your property back from someone who takes it. ",
            31: "Treat others as you would like people to treat you. ",
            32: "If you love those who love you, what credit can you expect? Even sinners love those who love them. ",
            33: "And if you do good to those who do good to you, what credit can you expect? For even sinners do that much. ",
            34: "And if you lend to those from whom you hope to get money back, what credit can you expect? Even sinners lend to sinners to get back the same amount. ",
            35: "Instead, love your enemies and do good to them, and lend without any hope of return. You will have a great reward, and you will be children of the Most High, for he himself is kind to the ungrateful and the wicked. ",
            36: "'Be compassionate just as your Father is compassionate. ",
            37: "Do not judge, and you will not be judged; do not condemn, and you will not be condemned; forgive, and you will be forgiven. ",
            38: "Give, and there will be gifts for you: a full measure, pressed down, shaken together, and overflowing, will be poured into your lap; because the standard you use will be the standard used for you.' ",
            39: "He also told them a parable, 'Can one blind person guide another? Surely both will fall into a pit? ",
            40: "Disciple is not superior to teacher; but fully trained disciple will be like teacher. ",
            41: "Why do you observe the splinter in your brother's eye and never notice the great log in your own? ",
            42: """How can you say to your brother, "Brother, let me take out that splinter in your eye," when you cannot see the great log in your own? Hypocrite! Take the log out of your own eye first, and then you will see clearly enough to take out the splinter in your brother's eyes. """,
            43: "'There is no sound tree that produces rotten fruit, nor again a rotten tree that produces sound fruit. ",
            44: "Every tree can be told by its own fruit: people do not pick figs from thorns, nor gather grapes from brambles. ",
            45: "Good people draw what is good from the store of goodness in their hearts; bad people draw what is bad from the store of badness. For the words of the mouth flow out of what fills the heart. ",
            46: """'Why do you call me, "Lord, Lord" and not do what I say? """,
            47: "'Everyone who comes to me and listens to my words and acts on them -- I will show you what such a person is like. ",
            48: "Such a person is like the man who, when he built a house, dug, and dug deep, and laid the foundations on rock; when the river was in flood it bore down on that house but could not shake it, it was so well built. ",
            49: "But someone who listens and does nothing is like the man who built a house on soil, with no foundations; as soon as the river bore down on it, it collapsed; and what a ruin that house became!'",
        }

        chapter_7 = {
            1: "When he had come to the end of all he wanted the people to hear, he went into Capernaum. ",
            2: "A centurion there had a servant, a favourite of his, who was sick and near death. ",
            3: "Having heard about Jesus he sent some Jewish elders to him to ask him to come and heal his servant. ",
            4: "When they came to Jesus they pleaded earnestly with him saying, 'He deserves this of you, ",
            5: "because he is well disposed towards our people; he built us our synagogue himself.' ",
            6: "So Jesus went with them, and was not very far from the house when the centurion sent word to him by some friends to say to him, 'Sir, do not put yourself to any trouble because I am not worthy to have you under my roof; ",
            7: "and that is why I did not presume to come to you myself; let my boy be cured by your giving the word. ",
            8: """For I am under authority myself, and have soldiers under me; and I say to one man, "Go," and he goes; to another, "Come here," and he comes; to my servant, "Do this," and he does it.' """,
            9: "When Jesus heard these words he was astonished at him and, turning round, said to the crowd following him, 'I tell you, not even in Israel have I found faith as great as this.' ",
            10: "And when the messengers got back to the house they found the servant in perfect health. ",
            11: "It happened that soon afterwards he went to a town called Nain, accompanied by his disciples and a great number of people. ",
            12: "Now when he was near the gate of the town there was a dead man being carried out, the only son of his mother, and she was a widow. And a considerable number of the townspeople was with her. ",
            13: "When the Lord saw her he felt sorry for her and said to her, 'Don't cry.' ",
            14: "Then he went up and touched the bier and the bearers stood still, and he said, 'Young man, I tell you: get up.' ",
            15: "And the dead man sat up and began to talk, and Jesus gave him to his mother. ",
            16: "Everyone was filled with awe and glorified God saying, 'A great prophet has risen up among us; God has visited his people.' ",
            17: "And this view of him spread throughout Judaea and all over the countryside. ",
            18: "The disciples of John gave him all this news, and John, summoning two of his disciples, ",
            19: "sent them to the Lord to ask, 'Are you the one who is to come, or are we to expect someone else?' ",
            20: """When the men reached Jesus they said, 'John the Baptist has sent us to you to ask, "Are you the one who is to come or are we to expect someone else?" ' """,
            21: "At that very time he cured many people of diseases and afflictions and of evil spirits, and gave the gift of sight to many who were blind. ",
            22: "Then he gave the messengers their answer, 'Go back and tell John what you have seen and heard: the blind see again, the lame walk, those suffering from virulent skin-diseases are cleansed, and the deaf hear, the dead are raised to life, the good news is proclaimed to the poor; ",
            23: "and blessed is anyone who does not find me a cause of falling.' ",
            24: "When John's messengers had gone he began to talk to the people about John, ",
            25: "'What did you go out into the desert to see? A reed swaying in the breeze? No! Then what did you go out to see? A man dressed in fine clothes? Look, those who go in magnificent clothes and live luxuriously are to be found at royal courts! ",
            26: "Then what did you go out to see? A prophet? Yes, I tell you, and much more than a prophet: ",
            27: "he is the one of whom scripture says: Look, I am going to send my messenger in front of you to prepare your way before you. ",
            28: "'I tell you, of all the children born to women, there is no one greater than John; yet the least in the kingdom of God is greater than he.' ",
            29: "All the people who heard him, and the tax collectors too, acknowledged God's saving justice by accepting baptism from John; ",
            30: "but by refusing baptism from him the Pharisees and the lawyers thwarted God's plan for them. ",
            31: "'What comparison, then, can I find for the people of this generation? What are they like? ",
            32: "They are like children shouting to one another while they sit in the market place: We played the pipes for you, and you wouldn't dance; we sang dirges, and you wouldn't cry. ",
            33: """'For John the Baptist has come, not eating bread, not drinking wine, and you say, "He is possessed." """,
            34: 'The Son of man has come, eating and drinking, and you say, "Look, a glutton and a drunkard, a friend of tax collectors and sinners." ',
            35: "Yet wisdom is justified by all her children.' ",
            36: "One of the Pharisees invited him to a meal. When he arrived at the Pharisee's house and took his place at table, ",
            37: "suddenly a woman came in, who had a bad name in the town. She had heard he was dining with the Pharisee and had brought with her an alabaster jar of ointment. ",
            38: "She waited behind him at his feet, weeping, and her tears fell on his feet, and she wiped them away with her hair; then she covered his feet with kisses and anointed them with the ointment. ",
            39: "When the Pharisee who had invited him saw this, he said to himself, 'If this man were a prophet, he would know who this woman is and what sort of person it is who is touching him and what a bad name she has.' ",
            40: "Then Jesus took him up and said, 'Simon, I have something to say to you.' He replied, 'Say on, Master.' ",
            41: "'There was once a creditor who had two men in his debt; one owed him five hundred denarii, the other fifty. ",
            42: "They were unable to pay, so he let them both off. Which of them will love him more?' ",
            43: "Simon answered, 'The one who was let off more, I suppose.' Jesus said, 'You are right.' ",
            44: "Then he turned to the woman and said to Simon, 'You see this woman? I came into your house, and you poured no water over my feet, but she has poured out her tears over my feet and wiped them away with her hair. ",
            45: "You gave me no kiss, but she has been covering my feet with kisses ever since I came in. ",
            46: "You did not anoint my head with oil, but she has anointed my feet with ointment. ",
            47: "For this reason I tell you that her sins, many as they are, have been forgiven her, because she has shown such great love. It is someone who is forgiven little who shows little love.' ",
            48: "Then he said to her, 'Your sins are forgiven.' ",
            49: "Those who were with him at table began to say to themselves, 'Who is this man, that even forgives sins?' ",
            50: "But he said to the woman, 'Your faith has saved you; go in peace.'"
        }

        chapter_8 = {
            1: "Now it happened that after this he made his way through towns and villages preaching and proclaiming the good news of the kingdom of God. With him went the Twelve, ",
            2: "as well as certain women who had been cured of evil spirits and ailments: Mary surnamed the Magdalene, from whom seven demons had gone out, ",
            3: "Joanna the wife of Herod's steward Chuza, Susanna, and many others who provided for them out of their own resources. ",
            4: "With a large crowd gathering and people from every town finding their way to him, he told this parable: ",
            5: "'A sower went out to sow his seed. Now as he sowed, some fell on the edge of the path and was trampled on; and the birds of the air ate it up. ",
            6: "Some seed fell on rock, and when it came up it withered away, having no moisture. ",
            7: "Some seed fell in the middle of thorns and the thorns grew with it and choked it. ",
            8: "And some seed fell into good soil and grew and produced its crop a hundredfold.' Saying this he cried, 'Anyone who has ears for listening should listen!' ",
            9: "His disciples asked him what this parable might mean, ",
            10: "and he said, 'To you is granted to understand the secrets of the kingdom of God; for the rest it remains in parables, so that they may look but not perceive, listen but not understand. ",
            11: "'This, then, is what the parable means: the seed is the word of God. ",
            12: "Those on the edge of the path are people who have heard it, and then the devil comes and carries away the word from their hearts in case they should believe and be saved. ",
            13: "Those on the rock are people who, when they first hear it, welcome the word with joy. But these have no root; they believe for a while, and in time of trial they give up. ",
            14: "As for the part that fell into thorns, this is people who have heard, but as they go on their way they are choked by the worries and riches and pleasures of life and never produce any crops. ",
            15: "As for the part in the rich soil, this is people with a noble and generous heart who have heard the word and take it to themselves and yield a harvest through their perseverance. ",
            16: "'No one lights a lamp to cover it with a bowl or to put it under a bed. No, it is put on a lamp-stand so that people may see the light when they come in. ",
            17: "For nothing is hidden but it will be made clear, nothing secret but it will be made known and brought to light. ",
            18: "So take care how you listen; anyone who has, will be given more; anyone who has not, will be deprived even of what he thinks he has.' ",
            19: "His mother and his brothers came looking for him, but they could not get to him because of the crowd. ",
            20: "He was told, 'Your mother and brothers are standing outside and want to see you.' ",
            21: "But he said in answer, 'My mother and my brothers are those who hear the word of God and put it into practice.' ",
            22: "It happened that one day he got into a boat with his disciples and said to them, 'Let us cross over to the other side of the lake.' So they set out, ",
            23: "and as they sailed he fell asleep. When a squall of wind came down on the lake the boat started shipping water and they found themselves in danger. ",
            24: "So they went to rouse him saying, 'Master! Master! We are lost!' Then he woke up and rebuked the wind and the rough water; and they subsided and it was calm again. ",
            25: "He said to them, 'Where is your faith?' They were awestruck and astounded and said to one another, 'Who can this be, that gives orders even to winds and waves and they obey him?' ",
            26: "They came to land in the territory of the Gerasenes, which is opposite Galilee. ",
            27: "He was stepping ashore when a man from the city who was possessed by devils came towards him; for a long time the man had been living with no clothes on, not in a house, but in the tombs. ",
            28: "Catching sight of Jesus he gave a shout, fell at his feet and cried out at the top of his voice, 'What do you want with me, Jesus, son of the Most High God? I implore you, do not torture me.' ",
            29: "For Jesus had been telling the unclean spirit to come out of the man. It had seized on him a great many times, and then they used to secure him with chains and fetters to restrain him, but he would always break the fastenings, and the devil would drive him out into the wilds. ",
            30: "Jesus asked him, 'What is your name?' He said, 'Legion' -- because many devils had gone into him. ",
            31: "And these begged him not to order them to depart into the Abyss. ",
            32: "Now there was a large herd of pigs feeding there on the mountain, and the devils begged him to let them go into these. So he gave them leave. ",
            33: "The devils came out of the man and went into the pigs, and the herd charged down the cliff into the lake and was drowned. ",
            34: "When the swineherds saw what had happened they ran off and told their story in the city and in the country round about; ",
            35: "and the people went out to see what had happened. When they came to Jesus they found the man from whom the devils had gone out sitting at the feet of Jesus, wearing clothes and in his right mind; and they were afraid. ",
            36: "Those who had witnessed it told them how the man who had been possessed came to be saved. ",
            37: "The entire population of the Gerasene territory was in great fear and asked Jesus to leave them. So he got into the boat and went back. ",
            38: "The man from whom the devils had gone out asked to be allowed to stay with him, but he sent him away saying, ",
            39: "'Go back home and report all that God has done for you.' So the man went off and proclaimed throughout the city all that Jesus had done for him. ",
            40: "On his return Jesus was welcomed by the crowd, for they were all there waiting for him. ",
            41: "And suddenly there came a man named Jairus, who was president of the synagogue. He fell at Jesus' feet and pleaded with him to come to his house, ",
            42: "because he had an only daughter about twelve years old, who was dying. And the crowds were almost stifling Jesus as he went. ",
            43: "Now there was a woman suffering from a haemorrhage for the past twelve years, whom no one had been able to cure. ",
            44: "She came up behind him and touched the fringe of his cloak; and the haemorrhage stopped at that very moment. ",
            45: "Jesus said, 'Who was it that touched me?' When they all denied it, Peter said, 'Master, it is the crowds round you, pushing.' ",
            46: "But Jesus said, 'Somebody touched me. I felt that power had gone out from me.' ",
            47: "Seeing herself discovered, the woman came forward trembling, and falling at his feet explained in front of all the people why she had touched him and how she had been cured at that very moment. ",
            48: "'My daughter,' he said, 'your faith has saved you; go in peace.' ",
            49: "While he was still speaking, someone arrived from the house of the president of the synagogue to say, 'Your daughter has died. Do not trouble the Master any further.' ",
            50: "But Jesus heard this, and he spoke to the man, 'Do not be afraid, only have faith and she will be saved.' ",
            51: "When he came to the house he allowed no one to go in with him except Peter and John and James, and the child's father and mother. ",
            52: "They were all crying and mourning for her, but Jesus said, 'Stop crying; she is not dead, but asleep.' ",
            53: "But they ridiculed him, knowing she was dead. ",
            54: "But taking her by the hand himself he spoke to her, 'Child, get up.' ",
            55: "And her spirit returned and she got up at that very moment. Then he told them to give her something to eat. ",
            56: "Her parents were astonished, but he ordered them not to tell anyone what had happened."
        }

        chapter_9 = {
            1: "He called the Twelve together and gave them power and authority over all devils and to cure diseases, ",
            2: "and he sent them out to proclaim the kingdom of God and to heal. ",
            3: "He said to them, 'Take nothing for the journey: neither staff, nor haversack, nor bread, nor money; and do not have a spare tunic. ",
            4: "Whatever house you enter, stay there; and when you leave let your departure be from there. ",
            5: "As for those who do not welcome you, when you leave their town shake the dust from your feet as evidence against them.' ",
            6: "So they set out and went from village to village proclaiming the good news and healing everywhere. ",
            7: "Meanwhile Herod the tetrarch had heard about all that was going on; and he was puzzled, because some people were saying that John had risen from the dead, ",
            8: "others that Elijah had reappeared, still others that one of the ancient prophets had come back to life. ",
            9: "But Herod said, 'John? I beheaded him. So who is this I hear such reports about?' And he was anxious to see him. ",
            10: "On their return the apostles gave him an account of all they had done. Then he took them with him and withdrew towards a town called Bethsaida where they could be by themselves. ",
            11: "But the crowds got to know and they went after him. He made them welcome and talked to them about the kingdom of God; and he cured those who were in need of healing. ",
            12: "It was late afternoon when the Twelve came up to him and said, 'Send the people away, and they can go to the villages and farms round about to find lodging and food; for we are in a lonely place here.' ",
            13: "He replied, 'Give them something to eat yourselves.' But they said, 'We have no more than five loaves and two fish, unless we are to go ourselves and buy food for all these people.' ",
            14: "For there were about five thousand men. But he said to his disciples, 'Get them to sit down in parties of about fifty.' ",
            15: "They did so and made them all sit down. ",
            16: "Then he took the five loaves and the two fish, raised his eyes to heaven, and said the blessing over them; then he broke them and handed them to his disciples to distribute among the crowd. ",
            17: "They all ate as much as they wanted, and when the scraps left over were collected they filled twelve baskets. ",
            18: "Now it happened that he was praying alone, and his disciples came to him and he put this question to them, 'Who do the crowds say I am?' ",
            19: "And they answered, 'Some say John the Baptist; others Elijah; others again one of the ancient prophets come back to life.' ",
            20: "'But you,' he said to them, 'who do you say I am?' It was Peter who spoke up. 'The Christ of God,' he said. ",
            21: "But he gave them strict orders and charged them not to say this to anyone. ",
            22: "He said, 'The Son of man is destined to suffer grievously, to be rejected by the elders and chief priests and scribes and to be put to death, and to be raised up on the third day.' ",
            23: "Then, speaking to all, he said, 'If anyone wants to be a follower of mine, let him renounce himself and take up his cross every day and follow me. ",
            24: "Anyone who wants to save his life will lose it; but anyone who loses his life for my sake, will save it. ",
            25: "What benefit is it to anyone to win the whole world and forfeit or lose his very self? ",
            26: "For if anyone is ashamed of me and of my words, of him the Son of man will be ashamed when he comes in his own glory and in the glory of the Father and the holy angels. ",
            27: "'I tell you truly, there are some standing here who will not taste death before they see the kingdom of God.' ",
            28: "Now about eight days after this had been said, he took with him Peter, John and James and went up the mountain to pray. ",
            29: "And it happened that, as he was praying, the aspect of his face was changed and his clothing became sparkling white. ",
            30: "And suddenly there were two men talking to him; they were Moses and Elijah ",
            31: "appearing in glory, and they were speaking of his passing which he was to accomplish in Jerusalem. ",
            32: "Peter and his companions were heavy with sleep, but they woke up and saw his glory and the two men standing with him. ",
            33: "As these were leaving him, Peter said to Jesus, 'Master, it is wonderful for us to be here; so let us make three shelters, one for you, one for Moses and one for Elijah.' He did not know what he was saying. ",
            34: "As he was saying this, a cloud came and covered them with shadow; and when they went into the cloud the disciples were afraid. ",
            35: "And a voice came from the cloud saying, 'This is my Son, the Chosen One. Listen to him.' ",
            36: "And after the voice had spoken, Jesus was found alone. The disciples kept silence and, at that time, told no one what they had seen. ",
            37: "Now it happened that on the following day when they were coming down from the mountain a large crowd came to meet him. ",
            38: "And suddenly a man in the crowd cried out. 'Master,' he said, 'I implore you to look at my son: he is my only child. ",
            39: "A spirit will suddenly take hold of him, and all at once it gives a sudden cry and throws the boy into convulsions with foaming at the mouth; it is slow to leave him, but when it does, it leaves the boy worn out. ",
            40: "I begged your disciples to drive it out, and they could not.'",
            41: "In reply Jesus said, 'Faithless and perverse generation! How much longer must I be among you and put up with you? Bring your son here.' ",
            42: "Even while the boy was coming, the devil threw him to the ground in convulsions. But Jesus rebuked the unclean spirit and cured the boy and gave him back to his father, ",
            43: "and everyone was awestruck by the greatness of God. But while everyone was full of admiration for all he did, he said to his disciples, ",
            44: "'For your part, you must have these words constantly in mind: The Son of man is going to be delivered into the power of men.' ",
            45: "But they did not understand what he said; it was hidden from them so that they should not see the meaning of it, and they were afraid to ask him about it. ",
            46: "An argument started between them about which of them was the greatest. ",
            47: "Jesus knew what thoughts were going through their minds, and he took a little child whom he set by his side ",
            48: "and then he said to them, 'Anyone who welcomes this little child in my name welcomes me; and anyone who welcomes me, welcomes the one who sent me. The least among you all is the one who is the greatest.' ",
            49: "John spoke up. 'Master,' he said, 'we saw someone driving out devils in your name, and because he is not with us we tried to stop him.' ",
            50: "But Jesus said to him, 'You must not stop him: anyone who is not against you is for you.' ",
            51: "Now it happened that as the time drew near for him to be taken up, he resolutely turned his face towards Jerusalem ",
            52: "and sent messengers ahead of him. These set out, and they went into a Samaritan village to make preparations for him, ",
            53: "but the people would not receive him because he was making for Jerusalem. ",
            54: "Seeing this, the disciples James and John said, 'Lord, do you want us to call down fire from heaven to burn them up?' ",
            55: "But he turned and rebuked them, ",
            56: "and they went on to another village. ",
            57: "As they travelled along they met a man on the road who said to him, 'I will follow you wherever you go.' ",
            58: "Jesus answered, 'Foxes have holes and the birds of the air have nests, but the Son of man has nowhere to lay his head.' ",
            59: "Another to whom he said, 'Follow me,' replied, 'Let me go and bury my father first.' ",
            60: "But he answered, 'Leave the dead to bury their dead; your duty is to go and spread the news of the kingdom of God.' ",
            61: "Another said, 'I will follow you, sir, but first let me go and say good -- bye to my people at home.' ",
            62: "Jesus said to him, 'Once the hand is laid on the plough, no one who looks back is fit for the kingdom of God.'"
        }

        chapter_10 = {
            1: "After this the Lord appointed seventy-two others and sent them out ahead of him in pairs, to all the towns and places he himself would be visiting. ",
            2: "And he said to them, 'The harvest is rich but the labourers are few, so ask the Lord of the harvest to send labourers to do his harvesting. ",
            3: "Start off now, but look, I am sending you out like lambs among wolves. ",
            4: "Take no purse with you, no haversack, no sandals. Salute no one on the road. ",
            5: 'Whatever house you enter, let your first words be, "Peace to this house!" ',
            6: "And if a man of peace lives there, your peace will go and rest on him; if not, it will come back to you. ",
            7: "Stay in the same house, taking what food and drink they have to offer, for the labourer deserves his wages; do not move from house to house. ",
            8: "Whenever you go into a town where they make you welcome, eat what is put before you. ",
            9: 'Cure those in it who are sick, and say, "The kingdom of God is very near to you." ',
            10: "But whenever you enter a town and they do not make you welcome, go out into its streets and say, ",
            11: '"We wipe off the very dust of your town that clings to our feet, and leave it with you. Yet be sure of this: the kingdom of God is very near." ',
            12: "I tell you, on the great Day it will be more bearable for Sodom than for that town.",
            13: "'Alas for you, Chorazin! Alas for you, Bethsaida! For if the miracles done in you had been done in Tyre and Sidon, they would have repented long ago, sitting in sackcloth and ashes. ",
            14: "And still, it will be more bearable for Tyre and Sidon at the Judgement than for you. ",
            15: "And as for you, Capernaum, did you want to be raised high as heaven? You shall be flung down to hell. ",
            16: "'Anyone who listens to you listens to me; anyone who rejects you rejects me, and those who reject me reject the one who sent me.' ",
            17: "The seventy-two came back rejoicing. 'Lord,' they said, 'even the devils submit to us when we use your name.' ",
            18: "He said to them, 'I watched Satan fall like lightning from heaven. ",
            19: "Look, I have given you power to tread down serpents and scorpions and the whole strength of the enemy; nothing shall ever hurt you. ",
            20: "Yet do not rejoice that the spirits submit to you; rejoice instead that your names are written in heaven.' ",
            21: "Just at this time, filled with joy by the Holy Spirit, he said, 'I bless you, Father, Lord of heaven and of earth, for hiding these things from the learned and the clever and revealing them to little children. Yes, Father, for that is what it has pleased you to do. ",
            22: "Everything has been entrusted to me by my Father; and no one knows who the Son is except the Father, and who the Father is except the Son and those to whom the Son chooses to reveal him.' ",
            23: "Then turning to his disciples he spoke to them by themselves, 'Blessed are the eyes that see what you see, ",
            24: "for I tell you that many prophets and kings wanted to see what you see, and never saw it; to hear what you hear, and never heard it.' ",
            25: "And now a lawyer stood up and, to test him, asked, 'Master, what must I do to inherit eternal life?' ",
            26: "He said to him, 'What is written in the Law? What is your reading of it?' ",
            27: "He replied, 'You must love the Lord your God with all your heart, with all your soul, with all your strength, and with all your mind, and your neighbour as yourself.' ",
            28: "Jesus said to him, 'You have answered right, do this and life is yours.' ",
            29: "But the man was anxious to justify himself and said to Jesus, 'And who is my neighbour?' ",
            30: "In answer Jesus said, 'A man was once on his way down from Jerusalem to Jericho and fell into the hands of bandits; they stripped him, beat him and then made off, leaving him half dead. ",
            31: "Now a priest happened to be travelling down the same road, but when he saw the man, he passed by on the other side. ",
            32: "In the same way a Levite who came to the place saw him, and passed by on the other side. ",
            33: "But a Samaritan traveller who came on him was moved with compassion when he saw him. ",
            34: "He went up to him and bandaged his wounds, pouring oil and wine on them. He then lifted him onto his own mount and took him to an inn and looked after him. ",
            35: 'Next day, he took out two denarii and handed them to the innkeeper and said, "Look after him, and on my way back I will make good any extra expense you have." ',
            36: "Which of these three, do you think, proved himself a neighbour to the man who fell into the bandits' hands?' ",
            37: "He replied, 'The one who showed pity towards him.' Jesus said to him, 'Go, and do the same yourself.' ",
            38: "In the course of their journey he came to a village, and a woman named Martha welcomed him into her house. ",
            39: "She had a sister called Mary, who sat down at the Lord's feet and listened to him speaking. ",
            40: "Now Martha, who was distracted with all the serving, came to him and said, 'Lord, do you not care that my sister is leaving me to do the serving all by myself? Please tell her to help me.' ",
            41: "But the Lord answered, 'Martha, Martha,' he said, 'you worry and fret about so many things, ",
            42: "and yet few are needed, indeed only one. It is Mary who has chosen the better part, and it is not to be taken from her.'"
        }

        chapter_11 = {
            1: "Now it happened that he was in a certain place praying, and when he had finished, one of his disciples said, 'Lord, teach us to pray, as John taught his disciples.' ",
            2: "He said to them, 'When you pray, this is what to say: Father, may your name be held holy, your kingdom come; ",
            3: "give us each day our daily bread, and forgive us our sins, ",
            4: "for we ourselves forgive each one who is in debt to us. And do not put us to the test.' ",
            5: '''He also said to them, 'Suppose one of you has a friend and goes to him in the middle of the night to say, "My friend, lend me three loaves, ''',
            6: 'because a friend of mine on his travels has just arrived at my house and I have nothing to offer him;" ',
            7: 'and the man answers from inside the house, "Do not bother me. The door is bolted now, and my children are with me in bed; I cannot get up to give it to you." ',
            8: "I tell you, if the man does not get up and give it to him for friendship's sake, persistence will make him get up and give his friend all he wants.",
            9: "'So I say to you: Ask, and it will be given to you; search, and you will find; knock, and the door will be opened to you. ",
            10: "For everyone who asks receives; everyone who searches finds; everyone who knocks will have the door opened. ",
            11: "What father among you, if his son asked for a fish, would hand him a snake? ",
            12: "Or if he asked for an egg, hand him a scorpion? ",
            13: "If you then, evil as you are, know how to give your children what is good, how much more will the heavenly Father give the Holy Spirit to those who ask him!' ",
            14: "He was driving out a devil and it was dumb; and it happened that when the devil had gone out the dumb man spoke, and the people were amazed. ",
            15: "But some of them said, 'It is through Beelzebul, the prince of devils, that he drives devils out.' ",
            16: "Others asked him, as a test, for a sign from heaven; ",
            17: "but, knowing what they were thinking, he said to them, 'Any kingdom which is divided against itself is heading for ruin, and house collapses against house. ",
            18: "So, too, with Satan: if he is divided against himself, how can his kingdom last? - since you claim that it is through Beelzebul that I drive devils out. ",
            19: "Now if it is through Beelzebul that I drive devils out, through whom do your own sons drive them out? They shall be your judges, then. ",
            20: "But if it is through the finger of God that I drive devils out, then the kingdom of God has indeed caught you unawares. ",
            21: "So long as a strong man fully armed guards his own home, his goods are undisturbed; ",
            22: "but when someone stronger than himself attacks and defeats him, the stronger man takes away all the weapons he relied on and shares out his spoil. ",
            23: "'Anyone who is not with me is against me; and anyone who does not gather in with me throws away. ",
            24: """'When an unclean spirit goes out of someone it wanders through waterless country looking for a place to rest, and not finding one it says, "I will go back to the home I came from." """,
            25: "But on arrival, finding it swept and tidied, ",
            26: "it then goes off and brings seven other spirits more wicked than itself, and they go in and set up house there, and so that person ends up worse off than before.' ",
            27: "It happened that as he was speaking, a woman in the crowd raised her voice and said, 'Blessed the womb that bore you and the breasts that fed you!' ",
            28: "But he replied, 'More blessed still are those who hear the word of God and keep it!' ",
            29: "The crowds got even bigger and he addressed them, 'This is an evil generation; it is asking for a sign. The only sign it will be given is the sign of Jonah. ",
            30: "For just as Jonah became a sign to the people of Nineveh, so will the Son of man be a sign to this generation. ",
            31: "On Judgement Day the Queen of the South will stand up against the people of this generation and be their condemnation, because she came from the ends of the earth to hear the wisdom of Solomon; and, look, there is something greater than Solomon here. ",
            32: "On Judgement Day the men of Nineveh will appear against this generation and be its condemnation, because when Jonah preached they repented; and, look, there is something greater than Jonah here. ",
            33: "'No one lights a lamp and puts it in some hidden place or under a tub; they put it on the lamp-stand so that people may see the light when they come in. ",
            34: "The lamp of the body is your eye. When your eye is clear, your whole body, too, is filled with light; but when it is diseased your body, too, will be darkened. ",
            35: "See to it then that the light inside you is not darkness. ",
            36: "If, therefore, your whole body is filled with light, and not darkened at all, it will be light entirely, as when the lamp shines on you with its rays.' ",
            37: "He had just finished speaking when a Pharisee invited him to dine at his house. He went in and sat down at table. ",
            38: "The Pharisee saw this and was surprised that he had not first washed before the meal. ",
            39: "But the Lord said to him, 'You Pharisees! You clean the outside of cup and plate, while inside yourselves you are filled with extortion and wickedness. ",
            40: "Fools! Did not he who made the outside make the inside too? ",
            41: "Instead, give alms from what you have and, look, everything will be clean for you. ",
            42: "But alas for you Pharisees, because you pay your tithe of mint and rue and all sorts of garden herbs and neglect justice and the love of God! These you should have practised, without neglecting the others. ",
            43: "Alas for you Pharisees, because you like to take the seats of honour in the synagogues and to be greeted respectfully in the market squares! ",
            44: "Alas for you, because you are like the unmarked tombs that people walk on without knowing it!' ",
            45: "A lawyer then spoke up. 'Master,' he said, 'when you speak like this you insult us too.' ",
            46: "But he said, 'Alas for you lawyers as well, because you load on people burdens that are unendurable, burdens that you yourselves do not touch with your fingertips. ",
            47: "'Alas for you because you build tombs for the prophets, the people your ancestors killed! ",
            48: "In this way you both witness to what your ancestors did and approve it; they did the killing, you do the building. ",
            49: """'And that is why the Wisdom of God said, "I will send them prophets and apostles; some they will slaughter and persecute, """,
            50: "so that this generation will have to answer for every prophet's blood that has been shed since the foundation of the world, ",
            51: 'from the blood of Abel to the blood of Zechariah, who perished between the altar and the Temple." Yes, I tell you, this generation will have to answer for it all. ',
            52: "'Alas for you lawyers who have taken away the key of knowledge! You have not gone in yourselves and have prevented others from going in who wanted to.' ",
            53: "When he left there, the scribes and the Pharisees began a furious attack on him and tried to force answers from him on innumerable questions, ",
            54: "lying in wait to catch him out in something he might say."
        }

        chapter_12 = {
            1: "Meanwhile the people had gathered in their thousands so that they were treading on one another. And he began to speak, first of all to his disciples. 'Be on your guard against the yeast of the Pharisees -- their hypocrisy. ",
            2: "Everything now covered up will be uncovered, and everything now hidden will be made clear. ",
            3: "For this reason, whatever you have said in the dark will be heard in the daylight, and what you have whispered in hidden places will be proclaimed from the housetops. ",
            4: "'To you my friends I say: Do not be afraid of those who kill the body and after that can do no more. ",
            5: "I will tell you whom to fear: fear him who, after he has killed, has the power to cast into hell. Yes, I tell you, he is the one to fear.",
            6: "Can you not buy five sparrows for two pennies? And yet not one is forgotten in God's sight. ",
            7: "Why, every hair on your head has been counted. There is no need to be afraid: you are worth more than many sparrows. ",
            8: "'I tell you, if anyone openly declares himself for me in the presence of human beings, the Son of man will declare himself for him in the presence of God's angels. ",
            9: "But anyone who disowns me in the presence of human beings will be disowned in the presence of God's angels. ",
            10: "'Everyone who says a word against the Son of man will be forgiven, but no one who blasphemes against the Holy Spirit will be forgiven. ",
            11: "'When they take you before synagogues and magistrates and authorities, do not worry about how to defend yourselves or what to say, ",
            12: "because when the time comes, the Holy Spirit will teach you what you should say.' ",
            13: "A man in the crowd said to him, 'Master, tell my brother to give me a share of our inheritance.' ",
            14: "He said to him, 'My friend, who appointed me your judge, or the arbitrator of your claims?' ",
            15: "Then he said to them, 'Watch, and be on your guard against avarice of any kind, for life does not consist in possessions, even when someone has more than he needs.' ",
            16: "Then he told them a parable, 'There was once a rich man who, having had a good harvest from his land, ",
            17: 'thought to himself, "What am I to do? I have not enough room to store my crops." ',
            18: 'Then he said, "This is what I will do: I will pull down my barns and build bigger ones, and store all my grain and my goods in them, ',
            19: 'and I will say to my soul: My soul, you have plenty of good things laid by for many years to come; take things easy, eat, drink, have a good time." ',
            20: 'But God said to him, "Fool! This very night the demand will be made for your soul; and this hoard of yours, whose will it be then?" ',
            21: "So it is when someone stores up treasure for himself instead of becoming rich in the sight of God.' ",
            22: "Then he said to his disciples, 'That is why I am telling you not to worry about your life and what you are to eat, nor about your body and how you are to clothe it. ",
            23: "For life is more than food, and the body more than clothing. ",
            24: "Think of the ravens. They do not sow or reap; they have no storehouses and no barns; yet God feeds them. And how much more you are worth than the birds! ",
            25: "Can any of you, however much you worry, add a single cubit to your span of life? ",
            26: "If a very small thing is beyond your powers, why worry about the rest? ",
            27: "Think how the flowers grow; they never have to spin or weave; yet, I assure you, not even Solomon in all his royal robes was clothed like one of them. ",
            28: "Now if that is how God clothes a flower which is growing wild today and is thrown into the furnace tomorrow, how much more will he look after you, who have so little faith! ",
            29: "But you must not set your hearts on things to eat and things to drink; nor must you worry. ",
            30: "It is the gentiles of this world who set their hearts on all these things. Your Father well knows you need them. ",
            31: "No; set your hearts on his kingdom, and these other things will be given you as well. ",
            32: "'There is no need to be afraid, little flock, for it has pleased your Father to give you the kingdom. ",
            33: "'Sell your possessions and give to those in need. Get yourselves purses that do not wear out, treasure that will not fail you, in heaven where no thief can reach it and no moth destroy it. ",
            34: "For wherever your treasure is, that is where your heart will be too. ",
            35: "'See that you have your belts done up and your lamps lit. ",
            36: "Be like people waiting for their master to return from the wedding feast, ready to open the door as soon as he comes and knocks. ",
            37: "Blessed those servants whom the master finds awake when he comes. In truth I tell you, he will do up his belt, sit them down at table and wait on them. ",
            38: "It may be in the second watch that he comes, or in the third, but blessed are those servants if he finds them ready. ",
            39: "You may be quite sure of this, that if the householder had known at what time the burglar would come, he would not have let anyone break through the wall of his house. ",
            40: "You too must stand ready, because the Son of man is coming at an hour you do not expect.' ",
            41: "Peter said, 'Lord, do you mean this parable for us, or for everyone?' ",
            42: "The Lord replied, 'Who, then, is the wise and trustworthy steward whom the master will place over his household to give them at the proper time their allowance of food? ",
            43: "Blessed that servant if his master's arrival finds him doing exactly that. ",
            44: "I tell you truly, he will put him in charge of everything that he owns.",
            45: 'But if the servant says to himself, "My master is taking his time coming," and sets about beating the menservants and the servant-girls, and eating and drinking and getting drunk, ',
            46: "his master will come on a day he does not expect and at an hour he does not know. The master will cut him off and send him to the same fate as the unfaithful. ",
            47: "'The servant who knows what his master wants, but has got nothing ready and done nothing in accord with those wishes, will be given a great many strokes of the lash. ",
            48: "The one who did not know, but has acted in such a way that he deserves a beating, will be given fewer strokes. When someone is given a great deal, a great deal will be demanded of that person; when someone is entrusted with a great deal, of that person even more will be expected. ",
            49: "'I have come to bring fire to the earth, and how I wish it were blazing already! ",
            50: "There is a baptism I must still receive, and what constraint I am under until it is completed! ",
            51: "'Do you suppose that I am here to bring peace on earth? No, I tell you, but rather division. ",
            52: "For from now on, a household of five will be divided: three against two and two against three; ",
            53: "father opposed to son, son to father, mother to daughter, daughter to mother, mother-in-law to daughter-in-law, daughter-in-law to mother-in-law.' ",
            54: "He said again to the crowds, 'When you see a cloud looming up in the west you say at once that rain is coming, and so it does. ",
            55: "And when the wind is from the south you say it's going to be hot, and it is. ",
            56: "Hypocrites! You know how to interpret the face of the earth and the sky. How is it you do not know how to interpret these times? ",
            57: "'Why not judge for yourselves what is upright? ",
            58: "For example: when you are going to court with your opponent, make an effort to settle with him on the way, or he may drag you before the judge and the judge hand you over to the officer and the officer have you thrown into prison. ",
            59: "I tell you, you will not get out till you have paid the very last penny.'"
        }

        chapter_13 = {
            1: "It was just about this time that some people arrived and told him about the Galileans whose blood Pilate had mingled with that of their sacrifices. At this he said to them, ",
            2: "'Do you suppose that these Galileans were worse sinners than any others, that this should have happened to them? ",
            3: "They were not, I tell you. No; but unless you repent you will all perish as they did. ",
            4: "Or those eighteen on whom the tower at Siloam fell, killing them all? Do you suppose that they were more guilty than all the other people living in Jerusalem? ",
            5: "They were not, I tell you. No; but unless you repent you will all perish as they did.' ",
            6: "He told this parable, 'A man had a fig tree planted in his vineyard, and he came looking for fruit on it but found none. ",
            7: 'He said to his vinedresser, "For three years now I have been coming to look for fruit on this fig tree and finding none. Cut it down: why should it be taking up the ground?" ',
            8: '"Sir," the man replied, "leave it one more year and give me time to dig round it and manure it: ',
            9: """it may bear fruit next year; if not, then you can cut it down." ' """,
            10: "One Sabbath day he was teaching in one of the synagogues, ",
            11: "and there before him was a woman who for eighteen years had been possessed by a spirit that crippled her; she was bent double and quite unable to stand upright. ",
            12: "When Jesus saw her he called her over and said, 'Woman, you are freed from your disability,' ",
            13: "and he laid his hands on her. And at once she straightened up, and she glorified God. ",
            14: "But the president of the synagogue was indignant because Jesus had healed on the Sabbath, and he addressed all those present saying, 'There are six days when work is to be done. Come and be healed on one of those days and not on the Sabbath.' ",
            15: "But the Lord answered him and said, 'Hypocrites! Is there one of you who does not untie his ox or his donkey from the manger on the Sabbath and take it out for watering? ",
            16: "And this woman, a daughter of Abraham whom Satan has held bound these eighteen years -- was it not right to untie this bond on the Sabbath day?' ",
            17: "When he said this, all his adversaries were covered with confusion, and all the people were overjoyed at all the wonders he worked. ",
            18: "He went on to say, 'What is the kingdom of God like? What shall I compare it with? ",
            19: "It is like a mustard seed which a man took and threw into his garden: it grew and became a tree, and the birds of the air sheltered in its branches.' ",
            20: "Again he said, 'What shall I compare the kingdom of God with? ",
            21: "It is like the yeast a woman took and mixed in with three measures of flour till it was leavened all through.' ",
            22: "Through towns and villages he went teaching, making his way to Jerusalem. ",
            23: "Someone said to him, 'Sir, will there be only a few saved?' He said to them, ",
            24: "'Try your hardest to enter by the narrow door, because, I tell you, many will try to enter and will not succeed. ",
            25: """'Once the master of the house has got up and locked the door, you may find yourself standing outside knocking on the door, saying, "Lord, open to us," but he will answer, "I do not know where you come from." """,
            26: 'Then you will start saying, "We once ate and drank in your company; you taught in our streets," ',
            27: 'but he will reply, "I do not know where you come from; away from me, all evil doers!" ',
            28: "'Then there will be weeping and grinding of teeth, when you see Abraham and Isaac and Jacob and all the prophets in the kingdom of God, and yourselves thrown out. ",
            29: "And people from east and west, from north and south, will come and sit down at the feast in the kingdom of God. ",
            30: "'Look, there are those now last who will be first, and those now first who will be last.' ",
            31: "Just at this time some Pharisees came up. 'Go away,' they said. 'Leave this place, because Herod means to kill you.' ",
            32: "He replied, 'You may go and give that fox this message: Look! Today and tomorrow I drive out devils and heal, and on the third day I attain my end. ",
            33: "But for today and tomorrow and the next day I must go on, since it would not be right for a prophet to die outside Jerusalem. ",
            34: "'Jerusalem, Jerusalem, you that kill the prophets and stone those who are sent to you! How often have I longed to gather your children together, as a hen gathers her brood under her wings, and you refused! ",
            35: "Look! Your house will be left to you. Yes, I promise you, you shall not see me till the time comes when you are saying: Blessed is he who is coming in the name of the Lord!'"
        }

        chapter_14 = {
            1: "Now it happened that on a Sabbath day he had gone to share a meal in the house of one of the leading Pharisees; and they watched him closely. ",
            2: "Now there in front of him was a man with dropsy, ",
            3: "and Jesus addressed the lawyers and Pharisees with the words, 'Is it against the law to cure someone on the Sabbath, or not?' ",
            4: "But they remained silent, so he took the man and cured him and sent him away. ",
            5: "Then he said to them, 'Which of you here, if his son falls into a well, or his ox, will not pull him out on a Sabbath day without any hesitation?' ",
            6: "And to this they could find no answer. ",
            7: "He then told the guests a parable, because he had noticed how they picked the places of honour. He said this, ",
            8: "'When someone invites you to a wedding feast, do not take your seat in the place of honour. A more distinguished person than you may have been invited, ",
            9: 'and the person who invited you both may come and say, "Give up your place to this man." And then, to your embarrassment, you will have to go and take the lowest place. ',
            10: 'No; when you are a guest, make your way to the lowest place and sit there, so that, when your host comes, he may say, "My friend, move up higher." Then, everyone with you at the table will see you honoured. ',
            11: "For everyone who raises himself up will be humbled, and the one who humbles himself will be raised up.' ",
            12: "Then he said to his host, 'When you give a lunch or a dinner, do not invite your friends or your brothers or your relations or rich neighbours, in case they invite you back and so repay you. ",
            13: "No; when you have a party, invite the poor, the crippled, the lame, the blind; ",
            14: "then you will be blessed, for they have no means to repay you and so you will be repaid when the upright rise again.' ",
            15: "On hearing this, one of those gathered round the table said to him, 'Blessed is anyone who will share the meal in the kingdom of God!' ",
            16: "But he said to him, 'There was a man who gave a great banquet, and he invited a large number of people. ",
            17: 'When the time for the banquet came, he sent his servant to say to those who had been invited, "Come along: everything is ready now." ',
            18: 'But all alike started to make excuses. The first said, "I have bought a piece of land and must go and see it. Please accept my apologies." ',
            19: 'Another said, "I have bought five yoke of oxen and am on my way to try them out. Please accept my apologies." ',
            20: 'Yet another said, "I have just got married and so am unable to come." ',
            21: """'The servant returned and reported this to his master. Then the householder, in a rage, said to his servant, "Go out quickly into the streets and alleys of the town and bring in here the poor, the crippled, the blind and the lame." """,
            22: '"Sir," said the servant, "your orders have been carried out and there is still room." ',
            23: 'Then the master said to his servant, "Go to the open roads and the hedgerows and press people to come in, to make sure my house is full; ',
            24: """because, I tell you, not one of those who were invited shall have a taste of my banquet." ' """,
            25: "Great crowds accompanied him on his way and he turned and spoke to them. ",
            26: "'Anyone who comes to me without hating father, mother, wife, children, brothers, sisters, yes and his own life too, cannot be my disciple. ",
            27: "No one who does not carry his cross and come after me can be my disciple. ",
            28: "'And indeed, which of you here, intending to build a tower, would not first sit down and work out the cost to see if he had enough to complete it? ",
            29: "Otherwise, if he laid the foundation and then found himself unable to finish the work, anyone who saw it would start making fun of him and saying, ",
            30: '"Here is someone who started to build and was unable to finish." ',
            31: "Or again, what king marching to war against another king would not first sit down and consider whether with ten thousand men he could stand up to the other who was advancing against him with twenty thousand? ",
            32: "If not, then while the other king was still a long way off, he would send envoys to sue for peace. ",
            33: "So in the same way, none of you can be my disciple without giving up all that he owns. ",
            34: "'Salt is a good thing. But if salt itself loses its taste, what can make it salty again? ",
            35: "It is good for neither soil nor manure heap. People throw it away. Anyone who has ears for listening should listen!'"
        }

        chapter_15 = {
            1: "The tax collectors and sinners, however, were all crowding round to listen to him, ",
            2: "and the Pharisees and scribes complained saying, 'This man welcomes sinners and eats with them.' ",
            3: "So he told them this parable: ",
            4: "'Which one of you with a hundred sheep, if he lost one, would fail to leave the ninety-nine in the desert and go after the missing one till he found it? ",
            5: "And when he found it, would he not joyfully take it on his shoulders ",
            6: 'and then, when he got home, call together his friends and neighbours, saying to them, "Rejoice with me, I have found my sheep that was lost." ',
            7: "In the same way, I tell you, there will be more rejoicing in heaven over one sinner repenting than over ninety-nine upright people who have no need of repentance. ",
            8: "'Or again, what woman with ten drachmas would not, if she lost one, light a lamp and sweep out the house and search thoroughly till she found it? ",
            9: 'And then, when she had found it, call together her friends and neighbours, saying to them, "Rejoice with me, I have found the drachma I lost." ',
            10: "In the same way, I tell you, there is rejoicing among the angels of God over one repentant sinner.' ",
            11: "Then he said, 'There was a man who had two sons. ",
            12: 'The younger one said to his father, "Father, let me have the share of the estate that will come to me." So the father divided the property between them. ',
            13: "A few days later, the younger son got together everything he had and left for a distant country where he squandered his money on a life of debauchery. ",
            14: "'When he had spent it all, that country experienced a severe famine, and now he began to feel the pinch; ",
            15: "so he hired himself out to one of the local inhabitants who put him on his farm to feed the pigs. ",
            16: "And he would willingly have filled himself with the husks the pigs were eating but no one would let him have them. ",
            17: '''Then he came to his senses and said, "How many of my father's hired men have all the food they want and more, and here am I dying of hunger! ''',
            18: "I will leave this place and go to my father and say: Father, I have sinned against heaven and against you;",
            19: 'I no longer deserve to be called your son; treat me as one of your hired men."',
            20: "So he left the place and went back to his father. 'While he was still a long way off, his father saw him and was moved with pity. He ran to the boy, clasped him in his arms and kissed him. ",
            21: 'Then his son said, "Father, I have sinned against heaven and against you. I no longer deserve to be called your son." ',
            22: 'But the father said to his servants, "Quick! Bring out the best robe and put it on him; put a ring on his finger and sandals on his feet. ',
            23: "Bring the calf we have been fattening, and kill it; we will celebrate by having a feast, ",
            24: 'because this son of mine was dead and has come back to life; he was lost and is found." And they began to celebrate. ',
            25: "'Now the elder son was out in the fields, and on his way back, as he drew near the house, he could hear music and dancing. ",
            26: "Calling one of the servants he asked what it was all about. ",
            27: 'The servant told him, "Your brother has come, and your father has killed the calf we had been fattening because he has got him back safe and sound." ',
            28: "He was angry then and refused to go in, and his father came out and began to urge him to come in; ",
            29: 'but he retorted to his father, "All these years I have slaved for you and never once disobeyed any orders of yours, yet you never offered me so much as a kid for me to celebrate with my friends. ',
            30: 'But, for this son of yours, when he comes back after swallowing up your property -- he and his loose women -- you kill the calf we had been fattening." ',
            31: """'The father said, "My son, you are with me always and all I have is yours. """,
            32: """But it was only right we should celebrate and rejoice, because your brother here was dead and has come to life; he was lost and is found." '"""
        }

        chapter_16 = {
            1: "He also said to his disciples, 'There was a rich man and he had a steward who was denounced to him for being wasteful with his property. ",
            2: 'He called for the man and said, "What is this I hear about you? Draw me up an account of your stewardship because you are not to be my steward any longer." ',
            3: 'Then the steward said to himself, "Now that my master is taking the stewardship from me, what am I to do? Dig? I am not strong enough. Go begging? I should be too ashamed. ',
            4: 'Ah, I know what I will do to make sure that when I am dismissed from office there will be some to welcome me into their homes." ',
            5: """'Then he called his master's debtors one by one. To the first he said, "How much do you owe my master?" """,
            6: '"One hundred measures of oil," he said. The steward said, "Here, take your bond; sit down and quickly write fifty." ',
            7: 'To another he said, "And you, sir, how much do you owe?" "One hundred measures of wheat," he said. The steward said, "Here, take your bond and write eighty." ',
            8: "'The master praised the dishonest steward for his astuteness. For the children of this world are more astute in dealing with their own kind than are the children of light.' ",
            9: "'And so I tell you this: use money, tainted as it is, to win you friends, and thus make sure that when it fails you, they will welcome you into eternal dwellings. ",
            10: "Anyone who is trustworthy in little things is trustworthy in great; anyone who is dishonest in little things is dishonest in great. ",
            11: "If then you are not trustworthy with money, that tainted thing, who will trust you with genuine riches? ",
            12: "And if you are not trustworthy with what is not yours, who will give you what is your very own? ",
            13: "'No servant can be the slave of two masters: he will either hate the first and love the second, or be attached to the first and despise the second. You cannot be the slave both of God and of money.' ",
            14: "The Pharisees, who loved money, heard all this and jeered at him. ",
            15: "He said to them, 'You are the very ones who pass yourselves off as upright in people's sight, but God knows your hearts. For what is highly esteemed in human eyes is loathsome in the sight of God. ",
            16: "'Up to the time of John it was the Law and the Prophets; from then onwards, the kingdom of God has been preached, and everyone is forcing their way into it. ",
            17: "'It is easier for heaven and earth to disappear than for one little stroke to drop out of the Law. ",
            18: "'Everyone who divorces his wife and marries another is guilty of adultery, and the man who marries a woman divorced by her husband commits adultery. ",
            19: "'There was a rich man who used to dress in purple and fine linen and feast magnificently every day. ",
            20: "And at his gate there used to lie a poor man called Lazarus, covered with sores, ",
            21: "who longed to fill himself with what fell from the rich man's table. Even dogs came and licked his sores. ",
            22: "Now it happened that the poor man died and was carried away by the angels into Abraham's embrace. The rich man also died and was buried. ",
            23: "'In his torment in Hades he looked up and saw Abraham a long way off with Lazarus in his embrace. ",
            24: 'So he cried out, "Father Abraham, pity me and send Lazarus to dip the tip of his finger in water and cool my tongue, for I am in agony in these flames." ',
            25: 'Abraham said, "My son, remember that during your life you had your fill of good things, just as Lazarus his fill of bad. Now he is being comforted here while you are in agony. ',
            26: 'But that is not all: between us and you a great gulf has been fixed, to prevent those who want to cross from our side to yours or from your side to ours." ',
            27: """'So he said, "Father, I beg you then to send Lazarus to my father's house, """,
            28: 'since I have five brothers, to give them warning so that they do not come to this place of torment too." ',
            29: 'Abraham said, "They have Moses and the prophets, let them listen to them." ',
            30: 'The rich man replied, "Ah no, father Abraham, but if someone comes to them from the dead, they will repent." ',
            31: 'Then Abraham said to him, "If they will not listen either to Moses or to the prophets, they will not be convinced even if someone should rise from the dead."'
        }

        chapter_17 = {
            1: "He said to his disciples, 'Causes of falling are sure to come, but alas for the one through whom they occur! ",
            2: "It would be better for such a person to be thrown into the sea with a millstone round the neck than to be the downfall of a single one of these little ones. ",
            3: "Keep watch on yourselves! 'If your brother does something wrong, rebuke him and, if he is sorry, forgive him. ",
            4: """And if he wrongs you seven times a day and seven times comes back to you and says, "I am sorry," you must forgive him.' """,
            5: "The apostles said to the Lord, 'Increase our faith.' ",
            6: """The Lord replied, 'If you had faith like a mustard seed you could say to this mulberry tree, "Be uprooted and planted in the sea," and it would obey you. """,
            7: """'Which of you, with a servant ploughing or minding sheep, would say to him when he returned from the fields, "Come and have your meal at once"? """,
            8: 'Would he not be more likely to say, "Get my supper ready; fasten your belt and wait on me while I eat and drink. You yourself can eat and drink afterwards"? ',
            9: "Must he be grateful to the servant for doing what he was told? ",
            10: """So with you: when you have done all you have been told to do, say, "We are useless servants: we have done no more than our duty." ' """,
            11: "Now it happened that on the way to Jerusalem he was travelling in the borderlands of Samaria and Galilee. ",
            12: "As he entered one of the villages, ten men suffering from a virulent skin-disease came to meet him. They stood some way off ",
            13: "and called to him, 'Jesus! Master! Take pity on us.' ",
            14: "When he saw them he said, 'Go and show yourselves to the priests.' Now as they were going away they were cleansed. ",
            15: "Finding himself cured, one of them turned back praising God at the top of his voice ",
            16: "and threw himself prostrate at the feet of Jesus and thanked him. The man was a Samaritan. ",
            17: "This led Jesus to say, 'Were not all ten made clean? The other nine, where are they? ",
            18: "It seems that no one has come back to give praise to God, except this foreigner.' ",
            19: "And he said to the man, 'Stand up and go on your way. Your faith has saved you.' ",
            20: "Asked by the Pharisees when the kingdom of God was to come, he gave them this answer, 'The coming of the kingdom of God does not admit of observation ",
            21: """and there will be no one to say, "Look, it is here! Look, it is there!" For look, the kingdom of God is among you.' """,
            22: "He said to the disciples, 'A time will come when you will long to see one of the days of the Son of man and will not see it. ",
            23: 'They will say to you, "Look, it is there!" or, "Look, it is here!" Make no move; do not set off in pursuit; ',
            24: "for as the lightning flashing from one part of heaven lights up the other, so will be the Son of man when his Day comes. ",
            25: "But first he is destined to suffer grievously and be rejected by this generation. ",
            26: "'As it was in Noah's day, so will it also be in the days of the Son of man. ",
            27: "People were eating and drinking, marrying wives and husbands, right up to the day Noah went into the ark, and the Flood came and destroyed them all. ",
            28: "It will be the same as it was in Lot's day: people were eating and drinking, buying and selling, planting and building, ",
            29: "but the day Lot left Sodom, it rained fire and brimstone from heaven and it destroyed them all. ",
            30: "It will be the same when the day comes for the Son of man to be revealed. ",
            31: "'When that Day comes, no one on the housetop, with his possessions in the house, must come down to collect them, nor must anyone in the fields turn back. ",
            32: "Remember Lot's wife. ",
            33: "Anyone who tries to preserve his life will lose it; and anyone who loses it will keep it safe. ",
            34: "I tell you, on that night, when two are in one bed, one will be taken, the other left;",
            35: "when two women are grinding corn together, one will be taken, the other left.' ",
            36: "",
            37: "The disciples spoke up and asked, 'Where, Lord?' He said, 'Where the body is, there too will the vultures gather.'"
        }

        chapter_18 = {
            1: "Then he told them a parable about the need to pray continually and never lose heart. ",
            2: "'There was a judge in a certain town,' he said, 'who had neither fear of God nor respect for anyone. ",
            3: 'In the same town there was also a widow who kept on coming to him and saying, "I want justice from you against my enemy!" ',
            4: 'For a long time he refused, but at last he said to himself, "Even though I have neither fear of God nor respect for any human person, ',
            5: """I must give this widow her just rights since she keeps pestering me, or she will come and slap me in the face." '""",
            6: "And the Lord said, 'You notice what the unjust judge has to say? ",
            7: "Now, will not God see justice done to his elect if they keep calling to him day and night even though he still delays to help them? ",
            8: "I promise you, he will see justice done to them, and done speedily. But when the Son of man comes, will he find any faith on earth?'",
            9: "He spoke the following parable to some people who prided themselves on being upright and despised everyone else, ",
            10: "'Two men went up to the Temple to pray, one a Pharisee, the other a tax collector. ",
            11: 'The Pharisee stood there and said this prayer to himself, "I thank you, God, that I am not grasping, unjust, adulterous like everyone else, and particularly that I am not like this tax collector here. ',
            12: 'I fast twice a week; I pay tithes on all I get."',
            13: 'The tax collector stood some distance away, not daring even to raise his eyes to heaven; but he beat his breast and said, "God, be merciful to me, a sinner." ',
            14: "This man, I tell you, went home again justified; the other did not. For everyone who raises himself up will be humbled, but anyone who humbles himself will be raised up.' ",
            15: "People even brought babies to him, for him to touch them; but when the disciples saw this they scolded them. ",
            16: "But Jesus called the children to him and said, 'Let the little children come to me, and do not stop them; for it is to such as these that the kingdom of God belongs. ",
            17: "In truth I tell you, anyone who does not welcome the kingdom of God like a little child will never enter it.' ",
            18: "One of the rulers put this question to him, 'Good Master, what shall I do to inherit eternal life?' ",
            19: "Jesus said to him, 'Why do you call me good? No one is good but God alone. ",
            20: "You know the commandments: You shall not commit adultery; You shall not kill; You shall not steal; You shall not give false witness; Honour your father and your mother.' ",
            21: "He replied, 'I have kept all these since my earliest days.' ",
            22: "And when Jesus heard this he said, 'There is still one thing you lack. Sell everything you own and distribute the money to the poor, and you will have treasure in heaven; then come, follow me.' ",
            23: "But when he heard this he was overcome with sadness, for he was very rich. ",
            24: "Jesus looked at him and said, 'How hard it is for those who have riches to make their way into the kingdom of God! ",
            25: "Yes, it is easier for a camel to pass through the eye of a needle than for someone rich to enter the kingdom of God.' ",
            26: "Those who were listening said, 'In that case, who can be saved?' ",
            27: "He replied, 'Things that are impossible by human resources, are possible for God.' ",
            28: "But Peter said, 'Look, we left all we had to follow you.' ",
            29: "He said to them, 'In truth I tell you, there is no one who has left house, wife, brothers, parents or children for the sake of the kingdom of God ",
            30: "who will not receive many times as much in this present age and, in the world to come, eternal life.' ",
            31: "Then taking the Twelve aside he said to them, 'Look, we are going up to Jerusalem, and everything that is written by the prophets about the Son of man is to come true. ",
            32: "For he will be handed over to the gentiles and will be mocked, maltreated and spat on, ",
            33: "and when they have scourged him they will put him to death; and on the third day he will rise again.' ",
            34: "But they could make nothing of this; what he said was quite obscure to them, they did not understand what he was telling them. ",
            35: "Now it happened that as he drew near to Jericho there was a blind man sitting at the side of the road begging. ",
            36: "When he heard the crowd going past he asked what it was all about, ",
            37: "and they told him that Jesus the Nazarene was passing by. ",
            38: "So he called out, 'Jesus, Son of David, have pity on me.' ",
            39: "The people in front scolded him and told him to keep quiet, but he only shouted all the louder, 'Son of David, have pity on me.' ",
            40: "Jesus stopped and ordered them to bring the man to him, and when he came up, asked him, ",
            41: "'What do you want me to do for you?' 'Sir,' he replied, 'let me see again.' ",
            42: "Jesus said to him, 'Receive your sight. Your faith has saved you.' ",
            43: "And instantly his sight returned and he followed him praising God, and all the people who saw it gave praise to God."
        }

        chapter_19 = {
            1: "He entered Jericho and was going through the town ",
            2: "and suddenly a man whose name was Zacchaeus made his appearance; he was one of the senior tax collectors and a wealthy man. ",
            3: "He kept trying to see which Jesus was, but he was too short and could not see him for the crowd; ",
            4: "so he ran ahead and climbed a sycamore tree to catch a glimpse of Jesus who was to pass that way. ",
            5: "When Jesus reached the spot he looked up and spoke to him, 'Zacchaeus, come down. Hurry, because I am to stay at your house today.' ",
            6: "And he hurried down and welcomed him joyfully. ",
            7: "They all complained when they saw what was happening. 'He has gone to stay at a sinner's house,' they said. ",
            8: "But Zacchaeus stood his ground and said to the Lord, 'Look, sir, I am going to give half my property to the poor, and if I have cheated anybody I will pay him back four times the amount.' ",
            9: "And Jesus said to him, 'Today salvation has come to this house, because this man too is a son of Abraham; ",
            10: "for the Son of man has come to seek out and save what was lost.' ",
            11: "While the people were listening to this he went on to tell a parable, because he was near Jerusalem and they thought that the kingdom of God was going to show itself then and there. ",
            12: "Accordingly he said, 'A man of noble birth went to a distant country to be appointed king and then return. ",
            13: 'He summoned ten of his servants and gave them ten pounds, telling them, "Trade with these, until I get back." ',
            14: 'But his compatriots detested him and sent a delegation to follow him with this message, "We do not want this man to be our king." ',
            15: "'Now it happened that on his return, having received his appointment as king, he sent for those servants to whom he had given the money, to find out what profit each had made by trading. ",
            16: 'The first came in, "Sir," he said, "your one pound has brought in ten." ',
            17: 'He replied, "Well done, my good servant! Since you have proved yourself trustworthy in a very small thing, you shall have the government of ten cities." ',
            18: 'Then came the second, "Sir," he said, "your one pound has made five." ',
            19: 'To this one also he said, "And you shall be in charge of five cities." ',
            20: 'Next came the other, "Sir," he said, "here is your pound. I put it away safely wrapped up in a cloth ',
            21: 'because I was afraid of you; for you are an exacting man: you gather in what you have not laid out and reap what you have not sown." ',
            22: 'He said to him, "You wicked servant! Out of your own mouth I condemn you. So you knew that I was an exacting man, gathering in what I have not laid out and reaping what I have not sown? ',
            23: 'Then why did you not put my money in the bank? On my return I could have drawn it out with interest." ',
            24: 'And he said to those standing by, "Take the pound from him and give it to the man who has ten pounds." ',
            25: 'And they said to him, "But, sir, he has ten pounds . . ." ',
            26: '"I tell you, to everyone who has will be given more; but anyone who has not will be deprived even of what he has. ',
            27: '''"As for my enemies who did not want me for their king, bring them here and execute them in my presence." ' ''',
            28: "When he had said this he went on ahead, going up to Jerusalem. ",
            29: "Now it happened that when he was near Bethphage and Bethany, close by the Mount of Olives as it is called, he sent two of the disciples, saying, ",
            30: "'Go to the village opposite, and as you enter it you will find a tethered colt that no one has ever yet ridden. Untie it and bring it here. ",
            31: """If anyone asks you, "Why are you untying it?" you are to say this, "The Master needs it." ' """,
            32: "The messengers went off and found everything just as he had told them. ",
            33: "As they were untying the colt, its owners said, 'Why are you untying it?' ",
            34: "and they answered, 'The Master needs it.' ",
            35: "So they took the colt to Jesus and, throwing their cloaks on its back, they lifted Jesus on to it. ",
            36: "As he moved off, they spread their cloaks in the road, ",
            37: "and now, as he was approaching the downward slope of the Mount of Olives, the whole group of disciples joyfully began to praise God at the top of their voices for all the miracles they had seen. ",
            38: "They cried out: Blessed is he who is coming as King in the name of the Lord! Peace in heaven and glory in the highest heavens! ",
            39: "Some Pharisees in the crowd said to him, 'Master, reprove your disciples,' ",
            40: "but he answered, 'I tell you, if these keep silence, the stones will cry out.' ",
            41: "As he drew near and came in sight of the city he shed tears over it ",
            42: "and said, 'If you too had only recognised on this day the way to peace! But in fact it is hidden from your eyes! ",
            43: "Yes, a time is coming when your enemies will raise fortifications all round you, when they will encircle you and hem you in on every side; ",
            44: "they will dash you and the children inside your walls to the ground; they will leave not one stone standing on another within you, because you did not recognise the moment of your visitation.' ",
            45: "Then he went into the Temple and began driving out those who were busy trading, saying to them, ",
            46: "'According to scripture, my house shall be a house of prayer but you have turned it into a bandits' den.' ",
            47: "He taught in the Temple every day. The chief priests and the scribes, in company with the leading citizens, tried to do away with him, ",
            48: "but they could not find a way to carry this out because the whole people hung on his words."
        }

        chapter_20 = {
            1: "Now it happened that one day while he was teaching the people in the Temple and proclaiming the good news, the chief priests and the scribes came up, together with the elders, ",
            2: "and spoke to him. 'Tell us,' they said, 'what authority have you for acting like this? Or who gives you this authority?' ",
            3: "In reply he said to them, 'And I will ask you a question, just one. Tell me: ",
            4: "John's baptism: what was its origin, heavenly or human?' ",
            5: '''And they debated this way among themselves, 'If we say heavenly, he will retort, "Why did you refuse to believe him?"; ''',
            6: "and if we say human, the whole people will stone us, for they are convinced that John was a prophet.' ",
            7: "So their reply was that they did not know where it came from. ",
            8: "And Jesus said to them, 'Nor will I tell you my authority for acting like this.' ",
            9: "And he went on to tell the people this parable, 'A man planted a vineyard and leased it to tenants, and went abroad for a long while. ",
            10: "When the right time came, he sent a servant to the tenants to get his share of the produce of the vineyard. But the tenants thrashed him, and sent him away empty-handed. ",
            11: "But he went on to send a second servant; they thrashed him too and treated him shamefully and sent him away empty-handed. ",
            12: "He still went on to send a third; they wounded this one too, and threw him out. ",
            13.: 'Then the owner of the vineyard thought, "What am I to do? I will send them my own beloved son. Perhaps they will respect him." ',
            14: 'But when the tenants saw him they put their heads together saying, "This is the heir, let us kill him so that the inheritance will be ours." ',
            15: "So they threw him out of the vineyard and killed him. 'Now what will the owner of the vineyard do to them? ",
            16: "He will come and make an end of these tenants and give the vineyard to others.' Hearing this they said, 'God forbid!' ",
            17: "But he looked hard at them and said, 'Then what does this text in the scriptures mean: The stone which the builders rejected has become the cornerstone? ",
            18: "Anyone who falls on that stone will be dashed to pieces; anyone it falls on will be crushed.' ",
            19: "And the scribes and the chief priests would have liked to lay hands on him that very moment, because they realised that this parable was aimed at them, but they were afraid of the people. ",
            20: "So they awaited their opportunity and sent agents to pose as upright men, and to catch him out in something he might say and so enable them to hand him over to the jurisdiction and authority of the governor. ",
            21: "They put to him this question, 'Master, we know that you say and teach what is right; you favour no one, but teach the way of God in all honesty. ",
            22: "Is it permissible for us to pay taxes to Caesar or not?' ",
            23: "But he was aware of their cunning and said, ",
            24: "'Show me a denarius. Whose portrait and title are on it?' They said, 'Caesar's.' ",
            25: "He said to them, 'Well then, pay Caesar what belongs to Caesar -- and God what belongs to God.' ",
            26: "They were unable to catch him out in anything he had to say in public; they were amazed at his answer and were silenced. ",
            27: "Some Sadducees -- those who argue that there is no resurrection -- approached him and they put this question to him, ",
            28: "'Master, Moses prescribed for us, if a man's married brother dies childless, the man must marry the widow to raise up children for his brother. ",
            29: "Well then, there were seven brothers; the first, having married a wife, died childless. ",
            30: "The second ",
            31: "and then the third married the widow. And the same with all seven, they died leaving no children. ",
            32: "Finally the woman herself died. ",
            33: "Now, at the resurrection, whose wife will she be, since she had been married to all seven?' ",
            34: "Jesus replied, 'The children of this world take wives and husbands, ",
            35: "but those who are judged worthy of a place in the other world and in the resurrection from the dead do not marry ",
            36: "because they can no longer die, for they are the same as the angels, and being children of the resurrection they are children of God. ",
            37: "And Moses himself implies that the dead rise again, in the passage about the bush where he calls the Lord the God of Abraham, the God of Isaac and the God of Jacob. ",
            38: "Now he is God, not of the dead, but of the living; for to him everyone is alive.' ",
            39: "Some scribes then spoke up. They said, 'Well put, Master.' ",
            40: "They did not dare to ask him any more questions. ",
            41: "He then said to them, 'How can people maintain that the Christ is son of David? ",
            42: "Why, David himself says in the Book of Psalms: The Lord declared to my Lord, take your seat at my right hand, ",
            43: "till I have made your enemies your footstool. ",
            44: "David here calls him Lord; how then can he be his son?' ",
            45: "While all the people were listening he said to the disciples, ",
            46: "'Beware of the scribes who like to walk about in long robes and love to be greeted respectfully in the market squares, to take the front seats in the synagogues and the places of honour at banquets, ",
            47: "who devour the property of widows, and for show offer long prayers. The more severe will be the sentence they receive.'"
        }

        chapter_21 = {
            1: "Looking up, he saw rich people putting their offerings into the treasury; ",
            2: "and he noticed a poverty-stricken widow putting in two small coins, ",
            3: "and he said, 'I tell you truly, this poor widow has put in more than any of them; ",
            4: "for these have all put in money they could spare, but she in her poverty has put in all she had to live on.' ",
            5: "When some were talking about the Temple, remarking how it was adorned with fine stonework and votive offerings, he said, ",
            6: "'All these things you are staring at now -- the time will come when not a single stone will be left on another; everything will be destroyed.' ",
            7: "And they put to him this question, 'Master,' they said, 'when will this happen, then, and what sign will there be that it is about to take place?' ",
            8: """But he said, 'Take care not to be deceived, because many will come using my name and saying, "I am the one" and "The time is near at hand." Refuse to join them. """,
            9: "And when you hear of wars and revolutions, do not be terrified, for this is something that must happen first, but the end will not come at once.' ",
            10: "Then he said to them, 'Nation will fight against nation, and kingdom against kingdom. ",
            11: "There will be great earthquakes and plagues and famines in various places; there will be terrifying events and great signs from heaven. ",
            12: "'But before all this happens, you will be seized and persecuted; you will be handed over to the synagogues and to imprisonment, and brought before kings and governors for the sake of my name ",
            13: "-and that will be your opportunity to bear witness. ",
            14: "Make up your minds not to prepare your defence, ",
            15: "because I myself shall give you an eloquence and a wisdom that none of your opponents will be able to resist or contradict. ",
            16: "You will be betrayed even by parents and brothers, relations and friends; and some of you will be put to death. ",
            17: "You will be hated universally on account of my name, ",
            18: "but not a hair of your head will be lost. ",
            19: "Your perseverance will win you your lives. ",
            20: "'When you see Jerusalem surrounded by armies, then you must realise that it will soon be laid desolate. ",
            21: "Then those in Judaea must escape to the mountains, those inside the city must leave it, and those in country districts must not take refuge in it. ",
            22: "For this is the time of retribution when all that scripture says must be fulfilled. ",
            23: "Alas for those with child, or with babies at the breast, when those days come! ",
            24: "'For great misery will descend on the land and retribution on this people. They will fall by the edge of the sword and be led captive to every gentile country; and Jerusalem will be trampled down by the gentiles until their time is complete. ",
            25: "'There will be signs in the sun and moon and stars; on earth nations in agony, bewildered by the turmoil of the ocean and its waves; ",
            26: "men fainting away with terror and fear at what menaces the world, for the powers of heaven will be shaken. ",
            27: "And then they will see the Son of man coming in a cloud with power and great glory. ",
            28: "When these things begin to take place, stand erect, hold your heads high, because your liberation is near at hand.' ",
            29: "And he told them a parable, 'Look at the fig tree and indeed every tree. ",
            30: "As soon as you see them bud, you can see for yourselves that summer is now near. ",
            31: "So with you when you see these things happening: know that the kingdom of God is near. ",
            32: "In truth I tell you, before this generation has passed away all will have taken place. ",
            33: "Sky and earth will pass away, but my words will never pass away. ",
            34: "'Watch yourselves, or your hearts will be coarsened by debauchery and drunkenness and the cares of life, and that day will come upon you unexpectedly, ",
            35: "like a trap. For it will come down on all those living on the face of the earth. ",
            36: "Stay awake, praying at all times for the strength to survive all that is going to happen, and to hold your ground before the Son of man.' ",
            37: "All day long he would be in the Temple teaching, but would spend the night in the open on the hill called the Mount of Olives. ",
            38: "And from early morning the people thronged to him in the Temple to listen to him."
        }

        chapter_22 = {
            1: "The feast of Unleavened Bread, called the Passover, was now drawing near, ",
            2: "and the chief priests and the scribes were looking for some way of doing away with him, because they were afraid of the people. ",
            3: "Then Satan entered into Judas, surnamed Iscariot, who was one of the Twelve. ",
            4: "He approached the chief priests and the officers of the guard to discuss some way of handing Jesus over to them. ",
            5: "They were delighted and agreed to give him money. ",
            6: "He accepted and began to look for an opportunity to betray him to them without people knowing about it. ",
            7: "The day of Unleavened Bread came round, on which the Passover had to be sacrificed, ",
            8: "and he sent Peter and John, saying, 'Go and make the preparations for us to eat the Passover.' ",
            9: "They asked him, 'Where do you want us to prepare it?' ",
            10: "He said to them, 'Look, as you go into the city you will meet a man carrying a pitcher of water. Follow him into the house he enters ",
            11: 'and tell the owner of the house, "The Master says this to you: Where is the room for me to eat the Passover with my disciples?" ',
            12: "The man will show you a large upper room furnished with couches. Make the preparations there.' ",
            13: "They set off and found everything as he had told them and prepared the Passover. ",
            14: "When the time came he took his place at table, and the apostles with him. ",
            15: "And he said to them, 'I have ardently longed to eat this Passover with you before I suffer; ",
            16: "because, I tell you, I shall not eat it until it is fulfilled in the kingdom of God.' ",
            17: "Then, taking a cup, he gave thanks and said, 'Take this and share it among you, ",
            18: "because from now on, I tell you, I shall never again drink wine until the kingdom of God comes.' ",
            19: "Then he took bread, and when he had given thanks, he broke it and gave it to them, saying, 'This is my body given for you; do this in remembrance of me.' ",
            20: "He did the same with the cup after supper, and said, 'This cup is the new covenant in my blood poured out for you. ",
            21: "'But look, here with me on the table is the hand of the man who is betraying me. ",
            22: "The Son of man is indeed on the path which was decreed, but alas for that man by whom he is betrayed!' ",
            23: "And they began to ask one another which of them it could be who was to do this. ",
            24: "An argument also began between them about who should be reckoned the greatest; ",
            25: "but he said to them, 'Among the gentiles it is the kings who lord it over them, and those who have authority over them are given the title Benefactor. ",
            26: "With you this must not happen. No; the greatest among you must behave as if he were the youngest, the leader as if he were the one who serves. ",
            27: "For who is the greater: the one at table or the one who serves? The one at table, surely? Yet here am I among you as one who serves! ",
            28: "'You are the men who have stood by me faithfully in my trials; ",
            29: "and now I confer a kingdom on you, just as my Father conferred one on me: ",
            30: "you will eat and drink at my table in my kingdom, and you will sit on thrones to judge the twelve tribes of Israel. ",
            31: "'Simon, Simon! Look, Satan has got his wish to sift you all like wheat; ",
            32: "but I have prayed for you, Simon, that your faith may not fail, and once you have recovered, you in your turn must strengthen your brothers.' ",
            33: "'Lord,' he answered, 'I would be ready to go to prison with you, and to death.' ",
            34: "Jesus replied, 'I tell you, Peter, by the time the cock crows today you will have denied three times that you know me.' ",
            35: "He said to them, 'When I sent you out without purse or haversack or sandals, were you short of anything?' ",
            36: "'No, nothing,' they said. He said to them, 'But now if you have a purse, take it, and the same with a haversack; if you have no sword, sell your cloak and buy one, ",
            37: "because I tell you these words of scripture are destined to be fulfilled in me: He was counted as one of the rebellious. Yes, what it says about me is even now reaching its fulfilment.' ",
            38: "They said, 'Lord, here are two swords.' He said to them, 'That is enough!' ",
            39: "He then left to make his way as usual to the Mount of Olives, with the disciples following. ",
            40: "When he reached the place he said to them, 'Pray not to be put to the test.' ",
            41: "Then he withdrew from them, about a stone's throw away, and knelt down and prayed. ",
            42: "'Father,' he said, 'if you are willing, take this cup away from me. Nevertheless, let your will be done, not mine.' ",
            43: "Then an angel appeared to him, coming from heaven to give him strength. ",
            44: "In his anguish he prayed even more earnestly, and his sweat fell to the ground like great drops of blood. ",
            45: "When he rose from prayer he went to the disciples and found them sleeping for sheer grief. ",
            46: "And he said to them, 'Why are you asleep? Get up and pray not to be put to the test.' ",
            47: "Suddenly, while he was still speaking, a number of men appeared, and at the head of them the man called Judas, one of the Twelve, who went up to Jesus to kiss him. ",
            48: "Jesus said, 'Judas, are you betraying the Son of man with a kiss?' ",
            49: "His followers, seeing what was about to happen, said, 'Lord, shall we use our swords?' ",
            50: "And one of them struck the high priest's servant and cut off his right ear. ",
            51: "But at this Jesus said, 'That is enough.' And touching the man's ear he healed him. ",
            52: "Then Jesus said to the chief priests and captains of the Temple guard and elders who had come for him, 'Am I a bandit, that you had to set out with swords and clubs? ",
            53: "When I was among you in the Temple day after day you never made a move to lay hands on me. But this is your hour; this is the reign of darkness.' ",
            54: "They seized him then and led him away, and they took him to the high priest's house. Peter followed at a distance. ",
            55: "They had lit a fire in the middle of the courtyard and Peter sat down among them, ",
            56: "and as he was sitting there by the blaze a servant-girl saw him, peered at him, and said, 'This man was with him too.' ",
            57: "But he denied it. 'Woman, I do not know him,' he said. ",
            58: "Shortly afterwards someone else saw him and said, 'You are one of them too.' But Peter replied, 'I am not, my friend.' ",
            59: "About an hour later another man insisted, saying, 'This fellow was certainly with him. Why, he is a Galilean.' ",
            60: "Peter said, 'My friend, I do not know what you are talking about.' At that instant, while he was still speaking, the cock crowed, ",
            61: """and the Lord turned and looked straight at Peter, and Peter remembered the Lord's words when he had said to him, 'Before the cock crows today, you will have disowned me three times.' """,
            62: "And he went outside and wept bitterly. ",
            63: "Meanwhile the men who guarded Jesus were mocking and beating him. ",
            64: "They blindfolded him and questioned him, saying, 'Prophesy! Who hit you then?' ",
            65: "And they heaped many other insults on him. ",
            66: "When day broke there was a meeting of the elders of the people, the chief priests and scribes. He was brought before their council, ",
            67: "and they said to him, 'If you are the Christ, tell us.' He replied, 'If I tell you, you will not believe, ",
            68: "and if I question you, you will not answer. ",
            69: "But from now on, the Son of man will be seated at the right hand of the Power of God.' ",
            70: "They all said, 'So you are the Son of God then?' He answered, 'It is you who say I am.' ",
            71: "Then they said, 'Why do we need any evidence? We have heard it for ourselves from his own lips.'"
        }

        chapter_23 = {
            1: "The whole assembly then rose, and they brought him before Pilate. ",
            2: "They began their accusation by saying, 'We found this man inciting our people to revolt, opposing payment of the tribute to Caesar, and claiming to be Christ, a king.' ",
            3: "Pilate put to him this question, 'Are you the king of the Jews?' He replied, 'It is you who say it.' ",
            4: "Pilate then said to the chief priests and the crowd, 'I find no case against this man.' ",
            5: "But they persisted, 'He is inflaming the people with his teaching all over Judaea and all the way from Galilee, where he started, down to here.' ",
            6: "When Pilate heard this, he asked if the man were a Galilean; ",
            7: "and finding that he came under Herod's jurisdiction, he passed him over to Herod, who was also in Jerusalem at that time. ",
            8: "Herod was delighted to see Jesus; he had heard about him and had been wanting for a long time to set eyes on him; moreover, he was hoping to see some miracle worked by him. ",
            9: "So he questioned him at some length, but without getting any reply. ",
            10: "Meanwhile the chief priests and the scribes were there, vigorously pressing their accusations. ",
            11: "Then Herod, together with his guards, treated him with contempt and made fun of him; he put a rich cloak on him and sent him back to Pilate. ",
            12: "And though Herod and Pilate had been enemies before, they were reconciled that same day. ",
            13: "Pilate then summoned the chief priests and the leading men and the people. ",
            14: "He said to them, 'You brought this man before me as a popular agitator. Now I have gone into the matter myself in your presence and found no grounds in the man for any of the charges you bring against him. ",
            15: "Nor has Herod either, since he has sent him back to us. As you can see, the man has done nothing that deserves death, ",
            16: "so I shall have him flogged and then let him go.' ",
            17: "",
            18: "But as one man they howled, 'Away with him! Give us Barabbas!' ",
            19: "(This man had been thrown into prison because of a riot in the city and murder.) ",
            20: "In his desire to set Jesus free, Pilate addressed them again, ",
            21: "but they shouted back, 'Crucify him! Crucify him!' ",
            22: "And for the third time he spoke to them, 'But what harm has this man done? I have found no case against him that deserves death, so I shall have him flogged and then let him go.' ",
            23: "But they kept on shouting at the top of their voices, demanding that he should be crucified. And their shouts kept growing louder. ",
            24: "Pilate then gave his verdict: their demand was to be granted. ",
            25: "He released the man they asked for, who had been imprisoned because of rioting and murder, and handed Jesus over to them to deal with as they pleased. ",
            26: "As they were leading him away they seized on a man, Simon from Cyrene, who was coming in from the country, and made him shoulder the cross and carry it behind Jesus. ",
            27: "Large numbers of people followed him, and women too, who mourned and lamented for him. ",
            28: "But Jesus turned to them and said, 'Daughters of Jerusalem, do not weep for me; weep rather for yourselves and for your children. ",
            29: 'For look, the days are surely coming when people will say, "Blessed are those who are barren, the wombs that have never borne children, the breasts that have never suckled!" ',
            30: 'Then they will begin to say to the mountains, "Fall on us!"; to the hills, "Cover us!" ',
            31: "For if this is what is done to green wood, what will be done when the wood is dry?' ",
            32: "Now they were also leading out two others, criminals, to be executed with him. ",
            33: "When they reached the place called The Skull, there they crucified him and the two criminals, one on his right, the other on his left. ",
            34: "Jesus said, 'Father, forgive them; they do not know what they are doing.' Then they cast lots to share out his clothing. ",
            35: "The people stayed there watching. As for the leaders, they jeered at him with the words, 'He saved others, let him save himself if he is the Christ of God, the Chosen One.' ",
            36: "The soldiers mocked him too, coming up to him, offering him vinegar, ",
            37: "and saying, 'If you are the king of the Jews, save yourself.' ",
            38: "Above him there was an inscription: 'This is the King of the Jews'. ",
            39: "One of the criminals hanging there abused him: 'Are you not the Christ? Save yourself and us as well.' ",
            40: "But the other spoke up and rebuked him. 'Have you no fear of God at all?' he said. 'You got the same sentence as he did, ",
            41: "but in our case we deserved it: we are paying for what we did. But this man has done nothing wrong.' ",
            42: "Then he said, 'Jesus, remember me when you come into your kingdom.' ",
            43: "He answered him, 'In truth I tell you, today you will be with me in paradise.' ",
            44: "It was now about the sixth hour and the sun's light failed, so that darkness came over the whole land until the ninth hour. ",
            45: "The veil of the Sanctuary was torn right down the middle. ",
            46: "Jesus cried out in a loud voice saying, 'Father, into your hands I commit my spirit.' With these words he breathed his last. ",
            47: "When the centurion saw what had taken place, he gave praise to God and said, 'Truly, this was an upright man.' ",
            48: "And when all the crowds who had gathered for the spectacle saw what had happened, they went home beating their breasts. ",
            49: "All his friends stood at a distance; so also did the women who had accompanied him from Galilee and saw all this happen. ",
            50: "And now a member of the Council arrived, a good and upright man named Joseph. ",
            51: "He had not consented to what the others had planned and carried out. He came from Arimathaea, a Jewish town, and he lived in the hope of seeing the kingdom of God. ",
            52: "This man went to Pilate and asked for the body of Jesus. ",
            53: "He then took it down, wrapped it in a shroud and put it in a tomb which was hewn in stone and which had never held a body. ",
            54: "It was Preparation day and the Sabbath was beginning to grow light. ",
            55: "Meanwhile the women who had come from Galilee with Jesus were following behind. They took note of the tomb and how the body had been laid. ",
            56: "Then they returned and prepared spices and ointments. And on the Sabbath day they rested, as the Law required."
        }

        chapter_24 = {
            1: "On the first day of the week, at the first sign of dawn, they went to the tomb with the spices they had prepared. ",
            2: "They found that the stone had been rolled away from the tomb, ",
            3: "but on entering they could not find the body of the Lord Jesus. ",
            4: "As they stood there puzzled about this, two men in brilliant clothes suddenly appeared at their side. ",
            5: "Terrified, the women bowed their heads to the ground. But the two said to them, 'Why look among the dead for someone who is alive? ",
            6: "He is not here; he has risen. Remember what he told you when he was still in Galilee: ",
            7: "that the Son of man was destined to be handed over into the power of sinful men and be crucified, and rise again on the third day.' ",
            8: "And they remembered his words. ",
            9: "And they returned from the tomb and told all this to the Eleven and to all the others. ",
            10: "The women were Mary of Magdala, Joanna, and Mary the mother of James. And the other women with them also told the apostles, ",
            11: "but this story of theirs seemed pure nonsense, and they did not believe them. ",
            12: "Peter, however, went off to the tomb, running. He bent down and looked in and saw the linen cloths but nothing else; he then went back home, amazed at what had happened. ",
            13: "Now that very same day, two of them were on their way to a village called Emmaus, seven miles from Jerusalem, ",
            14: "and they were talking together about all that had happened. ",
            15: "And it happened that as they were talking together and discussing it, Jesus himself came up and walked by their side; ",
            16: "but their eyes were prevented from recognising him. ",
            17: "He said to them, 'What are all these things that you are discussing as you walk along?' They stopped, their faces downcast. ",
            18: "Then one of them, called Cleopas, answered him, 'You must be the only person staying in Jerusalem who does not know the things that have been happening there these last few days.' ",
            19: "He asked, 'What things?' They answered, 'All about Jesus of Nazareth, who showed himself a prophet powerful in action and speech before God and the whole people; ",
            20: "and how our chief priests and our leaders handed him over to be sentenced to death, and had him crucified. ",
            21: "Our own hope had been that he would be the one to set Israel free. And this is not all: two whole days have now gone by since it all happened; ",
            22: "and some women from our group have astounded us: they went to the tomb in the early morning, ",
            23: "and when they could not find the body, they came back to tell us they had seen a vision of angels who declared he was alive. ",
            24: "Some of our friends went to the tomb and found everything exactly as the women had reported, but of him they saw nothing.' ",
            25: "Then he said to them, 'You foolish men! So slow to believe all that the prophets have said! ",
            26: "Was it not necessary that the Christ should suffer before entering into his glory?' ",
            27: "Then, starting with Moses and going through all the prophets, he explained to them the passages throughout the scriptures that were about himself. ",
            28: "When they drew near to the village to which they were going, he made as if to go on; ",
            29: "but they pressed him to stay with them saying, 'It is nearly evening, and the day is almost over.' So he went in to stay with them. ",
            30: "Now while he was with them at table, he took the bread and said the blessing; then he broke it and handed it to them. ",
            31: "And their eyes were opened and they recognised him; but he had vanished from their sight. ",
            32: "Then they said to each other, 'Did not our hearts burn within us as he talked to us on the road and explained the scriptures to us?' ",
            33: "They set out that instant and returned to Jerusalem. There they found the Eleven assembled together with their companions, ",
            34: "who said to them, 'The Lord has indeed risen and has appeared to Simon.' ",
            35: "Then they told their story of what had happened on the road and how they had recognised him at the breaking of bread. ",
            36: "They were still talking about all this when he himself stood among them and said to them, 'Peace be with you!' ",
            37: "In a state of alarm and fright, they thought they were seeing a ghost. ",
            38: "But he said, 'Why are you so agitated, and why are these doubts stirring in your hearts? ",
            39: "See by my hands and my feet that it is I myself. Touch me and see for yourselves; a ghost has no flesh and bones as you can see I have.' ",
            40: "And as he said this he showed them his hands and his feet. ",
            41: "Their joy was so great that they still could not believe it, as they were dumbfounded; so he said to them, 'Have you anything here to eat?' ",
            42: "And they offered him a piece of grilled fish, ",
            43: "which he took and ate before their eyes. ",
            44: "Then he told them, 'This is what I meant when I said, while I was still with you, that everything written about me in the Law of Moses, in the Prophets and in the Psalms, was destined to be fulfilled.' ",
            45: "He then opened their minds to understand the scriptures, ",
            46: "and he said to them, 'So it is written that the Christ would suffer and on the third day rise from the dead, '",
            47: "and that, in his name, repentance for the forgiveness of sins would be preached to all nations, beginning from Jerusalem. ",
            48: "You are witnesses to this. ",
            49: "'And now I am sending upon you what the Father has promised. Stay in the city, then, until you are clothed with the power from on high.' ",
            50: "Then he took them out as far as the outskirts of Bethany, and raising his hands he blessed them. ",
            51: "Now as he blessed them, he withdrew from them and was carried up to heaven. ",
            52: "They worshipped him and then went back to Jerusalem full of joy; ",
            53: "and they were continually in the Temple praising God."
        }

        chapters = {
            1: chapter_1,
            2: chapter_2,
            3: chapter_3,
            4: chapter_4,
            5: chapter_5,
            6: chapter_6,
            7: chapter_7,
            8: chapter_8,
            9: chapter_9,
            10: chapter_10,
            11: chapter_11,
            12: chapter_12,
            13: chapter_13,
            14: chapter_14,
            15: chapter_15,
            16: chapter_16,
            17: chapter_17,
            18: chapter_18,
            19: chapter_19,
            20: chapter_20,
            21: chapter_21,
            22: chapter_22,
            23: chapter_23,
            24: chapter_24
        }

        return get_text(chapters, chapter, verse_start, verse_end, 24)

    def john(self, chapter: int, verse_start: int, verse_end: int = None) -> Union[str, list]:
        """
        :param chapter: The chapter number
        :type chapter: int

        :param verse_start: The first verse
        :type verse_start: int

        :param verse_end: Alternatively, an end verse can be passed or let it
        as None to return only one verse
        :type verse_end: int or None
        """
        chapter_1 = {
            1: "In the beginning was the Word: the Word was with God and the Word was God. ",
            2: "He was with God in the beginning. ",
            3: "Through him all things came into being, not one thing came into being except through him. ",
            4: "What has come into being in him was life, life that was the light of men; ",
            5: "and light shines in darkness, and darkness could not overpower it. ",
            6: "A man came, sent by God. His name was John. ",
            7: "He came as a witness, to bear witness to the light, so that everyone might believe through him. ",
            8: "He was not the light, he was to bear witness to the light. ",
            9: "The Word was the real light that gives light to everyone; he was coming into the world. ",
            10: "He was in the world that had come into being through him, and the world did not recognise him. ",
            11: "He came to his own and his own people did not accept him. ",
            12: "But to those who did accept him he gave power to become children of God, to those who believed in his name ",
            13: "who were born not from human stock or human desire or human will but from God himself. ",
            14: "The Word became flesh, he lived among us, and we saw his glory, the glory that he has from the Father as only Son of the Father, full of grace and truth. ",
            15: "John witnesses to him. He proclaims: 'This is the one of whom I said: He who comes after me has passed ahead of me because he existed before me.' ",
            16: "Indeed, from his fullness we have, all of us, received -- one gift replacing another, ",
            17: "for the Law was given through Moses, grace and truth have come through Jesus Christ. ",
            18: "No one has ever seen God; it is the only Son, who is close to the Father's heart, who has made him known. ",
            19: "This was the witness of John, when the Jews sent to him priests and Levites from Jerusalem to ask him, 'Who are you?' ",
            20: "He declared, he did not deny but declared, 'I am not the Christ.' ",
            21: "So they asked, 'Then are you Elijah?' He replied, 'I am not.' 'Are you the Prophet?' He answered, 'No.' ",
            22: "So they said to him, 'Who are you? We must take back an answer to those who sent us. What have you to say about yourself?' ",
            23: "So he said, 'I am, as Isaiah prophesied: A voice of one that cries in the desert: Prepare a way for the Lord. Make his paths straight!' ",
            24: "Now those who had been sent were Pharisees, ",
            25: "and they put this question to him, 'Why are you baptising if you are not the Christ, and not Elijah, and not the Prophet?' ",
            26: "John answered them, 'I baptise with water; but standing among you -- unknown to you- ",
            27: "is the one who is coming after me; and I am not fit to undo the strap of his sandal.' ",
            28: "This happened at Bethany, on the far side of the Jordan, where John was baptising. ",
            29: "The next day, he saw Jesus coming towards him and said, 'Look, there is the lamb of God that takes away the sin of the world. ",
            30: 'It was of him that I said, "Behind me comes one who has passed ahead of me because he existed before me." ',
            31: "I did not know him myself, and yet my purpose in coming to baptise with water was so that he might be revealed to Israel.'",
            32: "And John declared, 'I saw the Spirit come down on him like a dove from heaven and rest on him. ",
            33: 'I did not know him myself, but he who sent me to baptise with water had said to me, "The man on whom you see the Spirit come down and rest is the one who is to baptise with the Holy Spirit."',
            34: "I have seen and I testify that he is the Chosen One of God.'",
            35: "The next day as John stood there again with two of his disciples, Jesus went past, ",
            36: "and John looked towards him and said, 'Look, there is the lamb of God.' ",
            37: "And the two disciples heard what he said and followed Jesus. ",
            38: "Jesus turned round, saw them following and said, 'What do you want?' They answered, 'Rabbi' -- which means Teacher -- 'where do you live?' ",
            39: "He replied, 'Come and see'; so they went and saw where he lived, and stayed with him that day. It was about the tenth hour. ",
            40: "One of these two who became followers of Jesus after hearing what John had said was Andrew, the brother of Simon Peter. ",
            41: "The first thing Andrew did was to find his brother and say to him, 'We have found the Messiah' -- which means the Christ- ",
            42: "and he took Simon to Jesus. Jesus looked at him and said, 'You are Simon son of John; you are to be called Cephas' -- which means Rock. ",
            43: "The next day, after Jesus had decided to leave for Galilee, he met Philip and said, 'Follow me.' ",
            44: "Philip came from the same town, Bethsaida, as Andrew and Peter. ",
            45: "Philip found Nathanael and said to him, 'We have found him of whom Moses in the Law and the prophets wrote, Jesus son of Joseph, from Nazareth.' ",
            46: "Nathanael said to him, 'From Nazareth? Can anything good come from that place?' Philip replied, 'Come and see.' ",
            47: "When Jesus saw Nathanael coming he said of him, 'There, truly, is an Israelite in whom there is no deception.' ",
            48: "Nathanael asked, 'How do you know me?' Jesus replied, 'Before Philip came to call you, I saw you under the fig tree.' ",
            49: "Nathanael answered, 'Rabbi, you are the Son of God, you are the king of Israel.' ",
            50: "Jesus replied, 'You believe that just because I said: I saw you under the fig tree. You are going to see greater things than that.' ",
            51: "And then he added, 'In all truth I tell you, you will see heaven open and the angels of God ascending and descending over the Son of man.' "
        }

        chapter_2 = {
            1: "On the third day there was a wedding at Cana in Galilee. The mother of Jesus was there, ",
            2: "and Jesus and his disciples had also been invited. ",
            3: "And they ran out of wine, since the wine provided for the feast had all been used, and the mother of Jesus said to him, 'They have no wine.' ",
            4: "Jesus said, 'Woman, what do you want from me? My hour has not come yet.' ",
            5: "His mother said to the servants, 'Do whatever he tells you.' ",
            6: "There were six stone water jars standing there, meant for the ablutions that are customary among the Jews: each could hold twenty or thirty gallons. ",
            7: "Jesus said to the servants, 'Fill the jars with water,' and they filled them to the brim. ",
            8: "Then he said to them, 'Draw some out now and take it to the president of the feast.' ",
            9: "They did this; the president tasted the water, and it had turned into wine. Having no idea where it came from -- though the servants who had drawn the water knew -- the president of the feast called the bridegroom ",
            10: "and said, 'Everyone serves good wine first and the worse wine when the guests are well wined; but you have kept the best wine till now.' ",
            11: "This was the first of Jesus' signs: it was at Cana in Galilee. He revealed his glory, and his disciples believed in him. ",
            12: "After this he went down to Capernaum with his mother and his brothers and his disciples, but they stayed there only a few days. ",
            13: "When the time of the Jewish Passover was near Jesus went up to Jerusalem, ",
            14: "and in the Temple he found people selling cattle and sheep and doves, and the money changers sitting there. ",
            15: "Making a whip out of cord, he drove them all out of the Temple, sheep and cattle as well, scattered the money changers' coins, knocked their tables over ",
            16: "and said to the dove sellers, 'Take all this out of here and stop using my Father's house as a market.' ",
            17: "Then his disciples remembered the words of scripture: I am eaten up with zeal for your house. ",
            18: "The Jews intervened and said, 'What sign can you show us that you should act like this?' ",
            19: "Jesus answered, 'Destroy this Temple, and in three days I will raise it up.' ",
            20: "The Jews replied, 'It has taken forty-six years to build this Temple: are you going to raise it up again in three days?' ",
            21: "But he was speaking of the Temple that was his body, ",
            22: "and when Jesus rose from the dead, his disciples remembered that he had said this, and they believed the scripture and what he had said. ",
            23: "During his stay in Jerusalem for the feast of the Passover many believed in his name when they saw the signs that he did, ",
            24: "but Jesus knew all people and did not trust himself to them; ",
            25: "he never needed evidence about anyone; he could tell what someone had within."
        }

        chapter_3 = {
            1: "There was one of the Pharisees called Nicodemus, a leader of the Jews, ",
            2: "who came to Jesus by night and said, 'Rabbi, we know that you have come from God as a teacher; for no one could perform the signs that you do unless God were with him.' ",
            3: "Jesus answered: In all truth I tell you, no one can see the kingdom of God without being born from above. ",
            4: "Nicodemus said, 'How can anyone who is already old be born? Is it possible to go back into the womb again and be born?' ",
            5: "Jesus replied: In all truth I tell you, no one can enter the kingdom of God without being born through water and the Spirit; ",
            6: "what is born of human nature is human; what is born of the Spirit is spirit. ",
            7: "Do not be surprised when I say: You must be born from above. ",
            8: "The wind blows where it pleases; you can hear its sound, but you cannot tell where it comes from or where it is going. So it is with everyone who is born of the Spirit. ",
            9: "'How is that possible?' asked Nicodemus. ",
            10: "Jesus replied, 'You are the Teacher of Israel, and you do not know these things! ",
            11: "'In all truth I tell you, we speak only about what we know and witness only to what we have seen and yet you people reject our evidence. ",
            12: "If you do not believe me when I speak to you about earthly things, how will you believe me when I speak to you about heavenly things? ",
            13: "No one has gone up to heaven except the one who came down from heaven, the Son of man; ",
            14: "as Moses lifted up the snake in the desert, so must the Son of man be lifted up ",
            15: "so that everyone who believes may have eternal life in him. ",
            16: "For this is how God loved the world: he gave his only Son, so that everyone who believes in him may not perish but may have eternal life. ",
            17: "For God sent his Son into the world not to judge the world, but so that through him the world might be saved. ",
            18: "No one who believes in him will be judged; but whoever does not believe is judged already, because that person does not believe in the Name of God's only Son. ",
            19: "And the judgement is this: though the light has come into the world people have preferred darkness to the light because their deeds were evil. ",
            20: "And indeed, everybody who does wrong hates the light and avoids it, to prevent his actions from being shown up; ",
            21: "but whoever does the truth comes out into the light, so that what he is doing may plainly appear as done in God.' ",
            22: "After this, Jesus went with his disciples into the Judaean countryside and stayed with them there and baptised. ",
            23: "John also was baptising at Aenon near Salim, where there was plenty of water, and people were going there and were being baptised. ",
            24: "For John had not yet been put in prison. ",
            25: "Now a discussion arose between some of John's disciples and a Jew about purification, ",
            26: "so they went to John and said, 'Rabbi, the man who was with you on the far side of the Jordan, the man to whom you bore witness, is baptising now, and everyone is going to him.' ",
            27: "John replied: 'No one can have anything except what is given him from heaven. ",
            28: """'You yourselves can bear me out. I said, "I am not the Christ; I am the one who has been sent to go in front of him." """,
            29: "'It is the bridegroom who has the bride; and yet the bridegroom's friend, who stands there and listens to him, is filled with joy at the bridegroom's voice. This is the joy I feel, and it is complete. ",
            30: "He must grow greater, I must grow less. ",
            31: "He who comes from above is above all others; he who is of the earth is earthly himself and speaks in an earthly way. He who comes from heaven ",
            32: "bears witness to the things he has seen and heard, but his testimony is not accepted by anybody; ",
            33: "though anyone who does accept his testimony is attesting that God is true, ",
            34: "since he whom God has sent speaks God's own words, for God gives him the Spirit without reserve. ",
            35: "The Father loves the Son and has entrusted everything to his hands. ",
            36: "Anyone who believes in the Son has eternal life, but anyone who refuses to believe in the Son will never see life: God's retribution hangs over him.'"
        }

        chapter_4 = {
            1: "When Jesus heard that the Pharisees had found out that he was making and baptising more disciples than John- ",
            2: "though in fact it was his disciples who baptised, not Jesus himself- ",
            3: "he left Judaea and went back to Galilee. ",
            4: "He had to pass through Samaria. ",
            5: "On the way he came to the Samaritan town called Sychar near the land that Jacob gave to his son Joseph. ",
            6: "Jacob's well was there and Jesus, tired by the journey, sat down by the well. It was about the sixth hour. ",
            7: "When a Samaritan woman came to draw water, Jesus said to her, 'Give me something to drink.' ",
            8: "His disciples had gone into the town to buy food. ",
            9: "The Samaritan woman said to him, 'You are a Jew. How is it that you ask me, a Samaritan, for something to drink?' -- Jews, of course, do not associate with Samaritans. ",
            10: "Jesus replied to her: If you only knew what God is offering and who it is that is saying to you, 'Give me something to drink,' you would have been the one to ask, and he would have given you living water. ",
            11: "'You have no bucket, sir,' she answered, 'and the well is deep: how do you get this living water? ",
            12: "Are you a greater man than our father Jacob, who gave us this well and drank from it himself with his sons and his cattle?' ",
            13: "Jesus replied: Whoever drinks this water will be thirsty again; ",
            14: "but no one who drinks the water that I shall give will ever be thirsty again: the water that I shall give will become a spring of water within, welling up for eternal life. ",
            15: "'Sir,' said the woman, 'give me some of that water, so that I may never be thirsty or come here again to draw water.' ",
            16: "'Go and call your husband,' said Jesus to her, 'and come back here.' ",
            17: """The woman answered, 'I have no husband.' Jesus said to her, 'You are right to say, "I have no husband"; """,
            18: "for although you have had five, the one you now have is not your husband. You spoke the truth there.' ",
            19: "'I see you are a prophet, sir,' said the woman. ",
            20: "'Our fathers worshipped on this mountain, though you say that Jerusalem is the place where one ought to worship.' ",
            21: "Jesus said: Believe me, woman, the hour is coming when you will worship the Father neither on this mountain nor in Jerusalem. ",
            22: "You worship what you do not know; we worship what we do know; for salvation comes from the Jews. ",
            23: "But the hour is coming -- indeed is already here -- when true worshippers will worship the Father in spirit and truth: that is the kind of worshipper the Father seeks. ",
            24: "God is spirit, and those who worship must worship in spirit and truth. ",
            25: "The woman said to him, 'I know that Messiah -- that is, Christ -- is coming; and when he comes he will explain everything.' ",
            26: "Jesus said, 'That is who I am, I who speak to you.' ",
            27: "At this point his disciples returned and were surprised to find him speaking to a woman, though none of them asked, 'What do you want from her?' or, 'What are you talking to her about?' ",
            28: "The woman put down her water jar and hurried back to the town to tell the people, ",
            29: "'Come and see a man who has told me everything I have done; could this be the Christ?' ",
            30: "This brought people out of the town and they made their way towards him. ",
            31: "Meanwhile, the disciples were urging him, 'Rabbi, do have something to eat'; ",
            32: "but he said, 'I have food to eat that you do not know about.' ",
            33: "So the disciples said to one another, 'Has someone brought him food?' ",
            34: "But Jesus said: My food is to do the will of the one who sent me, and to complete his work. ",
            35: "Do you not have a saying: Four months and then the harvest? Well, I tell you, look around you, look at the fields; already they are white, ready for harvest! ",
            36: "Already the reaper is being paid his wages, already he is bringing in the grain for eternal life, so that sower and reaper can rejoice together. ",
            37: "For here the proverb holds true: one sows, another reaps; ",
            38: "I sent you to reap a harvest you have not laboured for. Others have laboured for it; and you have come into the rewards of their labour.",
            39: "Many Samaritans of that town believed in him on the strength of the woman's words of testimony, 'He told me everything I have done.' ",
            40: "So, when the Samaritans came up to him, they begged him to stay with them. He stayed for two days, and ",
            41: "many more came to believe on the strength of the words he spoke to them; ",
            42: "and they said to the woman, 'Now we believe no longer because of what you told us; we have heard him ourselves and we know that he is indeed the Saviour of the world.' ",
            43: "When the two days were over Jesus left for Galilee. ",
            44: "He himself had declared that a prophet is not honoured in his own home town. ",
            45: "On his arrival the Galileans received him well, having seen all that he had done at Jerusalem during the festival which they too had attended. ",
            46: "He went again to Cana in Galilee, where he had changed the water into wine. And there was a royal official whose son was ill at Capernaum; ",
            47: "hearing that Jesus had arrived in Galilee from Judaea, he went and asked him to come and cure his son, as he was at the point of death. ",
            48: "Jesus said to him, 'Unless you see signs and portents you will not believe!' ",
            49: "'Sir,' answered the official, 'come down before my child dies.' ",
            50: "'Go home,' said Jesus, 'your son will live.' The man believed what Jesus had said and went on his way home; ",
            51: "and while he was still on the way his servants met him with the news that his boy was alive. ",
            52: "He asked them when the boy had begun to recover. They replied, 'The fever left him yesterday at the seventh hour.' ",
            53: "The father realised that this was exactly the time when Jesus had said, 'Your son will live'; and he and all his household believed. ",
            54: "This new sign, the second, Jesus performed on his return from Judaea to Galilee."
        }

        chapter_5 = {
            1: "After this there was a Jewish festival, and Jesus went up to Jerusalem. ",
            2: "Now in Jerusalem next to the Sheep Pool there is a pool called Bethesda in Hebrew, which has five porticos; ",
            3: "and under these were crowds of sick people, blind, lame, paralysed. ",
            4: "",
            5: "One man there had an illness which had lasted thirty-eight years, ",
            6: "and when Jesus saw him lying there and knew he had been in that condition for a long time, he said, 'Do you want to be well again?' ",
            7: "'Sir,' replied the sick man, 'I have no one to put me into the pool when the water is disturbed; and while I am still on the way, someone else gets down there before me.' ",
            8: "Jesus said, 'Get up, pick up your sleeping-mat and walk around.' ",
            9: "The man was cured at once, and he picked up his mat and started to walk around. Now that day happened to be the Sabbath, ",
            10: "so the Jews said to the man who had been cured, 'It is the Sabbath; you are not allowed to carry your sleeping-mat.' ",
            11: """He replied, 'But the man who cured me told me, "Pick up your sleeping-mat and walk around." ' """,
            12: """They asked, 'Who is the man who said to you, "Pick up your sleeping-mat and walk around"? ' """,
            13: "The man had no idea who it was, since Jesus had disappeared, as the place was crowded. ",
            14: "After a while Jesus met him in the Temple and said, 'Now you are well again, do not sin any more, or something worse may happen to you.' ",
            15: "The man went back and told the Jews that it was Jesus who had cured him. ",
            16: "It was because he did things like this on the Sabbath that the Jews began to harass Jesus. ",
            17: "His answer to them was, 'My Father still goes on working, and I am at work, too.' ",
            18: "But that only made the Jews even more intent on killing him, because not only was he breaking the Sabbath, but he spoke of God as his own Father and so made himself God's equal. ",
            19: "To this Jesus replied: In all truth I tell you, by himself the Son can do nothing; he can do only what he sees the Father doing: and whatever the Father does the Son does too. ",
            20: "For the Father loves the Son and shows him everything he himself does, and he will show him even greater things than these, works that will astonish you. ",
            21: "Thus, as the Father raises the dead and gives them life, so the Son gives life to anyone he chooses; ",
            22: "for the Father judges no one; he has entrusted all judgement to the Son, ",
            23: "so that all may honour the Son as they honour the Father. Whoever refuses honour to the Son refuses honour to the Father who sent him. ",
            24: "In all truth I tell you, whoever listens to my words, and believes in the one who sent me, has eternal life; without being brought to judgement such a person has passed from death to life. ",
            25: "In all truth I tell you, the hour is coming -- indeed it is already here -- when the dead will hear the voice of the Son of God, and all who hear it will live. ",
            26: "For as the Father has life in himself, so he has granted the Son also to have life in himself; ",
            27: "and, because he is the Son of man, has granted him power to give judgement. ",
            28: "Do not be surprised at this, for the hour is coming when the dead will leave their graves at the sound of his voice: ",
            29: "those who did good will come forth to life; and those who did evil will come forth to judgement. ",
            30: "By myself I can do nothing; I can judge only as I am told to judge, and my judging is just, because I seek to do not my own will but the will of him who sent me. ",
            31: "Were I to testify on my own behalf, my testimony would not be true; ",
            32: "but there is another witness who speaks on my behalf, and I know that his testimony is true. ",
            33: "You sent messengers to John, and he gave his testimony to the truth- ",
            34: "not that I depend on human testimony; no, it is for your salvation that I mention it. ",
            35: "John was a lamp lit and shining and for a time you were content to enjoy the light that he gave. ",
            36: "But my testimony is greater than John's: the deeds my Father has given me to perform, these same deeds of mine testify that the Father has sent me. ",
            37: "Besides, the Father who sent me bears witness to me himself. You have never heard his voice, you have never seen his shape, ",
            38: "and his word finds no home in you because you do not believe in the one whom he has sent. ",
            39: "You pore over the scriptures, believing that in them you can find eternal life; it is these scriptures that testify to me, ",
            40: "and yet you refuse to come to me to receive life! ",
            41: "Human glory means nothing to me. ",
            42: "Besides, I know you too well: you have no love of God in you. ",
            43: "I have come in the name of my Father and you refuse to accept me; if someone else should come in his own name you would accept him.",
            44: "How can you believe, since you look to each other for glory and are not concerned with the glory that comes from the one God? ",
            45: "Do not imagine that I am going to accuse you before the Father: you have placed your hopes on Moses, and Moses will be the one who accuses you. ",
            46: "If you really believed him you would believe me too, since it was about me that he was writing; 47.but if you will not believe what he wrote, how can you believe what I say?"
        }

        chapter_6 = {
            1: "After this, Jesus crossed the Sea of Galilee -- or of Tiberias- ",
            2: "and a large crowd followed him, impressed by the signs he had done in curing the sick. ",
            3: "Jesus climbed the hillside and sat down there with his disciples. ",
            4: "The time of the Jewish Passover was near. ",
            5: "Looking up, Jesus saw the crowds approaching and said to Philip, 'Where can we buy some bread for these people to eat?' ",
            6: "He said this only to put Philip to the test; he himself knew exactly what he was going to do. ",
            7: "Philip answered, 'Two hundred denarii would not buy enough to give them a little piece each.' ",
            8: "One of his disciples, Andrew, Simon Peter's brother, said, ",
            9: "'Here is a small boy with five barley loaves and two fish; but what is that among so many?' ",
            10: "Jesus said to them, 'Make the people sit down.' There was plenty of grass there, and as many as five thousand men sat down. ",
            11: "Then Jesus took the loaves, gave thanks, and distributed them to those who were sitting there; he then did the same with the fish, distributing as much as they wanted. ",
            12: "When they had eaten enough he said to the disciples, 'Pick up the pieces left over, so that nothing is wasted.' ",
            13: "So they picked them up and filled twelve large baskets with scraps left over from the meal of five barley loaves. ",
            14: "Seeing the sign that he had done, the people said, 'This is indeed the prophet who is to come into the world.' ",
            15: "Jesus, as he realised they were about to come and take him by force and make him king, fled back to the hills alone. ",
            16: "That evening the disciples went down to the shore of the sea ",
            17: "and got into a boat to make for Capernaum on the other side of the sea. It was getting dark by now and Jesus had still not rejoined them. ",
            18: "The wind was strong, and the sea was getting rough. ",
            19: "They had rowed three or four miles when they saw Jesus walking on the sea and coming towards the boat. They were afraid, ",
            20: "but he said, 'It's me. Don't be afraid.' ",
            21: "They were ready to take him into the boat, and immediately it reached the shore at the place they were making for. ",
            22: "Next day, the crowd that had stayed on the other side saw that only one boat had been there, and that Jesus had not got into the boat with his disciples, but that the disciples had set off by themselves. ",
            23: "Other boats, however, had put in from Tiberias, near the place where the bread had been eaten. ",
            24: "When the people saw that neither Jesus nor his disciples were there, they got into those boats and crossed to Capernaum to look for Jesus. ",
            25: "When they found him on the other side, they said to him, 'Rabbi, when did you come here?' ",
            26: "Jesus answered: In all truth I tell you, you are looking for me not because you have seen the signs but because you had all the bread you wanted to eat. ",
            27: "Do not work for food that goes bad, but work for food that endures for eternal life, which the Son of man will give you, for on him the Father, God himself, has set his seal. ",
            28: "Then they said to him, 'What must we do if we are to carry out God's work?' ",
            29: "Jesus gave them this answer, 'This is carrying out God's work: you must believe in the one he has sent.' ",
            30: "So they said, 'What sign will you yourself do, the sight of which will make us believe in you? What work will you do? ",
            31: "Our fathers ate manna in the desert; as scripture says: He gave them bread from heaven to eat.' ",
            32: "Jesus answered them: In all truth I tell you, it was not Moses who gave you the bread from heaven, it is my Father who gives you the bread from heaven, the true bread; ",
            33: "for the bread of God is the bread which comes down from heaven and gives life to the world. ",
            34: "'Sir,' they said, 'give us that bread always.' ",
            35: "Jesus answered them: I am the bread of life. No one who comes to me will ever hunger; no one who believes in me will ever thirst. ",
            36: "But, as I have told you, you can see me and still you do not believe. ",
            37: "Everyone whom the Father gives me will come to me; I will certainly not reject anyone who comes to me, ",
            38: "because I have come from heaven, not to do my own will, but to do the will of him who sent me. ",
            39: "Now the will of him who sent me is that I should lose nothing of all that he has given to me, but that I should raise it up on the last day. ",
            40: "It is my Father's will that whoever sees the Son and believes in him should have eternal life, and that I should raise that person up on the last day. ",
            41: "Meanwhile the Jews were complaining to each other about him, because he had said, 'I am the bread that has come down from heaven.' ",
            42: """They were saying, 'Surely this is Jesus son of Joseph, whose father and mother we know. How can he now say, "I have come down from heaven?" ' """,
            43: "Jesus said in reply to them, 'Stop complaining to each other. ",
            44: "'No one can come to me unless drawn by the Father who sent me, and I will raise that person up on the last day. ",
            45: "It is written in the prophets: They will all be taught by God; everyone who has listened to the Father, and learnt from him, comes to me. ",
            46: "Not that anybody has seen the Father, except him who has his being from God: he has seen the Father. ",
            47: "In all truth I tell you, everyone who believes has eternal life. ",
            48: "I am the bread of life.",
            49: "Your fathers ate manna in the desert and they are dead; ",
            50: "but this is the bread which comes down from heaven, so that a person may eat it and not die. ",
            51: "I am the living bread which has come down from heaven. Anyone who eats this bread will live for ever; and the bread that I shall give is my flesh, for the life of the world.'",
            52: "Then the Jews started arguing among themselves, 'How can this man give us his flesh to eat?' ",
            53: "Jesus replied to them: In all truth I tell you, if you do not eat the flesh of the Son of man and drink his blood, you have no life in you. ",
            54: "Anyone who does eat my flesh and drink my blood has eternal life, and I shall raise that person up on the last day. ",
            55: "For my flesh is real food and my blood is real drink. ",
            56: "Whoever eats my flesh and drinks my blood lives in me and I live in that person. ",
            57: "As the living Father sent me and I draw life from the Father, so whoever eats me will also draw life from me. ",
            58: "This is the bread which has come down from heaven; it is not like the bread our ancestors ate: they are dead, but anyone who eats this bread will live for ever. ",
            59: "This is what he taught at Capernaum in the synagogue. ",
            60: "After hearing it, many of his followers said, 'This is intolerable language. How could anyone accept it?' ",
            61: "Jesus was aware that his followers were complaining about it and said, 'Does this disturb you? ",
            62: "What if you should see the Son of man ascend to where he was before? ",
            63: "'It is the spirit that gives life, the flesh has nothing to offer. The words I have spoken to you are spirit and they are life. ",
            64: "'But there are some of you who do not believe.' For Jesus knew from the outset who did not believe and who was to betray him. ",
            65: "He went on, 'This is why I told you that no one could come to me except by the gift of the Father.' ",
            66: "After this, many of his disciples went away and accompanied him no more. ",
            67: "Then Jesus said to the Twelve, 'What about you, do you want to go away too?' ",
            68: "Simon Peter answered, 'Lord, to whom shall we go? You have the message of eternal life, ",
            69: "and we believe; we have come to know that you are the Holy One of God.' ",
            70: "Jesus replied to them, 'Did I not choose the Twelve of you? Yet one of you is a devil.' ",
            71: "He meant Judas son of Simon Iscariot, since this was the man, one of the Twelve, who was to betray him."
        }

        chapter_7 = {
            1: "After this Jesus travelled round Galilee; he could not travel round Judaea, because the Jews were seeking to kill him. ",
            2: "As the Jewish feast of Shelters drew near, ",
            3: "his brothers said to him, 'Leave this place and go to Judaea, so that your disciples, too, can see the works you are doing; ",
            4: "no one who wants to be publicly known acts in secret; if this is what you are doing, you should reveal yourself to the world.' ",
            5: "Not even his brothers had faith in him. ",
            6: "Jesus answered, 'For me the right time has not come yet, but for you any time is the right time. ",
            7: "The world cannot hate you, but it does hate me, because I give evidence that its ways are evil. ",
            8: "Go up to the festival yourselves: I am not going to this festival, because for me the time is not ripe yet.' ",
            9: "Having said that, he stayed behind in Galilee. ",
            10: "However, after his brothers had left for the festival, he went up as well, not publicly but secretly. ",
            11: "At the festival the Jews were on the look-out for him: 'Where is he?' they said. ",
            12: "There was a great deal of talk about him in the crowds. Some said, 'He is a good man'; others, 'No, he is leading the people astray.' ",
            13: "Yet no one spoke about him openly, for fear of the Jews. ",
            14: "When the festival was half over, Jesus went to the Temple and began to teach. ",
            15: "The Jews were astonished and said, 'How did he learn to read? He has not been educated.' ",
            16: "Jesus answered them: 'My teaching is not from myself: it comes from the one who sent me; ",
            17: "anyone who is prepared to do his will, will know whether my teaching is from God or whether I speak on my own account. ",
            18: "When someone speaks on his own account, he is seeking honour for himself; but when he is seeking the honour of the person who sent him, then he is true and altogether without dishonesty. ",
            19: "Did not Moses give you the Law? And yet not one of you keeps the Law! 'Why do you want to kill me?' ",
            20: "The crowd replied, 'You are mad! Who wants to kill you?' ",
            21: "Jesus answered, 'One work I did, and you are all amazed at it. ",
            22: "Moses ordered you to practise circumcision -- not that it began with him, it goes back to the patriarchs -- and you circumcise on the Sabbath. ",
            23: "Now if someone can be circumcised on the Sabbath so that the Law of Moses is not broken, why are you angry with me for making someone completely healthy on a Sabbath? ",
            24: "Do not keep judging according to appearances; let your judgement be according to what is right.' ",
            25: "Meanwhile some of the people of Jerusalem were saying, 'Isn't this the man they want to kill? ",
            26: "And here he is, speaking openly, and they have nothing to say to him! Can it be true the authorities have recognised that he is the Christ? ",
            27: "Yet we all know where he comes from, but when the Christ appears no one will know where he comes from.' ",
            28: "Then, as Jesus was teaching in the Temple, he cried out: You know me and you know where I came from. Yet I have not come of my own accord: but he who sent me is true; You do not know him, ",
            29: "but I know him because I have my being from him and it was he who sent me. ",
            30: "They wanted to arrest him then, but because his hour had not yet come no one laid a hand on him. ",
            31: "There were many people in the crowds, however, who believed in him; they were saying, 'When the Christ comes, will he give more signs than this man has?' ",
            32: "Hearing that talk like this about him was spreading among the people, the Pharisees sent the Temple guards to arrest him. ",
            33: "Then Jesus said: For a short time I am with you still; then I shall go back to the one who sent me. ",
            34: "You will look for me and will not find me; where I am you cannot come. ",
            35: "So the Jews said to one another, 'Where is he intending to go that we shall not be able to find him? Is he intending to go abroad to the people who are dispersed among the Greeks and to teach the Greeks? ",
            36: """What does he mean when he says: "You will look for me and will not find me; where I am, you cannot come?" ' """,
            37: "On the last day, the great day of the festival, Jesus stood and cried out: 'Let anyone who is thirsty come to me! ",
            38: """Let anyone who believes in me come and drink! As scripture says, "From his heart shall flow streams of living water." ' """,
            39: "He was speaking of the Spirit which those who believed in him were to receive; for there was no Spirit as yet because Jesus had not yet been glorified. ",
            40: "Some of the crowd who had been listening said, 'He is indeed the prophet,' ",
            41: "and some said, 'He is the Christ,' but others said, 'Would the Christ come from Galilee? ",
            42: "Does not scripture say that the Christ must be descended from David and come from Bethlehem, the village where David was?' ",
            43: "So the people could not agree about him. ",
            44: "Some wanted to arrest him, but no one actually laid a hand on him. ",
            45: "The guards went back to the chief priests and Pharisees who said to them, 'Why haven't you brought him?' ",
            46: "The guards replied, 'No one has ever spoken like this man.' ",
            47: "'So,' the Pharisees answered, 'you, too, have been led astray? ",
            48: "Have any of the authorities come to believe in him? Any of the Pharisees? ",
            49: "This rabble knows nothing about the Law -- they are damned.' ",
            50: "One of them, Nicodemus -- the same man who had come to Jesus earlier -- said to them, ",
            51: "'But surely our Law does not allow us to pass judgement on anyone without first giving him a hearing and discovering what he is doing?' ",
            52: "To this they answered, 'Are you a Galilean too? Go into the matter, and see for yourself: prophets do not arise in Galilee.' ",
            53: "They all went home."
        }

        chapter_8 = {
            1: "and Jesus went to the Mount of Olives. ",
            2: "At daybreak he appeared in the Temple again; and as all the people came to him, he sat down and began to teach them. ",
            3: "The scribes and Pharisees brought a woman along who had been caught committing adultery; and making her stand there in the middle ",
            4: "they said to Jesus, 'Master, this woman was caught in the very act of committing adultery, ",
            5: "and in the Law Moses has ordered us to stone women of this kind. What have you got to say?' ",
            6: "They asked him this as a test, looking for an accusation to use against him. But Jesus bent down and started writing on the ground with his finger. ",
            7: "As they persisted with their question, he straightened up and said, 'Let the one among you who is guiltless be the first to throw a stone at her.' ",
            8: "Then he bent down and continued writing on the ground. ",
            9: "When they heard this they went away one by one, beginning with the eldest, until the last one had gone and Jesus was left alone with the woman, who remained in the middle. ",
            10: "Jesus again straightened up and said, 'Woman, where are they? Has no one condemned you?' ",
            11: "'No one, sir,' she replied. 'Neither do I condemn you,' said Jesus. 'Go away, and from this moment sin no more.' ",
            12: "When Jesus spoke to the people again, he said: I am the light of the world; anyone who follows me will not be walking in the dark, but will have the light of life. ",
            13: "At this the Pharisees said to him, 'You are testifying on your own behalf; your testimony is not true.' ",
            14: "Jesus replied: Even though I am testifying on my own behalf, my testimony is still true, because I know where I have come from and where I am going; but you do not know where I come from or where I am going. ",
            15: "You judge by human standards; I judge no one, ",
            16: "but if I judge, my judgement will be true, because I am not alone: the one who sent me is with me; ",
            17: "and in your Law it is written that the testimony of two witnesses is true. ",
            18: "I testify on my own behalf, but the Father who sent me testifies on my behalf, too.",
            19: "They asked him, 'Where is your Father then?' Jesus answered: You do not know me, nor do you know my Father; if you did know me, you would know my Father as well. ",
            20: "He spoke these words in the Treasury, while teaching in the Temple. No one arrested him, because his hour had not yet come. ",
            21: "Again he said to them: I am going away; you will look for me and you will die in your sin. Where I am going, you cannot come. ",
            22: """So the Jews said to one another, 'Is he going to kill himself, that he says, "Where I am going, you cannot come?" ' """,
            23: "Jesus went on: You are from below; I am from above. You are of this world; I am not of this world. ",
            24: "I have told you already: You will die in your sins. Yes, if you do not believe that I am He, you will die in your sins.",
            25: "So they said to him, 'Who are you?' Jesus answered: What I have told you from the outset. ",
            26: "About you I have much to say and much to judge; but the one who sent me is true, and what I declare to the world I have learnt from him. ",
            27: "They did not recognise that he was talking to them about the Father. ",
            28: "So Jesus said: When you have lifted up the Son of man, then you will know that I am He and that I do nothing of my own accord. What I say is what the Father has taught me; ",
            29: "he who sent me is with me, and has not left me to myself, for I always do what pleases him. ",
            30: "As he was saying this, many came to believe in him. ",
            31: "To the Jews who believed in him Jesus said: If you make my word your home you will indeed be my disciples; ",
            32: "you will come to know the truth, and the truth will set you free. ",
            33: """They answered, 'We are descended from Abraham and we have never been the slaves of anyone; what do you mean, "You will be set free?" ' """,
            34: "Jesus replied: In all truth I tell you, everyone who commits sin is a slave. ",
            35: "Now a slave has no permanent standing in the household, but a son belongs to it for ever. ",
            36: "So if the Son sets you free, you will indeed be free. ",
            37: "I know that you are descended from Abraham; but you want to kill me because my word finds no place in you.",
            38: "What I speak of is what I have seen at my Father's side, and you too put into action the lessons you have learnt from your father. ",
            39: "They repeated, 'Our father is Abraham.' Jesus said to them: If you are Abraham's children, do as Abraham did. ",
            40: "As it is, you want to kill me, a man who has told you the truth as I have learnt it from God; that is not what Abraham did. ",
            41: "You are doing your father's work. They replied, 'We were not born illegitimate, the only father we have is God.' ",
            42: "Jesus answered: If God were your father, you would love me, since I have my origin in God and have come from him; I did not come of my own accord, but he sent me. ",
            43: "Why do you not understand what I say? Because you cannot bear to listen to my words. ",
            44: "You are from your father, the devil, and you prefer to do what your father wants. He was a murderer from the start; he was never grounded in the truth; there is no truth in him at all. When he lies he is speaking true to his nature, because he is a liar, and the father of lies. ",
            45: "But it is because I speak the truth that you do not believe me. ",
            46: "Can any of you convict me of sin? If I speak the truth, why do you not believe me? ",
            47: "Whoever comes from God listens to the words of God; the reason why you do not listen is that you are not from God. ",
            48: "The Jews replied, 'Are we not right in saying that you are a Samaritan and possessed by a devil?' Jesus answered: ",
            49: "I am not possessed; but I honour my Father, and you deny me honour.",
            50: "I do not seek my own glory; there is someone who does seek it and is the judge of it.",
            51: "In all truth I tell you, whoever keeps my word will never see death. ",
            52: """The Jews said, 'Now we know that you are possessed. Abraham is dead, and the prophets are dead, and yet you say, "Whoever keeps my word will never know the taste of death." """,
            53: "Are you greater than our father Abraham, who is dead? The prophets are dead too. Who are you claiming to be?' ",
            54: "Jesus answered: If I were to seek my own glory my glory would be worth nothing; in fact, my glory is conferred by the Father, by the one of whom you say, 'He is our God,' ",
            55: "although you do not know him. But I know him, and if I were to say, 'I do not know him,' I should be a liar, as you yourselves are. But I do know him, and I keep his word. ",
            56: "Your father Abraham rejoiced to think that he would see my Day; he saw it and was glad. ",
            57: "The Jews then said, 'You are not fifty yet, and you have seen Abraham!' ",
            58: "Jesus replied: In all truth I tell you, before Abraham ever was, I am. ",
            59: "At this they picked up stones to throw at him; but Jesus hid himself and left the Temple."
        }

        chapter_9 = {
            1: "As he went along, he saw a man who had been blind from birth. ",
            2: "His disciples asked him, 'Rabbi, who sinned, this man or his parents, that he should have been born blind?' ",
            3: "'Neither he nor his parents sinned,' Jesus answered, 'he was born blind so that the works of God might be revealed in him. ",
            4: "'As long as day lasts we must carry out the work of the one who sent me; the night will soon be here when no one can work. ",
            5: "As long as I am in the world I am the light of the world.' ",
            6: "Having said this, he spat on the ground, made a paste with the spittle, put this over the eyes of the blind man, ",
            7: "and said to him, 'Go and wash in the Pool of Siloam' (the name means 'one who has been sent'). So he went off and washed and came back able to see. ",
            8: "His neighbours and the people who used to see him before (for he was a beggar) said, 'Isn't this the man who used to sit and beg?' ",
            9: "Some said, 'Yes, it is the same one.' Others said, 'No, but he looks just like him.' The man himself said, 'Yes, I am the one.' ",
            10: "So they said to him, 'Then how is it that your eyes were opened?' ",
            11: """He answered, 'The man called Jesus made a paste, daubed my eyes with it and said to me, "Go off and wash at Siloam"; so I went, and when I washed I gained my sight.' """,
            12: "They asked, 'Where is he?' He answered, 'I don't know.' ",
            13: "They brought to the Pharisees the man who had been blind. ",
            14: "It had been a Sabbath day when Jesus made the paste and opened the man's eyes, ",
            15: "so when the Pharisees asked him how he had gained his sight, he said, 'He put a paste on my eyes, and I washed, and I can see.' ",
            16: "Then some of the Pharisees said, 'That man cannot be from God: he does not keep the Sabbath.' Others said, 'How can a sinner produce signs like this?' And there was division among them. ",
            17: "So they spoke to the blind man again, 'What have you to say about him yourself, now that he has opened your eyes?' The man answered, 'He is a prophet.' ",
            18: "However, the Jews would not believe that the man had been blind without first sending for the parents of the man who had gained his sight and ",
            19: "asking them, 'Is this man really the son of yours who you say was born blind? If so, how is it that he is now able to see?' ",
            20: "His parents answered, 'We know he is our son and we know he was born blind, ",
            21: "but how he can see, we don't know, nor who opened his eyes. Ask him. He is old enough: let him speak for himself.' ",
            22: "His parents spoke like this out of fear of the Jews, who had already agreed to ban from the synagogue anyone who should acknowledge Jesus as the Christ. ",
            23: "This was why his parents said, 'He is old enough; ask him.' ",
            24: "So the Jews sent for the man again and said to him, 'Give glory to God! We are satisfied that this man is a sinner.' ",
            25: "The man answered, 'Whether he is a sinner I don't know; all I know is that I was blind and now I can see.' ",
            26: "They said to him, 'What did he do to you? How did he open your eyes?' ",
            27: "He replied, 'I have told you once and you wouldn't listen. Why do you want to hear it all again? Do you want to become his disciples yourselves?' ",
            28: "At this they hurled abuse at him, 'It is you who are his disciple, we are disciples of Moses: ",
            29: "we know that God spoke to Moses, but as for this man, we don't know where he comes from.' ",
            30: "The man replied, 'That is just what is so amazing! You don't know where he comes from and he has opened my eyes! ",
            31: "We know that God doesn't listen to sinners, but God does listen to people who are devout and do his will. ",
            32: "Ever since the world began it is unheard of for anyone to open the eyes of someone born blind; ",
            33: "if this man were not from God, he wouldn't have been able to do anything.' ",
            34: "They retorted, 'Are you trying to teach us, and you a sinner through and through ever since you were born!' And they ejected him. ",
            35: "Jesus heard they had ejected him, and when he found him he said to him, 'Do you believe in the Son of man?' ",
            36: "'Sir,' the man replied, 'tell me who he is so that I may believe in him.' ",
            37: "Jesus said, 'You have seen him; he is speaking to you.' ",
            38: "The man said, 'Lord, I believe,' and worshipped him. ",
            39: "Jesus said: It is for judgement that I have come into this world, so that those without sight may see and those with sight may become blind. ",
            40: "Hearing this, some Pharisees who were present said to him, 'So we are blind, are we?' ",
            41: "Jesus replied: If you were blind, you would not be guilty, but since you say, 'We can see,' your guilt remains."
        }

        chapter_10 = {
            1: "'In all truth I tell you, anyone who does not enter the sheepfold through the gate, but climbs in some other way, is a thief and a bandit. ",
            2: "He who enters through the gate is the shepherd of the flock; ",
            3: "the gatekeeper lets him in, the sheep hear his voice, one by one he calls his own sheep and leads them out. ",
            4: "When he has brought out all those that are his, he goes ahead of them, and the sheep follow because they know his voice. ",
            5: "They will never follow a stranger, but will run away from him because they do not recognise the voice of strangers.' ",
            6: "Jesus told them this parable but they failed to understand what he was saying to them. ",
            7: "So Jesus spoke to them again: In all truth I tell you, I am the gate of the sheepfold. ",
            8: "All who have come before me are thieves and bandits, but the sheep took no notice of them. ",
            9: "I am the gate. Anyone who enters through me will be safe: such a one will go in and out and will find pasture.",
            10: "The thief comes only to steal and kill and destroy. I have come so that they may have life and have it to the full. ",
            11: "I am the good shepherd: the good shepherd lays down his life for his sheep.",
            12: "The hired man, since he is not the shepherd and the sheep do not belong to him, abandons the sheep as soon as he sees a wolf coming, and runs away, and then the wolf attacks and scatters the sheep; ",
            13: "he runs away because he is only a hired man and has no concern for the sheep. ",
            14: "I am the good shepherd; I know my own and my own know me,",
            15: "just as the Father knows me and I know the Father; and I lay down my life for my sheep. ",
            16: "And there are other sheep I have that are not of this fold, and I must lead these too. They too will listen to my voice, and there will be only one flock, one shepherd. ",
            17: "The Father loves me, because I lay down my life in order to take it up again. ",
            18: "No one takes it from me; I lay it down of my own free will, and as I have power to lay it down, so I have power to take it up again; and this is the command I have received from my Father. ",
            19: "These words caused a fresh division among the Jews. ",
            20: "Many said, 'He is possessed, he is raving; why do you listen to him?' ",
            21: "Others said, 'These are not the words of a man possessed by a devil: could a devil open the eyes of the blind?' ",
            22: "It was the time of the feast of Dedication in Jerusalem. It was winter, ",
            23: "and Jesus was in the Temple walking up and down in the Portico of Solomon. ",
            24: "The Jews gathered round him and said, 'How much longer are you going to keep us in suspense? If you are the Christ, tell us openly.' ",
            25: "Jesus replied: I have told you, but you do not believe. The works I do in my Father's name are my witness; ",
            26: "but you do not believe, because you are no sheep of mine. ",
            27: "The sheep that belong to me listen to my voice; I know them and they follow me. ",
            28: "I give them eternal life; they will never be lost and no one will ever steal them from my hand.",
            29: "The Father, for what he has given me, is greater than anyone, and no one can steal anything from the Father's hand. ",
            30: "The Father and I are one. ",
            31: "The Jews fetched stones to stone him, ",
            32: "so Jesus said to them, 'I have shown you many good works from my Father; for which of these are you stoning me?' ",
            33: "The Jews answered him, 'We are stoning you, not for doing a good work, but for blasphemy; though you are only a man, you claim to be God.' ",
            34: "Jesus answered: Is it not written in your Law: I said, you are gods? ",
            35: "So it uses the word 'gods' of those people to whom the word of God was addressed -- and scripture cannot be set aside. ",
            36: "Yet to someone whom the Father has consecrated and sent into the world you say, 'You are blaspheming' because I said, 'I am Son of God.' ",
            37: "If I am not doing my Father's work, there is no need to believe me; ",
            38: "but if I am doing it, then even if you refuse to believe in me, at least believe in the work I do; then you will know for certain that the Father is in me and I am in the Father. ",
            39: "They again wanted to arrest him then, but he eluded their clutches. ",
            40: "He went back again to the far side of the Jordan to the district where John had been baptising at first and he stayed there. ",
            41: "Many people who came to him said, 'John gave no signs, but all he said about this man was true'; ",
            42: "and many of them believed in him."
        }

        chapter_11 = {
            1: "There was a man named Lazarus of Bethany, the village of Mary and her sister, Martha, and he was ill. ",
            2: "It was the same Mary, the sister of the sick man Lazarus, who anointed the Lord with ointment and wiped his feet with her hair. ",
            3: "The sisters sent this message to Jesus, 'Lord, the man you love is ill.' ",
            4: "On receiving the message, Jesus said, 'This sickness will not end in death, but it is for God's glory so that through it the Son of God may be glorified.' ",
            5: "Jesus loved Martha and her sister and Lazarus, ",
            6: "yet when he heard that he was ill he stayed where he was for two more days ",
            7: "before saying to the disciples, 'Let us go back to Judaea.' ",
            8: "The disciples said, 'Rabbi, it is not long since the Jews were trying to stone you; are you going back there again?' ",
            9: "Jesus replied: Are there not twelve hours in the day? No one who walks in the daytime stumbles, having the light of this world to see by; ",
            10: "anyone who walks around at night stumbles, having no light as a guide. ",
            11: "He said that and then added, 'Our friend Lazarus is at rest; I am going to wake him.' ",
            12: "The disciples said to him, 'Lord, if he is at rest he will be saved.' ",
            13: "Jesus was speaking of the death of Lazarus, but they thought that by 'rest' he meant 'sleep'; ",
            14: "so Jesus put it plainly, 'Lazarus is dead; ",
            15: "and for your sake I am glad I was not there because now you will believe. But let us go to him.' ",
            16: "Then Thomas -- known as the Twin -- said to the other disciples, 'Let us also go to die with him.' ",
            17: "On arriving, Jesus found that Lazarus had been in the tomb for four days already. ",
            18: "Bethany is only about two miles from Jerusalem, ",
            19: "and many Jews had come to Martha and Mary to comfort them about their brother. ",
            20: "When Martha heard that Jesus was coming she went to meet him. Mary remained sitting in the house. ",
            21: "Martha said to Jesus, 'Lord, if you had been here, my brother would not have died, ",
            22: "but even now I know that God will grant whatever you ask of him.' ",
            23: "Jesus said to her, 'Your brother will rise again.' ",
            24: "Martha said, 'I know he will rise again at the resurrection on the last day.' ",
            25: "Jesus said: I am the resurrection. Anyone who believes in me, even though that person dies, will live, ",
            26: "and whoever lives and believes in me will never die. Do you believe this? ",
            27: "'Yes, Lord,' she said, 'I believe that you are the Christ, the Son of God, the one who was to come into this world.' ",
            28: "When she had said this, she went and called her sister Mary, saying in a low voice, 'The Master is here and wants to see you.' ",
            29: "Hearing this, Mary got up quickly and went to him. ",
            30: "Jesus had not yet come into the village; he was still at the place where Martha had met him. ",
            31: "When the Jews who were in the house comforting Mary saw her get up so quickly and go out, they followed her, thinking that she was going to the tomb to weep there. ",
            32: "Mary went to Jesus, and as soon as she saw him she threw herself at his feet, saying, 'Lord, if you had been here, my brother would not have died.' ",
            33: "At the sight of her tears, and those of the Jews who had come with her, Jesus was greatly distressed, and with a profound sigh he said, ",
            34: "'Where have you put him?' They said, 'Lord, come and see.' ",
            35: "Jesus wept; ",
            36: "and the Jews said, 'See how much he loved him!' ",
            37: "But there were some who remarked, 'He opened the eyes of the blind man. Could he not have prevented this man's death?' ",
            38: "Sighing again, Jesus reached the tomb: it was a cave with a stone to close the opening. ",
            39: "Jesus said, 'Take the stone away.' Martha, the dead man's sister, said to him, 'Lord, by now he will smell; this is the fourth day since he died.' ",
            40:  "Jesus replied, 'Have I not told you that if you believe you will see the glory of God?' ",
            41: "So they took the stone away. Then Jesus lifted up his eyes and said: Father, I thank you for hearing my prayer. ",
            42: "I myself knew that you hear me always, but I speak for the sake of all these who are standing around me, so that they may believe it was you who sent me.",
            43: "When he had said this, he cried in a loud voice, 'Lazarus, come out!' ",
            44: "The dead man came out, his feet and hands bound with strips of material, and a cloth over his face. Jesus said to them, 'Unbind him, let him go free.' ",
            45: "Many of the Jews who had come to visit Mary, and had seen what he did, believed in him, ",
            46: "but some of them went to the Pharisees to tell them what Jesus had done. ",
            47: "Then the chief priests and Pharisees called a meeting. 'Here is this man working all these signs,' they said, 'and what action are we taking? ",
            48: "If we let him go on in this way everybody will believe in him, and the Romans will come and suppress the Holy Place and our nation.' ",
            49: "One of them, Caiaphas, the high priest that year, said, 'You do not seem to have grasped the situation at all; ",
            50: "you fail to see that it is to your advantage that one man should die for the people, rather than that the whole nation should perish.' ",
            51: "He did not speak in his own person, but as high priest of that year he was prophesying that Jesus was to die for the nation- ",
            52: "and not for the nation only, but also to gather together into one the scattered children of God. ",
            53: "From that day onwards they were determined to kill him. ",
            54: "So Jesus no longer went about openly among the Jews, but left the district for a town called Ephraim, in the country bordering on the desert, and stayed there with his disciples. ",
            55: "The Jewish Passover was drawing near, and many of the country people who had gone up to Jerusalem before the Passover to purify themselves ",
            56: "were looking out for Jesus, saying to one another as they stood about in the Temple, 'What do you think? Will he come to the festival or not?' ",
            57: "The chief priests and Pharisees had by now given their orders: anyone who knew where he was must inform them so that they could arrest him."
        }

        chapter_12 = {
            1: "Six days before the Passover, Jesus went to Bethany, where Lazarus was, whom he had raised from the dead. ",
            2: "They gave a dinner for him there; Martha waited on them and Lazarus was among those at table. ",
            3: "Mary brought in a pound of very costly ointment, pure nard, and with it anointed the feet of Jesus, wiping them with her hair; the house was filled with the scent of the ointment. ",
            4: "Then Judas Iscariot -- one of his disciples, the man who was to betray him-said, ",
            5: "'Why was this ointment not sold for three hundred denarii and the money given to the poor?' ",
            6: "He said this, not because he cared about the poor, but because he was a thief; he was in charge of the common fund and used to help himself to the contents. ",
            7: "So Jesus said, 'Leave her alone; let her keep it for the day of my burial. ",
            8: "You have the poor with you always, you will not always have me.' ",
            9: "Meanwhile a large number of Jews heard that he was there and came not only on account of Jesus but also to see Lazarus whom he had raised from the dead. ",
            10: "Then the chief priests decided to kill Lazarus as well, ",
            11: "since it was on his account that many of the Jews were leaving them and believing in Jesus. ",
            12: "The next day the great crowd of people who had come up for the festival heard that Jesus was on his way to Jerusalem. ",
            13: "They took branches of palm and went out to receive him, shouting: 'Hosanna! Blessed is he who is coming in the name of the Lord, the king of Israel.' ",
            14: "Jesus found a young donkey and mounted it -- as scripture says: ",
            15: "Do not be afraid, daughter of Zion; look, your king is approaching, riding on the foal of a donkey. ",
            16: "At first his disciples did not understand this, but later, after Jesus had been glorified, they remembered that this had been written about him and that this was what had happened to him. ",
            17: "The crowd who had been with him when he called Lazarus out of the tomb and raised him from the dead kept bearing witness to it; ",
            18: "this was another reason why the crowd came out to receive him: they had heard that he had given this sign. ",
            19: "Then the Pharisees said to one another, 'You see, you are making no progress; look, the whole world has gone after him!' ",
            20: "Among those who went up to worship at the festival were some Greeks. ",
            21: "These approached Philip, who came from Bethsaida in Galilee, and put this request to him, 'Sir, we should like to see Jesus.' ",
            22: "Philip went to tell Andrew, and Andrew and Philip together went to tell Jesus. ",
            23: "Jesus replied to them: Now the hour has come for the Son of man to be glorified. ",
            24: "In all truth I tell you, unless a wheat grain falls into the earth and dies, it remains only a single grain; but if it dies it yields a rich harvest. ",
            25: "Anyone who loves his life loses it; anyone who hates his life in this world will keep it for eternal life. ",
            26: "Whoever serves me, must follow me, and my servant will be with me wherever I am. If anyone serves me, my Father will honour him. ",
            27: "Now my soul is troubled. What shall I say: Father, save me from this hour? But it is for this very reason that I have come to this hour. ",
            28: "Father, glorify your name! A voice came from heaven, 'I have glorified it, and I will again glorify it.' ",
            29: "The crowd standing by, who heard this, said it was a clap of thunder; others said, 'It was an angel speaking to him.' ",
            30: "Jesus answered, 'It was not for my sake that this voice came, but for yours. ",
            31: "'Now sentence is being passed on this world; now the prince of this world is to be driven out. ",
            32: "And when I am lifted up from the earth, I shall draw all people to myself.' ",
            33: "By these words he indicated the kind of death he would die. ",
            34: """The crowd answered, 'The Law has taught us that the Christ will remain for ever. So how can you say, "The Son of man must be lifted up"? Who is this Son of man?' """,
            35: "Jesus then said: The light will be with you only a little longer now. Go on your way while you have the light, or darkness will overtake you, and nobody who walks in the dark knows where he is going. ",
            36: "While you still have the light, believe in the light so that you may become children of light. Having said this, Jesus left them and was hidden from their sight. ",
            37: "Though they had been present when he gave so many signs, they did not believe in him; ",
            38: "this was to fulfil the words of the prophet Isaiah: Lord, who has given credence to what they have heard from us, and who has seen in it a revelation of the Lord's arm? ",
            39: "Indeed, they were unable to believe because, as Isaiah says again: ",
            40: "He has blinded their eyes, he has hardened their heart, to prevent them from using their eyes to see, using their heart to understand, changing their ways and being healed by me. ",
            41: "Isaiah said this because he saw his glory, and his words referred to Jesus. ",
            42: "And yet there were many who did believe in him, even among the leading men, but they did not admit it, because of the Pharisees and for fear of being banned from the synagogue: ",
            43: "they put human glory before God's glory. ",
            44: "Jesus declared publicly: Whoever believes in me believes not in me but in the one who sent me, ",
            45: "and whoever sees me, sees the one who sent me. ",
            46: "I have come into the world as light, to prevent anyone who believes in me from staying in the dark any more.",
            47: "If anyone hears my words and does not keep them faithfully, it is not I who shall judge such a person, since I have come not to judge the world, but to save the world: ",
            48: "anyone who rejects me and refuses my words has his judge already: the word itself that I have spoken will be his judge on the last day. ",
            49: "For I have not spoken of my own accord; but the Father who sent me commanded me what to say and what to speak, ",
            50: "and I know that his commands mean eternal life. And therefore what the Father has told me is what I speak."
        }

        chapter_13 = {
            1: "Before the festival of the Passover, Jesus, knowing that his hour had come to pass from this world to the Father, having loved those who were his in the world, loved them to the end. ",
            2: "They were at supper, and the devil had already put it into the mind of Judas Iscariot son of Simon, to betray him. ",
            3: "Jesus knew that the Father had put everything into his hands, and that he had come from God and was returning to God, ",
            4: "and he got up from table, removed his outer garments and, taking a towel, wrapped it round his waist; ",
            5: "he then poured water into a basin and began to wash the disciples' feet and to wipe them with the towel he was wearing. ",
            6: "He came to Simon Peter, who said to him, 'Lord, are you going to wash my feet?' ",
            7: "Jesus answered, 'At the moment you do not know what I am doing, but later you will understand.' ",
            8: "'Never!' said Peter. 'You shall never wash my feet.' Jesus replied, 'If I do not wash you, you can have no share with me.' Simon Peter said, ",
            9: "'Well then, Lord, not only my feet, but my hands and my head as well!' ",
            10: "Jesus said, 'No one who has had a bath needs washing, such a person is clean all over. You too are clean, though not all of you are.' ",
            11: "He knew who was going to betray him, and that was why he said, 'though not all of you are'. ",
            12: "When he had washed their feet and put on his outer garments again he went back to the table. 'Do you understand', he said, 'what I have done to you? ",
            13: "You call me Master and Lord, and rightly; so I am. ",
            14: "If I, then, the Lord and Master, have washed your feet, you must wash each other's feet. ",
            15: "I have given you an example so that you may copy what I have done to you.",
            16: "'In all truth I tell you, no servant is greater than his master, no messenger is greater than the one who sent him. ",
            17: "'Now that you know this, blessed are you if you behave accordingly. ",
            18: "I am not speaking about all of you: I know the ones I have chosen; but what scripture says must be fulfilled: 'He who shares my table takes advantage of me.",
            19: "I tell you this now, before it happens, so that when it does happen you may believe that I am He.",
            20: "In all truth I tell you, whoever welcomes the one I send, welcomes me, and whoever welcomes me, welcomes the one who sent me.' ",
            21: "Having said this, Jesus was deeply disturbed and declared, 'In all truth I tell you, one of you is going to betray me.' ",
            22: "The disciples looked at each other, wondering whom he meant. ",
            23: "The disciple Jesus loved was reclining next to Jesus; ",
            24: "Simon Peter signed to him and said, 'Ask who it is he means,' ",
            25: "so leaning back close to Jesus' chest he said, 'Who is it, Lord?' ",
            26: "Jesus answered, 'It is the one to whom I give the piece of bread that I dip in the dish.' And when he had dipped the piece of bread he gave it to Judas son of Simon Iscariot. ",
            27: "At that instant, after Judas had taken the bread, Satan entered him. Jesus then said, 'What you are going to do, do quickly.' ",
            28: "None of the others at table understood why he said this. ",
            29: "Since Judas had charge of the common fund, some of them thought Jesus was telling him, 'Buy what we need for the festival,' or telling him to give something to the poor. ",
            30: "As soon as Judas had taken the piece of bread he went out. It was night. ",
            31: "When he had gone, Jesus said: Now has the Son of man been glorified, and in him God has been glorified. ",
            32: "If God has been glorified in him, God will in turn glorify him in himself, and will glorify him very soon. ",
            33: "Little children, I shall be with you only a little longer. You will look for me, and, as I told the Jews, where I am going, you cannot come. ",
            34: "I give you a new commandment: love one another; you must love one another just as I have loved you",
            35: "It is by your love for one another, that everyone will recognise you as my disciples. ",
            36: "Simon Peter said, 'Lord, where are you going?' Jesus replied, 'Now you cannot follow me where I am going, but later you shall follow me.' ",
            37: "Peter said to him, 'Why can I not follow you now? I will lay down my life for you.' ",
            38: "'Lay down your life for me?' answered Jesus. 'In all truth I tell you, before the cock crows you will have disowned me three times.'"
        }

        chapter_14 = {
            1: "Do not let your hearts be troubled. You trust in God, trust also in me. ",
            2: "In my Father's house there are many places to live in; otherwise I would have told you. I am going now to prepare a place for you, ",
            3: "and after I have gone and prepared you a place, I shall return to take you to myself, so that you may be with me where I am. ",
            4: "You know the way to the place where I am going. ",
            5: "Thomas said, 'Lord, we do not know where you are going, so how can we know the way?' ",
            6: "Jesus said: I am the Way; I am Truth and Life. No one can come to the Father except through me. ",
            7: "If you know me, you will know my Father too. From this moment you know him and have seen him. ",
            8: "Philip said, 'Lord, show us the Father and then we shall be satisfied.' Jesus said to him, ",
            9: """'Have I been with you all this time, Philip, and you still do not know me? 'Anyone who has seen me has seen the Father, so how can you say, "Show us the Father"? """,
            10: "Do you not believe that I am in the Father and the Father is in me? What I say to you I do not speak of my own accord: it is the Father, living in me, who is doing his works. ",
            11: "You must believe me when I say that I am in the Father and the Father is in me; or at least believe it on the evidence of these works. ",
            12: "In all truth I tell you, whoever believes in me will perform the same works as I do myself, and will perform even greater works, because I am going to the Father. ",
            13: "Whatever you ask in my name I will do, so that the Father may be glorified in the Son. ",
            14: "If you ask me anything in my name, I will do it. ",
            15: "If you love me you will keep my commandments. ",
            16: "I shall ask the Father, and he will give you another Paraclete to be with you for ever,",
            17: "the Spirit of truth whom the world can never accept since it neither sees nor knows him; but you know him, because he is with you, he is in you. ",
            18: "I shall not leave you orphans; I shall come to you.",
            19: "In a short time the world will no longer see me; but you will see that I live and you also will live. ",
            20: "On that day you will know that I am in my Father and you in me and I in you. ",
            21: "Whoever holds to my commandments and keeps them is the one who loves me; and whoever loves me will be loved by my Father, and I shall love him and reveal myself to him.' ",
            22: "Judas -- not Judas Iscariot -- said to him, 'Lord, what has happened, that you intend to show yourself to us and not to the world?' ",
            23: "Jesus replied: Anyone who loves me will keep my word, and my Father will love him, and we shall come to him and make a home in him. ",
            24: "Anyone who does not love me does not keep my words. And the word that you hear is not my own: it is the word of the Father who sent me. ",
            25: "I have said these things to you while still with you;",
            26: "but the Paraclete, the Holy Spirit, whom the Father will send in my name, will teach you everything and remind you of all I have said to you. ",
            27: "Peace I bequeath to you, my own peace I give you, a peace which the world cannot give, this is my gift to you. Do not let your hearts be troubled or afraid. ",
            28: "You heard me say: I am going away and shall return. If you loved me you would be glad that I am going to the Father, for the Father is greater than I. ",
            29: "I have told you this now, before it happens, so that when it does happen you may believe.",
            30: "I shall not talk to you much longer, because the prince of this world is on his way. He has no power over me,",
            31: "but the world must recognise that I love the Father and that I act just as the Father commanded. Come now, let us go."
        }

        chapter_15 = {
            1: "I am the true vine, and my Father is the vinedresser.",
            2: "Every branch in me that bears no fruit he cuts away, and every branch that does bear fruit he prunes to make it bear even more. ",
            3: "You are clean already, by means of the word that I have spoken to you. ",
            4: "Remain in me, as I in you. As a branch cannot bear fruit all by itself, unless it remains part of the vine, neither can you unless you remain in me. ",
            5: "I am the vine, you are the branches. Whoever remains in me, with me in him, bears fruit in plenty; for cut off from me you can do nothing.",
            6: "Anyone who does not remain in me is thrown away like a branch -- and withers; these branches are collected and thrown on the fire and are burnt. ",
            7: "If you remain in me and my words remain in you, you may ask for whatever you please and you will get it. ",
            8: "It is to the glory of my Father that you should bear much fruit and be my disciples. ",
            9: "I have loved you just as the Father has loved me. Remain in my love.",
            10: "If you keep my commandments you will remain in my love, just as I have kept my Father's commandments and remain in his love. ",
            11: "I have told you this so that my own joy may be in you and your joy be complete.",
            12: "This is my commandment: love one another, as I have loved you. ",
            13: "No one can have greater love than to lay down his life for his friends. ",
            14: "You are my friends, if you do what I command you. ",
            15: "I shall no longer call you servants, because a servant does not know the master's business; I call you friends, because I have made known to you everything I have learnt from my Father.",
            16: "You did not choose me, no, I chose you; and I commissioned you to go out and to bear fruit, fruit that will last; so that the Father will give you anything you ask him in my name. ",
            17: "My command to you is to love one another. ",
            18: "If the world hates you, you must realise that it hated me before it hated you. ",
            19: "If you belonged to the world, the world would love you as its own; but because you do not belong to the world, because my choice of you has drawn you out of the world, that is why the world hates you. ",
            20: "Remember the words I said to you: A servant is not greater than his master. If they persecuted me, they will persecute you too; if they kept my word, they will keep yours as well. ",
            21: "But it will be on my account that they will do all this to you, because they do not know the one who sent me. ",
            22: "If I had not come, if I had not spoken to them, they would have been blameless; but as it is they have no excuse for their sin. ",
            23: "Anyone who hates me hates my Father. ",
            24: "If I had not performed such works among them as no one else has ever done, they would be blameless; but as it is, in spite of what they have seen, they hate both me and my Father. ",
            25: "But all this was only to fulfil the words written in their Law: They hated me without reason. ",
            26: "When the Paraclete comes, whom I shall send to you from the Father, the Spirit of truth who issues from the Father, he will be my witness. ",
            27: "And you too will be witnesses, because you have been with me from the beginning."
        }

        chapter_16 = {
            1: "I have told you all this so that you may not fall away.",
            2: "They will expel you from the synagogues, and indeed the time is coming when anyone who kills you will think he is doing a holy service to God. ",
            3: "They will do these things because they have never known either the Father or me. ",
            4: "But I have told you all this, so that when the time for it comes you may remember that I told you. I did not tell you this from the beginning, because I was with you; ",
            5: "but now I am going to the one who sent me. Not one of you asks, 'Where are you going?' ",
            6: "Yet you are sad at heart because I have told you this. ",
            7: "Still, I am telling you the truth: it is for your own good that I am going, because unless I go, the Paraclete will not come to you; but if I go, I will send him to you. ",
            8: "And when he comes, he will show the world how wrong it was, about sin, and about who was in the right, and about judgement: ",
            9: "about sin: in that they refuse to believe in me; ",
            10: "about who was in the right: in that I am going to the Father and you will see me no more; ",
            11: "about judgement: in that the prince of this world is already condemned. ",
            12: "I still have many things to say to you but they would be too much for you to bear now.",
            13: "However, when the Spirit of truth comes he will lead you to the complete truth, since he will not be speaking of his own accord, but will say only what he has been told; and he will reveal to you the things to come. ",
            14: "He will glorify me, since all he reveals to you will be taken from what is mine. ",
            15: "Everything the Father has is mine; that is why I said: all he reveals to you will be taken from what is mine. ",
            16: "In a short time you will no longer see me, and then a short time later you will see me again. ",
            17: """Then some of his disciples said to one another, 'What does he mean, "In a short time you will no longer see me, and then a short time later you will see me again," and, "I am going to the Father"? """,
            18: """What is this "short time"? We don't know what he means.' """,
            19: '''Jesus knew that they wanted to question him, so he said, 'You are asking one another what I meant by saying, "In a short time you will no longer see me, and then a short time later you will see me again." ''',
            20: "'In all truth I tell you, you will be weeping and wailing while the world will rejoice; you will be sorrowful, but your sorrow will turn to joy. ",
            21: "A woman in childbirth suffers, because her time has come; but when she has given birth to the child she forgets the suffering in her joy that a human being has been born into the world. ",
            22: "So it is with you: you are sad now, but I shall see you again, and your hearts will be full of joy, and that joy no one shall take from you. ",
            23: "When that day comes, you will not ask me any questions. In all truth I tell you, anything you ask from the Father he will grant in my name. ",
            24: "Until now you have not asked anything in my name. Ask and you will receive, and so your joy will be complete. ",
            25: "I have been telling you these things in veiled language. The hour is coming when I shall no longer speak to you in veiled language but tell you about the Father in plain words.",
            26: "When that day comes you will ask in my name; and I do not say that I shall pray to the Father for you, ",
            27: "because the Father himself loves you for loving me, and believing that I came from God. ",
            28: "I came from the Father and have come into the world and now I am leaving the world to go to the Father.'",
            29: "His disciples said, 'Now you are speaking plainly and not using veiled language. ",
            30: "Now we see that you know everything and need not wait for questions to be put into words; because of this we believe that you came from God.' ",
            31: "Jesus answered them: Do you believe at last? ",
            32: "Listen; the time will come -- indeed it has come already -- when you are going to be scattered, each going his own way and leaving me alone. And yet I am not alone, because the Father is with me. ",
            33: "I have told you all this so that you may find peace in me. In the world you will have hardship, but be courageous: I have conquered the world."
        }

        chapter_17 = {
            1: "After saying this, Jesus raised his eyes to heaven and said: Father, the hour has come: glorify your Son so that your Son may glorify you; ",
            2: "so that, just as you have given him power over all humanity, he may give eternal life to all those you have entrusted to him. ",
            3: "And eternal life is this: to know you, the only true God, and Jesus Christ whom you have sent. ",
            4: "I have glorified you on earth by finishing the work that you gave me to do.",
            5: "Now, Father, glorify me with that glory I had with you before ever the world existed. ",
            6: "I have revealed your name to those whom you took from the world to give me. They were yours and you gave them to me, and they have kept your word.",
            7: "Now at last they have recognised that all you have given me comes from you ",
            8: "for I have given them the teaching you gave to me, and they have indeed accepted it and know for certain that I came from you, and have believed that it was you who sent me. ",
            9: "It is for them that I pray. I am not praying for the world but for those you have given me, because they belong to you. ",
            10: "All I have is yours and all you have is mine, and in them I am glorified. ",
            11: "I am no longer in the world, but they are in the world, and I am coming to you. Holy Father, keep those you have given me true to your name, so that they may be one like us.",
            12: "While I was with them, I kept those you had given me true to your name. I have watched over them and not one is lost except one who was destined to be lost, and this was to fulfil the scriptures. ",
            13: "But now I am coming to you and I say these things in the world to share my joy with them to the full. ",
            14: "I passed your word on to them, and the world hated them, because they belong to the world no more than I belong to the world.",
            15: "I am not asking you to remove them from the world, but to protect them from the Evil One.",
            16: "They do not belong to the world any more than I belong to the world. ",
            17: "Consecrate them in the truth; your word is truth. ",
            18: "As you sent me into the world, I have sent them into the world, ",
            19: "and for their sake I consecrate myself so that they too may be consecrated in truth. ",
            20: "I pray not only for these but also for those who through their teaching will come to believe in me.",
            21: "May they all be one, just as, Father, you are in me and I am in you, so that they also may be in us, so that the world may believe it was you who sent me. ",
            22: "I have given them the glory you gave to me, that they may be one as we are one.",
            23: "With me in them and you in me, may they be so perfected in unity that the world will recognise that it was you who sent me and that you have loved them as you have loved me. ",
            24: "Father, I want those you have given me to be with me where I am, so that they may always see my glory which you have given me because you loved me before the foundation of the world. ",
            25: "Father, Upright One, the world has not known you, but I have known you, and these have known that you have sent me. ",
            26: "I have made your name known to them and will continue to make it known, so that the love with which you loved me may be in them, and so that I may be in them.",
        }

        chapter_18 = {
            1: "After he had said all this, Jesus left with his disciples and crossed the Kidron valley where there was a garden into which he went with his disciples. ",
            2: "Judas the traitor knew the place also, since Jesus had often met his disciples there, ",
            3: "so Judas brought the cohort to this place together with guards sent by the chief priests and the Pharisees, all with lanterns and torches and weapons. ",
            4: "Knowing everything that was to happen to him, Jesus came forward and said, 'Who are you looking for?' ",
            5: "They answered, 'Jesus the Nazarene.' He said, 'I am he.' Now Judas the traitor was standing among them. ",
            6: "When Jesus said to them, 'I am he,' they moved back and fell on the ground. ",
            7: "He asked them a second time, 'Who are you looking for?' They said, 'Jesus the Nazarene.' ",
            8: "Jesus replied, 'I have told you that I am he. If I am the one you are looking for, let these others go.' ",
            9: "This was to fulfil the words he had spoken, 'Not one of those you gave me have I lost.' ",
            10: "Simon Peter, who had a sword, drew it and struck the high priest's servant, cutting off his right ear. The servant's name was Malchus. ",
            11: "Jesus said to Peter, 'Put your sword back in its scabbard; am I not to drink the cup that the Father has given me?' ",
            12: "The cohort and its tribune and the Jewish guards seized Jesus and bound him. ",
            13: "They took him first to Annas, because Annas was the father-in-law of Caiaphas, who was high priest that year. ",
            14: "It was Caiaphas who had counselled the Jews, 'It is better for one man to die for the people.' ",
            15: "Simon Peter, with another disciple, followed Jesus. This disciple, who was known to the high priest, went with Jesus into the high priest's palace, ",
            16: "but Peter stayed outside the door. So the other disciple, the one known to the high priest, went out, spoke to the door-keeper and brought Peter in. ",
            17: "The girl on duty at the door said to Peter, 'Aren't you another of that man's disciples?' He answered, 'I am not.' ",
            18: "Now it was cold, and the servants and guards had lit a charcoal fire and were standing there warming themselves; so Peter stood there too, warming himself with the others. ",
            19: "The high priest questioned Jesus about his disciples and his teaching. ",
            20: "Jesus answered, 'I have spoken openly for all the world to hear; I have always taught in the synagogue and in the Temple where all the Jews meet together; I have said nothing in secret. ",
            21: "Why ask me? Ask my hearers what I taught; they know what I said.' ",
            22: "At these words, one of the guards standing by gave Jesus a slap in the face, saying, 'Is that the way you answer the high priest?' ",
            23: "Jesus replied, 'If there is some offence in what I said, point it out; but if not, why do you strike me?' ",
            24: "Then Annas sent him, bound, to Caiaphas the high priest. ",
            25: "As Simon Peter stood there warming himself, someone said to him, 'Aren't you another of his disciples?' He denied it saying, 'I am not.' ",
            26: "One of the high priest's servants, a relation of the man whose ear Peter had cut off, said, 'Didn't I see you in the garden with him?' ",
            27: "Again Peter denied it; and at once a cock crowed. ",
            28: "They then led Jesus from the house of Caiaphas to the Praetorium. It was now morning. They did not go into the Praetorium themselves to avoid becoming defiled and unable to eat the Passover. ",
            29: "So Pilate came outside to them and said, 'What charge do you bring against this man?' They replied, ",
            30: "'If he were not a criminal, we should not have handed him over to you.' ",
            31: "Pilate said, 'Take him yourselves, and try him by your own Law.' The Jews answered, 'We are not allowed to put anyone to death.' ",
            32: "This was to fulfil the words Jesus had spoken indicating the way he was going to die. ",
            33: "So Pilate went back into the Praetorium and called Jesus to him and asked him, 'Are you the king of the Jews?' ",
            34: "Jesus replied, 'Do you ask this of your own accord, or have others said it to you about me?' ",
            35: "Pilate answered, 'Am I a Jew? It is your own people and the chief priests who have handed you over to me: what have you done?' ",
            36: "Jesus replied, 'Mine is not a kingdom of this world; if my kingdom were of this world, my men would have fought to prevent my being surrendered to the Jews. As it is, my kingdom does not belong here.' ",
            37: "Pilate said, 'So, then you are a king?' Jesus answered, 'It is you who say that I am a king. I was born for this, I came into the world for this, to bear witness to the truth; and all who are on the side of truth listen to my voice.' ",
            38: "'Truth?' said Pilate. 'What is that?' And so saying he went out again to the Jews and said, 'I find no case against him. ",
            39: "But according to a custom of yours I should release one prisoner at the Passover; would you like me, then, to release for you the king of the Jews?' ",
            40: "At this they shouted, 'Not this man,' they said, 'but Barabbas.' Barabbas was a bandit."
        }

        chapter_19 = {
            1: "Pilate then had Jesus taken away and scourged; ",
            2: "and after this, the soldiers twisted some thorns into a crown and put it on his head and dressed him in a purple robe. ",
            3: "They kept coming up to him and saying, 'Hail, king of the Jews!' and slapping him in the face. ",
            4: "Pilate came outside again and said to them, 'Look, I am going to bring him out to you to let you see that I find no case against him.' ",
            5: "Jesus then came out wearing the crown of thorns and the purple robe. Pilate said, 'Here is the man.' ",
            6: "When they saw him, the chief priests and the guards shouted, 'Crucify him! Crucify him!' Pilate said, 'Take him yourselves and crucify him: I find no case against him.' ",
            7: "The Jews replied, 'We have a Law, and according to that Law he ought to be put to death, because he has claimed to be Son of God.' ",
            8: "When Pilate heard them say this his fears increased. ",
            9: "Re-entering the Praetorium, he said to Jesus, 'Where do you come from?' But Jesus made no answer. ",
            10: "Pilate then said to him, 'Are you refusing to speak to me? Surely you know I have power to release you and I have power to crucify you?' ",
            11: "Jesus replied, 'You would have no power over me at all if it had not been given you from above; that is why the man who handed me over to you has the greater guilt.' ",
            12: "From that moment Pilate was anxious to set him free, but the Jews shouted, 'If you set him free you are no friend of Caesar's; anyone who makes himself king is defying Caesar.' ",
            13: "Hearing these words, Pilate had Jesus brought out, and seated him on the chair of judgement at a place called the Pavement, in Hebrew Gabbatha. ",
            14: "It was the Day of Preparation, about the sixth hour. 'Here is your king,' said Pilate to the Jews. ",
            15: "But they shouted, 'Away with him, away with him, crucify him.' Pilate said, 'Shall I crucify your king?' The chief priests answered, 'We have no king except Caesar.' ",
            16: "So at that Pilate handed him over to them to be crucified. They then took charge of Jesus, ",
            17: "and carrying his own cross he went out to the Place of the Skull or, as it is called in Hebrew, Golgotha, ",
            18: "where they crucified him with two others, one on either side, Jesus being in the middle. ",
            19: "Pilate wrote out a notice and had it fixed to the cross; it ran: 'Jesus the Nazarene, King of the Jews'. ",
            20: "This notice was read by many of the Jews, because the place where Jesus was crucified was near the city, and the writing was in Hebrew, Latin and Greek. ",
            21: """So the Jewish chief priests said to Pilate, 'You should not write "King of the Jews", but that the man said, "I am King of the Jews". ' """,
            22: "Pilate answered, 'What I have written, I have written.' ",
            23: "When the soldiers had finished crucifying Jesus they took his clothing and divided it into four shares, one for each soldier. His undergarment was seamless, woven in one piece from neck to hem; ",
            24: "so they said to one another, 'Instead of tearing it, let's throw dice to decide who is to have it.' In this way the words of scripture were fulfilled: They divide my garments among them and cast lots for my clothes. That is what the soldiers did. ",
            25: "Near the cross of Jesus stood his mother and his mother's sister, Mary the wife of Clopas, and Mary of Magdala. ",
            26: "Seeing his mother and the disciple whom he loved standing near her, Jesus said to his mother, 'Woman, this is your son.' ",
            27: "Then to the disciple he said, 'This is your mother.' And from that hour the disciple took her into his home. ",
            28: "After this, Jesus knew that everything had now been completed and, so that the scripture should be completely fulfilled, he said: I am thirsty. ",
            29: "A jar full of sour wine stood there; so, putting a sponge soaked in the wine on a hyssop stick, they held it up to his mouth. ",
            30: "After Jesus had taken the wine he said, 'It is fulfilled'; and bowing his head he gave up his spirit. ",
            31: "It was the Day of Preparation, and to avoid the bodies' remaining on the cross during the Sabbath -- since that Sabbath was a day of special solemnity -- the Jews asked Pilate to have the legs broken and the bodies taken away. ",
            32: "Consequently the soldiers came and broke the legs of the first man who had been crucified with him and then of the other. ",
            33: "When they came to Jesus, they saw he was already dead, and so instead of breaking his legs ",
            34: "one of the soldiers pierced his side with a lance; and immediately there came out blood and water. ",
            35: "This is the evidence of one who saw it -- true evidence, and he knows that what he says is true -- and he gives it so that you may believe as well. ",
            36: "Because all this happened to fulfil the words of scripture: Not one bone of his will be broken; ",
            37: "and again, in another place scripture says: They will look to the one whom they have pierced. ",
            38: "After this, Joseph of Arimathaea, who was a disciple of Jesus -- though a secret one because he was afraid of the Jews -- asked Pilate to let him remove the body of Jesus. Pilate gave permission, so they came and took it away. ",
            39: "Nicodemus came as well -- the same one who had first come to Jesus at night-time -- and he brought a mixture of myrrh and aloes, weighing about a hundred pounds. ",
            40: "They took the body of Jesus and bound it in linen cloths with the spices, following the Jewish burial custom. ",
            41: "At the place where he had been crucified there was a garden, and in this garden a new tomb in which no one had yet been buried. ",
            42: "Since it was the Jewish Day of Preparation and the tomb was nearby, they laid Jesus there."
        }

        chapter_20 = {
            1: "It was very early on the first day of the week and still dark, when Mary of Magdala came to the tomb. She saw that the stone had been moved away from the tomb ",
            2: "and came running to Simon Peter and the other disciple, the one whom Jesus loved. 'They have taken the Lord out of the tomb,' she said, 'and we don't know where they have put him.' ",
            3: "So Peter set out with the other disciple to go to the tomb. ",
            4: "They ran together, but the other disciple, running faster than Peter, reached the tomb first; ",
            5: "he bent down and saw the linen cloths lying on the ground, but did not go in. ",
            6: "Simon Peter, following him, also came up, went into the tomb, saw the linen cloths lying on the ground ",
            7: "and also the cloth that had been over his head; this was not with the linen cloths but rolled up in a place by itself. ",
            8: "Then the other disciple who had reached the tomb first also went in; he saw and he believed. ",
            9: "Till this moment they had still not understood the scripture, that he must rise from the dead. ",
            10: "The disciples then went back home. ",
            11: "But Mary was standing outside near the tomb, weeping. Then, as she wept, she stooped to look inside, ",
            12: "and saw two angels in white sitting where the body of Jesus had been, one at the head, the other at the feet. ",
            13: "They said, 'Woman, why are you weeping?' 'They have taken my Lord away,' she replied, 'and I don't know where they have put him.' ",
            14: "As she said this she turned round and saw Jesus standing there, though she did not realise that it was Jesus. ",
            15: "Jesus said to her, 'Woman, why are you weeping? Who are you looking for?' Supposing him to be the gardener, she said, 'Sir, if you have taken him away, tell me where you have put him, and I will go and remove him.' ",
            16: "Jesus said, 'Mary!' She turned round then and said to him in Hebrew, 'Rabbuni!' -- which means Master. ",
            17: "Jesus said to her, 'Do not cling to me, because I have not yet ascended to the Father. But go to the brothers, and tell them: I am ascending to my Father and your Father, to my God and your God.' ",
            18: "So Mary of Magdala told the disciples, 'I have seen the Lord,' and that he had said these things to her. ",
            19: "In the evening of that same day, the first day of the week, the doors were closed in the room where the disciples were, for fear of the Jews. Jesus came and stood among them. He said to them, 'Peace be with you,' ",
            20: "and, after saying this, he showed them his hands and his side. The disciples were filled with joy at seeing the Lord, ",
            21: "and he said to them again, 'Peace be with you. 'As the Father sent me, so am I sending you.' ",
            22: "After saying this he breathed on them and said: Receive the Holy Spirit. ",
            23: "If you forgive anyone's sins, they are forgiven; if you retain anyone's sins, they are retained. ",
            24: "Thomas, called the Twin, who was one of the Twelve, was not with them when Jesus came. ",
            25: "So the other disciples said to him, 'We have seen the Lord,' but he answered, 'Unless I can see the holes that the nails made in his hands and can put my finger into the holes they made, and unless I can put my hand into his side, I refuse to believe.' ",
            26: "Eight days later the disciples were in the house again and Thomas was with them. The doors were closed, but Jesus came in and stood among them. 'Peace be with you,' he said. ",
            27: "Then he spoke to Thomas, 'Put your finger here; look, here are my hands. Give me your hand; put it into my side. Do not be unbelieving any more but believe.' ",
            28: "Thomas replied, 'My Lord and my God!' ",
            29: "Jesus said to him: You believe because you can see me. Blessed are those who have not seen and yet believe. ",
            30: "There were many other signs that Jesus worked in the sight of the disciples, but they are not recorded in this book. ",
            31: "These are recorded so that you may believe that Jesus is the Christ, the Son of God, and that believing this you may have life through his name."
        }

        chapter_21 = {
            1: "Later on, Jesus revealed himself again to the disciples. It was by the Sea of Tiberias, and it happened like this: ",
            2: "Simon Peter, Thomas called the Twin, Nathanael from Cana in Galilee, the sons of Zebedee and two more of his disciples were together. ",
            3: "Simon Peter said, 'I'm going fishing.' They replied, 'We'll come with you.' They went out and got into the boat but caught nothing that night. ",
            4: "When it was already light, there stood Jesus on the shore, though the disciples did not realise that it was Jesus. ",
            5: "Jesus called out, 'Haven't you caught anything, friends?' And when they answered, 'No,' ",
            6: "he said, 'Throw the net out to starboard and you'll find something.' So they threw the net out and could not haul it in because of the quantity of fish. ",
            7: "The disciple whom Jesus loved said to Peter, 'It is the Lord.' At these words, 'It is the Lord,' Simon Peter tied his outer garment round him (for he had nothing on) and jumped into the water. ",
            8: "The other disciples came on in the boat, towing the net with the fish; they were only about a hundred yards from land. ",
            9: "As soon as they came ashore they saw that there was some bread there and a charcoal fire with fish cooking on it. ",
            10: "Jesus said, 'Bring some of the fish you have just caught.' ",
            11: "Simon Peter went aboard and dragged the net ashore, full of big fish, one hundred and fifty-three of them; and in spite of there being so many the net was not broken. ",
            12: "Jesus said to them, 'Come and have breakfast.' None of the disciples was bold enough to ask, 'Who are you?'. They knew quite well it was the Lord. ",
            13: "Jesus then stepped forward, took the bread and gave it to them, and the same with the fish. ",
            14: "This was the third time that Jesus revealed himself to the disciples after rising from the dead. ",
            15: "When they had eaten, Jesus said to Simon Peter, 'Simon son of John, do you love me more than these others do?' He answered, 'Yes, Lord, you know I love you.' Jesus said to him, 'Feed my lambs.' ",
            16: "A second time he said to him, 'Simon son of John, do you love me?' He replied, 'Yes, Lord, you know I love you.' Jesus said to him, 'Look after my sheep.' ",
            17: "Then he said to him a third time, 'Simon son of John, do you love me?' Peter was hurt that he asked him a third time, 'Do you love me?' and said, 'Lord, you know everything; you know I love you.' Jesus said to him, 'Feed my sheep. ",
            18: "In all truth I tell you, when you were young you put on your own belt and walked where you liked; but when you grow old you will stretch out your hands, and somebody else will put a belt round you and take you where you would rather not go.' ",
            19: "In these words he indicated the kind of death by which Peter would give glory to God. After this he said, 'Follow me.' ",
            20: "Peter turned and saw the disciple whom Jesus loved following them -- the one who had leant back close to his chest at the supper and had said to him, 'Lord, who is it that will betray you?' ",
            21: "Seeing him, Peter said to Jesus, 'What about him, Lord?' ",
            22: "Jesus answered, 'If I want him to stay behind till I come, what does it matter to you? You are to follow me.' ",
            23: "The rumour then went out among the brothers that this disciple would not die. Yet Jesus had not said to Peter, 'He will not die,' but, 'If I want him to stay behind till I come.' ",
            24: "This disciple is the one who vouches for these things and has written them down, and we know that his testimony is true. ",
            25: "There was much else that Jesus did; if it were written down in detail, I do not suppose the world itself would hold all the books that would be written."
        }

        chapters = {
            1: chapter_1,
            2: chapter_2,
            3: chapter_3,
            4: chapter_4,
            5: chapter_5,
            6: chapter_6,
            7: chapter_7,
            8: chapter_8,
            9: chapter_9,
            10: chapter_10,
            11: chapter_11,
            12: chapter_12,
            13: chapter_13,
            14: chapter_14,
            15: chapter_15,
            16: chapter_16,
            17: chapter_17,
            18: chapter_18,
            19: chapter_19,
            20: chapter_20,
            21: chapter_21,
        }

        return get_text(chapters, chapter, verse_start, verse_end, 21)
