# -*- coding: utf-8 -*-

# Copyright 2020 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

import abc
import typing
import pkg_resources

from google import auth  # type: ignore
from google.api_core import exceptions  # type: ignore
from google.api_core import gapic_v1    # type: ignore
from google.api_core import retry as retries  # type: ignore
from google.auth import credentials  # type: ignore

from oceanbolt.com.tonnage_v3.types import service


try:
    DEFAULT_CLIENT_INFO = gapic_v1.client_info.ClientInfo(
        gapic_version=pkg_resources.get_distribution(
            'oceanbolt-com-tonnage',
        ).version,
    )
except pkg_resources.DistributionNotFound:
    DEFAULT_CLIENT_INFO = gapic_v1.client_info.ClientInfo()

class TonnageServiceTransport(abc.ABC):
    """Abstract transport class for TonnageService."""

    AUTH_SCOPES = (
    )

    def __init__(
            self, *,
            host: str = 'api.oceanbolt.com',
            credentials: credentials.Credentials = None,
            credentials_file: typing.Optional[str] = None,
            scopes: typing.Optional[typing.Sequence[str]] = AUTH_SCOPES,
            quota_project_id: typing.Optional[str] = None,
            client_info: gapic_v1.client_info.ClientInfo = DEFAULT_CLIENT_INFO,
            **kwargs,
            ) -> None:
        """Instantiate the transport.

        Args:
            host (Optional[str]): The hostname to connect to.
            credentials (Optional[google.auth.credentials.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify the application to the service; if none
                are specified, the client will attempt to ascertain the
                credentials from the environment.
            credentials_file (Optional[str]): A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is mutually exclusive with credentials.
            scope (Optional[Sequence[str]]): A list of scopes.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            client_info (google.api_core.gapic_v1.client_info.ClientInfo):	
                The client info used to send a user-agent string along with	
                API requests. If ``None``, then default info will be used.	
                Generally, you only need to set this if you're developing	
                your own client library.
        """
        # Save the hostname. Default to port 443 (HTTPS) if none is specified.
        if ':' not in host:
            host += ':443'
        self._host = host

        # If no credentials are provided, then determine the appropriate
        # defaults.
        if credentials and credentials_file:
            raise exceptions.DuplicateCredentialArgs("'credentials_file' and 'credentials' are mutually exclusive")

        if credentials_file is not None:
            credentials, _ = auth.load_credentials_from_file(
                                credentials_file,
                                scopes=scopes,
                                quota_project_id=quota_project_id
                            )

        elif credentials is None:
            credentials, _ = auth.default(scopes=scopes, quota_project_id=quota_project_id)

        # Save the credentials.
        self._credentials = credentials

        # Lifted into its own function so it can be stubbed out during tests.
        self._prep_wrapped_messages(client_info)

    def _prep_wrapped_messages(self, client_info):
        # Precompute the wrapped methods.
        self._wrapped_methods = {
            self.get_tonnage_zone_count: gapic_v1.method.wrap_method(
                self.get_tonnage_zone_count,
                default_timeout=None,
                client_info=client_info,
            ),
            self.get_tonnage_fleet_speed: gapic_v1.method.wrap_method(
                self.get_tonnage_fleet_speed,
                default_timeout=None,
                client_info=client_info,
            ),
            self.get_global_tonnage_status: gapic_v1.method.wrap_method(
                self.get_global_tonnage_status,
                default_timeout=None,
                client_info=client_info,
            ),
            self.get_tonnage_fleet_status: gapic_v1.method.wrap_method(
                self.get_tonnage_fleet_status,
                default_timeout=None,
                client_info=client_info,
            ),
            self.get_tonnage_fleet_growth: gapic_v1.method.wrap_method(
                self.get_tonnage_fleet_growth,
                default_timeout=None,
                client_info=client_info,
            ),
            self.get_tonnage_chinese_waters: gapic_v1.method.wrap_method(
                self.get_tonnage_chinese_waters,
                default_timeout=None,
                client_info=client_info,
            ),
            self.get_tonnage_zone_changes: gapic_v1.method.wrap_method(
                self.get_tonnage_zone_changes,
                default_timeout=None,
                client_info=client_info,
            ),
            self.get_tonnage_basin_count: gapic_v1.method.wrap_method(
                self.get_tonnage_basin_count,
                default_timeout=None,
                client_info=client_info,
            ),

        }

    @property
    def get_tonnage_zone_count(self) -> typing.Callable[
            [service.GetTonnageDataRequest],
            typing.Union[
                service.GetTonnageZoneCountResponse,
                typing.Awaitable[service.GetTonnageZoneCountResponse]
            ]]:
        raise NotImplementedError()

    @property
    def get_tonnage_fleet_speed(self) -> typing.Callable[
            [service.GetTonnageDataRequest],
            typing.Union[
                service.GetFleetSpeedResponse,
                typing.Awaitable[service.GetFleetSpeedResponse]
            ]]:
        raise NotImplementedError()

    @property
    def get_global_tonnage_status(self) -> typing.Callable[
            [service.GetTonnageDataRequest],
            typing.Union[
                service.GetGlobalTonnageStatusResponse,
                typing.Awaitable[service.GetGlobalTonnageStatusResponse]
            ]]:
        raise NotImplementedError()

    @property
    def get_tonnage_fleet_status(self) -> typing.Callable[
            [service.GetTonnageFleetRequest],
            typing.Union[
                service.GetTonnageFleetStatusResponse,
                typing.Awaitable[service.GetTonnageFleetStatusResponse]
            ]]:
        raise NotImplementedError()

    @property
    def get_tonnage_fleet_growth(self) -> typing.Callable[
            [service.GetTonnageFleetRequest],
            typing.Union[
                service.GetTonnageFleetGrowthResponse,
                typing.Awaitable[service.GetTonnageFleetGrowthResponse]
            ]]:
        raise NotImplementedError()

    @property
    def get_tonnage_chinese_waters(self) -> typing.Callable[
            [service.TonnageChineseWatersRequest],
            typing.Union[
                service.TonnageChineseWatersResponse,
                typing.Awaitable[service.TonnageChineseWatersResponse]
            ]]:
        raise NotImplementedError()

    @property
    def get_tonnage_zone_changes(self) -> typing.Callable[
            [service.GetTonnageZoneChangesRequest],
            typing.Union[
                service.GetTonnageZoneChangesResponse,
                typing.Awaitable[service.GetTonnageZoneChangesResponse]
            ]]:
        raise NotImplementedError()

    @property
    def get_tonnage_basin_count(self) -> typing.Callable[
            [service.GetTonnageBasinRequest],
            typing.Union[
                service.GetTonnageBasinResponse,
                typing.Awaitable[service.GetTonnageBasinResponse]
            ]]:
        raise NotImplementedError()


__all__ = (
    'TonnageServiceTransport',
)
