/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.client;

import java.io.Closeable;
import java.io.IOException;
import java.util.Collection;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.atomic.AtomicInteger;
import org.eclipse.jetty.client.api.Connection;
import org.eclipse.jetty.client.api.Destination;
import org.eclipse.jetty.util.BlockingArrayQueue;
import org.eclipse.jetty.util.Promise;
import org.eclipse.jetty.util.component.ContainerLifeCycle;
import org.eclipse.jetty.util.component.Dumpable;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;

public class ConnectionPool
implements Closeable,
Dumpable {
    protected static final Logger LOG = Log.getLogger(ConnectionPool.class);
    private final AtomicInteger connectionCount = new AtomicInteger();
    private final Destination destination;
    private final int maxConnections;
    private final Promise<Connection> connectionPromise;
    private final BlockingDeque<Connection> idleConnections;
    private final BlockingQueue<Connection> activeConnections;

    public ConnectionPool(Destination destination, int maxConnections, Promise<Connection> connectionPromise) {
        this.destination = destination;
        this.maxConnections = maxConnections;
        this.connectionPromise = connectionPromise;
        this.idleConnections = new LinkedBlockingDeque<Connection>(maxConnections);
        this.activeConnections = new BlockingArrayQueue(maxConnections);
    }

    public BlockingQueue<Connection> getIdleConnections() {
        return this.idleConnections;
    }

    public BlockingQueue<Connection> getActiveConnections() {
        return this.activeConnections;
    }

    public Connection acquire() {
        Connection connection = this.acquireIdleConnection();
        if (connection == null) {
            connection = this.tryCreate();
        }
        return connection;
    }

    private Connection tryCreate() {
        int next;
        int current;
        do {
            if ((next = (current = this.connectionCount.get()) + 1) <= this.maxConnections) continue;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Max connections {}/{} reached", new Object[]{current, this.maxConnections});
            }
            return this.acquireIdleConnection();
        } while (!this.connectionCount.compareAndSet(current, next));
        if (LOG.isDebugEnabled()) {
            LOG.debug("Connection {}/{} creation", new Object[]{next, this.maxConnections});
        }
        this.destination.newConnection(new Promise<Connection>(){

            public void succeeded(Connection connection) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Connection {}/{} creation succeeded {}", new Object[]{next, ConnectionPool.this.maxConnections, connection});
                }
                if (ConnectionPool.this.activate(connection)) {
                    ConnectionPool.this.connectionPromise.succeeded((Object)connection);
                }
            }

            public void failed(Throwable x) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Connection " + next + "/" + ConnectionPool.this.maxConnections + " creation failed", x);
                }
                ConnectionPool.this.connectionCount.decrementAndGet();
                ConnectionPool.this.connectionPromise.failed(x);
            }
        });
        return this.acquireIdleConnection();
    }

    private Connection acquireIdleConnection() {
        Connection connection = (Connection)this.idleConnections.pollFirst();
        if (connection == null) {
            return null;
        }
        return this.activate(connection) ? connection : null;
    }

    private boolean activate(Connection connection) {
        if (this.activeConnections.offer(connection)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Connection active {}", new Object[]{connection});
            }
            this.acquired(connection);
            return true;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Connection active overflow {}", new Object[]{connection});
        }
        connection.close();
        return false;
    }

    protected void acquired(Connection connection) {
    }

    public boolean release(Connection connection) {
        this.released(connection);
        if (this.activeConnections.remove(connection)) {
            if (this.idleConnections.offerFirst(connection)) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Connection idle {}", new Object[]{connection});
                }
                return true;
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("Connection idle overflow {}", new Object[]{connection});
            }
            connection.close();
        }
        return false;
    }

    protected void released(Connection connection) {
    }

    public boolean remove(Connection connection) {
        boolean removed;
        boolean activeRemoved = this.activeConnections.remove(connection);
        boolean idleRemoved = this.idleConnections.remove(connection);
        if (!idleRemoved) {
            this.released(connection);
        }
        boolean bl = removed = activeRemoved || idleRemoved;
        if (removed) {
            int pooled = this.connectionCount.decrementAndGet();
            if (LOG.isDebugEnabled()) {
                LOG.debug("Connection removed {} - pooled: {}", new Object[]{connection, pooled});
            }
        }
        return removed;
    }

    public boolean isActive(Connection connection) {
        return this.activeConnections.contains(connection);
    }

    public boolean isIdle(Connection connection) {
        return this.idleConnections.contains(connection);
    }

    public boolean isEmpty() {
        return this.connectionCount.get() == 0;
    }

    @Override
    public void close() {
        for (Connection connection : this.idleConnections) {
            connection.close();
        }
        this.idleConnections.clear();
        for (Connection connection : this.activeConnections) {
            connection.close();
        }
        this.activeConnections.clear();
        this.connectionCount.set(0);
    }

    public String dump() {
        return ContainerLifeCycle.dump((Dumpable)this);
    }

    public void dump(Appendable out, String indent) throws IOException {
        ContainerLifeCycle.dumpObject((Appendable)out, (Object)this);
        ContainerLifeCycle.dump((Appendable)out, (String)indent, (Collection[])new Collection[]{this.activeConnections, this.idleConnections});
    }

    public String toString() {
        return String.format("%s[c=%d/%d,a=%d,i=%d]", this.getClass().getSimpleName(), this.connectionCount.get(), this.maxConnections, this.activeConnections.size(), this.idleConnections.size());
    }
}

