import re
from collections import defaultdict
from dataclasses import _MISSING_TYPE, dataclass, is_dataclass
from logging import getLogger
from os import environ
from pathlib import Path
from typing import Dict, List

import yaml
from yaml.loader import SafeLoader

from hartware_lib.utils.casing import pascal_to_snake_case

logger = getLogger(__name__)


def transform(target_type, value):
    if value is None:
        return value

    if target_type is bool:
        if not isinstance(value, str):
            value = str(value)
        if value.lower() in ("0", "no", "false"):
            return False
        if value.lower() in ("1", "yes", "true"):
            return True

        raise Exception(f"Can't parse '{value}' as boolean")

    return target_type(value)


def load_settings(  # noqa: C901
    cls, config: Dict, path: List[str] = [], hide_prefix: bool = False
):
    settings = {}

    if not hide_prefix:
        path = path + [
            getattr(
                cls,
                "_prefix",
                pascal_to_snake_case(cls.__name__.replace("Settings", "")),
            )
        ]

    for option_name, option in cls.__dataclass_fields__.items():
        env_path = "_".join(path + [option_name]).upper()
        config_value = config.get(option_name)

        if is_dataclass(option.type):
            if option.metadata.get("as_list"):
                config_values = {
                    i: load_settings(
                        option.type,
                        _config_value,
                        path + [option_name, str(i)],
                        hide_prefix=True,
                    )
                    for i, _config_value in enumerate(config_value or [])
                }

                extra_conf: Dict[int, Dict] = defaultdict(dict)
                for k, v in environ.items():
                    if env_path not in k:
                        continue

                    m = re.match(rf"{env_path}_(?P<index>\d+)_(?P<attr>\w+)", k)
                    if not m:
                        logger.warning(f"Could not parse key: {k}")
                        continue
                    index = int(m.group("index"))
                    if index in config_values:
                        continue
                    extra_conf[index][m.group("attr")] = v
                for i, config_value in extra_conf.items():
                    try:
                        config_values[i] = load_settings(
                            option.type,
                            config_value,
                            path + [option_name, str(i)],
                            hide_prefix=True,
                        )
                    except Exception as exc:
                        logger.warning(f"Incomplete settings: {exc}")
                        # raise Exception(f"Incomplete settings: {exc}")
                        # print(f"Info: {exc}")

                config_value = [
                    i[1] for i in sorted(config_values.items(), key=lambda x: x[0])
                ]
            else:
                config_value = load_settings(option.type, config_value or {}, path)
        else:
            if env_value := environ.get(env_path):
                config_value = env_value
            if not isinstance(config_value, option.type):
                config_value = transform(option.type, config_value)
        if config_value is None:
            if not isinstance(option.default, _MISSING_TYPE):
                config_value = option.default
            elif not isinstance(option.default_factory, _MISSING_TYPE):
                config_value = option.default_factory()

            if not isinstance(config_value, option.type):
                raise Exception(f"No value for {'.'.join(path + [option_name])}")

        settings[option_name] = config_value

    return cls(**settings)


def load_settings_from_file(base_dataclass, config_path: Path):
    with open(config_path) as f:
        config = yaml.load(f, Loader=SafeLoader)

        return load_settings(base_dataclass, config or {})


@dataclass
class RabbitMQSettings:
    _prefix = "rabbit_mq"

    username: str
    password: str
    host: str
    port: int = 5672


@dataclass
class SlackBotSettings:
    api_token: str
    default_channel: str
