#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
:Title: This modules contains read quality check function

:Created: May 19, 2021

:Author: Naveen Duhan
"""


import os
import sys
import shutil
import subprocess
import pkg_resources
from pyseqrna.pyseqrna_utils import PyseqrnaLogger
from pyseqrna import pyseqrna_utils as pu

log = PyseqrnaLogger(mode='a', log='qc')

def fastqcRun(sampleDict=None, configFile=None,slurm=False, mem=10, cpu=8, task=1, pairedEND =False, afterTrim=False, outDir=None, dep=''):

    """This function perform fastqc quality using FastQC (http://www.bioinformatics.babraham.ac.uk/projects/fastqc/)
           
    :param sampleDict: Samples dictionary generated by *pyseqrna_utils.read_input_file* function.

    :param configFile: Parameters file for FastQC tool. Default from pyseqrna params

    :param slurm: True to enable slurm job scheduling on HPC

    :param mem: Memory if slurm True

    :param cpu: Number of threads to use if slurm True. 

    :param task: Number of tasks per cpu if slurm True.

    :param pairedEND: True if samples are paired

    :param afterTrim: True if checking quality after trimming.
        
    :param outDir: Output directory for results. Default is current working directory.
    
    :param dep: Slurm job dependency.
    """

    if configFile != None:

        config = pu.parse_config_file(configFile)

    else:
        stream = pkg_resources.resource_stream('pyseqrna', "param/fastqc.ini")

        config = pu.parse_config_file(stream.name)

        log.info("Using default config file fastqc.ini")

    fastqc_config = config[list(config.keys())[0]]

    if afterTrim:

        out ='trim_fastqc_results'

    else:

        out = "fastqc_results"

    if os.path.exists(outDir):

        output1 = os.path.join(outDir,out)

        output = pu.make_directory(output1)

    else:
        
        output = pu.make_directory(out)

    const = ['threads', '-t']  # require when change number of CPUs

    if not slurm:

        fastqc_config = pu.replace_cpu(fastqc_config, const)

    args = ' '.join(fastqc_config[0:])

    job_id = []

    fastqcOut = {}

    for key, value in sampleDict.items():  # Iterate thorough total number of samloges

        if pairedEND:
            try:
                input1 = value[2]
                input2 = value[3]

            except Exception:

                log.error(
                    "logease provide a paired END samloge file or input Path is wrong")

            inputPair = pu.get_basename(input1)+" and " + pu.get_basename(input2)
            
            if pu.get_file_extension(value[2]) == "gz":
                r1 = pu.get_basename(value[2])+".zip"
                r2 = pu.get_basename(value[3])+".zip"
            
                fastqcOut[key]=[value[0], value[1], r1, r2]
            else: 

                r1 = value[2]+".zip"
                r2 = value[3]+".zip"
                fastqcOut[key]=[value[0], value[1], r1, r2]
        else:
            try:
                inputFile = value[2]

            except Exception:

                log.error("logease provide a valid input data path")
            
            inputPair = pu.get_basename(value[2])

        

        execPATH = shutil.which('fastqc')
              # get absolute path of flexbar

        if execPATH is None:

            log.error("fastqc command not found in path")
            sys.exit()
        else:

            if pairedEND:

                fastqcCmd = f"{execPATH} -o {output} {args} {input1} {input2}"

            else:

                fastqcCmd = f"{execPATH} -o {output} {args} {inputFile}"
            

            if slurm == True:  # check if slurm job scheduling is enabled or not

                try:
                    job = pu.clusterRun(job_name=
                        'fastqc', sout= os.path.join(output,"fastqc.out"), serror= os.path.join(output,"fastqc.err"), command=fastqcCmd, mem=mem, cpu=cpu, tasks=task, dep=dep)

                    job_id.append(job)

                    log.info(
                        "Job submitted on slurm successfully for {} with {}".format(inputPair, job))

                except Exception:

                    log.error("Slurm job sumission failed")

            else:

                try:
                
                    with open(os.path.join(output,"fastqc.out"), 'w+') as fout:
                        with open(os.path.join(output,"fastqc.err"), 'w+') as ferr:
                            job = subprocess.call(fastqcCmd, shell=True, stdout=fout, stderr=ferr)

                            job_id.append(" ")    

                            log.info("Job successfully completed for {} with status {}".format(inputPair, job))

                except Exception:

                    log.info("Job submition failed for {} error present in fastqc.err  ".format(pu.get_basename(out)))
                

    return job_id, fastqcOut

