from pathlib import Path
from typing import Any, Dict, List, NamedTuple
import pandas
import vcf


class VCFColumns(NamedTuple):
	sequence_id: str = 'seq id'
	position: str = 'position'
	alternate: str = 'alt'
	reference: str = 'ref'
	quality: str = 'quality'
	depth: str = 'readDepth'
	variant_type: str = 'variantType'

VCFColumns = VCFColumns() # Overwritting the class, but we only need the object here. This is entirely because we want the class iterable.


def _convert_record_to_dictionary(record: Any) -> Dict[str, str]:
	alt = "".join(str(i) for i in record.ALT)
	ref = "".join(str(i) for i in record.REF)
	qual = record.QUAL
	depth = record.INFO.get('DP')
	record_data = {
		VCFColumns.sequence_id:  record.CHROM,
		VCFColumns.position:     record.POS,
		VCFColumns.alternate:    alt,
		VCFColumns.reference:    ref,
		VCFColumns.quality:      qual,
		VCFColumns.depth:        depth,
		VCFColumns.variant_type: record.var_type
	}

	return record_data


def _convert_vcf_to_table(vcf_filename: Path) -> List[Dict[str, Any]]:
	"""Converts all records in a vcf file into a list of dictionaries."""
	table: List[Dict[str, str]] = list()
	seen_positions = set()
	with vcf_filename.open('r') as file1:
		vcf_reader = vcf.Reader(file1)
		for record in vcf_reader:
			data = _convert_record_to_dictionary(record)
			# VCF files sometimes record separate mutations as occuring at the same position.
			# The gd file will instead increment the second mutations position by 1. Do this to maintain compatibility.
			if (data['seq id'], data['position']) in seen_positions:
				data['position'] += 1
			seen_positions.add((data['seq id'], data['position']))
			table.append(data)
	return table


def parse_vcf_file(filename: Path, set_index: bool = True) -> pandas.DataFrame:
	"""
		Converts the VCF file generated by breseq into a pandas Dataframe.
	Parameters
	----------
	filename: Path
		Either a folder containing a single breseq run or a path to the vcf file itself.
	set_index:bool; default True
		Whether to set the index of the dataframe.
	Returns
	-------
	pandas.DataFrame
		- Index -> (VCFColumns.sample_name, VCFColumns.sequence_id, VCFColumns.position)
		- Columns-> VCFColumns
	"""

	table = _convert_vcf_to_table(filename)

	# Columns are defined in VCFColumns
	vcf_df: pandas.DataFrame = pandas.DataFrame(table)

	# Order the columns correctly
	filtered_df = vcf_df[list(VCFColumns)]

	if set_index:
		filtered_df.set_index(keys = [VCFColumns.sequence_id, VCFColumns.position], inplace = True)

	return filtered_df
