# -*- coding: utf-8 -*-
#
# This file is part of SENAITE.CORE.
#
# SENAITE.CORE is free software: you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation, version 2.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc., 51
# Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
#
# Copyright 2018-2021 by it's authors.
# Some rights reserved, see README and LICENSE.

import collections

from Products.ATContentTypes.content import schemata
from Products.Archetypes import atapi
from bika.lims import api
from bika.lims import bikaMessageFactory as _
from bika.lims.browser.bika_listing import BikaListingView
from bika.lims.config import PROJECTNAME
from bika.lims.interfaces import ISuppliers
from bika.lims.permissions import AddSupplier
from bika.lims.utils import get_link
from plone.app.folder.folder import ATFolder
from plone.app.folder.folder import ATFolderSchema
from senaite.core.interfaces import IHideActionsMenu
from zope.interface.declarations import implements


# TODO: Separate content and view into own modules!


class SuppliersView(BikaListingView):

    def __init__(self, context, request):
        super(SuppliersView, self).__init__(context, request)

        self.catalog = "senaite_catalog_setup"

        self.contentFilter = {
            "portal_type": "Supplier",
            "sort_on": "sortable_title",
            "sort_order": "ascending",
        }

        self.context_actions = {
            _("Add"): {
                "url": "createObject?type_name=Supplier",
                "permission": AddSupplier,
                "icon": "++resource++bika.lims.images/add.png"}
        }

        self.title = self.context.translate(_("Suppliers"))
        self.icon = "{}/{}".format(
            self.portal_url,
            "/++resource++bika.lims.images/supplier_big.png"
        )

        self.description = ""

        self.show_select_row = False
        self.show_select_column = True
        self.pagesize = 25

        self.columns = collections.OrderedDict((
            ("Name", {
                "title": _("Name"),
                "index": "sortable_title"}),
            ("Email", {
                "title": _("Email"),
                "toggle": True}),
            ("Phone", {
                "title": _("Phone"),
                "toggle": True}),
            ("Fax", {"title": _("Fax"),
                     "toggle": True}),
        ))

        self.review_states = [
            {
                "id": "default",
                "title": _("Active"),
                "contentFilter": {"is_active": True},
                "transitions": [{"id": "deactivate"}, ],
                "columns": self.columns.keys(),
            }, {
                "id": "inactive",
                "title": _("Inactive"),
                "contentFilter": {'is_active': False},
                "transitions": [{"id": "activate"}, ],
                "columns": self.columns.keys(),
            }, {
                "id": "all",
                "title": _("All"),
                "contentFilter": {},
                "columns": self.columns.keys(),
            },
        ]

    def folderitem(self, obj, item, index):
        """Service triggered each time an item is iterated in folderitems.

        The use of this service prevents the extra-loops in child objects.

        :obj: the instance of the class to be foldered
        :item: dict containing the properties of the object to be used by
            the template
        :index: current index of the item
        """
        obj = api.get_object(obj)
        name = obj.getName()
        email = obj.getEmailAddress()
        phone = obj.getPhone()
        fax = obj.getFax()
        url = obj.absolute_url()

        item["Name"] = name
        item["Email"] = email
        item["Phone"] = phone
        item["Fax"] = fax
        item["replace"]["Name"] = get_link(url, value=name)

        return item


schema = ATFolderSchema.copy()


class Suppliers(ATFolder):
    implements(ISuppliers, IHideActionsMenu)
    displayContentsTab = False
    schema = schema


schemata.finalizeATCTSchema(schema, folderish=True, moveDiscussion=False)
atapi.registerType(Suppliers, PROJECTNAME)
