import typing
import pandas as pd

from . import validation
from .validation_warning import ValidationWarning

class Column:
    def __init__(self, name: str, validations: typing.Iterable['validation._BaseValidation'] = [], allow_empty=False):
        """
        Creates a new Column object

        :param name: The column header that defines this column. This must be identical to the header used in the CSV/Data Frame you are validating.
        :param validations: An iterable of objects implementing _BaseValidation that will generate ValidationErrors
        :param allow_empty: True if an empty column is considered valid. False if we leave that logic up to the Validation
        """
        self.name = name
        self.validations = list(validations)
        self.allow_empty = allow_empty

    def validate(self, series: pd.Series) -> typing.List[ValidationWarning]:
        """
        Creates a list of validation errors using the Validation objects contained in the Column

        :param series: A pandas Series to validate
        :return: An iterable of ValidationError instances generated by the validation
        """
        return [error for validation in self.validations for error in validation.get_errors(series, self)]
