#<=====>#
# Description
#<=====>#
"""FSTrent Charts - Color Chart options for terminal output."""
__version__ = "1.0.0"


#<=====>#
# Known To Do List
#<=====>#



#<=====>#
# Imports
#<=====>#
import re
from fstrent_colors import cs, cp
import pandas as pd
import numpy as np
import pandas_ta as pta
import datetime
from decimal import Decimal
import pytz

#<=====>#
# Variables
#<=====>#


#<=====>#
# Assignments Pre
#<=====>#

'''
Single-Line Box Drawing Characters:
┬ (U+252C) — Top T-junction
┴ (U+2534) — Bottom T-junction

Double-Line Box Drawing Characters:
╦ (U+2566) — Top T-junction (double)
╩ (U+2569) — Bottom T-junction (double)

Single-Line Box Drawing Characters:
┼ (U+253C) — Intersection

Double-Line Box Drawing Characters:
╬ (U+256C) — Intersection (double)

╔ ========== ========== ========== ╦ ========== ========== ========== ╗
║                                  ║                                  ║
╠ ========== ========== ========== ╬ ========== ========== ========== ╣
║                                  ║                                  ║
║                                  ║                                  ║
║                                  ║                                  ║
║                                  ║                                  ║
╚ ========== ========== ========== ╩ ========== ========== ========== ╝
'''

'''
Other Vertical and Horizontal Line Variations:
╶ (U+2576) — Light horizontal line, starting right
╴ (U+2574) — Light horizontal line, starting left
╵ (U+2575) — Light vertical line, starting top
╷ (U+2577) — Light vertical line, starting bottom
'''

'''
Junctions and Intersections
┬ (U+252C) — Top T-junction
┼ (U+253C) — Intersection
┴ (U+2534) — Bottom T-junction
╦ (U+2566) — Top T-junction (double)
╬ (U+256C) — Intersection (double)
╩ (U+2569) — Bottom T-junction (double)
'''


'''
Style 1 - Single-Line Box Drawing Characters:
┌ (U+250C) — Upper left corner
├ (U+251C) — Left T-junction
└ (U+2514) — Lower left corner
┐ (U+2510) — Upper right corner
┤ (U+2524) — Right T-junction
┘ (U+2518) — Lower right corner
│ (U+2502) — Vertical line
─ (U+2500) — Horizontal line
┌──────────┐
├──────────┤
│          │
└──────────┘
'''


'''
Style 2 - Vertical and Horizontal Line Variations:
╭ (U+256D) — Rounded upper left corner
├ (U+251C) — Left T-junction
╰ (U+2570) — Rounded lower left corner
╮ (U+256E) — Rounded upper right corner
┤ (U+2524) — Right T-junction
╯ (U+256F) — Rounded lower right corner
│ (U+2502) — Vertical line
─ (U+2500) — Horizontal line
╭──────────╮
├──────────┤
│          │
╰──────────╯
'''
'''
Style 3 - Mixed Single/Double Line Characters:
╓ (U+2553) — Upper left corner (single horizontal, double vertical)
╠ (U+2560) — Left T-junction (double)
╙ (U+2559) — Lower left corner (single horizontal, double vertical)
╖ (U+2556) — Upper right corner (single horizontal, double vertical)
╣ (U+2563) — Right T-junction (double)
╜ (U+255C) — Lower right corner (single horizontal, double vertical)
║ (U+2551) — Vertical line (double)
─ (U+2500) — Horizontal line
╓──────────╖
╠──────────╣
║          ║
╙──────────╜
'''
'''
Style 4 - Mixed Single/Double Line Characters:
╒ (U+2552) — Upper left corner (double horizontal, single vertical)
├ (U+251C) — Left T-junction
╘ (U+2558) — Lower left corner (double horizontal, single vertical)
╕ (U+2555) — Upper right corner (double horizontal, single vertical)
┤ (U+2524) — Right T-junction
╛ (U+255B) — Lower right corner (double horizontal, single vertical)
│ (U+2502) — Vertical line
═ (U+2550) — Horizontal line (double)
╒══════════╕
├──────────┤
│          │
╘══════════╛
'''
'''
Style 5 - Double-Line Box Drawing Characters:
╔ (U+2554) — Upper left corner (double)
╠ (U+2560) — Left T-junction (double)
╚ (U+255A) — Lower left corner (double)
╗ (U+2557) — Upper right corner (double)
╣ (U+2563) — Right T-junction (double)
╝ (U+255D) — Lower right corner (double)
║ (U+2551) — Vertical line (double)
═ (U+2550) — Horizontal line (double)
╔══════════╗
╠══════════╣
║          ║
╚══════════╝
'''

#<=====>#
# Classes
#<=====>#

class CHRT:

	def __init__(
			self, 
			len_cnt=220, 
			align='left', 
			bold=True, 
			italic=False, 
			header_font_color='white', 
			header_bg_color='blue',
			font_color='#00FF00', 
			bg_color='#4B0082', 
			border_font_color='#00FF00', 
			border_bg_color='#4B0082', 
			style=2, 
			formatted=False, 
			in_json=None
		):
		self.len_cnt                    = len_cnt
		self.align                      = align
		self.bold                       = bold
		self.italic                     = italic
		self.header_font_color          = header_font_color
		self.header_bg_color            = header_bg_color
		self.font_color                 = font_color
		self.bg_color                   = bg_color
		self.border_font_color          = border_font_color
		self.border_bg_color            = border_bg_color
		self.style                      = style
		self.formatted                  = formatted
		self.in_json                    = self._prepare_data(in_json)

	#<=====>#

	def _prepare_data(self, in_json):
		if in_json is None:
			return None
		if isinstance(in_json, pd.DataFrame):
			return {'title': 'Data', 'data': in_json.to_dict(orient='records')}
		elif isinstance(in_json, dict) and 'data' in in_json and isinstance(in_json['data'], pd.DataFrame):
			in_json['data'] = in_json['data'].to_dict(orient='records')
		return in_json

	#<=====>#

	def chart_shapes(self, part, style):
		if style == 1:
			if part == 'top':
				l = '┌'
				s = '─'
				r = '┐'
			elif part == 'mid':
				l = '├'
				s = '─'
				r = '┤'
			elif part == 'row':
				l = '│'
				s = ' '
				r = '│'
			elif part == 'bottom':
				l = '└'
				s = '─'
				r = '┘'

		elif style == 2:
			if part == 'top':
				l = '╭'
				s = '─'
				r = '╮'
			elif part == 'mid':
				l = '├'
				s = '─'
				r = '┤'
			elif part == 'row':
				l = '│'
				s = ' '
				r = '│'
			elif part == 'bottom':
				l = '╰'
				s = '─'
				r = '╯'
		elif style == 3:
			if part == 'top':
				l = '╓'
				s = '─'
				r = '╖'
			elif part == 'mid':
				l = '╠'
				s = '═'
				r = '╣'
			elif part == 'row':
				l = '║'
				s = ' '
				r = '║'
			elif part == 'bottom':
				l = '╙'
				s = '─'
				r = '╜'
		elif style == 4:
			if part == 'top':
				l = '╒'
				s = '═'
				r = '╗'
			elif part == 'mid':
				l = '├'
				s = '─'
				r = '┤'
			elif part == 'row':
				l = '│'
				s = ' '
				r = '│'
			elif part == 'bottom':
				l = '╘'
				s = '═'
				r = '╛'
			if part == 'top':
				pass
			elif part == 'mid':
				pass
			elif part == 'row':
				pass
			elif part == 'bottom':
				pass
		elif style == 5:
			if part == 'top':
				l = '╔'
				s = '═'
				r = '╗'
			elif part == 'mid':
				l = '╠'
				s = '═'
				r = '╣'
			elif part == 'row':
				l = '║'
				s = ' '
				r = '║'
			elif part == 'bottom':
				l = '╚'
				s = '═'
				r = '╝'
		return l, s, r

	#<=====>#

	def chart_rep_str(self, s, border_font_color, border_bg_color, in_str=''):

		front_match = re.match(r'^(' + re.escape(s) + r'+)', in_str)
		rear_match  = re.search(r'(' + re.escape(s) + r'+)$', in_str)

		front_str = front_match.group(1) if front_match else ''
		rear_str  = rear_match.group(1) if rear_match else ''

		return front_str, rear_str

	#<=====>#

	def chart_embed(
			self, 
			l, 
			s, 
			r, 
			align=None, 
			bold=None, 
			italic=None, 
			header_font_color=None,
			header_bg_color=None,
			font_color=None, 
			bg_color=None, 
			border_font_color=None, 
			border_bg_color=None, 
			len_cnt=None, 
			formatted=None, 
			in_str=''
			):

		align = align if align is not None else self.align
		bold = bold if bold is not None else self.bold
		italic = italic if italic is not None else self.italic
		font_color = font_color or self.font_color
		bg_color = bg_color or self.bg_color
		border_font_color = border_font_color or self.border_font_color
		border_bg_color = border_bg_color or self.border_bg_color
		len_cnt = len_cnt or self.len_cnt
		formatted = formatted if formatted is not None else self.formatted
		in_str = in_str

		fore = cs(l, font_color=border_font_color, bg_color=border_bg_color)
		aft  = cs(r, font_color=border_font_color, bg_color=border_bg_color)

		if in_str == '':
			in_str = s * len_cnt
			disp_str = cs(text=in_str, font_color=border_font_color, bg_color=border_bg_color, bold=bold, italic=italic)
			disp_str = f"{fore}{disp_str}{aft}"
		else:
			in_str = f' {in_str} '
			if align == 'left':
				temp_str = f'{s*2}{in_str}'
				temp_str = self.rpad(temp_str, len_cnt, s)
			if align == 'right':
				temp_str = f'{in_str}{s*2}'
				temp_str = self.lpad(temp_str, len_cnt, s)
			if align == 'center':
				temp_str = f'{in_str}'
				temp_str = self.cpad(temp_str, len_cnt, s)

			front_str, rear_str = self.chart_rep_str(s, border_font_color, border_bg_color, in_str=temp_str)
			front_str = cs(text=front_str, font_color=border_font_color, bg_color=border_bg_color, bold=bold, italic=italic)
			rear_str  = cs(text=rear_str,  font_color=border_font_color, bg_color=border_bg_color, bold=bold, italic=italic)
			disp_str  = cs(text=in_str, font_color=font_color, bg_color=bg_color, bold=bold, italic=italic)

			disp_str = f"{fore}{front_str}{disp_str}{rear_str}{aft}"

		print(disp_str) 

	#<=====>#

	def chart_string(
			self, 
			l, 
			s, 
			r, 
			align, 
			bold, 
			italic, 
			header_font_color,
			header_bg_color,
			font_color, 
			bg_color, 
			border_font_color, 
			border_bg_color, 
			len_cnt, 
			formatted, 
			in_str=''
			):
		align = align if align is not None else self.align
		bold = bold if bold is not None else self.bold
		italic = italic if italic is not None else self.italic
		font_color = font_color or self.font_color
		bg_color = bg_color or self.bg_color
		border_font_color = border_font_color or self.border_font_color
		border_bg_color = border_bg_color or self.border_bg_color
		len_cnt = len_cnt or self.len_cnt
		formatted = formatted if formatted is not None else self.formatted
		in_str = in_str

		fore = cs(l, font_color=border_font_color, bg_color=border_bg_color, bold=bold, italic=italic)
		aft  = cs(r, font_color=border_font_color, bg_color=border_bg_color, bold=bold, italic=italic)

		true_len = self.display_length(in_str)

		if in_str == '':
			in_str = s * len_cnt
			disp_str = cs(text=in_str, font_color=border_font_color, bg_color=border_bg_color, bold=bold, italic=italic)
			disp_str = f"{fore}{disp_str}{aft}"
		else:
			if align == 'left':
				true_len = self.display_length(in_str)
				in_str = in_str + ' ' * (len_cnt - true_len)
			elif align == 'right':
				true_len = self.display_length(in_str)
				in_str = ' ' * (len_cnt - true_len) + in_str
			elif align == 'center':
				needed_pad_len = len_cnt - true_len
				lead_pad_len = int(needed_pad_len / 2)
				rear_pad_len = len_cnt - true_len - lead_pad_len
				in_str = ' ' * lead_pad_len + in_str + ' ' * rear_pad_len

			if not formatted:
				disp_str  = cs(text=in_str, font_color=font_color, bg_color=bg_color, bold=bold, italic=italic)
			else:
				disp_str = in_str  # If already formatted

			disp_str = f"{fore}{disp_str}{aft}"

		print(disp_str) 

	#<=====>#

	def lpad(self, in_str, length, pad_char):
		out_str = in_str.rjust(length, pad_char)
		return out_str

	#<=====>#

	def rpad(self, in_str, length, pad_char):
		out_str = in_str.ljust(length, pad_char)
		return out_str

	#<=====>#

	def cpad(self, in_str, length, pad_char):
		out_str = in_str.center(length, pad_char)
		return out_str

	#<=====>#

	def left(self, in_str, length):
		out_str = in_str[0:length]
		return out_str

	#<=====>#

	def right(self, in_str, length):
		out_str = in_str[length*-1:]
		return out_str

	#<=====>#

	def mid(self, in_str, start_position, length):
		out_str = in_str[start_position:start_position + length]
		return out_str

	#<=====>#

	def strip_formatting(self, in_str):
		# Define a regex pattern to match ANSI escape sequences
		ansi_escape = re.compile(r'''
			\x1B     # ESC
			\[       # CSI (Control Sequence Introducer)
			[0-?]*   # Parameter bytes
			[ -/]*   # Intermediate bytes
			[@-~]    # Final byte
			''', re.VERBOSE)
		strip_str = ansi_escape.sub('', in_str)
		return strip_str

	#<=====>#

	def is_even(self, number: int) -> bool:
		return number % 2 == 0

	#<=====>#

	def is_odd(self, number: int) -> bool:
		return number % 2 != 0

	#<=====>#

	def display_length(self, in_str):
		strip_str = self.strip_formatting(in_str)
		display_length = len(strip_str)
		return display_length

	#<=====>#

	def chart_top(
			self, 
			in_str='', 
			len_cnt=None, 
			align=None, 
			bold=None, 
			italic=None, 
			header_font_color=None,
			header_bg_color=None,
			font_color=None, 
			bg_color=None, 
			border_font_color=None, 
			border_bg_color=None,
			style=None,
			formatted=None
			):
		l, s, r = self.chart_shapes(part='top', style=style or self.style)
		self.chart_embed(
			l, s, r,
			align or self.align,
			bold if bold is not None else self.bold,
			italic if italic is not None else self.italic,
			header_font_color or self.header_font_color,
			header_bg_color or self.header_bg_color,
			font_color or self.font_color,
			bg_color or self.bg_color,
			border_font_color or self.border_font_color,
			border_bg_color or self.border_bg_color,
			len_cnt or self.len_cnt,
			formatted if formatted is not None else self.formatted,
			in_str=in_str
		)

	#<=====>#

	def chart_title(
			self, 
			in_str='', 
			len_cnt=None, 
			align=None, 
			bold=None, 
			italic=None, 
			header_font_color=None,
			header_bg_color=None,
			font_color=None, 
			bg_color=None, 
			border_font_color=None, 
			border_bg_color=None,
			style=None,
			formatted=None
			):
		l, s, r = self.chart_shapes(part='row', style=style or self.style)
		self.chart_string(
			l, s, r,
			align or self.align,
			bold if bold is not None else self.bold,
			italic if italic is not None else self.italic,
			header_font_color or self.header_font_color,
			header_bg_color or self.header_bg_color,
			font_color or self.font_color,
			bg_color or self.bg_color,
			border_font_color or self.border_font_color,
			border_bg_color or self.border_bg_color,
			len_cnt or self.len_cnt,
			formatted if formatted is not None else self.formatted,
			in_str=in_str
		)

	#<=====>#

	def chart_headers(
			self, 
			in_str='', 
			len_cnt=None, 
			align=None, 
			bold=None, 
			italic=None, 
			header_font_color=None,
			header_bg_color=None,
			font_color=None, 
			bg_color=None, 
			border_font_color=None, 
			border_bg_color=None,
			style=None,
			formatted=None
			):
		l, s, r = self.chart_shapes(part='row', style=style or self.style)
		self.chart_string(
			l, s, r,
			align or self.align,
			bold if bold is not None else self.bold,
			italic if italic is not None else self.italic,
			header_font_color or self.header_font_color,
			header_bg_color or self.header_bg_color,
			font_color or self.header_font_color,
			bg_color or self.header_bg_color,
			border_font_color or self.border_font_color,
			border_bg_color or self.border_bg_color,
			len_cnt or self.len_cnt,
			formatted if formatted is not None else self.formatted,
			in_str=in_str
		)

	#<=====>#

	def chart_mid(
			self, 
			in_str='', 
			len_cnt=None, 
			align=None, 
			bold=None, 
			italic=None, 
			header_font_color=None,
			header_bg_color=None,
			font_color=None, 
			bg_color=None, 
			border_font_color=None, 
			border_bg_color=None,
			style=None,
			formatted=None
			):
		l, s, r = self.chart_shapes(part='mid', style=style or self.style)
		self.chart_embed(
			l, s, r,
			align or self.align,
			bold if bold is not None else self.bold,
			italic if italic is not None else self.italic,
			header_font_color or self.header_font_color,
			header_bg_color or self.header_bg_color,
			font_color or self.font_color,
			bg_color or self.bg_color,
			border_font_color or self.border_font_color,
			border_bg_color or self.border_bg_color,
			len_cnt or self.len_cnt,
			formatted if formatted is not None else self.formatted,
			in_str=in_str
		)

	#<=====>#

	def chart_row(
			self, 
			in_str='', 
			len_cnt=None, 
			align=None, 
			bold=None, 
			italic=None, 
			header_font_color=None,
			header_bg_color=None,
			font_color=None, 
			bg_color=None, 
			border_font_color=None, 
			border_bg_color=None,
			style=None,
			formatted=None
			):
		l, s, r = self.chart_shapes(part='row', style=style or self.style)
		self.chart_string(
			l, s, r,
			align or self.align,
			bold if bold is not None else self.bold,
			italic if italic is not None else self.italic,
			header_font_color or self.header_font_color,
			header_bg_color or self.header_bg_color,
			font_color or self.font_color,
			bg_color or self.bg_color,
			border_font_color or self.border_font_color,
			border_bg_color or self.border_bg_color,
			len_cnt or self.len_cnt,
			formatted if formatted is not None else self.formatted,
			in_str=in_str
		)

	#<=====>#

	def chart_bottom(
			self, 
			in_str='', 
			len_cnt=None, 
			align=None, 
			bold=None, 
			italic=None, 
			header_font_color=None,
			header_bg_color=None,
			font_color=None, 
			bg_color=None, 
			border_font_color=None, 
			border_bg_color=None,
			style=None,
			formatted=None
			):
		l, s, r = self.chart_shapes(part='bottom', style=style or self.style)
		self.chart_embed(
			l, s, r,
			align or self.align,
			bold if bold is not None else self.bold,
			italic if italic is not None else self.italic,
			header_font_color or self.header_font_color,
			header_bg_color or self.header_bg_color,
			font_color or self.font_color,
			bg_color or self.bg_color,
			border_font_color or self.border_font_color,
			border_bg_color or self.border_bg_color,
			len_cnt or self.len_cnt,
			formatted if formatted is not None else self.formatted,
			in_str=in_str
		)

	#<=====>#

	def print(self):
		if not self.in_json:
			raise ValueError("No JSON data provided for printing.")

		if isinstance(self.in_json['data'], pd.DataFrame):
			data = self.in_json['data']
		else:
			data = pd.DataFrame(self.in_json['data'])

		# Extract headers
		headers = data.columns.tolist()

		# Determine max length for each column and max decimal places
		col_widths = {col: max(data[col].astype(str).apply(len).max(), len(col)) for col in headers}
		decimal_precisions = {}
		for col in headers:
			numeric_values = [value for value in data[col] if isinstance(value, (int, float))]
			if numeric_values:
				decimal_precisions[col] = max(
					(str(value)[::-1].find('.') if '.' in str(value) else 0) for value in numeric_values
				)
			else:
				decimal_precisions[col] = 0  # Default precision if no numeric values

		# Calculate total length needed for the table
		self.len_cnt = sum(col_widths.values()) + 3 * (len(headers) - 1) + 2  # 3 spaces for ' | ' and 2 for padding

		# Print the chart
		self.chart_top(in_str=self.in_json['title'])

		# Print headers
		header_str = ' | '.join([f"{col.center(col_widths[col])}" for col in headers])
		self.chart_headers(in_str=header_str)

		# Print each row
		for _, row in data.iterrows():
			row_str = ' | '.join([
				f"{format(value, f'.{decimal_precisions[col]}f').rjust(col_widths[col])}" if isinstance(value, (int, float)) else f"{str(value).ljust(col_widths[col])}"
				for col, value in row.items()
			])
			self.chart_row(in_str=row_str, align='left', font_color='white', bg_color='black', border_font_color='yellow', border_bg_color='#4B0082', style=2)

		self.chart_bottom()

#<=====>#
# Assignments Mid
#<=====>#

chrt = CHRT()

#<=====>#
# Functions
#<=====>#

def chart_top(in_str='', len_cnt=250, align='left', bold=True, italic=False, header_font_color='white', header_bg_color='blue', font_color='#00FF00', bg_color='#4B0082', border_font_color='#00FF00', border_bg_color='#4B0082', style=2, formatted=False):
	# print('* chart_top')
	# print(f'	in_str: {in_str}')
	# print(f'	len_cnt: {len_cnt}')
	# print(f'	align: {align}')
	# print(f'	bold: {bold}')
	# print(f'	italic: {italic}')
	# print(f'	header_font_color: {header_font_color}')
	# print(f'	header_bg_color: {header_bg_color}')
	# print(f'	font_color: {font_color}')
	# print(f'	bg_color: {bg_color}')
	# print(f'	border_font_color: {border_font_color}')
	# print(f'	border_bg_color: {border_bg_color}')
	# print(f'	style: {style}')
	# print(f'	formatted: {formatted}')
	chrt = CHRT(len_cnt=len_cnt, align=align, bold=bold, italic=italic, header_font_color=header_font_color, header_bg_color=header_bg_color, font_color=font_color, bg_color=bg_color, border_font_color=border_font_color, border_bg_color=border_bg_color, style=style, formatted=formatted)
	chrt.chart_top(in_str=in_str)

#<=====>#

def chart_title(in_str='', len_cnt=250, align='left', bold=True, italic=False, header_font_color='white', header_bg_color='blue', font_color='#00FF00', bg_color='#4B0082', border_font_color='#00FF00', border_bg_color='#4B0082', style=2, formatted=False):
	# print('* chart_title')
	# print(f'	in_str: {in_str}')
	# print(f'	len_cnt: {len_cnt}')
	# print(f'	align: {align}')
	# print(f'	bold: {bold}')
	# print(f'	italic: {italic}')
	# print(f'	header_font_color: {header_font_color}')
	# print(f'	header_bg_color: {header_bg_color}')
	# print(f'	font_color: {font_color}')
	# print(f'	bg_color: {bg_color}')
	# print(f'	border_font_color: {border_font_color}')
	# print(f'	border_bg_color: {border_bg_color}')
	# print(f'	style: {style}')
	# print(f'	formatted: {formatted}')
	chrt = CHRT(len_cnt=len_cnt, align=align, bold=bold, italic=italic, header_font_color=header_font_color, header_bg_color=header_bg_color, font_color=font_color, bg_color=bg_color, border_font_color=border_font_color, border_bg_color=border_bg_color, style=style, formatted=formatted)
	chrt.chart_title(in_str=in_str)

#<=====>#

def chart_headers(in_str='', len_cnt=250, align='left', bold=True, italic=False, header_font_color='white', header_bg_color='blue', font_color='gold', bg_color='blue', border_font_color='#00FF00', border_bg_color='#4B0082', style=2, formatted=False):
	# print('* chart_headers')
	# print(f'	in_str: {in_str}')
	# print(f'	len_cnt: {len_cnt}')
	# print(f'	align: {align}')
	# print(f'	bold: {bold}')
	# print(f'	italic: {italic}')
	# print(f'	header_font_color: {header_font_color}')
	# print(f'	header_bg_color: {header_bg_color}')
	# print(f'	font_color: {font_color}')
	# print(f'	bg_color: {bg_color}')
	# print(f'	border_font_color: {border_font_color}')
	# print(f'	border_bg_color: {border_bg_color}')
	# print(f'	style: {style}')
	# print(f'	formatted: {formatted}')
	chrt = CHRT(len_cnt=len_cnt, align=align, bold=bold, italic=italic, header_font_color=header_font_color, header_bg_color=header_bg_color, font_color=font_color, bg_color=bg_color, border_font_color=border_font_color, border_bg_color=border_bg_color, style=style, formatted=formatted)
	chrt.chart_headers(in_str=in_str)

#<=====>#

def chart_mid(in_str='', len_cnt=250, align='left', bold=True, italic=False, header_font_color='white', header_bg_color='blue', font_color='#00FF00', bg_color='#4B0082', border_font_color='#00FF00', border_bg_color='#4B0082', style=2, formatted=False):
	# print('* chart_mid')
	# print(f'	in_str: {in_str}')
	# print(f'	len_cnt: {len_cnt}')
	# print(f'	align: {align}')
	# print(f'	bold: {bold}')
	# print(f'	italic: {italic}')
	# print(f'	header_font_color: {header_font_color}')
	# print(f'	header_bg_color: {header_bg_color}')
	# print(f'	font_color: {font_color}')
	# print(f'	bg_color: {bg_color}')
	# print(f'	border_font_color: {border_font_color}')
	# print(f'	border_bg_color: {border_bg_color}')
	# print(f'	style: {style}')
	# print(f'	formatted: {formatted}')
	chrt = CHRT(len_cnt=len_cnt, align=align, bold=bold, italic=italic, header_font_color=header_font_color, header_bg_color=header_bg_color, font_color=font_color, bg_color=bg_color, border_font_color=border_font_color, border_bg_color=border_bg_color, style=style, formatted=formatted)
	chrt.chart_mid(in_str=in_str)

#<=====>#

def chart_row(in_str='', len_cnt=250, align='left', bold=False, italic=False, header_font_color='white', header_bg_color='blue', font_color='white', bg_color='black', border_font_color='#00FF00', border_bg_color='#4B0082', style=2, formatted=False):
	# print('* chart_row')
	# print(f'	in_str: {in_str}')
	# print(f'	len_cnt: {len_cnt}')
	# print(f'	align: {align}')
	# print(f'	bold: {bold}')
	# print(f'	italic: {italic}')
	# print(f'	header_font_color: {header_font_color}')
	# print(f'	header_bg_color: {header_bg_color}')
	# print(f'	font_color: {font_color}')
	# print(f'	bg_color: {bg_color}')
	# print(f'	border_font_color: {border_font_color}')
	# print(f'	border_bg_color: {border_bg_color}')
	# print(f'	style: {style}')
	# print(f'	formatted: {formatted}')
	chrt = CHRT(len_cnt=len_cnt, align=align, bold=bold, italic=italic, header_font_color=header_font_color, header_bg_color=header_bg_color, font_color=font_color, bg_color=bg_color, border_font_color=border_font_color, border_bg_color=border_bg_color, style=style, formatted=formatted)
	chrt.chart_row(in_str=in_str)

#<=====>#

def chart_bottom(in_str='', len_cnt=250, align='left', bold=True, italic=False, header_font_color='white', header_bg_color='blue', font_color='#00FF00', bg_color='#4B0082', border_font_color='#00FF00', border_bg_color='#4B0082', style=2, formatted=False):
	# print('* chart_bottom')
	# print(f'	in_str: {in_str}')
	# print(f'	len_cnt: {len_cnt}')
	# print(f'	align: {align}')
	# print(f'	bold: {bold}')
	# print(f'	italic: {italic}')
	# print(f'	header_font_color: {header_font_color}')
	# print(f'	header_bg_color: {header_bg_color}')
	# print(f'	font_color: {font_color}')
	# print(f'	bg_color: {bg_color}')
	# print(f'	border_font_color: {border_font_color}')
	# print(f'	border_bg_color: {border_bg_color}')
	# print(f'	style: {style}')
	# print(f'	formatted: {formatted}')
	chrt = CHRT(len_cnt=len_cnt, align=align, bold=bold, italic=italic, header_font_color=header_font_color, header_bg_color=header_bg_color, font_color=font_color, bg_color=bg_color, border_font_color=border_font_color, border_bg_color=border_bg_color, style=style, formatted=formatted)
	chrt.chart_bottom(in_str=in_str)

#<=====>#

def demo1():
	print('')
	print('')
	print('demo1')

	chart_json = {
		'title': 'BTC OHLCV Data',
		'data': [
			{"timestamp":"2024-12-01 08:48:00", "open":97140.78, "high":97147.51, "low":97112.59, "close": 97124.75, "volume": 3.06128143},
			{"timestamp":"2024-12-01 08:49:00", "open":97124.76, "high":97169.10, "low":97108.75, "close": 97168.99, "volume": 1.61392088},
			{"timestamp":"2024-12-01 08:50:00", "open":97168.81, "high":97187.52, "low":97131.78, "close": 97181.35, "volume": 1.38697039},
			{"timestamp":"2024-12-01 08:51:00", "open":97181.35, "high":97211.74, "low":97170.02, "close": 97211.73, "volume": 0.66927336},
			{"timestamp":"2024-12-01 08:52:00", "open":97211.74, "high":97211.74, "low":97137.00, "close": 97147.47, "volume": 9.05403498},
			{"timestamp":"2024-12-01 08:53:00", "open":97147.47, "high":97184.49, "low":97147.46, "close": 97175.42, "volume": 1.39944668},
			{"timestamp":"2024-12-01 08:54:00", "open":97163.55, "high":97218.04, "low":97156.44, "close": 97204.37, "volume": 2.23422710},
			{"timestamp":"2024-12-01 08:55:00", "open":97204.37, "high":97204.37, "low":97101.29, "close": 97103.36, "volume": 2.37993227},
			{"timestamp":"2024-12-01 08:56:00", "open":97103.37, "high":97136.74, "low":97078.64, "close": 97128.86, "volume": 1.88873265},
			{"timestamp":"2024-12-01 08:57:00", "open":97123.33, "high":97123.33, "low":97046.65, "close": 97049.47, "volume": 2.82729602},
			{"timestamp":"2024-12-01 08:58:00", "open":97047.43, "high":97110.61, "low":97047.43, "close": 97099.68, "volume": 1.26491525},
			{"timestamp":"2024-12-01 08:59:00", "open":97100.19, "high":97110.66, "low":97089.61, "close": 97098.97, "volume": 1.42565733},
			{"timestamp":"2024-12-01 09:00:00", "open":97100.36, "high":97160.55, "low":97083.65, "close": 97083.65, "volume": 3.35403027},
			{"timestamp":"2024-12-01 09:01:00", "open":97083.65, "high":97091.91, "low":97005.80, "close": 97015.58, "volume": 2.32681268},
			{"timestamp":"2024-12-01 09:02:00", "open":97019.39, "high":97048.56, "low":96998.16, "close": 97032.76, "volume": 1.58268383},
			{"timestamp":"2024-12-01 09:03:00", "open":97032.76, "high":97046.59, "low":96995.95, "close": 96995.95, "volume": 1.51670571},
			{"timestamp":"2024-12-01 09:04:00", "open":96980.96, "high":97002.96, "low":96963.74, "close": 96965.22, "volume": 1.81746833},
			{"timestamp":"2024-12-01 09:05:00", "open":96965.22, "high":97003.98, "low":96956.36, "close": 96995.08, "volume": 1.53638046},
			{"timestamp":"2024-12-01 09:06:00", "open":96995.08, "high":97003.20, "low":96960.01, "close": 96989.67, "volume": 2.06595389},
			{"timestamp":"2024-12-01 09:07:00", "open":96989.67, "high":97001.15, "low":96975.30, "close": 96993.11, "volume": 1.33001954},
			{"timestamp":"2024-12-01 09:08:00", "open":96993.11, "high":96996.38, "low":96961.22, "close": 96975.32, "volume": 1.77676409},
			{"timestamp":"2024-12-01 09:09:00", "open":96975.32, "high":96980.63, "low":96946.97, "close": 96946.97, "volume": 1.24955840},
			{"timestamp":"2024-12-01 09:10:00", "open":96946.96, "high":96948.28, "low":96920.65, "close": 96942.03, "volume": 1.85760543},
			{"timestamp":"2024-12-01 09:11:00", "open":96942.04, "high":96973.10, "low":96942.03, "close": 96963.23, "volume": 0.49510429},
			{"timestamp":"2024-12-01 09:12:00", "open":96963.34, "high":96968.74, "low":96955.69, "close": 96957.61, "volume": 1.15413387},
			{"timestamp":"2024-12-01 09:13:00", "open":96960.19, "high":96969.00, "low":96953.92, "close": 96966.26, "volume": 1.37276975},
			{"timestamp":"2024-12-01 09:14:00", "open":96961.82, "high":97029.16, "low":96960.74, "close": 97010.49, "volume": 2.03371255},
			{"timestamp":"2024-12-01 09:15:00", "open":97001.88, "high":97028.89, "low":96994.76, "close": 97023.05, "volume": 1.17327599},
			{"timestamp":"2024-12-01 09:16:00", "open":97023.05, "high":97073.27, "low":97023.04, "close": 97064.68, "volume": 2.07387944},
			{"timestamp":"2024-12-01 09:17:00", "open":97064.68, "high":97065.26, "low":97031.86, "close": 97031.89, "volume": 0.42650327},
			{"timestamp":"2024-12-01 09:18:00", "open":97031.88, "high":97047.05, "low":97017.48, "close": 97035.62, "volume": 0.49421106}
		]
	}
	chrt = CHRT(in_json=chart_json)
	chrt.print()

#<=====>#

def demo2():

	print('')
	print('')
	print('demo2')

	df = pd.DataFrame([
			{"timestamp":"2024-12-01 08:48:00", "open":97140.78, "high":97147.51, "low":97112.59, "close": 97124.75, "volume": 3.06128143},
			{"timestamp":"2024-12-01 08:49:00", "open":97124.76, "high":97169.10, "low":97108.75, "close": 97168.99, "volume": 1.61392088},
			{"timestamp":"2024-12-01 08:50:00", "open":97168.81, "high":97187.52, "low":97131.78, "close": 97181.35, "volume": 1.38697039},
			{"timestamp":"2024-12-01 08:51:00", "open":97181.35, "high":97211.74, "low":97170.02, "close": 97211.73, "volume": 0.66927336},
			{"timestamp":"2024-12-01 08:52:00", "open":97211.74, "high":97211.74, "low":97137.00, "close": 97147.47, "volume": 9.05403498},
			{"timestamp":"2024-12-01 08:53:00", "open":97147.47, "high":97184.49, "low":97147.46, "close": 97175.42, "volume": 1.39944668},
			{"timestamp":"2024-12-01 08:54:00", "open":97163.55, "high":97218.04, "low":97156.44, "close": 97204.37, "volume": 2.23422710},
			{"timestamp":"2024-12-01 08:55:00", "open":97204.37, "high":97204.37, "low":97101.29, "close": 97103.36, "volume": 2.37993227},
			{"timestamp":"2024-12-01 08:56:00", "open":97103.37, "high":97136.74, "low":97078.64, "close": 97128.86, "volume": 1.88873265},
			{"timestamp":"2024-12-01 08:57:00", "open":97123.33, "high":97123.33, "low":97046.65, "close": 97049.47, "volume": 2.82729602},
			{"timestamp":"2024-12-01 08:58:00", "open":97047.43, "high":97110.61, "low":97047.43, "close": 97099.68, "volume": 1.26491525},
			{"timestamp":"2024-12-01 08:59:00", "open":97100.19, "high":97110.66, "low":97089.61, "close": 97098.97, "volume": 1.42565733},
			{"timestamp":"2024-12-01 09:00:00", "open":97100.36, "high":97160.55, "low":97083.65, "close": 97083.65, "volume": 3.35403027},
			{"timestamp":"2024-12-01 09:01:00", "open":97083.65, "high":97091.91, "low":97005.80, "close": 97015.58, "volume": 2.32681268},
			{"timestamp":"2024-12-01 09:02:00", "open":97019.39, "high":97048.56, "low":96998.16, "close": 97032.76, "volume": 1.58268383},
			{"timestamp":"2024-12-01 09:03:00", "open":97032.76, "high":97046.59, "low":96995.95, "close": 96995.95, "volume": 1.51670571},
			{"timestamp":"2024-12-01 09:04:00", "open":96980.96, "high":97002.96, "low":96963.74, "close": 96965.22, "volume": 1.81746833},
			{"timestamp":"2024-12-01 09:05:00", "open":96965.22, "high":97003.98, "low":96956.36, "close": 96995.08, "volume": 1.53638046},
			{"timestamp":"2024-12-01 09:06:00", "open":96995.08, "high":97003.20, "low":96960.01, "close": 96989.67, "volume": 2.06595389},
			{"timestamp":"2024-12-01 09:07:00", "open":96989.67, "high":97001.15, "low":96975.30, "close": 96993.11, "volume": 1.33001954},
			{"timestamp":"2024-12-01 09:08:00", "open":96993.11, "high":96996.38, "low":96961.22, "close": 96975.32, "volume": 1.77676409},
			{"timestamp":"2024-12-01 09:09:00", "open":96975.32, "high":96980.63, "low":96946.97, "close": 96946.97, "volume": 1.24955840},
			{"timestamp":"2024-12-01 09:10:00", "open":96946.96, "high":96948.28, "low":96920.65, "close": 96942.03, "volume": 1.85760543},
			{"timestamp":"2024-12-01 09:11:00", "open":96942.04, "high":96973.10, "low":96942.03, "close": 96963.23, "volume": 0.49510429},
			{"timestamp":"2024-12-01 09:12:00", "open":96963.34, "high":96968.74, "low":96955.69, "close": 96957.61, "volume": 1.15413387},
			{"timestamp":"2024-12-01 09:13:00", "open":96960.19, "high":96969.00, "low":96953.92, "close": 96966.26, "volume": 1.37276975},
			{"timestamp":"2024-12-01 09:14:00", "open":96961.82, "high":97029.16, "low":96960.74, "close": 97010.49, "volume": 2.03371255},
			{"timestamp":"2024-12-01 09:15:00", "open":97001.88, "high":97028.89, "low":96994.76, "close": 97023.05, "volume": 1.17327599},
			{"timestamp":"2024-12-01 09:16:00", "open":97023.05, "high":97073.27, "low":97023.04, "close": 97064.68, "volume": 2.07387944},
			{"timestamp":"2024-12-01 09:17:00", "open":97064.68, "high":97065.26, "low":97031.86, "close": 97031.89, "volume": 0.42650327},
			{"timestamp":"2024-12-01 09:18:00", "open":97031.88, "high":97047.05, "low":97017.48, "close": 97035.62, "volume": 0.49421106}
		])

	df['color'] = np.where(df['open'] < df['close'], 'green', 'red')

	df['rsi'] = round(pta.rsi(df['close'], length=14), 2)
	df['rsi_roc']   = pta.roc(df['rsi'], length=3)
	df['rsi_roc_up']  = df['rsi_roc'] > df['rsi_roc'].shift(1)
	df['rsi_roc_dn']  = df['rsi_roc'] < df['rsi_roc'].shift(1)
	df['rsi_dir'] = np.where(df['rsi_roc_up'], 'up', np.where(df['rsi_roc_dn'], 'down', 'flat'))
	df = df.drop(columns=['rsi_roc_up', 'rsi_roc_dn', 'rsi_roc'])

	df['ema8'] = round(pta.ema(df['close'], length=8), 2)
	df['ema8_roc']  = pta.roc(df['ema8'], length=3)
	df['ema8_roc_up']  = df['ema8_roc'] > df['ema8_roc'].shift(1)
	df['ema8_roc_dn']  = df['ema8_roc'] < df['ema8_roc'].shift(1)
	df['ema8_dir'] = np.where(df['ema8_roc_up'], 'up', np.where(df['ema8_roc_dn'], 'down', 'flat'))
	df = df.drop(columns=['ema8_roc_up', 'ema8_roc_dn', 'ema8_roc'])	

	df['ema13'] = round(pta.ema(df['close'], length=13), 2)
	df['ema13_roc'] = pta.roc(df['ema13'], length=3)
	df['ema13_roc_up']  = df['ema13_roc'] > df['ema13_roc'].shift(1)
	df['ema13_roc_dn']  = df['ema13_roc'] < df['ema13_roc'].shift(1)
	df['ema13_dir'] = np.where(df['ema13_roc_up'], 'up', np.where(df['ema13_roc_dn'], 'down', 'flat'))
	df = df.drop(columns=['ema13_roc_up', 'ema13_roc_dn', 'ema13_roc'])	

	df['ema21'] = round(pta.ema(df['close'], length=21), 2)
	df['ema21_roc'] = pta.roc(df['ema21'], length=3)
	df['ema21_roc_up']  = df['ema21_roc'] > df['ema21_roc'].shift(1)
	df['ema21_roc_dn']  = df['ema21_roc'] < df['ema21_roc'].shift(1)
	df['ema21_dir'] = np.where(df['ema21_roc_up'], 'up', np.where(df['ema21_roc_dn'], 'down', 'flat'))
	df = df.drop(columns=['ema21_roc_up', 'ema21_roc_dn', 'ema21_roc'])	

	# Calculate Heikin Ashi Close
	df['ha_close']                     = (df['open'] + df['high'] + df['low'] + df['close']) / 4
	# Initialize HA Open
	df['ha_open']                      = np.nan
	df.at[df.index[0], 'ha_open']      = (df.at[df.index[0], 'open'] + df.at[df.index[0], 'close']) / 2
	# Calculate HA Open for the rest of the rows
	for i in range(1, len(df)):
		df.at[df.index[i], 'ha_open']  = (df.at[df.index[i-1], 'ha_open'] + df.at[df.index[i-1], 'ha_close']) / 2
	# Calculate HA High and Low
	df['ha_high']                      = df[['ha_open','ha_close','high']].max(axis=1)
	df['ha_low']                       = df[['ha_open','ha_close','low']].min(axis=1)
	# HA Candle Height
	df['ha_span']                      = df['ha_high'] - df['ha_low']
	# HA Candle Body Height
	df['ha_body']                      = (df['ha_close'] - df['ha_open']).abs()
	# HA Candle Upper And Lower Wick Heights
	df['ha_wick_upper']                = df['ha_high'] - df[['ha_open','ha_close']].max(axis=1)
	df['ha_wick_lower']                = df[['ha_open','ha_close']].min(axis=1) - df['ha_low']
	# What percentage of each candle is body, uppper and lower wick
	df['ha_body_pct']                  = round((df['ha_body'] / df['ha_span']) * 100, 2)
	df['ha_wick_upper_pct']            = round((df['ha_wick_upper'] / df['ha_span']) * 100, 2)
	df['ha_wick_lower_pct']            = round((df['ha_wick_lower'] / df['ha_span']) * 100, 2)
	# Existence of upper and lower wicks
	df['ha_wick_upper_none']           = df['ha_wick_upper_pct'] < 3
	df['ha_wick_lower_none']           = df['ha_wick_lower_pct'] < 3
	# HA Candle Color
	df['ha_color']                     = np.where(df['ha_open'] < df['ha_close'], 'green', 'red')
	df = df.drop(columns=['ha_open', 'ha_close', 'ha_high', 'ha_low', 'ha_span', 'ha_body', 'ha_wick_upper', 'ha_wick_lower', 'ha_body_pct', 'ha_wick_upper_pct', 'ha_wick_lower_pct', 'ha_wick_upper_none', 'ha_wick_lower_none'])

	chart_json = {
		'title': 'BTC OHLCV Data',
		'data': df
		}

	chrt = CHRT(in_json=chart_json)
	chrt.print()

#<=====>#

def demo3():
	print('')
	print('')
	print('demo3')

	poss = [
		{'pos_id': 32814, 'symb': 'USDC', 'prod_id': 'PNG-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 12, 1, 8, 10, 26), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 18, 45, 22), 'age_mins': 335, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('5.982007500000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.014992500000'), 'fees_cnt_tot': Decimal('0.029955093516'), 'buy_cnt': Decimal('15.151993434136'), 'sell_cnt_tot': Decimal('15.000000000000'), 'hold_cnt': Decimal('0.151993434136'), 'pocket_cnt': Decimal('0.151993434136'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.395987500000'), 'prc_curr': Decimal('0.398800500000'), 'prc_high': Decimal('0.428580000000'), 'prc_low': Decimal('0.372900000000'), 'prc_chg_pct': Decimal('0.710400000000'), 'prc_chg_pct_high': Decimal('6.973100000000'), 'prc_chg_pct_low': Decimal('-5.830400000000'), 'prc_chg_pct_drop': Decimal('-6.260000000000'), 'prc_sell_avg': Decimal('0.398800500000'), 'val_curr': Decimal('0.060615057530'), 'val_tot': Decimal('0.060615057530'), 'gain_loss_amt_est': Decimal('0.083525363806'), 'gain_loss_amt_est_high': Decimal('0.418384418700'), 'gain_loss_amt_est_low': Decimal('-0.349821648411'), 'gain_loss_amt': Decimal('0.042622557530'), 'gain_loss_amt_net': Decimal('0.103237615060'), 'gain_loss_pct_est': Decimal('1.392100000000'), 'gain_loss_pct_est_high': Decimal('6.973100000000'), 'gain_loss_pct_est_low': Decimal('-5.830400000000'), 'gain_loss_pct': Decimal('0.710400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32468, 'bo_uuid': 'd1e7d87f-f530-4a97-98f7-fc99a0e344a5', 'buy_curr_symb': 'PNG', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PNG', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PNG', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('116620741.418695990000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 12, 1, 8, 10, 56), 'check_last_dttm': datetime.datetime(2024, 12, 1, 13, 45, 17), 'add_dttm': datetime.datetime(2024, 12, 1, 8, 10, 56), 'upd_dttm': datetime.datetime(2024, 12, 1, 8, 10, 56), 'del_dttm': datetime.datetime(2024, 12, 1, 8, 10, 56), 'dlm': datetime.datetime(2024, 12, 1, 13, 45, 56)},
		{'pos_id': 32548, 'symb': 'USDC', 'prod_id': 'PNG-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 18, 26, 32), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 18, 45, 6), 'age_mins': 1159, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.014925000000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015075000000'), 'fees_cnt_tot': Decimal('0.030037593516'), 'buy_cnt': Decimal('15.386491352984'), 'sell_cnt_tot': Decimal('15.000000000000'), 'hold_cnt': Decimal('0.386491352984'), 'pocket_cnt': Decimal('0.386491352984'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.389952450000'), 'prc_curr': Decimal('0.400995000000'), 'prc_high': Decimal('0.428580000000'), 'prc_low': Decimal('0.343510000000'), 'prc_chg_pct': Decimal('2.831800000000'), 'prc_chg_pct_high': Decimal('8.628600000000'), 'prc_chg_pct_low': Decimal('-11.909800000000'), 'prc_chg_pct_drop': Decimal('-5.800000000000'), 'prc_sell_avg': Decimal('0.400995000000'), 'val_curr': Decimal('0.154981100090'), 'val_tot': Decimal('0.154981100090'), 'gain_loss_amt_est': Decimal('0.177676278223'), 'gain_loss_amt_est_high': Decimal('0.517717737124'), 'gain_loss_amt_est_low': Decimal('-0.714586355336'), 'gain_loss_amt': Decimal('0.169906100090'), 'gain_loss_amt_net': Decimal('0.324887200180'), 'gain_loss_pct_est': Decimal('2.961300000000'), 'gain_loss_pct_est_high': Decimal('8.628600000000'), 'gain_loss_pct_est_low': Decimal('-11.909800000000'), 'gain_loss_pct': Decimal('2.831800000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32202, 'bo_uuid': 'bf49d6af-2bfc-4df3-985f-5b915714016f', 'buy_curr_symb': 'PNG', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PNG', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PNG', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('116620741.418695990000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 18, 26, 59), 'check_last_dttm': datetime.datetime(2024, 12, 1, 13, 45, 1), 'add_dttm': datetime.datetime(2024, 11, 30, 18, 26, 59), 'upd_dttm': datetime.datetime(2024, 11, 30, 18, 26, 59), 'del_dttm': datetime.datetime(2024, 11, 30, 18, 26, 59), 'dlm': datetime.datetime(2024, 12, 1, 13, 45, 49)},
		{'pos_id': 32129, 'symb': 'USDC', 'prod_id': 'BONK-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 23, 50, 59), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 17, 19, 44), 'age_mins': 2189, 'tot_out_cnt': Decimal('10.000000000000'), 'tot_in_cnt': Decimal('10.209681386100'), 'buy_fees_cnt': Decimal('0.024937655860'), 'sell_fees_cnt_tot': Decimal('0.025588173900'), 'fees_cnt_tot': Decimal('0.050525829760'), 'buy_cnt': Decimal('218655.465676011630'), 'sell_cnt_tot': Decimal('218143.000000000000'), 'hold_cnt': Decimal('512.465676011634'), 'pocket_cnt': Decimal('512.465676011634'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.000045730000'), 'prc_curr': Decimal('0.000046800000'), 'prc_high': Decimal('0.000048750000'), 'prc_low': Decimal('0.000044200000'), 'prc_chg_pct': Decimal('2.339800000000'), 'prc_chg_pct_high': Decimal('5.401300000000'), 'prc_chg_pct_low': Decimal('-3.345700000000'), 'prc_chg_pct_drop': Decimal('-3.060000000000'), 'prc_sell_avg': Decimal('0.000046800000'), 'val_curr': Decimal('0.023983393637'), 'val_tot': Decimal('0.023983393637'), 'gain_loss_amt_est': Decimal('0.233075793637'), 'gain_loss_amt_est_high': Decimal('0.539193445584'), 'gain_loss_amt_est_low': Decimal('-0.335428417120'), 'gain_loss_amt': Decimal('0.233664779737'), 'gain_loss_amt_net': Decimal('0.257648173375'), 'gain_loss_pct_est': Decimal('2.330800000000'), 'gain_loss_pct_est_high': Decimal('5.391900000000'), 'gain_loss_pct_est_low': Decimal('-3.354300000000'), 'gain_loss_pct': Decimal('2.336600000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31783, 'bo_uuid': '508ff38e-c71d-4330-ba5c-43a42106348d', 'buy_curr_symb': 'BONK', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'BONK', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'BONK', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('357142857143.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('1.0000E-8'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 23, 51, 40), 'check_last_dttm': datetime.datetime(2024, 12, 1, 12, 19, 39), 'add_dttm': datetime.datetime(2024, 11, 29, 23, 51, 40), 'upd_dttm': datetime.datetime(2024, 11, 29, 23, 51, 40), 'del_dttm': datetime.datetime(2024, 11, 29, 23, 51, 40), 'dlm': datetime.datetime(2024, 12, 1, 12, 20, 42)},
		{'pos_id': 32738, 'symb': 'USDC', 'prod_id': 'CVX-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 12, 1, 2, 56, 7), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 17, 18, 54), 'age_mins': 562, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.282825570000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015746430000'), 'fees_cnt_tot': Decimal('0.030709023516'), 'buy_cnt': Decimal('1.356344239220'), 'sell_cnt_tot': Decimal('1.347000000000'), 'hold_cnt': Decimal('0.009344239220'), 'pocket_cnt': Decimal('0.009344239220'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('4.423655760000'), 'prc_curr': Decimal('4.664310000000'), 'prc_high': Decimal('5.110000000000'), 'prc_low': Decimal('4.287000000000'), 'prc_chg_pct': Decimal('5.440200000000'), 'prc_chg_pct_high': Decimal('9.909100000000'), 'prc_chg_pct_low': Decimal('-3.089200000000'), 'prc_chg_pct_drop': Decimal('-4.470000000000'), 'prc_sell_avg': Decimal('4.664310000000'), 'val_curr': Decimal('0.043584428436'), 'val_tot': Decimal('0.043584428436'), 'gain_loss_amt_est': Decimal('0.400588464879'), 'gain_loss_amt_est_high': Decimal('0.594545691088'), 'gain_loss_amt_est_low': Decimal('-0.185352246464'), 'gain_loss_amt': Decimal('0.326409998436'), 'gain_loss_amt_net': Decimal('0.369994426872'), 'gain_loss_pct_est': Decimal('6.676500000000'), 'gain_loss_pct_est_high': Decimal('9.909100000000'), 'gain_loss_pct_est_low': Decimal('-3.089200000000'), 'gain_loss_pct': Decimal('5.440200000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32392, 'bo_uuid': '97316d98-50da-449f-994b-08a4697488a5', 'buy_curr_symb': 'CVX', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'CVX', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'CVX', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('1934876.697921781300'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 12, 1, 2, 56, 32), 'check_last_dttm': datetime.datetime(2024, 12, 1, 12, 18, 49), 'add_dttm': datetime.datetime(2024, 12, 1, 2, 56, 32), 'upd_dttm': datetime.datetime(2024, 12, 1, 2, 56, 32), 'del_dttm': datetime.datetime(2024, 12, 1, 2, 56, 32), 'dlm': datetime.datetime(2024, 12, 1, 12, 20, 11)},
		{'pos_id': 32542, 'symb': 'USDC', 'prod_id': 'ALEO-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 18, 9, 56), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 15, 47, 58), 'age_mins': 998, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.209236800000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.008043200000'), 'fees_cnt_tot': Decimal('0.015524496758'), 'buy_cnt': Decimal('1.779143105376'), 'sell_cnt_tot': Decimal('1.760000000000'), 'hold_cnt': Decimal('0.019143105376'), 'pocket_cnt': Decimal('0.019143105376'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.686205000000'), 'prc_curr': Decimal('1.823430000000'), 'prc_high': Decimal('1.905000000000'), 'prc_low': Decimal('1.583000000000'), 'prc_chg_pct': Decimal('8.138100000000'), 'prc_chg_pct_high': Decimal('11.967400000000'), 'prc_chg_pct_low': Decimal('-6.120500000000'), 'prc_chg_pct_drop': Decimal('-3.830000000000'), 'prc_sell_avg': Decimal('1.823430000000'), 'val_curr': Decimal('0.034906112636'), 'val_tot': Decimal('0.034906112636'), 'gain_loss_amt_est': Decimal('0.252273596627'), 'gain_loss_amt_est_high': Decimal('0.359022182950'), 'gain_loss_amt_est_low': Decimal('-0.183616464190'), 'gain_loss_amt': Decimal('0.244142912636'), 'gain_loss_amt_net': Decimal('0.279049025272'), 'gain_loss_pct_est': Decimal('8.409100000000'), 'gain_loss_pct_est_high': Decimal('11.967400000000'), 'gain_loss_pct_est_low': Decimal('-6.120500000000'), 'gain_loss_pct': Decimal('8.138100000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32196, 'bo_uuid': '88716929-a5ec-4d31-bcd7-ee1214bd4fc3', 'buy_curr_symb': 'ALEO', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ALEO', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ALEO', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('8000000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 18, 10, 32), 'check_last_dttm': datetime.datetime(2024, 12, 1, 10, 47, 53), 'add_dttm': datetime.datetime(2024, 11, 30, 18, 10, 32), 'upd_dttm': datetime.datetime(2024, 11, 30, 18, 10, 32), 'del_dttm': datetime.datetime(2024, 11, 30, 18, 10, 32), 'dlm': datetime.datetime(2024, 12, 1, 10, 49, 10)},
		{'pos_id': 32520, 'symb': 'USDC', 'prod_id': 'ALEO-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 17, 45, 33), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 15, 19, 55), 'age_mins': 995, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.217895100000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.008064900000'), 'fees_cnt_tot': Decimal('0.015546196758'), 'buy_cnt': Decimal('1.764456782572'), 'sell_cnt_tot': Decimal('1.740000000000'), 'hold_cnt': Decimal('0.024456782572'), 'pocket_cnt': Decimal('0.024456782572'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.700240000000'), 'prc_curr': Decimal('1.849365000000'), 'prc_high': Decimal('1.905000000000'), 'prc_low': Decimal('1.583000000000'), 'prc_chg_pct': Decimal('8.770800000000'), 'prc_chg_pct_high': Decimal('11.043100000000'), 'prc_chg_pct_low': Decimal('-6.895500000000'), 'prc_chg_pct_drop': Decimal('-2.270000000000'), 'prc_sell_avg': Decimal('1.849365000000'), 'val_curr': Decimal('0.045229517701'), 'val_tot': Decimal('0.045229517701'), 'gain_loss_amt_est': Decimal('0.271302874888'), 'gain_loss_amt_est_high': Decimal('0.331294405496'), 'gain_loss_amt_est_low': Decimal('-0.206864913189'), 'gain_loss_amt': Decimal('0.263124617701'), 'gain_loss_amt_net': Decimal('0.308354135403'), 'gain_loss_pct_est': Decimal('9.043400000000'), 'gain_loss_pct_est_high': Decimal('11.043100000000'), 'gain_loss_pct_est_low': Decimal('-6.895500000000'), 'gain_loss_pct': Decimal('8.770800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32174, 'bo_uuid': '767e4077-a70d-4967-878b-a6f6c58d5d70', 'buy_curr_symb': 'ALEO', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ALEO', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ALEO', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('8000000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 17, 46, 26), 'check_last_dttm': datetime.datetime(2024, 12, 1, 10, 19, 50), 'add_dttm': datetime.datetime(2024, 11, 30, 17, 46, 26), 'upd_dttm': datetime.datetime(2024, 11, 30, 17, 46, 26), 'del_dttm': datetime.datetime(2024, 11, 30, 17, 46, 26), 'dlm': datetime.datetime(2024, 12, 1, 10, 20, 38)},
		{'pos_id': 32744, 'symb': 'USDC', 'prod_id': 'PNG-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 12, 1, 3, 54, 8), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 13, 40, 30), 'age_mins': 286, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('2.978774400000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007465600000'), 'fees_cnt_tot': Decimal('0.014946896758'), 'buy_cnt': Decimal('8.381935754977'), 'sell_cnt_tot': Decimal('8.000000000000'), 'hold_cnt': Decimal('0.381935754977'), 'pocket_cnt': Decimal('0.381935754977'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.357912550000'), 'prc_curr': Decimal('0.372346800000'), 'prc_high': Decimal('0.402850000000'), 'prc_low': Decimal('0.355110000000'), 'prc_chg_pct': Decimal('4.032900000000'), 'prc_chg_pct_high': Decimal('11.060600000000'), 'prc_chg_pct_low': Decimal('-0.783000000000'), 'prc_chg_pct_drop': Decimal('-7.030000000000'), 'prc_sell_avg': Decimal('0.372346800000'), 'val_curr': Decimal('0.142212556171'), 'val_tot': Decimal('0.142212556171'), 'gain_loss_amt_est': Decimal('0.125623843031'), 'gain_loss_amt_est_high': Decimal('0.331819462603'), 'gain_loss_amt_est_low': Decimal('-0.023490794050'), 'gain_loss_amt': Decimal('0.120986956171'), 'gain_loss_amt_net': Decimal('0.263199512343'), 'gain_loss_pct_est': Decimal('4.187500000000'), 'gain_loss_pct_est_high': Decimal('11.060600000000'), 'gain_loss_pct_est_low': Decimal('-0.783000000000'), 'gain_loss_pct': Decimal('4.032900000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32398, 'bo_uuid': 'fb25ab33-372b-409d-ab9d-5641f3da7184', 'buy_curr_symb': 'PNG', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PNG', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PNG', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('116620741.418695990000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 12, 1, 3, 54, 26), 'check_last_dttm': datetime.datetime(2024, 12, 1, 8, 40, 26), 'add_dttm': datetime.datetime(2024, 12, 1, 3, 54, 26), 'upd_dttm': datetime.datetime(2024, 12, 1, 3, 54, 26), 'del_dttm': datetime.datetime(2024, 12, 1, 3, 54, 26), 'dlm': datetime.datetime(2024, 12, 1, 8, 42, 6)},
		{'pos_id': 32669, 'symb': 'USDC', 'prod_id': 'PNG-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 22, 21, 8), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 13, 40, 11), 'age_mins': 619, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('5.959783200000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.014936800000'), 'fees_cnt_tot': Decimal('0.029899393516'), 'buy_cnt': Decimal('16.886373632265'), 'sell_cnt_tot': Decimal('16.000000000000'), 'hold_cnt': Decimal('0.886373632265'), 'pocket_cnt': Decimal('0.886373632265'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.355316080000'), 'prc_curr': Decimal('0.372486450000'), 'prc_high': Decimal('0.402850000000'), 'prc_low': Decimal('0.343510000000'), 'prc_chg_pct': Decimal('4.832400000000'), 'prc_chg_pct_high': Decimal('11.872200000000'), 'prc_chg_pct_low': Decimal('-3.322700000000'), 'prc_chg_pct_drop': Decimal('-7.040000000000'), 'prc_sell_avg': Decimal('0.372486450000'), 'val_curr': Decimal('0.330162167656'), 'val_tot': Decimal('0.330162167656'), 'gain_loss_amt_est': Decimal('0.296928727472'), 'gain_loss_amt_est_high': Decimal('0.712333518825'), 'gain_loss_amt_est_low': Decimal('-0.199361793581'), 'gain_loss_amt': Decimal('0.289945367656'), 'gain_loss_amt_net': Decimal('0.620107535312'), 'gain_loss_pct_est': Decimal('4.948800000000'), 'gain_loss_pct_est_high': Decimal('11.872200000000'), 'gain_loss_pct_est_low': Decimal('-3.322700000000'), 'gain_loss_pct': Decimal('4.832400000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32323, 'bo_uuid': '2a053679-772d-4446-97c0-c06a95bbbbfa', 'buy_curr_symb': 'PNG', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PNG', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PNG', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('116620741.418695990000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 22, 22, 8), 'check_last_dttm': datetime.datetime(2024, 12, 1, 8, 40, 7), 'add_dttm': datetime.datetime(2024, 11, 30, 22, 22, 8), 'upd_dttm': datetime.datetime(2024, 11, 30, 22, 22, 8), 'del_dttm': datetime.datetime(2024, 11, 30, 22, 22, 8), 'dlm': datetime.datetime(2024, 12, 1, 8, 40, 43)},
		{'pos_id': 32631, 'symb': 'USDC', 'prod_id': 'PNG-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 20, 53, 47), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 13, 39, 55), 'age_mins': 706, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('5.953080000000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.014920000000'), 'fees_cnt_tot': Decimal('0.029882593516'), 'buy_cnt': Decimal('16.730599632359'), 'sell_cnt_tot': Decimal('16.000000000000'), 'hold_cnt': Decimal('0.730599632359'), 'pocket_cnt': Decimal('0.730599632359'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.358624320000'), 'prc_curr': Decimal('0.372067500000'), 'prc_high': Decimal('0.402850000000'), 'prc_low': Decimal('0.343510000000'), 'prc_chg_pct': Decimal('3.748500000000'), 'prc_chg_pct_high': Decimal('10.840200000000'), 'prc_chg_pct_low': Decimal('-4.214500000000'), 'prc_chg_pct_drop': Decimal('-7.090000000000'), 'prc_sell_avg': Decimal('0.372067500000'), 'val_curr': Decimal('0.271832378713'), 'val_tot': Decimal('0.271832378713'), 'gain_loss_amt_est': Decimal('0.238840602907'), 'gain_loss_amt_est_high': Decimal('0.650413353863'), 'gain_loss_amt_est_low': Decimal('-0.252871720288'), 'gain_loss_amt': Decimal('0.224912378713'), 'gain_loss_amt_net': Decimal('0.496744757425'), 'gain_loss_pct_est': Decimal('3.980700000000'), 'gain_loss_pct_est_high': Decimal('10.840200000000'), 'gain_loss_pct_est_low': Decimal('-4.214500000000'), 'gain_loss_pct': Decimal('3.748500000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32285, 'bo_uuid': 'dc21dc13-3127-4291-8826-84dfc03db641', 'buy_curr_symb': 'PNG', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PNG', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PNG', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('116620741.418695990000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 20, 55, 10), 'check_last_dttm': datetime.datetime(2024, 12, 1, 8, 39, 50), 'add_dttm': datetime.datetime(2024, 11, 30, 20, 55, 10), 'upd_dttm': datetime.datetime(2024, 11, 30, 20, 55, 10), 'del_dttm': datetime.datetime(2024, 11, 30, 20, 55, 10), 'dlm': datetime.datetime(2024, 12, 1, 8, 40, 35)},
		{'pos_id': 32630, 'symb': 'USDC', 'prod_id': 'PNG-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 20, 45, 24), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 13, 39, 39), 'age_mins': 714, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('5.951484000000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.014916000000'), 'fees_cnt_tot': Decimal('0.029878593516'), 'buy_cnt': Decimal('16.745767785349'), 'sell_cnt_tot': Decimal('16.000000000000'), 'hold_cnt': Decimal('0.745767785349'), 'pocket_cnt': Decimal('0.745767785349'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.358299490000'), 'prc_curr': Decimal('0.371967750000'), 'prc_high': Decimal('0.402850000000'), 'prc_low': Decimal('0.343510000000'), 'prc_chg_pct': Decimal('3.814800000000'), 'prc_chg_pct_high': Decimal('10.940700000000'), 'prc_chg_pct_low': Decimal('-4.127700000000'), 'prc_chg_pct_drop': Decimal('-7.130000000000'), 'prc_sell_avg': Decimal('0.371967750000'), 'val_curr': Decimal('0.277401565139'), 'val_tot': Decimal('0.277401565139'), 'gain_loss_amt_est': Decimal('0.244496807157'), 'gain_loss_amt_est_high': Decimal('0.656442694676'), 'gain_loss_amt_est_low': Decimal('-0.247661308055'), 'gain_loss_amt': Decimal('0.228885565139'), 'gain_loss_amt_net': Decimal('0.506287130278'), 'gain_loss_pct_est': Decimal('4.074900000000'), 'gain_loss_pct_est_high': Decimal('10.940700000000'), 'gain_loss_pct_est_low': Decimal('-4.127700000000'), 'gain_loss_pct': Decimal('3.814800000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32284, 'bo_uuid': '6a004633-175f-4650-b09b-67a5eca4aeae', 'buy_curr_symb': 'PNG', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PNG', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PNG', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('116620741.418695990000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 20, 45, 52), 'check_last_dttm': datetime.datetime(2024, 12, 1, 8, 39, 34), 'add_dttm': datetime.datetime(2024, 11, 30, 20, 45, 52), 'upd_dttm': datetime.datetime(2024, 11, 30, 20, 45, 52), 'del_dttm': datetime.datetime(2024, 11, 30, 20, 45, 52), 'dlm': datetime.datetime(2024, 12, 1, 8, 40, 27)},
		{'pos_id': 32783, 'symb': 'USDC', 'prod_id': 'HNT-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 12, 1, 7, 12, 48), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 13, 4), 'age_mins': 52, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.015003600000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007556400000'), 'fees_cnt_tot': Decimal('0.015037696758'), 'buy_cnt': Decimal('0.360631321191'), 'sell_cnt_tot': Decimal('0.360000000000'), 'hold_cnt': Decimal('0.000631321191'), 'pocket_cnt': Decimal('0.000631321191'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('8.318745000000'), 'prc_curr': Decimal('8.375010000000'), 'prc_high': Decimal('8.865000000000'), 'prc_low': Decimal('8.298000000000'), 'prc_chg_pct': Decimal('0.676400000000'), 'prc_chg_pct_high': Decimal('5.701000000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-5.020000000000'), 'prc_sell_avg': Decimal('8.375010000000'), 'val_curr': Decimal('0.005287321288'), 'val_tot': Decimal('0.005287321288'), 'gain_loss_amt_est': Decimal('0.020287314975'), 'gain_loss_amt_est_high': Decimal('0.171031207232'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.020290921288'), 'gain_loss_amt_net': Decimal('0.025578242576'), 'gain_loss_pct_est': Decimal('0.676200000000'), 'gain_loss_pct_est_high': Decimal('5.701000000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('0.676400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32437, 'bo_uuid': 'f6ed0e0f-fd99-4a87-87dd-a48641f1fdb5', 'buy_curr_symb': 'HNT', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'HNT', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'HNT', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('7031336.375555180000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 12, 1, 7, 13, 36), 'check_last_dttm': datetime.datetime(2024, 12, 1, 8, 3, 55), 'add_dttm': datetime.datetime(2024, 12, 1, 7, 13, 36), 'upd_dttm': datetime.datetime(2024, 12, 1, 7, 13, 36), 'del_dttm': datetime.datetime(2024, 12, 1, 7, 13, 36), 'dlm': datetime.datetime(2024, 12, 1, 8, 5, 9)},
		{'pos_id': 32739, 'symb': 'USDC', 'prod_id': 'MOBILE-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 12, 1, 2, 57, 30), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 13, 1, 22), 'age_mins': 304, 'tot_out_cnt': Decimal('10.000000000000'), 'tot_in_cnt': Decimal('11.114513077500'), 'buy_fees_cnt': Decimal('0.024937655860'), 'sell_fees_cnt_tot': Decimal('0.027855922500'), 'fees_cnt_tot': Decimal('0.052793578360'), 'buy_cnt': Decimal('9509.115676014920'), 'sell_cnt_tot': Decimal('9387.000000000000'), 'hold_cnt': Decimal('122.115676014921'), 'pocket_cnt': Decimal('122.115676014921'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.001051620000'), 'prc_curr': Decimal('0.001184030000'), 'prc_high': Decimal('0.001273000000'), 'prc_low': Decimal('0.001049000000'), 'prc_chg_pct': Decimal('12.591000000000'), 'prc_chg_pct_high': Decimal('16.011500000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-3.420000000000'), 'prc_sell_avg': Decimal('0.001184030000'), 'val_curr': Decimal('0.144588623872'), 'val_tot': Decimal('0.144588623872'), 'gain_loss_amt_est': Decimal('1.277811191754'), 'gain_loss_amt_est_high': Decimal('1.601121124738'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('1.259101701372'), 'gain_loss_amt_net': Decimal('1.403690325244'), 'gain_loss_pct_est': Decimal('12.778100000000'), 'gain_loss_pct_est_high': Decimal('16.011200000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('12.591000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32393, 'bo_uuid': 'f92dbfcf-9f5d-42be-a74e-0386926addd7', 'buy_curr_symb': 'MOBILE', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'MOBILE', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'MOBILE', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('2779713425.988456200000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 12, 1, 2, 57, 58), 'check_last_dttm': datetime.datetime(2024, 12, 1, 8, 1, 17), 'add_dttm': datetime.datetime(2024, 12, 1, 2, 57, 58), 'upd_dttm': datetime.datetime(2024, 12, 1, 2, 57, 58), 'del_dttm': datetime.datetime(2024, 12, 1, 2, 57, 58), 'dlm': datetime.datetime(2024, 12, 1, 8, 2, 25)},
		{'pos_id': 32627, 'symb': 'USDC', 'prod_id': 'ALGO-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 20, 41, 9), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 12, 0, 44), 'age_mins': 619, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.857014500000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.017185500000'), 'fees_cnt_tot': Decimal('0.032148093516'), 'buy_cnt': Decimal('13.796766727717'), 'sell_cnt_tot': Decimal('13.500000000000'), 'hold_cnt': Decimal('0.296766727717'), 'pocket_cnt': Decimal('0.296766727717'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.434884500000'), 'prc_curr': Decimal('0.507927000000'), 'prc_high': Decimal('0.523300000000'), 'prc_low': Decimal('0.427200000000'), 'prc_chg_pct': Decimal('16.795800000000'), 'prc_chg_pct_high': Decimal('18.353300000000'), 'prc_chg_pct_low': Decimal('-1.767000000000'), 'prc_chg_pct_drop': Decimal('-1.560000000000'), 'prc_sell_avg': Decimal('0.507927000000'), 'val_curr': Decimal('0.150735833709'), 'val_tot': Decimal('0.150735833709'), 'gain_loss_amt_est': Decimal('0.935634634023'), 'gain_loss_amt_est_high': Decimal('1.101195834756'), 'gain_loss_amt_est_low': Decimal('-0.106021253919'), 'gain_loss_amt': Decimal('1.007750333709'), 'gain_loss_amt_net': Decimal('1.158486167418'), 'gain_loss_pct_est': Decimal('15.593900000000'), 'gain_loss_pct_est_high': Decimal('18.353300000000'), 'gain_loss_pct_est_low': Decimal('-1.767000000000'), 'gain_loss_pct': Decimal('16.795800000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32281, 'bo_uuid': 'c71ec60d-0413-4a32-b003-c854c4184791', 'buy_curr_symb': 'ALGO', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ALGO', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ALGO', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('48877100.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 20, 41, 44), 'check_last_dttm': datetime.datetime(2024, 12, 1, 7, 0, 39), 'add_dttm': datetime.datetime(2024, 11, 30, 20, 41, 44), 'upd_dttm': datetime.datetime(2024, 11, 30, 20, 41, 44), 'del_dttm': datetime.datetime(2024, 11, 30, 20, 41, 44), 'dlm': datetime.datetime(2024, 12, 1, 7, 2, 31)},
		{'pos_id': 32621, 'symb': 'USDC', 'prod_id': 'ALGO-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 20, 32, 52), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 12, 0, 25), 'age_mins': 627, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.914590200000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.017329800000'), 'fees_cnt_tot': Decimal('0.032292393516'), 'buy_cnt': Decimal('13.825450234428'), 'sell_cnt_tot': Decimal('13.600000000000'), 'hold_cnt': Decimal('0.225450234428'), 'pocket_cnt': Decimal('0.225450234428'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.433982250000'), 'prc_curr': Decimal('0.508425750000'), 'prc_high': Decimal('0.523300000000'), 'prc_low': Decimal('0.427200000000'), 'prc_chg_pct': Decimal('17.153600000000'), 'prc_chg_pct_high': Decimal('18.599300000000'), 'prc_chg_pct_low': Decimal('-1.562800000000'), 'prc_chg_pct_drop': Decimal('-1.450000000000'), 'prc_sell_avg': Decimal('0.508425750000'), 'val_curr': Decimal('0.114624704527'), 'val_tot': Decimal('0.114624704527'), 'gain_loss_amt_est': Decimal('0.950053832847'), 'gain_loss_amt_est_high': Decimal('1.115959235660'), 'gain_loss_amt_est_low': Decimal('-0.093767659852'), 'gain_loss_amt': Decimal('1.029214904527'), 'gain_loss_amt_net': Decimal('1.143839609053'), 'gain_loss_pct_est': Decimal('15.834200000000'), 'gain_loss_pct_est_high': Decimal('18.599300000000'), 'gain_loss_pct_est_low': Decimal('-1.562800000000'), 'gain_loss_pct': Decimal('17.153600000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32275, 'bo_uuid': '32010bd5-0488-4796-ade6-fbc8bfa0bc2f', 'buy_curr_symb': 'ALGO', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ALGO', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ALGO', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('48877100.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 20, 33, 16), 'check_last_dttm': datetime.datetime(2024, 12, 1, 7, 0, 20), 'add_dttm': datetime.datetime(2024, 11, 30, 20, 33, 16), 'upd_dttm': datetime.datetime(2024, 11, 30, 20, 33, 16), 'del_dttm': datetime.datetime(2024, 11, 30, 20, 33, 16), 'dlm': datetime.datetime(2024, 12, 1, 7, 1)},
		{'pos_id': 32312, 'symb': 'USDC', 'prod_id': 'ALGO-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 8, 21, 35), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 11, 59, 31), 'age_mins': 1358, 'tot_out_cnt': Decimal('10.000000000000'), 'tot_in_cnt': Decimal('10.597340250000'), 'buy_fees_cnt': Decimal('0.024937655860'), 'sell_fees_cnt_tot': Decimal('0.026559750000'), 'fees_cnt_tot': Decimal('0.051497405860'), 'buy_cnt': Decimal('21.200982665547'), 'sell_cnt_tot': Decimal('21.000000000000'), 'hold_cnt': Decimal('0.200982665547'), 'pocket_cnt': Decimal('0.200982665547'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.471676250000'), 'prc_curr': Decimal('0.504635250000'), 'prc_high': Decimal('0.523300000000'), 'prc_low': Decimal('0.427200000000'), 'prc_chg_pct': Decimal('6.987600000000'), 'prc_chg_pct_high': Decimal('9.121500000000'), 'prc_chg_pct_low': Decimal('-9.429400000000'), 'prc_chg_pct_drop': Decimal('-2.130000000000'), 'prc_sell_avg': Decimal('0.504635250000'), 'val_curr': Decimal('0.101422937674'), 'val_tot': Decimal('0.101422937674'), 'gain_loss_amt_est': Decimal('0.657733985970'), 'gain_loss_amt_est_high': Decimal('0.912145777957'), 'gain_loss_amt_est_low': Decimal('-0.942940205278'), 'gain_loss_amt': Decimal('0.698763187674'), 'gain_loss_amt_net': Decimal('0.800186125348'), 'gain_loss_pct_est': Decimal('6.577300000000'), 'gain_loss_pct_est_high': Decimal('9.121500000000'), 'gain_loss_pct_est_low': Decimal('-9.429400000000'), 'gain_loss_pct': Decimal('6.987600000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31966, 'bo_uuid': 'c0fe98b6-fe95-4a2a-91a5-792be45d9053', 'buy_curr_symb': 'ALGO', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ALGO', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ALGO', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('48877100.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 8, 22, 25), 'check_last_dttm': datetime.datetime(2024, 12, 1, 6, 59, 27), 'add_dttm': datetime.datetime(2024, 11, 30, 8, 22, 25), 'upd_dttm': datetime.datetime(2024, 11, 30, 8, 22, 25), 'del_dttm': datetime.datetime(2024, 11, 30, 8, 22, 25), 'dlm': datetime.datetime(2024, 12, 1, 7, 0, 32)},
		{'pos_id': 32655, 'symb': 'USDC', 'prod_id': 'CVX-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 21, 48, 58), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 7, 52, 42), 'age_mins': 304, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.113885775000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007804225000'), 'fees_cnt_tot': Decimal('0.015285521758'), 'buy_cnt': Decimal('0.718491885532'), 'sell_cnt_tot': Decimal('0.715000000000'), 'hold_cnt': Decimal('0.003491885532'), 'pocket_cnt': Decimal('0.003491885532'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('4.175412500000'), 'prc_curr': Decimal('4.355085000000'), 'prc_high': Decimal('4.560000000000'), 'prc_low': Decimal('3.880000000000'), 'prc_chg_pct': Decimal('4.303100000000'), 'prc_chg_pct_high': Decimal('6.193100000000'), 'prc_chg_pct_low': Decimal('-7.075000000000'), 'prc_chg_pct_drop': Decimal('-1.890000000000'), 'prc_sell_avg': Decimal('4.355085000000'), 'val_curr': Decimal('0.015207458302'), 'val_tot': Decimal('0.015207458302'), 'gain_loss_amt_est': Decimal('0.124721210179'), 'gain_loss_amt_est_high': Decimal('0.185793020449'), 'gain_loss_amt_est_low': Decimal('-0.212251484136'), 'gain_loss_amt': Decimal('0.129093233302'), 'gain_loss_amt_net': Decimal('0.144300691604'), 'gain_loss_pct_est': Decimal('4.157400000000'), 'gain_loss_pct_est_high': Decimal('6.193100000000'), 'gain_loss_pct_est_low': Decimal('-7.075000000000'), 'gain_loss_pct': Decimal('4.303100000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32309, 'bo_uuid': '818cbf8f-d71a-4da6-9849-ea4314047cd4', 'buy_curr_symb': 'CVX', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'CVX', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'CVX', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('1934876.697921781300'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 21, 49, 9), 'check_last_dttm': datetime.datetime(2024, 12, 1, 2, 52, 37), 'add_dttm': datetime.datetime(2024, 11, 30, 21, 49, 9), 'upd_dttm': datetime.datetime(2024, 11, 30, 21, 49, 9), 'del_dttm': datetime.datetime(2024, 11, 30, 21, 49, 9), 'dlm': datetime.datetime(2024, 12, 1, 2, 53, 18)},
		{'pos_id': 32706, 'symb': 'USDC', 'prod_id': 'POLS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 12, 1, 1, 15, 35), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 7, 52, 33), 'age_mins': 97, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.160378455000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015439545000'), 'fees_cnt_tot': Decimal('0.030402138516'), 'buy_cnt': Decimal('10.026867827917'), 'sell_cnt_tot': Decimal('9.990000000000'), 'hold_cnt': Decimal('0.036867827917'), 'pocket_cnt': Decimal('0.036867827917'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.598392250000'), 'prc_curr': Decimal('0.616654500000'), 'prc_high': Decimal('0.647000000000'), 'prc_low': Decimal('0.596900000000'), 'prc_chg_pct': Decimal('3.051900000000'), 'prc_chg_pct_high': Decimal('7.638400000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-4.590000000000'), 'prc_sell_avg': Decimal('0.616654500000'), 'val_curr': Decimal('0.022734711990'), 'val_tot': Decimal('0.022734711990'), 'gain_loss_amt_est': Decimal('0.198609691218'), 'gain_loss_amt_est_high': Decimal('0.458305567961'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.183113166990'), 'gain_loss_amt_net': Decimal('0.205847878980'), 'gain_loss_pct_est': Decimal('3.310200000000'), 'gain_loss_pct_est_high': Decimal('7.638400000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('3.051900000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32360, 'bo_uuid': 'c6f35a8e-16ee-4d68-8de5-c80c3726b5d2', 'buy_curr_symb': 'POLS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'POLS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'POLS', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('24138100.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 12, 1, 1, 16, 36), 'check_last_dttm': datetime.datetime(2024, 12, 1, 2, 52, 29), 'add_dttm': datetime.datetime(2024, 12, 1, 1, 16, 36), 'upd_dttm': datetime.datetime(2024, 12, 1, 1, 16, 36), 'del_dttm': datetime.datetime(2024, 12, 1, 1, 16, 36), 'dlm': datetime.datetime(2024, 12, 1, 2, 53, 11)},
		{'pos_id': 32647, 'symb': 'USDC', 'prod_id': 'CVX-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 21, 25, 25), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 7, 52, 26), 'age_mins': 327, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.129032812500'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007842187500'), 'fees_cnt_tot': Decimal('0.015323484258'), 'buy_cnt': Decimal('0.721088844155'), 'sell_cnt_tot': Decimal('0.717000000000'), 'hold_cnt': Decimal('0.004088844155'), 'pocket_cnt': Decimal('0.004088844155'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('4.160375000000'), 'prc_curr': Decimal('4.364062500000'), 'prc_high': Decimal('4.560000000000'), 'prc_low': Decimal('3.880000000000'), 'prc_chg_pct': Decimal('4.895900000000'), 'prc_chg_pct_high': Decimal('6.576900000000'), 'prc_chg_pct_low': Decimal('-6.739200000000'), 'prc_chg_pct_drop': Decimal('-1.680000000000'), 'prc_sell_avg': Decimal('4.364062500000'), 'val_curr': Decimal('0.017843971445'), 'val_tot': Decimal('0.017843971445'), 'gain_loss_amt_est': Decimal('0.136015383230'), 'gain_loss_amt_est_high': Decimal('0.197307934983'), 'gain_loss_amt_est_low': Decimal('-0.202175284679'), 'gain_loss_amt': Decimal('0.146876783945'), 'gain_loss_amt_net': Decimal('0.164720755390'), 'gain_loss_pct_est': Decimal('4.533800000000'), 'gain_loss_pct_est_high': Decimal('6.576900000000'), 'gain_loss_pct_est_low': Decimal('-6.739200000000'), 'gain_loss_pct': Decimal('4.895900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32301, 'bo_uuid': '4a577547-2fc8-4ee8-8207-fa6b0d8cd219', 'buy_curr_symb': 'CVX', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'CVX', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'CVX', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('1934876.697921781300'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 21, 25, 46), 'check_last_dttm': datetime.datetime(2024, 12, 1, 2, 52, 21), 'add_dttm': datetime.datetime(2024, 11, 30, 21, 25, 46), 'upd_dttm': datetime.datetime(2024, 11, 30, 21, 25, 46), 'del_dttm': datetime.datetime(2024, 11, 30, 21, 25, 46), 'dlm': datetime.datetime(2024, 12, 1, 2, 53, 4)},
		{'pos_id': 32623, 'symb': 'USDC', 'prod_id': 'CVX-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 20, 36, 57), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 7, 52, 10), 'age_mins': 376, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.125568495000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007833505000'), 'fees_cnt_tot': Decimal('0.015314801758'), 'buy_cnt': Decimal('0.723005243596'), 'sell_cnt_tot': Decimal('0.719000000000'), 'hold_cnt': Decimal('0.004005243596'), 'pocket_cnt': Decimal('0.004005243596'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('4.149347500000'), 'prc_curr': Decimal('4.347105000000'), 'prc_high': Decimal('4.560000000000'), 'prc_low': Decimal('3.880000000000'), 'prc_chg_pct': Decimal('4.766000000000'), 'prc_chg_pct_high': Decimal('6.860200000000'), 'prc_chg_pct_low': Decimal('-6.491300000000'), 'prc_chg_pct_drop': Decimal('-2.090000000000'), 'prc_sell_avg': Decimal('4.347105000000'), 'val_curr': Decimal('0.017411214462'), 'val_tot': Decimal('0.017411214462'), 'gain_loss_amt_est': Decimal('0.144349804399'), 'gain_loss_amt_est_high': Decimal('0.205805250105'), 'gain_loss_amt_est_low': Decimal('-0.194739654848'), 'gain_loss_amt': Decimal('0.142979709462'), 'gain_loss_amt_net': Decimal('0.160390923925'), 'gain_loss_pct_est': Decimal('4.811700000000'), 'gain_loss_pct_est_high': Decimal('6.860200000000'), 'gain_loss_pct_est_low': Decimal('-6.491300000000'), 'gain_loss_pct': Decimal('4.766000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32277, 'bo_uuid': '8e0c3b12-d4ed-4a67-9963-e45bc4023b2b', 'buy_curr_symb': 'CVX', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'CVX', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'CVX', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('1934876.697921781300'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 20, 37, 7), 'check_last_dttm': datetime.datetime(2024, 12, 1, 2, 52, 6), 'add_dttm': datetime.datetime(2024, 11, 30, 20, 37, 7), 'upd_dttm': datetime.datetime(2024, 11, 30, 20, 37, 7), 'del_dttm': datetime.datetime(2024, 11, 30, 20, 37, 7), 'dlm': datetime.datetime(2024, 12, 1, 2, 52, 56)},
		{'pos_id': 32705, 'symb': 'USDC', 'prod_id': 'POLS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 12, 1, 1, 7, 22), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 7, 17, 35), 'age_mins': 71, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.211825515000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015568485000'), 'fees_cnt_tot': Decimal('0.030531078516'), 'buy_cnt': Decimal('9.913926464277'), 'sell_cnt_tot': Decimal('9.870000000000'), 'hold_cnt': Decimal('0.043926464277'), 'pocket_cnt': Decimal('0.043926464277'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.605209250000'), 'prc_curr': Decimal('0.629364290000'), 'prc_high': Decimal('0.641000000000'), 'prc_low': Decimal('0.603700000000'), 'prc_chg_pct': Decimal('3.991200000000'), 'prc_chg_pct_high': Decimal('5.120700000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-1.130000000000'), 'prc_sell_avg': Decimal('0.629364290000'), 'val_curr': Decimal('0.027645748002'), 'val_tot': Decimal('0.027645748002'), 'gain_loss_amt_est': Decimal('0.254696206312'), 'gain_loss_amt_est_high': Decimal('0.307240016573'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.239471263002'), 'gain_loss_amt_net': Decimal('0.267117011004'), 'gain_loss_pct_est': Decimal('4.244900000000'), 'gain_loss_pct_est_high': Decimal('5.120700000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('3.991200000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32359, 'bo_uuid': '407584a0-2dbe-41e7-89b7-9e0bf43e3b4e', 'buy_curr_symb': 'POLS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'POLS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'POLS', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('24138100.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 12, 1, 1, 7, 34), 'check_last_dttm': datetime.datetime(2024, 12, 1, 2, 17, 31), 'add_dttm': datetime.datetime(2024, 12, 1, 1, 7, 34), 'upd_dttm': datetime.datetime(2024, 12, 1, 1, 7, 34), 'del_dttm': datetime.datetime(2024, 12, 1, 1, 7, 34), 'dlm': datetime.datetime(2024, 12, 1, 2, 18, 48)},
		{'pos_id': 32222, 'symb': 'USDC', 'prod_id': 'SHIB-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 5, 6, 19), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 1, 36, 48), 'age_mins': 931, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.191203915200'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007998004800'), 'fees_cnt_tot': Decimal('0.015479301558'), 'buy_cnt': Decimal('110506.599085742070'), 'sell_cnt_tot': Decimal('109712.000000000000'), 'hold_cnt': Decimal('794.599085742069'), 'pocket_cnt': Decimal('794.599085742069'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.000027150000'), 'prc_curr': Decimal('0.000029090000'), 'prc_high': Decimal('0.000030000000'), 'prc_low': Decimal('0.000026700000'), 'prc_chg_pct': Decimal('7.145500000000'), 'prc_chg_pct_high': Decimal('10.128900000000'), 'prc_chg_pct_low': Decimal('-1.657500000000'), 'prc_chg_pct_drop': Decimal('-2.980000000000'), 'prc_sell_avg': Decimal('0.000029090000'), 'val_curr': Decimal('0.023114887404'), 'val_tot': Decimal('0.023114887404'), 'gain_loss_amt_est': Decimal('0.215742033395'), 'gain_loss_amt_est_high': Decimal('0.304147312664'), 'gain_loss_amt_est_low': Decimal('-0.049473804411'), 'gain_loss_amt': Decimal('0.214318802604'), 'gain_loss_amt_net': Decimal('0.237433690008'), 'gain_loss_pct_est': Decimal('7.191400000000'), 'gain_loss_pct_est_high': Decimal('10.138200000000'), 'gain_loss_pct_est_low': Decimal('-1.649100000000'), 'gain_loss_pct': Decimal('7.144000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31876, 'bo_uuid': 'b37bdc8b-9087-4fc1-9d0c-e2113ba4c37e', 'buy_curr_symb': 'SHIB', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SHIB', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SHIB', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('970443076600.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('1.0000E-8'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 5, 7, 28), 'check_last_dttm': datetime.datetime(2024, 11, 30, 20, 36, 44), 'add_dttm': datetime.datetime(2024, 11, 30, 5, 7, 28), 'upd_dttm': datetime.datetime(2024, 11, 30, 5, 7, 28), 'del_dttm': datetime.datetime(2024, 11, 30, 5, 7, 28), 'dlm': datetime.datetime(2024, 11, 30, 20, 37, 26)},
		{'pos_id': 32017, 'symb': 'USDC', 'prod_id': 'AAVE-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 19, 28, 17), 'pos_end_dttm': datetime.datetime(2024, 12, 1, 0, 54, 17), 'age_mins': 1466, 'tot_out_cnt': Decimal('10.000000000000'), 'tot_in_cnt': Decimal('10.311686175000'), 'buy_fees_cnt': Decimal('0.024937655860'), 'sell_fees_cnt_tot': Decimal('0.025843825000'), 'fees_cnt_tot': Decimal('0.050781480860'), 'buy_cnt': Decimal('0.049750934385'), 'sell_cnt_tot': Decimal('0.049000000000'), 'hold_cnt': Decimal('0.000750934385'), 'pocket_cnt': Decimal('0.000750934385'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('201.001250000000'), 'prc_curr': Decimal('210.442575000000'), 'prc_high': Decimal('214.990000000000'), 'prc_low': Decimal('198.580000000000'), 'prc_chg_pct': Decimal('4.697100000000'), 'prc_chg_pct_high': Decimal('6.656100000000'), 'prc_chg_pct_low': Decimal('-1.204600000000'), 'prc_chg_pct_drop': Decimal('-1.960000000000'), 'prc_sell_avg': Decimal('210.442575000000'), 'val_curr': Decimal('0.158028565635'), 'val_tot': Decimal('0.158028565635'), 'gain_loss_amt_est': Decimal('0.498939683267'), 'gain_loss_amt_est_high': Decimal('0.665605313456'), 'gain_loss_amt_est_low': Decimal('-0.120459449827'), 'gain_loss_amt': Decimal('0.469714740635'), 'gain_loss_amt_net': Decimal('0.627743306271'), 'gain_loss_pct_est': Decimal('4.989400000000'), 'gain_loss_pct_est_high': Decimal('6.656100000000'), 'gain_loss_pct_est_low': Decimal('-1.204600000000'), 'gain_loss_pct': Decimal('4.697100000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31671, 'bo_uuid': '605c044b-f781-4303-8768-299a3a071d96', 'buy_curr_symb': 'AAVE', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'AAVE', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'AAVE', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('141000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 19, 29, 15), 'check_last_dttm': datetime.datetime(2024, 11, 30, 19, 54, 13), 'add_dttm': datetime.datetime(2024, 11, 29, 19, 29, 15), 'upd_dttm': datetime.datetime(2024, 11, 29, 19, 29, 15), 'del_dttm': datetime.datetime(2024, 11, 29, 19, 29, 15), 'dlm': datetime.datetime(2024, 11, 30, 19, 54, 32)},
		{'pos_id': 32483, 'symb': 'USDC', 'prod_id': 'PNG-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 15, 4, 23), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 22, 48, 44), 'age_mins': 163, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.838620600000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.017139400000'), 'fees_cnt_tot': Decimal('0.032101993516'), 'buy_cnt': Decimal('17.288302396036'), 'sell_cnt_tot': Decimal('17.000000000000'), 'hold_cnt': Decimal('0.288302396036'), 'pocket_cnt': Decimal('0.288302396036'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.347055480000'), 'prc_curr': Decimal('0.402271800000'), 'prc_high': Decimal('0.423900000000'), 'prc_low': Decimal('0.345620000000'), 'prc_chg_pct': Decimal('15.909900000000'), 'prc_chg_pct_high': Decimal('20.441800000000'), 'prc_chg_pct_low': Decimal('-0.413600000000'), 'prc_chg_pct_drop': Decimal('-4.530000000000'), 'prc_sell_avg': Decimal('0.402271800000'), 'val_curr': Decimal('0.115975923798'), 'val_tot': Decimal('0.115975923798'), 'gain_loss_amt_est': Decimal('0.993636968268'), 'gain_loss_amt_est_high': Decimal('1.226510401543'), 'gain_loss_amt_est_low': Decimal('-0.024816925882'), 'gain_loss_amt': Decimal('0.954596523798'), 'gain_loss_amt_net': Decimal('1.070572447595'), 'gain_loss_pct_est': Decimal('16.560600000000'), 'gain_loss_pct_est_high': Decimal('20.441800000000'), 'gain_loss_pct_est_low': Decimal('-0.413600000000'), 'gain_loss_pct': Decimal('15.909900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32137, 'bo_uuid': '8bd186cc-ca52-4d5b-97ed-9f6aa110e8f9', 'buy_curr_symb': 'PNG', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PNG', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PNG', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('116620741.418695990000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 15, 5, 12), 'check_last_dttm': datetime.datetime(2024, 11, 30, 17, 48, 40), 'add_dttm': datetime.datetime(2024, 11, 30, 15, 5, 12), 'upd_dttm': datetime.datetime(2024, 11, 30, 15, 5, 12), 'del_dttm': datetime.datetime(2024, 11, 30, 15, 5, 12), 'dlm': datetime.datetime(2024, 11, 30, 17, 50)},
		{'pos_id': 32256, 'symb': 'USDC', 'prod_id': 'PNG-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 6, 32, 59), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 22, 48, 16), 'age_mins': 675, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.436348800000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.016131200000'), 'fees_cnt_tot': Decimal('0.031093793516'), 'buy_cnt': Decimal('17.253415798910'), 'sell_cnt_tot': Decimal('16.000000000000'), 'hold_cnt': Decimal('1.253415798910'), 'pocket_cnt': Decimal('1.253415798910'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.347757230000'), 'prc_curr': Decimal('0.402271800000'), 'prc_high': Decimal('0.423900000000'), 'prc_low': Decimal('0.328000000000'), 'prc_chg_pct': Decimal('15.676000000000'), 'prc_chg_pct_high': Decimal('20.198800000000'), 'prc_chg_pct_low': Decimal('-5.681300000000'), 'prc_chg_pct_drop': Decimal('-4.520000000000'), 'prc_sell_avg': Decimal('0.402271800000'), 'val_curr': Decimal('0.504213829576'), 'val_tot': Decimal('0.504213829576'), 'gain_loss_amt_est': Decimal('0.979524293133'), 'gain_loss_amt_est_high': Decimal('1.211927803944'), 'gain_loss_amt_est_low': Decimal('-0.340879617958'), 'gain_loss_amt': Decimal('0.940562629576'), 'gain_loss_amt_net': Decimal('1.444776459152'), 'gain_loss_pct_est': Decimal('16.325400000000'), 'gain_loss_pct_est_high': Decimal('20.198800000000'), 'gain_loss_pct_est_low': Decimal('-5.681300000000'), 'gain_loss_pct': Decimal('15.676000000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31910, 'bo_uuid': 'f86e89e9-a101-4b97-bdb7-f2dce1b8c943', 'buy_curr_symb': 'PNG', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PNG', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PNG', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('116620741.418695990000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 6, 33, 58), 'check_last_dttm': datetime.datetime(2024, 11, 30, 17, 48, 12), 'add_dttm': datetime.datetime(2024, 11, 30, 6, 33, 58), 'upd_dttm': datetime.datetime(2024, 11, 30, 6, 33, 58), 'del_dttm': datetime.datetime(2024, 11, 30, 6, 33, 58), 'dlm': datetime.datetime(2024, 11, 30, 17, 49, 47)},
		{'pos_id': 31815, 'symb': 'USDC', 'prod_id': 'AVAX-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 12, 3, 10), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 22, 48, 4), 'age_mins': 1785, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.054395910413'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015173924588'), 'fees_cnt_tot': Decimal('0.030136518104'), 'buy_cnt': Decimal('0.134314124921'), 'sell_cnt_tot': Decimal('0.134134140000'), 'hold_cnt': Decimal('0.000179984921'), 'pocket_cnt': Decimal('0.000179984921'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('44.671400000000'), 'prc_curr': Decimal('45.136875000000'), 'prc_high': Decimal('46.290000000000'), 'prc_low': Decimal('43.730000000000'), 'prc_chg_pct': Decimal('1.042000000000'), 'prc_chg_pct_high': Decimal('3.309900000000'), 'prc_chg_pct_low': Decimal('-2.107400000000'), 'prc_chg_pct_drop': Decimal('-2.270000000000'), 'prc_sell_avg': Decimal('45.136875000000'), 'val_curr': Decimal('0.008123956881'), 'val_tot': Decimal('0.008123956881'), 'gain_loss_amt_est': Decimal('0.080400435174'), 'gain_loss_amt_est_high': Decimal('0.198596865104'), 'gain_loss_amt_est_low': Decimal('-0.126443317205'), 'gain_loss_amt': Decimal('0.062519867294'), 'gain_loss_amt_net': Decimal('0.070643824175'), 'gain_loss_pct_est': Decimal('1.340000000000'), 'gain_loss_pct_est_high': Decimal('3.309900000000'), 'gain_loss_pct_est_low': Decimal('-2.107400000000'), 'gain_loss_pct': Decimal('1.042000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '1h', 'sell_strat_type': 'strat', 'sell_strat_name': 'sha', 'sell_strat_freq': '1h', 'bo_id': 31469, 'bo_uuid': '5a1419bb-aa08-4416-add5-b110e9581a5b', 'buy_curr_symb': 'AVAX', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'AVAX', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'AVAX', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('631600.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 12, 3, 59), 'check_last_dttm': datetime.datetime(2024, 11, 30, 17, 48, 1), 'add_dttm': datetime.datetime(2024, 11, 29, 12, 3, 59), 'upd_dttm': datetime.datetime(2024, 11, 29, 12, 3, 59), 'del_dttm': datetime.datetime(2024, 11, 29, 12, 3, 59), 'dlm': datetime.datetime(2024, 11, 30, 17, 49, 40)},
		{'pos_id': 32251, 'symb': 'USDC', 'prod_id': 'PNG-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 6, 25, 6), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 22, 47, 57), 'age_mins': 683, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.838840050000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.017139950000'), 'fees_cnt_tot': Decimal('0.032102543516'), 'buy_cnt': Decimal('17.373112936092'), 'sell_cnt_tot': Decimal('17.000000000000'), 'hold_cnt': Decimal('0.373112936092'), 'pocket_cnt': Decimal('0.373112936092'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.345361250000'), 'prc_curr': Decimal('0.402284710000'), 'prc_high': Decimal('0.423900000000'), 'prc_low': Decimal('0.328000000000'), 'prc_chg_pct': Decimal('16.482300000000'), 'prc_chg_pct_high': Decimal('21.032700000000'), 'prc_chg_pct_low': Decimal('-5.027000000000'), 'prc_chg_pct_drop': Decimal('-4.550000000000'), 'prc_sell_avg': Decimal('0.402284710000'), 'val_curr': Decimal('0.150097629293'), 'val_tot': Decimal('0.150097629293'), 'gain_loss_amt_est': Decimal('1.027945376037'), 'gain_loss_amt_est_high': Decimal('1.261961207286'), 'gain_loss_amt_est_low': Decimal('-0.301618956962'), 'gain_loss_amt': Decimal('0.988937679293'), 'gain_loss_amt_net': Decimal('1.139035308586'), 'gain_loss_pct_est': Decimal('17.132400000000'), 'gain_loss_pct_est_high': Decimal('21.032700000000'), 'gain_loss_pct_est_low': Decimal('-5.027000000000'), 'gain_loss_pct': Decimal('16.482300000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31905, 'bo_uuid': 'e5591ca7-0473-436b-a5e3-3cbf10676f7f', 'buy_curr_symb': 'PNG', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PNG', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PNG', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('116620741.418695990000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 6, 25, 17), 'check_last_dttm': datetime.datetime(2024, 11, 30, 17, 47, 53), 'add_dttm': datetime.datetime(2024, 11, 30, 6, 25, 17), 'upd_dttm': datetime.datetime(2024, 11, 30, 6, 25, 17), 'del_dttm': datetime.datetime(2024, 11, 30, 6, 25, 17), 'dlm': datetime.datetime(2024, 11, 30, 17, 48, 17)},
		{'pos_id': 32249, 'symb': 'USDC', 'prod_id': 'PNG-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 6, 17, 27), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 22, 47, 39), 'age_mins': 690, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.850830000000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.017170000000'), 'fees_cnt_tot': Decimal('0.032132593516'), 'buy_cnt': Decimal('17.444511371605'), 'sell_cnt_tot': Decimal('17.000000000000'), 'hold_cnt': Decimal('0.444511371605'), 'pocket_cnt': Decimal('0.444511371605'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.343947720000'), 'prc_curr': Decimal('0.402990000000'), 'prc_high': Decimal('0.423900000000'), 'prc_low': Decimal('0.328000000000'), 'prc_chg_pct': Decimal('17.166100000000'), 'prc_chg_pct_high': Decimal('21.530100000000'), 'prc_chg_pct_low': Decimal('-4.636700000000'), 'prc_chg_pct_drop': Decimal('-4.360000000000'), 'prc_sell_avg': Decimal('0.402990000000'), 'val_curr': Decimal('0.179133637643'), 'val_tot': Decimal('0.179133637643'), 'gain_loss_amt_est': Decimal('1.056828185155'), 'gain_loss_amt_est_high': Decimal('1.291805753331'), 'gain_loss_amt_est_low': Decimal('-0.278200270114'), 'gain_loss_amt': Decimal('1.029963637643'), 'gain_loss_amt_net': Decimal('1.209097275286'), 'gain_loss_pct_est': Decimal('17.613800000000'), 'gain_loss_pct_est_high': Decimal('21.530100000000'), 'gain_loss_pct_est_low': Decimal('-4.636700000000'), 'gain_loss_pct': Decimal('17.166100000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31903, 'bo_uuid': 'feeb1278-ba11-4cab-9f7f-73af48f22c1d', 'buy_curr_symb': 'PNG', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PNG', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PNG', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('116620741.418695990000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 6, 17, 48), 'check_last_dttm': datetime.datetime(2024, 11, 30, 17, 47, 35), 'add_dttm': datetime.datetime(2024, 11, 30, 6, 17, 48), 'upd_dttm': datetime.datetime(2024, 11, 30, 6, 17, 48), 'del_dttm': datetime.datetime(2024, 11, 30, 6, 17, 48), 'dlm': datetime.datetime(2024, 11, 30, 17, 48, 10)},
		{'pos_id': 32133, 'symb': 'USDC', 'prod_id': 'PNG-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 0, 1, 28), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 22, 47, 22), 'age_mins': 1066, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.044850000000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015150000000'), 'fees_cnt_tot': Decimal('0.030112593516'), 'buy_cnt': Decimal('15.924005338523'), 'sell_cnt_tot': Decimal('15.000000000000'), 'hold_cnt': Decimal('0.924005338523'), 'pocket_cnt': Decimal('0.924005338523'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.376789630000'), 'prc_curr': Decimal('0.402990000000'), 'prc_high': Decimal('0.423900000000'), 'prc_low': Decimal('0.328000000000'), 'prc_chg_pct': Decimal('6.953600000000'), 'prc_chg_pct_high': Decimal('10.937200000000'), 'prc_chg_pct_low': Decimal('-12.948800000000'), 'prc_chg_pct_drop': Decimal('-3.980000000000'), 'prc_sell_avg': Decimal('0.402990000000'), 'val_curr': Decimal('0.372364911371'), 'val_tot': Decimal('0.372364911371'), 'gain_loss_amt_est': Decimal('0.441737879593'), 'gain_loss_amt_est_high': Decimal('0.656234231503'), 'gain_loss_amt_est_low': Decimal('-0.776926248964'), 'gain_loss_amt': Decimal('0.417214911371'), 'gain_loss_amt_net': Decimal('0.789579822743'), 'gain_loss_pct_est': Decimal('7.362300000000'), 'gain_loss_pct_est_high': Decimal('10.937200000000'), 'gain_loss_pct_est_low': Decimal('-12.948800000000'), 'gain_loss_pct': Decimal('6.953600000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31787, 'bo_uuid': 'c192fe23-0eb0-4ea5-9700-c6237eb78267', 'buy_curr_symb': 'PNG', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PNG', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PNG', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('116620741.418695990000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 0, 1, 37), 'check_last_dttm': datetime.datetime(2024, 11, 30, 17, 47, 19), 'add_dttm': datetime.datetime(2024, 11, 30, 0, 1, 37), 'upd_dttm': datetime.datetime(2024, 11, 30, 0, 1, 37), 'del_dttm': datetime.datetime(2024, 11, 30, 0, 1, 37), 'dlm': datetime.datetime(2024, 11, 30, 17, 48, 3)},
		{'pos_id': 31872, 'symb': 'USDC', 'prod_id': 'DOT-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 14, 57, 35), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 22, 29, 35), 'age_mins': 1592, 'tot_out_cnt': Decimal('15.000000000000'), 'tot_in_cnt': Decimal('15.543447839697'), 'buy_fees_cnt': Decimal('0.037406483791'), 'sell_fees_cnt_tot': Decimal('0.038956009623'), 'fees_cnt_tot': Decimal('0.076362493414'), 'buy_cnt': Decimal('1.715893751859'), 'sell_cnt_tot': Decimal('1.708784280000'), 'hold_cnt': Decimal('0.007109471859'), 'pocket_cnt': Decimal('0.007109471859'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('8.741800000000'), 'prc_curr': Decimal('9.096202500000'), 'prc_high': Decimal('9.199000000000'), 'prc_low': Decimal('8.624000000000'), 'prc_chg_pct': Decimal('4.054100000000'), 'prc_chg_pct_high': Decimal('5.138500000000'), 'prc_chg_pct_low': Decimal('-1.347500000000'), 'prc_chg_pct_drop': Decimal('-1.080000000000'), 'prc_sell_avg': Decimal('9.096202500000'), 'val_curr': Decimal('0.064669195698'), 'val_tot': Decimal('0.064669195698'), 'gain_loss_amt_est': Decimal('0.621496716924'), 'gain_loss_amt_est_high': Decimal('0.770779473336'), 'gain_loss_amt_est_low': Decimal('-0.202132283968'), 'gain_loss_amt': Decimal('0.608117035395'), 'gain_loss_amt_net': Decimal('0.672786231092'), 'gain_loss_pct_est': Decimal('4.143300000000'), 'gain_loss_pct_est_high': Decimal('5.138500000000'), 'gain_loss_pct_est_low': Decimal('-1.347500000000'), 'gain_loss_pct': Decimal('4.054100000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31526, 'bo_uuid': '92c3fc04-3310-4e93-a0d6-e16500f3b12d', 'buy_curr_symb': 'DOT', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOT', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOT', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('1732900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 14, 58, 15), 'check_last_dttm': datetime.datetime(2024, 11, 30, 17, 29, 31), 'add_dttm': datetime.datetime(2024, 11, 29, 14, 58, 15), 'upd_dttm': datetime.datetime(2024, 11, 29, 14, 58, 15), 'del_dttm': datetime.datetime(2024, 11, 29, 14, 58, 15), 'dlm': datetime.datetime(2024, 11, 30, 17, 30, 4)},
		{'pos_id': 30152, 'symb': 'USDC', 'prod_id': 'POL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 11, 37, 30), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 21, 41, 32), 'age_mins': 4624, 'tot_out_cnt': Decimal('2.000000000000'), 'tot_in_cnt': Decimal('2.078402970000'), 'buy_fees_cnt': Decimal('0.004987531172'), 'sell_fees_cnt_tot': Decimal('0.005209030000'), 'fees_cnt_tot': Decimal('0.010196561172'), 'buy_cnt': Decimal('3.480482325241'), 'sell_cnt_tot': Decimal('3.460000000000'), 'hold_cnt': Decimal('0.020482325241'), 'pocket_cnt': Decimal('0.020482325241'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.574633000000'), 'prc_curr': Decimal('0.600694500000'), 'prc_high': Decimal('0.618300000000'), 'prc_low': Decimal('0.548800000000'), 'prc_chg_pct': Decimal('4.535300000000'), 'prc_chg_pct_high': Decimal('6.624600000000'), 'prc_chg_pct_low': Decimal('-4.495600000000'), 'prc_chg_pct_drop': Decimal('-2.090000000000'), 'prc_sell_avg': Decimal('0.600694500000'), 'val_curr': Decimal('0.012303620119'), 'val_tot': Decimal('0.012303620119'), 'gain_loss_amt_est': Decimal('0.097338649190'), 'gain_loss_amt_est_high': Decimal('0.132491520675'), 'gain_loss_amt_est_low': Decimal('-0.089911299908'), 'gain_loss_amt': Decimal('0.090706590119'), 'gain_loss_amt_net': Decimal('0.103010210239'), 'gain_loss_pct_est': Decimal('4.866900000000'), 'gain_loss_pct_est_high': Decimal('6.624600000000'), 'gain_loss_pct_est_low': Decimal('-4.495600000000'), 'gain_loss_pct': Decimal('4.535300000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29806, 'bo_uuid': '031bbf87-3076-40f8-b9cb-f2393102bb81', 'buy_curr_symb': 'POL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'POL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'POL', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('26020434.110127510000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 11, 37, 46), 'check_last_dttm': datetime.datetime(2024, 11, 30, 16, 41, 28), 'add_dttm': datetime.datetime(2024, 11, 27, 11, 37, 46), 'upd_dttm': datetime.datetime(2024, 11, 27, 11, 37, 46), 'del_dttm': datetime.datetime(2024, 11, 27, 11, 37, 46), 'dlm': datetime.datetime(2024, 11, 30, 16, 42, 5)},
		{'pos_id': 32407, 'symb': 'USDC', 'prod_id': 'XTZ-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 12, 30, 51), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 21, 34, 53), 'age_mins': 244, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.358383000000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.008417000000'), 'fees_cnt_tot': Decimal('0.015898296758'), 'buy_cnt': Decimal('1.931903617329'), 'sell_cnt_tot': Decimal('1.900000000000'), 'hold_cnt': Decimal('0.031903617329'), 'pocket_cnt': Decimal('0.031903617329'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.552872500000'), 'prc_curr': Decimal('1.767570000000'), 'prc_high': Decimal('1.854000000000'), 'prc_low': Decimal('1.548000000000'), 'prc_chg_pct': Decimal('13.825800000000'), 'prc_chg_pct_high': Decimal('19.327200000000'), 'prc_chg_pct_low': Decimal('-0.313800000000'), 'prc_chg_pct_drop': Decimal('-5.500000000000'), 'prc_sell_avg': Decimal('1.767570000000'), 'val_curr': Decimal('0.056391876882'), 'val_tot': Decimal('0.056391876882'), 'gain_loss_amt_est': Decimal('0.417537499055'), 'gain_loss_amt_est_high': Decimal('0.579817402911'), 'gain_loss_amt_est_low': Decimal('-0.009413200375'), 'gain_loss_amt': Decimal('0.414774876882'), 'gain_loss_amt_net': Decimal('0.471166753764'), 'gain_loss_pct_est': Decimal('13.917900000000'), 'gain_loss_pct_est_high': Decimal('19.327200000000'), 'gain_loss_pct_est_low': Decimal('-0.313800000000'), 'gain_loss_pct': Decimal('13.825800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 32061, 'bo_uuid': '05b4d373-b06d-433f-b1ea-71f3709c14f3', 'buy_curr_symb': 'XTZ', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XTZ', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XTZ', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('9339300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 12, 31, 32), 'check_last_dttm': datetime.datetime(2024, 11, 30, 16, 34, 50), 'add_dttm': datetime.datetime(2024, 11, 30, 12, 31, 32), 'upd_dttm': datetime.datetime(2024, 11, 30, 12, 31, 32), 'del_dttm': datetime.datetime(2024, 11, 30, 12, 31, 32), 'dlm': datetime.datetime(2024, 11, 30, 16, 35, 59)},
		{'pos_id': 32291, 'symb': 'USDC', 'prod_id': 'ALGO-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 7, 30, 9), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 13, 40, 54), 'age_mins': 71, 'tot_out_cnt': Decimal('25.000000000000'), 'tot_in_cnt': Decimal('25.407930975000'), 'buy_fees_cnt': Decimal('0.062344139651'), 'sell_fees_cnt_tot': Decimal('0.063679025000'), 'fees_cnt_tot': Decimal('0.126023164651'), 'buy_cnt': Decimal('55.951662239958'), 'sell_cnt_tot': Decimal('55.700000000000'), 'hold_cnt': Decimal('0.251662239958'), 'pocket_cnt': Decimal('0.251662239958'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.446814250000'), 'prc_curr': Decimal('0.456156750000'), 'prc_high': Decimal('0.483200000000'), 'prc_low': Decimal('0.440800000000'), 'prc_chg_pct': Decimal('2.090900000000'), 'prc_chg_pct_high': Decimal('6.755800000000'), 'prc_chg_pct_low': Decimal('-1.346000000000'), 'prc_chg_pct_drop': Decimal('-4.660000000000'), 'prc_sell_avg': Decimal('0.456156750000'), 'val_curr': Decimal('0.114797429477'), 'val_tot': Decimal('0.114797429477'), 'gain_loss_amt_est': Decimal('0.726574297933'), 'gain_loss_amt_est_high': Decimal('1.688942888460'), 'gain_loss_amt_est_low': Decimal('-0.336507284627'), 'gain_loss_amt': Decimal('0.522728404477'), 'gain_loss_amt_net': Decimal('0.637525833954'), 'gain_loss_pct_est': Decimal('2.906300000000'), 'gain_loss_pct_est_high': Decimal('6.755800000000'), 'gain_loss_pct_est_low': Decimal('-1.346000000000'), 'gain_loss_pct': Decimal('2.090900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31945, 'bo_uuid': 'dd981347-8b8b-4b78-aab2-8c9edc137871', 'buy_curr_symb': 'ALGO', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ALGO', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ALGO', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('48877100.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 7, 30, 28), 'check_last_dttm': datetime.datetime(2024, 11, 30, 8, 40, 51), 'add_dttm': datetime.datetime(2024, 11, 30, 7, 30, 28), 'upd_dttm': datetime.datetime(2024, 11, 30, 7, 30, 28), 'del_dttm': datetime.datetime(2024, 11, 30, 7, 30, 28), 'dlm': datetime.datetime(2024, 11, 30, 8, 42, 23)},
		{'pos_id': 32147, 'symb': 'USDC', 'prod_id': 'ENS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 0, 39, 10), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 11, 49, 55), 'age_mins': 370, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.277267500000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015732500000'), 'fees_cnt_tot': Decimal('0.030695093516'), 'buy_cnt': Decimal('0.140990280483'), 'sell_cnt_tot': Decimal('0.140000000000'), 'hold_cnt': Decimal('0.000990280483'), 'pocket_cnt': Decimal('0.000990280483'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('42.556125000000'), 'prc_curr': Decimal('44.837625000000'), 'prc_high': Decimal('47.050000000000'), 'prc_low': Decimal('42.450000000000'), 'prc_chg_pct': Decimal('5.361200000000'), 'prc_chg_pct_high': Decimal('9.032500000000'), 'prc_chg_pct_low': Decimal('-0.014400000000'), 'prc_chg_pct_drop': Decimal('-3.670000000000'), 'prc_sell_avg': Decimal('44.837625000000'), 'val_curr': Decimal('0.044401824942'), 'val_tot': Decimal('0.044401824942'), 'gain_loss_amt_est': Decimal('0.360071552588'), 'gain_loss_amt_est_high': Decimal('0.541949014411'), 'gain_loss_amt_est_low': Decimal('-0.000863565448'), 'gain_loss_amt': Decimal('0.321669324942'), 'gain_loss_amt_net': Decimal('0.366071149883'), 'gain_loss_pct_est': Decimal('6.001200000000'), 'gain_loss_pct_est_high': Decimal('9.032500000000'), 'gain_loss_pct_est_low': Decimal('-0.014400000000'), 'gain_loss_pct': Decimal('5.361200000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31801, 'bo_uuid': 'bfa7b866-c5ec-4799-88bf-0f41005e1237', 'buy_curr_symb': 'ENS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ENS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ENS', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('775300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 0, 40, 34), 'check_last_dttm': datetime.datetime(2024, 11, 30, 6, 49, 52), 'add_dttm': datetime.datetime(2024, 11, 30, 0, 40, 34), 'upd_dttm': datetime.datetime(2024, 11, 30, 0, 40, 34), 'del_dttm': datetime.datetime(2024, 11, 30, 0, 40, 34), 'dlm': datetime.datetime(2024, 11, 30, 6, 51, 19)},
		{'pos_id': 32088, 'symb': 'USDC', 'prod_id': 'ENS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 22, 37, 28), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 11, 49, 31), 'age_mins': 492, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.330543975000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015866025000'), 'fees_cnt_tot': Decimal('0.030828618516'), 'buy_cnt': Decimal('0.142263784323'), 'sell_cnt_tot': Decimal('0.141000000000'), 'hold_cnt': Decimal('0.001263784323'), 'pocket_cnt': Decimal('0.001263784323'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('42.175175000000'), 'prc_curr': Decimal('44.897475000000'), 'prc_high': Decimal('47.050000000000'), 'prc_low': Decimal('41.500000000000'), 'prc_chg_pct': Decimal('6.454700000000'), 'prc_chg_pct_high': Decimal('10.017300000000'), 'prc_chg_pct_low': Decimal('-1.600900000000'), 'prc_chg_pct_drop': Decimal('-3.560000000000'), 'prc_sell_avg': Decimal('44.897475000000'), 'val_curr': Decimal('0.056740725047'), 'val_tot': Decimal('0.056740725047'), 'gain_loss_amt_est': Decimal('0.417519310811'), 'gain_loss_amt_est_high': Decimal('0.601039592587'), 'gain_loss_amt_est_low': Decimal('-0.096052950596'), 'gain_loss_amt': Decimal('0.387284700047'), 'gain_loss_amt_net': Decimal('0.444025425095'), 'gain_loss_pct_est': Decimal('6.958700000000'), 'gain_loss_pct_est_high': Decimal('10.017300000000'), 'gain_loss_pct_est_low': Decimal('-1.600900000000'), 'gain_loss_pct': Decimal('6.454700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31742, 'bo_uuid': '12ad8493-1d47-4187-9e0b-f201cfae4d60', 'buy_curr_symb': 'ENS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ENS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ENS', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('775300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 22, 38, 19), 'check_last_dttm': datetime.datetime(2024, 11, 30, 6, 49, 28), 'add_dttm': datetime.datetime(2024, 11, 29, 22, 38, 19), 'upd_dttm': datetime.datetime(2024, 11, 29, 22, 38, 19), 'del_dttm': datetime.datetime(2024, 11, 29, 22, 38, 19), 'dlm': datetime.datetime(2024, 11, 30, 6, 49, 51)},
		{'pos_id': 32084, 'symb': 'USDC', 'prod_id': 'ENS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 22, 21, 54), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 11, 49, 15), 'age_mins': 508, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.147711000000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007889000000'), 'fees_cnt_tot': Decimal('0.015370296758'), 'buy_cnt': Decimal('0.070594921048'), 'sell_cnt_tot': Decimal('0.070000000000'), 'hold_cnt': Decimal('0.000594921048'), 'pocket_cnt': Decimal('0.000594921048'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('42.495975000000'), 'prc_curr': Decimal('44.967300000000'), 'prc_high': Decimal('47.050000000000'), 'prc_low': Decimal('41.500000000000'), 'prc_chg_pct': Decimal('5.815400000000'), 'prc_chg_pct_high': Decimal('9.186800000000'), 'prc_chg_pct_low': Decimal('-2.343700000000'), 'prc_chg_pct_drop': Decimal('-3.370000000000'), 'prc_sell_avg': Decimal('44.967300000000'), 'val_curr': Decimal('0.026751993242'), 'val_tot': Decimal('0.026751993242'), 'gain_loss_amt_est': Decimal('0.184536888475'), 'gain_loss_amt_est_high': Decimal('0.275604336627'), 'gain_loss_amt_est_low': Decimal('-0.070310776508'), 'gain_loss_amt': Decimal('0.174462993242'), 'gain_loss_amt_net': Decimal('0.201214986483'), 'gain_loss_pct_est': Decimal('6.151200000000'), 'gain_loss_pct_est_high': Decimal('9.186800000000'), 'gain_loss_pct_est_low': Decimal('-2.343700000000'), 'gain_loss_pct': Decimal('5.815400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'bb_bo', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31738, 'bo_uuid': 'd9374d4d-626d-4de7-9899-7caa81c62f68', 'buy_curr_symb': 'ENS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ENS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ENS', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('775300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 22, 22, 11), 'check_last_dttm': datetime.datetime(2024, 11, 30, 6, 49, 12), 'add_dttm': datetime.datetime(2024, 11, 29, 22, 22, 11), 'upd_dttm': datetime.datetime(2024, 11, 29, 22, 22, 11), 'del_dttm': datetime.datetime(2024, 11, 29, 22, 22, 11), 'dlm': datetime.datetime(2024, 11, 30, 6, 49, 44)},
		{'pos_id': 32150, 'symb': 'USDC', 'prod_id': 'ENS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 0, 53, 57), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 11, 31, 53), 'age_mins': 337, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.790493500000'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.052106500000'), 'fees_cnt_tot': Decimal('0.101981811721'), 'buy_cnt': Decimal('0.463202337782'), 'sell_cnt_tot': Decimal('0.460000000000'), 'hold_cnt': Decimal('0.003202337782'), 'pocket_cnt': Decimal('0.003202337782'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('43.177675000000'), 'prc_curr': Decimal('45.196725000000'), 'prc_high': Decimal('47.050000000000'), 'prc_low': Decimal('42.550000000000'), 'prc_chg_pct': Decimal('4.676100000000'), 'prc_chg_pct_high': Decimal('7.462900000000'), 'prc_chg_pct_low': Decimal('-1.453700000000'), 'prc_chg_pct_drop': Decimal('-2.790000000000'), 'prc_sell_avg': Decimal('45.196725000000'), 'val_curr': Decimal('0.144735180090'), 'val_tot': Decimal('0.144735180090'), 'gain_loss_amt_est': Decimal('1.205403023660'), 'gain_loss_amt_est_high': Decimal('1.492588473085'), 'gain_loss_amt_est_low': Decimal('-0.290740527376'), 'gain_loss_amt': Decimal('0.935228680090'), 'gain_loss_amt_net': Decimal('1.079963860180'), 'gain_loss_pct_est': Decimal('6.027000000000'), 'gain_loss_pct_est_high': Decimal('7.462900000000'), 'gain_loss_pct_est_low': Decimal('-1.453700000000'), 'gain_loss_pct': Decimal('4.676100000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31804, 'bo_uuid': '077683a5-019c-4ce5-838e-72535a55c1c2', 'buy_curr_symb': 'ENS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ENS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ENS', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('775300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 0, 54, 40), 'check_last_dttm': datetime.datetime(2024, 11, 30, 6, 31, 50), 'add_dttm': datetime.datetime(2024, 11, 30, 0, 54, 40), 'upd_dttm': datetime.datetime(2024, 11, 30, 0, 54, 40), 'del_dttm': datetime.datetime(2024, 11, 30, 0, 54, 40), 'dlm': datetime.datetime(2024, 11, 30, 6, 32, 36)},
		{'pos_id': 32149, 'symb': 'USDC', 'prod_id': 'ENS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 30, 0, 46, 26), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 11, 31, 39), 'age_mins': 344, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.214155675000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015574325000'), 'fees_cnt_tot': Decimal('0.030536918516'), 'buy_cnt': Decimal('0.138703068516'), 'sell_cnt_tot': Decimal('0.137000000000'), 'hold_cnt': Decimal('0.001703068516'), 'pocket_cnt': Decimal('0.001703068516'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('43.257875000000'), 'prc_curr': Decimal('45.358800550000'), 'prc_high': Decimal('47.050000000000'), 'prc_low': Decimal('42.550000000000'), 'prc_chg_pct': Decimal('4.856700000000'), 'prc_chg_pct_high': Decimal('7.263700000000'), 'prc_chg_pct_low': Decimal('-1.636400000000'), 'prc_chg_pct_drop': Decimal('-2.410000000000'), 'prc_sell_avg': Decimal('45.358800550000'), 'val_curr': Decimal('0.077249145140'), 'val_tot': Decimal('0.077249145140'), 'gain_loss_amt_est': Decimal('0.349826476662'), 'gain_loss_amt_est_high': Decimal('0.435822379142'), 'gain_loss_amt_est_low': Decimal('-0.098184434644'), 'gain_loss_amt': Decimal('0.291404820140'), 'gain_loss_amt_net': Decimal('0.368653965280'), 'gain_loss_pct_est': Decimal('5.830400000000'), 'gain_loss_pct_est_high': Decimal('7.263700000000'), 'gain_loss_pct_est_low': Decimal('-1.636400000000'), 'gain_loss_pct': Decimal('4.856700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31803, 'bo_uuid': '08dec8c4-f2f8-4697-b109-058cad40bad2', 'buy_curr_symb': 'ENS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ENS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ENS', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('775300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 30, 0, 46, 56), 'check_last_dttm': datetime.datetime(2024, 11, 30, 6, 31, 35), 'add_dttm': datetime.datetime(2024, 11, 30, 0, 46, 56), 'upd_dttm': datetime.datetime(2024, 11, 30, 0, 46, 56), 'del_dttm': datetime.datetime(2024, 11, 30, 0, 46, 56), 'dlm': datetime.datetime(2024, 11, 30, 6, 32, 28)},
		{'pos_id': 32115, 'symb': 'USDC', 'prod_id': 'ENS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 23, 8, 33), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 11, 31, 18), 'age_mins': 442, 'tot_out_cnt': Decimal('2.000000000000'), 'tot_in_cnt': Decimal('2.051358750000'), 'buy_fees_cnt': Decimal('0.004987531172'), 'sell_fees_cnt_tot': Decimal('0.005141250000'), 'fees_cnt_tot': Decimal('0.010128781172'), 'buy_cnt': Decimal('0.045372128015'), 'sell_cnt_tot': Decimal('0.045000000000'), 'hold_cnt': Decimal('0.000372128015'), 'pocket_cnt': Decimal('0.000372128015'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('44.079925000000'), 'prc_curr': Decimal('45.585750000000'), 'prc_high': Decimal('47.050000000000'), 'prc_low': Decimal('41.500000000000'), 'prc_chg_pct': Decimal('3.416100000000'), 'prc_chg_pct_high': Decimal('5.263300000000'), 'prc_chg_pct_low': Decimal('-5.852800000000'), 'prc_chg_pct_drop': Decimal('-1.850000000000'), 'prc_sell_avg': Decimal('45.585750000000'), 'val_curr': Decimal('0.016963734660'), 'val_tot': Decimal('0.016963734660'), 'gain_loss_amt_est': Decimal('0.077136020527'), 'gain_loss_amt_est_high': Decimal('0.105266739896'), 'gain_loss_amt_est_low': Decimal('-0.117056687378'), 'gain_loss_amt': Decimal('0.068322484660'), 'gain_loss_amt_net': Decimal('0.085286219320'), 'gain_loss_pct_est': Decimal('3.856800000000'), 'gain_loss_pct_est_high': Decimal('5.263300000000'), 'gain_loss_pct_est_low': Decimal('-5.852800000000'), 'gain_loss_pct': Decimal('3.416100000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'bb_bo', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31769, 'bo_uuid': '0348451b-6c78-444b-9ce7-fcf98845932f', 'buy_curr_symb': 'ENS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ENS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ENS', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('775300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 23, 9, 5), 'check_last_dttm': datetime.datetime(2024, 11, 30, 6, 31, 15), 'add_dttm': datetime.datetime(2024, 11, 29, 23, 9, 5), 'upd_dttm': datetime.datetime(2024, 11, 29, 23, 9, 5), 'del_dttm': datetime.datetime(2024, 11, 29, 23, 9, 5), 'dlm': datetime.datetime(2024, 11, 30, 6, 32, 21)},
		{'pos_id': 32109, 'symb': 'USDC', 'prod_id': 'ENS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 23, 0, 25), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 11, 31, 3), 'age_mins': 450, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.252080625000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015669375000'), 'fees_cnt_tot': Decimal('0.030631968516'), 'buy_cnt': Decimal('0.137999479052'), 'sell_cnt_tot': Decimal('0.137000000000'), 'hold_cnt': Decimal('0.000999479052'), 'pocket_cnt': Decimal('0.000999479052'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('43.478425000000'), 'prc_curr': Decimal('45.635625000000'), 'prc_high': Decimal('47.050000000000'), 'prc_low': Decimal('41.500000000000'), 'prc_chg_pct': Decimal('4.961500000000'), 'prc_chg_pct_high': Decimal('6.719600000000'), 'prc_chg_pct_low': Decimal('-4.550400000000'), 'prc_chg_pct_drop': Decimal('-1.760000000000'), 'prc_sell_avg': Decimal('45.635625000000'), 'val_curr': Decimal('0.045611851212'), 'val_tot': Decimal('0.045611851212'), 'gain_loss_amt_est': Decimal('0.317616151001'), 'gain_loss_amt_est_high': Decimal('0.403175828013'), 'gain_loss_amt_est_low': Decimal('-0.273021619342'), 'gain_loss_amt': Decimal('0.297692476212'), 'gain_loss_amt_net': Decimal('0.343304327425'), 'gain_loss_pct_est': Decimal('5.293600000000'), 'gain_loss_pct_est_high': Decimal('6.719600000000'), 'gain_loss_pct_est_low': Decimal('-4.550400000000'), 'gain_loss_pct': Decimal('4.961500000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31763, 'bo_uuid': '7d91d53f-e256-42d8-a886-a3f2a84dca22', 'buy_curr_symb': 'ENS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ENS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ENS', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('775300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 23, 0, 50), 'check_last_dttm': datetime.datetime(2024, 11, 30, 6, 30, 59), 'add_dttm': datetime.datetime(2024, 11, 29, 23, 0, 50), 'upd_dttm': datetime.datetime(2024, 11, 29, 23, 0, 50), 'del_dttm': datetime.datetime(2024, 11, 29, 23, 0, 50), 'dlm': datetime.datetime(2024, 11, 30, 6, 32, 13)},
		{'pos_id': 32085, 'symb': 'USDC', 'prod_id': 'ENS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 22, 29, 36), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 11, 30, 34), 'age_mins': 481, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.283950750000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015749250000'), 'fees_cnt_tot': Decimal('0.030711843516'), 'buy_cnt': Decimal('0.139837322581'), 'sell_cnt_tot': Decimal('0.138000000000'), 'hold_cnt': Decimal('0.001837322581'), 'pocket_cnt': Decimal('0.001837322581'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('42.907000000000'), 'prc_curr': Decimal('45.535875000000'), 'prc_high': Decimal('47.050000000000'), 'prc_low': Decimal('41.500000000000'), 'prc_chg_pct': Decimal('6.126900000000'), 'prc_chg_pct_high': Decimal('8.140900000000'), 'prc_chg_pct_low': Decimal('-3.279200000000'), 'prc_chg_pct_drop': Decimal('-2.010000000000'), 'prc_sell_avg': Decimal('45.535875000000'), 'val_curr': Decimal('0.083664091383'), 'val_tot': Decimal('0.083664091383'), 'gain_loss_amt_est': Decimal('0.401752627758'), 'gain_loss_amt_est_high': Decimal('0.488451767758'), 'gain_loss_amt_est_low': Decimal('-0.196751112888'), 'gain_loss_amt': Decimal('0.367614841383'), 'gain_loss_amt_net': Decimal('0.451278932766'), 'gain_loss_pct_est': Decimal('6.695900000000'), 'gain_loss_pct_est_high': Decimal('8.140900000000'), 'gain_loss_pct_est_low': Decimal('-3.279200000000'), 'gain_loss_pct': Decimal('6.126900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31739, 'bo_uuid': 'a10111b9-5dd5-4e5b-8e3f-bef11a387c4d', 'buy_curr_symb': 'ENS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ENS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ENS', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('775300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 22, 30, 8), 'check_last_dttm': datetime.datetime(2024, 11, 30, 6, 30, 31), 'add_dttm': datetime.datetime(2024, 11, 29, 22, 30, 8), 'upd_dttm': datetime.datetime(2024, 11, 29, 22, 30, 8), 'del_dttm': datetime.datetime(2024, 11, 29, 22, 30, 8), 'dlm': datetime.datetime(2024, 11, 30, 6, 30, 46)},
		{'pos_id': 31888, 'symb': 'USDC', 'prod_id': 'ADA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 15, 24, 25), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 8, 25, 25), 'age_mins': 721, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.970820797801'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.052558448115'), 'fees_cnt_tot': Decimal('0.102433759836'), 'buy_cnt': Decimal('18.780123023891'), 'sell_cnt_tot': Decimal('18.667536180000'), 'hold_cnt': Decimal('0.112586843891'), 'pocket_cnt': Decimal('0.112586843891'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.064955750000'), 'prc_curr': Decimal('1.123384500000'), 'prc_high': Decimal('1.148100000000'), 'prc_low': Decimal('1.046100000000'), 'prc_chg_pct': Decimal('5.486500000000'), 'prc_chg_pct_high': Decimal('7.525600000000'), 'prc_chg_pct_low': Decimal('-1.770600000000'), 'prc_chg_pct_drop': Decimal('-2.040000000000'), 'prc_sell_avg': Decimal('1.123384500000'), 'val_curr': Decimal('0.126478315331'), 'val_tot': Decimal('0.126478315331'), 'gain_loss_amt_est': Decimal('1.199002869368'), 'gain_loss_amt_est_high': Decimal('1.505118874658'), 'gain_loss_amt_est_low': Decimal('-0.354113304708'), 'gain_loss_amt': Decimal('1.097299113132'), 'gain_loss_amt_net': Decimal('1.223777428463'), 'gain_loss_pct_est': Decimal('5.995000000000'), 'gain_loss_pct_est_high': Decimal('7.525600000000'), 'gain_loss_pct_est_low': Decimal('-1.770600000000'), 'gain_loss_pct': Decimal('5.486500000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31542, 'bo_uuid': '618d808f-cfc8-47d1-b5be-024127fb2dba', 'buy_curr_symb': 'ADA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ADA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ADA', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('30176000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 15, 24, 46), 'check_last_dttm': datetime.datetime(2024, 11, 30, 3, 25, 22), 'add_dttm': datetime.datetime(2024, 11, 29, 15, 24, 46), 'upd_dttm': datetime.datetime(2024, 11, 29, 15, 24, 46), 'del_dttm': datetime.datetime(2024, 11, 29, 15, 24, 46), 'dlm': datetime.datetime(2024, 11, 30, 3, 26, 28)},
		{'pos_id': 31733, 'symb': 'USDC', 'prod_id': 'ADA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 10, 6, 20), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 8, 25, 4), 'age_mins': 1039, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.228318332445'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015609820382'), 'fees_cnt_tot': Decimal('0.030572413898'), 'buy_cnt': Decimal('5.555074630113'), 'sell_cnt_tot': Decimal('5.530024020000'), 'hold_cnt': Decimal('0.025050610113'), 'pocket_cnt': Decimal('0.025050610113'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.080093500000'), 'prc_curr': Decimal('1.126273290000'), 'prc_high': Decimal('1.148100000000'), 'prc_low': Decimal('1.039800000000'), 'prc_chg_pct': Decimal('4.275500000000'), 'prc_chg_pct_high': Decimal('6.018600000000'), 'prc_chg_pct_low': Decimal('-3.730600000000'), 'prc_chg_pct_drop': Decimal('-1.740000000000'), 'prc_sell_avg': Decimal('1.126273290000'), 'val_curr': Decimal('0.028213833068'), 'val_tot': Decimal('0.028213833068'), 'gain_loss_amt_est': Decimal('0.270568242472'), 'gain_loss_amt_est_high': Decimal('0.361115958942'), 'gain_loss_amt_est_low': Decimal('-0.223833399609'), 'gain_loss_amt': Decimal('0.256532165513'), 'gain_loss_amt_net': Decimal('0.284745998582'), 'gain_loss_pct_est': Decimal('4.509500000000'), 'gain_loss_pct_est_high': Decimal('6.018600000000'), 'gain_loss_pct_est_low': Decimal('-3.730600000000'), 'gain_loss_pct': Decimal('4.275500000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31387, 'bo_uuid': '7f034a19-f684-4c68-b63a-ae8ebaa0be3e', 'buy_curr_symb': 'ADA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ADA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ADA', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('30176000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 10, 6, 41), 'check_last_dttm': datetime.datetime(2024, 11, 30, 3, 25, 1), 'add_dttm': datetime.datetime(2024, 11, 29, 10, 6, 41), 'upd_dttm': datetime.datetime(2024, 11, 29, 10, 6, 41), 'del_dttm': datetime.datetime(2024, 11, 29, 10, 6, 41), 'dlm': datetime.datetime(2024, 11, 30, 3, 26, 21)},
		{'pos_id': 31962, 'symb': 'USDC', 'prod_id': 'ADA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 16, 53, 12), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 7, 1, 29), 'age_mins': 548, 'tot_out_cnt': Decimal('25.000000000000'), 'tot_in_cnt': Decimal('25.677802269712'), 'buy_fees_cnt': Decimal('0.062344139651'), 'sell_fees_cnt_tot': Decimal('0.064355394160'), 'fees_cnt_tot': Decimal('0.126699533811'), 'buy_cnt': Decimal('23.581707669361'), 'sell_cnt_tot': Decimal('23.485489580000'), 'hold_cnt': Decimal('0.096218089361'), 'pocket_cnt': Decimal('0.096218089361'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.060143750000'), 'prc_curr': Decimal('1.093347540000'), 'prc_high': Decimal('1.121800000000'), 'prc_low': Decimal('1.053000000000'), 'prc_chg_pct': Decimal('3.132000000000'), 'prc_chg_pct_high': Decimal('5.174400000000'), 'prc_chg_pct_low': Decimal('-0.673800000000'), 'prc_chg_pct_drop': Decimal('-2.040000000000'), 'prc_sell_avg': Decimal('1.093347540000'), 'val_curr': Decimal('0.105199811306'), 'val_tot': Decimal('0.105199811306'), 'gain_loss_amt_est': Decimal('1.020056242373'), 'gain_loss_amt_est_high': Decimal('1.293604051338'), 'gain_loss_amt_est_low': Decimal('-0.168461824163'), 'gain_loss_amt': Decimal('0.783002081018'), 'gain_loss_amt_net': Decimal('0.888201892325'), 'gain_loss_pct_est': Decimal('4.080200000000'), 'gain_loss_pct_est_high': Decimal('5.174400000000'), 'gain_loss_pct_est_low': Decimal('-0.673800000000'), 'gain_loss_pct': Decimal('3.132000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31616, 'bo_uuid': 'b4fd970b-2b82-4ca1-9260-ecf52300843b', 'buy_curr_symb': 'ADA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ADA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ADA', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('30176000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 16, 54, 9), 'check_last_dttm': datetime.datetime(2024, 11, 30, 2, 1, 26), 'add_dttm': datetime.datetime(2024, 11, 29, 16, 54, 9), 'upd_dttm': datetime.datetime(2024, 11, 29, 16, 54, 9), 'del_dttm': datetime.datetime(2024, 11, 29, 16, 54, 9), 'dlm': datetime.datetime(2024, 11, 30, 2, 2, 24)},
		{'pos_id': 32050, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 21, 2, 14), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 7, 0, 35), 'age_mins': 298, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.187452197409'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015507398991'), 'fees_cnt_tot': Decimal('0.030469992507'), 'buy_cnt': Decimal('3.265159523450'), 'sell_cnt_tot': Decimal('3.253073000000'), 'hold_cnt': Decimal('0.012086523450'), 'pocket_cnt': Decimal('0.012086523450'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.837582500000'), 'prc_curr': Decimal('1.902033000000'), 'prc_high': Decimal('1.939900000000'), 'prc_low': Decimal('1.781500000000'), 'prc_chg_pct': Decimal('3.507400000000'), 'prc_chg_pct_high': Decimal('5.415700000000'), 'prc_chg_pct_low': Decimal('-3.052000000000'), 'prc_chg_pct_drop': Decimal('-1.910000000000'), 'prc_sell_avg': Decimal('1.902033000000'), 'val_curr': Decimal('0.022988966457'), 'val_tot': Decimal('0.022988966457'), 'gain_loss_amt_est': Decimal('0.222087987886'), 'gain_loss_amt_est_high': Decimal('0.324940512875'), 'gain_loss_amt_est_low': Decimal('-0.183118308974'), 'gain_loss_amt': Decimal('0.210441163866'), 'gain_loss_amt_net': Decimal('0.233430130323'), 'gain_loss_pct_est': Decimal('3.701500000000'), 'gain_loss_pct_est_high': Decimal('5.415700000000'), 'gain_loss_pct_est_low': Decimal('-3.052000000000'), 'gain_loss_pct': Decimal('3.507400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31704, 'bo_uuid': '54d57c06-6999-42f1-87fc-7a2df593a37c', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 21, 2, 26), 'check_last_dttm': datetime.datetime(2024, 11, 30, 2, 0, 32), 'add_dttm': datetime.datetime(2024, 11, 29, 21, 2, 26), 'upd_dttm': datetime.datetime(2024, 11, 29, 21, 2, 26), 'del_dttm': datetime.datetime(2024, 11, 29, 21, 2, 26), 'dlm': datetime.datetime(2024, 11, 30, 2, 2, 12)},
		{'pos_id': 32032, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 20, 33, 48), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 7, 0, 17), 'age_mins': 326, 'tot_out_cnt': Decimal('25.000000000000'), 'tot_in_cnt': Decimal('25.888668969272'), 'buy_fees_cnt': Decimal('0.062344139651'), 'sell_fees_cnt_tot': Decimal('0.064883882129'), 'fees_cnt_tot': Decimal('0.127228021780'), 'buy_cnt': Decimal('13.691476809240'), 'sell_cnt_tot': Decimal('13.631755000000'), 'hold_cnt': Decimal('0.059721809240'), 'pocket_cnt': Decimal('0.059721809240'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.825953500000'), 'prc_curr': Decimal('1.899144240000'), 'prc_high': Decimal('1.939900000000'), 'prc_low': Decimal('1.781500000000'), 'prc_chg_pct': Decimal('4.008400000000'), 'prc_chg_pct_high': Decimal('6.087000000000'), 'prc_chg_pct_low': Decimal('-2.434500000000'), 'prc_chg_pct_drop': Decimal('-2.080000000000'), 'prc_sell_avg': Decimal('1.899144240000'), 'val_curr': Decimal('0.113420330021'), 'val_tot': Decimal('0.113420330021'), 'gain_loss_amt_est': Decimal('1.090478207688'), 'gain_loss_amt_est_high': Decimal('1.521759727179'), 'gain_loss_amt_est_low': Decimal('-0.608634064339'), 'gain_loss_amt': Decimal('1.002089299293'), 'gain_loss_amt_net': Decimal('1.115509629313'), 'gain_loss_pct_est': Decimal('4.361900000000'), 'gain_loss_pct_est_high': Decimal('6.087000000000'), 'gain_loss_pct_est_low': Decimal('-2.434500000000'), 'gain_loss_pct': Decimal('4.008400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31686, 'bo_uuid': '46ef4a46-70ca-48ca-aa5b-f423cd2e2b04', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 20, 35, 2), 'check_last_dttm': datetime.datetime(2024, 11, 30, 2, 0, 14), 'add_dttm': datetime.datetime(2024, 11, 29, 20, 35, 2), 'upd_dttm': datetime.datetime(2024, 11, 29, 20, 35, 2), 'del_dttm': datetime.datetime(2024, 11, 29, 20, 35, 2), 'dlm': datetime.datetime(2024, 11, 30, 2, 0, 46)},
		{'pos_id': 32028, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 20, 26, 43), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 7, 0, 1), 'age_mins': 333, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.710905871241'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.051907032259'), 'fees_cnt_tot': Decimal('0.101782343980'), 'buy_cnt': Decimal('10.990593151322'), 'sell_cnt_tot': Decimal('10.938735000000'), 'hold_cnt': Decimal('0.051858151322'), 'pocket_cnt': Decimal('0.051858151322'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.819738000000'), 'prc_curr': Decimal('1.893354750000'), 'prc_high': Decimal('1.939900000000'), 'prc_low': Decimal('1.781500000000'), 'prc_chg_pct': Decimal('4.045500000000'), 'prc_chg_pct_high': Decimal('6.449400000000'), 'prc_chg_pct_low': Decimal('-2.101300000000'), 'prc_chg_pct_drop': Decimal('-2.400000000000'), 'prc_sell_avg': Decimal('1.893354750000'), 'val_curr': Decimal('0.098185877132'), 'val_tot': Decimal('0.098185877132'), 'gain_loss_amt_est': Decimal('0.943674309159'), 'gain_loss_amt_est_high': Decimal('1.289877993426'), 'gain_loss_amt_est_low': Decimal('-0.420258300920'), 'gain_loss_amt': Decimal('0.809091748373'), 'gain_loss_amt_net': Decimal('0.907277625504'), 'gain_loss_pct_est': Decimal('4.718400000000'), 'gain_loss_pct_est_high': Decimal('6.449400000000'), 'gain_loss_pct_est_low': Decimal('-2.101300000000'), 'gain_loss_pct': Decimal('4.045500000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31682, 'bo_uuid': '0f863baf-ebd0-40fd-bf70-ab89c0b5c5ca', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 20, 27, 32), 'check_last_dttm': datetime.datetime(2024, 11, 30, 1, 59, 57), 'add_dttm': datetime.datetime(2024, 11, 29, 20, 27, 32), 'upd_dttm': datetime.datetime(2024, 11, 29, 20, 27, 32), 'del_dttm': datetime.datetime(2024, 11, 29, 20, 27, 32), 'dlm': datetime.datetime(2024, 11, 30, 2, 0, 40)},
		{'pos_id': 32026, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 20, 19, 19), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 6, 59, 43), 'age_mins': 341, 'tot_out_cnt': Decimal('25.000000000000'), 'tot_in_cnt': Decimal('26.219646554115'), 'buy_fees_cnt': Decimal('0.062344139651'), 'sell_fees_cnt_tot': Decimal('0.065713399885'), 'fees_cnt_tot': Decimal('0.128057539536'), 'buy_cnt': Decimal('13.918432695401'), 'sell_cnt_tot': Decimal('13.833642000000'), 'hold_cnt': Decimal('0.084790695401'), 'pocket_cnt': Decimal('0.084790695401'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.796179250000'), 'prc_curr': Decimal('1.895353840000'), 'prc_high': Decimal('1.939900000000'), 'prc_low': Decimal('1.781500000000'), 'prc_chg_pct': Decimal('5.521400000000'), 'prc_chg_pct_high': Decimal('7.845600000000'), 'prc_chg_pct_low': Decimal('-0.817200000000'), 'prc_chg_pct_drop': Decimal('-2.320000000000'), 'prc_sell_avg': Decimal('1.895353840000'), 'val_curr': Decimal('0.160708370125'), 'val_tot': Decimal('0.160708370125'), 'gain_loss_amt_est': Decimal('1.522965344356'), 'gain_loss_amt_est_high': Decimal('1.961395974261'), 'gain_loss_amt_est_low': Decimal('-0.204312153143'), 'gain_loss_amt': Decimal('1.380354924240'), 'gain_loss_amt_net': Decimal('1.541063294364'), 'gain_loss_pct_est': Decimal('6.091900000000'), 'gain_loss_pct_est_high': Decimal('7.845600000000'), 'gain_loss_pct_est_low': Decimal('-0.817200000000'), 'gain_loss_pct': Decimal('5.521400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31680, 'bo_uuid': '807da8c8-f5a7-4c51-9556-2ba1fccf052b', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 20, 19, 42), 'check_last_dttm': datetime.datetime(2024, 11, 30, 1, 59, 40), 'add_dttm': datetime.datetime(2024, 11, 29, 20, 19, 42), 'upd_dttm': datetime.datetime(2024, 11, 29, 20, 19, 42), 'del_dttm': datetime.datetime(2024, 11, 29, 20, 19, 42), 'dlm': datetime.datetime(2024, 11, 30, 2, 0, 33)},
		{'pos_id': 31981, 'symb': 'USDC', 'prod_id': 'FLR-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 17, 34, 2), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 6, 49, 39), 'age_mins': 496, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.120289725000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007820275000'), 'fees_cnt_tot': Decimal('0.015301571758'), 'buy_cnt': Decimal('103.619068671811'), 'sell_cnt_tot': Decimal('103.000000000000'), 'hold_cnt': Decimal('0.619068671811'), 'pocket_cnt': Decimal('0.619068671811'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.028952200000'), 'prc_curr': Decimal('0.030294080000'), 'prc_high': Decimal('0.032400000000'), 'prc_low': Decimal('0.027100000000'), 'prc_chg_pct': Decimal('4.634800000000'), 'prc_chg_pct_high': Decimal('6.796700000000'), 'prc_chg_pct_low': Decimal('-6.397400000000'), 'prc_chg_pct_drop': Decimal('-2.160000000000'), 'prc_sell_avg': Decimal('0.030294080000'), 'val_curr': Decimal('0.018754115869'), 'val_tot': Decimal('0.018754115869'), 'gain_loss_amt_est': Decimal('0.151055878310'), 'gain_loss_amt_est_high': Decimal('0.203901603332'), 'gain_loss_amt_est_low': Decimal('-0.191923238994'), 'gain_loss_amt': Decimal('0.139043840869'), 'gain_loss_amt_net': Decimal('0.157797956739'), 'gain_loss_pct_est': Decimal('5.035200000000'), 'gain_loss_pct_est_high': Decimal('6.796700000000'), 'gain_loss_pct_est_low': Decimal('-6.397400000000'), 'gain_loss_pct': Decimal('4.634800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31635, 'bo_uuid': '58d30f43-955a-4d91-8089-85f7e96650b1', 'buy_curr_symb': 'FLR', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'FLR', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'FLR', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('259257334.645080500000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 17, 35, 9), 'check_last_dttm': datetime.datetime(2024, 11, 30, 1, 49, 36), 'add_dttm': datetime.datetime(2024, 11, 29, 17, 35, 9), 'upd_dttm': datetime.datetime(2024, 11, 29, 17, 35, 9), 'del_dttm': datetime.datetime(2024, 11, 29, 17, 35, 9), 'dlm': datetime.datetime(2024, 11, 30, 1, 50, 34)},
		{'pos_id': 32057, 'symb': 'USDC', 'prod_id': 'ENS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 21, 16, 48), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 4, 52, 24), 'age_mins': 155, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('7.037641800000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.017638200000'), 'fees_cnt_tot': Decimal('0.032600793516'), 'buy_cnt': Decimal('0.168119028272'), 'sell_cnt_tot': Decimal('0.164000000000'), 'hold_cnt': Decimal('0.004119028272'), 'pocket_cnt': Decimal('0.004119028272'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('35.689000000000'), 'prc_curr': Decimal('42.912450000000'), 'prc_high': Decimal('45.100000000000'), 'prc_low': Decimal('35.530000000000'), 'prc_chg_pct': Decimal('20.240000000000'), 'prc_chg_pct_high': Decimal('22.811000000000'), 'prc_chg_pct_low': Decimal('-0.445500000000'), 'prc_chg_pct_drop': Decimal('-2.570000000000'), 'prc_sell_avg': Decimal('42.912450000000'), 'val_curr': Decimal('0.176757594771'), 'val_tot': Decimal('0.176757594771'), 'gain_loss_amt_est': Decimal('1.210625122586'), 'gain_loss_amt_est_high': Decimal('1.368657009162'), 'gain_loss_amt_est_low': Decimal('-0.026730925496'), 'gain_loss_amt': Decimal('1.214399394771'), 'gain_loss_amt_net': Decimal('1.391156989542'), 'gain_loss_pct_est': Decimal('20.177100000000'), 'gain_loss_pct_est_high': Decimal('22.811000000000'), 'gain_loss_pct_est_low': Decimal('-0.445500000000'), 'gain_loss_pct': Decimal('20.240000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31711, 'bo_uuid': '6be6484b-b61a-4e9c-88a2-239c30949d0c', 'buy_curr_symb': 'ENS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ENS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ENS', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('775300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 21, 17, 49), 'check_last_dttm': datetime.datetime(2024, 11, 29, 23, 52, 21), 'add_dttm': datetime.datetime(2024, 11, 29, 21, 17, 49), 'upd_dttm': datetime.datetime(2024, 11, 29, 21, 17, 49), 'del_dttm': datetime.datetime(2024, 11, 29, 21, 17, 49), 'dlm': datetime.datetime(2024, 11, 29, 23, 53, 33)},
		{'pos_id': 32052, 'symb': 'USDC', 'prod_id': 'ENS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 21, 10, 47), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 4, 52, 9), 'age_mins': 161, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('7.070679000000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.017721000000'), 'fees_cnt_tot': Decimal('0.032683593516'), 'buy_cnt': Decimal('0.169260107649'), 'sell_cnt_tot': Decimal('0.165000000000'), 'hold_cnt': Decimal('0.004260107649'), 'pocket_cnt': Decimal('0.004260107649'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('35.448400000000'), 'prc_curr': Decimal('42.852600000000'), 'prc_high': Decimal('45.100000000000'), 'prc_low': Decimal('35.360000000000'), 'prc_chg_pct': Decimal('20.887300000000'), 'prc_chg_pct_high': Decimal('23.644500000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-2.760000000000'), 'prc_sell_avg': Decimal('42.852600000000'), 'val_curr': Decimal('0.182556689040'), 'val_tot': Decimal('0.182556689040'), 'gain_loss_amt_est': Decimal('1.259566017066'), 'gain_loss_amt_est_high': Decimal('1.418670518256'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('1.253235689040'), 'gain_loss_amt_net': Decimal('1.435792378079'), 'gain_loss_pct_est': Decimal('20.992800000000'), 'gain_loss_pct_est_high': Decimal('23.644500000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('20.887300000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31706, 'bo_uuid': '7b35f91b-01f8-43eb-88f8-0f1d3f9ffdd2', 'buy_curr_symb': 'ENS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ENS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ENS', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('775300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 21, 11, 7), 'check_last_dttm': datetime.datetime(2024, 11, 29, 23, 52, 6), 'add_dttm': datetime.datetime(2024, 11, 29, 21, 11, 7), 'upd_dttm': datetime.datetime(2024, 11, 29, 21, 11, 7), 'del_dttm': datetime.datetime(2024, 11, 29, 21, 11, 7), 'dlm': datetime.datetime(2024, 11, 29, 23, 53, 24)},
		{'pos_id': 32046, 'symb': 'USDC', 'prod_id': 'ENS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 20, 49, 4), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 4, 51, 36), 'age_mins': 183, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('7.101940650000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.017799350000'), 'fees_cnt_tot': Decimal('0.032761943516'), 'buy_cnt': Decimal('0.170174506866'), 'sell_cnt_tot': Decimal('0.166000000000'), 'hold_cnt': Decimal('0.004174506866'), 'pocket_cnt': Decimal('0.004174506866'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('35.257925000000'), 'prc_curr': Decimal('42.782775000000'), 'prc_high': Decimal('45.100000000000'), 'prc_low': Decimal('35.050000000000'), 'prc_chg_pct': Decimal('21.342300000000'), 'prc_chg_pct_high': Decimal('24.312500000000'), 'prc_chg_pct_low': Decimal('-0.589700000000'), 'prc_chg_pct_drop': Decimal('-2.970000000000'), 'prc_sell_avg': Decimal('42.782775000000'), 'val_curr': Decimal('0.178596987984'), 'val_tot': Decimal('0.178596987984'), 'gain_loss_amt_est': Decimal('1.298784599483'), 'gain_loss_amt_est_high': Decimal('1.458748635937'), 'gain_loss_amt_est_low': Decimal('-0.035383534347'), 'gain_loss_amt': Decimal('1.280537637984'), 'gain_loss_amt_net': Decimal('1.459134625968'), 'gain_loss_pct_est': Decimal('21.646400000000'), 'gain_loss_pct_est_high': Decimal('24.312500000000'), 'gain_loss_pct_est_low': Decimal('-0.589700000000'), 'gain_loss_pct': Decimal('21.342300000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31700, 'bo_uuid': '9d5cb901-4292-42df-bbb7-526c130fdb71', 'buy_curr_symb': 'ENS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ENS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ENS', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('775300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 20, 49, 46), 'check_last_dttm': datetime.datetime(2024, 11, 29, 23, 51, 33), 'add_dttm': datetime.datetime(2024, 11, 29, 20, 49, 46), 'upd_dttm': datetime.datetime(2024, 11, 29, 20, 49, 46), 'del_dttm': datetime.datetime(2024, 11, 29, 20, 49, 46), 'dlm': datetime.datetime(2024, 11, 29, 23, 51, 55)},
		{'pos_id': 30838, 'symb': 'USDC', 'prod_id': 'ENS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 7, 10, 17), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 4, 51, 20), 'age_mins': 2441, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.686561700000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.016758300000'), 'fees_cnt_tot': Decimal('0.031720893516'), 'buy_cnt': Decimal('0.158418142046'), 'sell_cnt_tot': Decimal('0.156000000000'), 'hold_cnt': Decimal('0.002418142046'), 'pocket_cnt': Decimal('0.002418142046'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('37.874450000000'), 'prc_curr': Decimal('42.862575000000'), 'prc_high': Decimal('45.100000000000'), 'prc_low': Decimal('32.410000000000'), 'prc_chg_pct': Decimal('13.170200000000'), 'prc_chg_pct_high': Decimal('15.724500000000'), 'prc_chg_pct_low': Decimal('-14.427800000000'), 'prc_chg_pct_drop': Decimal('-2.550000000000'), 'prc_sell_avg': Decimal('42.862575000000'), 'val_curr': Decimal('0.103647794807'), 'val_tot': Decimal('0.103647794807'), 'gain_loss_amt_est': Decimal('0.794554112353'), 'gain_loss_amt_est_high': Decimal('0.943467165876'), 'gain_loss_amt_est_low': Decimal('-0.865668016289'), 'gain_loss_amt': Decimal('0.790209494807'), 'gain_loss_amt_net': Decimal('0.893857289615'), 'gain_loss_pct_est': Decimal('13.242600000000'), 'gain_loss_pct_est_high': Decimal('15.724500000000'), 'gain_loss_pct_est_low': Decimal('-14.427800000000'), 'gain_loss_pct': Decimal('13.170200000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30492, 'bo_uuid': '64f323e5-8289-4c49-99dc-0c325877c4b9', 'buy_curr_symb': 'ENS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ENS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ENS', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('775300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 7, 11, 9), 'check_last_dttm': datetime.datetime(2024, 11, 29, 23, 51, 17), 'add_dttm': datetime.datetime(2024, 11, 28, 7, 11, 9), 'upd_dttm': datetime.datetime(2024, 11, 28, 7, 11, 9), 'del_dttm': datetime.datetime(2024, 11, 28, 7, 11, 9), 'dlm': datetime.datetime(2024, 11, 29, 23, 51, 48)},
		{'pos_id': 32081, 'symb': 'USDC', 'prod_id': 'ENS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 22, 14, 50), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 4, 38, 11), 'age_mins': 83, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.105496800000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007783200000'), 'fees_cnt_tot': Decimal('0.015264496758'), 'buy_cnt': Decimal('0.072846122280'), 'sell_cnt_tot': Decimal('0.072000000000'), 'hold_cnt': Decimal('0.000846122280'), 'pocket_cnt': Decimal('0.000846122280'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('41.182700000000'), 'prc_curr': Decimal('43.131900000000'), 'prc_high': Decimal('45.100000000000'), 'prc_low': Decimal('41.080000000000'), 'prc_chg_pct': Decimal('4.733100000000'), 'prc_chg_pct_high': Decimal('6.428200000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-1.700000000000'), 'prc_sell_avg': Decimal('43.131900000000'), 'val_curr': Decimal('0.036494861569'), 'val_tot': Decimal('0.036494861569'), 'gain_loss_amt_est': Decimal('0.135297102931'), 'gain_loss_amt_est_high': Decimal('0.192845539532'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.141991661569'), 'gain_loss_amt_net': Decimal('0.178486523137'), 'gain_loss_pct_est': Decimal('4.509900000000'), 'gain_loss_pct_est_high': Decimal('6.428200000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('4.733100000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'bb_bo', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31735, 'bo_uuid': 'd19c6c81-9588-4d42-955a-2228fc9815b8', 'buy_curr_symb': 'ENS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ENS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ENS', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('775300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 22, 15, 33), 'check_last_dttm': datetime.datetime(2024, 11, 29, 23, 38, 8), 'add_dttm': datetime.datetime(2024, 11, 29, 22, 15, 33), 'upd_dttm': datetime.datetime(2024, 11, 29, 22, 15, 33), 'del_dttm': datetime.datetime(2024, 11, 29, 22, 15, 33), 'dlm': datetime.datetime(2024, 11, 29, 23, 38, 52)},
		{'pos_id': 30862, 'symb': 'USDC', 'prod_id': 'ENS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 7, 33, 4), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 4, 37, 31), 'age_mins': 2405, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.324996675000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.008333325000'), 'fees_cnt_tot': Decimal('0.015814621758'), 'buy_cnt': Decimal('0.077930174564'), 'sell_cnt_tot': Decimal('0.077000000000'), 'hold_cnt': Decimal('0.000930174564'), 'pocket_cnt': Decimal('0.000930174564'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('38.496000000000'), 'prc_curr': Decimal('43.181775000000'), 'prc_high': Decimal('45.100000000000'), 'prc_low': Decimal('32.410000000000'), 'prc_chg_pct': Decimal('12.172100000000'), 'prc_chg_pct_high': Decimal('13.856000000000'), 'prc_chg_pct_low': Decimal('-15.809400000000'), 'prc_chg_pct_drop': Decimal('-1.680000000000'), 'prc_sell_avg': Decimal('43.181775000000'), 'val_curr': Decimal('0.040166588733'), 'val_tot': Decimal('0.040166588733'), 'gain_loss_amt_est': Decimal('0.354114713235'), 'gain_loss_amt_est_high': Decimal('0.415679551140'), 'gain_loss_amt_est_low': Decimal('-0.474283042381'), 'gain_loss_amt': Decimal('0.365163263733'), 'gain_loss_amt_net': Decimal('0.405329852467'), 'gain_loss_pct_est': Decimal('11.803800000000'), 'gain_loss_pct_est_high': Decimal('13.856000000000'), 'gain_loss_pct_est_low': Decimal('-15.809400000000'), 'gain_loss_pct': Decimal('12.172100000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30516, 'bo_uuid': '49a124fa-a8a8-46a0-ac76-cd02717a0cf9', 'buy_curr_symb': 'ENS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ENS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ENS', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('775300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 7, 34, 13), 'check_last_dttm': datetime.datetime(2024, 11, 29, 23, 37, 28), 'add_dttm': datetime.datetime(2024, 11, 28, 7, 34, 13), 'upd_dttm': datetime.datetime(2024, 11, 28, 7, 34, 13), 'del_dttm': datetime.datetime(2024, 11, 28, 7, 34, 13), 'dlm': datetime.datetime(2024, 11, 29, 23, 38, 45)},
		{'pos_id': 32051, 'symb': 'USDC', 'prod_id': 'PNG-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 21, 8, 56), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 3, 27, 34), 'age_mins': 79, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.163203375000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015446625000'), 'fees_cnt_tot': Decimal('0.030409218516'), 'buy_cnt': Decimal('15.347430331779'), 'sell_cnt_tot': Decimal('15.000000000000'), 'hold_cnt': Decimal('0.347430331779'), 'pocket_cnt': Decimal('0.347430331779'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.390944920000'), 'prc_curr': Decimal('0.410880220000'), 'prc_high': Decimal('0.426150000000'), 'prc_low': Decimal('0.389970000000'), 'prc_chg_pct': Decimal('5.099300000000'), 'prc_chg_pct_high': Decimal('8.237800000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-3.140000000000'), 'prc_sell_avg': Decimal('0.410880220000'), 'val_curr': Decimal('0.142752251156'), 'val_tot': Decimal('0.142752251156'), 'gain_loss_amt_est': Decimal('0.290911692996'), 'gain_loss_amt_est_high': Decimal('0.494265144892'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.305955626156'), 'gain_loss_amt_net': Decimal('0.448707877312'), 'gain_loss_pct_est': Decimal('4.848500000000'), 'gain_loss_pct_est_high': Decimal('8.237800000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('5.099300000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31705, 'bo_uuid': '0ff48f72-7a83-41e0-b252-08a245b47176', 'buy_curr_symb': 'PNG', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PNG', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PNG', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('116620741.418695990000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 21, 9, 47), 'check_last_dttm': datetime.datetime(2024, 11, 29, 22, 27, 31), 'add_dttm': datetime.datetime(2024, 11, 29, 21, 9, 47), 'upd_dttm': datetime.datetime(2024, 11, 29, 21, 9, 47), 'del_dttm': datetime.datetime(2024, 11, 29, 21, 9, 47), 'dlm': datetime.datetime(2024, 11, 29, 22, 28, 31)},
		{'pos_id': 32031, 'symb': 'USDC', 'prod_id': 'PNG-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 20, 32, 6), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 2, 33, 54), 'age_mins': 61, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.224400000000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015600000000'), 'fees_cnt_tot': Decimal('0.030562593516'), 'buy_cnt': Decimal('16.286259235581'), 'sell_cnt_tot': Decimal('16.000000000000'), 'hold_cnt': Decimal('0.286259235581'), 'pocket_cnt': Decimal('0.286259235581'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.368408720000'), 'prc_curr': Decimal('0.389025000000'), 'prc_high': Decimal('0.410000000000'), 'prc_low': Decimal('0.363050000000'), 'prc_chg_pct': Decimal('5.596000000000'), 'prc_chg_pct_high': Decimal('10.749300000000'), 'prc_chg_pct_low': Decimal('-1.454600000000'), 'prc_chg_pct_drop': Decimal('-5.150000000000'), 'prc_sell_avg': Decimal('0.389025000000'), 'val_curr': Decimal('0.111361999122'), 'val_tot': Decimal('0.111361999122'), 'gain_loss_amt_est': Decimal('0.391379574411'), 'gain_loss_amt_est_high': Decimal('0.644956630709'), 'gain_loss_amt_est_low': Decimal('-0.087273584522'), 'gain_loss_amt': Decimal('0.335761999122'), 'gain_loss_amt_net': Decimal('0.447123998244'), 'gain_loss_pct_est': Decimal('6.523000000000'), 'gain_loss_pct_est_high': Decimal('10.749300000000'), 'gain_loss_pct_est_low': Decimal('-1.454600000000'), 'gain_loss_pct': Decimal('5.596000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31685, 'bo_uuid': '4eb2dc88-cc27-4588-917a-0bff177673a8', 'buy_curr_symb': 'PNG', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PNG', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PNG', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('116620741.418695990000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 20, 32, 33), 'check_last_dttm': datetime.datetime(2024, 11, 29, 21, 33, 52), 'add_dttm': datetime.datetime(2024, 11, 29, 20, 32, 33), 'upd_dttm': datetime.datetime(2024, 11, 29, 20, 32, 33), 'del_dttm': datetime.datetime(2024, 11, 29, 20, 32, 33), 'dlm': datetime.datetime(2024, 11, 29, 21, 35, 8)},
		{'pos_id': 32027, 'symb': 'USDC', 'prod_id': 'PNG-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 20, 24, 59), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 2, 33, 37), 'age_mins': 68, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.202215600000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015544400000'), 'fees_cnt_tot': Decimal('0.030506993516'), 'buy_cnt': Decimal('16.722289484448'), 'sell_cnt_tot': Decimal('16.000000000000'), 'hold_cnt': Decimal('0.722289484448'), 'pocket_cnt': Decimal('0.722289484448'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.358802540000'), 'prc_curr': Decimal('0.387638470000'), 'prc_high': Decimal('0.410000000000'), 'prc_low': Decimal('0.357907774055'), 'prc_chg_pct': Decimal('8.036700000000'), 'prc_chg_pct_high': Decimal('13.714400000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-5.680000000000'), 'prc_sell_avg': Decimal('0.387638470000'), 'val_curr': Decimal('0.279987190649'), 'val_tot': Decimal('0.279987190649'), 'gain_loss_amt_est': Decimal('0.562495285277'), 'gain_loss_amt_est_high': Decimal('0.822861332550'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.482202790649'), 'gain_loss_amt_net': Decimal('0.762189981297'), 'gain_loss_pct_est': Decimal('9.374900000000'), 'gain_loss_pct_est_high': Decimal('13.714400000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('8.036700000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31681, 'bo_uuid': 'a50a72f1-144c-4e9d-bd50-659493a86ede', 'buy_curr_symb': 'PNG', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PNG', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PNG', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('116620741.418695990000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 20, 26, 11), 'check_last_dttm': datetime.datetime(2024, 11, 29, 21, 33, 34), 'add_dttm': datetime.datetime(2024, 11, 29, 20, 26, 11), 'upd_dttm': datetime.datetime(2024, 11, 29, 20, 26, 11), 'del_dttm': datetime.datetime(2024, 11, 29, 20, 26, 11), 'dlm': datetime.datetime(2024, 11, 29, 21, 35, 1)},
		{'pos_id': 32025, 'symb': 'USDC', 'prod_id': 'PNG-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 20, 17, 54), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 2, 33, 20), 'age_mins': 75, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.196310400000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015529600000'), 'fees_cnt_tot': Decimal('0.030492193516'), 'buy_cnt': Decimal('16.998600944315'), 'sell_cnt_tot': Decimal('16.000000000000'), 'hold_cnt': Decimal('0.998600944315'), 'pocket_cnt': Decimal('0.998600944315'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.352970230000'), 'prc_curr': Decimal('0.387269400000'), 'prc_high': Decimal('0.410000000000'), 'prc_low': Decimal('0.351660000000'), 'prc_chg_pct': Decimal('9.717300000000'), 'prc_chg_pct_high': Decimal('15.593300000000'), 'prc_chg_pct_low': Decimal('-0.371200000000'), 'prc_chg_pct_drop': Decimal('-5.880000000000'), 'prc_sell_avg': Decimal('0.387269400000'), 'val_curr': Decimal('0.386727588544'), 'val_tot': Decimal('0.386727588544'), 'gain_loss_amt_est': Decimal('0.670930954587'), 'gain_loss_amt_est_high': Decimal('0.935599171290'), 'gain_loss_amt_est_low': Decimal('-0.022271991922'), 'gain_loss_amt': Decimal('0.583037988544'), 'gain_loss_amt_net': Decimal('0.969765577089'), 'gain_loss_pct_est': Decimal('11.182200000000'), 'gain_loss_pct_est_high': Decimal('15.593300000000'), 'gain_loss_pct_est_low': Decimal('-0.371200000000'), 'gain_loss_pct': Decimal('9.717300000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31679, 'bo_uuid': '4152b0d4-696e-4efc-9523-5f58cc1b2281', 'buy_curr_symb': 'PNG', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PNG', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PNG', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('116620741.418695990000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 20, 18, 26), 'check_last_dttm': datetime.datetime(2024, 11, 29, 21, 33, 17), 'add_dttm': datetime.datetime(2024, 11, 29, 20, 18, 26), 'upd_dttm': datetime.datetime(2024, 11, 29, 20, 18, 26), 'del_dttm': datetime.datetime(2024, 11, 29, 20, 18, 26), 'dlm': datetime.datetime(2024, 11, 29, 21, 34, 52)},
		{'pos_id': 32012, 'symb': 'USDC', 'prod_id': 'PNG-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 18, 59, 50), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 2, 33, 4), 'age_mins': 153, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.203013600000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015546400000'), 'fees_cnt_tot': Decimal('0.030508993516'), 'buy_cnt': Decimal('16.484527519442'), 'sell_cnt_tot': Decimal('16.000000000000'), 'hold_cnt': Decimal('0.484527519442'), 'pocket_cnt': Decimal('0.484527519442'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.363977680000'), 'prc_curr': Decimal('0.387688350000'), 'prc_high': Decimal('0.410000000000'), 'prc_low': Decimal('0.344300000000'), 'prc_chg_pct': Decimal('6.514300000000'), 'prc_chg_pct_high': Decimal('12.097500000000'), 'prc_chg_pct_low': Decimal('-5.406300000000'), 'prc_chg_pct_drop': Decimal('-5.580000000000'), 'prc_sell_avg': Decimal('0.387688350000'), 'val_curr': Decimal('0.187845674542'), 'val_tot': Decimal('0.187845674542'), 'gain_loss_amt_est': Decimal('0.469187979730'), 'gain_loss_amt_est_high': Decimal('0.725852073208'), 'gain_loss_amt_est_low': Decimal('-0.324377175056'), 'gain_loss_amt': Decimal('0.390859274542'), 'gain_loss_amt_net': Decimal('0.578704949084'), 'gain_loss_pct_est': Decimal('7.819800000000'), 'gain_loss_pct_est_high': Decimal('12.097500000000'), 'gain_loss_pct_est_low': Decimal('-5.406300000000'), 'gain_loss_pct': Decimal('6.514300000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31666, 'bo_uuid': '37abc18c-ba14-4c01-8fb2-fb71603e63a3', 'buy_curr_symb': 'PNG', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PNG', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PNG', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('116620741.418695990000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 19, 0, 45), 'check_last_dttm': datetime.datetime(2024, 11, 29, 21, 33, 1), 'add_dttm': datetime.datetime(2024, 11, 29, 19, 0, 45), 'upd_dttm': datetime.datetime(2024, 11, 29, 19, 0, 45), 'del_dttm': datetime.datetime(2024, 11, 29, 19, 0, 45), 'dlm': datetime.datetime(2024, 11, 29, 21, 33, 32)},
		{'pos_id': 32010, 'symb': 'USDC', 'prod_id': 'PNG-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 18, 38, 34), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 2, 32, 47), 'age_mins': 175, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.248419800000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015660200000'), 'fees_cnt_tot': Decimal('0.030622793516'), 'buy_cnt': Decimal('16.433381127083'), 'sell_cnt_tot': Decimal('16.000000000000'), 'hold_cnt': Decimal('0.433381127083'), 'pocket_cnt': Decimal('0.433381127083'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.365110500000'), 'prc_curr': Decimal('0.390526240000'), 'prc_high': Decimal('0.410000000000'), 'prc_low': Decimal('0.344300000000'), 'prc_chg_pct': Decimal('6.961100000000'), 'prc_chg_pct_high': Decimal('11.749700000000'), 'prc_chg_pct_low': Decimal('-5.699800000000'), 'prc_chg_pct_drop': Decimal('-4.790000000000'), 'prc_sell_avg': Decimal('0.390526240000'), 'val_curr': Decimal('0.169246702047'), 'val_tot': Decimal('0.169246702047'), 'gain_loss_amt_est': Decimal('0.449116089512'), 'gain_loss_amt_est_high': Decimal('0.704983833661'), 'gain_loss_amt_est_low': Decimal('-0.341986877945'), 'gain_loss_amt': Decimal('0.417666502047'), 'gain_loss_amt_net': Decimal('0.586913204093'), 'gain_loss_pct_est': Decimal('7.485300000000'), 'gain_loss_pct_est_high': Decimal('11.749700000000'), 'gain_loss_pct_est_low': Decimal('-5.699800000000'), 'gain_loss_pct': Decimal('6.961100000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31664, 'bo_uuid': 'c6503d81-4c87-4555-8ea8-6a129a03b659', 'buy_curr_symb': 'PNG', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PNG', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PNG', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('116620741.418695990000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 18, 39, 9), 'check_last_dttm': datetime.datetime(2024, 11, 29, 21, 32, 45), 'add_dttm': datetime.datetime(2024, 11, 29, 18, 39, 9), 'upd_dttm': datetime.datetime(2024, 11, 29, 18, 39, 9), 'del_dttm': datetime.datetime(2024, 11, 29, 18, 39, 9), 'dlm': datetime.datetime(2024, 11, 29, 21, 33, 25)},
		{'pos_id': 31846, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 13, 51, 22), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 0, 41, 53), 'age_mins': 351, 'tot_out_cnt': Decimal('25.000000000000'), 'tot_in_cnt': Decimal('25.728236897664'), 'buy_fees_cnt': Decimal('0.062344139651'), 'sell_fees_cnt_tot': Decimal('0.064481796736'), 'fees_cnt_tot': Decimal('0.126825936387'), 'buy_cnt': Decimal('14.582571697766'), 'sell_cnt_tot': Decimal('14.529472000000'), 'hold_cnt': Decimal('0.053099697766'), 'pocket_cnt': Decimal('0.053099697766'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.714375250000'), 'prc_curr': Decimal('1.770762000000'), 'prc_high': Decimal('1.868900000000'), 'prc_low': Decimal('1.710100000000'), 'prc_chg_pct': Decimal('3.289100000000'), 'prc_chg_pct_high': Decimal('7.351100000000'), 'prc_chg_pct_low': Decimal('-0.214400000000'), 'prc_chg_pct_drop': Decimal('-4.060000000000'), 'prc_sell_avg': Decimal('1.770762000000'), 'val_curr': Decimal('0.094026927016'), 'val_tot': Decimal('0.094026927016'), 'gain_loss_amt_est': Decimal('0.910313392591'), 'gain_loss_amt_est_high': Decimal('1.837764952569'), 'gain_loss_amt_est_low': Decimal('-0.053594596632'), 'gain_loss_amt': Decimal('0.822263824680'), 'gain_loss_amt_net': Decimal('0.916290751695'), 'gain_loss_pct_est': Decimal('3.641300000000'), 'gain_loss_pct_est_high': Decimal('7.351100000000'), 'gain_loss_pct_est_low': Decimal('-0.214400000000'), 'gain_loss_pct': Decimal('3.289100000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31500, 'bo_uuid': '873046b2-fd60-4725-9061-651a66d3fab5', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 13, 52, 9), 'check_last_dttm': datetime.datetime(2024, 11, 29, 19, 41, 50), 'add_dttm': datetime.datetime(2024, 11, 29, 13, 52, 9), 'upd_dttm': datetime.datetime(2024, 11, 29, 13, 52, 9), 'del_dttm': datetime.datetime(2024, 11, 29, 13, 52, 9), 'dlm': datetime.datetime(2024, 11, 29, 19, 42, 32)},
		{'pos_id': 31832, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 12, 46, 8), 'pos_end_dttm': datetime.datetime(2024, 11, 30, 0, 41, 37), 'age_mins': 416, 'tot_out_cnt': Decimal('25.000000000000'), 'tot_in_cnt': Decimal('25.666663763496'), 'buy_fees_cnt': Decimal('0.062344139651'), 'sell_fees_cnt_tot': Decimal('0.064327478104'), 'fees_cnt_tot': Decimal('0.126671617755'), 'buy_cnt': Decimal('14.564686286853'), 'sell_cnt_tot': Decimal('14.513504000000'), 'hold_cnt': Decimal('0.051182286853'), 'pocket_cnt': Decimal('0.051182286853'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.716480500000'), 'prc_curr': Decimal('1.768467750000'), 'prc_high': Decimal('1.868900000000'), 'prc_low': Decimal('1.684700000000'), 'prc_chg_pct': Decimal('3.028700000000'), 'prc_chg_pct_high': Decimal('7.219400000000'), 'prc_chg_pct_low': Decimal('-1.851500000000'), 'prc_chg_pct_drop': Decimal('-4.190000000000'), 'prc_sell_avg': Decimal('1.768467750000'), 'val_curr': Decimal('0.090514223671'), 'val_tot': Decimal('0.090514223671'), 'gain_loss_amt_est': Decimal('0.878534594480'), 'gain_loss_amt_est_high': Decimal('1.804848642324'), 'gain_loss_amt_est_low': Decimal('-0.462873012539'), 'gain_loss_amt': Decimal('0.757177987167'), 'gain_loss_amt_net': Decimal('0.847692210838'), 'gain_loss_pct_est': Decimal('3.514100000000'), 'gain_loss_pct_est_high': Decimal('7.219400000000'), 'gain_loss_pct_est_low': Decimal('-1.851500000000'), 'gain_loss_pct': Decimal('3.028700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31486, 'bo_uuid': '2fb0e384-3eee-4c58-990d-2cc60bc8793a', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 12, 47, 8), 'check_last_dttm': datetime.datetime(2024, 11, 29, 19, 41, 34), 'add_dttm': datetime.datetime(2024, 11, 29, 12, 47, 8), 'upd_dttm': datetime.datetime(2024, 11, 29, 12, 47, 8), 'del_dttm': datetime.datetime(2024, 11, 29, 12, 47, 8), 'dlm': datetime.datetime(2024, 11, 29, 19, 42, 25)},
		{'pos_id': 31904, 'symb': 'USDC', 'prod_id': 'ALGO-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 15, 33, 36), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 23, 31, 35), 'age_mins': 178, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.081516900000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007723100000'), 'fees_cnt_tot': Decimal('0.015204396758'), 'buy_cnt': Decimal('6.885685005159'), 'sell_cnt_tot': Decimal('6.800000000000'), 'hold_cnt': Decimal('0.085685005159'), 'pocket_cnt': Decimal('0.085685005159'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.435686500000'), 'prc_curr': Decimal('0.453164250000'), 'prc_high': Decimal('0.468000000000'), 'prc_low': Decimal('0.429800000000'), 'prc_chg_pct': Decimal('4.011500000000'), 'prc_chg_pct_high': Decimal('6.728100000000'), 'prc_chg_pct_low': Decimal('-1.351100000000'), 'prc_chg_pct_drop': Decimal('-2.720000000000'), 'prc_sell_avg': Decimal('0.453164250000'), 'val_curr': Decimal('0.038829381099'), 'val_tot': Decimal('0.038829381099'), 'gain_loss_amt_est': Decimal('0.150889458361'), 'gain_loss_amt_est_high': Decimal('0.201843527399'), 'gain_loss_amt_est_low': Decimal('-0.040532584783'), 'gain_loss_amt': Decimal('0.120346281099'), 'gain_loss_amt_net': Decimal('0.159175662198'), 'gain_loss_pct_est': Decimal('5.029600000000'), 'gain_loss_pct_est_high': Decimal('6.728100000000'), 'gain_loss_pct_est_low': Decimal('-1.351100000000'), 'gain_loss_pct': Decimal('4.011500000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'bb_bo', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31558, 'bo_uuid': '8ffd16cf-a9eb-4109-96c2-75d820996dd2', 'buy_curr_symb': 'ALGO', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ALGO', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ALGO', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('48877100.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 15, 34, 23), 'check_last_dttm': datetime.datetime(2024, 11, 29, 18, 31, 32), 'add_dttm': datetime.datetime(2024, 11, 29, 15, 34, 23), 'upd_dttm': datetime.datetime(2024, 11, 29, 15, 34, 23), 'del_dttm': datetime.datetime(2024, 11, 29, 15, 34, 23), 'dlm': datetime.datetime(2024, 11, 29, 18, 32, 15)},
		{'pos_id': 31820, 'symb': 'USDC', 'prod_id': 'A8-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 12, 18, 57), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 23, 4, 3), 'age_mins': 345, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.042527820000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015144180000'), 'fees_cnt_tot': Decimal('0.030106773516'), 'buy_cnt': Decimal('49.422274207133'), 'sell_cnt_tot': Decimal('49.360000000000'), 'hold_cnt': Decimal('0.062274207133'), 'pocket_cnt': Decimal('0.062274207133'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.121402750000'), 'prc_curr': Decimal('0.122417500000'), 'prc_high': Decimal('0.126100000000'), 'prc_low': Decimal('0.113600000000'), 'prc_chg_pct': Decimal('0.835900000000'), 'prc_chg_pct_high': Decimal('3.292600000000'), 'prc_chg_pct_low': Decimal('-6.427200000000'), 'prc_chg_pct_drop': Decimal('-2.460000000000'), 'prc_sell_avg': Decimal('0.122417500000'), 'val_curr': Decimal('0.007623452752'), 'val_tot': Decimal('0.007623452752'), 'gain_loss_amt_est': Decimal('0.073997500057'), 'gain_loss_amt_est_high': Decimal('0.197553185574'), 'gain_loss_amt_est_low': Decimal('-0.385629650070'), 'gain_loss_amt': Decimal('0.050151272752'), 'gain_loss_amt_net': Decimal('0.057774725503'), 'gain_loss_pct_est': Decimal('1.233300000000'), 'gain_loss_pct_est_high': Decimal('3.292600000000'), 'gain_loss_pct_est_low': Decimal('-6.427200000000'), 'gain_loss_pct': Decimal('0.835900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31474, 'bo_uuid': 'dd51afc5-d38b-4b69-8066-94c86d8b52da', 'buy_curr_symb': 'A8', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'A8', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'A8', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('10000000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 12, 19, 50), 'check_last_dttm': datetime.datetime(2024, 11, 29, 18, 4), 'add_dttm': datetime.datetime(2024, 11, 29, 12, 19, 50), 'upd_dttm': datetime.datetime(2024, 11, 29, 12, 19, 50), 'del_dttm': datetime.datetime(2024, 11, 29, 12, 19, 50), 'dlm': datetime.datetime(2024, 11, 29, 18, 4, 29)},
		{'pos_id': 31803, 'symb': 'USDC', 'prod_id': 'A8-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 11, 44, 51), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 23, 3, 44), 'age_mins': 379, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.025597050000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007582950000'), 'fees_cnt_tot': Decimal('0.015064246758'), 'buy_cnt': Decimal('24.690748376583'), 'sell_cnt_tot': Decimal('24.660000000000'), 'hold_cnt': Decimal('0.030748376583'), 'pocket_cnt': Decimal('0.030748376583'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.121503000000'), 'prc_curr': Decimal('0.122692500000'), 'prc_high': Decimal('0.126100000000'), 'prc_low': Decimal('0.113600000000'), 'prc_chg_pct': Decimal('0.979000000000'), 'prc_chg_pct_high': Decimal('3.207300000000'), 'prc_chg_pct_low': Decimal('-6.504400000000'), 'prc_chg_pct_drop': Decimal('-2.230000000000'), 'prc_sell_avg': Decimal('0.122692500000'), 'val_curr': Decimal('0.003772595194'), 'val_tot': Decimal('0.003772595194'), 'gain_loss_amt_est': Decimal('0.034492975482'), 'gain_loss_amt_est_high': Decimal('0.096219846424'), 'gain_loss_amt_est_low': Decimal('-0.195130984420'), 'gain_loss_amt': Decimal('0.029369645194'), 'gain_loss_amt_net': Decimal('0.033142240388'), 'gain_loss_pct_est': Decimal('1.149800000000'), 'gain_loss_pct_est_high': Decimal('3.207300000000'), 'gain_loss_pct_est_low': Decimal('-6.504400000000'), 'gain_loss_pct': Decimal('0.979000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31457, 'bo_uuid': '033f12f3-b355-477d-837e-0ca9bd8a471d', 'buy_curr_symb': 'A8', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'A8', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'A8', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('10000000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 11, 45, 10), 'check_last_dttm': datetime.datetime(2024, 11, 29, 18, 3, 41), 'add_dttm': datetime.datetime(2024, 11, 29, 11, 45, 10), 'upd_dttm': datetime.datetime(2024, 11, 29, 11, 45, 10), 'del_dttm': datetime.datetime(2024, 11, 29, 11, 45, 10), 'dlm': datetime.datetime(2024, 11, 29, 18, 4, 21)},
		{'pos_id': 31961, 'symb': 'USDC', 'prod_id': 'SKL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 16, 51, 39), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 22, 22, 9), 'age_mins': 30, 'tot_out_cnt': Decimal('10.000000000000'), 'tot_in_cnt': Decimal('10.152475200000'), 'buy_fees_cnt': Decimal('0.024937655860'), 'sell_fees_cnt_tot': Decimal('0.025444800000'), 'fees_cnt_tot': Decimal('0.050382455860'), 'buy_cnt': Decimal('121.944527434470'), 'sell_cnt_tot': Decimal('121.600000000000'), 'hold_cnt': Decimal('0.344527434470'), 'pocket_cnt': Decimal('0.344527434470'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.082004500000'), 'prc_curr': Decimal('0.083490750000'), 'prc_high': Decimal('0.086000000000'), 'prc_low': Decimal('0.081800000000'), 'prc_chg_pct': Decimal('1.812400000000'), 'prc_chg_pct_high': Decimal('3.530900000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-1.720000000000'), 'prc_sell_avg': Decimal('0.083490750000'), 'val_curr': Decimal('0.028764853899'), 'val_tot': Decimal('0.028764853899'), 'gain_loss_amt_est': Decimal('0.231145851752'), 'gain_loss_amt_est_high': Decimal('0.353090379187'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.181240053899'), 'gain_loss_amt_net': Decimal('0.210004907799'), 'gain_loss_pct_est': Decimal('2.311500000000'), 'gain_loss_pct_est_high': Decimal('3.530900000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('1.812400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31615, 'bo_uuid': 'e3cef7f8-7b5f-4cb3-8c18-6e963c3696eb', 'buy_curr_symb': 'SKL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SKL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SKL', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('257817000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 16, 52, 50), 'check_last_dttm': datetime.datetime(2024, 11, 29, 17, 22, 6), 'add_dttm': datetime.datetime(2024, 11, 29, 16, 52, 50), 'upd_dttm': datetime.datetime(2024, 11, 29, 16, 52, 50), 'del_dttm': datetime.datetime(2024, 11, 29, 16, 52, 50), 'dlm': datetime.datetime(2024, 11, 29, 17, 23, 2)},
		{'pos_id': 31833, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 12, 59, 29), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 22, 18, 46), 'age_mins': 260, 'tot_out_cnt': Decimal('25.000000000000'), 'tot_in_cnt': Decimal('25.321080107547'), 'buy_fees_cnt': Decimal('0.062344139651'), 'sell_fees_cnt_tot': Decimal('0.063461353653'), 'fees_cnt_tot': Decimal('0.125805493304'), 'buy_cnt': Decimal('14.616760952083'), 'sell_cnt_tot': Decimal('14.590494000000'), 'hold_cnt': Decimal('0.026266952083'), 'pocket_cnt': Decimal('0.026266952083'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.710365250000'), 'prc_curr': Decimal('1.735450500000'), 'prc_high': Decimal('1.777800000000'), 'prc_low': Decimal('1.684700000000'), 'prc_chg_pct': Decimal('1.466700000000'), 'prc_chg_pct_high': Decimal('3.018900000000'), 'prc_chg_pct_low': Decimal('-1.500600000000'), 'prc_chg_pct_drop': Decimal('-1.550000000000'), 'prc_sell_avg': Decimal('1.735450500000'), 'val_curr': Decimal('0.045584995126'), 'val_tot': Decimal('0.045584995126'), 'gain_loss_amt_est': Decimal('0.449242493672'), 'gain_loss_amt_est_high': Decimal('0.754732797570'), 'gain_loss_amt_est_low': Decimal('-0.375142824026'), 'gain_loss_amt': Decimal('0.366665102673'), 'gain_loss_amt_net': Decimal('0.412250097799'), 'gain_loss_pct_est': Decimal('1.797000000000'), 'gain_loss_pct_est_high': Decimal('3.018900000000'), 'gain_loss_pct_est_low': Decimal('-1.500600000000'), 'gain_loss_pct': Decimal('1.466700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31487, 'bo_uuid': '6f5cf1da-3a0a-4e04-a447-ba83bedf7584', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 12, 59, 46), 'check_last_dttm': datetime.datetime(2024, 11, 29, 17, 18, 43), 'add_dttm': datetime.datetime(2024, 11, 29, 12, 59, 46), 'upd_dttm': datetime.datetime(2024, 11, 29, 12, 59, 46), 'del_dttm': datetime.datetime(2024, 11, 29, 12, 59, 46), 'dlm': datetime.datetime(2024, 11, 29, 17, 19, 41)},
		{'pos_id': 31029, 'symb': 'USDC', 'prod_id': 'SKL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 13, 18, 17), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 21, 46, 27), 'age_mins': 1648, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.212908800000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015571200000'), 'fees_cnt_tot': Decimal('0.030533793516'), 'buy_cnt': Decimal('77.226289115920'), 'sell_cnt_tot': Decimal('76.800000000000'), 'hold_cnt': Decimal('0.426289115920'), 'pocket_cnt': Decimal('0.426289115920'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.077693750000'), 'prc_curr': Decimal('0.080897250000'), 'prc_high': Decimal('0.083200000000'), 'prc_low': Decimal('0.071600000000'), 'prc_chg_pct': Decimal('4.123200000000'), 'prc_chg_pct_high': Decimal('6.443600000000'), 'prc_chg_pct_low': Decimal('-7.843300000000'), 'prc_chg_pct_drop': Decimal('-2.320000000000'), 'prc_sell_avg': Decimal('0.080897250000'), 'val_curr': Decimal('0.034485617183'), 'val_tot': Decimal('0.034485617183'), 'gain_loss_amt_est': Decimal('0.255329418390'), 'gain_loss_amt_est_high': Decimal('0.386614109887'), 'gain_loss_amt_est_low': Decimal('-0.470597699300'), 'gain_loss_amt': Decimal('0.247394417183'), 'gain_loss_amt_net': Decimal('0.281880034366'), 'gain_loss_pct_est': Decimal('4.255500000000'), 'gain_loss_pct_est_high': Decimal('6.443600000000'), 'gain_loss_pct_est_low': Decimal('-7.843300000000'), 'gain_loss_pct': Decimal('4.123200000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30683, 'bo_uuid': '363ac1db-101c-4bfd-9642-d760326f66a4', 'buy_curr_symb': 'SKL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SKL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SKL', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('257817000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 13, 18, 47), 'check_last_dttm': datetime.datetime(2024, 11, 29, 16, 46, 24), 'add_dttm': datetime.datetime(2024, 11, 28, 13, 18, 47), 'upd_dttm': datetime.datetime(2024, 11, 28, 13, 18, 47), 'del_dttm': datetime.datetime(2024, 11, 28, 13, 18, 47), 'dlm': datetime.datetime(2024, 11, 29, 16, 47, 54)},
		{'pos_id': 31027, 'symb': 'USDC', 'prod_id': 'SKL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 13, 5, 58), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 21, 46, 10), 'age_mins': 1660, 'tot_out_cnt': Decimal('10.000000000000'), 'tot_in_cnt': Decimal('10.342080000000'), 'buy_fees_cnt': Decimal('0.024937655860'), 'sell_fees_cnt_tot': Decimal('0.025920000000'), 'fees_cnt_tot': Decimal('0.050857655860'), 'buy_cnt': Decimal('128.544617836851'), 'sell_cnt_tot': Decimal('128.000000000000'), 'hold_cnt': Decimal('0.544617836851'), 'pocket_cnt': Decimal('0.544617836851'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.077794000000'), 'prc_curr': Decimal('0.080797500000'), 'prc_high': Decimal('0.083200000000'), 'prc_low': Decimal('0.071600000000'), 'prc_chg_pct': Decimal('3.860800000000'), 'prc_chg_pct_high': Decimal('6.306400000000'), 'prc_chg_pct_low': Decimal('-7.962100000000'), 'prc_chg_pct_drop': Decimal('-2.450000000000'), 'prc_sell_avg': Decimal('0.080797500000'), 'val_curr': Decimal('0.044003759673'), 'val_tot': Decimal('0.044003759673'), 'gain_loss_amt_est': Decimal('0.412114044785'), 'gain_loss_amt_est_high': Decimal('0.630639895108'), 'gain_loss_amt_est_low': Decimal('-0.796205362881'), 'gain_loss_amt': Decimal('0.386083759673'), 'gain_loss_amt_net': Decimal('0.430087519346'), 'gain_loss_pct_est': Decimal('4.121100000000'), 'gain_loss_pct_est_high': Decimal('6.306400000000'), 'gain_loss_pct_est_low': Decimal('-7.962100000000'), 'gain_loss_pct': Decimal('3.860800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30681, 'bo_uuid': 'fcbc1a40-89a8-411f-af60-6b4148d44517', 'buy_curr_symb': 'SKL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SKL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SKL', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('257817000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 13, 6, 17), 'check_last_dttm': datetime.datetime(2024, 11, 29, 16, 46, 7), 'add_dttm': datetime.datetime(2024, 11, 28, 13, 6, 17), 'upd_dttm': datetime.datetime(2024, 11, 28, 13, 6, 17), 'del_dttm': datetime.datetime(2024, 11, 28, 13, 6, 17), 'dlm': datetime.datetime(2024, 11, 29, 16, 46, 35)},
		{'pos_id': 31024, 'symb': 'USDC', 'prod_id': 'SKL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 12, 52, 49), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 21, 45, 54), 'age_mins': 1674, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.077886000000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007714000000'), 'fees_cnt_tot': Decimal('0.015195296758'), 'buy_cnt': Decimal('38.169881418902'), 'sell_cnt_tot': Decimal('38.000000000000'), 'hold_cnt': Decimal('0.169881418902'), 'pocket_cnt': Decimal('0.169881418902'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.078596000000'), 'prc_curr': Decimal('0.080997000000'), 'prc_high': Decimal('0.083200000000'), 'prc_low': Decimal('0.071600000000'), 'prc_chg_pct': Decimal('3.054900000000'), 'prc_chg_pct_high': Decimal('5.221600000000'), 'prc_chg_pct_low': Decimal('-8.901200000000'), 'prc_chg_pct_drop': Decimal('-2.170000000000'), 'prc_sell_avg': Decimal('0.080997000000'), 'val_curr': Decimal('0.013759885287'), 'val_tot': Decimal('0.013759885287'), 'gain_loss_amt_est': Decimal('0.091760394931'), 'gain_loss_amt_est_high': Decimal('0.156649193343'), 'gain_loss_amt_est_low': Decimal('-0.267036490407'), 'gain_loss_amt': Decimal('0.091645885287'), 'gain_loss_amt_net': Decimal('0.105405770574'), 'gain_loss_pct_est': Decimal('3.058700000000'), 'gain_loss_pct_est_high': Decimal('5.221600000000'), 'gain_loss_pct_est_low': Decimal('-8.901200000000'), 'gain_loss_pct': Decimal('3.054900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30678, 'bo_uuid': '81f5461f-5cc3-40ee-9b24-003a4241bca8', 'buy_curr_symb': 'SKL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SKL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SKL', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('257817000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 12, 53, 26), 'check_last_dttm': datetime.datetime(2024, 11, 29, 16, 45, 52), 'add_dttm': datetime.datetime(2024, 11, 28, 12, 53, 26), 'upd_dttm': datetime.datetime(2024, 11, 28, 12, 53, 26), 'del_dttm': datetime.datetime(2024, 11, 28, 12, 53, 26), 'dlm': datetime.datetime(2024, 11, 29, 16, 46, 29)},
		{'pos_id': 30554, 'symb': 'USDC', 'prod_id': 'DOGE-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 22, 16, 30), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 21, 15, 10), 'age_mins': 2519, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.199165510000'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.013030490000'), 'fees_cnt_tot': Decimal('0.025499317930'), 'buy_cnt': Decimal('12.170350094116'), 'sell_cnt_tot': Decimal('12.100000000000'), 'hold_cnt': Decimal('0.070350094116'), 'pocket_cnt': Decimal('0.070350094116'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.410834530000'), 'prc_curr': Decimal('0.429683100000'), 'prc_high': Decimal('0.437820000000'), 'prc_low': Decimal('0.391680000000'), 'prc_chg_pct': Decimal('4.587900000000'), 'prc_chg_pct_high': Decimal('6.147400000000'), 'prc_chg_pct_low': Decimal('-4.662300000000'), 'prc_chg_pct_drop': Decimal('-1.560000000000'), 'prc_sell_avg': Decimal('0.429683100000'), 'val_curr': Decimal('0.030228246525'), 'val_tot': Decimal('0.030228246525'), 'gain_loss_amt_est': Decimal('0.247489850080'), 'gain_loss_amt_est_high': Decimal('0.307367972543'), 'gain_loss_amt_est_low': Decimal('-0.233117275137'), 'gain_loss_amt': Decimal('0.229393756525'), 'gain_loss_amt_net': Decimal('0.259622003050'), 'gain_loss_pct_est': Decimal('4.949800000000'), 'gain_loss_pct_est_high': Decimal('6.147400000000'), 'gain_loss_pct_est_low': Decimal('-4.662300000000'), 'gain_loss_pct': Decimal('4.587900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30208, 'bo_uuid': '6ed78868-9652-4b1c-94ef-2c95a140a197', 'buy_curr_symb': 'DOGE', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOGE', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOGE', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('141834300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 22, 17, 59), 'check_last_dttm': datetime.datetime(2024, 11, 29, 16, 15, 8), 'add_dttm': datetime.datetime(2024, 11, 27, 22, 17, 59), 'upd_dttm': datetime.datetime(2024, 11, 27, 22, 17, 59), 'del_dttm': datetime.datetime(2024, 11, 27, 22, 17, 59), 'dlm': datetime.datetime(2024, 11, 29, 16, 16, 27)},
		{'pos_id': 31891, 'symb': 'USDC', 'prod_id': 'ALGO-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 15, 26, 38), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 21, 12, 8), 'age_mins': 45, 'tot_out_cnt': Decimal('10.000000000000'), 'tot_in_cnt': Decimal('10.297741125000'), 'buy_fees_cnt': Decimal('0.024937655860'), 'sell_fees_cnt_tot': Decimal('0.025808875000'), 'fees_cnt_tot': Decimal('0.050746530860'), 'buy_cnt': Decimal('23.665628337223'), 'sell_cnt_tot': Decimal('23.500000000000'), 'hold_cnt': Decimal('0.165628337223'), 'pocket_cnt': Decimal('0.165628337223'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.422553750000'), 'prc_curr': Decimal('0.438201750000'), 'prc_high': Decimal('0.445000000000'), 'prc_low': Decimal('0.421500000000'), 'prc_chg_pct': Decimal('3.703200000000'), 'prc_chg_pct_high': Decimal('5.264700000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-1.560000000000'), 'prc_sell_avg': Decimal('0.438201750000'), 'val_curr': Decimal('0.072578627221'), 'val_tot': Decimal('0.072578627221'), 'gain_loss_amt_est': Decimal('0.370278337371'), 'gain_loss_amt_est_high': Decimal('0.526471484397'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.370319752221'), 'gain_loss_amt_net': Decimal('0.442898379441'), 'gain_loss_pct_est': Decimal('3.702800000000'), 'gain_loss_pct_est_high': Decimal('5.264700000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('3.703200000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31545, 'bo_uuid': '81a54b03-6606-4eec-9735-ddc373a76c0c', 'buy_curr_symb': 'ALGO', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ALGO', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ALGO', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('48877100.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 15, 27, 22), 'check_last_dttm': datetime.datetime(2024, 11, 29, 16, 12, 5), 'add_dttm': datetime.datetime(2024, 11, 29, 15, 27, 22), 'upd_dttm': datetime.datetime(2024, 11, 29, 15, 27, 22), 'del_dttm': datetime.datetime(2024, 11, 29, 15, 27, 22), 'dlm': datetime.datetime(2024, 11, 29, 16, 12, 38)},
		{'pos_id': 31719, 'symb': 'USDC', 'prod_id': 'DOGE-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 9, 48, 26), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 20, 58, 50), 'age_mins': 370, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.018704325000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007565675000'), 'fees_cnt_tot': Decimal('0.015046971758'), 'buy_cnt': Decimal('7.096657899929'), 'sell_cnt_tot': Decimal('7.000000000000'), 'hold_cnt': Decimal('0.096657899929'), 'pocket_cnt': Decimal('0.096657899929'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.422734200000'), 'prc_curr': Decimal('0.431243470000'), 'prc_high': Decimal('0.437820000000'), 'prc_low': Decimal('0.417410000000'), 'prc_chg_pct': Decimal('2.012900000000'), 'prc_chg_pct_high': Decimal('3.159400000000'), 'prc_chg_pct_low': Decimal('-1.259500000000'), 'prc_chg_pct_drop': Decimal('-1.150000000000'), 'prc_sell_avg': Decimal('0.431243470000'), 'val_curr': Decimal('0.041683088168'), 'val_tot': Decimal('0.041683088168'), 'gain_loss_amt_est': Decimal('0.067388444086'), 'gain_loss_amt_est_high': Decimal('0.094781543580'), 'gain_loss_amt_est_low': Decimal('-0.037784025991'), 'gain_loss_amt': Decimal('0.060387413168'), 'gain_loss_amt_net': Decimal('0.102070501337'), 'gain_loss_pct_est': Decimal('2.246300000000'), 'gain_loss_pct_est_high': Decimal('3.159400000000'), 'gain_loss_pct_est_low': Decimal('-1.259500000000'), 'gain_loss_pct': Decimal('2.012900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'bb_bo', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31373, 'bo_uuid': '607828b1-08e5-4efa-bdc2-b97402810f0d', 'buy_curr_symb': 'DOGE', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOGE', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOGE', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('141834300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 9, 48, 45), 'check_last_dttm': datetime.datetime(2024, 11, 29, 15, 58, 48), 'add_dttm': datetime.datetime(2024, 11, 29, 9, 48, 45), 'upd_dttm': datetime.datetime(2024, 11, 29, 9, 48, 45), 'del_dttm': datetime.datetime(2024, 11, 29, 9, 48, 45), 'dlm': datetime.datetime(2024, 11, 29, 15, 59, 33)},
		{'pos_id': 31706, 'symb': 'USDC', 'prod_id': 'DOGE-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 9, 28, 39), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 20, 58, 35), 'age_mins': 390, 'tot_out_cnt': Decimal('10.000000000000'), 'tot_in_cnt': Decimal('10.177715940000'), 'buy_fees_cnt': Decimal('0.024937655860'), 'sell_fees_cnt_tot': Decimal('0.025508060000'), 'fees_cnt_tot': Decimal('0.050445715860'), 'buy_cnt': Decimal('23.713449052989'), 'sell_cnt_tot': Decimal('23.600000000000'), 'hold_cnt': Decimal('0.113449052989'), 'pocket_cnt': Decimal('0.113449052989'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.421701630000'), 'prc_curr': Decimal('0.431259150000'), 'prc_high': Decimal('0.437820000000'), 'prc_low': Decimal('0.417410000000'), 'prc_chg_pct': Decimal('2.266400000000'), 'prc_chg_pct_high': Decimal('3.412000000000'), 'prc_chg_pct_low': Decimal('-1.017700000000'), 'prc_chg_pct_drop': Decimal('-1.150000000000'), 'prc_sell_avg': Decimal('0.431259150000'), 'val_curr': Decimal('0.048925942160'), 'val_tot': Decimal('0.048925942160'), 'gain_loss_amt_est': Decimal('0.249664084173'), 'gain_loss_amt_est_high': Decimal('0.341197997518'), 'gain_loss_amt_est_low': Decimal('-0.101769230792'), 'gain_loss_amt': Decimal('0.226641882160'), 'gain_loss_amt_net': Decimal('0.275567824321'), 'gain_loss_pct_est': Decimal('2.496600000000'), 'gain_loss_pct_est_high': Decimal('3.412000000000'), 'gain_loss_pct_est_low': Decimal('-1.017700000000'), 'gain_loss_pct': Decimal('2.266400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31360, 'bo_uuid': '18cf02ba-0cfb-4465-9238-1c53a51ae3a8', 'buy_curr_symb': 'DOGE', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOGE', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOGE', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('141834300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 9, 29, 24), 'check_last_dttm': datetime.datetime(2024, 11, 29, 15, 58, 32), 'add_dttm': datetime.datetime(2024, 11, 29, 9, 29, 24), 'upd_dttm': datetime.datetime(2024, 11, 29, 9, 29, 24), 'del_dttm': datetime.datetime(2024, 11, 29, 9, 29, 24), 'dlm': datetime.datetime(2024, 11, 29, 15, 59, 27)},
		{'pos_id': 31728, 'symb': 'USDC', 'prod_id': 'ALGO-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 10, 2, 11), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 20, 12, 28), 'age_mins': 311, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.243851250000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015648750000'), 'fees_cnt_tot': Decimal('0.030611343516'), 'buy_cnt': Decimal('15.098479834722'), 'sell_cnt_tot': Decimal('15.000000000000'), 'hold_cnt': Decimal('0.098479834722'), 'pocket_cnt': Decimal('0.098479834722'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.397391000000'), 'prc_curr': Decimal('0.416256750000'), 'prc_high': Decimal('0.424300000000'), 'prc_low': Decimal('0.381900000000'), 'prc_chg_pct': Decimal('4.747400000000'), 'prc_chg_pct_high': Decimal('6.318500000000'), 'prc_chg_pct_low': Decimal('-3.898200000000'), 'prc_chg_pct_drop': Decimal('-1.570000000000'), 'prc_sell_avg': Decimal('0.416256750000'), 'val_curr': Decimal('0.040992895942'), 'val_tot': Decimal('0.040992895942'), 'gain_loss_amt_est': Decimal('0.314184266881'), 'gain_loss_amt_est_high': Decimal('0.379107730170'), 'gain_loss_amt_est_low': Decimal('-0.233890551120'), 'gain_loss_amt': Decimal('0.284844145942'), 'gain_loss_amt_net': Decimal('0.325837041884'), 'gain_loss_pct_est': Decimal('5.236400000000'), 'gain_loss_pct_est_high': Decimal('6.318500000000'), 'gain_loss_pct_est_low': Decimal('-3.898200000000'), 'gain_loss_pct': Decimal('4.747400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31382, 'bo_uuid': 'bf89d048-8c40-4cbf-8913-ec1a337986de', 'buy_curr_symb': 'ALGO', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ALGO', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ALGO', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('48877100.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 10, 3, 15), 'check_last_dttm': datetime.datetime(2024, 11, 29, 15, 12, 25), 'add_dttm': datetime.datetime(2024, 11, 29, 10, 3, 15), 'upd_dttm': datetime.datetime(2024, 11, 29, 10, 3, 15), 'del_dttm': datetime.datetime(2024, 11, 29, 10, 3, 15), 'dlm': datetime.datetime(2024, 11, 29, 15, 13, 39)},
		{'pos_id': 31721, 'symb': 'USDC', 'prod_id': 'ALGO-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 9, 55, 23), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 17, 3, 59), 'age_mins': 129, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('5.996571000000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015029000000'), 'fees_cnt_tot': Decimal('0.029991593516'), 'buy_cnt': Decimal('15.260166768189'), 'sell_cnt_tot': Decimal('15.200000000000'), 'hold_cnt': Decimal('0.060166768189'), 'pocket_cnt': Decimal('0.060166768189'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.393180500000'), 'prc_curr': Decimal('0.394511250000'), 'prc_high': Decimal('0.409900000000'), 'prc_low': Decimal('0.381900000000'), 'prc_chg_pct': Decimal('0.338500000000'), 'prc_chg_pct_high': Decimal('3.514800000000'), 'prc_chg_pct_low': Decimal('-2.869000000000'), 'prc_chg_pct_drop': Decimal('-3.180000000000'), 'prc_sell_avg': Decimal('0.394511250000'), 'val_curr': Decimal('0.023736466927'), 'val_tot': Decimal('0.023736466927'), 'gain_loss_amt_est': Decimal('0.047604090233'), 'gain_loss_amt_est_high': Decimal('0.210887874653'), 'gain_loss_amt_est_low': Decimal('-0.172142311229'), 'gain_loss_amt': Decimal('0.020307466927'), 'gain_loss_amt_net': Decimal('0.044043933853'), 'gain_loss_pct_est': Decimal('0.793400000000'), 'gain_loss_pct_est_high': Decimal('3.514800000000'), 'gain_loss_pct_est_low': Decimal('-2.869000000000'), 'gain_loss_pct': Decimal('0.338500000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31375, 'bo_uuid': '0f3a9a25-4f8b-4137-bbd7-2126830d25b1', 'buy_curr_symb': 'ALGO', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ALGO', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ALGO', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('48877100.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 9, 56, 18), 'check_last_dttm': datetime.datetime(2024, 11, 29, 12, 3, 56), 'add_dttm': datetime.datetime(2024, 11, 29, 9, 56, 18), 'upd_dttm': datetime.datetime(2024, 11, 29, 9, 56, 18), 'del_dttm': datetime.datetime(2024, 11, 29, 9, 56, 18), 'dlm': datetime.datetime(2024, 11, 29, 12, 4, 12)},
		{'pos_id': 31761, 'symb': 'USDC', 'prod_id': 'A8-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 10, 44, 36), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 16, 52, 45), 'age_mins': 68, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.026892802500'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007586197500'), 'fees_cnt_tot': Decimal('0.015067494258'), 'buy_cnt': Decimal('25.317417117106'), 'sell_cnt_tot': Decimal('25.280000000000'), 'hold_cnt': Decimal('0.037417117106'), 'pocket_cnt': Decimal('0.037417117106'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.118495500000'), 'prc_curr': Decimal('0.119734680000'), 'prc_high': Decimal('0.123900000000'), 'prc_low': Decimal('0.117300000000'), 'prc_chg_pct': Decimal('1.045800000000'), 'prc_chg_pct_high': Decimal('3.041900000000'), 'prc_chg_pct_low': Decimal('-1.008900000000'), 'prc_chg_pct_drop': Decimal('-2.000000000000'), 'prc_sell_avg': Decimal('0.119734680000'), 'val_curr': Decimal('0.004480126543'), 'val_tot': Decimal('0.004480126543'), 'gain_loss_amt_est': Decimal('0.035558312341'), 'gain_loss_amt_est_high': Decimal('0.091256629999'), 'gain_loss_amt_est_low': Decimal('-0.030266972163'), 'gain_loss_amt': Decimal('0.031372929043'), 'gain_loss_amt_net': Decimal('0.035853055586'), 'gain_loss_pct_est': Decimal('1.185300000000'), 'gain_loss_pct_est_high': Decimal('3.041900000000'), 'gain_loss_pct_est_low': Decimal('-1.008900000000'), 'gain_loss_pct': Decimal('1.045800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31415, 'bo_uuid': '3bdc6c5d-e3ed-4ad2-8385-ee52c823869e', 'buy_curr_symb': 'A8', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'A8', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'A8', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('10000000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 10, 45), 'check_last_dttm': datetime.datetime(2024, 11, 29, 11, 52, 43), 'add_dttm': datetime.datetime(2024, 11, 29, 10, 45), 'upd_dttm': datetime.datetime(2024, 11, 29, 10, 45), 'del_dttm': datetime.datetime(2024, 11, 29, 10, 45), 'dlm': datetime.datetime(2024, 11, 29, 11, 54, 26)},
		{'pos_id': 28796, 'symb': 'USDC', 'prod_id': 'AVAX-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 21, 36, 56), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 16, 26, 4), 'age_mins': 5149, 'tot_out_cnt': Decimal('19.000000000000'), 'tot_in_cnt': Decimal('19.373068917563'), 'buy_fees_cnt': Decimal('0.047381546135'), 'sell_fees_cnt_tot': Decimal('0.048554057438'), 'fees_cnt_tot': Decimal('0.095935603573'), 'buy_cnt': Decimal('0.441888982370'), 'sell_cnt_tot': Decimal('0.440899500000'), 'hold_cnt': Decimal('0.000989482370'), 'pocket_cnt': Decimal('0.000989482370'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('42.997225000000'), 'prc_curr': Decimal('43.939875000000'), 'prc_high': Decimal('44.670000000000'), 'prc_low': Decimal('38.700000000000'), 'prc_chg_pct': Decimal('2.192400000000'), 'prc_chg_pct_high': Decimal('3.727600000000'), 'prc_chg_pct_low': Decimal('-9.994200000000'), 'prc_chg_pct_drop': Decimal('-1.540000000000'), 'prc_sell_avg': Decimal('43.939875000000'), 'val_curr': Decimal('0.043477731653'), 'val_tot': Decimal('0.043477731653'), 'gain_loss_amt_est': Decimal('0.425439664985'), 'gain_loss_amt_est_high': Decimal('0.708248613702'), 'gain_loss_amt_est_low': Decimal('-1.898896382281'), 'gain_loss_amt': Decimal('0.416546649216'), 'gain_loss_amt_net': Decimal('0.460024380868'), 'gain_loss_pct_est': Decimal('2.239200000000'), 'gain_loss_pct_est_high': Decimal('3.727600000000'), 'gain_loss_pct_est_low': Decimal('-9.994200000000'), 'gain_loss_pct': Decimal('2.192400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28449, 'bo_uuid': 'dc5133c4-2c74-4fe4-8bfa-ba7902e7b7fe', 'buy_curr_symb': 'AVAX', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'AVAX', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'AVAX', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('631600.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 21, 38, 21), 'check_last_dttm': datetime.datetime(2024, 11, 29, 11, 26, 2), 'add_dttm': datetime.datetime(2024, 11, 25, 21, 38, 21), 'upd_dttm': datetime.datetime(2024, 11, 25, 21, 38, 21), 'del_dttm': datetime.datetime(2024, 11, 25, 21, 38, 21), 'dlm': datetime.datetime(2024, 11, 29, 11, 27, 21)},
		{'pos_id': 31509, 'symb': 'USDC', 'prod_id': 'SUI-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 3, 53, 40), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 16, 3, 17), 'age_mins': 429, 'tot_out_cnt': Decimal('4.000000000000'), 'tot_in_cnt': Decimal('3.862320000000'), 'buy_fees_cnt': Decimal('0.009975062344'), 'sell_fees_cnt_tot': Decimal('0.009680000000'), 'fees_cnt_tot': Decimal('0.019655062344'), 'buy_cnt': Decimal('1.180026894288'), 'sell_cnt_tot': Decimal('1.100000000000'), 'hold_cnt': Decimal('0.080026894288'), 'pocket_cnt': Decimal('0.080026894288'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('3.389753250000'), 'prc_curr': Decimal('3.511200000000'), 'prc_high': Decimal('3.557700000000'), 'prc_low': Decimal('3.381300000000'), 'prc_chg_pct': Decimal('3.582800000000'), 'prc_chg_pct_high': Decimal('4.810000000000'), 'prc_chg_pct_low': Decimal('-0.140200000000'), 'prc_chg_pct_drop': Decimal('-1.230000000000'), 'prc_sell_avg': Decimal('3.511200000000'), 'val_curr': Decimal('0.280990431224'), 'val_tot': Decimal('0.280990431224'), 'gain_loss_amt_est': Decimal('0.147086517286'), 'gain_loss_amt_est_high': Decimal('0.192399550026'), 'gain_loss_amt_est_low': Decimal('-0.005608962835'), 'gain_loss_amt': Decimal('0.143310431224'), 'gain_loss_amt_net': Decimal('0.424300862448'), 'gain_loss_pct_est': Decimal('3.677200000000'), 'gain_loss_pct_est_high': Decimal('4.810000000000'), 'gain_loss_pct_est_low': Decimal('-0.140200000000'), 'gain_loss_pct': Decimal('3.582800000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31163, 'bo_uuid': '09244e84-ff11-426b-853c-35851902d4d5', 'buy_curr_symb': 'SUI', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SUI', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SUI', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('8632925.224713609000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 3, 54, 1), 'check_last_dttm': datetime.datetime(2024, 11, 29, 11, 3, 15), 'add_dttm': datetime.datetime(2024, 11, 29, 3, 54, 1), 'upd_dttm': datetime.datetime(2024, 11, 29, 3, 54, 1), 'del_dttm': datetime.datetime(2024, 11, 29, 3, 54, 1), 'dlm': datetime.datetime(2024, 11, 29, 11, 4, 49)},
		{'pos_id': 31506, 'symb': 'USDC', 'prod_id': 'SUI-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 3, 47, 16), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 16, 3, 2), 'age_mins': 436, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('5.965139775000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.014950225000'), 'fees_cnt_tot': Decimal('0.029912818516'), 'buy_cnt': Decimal('1.769726309614'), 'sell_cnt_tot': Decimal('1.700000000000'), 'hold_cnt': Decimal('0.069726309614'), 'pocket_cnt': Decimal('0.069726309614'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('3.390354750000'), 'prc_curr': Decimal('3.508905750000'), 'prc_high': Decimal('3.557700000000'), 'prc_low': Decimal('3.380500000000'), 'prc_chg_pct': Decimal('3.496700000000'), 'prc_chg_pct_high': Decimal('4.791400000000'), 'prc_chg_pct_low': Decimal('-0.290700000000'), 'prc_chg_pct_drop': Decimal('-1.290000000000'), 'prc_sell_avg': Decimal('3.508905750000'), 'val_curr': Decimal('0.244663048731'), 'val_tot': Decimal('0.244663048731'), 'gain_loss_amt_est': Decimal('0.219526142507'), 'gain_loss_amt_est_high': Decimal('0.287483632797'), 'gain_loss_amt_est_low': Decimal('-0.017440210350'), 'gain_loss_amt': Decimal('0.209802823731'), 'gain_loss_amt_net': Decimal('0.454465872462'), 'gain_loss_pct_est': Decimal('3.658800000000'), 'gain_loss_pct_est_high': Decimal('4.791400000000'), 'gain_loss_pct_est_low': Decimal('-0.290700000000'), 'gain_loss_pct': Decimal('3.496700000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31160, 'bo_uuid': '1736d722-0077-43ca-ad49-bf60089e6c8b', 'buy_curr_symb': 'SUI', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SUI', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SUI', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('8632925.224713609000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 3, 47, 45), 'check_last_dttm': datetime.datetime(2024, 11, 29, 11, 3), 'add_dttm': datetime.datetime(2024, 11, 29, 3, 47, 45), 'upd_dttm': datetime.datetime(2024, 11, 29, 3, 47, 45), 'del_dttm': datetime.datetime(2024, 11, 29, 3, 47, 45), 'dlm': datetime.datetime(2024, 11, 29, 11, 4, 41)},
		{'pos_id': 31486, 'symb': 'USDC', 'prod_id': 'SUI-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 3, 10, 31), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 16, 2, 45), 'age_mins': 473, 'tot_out_cnt': Decimal('10.000000000000'), 'tot_in_cnt': Decimal('10.175537400000'), 'buy_fees_cnt': Decimal('0.024937655860'), 'sell_fees_cnt_tot': Decimal('0.025502600000'), 'fees_cnt_tot': Decimal('0.050440255860'), 'buy_cnt': Decimal('2.942062334210'), 'sell_cnt_tot': Decimal('2.900000000000'), 'hold_cnt': Decimal('0.042062334210'), 'pocket_cnt': Decimal('0.042062334210'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('3.398976250000'), 'prc_curr': Decimal('3.508806000000'), 'prc_high': Decimal('3.557700000000'), 'prc_low': Decimal('3.374800000000'), 'prc_chg_pct': Decimal('3.231300000000'), 'prc_chg_pct_high': Decimal('4.525600000000'), 'prc_chg_pct_low': Decimal('-0.711300000000'), 'prc_chg_pct_drop': Decimal('-1.290000000000'), 'prc_sell_avg': Decimal('3.508806000000'), 'val_curr': Decimal('0.147588570650'), 'val_tot': Decimal('0.147588570650'), 'gain_loss_amt_est': Decimal('0.339583867348'), 'gain_loss_amt_est_high': Decimal('0.452559060981'), 'gain_loss_amt_est_low': Decimal('-0.071128034508'), 'gain_loss_amt': Decimal('0.323125970650'), 'gain_loss_amt_net': Decimal('0.470714541300'), 'gain_loss_pct_est': Decimal('3.395800000000'), 'gain_loss_pct_est_high': Decimal('4.525600000000'), 'gain_loss_pct_est_low': Decimal('-0.711300000000'), 'gain_loss_pct': Decimal('3.231300000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31140, 'bo_uuid': '8f5de8d6-e1f5-4bf9-b6de-40a26fc938fe', 'buy_curr_symb': 'SUI', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SUI', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SUI', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('8632925.224713609000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 3, 11, 54), 'check_last_dttm': datetime.datetime(2024, 11, 29, 11, 2, 43), 'add_dttm': datetime.datetime(2024, 11, 29, 3, 11, 54), 'upd_dttm': datetime.datetime(2024, 11, 29, 3, 11, 54), 'del_dttm': datetime.datetime(2024, 11, 29, 3, 11, 54), 'dlm': datetime.datetime(2024, 11, 29, 11, 3, 4)},
		{'pos_id': 27791, 'symb': 'USDC', 'prod_id': 'ADA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 0, 19, 4), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 15, 39, 59), 'age_mins': 6381, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.050879916255'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007646315580'), 'fees_cnt_tot': Decimal('0.015127612338'), 'buy_cnt': Decimal('2.847848023641'), 'sell_cnt_tot': Decimal('2.842232350000'), 'hold_cnt': Decimal('0.005615673641'), 'pocket_cnt': Decimal('0.005615673641'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.053427000000'), 'prc_curr': Decimal('1.073409750000'), 'prc_high': Decimal('1.100000000000'), 'prc_low': Decimal('0.879800000000'), 'prc_chg_pct': Decimal('1.896900000000'), 'prc_chg_pct_high': Decimal('4.117300000000'), 'prc_chg_pct_low': Decimal('-16.482100000000'), 'prc_chg_pct_drop': Decimal('-2.220000000000'), 'prc_sell_avg': Decimal('1.073409750000'), 'val_curr': Decimal('0.006027918839'), 'val_tot': Decimal('0.006027918839'), 'gain_loss_amt_est': Decimal('0.059158346995'), 'gain_loss_amt_est_high': Decimal('0.123519712329'), 'gain_loss_amt_est_low': Decimal('-0.494463308801'), 'gain_loss_amt': Decimal('0.056907835094'), 'gain_loss_amt_net': Decimal('0.062935753933'), 'gain_loss_pct_est': Decimal('1.971900000000'), 'gain_loss_pct_est_high': Decimal('4.117300000000'), 'gain_loss_pct_est_low': Decimal('-16.482100000000'), 'gain_loss_pct': Decimal('1.896900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 27444, 'bo_uuid': '2ce730db-257a-4baf-8eeb-810697fe7a04', 'buy_curr_symb': 'ADA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ADA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ADA', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('30176000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 0, 19, 33), 'check_last_dttm': datetime.datetime(2024, 11, 29, 10, 39, 57), 'add_dttm': datetime.datetime(2024, 11, 25, 0, 19, 33), 'upd_dttm': datetime.datetime(2024, 11, 25, 0, 19, 33), 'del_dttm': datetime.datetime(2024, 11, 25, 0, 19, 33), 'dlm': datetime.datetime(2024, 11, 29, 10, 42, 6)},
		{'pos_id': 27790, 'symb': 'USDC', 'prod_id': 'ADA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 0, 13, 57), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 15, 39, 43), 'age_mins': 6386, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.278399455822'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.050823056280'), 'fees_cnt_tot': Decimal('0.100698368001'), 'buy_cnt': Decimal('18.945988079993'), 'sell_cnt_tot': Decimal('18.912663980000'), 'hold_cnt': Decimal('0.033324099993'), 'pocket_cnt': Decimal('0.033324099993'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.055632460000'), 'prc_curr': Decimal('1.072212750000'), 'prc_high': Decimal('1.100000000000'), 'prc_low': Decimal('0.879800000000'), 'prc_chg_pct': Decimal('1.570600000000'), 'prc_chg_pct_high': Decimal('3.899800000000'), 'prc_chg_pct_low': Decimal('-16.656600000000'), 'prc_chg_pct_drop': Decimal('-2.330000000000'), 'prc_sell_avg': Decimal('1.072212750000'), 'val_curr': Decimal('0.035730524895'), 'val_tot': Decimal('0.035730524895'), 'gain_loss_amt_est': Decimal('0.351780395528'), 'gain_loss_amt_est_high': Decimal('0.779959726136'), 'gain_loss_amt_est_low': Decimal('-3.331319687222'), 'gain_loss_amt': Decimal('0.314129980717'), 'gain_loss_amt_net': Decimal('0.349860505612'), 'gain_loss_pct_est': Decimal('1.758900000000'), 'gain_loss_pct_est_high': Decimal('3.899800000000'), 'gain_loss_pct_est_low': Decimal('-16.656600000000'), 'gain_loss_pct': Decimal('1.570600000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 27443, 'bo_uuid': 'a2f19ffc-3001-4303-b94d-db1214d29da4', 'buy_curr_symb': 'ADA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ADA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ADA', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('30176000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 0, 15, 5), 'check_last_dttm': datetime.datetime(2024, 11, 29, 10, 39, 40), 'add_dttm': datetime.datetime(2024, 11, 25, 0, 15, 5), 'upd_dttm': datetime.datetime(2024, 11, 25, 0, 15, 5), 'del_dttm': datetime.datetime(2024, 11, 25, 0, 15, 5), 'dlm': datetime.datetime(2024, 11, 29, 10, 41, 52)},
		{'pos_id': 27789, 'symb': 'USDC', 'prod_id': 'ADA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 0, 8, 28), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 15, 39, 27), 'age_mins': 6391, 'tot_out_cnt': Decimal('25.000000000000'), 'tot_in_cnt': Decimal('25.364150651310'), 'buy_fees_cnt': Decimal('0.062344139651'), 'sell_fees_cnt_tot': Decimal('0.063569299878'), 'fees_cnt_tot': Decimal('0.125913439529'), 'buy_cnt': Decimal('23.741104208253'), 'sell_cnt_tot': Decimal('23.693365590000'), 'hold_cnt': Decimal('0.047738618253'), 'pocket_cnt': Decimal('0.047738618253'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.053026000000'), 'prc_curr': Decimal('1.070517000000'), 'prc_high': Decimal('1.100000000000'), 'prc_low': Decimal('0.879800000000'), 'prc_chg_pct': Decimal('1.661000000000'), 'prc_chg_pct_high': Decimal('4.157000000000'), 'prc_chg_pct_low': Decimal('-16.450300000000'), 'prc_chg_pct_drop': Decimal('-2.500000000000'), 'prc_sell_avg': Decimal('1.070517000000'), 'val_curr': Decimal('0.051105002396'), 'val_tot': Decimal('0.051105002396'), 'gain_loss_amt_est': Decimal('0.502694140505'), 'gain_loss_amt_est_high': Decimal('1.039243095612'), 'gain_loss_amt_est_low': Decimal('-4.112576517579'), 'gain_loss_amt': Decimal('0.415255653706'), 'gain_loss_amt_net': Decimal('0.466360656103'), 'gain_loss_pct_est': Decimal('2.010800000000'), 'gain_loss_pct_est_high': Decimal('4.157000000000'), 'gain_loss_pct_est_low': Decimal('-16.450300000000'), 'gain_loss_pct': Decimal('1.661000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 27442, 'bo_uuid': '71952c91-62af-43e4-ada2-6d9b9d755b35', 'buy_curr_symb': 'ADA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ADA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ADA', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('30176000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 0, 10, 17), 'check_last_dttm': datetime.datetime(2024, 11, 29, 10, 39, 25), 'add_dttm': datetime.datetime(2024, 11, 25, 0, 10, 17), 'upd_dttm': datetime.datetime(2024, 11, 25, 0, 10, 17), 'del_dttm': datetime.datetime(2024, 11, 25, 0, 10, 17), 'dlm': datetime.datetime(2024, 11, 29, 10, 40, 32)},
		{'pos_id': 31698, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 9, 18, 43), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 15, 38, 42), 'age_mins': 79, 'tot_out_cnt': Decimal('15.000000000000'), 'tot_in_cnt': Decimal('15.324897415601'), 'buy_fees_cnt': Decimal('0.037406483791'), 'sell_fees_cnt_tot': Decimal('0.038408264200'), 'fees_cnt_tot': Decimal('0.075814747991'), 'buy_cnt': Decimal('8.892014926136'), 'sell_cnt_tot': Decimal('8.865662000000'), 'hold_cnt': Decimal('0.026352926136'), 'pocket_cnt': Decimal('0.026352926136'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.686906750000'), 'prc_curr': Decimal('1.728567750000'), 'prc_high': Decimal('1.777400000000'), 'prc_low': Decimal('1.680900000000'), 'prc_chg_pct': Decimal('2.469700000000'), 'prc_chg_pct_high': Decimal('4.783500000000'), 'prc_chg_pct_low': Decimal('-0.356100000000'), 'prc_chg_pct_drop': Decimal('-2.310000000000'), 'prc_sell_avg': Decimal('1.728567750000'), 'val_curr': Decimal('0.045552818237'), 'val_tot': Decimal('0.045552818237'), 'gain_loss_amt_est': Decimal('0.444540725206'), 'gain_loss_amt_est_high': Decimal('0.717525583438'), 'gain_loss_amt_est_low': Decimal('-0.053412110658'), 'gain_loss_amt': Decimal('0.370450233838'), 'gain_loss_amt_net': Decimal('0.416003052075'), 'gain_loss_pct_est': Decimal('2.963600000000'), 'gain_loss_pct_est_high': Decimal('4.783500000000'), 'gain_loss_pct_est_low': Decimal('-0.356100000000'), 'gain_loss_pct': Decimal('2.469700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31352, 'bo_uuid': '381e26e6-4a90-4497-b7a2-5b85742d49b5', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 9, 18, 50), 'check_last_dttm': datetime.datetime(2024, 11, 29, 10, 38, 40), 'add_dttm': datetime.datetime(2024, 11, 29, 9, 18, 50), 'upd_dttm': datetime.datetime(2024, 11, 29, 9, 18, 50), 'del_dttm': datetime.datetime(2024, 11, 29, 9, 18, 50), 'dlm': datetime.datetime(2024, 11, 29, 10, 40, 16)},
		{'pos_id': 31654, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 7, 46, 44), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 15, 38, 25), 'age_mins': 171, 'tot_out_cnt': Decimal('19.000000000000'), 'tot_in_cnt': Decimal('19.560998741099'), 'buy_fees_cnt': Decimal('0.047381546135'), 'sell_fees_cnt_tot': Decimal('0.049025059502'), 'fees_cnt_tot': Decimal('0.096406605637'), 'buy_cnt': Decimal('11.336654177453'), 'sell_cnt_tot': Decimal('11.295446000000'), 'hold_cnt': Decimal('0.041208177453'), 'pocket_cnt': Decimal('0.041208177453'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.675979500000'), 'prc_curr': Decimal('1.731759750000'), 'prc_high': Decimal('1.777400000000'), 'prc_low': Decimal('1.655400000000'), 'prc_chg_pct': Decimal('3.328200000000'), 'prc_chg_pct_high': Decimal('5.466700000000'), 'prc_chg_pct_low': Decimal('-1.227900000000'), 'prc_chg_pct_drop': Decimal('-2.140000000000'), 'prc_sell_avg': Decimal('1.731759750000'), 'val_curr': Decimal('0.071362663084'), 'val_tot': Decimal('0.071362663084'), 'gain_loss_amt_est': Decimal('0.690634640818'), 'gain_loss_amt_est_high': Decimal('1.038669924066'), 'gain_loss_amt_est_low': Decimal('-0.233302674644'), 'gain_loss_amt': Decimal('0.632361404183'), 'gain_loss_amt_net': Decimal('0.703724067267'), 'gain_loss_pct_est': Decimal('3.634900000000'), 'gain_loss_pct_est_high': Decimal('5.466700000000'), 'gain_loss_pct_est_low': Decimal('-1.227900000000'), 'gain_loss_pct': Decimal('3.328200000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31308, 'bo_uuid': 'e35a5890-8f16-4734-86d4-d1e221491054', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 7, 48, 53), 'check_last_dttm': datetime.datetime(2024, 11, 29, 10, 38, 23), 'add_dttm': datetime.datetime(2024, 11, 29, 7, 48, 53), 'upd_dttm': datetime.datetime(2024, 11, 29, 7, 48, 53), 'del_dttm': datetime.datetime(2024, 11, 29, 7, 48, 53), 'dlm': datetime.datetime(2024, 11, 29, 10, 40)},
		{'pos_id': 31617, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 7, 12, 45), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 15, 38, 1), 'age_mins': 205, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.761262131974'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.052033238426'), 'fees_cnt_tot': Decimal('0.101908550147'), 'buy_cnt': Decimal('12.038453227299'), 'sell_cnt_tot': Decimal('11.983703000000'), 'hold_cnt': Decimal('0.054750227299'), 'pocket_cnt': Decimal('0.054750227299'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.661343000000'), 'prc_curr': Decimal('1.732458000000'), 'prc_high': Decimal('1.777400000000'), 'prc_low': Decimal('1.651500000000'), 'prc_chg_pct': Decimal('4.280600000000'), 'prc_chg_pct_high': Decimal('6.395800000000'), 'prc_chg_pct_low': Decimal('-0.592500000000'), 'prc_chg_pct_drop': Decimal('-2.120000000000'), 'prc_sell_avg': Decimal('1.732458000000'), 'val_curr': Decimal('0.094852469286'), 'val_tot': Decimal('0.094852469286'), 'gain_loss_amt_est': Decimal('0.909589410496'), 'gain_loss_amt_est_high': Decimal('1.279169924574'), 'gain_loss_amt_est_low': Decimal('-0.118494495116'), 'gain_loss_amt': Decimal('0.856114601260'), 'gain_loss_amt_net': Decimal('0.950967070546'), 'gain_loss_pct_est': Decimal('4.547900000000'), 'gain_loss_pct_est_high': Decimal('6.395800000000'), 'gain_loss_pct_est_low': Decimal('-0.592500000000'), 'gain_loss_pct': Decimal('4.280600000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31271, 'bo_uuid': 'e0e5353a-02cb-4cc6-a04a-c9c2d730d835', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 7, 13, 3), 'check_last_dttm': datetime.datetime(2024, 11, 29, 10, 37, 59), 'add_dttm': datetime.datetime(2024, 11, 29, 7, 13, 3), 'upd_dttm': datetime.datetime(2024, 11, 29, 7, 13, 3), 'del_dttm': datetime.datetime(2024, 11, 29, 7, 13, 3), 'dlm': datetime.datetime(2024, 11, 29, 10, 38, 25)},
		{'pos_id': 31616, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 7, 6, 27), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 15, 37, 45), 'age_mins': 212, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.688477594561'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.051850821039'), 'fees_cnt_tot': Decimal('0.101726132760'), 'buy_cnt': Decimal('11.992861249341'), 'sell_cnt_tot': Decimal('11.943066000000'), 'hold_cnt': Decimal('0.049795249341'), 'pocket_cnt': Decimal('0.049795249341'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.667658750000'), 'prc_curr': Decimal('1.732258500000'), 'prc_high': Decimal('1.777400000000'), 'prc_low': Decimal('1.651500000000'), 'prc_chg_pct': Decimal('3.873700000000'), 'prc_chg_pct_high': Decimal('5.992900000000'), 'prc_chg_pct_low': Decimal('-0.968900000000'), 'prc_chg_pct_drop': Decimal('-2.120000000000'), 'prc_sell_avg': Decimal('1.732258500000'), 'val_curr': Decimal('0.086258243931'), 'val_tot': Decimal('0.086258243931'), 'gain_loss_amt_est': Decimal('0.830400703980'), 'gain_loss_amt_est_high': Decimal('1.198581544335'), 'gain_loss_amt_est_low': Decimal('-0.193789646713'), 'gain_loss_amt': Decimal('0.774735838492'), 'gain_loss_amt_net': Decimal('0.860994082422'), 'gain_loss_pct_est': Decimal('4.152000000000'), 'gain_loss_pct_est_high': Decimal('5.992900000000'), 'gain_loss_pct_est_low': Decimal('-0.968900000000'), 'gain_loss_pct': Decimal('3.873700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31270, 'bo_uuid': '4d2d825b-6bd2-4390-881a-69e01afd4879', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 7, 6, 37), 'check_last_dttm': datetime.datetime(2024, 11, 29, 10, 37, 43), 'add_dttm': datetime.datetime(2024, 11, 29, 7, 6, 37), 'upd_dttm': datetime.datetime(2024, 11, 29, 7, 6, 37), 'del_dttm': datetime.datetime(2024, 11, 29, 7, 6, 37), 'dlm': datetime.datetime(2024, 11, 29, 10, 38, 18)},
		{'pos_id': 31677, 'symb': 'USDC', 'prod_id': 'DIMO-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 8, 57, 5), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 15, 26, 50), 'age_mins': 90, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.274727865000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015726135000'), 'fees_cnt_tot': Decimal('0.030688728516'), 'buy_cnt': Decimal('23.908590286757'), 'sell_cnt_tot': Decimal('23.700000000000'), 'hold_cnt': Decimal('0.208590286757'), 'pocket_cnt': Decimal('0.208590286757'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.250955830000'), 'prc_curr': Decimal('0.264756450000'), 'prc_high': Decimal('0.289000000000'), 'prc_low': Decimal('0.250330000000'), 'prc_chg_pct': Decimal('5.499200000000'), 'prc_chg_pct_high': Decimal('10.668900000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-5.170000000000'), 'prc_sell_avg': Decimal('0.264756450000'), 'val_curr': Decimal('0.055225623826'), 'val_tot': Decimal('0.055225623826'), 'gain_loss_amt_est': Decimal('0.357772329054'), 'gain_loss_amt_est_high': Decimal('0.640132780341'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.329953488826'), 'gain_loss_amt_net': Decimal('0.385179112653'), 'gain_loss_pct_est': Decimal('5.962900000000'), 'gain_loss_pct_est_high': Decimal('10.668900000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('5.499200000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31331, 'bo_uuid': 'd24a1df9-a524-4c99-b766-e35a210fa80b', 'buy_curr_symb': 'DIMO', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DIMO', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DIMO', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('78083813.347947510000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 8, 57, 22), 'check_last_dttm': datetime.datetime(2024, 11, 29, 10, 26, 47), 'add_dttm': datetime.datetime(2024, 11, 29, 8, 57, 22), 'upd_dttm': datetime.datetime(2024, 11, 29, 8, 57, 22), 'del_dttm': datetime.datetime(2024, 11, 29, 8, 57, 22), 'dlm': datetime.datetime(2024, 11, 29, 10, 27, 23)},
		{'pos_id': 31278, 'symb': 'USDC', 'prod_id': 'DOT-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 20, 32, 43), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 15, 25, 53), 'age_mins': 832, 'tot_out_cnt': Decimal('13.000000000000'), 'tot_in_cnt': Decimal('13.280600732245'), 'buy_fees_cnt': Decimal('0.032418952618'), 'sell_fees_cnt_tot': Decimal('0.033284713615'), 'fees_cnt_tot': Decimal('0.065703666233'), 'buy_cnt': Decimal('1.506106974144'), 'sell_cnt_tot': Decimal('1.501509580000'), 'hold_cnt': Decimal('0.004597394144'), 'pocket_cnt': Decimal('0.004597394144'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('8.631525000000'), 'prc_curr': Decimal('8.844832500000'), 'prc_high': Decimal('9.004000000000'), 'prc_low': Decimal('8.448000000000'), 'prc_chg_pct': Decimal('2.471300000000'), 'prc_chg_pct_high': Decimal('4.129900000000'), 'prc_chg_pct_low': Decimal('-2.126200000000'), 'prc_chg_pct_drop': Decimal('-1.660000000000'), 'prc_sell_avg': Decimal('8.844832500000'), 'val_curr': Decimal('0.040663181140'), 'val_tot': Decimal('0.040663181140'), 'gain_loss_amt_est': Decimal('0.396821535011'), 'gain_loss_amt_est_high': Decimal('0.536889483606'), 'gain_loss_amt_est_low': Decimal('-0.276408282431'), 'gain_loss_amt': Decimal('0.321263913385'), 'gain_loss_amt_net': Decimal('0.361927094525'), 'gain_loss_pct_est': Decimal('3.052500000000'), 'gain_loss_pct_est_high': Decimal('4.129900000000'), 'gain_loss_pct_est_low': Decimal('-2.126200000000'), 'gain_loss_pct': Decimal('2.471300000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30922, 'bo_uuid': 'dee0dcdd-52fd-46f3-8b5d-02f9e247c904', 'buy_curr_symb': 'DOT', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOT', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOT', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('1732900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 21, 24, 13), 'check_last_dttm': datetime.datetime(2024, 11, 29, 10, 25, 51), 'add_dttm': datetime.datetime(2024, 11, 28, 21, 24, 13), 'upd_dttm': datetime.datetime(2024, 11, 28, 21, 24, 13), 'del_dttm': datetime.datetime(2024, 11, 28, 21, 24, 13), 'dlm': datetime.datetime(2024, 11, 29, 10, 27, 11)},
		{'pos_id': 31277, 'symb': 'USDC', 'prod_id': 'DOT-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 20, 3, 53), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 15, 25, 38), 'age_mins': 861, 'tot_out_cnt': Decimal('13.000000000000'), 'tot_in_cnt': Decimal('13.230757547697'), 'buy_fees_cnt': Decimal('0.032418952618'), 'sell_fees_cnt_tot': Decimal('0.033159793353'), 'fees_cnt_tot': Decimal('0.065578745971'), 'buy_cnt': Decimal('1.500182906916'), 'sell_cnt_tot': Decimal('1.496211770000'), 'hold_cnt': Decimal('0.003971136916'), 'pocket_cnt': Decimal('0.003971136916'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('8.665610000000'), 'prc_curr': Decimal('8.842837500000'), 'prc_high': Decimal('9.004000000000'), 'prc_low': Decimal('8.448000000000'), 'prc_chg_pct': Decimal('2.045200000000'), 'prc_chg_pct_high': Decimal('3.720300000000'), 'prc_chg_pct_low': Decimal('-2.511200000000'), 'prc_chg_pct_drop': Decimal('-1.680000000000'), 'prc_sell_avg': Decimal('8.842837500000'), 'val_curr': Decimal('0.035116118438'), 'val_tot': Decimal('0.035116118438'), 'gain_loss_amt_est': Decimal('0.344126957018'), 'gain_loss_amt_est_high': Decimal('0.483643967361'), 'gain_loss_amt_est_low': Decimal('-0.326454802374'), 'gain_loss_amt': Decimal('0.265873666135'), 'gain_loss_amt_net': Decimal('0.300989784574'), 'gain_loss_pct_est': Decimal('2.647100000000'), 'gain_loss_pct_est_high': Decimal('3.720300000000'), 'gain_loss_pct_est_low': Decimal('-2.511200000000'), 'gain_loss_pct': Decimal('2.045200000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30907, 'bo_uuid': '03bc7e24-78a7-4eb1-a431-7f5009a7358f', 'buy_curr_symb': 'DOT', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOT', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOT', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('1732900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 21, 20, 4), 'check_last_dttm': datetime.datetime(2024, 11, 29, 10, 25, 35), 'add_dttm': datetime.datetime(2024, 11, 28, 21, 20, 4), 'upd_dttm': datetime.datetime(2024, 11, 28, 21, 20, 4), 'del_dttm': datetime.datetime(2024, 11, 28, 21, 20, 4), 'dlm': datetime.datetime(2024, 11, 29, 10, 27, 4)},
		{'pos_id': 31274, 'symb': 'USDC', 'prod_id': 'DOT-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 19, 22, 10), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 15, 25, 20), 'age_mins': 902, 'tot_out_cnt': Decimal('12.000000000000'), 'tot_in_cnt': Decimal('12.232698323043'), 'buy_fees_cnt': Decimal('0.029925187032'), 'sell_fees_cnt_tot': Decimal('0.030658391787'), 'fees_cnt_tot': Decimal('0.060583578819'), 'buy_cnt': Decimal('1.386227540587'), 'sell_cnt_tot': Decimal('1.382409730000'), 'hold_cnt': Decimal('0.003817810587'), 'pocket_cnt': Decimal('0.003817810587'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('8.656587500000'), 'prc_curr': Decimal('8.848822500000'), 'prc_high': Decimal('9.004000000000'), 'prc_low': Decimal('8.448000000000'), 'prc_chg_pct': Decimal('2.220700000000'), 'prc_chg_pct_high': Decimal('3.828400000000'), 'prc_chg_pct_low': Decimal('-2.409600000000'), 'prc_chg_pct_drop': Decimal('-1.610000000000'), 'prc_sell_avg': Decimal('8.848822500000'), 'val_curr': Decimal('0.033783128223'), 'val_tot': Decimal('0.033783128223'), 'gain_loss_amt_est': Decimal('0.330493973521'), 'gain_loss_amt_est_high': Decimal('0.459413134796'), 'gain_loss_amt_est_low': Decimal('-0.289149737121'), 'gain_loss_amt': Decimal('0.266481451266'), 'gain_loss_amt_net': Decimal('0.300264579489'), 'gain_loss_pct_est': Decimal('2.754100000000'), 'gain_loss_pct_est_high': Decimal('3.828400000000'), 'gain_loss_pct_est_low': Decimal('-2.409600000000'), 'gain_loss_pct': Decimal('2.220700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30877, 'bo_uuid': '8c22b213-815e-48a2-ade5-e2931de6e227', 'buy_curr_symb': 'DOT', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOT', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOT', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('1732900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 21, 8, 10), 'check_last_dttm': datetime.datetime(2024, 11, 29, 10, 25, 17), 'add_dttm': datetime.datetime(2024, 11, 28, 21, 8, 10), 'upd_dttm': datetime.datetime(2024, 11, 28, 21, 8, 10), 'del_dttm': datetime.datetime(2024, 11, 28, 21, 8, 10), 'dlm': datetime.datetime(2024, 11, 29, 10, 26, 58)},
		{'pos_id': 31268, 'symb': 'USDC', 'prod_id': 'DOT-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 19, 2, 34), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 15, 25, 3), 'age_mins': 922, 'tot_out_cnt': Decimal('12.000000000000'), 'tot_in_cnt': Decimal('12.239162825812'), 'buy_fees_cnt': Decimal('0.029925187032'), 'sell_fees_cnt_tot': Decimal('0.030674593548'), 'fees_cnt_tot': Decimal('0.060599780580'), 'buy_cnt': Decimal('1.386869981806'), 'sell_cnt_tot': Decimal('1.382984380000'), 'hold_cnt': Decimal('0.003885601806'), 'pocket_cnt': Decimal('0.003885601806'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('8.652577500000'), 'prc_curr': Decimal('8.849820000000'), 'prc_high': Decimal('9.004000000000'), 'prc_low': Decimal('8.448000000000'), 'prc_chg_pct': Decimal('2.279600000000'), 'prc_chg_pct_high': Decimal('3.876600000000'), 'prc_chg_pct_low': Decimal('-2.364400000000'), 'prc_chg_pct_drop': Decimal('-1.600000000000'), 'prc_sell_avg': Decimal('8.849820000000'), 'val_curr': Decimal('0.034386876575'), 'val_tot': Decimal('0.034386876575'), 'gain_loss_amt_est': Decimal('0.336208488164'), 'gain_loss_amt_est_high': Decimal('0.465187396472'), 'gain_loss_amt_est_low': Decimal('-0.283722393703'), 'gain_loss_amt': Decimal('0.273549702387'), 'gain_loss_amt_net': Decimal('0.307936578962'), 'gain_loss_pct_est': Decimal('2.801700000000'), 'gain_loss_pct_est_high': Decimal('3.876600000000'), 'gain_loss_pct_est_low': Decimal('-2.364400000000'), 'gain_loss_pct': Decimal('2.279600000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30873, 'bo_uuid': 'c287af7d-8577-402d-a933-84452fcc09e3', 'buy_curr_symb': 'DOT', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOT', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOT', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('1732900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 20, 54, 56), 'check_last_dttm': datetime.datetime(2024, 11, 29, 10, 25, 1), 'add_dttm': datetime.datetime(2024, 11, 28, 20, 54, 56), 'upd_dttm': datetime.datetime(2024, 11, 28, 20, 54, 56), 'del_dttm': datetime.datetime(2024, 11, 28, 20, 54, 56), 'dlm': datetime.datetime(2024, 11, 29, 10, 25, 39)},
		{'pos_id': 31182, 'symb': 'USDC', 'prod_id': 'DOT-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 18, 15, 8), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 15, 24, 34), 'age_mins': 969, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.356126668426'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.051017861324'), 'fees_cnt_tot': Decimal('0.100893173045'), 'buy_cnt': Decimal('2.298289182724'), 'sell_cnt_tot': Decimal('2.293195250000'), 'hold_cnt': Decimal('0.005093932724'), 'pocket_cnt': Decimal('0.005093932724'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('8.702125110000'), 'prc_curr': Decimal('8.876752500000'), 'prc_high': Decimal('9.004000000000'), 'prc_low': Decimal('8.448000000000'), 'prc_chg_pct': Decimal('2.006700000000'), 'prc_chg_pct_high': Decimal('3.285100000000'), 'prc_chg_pct_low': Decimal('-2.920300000000'), 'prc_chg_pct_drop': Decimal('-1.280000000000'), 'prc_sell_avg': Decimal('8.876752500000'), 'val_curr': Decimal('0.045217580043'), 'val_tot': Decimal('0.045217580043'), 'gain_loss_amt_est': Decimal('0.443282280330'), 'gain_loss_amt_est_high': Decimal('0.657023174323'), 'gain_loss_amt_est_low': Decimal('-0.584052984348'), 'gain_loss_amt': Decimal('0.401344248469'), 'gain_loss_amt_net': Decimal('0.446561828511'), 'gain_loss_pct_est': Decimal('2.216400000000'), 'gain_loss_pct_est_high': Decimal('3.285100000000'), 'gain_loss_pct_est_low': Decimal('-2.920300000000'), 'gain_loss_pct': Decimal('2.006700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30836, 'bo_uuid': 'ac9b071c-147c-465d-bfaf-88bfedcaf29d', 'buy_curr_symb': 'DOT', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOT', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOT', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('1732900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 18, 15, 37), 'check_last_dttm': datetime.datetime(2024, 11, 29, 10, 24, 32), 'add_dttm': datetime.datetime(2024, 11, 28, 18, 15, 37), 'upd_dttm': datetime.datetime(2024, 11, 28, 18, 15, 37), 'del_dttm': datetime.datetime(2024, 11, 28, 18, 15, 37), 'dlm': datetime.datetime(2024, 11, 29, 10, 25, 22)},
		{'pos_id': 30486, 'symb': 'USDC', 'prod_id': 'DOT-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 20, 15, 35), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 15, 24, 12), 'age_mins': 2289, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.200925403539'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.013034900761'), 'fees_cnt_tot': Decimal('0.025503728691'), 'buy_cnt': Decimal('0.587874961347'), 'sell_cnt_tot': Decimal('0.585180730000'), 'hold_cnt': Decimal('0.002694231347'), 'pocket_cnt': Decimal('0.002694231347'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('8.505210000000'), 'prc_curr': Decimal('8.887725000000'), 'prc_high': Decimal('9.004000000000'), 'prc_low': Decimal('8.020000000000'), 'prc_chg_pct': Decimal('4.497400000000'), 'prc_chg_pct_high': Decimal('5.676400000000'), 'prc_chg_pct_low': Decimal('-5.704900000000'), 'prc_chg_pct_drop': Decimal('-1.180000000000'), 'prc_sell_avg': Decimal('8.887725000000'), 'val_curr': Decimal('0.023945587299'), 'val_tot': Decimal('0.023945587299'), 'gain_loss_amt_est': Decimal('0.229147781182'), 'gain_loss_amt_est_high': Decimal('0.283820152587'), 'gain_loss_amt_est_low': Decimal('-0.285242809997'), 'gain_loss_amt': Decimal('0.224870990838'), 'gain_loss_amt_net': Decimal('0.248816578136'), 'gain_loss_pct_est': Decimal('4.583000000000'), 'gain_loss_pct_est_high': Decimal('5.676400000000'), 'gain_loss_pct_est_low': Decimal('-5.704900000000'), 'gain_loss_pct': Decimal('4.497400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30140, 'bo_uuid': 'f72666ed-ad7c-4b7e-a5c9-d4dc1604ecca', 'buy_curr_symb': 'DOT', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOT', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOT', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('1732900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 20, 16, 19), 'check_last_dttm': datetime.datetime(2024, 11, 29, 10, 24, 10), 'add_dttm': datetime.datetime(2024, 11, 27, 20, 16, 19), 'upd_dttm': datetime.datetime(2024, 11, 27, 20, 16, 19), 'del_dttm': datetime.datetime(2024, 11, 27, 20, 16, 19), 'dlm': datetime.datetime(2024, 11, 29, 10, 25, 10)},
		{'pos_id': 31675, 'symb': 'USDC', 'prod_id': 'ALGO-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 8, 48, 51), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 14, 42, 17), 'age_mins': 53, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.066545625000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015204375000'), 'fees_cnt_tot': Decimal('0.030166968516'), 'buy_cnt': Decimal('16.019907404935'), 'sell_cnt_tot': Decimal('15.900000000000'), 'hold_cnt': Decimal('0.119907404935'), 'pocket_cnt': Decimal('0.119907404935'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.374534000000'), 'prc_curr': Decimal('0.381543750000'), 'prc_high': Decimal('0.389600000000'), 'prc_low': Decimal('0.373300000000'), 'prc_chg_pct': Decimal('1.871600000000'), 'prc_chg_pct_high': Decimal('3.221600000000'), 'prc_chg_pct_low': Decimal('-0.329500000000'), 'prc_chg_pct_drop': Decimal('-1.350000000000'), 'prc_sell_avg': Decimal('0.381543750000'), 'val_curr': Decimal('0.045749920932'), 'val_tot': Decimal('0.045749920932'), 'gain_loss_amt_est': Decimal('0.126012591647'), 'gain_loss_amt_est_high': Decimal('0.193296202748'), 'gain_loss_amt_est_low': Decimal('-0.019768565738'), 'gain_loss_amt': Decimal('0.112295545932'), 'gain_loss_amt_net': Decimal('0.158045466863'), 'gain_loss_pct_est': Decimal('2.100200000000'), 'gain_loss_pct_est_high': Decimal('3.221600000000'), 'gain_loss_pct_est_low': Decimal('-0.329500000000'), 'gain_loss_pct': Decimal('1.871600000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31329, 'bo_uuid': 'b03a4d15-7e35-481d-9f61-5505451fce02', 'buy_curr_symb': 'ALGO', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ALGO', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ALGO', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('48877100.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 8, 49, 31), 'check_last_dttm': datetime.datetime(2024, 11, 29, 9, 42, 15), 'add_dttm': datetime.datetime(2024, 11, 29, 8, 49, 31), 'upd_dttm': datetime.datetime(2024, 11, 29, 8, 49, 31), 'del_dttm': datetime.datetime(2024, 11, 29, 8, 49, 31), 'dlm': datetime.datetime(2024, 11, 29, 9, 44, 12)},
		{'pos_id': 31295, 'symb': 'USDC', 'prod_id': 'DOGE-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 21, 35, 35), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 14, 18, 17), 'age_mins': 703, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.147127665000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015406335000'), 'fees_cnt_tot': Decimal('0.030368928516'), 'buy_cnt': Decimal('14.773492808264'), 'sell_cnt_tot': Decimal('14.700000000000'), 'hold_cnt': Decimal('0.073492808264'), 'pocket_cnt': Decimal('0.073492808264'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.406132800000'), 'prc_curr': Decimal('0.418171950000'), 'prc_high': Decimal('0.422970000000'), 'prc_low': Decimal('0.400550000000'), 'prc_chg_pct': Decimal('2.964300000000'), 'prc_chg_pct_high': Decimal('3.838000000000'), 'prc_chg_pct_low': Decimal('-1.374600000000'), 'prc_chg_pct_drop': Decimal('-0.870000000000'), 'prc_sell_avg': Decimal('0.418171950000'), 'val_curr': Decimal('0.030732630943'), 'val_tot': Decimal('0.030732630943'), 'gain_loss_amt_est': Decimal('0.176058668495'), 'gain_loss_amt_est_high': Decimal('0.230277387101'), 'gain_loss_amt_est_low': Decimal('-0.082477455650'), 'gain_loss_amt': Decimal('0.177860295943'), 'gain_loss_amt_net': Decimal('0.208592926885'), 'gain_loss_pct_est': Decimal('2.934300000000'), 'gain_loss_pct_est_high': Decimal('3.838000000000'), 'gain_loss_pct_est_low': Decimal('-1.374600000000'), 'gain_loss_pct': Decimal('2.964300000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30949, 'bo_uuid': '89d197c0-7a3c-41fa-974d-a8368ad41c7f', 'buy_curr_symb': 'DOGE', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOGE', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOGE', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('141834300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 21, 36, 18), 'check_last_dttm': datetime.datetime(2024, 11, 29, 9, 18, 14), 'add_dttm': datetime.datetime(2024, 11, 28, 21, 36, 18), 'upd_dttm': datetime.datetime(2024, 11, 28, 21, 36, 18), 'del_dttm': datetime.datetime(2024, 11, 28, 21, 36, 18), 'dlm': datetime.datetime(2024, 11, 29, 9, 19, 4)},
		{'pos_id': 31550, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 5, 5, 38), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 13, 50, 25), 'age_mins': 225, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.105118095805'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.012794782195'), 'fees_cnt_tot': Decimal('0.025263610125'), 'buy_cnt': Decimal('3.044333255246'), 'sell_cnt_tot': Decimal('3.036436000000'), 'hold_cnt': Decimal('0.007897255246'), 'pocket_cnt': Decimal('0.007897255246'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.642395750000'), 'prc_curr': Decimal('1.681286250000'), 'prc_high': Decimal('1.703900000000'), 'prc_low': Decimal('1.638300000000'), 'prc_chg_pct': Decimal('2.367900000000'), 'prc_chg_pct_high': Decimal('3.708300000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-1.340000000000'), 'prc_sell_avg': Decimal('1.681286250000'), 'val_curr': Decimal('0.013277546658'), 'val_tot': Decimal('0.013277546658'), 'gain_loss_amt_est': Decimal('0.129701535090'), 'gain_loss_amt_est_high': Decimal('0.185412833661'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.118395642463'), 'gain_loss_amt_net': Decimal('0.131673189121'), 'gain_loss_pct_est': Decimal('2.594000000000'), 'gain_loss_pct_est_high': Decimal('3.708300000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('2.367900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31204, 'bo_uuid': '0d79a2a2-70c0-4ab8-8e58-5c7666041051', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 5, 6, 49), 'check_last_dttm': datetime.datetime(2024, 11, 29, 8, 50, 23), 'add_dttm': datetime.datetime(2024, 11, 29, 5, 6, 49), 'upd_dttm': datetime.datetime(2024, 11, 29, 5, 6, 49), 'del_dttm': datetime.datetime(2024, 11, 29, 5, 6, 49), 'dlm': datetime.datetime(2024, 11, 29, 8, 51, 16)},
		{'pos_id': 31527, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 4, 32, 27), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 13, 50, 10), 'age_mins': 258, 'tot_out_cnt': Decimal('10.000000000000'), 'tot_in_cnt': Decimal('10.261764814392'), 'buy_fees_cnt': Decimal('0.024937655860'), 'sell_fees_cnt_tot': Decimal('0.025718708808'), 'fees_cnt_tot': Decimal('0.050656364668'), 'buy_cnt': Decimal('6.119294732924'), 'sell_cnt_tot': Decimal('6.100263000000'), 'hold_cnt': Decimal('0.019031732924'), 'pocket_cnt': Decimal('0.019031732924'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.634175250000'), 'prc_curr': Decimal('1.682184000000'), 'prc_high': Decimal('1.703900000000'), 'prc_low': Decimal('1.626200000000'), 'prc_chg_pct': Decimal('2.937800000000'), 'prc_chg_pct_high': Decimal('4.229900000000'), 'prc_chg_pct_low': Decimal('-0.488000000000'), 'prc_chg_pct_drop': Decimal('-1.290000000000'), 'prc_sell_avg': Decimal('1.682184000000'), 'val_curr': Decimal('0.032014876617'), 'val_tot': Decimal('0.032014876617'), 'gain_loss_amt_est': Decimal('0.311011624977'), 'gain_loss_amt_est_high': Decimal('0.422994718589'), 'gain_loss_amt_est_low': Decimal('-0.048802905319'), 'gain_loss_amt': Decimal('0.293779691009'), 'gain_loss_amt_net': Decimal('0.325794567626'), 'gain_loss_pct_est': Decimal('3.110100000000'), 'gain_loss_pct_est_high': Decimal('4.229900000000'), 'gain_loss_pct_est_low': Decimal('-0.488000000000'), 'gain_loss_pct': Decimal('2.937800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31181, 'bo_uuid': '8b91ec14-d5d8-4ed8-aa23-67bba001cff5', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 4, 32, 56), 'check_last_dttm': datetime.datetime(2024, 11, 29, 8, 50, 8), 'add_dttm': datetime.datetime(2024, 11, 29, 4, 32, 56), 'upd_dttm': datetime.datetime(2024, 11, 29, 4, 32, 56), 'del_dttm': datetime.datetime(2024, 11, 29, 4, 32, 56), 'dlm': datetime.datetime(2024, 11, 29, 8, 51, 9)},
		{'pos_id': 31589, 'symb': 'USDC', 'prod_id': 'ALGO-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 6, 22, 1), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 12, 44, 4), 'age_mins': 82, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.042774000000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007626000000'), 'fees_cnt_tot': Decimal('0.015107296758'), 'buy_cnt': Decimal('8.246124836710'), 'sell_cnt_tot': Decimal('8.200000000000'), 'hold_cnt': Decimal('0.046124836710'), 'pocket_cnt': Decimal('0.046124836710'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.363807250000'), 'prc_curr': Decimal('0.371070000000'), 'prc_high': Decimal('0.375800000000'), 'prc_low': Decimal('0.359400000000'), 'prc_chg_pct': Decimal('1.996300000000'), 'prc_chg_pct_high': Decimal('3.269000000000'), 'prc_chg_pct_low': Decimal('-1.211400000000'), 'prc_chg_pct_drop': Decimal('-1.270000000000'), 'prc_sell_avg': Decimal('0.371070000000'), 'val_curr': Decimal('0.017115543158'), 'val_tot': Decimal('0.017115543158'), 'gain_loss_amt_est': Decimal('0.066733826772'), 'gain_loss_amt_est_high': Decimal('0.098069101152'), 'gain_loss_amt_est_low': Decimal('-0.036342733686'), 'gain_loss_amt': Decimal('0.059889543158'), 'gain_loss_amt_net': Decimal('0.077005086316'), 'gain_loss_pct_est': Decimal('2.224500000000'), 'gain_loss_pct_est_high': Decimal('3.269000000000'), 'gain_loss_pct_est_low': Decimal('-1.211400000000'), 'gain_loss_pct': Decimal('1.996300000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31243, 'bo_uuid': '057c4c68-073c-489e-892d-0c2910090936', 'buy_curr_symb': 'ALGO', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ALGO', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ALGO', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('48877100.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 6, 22, 46), 'check_last_dttm': datetime.datetime(2024, 11, 29, 7, 44, 2), 'add_dttm': datetime.datetime(2024, 11, 29, 6, 22, 46), 'upd_dttm': datetime.datetime(2024, 11, 29, 6, 22, 46), 'del_dttm': datetime.datetime(2024, 11, 29, 6, 22, 46), 'dlm': datetime.datetime(2024, 11, 29, 7, 44, 54)},
		{'pos_id': 31070, 'symb': 'USDC', 'prod_id': 'RENDER-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 15, 3, 2), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 11, 44, 54), 'age_mins': 942, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.134631250000'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.012868750000'), 'fees_cnt_tot': Decimal('0.025337577930'), 'buy_cnt': Decimal('0.585734723672'), 'sell_cnt_tot': Decimal('0.580000000000'), 'hold_cnt': Decimal('0.005734723672'), 'pocket_cnt': Decimal('0.005734723672'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('8.536287500000'), 'prc_curr': Decimal('8.852812500000'), 'prc_high': Decimal('8.988000000000'), 'prc_low': Decimal('8.370000000000'), 'prc_chg_pct': Decimal('3.708000000000'), 'prc_chg_pct_high': Decimal('5.139400000000'), 'prc_chg_pct_low': Decimal('-1.948000000000'), 'prc_chg_pct_drop': Decimal('-1.430000000000'), 'prc_sell_avg': Decimal('8.852812500000'), 'val_curr': Decimal('0.050768433408'), 'val_tot': Decimal('0.050768433408'), 'gain_loss_amt_est': Decimal('0.211867571233'), 'gain_loss_amt_est_high': Decimal('0.256969144956'), 'gain_loss_amt_est_low': Decimal('-0.097400362865'), 'gain_loss_amt': Decimal('0.185399683408'), 'gain_loss_amt_net': Decimal('0.236168116815'), 'gain_loss_pct_est': Decimal('4.237400000000'), 'gain_loss_pct_est_high': Decimal('5.139400000000'), 'gain_loss_pct_est_low': Decimal('-1.948000000000'), 'gain_loss_pct': Decimal('3.708000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30724, 'bo_uuid': '067561aa-b8ae-4097-9382-7726e2e1d819', 'buy_curr_symb': 'RENDER', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'RENDER', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'RENDER', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('1982431.303836978700'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 15, 3, 38), 'check_last_dttm': datetime.datetime(2024, 11, 29, 6, 44, 52), 'add_dttm': datetime.datetime(2024, 11, 28, 15, 3, 38), 'upd_dttm': datetime.datetime(2024, 11, 28, 15, 3, 38), 'del_dttm': datetime.datetime(2024, 11, 28, 15, 3, 38), 'dlm': datetime.datetime(2024, 11, 29, 6, 45, 30)},
		{'pos_id': 31503, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 3, 36, 58), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 11, 41, 40), 'age_mins': 184, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.388371868837'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.051098676363'), 'fees_cnt_tot': Decimal('0.100973988084'), 'buy_cnt': Decimal('12.383690369346'), 'sell_cnt_tot': Decimal('12.356106000000'), 'hold_cnt': Decimal('0.027584369346'), 'pocket_cnt': Decimal('0.027584369346'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.615027460000'), 'prc_curr': Decimal('1.650064500000'), 'prc_high': Decimal('1.683800000000'), 'prc_low': Decimal('1.605700000000'), 'prc_chg_pct': Decimal('2.169400000000'), 'prc_chg_pct_high': Decimal('3.453300000000'), 'prc_chg_pct_low': Decimal('-0.577500000000'), 'prc_chg_pct_drop': Decimal('-1.280000000000'), 'prc_sell_avg': Decimal('1.650064500000'), 'val_curr': Decimal('0.045515988613'), 'val_tot': Decimal('0.045515988613'), 'gain_loss_amt_est': Decimal('0.445472799790'), 'gain_loss_amt_est_high': Decimal('0.690669869103'), 'gain_loss_amt_est_low': Decimal('-0.115508373941'), 'gain_loss_amt': Decimal('0.433887857450'), 'gain_loss_amt_net': Decimal('0.479403846062'), 'gain_loss_pct_est': Decimal('2.227400000000'), 'gain_loss_pct_est_high': Decimal('3.453300000000'), 'gain_loss_pct_est_low': Decimal('-0.577500000000'), 'gain_loss_pct': Decimal('2.169400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31157, 'bo_uuid': 'df552df6-7ba2-4bba-9335-ab7409c9fcf6', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 3, 37, 26), 'check_last_dttm': datetime.datetime(2024, 11, 29, 6, 41, 38), 'add_dttm': datetime.datetime(2024, 11, 29, 3, 37, 26), 'upd_dttm': datetime.datetime(2024, 11, 29, 3, 37, 26), 'del_dttm': datetime.datetime(2024, 11, 29, 3, 37, 26), 'dlm': datetime.datetime(2024, 11, 29, 6, 43, 2)},
		{'pos_id': 31491, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 3, 25, 2), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 11, 41, 24), 'age_mins': 196, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.427449779711'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.051196615989'), 'fees_cnt_tot': Decimal('0.101071927710'), 'buy_cnt': Decimal('12.403708460756'), 'sell_cnt_tot': Decimal('12.374031000000'), 'hold_cnt': Decimal('0.029677460756'), 'pocket_cnt': Decimal('0.029677460756'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.612421000000'), 'prc_curr': Decimal('1.650832280000'), 'prc_high': Decimal('1.683800000000'), 'prc_low': Decimal('1.605700000000'), 'prc_chg_pct': Decimal('2.382200000000'), 'prc_chg_pct_high': Decimal('3.620600000000'), 'prc_chg_pct_low': Decimal('-0.416800000000'), 'prc_chg_pct_drop': Decimal('-1.240000000000'), 'prc_sell_avg': Decimal('1.650832280000'), 'val_curr': Decimal('0.048992510204'), 'val_tot': Decimal('0.048992510204'), 'gain_loss_amt_est': Decimal('0.478522668708'), 'gain_loss_amt_est_high': Decimal('0.724116096231'), 'gain_loss_amt_est_low': Decimal('-0.083365324564'), 'gain_loss_amt': Decimal('0.476442289915'), 'gain_loss_amt_net': Decimal('0.525434800120'), 'gain_loss_pct_est': Decimal('2.392600000000'), 'gain_loss_pct_est_high': Decimal('3.620600000000'), 'gain_loss_pct_est_low': Decimal('-0.416800000000'), 'gain_loss_pct': Decimal('2.382200000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31145, 'bo_uuid': '0d20fab2-a2b6-4e23-aa3f-e93b7923bf4a', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 3, 25, 13), 'check_last_dttm': datetime.datetime(2024, 11, 29, 6, 41, 22), 'add_dttm': datetime.datetime(2024, 11, 29, 3, 25, 13), 'upd_dttm': datetime.datetime(2024, 11, 29, 3, 25, 13), 'del_dttm': datetime.datetime(2024, 11, 29, 3, 25, 13), 'dlm': datetime.datetime(2024, 11, 29, 6, 41, 41)},
		{'pos_id': 31490, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 29, 3, 18, 55), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 11, 41, 8), 'age_mins': 203, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.385133012272'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.051090558928'), 'fees_cnt_tot': Decimal('0.100965870649'), 'buy_cnt': Decimal('12.389842683070'), 'sell_cnt_tot': Decimal('12.361616000000'), 'hold_cnt': Decimal('0.028226683070'), 'pocket_cnt': Decimal('0.028226683070'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.614225500000'), 'prc_curr': Decimal('1.649067000000'), 'prc_high': Decimal('1.683800000000'), 'prc_low': Decimal('1.603300000000'), 'prc_chg_pct': Decimal('2.158400000000'), 'prc_chg_pct_high': Decimal('3.504700000000'), 'prc_chg_pct_low': Decimal('-0.676800000000'), 'prc_chg_pct_drop': Decimal('-1.350000000000'), 'prc_sell_avg': Decimal('1.649067000000'), 'val_curr': Decimal('0.046547691570'), 'val_tot': Decimal('0.046547691570'), 'gain_loss_amt_est': Decimal('0.455630269749'), 'gain_loss_amt_est_high': Decimal('0.700949154873'), 'gain_loss_amt_est_low': Decimal('-0.135365226234'), 'gain_loss_amt': Decimal('0.431680703842'), 'gain_loss_amt_net': Decimal('0.478228395412'), 'gain_loss_pct_est': Decimal('2.278200000000'), 'gain_loss_pct_est_high': Decimal('3.504700000000'), 'gain_loss_pct_est_low': Decimal('-0.676800000000'), 'gain_loss_pct': Decimal('2.158400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 31144, 'bo_uuid': '30d60463-e760-494b-8707-d06f429bb12f', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 29, 3, 19, 59), 'check_last_dttm': datetime.datetime(2024, 11, 29, 6, 41, 6), 'add_dttm': datetime.datetime(2024, 11, 29, 3, 19, 59), 'upd_dttm': datetime.datetime(2024, 11, 29, 3, 19, 59), 'del_dttm': datetime.datetime(2024, 11, 29, 3, 19, 59), 'dlm': datetime.datetime(2024, 11, 29, 6, 41, 34)},
		{'pos_id': 29896, 'symb': 'USDC', 'prod_id': 'XLM-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 5, 45, 22), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 11, 17, 55), 'age_mins': 2912, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.446206452529'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.051243625194'), 'fees_cnt_tot': Decimal('0.101118936915'), 'buy_cnt': Decimal('38.229908255628'), 'sell_cnt_tot': Decimal('38.114935620000'), 'hold_cnt': Decimal('0.114972635628'), 'pocket_cnt': Decimal('0.114972635628'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.523150610000'), 'prc_curr': Decimal('0.536435550000'), 'prc_high': Decimal('0.551179000000'), 'prc_low': Decimal('0.464231000000'), 'prc_chg_pct': Decimal('2.539400000000'), 'prc_chg_pct_high': Decimal('4.302900000000'), 'prc_chg_pct_low': Decimal('-11.262500000000'), 'prc_chg_pct_drop': Decimal('-1.760000000000'), 'prc_sell_avg': Decimal('0.536435550000'), 'val_curr': Decimal('0.061675409028'), 'val_tot': Decimal('0.061675409028'), 'gain_loss_amt_est': Decimal('0.601485880426'), 'gain_loss_amt_est_high': Decimal('0.860569968674'), 'gain_loss_amt_est_low': Decimal('-2.252491460582'), 'gain_loss_amt': Decimal('0.507881861557'), 'gain_loss_amt_net': Decimal('0.569557270585'), 'gain_loss_pct_est': Decimal('3.007400000000'), 'gain_loss_pct_est_high': Decimal('4.302800000000'), 'gain_loss_pct_est_low': Decimal('-11.262500000000'), 'gain_loss_pct': Decimal('2.539400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29550, 'bo_uuid': '6c8832b6-1353-484c-9878-42c2a70e809e', 'buy_curr_symb': 'XLM', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XLM', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XLM', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('116945400.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 5, 45, 53), 'check_last_dttm': datetime.datetime(2024, 11, 29, 6, 17, 53), 'add_dttm': datetime.datetime(2024, 11, 27, 5, 45, 53), 'upd_dttm': datetime.datetime(2024, 11, 27, 5, 45, 53), 'del_dttm': datetime.datetime(2024, 11, 27, 5, 45, 53), 'dlm': datetime.datetime(2024, 11, 29, 6, 19, 26)},
		{'pos_id': 28427, 'symb': 'USDC', 'prod_id': 'XLM-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 12, 34, 58), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 11, 17, 40), 'age_mins': 5383, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.141100450915'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.012884963536'), 'fees_cnt_tot': Decimal('0.025353791466'), 'buy_cnt': Decimal('9.617243201507'), 'sell_cnt_tot': Decimal('9.582124870000'), 'hold_cnt': Decimal('0.035118331507'), 'pocket_cnt': Decimal('0.035118331507'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.519899510000'), 'prc_curr': Decimal('0.536530310000'), 'prc_high': Decimal('0.551179000000'), 'prc_low': Decimal('0.416771000000'), 'prc_chg_pct': Decimal('3.198800000000'), 'prc_chg_pct_high': Decimal('4.955100000000'), 'prc_chg_pct_low': Decimal('-19.836200000000'), 'prc_chg_pct_drop': Decimal('-1.760000000000'), 'prc_sell_avg': Decimal('0.536530310000'), 'val_curr': Decimal('0.018842049290'), 'val_tot': Decimal('0.018842049290'), 'gain_loss_amt_est': Decimal('0.182578485401'), 'gain_loss_amt_est_high': Decimal('0.247754542578'), 'gain_loss_amt_est_low': Decimal('-0.991811933665'), 'gain_loss_amt': Decimal('0.159942500205'), 'gain_loss_amt_net': Decimal('0.178784549495'), 'gain_loss_pct_est': Decimal('3.651600000000'), 'gain_loss_pct_est_high': Decimal('4.955100000000'), 'gain_loss_pct_est_low': Decimal('-19.836200000000'), 'gain_loss_pct': Decimal('3.198900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28080, 'bo_uuid': 'd8aaefbe-9582-487c-a274-8a71c0221b39', 'buy_curr_symb': 'XLM', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XLM', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XLM', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('116945400.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 12, 35, 40), 'check_last_dttm': datetime.datetime(2024, 11, 29, 6, 17, 38), 'add_dttm': datetime.datetime(2024, 11, 25, 12, 35, 40), 'upd_dttm': datetime.datetime(2024, 11, 25, 12, 35, 40), 'del_dttm': datetime.datetime(2024, 11, 25, 12, 35, 40), 'dlm': datetime.datetime(2024, 11, 29, 6, 19, 19)},
		{'pos_id': 28396, 'symb': 'USDC', 'prod_id': 'XLM-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 11, 56, 24), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 11, 17, 24), 'age_mins': 5421, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.399433700238'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.051126400251'), 'fees_cnt_tot': Decimal('0.101001711972'), 'buy_cnt': Decimal('38.095367075267'), 'sell_cnt_tot': Decimal('37.994608630000'), 'hold_cnt': Decimal('0.100758445267'), 'pocket_cnt': Decimal('0.100758445267'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.524998220000'), 'prc_curr': Decimal('0.536903380000'), 'prc_high': Decimal('0.551179000000'), 'prc_low': Decimal('0.416771000000'), 'prc_chg_pct': Decimal('2.267700000000'), 'prc_chg_pct_high': Decimal('3.935800000000'), 'prc_chg_pct_low': Decimal('-20.614800000000'), 'prc_chg_pct_drop': Decimal('-1.670000000000'), 'prc_sell_avg': Decimal('0.536903380000'), 'val_curr': Decimal('0.054097549827'), 'val_tot': Decimal('0.054097549827'), 'gain_loss_amt_est': Decimal('0.528983790988'), 'gain_loss_amt_est_high': Decimal('0.787156093657'), 'gain_loss_amt_est_low': Decimal('-4.122955768674'), 'gain_loss_amt': Decimal('0.453531250065'), 'gain_loss_amt_net': Decimal('0.507628799893'), 'gain_loss_pct_est': Decimal('2.644900000000'), 'gain_loss_pct_est_high': Decimal('3.935800000000'), 'gain_loss_pct_est_low': Decimal('-20.614800000000'), 'gain_loss_pct': Decimal('2.267700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28049, 'bo_uuid': 'a2632085-8ad0-4fa1-9cf1-dda1fd63c1e4', 'buy_curr_symb': 'XLM', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XLM', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XLM', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('116945400.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 11, 57, 31), 'check_last_dttm': datetime.datetime(2024, 11, 29, 6, 17, 22), 'add_dttm': datetime.datetime(2024, 11, 25, 11, 57, 31), 'upd_dttm': datetime.datetime(2024, 11, 25, 11, 57, 31), 'del_dttm': datetime.datetime(2024, 11, 25, 11, 57, 31), 'dlm': datetime.datetime(2024, 11, 29, 6, 19, 12)},
		{'pos_id': 28116, 'symb': 'USDC', 'prod_id': 'XLM-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 7, 46, 18), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 11, 17, 9), 'age_mins': 5671, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.210503242122'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015565171033'), 'fees_cnt_tot': Decimal('0.030527764549'), 'buy_cnt': Decimal('11.597651042685'), 'sell_cnt_tot': Decimal('11.549368860000'), 'hold_cnt': Decimal('0.048282182685'), 'pocket_cnt': Decimal('0.048282182685'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('10.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.517346140000'), 'prc_curr': Decimal('0.537735290000'), 'prc_high': Decimal('0.551179000000'), 'prc_low': Decimal('0.416771000000'), 'prc_chg_pct': Decimal('3.941100000000'), 'prc_chg_pct_high': Decimal('5.473100000000'), 'prc_chg_pct_low': Decimal('-19.440600000000'), 'prc_chg_pct_drop': Decimal('-1.530000000000'), 'prc_sell_avg': Decimal('0.537735290000'), 'val_curr': Decimal('0.025963033508'), 'val_tot': Decimal('0.025963033508'), 'gain_loss_amt_est': Decimal('0.249788584486'), 'gain_loss_amt_est_high': Decimal('0.328385865603'), 'gain_loss_amt_est_low': Decimal('-1.166435377289'), 'gain_loss_amt': Decimal('0.236466275630'), 'gain_loss_amt_net': Decimal('0.262429309138'), 'gain_loss_pct_est': Decimal('4.163100000000'), 'gain_loss_pct_est_high': Decimal('5.473100000000'), 'gain_loss_pct_est_low': Decimal('-19.440600000000'), 'gain_loss_pct': Decimal('3.941100000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 27769, 'bo_uuid': '56d0f0bb-3abd-4c82-96e0-42456e45a557', 'buy_curr_symb': 'XLM', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XLM', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XLM', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('116945400.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 7, 47, 2), 'check_last_dttm': datetime.datetime(2024, 11, 29, 6, 17, 7), 'add_dttm': datetime.datetime(2024, 11, 25, 7, 47, 2), 'upd_dttm': datetime.datetime(2024, 11, 25, 7, 47, 2), 'del_dttm': datetime.datetime(2024, 11, 25, 7, 47, 2), 'dlm': datetime.datetime(2024, 11, 29, 6, 19, 5)},
		{'pos_id': 31324, 'symb': 'USDC', 'prod_id': 'ALGO-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 22, 5, 45), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 11, 6, 14), 'age_mins': 481, 'tot_out_cnt': Decimal('18.000000000000'), 'tot_in_cnt': Decimal('18.188165625000'), 'buy_fees_cnt': Decimal('0.044887780549'), 'sell_fees_cnt_tot': Decimal('0.045584375000'), 'fees_cnt_tot': Decimal('0.090472155549'), 'buy_cnt': Decimal('50.336731761849'), 'sell_cnt_tot': Decimal('50.300000000000'), 'hold_cnt': Decimal('0.036731761849'), 'pocket_cnt': Decimal('0.036731761849'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.357591750000'), 'prc_curr': Decimal('0.361593750000'), 'prc_high': Decimal('0.370000000000'), 'prc_low': Decimal('0.332500000000'), 'prc_chg_pct': Decimal('1.119200000000'), 'prc_chg_pct_high': Decimal('3.414000000000'), 'prc_chg_pct_low': Decimal('-7.016900000000'), 'prc_chg_pct_drop': Decimal('-2.290000000000'), 'prc_sell_avg': Decimal('0.361593750000'), 'val_curr': Decimal('0.013281975511'), 'val_tot': Decimal('0.013281975511'), 'gain_loss_amt_est': Decimal('0.327604034489'), 'gain_loss_amt_est_high': Decimal('0.614523405532'), 'gain_loss_amt_est_low': Decimal('-1.263036689185'), 'gain_loss_amt': Decimal('0.201447600511'), 'gain_loss_amt_net': Decimal('0.214729576022'), 'gain_loss_pct_est': Decimal('1.820000000000'), 'gain_loss_pct_est_high': Decimal('3.414000000000'), 'gain_loss_pct_est_low': Decimal('-7.016900000000'), 'gain_loss_pct': Decimal('1.119200000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30978, 'bo_uuid': 'c8834ba9-4843-47a5-a260-338002d5e543', 'buy_curr_symb': 'ALGO', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ALGO', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ALGO', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('48877100.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 22, 6, 34), 'check_last_dttm': datetime.datetime(2024, 11, 29, 6, 6, 12), 'add_dttm': datetime.datetime(2024, 11, 28, 22, 6, 34), 'upd_dttm': datetime.datetime(2024, 11, 28, 22, 6, 34), 'del_dttm': datetime.datetime(2024, 11, 28, 22, 6, 34), 'dlm': datetime.datetime(2024, 11, 29, 6, 7, 39)},
		{'pos_id': 31331, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 22, 8, 58), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 7, 35, 22), 'age_mins': 266, 'tot_out_cnt': Decimal('2.000000000000'), 'tot_in_cnt': Decimal('2.050831249058'), 'buy_fees_cnt': Decimal('0.004987531172'), 'sell_fees_cnt_tot': Decimal('0.005139927943'), 'fees_cnt_tot': Decimal('0.010127459115'), 'buy_cnt': Decimal('1.266996360236'), 'sell_cnt_tot': Decimal('1.266382000000'), 'hold_cnt': Decimal('0.000614360236'), 'pocket_cnt': Decimal('0.000614360236'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.578536500000'), 'prc_curr': Decimal('1.619441250000'), 'prc_high': Decimal('1.646500000000'), 'prc_low': Decimal('1.541400000000'), 'prc_chg_pct': Decimal('2.591300000000'), 'prc_chg_pct_high': Decimal('3.165200000000'), 'prc_chg_pct_low': Decimal('-2.352600000000'), 'prc_chg_pct_drop': Decimal('-0.570000000000'), 'prc_sell_avg': Decimal('1.619441250000'), 'val_curr': Decimal('0.000994920309'), 'val_tot': Decimal('0.000994920309'), 'gain_loss_amt_est': Decimal('0.048479715230'), 'gain_loss_amt_est_high': Decimal('0.063303572644'), 'gain_loss_amt_est_low': Decimal('-0.047051810332'), 'gain_loss_amt': Decimal('0.051826169367'), 'gain_loss_amt_net': Decimal('0.052821089675'), 'gain_loss_pct_est': Decimal('2.424000000000'), 'gain_loss_pct_est_high': Decimal('3.165200000000'), 'gain_loss_pct_est_low': Decimal('-2.352600000000'), 'gain_loss_pct': Decimal('2.591300000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30985, 'bo_uuid': '55b9dcc8-492c-4e2b-81e8-7991cba05a23', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 22, 9, 23), 'check_last_dttm': datetime.datetime(2024, 11, 29, 2, 35, 20), 'add_dttm': datetime.datetime(2024, 11, 28, 22, 9, 23), 'upd_dttm': datetime.datetime(2024, 11, 28, 22, 9, 23), 'del_dttm': datetime.datetime(2024, 11, 28, 22, 9, 23), 'dlm': datetime.datetime(2024, 11, 29, 2, 37, 21)},
		{'pos_id': 31314, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 21, 50, 14), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 7, 35, 8), 'age_mins': 284, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.563558353765'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.051537740235'), 'fees_cnt_tot': Decimal('0.101413051956'), 'buy_cnt': Decimal('12.716805640158'), 'sell_cnt_tot': Decimal('12.709677000000'), 'hold_cnt': Decimal('0.007128640158'), 'pocket_cnt': Decimal('0.007128640158'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.572722000000'), 'prc_curr': Decimal('1.617945000000'), 'prc_high': Decimal('1.646500000000'), 'prc_low': Decimal('1.541400000000'), 'prc_chg_pct': Decimal('2.875500000000'), 'prc_chg_pct_high': Decimal('3.546600000000'), 'prc_chg_pct_low': Decimal('-1.991600000000'), 'prc_chg_pct_drop': Decimal('-0.670000000000'), 'prc_sell_avg': Decimal('1.617945000000'), 'val_curr': Decimal('0.011533747700'), 'val_tot': Decimal('0.011533747700'), 'gain_loss_amt_est': Decimal('0.560531359007'), 'gain_loss_amt_est_high': Decimal('0.709317984997'), 'gain_loss_amt_est_low': Decimal('-0.398315786260'), 'gain_loss_amt': Decimal('0.575092101465'), 'gain_loss_amt_net': Decimal('0.586625849166'), 'gain_loss_pct_est': Decimal('2.802700000000'), 'gain_loss_pct_est_high': Decimal('3.546600000000'), 'gain_loss_pct_est_low': Decimal('-1.991600000000'), 'gain_loss_pct': Decimal('2.875500000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30968, 'bo_uuid': '32cd339d-eddd-4e25-80af-dea520a3c5a7', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 21, 51, 15), 'check_last_dttm': datetime.datetime(2024, 11, 29, 2, 35, 6), 'add_dttm': datetime.datetime(2024, 11, 28, 21, 51, 15), 'upd_dttm': datetime.datetime(2024, 11, 28, 21, 51, 15), 'del_dttm': datetime.datetime(2024, 11, 28, 21, 51, 15), 'dlm': datetime.datetime(2024, 11, 29, 2, 35, 40)},
		{'pos_id': 31313, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 21, 44, 19), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 7, 34, 52), 'age_mins': 290, 'tot_out_cnt': Decimal('10.000000000000'), 'tot_in_cnt': Decimal('10.230169661838'), 'buy_fees_cnt': Decimal('0.024937655860'), 'sell_fees_cnt_tot': Decimal('0.025639522962'), 'fees_cnt_tot': Decimal('0.050577178822'), 'buy_cnt': Decimal('6.336189000914'), 'sell_cnt_tot': Decimal('6.333092000000'), 'hold_cnt': Decimal('0.003097000914'), 'pocket_cnt': Decimal('0.003097000914'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.578235750000'), 'prc_curr': Decimal('1.615351500000'), 'prc_high': Decimal('1.646500000000'), 'prc_low': Decimal('1.541400000000'), 'prc_chg_pct': Decimal('2.351700000000'), 'prc_chg_pct_high': Decimal('3.184800000000'), 'prc_chg_pct_low': Decimal('-2.334000000000'), 'prc_chg_pct_drop': Decimal('-0.830000000000'), 'prc_sell_avg': Decimal('1.615351500000'), 'val_curr': Decimal('0.005002745072'), 'val_tot': Decimal('0.005002745072'), 'gain_loss_amt_est': Decimal('0.244350376678'), 'gain_loss_amt_est_high': Decimal('0.318483787988'), 'gain_loss_amt_est_low': Decimal('-0.233398273991'), 'gain_loss_amt': Decimal('0.235172406910'), 'gain_loss_amt_net': Decimal('0.240175151982'), 'gain_loss_pct_est': Decimal('2.443500000000'), 'gain_loss_pct_est_high': Decimal('3.184800000000'), 'gain_loss_pct_est_low': Decimal('-2.334000000000'), 'gain_loss_pct': Decimal('2.351700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30967, 'bo_uuid': 'de6b12af-5454-416c-915a-967a58a3c0ee', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 21, 44, 56), 'check_last_dttm': datetime.datetime(2024, 11, 29, 2, 34, 51), 'add_dttm': datetime.datetime(2024, 11, 28, 21, 44, 56), 'upd_dttm': datetime.datetime(2024, 11, 28, 21, 44, 56), 'del_dttm': datetime.datetime(2024, 11, 28, 21, 44, 56), 'dlm': datetime.datetime(2024, 11, 29, 2, 35, 34)},
		{'pos_id': 31309, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 21, 38, 25), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 7, 34, 35), 'age_mins': 296, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.137360241159'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015381855241'), 'fees_cnt_tot': Decimal('0.030344448757'), 'buy_cnt': Decimal('3.803162868707'), 'sell_cnt_tot': Decimal('3.801274000000'), 'hold_cnt': Decimal('0.001888868707'), 'pocket_cnt': Decimal('0.001888868707'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.577634250000'), 'prc_curr': Decimal('1.614553500000'), 'prc_high': Decimal('1.646500000000'), 'prc_low': Decimal('1.541400000000'), 'prc_chg_pct': Decimal('2.340200000000'), 'prc_chg_pct_high': Decimal('3.224200000000'), 'prc_chg_pct_low': Decimal('-2.296700000000'), 'prc_chg_pct_drop': Decimal('-0.880000000000'), 'prc_sell_avg': Decimal('1.614553500000'), 'val_curr': Decimal('0.003049679582'), 'val_tot': Decimal('0.003049679582'), 'gain_loss_amt_est': Decimal('0.148953726125'), 'gain_loss_amt_est_high': Decimal('0.193450731689'), 'gain_loss_amt_est_low': Decimal('-0.137804754175'), 'gain_loss_amt': Decimal('0.140409920741'), 'gain_loss_amt_net': Decimal('0.143459600323'), 'gain_loss_pct_est': Decimal('2.482600000000'), 'gain_loss_pct_est_high': Decimal('3.224200000000'), 'gain_loss_pct_est_low': Decimal('-2.296700000000'), 'gain_loss_pct': Decimal('2.340200000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'bb_bo', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30963, 'bo_uuid': 'bb29b106-77b4-46b0-a721-18993a3d8fd8', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 21, 39, 57), 'check_last_dttm': datetime.datetime(2024, 11, 29, 2, 34, 33), 'add_dttm': datetime.datetime(2024, 11, 28, 21, 39, 57), 'upd_dttm': datetime.datetime(2024, 11, 28, 21, 39, 57), 'del_dttm': datetime.datetime(2024, 11, 28, 21, 39, 57), 'dlm': datetime.datetime(2024, 11, 29, 2, 35, 28)},
		{'pos_id': 31293, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 21, 31, 54), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 7, 34, 20), 'age_mins': 303, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.520969961690'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.051431002410'), 'fees_cnt_tot': Decimal('0.101306314131'), 'buy_cnt': Decimal('12.701036370753'), 'sell_cnt_tot': Decimal('12.694240000000'), 'hold_cnt': Decimal('0.006796370753'), 'pocket_cnt': Decimal('0.006796370753'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.574674650000'), 'prc_curr': Decimal('1.616557590000'), 'prc_high': Decimal('1.646500000000'), 'prc_low': Decimal('1.541400000000'), 'prc_chg_pct': Decimal('2.659800000000'), 'prc_chg_pct_high': Decimal('3.418200000000'), 'prc_chg_pct_low': Decimal('-2.113100000000'), 'prc_chg_pct_drop': Decimal('-0.760000000000'), 'prc_sell_avg': Decimal('1.616557590000'), 'val_curr': Decimal('0.010986724725'), 'val_tot': Decimal('0.010986724725'), 'gain_loss_amt_est': Decimal('0.535035604233'), 'gain_loss_amt_est_high': Decimal('0.683637729771'), 'gain_loss_amt_est_low': Decimal('-0.422622538121'), 'gain_loss_amt': Decimal('0.531956686415'), 'gain_loss_amt_net': Decimal('0.542943411140'), 'gain_loss_pct_est': Decimal('2.675200000000'), 'gain_loss_pct_est_high': Decimal('3.418200000000'), 'gain_loss_pct_est_low': Decimal('-2.113100000000'), 'gain_loss_pct': Decimal('2.659800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30947, 'bo_uuid': '8e62d20e-f248-41a3-9b70-90495761954d', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 21, 34, 16), 'check_last_dttm': datetime.datetime(2024, 11, 29, 2, 34, 19), 'add_dttm': datetime.datetime(2024, 11, 28, 21, 34, 16), 'upd_dttm': datetime.datetime(2024, 11, 28, 21, 34, 16), 'del_dttm': datetime.datetime(2024, 11, 28, 21, 34, 16), 'dlm': datetime.datetime(2024, 11, 29, 2, 35, 22)},
		{'pos_id': 31285, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 21, 13, 23), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 7, 34, 5), 'age_mins': 321, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.456483882835'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.051269383165'), 'fees_cnt_tot': Decimal('0.101144694886'), 'buy_cnt': Decimal('12.666745833828'), 'sell_cnt_tot': Decimal('12.660670000000'), 'hold_cnt': Decimal('0.006075833828'), 'pocket_cnt': Decimal('0.006075833828'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.578937500000'), 'prc_curr': Decimal('1.615750500000'), 'prc_high': Decimal('1.646500000000'), 'prc_low': Decimal('1.541400000000'), 'prc_chg_pct': Decimal('2.331500000000'), 'prc_chg_pct_high': Decimal('3.139000000000'), 'prc_chg_pct_low': Decimal('-2.377400000000'), 'prc_chg_pct_drop': Decimal('-0.810000000000'), 'prc_sell_avg': Decimal('1.615750500000'), 'val_curr': Decimal('0.009817031546'), 'val_tot': Decimal('0.009817031546'), 'gain_loss_amt_est': Decimal('0.479594664133'), 'gain_loss_amt_est_high': Decimal('0.627795590389'), 'gain_loss_amt_est_low': Decimal('-0.475477971738'), 'gain_loss_amt': Decimal('0.466300914381'), 'gain_loss_amt_net': Decimal('0.476117945926'), 'gain_loss_pct_est': Decimal('2.398000000000'), 'gain_loss_pct_est_high': Decimal('3.139000000000'), 'gain_loss_pct_est_low': Decimal('-2.377400000000'), 'gain_loss_pct': Decimal('2.331500000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30934, 'bo_uuid': '2a22fd71-9bb7-43a8-9ebf-9f90ec1c1c85', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 21, 31, 24), 'check_last_dttm': datetime.datetime(2024, 11, 29, 2, 34, 3), 'add_dttm': datetime.datetime(2024, 11, 28, 21, 31, 24), 'upd_dttm': datetime.datetime(2024, 11, 28, 21, 31, 24), 'del_dttm': datetime.datetime(2024, 11, 28, 21, 31, 24), 'dlm': datetime.datetime(2024, 11, 29, 2, 35, 16)},
		{'pos_id': 30979, 'symb': 'USDC', 'prod_id': 'JTO-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 11, 13, 38), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 4, 12, 1), 'age_mins': 719, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.204579675000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015550325000'), 'fees_cnt_tot': Decimal('0.030512918516'), 'buy_cnt': Decimal('1.712901375639'), 'sell_cnt_tot': Decimal('1.700000000000'), 'hold_cnt': Decimal('0.012901375639'), 'pocket_cnt': Decimal('0.012901375639'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('3.502828640000'), 'prc_curr': Decimal('3.649752750000'), 'prc_high': Decimal('3.721000000000'), 'prc_low': Decimal('3.451500000000'), 'prc_chg_pct': Decimal('4.194400000000'), 'prc_chg_pct_high': Decimal('5.594700000000'), 'prc_chg_pct_low': Decimal('-1.465300000000'), 'prc_chg_pct_drop': Decimal('-1.400000000000'), 'prc_sell_avg': Decimal('3.649752750000'), 'val_curr': Decimal('0.047086831217'), 'val_tot': Decimal('0.047086831217'), 'gain_loss_amt_est': Decimal('0.268020003876'), 'gain_loss_amt_est_high': Decimal('0.335679608214'), 'gain_loss_amt_est_low': Decimal('-0.087920901982'), 'gain_loss_amt': Decimal('0.251666506217'), 'gain_loss_amt_net': Decimal('0.298753337434'), 'gain_loss_pct_est': Decimal('4.467000000000'), 'gain_loss_pct_est_high': Decimal('5.594700000000'), 'gain_loss_pct_est_low': Decimal('-1.465300000000'), 'gain_loss_pct': Decimal('4.194400000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30633, 'bo_uuid': '1a7ed598-767f-478e-b9b3-a314381bacc0', 'buy_curr_symb': 'JTO', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'JTO', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'JTO', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('50000000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 11, 14, 39), 'check_last_dttm': datetime.datetime(2024, 11, 28, 23, 12), 'add_dttm': datetime.datetime(2024, 11, 28, 11, 14, 39), 'upd_dttm': datetime.datetime(2024, 11, 28, 11, 14, 39), 'del_dttm': datetime.datetime(2024, 11, 28, 11, 14, 39), 'dlm': datetime.datetime(2024, 11, 28, 23, 13, 28)},
		{'pos_id': 30978, 'symb': 'USDC', 'prod_id': 'JTO-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 11, 8, 4), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 1, 44, 48), 'age_mins': 577, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.030087000000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015113000000'), 'fees_cnt_tot': Decimal('0.030075593516'), 'buy_cnt': Decimal('1.720332683669'), 'sell_cnt_tot': Decimal('1.700000000000'), 'hold_cnt': Decimal('0.020332683669'), 'pocket_cnt': Decimal('0.020332683669'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('3.487697500000'), 'prc_curr': Decimal('3.547110000000'), 'prc_high': Decimal('3.597600000000'), 'prc_low': Decimal('3.451500000000'), 'prc_chg_pct': Decimal('1.703500000000'), 'prc_chg_pct_high': Decimal('3.007800000000'), 'prc_chg_pct_low': Decimal('-1.037900000000'), 'prc_chg_pct_drop': Decimal('-1.300000000000'), 'prc_sell_avg': Decimal('3.547110000000'), 'val_curr': Decimal('0.072122265569'), 'val_tot': Decimal('0.072122265569'), 'gain_loss_amt_est': Decimal('0.115782690443'), 'gain_loss_amt_est_high': Decimal('0.180467199349'), 'gain_loss_amt_est_low': Decimal('-0.062271742316'), 'gain_loss_amt': Decimal('0.102209265569'), 'gain_loss_amt_net': Decimal('0.174331531138'), 'gain_loss_pct_est': Decimal('1.929700000000'), 'gain_loss_pct_est_high': Decimal('3.007800000000'), 'gain_loss_pct_est_low': Decimal('-1.037900000000'), 'gain_loss_pct': Decimal('1.703500000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30632, 'bo_uuid': '40ab490f-0b27-4812-8b7a-2ddcc2b5e1fd', 'buy_curr_symb': 'JTO', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'JTO', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'JTO', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('50000000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 11, 8, 14), 'check_last_dttm': datetime.datetime(2024, 11, 28, 20, 44, 46), 'add_dttm': datetime.datetime(2024, 11, 28, 11, 8, 14), 'upd_dttm': datetime.datetime(2024, 11, 28, 11, 8, 14), 'del_dttm': datetime.datetime(2024, 11, 28, 11, 8, 14), 'dlm': datetime.datetime(2024, 11, 28, 21, 27, 55)},
		{'pos_id': 30301, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 15, 32, 37), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 0, 25, 34), 'age_mins': 1673, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.076676112507'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015229764693'), 'fees_cnt_tot': Decimal('0.030192358209'), 'buy_cnt': Decimal('3.945831623473'), 'sell_cnt_tot': Decimal('3.944513000000'), 'hold_cnt': Decimal('0.001318623473'), 'pocket_cnt': Decimal('0.001318623473'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.520592000000'), 'prc_curr': Decimal('1.540539000000'), 'prc_high': Decimal('1.574300000000'), 'prc_low': Decimal('1.433600000000'), 'prc_chg_pct': Decimal('1.311800000000'), 'prc_chg_pct_high': Decimal('3.492600000000'), 'prc_chg_pct_low': Decimal('-5.720900000000'), 'prc_chg_pct_drop': Decimal('-2.180000000000'), 'prc_sell_avg': Decimal('1.540539000000'), 'val_curr': Decimal('0.002031390886'), 'val_tot': Decimal('0.002031390886'), 'gain_loss_amt_est': Decimal('0.100255689889'), 'gain_loss_amt_est_high': Decimal('0.209555225859'), 'gain_loss_amt_est_low': Decimal('-0.343255784589'), 'gain_loss_amt': Decimal('0.078707503393'), 'gain_loss_amt_net': Decimal('0.080738894280'), 'gain_loss_pct_est': Decimal('1.670900000000'), 'gain_loss_pct_est_high': Decimal('3.492600000000'), 'gain_loss_pct_est_low': Decimal('-5.720900000000'), 'gain_loss_pct': Decimal('1.311800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29955, 'bo_uuid': '72804003-a91a-4e95-8ca9-afed8d00a57f', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 15, 34, 4), 'check_last_dttm': datetime.datetime(2024, 11, 28, 19, 25, 32), 'add_dttm': datetime.datetime(2024, 11, 27, 15, 34, 4), 'upd_dttm': datetime.datetime(2024, 11, 27, 15, 34, 4), 'del_dttm': datetime.datetime(2024, 11, 27, 15, 34, 4), 'dlm': datetime.datetime(2024, 11, 28, 21, 9, 54)},
		{'pos_id': 30288, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 15, 26, 18), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 0, 25, 19), 'age_mins': 1679, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.300160643403'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.050877595598'), 'fees_cnt_tot': Decimal('0.100752907319'), 'buy_cnt': Decimal('13.190165083160'), 'sell_cnt_tot': Decimal('13.184994000000'), 'hold_cnt': Decimal('0.005171083160'), 'pocket_cnt': Decimal('0.005171083160'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.516281250000'), 'prc_curr': Decimal('1.539641250000'), 'prc_high': Decimal('1.574300000000'), 'prc_low': Decimal('1.433600000000'), 'prc_chg_pct': Decimal('1.540600000000'), 'prc_chg_pct_high': Decimal('3.786800000000'), 'prc_chg_pct_low': Decimal('-5.452900000000'), 'prc_chg_pct_drop': Decimal('-2.250000000000'), 'prc_sell_avg': Decimal('1.539641250000'), 'val_curr': Decimal('0.007961612940'), 'val_tot': Decimal('0.007961612940'), 'gain_loss_amt_est': Decimal('0.391995218565'), 'gain_loss_amt_est_high': Decimal('0.757362791369'), 'gain_loss_amt_est_low': Decimal('-1.090579336782'), 'gain_loss_amt': Decimal('0.308122256343'), 'gain_loss_amt_net': Decimal('0.316083869284'), 'gain_loss_pct_est': Decimal('1.960000000000'), 'gain_loss_pct_est_high': Decimal('3.786800000000'), 'gain_loss_pct_est_low': Decimal('-5.452900000000'), 'gain_loss_pct': Decimal('1.540600000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29942, 'bo_uuid': 'c99ad253-4d20-4f18-bd37-8b0d78015898', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 15, 28, 10), 'check_last_dttm': datetime.datetime(2024, 11, 28, 19, 25, 18), 'add_dttm': datetime.datetime(2024, 11, 27, 15, 28, 10), 'upd_dttm': datetime.datetime(2024, 11, 27, 15, 28, 10), 'del_dttm': datetime.datetime(2024, 11, 27, 15, 28, 10), 'dlm': datetime.datetime(2024, 11, 28, 21, 9, 47)},
		{'pos_id': 27915, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 3, 17, 18), 'pos_end_dttm': datetime.datetime(2024, 11, 29, 0, 25, 4), 'age_mins': 5288, 'tot_out_cnt': Decimal('12.000000000000'), 'tot_in_cnt': Decimal('12.332620347275'), 'buy_fees_cnt': Decimal('0.029925187032'), 'sell_fees_cnt_tot': Decimal('0.030908822926'), 'fees_cnt_tot': Decimal('0.060834009958'), 'buy_cnt': Decimal('8.005231215401'), 'sell_cnt_tot': Decimal('8.000213000000'), 'hold_cnt': Decimal('0.005018215401'), 'pocket_cnt': Decimal('0.005018215401'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.499019790000'), 'prc_curr': Decimal('1.541536500000'), 'prc_high': Decimal('1.574300000000'), 'prc_low': Decimal('1.284800000000'), 'prc_chg_pct': Decimal('2.836300000000'), 'prc_chg_pct_high': Decimal('4.981900000000'), 'prc_chg_pct_low': Decimal('-14.290700000000'), 'prc_chg_pct_drop': Decimal('-2.150000000000'), 'prc_sell_avg': Decimal('1.541536500000'), 'val_curr': Decimal('0.007735762206'), 'val_tot': Decimal('0.007735762206'), 'gain_loss_amt_est': Decimal('0.376087459010'), 'gain_loss_amt_est_high': Decimal('0.597832363677'), 'gain_loss_amt_est_low': Decimal('-1.714878934453'), 'gain_loss_amt': Decimal('0.340356109481'), 'gain_loss_amt_net': Decimal('0.348091871686'), 'gain_loss_pct_est': Decimal('3.134100000000'), 'gain_loss_pct_est_high': Decimal('4.981900000000'), 'gain_loss_pct_est_low': Decimal('-14.290700000000'), 'gain_loss_pct': Decimal('2.836300000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 27568, 'bo_uuid': 'f8584c55-b152-4e43-9637-60df5e369707', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 3, 18, 8), 'check_last_dttm': datetime.datetime(2024, 11, 28, 19, 25, 3), 'add_dttm': datetime.datetime(2024, 11, 25, 3, 18, 8), 'upd_dttm': datetime.datetime(2024, 11, 25, 3, 18, 8), 'del_dttm': datetime.datetime(2024, 11, 25, 3, 18, 8), 'dlm': datetime.datetime(2024, 11, 28, 21, 9, 41)},
		{'pos_id': 28395, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 11, 56, 6), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 23, 42), 'age_mins': 4726, 'tot_out_cnt': Decimal('18.000000000000'), 'tot_in_cnt': Decimal('18.556140252227'), 'buy_fees_cnt': Decimal('0.044887780549'), 'sell_fees_cnt_tot': Decimal('0.046506617174'), 'fees_cnt_tot': Decimal('0.091394397723'), 'buy_cnt': Decimal('12.221844816181'), 'sell_cnt_tot': Decimal('12.213674000000'), 'hold_cnt': Decimal('0.008170816181'), 'pocket_cnt': Decimal('0.008170816181'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.472772750000'), 'prc_curr': Decimal('1.519292250000'), 'prc_high': Decimal('1.544600000000'), 'prc_low': Decimal('1.284800000000'), 'prc_chg_pct': Decimal('3.158600000000'), 'prc_chg_pct_high': Decimal('4.299900000000'), 'prc_chg_pct_low': Decimal('-12.763200000000'), 'prc_chg_pct_drop': Decimal('-1.140000000000'), 'prc_sell_avg': Decimal('1.519292250000'), 'val_curr': Decimal('0.012413857700'), 'val_tot': Decimal('0.012413857700'), 'gain_loss_amt_est': Decimal('0.601647810227'), 'gain_loss_amt_est_high': Decimal('0.773975822136'), 'gain_loss_amt_est_low': Decimal('-2.297373780171'), 'gain_loss_amt': Decimal('0.568554109927'), 'gain_loss_amt_net': Decimal('0.580967967627'), 'gain_loss_pct_est': Decimal('3.342500000000'), 'gain_loss_pct_est_high': Decimal('4.299900000000'), 'gain_loss_pct_est_low': Decimal('-12.763200000000'), 'gain_loss_pct': Decimal('3.158600000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28048, 'bo_uuid': 'd233452e-7f66-4865-b69d-27cd4a7412f5', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 11, 57, 25), 'check_last_dttm': datetime.datetime(2024, 11, 28, 18, 41, 59), 'add_dttm': datetime.datetime(2024, 11, 25, 11, 57, 25), 'upd_dttm': datetime.datetime(2024, 11, 25, 11, 57, 25), 'del_dttm': datetime.datetime(2024, 11, 25, 11, 57, 25), 'dlm': datetime.datetime(2024, 11, 28, 20, 35, 46)},
		{'pos_id': 27911, 'symb': 'USDC', 'prod_id': 'XRP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 3, 5, 50), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 23, 31, 59), 'age_mins': 5247, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.455221684240'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.051266219760'), 'fees_cnt_tot': Decimal('0.101141531481'), 'buy_cnt': Decimal('13.425386735047'), 'sell_cnt_tot': Decimal('13.418720000000'), 'hold_cnt': Decimal('0.006666735047'), 'pocket_cnt': Decimal('0.006666735047'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.489715000000'), 'prc_curr': Decimal('1.524379500000'), 'prc_high': Decimal('1.544600000000'), 'prc_low': Decimal('1.284800000000'), 'prc_chg_pct': Decimal('2.326900000000'), 'prc_chg_pct_high': Decimal('3.113700000000'), 'prc_chg_pct_low': Decimal('-13.755300000000'), 'prc_chg_pct_drop': Decimal('-0.790000000000'), 'prc_sell_avg': Decimal('1.524379500000'), 'val_curr': Decimal('0.010162634238'), 'val_tot': Decimal('0.010162634238'), 'gain_loss_amt_est': Decimal('0.496537928396'), 'gain_loss_amt_est_high': Decimal('0.622736563706'), 'gain_loss_amt_est_low': Decimal('-2.751063122812'), 'gain_loss_amt': Decimal('0.465384318478'), 'gain_loss_amt_net': Decimal('0.475546952715'), 'gain_loss_pct_est': Decimal('2.482700000000'), 'gain_loss_pct_est_high': Decimal('3.113700000000'), 'gain_loss_pct_est_low': Decimal('-13.755300000000'), 'gain_loss_pct': Decimal('2.326900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 27564, 'bo_uuid': '04008695-bded-4d26-afd7-27285244bc06', 'buy_curr_symb': 'XRP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XRP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XRP', 'base_size_incr': Decimal('0.000001000000'), 'base_size_min': Decimal('0.000001000000'), 'base_size_max': Decimal('11996772.840729220000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 3, 6, 48), 'check_last_dttm': datetime.datetime(2024, 11, 28, 18, 31, 58), 'add_dttm': datetime.datetime(2024, 11, 25, 3, 6, 48), 'upd_dttm': datetime.datetime(2024, 11, 25, 3, 6, 48), 'del_dttm': datetime.datetime(2024, 11, 25, 3, 6, 48), 'dlm': datetime.datetime(2024, 11, 28, 20, 14, 38)},
		{'pos_id': 30141, 'symb': 'USDC', 'prod_id': 'DOT-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 11, 29, 57), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 23, 4, 28), 'age_mins': 1835, 'tot_out_cnt': Decimal('15.000000000000'), 'tot_in_cnt': Decimal('15.349875827864'), 'buy_fees_cnt': Decimal('0.037406483791'), 'sell_fees_cnt_tot': Decimal('0.038470866737'), 'fees_cnt_tot': Decimal('0.075877350528'), 'buy_cnt': Decimal('1.789998028019'), 'sell_cnt_tot': Decimal('1.789134600000'), 'hold_cnt': Decimal('0.000863428019'), 'pocket_cnt': Decimal('0.000863428019'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('8.379897500000'), 'prc_curr': Decimal('8.579497500000'), 'prc_high': Decimal('8.754000000000'), 'prc_low': Decimal('8.020000000000'), 'prc_chg_pct': Decimal('2.381900000000'), 'prc_chg_pct_high': Decimal('3.569300000000'), 'prc_chg_pct_low': Decimal('-4.294800000000'), 'prc_chg_pct_drop': Decimal('-1.190000000000'), 'prc_sell_avg': Decimal('8.579497500000'), 'val_curr': Decimal('0.007407778530'), 'val_tot': Decimal('0.007407778530'), 'gain_loss_amt_est': Decimal('0.361763076459'), 'gain_loss_amt_est_high': Decimal('0.535392885177'), 'gain_loss_amt_est_low': Decimal('-0.644215815288'), 'gain_loss_amt': Decimal('0.357283606394'), 'gain_loss_amt_net': Decimal('0.364691384925'), 'gain_loss_pct_est': Decimal('2.411800000000'), 'gain_loss_pct_est_high': Decimal('3.569300000000'), 'gain_loss_pct_est_low': Decimal('-4.294800000000'), 'gain_loss_pct': Decimal('2.381900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29795, 'bo_uuid': 'b0b11f86-c29a-45cf-9416-ac8d07f4656e', 'buy_curr_symb': 'DOT', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOT', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOT', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('1732900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 11, 31), 'check_last_dttm': datetime.datetime(2024, 11, 28, 18, 4, 27), 'add_dttm': datetime.datetime(2024, 11, 27, 11, 31), 'upd_dttm': datetime.datetime(2024, 11, 27, 11, 31), 'del_dttm': datetime.datetime(2024, 11, 27, 11, 31), 'dlm': datetime.datetime(2024, 11, 28, 18, 5, 21)},
		{'pos_id': 29919, 'symb': 'USDC', 'prod_id': 'DOT-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 6, 46, 9), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 23, 4, 1), 'age_mins': 2118, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.121505171260'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.012835852560'), 'fees_cnt_tot': Decimal('0.025304680490'), 'buy_cnt': Decimal('0.598312280719'), 'sell_cnt_tot': Decimal('0.597989870000'), 'hold_cnt': Decimal('0.000322410719'), 'pocket_cnt': Decimal('0.000322410719'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('8.356840000000'), 'prc_curr': Decimal('8.564535000000'), 'prc_high': Decimal('8.754000000000'), 'prc_low': Decimal('8.020000000000'), 'prc_chg_pct': Decimal('2.485300000000'), 'prc_chg_pct_high': Decimal('3.855000000000'), 'prc_chg_pct_low': Decimal('-4.030700000000'), 'prc_chg_pct_drop': Decimal('-1.370000000000'), 'prc_sell_avg': Decimal('8.564535000000'), 'val_curr': Decimal('0.002761297887'), 'val_tot': Decimal('0.002761297887'), 'gain_loss_amt_est': Decimal('0.134715993130'), 'gain_loss_amt_est_high': Decimal('0.192752284360'), 'gain_loss_amt_est_low': Decimal('-0.201535508634'), 'gain_loss_amt': Decimal('0.124266469147'), 'gain_loss_amt_net': Decimal('0.127027767035'), 'gain_loss_pct_est': Decimal('2.694300000000'), 'gain_loss_pct_est_high': Decimal('3.855000000000'), 'gain_loss_pct_est_low': Decimal('-4.030700000000'), 'gain_loss_pct': Decimal('2.485300000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29573, 'bo_uuid': 'cb31ce1f-0598-46b2-9c6f-6ab4e3ce006d', 'buy_curr_symb': 'DOT', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOT', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOT', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('1732900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 6, 47, 14), 'check_last_dttm': datetime.datetime(2024, 11, 28, 18, 3, 59), 'add_dttm': datetime.datetime(2024, 11, 27, 6, 47, 14), 'upd_dttm': datetime.datetime(2024, 11, 27, 6, 47, 14), 'del_dttm': datetime.datetime(2024, 11, 27, 6, 47, 14), 'dlm': datetime.datetime(2024, 11, 28, 18, 5, 6)},
		{'pos_id': 31069, 'symb': 'USDC', 'prod_id': 'RNDR-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 15, 2, 31), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 22, 56, 24), 'age_mins': 174, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.038534095000'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.012627905000'), 'fees_cnt_tot': Decimal('0.025096732930'), 'buy_cnt': Decimal('0.586968633071'), 'sell_cnt_tot': Decimal('0.580000000000'), 'hold_cnt': Decimal('0.006968633071'), 'pocket_cnt': Decimal('0.006968633071'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('8.518342750000'), 'prc_curr': Decimal('8.687127750000'), 'prc_high': Decimal('8.784900000000'), 'prc_low': Decimal('8.406500000000'), 'prc_chg_pct': Decimal('1.981400000000'), 'prc_chg_pct_high': Decimal('3.071700000000'), 'prc_chg_pct_low': Decimal('-1.313000000000'), 'prc_chg_pct_drop': Decimal('-1.090000000000'), 'prc_sell_avg': Decimal('8.687127750000'), 'val_curr': Decimal('0.060537405731'), 'val_tot': Decimal('0.060537405731'), 'gain_loss_amt_est': Decimal('0.112438097185'), 'gain_loss_amt_est_high': Decimal('0.153584598363'), 'gain_loss_amt_est_low': Decimal('-0.065648186089'), 'gain_loss_amt': Decimal('0.099071500731'), 'gain_loss_amt_net': Decimal('0.159608906461'), 'gain_loss_pct_est': Decimal('2.248800000000'), 'gain_loss_pct_est_high': Decimal('3.071700000000'), 'gain_loss_pct_est_low': Decimal('-1.313000000000'), 'gain_loss_pct': Decimal('1.981400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30723, 'bo_uuid': '6c650d30-e093-448e-93e1-e3ca7058b45d', 'buy_curr_symb': 'RNDR', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'RNDR', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'RNDR', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('8235000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 15, 3, 33), 'check_last_dttm': datetime.datetime(2024, 11, 28, 17, 56, 22), 'add_dttm': datetime.datetime(2024, 11, 28, 15, 3, 33), 'upd_dttm': datetime.datetime(2024, 11, 28, 15, 3, 33), 'del_dttm': datetime.datetime(2024, 11, 28, 15, 3, 33), 'dlm': datetime.datetime(2024, 11, 28, 17, 57, 7)},
		{'pos_id': 31049, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 14, 3, 20), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 19, 44, 19), 'age_mins': 41, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.265815000000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.008185000000'), 'fees_cnt_tot': Decimal('0.015666296758'), 'buy_cnt': Decimal('100.761598142762'), 'sell_cnt_tot': Decimal('100.000000000000'), 'hold_cnt': Decimal('0.761598142762'), 'pocket_cnt': Decimal('0.761598142762'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.029773250000'), 'prc_curr': Decimal('0.032658150000'), 'prc_high': Decimal('0.034499000000'), 'prc_low': Decimal('0.029699000000'), 'prc_chg_pct': Decimal('9.689600000000'), 'prc_chg_pct_high': Decimal('14.979000000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-5.290000000000'), 'prc_sell_avg': Decimal('0.032658150000'), 'val_curr': Decimal('0.024872386386'), 'val_tot': Decimal('0.024872386386'), 'gain_loss_amt_est': Decimal('0.273845085256'), 'gain_loss_amt_est_high': Decimal('0.449371789221'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.290687386386'), 'gain_loss_amt_net': Decimal('0.315559772772'), 'gain_loss_pct_est': Decimal('9.128200000000'), 'gain_loss_pct_est_high': Decimal('14.979100000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('9.689600000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30703, 'bo_uuid': '901a5e43-67b4-4008-ad4a-657eca3f7d32', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 14, 4, 28), 'check_last_dttm': datetime.datetime(2024, 11, 28, 14, 44, 18), 'add_dttm': datetime.datetime(2024, 11, 28, 14, 4, 28), 'upd_dttm': datetime.datetime(2024, 11, 28, 14, 4, 28), 'del_dttm': datetime.datetime(2024, 11, 28, 14, 4, 28), 'dlm': datetime.datetime(2024, 11, 28, 14, 44, 42)},
		{'pos_id': 31038, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 13, 57, 19), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 19, 44, 5), 'age_mins': 47, 'tot_out_cnt': Decimal('10.000000000000'), 'tot_in_cnt': Decimal('11.171816460000'), 'buy_fees_cnt': Decimal('0.024937655860'), 'sell_fees_cnt_tot': Decimal('0.027999540000'), 'fees_cnt_tot': Decimal('0.052937195860'), 'buy_cnt': Decimal('342.903483813670'), 'sell_cnt_tot': Decimal('342.000000000000'), 'hold_cnt': Decimal('0.903483813670'), 'pocket_cnt': Decimal('0.903483813670'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.029162730000'), 'prc_curr': Decimal('0.032666130000'), 'prc_high': Decimal('0.034499000000'), 'prc_low': Decimal('0.029090000000'), 'prc_chg_pct': Decimal('12.013300000000'), 'prc_chg_pct_high': Decimal('17.386100000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-5.370000000000'), 'prc_sell_avg': Decimal('0.032666130000'), 'val_curr': Decimal('0.029513319710'), 'val_tot': Decimal('0.029513319710'), 'gain_loss_amt_est': Decimal('1.141277092590'), 'gain_loss_amt_est_high': Decimal('1.738614961393'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('1.201329779710'), 'gain_loss_amt_net': Decimal('1.230843099420'), 'gain_loss_pct_est': Decimal('11.412800000000'), 'gain_loss_pct_est_high': Decimal('17.386100000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('12.013300000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30692, 'bo_uuid': '4ef3015c-6650-4efd-8b82-55acd2fe5d38', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 13, 58, 14), 'check_last_dttm': datetime.datetime(2024, 11, 28, 14, 44, 3), 'add_dttm': datetime.datetime(2024, 11, 28, 13, 58, 14), 'upd_dttm': datetime.datetime(2024, 11, 28, 13, 58, 14), 'del_dttm': datetime.datetime(2024, 11, 28, 13, 58, 14), 'dlm': datetime.datetime(2024, 11, 28, 14, 44, 37)},
		{'pos_id': 30972, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 10, 51, 43), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 19, 43, 44), 'age_mins': 232, 'tot_out_cnt': Decimal('4.000000000000'), 'tot_in_cnt': Decimal('4.036994220000'), 'buy_fees_cnt': Decimal('0.009975062344'), 'sell_fees_cnt_tot': Decimal('0.010117780000'), 'fees_cnt_tot': Decimal('0.020092842344'), 'buy_cnt': Decimal('124.809188202817'), 'sell_cnt_tot': Decimal('124.000000000000'), 'hold_cnt': Decimal('0.809188202817'), 'pocket_cnt': Decimal('0.809188202817'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.032048920000'), 'prc_curr': Decimal('0.032556410000'), 'prc_high': Decimal('0.034499000000'), 'prc_low': Decimal('0.023926000000'), 'prc_chg_pct': Decimal('1.583500000000'), 'prc_chg_pct_high': Decimal('6.814800000000'), 'prc_chg_pct_low': Decimal('-25.345400000000'), 'prc_chg_pct_drop': Decimal('-5.230000000000'), 'prc_sell_avg': Decimal('0.032556410000'), 'val_curr': Decimal('0.026344262898'), 'val_tot': Decimal('0.026344262898'), 'gain_loss_amt_est': Decimal('0.055175333898'), 'gain_loss_amt_est_high': Decimal('0.272592939747'), 'gain_loss_amt_est_low': Decimal('-1.013815363059'), 'gain_loss_amt': Decimal('0.063338482898'), 'gain_loss_amt_net': Decimal('0.089682745796'), 'gain_loss_pct_est': Decimal('1.379400000000'), 'gain_loss_pct_est_high': Decimal('6.814800000000'), 'gain_loss_pct_est_low': Decimal('-25.345400000000'), 'gain_loss_pct': Decimal('1.583500000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30626, 'bo_uuid': '745cd307-8b1b-4759-a4af-fa6f0826aa47', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 10, 52, 56), 'check_last_dttm': datetime.datetime(2024, 11, 28, 14, 43, 42), 'add_dttm': datetime.datetime(2024, 11, 28, 10, 52, 56), 'upd_dttm': datetime.datetime(2024, 11, 28, 10, 52, 56), 'del_dttm': datetime.datetime(2024, 11, 28, 10, 52, 56), 'dlm': datetime.datetime(2024, 11, 28, 14, 44, 26)},
		{'pos_id': 30549, 'symb': 'USDC', 'prod_id': 'RNDR-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 22, 15, 46), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 19, 28, 32), 'age_mins': 973, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.087250000000'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.012750000000'), 'fees_cnt_tot': Decimal('0.025218827930'), 'buy_cnt': Decimal('0.605312293324'), 'sell_cnt_tot': Decimal('0.600000000000'), 'hold_cnt': Decimal('0.005312293324'), 'pocket_cnt': Decimal('0.005312293324'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('8.260199000000'), 'prc_curr': Decimal('8.478750000000'), 'prc_high': Decimal('8.660000000000'), 'prc_low': Decimal('7.952900000000'), 'prc_chg_pct': Decimal('2.645800000000'), 'prc_chg_pct_high': Decimal('4.606400000000'), 'prc_chg_pct_low': Decimal('-3.720200000000'), 'prc_chg_pct_drop': Decimal('-1.960000000000'), 'prc_sell_avg': Decimal('8.478750000000'), 'val_curr': Decimal('0.045041607021'), 'val_tot': Decimal('0.045041607021'), 'gain_loss_amt_est': Decimal('0.145336086942'), 'gain_loss_amt_est_high': Decimal('0.230321932925'), 'gain_loss_amt_est_low': Decimal('-0.186011862424'), 'gain_loss_amt': Decimal('0.132291607021'), 'gain_loss_amt_net': Decimal('0.177333214042'), 'gain_loss_pct_est': Decimal('2.906700000000'), 'gain_loss_pct_est_high': Decimal('4.606400000000'), 'gain_loss_pct_est_low': Decimal('-3.720200000000'), 'gain_loss_pct': Decimal('2.645800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30203, 'bo_uuid': '2b143b58-87e8-4933-b660-f0adbca7e00c', 'buy_curr_symb': 'RNDR', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'RNDR', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'RNDR', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('8235000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 22, 16, 30), 'check_last_dttm': datetime.datetime(2024, 11, 28, 14, 28, 31), 'add_dttm': datetime.datetime(2024, 11, 27, 22, 16, 30), 'upd_dttm': datetime.datetime(2024, 11, 27, 22, 16, 30), 'del_dttm': datetime.datetime(2024, 11, 27, 22, 16, 30), 'dlm': datetime.datetime(2024, 11, 28, 14, 29, 51)},
		{'pos_id': 30534, 'symb': 'USDC', 'prod_id': 'RENDER-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 21, 42, 51), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 19, 28, 9), 'age_mins': 1006, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.089225050000'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.012754950000'), 'fees_cnt_tot': Decimal('0.025223777930'), 'buy_cnt': Decimal('0.605871133633'), 'sell_cnt_tot': Decimal('0.600000000000'), 'hold_cnt': Decimal('0.005871133633'), 'pocket_cnt': Decimal('0.005871133633'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('8.252580000000'), 'prc_curr': Decimal('8.482041750000'), 'prc_high': Decimal('8.690000000000'), 'prc_low': Decimal('7.980000000000'), 'prc_chg_pct': Decimal('2.780500000000'), 'prc_chg_pct_high': Decimal('4.815700000000'), 'prc_chg_pct_low': Decimal('-3.303000000000'), 'prc_chg_pct_drop': Decimal('-2.040000000000'), 'prc_sell_avg': Decimal('8.482041750000'), 'val_curr': Decimal('0.049799200595'), 'val_tot': Decimal('0.049799200595'), 'gain_loss_amt_est': Decimal('0.154751604950'), 'gain_loss_amt_est_high': Decimal('0.240785305925'), 'gain_loss_amt_est_low': Decimal('-0.165148353609'), 'gain_loss_amt': Decimal('0.139024250595'), 'gain_loss_amt_net': Decimal('0.188823451190'), 'gain_loss_pct_est': Decimal('3.095000000000'), 'gain_loss_pct_est_high': Decimal('4.815700000000'), 'gain_loss_pct_est_low': Decimal('-3.303000000000'), 'gain_loss_pct': Decimal('2.780500000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30188, 'bo_uuid': 'b7779363-73ae-4f15-b1ca-86c9c6bb7fc7', 'buy_curr_symb': 'RENDER', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'RENDER', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'RENDER', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('1982431.303836978700'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 21, 43, 24), 'check_last_dttm': datetime.datetime(2024, 11, 28, 14, 28, 7), 'add_dttm': datetime.datetime(2024, 11, 27, 21, 43, 24), 'upd_dttm': datetime.datetime(2024, 11, 27, 21, 43, 24), 'del_dttm': datetime.datetime(2024, 11, 27, 21, 43, 24), 'dlm': datetime.datetime(2024, 11, 28, 14, 28, 26)},
		{'pos_id': 31004, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 12, 26, 28), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 18, 30, 32), 'age_mins': 64, 'tot_out_cnt': Decimal('10.000000000000'), 'tot_in_cnt': Decimal('10.786240815000'), 'buy_fees_cnt': Decimal('0.024937655860'), 'sell_fees_cnt_tot': Decimal('0.027033185000'), 'fees_cnt_tot': Decimal('0.051970840860'), 'buy_cnt': Decimal('383.656244005371'), 'sell_cnt_tot': Decimal('382.000000000000'), 'hold_cnt': Decimal('1.656244005371'), 'pocket_cnt': Decimal('1.656244005371'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.026065000000'), 'prc_curr': Decimal('0.028236230000'), 'prc_high': Decimal('0.030799000000'), 'prc_low': Decimal('0.026000000000'), 'prc_chg_pct': Decimal('8.330100000000'), 'prc_chg_pct_high': Decimal('16.831000000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-8.500000000000'), 'prc_sell_avg': Decimal('0.028236230000'), 'val_curr': Decimal('0.046766086672'), 'val_tot': Decimal('0.046766086672'), 'gain_loss_amt_est': Decimal('1.302512948398'), 'gain_loss_amt_est_high': Decimal('1.683099942452'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.833006901672'), 'gain_loss_amt_net': Decimal('0.879772988344'), 'gain_loss_pct_est': Decimal('13.025100000000'), 'gain_loss_pct_est_high': Decimal('16.831000000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('8.330100000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30658, 'bo_uuid': '01875dba-5b4b-4d17-80d6-ce8fcb3d7810', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 12, 27, 4), 'check_last_dttm': datetime.datetime(2024, 11, 28, 13, 30, 30), 'add_dttm': datetime.datetime(2024, 11, 28, 12, 27, 4), 'upd_dttm': datetime.datetime(2024, 11, 28, 12, 27, 4), 'del_dttm': datetime.datetime(2024, 11, 28, 12, 27, 4), 'dlm': datetime.datetime(2024, 11, 28, 13, 31, 24)},
		{'pos_id': 30826, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 6, 43, 6), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 15, 32, 43), 'age_mins': 230, 'tot_out_cnt': Decimal('10.000000000000'), 'tot_in_cnt': Decimal('13.020205905000'), 'buy_fees_cnt': Decimal('0.024937655860'), 'sell_fees_cnt_tot': Decimal('0.032632095000'), 'fees_cnt_tot': Decimal('0.057569750860'), 'buy_cnt': Decimal('441.296334460257'), 'sell_cnt_tot': Decimal('438.000000000000'), 'hold_cnt': Decimal('3.296334460257'), 'pocket_cnt': Decimal('3.296334460257'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.022660510000'), 'prc_curr': Decimal('0.029726500000'), 'prc_high': Decimal('0.035687000000'), 'prc_low': Decimal('0.019771000000'), 'prc_chg_pct': Decimal('31.182000000000'), 'prc_chg_pct_high': Decimal('57.485400000000'), 'prc_chg_pct_low': Decimal('-12.751300000000'), 'prc_chg_pct_drop': Decimal('-26.300000000000'), 'prc_sell_avg': Decimal('0.029726500000'), 'val_curr': Decimal('0.097988486333'), 'val_tot': Decimal('0.097988486333'), 'gain_loss_amt_est': Decimal('3.115327060159'), 'gain_loss_amt_est_high': Decimal('5.748542287883'), 'gain_loss_amt_est_low': Decimal('-1.275130171386'), 'gain_loss_amt': Decimal('3.118194391333'), 'gain_loss_amt_net': Decimal('3.216182877666'), 'gain_loss_pct_est': Decimal('31.153300000000'), 'gain_loss_pct_est_high': Decimal('57.485400000000'), 'gain_loss_pct_est_low': Decimal('-12.751300000000'), 'gain_loss_pct': Decimal('31.181900000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30480, 'bo_uuid': 'ada77ebe-7ba5-4403-9ee7-111d5958c2b0', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 6, 43, 42), 'check_last_dttm': datetime.datetime(2024, 11, 28, 10, 32, 42), 'add_dttm': datetime.datetime(2024, 11, 28, 6, 43, 42), 'upd_dttm': datetime.datetime(2024, 11, 28, 6, 43, 42), 'del_dttm': datetime.datetime(2024, 11, 28, 6, 43, 42), 'dlm': datetime.datetime(2024, 11, 28, 10, 34, 12)},
		{'pos_id': 30579, 'symb': 'USDC', 'prod_id': 'SKL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 22, 45, 42), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 15, 12, 56), 'age_mins': 688, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.061447200000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007672800000'), 'fees_cnt_tot': Decimal('0.015154096758'), 'buy_cnt': Decimal('41.736662527781'), 'sell_cnt_tot': Decimal('41.700000000000'), 'hold_cnt': Decimal('0.036662527781'), 'pocket_cnt': Decimal('0.036662527781'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.071879250000'), 'prc_curr': Decimal('0.073416000000'), 'prc_high': Decimal('0.079200000000'), 'prc_low': Decimal('0.069300000000'), 'prc_chg_pct': Decimal('2.138000000000'), 'prc_chg_pct_high': Decimal('6.845900000000'), 'prc_chg_pct_low': Decimal('-3.588300000000'), 'prc_chg_pct_drop': Decimal('-4.710000000000'), 'prc_sell_avg': Decimal('0.073416000000'), 'val_curr': Decimal('0.002691616140'), 'val_tot': Decimal('0.002691616140'), 'gain_loss_amt_est': Decimal('0.080165694550'), 'gain_loss_amt_est_high': Decimal('0.205375682134'), 'gain_loss_amt_est_low': Decimal('-0.107649286825'), 'gain_loss_amt': Decimal('0.064138816140'), 'gain_loss_amt_net': Decimal('0.066830432279'), 'gain_loss_pct_est': Decimal('2.672200000000'), 'gain_loss_pct_est_high': Decimal('6.845900000000'), 'gain_loss_pct_est_low': Decimal('-3.588300000000'), 'gain_loss_pct': Decimal('2.138000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30233, 'bo_uuid': '0f294d7e-b145-4f5e-afdb-73d238fe632c', 'buy_curr_symb': 'SKL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SKL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SKL', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('257817000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 22, 46, 36), 'check_last_dttm': datetime.datetime(2024, 11, 28, 10, 12, 54), 'add_dttm': datetime.datetime(2024, 11, 27, 22, 46, 36), 'upd_dttm': datetime.datetime(2024, 11, 27, 22, 46, 36), 'del_dttm': datetime.datetime(2024, 11, 27, 22, 46, 36), 'dlm': datetime.datetime(2024, 11, 28, 10, 14, 4)},
		{'pos_id': 30699, 'symb': 'USDC', 'prod_id': 'SKL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 2, 2, 36), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 15, 2, 16), 'age_mins': 480, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.152929125000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015420875000'), 'fees_cnt_tot': Decimal('0.030383468516'), 'buy_cnt': Decimal('81.762806099505'), 'sell_cnt_tot': Decimal('81.700000000000'), 'hold_cnt': Decimal('0.062806099505'), 'pocket_cnt': Decimal('0.062806099505'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.073383000000'), 'prc_curr': Decimal('0.075311250000'), 'prc_high': Decimal('0.076900000000'), 'prc_low': Decimal('0.069300000000'), 'prc_chg_pct': Decimal('2.627700000000'), 'prc_chg_pct_high': Decimal('4.656400000000'), 'prc_chg_pct_low': Decimal('-5.564000000000'), 'prc_chg_pct_drop': Decimal('-2.030000000000'), 'prc_sell_avg': Decimal('0.075311250000'), 'val_curr': Decimal('0.004730005861'), 'val_tot': Decimal('0.004730005861'), 'gain_loss_amt_est': Decimal('0.213973263562'), 'gain_loss_amt_est_high': Decimal('0.279383508442'), 'gain_loss_amt_est_low': Decimal('-0.333837537304'), 'gain_loss_amt': Decimal('0.157659130861'), 'gain_loss_amt_net': Decimal('0.162389136723'), 'gain_loss_pct_est': Decimal('3.566200000000'), 'gain_loss_pct_est_high': Decimal('4.656400000000'), 'gain_loss_pct_est_low': Decimal('-5.564000000000'), 'gain_loss_pct': Decimal('2.627700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30353, 'bo_uuid': '9cfe1110-f6b2-4f6d-b7dd-7d32b8045257', 'buy_curr_symb': 'SKL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SKL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SKL', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('257817000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 2, 4, 10), 'check_last_dttm': datetime.datetime(2024, 11, 28, 10, 2, 15), 'add_dttm': datetime.datetime(2024, 11, 28, 2, 4, 10), 'upd_dttm': datetime.datetime(2024, 11, 28, 2, 4, 10), 'del_dttm': datetime.datetime(2024, 11, 28, 2, 4, 10), 'dlm': datetime.datetime(2024, 11, 28, 10, 3, 48)},
		{'pos_id': 30546, 'symb': 'USDC', 'prod_id': 'SKL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 22, 1, 53), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 15, 1, 57), 'age_mins': 721, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.246744000000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015656000000'), 'fees_cnt_tot': Decimal('0.030618593516'), 'buy_cnt': Decimal('82.552240089432'), 'sell_cnt_tot': Decimal('82.400000000000'), 'hold_cnt': Decimal('0.152240089432'), 'pocket_cnt': Decimal('0.152240089432'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.072681250000'), 'prc_curr': Decimal('0.075810000000'), 'prc_high': Decimal('0.076900000000'), 'prc_low': Decimal('0.069300000000'), 'prc_chg_pct': Decimal('4.304800000000'), 'prc_chg_pct_high': Decimal('5.666900000000'), 'prc_chg_pct_low': Decimal('-4.652200000000'), 'prc_chg_pct_drop': Decimal('-1.360000000000'), 'prc_sell_avg': Decimal('0.075810000000'), 'val_curr': Decimal('0.011541321180'), 'val_tot': Decimal('0.011541321180'), 'gain_loss_amt_est': Decimal('0.273970246797'), 'gain_loss_amt_est_high': Decimal('0.340012038868'), 'gain_loss_amt_est_low': Decimal('-0.279129761802'), 'gain_loss_amt': Decimal('0.258285321180'), 'gain_loss_amt_net': Decimal('0.269826642360'), 'gain_loss_pct_est': Decimal('4.566200000000'), 'gain_loss_pct_est_high': Decimal('5.666900000000'), 'gain_loss_pct_est_low': Decimal('-4.652200000000'), 'gain_loss_pct': Decimal('4.304800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30200, 'bo_uuid': '24b058c6-3eab-4f9c-a31a-13ed27d81485', 'buy_curr_symb': 'SKL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SKL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SKL', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('257817000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 22, 2, 17), 'check_last_dttm': datetime.datetime(2024, 11, 28, 10, 1, 55), 'add_dttm': datetime.datetime(2024, 11, 27, 22, 2, 17), 'upd_dttm': datetime.datetime(2024, 11, 27, 22, 2, 17), 'del_dttm': datetime.datetime(2024, 11, 27, 22, 2, 17), 'dlm': datetime.datetime(2024, 11, 28, 10, 3, 38)},
		{'pos_id': 30829, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 6, 48, 36), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 15, 0, 42), 'age_mins': 192, 'tot_out_cnt': Decimal('18.000000000000'), 'tot_in_cnt': Decimal('27.774886755000'), 'buy_fees_cnt': Decimal('0.044887780549'), 'sell_fees_cnt_tot': Decimal('0.069611245000'), 'fees_cnt_tot': Decimal('0.114499025549'), 'buy_cnt': Decimal('823.629000892265'), 'sell_cnt_tot': Decimal('814.000000000000'), 'hold_cnt': Decimal('9.629000892265'), 'pocket_cnt': Decimal('9.629000892265'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.021854500000'), 'prc_curr': Decimal('0.034121480000'), 'prc_high': Decimal('0.036500000000'), 'prc_low': Decimal('0.019771000000'), 'prc_chg_pct': Decimal('56.130200000000'), 'prc_chg_pct_high': Decimal('63.293600000000'), 'prc_chg_pct_low': Decimal('-9.533500000000'), 'prc_chg_pct_drop': Decimal('-7.160000000000'), 'prc_sell_avg': Decimal('0.034121480000'), 'val_curr': Decimal('0.328555761365'), 'val_tot': Decimal('0.328555761365'), 'gain_loss_amt_est': Decimal('10.275183600631'), 'gain_loss_amt_est_high': Decimal('11.392848154842'), 'gain_loss_amt_est_low': Decimal('-1.716031023359'), 'gain_loss_amt': Decimal('10.103442516365'), 'gain_loss_amt_net': Decimal('10.431998277731'), 'gain_loss_pct_est': Decimal('57.084400000000'), 'gain_loss_pct_est_high': Decimal('63.293600000000'), 'gain_loss_pct_est_low': Decimal('-9.533500000000'), 'gain_loss_pct': Decimal('56.130200000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30483, 'bo_uuid': '4c5729e6-981a-4acf-b32a-ab0e26c699e4', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 6, 48, 54), 'check_last_dttm': datetime.datetime(2024, 11, 28, 10, 0, 41), 'add_dttm': datetime.datetime(2024, 11, 28, 6, 48, 54), 'upd_dttm': datetime.datetime(2024, 11, 28, 6, 48, 54), 'del_dttm': datetime.datetime(2024, 11, 28, 6, 48, 54), 'dlm': datetime.datetime(2024, 11, 28, 10, 1, 45)},
		{'pos_id': 30861, 'symb': 'USDC', 'prod_id': 'AMP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 7, 32, 46), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 13, 5, 58), 'age_mins': 33, 'tot_out_cnt': Decimal('2.000000000000'), 'tot_in_cnt': Decimal('2.015648250000'), 'buy_fees_cnt': Decimal('0.004987531172'), 'sell_fees_cnt_tot': Decimal('0.005051750000'), 'fees_cnt_tot': Decimal('0.010039281172'), 'buy_cnt': Decimal('242.407347366699'), 'sell_cnt_tot': Decimal('242.000000000000'), 'hold_cnt': Decimal('0.407347366699'), 'pocket_cnt': Decimal('0.407347366699'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.008250570000'), 'prc_curr': Decimal('0.008329120000'), 'prc_high': Decimal('0.009110000000'), 'prc_low': Decimal('0.008200000000'), 'prc_chg_pct': Decimal('0.952100000000'), 'prc_chg_pct_high': Decimal('4.477600000000'), 'prc_chg_pct_low': Decimal('-0.612900000000'), 'prc_chg_pct_drop': Decimal('-3.530000000000'), 'prc_sell_avg': Decimal('0.008329120000'), 'val_curr': Decimal('0.003392845099'), 'val_tot': Decimal('0.003392845099'), 'gain_loss_amt_est': Decimal('0.065310599564'), 'gain_loss_amt_est_high': Decimal('0.089551334301'), 'gain_loss_amt_est_low': Decimal('-0.012259751593'), 'gain_loss_amt': Decimal('0.019041095099'), 'gain_loss_amt_net': Decimal('0.022433940198'), 'gain_loss_pct_est': Decimal('3.265500000000'), 'gain_loss_pct_est_high': Decimal('4.477600000000'), 'gain_loss_pct_est_low': Decimal('-0.613000000000'), 'gain_loss_pct': Decimal('0.952100000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30515, 'bo_uuid': '5645c45b-be0f-4b08-a8af-2787c240d66f', 'buy_curr_symb': 'AMP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'AMP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'AMP', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('2506153200.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 7, 32, 57), 'check_last_dttm': datetime.datetime(2024, 11, 28, 8, 5, 57), 'add_dttm': datetime.datetime(2024, 11, 28, 7, 32, 57), 'upd_dttm': datetime.datetime(2024, 11, 28, 7, 32, 57), 'del_dttm': datetime.datetime(2024, 11, 28, 7, 32, 57), 'dlm': datetime.datetime(2024, 11, 28, 8, 7, 3)},
		{'pos_id': 30852, 'symb': 'USDC', 'prod_id': 'AMP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 7, 21, 18), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 13, 5, 42), 'age_mins': 44, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.090395850000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015264150000'), 'fees_cnt_tot': Decimal('0.030226743516'), 'buy_cnt': Decimal('727.222042100096'), 'sell_cnt_tot': Decimal('726.000000000000'), 'hold_cnt': Decimal('1.222042100096'), 'pocket_cnt': Decimal('1.222042100096'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.008250570000'), 'prc_curr': Decimal('0.008388970000'), 'prc_high': Decimal('0.009110000000'), 'prc_low': Decimal('0.008090000000'), 'prc_chg_pct': Decimal('1.677500000000'), 'prc_chg_pct_high': Decimal('4.477600000000'), 'prc_chg_pct_low': Decimal('-1.946200000000'), 'prc_chg_pct_drop': Decimal('-2.800000000000'), 'prc_sell_avg': Decimal('0.008388970000'), 'val_curr': Decimal('0.010251674516'), 'val_tot': Decimal('0.010251674516'), 'gain_loss_amt_est': Decimal('0.195931798693'), 'gain_loss_amt_est_high': Decimal('0.268654002903'), 'gain_loss_amt_est_low': Decimal('-0.116773679410'), 'gain_loss_amt': Decimal('0.100647524516'), 'gain_loss_amt_net': Decimal('0.110899199033'), 'gain_loss_pct_est': Decimal('3.265500000000'), 'gain_loss_pct_est_high': Decimal('4.477600000000'), 'gain_loss_pct_est_low': Decimal('-1.946200000000'), 'gain_loss_pct': Decimal('1.677500000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30506, 'bo_uuid': '12c7fa97-80e9-4237-a996-669a3360d18a', 'buy_curr_symb': 'AMP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'AMP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'AMP', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('2506153200.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 7, 21, 35), 'check_last_dttm': datetime.datetime(2024, 11, 28, 8, 5, 41), 'add_dttm': datetime.datetime(2024, 11, 28, 7, 21, 35), 'upd_dttm': datetime.datetime(2024, 11, 28, 7, 21, 35), 'del_dttm': datetime.datetime(2024, 11, 28, 7, 21, 35), 'dlm': datetime.datetime(2024, 11, 28, 8, 6, 55)},
		{'pos_id': 30845, 'symb': 'USDC', 'prod_id': 'AMP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 7, 15, 36), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 13, 5, 25), 'age_mins': 50, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.168799350000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015460650000'), 'fees_cnt_tot': Decimal('0.030423243516'), 'buy_cnt': Decimal('731.667164606820'), 'sell_cnt_tot': Decimal('731.000000000000'), 'hold_cnt': Decimal('0.667164606820'), 'pocket_cnt': Decimal('0.667164606820'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.008200450000'), 'prc_curr': Decimal('0.008438850000'), 'prc_high': Decimal('0.009110000000'), 'prc_low': Decimal('0.008090000000'), 'prc_chg_pct': Decimal('2.907200000000'), 'prc_chg_pct_high': Decimal('5.116200000000'), 'prc_chg_pct_low': Decimal('-1.346900000000'), 'prc_chg_pct_drop': Decimal('-2.210000000000'), 'prc_sell_avg': Decimal('0.008438850000'), 'val_curr': Decimal('0.005630102042'), 'val_tot': Decimal('0.005630102042'), 'gain_loss_amt_est': Decimal('0.233804242450'), 'gain_loss_amt_est_high': Decimal('0.306970958911'), 'gain_loss_amt_est_low': Decimal('-0.080812638331'), 'gain_loss_amt': Decimal('0.174429452042'), 'gain_loss_amt_net': Decimal('0.180059554085'), 'gain_loss_pct_est': Decimal('3.896700000000'), 'gain_loss_pct_est_high': Decimal('5.116200000000'), 'gain_loss_pct_est_low': Decimal('-1.346900000000'), 'gain_loss_pct': Decimal('2.907200000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30499, 'bo_uuid': '2d92d5a6-69e3-4d8c-a271-e8e344fd8d05', 'buy_curr_symb': 'AMP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'AMP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'AMP', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('2506153200.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 7, 16, 23), 'check_last_dttm': datetime.datetime(2024, 11, 28, 8, 5, 24), 'add_dttm': datetime.datetime(2024, 11, 28, 7, 16, 23), 'upd_dttm': datetime.datetime(2024, 11, 28, 7, 16, 23), 'del_dttm': datetime.datetime(2024, 11, 28, 7, 16, 23), 'dlm': datetime.datetime(2024, 11, 28, 8, 6, 46)},
		{'pos_id': 30794, 'symb': 'USDC', 'prod_id': 'ENS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 5, 21, 49), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 11, 58, 43), 'age_mins': 96, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.078634125000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007715875000'), 'fees_cnt_tot': Decimal('0.015197171758'), 'buy_cnt': Decimal('0.085770097542'), 'sell_cnt_tot': Decimal('0.085000000000'), 'hold_cnt': Decimal('0.000770097542'), 'pocket_cnt': Decimal('0.000770097542'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('34.977225000000'), 'prc_curr': Decimal('36.219225000000'), 'prc_high': Decimal('37.100000000000'), 'prc_low': Decimal('34.820000000000'), 'prc_chg_pct': Decimal('3.550900000000'), 'prc_chg_pct_high': Decimal('5.497200000000'), 'prc_chg_pct_low': Decimal('-0.449500000000'), 'prc_chg_pct_drop': Decimal('-1.950000000000'), 'prc_sell_avg': Decimal('36.219225000000'), 'val_curr': Decimal('0.027892336146'), 'val_tot': Decimal('0.027892336146'), 'gain_loss_amt_est': Decimal('0.104877531020'), 'gain_loss_amt_est_high': Decimal('0.164916599300'), 'gain_loss_amt_est_low': Decimal('-0.013485203588'), 'gain_loss_amt': Decimal('0.106526461146'), 'gain_loss_amt_net': Decimal('0.134418797291'), 'gain_loss_pct_est': Decimal('3.495900000000'), 'gain_loss_pct_est_high': Decimal('5.497200000000'), 'gain_loss_pct_est_low': Decimal('-0.449500000000'), 'gain_loss_pct': Decimal('3.550900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30448, 'bo_uuid': '255db8d5-f1a6-4aef-8c71-ac06c51ae0df', 'buy_curr_symb': 'ENS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ENS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ENS', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('775300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 5, 22, 43), 'check_last_dttm': datetime.datetime(2024, 11, 28, 6, 58, 42), 'add_dttm': datetime.datetime(2024, 11, 28, 5, 22, 43), 'upd_dttm': datetime.datetime(2024, 11, 28, 5, 22, 43), 'del_dttm': datetime.datetime(2024, 11, 28, 5, 22, 43), 'dlm': datetime.datetime(2024, 11, 28, 6, 59, 32)},
		{'pos_id': 30638, 'symb': 'USDC', 'prod_id': 'ENS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 0, 8), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 11, 57, 57), 'age_mins': 410, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.155718400000'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.012921600000'), 'fees_cnt_tot': Decimal('0.025390427930'), 'buy_cnt': Decimal('0.143155314927'), 'sell_cnt_tot': Decimal('0.143000000000'), 'hold_cnt': Decimal('0.000155314927'), 'pocket_cnt': Decimal('0.000155314927'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('34.927100000000'), 'prc_curr': Decimal('36.053974830000'), 'prc_high': Decimal('37.100000000000'), 'prc_low': Decimal('33.520000000000'), 'prc_chg_pct': Decimal('3.226400000000'), 'prc_chg_pct_high': Decimal('5.648600000000'), 'prc_chg_pct_low': Decimal('-4.028700000000'), 'prc_chg_pct_drop': Decimal('-2.420000000000'), 'prc_sell_avg': Decimal('36.053974830000'), 'val_curr': Decimal('0.005599720469'), 'val_tot': Decimal('0.005599720469'), 'gain_loss_amt_est': Decimal('0.182222400357'), 'gain_loss_amt_est_high': Decimal('0.282431120806'), 'gain_loss_amt_est_low': Decimal('-0.201433843647'), 'gain_loss_amt': Decimal('0.161318120469'), 'gain_loss_amt_net': Decimal('0.166917840938'), 'gain_loss_pct_est': Decimal('3.644400000000'), 'gain_loss_pct_est_high': Decimal('5.648600000000'), 'gain_loss_pct_est_low': Decimal('-4.028700000000'), 'gain_loss_pct': Decimal('3.226400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30292, 'bo_uuid': 'c2707295-3a86-4d4e-82bd-de130c70f624', 'buy_curr_symb': 'ENS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ENS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ENS', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('775300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 0, 8, 45), 'check_last_dttm': datetime.datetime(2024, 11, 28, 6, 57, 56), 'add_dttm': datetime.datetime(2024, 11, 28, 0, 8, 45), 'upd_dttm': datetime.datetime(2024, 11, 28, 0, 8, 45), 'del_dttm': datetime.datetime(2024, 11, 28, 0, 8, 45), 'dlm': datetime.datetime(2024, 11, 28, 6, 59, 4)},
		{'pos_id': 30487, 'symb': 'USDC', 'prod_id': 'ENS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 20, 19, 18), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 11, 57, 43), 'age_mins': 639, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.243841275000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015648725000'), 'fees_cnt_tot': Decimal('0.030611318516'), 'buy_cnt': Decimal('0.173228289623'), 'sell_cnt_tot': Decimal('0.173000000000'), 'hold_cnt': Decimal('0.000228289623'), 'pocket_cnt': Decimal('0.000228289623'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('34.636375000000'), 'prc_curr': Decimal('36.091568060000'), 'prc_high': Decimal('37.100000000000'), 'prc_low': Decimal('32.540000000000'), 'prc_chg_pct': Decimal('4.201300000000'), 'prc_chg_pct_high': Decimal('6.535400000000'), 'prc_chg_pct_low': Decimal('-6.052500000000'), 'prc_chg_pct_drop': Decimal('-2.330000000000'), 'prc_sell_avg': Decimal('36.091568060000'), 'val_curr': Decimal('0.008239330466'), 'val_tot': Decimal('0.008239330466'), 'gain_loss_amt_est': Decimal('0.270864084353'), 'gain_loss_amt_est_high': Decimal('0.392123887089'), 'gain_loss_amt_est_low': Decimal('-0.363151455668'), 'gain_loss_amt': Decimal('0.252080605466'), 'gain_loss_amt_net': Decimal('0.260319935932'), 'gain_loss_pct_est': Decimal('4.514400000000'), 'gain_loss_pct_est_high': Decimal('6.535400000000'), 'gain_loss_pct_est_low': Decimal('-6.052500000000'), 'gain_loss_pct': Decimal('4.201300000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30141, 'bo_uuid': '13e76c12-ae7e-43d4-a217-56f6ef22f66d', 'buy_curr_symb': 'ENS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ENS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ENS', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('775300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 20, 19, 38), 'check_last_dttm': datetime.datetime(2024, 11, 28, 6, 57, 42), 'add_dttm': datetime.datetime(2024, 11, 27, 20, 19, 38), 'upd_dttm': datetime.datetime(2024, 11, 27, 20, 19, 38), 'del_dttm': datetime.datetime(2024, 11, 27, 20, 19, 38), 'dlm': datetime.datetime(2024, 11, 28, 6, 58, 57)},
		{'pos_id': 30800, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 5, 54, 29), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 11, 24, 45), 'age_mins': 31, 'tot_out_cnt': Decimal('18.000000000000'), 'tot_in_cnt': Decimal('16.045365052500'), 'buy_fees_cnt': Decimal('0.044887780549'), 'sell_fees_cnt_tot': Decimal('0.040213947500'), 'fees_cnt_tot': Decimal('0.085101728049'), 'buy_cnt': Decimal('739.075994873276'), 'sell_cnt_tot': Decimal('739.000000000000'), 'hold_cnt': Decimal('0.075994873276'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.075994873276'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.024354740000'), 'prc_curr': Decimal('0.021712270000'), 'prc_high': Decimal('0.024412000000'), 'prc_low': Decimal('0.021712270000'), 'prc_chg_pct': Decimal('-10.849900000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-10.849900000000'), 'prc_chg_pct_drop': Decimal('-10.850000000000'), 'prc_sell_avg': Decimal('0.021712270000'), 'val_curr': Decimal('0.001650021207'), 'val_tot': Decimal('0.001650021207'), 'gain_loss_amt_est': Decimal('-1.852667664009'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-1.852667664009'), 'gain_loss_amt': Decimal('-1.952984926293'), 'gain_loss_amt_net': Decimal('-1.951334905086'), 'gain_loss_pct_est': Decimal('-10.292600000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-10.292600000000'), 'gain_loss_pct': Decimal('-10.849900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 30454, 'bo_uuid': '5fb43982-dd20-4be4-8647-34ac9261c4a7', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 5, 54, 56), 'check_last_dttm': datetime.datetime(2024, 11, 28, 6, 24, 44), 'add_dttm': datetime.datetime(2024, 11, 28, 5, 54, 56), 'upd_dttm': datetime.datetime(2024, 11, 28, 5, 54, 56), 'del_dttm': datetime.datetime(2024, 11, 28, 5, 54, 56), 'dlm': datetime.datetime(2024, 11, 28, 6, 25, 40)},
		{'pos_id': 30799, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 5, 49, 12), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 11, 7, 6), 'age_mins': 18, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.220011000000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015589000000'), 'fees_cnt_tot': Decimal('0.030551593516'), 'buy_cnt': Decimal('262.616823452558'), 'sell_cnt_tot': Decimal('262.000000000000'), 'hold_cnt': Decimal('0.616823452558'), 'pocket_cnt': Decimal('0.616823452558'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.022846980000'), 'prc_curr': Decimal('0.023740500000'), 'prc_high': Decimal('0.024412000000'), 'prc_low': Decimal('0.022790000000'), 'prc_chg_pct': Decimal('3.910900000000'), 'prc_chg_pct_high': Decimal('4.670300000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-0.760000000000'), 'prc_sell_avg': Decimal('0.023740500000'), 'val_curr': Decimal('0.014643697175'), 'val_tot': Decimal('0.014643697175'), 'gain_loss_amt_est': Decimal('0.190403762424'), 'gain_loss_amt_est_high': Decimal('0.280218716044'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.234654697175'), 'gain_loss_amt_net': Decimal('0.249298394351'), 'gain_loss_pct_est': Decimal('3.173400000000'), 'gain_loss_pct_est_high': Decimal('4.670300000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('3.910900000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30453, 'bo_uuid': 'bc1fcc48-da59-4dda-a46f-8999a8f08c2a', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 5, 49, 58), 'check_last_dttm': datetime.datetime(2024, 11, 28, 6, 7, 5), 'add_dttm': datetime.datetime(2024, 11, 28, 5, 49, 58), 'upd_dttm': datetime.datetime(2024, 11, 28, 5, 49, 58), 'del_dttm': datetime.datetime(2024, 11, 28, 5, 49, 58), 'dlm': datetime.datetime(2024, 11, 28, 6, 8, 21)},
		{'pos_id': 30750, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 3, 21, 9), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 10, 13, 35), 'age_mins': 112, 'tot_out_cnt': Decimal('15.000000000000'), 'tot_in_cnt': Decimal('16.645302450000'), 'buy_fees_cnt': Decimal('0.037406483791'), 'sell_fees_cnt_tot': Decimal('0.041717550000'), 'fees_cnt_tot': Decimal('0.079124033791'), 'buy_cnt': Decimal('745.111972322568'), 'sell_cnt_tot': Decimal('742.000000000000'), 'hold_cnt': Decimal('3.111972322568'), 'pocket_cnt': Decimal('3.111972322568'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.020131200000'), 'prc_curr': Decimal('0.022433020000'), 'prc_high': Decimal('0.025000000000'), 'prc_low': Decimal('0.020005000000'), 'prc_chg_pct': Decimal('11.434100000000'), 'prc_chg_pct_high': Decimal('23.683600000000'), 'prc_chg_pct_low': Decimal('-0.626900000000'), 'prc_chg_pct_drop': Decimal('-12.250000000000'), 'prc_sell_avg': Decimal('0.022433020000'), 'val_curr': Decimal('0.069810937352'), 'val_tot': Decimal('0.069810937352'), 'gain_loss_amt_est': Decimal('2.890138455465'), 'gain_loss_amt_est_high': Decimal('3.552542998860'), 'gain_loss_amt_est_low': Decimal('-0.094034993687'), 'gain_loss_amt': Decimal('1.715113387352'), 'gain_loss_amt_net': Decimal('1.784924324703'), 'gain_loss_pct_est': Decimal('19.267600000000'), 'gain_loss_pct_est_high': Decimal('23.683600000000'), 'gain_loss_pct_est_low': Decimal('-0.626900000000'), 'gain_loss_pct': Decimal('11.434100000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30404, 'bo_uuid': 'ee078313-e638-4cb0-ade3-c2df499439e3', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 3, 22, 37), 'check_last_dttm': datetime.datetime(2024, 11, 28, 5, 13, 34), 'add_dttm': datetime.datetime(2024, 11, 28, 3, 22, 37), 'upd_dttm': datetime.datetime(2024, 11, 28, 3, 22, 37), 'del_dttm': datetime.datetime(2024, 11, 28, 3, 22, 37), 'dlm': datetime.datetime(2024, 11, 28, 5, 15, 30)},
		{'pos_id': 30681, 'symb': 'USDC', 'prod_id': 'AMP-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 1, 23, 42), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 9, 41, 15), 'age_mins': 198, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('2.993298000000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007502000000'), 'fees_cnt_tot': Decimal('0.014983296758'), 'buy_cnt': Decimal('440.076279888514'), 'sell_cnt_tot': Decimal('440.000000000000'), 'hold_cnt': Decimal('0.076279888514'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.076279888514'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.006817000000'), 'prc_curr': Decimal('0.006802950000'), 'prc_high': Decimal('0.007200000000'), 'prc_low': Decimal('0.006490000000'), 'prc_chg_pct': Decimal('-0.206100000000'), 'prc_chg_pct_high': Decimal('4.298100000000'), 'prc_chg_pct_low': Decimal('-4.796800000000'), 'prc_chg_pct_drop': Decimal('-4.500000000000'), 'prc_sell_avg': Decimal('0.006802950000'), 'val_curr': Decimal('0.000518928268'), 'val_tot': Decimal('0.000518928268'), 'gain_loss_amt_est': Decimal('0.023324042834'), 'gain_loss_amt_est_high': Decimal('0.128942350007'), 'gain_loss_amt_est_low': Decimal('-0.143904943524'), 'gain_loss_amt': Decimal('-0.006183071732'), 'gain_loss_amt_net': Decimal('-0.005664143465'), 'gain_loss_pct_est': Decimal('0.777500000000'), 'gain_loss_pct_est_high': Decimal('4.298100000000'), 'gain_loss_pct_est_low': Decimal('-4.796800000000'), 'gain_loss_pct': Decimal('-0.206100000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30335, 'bo_uuid': 'f76bca7d-4ebd-4702-b0c8-818fcdf305b8', 'buy_curr_symb': 'AMP', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'AMP', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'AMP', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('2506153200.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 1, 24, 37), 'check_last_dttm': datetime.datetime(2024, 11, 28, 4, 41, 14), 'add_dttm': datetime.datetime(2024, 11, 28, 1, 24, 37), 'upd_dttm': datetime.datetime(2024, 11, 28, 1, 24, 37), 'del_dttm': datetime.datetime(2024, 11, 28, 1, 24, 37), 'dlm': datetime.datetime(2024, 11, 28, 4, 42, 23)},
		{'pos_id': 30516, 'symb': 'USDC', 'prod_id': 'ENS-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 21, 1, 56), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 9, 12, 24), 'age_mins': 431, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.088181400000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015258600000'), 'fees_cnt_tot': Decimal('0.030221193516'), 'buy_cnt': Decimal('0.174084857664'), 'sell_cnt_tot': Decimal('0.173000000000'), 'hold_cnt': Decimal('0.001084857664'), 'pocket_cnt': Decimal('0.001084857664'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('34.465950000000'), 'prc_curr': Decimal('35.191800000000'), 'prc_high': Decimal('36.280000000000'), 'prc_low': Decimal('32.540000000000'), 'prc_chg_pct': Decimal('2.106000000000'), 'prc_chg_pct_high': Decimal('3.319400000000'), 'prc_chg_pct_low': Decimal('-5.588000000000'), 'prc_chg_pct_drop': Decimal('-1.210000000000'), 'prc_sell_avg': Decimal('35.191800000000'), 'val_curr': Decimal('0.038178093940'), 'val_tot': Decimal('0.038178093940'), 'gain_loss_amt_est': Decimal('0.148677172692'), 'gain_loss_amt_est_high': Decimal('0.199161781415'), 'gain_loss_amt_est_low': Decimal('-0.335278731613'), 'gain_loss_amt': Decimal('0.126359493940'), 'gain_loss_amt_net': Decimal('0.164537587880'), 'gain_loss_pct_est': Decimal('2.478000000000'), 'gain_loss_pct_est_high': Decimal('3.319400000000'), 'gain_loss_pct_est_low': Decimal('-5.588000000000'), 'gain_loss_pct': Decimal('2.106000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30170, 'bo_uuid': 'c7f88562-f5a5-4b75-8f1c-de056e44bbe9', 'buy_curr_symb': 'ENS', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ENS', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ENS', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('775300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 21, 2, 14), 'check_last_dttm': datetime.datetime(2024, 11, 28, 4, 12, 23), 'add_dttm': datetime.datetime(2024, 11, 27, 21, 2, 14), 'upd_dttm': datetime.datetime(2024, 11, 27, 21, 2, 14), 'del_dttm': datetime.datetime(2024, 11, 27, 21, 2, 14), 'dlm': datetime.datetime(2024, 11, 28, 4, 13, 3)},
		{'pos_id': 30691, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 1, 56, 22), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 8, 6, 15), 'age_mins': 70, 'tot_out_cnt': Decimal('19.000000000000'), 'tot_in_cnt': Decimal('19.544695800000'), 'buy_fees_cnt': Decimal('0.047381546135'), 'sell_fees_cnt_tot': Decimal('0.048984200000'), 'fees_cnt_tot': Decimal('0.096365746135'), 'buy_cnt': Decimal('1034.927016538270'), 'sell_cnt_tot': Decimal('1033.000000000000'), 'hold_cnt': Decimal('1.927016538270'), 'pocket_cnt': Decimal('1.927016538270'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.018358780000'), 'prc_curr': Decimal('0.018920330000'), 'prc_high': Decimal('0.020900000000'), 'prc_low': Decimal('0.018023000000'), 'prc_chg_pct': Decimal('3.058800000000'), 'prc_chg_pct_high': Decimal('8.945100000000'), 'prc_chg_pct_low': Decimal('-1.829000000000'), 'prc_chg_pct_drop': Decimal('-5.890000000000'), 'prc_sell_avg': Decimal('0.018920330000'), 'val_curr': Decimal('0.036459788820'), 'val_tot': Decimal('0.036459788820'), 'gain_loss_amt_est': Decimal('0.875773352617'), 'gain_loss_amt_est_high': Decimal('1.699575257782'), 'gain_loss_amt_est_low': Decimal('-0.347510380931'), 'gain_loss_amt': Decimal('0.581155588820'), 'gain_loss_amt_net': Decimal('0.617615377639'), 'gain_loss_pct_est': Decimal('4.609300000000'), 'gain_loss_pct_est_high': Decimal('8.945100000000'), 'gain_loss_pct_est_low': Decimal('-1.829000000000'), 'gain_loss_pct': Decimal('3.058700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30345, 'bo_uuid': '2b913b5a-96b2-4e05-8edf-de851084ec7b', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 1, 57, 40), 'check_last_dttm': datetime.datetime(2024, 11, 28, 3, 6, 8), 'add_dttm': datetime.datetime(2024, 11, 28, 1, 57, 40), 'upd_dttm': datetime.datetime(2024, 11, 28, 1, 57, 40), 'del_dttm': datetime.datetime(2024, 11, 28, 1, 57, 40), 'dlm': datetime.datetime(2024, 11, 28, 3, 6, 42)},
		{'pos_id': 30690, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 1, 50, 58), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 8, 6, 1), 'age_mins': 75, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.435407160000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.016128840000'), 'fees_cnt_tot': Decimal('0.031091433516'), 'buy_cnt': Decimal('336.616277080078'), 'sell_cnt_tot': Decimal('336.000000000000'), 'hold_cnt': Decimal('0.616277080078'), 'pocket_cnt': Decimal('0.616277080078'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.017824450000'), 'prc_curr': Decimal('0.019153000000'), 'prc_high': Decimal('0.020900000000'), 'prc_low': Decimal('0.017780000000'), 'prc_chg_pct': Decimal('7.453500000000'), 'prc_chg_pct_high': Decimal('12.211000000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-4.760000000000'), 'prc_sell_avg': Decimal('0.019153000000'), 'val_curr': Decimal('0.011803554915'), 'val_tot': Decimal('0.011803554915'), 'gain_loss_amt_est': Decimal('0.464715601323'), 'gain_loss_amt_est_high': Decimal('0.732662157879'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.447210714915'), 'gain_loss_amt_net': Decimal('0.459014269829'), 'gain_loss_pct_est': Decimal('7.745300000000'), 'gain_loss_pct_est_high': Decimal('12.211000000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('7.453500000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30344, 'bo_uuid': '6535df54-fc00-45f5-b895-882b7bac538a', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 1, 52, 15), 'check_last_dttm': datetime.datetime(2024, 11, 28, 3, 5, 54), 'add_dttm': datetime.datetime(2024, 11, 28, 1, 52, 15), 'upd_dttm': datetime.datetime(2024, 11, 28, 1, 52, 15), 'del_dttm': datetime.datetime(2024, 11, 28, 1, 52, 15), 'dlm': datetime.datetime(2024, 11, 28, 3, 6, 36)},
		{'pos_id': 30621, 'symb': 'USDC', 'prod_id': 'TNSR-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 23, 49, 14), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 7, 53, 55), 'age_mins': 185, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('2.776920300000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.006959700000'), 'fees_cnt_tot': Decimal('0.014440996758'), 'buy_cnt': Decimal('4.446535963212'), 'sell_cnt_tot': Decimal('4.440000000000'), 'hold_cnt': Decimal('0.006535963212'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.006535963212'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.674682500000'), 'prc_curr': Decimal('0.625432500000'), 'prc_high': Decimal('0.680000000000'), 'prc_low': Decimal('0.625432500000'), 'prc_chg_pct': Decimal('-7.299700000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-7.299700000000'), 'prc_chg_pct_drop': Decimal('-7.300000000000'), 'prc_sell_avg': Decimal('0.625432500000'), 'val_curr': Decimal('0.004087803812'), 'val_tot': Decimal('0.004087803812'), 'gain_loss_amt_est': Decimal('-0.212021951066'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-0.212021951066'), 'gain_loss_amt': Decimal('-0.218991896188'), 'gain_loss_amt_net': Decimal('-0.214904092377'), 'gain_loss_pct_est': Decimal('-7.067400000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-7.067400000000'), 'gain_loss_pct': Decimal('-7.299700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '1h', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 30275, 'bo_uuid': 'd71b4fe5-b9ca-4342-9722-01e1d7fad0e2', 'buy_curr_symb': 'TNSR', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'TNSR', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'TNSR', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('8000000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 23, 50, 6), 'check_last_dttm': datetime.datetime(2024, 11, 28, 2, 53, 48), 'add_dttm': datetime.datetime(2024, 11, 27, 23, 50, 6), 'upd_dttm': datetime.datetime(2024, 11, 27, 23, 50, 6), 'del_dttm': datetime.datetime(2024, 11, 27, 23, 50, 6), 'dlm': datetime.datetime(2024, 11, 28, 2, 54, 56)},
		{'pos_id': 30639, 'symb': 'USDC', 'prod_id': 'TNSR-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 0, 11, 32), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 6, 52, 38), 'age_mins': 101, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('2.784491325000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.006978675000'), 'fees_cnt_tot': Decimal('0.014459971758'), 'buy_cnt': Decimal('4.426802815447'), 'sell_cnt_tot': Decimal('4.420000000000'), 'hold_cnt': Decimal('0.006802815447'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.006802815447'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.677690000000'), 'prc_curr': Decimal('0.629975410000'), 'prc_high': Decimal('0.676000000000'), 'prc_low': Decimal('0.629975410000'), 'prc_chg_pct': Decimal('-7.040800000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-7.040800000000'), 'prc_chg_pct_drop': Decimal('-7.040000000000'), 'prc_sell_avg': Decimal('0.629975410000'), 'val_curr': Decimal('0.004285606450'), 'val_tot': Decimal('0.004285606450'), 'gain_loss_amt_est': Decimal('-0.149138986852'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-0.149138986852'), 'gain_loss_amt': Decimal('-0.211223068550'), 'gain_loss_amt_net': Decimal('-0.206937462099'), 'gain_loss_pct_est': Decimal('-4.971300000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-4.971300000000'), 'gain_loss_pct': Decimal('-7.040800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '30min', 'sell_strat_type': 'strat', 'sell_strat_name': 'sha', 'sell_strat_freq': '30min', 'bo_id': 30293, 'bo_uuid': 'a2e050fb-d834-4a3f-aa7f-fb85c334699c', 'buy_curr_symb': 'TNSR', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'TNSR', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'TNSR', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('8000000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 0, 12, 35), 'check_last_dttm': datetime.datetime(2024, 11, 28, 1, 52, 30), 'add_dttm': datetime.datetime(2024, 11, 28, 0, 12, 35), 'upd_dttm': datetime.datetime(2024, 11, 28, 0, 12, 35), 'del_dttm': datetime.datetime(2024, 11, 28, 0, 12, 35), 'dlm': datetime.datetime(2024, 11, 28, 1, 53, 47)},
		{'pos_id': 30635, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 28, 0, 2, 18), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 6, 24, 30), 'age_mins': 82, 'tot_out_cnt': Decimal('4.000000000000'), 'tot_in_cnt': Decimal('3.658642470000'), 'buy_fees_cnt': Decimal('0.009975062344'), 'sell_fees_cnt_tot': Decimal('0.009169530000'), 'fees_cnt_tot': Decimal('0.019144592344'), 'buy_cnt': Decimal('212.869448231747'), 'sell_cnt_tot': Decimal('212.000000000000'), 'hold_cnt': Decimal('0.869448231747'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.869448231747'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.018790860000'), 'prc_curr': Decimal('0.017257750000'), 'prc_high': Decimal('0.018999000000'), 'prc_low': Decimal('0.017257750000'), 'prc_chg_pct': Decimal('-8.158800000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-8.158800000000'), 'prc_chg_pct_drop': Decimal('-8.160000000000'), 'prc_sell_avg': Decimal('0.017257750000'), 'val_curr': Decimal('0.015004720221'), 'val_tot': Decimal('0.015004720221'), 'gain_loss_amt_est': Decimal('-0.300541859180'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-0.300541859180'), 'gain_loss_amt': Decimal('-0.326352809779'), 'gain_loss_amt_net': Decimal('-0.311348089557'), 'gain_loss_pct_est': Decimal('-7.513500000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-7.513500000000'), 'gain_loss_pct': Decimal('-8.158800000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '30min', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 30289, 'bo_uuid': '5a21fe0a-ba6b-4b24-9211-79a7100646a7', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 28, 0, 2, 46), 'check_last_dttm': datetime.datetime(2024, 11, 28, 1, 24, 22), 'add_dttm': datetime.datetime(2024, 11, 28, 0, 2, 46), 'upd_dttm': datetime.datetime(2024, 11, 28, 0, 2, 46), 'del_dttm': datetime.datetime(2024, 11, 28, 0, 2, 46), 'dlm': datetime.datetime(2024, 11, 28, 1, 24, 57)},
		{'pos_id': 30625, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 23, 56, 29), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 6, 24, 16), 'age_mins': 88, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('5.505221452500'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.013797547500'), 'fees_cnt_tot': Decimal('0.028760141016'), 'buy_cnt': Decimal('319.696458868853'), 'sell_cnt_tot': Decimal('319.000000000000'), 'hold_cnt': Decimal('0.696458868853'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.696458868853'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.018767800000'), 'prc_curr': Decimal('0.017257750000'), 'prc_high': Decimal('0.018999000000'), 'prc_low': Decimal('0.017257750000'), 'prc_chg_pct': Decimal('-8.046000000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-8.046000000000'), 'prc_chg_pct_drop': Decimal('-8.050000000000'), 'prc_sell_avg': Decimal('0.017257750000'), 'val_curr': Decimal('0.012019313044'), 'val_tot': Decimal('0.012019313044'), 'gain_loss_amt_est': Decimal('-0.443995241318'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-0.443995241318'), 'gain_loss_amt': Decimal('-0.482759234456'), 'gain_loss_amt_net': Decimal('-0.470739921412'), 'gain_loss_pct_est': Decimal('-7.399900000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-7.399900000000'), 'gain_loss_pct': Decimal('-8.046000000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '15min', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 30279, 'bo_uuid': 'f5f81f34-f6c3-4f98-935c-7a2df0423a8b', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 23, 57, 18), 'check_last_dttm': datetime.datetime(2024, 11, 28, 1, 24, 8), 'add_dttm': datetime.datetime(2024, 11, 27, 23, 57, 18), 'upd_dttm': datetime.datetime(2024, 11, 27, 23, 57, 18), 'del_dttm': datetime.datetime(2024, 11, 27, 23, 57, 18), 'dlm': datetime.datetime(2024, 11, 28, 1, 24, 51)},
		{'pos_id': 30622, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 23, 51, 8), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 6, 23, 55), 'age_mins': 93, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('18.643394700000'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.046725300000'), 'fees_cnt_tot': Decimal('0.096600611721'), 'buy_cnt': Decimal('1076.813552560010'), 'sell_cnt_tot': Decimal('1076.000000000000'), 'hold_cnt': Decimal('0.813552560010'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.813552560010'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.018573320000'), 'prc_curr': Decimal('0.017326580000'), 'prc_high': Decimal('0.018999000000'), 'prc_low': Decimal('0.017326580000'), 'prc_chg_pct': Decimal('-6.712500000000'), 'prc_chg_pct_high': Decimal('0.843600000000'), 'prc_chg_pct_low': Decimal('-6.712500000000'), 'prc_chg_pct_drop': Decimal('-7.560000000000'), 'prc_sell_avg': Decimal('0.017326580000'), 'val_curr': Decimal('0.014096083515'), 'val_tot': Decimal('0.014096083515'), 'gain_loss_amt_est': Decimal('-1.286057270060'), 'gain_loss_amt_est_high': Decimal('0.168717839449'), 'gain_loss_amt_est_low': Decimal('-1.286057270060'), 'gain_loss_amt': Decimal('-1.342509216485'), 'gain_loss_amt_net': Decimal('-1.328413132970'), 'gain_loss_pct_est': Decimal('-6.430300000000'), 'gain_loss_pct_est_high': Decimal('0.843600000000'), 'gain_loss_pct_est_low': Decimal('-6.430300000000'), 'gain_loss_pct': Decimal('-6.712500000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'trail_stop', 'sell_strat_freq': '', 'bo_id': 30276, 'bo_uuid': '0a8ae3fe-1a52-4189-8063-2a4784c01d73', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 23, 51, 18), 'check_last_dttm': datetime.datetime(2024, 11, 28, 1, 23, 48), 'add_dttm': datetime.datetime(2024, 11, 27, 23, 51, 18), 'upd_dttm': datetime.datetime(2024, 11, 27, 23, 51, 18), 'del_dttm': datetime.datetime(2024, 11, 27, 23, 51, 18), 'dlm': datetime.datetime(2024, 11, 28, 1, 24, 44)},
		{'pos_id': 30532, 'symb': 'USDC', 'prod_id': 'SKL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 21, 28, 44), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 6, 9, 23), 'age_mins': 221, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('5.534249700000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.013870300000'), 'fees_cnt_tot': Decimal('0.028832893516'), 'buy_cnt': Decimal('79.694239766762'), 'sell_cnt_tot': Decimal('79.600000000000'), 'hold_cnt': Decimal('0.094239766762'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.094239766762'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.075287750000'), 'prc_curr': Decimal('0.069525750000'), 'prc_high': Decimal('0.075200000000'), 'prc_low': Decimal('0.069525750000'), 'prc_chg_pct': Decimal('-7.653300000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-7.653300000000'), 'prc_chg_pct_drop': Decimal('-7.650000000000'), 'prc_sell_avg': Decimal('0.069525750000'), 'val_curr': Decimal('0.006552090464'), 'val_tot': Decimal('0.006552090464'), 'gain_loss_amt_est': Decimal('-0.453280912233'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-0.453280912233'), 'gain_loss_amt': Decimal('-0.459198209536'), 'gain_loss_amt_net': Decimal('-0.452646119072'), 'gain_loss_pct_est': Decimal('-7.554700000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-7.554700000000'), 'gain_loss_pct': Decimal('-7.653300000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 30186, 'bo_uuid': 'db34aa0c-3a6a-4ae2-a6c4-6475a6fc52e3', 'buy_curr_symb': 'SKL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SKL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SKL', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('257817000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 21, 29, 39), 'check_last_dttm': datetime.datetime(2024, 11, 28, 1, 9, 16), 'add_dttm': datetime.datetime(2024, 11, 27, 21, 29, 39), 'upd_dttm': datetime.datetime(2024, 11, 27, 21, 29, 39), 'del_dttm': datetime.datetime(2024, 11, 27, 21, 29, 39), 'dlm': datetime.datetime(2024, 11, 28, 1, 9, 58)},
		{'pos_id': 30560, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 22, 24, 10), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 4, 23, 17), 'age_mins': 59, 'tot_out_cnt': Decimal('12.000000000000'), 'tot_in_cnt': Decimal('12.800699047500'), 'buy_fees_cnt': Decimal('0.029925187032'), 'sell_fees_cnt_tot': Decimal('0.032081952500'), 'fees_cnt_tot': Decimal('0.062007139532'), 'buy_cnt': Decimal('652.497945650999'), 'sell_cnt_tot': Decimal('652.000000000000'), 'hold_cnt': Decimal('0.497945650999'), 'pocket_cnt': Decimal('0.497945650999'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.018390860000'), 'prc_curr': Decimal('0.019632970000'), 'prc_high': Decimal('0.020800000000'), 'prc_low': Decimal('0.018241000000'), 'prc_chg_pct': Decimal('6.754000000000'), 'prc_chg_pct_high': Decimal('11.539100000000'), 'prc_chg_pct_low': Decimal('-0.814900000000'), 'prc_chg_pct_drop': Decimal('-4.790000000000'), 'prc_sell_avg': Decimal('0.019632970000'), 'val_curr': Decimal('0.009776152028'), 'val_tot': Decimal('0.009776152028'), 'gain_loss_amt_est': Decimal('0.268918872076'), 'gain_loss_amt_est_high': Decimal('1.384690359139'), 'gain_loss_amt_est_low': Decimal('-0.097784973380'), 'gain_loss_amt': Decimal('0.810475199528'), 'gain_loss_amt_net': Decimal('0.820251351555'), 'gain_loss_pct_est': Decimal('2.241000000000'), 'gain_loss_pct_est_high': Decimal('11.539100000000'), 'gain_loss_pct_est_low': Decimal('-0.814900000000'), 'gain_loss_pct': Decimal('6.754000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30214, 'bo_uuid': 'e75ba7de-17f4-4b59-b6a2-c335e2ec8986', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 22, 24, 43), 'check_last_dttm': datetime.datetime(2024, 11, 27, 23, 23, 10), 'add_dttm': datetime.datetime(2024, 11, 27, 22, 24, 43), 'upd_dttm': datetime.datetime(2024, 11, 27, 22, 24, 43), 'del_dttm': datetime.datetime(2024, 11, 27, 22, 24, 43), 'dlm': datetime.datetime(2024, 11, 27, 23, 24, 27)},
		{'pos_id': 30558, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 22, 18, 38), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 4, 23, 5), 'age_mins': 64, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.219211800000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.008068200000'), 'fees_cnt_tot': Decimal('0.015549496758'), 'buy_cnt': Decimal('169.068853290503'), 'sell_cnt_tot': Decimal('168.000000000000'), 'hold_cnt': Decimal('1.068853290503'), 'pocket_cnt': Decimal('1.068853290503'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.017744250000'), 'prc_curr': Decimal('0.019161980000'), 'prc_high': Decimal('0.020800000000'), 'prc_low': Decimal('0.017700000000'), 'prc_chg_pct': Decimal('7.989800000000'), 'prc_chg_pct_high': Decimal('15.603600000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-7.610000000000'), 'prc_sell_avg': Decimal('0.019161980000'), 'val_curr': Decimal('0.020481345376'), 'val_tot': Decimal('0.020481345376'), 'gain_loss_amt_est': Decimal('0.179001648421'), 'gain_loss_amt_est_high': Decimal('0.468109387548'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.239693145376'), 'gain_loss_amt_net': Decimal('0.260174490751'), 'gain_loss_pct_est': Decimal('5.966700000000'), 'gain_loss_pct_est_high': Decimal('15.603600000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('7.989800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30212, 'bo_uuid': 'a8b96e5d-4c56-4e28-8ee6-8cc232bc3452', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 22, 19, 32), 'check_last_dttm': datetime.datetime(2024, 11, 27, 23, 22, 57), 'add_dttm': datetime.datetime(2024, 11, 27, 22, 19, 32), 'upd_dttm': datetime.datetime(2024, 11, 27, 22, 19, 32), 'del_dttm': datetime.datetime(2024, 11, 27, 22, 19, 32), 'dlm': datetime.datetime(2024, 11, 27, 23, 24, 21)},
		{'pos_id': 28161, 'symb': 'USDC', 'prod_id': 'POL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 9, 14, 54), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 3, 36, 13), 'age_mins': 3682, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('18.556043625000'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.046506375000'), 'fees_cnt_tot': Decimal('0.096381686721'), 'buy_cnt': Decimal('33.503778448571'), 'sell_cnt_tot': Decimal('33.500000000000'), 'hold_cnt': Decimal('0.003778448571'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.003778448571'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.596947600000'), 'prc_curr': Decimal('0.553911750000'), 'prc_high': Decimal('0.595458948575'), 'prc_low': Decimal('0.501400000000'), 'prc_chg_pct': Decimal('-7.209300000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-16.006000000000'), 'prc_chg_pct_drop': Decimal('-7.210000000000'), 'prc_sell_avg': Decimal('0.553911750000'), 'val_curr': Decimal('0.002092927060'), 'val_tot': Decimal('0.002092927060'), 'gain_loss_amt_est': Decimal('-1.422154850267'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-3.201205485887'), 'gain_loss_amt': Decimal('-1.441863447940'), 'gain_loss_amt_net': Decimal('-1.439770520880'), 'gain_loss_pct_est': Decimal('-7.110800000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-16.006000000000'), 'gain_loss_pct': Decimal('-7.209300000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1d', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 27814, 'bo_uuid': '372057d9-4027-4807-b1f3-771af243ce46', 'buy_curr_symb': 'POL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'POL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'POL', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('26020434.110127510000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 9, 15, 50), 'check_last_dttm': datetime.datetime(2024, 11, 27, 22, 36, 6), 'add_dttm': datetime.datetime(2024, 11, 25, 9, 15, 50), 'upd_dttm': datetime.datetime(2024, 11, 25, 9, 15, 50), 'del_dttm': datetime.datetime(2024, 11, 25, 9, 15, 50), 'dlm': datetime.datetime(2024, 11, 27, 22, 36, 34)},
		{'pos_id': 30307, 'symb': 'USDC', 'prod_id': 'PEPE-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 15, 36, 38), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 3, 22, 20), 'age_mins': 406, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('5.806627987650'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.014552952350'), 'fees_cnt_tot': Decimal('0.029515545866'), 'buy_cnt': Decimal('297911.269610940300'), 'sell_cnt_tot': Decimal('297911.000000000000'), 'hold_cnt': Decimal('0.269610940304'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.269610940304'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.000020140000'), 'prc_curr': Decimal('0.000019490000'), 'prc_high': Decimal('0.000020410000'), 'prc_low': Decimal('0.000019490000'), 'prc_chg_pct': Decimal('-3.227400000000'), 'prc_chg_pct_high': Decimal('0.794400000000'), 'prc_chg_pct_low': Decimal('-3.227400000000'), 'prc_chg_pct_drop': Decimal('-4.020000000000'), 'prc_sell_avg': Decimal('0.000019490000'), 'val_curr': Decimal('0.000005254717'), 'val_tot': Decimal('0.000005254717'), 'gain_loss_amt_est': Decimal('-0.160939115626'), 'gain_loss_amt_est_high': Decimal('0.047598773102'), 'gain_loss_amt_est_low': Decimal('-0.160939115626'), 'gain_loss_amt': Decimal('-0.193366757633'), 'gain_loss_amt_net': Decimal('-0.193361502916'), 'gain_loss_pct_est': Decimal('-2.682300000000'), 'gain_loss_pct_est_high': Decimal('0.793300000000'), 'gain_loss_pct_est_low': Decimal('-2.682300000000'), 'gain_loss_pct': Decimal('-3.222800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '1h', 'sell_strat_type': 'strat', 'sell_strat_name': 'sha', 'sell_strat_freq': '1h', 'bo_id': 29961, 'bo_uuid': 'ecafc17b-7bab-41da-ae9f-44b8ec43a78e', 'buy_curr_symb': 'PEPE', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PEPE', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PEPE', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('539937939102.841000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('1.0000E-8'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 15, 37, 2), 'check_last_dttm': datetime.datetime(2024, 11, 27, 22, 22, 13), 'add_dttm': datetime.datetime(2024, 11, 27, 15, 37, 2), 'upd_dttm': datetime.datetime(2024, 11, 27, 15, 37, 2), 'del_dttm': datetime.datetime(2024, 11, 27, 15, 37, 2), 'dlm': datetime.datetime(2024, 11, 27, 22, 23, 9)},
		{'pos_id': 30533, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 21, 39, 33), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 3, 13, 45), 'age_mins': 35, 'tot_out_cnt': Decimal('17.000000000000'), 'tot_in_cnt': Decimal('17.076781050000'), 'buy_fees_cnt': Decimal('0.042394014963'), 'sell_fees_cnt_tot': Decimal('0.042798950000'), 'fees_cnt_tot': Decimal('0.085192964963'), 'buy_cnt': Decimal('1003.408638167894'), 'sell_cnt_tot': Decimal('1003.000000000000'), 'hold_cnt': Decimal('0.408638167894'), 'pocket_cnt': Decimal('0.408638167894'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.016942250000'), 'prc_curr': Decimal('0.017025700000'), 'prc_high': Decimal('0.017599000000'), 'prc_low': Decimal('0.016840000000'), 'prc_chg_pct': Decimal('0.492600000000'), 'prc_chg_pct_high': Decimal('3.675700000000'), 'prc_chg_pct_low': Decimal('-0.603500000000'), 'prc_chg_pct_drop': Decimal('-3.180000000000'), 'prc_sell_avg': Decimal('0.017025700000'), 'val_curr': Decimal('0.006957350855'), 'val_tot': Decimal('0.006957350855'), 'gain_loss_amt_est': Decimal('0.298764922014'), 'gain_loss_amt_est_high': Decimal('0.624872729419'), 'gain_loss_amt_est_low': Decimal('-0.102598533253'), 'gain_loss_amt': Decimal('0.083738400855'), 'gain_loss_amt_net': Decimal('0.090695751710'), 'gain_loss_pct_est': Decimal('1.757400000000'), 'gain_loss_pct_est_high': Decimal('3.675700000000'), 'gain_loss_pct_est_low': Decimal('-0.603500000000'), 'gain_loss_pct': Decimal('0.492600000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30187, 'bo_uuid': 'a898d6cd-55c7-4b6f-89c8-4d568b53108d', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 21, 39, 47), 'check_last_dttm': datetime.datetime(2024, 11, 27, 22, 13, 38), 'add_dttm': datetime.datetime(2024, 11, 27, 21, 39, 47), 'upd_dttm': datetime.datetime(2024, 11, 27, 21, 39, 47), 'del_dttm': datetime.datetime(2024, 11, 27, 21, 39, 47), 'dlm': datetime.datetime(2024, 11, 27, 22, 13, 54)},
		{'pos_id': 30531, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 21, 28, 37), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 2, 59, 2), 'age_mins': 31, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.061242915000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015191085000'), 'fees_cnt_tot': Decimal('0.030153678516'), 'buy_cnt': Decimal('359.980597045819'), 'sell_cnt_tot': Decimal('359.000000000000'), 'hold_cnt': Decimal('0.980597045819'), 'pocket_cnt': Decimal('0.980597045819'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.016667560000'), 'prc_curr': Decimal('0.016883680000'), 'prc_high': Decimal('0.017499000000'), 'prc_low': Decimal('0.016050000000'), 'prc_chg_pct': Decimal('1.296700000000'), 'prc_chg_pct_high': Decimal('4.094400000000'), 'prc_chg_pct_low': Decimal('-3.705200000000'), 'prc_chg_pct_drop': Decimal('-2.800000000000'), 'prc_sell_avg': Decimal('0.016883680000'), 'val_curr': Decimal('0.016556086731'), 'val_tot': Decimal('0.016556086731'), 'gain_loss_amt_est': Decimal('0.111030615450'), 'gain_loss_amt_est_high': Decimal('0.245663358745'), 'gain_loss_amt_est_low': Decimal('-0.222311417415'), 'gain_loss_amt': Decimal('0.077799001731'), 'gain_loss_amt_net': Decimal('0.094355088461'), 'gain_loss_pct_est': Decimal('1.850500000000'), 'gain_loss_pct_est_high': Decimal('4.094400000000'), 'gain_loss_pct_est_low': Decimal('-3.705200000000'), 'gain_loss_pct': Decimal('1.296700000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30185, 'bo_uuid': '9ede0813-17c0-4a0f-b80b-5d93a7d447a2', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 21, 29, 35), 'check_last_dttm': datetime.datetime(2024, 11, 27, 21, 58, 55), 'add_dttm': datetime.datetime(2024, 11, 27, 21, 29, 35), 'upd_dttm': datetime.datetime(2024, 11, 27, 21, 29, 35), 'del_dttm': datetime.datetime(2024, 11, 27, 21, 29, 35), 'dlm': datetime.datetime(2024, 11, 27, 21, 59, 44)},
		{'pos_id': 30178, 'symb': 'USDC', 'prod_id': 'DOGE-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 12, 20, 58), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 2, 52, 15), 'age_mins': 572, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.166407610000'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.012948390000'), 'fees_cnt_tot': Decimal('0.025417217930'), 'buy_cnt': Decimal('12.460417148599'), 'sell_cnt_tot': Decimal('12.400000000000'), 'hold_cnt': Decimal('0.060417148599'), 'pocket_cnt': Decimal('0.060417148599'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.401270680000'), 'prc_curr': Decimal('0.416645770000'), 'prc_high': Decimal('0.429340000000'), 'prc_low': Decimal('0.397840000000'), 'prc_chg_pct': Decimal('3.831600000000'), 'prc_chg_pct_high': Decimal('5.786400000000'), 'prc_chg_pct_low': Decimal('-0.855000000000'), 'prc_chg_pct_drop': Decimal('-1.950000000000'), 'prc_sell_avg': Decimal('0.416645770000'), 'val_curr': Decimal('0.025172549399'), 'val_tot': Decimal('0.025172549399'), 'gain_loss_amt_est': Decimal('0.195993950966'), 'gain_loss_amt_est_high': Decimal('0.289322475409'), 'gain_loss_amt_est_low': Decimal('-0.042747641601'), 'gain_loss_amt': Decimal('0.191580159399'), 'gain_loss_amt_net': Decimal('0.216752708798'), 'gain_loss_pct_est': Decimal('3.919900000000'), 'gain_loss_pct_est_high': Decimal('5.786400000000'), 'gain_loss_pct_est_low': Decimal('-0.855000000000'), 'gain_loss_pct': Decimal('3.831600000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29832, 'bo_uuid': 'ebb0af83-29b0-4aae-b90c-393cd68ba9af', 'buy_curr_symb': 'DOGE', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOGE', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOGE', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('141834300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 12, 21, 36), 'check_last_dttm': datetime.datetime(2024, 11, 27, 21, 52, 8), 'add_dttm': datetime.datetime(2024, 11, 27, 12, 21, 36), 'upd_dttm': datetime.datetime(2024, 11, 27, 12, 21, 36), 'del_dttm': datetime.datetime(2024, 11, 27, 12, 21, 36), 'dlm': datetime.datetime(2024, 11, 27, 21, 52, 46)},
		{'pos_id': 30396, 'symb': 'USDC', 'prod_id': 'RNDR-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 17, 34, 16), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 2, 37, 6), 'age_mins': 243, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.075062545000'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.012719455000'), 'fees_cnt_tot': Decimal('0.025188282930'), 'buy_cnt': Decimal('0.629706980969'), 'sell_cnt_tot': Decimal('0.620000000000'), 'hold_cnt': Decimal('0.009706980969'), 'pocket_cnt': Decimal('0.009706980969'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('7.940201000000'), 'prc_curr': Decimal('8.185584750000'), 'prc_high': Decimal('8.300000000000'), 'prc_low': Decimal('7.852300000000'), 'prc_chg_pct': Decimal('3.090400000000'), 'prc_chg_pct_high': Decimal('4.071700000000'), 'prc_chg_pct_low': Decimal('-1.107000000000'), 'prc_chg_pct_drop': Decimal('-0.980000000000'), 'prc_sell_avg': Decimal('8.185584750000'), 'val_curr': Decimal('0.079457315388'), 'val_tot': Decimal('0.079457315388'), 'gain_loss_amt_est': Decimal('0.156229672268'), 'gain_loss_amt_est_high': Decimal('0.203583637237'), 'gain_loss_amt_est_low': Decimal('-0.055351873337'), 'gain_loss_amt': Decimal('0.154519860388'), 'gain_loss_amt_net': Decimal('0.233977175777'), 'gain_loss_pct_est': Decimal('3.124600000000'), 'gain_loss_pct_est_high': Decimal('4.071700000000'), 'gain_loss_pct_est_low': Decimal('-1.107000000000'), 'gain_loss_pct': Decimal('3.090400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30050, 'bo_uuid': '8a4651a9-a35f-4a00-8546-bbb2686df5d3', 'buy_curr_symb': 'RNDR', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'RNDR', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'RNDR', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('8235000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 17, 34, 38), 'check_last_dttm': datetime.datetime(2024, 11, 27, 21, 36, 59), 'add_dttm': datetime.datetime(2024, 11, 27, 17, 34, 38), 'upd_dttm': datetime.datetime(2024, 11, 27, 17, 34, 38), 'del_dttm': datetime.datetime(2024, 11, 27, 17, 34, 38), 'dlm': datetime.datetime(2024, 11, 27, 21, 38, 18)},
		{'pos_id': 30397, 'symb': 'USDC', 'prod_id': 'RENDER-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 17, 34, 52), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 2, 37, 3), 'age_mins': 242, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.078711400000'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.012728600000'), 'fees_cnt_tot': Decimal('0.025197427930'), 'buy_cnt': Decimal('0.629500337255'), 'sell_cnt_tot': Decimal('0.620000000000'), 'hold_cnt': Decimal('0.009500337255'), 'pocket_cnt': Decimal('0.009500337255'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('7.942807500000'), 'prc_curr': Decimal('8.191470000000'), 'prc_high': Decimal('8.303000000000'), 'prc_low': Decimal('7.839000000000'), 'prc_chg_pct': Decimal('3.130700000000'), 'prc_chg_pct_high': Decimal('4.031200000000'), 'prc_chg_pct_low': Decimal('-1.306900000000'), 'prc_chg_pct_drop': Decimal('-0.900000000000'), 'prc_sell_avg': Decimal('8.191470000000'), 'val_curr': Decimal('0.077821727614'), 'val_tot': Decimal('0.077821727614'), 'gain_loss_amt_est': Decimal('0.158125763467'), 'gain_loss_amt_est_high': Decimal('0.201561286738'), 'gain_loss_amt_est_low': Decimal('-0.065346856258'), 'gain_loss_amt': Decimal('0.156533127614'), 'gain_loss_amt_net': Decimal('0.234354855228'), 'gain_loss_pct_est': Decimal('3.162500000000'), 'gain_loss_pct_est_high': Decimal('4.031200000000'), 'gain_loss_pct_est_low': Decimal('-1.306900000000'), 'gain_loss_pct': Decimal('3.130700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30051, 'bo_uuid': '1a18cb93-2504-414f-971f-4a083a991e49', 'buy_curr_symb': 'RENDER', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'RENDER', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'RENDER', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('1982431.303836978700'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 17, 35, 54), 'check_last_dttm': datetime.datetime(2024, 11, 27, 21, 36, 56), 'add_dttm': datetime.datetime(2024, 11, 27, 17, 35, 54), 'upd_dttm': datetime.datetime(2024, 11, 27, 17, 35, 54), 'del_dttm': datetime.datetime(2024, 11, 27, 17, 35, 54), 'dlm': datetime.datetime(2024, 11, 27, 21, 38, 12)},
		{'pos_id': 30470, 'symb': 'USDC', 'prod_id': 'ZEN-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 19, 52, 36), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 2, 35, 12), 'age_mins': 102, 'tot_out_cnt': Decimal('4.000000000000'), 'tot_in_cnt': Decimal('3.668225452500'), 'buy_fees_cnt': Decimal('0.009975062344'), 'sell_fees_cnt_tot': Decimal('0.009193547500'), 'fees_cnt_tot': Decimal('0.019168609844'), 'buy_cnt': Decimal('0.241293235224'), 'sell_cnt_tot': Decimal('0.241000000000'), 'hold_cnt': Decimal('0.000293235224'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.000293235224'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('16.577340000000'), 'prc_curr': Decimal('15.220852500000'), 'prc_high': Decimal('16.555000000000'), 'prc_low': Decimal('15.220852500000'), 'prc_chg_pct': Decimal('-8.182800000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-8.182800000000'), 'prc_chg_pct_drop': Decimal('-8.180000000000'), 'prc_sell_avg': Decimal('15.220852500000'), 'val_curr': Decimal('0.004463290092'), 'val_tot': Decimal('0.004463290092'), 'gain_loss_amt_est': Decimal('-0.315693591365'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-0.315693591365'), 'gain_loss_amt': Decimal('-0.327311257408'), 'gain_loss_amt_net': Decimal('-0.322847967315'), 'gain_loss_pct_est': Decimal('-7.892300000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-7.892300000000'), 'gain_loss_pct': Decimal('-8.182800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 30124, 'bo_uuid': '20198fcb-83b9-43a7-95df-a93d467c160a', 'buy_curr_symb': 'ZEN', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ZEN', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ZEN', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('1005700.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 19, 52, 48), 'check_last_dttm': datetime.datetime(2024, 11, 27, 21, 35, 5), 'add_dttm': datetime.datetime(2024, 11, 27, 19, 52, 48), 'upd_dttm': datetime.datetime(2024, 11, 27, 19, 52, 48), 'del_dttm': datetime.datetime(2024, 11, 27, 19, 52, 48), 'dlm': datetime.datetime(2024, 11, 27, 21, 36, 13)},
		{'pos_id': 30505, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 20, 40, 37), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 2, 35, 12), 'age_mins': 55, 'tot_out_cnt': Decimal('8.000000000000'), 'tot_in_cnt': Decimal('7.299542407500'), 'buy_fees_cnt': Decimal('0.019950124688'), 'sell_fees_cnt_tot': Decimal('0.018294592500'), 'fees_cnt_tot': Decimal('0.038244717188'), 'buy_cnt': Decimal('459.283446061106'), 'sell_cnt_tot': Decimal('459.000000000000'), 'hold_cnt': Decimal('0.283446061106'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.283446061106'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.017418440000'), 'prc_curr': Decimal('0.015903140000'), 'prc_high': Decimal('0.017600000000'), 'prc_low': Decimal('0.015903140000'), 'prc_chg_pct': Decimal('-8.699400000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-8.699400000000'), 'prc_chg_pct_drop': Decimal('-8.700000000000'), 'prc_sell_avg': Decimal('0.015903140000'), 'val_curr': Decimal('0.004507682392'), 'val_tot': Decimal('0.004507682392'), 'gain_loss_amt_est': Decimal('-0.628500690719'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-0.628500690719'), 'gain_loss_amt': Decimal('-0.695949910108'), 'gain_loss_amt_net': Decimal('-0.691442227716'), 'gain_loss_pct_est': Decimal('-7.856300000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-7.856300000000'), 'gain_loss_pct': Decimal('-8.699400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 30159, 'bo_uuid': '0dfe9a7d-dc41-4db9-92f7-59ad00362c3b', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 20, 42, 15), 'check_last_dttm': datetime.datetime(2024, 11, 27, 21, 35, 5), 'add_dttm': datetime.datetime(2024, 11, 27, 20, 42, 15), 'upd_dttm': datetime.datetime(2024, 11, 27, 20, 42, 15), 'del_dttm': datetime.datetime(2024, 11, 27, 20, 42, 15), 'dlm': datetime.datetime(2024, 11, 27, 21, 36, 20)},
		{'pos_id': 30449, 'symb': 'USDC', 'prod_id': 'ZEN-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 19, 41, 46), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 2, 34, 34), 'age_mins': 113, 'tot_out_cnt': Decimal('10.000000000000'), 'tot_in_cnt': Decimal('9.197009850000'), 'buy_fees_cnt': Decimal('0.024937655860'), 'sell_fees_cnt_tot': Decimal('0.023050150000'), 'fees_cnt_tot': Decimal('0.047987805860'), 'buy_cnt': Decimal('0.604585874546'), 'sell_cnt_tot': Decimal('0.604000000000'), 'hold_cnt': Decimal('0.000585874546'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.000585874546'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('16.540247500000'), 'prc_curr': Decimal('15.226837500000'), 'prc_high': Decimal('16.692000000000'), 'prc_low': Decimal('15.226837500000'), 'prc_chg_pct': Decimal('-7.940700000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-7.940700000000'), 'prc_chg_pct_drop': Decimal('-7.940000000000'), 'prc_sell_avg': Decimal('15.226837500000'), 'val_curr': Decimal('0.008921016507'), 'val_tot': Decimal('0.008921016507'), 'gain_loss_amt_est': Decimal('-0.768578281557'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-0.768578281557'), 'gain_loss_amt': Decimal('-0.794069133493'), 'gain_loss_amt_net': Decimal('-0.785148116985'), 'gain_loss_pct_est': Decimal('-7.685800000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-7.685800000000'), 'gain_loss_pct': Decimal('-7.940700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 30103, 'bo_uuid': 'f7d34814-c6f0-4c99-ad22-7d0f42ad0786', 'buy_curr_symb': 'ZEN', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ZEN', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ZEN', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('1005700.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 19, 42, 7), 'check_last_dttm': datetime.datetime(2024, 11, 27, 21, 34, 26), 'add_dttm': datetime.datetime(2024, 11, 27, 19, 42, 7), 'upd_dttm': datetime.datetime(2024, 11, 27, 19, 42, 7), 'del_dttm': datetime.datetime(2024, 11, 27, 19, 42, 7), 'dlm': datetime.datetime(2024, 11, 27, 21, 34, 40)},
		{'pos_id': 30508, 'symb': 'USDC', 'prod_id': 'SKL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 20, 51, 29), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 2, 27, 17), 'age_mins': 36, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.174185850000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015474150000'), 'fees_cnt_tot': Decimal('0.030436743516'), 'buy_cnt': Decimal('82.325136265252'), 'sell_cnt_tot': Decimal('82.200000000000'), 'hold_cnt': Decimal('0.125136265252'), 'pocket_cnt': Decimal('0.125136265252'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.072881750000'), 'prc_curr': Decimal('0.075111750000'), 'prc_high': Decimal('0.077000000000'), 'prc_low': Decimal('0.072300000000'), 'prc_chg_pct': Decimal('3.059800000000'), 'prc_chg_pct_high': Decimal('3.455300000000'), 'prc_chg_pct_low': Decimal('-0.798200000000'), 'prc_chg_pct_drop': Decimal('-0.400000000000'), 'prc_sell_avg': Decimal('0.075111750000'), 'val_curr': Decimal('0.009399203872'), 'val_tot': Decimal('0.009399203872'), 'gain_loss_amt_est': Decimal('0.166152706267'), 'gain_loss_amt_est_high': Decimal('0.207315274400'), 'gain_loss_amt_est_low': Decimal('-0.047892648022'), 'gain_loss_amt': Decimal('0.183585053872'), 'gain_loss_amt_net': Decimal('0.192984257743'), 'gain_loss_pct_est': Decimal('2.769200000000'), 'gain_loss_pct_est_high': Decimal('3.455300000000'), 'gain_loss_pct_est_low': Decimal('-0.798200000000'), 'gain_loss_pct': Decimal('3.059800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30162, 'bo_uuid': 'afad7404-7a5f-447c-967c-39b9133a32de', 'buy_curr_symb': 'SKL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SKL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SKL', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('257817000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 20, 52, 20), 'check_last_dttm': datetime.datetime(2024, 11, 27, 21, 27, 10), 'add_dttm': datetime.datetime(2024, 11, 27, 20, 52, 20), 'upd_dttm': datetime.datetime(2024, 11, 27, 20, 52, 20), 'del_dttm': datetime.datetime(2024, 11, 27, 20, 52, 20), 'dlm': datetime.datetime(2024, 11, 27, 21, 28, 3)},
		{'pos_id': 30507, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 20, 51, 11), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 2, 11, 5), 'age_mins': 19, 'tot_out_cnt': Decimal('4.000000000000'), 'tot_in_cnt': Decimal('3.678197460000'), 'buy_fees_cnt': Decimal('0.009975062344'), 'sell_fees_cnt_tot': Decimal('0.009218540000'), 'fees_cnt_tot': Decimal('0.019193602344'), 'buy_cnt': Decimal('226.963875862108'), 'sell_cnt_tot': Decimal('226.000000000000'), 'hold_cnt': Decimal('0.963875862108'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.963875862108'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.017623950000'), 'prc_curr': Decimal('0.016275210000'), 'prc_high': Decimal('0.017580000000'), 'prc_low': Decimal('0.016275210000'), 'prc_chg_pct': Decimal('-7.652900000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-7.652900000000'), 'prc_chg_pct_drop': Decimal('-7.650000000000'), 'prc_sell_avg': Decimal('0.016275210000'), 'val_curr': Decimal('0.015687282070'), 'val_tot': Decimal('0.015687282070'), 'gain_loss_amt_est': Decimal('-0.302531498330'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-0.302531498330'), 'gain_loss_amt': Decimal('-0.306115257930'), 'gain_loss_amt_net': Decimal('-0.290427975861'), 'gain_loss_pct_est': Decimal('-7.563300000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-7.563300000000'), 'gain_loss_pct': Decimal('-7.652900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 30161, 'bo_uuid': '995d4c76-de2a-4a07-90cb-6b4fabff28bc', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 20, 52, 15), 'check_last_dttm': datetime.datetime(2024, 11, 27, 21, 10, 58), 'add_dttm': datetime.datetime(2024, 11, 27, 20, 52, 15), 'upd_dttm': datetime.datetime(2024, 11, 27, 20, 52, 15), 'del_dttm': datetime.datetime(2024, 11, 27, 20, 52, 15), 'dlm': datetime.datetime(2024, 11, 27, 21, 12, 34)},
		{'pos_id': 30476, 'symb': 'USDC', 'prod_id': 'LCX-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 20, 2, 20), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 1, 46, 5), 'age_mins': 44, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('18.442867275000'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.046222725000'), 'fees_cnt_tot': Decimal('0.096098036721'), 'buy_cnt': Decimal('59.766700683880'), 'sell_cnt_tot': Decimal('59.700000000000'), 'hold_cnt': Decimal('0.066700683880'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.066700683880'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.334634500000'), 'prc_curr': Decimal('0.308925750000'), 'prc_high': Decimal('0.333800000000'), 'prc_low': Decimal('0.308900000000'), 'prc_chg_pct': Decimal('-7.682600000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-7.690300000000'), 'prc_chg_pct_drop': Decimal('-7.680000000000'), 'prc_sell_avg': Decimal('0.308925750000'), 'val_curr': Decimal('0.020605558793'), 'val_tot': Decimal('0.020605558793'), 'gain_loss_amt_est': Decimal('-1.538066158749'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-1.538066158749'), 'gain_loss_amt': Decimal('-1.536527166207'), 'gain_loss_amt_net': Decimal('-1.515921607414'), 'gain_loss_pct_est': Decimal('-7.690300000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-7.690300000000'), 'gain_loss_pct': Decimal('-7.682600000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 30130, 'bo_uuid': '63e927ed-8e6f-4d35-a39d-cf89ed432bea', 'buy_curr_symb': 'LCX', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'LCX', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'LCX', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('117651200.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 20, 2, 47), 'check_last_dttm': datetime.datetime(2024, 11, 27, 20, 45, 58), 'add_dttm': datetime.datetime(2024, 11, 27, 20, 2, 47), 'upd_dttm': datetime.datetime(2024, 11, 27, 20, 2, 47), 'del_dttm': datetime.datetime(2024, 11, 27, 20, 2, 47), 'dlm': datetime.datetime(2024, 11, 27, 20, 47, 19)},
		{'pos_id': 30469, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 19, 52, 22), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 1, 28, 56), 'age_mins': 37, 'tot_out_cnt': Decimal('9.000000000000'), 'tot_in_cnt': Decimal('9.239119312500'), 'buy_fees_cnt': Decimal('0.022443890274'), 'sell_fees_cnt_tot': Decimal('0.023155687500'), 'fees_cnt_tot': Decimal('0.045599577774'), 'buy_cnt': Decimal('545.515957326711'), 'sell_cnt_tot': Decimal('545.000000000000'), 'hold_cnt': Decimal('0.515957326711'), 'pocket_cnt': Decimal('0.515957326711'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.016498140000'), 'prc_curr': Decimal('0.016952510000'), 'prc_high': Decimal('0.018500000000'), 'prc_low': Decimal('0.016457000000'), 'prc_chg_pct': Decimal('2.754100000000'), 'prc_chg_pct_high': Decimal('7.224200000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-4.470000000000'), 'prc_sell_avg': Decimal('0.016952510000'), 'val_curr': Decimal('0.008746771741'), 'val_tot': Decimal('0.008746771741'), 'gain_loss_amt_est': Decimal('0.271043694767'), 'gain_loss_amt_est_high': Decimal('0.650177285110'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.247866084241'), 'gain_loss_amt_net': Decimal('0.256612855981'), 'gain_loss_pct_est': Decimal('3.011600000000'), 'gain_loss_pct_est_high': Decimal('7.224200000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('2.754100000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30123, 'bo_uuid': 'd6566793-512c-490f-b110-df77a5999c22', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 19, 52, 44), 'check_last_dttm': datetime.datetime(2024, 11, 27, 20, 28, 49), 'add_dttm': datetime.datetime(2024, 11, 27, 19, 52, 44), 'upd_dttm': datetime.datetime(2024, 11, 27, 19, 52, 44), 'del_dttm': datetime.datetime(2024, 11, 27, 19, 52, 44), 'dlm': datetime.datetime(2024, 11, 27, 20, 30, 6)},
		{'pos_id': 30446, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 19, 31, 9), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 1, 28, 36), 'age_mins': 58, 'tot_out_cnt': Decimal('10.000000000000'), 'tot_in_cnt': Decimal('10.327724977500'), 'buy_fees_cnt': Decimal('0.024937655860'), 'sell_fees_cnt_tot': Decimal('0.025884022500'), 'fees_cnt_tot': Decimal('0.050821678360'), 'buy_cnt': Decimal('609.610850341603'), 'sell_cnt_tot': Decimal('609.000000000000'), 'hold_cnt': Decimal('0.610850341603'), 'pocket_cnt': Decimal('0.610850341603'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.016403910000'), 'prc_curr': Decimal('0.016958500000'), 'prc_high': Decimal('0.018500000000'), 'prc_low': Decimal('0.016201000000'), 'prc_chg_pct': Decimal('3.380800000000'), 'prc_chg_pct_high': Decimal('7.840100000000'), 'prc_chg_pct_low': Decimal('-1.237000000000'), 'prc_chg_pct_drop': Decimal('-4.460000000000'), 'prc_sell_avg': Decimal('0.016958500000'), 'val_curr': Decimal('0.010359105518'), 'val_tot': Decimal('0.010359105518'), 'gain_loss_amt_est': Decimal('0.360336401556'), 'gain_loss_amt_est_high': Decimal('0.784015942543'), 'gain_loss_amt_est_low': Decimal('-0.123694613616'), 'gain_loss_amt': Decimal('0.338084083018'), 'gain_loss_amt_net': Decimal('0.348443188536'), 'gain_loss_pct_est': Decimal('3.603400000000'), 'gain_loss_pct_est_high': Decimal('7.840200000000'), 'gain_loss_pct_est_low': Decimal('-1.236900000000'), 'gain_loss_pct': Decimal('3.380800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30100, 'bo_uuid': 'd153f820-add8-413f-a685-7d4e0216b03f', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 19, 32, 21), 'check_last_dttm': datetime.datetime(2024, 11, 27, 20, 28, 29), 'add_dttm': datetime.datetime(2024, 11, 27, 19, 32, 21), 'upd_dttm': datetime.datetime(2024, 11, 27, 19, 32, 21), 'del_dttm': datetime.datetime(2024, 11, 27, 19, 32, 21), 'dlm': datetime.datetime(2024, 11, 27, 20, 28, 45)},
		{'pos_id': 29008, 'symb': 'USDC', 'prod_id': 'DRIFT-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 26, 4, 16, 25), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 1, 17, 36), 'age_mins': 2402, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.120758550000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007821450000'), 'fees_cnt_tot': Decimal('0.015302746758'), 'buy_cnt': Decimal('2.348915779625'), 'sell_cnt_tot': Decimal('2.340000000000'), 'hold_cnt': Decimal('0.008915779625'), 'pocket_cnt': Decimal('0.008915779625'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.277185000000'), 'prc_curr': Decimal('1.333657500000'), 'prc_high': Decimal('1.375000000000'), 'prc_low': Decimal('1.213000000000'), 'prc_chg_pct': Decimal('4.421600000000'), 'prc_chg_pct_high': Decimal('6.797400000000'), 'prc_chg_pct_low': Decimal('-5.025500000000'), 'prc_chg_pct_drop': Decimal('-2.380000000000'), 'prc_sell_avg': Decimal('1.333657500000'), 'val_curr': Decimal('0.011890596365'), 'val_tot': Decimal('0.011890596365'), 'gain_loss_amt_est': Decimal('0.142849313138'), 'gain_loss_amt_est_high': Decimal('0.203921123409'), 'gain_loss_amt_est_low': Decimal('-0.150765159315'), 'gain_loss_amt': Decimal('0.132649146365'), 'gain_loss_amt_net': Decimal('0.144539742730'), 'gain_loss_pct_est': Decimal('4.761600000000'), 'gain_loss_pct_est_high': Decimal('6.797400000000'), 'gain_loss_pct_est_low': Decimal('-5.025500000000'), 'gain_loss_pct': Decimal('4.421600000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28661, 'bo_uuid': '3635c198-cf4d-4ddd-8a2d-3d44b06132c9', 'buy_curr_symb': 'DRIFT', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DRIFT', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DRIFT', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('6666666.666666667000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 26, 4, 16, 48), 'check_last_dttm': datetime.datetime(2024, 11, 27, 20, 17, 29), 'add_dttm': datetime.datetime(2024, 11, 26, 4, 16, 48), 'upd_dttm': datetime.datetime(2024, 11, 26, 4, 16, 48), 'del_dttm': datetime.datetime(2024, 11, 26, 4, 16, 48), 'dlm': datetime.datetime(2024, 11, 27, 20, 18, 26)},
		{'pos_id': 28693, 'symb': 'USDC', 'prod_id': 'ETH-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 19, 27, 30), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 1, 10, 6), 'age_mins': 2922, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.740804013155'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.051981964945'), 'fees_cnt_tot': Decimal('0.101857276666'), 'buy_cnt': Decimal('0.005803014258'), 'sell_cnt_tot': Decimal('0.005725690000'), 'hold_cnt': Decimal('0.000077324258'), 'pocket_cnt': Decimal('0.000077324258'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('25.000000000000'), 'clip_pct': Decimal('5.000000000000'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('3446.484725000000'), 'prc_curr': Decimal('3622.411275000000'), 'prc_high': Decimal('3688.540000000000'), 'prc_low': Decimal('3255.890000000000'), 'prc_chg_pct': Decimal('5.104500000000'), 'prc_chg_pct_high': Decimal('6.795800000000'), 'prc_chg_pct_low': Decimal('-5.530100000000'), 'prc_chg_pct_drop': Decimal('-1.690000000000'), 'prc_sell_avg': Decimal('3622.411275000000'), 'val_curr': Decimal('0.280100264010'), 'val_tot': Decimal('0.280100264010'), 'gain_loss_amt_est': Decimal('1.065986299106'), 'gain_loss_amt_est_high': Decimal('1.359154579421'), 'gain_loss_amt_est_low': Decimal('-1.106023907520'), 'gain_loss_amt': Decimal('1.020904277165'), 'gain_loss_amt_net': Decimal('1.301004541175'), 'gain_loss_pct_est': Decimal('5.329900000000'), 'gain_loss_pct_est_high': Decimal('6.795800000000'), 'gain_loss_pct_est_low': Decimal('-5.530100000000'), 'gain_loss_pct': Decimal('5.104500000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28346, 'bo_uuid': '690261c0-d561-43a9-a2ee-3e7c99634ffc', 'buy_curr_symb': 'ETH', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ETH', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ETH', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('42000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('150000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 19, 28, 29), 'check_last_dttm': datetime.datetime(2024, 11, 27, 20, 9, 59), 'add_dttm': datetime.datetime(2024, 11, 25, 19, 28, 29), 'upd_dttm': datetime.datetime(2024, 11, 25, 19, 28, 29), 'del_dttm': datetime.datetime(2024, 11, 25, 19, 28, 29), 'dlm': datetime.datetime(2024, 11, 27, 20, 11, 9)},
		{'pos_id': 28692, 'symb': 'USDC', 'prod_id': 'ETH-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 19, 27, 21), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 1, 9, 50), 'age_mins': 2923, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.727798244865'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.051949369035'), 'fees_cnt_tot': Decimal('0.101824680756'), 'buy_cnt': Decimal('0.005805226326'), 'sell_cnt_tot': Decimal('0.005727280000'), 'hold_cnt': Decimal('0.000077946326'), 'pocket_cnt': Decimal('0.000077946326'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('25.000000000000'), 'clip_pct': Decimal('5.000000000000'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('3445.171450000000'), 'prc_curr': Decimal('3619.134780360000'), 'prc_high': Decimal('3688.540000000000'), 'prc_low': Decimal('3255.890000000000'), 'prc_chg_pct': Decimal('5.049500000000'), 'prc_chg_pct_high': Decimal('6.836500000000'), 'prc_chg_pct_low': Decimal('-5.494100000000'), 'prc_chg_pct_drop': Decimal('-1.790000000000'), 'prc_sell_avg': Decimal('3619.134780360000'), 'val_curr': Decimal('0.282098259428'), 'val_tot': Decimal('0.282098259428'), 'gain_loss_amt_est': Decimal('1.074016504119'), 'gain_loss_amt_est_high': Decimal('1.367296538108'), 'gain_loss_amt_est_low': Decimal('-1.098821657440'), 'gain_loss_amt': Decimal('1.009896504293'), 'gain_loss_amt_net': Decimal('1.291994763721'), 'gain_loss_pct_est': Decimal('5.370100000000'), 'gain_loss_pct_est_high': Decimal('6.836500000000'), 'gain_loss_pct_est_low': Decimal('-5.494100000000'), 'gain_loss_pct': Decimal('5.049500000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28345, 'bo_uuid': '749dc71c-5a4f-4979-a26d-83e689f184ad', 'buy_curr_symb': 'ETH', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ETH', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ETH', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('42000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('150000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 19, 28, 24), 'check_last_dttm': datetime.datetime(2024, 11, 27, 20, 9, 43), 'add_dttm': datetime.datetime(2024, 11, 25, 19, 28, 24), 'upd_dttm': datetime.datetime(2024, 11, 25, 19, 28, 24), 'del_dttm': datetime.datetime(2024, 11, 25, 19, 28, 24), 'dlm': datetime.datetime(2024, 11, 27, 20, 11, 4)},
		{'pos_id': 28691, 'symb': 'USDC', 'prod_id': 'ETH-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 19, 27, 11), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 1, 9, 36), 'age_mins': 2923, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.754894443493'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.052017279307'), 'fees_cnt_tot': Decimal('0.101892591028'), 'buy_cnt': Decimal('0.005810467714'), 'sell_cnt_tot': Decimal('0.005731070000'), 'hold_cnt': Decimal('0.000079397714'), 'pocket_cnt': Decimal('0.000079397714'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('25.000000000000'), 'clip_pct': Decimal('5.000000000000'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('3442.063700000000'), 'prc_curr': Decimal('3621.469366710000'), 'prc_high': Decimal('3688.540000000000'), 'prc_low': Decimal('3255.890000000000'), 'prc_chg_pct': Decimal('5.212200000000'), 'prc_chg_pct_high': Decimal('6.932900000000'), 'prc_chg_pct_low': Decimal('-5.408800000000'), 'prc_chg_pct_drop': Decimal('-1.720000000000'), 'prc_sell_avg': Decimal('3621.469366710000'), 'val_curr': Decimal('0.287536389038'), 'val_tot': Decimal('0.287536389038'), 'gain_loss_amt_est': Decimal('1.093043686009'), 'gain_loss_amt_est_high': Decimal('1.386588514920'), 'gain_loss_amt_est_low': Decimal('-1.081756274665'), 'gain_loss_amt': Decimal('1.042430832531'), 'gain_loss_amt_net': Decimal('1.329967221569'), 'gain_loss_pct_est': Decimal('5.465200000000'), 'gain_loss_pct_est_high': Decimal('6.932900000000'), 'gain_loss_pct_est_low': Decimal('-5.408800000000'), 'gain_loss_pct': Decimal('5.212200000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28344, 'bo_uuid': '7be96789-01c8-4def-a265-3d4a90ebd506', 'buy_curr_symb': 'ETH', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ETH', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ETH', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('42000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('150000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 19, 28, 18), 'check_last_dttm': datetime.datetime(2024, 11, 27, 20, 9, 29), 'add_dttm': datetime.datetime(2024, 11, 25, 19, 28, 18), 'upd_dttm': datetime.datetime(2024, 11, 25, 19, 28, 18), 'del_dttm': datetime.datetime(2024, 11, 25, 19, 28, 18), 'dlm': datetime.datetime(2024, 11, 27, 20, 9, 49)},
		{'pos_id': 30434, 'symb': 'USDC', 'prod_id': 'LCX-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 19, 24, 46), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 0, 46, 45), 'age_mins': 22, 'tot_out_cnt': Decimal('18.000000000000'), 'tot_in_cnt': Decimal('16.378870200000'), 'buy_fees_cnt': Decimal('0.044887780549'), 'sell_fees_cnt_tot': Decimal('0.041049800000'), 'fees_cnt_tot': Decimal('0.085937580549'), 'buy_cnt': Decimal('51.714032890125'), 'sell_cnt_tot': Decimal('51.700000000000'), 'hold_cnt': Decimal('0.014032890125'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.014032890125'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.348068000000'), 'prc_curr': Decimal('0.316806000000'), 'prc_high': Decimal('0.347200000000'), 'prc_low': Decimal('0.316806000000'), 'prc_chg_pct': Decimal('-8.981600000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-8.981600000000'), 'prc_chg_pct_drop': Decimal('-8.980000000000'), 'prc_sell_avg': Decimal('0.316806000000'), 'val_curr': Decimal('0.004445703789'), 'val_tot': Decimal('0.004445703789'), 'gain_loss_amt_est': Decimal('-1.560108944229'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-1.560108944229'), 'gain_loss_amt': Decimal('-1.616684096211'), 'gain_loss_amt_net': Decimal('-1.612238392422'), 'gain_loss_pct_est': Decimal('-8.667300000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-8.667300000000'), 'gain_loss_pct': Decimal('-8.981600000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 30088, 'bo_uuid': 'c4062946-3176-4350-b441-d71283e654f5', 'buy_curr_symb': 'LCX', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'LCX', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'LCX', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('117651200.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 19, 25, 5), 'check_last_dttm': datetime.datetime(2024, 11, 27, 19, 46, 38), 'add_dttm': datetime.datetime(2024, 11, 27, 19, 25, 5), 'upd_dttm': datetime.datetime(2024, 11, 27, 19, 25, 5), 'del_dttm': datetime.datetime(2024, 11, 27, 19, 25, 5), 'dlm': datetime.datetime(2024, 11, 27, 19, 47, 14)},
		{'pos_id': 30406, 'symb': 'USDC', 'prod_id': 'ZEN-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 17, 43, 37), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 0, 39), 'age_mins': 115, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.187000000000'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.013000000000'), 'fees_cnt_tot': Decimal('0.025468827930'), 'buy_cnt': Decimal('0.321237354893'), 'sell_cnt_tot': Decimal('0.320000000000'), 'hold_cnt': Decimal('0.001237354893'), 'pocket_cnt': Decimal('0.001237354893'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('15.564815000000'), 'prc_curr': Decimal('16.209375000000'), 'prc_high': Decimal('16.524000000000'), 'prc_low': Decimal('15.526000000000'), 'prc_chg_pct': Decimal('4.141100000000'), 'prc_chg_pct_high': Decimal('6.066100000000'), 'prc_chg_pct_low': Decimal('-0.210800000000'), 'prc_chg_pct_drop': Decimal('-1.920000000000'), 'prc_sell_avg': Decimal('16.209375000000'), 'val_curr': Decimal('0.020056749469'), 'val_tot': Decimal('0.020056749469'), 'gain_loss_amt_est': Decimal('0.190874417716'), 'gain_loss_amt_est_high': Decimal('0.303307491929'), 'gain_loss_amt_est_low': Decimal('-0.010541403802'), 'gain_loss_amt': Decimal('0.207056749469'), 'gain_loss_amt_net': Decimal('0.227113498937'), 'gain_loss_pct_est': Decimal('3.817500000000'), 'gain_loss_pct_est_high': Decimal('6.066100000000'), 'gain_loss_pct_est_low': Decimal('-0.210800000000'), 'gain_loss_pct': Decimal('4.141100000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30060, 'bo_uuid': '0da7cbb1-4b8b-4395-80ac-6bfa6420676a', 'buy_curr_symb': 'ZEN', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ZEN', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ZEN', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('1005700.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 17, 44, 28), 'check_last_dttm': datetime.datetime(2024, 11, 27, 19, 38, 53), 'add_dttm': datetime.datetime(2024, 11, 27, 17, 44, 28), 'upd_dttm': datetime.datetime(2024, 11, 27, 17, 44, 28), 'del_dttm': datetime.datetime(2024, 11, 27, 17, 44, 28), 'dlm': datetime.datetime(2024, 11, 27, 19, 40, 39)},
		{'pos_id': 30401, 'symb': 'USDC', 'prod_id': 'ZEN-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 17, 37, 35), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 0, 38, 46), 'age_mins': 121, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.195583222500'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015527777500'), 'fees_cnt_tot': Decimal('0.030490371016'), 'buy_cnt': Decimal('0.384222726230'), 'sell_cnt_tot': Decimal('0.383000000000'), 'hold_cnt': Decimal('0.001222726230'), 'pocket_cnt': Decimal('0.001222726230'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('15.615942500000'), 'prc_curr': Decimal('16.176457500000'), 'prc_high': Decimal('16.524000000000'), 'prc_low': Decimal('15.504000000000'), 'prc_chg_pct': Decimal('3.589400000000'), 'prc_chg_pct_high': Decimal('5.718900000000'), 'prc_chg_pct_low': Decimal('-0.716800000000'), 'prc_chg_pct_drop': Decimal('-2.130000000000'), 'prc_sell_avg': Decimal('16.176457500000'), 'val_curr': Decimal('0.019779378894'), 'val_tot': Decimal('0.019779378894'), 'gain_loss_amt_est': Decimal('0.208655033151'), 'gain_loss_amt_est_high': Decimal('0.343132987331'), 'gain_loss_amt_est_low': Decimal('-0.043010852530'), 'gain_loss_amt': Decimal('0.215362601394'), 'gain_loss_amt_net': Decimal('0.235141980287'), 'gain_loss_pct_est': Decimal('3.477600000000'), 'gain_loss_pct_est_high': Decimal('5.718900000000'), 'gain_loss_pct_est_low': Decimal('-0.716800000000'), 'gain_loss_pct': Decimal('3.589400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30055, 'bo_uuid': '9d1b3db2-3b89-478b-8735-cf1e6e8cbf5a', 'buy_curr_symb': 'ZEN', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ZEN', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ZEN', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('1005700.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 17, 38, 32), 'check_last_dttm': datetime.datetime(2024, 11, 27, 19, 38, 39), 'add_dttm': datetime.datetime(2024, 11, 27, 17, 38, 32), 'upd_dttm': datetime.datetime(2024, 11, 27, 17, 38, 32), 'del_dttm': datetime.datetime(2024, 11, 27, 17, 38, 32), 'dlm': datetime.datetime(2024, 11, 27, 19, 40, 33)},
		{'pos_id': 30394, 'symb': 'USDC', 'prod_id': 'ZEN-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 17, 31, 34), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 0, 38, 33), 'age_mins': 127, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.203606115000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015547885000'), 'fees_cnt_tot': Decimal('0.030510478516'), 'buy_cnt': Decimal('0.384173400506'), 'sell_cnt_tot': Decimal('0.383000000000'), 'hold_cnt': Decimal('0.001173400506'), 'pocket_cnt': Decimal('0.001173400506'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('15.617947500000'), 'prc_curr': Decimal('16.197405000000'), 'prc_high': Decimal('16.524000000000'), 'prc_low': Decimal('15.504000000000'), 'prc_chg_pct': Decimal('3.710200000000'), 'prc_chg_pct_high': Decimal('5.705300000000'), 'prc_chg_pct_low': Decimal('-0.729600000000'), 'prc_chg_pct_drop': Decimal('-2.000000000000'), 'prc_sell_avg': Decimal('16.197405000000'), 'val_curr': Decimal('0.019006043223'), 'val_tot': Decimal('0.019006043223'), 'gain_loss_amt_est': Decimal('0.207857978776'), 'gain_loss_amt_est_high': Decimal('0.342318668954'), 'gain_loss_amt_est_low': Decimal('-0.043775598555'), 'gain_loss_amt': Decimal('0.222612158223'), 'gain_loss_amt_net': Decimal('0.241618201446'), 'gain_loss_pct_est': Decimal('3.464300000000'), 'gain_loss_pct_est_high': Decimal('5.705300000000'), 'gain_loss_pct_est_low': Decimal('-0.729600000000'), 'gain_loss_pct': Decimal('3.710200000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30048, 'bo_uuid': '705bd642-b660-4b5a-a442-5ed1f74bf2f2', 'buy_curr_symb': 'ZEN', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ZEN', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ZEN', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('1005700.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 17, 32, 12), 'check_last_dttm': datetime.datetime(2024, 11, 27, 19, 38, 26), 'add_dttm': datetime.datetime(2024, 11, 27, 17, 32, 12), 'upd_dttm': datetime.datetime(2024, 11, 27, 17, 32, 12), 'del_dttm': datetime.datetime(2024, 11, 27, 17, 32, 12), 'dlm': datetime.datetime(2024, 11, 27, 19, 39, 3)},
		{'pos_id': 30302, 'symb': 'USDC', 'prod_id': 'DOT-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 15, 33, 26), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 0, 37, 45), 'age_mins': 244, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('5.866515083957'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.014703045323'), 'fees_cnt_tot': Decimal('0.029665638839'), 'buy_cnt': Decimal('0.701481177506'), 'sell_cnt_tot': Decimal('0.701481170000'), 'hold_cnt': Decimal('7.506E-9'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('7.506E-9'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('8.553330000000'), 'prc_curr': Decimal('8.363040000000'), 'prc_high': Decimal('8.532000000000'), 'prc_low': Decimal('8.356000000000'), 'prc_chg_pct': Decimal('-2.224700000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-2.307100000000'), 'prc_chg_pct_drop': Decimal('-2.220000000000'), 'prc_sell_avg': Decimal('8.363040000000'), 'val_curr': Decimal('6.2773E-8'), 'val_tot': Decimal('6.2773E-8'), 'gain_loss_amt_est': Decimal('-0.120184770145'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-0.138423280760'), 'gain_loss_amt': Decimal('-0.133484853270'), 'gain_loss_amt_net': Decimal('-0.133484790497'), 'gain_loss_pct_est': Decimal('-2.003100000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-2.307100000000'), 'gain_loss_pct': Decimal('-2.224700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '30min', 'sell_strat_type': 'strat', 'sell_strat_name': 'sha', 'sell_strat_freq': '30min', 'bo_id': 29956, 'bo_uuid': '42fd9189-0a44-4fac-b838-724c279ec1bc', 'buy_curr_symb': 'DOT', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOT', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOT', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('1732900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 15, 34, 8), 'check_last_dttm': datetime.datetime(2024, 11, 27, 19, 37, 38), 'add_dttm': datetime.datetime(2024, 11, 27, 15, 34, 8), 'upd_dttm': datetime.datetime(2024, 11, 27, 15, 34, 8), 'del_dttm': datetime.datetime(2024, 11, 27, 15, 34, 8), 'dlm': datetime.datetime(2024, 11, 27, 19, 38, 38)},
		{'pos_id': 30428, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 19, 9, 39), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 0, 30, 9), 'age_mins': 21, 'tot_out_cnt': Decimal('17.000000000000'), 'tot_in_cnt': Decimal('17.383742722500'), 'buy_fees_cnt': Decimal('0.042394014963'), 'sell_fees_cnt_tot': Decimal('0.043568277500'), 'fees_cnt_tot': Decimal('0.085962292463'), 'buy_cnt': Decimal('1067.863280307296'), 'sell_cnt_tot': Decimal('1067.000000000000'), 'hold_cnt': Decimal('0.863280307296'), 'pocket_cnt': Decimal('0.863280307296'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.015919640000'), 'prc_curr': Decimal('0.016292170000'), 'prc_high': Decimal('0.017200000000'), 'prc_low': Decimal('0.015879941091'), 'prc_chg_pct': Decimal('2.340100000000'), 'prc_chg_pct_high': Decimal('5.881800000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-3.540000000000'), 'prc_sell_avg': Decimal('0.016292170000'), 'val_curr': Decimal('0.014064709524'), 'val_tot': Decimal('0.014064709524'), 'gain_loss_amt_est': Decimal('0.566350961055'), 'gain_loss_amt_est_high': Decimal('0.999903452860'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.397807432024'), 'gain_loss_amt_net': Decimal('0.411872141548'), 'gain_loss_pct_est': Decimal('3.331500000000'), 'gain_loss_pct_est_high': Decimal('5.881800000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('2.340000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 30082, 'bo_uuid': '23edb3fe-4d13-42bd-8e5e-ba43c6d39cfd', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 19, 9, 54), 'check_last_dttm': datetime.datetime(2024, 11, 27, 19, 30, 2), 'add_dttm': datetime.datetime(2024, 11, 27, 19, 9, 54), 'upd_dttm': datetime.datetime(2024, 11, 27, 19, 9, 54), 'del_dttm': datetime.datetime(2024, 11, 27, 19, 9, 54), 'dlm': datetime.datetime(2024, 11, 27, 19, 30, 52)},
		{'pos_id': 30353, 'symb': 'USDC', 'prod_id': 'WELL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 16, 45, 57), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 0, 25, 43), 'age_mins': 160, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('2.655017820000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.006654180000'), 'fees_cnt_tot': Decimal('0.014135476758'), 'buy_cnt': Decimal('38.540539155164'), 'sell_cnt_tot': Decimal('38.000000000000'), 'hold_cnt': Decimal('0.540539155164'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.540539155164'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.077840120000'), 'prc_curr': Decimal('0.069868890000'), 'prc_high': Decimal('0.077646000000'), 'prc_low': Decimal('0.069868890000'), 'prc_chg_pct': Decimal('-10.240500000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-10.240500000000'), 'prc_chg_pct_drop': Decimal('-10.240000000000'), 'prc_sell_avg': Decimal('0.069868890000'), 'val_curr': Decimal('0.037766870773'), 'val_tot': Decimal('0.037766870773'), 'gain_loss_amt_est': Decimal('-0.300582097033'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-0.300582097033'), 'gain_loss_amt': Decimal('-0.307215309227'), 'gain_loss_amt_net': Decimal('-0.269448438454'), 'gain_loss_pct_est': Decimal('-10.019400000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-10.019400000000'), 'gain_loss_pct': Decimal('-10.240500000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '15min', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 30007, 'bo_uuid': '04971b1a-10b9-4bd5-92af-e060c5806c81', 'buy_curr_symb': 'WELL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'WELL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'WELL', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('289584259.672040800000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 16, 47), 'check_last_dttm': datetime.datetime(2024, 11, 27, 19, 25, 36), 'add_dttm': datetime.datetime(2024, 11, 27, 16, 47), 'upd_dttm': datetime.datetime(2024, 11, 27, 16, 47), 'del_dttm': datetime.datetime(2024, 11, 27, 16, 47), 'dlm': datetime.datetime(2024, 11, 27, 19, 27, 6)},
		{'pos_id': 28448, 'symb': 'USDC', 'prod_id': 'ETH-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 13, 8, 51), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 0, 10, 22), 'age_mins': 3241, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.097227586446'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007762475154'), 'fees_cnt_tot': Decimal('0.015243771912'), 'buy_cnt': Decimal('0.000860928532'), 'sell_cnt_tot': Decimal('0.000850920000'), 'hold_cnt': Decimal('0.000010008532'), 'pocket_cnt': Decimal('0.000010008532'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('25.000000000000'), 'clip_pct': Decimal('5.000000000000'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('3484.609800000000'), 'prc_curr': Decimal('3639.857550000000'), 'prc_high': Decimal('3688.540000000000'), 'prc_low': Decimal('3255.890000000000'), 'prc_chg_pct': Decimal('4.455200000000'), 'prc_chg_pct_high': Decimal('5.627300000000'), 'prc_chg_pct_low': Decimal('-6.563700000000'), 'prc_chg_pct_drop': Decimal('-1.170000000000'), 'prc_sell_avg': Decimal('3639.857550000000'), 'val_curr': Decimal('0.036429630765'), 'val_tot': Decimal('0.036429630765'), 'gain_loss_amt_est': Decimal('0.139444766221'), 'gain_loss_amt_est_high': Decimal('0.168819647732'), 'gain_loss_amt_est_low': Decimal('-0.196911401947'), 'gain_loss_amt': Decimal('0.133657217211'), 'gain_loss_amt_net': Decimal('0.170086847975'), 'gain_loss_pct_est': Decimal('4.648200000000'), 'gain_loss_pct_est_high': Decimal('5.627300000000'), 'gain_loss_pct_est_low': Decimal('-6.563700000000'), 'gain_loss_pct': Decimal('4.455200000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28101, 'bo_uuid': 'c259cb35-f5ec-4153-8523-cae22d1eb23b', 'buy_curr_symb': 'ETH', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ETH', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ETH', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('42000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('150000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 13, 11, 10), 'check_last_dttm': datetime.datetime(2024, 11, 27, 19, 10, 15), 'add_dttm': datetime.datetime(2024, 11, 25, 13, 11, 10), 'upd_dttm': datetime.datetime(2024, 11, 25, 13, 11, 10), 'del_dttm': datetime.datetime(2024, 11, 25, 13, 11, 10), 'dlm': datetime.datetime(2024, 11, 27, 19, 11, 44)},
		{'pos_id': 28447, 'symb': 'USDC', 'prod_id': 'ETH-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 13, 8, 38), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 0, 10, 7), 'age_mins': 3241, 'tot_out_cnt': Decimal('25.000000000000'), 'tot_in_cnt': Decimal('25.787612472101'), 'buy_fees_cnt': Decimal('0.062344139651'), 'sell_fees_cnt_tot': Decimal('0.064630607700'), 'fees_cnt_tot': Decimal('0.126974747351'), 'buy_cnt': Decimal('0.007167022823'), 'sell_cnt_tot': Decimal('0.007085660000'), 'hold_cnt': Decimal('0.000081362823'), 'pocket_cnt': Decimal('0.000081362823'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('25.000000000000'), 'clip_pct': Decimal('5.000000000000'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('3488.198742900000'), 'prc_curr': Decimal('3639.408675000000'), 'prc_high': Decimal('3688.540000000000'), 'prc_low': Decimal('3255.890000000000'), 'prc_chg_pct': Decimal('4.334900000000'), 'prc_chg_pct_high': Decimal('5.518600000000'), 'prc_chg_pct_low': Decimal('-6.659800000000'), 'prc_chg_pct_drop': Decimal('-1.180000000000'), 'prc_sell_avg': Decimal('3639.408675000000'), 'val_curr': Decimal('0.296112563849'), 'val_tot': Decimal('0.296112563849'), 'gain_loss_amt_est': Decimal('1.135122085895'), 'gain_loss_amt_est_high': Decimal('1.379660904616'), 'gain_loss_amt_est_low': Decimal('-1.664962060823'), 'gain_loss_amt': Decimal('1.083725035950'), 'gain_loss_amt_net': Decimal('1.379837599798'), 'gain_loss_pct_est': Decimal('4.540500000000'), 'gain_loss_pct_est_high': Decimal('5.518600000000'), 'gain_loss_pct_est_low': Decimal('-6.659800000000'), 'gain_loss_pct': Decimal('4.334900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28100, 'bo_uuid': 'e37c639e-8877-45f4-8e6b-45bbf2664315', 'buy_curr_symb': 'ETH', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ETH', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ETH', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('42000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('150000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 13, 9, 1), 'check_last_dttm': datetime.datetime(2024, 11, 27, 19, 10), 'add_dttm': datetime.datetime(2024, 11, 25, 13, 9, 1), 'upd_dttm': datetime.datetime(2024, 11, 25, 13, 9, 1), 'del_dttm': datetime.datetime(2024, 11, 25, 13, 9, 1), 'dlm': datetime.datetime(2024, 11, 27, 19, 11, 39)},
		{'pos_id': 28446, 'symb': 'USDC', 'prod_id': 'ETH-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 13, 8, 27), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 0, 9, 45), 'age_mins': 3241, 'tot_out_cnt': Decimal('25.000000000000'), 'tot_in_cnt': Decimal('25.803627441832'), 'buy_fees_cnt': Decimal('0.062344139651'), 'sell_fees_cnt_tot': Decimal('0.064670745468'), 'fees_cnt_tot': Decimal('0.127014885119'), 'buy_cnt': Decimal('0.007174578679'), 'sell_cnt_tot': Decimal('0.007091160000'), 'hold_cnt': Decimal('0.000083418679'), 'pocket_cnt': Decimal('0.000083418679'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('25.000000000000'), 'clip_pct': Decimal('5.000000000000'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('3484.525171150000'), 'prc_curr': Decimal('3638.844341660000'), 'prc_high': Decimal('3688.540000000000'), 'prc_low': Decimal('3255.890000000000'), 'prc_chg_pct': Decimal('4.428700000000'), 'prc_chg_pct_high': Decimal('5.629900000000'), 'prc_chg_pct_low': Decimal('-6.561400000000'), 'prc_chg_pct_drop': Decimal('-1.200000000000'), 'prc_sell_avg': Decimal('3638.844341660000'), 'val_curr': Decimal('0.303547588068'), 'val_tot': Decimal('0.303547588068'), 'gain_loss_amt_est': Decimal('1.162675119268'), 'gain_loss_amt_est_high': Decimal('1.407471743795'), 'gain_loss_amt_est_low': Decimal('-1.640361024831'), 'gain_loss_amt': Decimal('1.107175029900'), 'gain_loss_amt_net': Decimal('1.410722617968'), 'gain_loss_pct_est': Decimal('4.650700000000'), 'gain_loss_pct_est_high': Decimal('5.629900000000'), 'gain_loss_pct_est_low': Decimal('-6.561400000000'), 'gain_loss_pct': Decimal('4.428700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28099, 'bo_uuid': '8ff20dbb-de51-429d-96d6-2187192d0ad9', 'buy_curr_symb': 'ETH', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ETH', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ETH', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('42000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('150000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 13, 8, 53), 'check_last_dttm': datetime.datetime(2024, 11, 27, 19, 9, 38), 'add_dttm': datetime.datetime(2024, 11, 25, 13, 8, 53), 'upd_dttm': datetime.datetime(2024, 11, 25, 13, 8, 53), 'del_dttm': datetime.datetime(2024, 11, 25, 13, 8, 53), 'dlm': datetime.datetime(2024, 11, 27, 19, 10, 20)},
		{'pos_id': 28445, 'symb': 'USDC', 'prod_id': 'ETH-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 13, 8, 15), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 0, 9, 32), 'age_mins': 3241, 'tot_out_cnt': Decimal('50.000000000000'), 'tot_in_cnt': Decimal('51.621902561093'), 'buy_fees_cnt': Decimal('0.124688279302'), 'sell_fees_cnt_tot': Decimal('0.129378201908'), 'fees_cnt_tot': Decimal('0.254066481210'), 'buy_cnt': Decimal('0.014348620933'), 'sell_cnt_tot': Decimal('0.014181930000'), 'hold_cnt': Decimal('0.000166690933'), 'pocket_cnt': Decimal('0.000166690933'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('25.000000000000'), 'clip_pct': Decimal('5.000000000000'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('3484.655440600000'), 'prc_curr': Decimal('3639.977250000000'), 'prc_high': Decimal('3688.540000000000'), 'prc_low': Decimal('3255.890000000000'), 'prc_chg_pct': Decimal('4.457300000000'), 'prc_chg_pct_high': Decimal('5.625900000000'), 'prc_chg_pct_low': Decimal('-6.564900000000'), 'prc_chg_pct_drop': Decimal('-1.170000000000'), 'prc_sell_avg': Decimal('3639.977250000000'), 'val_curr': Decimal('0.606751203901'), 'val_tot': Decimal('0.606751203901'), 'gain_loss_amt_est': Decimal('2.323394121859'), 'gain_loss_amt_est_high': Decimal('2.812969068093'), 'gain_loss_amt_est_low': Decimal('-3.282468590455'), 'gain_loss_amt': Decimal('2.228653764994'), 'gain_loss_amt_net': Decimal('2.835404968896'), 'gain_loss_pct_est': Decimal('4.646800000000'), 'gain_loss_pct_est_high': Decimal('5.625900000000'), 'gain_loss_pct_est_low': Decimal('-6.564900000000'), 'gain_loss_pct': Decimal('4.457300000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28098, 'bo_uuid': '4fa815ec-d052-4add-a3f0-dd6f927f8041', 'buy_curr_symb': 'ETH', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ETH', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ETH', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('42000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('150000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 13, 8, 46), 'check_last_dttm': datetime.datetime(2024, 11, 27, 19, 9, 25), 'add_dttm': datetime.datetime(2024, 11, 25, 13, 8, 46), 'upd_dttm': datetime.datetime(2024, 11, 25, 13, 8, 46), 'del_dttm': datetime.datetime(2024, 11, 25, 13, 8, 46), 'dlm': datetime.datetime(2024, 11, 27, 19, 10, 14)},
		{'pos_id': 28095, 'symb': 'USDC', 'prod_id': 'ETH-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 7, 17, 59), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 0, 9, 18), 'age_mins': 3591, 'tot_out_cnt': Decimal('25.000000000000'), 'tot_in_cnt': Decimal('25.736787431355'), 'buy_fees_cnt': Decimal('0.062344139651'), 'sell_fees_cnt_tot': Decimal('0.064503226645'), 'fees_cnt_tot': Decimal('0.126847366296'), 'buy_cnt': Decimal('0.007149782780'), 'sell_cnt_tot': Decimal('0.007073110000'), 'hold_cnt': Decimal('0.000076672780'), 'pocket_cnt': Decimal('0.000076672780'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('25.000000000000'), 'clip_pct': Decimal('5.000000000000'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('3496.609725000000'), 'prc_curr': Decimal('3638.680500000000'), 'prc_high': Decimal('3688.540000000000'), 'prc_low': Decimal('3255.890000000000'), 'prc_chg_pct': Decimal('4.063100000000'), 'prc_chg_pct_high': Decimal('5.264800000000'), 'prc_chg_pct_low': Decimal('-6.884400000000'), 'prc_chg_pct_drop': Decimal('-1.200000000000'), 'prc_sell_avg': Decimal('3638.680500000000'), 'val_curr': Decimal('0.278987749467'), 'val_tot': Decimal('0.278987749467'), 'gain_loss_amt_est': Decimal('1.072254889892'), 'gain_loss_amt_est_high': Decimal('1.316205478346'), 'gain_loss_amt_est_low': Decimal('-1.721093744426'), 'gain_loss_amt': Decimal('1.015775180822'), 'gain_loss_amt_net': Decimal('1.294762930289'), 'gain_loss_pct_est': Decimal('4.289000000000'), 'gain_loss_pct_est_high': Decimal('5.264800000000'), 'gain_loss_pct_est_low': Decimal('-6.884400000000'), 'gain_loss_pct': Decimal('4.063100000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 27748, 'bo_uuid': '79d2cfcc-8720-42a2-a40a-d9b2e77dffc1', 'buy_curr_symb': 'ETH', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ETH', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ETH', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('42000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('150000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 7, 19, 37), 'check_last_dttm': datetime.datetime(2024, 11, 27, 19, 9, 11), 'add_dttm': datetime.datetime(2024, 11, 25, 7, 19, 37), 'upd_dttm': datetime.datetime(2024, 11, 25, 7, 19, 37), 'del_dttm': datetime.datetime(2024, 11, 25, 7, 19, 37), 'dlm': datetime.datetime(2024, 11, 27, 19, 10, 8)},
		{'pos_id': 28094, 'symb': 'USDC', 'prod_id': 'ETH-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 7, 17, 47), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 0, 9, 4), 'age_mins': 3592, 'tot_out_cnt': Decimal('25.000000000000'), 'tot_in_cnt': Decimal('25.712219439270'), 'buy_fees_cnt': Decimal('0.062344139651'), 'sell_fees_cnt_tot': Decimal('0.064441652730'), 'fees_cnt_tot': Decimal('0.126785792381'), 'buy_cnt': Decimal('0.007146463504'), 'sell_cnt_tot': Decimal('0.007070700000'), 'hold_cnt': Decimal('0.000075763504'), 'pocket_cnt': Decimal('0.000075763504'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('25.000000000000'), 'clip_pct': Decimal('5.000000000000'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('3498.233775000000'), 'prc_curr': Decimal('3636.446100000000'), 'prc_high': Decimal('3688.540000000000'), 'prc_low': Decimal('3255.890000000000'), 'prc_chg_pct': Decimal('3.950900000000'), 'prc_chg_pct_high': Decimal('5.216000000000'), 'prc_chg_pct_low': Decimal('-6.927600000000'), 'prc_chg_pct_drop': Decimal('-1.270000000000'), 'prc_sell_avg': Decimal('3636.446100000000'), 'val_curr': Decimal('0.275509898643'), 'val_tot': Decimal('0.275509898643'), 'gain_loss_amt_est': Decimal('1.060150884416'), 'gain_loss_amt_est_high': Decimal('1.303988219173'), 'gain_loss_amt_est_low': Decimal('-1.731900941961'), 'gain_loss_amt': Decimal('0.987729337913'), 'gain_loss_amt_net': Decimal('1.263239236556'), 'gain_loss_pct_est': Decimal('4.240600000000'), 'gain_loss_pct_est_high': Decimal('5.216000000000'), 'gain_loss_pct_est_low': Decimal('-6.927600000000'), 'gain_loss_pct': Decimal('3.950900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 27747, 'bo_uuid': '79343ebf-1698-4604-8c10-39f031bbcec4', 'buy_curr_symb': 'ETH', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ETH', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ETH', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('42000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('150000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 7, 18, 4), 'check_last_dttm': datetime.datetime(2024, 11, 27, 19, 8, 57), 'add_dttm': datetime.datetime(2024, 11, 25, 7, 18, 4), 'upd_dttm': datetime.datetime(2024, 11, 25, 7, 18, 4), 'del_dttm': datetime.datetime(2024, 11, 25, 7, 18, 4), 'dlm': datetime.datetime(2024, 11, 27, 19, 10, 2)},
		{'pos_id': 30258, 'symb': 'USDC', 'prod_id': 'LCX-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 14, 55, 48), 'pos_end_dttm': datetime.datetime(2024, 11, 28, 0, 2, 47), 'age_mins': 247, 'tot_out_cnt': Decimal('15.000000000000'), 'tot_in_cnt': Decimal('15.965506200000'), 'buy_fees_cnt': Decimal('0.037406483791'), 'sell_fees_cnt_tot': Decimal('0.040013800000'), 'fees_cnt_tot': Decimal('0.077420283791'), 'buy_cnt': Decimal('47.364968395725'), 'sell_cnt_tot': Decimal('47.200000000000'), 'hold_cnt': Decimal('0.164968395725'), 'pocket_cnt': Decimal('0.164968395725'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.316689750000'), 'prc_curr': Decimal('0.338252250000'), 'prc_high': Decimal('0.390000000000'), 'prc_low': Decimal('0.302500000000'), 'prc_chg_pct': Decimal('6.808700000000'), 'prc_chg_pct_high': Decimal('13.044400000000'), 'prc_chg_pct_low': Decimal('-4.480600000000'), 'prc_chg_pct_drop': Decimal('-6.240000000000'), 'prc_sell_avg': Decimal('0.338252250000'), 'val_curr': Decimal('0.055800931033'), 'val_tot': Decimal('0.055800931033'), 'gain_loss_amt_est': Decimal('1.104089254547'), 'gain_loss_amt_est_high': Decimal('1.956658685670'), 'gain_loss_amt_est_low': Decimal('-0.672097060293'), 'gain_loss_amt': Decimal('1.021307131033'), 'gain_loss_amt_net': Decimal('1.077108062066'), 'gain_loss_pct_est': Decimal('7.360600000000'), 'gain_loss_pct_est_high': Decimal('13.044400000000'), 'gain_loss_pct_est_low': Decimal('-4.480600000000'), 'gain_loss_pct': Decimal('6.808700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29912, 'bo_uuid': 'd7273469-a8c0-4609-944d-7d1c4044f0d7', 'buy_curr_symb': 'LCX', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'LCX', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'LCX', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('117651200.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 14, 56, 12), 'check_last_dttm': datetime.datetime(2024, 11, 27, 19, 2, 40), 'add_dttm': datetime.datetime(2024, 11, 27, 14, 56, 12), 'upd_dttm': datetime.datetime(2024, 11, 27, 14, 56, 12), 'del_dttm': datetime.datetime(2024, 11, 27, 14, 56, 12), 'dlm': datetime.datetime(2024, 11, 27, 19, 3, 32)},
		{'pos_id': 30393, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 17, 31, 22), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 23, 19, 20), 'age_mins': 48, 'tot_out_cnt': Decimal('10.000000000000'), 'tot_in_cnt': Decimal('9.402425025000'), 'buy_fees_cnt': Decimal('0.024937655860'), 'sell_fees_cnt_tot': Decimal('0.023564975000'), 'fees_cnt_tot': Decimal('0.048502630860'), 'buy_cnt': Decimal('623.246631936248'), 'sell_cnt_tot': Decimal('623.000000000000'), 'hold_cnt': Decimal('0.246631936248'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.246631936248'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.016045010000'), 'prc_curr': Decimal('0.015092170000'), 'prc_high': Decimal('0.016514000000'), 'prc_low': Decimal('0.015092170000'), 'prc_chg_pct': Decimal('-5.938500000000'), 'prc_chg_pct_high': Decimal('2.119000000000'), 'prc_chg_pct_low': Decimal('-5.938500000000'), 'prc_chg_pct_drop': Decimal('-8.060000000000'), 'prc_sell_avg': Decimal('0.015092170000'), 'val_curr': Decimal('0.003722211109'), 'val_tot': Decimal('0.003722211109'), 'gain_loss_amt_est': Decimal('-0.581496898179'), 'gain_loss_amt_est_high': Decimal('0.211896064275'), 'gain_loss_amt_est_low': Decimal('-0.581496898179'), 'gain_loss_amt': Decimal('-0.593852763891'), 'gain_loss_amt_net': Decimal('-0.590130552781'), 'gain_loss_pct_est': Decimal('-5.815000000000'), 'gain_loss_pct_est_high': Decimal('2.119000000000'), 'gain_loss_pct_est_low': Decimal('-5.815000000000'), 'gain_loss_pct': Decimal('-5.938500000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'trail_stop', 'sell_strat_freq': '', 'bo_id': 30047, 'bo_uuid': '273f424b-8f70-4630-91be-3d1ea2c2cfef', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 17, 32, 8), 'check_last_dttm': datetime.datetime(2024, 11, 27, 18, 19, 13), 'add_dttm': datetime.datetime(2024, 11, 27, 17, 32, 8), 'upd_dttm': datetime.datetime(2024, 11, 27, 17, 32, 8), 'del_dttm': datetime.datetime(2024, 11, 27, 17, 32, 8), 'dlm': datetime.datetime(2024, 11, 27, 18, 19, 51)},
		{'pos_id': 29844, 'symb': 'USDC', 'prod_id': 'LQTY-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 4, 45, 28), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 23, 18, 59), 'age_mins': 814, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('23.264699482500'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.058307517500'), 'fees_cnt_tot': Decimal('0.108182829221'), 'buy_cnt': Decimal('14.233924144331'), 'sell_cnt_tot': Decimal('14.180000000000'), 'hold_cnt': Decimal('0.053924144331'), 'pocket_cnt': Decimal('0.053924144331'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.405093900000'), 'prc_curr': Decimal('1.640669920000'), 'prc_high': Decimal('1.777700000000'), 'prc_low': Decimal('1.401589926010'), 'prc_chg_pct': Decimal('16.765900000000'), 'prc_chg_pct_high': Decimal('21.315700000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-4.550000000000'), 'prc_sell_avg': Decimal('1.640669920000'), 'val_curr': Decimal('0.088471721566'), 'val_tot': Decimal('0.088471721566'), 'gain_loss_amt_est': Decimal('3.387760761550'), 'gain_loss_amt_est_high': Decimal('4.263147096427'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('3.353171204066'), 'gain_loss_amt_net': Decimal('3.441642925631'), 'gain_loss_pct_est': Decimal('16.938800000000'), 'gain_loss_pct_est_high': Decimal('21.315700000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('16.765900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29498, 'bo_uuid': 'b602482d-3cdf-4629-a57e-088606033259', 'buy_curr_symb': 'LQTY', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'LQTY', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'LQTY', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('4687700.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 4, 45, 52), 'check_last_dttm': datetime.datetime(2024, 11, 27, 18, 18, 52), 'add_dttm': datetime.datetime(2024, 11, 27, 4, 45, 52), 'upd_dttm': datetime.datetime(2024, 11, 27, 4, 45, 52), 'del_dttm': datetime.datetime(2024, 11, 27, 4, 45, 52), 'dlm': datetime.datetime(2024, 11, 27, 18, 19, 41)},
		{'pos_id': 30287, 'symb': 'USDC', 'prod_id': 'WELL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 15, 25, 38), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 23, 12, 23), 'age_mins': 167, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('2.756229157500'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.006907842500'), 'fees_cnt_tot': Decimal('0.014389139258'), 'buy_cnt': Decimal('37.309013991471'), 'sell_cnt_tot': Decimal('37.000000000000'), 'hold_cnt': Decimal('0.309013991471'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.309013991471'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.080409520000'), 'prc_curr': Decimal('0.074492680000'), 'prc_high': Decimal('0.080209000000'), 'prc_low': Decimal('0.074492680000'), 'prc_chg_pct': Decimal('-7.358400000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-7.358400000000'), 'prc_chg_pct_drop': Decimal('-7.360000000000'), 'prc_sell_avg': Decimal('0.074492680000'), 'val_curr': Decimal('0.023019280382'), 'val_tot': Decimal('0.023019280382'), 'gain_loss_amt_est': Decimal('-0.215516358775'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-0.215516358775'), 'gain_loss_amt': Decimal('-0.220751562118'), 'gain_loss_amt_net': Decimal('-0.197732281736'), 'gain_loss_pct_est': Decimal('-7.183900000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-7.183900000000'), 'gain_loss_pct': Decimal('-7.358400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1d', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 29941, 'bo_uuid': '2a7bbb13-a979-4775-8550-f524a8f3b96b', 'buy_curr_symb': 'WELL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'WELL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'WELL', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('289584259.672040800000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 15, 28, 5), 'check_last_dttm': datetime.datetime(2024, 11, 27, 18, 12, 16), 'add_dttm': datetime.datetime(2024, 11, 27, 15, 28, 5), 'upd_dttm': datetime.datetime(2024, 11, 27, 15, 28, 5), 'del_dttm': datetime.datetime(2024, 11, 27, 15, 28, 5), 'dlm': datetime.datetime(2024, 11, 27, 18, 13, 29)},
		{'pos_id': 30269, 'symb': 'USDC', 'prod_id': 'LQTY-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 15, 4, 47), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 23, 8, 29), 'age_mins': 184, 'tot_out_cnt': Decimal('2.000000000000'), 'tot_in_cnt': Decimal('2.111417227500'), 'buy_fees_cnt': Decimal('0.004987531172'), 'sell_fees_cnt_tot': Decimal('0.005291772500'), 'fees_cnt_tot': Decimal('0.010279303672'), 'buy_cnt': Decimal('1.275066131170'), 'sell_cnt_tot': Decimal('1.270000000000'), 'hold_cnt': Decimal('0.005066131170'), 'pocket_cnt': Decimal('0.005066131170'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.568546090000'), 'prc_curr': Decimal('1.662533250000'), 'prc_high': Decimal('1.777700000000'), 'prc_low': Decimal('1.552100000000'), 'prc_chg_pct': Decimal('5.992000000000'), 'prc_chg_pct_high': Decimal('8.673900000000'), 'prc_chg_pct_low': Decimal('-1.048500000000'), 'prc_chg_pct_drop': Decimal('-2.680000000000'), 'prc_sell_avg': Decimal('1.662533250000'), 'val_curr': Decimal('0.008422611519'), 'val_tot': Decimal('0.008422611519'), 'gain_loss_amt_est': Decimal('0.135990782936'), 'gain_loss_amt_est_high': Decimal('0.173477727192'), 'gain_loss_amt_est_low': Decimal('-0.020969857811'), 'gain_loss_amt': Decimal('0.119839839019'), 'gain_loss_amt_net': Decimal('0.128262450538'), 'gain_loss_pct_est': Decimal('6.799500000000'), 'gain_loss_pct_est_high': Decimal('8.673900000000'), 'gain_loss_pct_est_low': Decimal('-1.048500000000'), 'gain_loss_pct': Decimal('5.992000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29923, 'bo_uuid': '4b84dbc4-61ad-4fb9-a1af-a0e5208b96cc', 'buy_curr_symb': 'LQTY', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'LQTY', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'LQTY', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('4687700.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 15, 5, 21), 'check_last_dttm': datetime.datetime(2024, 11, 27, 18, 8, 22), 'add_dttm': datetime.datetime(2024, 11, 27, 15, 5, 21), 'upd_dttm': datetime.datetime(2024, 11, 27, 15, 5, 21), 'del_dttm': datetime.datetime(2024, 11, 27, 15, 5, 21), 'dlm': datetime.datetime(2024, 11, 27, 18, 9, 30)},
		{'pos_id': 30264, 'symb': 'USDC', 'prod_id': 'LQTY-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 14, 57, 11), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 23, 8, 16), 'age_mins': 191, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.207002400000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.008037600000'), 'fees_cnt_tot': Decimal('0.015518896758'), 'buy_cnt': Decimal('1.927051776188'), 'sell_cnt_tot': Decimal('1.920000000000'), 'hold_cnt': Decimal('0.007051776188'), 'pocket_cnt': Decimal('0.007051776188'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.556782250000'), 'prc_curr': Decimal('1.670313750000'), 'prc_high': Decimal('1.777700000000'), 'prc_low': Decimal('1.545200000000'), 'prc_chg_pct': Decimal('7.292700000000'), 'prc_chg_pct_high': Decimal('9.495100000000'), 'prc_chg_pct_low': Decimal('-0.744000000000'), 'prc_chg_pct_drop': Decimal('-2.200000000000'), 'prc_sell_avg': Decimal('1.670313750000'), 'val_curr': Decimal('0.011778678729'), 'val_tot': Decimal('0.011778678729'), 'gain_loss_amt_est': Decimal('0.228197135470'), 'gain_loss_amt_est_high': Decimal('0.284852457690'), 'gain_loss_amt_est_low': Decimal('-0.022319595434'), 'gain_loss_amt': Decimal('0.218781078729'), 'gain_loss_amt_net': Decimal('0.230559757457'), 'gain_loss_pct_est': Decimal('7.606600000000'), 'gain_loss_pct_est_high': Decimal('9.495100000000'), 'gain_loss_pct_est_low': Decimal('-0.744000000000'), 'gain_loss_pct': Decimal('7.292700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29918, 'bo_uuid': '57e42ef0-4b26-4cf1-85b9-de9d2c798614', 'buy_curr_symb': 'LQTY', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'LQTY', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'LQTY', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('4687700.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 14, 58, 11), 'check_last_dttm': datetime.datetime(2024, 11, 27, 18, 8, 9), 'add_dttm': datetime.datetime(2024, 11, 27, 14, 58, 11), 'upd_dttm': datetime.datetime(2024, 11, 27, 14, 58, 11), 'del_dttm': datetime.datetime(2024, 11, 27, 14, 58, 11), 'dlm': datetime.datetime(2024, 11, 27, 18, 9, 23)},
		{'pos_id': 30173, 'symb': 'USDC', 'prod_id': 'ZEN-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 12, 7, 41), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 22, 22, 50), 'age_mins': 315, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.163353000000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015447000000'), 'fees_cnt_tot': Decimal('0.030409593516'), 'buy_cnt': Decimal('0.400685372329'), 'sell_cnt_tot': Decimal('0.400000000000'), 'hold_cnt': Decimal('0.000685372329'), 'pocket_cnt': Decimal('0.000685372329'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('14.974342500000'), 'prc_curr': Decimal('15.408382500000'), 'prc_high': Decimal('15.820000000000'), 'prc_low': Decimal('14.901000000000'), 'prc_chg_pct': Decimal('2.898600000000'), 'prc_chg_pct_high': Decimal('4.325100000000'), 'prc_chg_pct_low': Decimal('-0.489800000000'), 'prc_chg_pct_drop': Decimal('-1.430000000000'), 'prc_sell_avg': Decimal('15.408382500000'), 'val_curr': Decimal('0.010560479000'), 'val_tot': Decimal('0.010560479000'), 'gain_loss_amt_est': Decimal('0.184578721898'), 'gain_loss_amt_est_high': Decimal('0.259506886524'), 'gain_loss_amt_est_low': Decimal('-0.029387266926'), 'gain_loss_amt': Decimal('0.173913479000'), 'gain_loss_amt_net': Decimal('0.184473958000'), 'gain_loss_pct_est': Decimal('3.076300000000'), 'gain_loss_pct_est_high': Decimal('4.325100000000'), 'gain_loss_pct_est_low': Decimal('-0.489800000000'), 'gain_loss_pct': Decimal('2.898600000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29827, 'bo_uuid': 'f424966e-7e1c-47ef-abc6-a1cacbf235df', 'buy_curr_symb': 'ZEN', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ZEN', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ZEN', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('1005700.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 12, 8, 49), 'check_last_dttm': datetime.datetime(2024, 11, 27, 17, 22, 43), 'add_dttm': datetime.datetime(2024, 11, 27, 12, 8, 49), 'upd_dttm': datetime.datetime(2024, 11, 27, 12, 8, 49), 'del_dttm': datetime.datetime(2024, 11, 27, 12, 8, 49), 'dlm': datetime.datetime(2024, 11, 27, 17, 23, 26)},
		{'pos_id': 30170, 'symb': 'USDC', 'prod_id': 'ZEN-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 12, 2, 7), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 22, 22, 37), 'age_mins': 321, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.158565000000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015435000000'), 'fees_cnt_tot': Decimal('0.030397593516'), 'buy_cnt': Decimal('0.400604913419'), 'sell_cnt_tot': Decimal('0.400000000000'), 'hold_cnt': Decimal('0.000604913419'), 'pocket_cnt': Decimal('0.000604913419'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('14.977350000000'), 'prc_curr': Decimal('15.396412500000'), 'prc_high': Decimal('15.820000000000'), 'prc_low': Decimal('14.901000000000'), 'prc_chg_pct': Decimal('2.798000000000'), 'prc_chg_pct_high': Decimal('4.304200000000'), 'prc_chg_pct_low': Decimal('-0.509800000000'), 'prc_chg_pct_drop': Decimal('-1.510000000000'), 'prc_sell_avg': Decimal('15.396412500000'), 'val_curr': Decimal('0.009313496526'), 'val_tot': Decimal('0.009313496526'), 'gain_loss_amt_est': Decimal('0.183336838622'), 'gain_loss_amt_est_high': Decimal('0.258249957432'), 'gain_loss_amt_est_low': Decimal('-0.030586185143'), 'gain_loss_amt': Decimal('0.167878496526'), 'gain_loss_amt_net': Decimal('0.177191993051'), 'gain_loss_pct_est': Decimal('3.055600000000'), 'gain_loss_pct_est_high': Decimal('4.304200000000'), 'gain_loss_pct_est_low': Decimal('-0.509800000000'), 'gain_loss_pct': Decimal('2.798000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29824, 'bo_uuid': '26d4c479-0a82-45ea-825e-bbc4f8133d54', 'buy_curr_symb': 'ZEN', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ZEN', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ZEN', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('1005700.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 12, 3, 7), 'check_last_dttm': datetime.datetime(2024, 11, 27, 17, 22, 30), 'add_dttm': datetime.datetime(2024, 11, 27, 12, 3, 7), 'upd_dttm': datetime.datetime(2024, 11, 27, 12, 3, 7), 'del_dttm': datetime.datetime(2024, 11, 27, 12, 3, 7), 'dlm': datetime.datetime(2024, 11, 27, 17, 23, 20)},
		{'pos_id': 29051, 'symb': 'USDC', 'prod_id': 'DRIFT-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 26, 6, 13, 13), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 21, 55, 35), 'age_mins': 2083, 'tot_out_cnt': Decimal('2.000000000000'), 'tot_in_cnt': Decimal('2.063628000000'), 'buy_fees_cnt': Decimal('0.004987531172'), 'sell_fees_cnt_tot': Decimal('0.005172000000'), 'fees_cnt_tot': Decimal('0.010159531172'), 'buy_cnt': Decimal('1.603707772370'), 'sell_cnt_tot': Decimal('1.600000000000'), 'hold_cnt': Decimal('0.003707772370'), 'pocket_cnt': Decimal('0.003707772370'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.247110000000'), 'prc_curr': Decimal('1.289767500000'), 'prc_high': Decimal('1.311000000000'), 'prc_low': Decimal('1.213000000000'), 'prc_chg_pct': Decimal('3.420500000000'), 'prc_chg_pct_high': Decimal('4.882500000000'), 'prc_chg_pct_low': Decimal('-2.735100000000'), 'prc_chg_pct_drop': Decimal('-1.460000000000'), 'prc_sell_avg': Decimal('1.289767500000'), 'val_curr': Decimal('0.004782164300'), 'val_tot': Decimal('0.004782164300'), 'gain_loss_amt_est': Decimal('0.073594149674'), 'gain_loss_amt_est_high': Decimal('0.097649766260'), 'gain_loss_amt_est_low': Decimal('-0.054702472115'), 'gain_loss_amt': Decimal('0.068410164300'), 'gain_loss_amt_net': Decimal('0.073192328600'), 'gain_loss_pct_est': Decimal('3.679700000000'), 'gain_loss_pct_est_high': Decimal('4.882500000000'), 'gain_loss_pct_est_low': Decimal('-2.735100000000'), 'gain_loss_pct': Decimal('3.420500000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28704, 'bo_uuid': '2e697074-c85b-4805-aa58-b4a2658e0c1f', 'buy_curr_symb': 'DRIFT', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DRIFT', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DRIFT', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('6666666.666666667000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 26, 6, 14, 3), 'check_last_dttm': datetime.datetime(2024, 11, 27, 16, 55, 28), 'add_dttm': datetime.datetime(2024, 11, 26, 6, 14, 3), 'upd_dttm': datetime.datetime(2024, 11, 26, 6, 14, 3), 'del_dttm': datetime.datetime(2024, 11, 26, 6, 14, 3), 'dlm': datetime.datetime(2024, 11, 27, 16, 55, 43)},
		{'pos_id': 30244, 'symb': 'USDC', 'prod_id': 'SWELL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 14, 11, 42), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 21, 51, 33), 'age_mins': 160, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('2.805768000000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007032000000'), 'fees_cnt_tot': Decimal('0.014513296758'), 'buy_cnt': Decimal('64.759114980348'), 'sell_cnt_tot': Decimal('64.000000000000'), 'hold_cnt': Decimal('0.759114980348'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.759114980348'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.046325530000'), 'prc_curr': Decimal('0.043840130000'), 'prc_high': Decimal('0.048000000000'), 'prc_low': Decimal('0.043840130000'), 'prc_chg_pct': Decimal('-5.365100000000'), 'prc_chg_pct_high': Decimal('2.707900000000'), 'prc_chg_pct_low': Decimal('-5.365100000000'), 'prc_chg_pct_drop': Decimal('-8.070000000000'), 'prc_sell_avg': Decimal('0.043840130000'), 'val_curr': Decimal('0.033279699423'), 'val_tot': Decimal('0.033279699423'), 'gain_loss_amt_est': Decimal('-0.152541714314'), 'gain_loss_amt_est_high': Decimal('0.081238690765'), 'gain_loss_amt_est_low': Decimal('-0.152541714314'), 'gain_loss_amt': Decimal('-0.160952300577'), 'gain_loss_amt_net': Decimal('-0.127672601153'), 'gain_loss_pct_est': Decimal('-5.084700000000'), 'gain_loss_pct_est_high': Decimal('2.708000000000'), 'gain_loss_pct_est_low': Decimal('-5.084700000000'), 'gain_loss_pct': Decimal('-5.365100000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'trail_stop', 'sell_strat_freq': '', 'bo_id': 29898, 'bo_uuid': '2e33467f-2f76-45d0-934e-e6a78adf2bba', 'buy_curr_symb': 'SWELL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SWELL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SWELL', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('333333333.333333300000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 14, 12, 16), 'check_last_dttm': datetime.datetime(2024, 11, 27, 16, 51, 27), 'add_dttm': datetime.datetime(2024, 11, 27, 14, 12, 16), 'upd_dttm': datetime.datetime(2024, 11, 27, 14, 12, 16), 'del_dttm': datetime.datetime(2024, 11, 27, 14, 12, 16), 'dlm': datetime.datetime(2024, 11, 27, 16, 53)},
		{'pos_id': 30201, 'symb': 'USDC', 'prod_id': 'SWELL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 13, 0, 54), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 21, 51, 17), 'age_mins': 231, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('5.648942250000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.014157750000'), 'fees_cnt_tot': Decimal('0.029120343516'), 'buy_cnt': Decimal('129.015680243238'), 'sell_cnt_tot': Decimal('129.000000000000'), 'hold_cnt': Decimal('0.015680243238'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.015680243238'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.046505970000'), 'prc_curr': Decimal('0.043790250000'), 'prc_high': Decimal('0.048000000000'), 'prc_low': Decimal('0.043790250000'), 'prc_chg_pct': Decimal('-5.839500000000'), 'prc_chg_pct_high': Decimal('2.309400000000'), 'prc_chg_pct_low': Decimal('-5.839500000000'), 'prc_chg_pct_drop': Decimal('-8.150000000000'), 'prc_sell_avg': Decimal('0.043790250000'), 'val_curr': Decimal('0.000686641771'), 'val_tot': Decimal('0.000686641771'), 'gain_loss_amt_est': Decimal('-0.327180539705'), 'gain_loss_amt_est_high': Decimal('0.138566065973'), 'gain_loss_amt_est_low': Decimal('-0.346532891741'), 'gain_loss_amt': Decimal('-0.350371108229'), 'gain_loss_amt_net': Decimal('-0.349684466457'), 'gain_loss_pct_est': Decimal('-5.453000000000'), 'gain_loss_pct_est_high': Decimal('2.309400000000'), 'gain_loss_pct_est_low': Decimal('-5.775500000000'), 'gain_loss_pct': Decimal('-5.839500000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'trail_stop', 'sell_strat_freq': '', 'bo_id': 29855, 'bo_uuid': '0a1c6e0e-e12b-4357-bf2d-55f222a3f3bb', 'buy_curr_symb': 'SWELL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SWELL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SWELL', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('333333333.333333300000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 13, 1, 42), 'check_last_dttm': datetime.datetime(2024, 11, 27, 16, 51, 10), 'add_dttm': datetime.datetime(2024, 11, 27, 13, 1, 42), 'upd_dttm': datetime.datetime(2024, 11, 27, 13, 1, 42), 'del_dttm': datetime.datetime(2024, 11, 27, 13, 1, 42), 'dlm': datetime.datetime(2024, 11, 27, 16, 52, 53)},
		{'pos_id': 30332, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 16, 11, 30), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 21, 50, 28), 'age_mins': 39, 'tot_out_cnt': Decimal('8.000000000000'), 'tot_in_cnt': Decimal('8.373164625000'), 'buy_fees_cnt': Decimal('0.019950124688'), 'sell_fees_cnt_tot': Decimal('0.020985375000'), 'fees_cnt_tot': Decimal('0.040935499688'), 'buy_cnt': Decimal('535.573817135015'), 'sell_cnt_tot': Decimal('535.000000000000'), 'hold_cnt': Decimal('0.573817135015'), 'pocket_cnt': Decimal('0.573817135015'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.014937250000'), 'prc_curr': Decimal('0.015650770000'), 'prc_high': Decimal('0.016340000000'), 'prc_low': Decimal('0.014900000000'), 'prc_chg_pct': Decimal('4.776800000000'), 'prc_chg_pct_high': Decimal('7.871300000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-3.090000000000'), 'prc_sell_avg': Decimal('0.015650770000'), 'val_curr': Decimal('0.008980680002'), 'val_tot': Decimal('0.008980680002'), 'gain_loss_amt_est': Decimal('0.322281544461'), 'gain_loss_amt_est_high': Decimal('0.629700915496'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.382145305002'), 'gain_loss_amt_net': Decimal('0.391125985004'), 'gain_loss_pct_est': Decimal('4.028500000000'), 'gain_loss_pct_est_high': Decimal('7.871300000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('4.776800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29986, 'bo_uuid': 'd90cb1fa-8e95-4868-8a3a-3327cb19195f', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 16, 12, 57), 'check_last_dttm': datetime.datetime(2024, 11, 27, 16, 50, 21), 'add_dttm': datetime.datetime(2024, 11, 27, 16, 12, 57), 'upd_dttm': datetime.datetime(2024, 11, 27, 16, 12, 57), 'del_dttm': datetime.datetime(2024, 11, 27, 16, 12, 57), 'dlm': datetime.datetime(2024, 11, 27, 16, 51, 17)},
		{'pos_id': 30286, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 15, 25, 21), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 21, 50, 7), 'age_mins': 85, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.816765375000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.017084625000'), 'fees_cnt_tot': Decimal('0.032047218516'), 'buy_cnt': Decimal('436.227216216020'), 'sell_cnt_tot': Decimal('435.000000000000'), 'hold_cnt': Decimal('1.227216216020'), 'pocket_cnt': Decimal('1.227216216020'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.013754300000'), 'prc_curr': Decimal('0.015670720000'), 'prc_high': Decimal('0.016340000000'), 'prc_low': Decimal('0.013720000000'), 'prc_chg_pct': Decimal('13.933200000000'), 'prc_chg_pct_high': Decimal('17.148800000000'), 'prc_chg_pct_low': Decimal('0E-12'), 'prc_chg_pct_drop': Decimal('-3.220000000000'), 'prc_sell_avg': Decimal('0.015670720000'), 'val_curr': Decimal('0.019231361701'), 'val_tot': Decimal('0.019231361701'), 'gain_loss_amt_est': Decimal('0.778534712781'), 'gain_loss_amt_est_high': Decimal('1.028929134889'), 'gain_loss_amt_est_low': Decimal('0E-12'), 'gain_loss_amt': Decimal('0.835996736701'), 'gain_loss_amt_net': Decimal('0.855228098401'), 'gain_loss_pct_est': Decimal('12.975600000000'), 'gain_loss_pct_est_high': Decimal('17.148800000000'), 'gain_loss_pct_est_low': Decimal('0E-12'), 'gain_loss_pct': Decimal('13.933300000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29940, 'bo_uuid': 'b6f16099-ae08-4b0c-a526-8da96cdf7ba7', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 15, 28), 'check_last_dttm': datetime.datetime(2024, 11, 27, 16, 50), 'add_dttm': datetime.datetime(2024, 11, 27, 15, 28), 'upd_dttm': datetime.datetime(2024, 11, 27, 15, 28), 'del_dttm': datetime.datetime(2024, 11, 27, 15, 28), 'dlm': datetime.datetime(2024, 11, 27, 16, 51, 8)},
		{'pos_id': 29389, 'symb': 'USDC', 'prod_id': 'RENDER-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 26, 12, 21, 43), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 21, 32, 31), 'age_mins': 1691, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.247677925000'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.013152075000'), 'fees_cnt_tot': Decimal('0.025620902930'), 'buy_cnt': Decimal('0.675268233424'), 'sell_cnt_tot': Decimal('0.670000000000'), 'hold_cnt': Decimal('0.005268233424'), 'pocket_cnt': Decimal('0.005268233424'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('7.404465000000'), 'prc_curr': Decimal('7.832355110000'), 'prc_high': Decimal('7.988000000000'), 'prc_low': Decimal('7.149000000000'), 'prc_chg_pct': Decimal('5.778800000000'), 'prc_chg_pct_high': Decimal('7.489200000000'), 'prc_chg_pct_low': Decimal('-3.450100000000'), 'prc_chg_pct_drop': Decimal('-1.710000000000'), 'prc_sell_avg': Decimal('7.832355110000'), 'val_curr': Decimal('0.041262674979'), 'val_tot': Decimal('0.041262674979'), 'gain_loss_amt_est': Decimal('0.302206168845'), 'gain_loss_amt_est_high': Decimal('0.374459869822'), 'gain_loss_amt_est_low': Decimal('-0.172507399252'), 'gain_loss_amt': Decimal('0.288940599979'), 'gain_loss_amt_net': Decimal('0.330203274958'), 'gain_loss_pct_est': Decimal('6.044100000000'), 'gain_loss_pct_est_high': Decimal('7.489200000000'), 'gain_loss_pct_est_low': Decimal('-3.450100000000'), 'gain_loss_pct': Decimal('5.778800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29043, 'bo_uuid': '265b7926-a5d3-47f6-84e7-362c21651af8', 'buy_curr_symb': 'RENDER', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'RENDER', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'RENDER', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('1982431.303836978700'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 26, 12, 23, 38), 'check_last_dttm': datetime.datetime(2024, 11, 27, 16, 32, 24), 'add_dttm': datetime.datetime(2024, 11, 26, 12, 23, 38), 'upd_dttm': datetime.datetime(2024, 11, 26, 12, 23, 38), 'del_dttm': datetime.datetime(2024, 11, 26, 12, 23, 38), 'dlm': datetime.datetime(2024, 11, 27, 16, 33, 52)},
		{'pos_id': 29385, 'symb': 'USDC', 'prod_id': 'RNDR-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 26, 12, 21, 3), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 21, 31, 36), 'age_mins': 1691, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.241071482500'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.013135517500'), 'fees_cnt_tot': Decimal('0.025604345430'), 'buy_cnt': Decimal('0.675496874392'), 'sell_cnt_tot': Decimal('0.670000000000'), 'hold_cnt': Decimal('0.005496874392'), 'pocket_cnt': Decimal('0.005496874392'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('7.401958750000'), 'prc_curr': Decimal('7.822494750000'), 'prc_high': Decimal('7.987900000000'), 'prc_low': Decimal('7.140100000000'), 'prc_chg_pct': Decimal('5.681400000000'), 'prc_chg_pct_high': Decimal('7.372900000000'), 'prc_chg_pct_low': Decimal('-3.537700000000'), 'prc_chg_pct_drop': Decimal('-1.690000000000'), 'prc_sell_avg': Decimal('7.822494750000'), 'val_curr': Decimal('0.042999271073'), 'val_tot': Decimal('0.042999271073'), 'gain_loss_amt_est': Decimal('0.293261057423'), 'gain_loss_amt_est_high': Decimal('0.368646508605'), 'gain_loss_amt_est_low': Decimal('-0.176884767154'), 'gain_loss_amt': Decimal('0.284070753573'), 'gain_loss_amt_net': Decimal('0.327070024646'), 'gain_loss_pct_est': Decimal('5.865200000000'), 'gain_loss_pct_est_high': Decimal('7.372900000000'), 'gain_loss_pct_est_low': Decimal('-3.537700000000'), 'gain_loss_pct': Decimal('5.681400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29039, 'bo_uuid': '0dfe7c3a-9b7c-495e-b92f-262fc0c7408f', 'buy_curr_symb': 'RNDR', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'RNDR', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'RNDR', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('8235000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 26, 12, 23, 16), 'check_last_dttm': datetime.datetime(2024, 11, 27, 16, 31, 29), 'add_dttm': datetime.datetime(2024, 11, 26, 12, 23, 16), 'upd_dttm': datetime.datetime(2024, 11, 26, 12, 23, 16), 'del_dttm': datetime.datetime(2024, 11, 26, 12, 23, 16), 'dlm': datetime.datetime(2024, 11, 27, 16, 32, 12)},
		{'pos_id': 29159, 'symb': 'USDC', 'prod_id': 'RNDR-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 26, 7, 17, 24), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 21, 31, 22), 'age_mins': 1994, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.334281607500'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015875392500'), 'fees_cnt_tot': Decimal('0.030837986016'), 'buy_cnt': Decimal('0.814145769590'), 'sell_cnt_tot': Decimal('0.810000000000'), 'hold_cnt': Decimal('0.004145769590'), 'pocket_cnt': Decimal('0.004145769590'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('7.369687620000'), 'prc_curr': Decimal('7.820100750000'), 'prc_high': Decimal('7.987900000000'), 'prc_low': Decimal('7.117800000000'), 'prc_chg_pct': Decimal('6.111700000000'), 'prc_chg_pct_high': Decimal('7.843100000000'), 'prc_chg_pct_low': Decimal('-3.417900000000'), 'prc_chg_pct_drop': Decimal('-1.730000000000'), 'prc_sell_avg': Decimal('7.820100750000'), 'val_curr': Decimal('0.032420335880'), 'val_tot': Decimal('0.032420335880'), 'gain_loss_amt_est': Decimal('0.379727665084'), 'gain_loss_amt_est_high': Decimal('0.470586332970'), 'gain_loss_amt_est_low': Decimal('-0.205073241212'), 'gain_loss_amt': Decimal('0.366701943380'), 'gain_loss_amt_net': Decimal('0.399122279260'), 'gain_loss_pct_est': Decimal('6.328800000000'), 'gain_loss_pct_est_high': Decimal('7.843100000000'), 'gain_loss_pct_est_low': Decimal('-3.417900000000'), 'gain_loss_pct': Decimal('6.111700000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28812, 'bo_uuid': 'c7cecf27-d31f-4c45-a014-34c1352b1926', 'buy_curr_symb': 'RNDR', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'RNDR', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'RNDR', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('8235000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 26, 7, 18, 35), 'check_last_dttm': datetime.datetime(2024, 11, 27, 16, 31, 15), 'add_dttm': datetime.datetime(2024, 11, 26, 7, 18, 35), 'upd_dttm': datetime.datetime(2024, 11, 26, 7, 18, 35), 'del_dttm': datetime.datetime(2024, 11, 26, 7, 18, 35), 'dlm': datetime.datetime(2024, 11, 27, 16, 32, 3)},
		{'pos_id': 28740, 'symb': 'USDC', 'prod_id': 'KRL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 20, 25, 31), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 21, 28, 56), 'age_mins': 2644, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('18.257960700000'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.045759300000'), 'fees_cnt_tot': Decimal('0.095634611721'), 'buy_cnt': Decimal('31.526745714727'), 'sell_cnt_tot': Decimal('31.500000000000'), 'hold_cnt': Decimal('0.026745714727'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.026745714727'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.634382000000'), 'prc_curr': Decimal('0.579617800000'), 'prc_high': Decimal('0.634600000000'), 'prc_low': Decimal('0.535800000000'), 'prc_chg_pct': Decimal('-8.632700000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-15.539800000000'), 'prc_chg_pct_drop': Decimal('-8.630000000000'), 'prc_sell_avg': Decimal('0.579617800000'), 'val_curr': Decimal('0.015502292329'), 'val_tot': Decimal('0.015502292329'), 'gain_loss_amt_est': Decimal('-1.686113414315'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-3.107969646049'), 'gain_loss_amt': Decimal('-1.726537007671'), 'gain_loss_amt_net': Decimal('-1.711034715341'), 'gain_loss_pct_est': Decimal('-8.430600000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-15.539800000000'), 'gain_loss_pct': Decimal('-8.632700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 28393, 'bo_uuid': 'e917f41c-0460-4e2a-9f8d-896ba47b5208', 'buy_curr_symb': 'KRL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'KRL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'KRL', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('35449200.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 20, 26, 29), 'check_last_dttm': datetime.datetime(2024, 11, 27, 16, 28, 49), 'add_dttm': datetime.datetime(2024, 11, 25, 20, 26, 29), 'upd_dttm': datetime.datetime(2024, 11, 25, 20, 26, 29), 'del_dttm': datetime.datetime(2024, 11, 25, 20, 26, 29), 'dlm': datetime.datetime(2024, 11, 27, 16, 29, 36)},
		{'pos_id': 29848, 'symb': 'USDC', 'prod_id': 'TRB-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 5, 1, 9), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 21, 24, 59), 'age_mins': 684, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.106395750000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015304250000'), 'fees_cnt_tot': Decimal('0.030266843516'), 'buy_cnt': Decimal('0.085439506160'), 'sell_cnt_tot': Decimal('0.085000000000'), 'hold_cnt': Decimal('0.000439506160'), 'pocket_cnt': Decimal('0.000439506160'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('70.225125000000'), 'prc_curr': Decimal('71.839950000000'), 'prc_high': Decimal('72.800000000000'), 'prc_low': Decimal('69.460000000000'), 'prc_chg_pct': Decimal('2.299500000000'), 'prc_chg_pct_high': Decimal('3.495700000000'), 'prc_chg_pct_low': Decimal('-1.089500000000'), 'prc_chg_pct_drop': Decimal('-1.200000000000'), 'prc_sell_avg': Decimal('71.839950000000'), 'val_curr': Decimal('0.031574100559'), 'val_tot': Decimal('0.031574100559'), 'gain_loss_amt_est': Decimal('0.155062023766'), 'gain_loss_amt_est_high': Decimal('0.209743307709'), 'gain_loss_amt_est_low': Decimal('-0.065371902126'), 'gain_loss_amt': Decimal('0.137969850559'), 'gain_loss_amt_net': Decimal('0.169543951118'), 'gain_loss_pct_est': Decimal('2.584400000000'), 'gain_loss_pct_est_high': Decimal('3.495700000000'), 'gain_loss_pct_est_low': Decimal('-1.089500000000'), 'gain_loss_pct': Decimal('2.299500000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29502, 'bo_uuid': 'c99c58ae-3270-47fa-91dd-3b8b83c9928e', 'buy_curr_symb': 'TRB', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'TRB', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'TRB', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('719900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 5, 2, 11), 'check_last_dttm': datetime.datetime(2024, 11, 27, 16, 24, 52), 'add_dttm': datetime.datetime(2024, 11, 27, 5, 2, 11), 'upd_dttm': datetime.datetime(2024, 11, 27, 5, 2, 11), 'del_dttm': datetime.datetime(2024, 11, 27, 5, 2, 11), 'dlm': datetime.datetime(2024, 11, 27, 16, 25, 57)},
		{'pos_id': 28471, 'symb': 'USDC', 'prod_id': 'SAND-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 14, 8, 39), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 21, 20, 28), 'age_mins': 3012, 'tot_out_cnt': Decimal('9.000000000000'), 'tot_in_cnt': Decimal('7.616614740000'), 'buy_fees_cnt': Decimal('0.022443890274'), 'sell_fees_cnt_tot': Decimal('0.019089260000'), 'fees_cnt_tot': Decimal('0.041533150274'), 'buy_cnt': Decimal('12.289604530768'), 'sell_cnt_tot': Decimal('12.280000000000'), 'hold_cnt': Decimal('0.009604530768'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.009604530768'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.732326250000'), 'prc_curr': Decimal('0.620245500000'), 'prc_high': Decimal('0.735600000000'), 'prc_low': Decimal('0.581000000000'), 'prc_chg_pct': Decimal('-15.304800000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-20.663800000000'), 'prc_chg_pct_drop': Decimal('-15.300000000000'), 'prc_sell_avg': Decimal('0.620245500000'), 'val_curr': Decimal('0.005957166988'), 'val_tot': Decimal('0.005957166988'), 'gain_loss_amt_est': Decimal('-1.370613507299'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-1.859739767624'), 'gain_loss_amt': Decimal('-1.377428093012'), 'gain_loss_amt_net': Decimal('-1.371470926023'), 'gain_loss_pct_est': Decimal('-15.229000000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-20.663800000000'), 'gain_loss_pct': Decimal('-15.304800000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '15min', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 28124, 'bo_uuid': '8951126f-880d-49d9-a150-b2180a9ee118', 'buy_curr_symb': 'SAND', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SAND', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SAND', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('16861000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 14, 9, 27), 'check_last_dttm': datetime.datetime(2024, 11, 27, 16, 20, 21), 'add_dttm': datetime.datetime(2024, 11, 25, 14, 9, 27), 'upd_dttm': datetime.datetime(2024, 11, 25, 14, 9, 27), 'del_dttm': datetime.datetime(2024, 11, 25, 14, 9, 27), 'dlm': datetime.datetime(2024, 11, 27, 16, 21, 54)},
		{'pos_id': 28322, 'symb': 'USDC', 'prod_id': 'SAND-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 10, 49, 33), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 21, 20, 9), 'age_mins': 3211, 'tot_out_cnt': Decimal('12.000000000000'), 'tot_in_cnt': Decimal('9.579766560000'), 'buy_fees_cnt': Decimal('0.029925187032'), 'sell_fees_cnt_tot': Decimal('0.024009440000'), 'fees_cnt_tot': Decimal('0.053934627032'), 'buy_cnt': Decimal('15.470888991815'), 'sell_cnt_tot': Decimal('15.470000000000'), 'hold_cnt': Decimal('0.000888991815'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.000888991815'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.775650320000'), 'prc_curr': Decimal('0.619248000000'), 'prc_high': Decimal('0.781900000000'), 'prc_low': Decimal('0.581000000000'), 'prc_chg_pct': Decimal('-20.164000000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-25.095100000000'), 'prc_chg_pct_drop': Decimal('-20.160000000000'), 'prc_sell_avg': Decimal('0.619248000000'), 'val_curr': Decimal('0.000550506403'), 'val_tot': Decimal('0.000550506403'), 'gain_loss_amt_est': Decimal('-2.395672113881'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-3.011413495755'), 'gain_loss_amt': Decimal('-2.419682933597'), 'gain_loss_amt_net': Decimal('-2.419132427193'), 'gain_loss_pct_est': Decimal('-19.963900000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-25.095100000000'), 'gain_loss_pct': Decimal('-20.164000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 27975, 'bo_uuid': 'bc649305-b165-43a2-b316-7fccf74b978c', 'buy_curr_symb': 'SAND', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SAND', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SAND', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('16861000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 10, 50, 48), 'check_last_dttm': datetime.datetime(2024, 11, 27, 16, 20, 2), 'add_dttm': datetime.datetime(2024, 11, 25, 10, 50, 48), 'upd_dttm': datetime.datetime(2024, 11, 25, 10, 50, 48), 'del_dttm': datetime.datetime(2024, 11, 25, 10, 50, 48), 'dlm': datetime.datetime(2024, 11, 27, 16, 20, 26)},
		{'pos_id': 28317, 'symb': 'USDC', 'prod_id': 'SAND-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 10, 43, 27), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 21, 19, 54), 'age_mins': 3217, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('2.328747540000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.005836460000'), 'fees_cnt_tot': Decimal('0.013317756758'), 'buy_cnt': Decimal('3.766543364685'), 'sell_cnt_tot': Decimal('3.760000000000'), 'hold_cnt': Decimal('0.006543364685'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.006543364685'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.796486250000'), 'prc_curr': Decimal('0.619347750000'), 'prc_high': Decimal('0.801000000000'), 'prc_low': Decimal('0.581000000000'), 'prc_chg_pct': Decimal('-22.240000000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-27.054600000000'), 'prc_chg_pct_drop': Decimal('-22.240000000000'), 'prc_sell_avg': Decimal('0.619347750000'), 'val_curr': Decimal('0.004052618195'), 'val_tot': Decimal('0.004052618195'), 'gain_loss_amt_est': Decimal('-0.661729879204'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-0.811638305118'), 'gain_loss_amt': Decimal('-0.667199841805'), 'gain_loss_amt_net': Decimal('-0.663147223610'), 'gain_loss_pct_est': Decimal('-22.057700000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-27.054600000000'), 'gain_loss_pct': Decimal('-22.240000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1d', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 27970, 'bo_uuid': '248a2aeb-11e5-4878-973b-56a845bff4ac', 'buy_curr_symb': 'SAND', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SAND', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SAND', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('16861000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 10, 44, 2), 'check_last_dttm': datetime.datetime(2024, 11, 27, 16, 19, 47), 'add_dttm': datetime.datetime(2024, 11, 25, 10, 44, 2), 'upd_dttm': datetime.datetime(2024, 11, 25, 10, 44, 2), 'del_dttm': datetime.datetime(2024, 11, 25, 10, 44, 2), 'dlm': datetime.datetime(2024, 11, 27, 16, 20, 20)},
		{'pos_id': 30308, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 15, 37, 30), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 21, 9, 23), 'age_mins': 32, 'tot_out_cnt': Decimal('4.000000000000'), 'tot_in_cnt': Decimal('4.193250600000'), 'buy_fees_cnt': Decimal('0.009975062344'), 'sell_fees_cnt_tot': Decimal('0.010509400000'), 'fees_cnt_tot': Decimal('0.020484462344'), 'buy_cnt': Decimal('281.880956386850'), 'sell_cnt_tot': Decimal('281.000000000000'), 'hold_cnt': Decimal('0.880956386850'), 'pocket_cnt': Decimal('0.880956386850'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.014190390000'), 'prc_curr': Decimal('0.014922600000'), 'prc_high': Decimal('0.015263000000'), 'prc_low': Decimal('0.014025000000'), 'prc_chg_pct': Decimal('5.159900000000'), 'prc_chg_pct_high': Decimal('6.924500000000'), 'prc_chg_pct_low': Decimal('-1.165500000000'), 'prc_chg_pct_drop': Decimal('-1.760000000000'), 'prc_sell_avg': Decimal('0.014922600000'), 'val_curr': Decimal('0.013146159778'), 'val_tot': Decimal('0.013146159778'), 'gain_loss_amt_est': Decimal('0.219757917111'), 'gain_loss_amt_est_high': Decimal('0.276979751258'), 'gain_loss_amt_est_low': Decimal('-0.046619586674'), 'gain_loss_amt': Decimal('0.206396759778'), 'gain_loss_amt_net': Decimal('0.219542919557'), 'gain_loss_pct_est': Decimal('5.493900000000'), 'gain_loss_pct_est_high': Decimal('6.924500000000'), 'gain_loss_pct_est_low': Decimal('-1.165500000000'), 'gain_loss_pct': Decimal('5.159900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29962, 'bo_uuid': 'c9adace4-7252-4daf-9fcb-e231c764d390', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 15, 38, 15), 'check_last_dttm': datetime.datetime(2024, 11, 27, 16, 9, 16), 'add_dttm': datetime.datetime(2024, 11, 27, 15, 38, 15), 'upd_dttm': datetime.datetime(2024, 11, 27, 15, 38, 15), 'del_dttm': datetime.datetime(2024, 11, 27, 15, 38, 15), 'dlm': datetime.datetime(2024, 11, 27, 16, 10, 4)},
		{'pos_id': 30299, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 15, 31, 30), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 21, 9, 10), 'age_mins': 38, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.344945880000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015902120000'), 'fees_cnt_tot': Decimal('0.030864713516'), 'buy_cnt': Decimal('424.591189449758'), 'sell_cnt_tot': Decimal('424.000000000000'), 'hold_cnt': Decimal('0.591189449758'), 'pocket_cnt': Decimal('0.591189449758'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.014131240000'), 'prc_curr': Decimal('0.014964490000'), 'prc_high': Decimal('0.015263000000'), 'prc_low': Decimal('0.014025000000'), 'prc_chg_pct': Decimal('5.896500000000'), 'prc_chg_pct_high': Decimal('7.372000000000'), 'prc_chg_pct_low': Decimal('-0.751800000000'), 'prc_chg_pct_drop': Decimal('-1.480000000000'), 'prc_sell_avg': Decimal('0.014964490000'), 'val_curr': Decimal('0.008846848609'), 'val_tot': Decimal('0.008846848609'), 'gain_loss_amt_est': Decimal('0.356130106063'), 'gain_loss_amt_est_high': Decimal('0.442322117521'), 'gain_loss_amt_est_low': Decimal('-0.045108567967'), 'gain_loss_amt': Decimal('0.353792728609'), 'gain_loss_amt_net': Decimal('0.362639577218'), 'gain_loss_pct_est': Decimal('5.935500000000'), 'gain_loss_pct_est_high': Decimal('7.372000000000'), 'gain_loss_pct_est_low': Decimal('-0.751800000000'), 'gain_loss_pct': Decimal('5.896500000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29953, 'bo_uuid': '35e6c198-d11e-4233-b95a-de9eae3aeef9', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 15, 33, 58), 'check_last_dttm': datetime.datetime(2024, 11, 27, 16, 9, 3), 'add_dttm': datetime.datetime(2024, 11, 27, 15, 33, 58), 'upd_dttm': datetime.datetime(2024, 11, 27, 15, 33, 58), 'del_dttm': datetime.datetime(2024, 11, 27, 15, 33, 58), 'dlm': datetime.datetime(2024, 11, 27, 16, 9, 58)},
		{'pos_id': 28512, 'symb': 'USDC', 'prod_id': 'MUSE-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 15, 19, 39), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 20, 58, 51), 'age_mins': 2919, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('1.907442442500'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.004780557500'), 'fees_cnt_tot': Decimal('0.012261854258'), 'buy_cnt': Decimal('0.139005885509'), 'sell_cnt_tot': Decimal('0.139000000000'), 'hold_cnt': Decimal('0.000005885509'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.000005885509'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('21.581820000000'), 'prc_curr': Decimal('13.722607500000'), 'prc_high': Decimal('22.361000000000'), 'prc_low': Decimal('13.201000000000'), 'prc_chg_pct': Decimal('-36.415900000000'), 'prc_chg_pct_high': Decimal('3.114600000000'), 'prc_chg_pct_low': Decimal('-38.832800000000'), 'prc_chg_pct_drop': Decimal('-39.530000000000'), 'prc_sell_avg': Decimal('13.722607500000'), 'val_curr': Decimal('0.000080764530'), 'val_tot': Decimal('0.000080764530'), 'gain_loss_amt_est': Decimal('-1.085610944770'), 'gain_loss_amt_est_high': Decimal('0.093436976117'), 'gain_loss_amt_est_low': Decimal('-1.164983305396'), 'gain_loss_amt': Decimal('-1.092476792970'), 'gain_loss_amt_net': Decimal('-1.092396028440'), 'gain_loss_pct_est': Decimal('-36.187000000000'), 'gain_loss_pct_est_high': Decimal('3.114600000000'), 'gain_loss_pct_est_low': Decimal('-38.832800000000'), 'gain_loss_pct': Decimal('-36.415900000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1h', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 28165, 'bo_uuid': '6ddec705-39d4-4485-95d2-d3505dafee9c', 'buy_curr_symb': 'MUSE', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'MUSE', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'MUSE', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('1763900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 15, 20, 53), 'check_last_dttm': datetime.datetime(2024, 11, 27, 15, 58, 45), 'add_dttm': datetime.datetime(2024, 11, 25, 15, 20, 53), 'upd_dttm': datetime.datetime(2024, 11, 25, 15, 20, 53), 'del_dttm': datetime.datetime(2024, 11, 25, 15, 20, 53), 'dlm': datetime.datetime(2024, 11, 27, 15, 59, 49)},
		{'pos_id': 28496, 'symb': 'USDC', 'prod_id': 'MUSE-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 14, 40, 6), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 20, 58, 32), 'age_mins': 2958, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('3.747088800000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.009391200000'), 'fees_cnt_tot': Decimal('0.024353793516'), 'buy_cnt': Decimal('0.273539186768'), 'sell_cnt_tot': Decimal('0.273000000000'), 'hold_cnt': Decimal('0.000539186768'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.000539186768'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('21.934700000000'), 'prc_curr': Decimal('13.725600000000'), 'prc_high': Decimal('22.361000000000'), 'prc_low': Decimal('13.201000000000'), 'prc_chg_pct': Decimal('-37.425200000000'), 'prc_chg_pct_high': Decimal('1.455700000000'), 'prc_chg_pct_low': Decimal('-39.816800000000'), 'prc_chg_pct_drop': Decimal('-38.880000000000'), 'prc_sell_avg': Decimal('13.725600000000'), 'val_curr': Decimal('0.007400661903'), 'val_tot': Decimal('0.007400661903'), 'gain_loss_amt_est': Decimal('-2.232818319831'), 'gain_loss_amt_est_high': Decimal('0.087341062335'), 'gain_loss_amt_est_low': Decimal('-2.389009195476'), 'gain_loss_amt': Decimal('-2.245510538097'), 'gain_loss_amt_net': Decimal('-2.238109876194'), 'gain_loss_pct_est': Decimal('-37.213600000000'), 'gain_loss_pct_est_high': Decimal('1.455700000000'), 'gain_loss_pct_est_low': Decimal('-39.816800000000'), 'gain_loss_pct': Decimal('-37.425200000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1d', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 28149, 'bo_uuid': '4aa36967-a5e2-480f-9780-6572afe69c05', 'buy_curr_symb': 'MUSE', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'MUSE', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'MUSE', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('1763900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 14, 41, 49), 'check_last_dttm': datetime.datetime(2024, 11, 27, 15, 58, 25), 'add_dttm': datetime.datetime(2024, 11, 25, 14, 41, 49), 'upd_dttm': datetime.datetime(2024, 11, 25, 14, 41, 49), 'del_dttm': datetime.datetime(2024, 11, 25, 14, 41, 49), 'dlm': datetime.datetime(2024, 11, 27, 15, 59, 44)},
		{'pos_id': 28425, 'symb': 'USDC', 'prod_id': 'MUSE-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 12, 34, 18), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 20, 58, 12), 'age_mins': 3084, 'tot_out_cnt': Decimal('12.000000000000'), 'tot_in_cnt': Decimal('6.603215587500'), 'buy_fees_cnt': Decimal('0.029925187032'), 'sell_fees_cnt_tot': Decimal('0.016549412500'), 'fees_cnt_tot': Decimal('0.046474599532'), 'buy_cnt': Decimal('0.485975998253'), 'sell_cnt_tot': Decimal('0.485000000000'), 'hold_cnt': Decimal('0.000975998253'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.000975998253'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('24.692577500000'), 'prc_curr': Decimal('13.614877500000'), 'prc_high': Decimal('26.690000000000'), 'prc_low': Decimal('13.201000000000'), 'prc_chg_pct': Decimal('-44.862500000000'), 'prc_chg_pct_high': Decimal('3.330600000000'), 'prc_chg_pct_low': Decimal('-46.538600000000'), 'prc_chg_pct_drop': Decimal('-48.190000000000'), 'prc_sell_avg': Decimal('13.614877500000'), 'val_curr': Decimal('0.013288096655'), 'val_tot': Decimal('0.013288096655'), 'gain_loss_amt_est': Decimal('-5.307138552060'), 'gain_loss_amt_est_high': Decimal('0.399677595425'), 'gain_loss_amt_est_low': Decimal('-5.584630847062'), 'gain_loss_amt': Decimal('-5.383496315845'), 'gain_loss_amt_net': Decimal('-5.370208219190'), 'gain_loss_pct_est': Decimal('-44.226200000000'), 'gain_loss_pct_est_high': Decimal('3.330600000000'), 'gain_loss_pct_est_low': Decimal('-46.538600000000'), 'gain_loss_pct': Decimal('-44.862500000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 28078, 'bo_uuid': 'fddf3d50-23a9-4a6f-a501-053a49c6de6a', 'buy_curr_symb': 'MUSE', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'MUSE', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'MUSE', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('1763900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 12, 35, 30), 'check_last_dttm': datetime.datetime(2024, 11, 27, 15, 58, 6), 'add_dttm': datetime.datetime(2024, 11, 25, 12, 35, 30), 'upd_dttm': datetime.datetime(2024, 11, 25, 12, 35, 30), 'del_dttm': datetime.datetime(2024, 11, 25, 12, 35, 30), 'dlm': datetime.datetime(2024, 11, 27, 15, 58, 28)},
		{'pos_id': 28620, 'symb': 'USDC', 'prod_id': 'POL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 17, 58, 41), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 20, 48, 53), 'age_mins': 2750, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.063601800000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007678200000'), 'fees_cnt_tot': Decimal('0.015159496758'), 'buy_cnt': Decimal('5.365821594481'), 'sell_cnt_tot': Decimal('5.360000000000'), 'hold_cnt': Decimal('0.005821594481'), 'pocket_cnt': Decimal('0.005821594481'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.559094250000'), 'prc_curr': Decimal('0.571567500000'), 'prc_high': Decimal('0.577700000000'), 'prc_low': Decimal('0.501400000000'), 'prc_chg_pct': Decimal('2.231000000000'), 'prc_chg_pct_high': Decimal('3.184700000000'), 'prc_chg_pct_low': Decimal('-10.319200000000'), 'prc_chg_pct_drop': Decimal('-0.950000000000'), 'prc_sell_avg': Decimal('0.571567500000'), 'val_curr': Decimal('0.003327434204'), 'val_tot': Decimal('0.003327434204'), 'gain_loss_amt_est': Decimal('0.076225520116'), 'gain_loss_amt_est_high': Decimal('0.095542477856'), 'gain_loss_amt_est_low': Decimal('-0.309577052527'), 'gain_loss_amt': Decimal('0.066929234204'), 'gain_loss_amt_net': Decimal('0.070256668407'), 'gain_loss_pct_est': Decimal('2.540900000000'), 'gain_loss_pct_est_high': Decimal('3.184700000000'), 'gain_loss_pct_est_low': Decimal('-10.319200000000'), 'gain_loss_pct': Decimal('2.231000000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28273, 'bo_uuid': 'bde908db-ae34-4855-9b66-4589a31a3b9d', 'buy_curr_symb': 'POL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'POL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'POL', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('26020434.110127510000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 18, 0, 26), 'check_last_dttm': datetime.datetime(2024, 11, 27, 15, 48, 46), 'add_dttm': datetime.datetime(2024, 11, 25, 18, 0, 26), 'upd_dttm': datetime.datetime(2024, 11, 25, 18, 0, 26), 'del_dttm': datetime.datetime(2024, 11, 25, 18, 0, 26), 'dlm': datetime.datetime(2024, 11, 27, 15, 49, 49)},
		{'pos_id': 30237, 'symb': 'USDC', 'prod_id': 'LCX-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 13, 50, 23), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 19, 38, 31), 'age_mins': 48, 'tot_out_cnt': Decimal('15.000000000000'), 'tot_in_cnt': Decimal('15.593139450000'), 'buy_fees_cnt': Decimal('0.037406483791'), 'sell_fees_cnt_tot': Decimal('0.039080550000'), 'fees_cnt_tot': Decimal('0.076487033791'), 'buy_cnt': Decimal('49.875311720698'), 'sell_cnt_tot': Decimal('49.800000000000'), 'hold_cnt': Decimal('0.075311720698'), 'pocket_cnt': Decimal('0.075311720698'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.300750000000'), 'prc_curr': Decimal('0.313115250000'), 'prc_high': Decimal('0.319900000000'), 'prc_low': Decimal('0.300000000000'), 'prc_chg_pct': Decimal('4.111500000000'), 'prc_chg_pct_high': Decimal('5.602700000000'), 'prc_chg_pct_low': Decimal('-0.016600000000'), 'prc_chg_pct_drop': Decimal('-1.490000000000'), 'prc_sell_avg': Decimal('0.313115250000'), 'val_curr': Decimal('0.023581248254'), 'val_tot': Decimal('0.023581248254'), 'gain_loss_amt_est': Decimal('0.655860349127'), 'gain_loss_amt_est_high': Decimal('0.840399002494'), 'gain_loss_amt_est_low': Decimal('-0.002493765586'), 'gain_loss_amt': Decimal('0.616720698254'), 'gain_loss_amt_net': Decimal('0.640301946509'), 'gain_loss_pct_est': Decimal('4.372400000000'), 'gain_loss_pct_est_high': Decimal('5.602700000000'), 'gain_loss_pct_est_low': Decimal('-0.016600000000'), 'gain_loss_pct': Decimal('4.111500000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29891, 'bo_uuid': '5c9d65de-dbd8-4178-bc95-3ff317535ed4', 'buy_curr_symb': 'LCX', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'LCX', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'LCX', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('117651200.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 13, 51, 26), 'check_last_dttm': datetime.datetime(2024, 11, 27, 14, 38, 25), 'add_dttm': datetime.datetime(2024, 11, 27, 13, 51, 26), 'upd_dttm': datetime.datetime(2024, 11, 27, 13, 51, 26), 'del_dttm': datetime.datetime(2024, 11, 27, 13, 51, 26), 'dlm': datetime.datetime(2024, 11, 27, 14, 39, 26)},
		{'pos_id': 30159, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 11, 42, 58), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 18, 3, 8), 'age_mins': 80, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.373923520000'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.013468480000'), 'fees_cnt_tot': Decimal('0.025937307930'), 'buy_cnt': Decimal('369.446753486654'), 'sell_cnt_tot': Decimal('368.000000000000'), 'hold_cnt': Decimal('1.446753486654'), 'pocket_cnt': Decimal('1.446753486654'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.013533750000'), 'prc_curr': Decimal('0.014603050000'), 'prc_high': Decimal('0.016430000000'), 'prc_low': Decimal('0.013297000000'), 'prc_chg_pct': Decimal('7.901000000000'), 'prc_chg_pct_high': Decimal('15.681200000000'), 'prc_chg_pct_low': Decimal('-1.749300000000'), 'prc_chg_pct_drop': Decimal('-7.780000000000'), 'prc_sell_avg': Decimal('0.014603050000'), 'val_curr': Decimal('0.021127013503'), 'val_tot': Decimal('0.021127013503'), 'gain_loss_amt_est': Decimal('0.468181398356'), 'gain_loss_amt_est_high': Decimal('0.784058372587'), 'gain_loss_amt_est_low': Decimal('-0.087466518888'), 'gain_loss_amt': Decimal('0.395050533503'), 'gain_loss_amt_net': Decimal('0.416177547007'), 'gain_loss_pct_est': Decimal('9.363600000000'), 'gain_loss_pct_est_high': Decimal('15.681200000000'), 'gain_loss_pct_est_low': Decimal('-1.749300000000'), 'gain_loss_pct': Decimal('7.901000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29813, 'bo_uuid': '0b1a6d9b-82cb-4b3c-92d7-caf246a6a134', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 11, 43, 31), 'check_last_dttm': datetime.datetime(2024, 11, 27, 13, 3, 1), 'add_dttm': datetime.datetime(2024, 11, 27, 11, 43, 31), 'upd_dttm': datetime.datetime(2024, 11, 27, 11, 43, 31), 'del_dttm': datetime.datetime(2024, 11, 27, 11, 43, 31), 'dlm': datetime.datetime(2024, 11, 27, 13, 3, 42)},
		{'pos_id': 30151, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 11, 35, 44), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 18, 2, 54), 'age_mins': 87, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.542770080000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.016397920000'), 'fees_cnt_tot': Decimal('0.031360513516'), 'buy_cnt': Decimal('449.394609286964'), 'sell_cnt_tot': Decimal('448.000000000000'), 'hold_cnt': Decimal('1.394609286964'), 'pocket_cnt': Decimal('1.394609286964'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.013351300000'), 'prc_curr': Decimal('0.014604400000'), 'prc_high': Decimal('0.016430000000'), 'prc_low': Decimal('0.013297000000'), 'prc_chg_pct': Decimal('9.385600000000'), 'prc_chg_pct_high': Decimal('17.262000000000'), 'prc_chg_pct_low': Decimal('-0.406700000000'), 'prc_chg_pct_drop': Decimal('-7.880000000000'), 'prc_sell_avg': Decimal('0.014604400000'), 'val_curr': Decimal('0.020367431871'), 'val_tot': Decimal('0.020367431871'), 'gain_loss_amt_est': Decimal('0.651489612056'), 'gain_loss_amt_est_high': Decimal('1.035722002997'), 'gain_loss_amt_est_low': Decimal('-0.024399880311'), 'gain_loss_amt': Decimal('0.563137511871'), 'gain_loss_amt_net': Decimal('0.583504943741'), 'gain_loss_pct_est': Decimal('10.858200000000'), 'gain_loss_pct_est_high': Decimal('17.262000000000'), 'gain_loss_pct_est_low': Decimal('-0.406700000000'), 'gain_loss_pct': Decimal('9.385600000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29805, 'bo_uuid': 'dcf20202-5dbe-4666-b531-3e79d29adcdf', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 11, 36, 32), 'check_last_dttm': datetime.datetime(2024, 11, 27, 13, 2, 47), 'add_dttm': datetime.datetime(2024, 11, 27, 11, 36, 32), 'upd_dttm': datetime.datetime(2024, 11, 27, 11, 36, 32), 'del_dttm': datetime.datetime(2024, 11, 27, 11, 36, 32), 'dlm': datetime.datetime(2024, 11, 27, 13, 3, 36)},
		{'pos_id': 30140, 'symb': 'USDC', 'prod_id': 'SPA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 11, 28, 24), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 18, 2, 41), 'age_mins': 94, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.285989437500'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.008235562500'), 'fees_cnt_tot': Decimal('0.015716859258'), 'buy_cnt': Decimal('225.901615704831'), 'sell_cnt_tot': Decimal('225.000000000000'), 'hold_cnt': Decimal('0.901615704831'), 'pocket_cnt': Decimal('0.901615704831'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.013280120000'), 'prc_curr': Decimal('0.014604400000'), 'prc_high': Decimal('0.016430000000'), 'prc_low': Decimal('0.013209000000'), 'prc_chg_pct': Decimal('9.971900000000'), 'prc_chg_pct_high': Decimal('17.890500000000'), 'prc_chg_pct_low': Decimal('-0.535500000000'), 'prc_chg_pct_drop': Decimal('-7.920000000000'), 'prc_sell_avg': Decimal('0.014604400000'), 'val_curr': Decimal('0.013167556400'), 'val_tot': Decimal('0.013167556400'), 'gain_loss_amt_est': Decimal('0.343569814047'), 'gain_loss_amt_est_high': Decimal('0.536715695475'), 'gain_loss_amt_est_low': Decimal('-0.016065558155'), 'gain_loss_amt': Decimal('0.299156993900'), 'gain_loss_amt_net': Decimal('0.312324550299'), 'gain_loss_pct_est': Decimal('11.452300000000'), 'gain_loss_pct_est_high': Decimal('17.890500000000'), 'gain_loss_pct_est_low': Decimal('-0.535500000000'), 'gain_loss_pct': Decimal('9.971900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'bb_bo', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29794, 'bo_uuid': '3d020dfd-9e86-4528-9bec-c1d1258f4714', 'buy_curr_symb': 'SPA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SPA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SPA', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('780387539.540182100000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 11, 29, 40), 'check_last_dttm': datetime.datetime(2024, 11, 27, 13, 2, 34), 'add_dttm': datetime.datetime(2024, 11, 27, 11, 29, 40), 'upd_dttm': datetime.datetime(2024, 11, 27, 11, 29, 40), 'del_dttm': datetime.datetime(2024, 11, 27, 11, 29, 40), 'dlm': datetime.datetime(2024, 11, 27, 13, 3, 30)},
		{'pos_id': 30077, 'symb': 'USDC', 'prod_id': 'LCX-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 10, 30, 9), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 17, 27, 40), 'age_mins': 118, 'tot_out_cnt': Decimal('15.000000000000'), 'tot_in_cnt': Decimal('14.295451800000'), 'buy_fees_cnt': Decimal('0.037406483791'), 'sell_fees_cnt_tot': Decimal('0.035828200000'), 'fees_cnt_tot': Decimal('0.073234683791'), 'buy_cnt': Decimal('49.941900921927'), 'sell_cnt_tot': Decimal('49.900000000000'), 'hold_cnt': Decimal('0.041900921927'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.041900921927'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.300349000000'), 'prc_curr': Decimal('0.286482000000'), 'prc_high': Decimal('0.320000000000'), 'prc_low': Decimal('0.286482000000'), 'prc_chg_pct': Decimal('-4.617000000000'), 'prc_chg_pct_high': Decimal('3.479600000000'), 'prc_chg_pct_low': Decimal('-4.617000000000'), 'prc_chg_pct_drop': Decimal('-8.100000000000'), 'prc_sell_avg': Decimal('0.286482000000'), 'val_curr': Decimal('0.012003859915'), 'val_tot': Decimal('0.012003859915'), 'gain_loss_amt_est': Decimal('-0.636709294854'), 'gain_loss_amt_est_high': Decimal('0.521942806535'), 'gain_loss_amt_est_low': Decimal('-0.636709294854'), 'gain_loss_amt': Decimal('-0.692544340085'), 'gain_loss_amt_net': Decimal('-0.680540480169'), 'gain_loss_pct_est': Decimal('-4.244700000000'), 'gain_loss_pct_est_high': Decimal('3.479600000000'), 'gain_loss_pct_est_low': Decimal('-4.244700000000'), 'gain_loss_pct': Decimal('-4.617000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'trail_stop', 'sell_strat_freq': '', 'bo_id': 29731, 'bo_uuid': 'c6921444-d97b-4f26-9ece-6cdba647d92d', 'buy_curr_symb': 'LCX', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'LCX', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'LCX', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('117651200.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 10, 30, 51), 'check_last_dttm': datetime.datetime(2024, 11, 27, 12, 27, 33), 'add_dttm': datetime.datetime(2024, 11, 27, 10, 30, 51), 'upd_dttm': datetime.datetime(2024, 11, 27, 10, 30, 51), 'del_dttm': datetime.datetime(2024, 11, 27, 10, 30, 51), 'dlm': datetime.datetime(2024, 11, 27, 12, 28, 1)},
		{'pos_id': 29006, 'symb': 'USDC', 'prod_id': 'SHIB-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 26, 4, 15, 25), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 17, 9, 47), 'age_mins': 1915, 'tot_out_cnt': Decimal('2.000000000000'), 'tot_in_cnt': Decimal('2.051514519600'), 'buy_fees_cnt': Decimal('0.004987531172'), 'sell_fees_cnt_tot': Decimal('0.005141640400'), 'fees_cnt_tot': Decimal('0.010129171572'), 'buy_cnt': Decimal('81528.911680749100'), 'sell_cnt_tot': Decimal('81484.000000000000'), 'hold_cnt': Decimal('44.911680749094'), 'pocket_cnt': Decimal('44.911680749094'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.000024530000'), 'prc_curr': Decimal('0.000025180000'), 'prc_high': Decimal('0.000025590000'), 'prc_low': Decimal('0.000023500000'), 'prc_chg_pct': Decimal('2.649800000000'), 'prc_chg_pct_high': Decimal('3.546700000000'), 'prc_chg_pct_low': Decimal('-4.198900000000'), 'prc_chg_pct_drop': Decimal('-0.900000000000'), 'prc_sell_avg': Decimal('0.000025180000'), 'val_curr': Decimal('0.001130876121'), 'val_tot': Decimal('0.001130876121'), 'gain_loss_amt_est': Decimal('0.054528574355'), 'gain_loss_amt_est_high': Decimal('0.070834356691'), 'gain_loss_amt_est_low': Decimal('-0.084070575502'), 'gain_loss_amt': Decimal('0.052645395721'), 'gain_loss_amt_net': Decimal('0.053776271843'), 'gain_loss_pct_est': Decimal('2.726400000000'), 'gain_loss_pct_est_high': Decimal('3.541700000000'), 'gain_loss_pct_est_low': Decimal('-4.203500000000'), 'gain_loss_pct': Decimal('2.632300000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28659, 'bo_uuid': 'a05296b1-eda9-479b-980a-d44763870672', 'buy_curr_symb': 'SHIB', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'SHIB', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'SHIB', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('970443076600.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('1.0000E-8'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 26, 4, 16, 37), 'check_last_dttm': datetime.datetime(2024, 11, 27, 12, 9, 41), 'add_dttm': datetime.datetime(2024, 11, 26, 4, 16, 37), 'upd_dttm': datetime.datetime(2024, 11, 26, 4, 16, 37), 'del_dttm': datetime.datetime(2024, 11, 26, 4, 16, 37), 'dlm': datetime.datetime(2024, 11, 27, 12, 10, 49)},
		{'pos_id': 30103, 'symb': 'USDC', 'prod_id': 'MOBILE-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 10, 57, 25), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 17, 6, 3), 'age_mins': 68, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.109248600000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015311400000'), 'fees_cnt_tot': Decimal('0.030273993516'), 'buy_cnt': Decimal('6762.754131620102'), 'sell_cnt_tot': Decimal('6760.000000000000'), 'hold_cnt': Decimal('2.754131620102'), 'pocket_cnt': Decimal('2.754131620102'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.000887210000'), 'prc_curr': Decimal('0.000903740000'), 'prc_high': Decimal('0.000950000000'), 'prc_low': Decimal('0.000860000000'), 'prc_chg_pct': Decimal('1.863100000000'), 'prc_chg_pct_high': Decimal('4.259400000000'), 'prc_chg_pct_low': Decimal('-3.066900000000'), 'prc_chg_pct_drop': Decimal('-2.400000000000'), 'prc_sell_avg': Decimal('0.000903740000'), 'val_curr': Decimal('0.002489018910'), 'val_tot': Decimal('0.002489018910'), 'gain_loss_amt_est': Decimal('0.100004226721'), 'gain_loss_amt_est_high': Decimal('0.255547571749'), 'gain_loss_amt_est_low': Decimal('-0.184031446807'), 'gain_loss_amt': Decimal('0.111737618910'), 'gain_loss_amt_net': Decimal('0.114226637821'), 'gain_loss_pct_est': Decimal('1.666700000000'), 'gain_loss_pct_est_high': Decimal('4.259100000000'), 'gain_loss_pct_est_low': Decimal('-3.067200000000'), 'gain_loss_pct': Decimal('1.862300000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29757, 'bo_uuid': 'c7a33dca-a5b6-48be-ab1c-10540c3ed9c0', 'buy_curr_symb': 'MOBILE', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'MOBILE', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'MOBILE', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('2779713425.988456200000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 10, 58, 6), 'check_last_dttm': datetime.datetime(2024, 11, 27, 12, 5, 57), 'add_dttm': datetime.datetime(2024, 11, 27, 10, 58, 6), 'upd_dttm': datetime.datetime(2024, 11, 27, 10, 58, 6), 'del_dttm': datetime.datetime(2024, 11, 27, 10, 58, 6), 'dlm': datetime.datetime(2024, 11, 27, 12, 7, 14)},
		{'pos_id': 30091, 'symb': 'USDC', 'prod_id': 'MOBILE-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 10, 44, 9), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 17, 5, 49), 'age_mins': 82, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.081736350000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007723650000'), 'fees_cnt_tot': Decimal('0.015204946758'), 'buy_cnt': Decimal('3412.222010538079'), 'sell_cnt_tot': Decimal('3410.000000000000'), 'hold_cnt': Decimal('2.222010538079'), 'pocket_cnt': Decimal('2.222010538079'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.000879190000'), 'prc_curr': Decimal('0.000903740000'), 'prc_high': Decimal('0.000950000000'), 'prc_low': Decimal('0.000860000000'), 'prc_chg_pct': Decimal('2.792300000000'), 'prc_chg_pct_high': Decimal('5.210500000000'), 'prc_chg_pct_low': Decimal('-2.182700000000'), 'prc_chg_pct_drop': Decimal('-2.420000000000'), 'prc_sell_avg': Decimal('0.000903740000'), 'val_curr': Decimal('0.002008119804'), 'val_tot': Decimal('0.002008119804'), 'gain_loss_amt_est': Decimal('0.077824253505'), 'gain_loss_amt_est_high': Decimal('0.156305359748'), 'gain_loss_amt_est_low': Decimal('-0.065489070937'), 'gain_loss_amt': Decimal('0.083744469804'), 'gain_loss_amt_net': Decimal('0.085752589607'), 'gain_loss_pct_est': Decimal('2.594100000000'), 'gain_loss_pct_est_high': Decimal('5.210200000000'), 'gain_loss_pct_est_low': Decimal('-2.183000000000'), 'gain_loss_pct': Decimal('2.791500000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29745, 'bo_uuid': 'bb6b5547-fdba-4cd1-9fe7-88806560cae4', 'buy_curr_symb': 'MOBILE', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'MOBILE', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'MOBILE', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('2779713425.988456200000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 10, 44, 29), 'check_last_dttm': datetime.datetime(2024, 11, 27, 12, 5, 42), 'add_dttm': datetime.datetime(2024, 11, 27, 10, 44, 29), 'upd_dttm': datetime.datetime(2024, 11, 27, 10, 44, 29), 'del_dttm': datetime.datetime(2024, 11, 27, 10, 44, 29), 'dlm': datetime.datetime(2024, 11, 27, 12, 7, 8)},
		{'pos_id': 30076, 'symb': 'USDC', 'prod_id': 'MOBILE-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 10, 23, 29), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 17, 5, 35), 'age_mins': 102, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.212639475000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015570525000'), 'fees_cnt_tot': Decimal('0.030533118516'), 'buy_cnt': Decimal('6887.269742789172'), 'sell_cnt_tot': Decimal('6882.000000000000'), 'hold_cnt': Decimal('5.269742789172'), 'pocket_cnt': Decimal('5.269742789172'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.000871170000'), 'prc_curr': Decimal('0.000902740000'), 'prc_high': Decimal('0.000950000000'), 'prc_low': Decimal('0.000860000000'), 'prc_chg_pct': Decimal('3.623900000000'), 'prc_chg_pct_high': Decimal('6.179000000000'), 'prc_chg_pct_low': Decimal('-1.282200000000'), 'prc_chg_pct_drop': Decimal('-2.560000000000'), 'prc_sell_avg': Decimal('0.000902740000'), 'val_curr': Decimal('0.004757207605'), 'val_tot': Decimal('0.004757207605'), 'gain_loss_amt_est': Decimal('0.212317307996'), 'gain_loss_amt_est_high': Decimal('0.370724512080'), 'gain_loss_amt_est_low': Decimal('-0.076948021201'), 'gain_loss_amt': Decimal('0.217396682605'), 'gain_loss_amt_net': Decimal('0.222153890211'), 'gain_loss_pct_est': Decimal('3.538600000000'), 'gain_loss_pct_est_high': Decimal('6.178700000000'), 'gain_loss_pct_est_low': Decimal('-1.282500000000'), 'gain_loss_pct': Decimal('3.623300000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29730, 'bo_uuid': 'fe2e3dde-f6f7-428c-8037-d473b5fd4aa1', 'buy_curr_symb': 'MOBILE', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'MOBILE', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'MOBILE', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('2779713425.988456200000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 10, 24, 42), 'check_last_dttm': datetime.datetime(2024, 11, 27, 12, 5, 29), 'add_dttm': datetime.datetime(2024, 11, 27, 10, 24, 42), 'upd_dttm': datetime.datetime(2024, 11, 27, 10, 24, 42), 'del_dttm': datetime.datetime(2024, 11, 27, 10, 24, 42), 'dlm': datetime.datetime(2024, 11, 27, 12, 5, 50)},
		{'pos_id': 29387, 'symb': 'USDC', 'prod_id': 'DOGE-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 26, 12, 21, 30), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 16, 55, 49), 'age_mins': 1415, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.107072320000'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.012799680000'), 'fees_cnt_tot': Decimal('0.025268507930'), 'buy_cnt': Decimal('12.854793092785'), 'sell_cnt_tot': Decimal('12.800000000000'), 'hold_cnt': Decimal('0.054793092785'), 'pocket_cnt': Decimal('0.054793092785'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.388959980000'), 'prc_curr': Decimal('0.398990030000'), 'prc_high': Decimal('0.405850000000'), 'prc_low': Decimal('0.370680000000'), 'prc_chg_pct': Decimal('2.578700000000'), 'prc_chg_pct_high': Decimal('3.581400000000'), 'prc_chg_pct_low': Decimal('-4.699700000000'), 'prc_chg_pct_drop': Decimal('-1.000000000000'), 'prc_sell_avg': Decimal('0.398990030000'), 'val_curr': Decimal('0.021861897734'), 'val_tot': Decimal('0.021861897734'), 'gain_loss_amt_est': Decimal('0.133175977811'), 'gain_loss_amt_est_high': Decimal('0.179067589152'), 'gain_loss_amt_est_low': Decimal('-0.234985296366'), 'gain_loss_amt': Decimal('0.128934217734'), 'gain_loss_amt_net': Decimal('0.150796115468'), 'gain_loss_pct_est': Decimal('2.663500000000'), 'gain_loss_pct_est_high': Decimal('3.581400000000'), 'gain_loss_pct_est_low': Decimal('-4.699700000000'), 'gain_loss_pct': Decimal('2.578700000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29041, 'bo_uuid': '3ae78cee-f41b-474b-9de5-40c2d5fe40a2', 'buy_curr_symb': 'DOGE', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOGE', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOGE', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('141834300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 26, 12, 23, 25), 'check_last_dttm': datetime.datetime(2024, 11, 27, 11, 55, 42), 'add_dttm': datetime.datetime(2024, 11, 26, 12, 23, 25), 'upd_dttm': datetime.datetime(2024, 11, 26, 12, 23, 25), 'del_dttm': datetime.datetime(2024, 11, 26, 12, 23, 25), 'dlm': datetime.datetime(2024, 11, 27, 11, 56, 16)},
		{'pos_id': 28275, 'symb': 'USDC', 'prod_id': 'ADA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 10, 26, 16), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 16, 50, 27), 'age_mins': 2964, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.052042341913'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015168025920'), 'fees_cnt_tot': Decimal('0.030130619436'), 'buy_cnt': Decimal('5.967730986623'), 'sell_cnt_tot': Decimal('5.966378570000'), 'hold_cnt': Decimal('0.001352416623'), 'pocket_cnt': Decimal('0.001352416623'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('1.005407250000'), 'prc_curr': Decimal('1.014357750000'), 'prc_high': Decimal('1.042700000000'), 'prc_low': Decimal('0.879800000000'), 'prc_chg_pct': Decimal('0.890200000000'), 'prc_chg_pct_high': Decimal('3.251700000000'), 'prc_chg_pct_low': Decimal('-12.493200000000'), 'prc_chg_pct_drop': Decimal('-2.360000000000'), 'prc_sell_avg': Decimal('1.014357750000'), 'val_curr': Decimal('0.001371834283'), 'val_tot': Decimal('0.001371834283'), 'gain_loss_amt_est': Decimal('0.067988867198'), 'gain_loss_amt_est_high': Decimal('0.195101537213'), 'gain_loss_amt_est_low': Decimal('-0.749590277969'), 'gain_loss_amt': Decimal('0.053414176196'), 'gain_loss_amt_net': Decimal('0.054786010479'), 'gain_loss_pct_est': Decimal('1.133100000000'), 'gain_loss_pct_est_high': Decimal('3.251700000000'), 'gain_loss_pct_est_low': Decimal('-12.493200000000'), 'gain_loss_pct': Decimal('0.890200000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 27928, 'bo_uuid': '40092155-3144-4c2f-b78d-67e9a45bc747', 'buy_curr_symb': 'ADA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ADA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ADA', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('30176000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 10, 27, 7), 'check_last_dttm': datetime.datetime(2024, 11, 27, 11, 50, 20), 'add_dttm': datetime.datetime(2024, 11, 25, 10, 27, 7), 'upd_dttm': datetime.datetime(2024, 11, 25, 10, 27, 7), 'del_dttm': datetime.datetime(2024, 11, 25, 10, 27, 7), 'dlm': datetime.datetime(2024, 11, 27, 11, 50, 37)},
		{'pos_id': 29847, 'symb': 'USDC', 'prod_id': 'WELL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 4, 56, 51), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 16, 43, 37), 'age_mins': 407, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('5.437609905000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.013628095000'), 'fees_cnt_tot': Decimal('0.028590688516'), 'buy_cnt': Decimal('71.049982863632'), 'sell_cnt_tot': Decimal('71.000000000000'), 'hold_cnt': Decimal('0.049982863632'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.049982863632'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.084447590000'), 'prc_curr': Decimal('0.076586060000'), 'prc_high': Decimal('0.085000000000'), 'prc_low': Decimal('0.076586060000'), 'prc_chg_pct': Decimal('-9.309400000000'), 'prc_chg_pct_high': Decimal('0.307200000000'), 'prc_chg_pct_low': Decimal('-9.309400000000'), 'prc_chg_pct_drop': Decimal('-9.620000000000'), 'prc_sell_avg': Decimal('0.076586060000'), 'val_curr': Decimal('0.003827990593'), 'val_tot': Decimal('0.003827990593'), 'gain_loss_amt_est': Decimal('-0.544853365713'), 'gain_loss_amt_est_high': Decimal('0.018430898430'), 'gain_loss_amt_est_low': Decimal('-0.544853365713'), 'gain_loss_amt': Decimal('-0.558562104407'), 'gain_loss_amt_net': Decimal('-0.554734113814'), 'gain_loss_pct_est': Decimal('-9.080900000000'), 'gain_loss_pct_est_high': Decimal('0.307200000000'), 'gain_loss_pct_est_low': Decimal('-9.080900000000'), 'gain_loss_pct': Decimal('-9.309400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '30min', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 29501, 'bo_uuid': 'c8330ddb-4bef-4a56-81bd-3232ec727486', 'buy_curr_symb': 'WELL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'WELL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'WELL', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('289584259.672040800000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 4, 57, 13), 'check_last_dttm': datetime.datetime(2024, 11, 27, 11, 43, 30), 'add_dttm': datetime.datetime(2024, 11, 27, 4, 57, 13), 'upd_dttm': datetime.datetime(2024, 11, 27, 4, 57, 13), 'del_dttm': datetime.datetime(2024, 11, 27, 4, 57, 13), 'dlm': datetime.datetime(2024, 11, 27, 11, 43, 50)},
		{'pos_id': 29897, 'symb': 'USDC', 'prod_id': 'DEGEN-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 5, 57, 15), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 16, 41, 19), 'age_mins': 344, 'tot_out_cnt': Decimal('17.000000000000'), 'tot_in_cnt': Decimal('17.442895470000'), 'buy_fees_cnt': Decimal('0.042394014963'), 'sell_fees_cnt_tot': Decimal('0.043716530000'), 'fees_cnt_tot': Decimal('0.086110544963'), 'buy_cnt': Decimal('836.834089273461'), 'sell_cnt_tot': Decimal('836.000000000000'), 'hold_cnt': Decimal('0.834089273461'), 'pocket_cnt': Decimal('0.834089273461'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.020314660000'), 'prc_curr': Decimal('0.020864710000'), 'prc_high': Decimal('0.022019000000'), 'prc_low': Decimal('0.019129000000'), 'prc_chg_pct': Decimal('2.707700000000'), 'prc_chg_pct_high': Decimal('5.003000000000'), 'prc_chg_pct_low': Decimal('-5.836500000000'), 'prc_chg_pct_drop': Decimal('-2.300000000000'), 'prc_sell_avg': Decimal('0.020864710000'), 'val_curr': Decimal('0.017403030805'), 'val_tot': Decimal('0.017403030805'), 'gain_loss_amt_est': Decimal('0.518284824851'), 'gain_loss_amt_est_high': Decimal('0.850507958292'), 'gain_loss_amt_est_low': Decimal('-0.992200706288'), 'gain_loss_amt': Decimal('0.460298500805'), 'gain_loss_amt_net': Decimal('0.477701531610'), 'gain_loss_pct_est': Decimal('3.048700000000'), 'gain_loss_pct_est_high': Decimal('5.003000000000'), 'gain_loss_pct_est_low': Decimal('-5.836500000000'), 'gain_loss_pct': Decimal('2.707600000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29551, 'bo_uuid': 'dac5b562-468b-423e-8a16-13ed87b3568d', 'buy_curr_symb': 'DEGEN', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DEGEN', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DEGEN', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('870720519.417491400000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 5, 57, 51), 'check_last_dttm': datetime.datetime(2024, 11, 27, 11, 41, 12), 'add_dttm': datetime.datetime(2024, 11, 27, 5, 57, 51), 'upd_dttm': datetime.datetime(2024, 11, 27, 5, 57, 51), 'del_dttm': datetime.datetime(2024, 11, 27, 5, 57, 51), 'dlm': datetime.datetime(2024, 11, 27, 11, 42, 2)},
		{'pos_id': 28184, 'symb': 'USDC', 'prod_id': 'XLM-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 9, 38, 29), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 16, 37, 44), 'age_mins': 2999, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.033370283288'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015121228780'), 'fees_cnt_tot': Decimal('0.030083822296'), 'buy_cnt': Decimal('11.834715370141'), 'sell_cnt_tot': Decimal('11.833289990000'), 'hold_cnt': Decimal('0.001425380141'), 'pocket_cnt': Decimal('0.001425380141'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.506983040000'), 'prc_curr': Decimal('0.509864140000'), 'prc_high': Decimal('0.532614000000'), 'prc_low': Decimal('0.416771000000'), 'prc_chg_pct': Decimal('0.568300000000'), 'prc_chg_pct_high': Decimal('3.759300000000'), 'prc_chg_pct_low': Decimal('-17.793900000000'), 'prc_chg_pct_drop': Decimal('-3.190000000000'), 'prc_sell_avg': Decimal('0.509864140000'), 'val_curr': Decimal('0.000726750220'), 'val_tot': Decimal('0.000726750220'), 'gain_loss_amt_est': Decimal('0.036130888525'), 'gain_loss_amt_est_high': Decimal('0.225557342740'), 'gain_loss_amt_est_low': Decimal('-1.067633840471'), 'gain_loss_amt': Decimal('0.034097033508'), 'gain_loss_amt_net': Decimal('0.034823783728'), 'gain_loss_pct_est': Decimal('0.602200000000'), 'gain_loss_pct_est_high': Decimal('3.759300000000'), 'gain_loss_pct_est_low': Decimal('-17.793900000000'), 'gain_loss_pct': Decimal('0.568300000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1d', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 27837, 'bo_uuid': 'e10ab706-a1e2-42ea-b088-a14067d01b8d', 'buy_curr_symb': 'XLM', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XLM', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XLM', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('116945400.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 9, 38, 50), 'check_last_dttm': datetime.datetime(2024, 11, 27, 11, 37, 37), 'add_dttm': datetime.datetime(2024, 11, 25, 9, 38, 50), 'upd_dttm': datetime.datetime(2024, 11, 25, 9, 38, 50), 'del_dttm': datetime.datetime(2024, 11, 25, 9, 38, 50), 'dlm': datetime.datetime(2024, 11, 27, 11, 37, 58)},
		{'pos_id': 28160, 'symb': 'USDC', 'prod_id': 'XLM-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 9, 14, 30), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 16, 37, 30), 'age_mins': 3023, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.033701704114'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015122059409'), 'fees_cnt_tot': Decimal('0.030084652925'), 'buy_cnt': Decimal('11.831457466750'), 'sell_cnt_tot': Decimal('11.830098030000'), 'hold_cnt': Decimal('0.001359436750'), 'pocket_cnt': Decimal('0.001359436750'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.507122640000'), 'prc_curr': Decimal('0.510029730000'), 'prc_high': Decimal('0.532614000000'), 'prc_low': Decimal('0.416771000000'), 'prc_chg_pct': Decimal('0.573300000000'), 'prc_chg_pct_high': Decimal('3.730700000000'), 'prc_chg_pct_low': Decimal('-17.816500000000'), 'prc_chg_pct_drop': Decimal('-3.160000000000'), 'prc_sell_avg': Decimal('0.510029730000'), 'val_curr': Decimal('0.000693353159'), 'val_tot': Decimal('0.000693353159'), 'gain_loss_amt_est': Decimal('0.034469240511'), 'gain_loss_amt_est_high': Decimal('0.223843548724'), 'gain_loss_amt_est_low': Decimal('-1.068991640125'), 'gain_loss_amt': Decimal('0.034395057273'), 'gain_loss_amt_net': Decimal('0.035088410431'), 'gain_loss_pct_est': Decimal('0.574500000000'), 'gain_loss_pct_est_high': Decimal('3.730700000000'), 'gain_loss_pct_est_low': Decimal('-17.816500000000'), 'gain_loss_pct': Decimal('0.573300000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 27813, 'bo_uuid': '51f30c39-62ea-4eff-8006-a1bc30390717', 'buy_curr_symb': 'XLM', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'XLM', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'XLM', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('116945400.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 9, 15, 44), 'check_last_dttm': datetime.datetime(2024, 11, 27, 11, 37, 24), 'add_dttm': datetime.datetime(2024, 11, 25, 9, 15, 44), 'upd_dttm': datetime.datetime(2024, 11, 25, 9, 15, 44), 'del_dttm': datetime.datetime(2024, 11, 25, 9, 15, 44), 'dlm': datetime.datetime(2024, 11, 27, 11, 37, 52)},
		{'pos_id': 29826, 'symb': 'USDC', 'prod_id': 'WELL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 3, 51, 13), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 16, 28, 47), 'age_mins': 458, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('5.487965700000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.013754300000'), 'fees_cnt_tot': Decimal('0.028716893516'), 'buy_cnt': Decimal('70.563293245346'), 'sell_cnt_tot': Decimal('70.000000000000'), 'hold_cnt': Decimal('0.563293245346'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.563293245346'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.085030040000'), 'prc_curr': Decimal('0.078399510000'), 'prc_high': Decimal('0.085000000000'), 'prc_low': Decimal('0.078399510000'), 'prc_chg_pct': Decimal('-7.797900000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-7.797900000000'), 'prc_chg_pct_drop': Decimal('-7.800000000000'), 'prc_sell_avg': Decimal('0.078399510000'), 'val_curr': Decimal('0.044161914421'), 'val_tot': Decimal('0.044161914421'), 'gain_loss_amt_est': Decimal('-0.428816308400'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-0.428816308400'), 'gain_loss_amt': Decimal('-0.467872385579'), 'gain_loss_amt_net': Decimal('-0.423710471157'), 'gain_loss_pct_est': Decimal('-7.146900000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-7.146900000000'), 'gain_loss_pct': Decimal('-7.797900000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '4h', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 29480, 'bo_uuid': '76df9c4f-a3d3-4394-a500-d7252bde6562', 'buy_curr_symb': 'WELL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'WELL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'WELL', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('289584259.672040800000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 3, 52, 25), 'check_last_dttm': datetime.datetime(2024, 11, 27, 11, 28, 40), 'add_dttm': datetime.datetime(2024, 11, 27, 3, 52, 25), 'upd_dttm': datetime.datetime(2024, 11, 27, 3, 52, 25), 'del_dttm': datetime.datetime(2024, 11, 27, 3, 52, 25), 'dlm': datetime.datetime(2024, 11, 27, 11, 29, 47)},
		{'pos_id': 29395, 'symb': 'USDC', 'prod_id': 'DOT-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 26, 12, 26, 26), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 16, 27, 18), 'age_mins': 1381, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('20.535836810198'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.051468262682'), 'fees_cnt_tot': Decimal('0.101343574403'), 'buy_cnt': Decimal('2.460548185530'), 'sell_cnt_tot': Decimal('2.459066540000'), 'hold_cnt': Decimal('0.001481645530'), 'pocket_cnt': Decimal('0.001481645530'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('8.128270000000'), 'prc_curr': Decimal('8.351070000000'), 'prc_high': Decimal('8.478000000000'), 'prc_low': Decimal('7.857000000000'), 'prc_chg_pct': Decimal('2.741100000000'), 'prc_chg_pct_high': Decimal('3.958200000000'), 'prc_chg_pct_low': Decimal('-3.337400000000'), 'prc_chg_pct_drop': Decimal('-1.220000000000'), 'prc_sell_avg': Decimal('8.351070000000'), 'val_curr': Decimal('0.012373325536'), 'val_tot': Decimal('0.012373325536'), 'gain_loss_amt_est': Decimal('0.602169957443'), 'gain_loss_amt_est_high': Decimal('0.791632167728'), 'gain_loss_amt_est_low': Decimal('-0.667472906291'), 'gain_loss_amt': Decimal('0.548210135734'), 'gain_loss_amt_net': Decimal('0.560583461270'), 'gain_loss_pct_est': Decimal('3.010800000000'), 'gain_loss_pct_est_high': Decimal('3.958200000000'), 'gain_loss_pct_est_low': Decimal('-3.337400000000'), 'gain_loss_pct': Decimal('2.741100000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29049, 'bo_uuid': 'd1456b56-5b62-4843-a923-52fafa3df5cb', 'buy_curr_symb': 'DOT', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOT', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOT', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('1732900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 26, 12, 28, 24), 'check_last_dttm': datetime.datetime(2024, 11, 27, 11, 27, 11), 'add_dttm': datetime.datetime(2024, 11, 26, 12, 28, 24), 'upd_dttm': datetime.datetime(2024, 11, 26, 12, 28, 24), 'del_dttm': datetime.datetime(2024, 11, 26, 12, 28, 24), 'dlm': datetime.datetime(2024, 11, 27, 11, 28)},
		{'pos_id': 28683, 'symb': 'USDC', 'prod_id': 'POL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 19, 13, 25), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 16, 26, 43), 'age_mins': 2414, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.070322955000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007695045000'), 'fees_cnt_tot': Decimal('0.015176341758'), 'buy_cnt': Decimal('5.428113011504'), 'sell_cnt_tot': Decimal('5.420000000000'), 'hold_cnt': Decimal('0.008113011504'), 'pocket_cnt': Decimal('0.008113011504'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.552678250000'), 'prc_curr': Decimal('0.566480250000'), 'prc_high': Decimal('0.573900000000'), 'prc_low': Decimal('0.501400000000'), 'prc_chg_pct': Decimal('2.497300000000'), 'prc_chg_pct_high': Decimal('3.206500000000'), 'prc_chg_pct_low': Decimal('-9.278100000000'), 'prc_chg_pct_drop': Decimal('-0.710000000000'), 'prc_sell_avg': Decimal('0.566480250000'), 'val_curr': Decimal('0.004595860785'), 'val_tot': Decimal('0.004595860785'), 'gain_loss_amt_est': Decimal('0.076654454920'), 'gain_loss_amt_est_high': Decimal('0.096195661762'), 'gain_loss_amt_est_low': Decimal('-0.278344136032'), 'gain_loss_amt': Decimal('0.074918815785'), 'gain_loss_amt_net': Decimal('0.079514676570'), 'gain_loss_pct_est': Decimal('2.555100000000'), 'gain_loss_pct_est_high': Decimal('3.206500000000'), 'gain_loss_pct_est_low': Decimal('-9.278100000000'), 'gain_loss_pct': Decimal('2.497300000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28336, 'bo_uuid': 'c2c3ae3b-583b-42d7-af19-e2f5ee765f9e', 'buy_curr_symb': 'POL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'POL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'POL', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('26020434.110127510000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 19, 14, 29), 'check_last_dttm': datetime.datetime(2024, 11, 27, 11, 26, 37), 'add_dttm': datetime.datetime(2024, 11, 25, 19, 14, 29), 'upd_dttm': datetime.datetime(2024, 11, 25, 19, 14, 29), 'del_dttm': datetime.datetime(2024, 11, 25, 19, 14, 29), 'dlm': datetime.datetime(2024, 11, 27, 11, 27, 54)},
		{'pos_id': 29917, 'symb': 'USDC', 'prod_id': 'PLU-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 6, 41, 40), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 15, 28, 16), 'age_mins': 227, 'tot_out_cnt': Decimal('20.000000000000'), 'tot_in_cnt': Decimal('18.528063750000'), 'buy_fees_cnt': Decimal('0.049875311721'), 'sell_fees_cnt_tot': Decimal('0.046436250000'), 'fees_cnt_tot': Decimal('0.096311561721'), 'buy_cnt': Decimal('9.151433343247'), 'sell_cnt_tot': Decimal('9.150000000000'), 'hold_cnt': Decimal('0.001433343247'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.001433343247'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('2.185450000000'), 'prc_curr': Decimal('2.024925000000'), 'prc_high': Decimal('2.180000000000'), 'prc_low': Decimal('2.024925000000'), 'prc_chg_pct': Decimal('-7.345200000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-7.345200000000'), 'prc_chg_pct_drop': Decimal('-7.350000000000'), 'prc_sell_avg': Decimal('2.024925000000'), 'val_curr': Decimal('0.002902412574'), 'val_tot': Decimal('0.002902412574'), 'gain_loss_amt_est': Decimal('-1.422590313209'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-1.422590313209'), 'gain_loss_amt': Decimal('-1.469033837426'), 'gain_loss_amt_net': Decimal('-1.466131424851'), 'gain_loss_pct_est': Decimal('-7.113000000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-7.113000000000'), 'gain_loss_pct': Decimal('-7.345200000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '1d', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'hard_stop_loss', 'sell_strat_freq': '', 'bo_id': 29571, 'bo_uuid': 'd0f713e6-6249-4e64-ab56-d26684c0a346', 'buy_curr_symb': 'PLU', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'PLU', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'PLU', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('1115900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 6, 42, 49), 'check_last_dttm': datetime.datetime(2024, 11, 27, 10, 28, 10), 'add_dttm': datetime.datetime(2024, 11, 27, 6, 42, 49), 'upd_dttm': datetime.datetime(2024, 11, 27, 6, 42, 49), 'del_dttm': datetime.datetime(2024, 11, 27, 6, 42, 49), 'dlm': datetime.datetime(2024, 11, 27, 10, 29, 38)},
		{'pos_id': 29997, 'symb': 'USDC', 'prod_id': 'LCX-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 8, 28, 51), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 15, 6, 58), 'age_mins': 98, 'tot_out_cnt': Decimal('13.000000000000'), 'tot_in_cnt': Decimal('13.214730375000'), 'buy_fees_cnt': Decimal('0.032418952618'), 'sell_fees_cnt_tot': Decimal('0.033119625000'), 'fees_cnt_tot': Decimal('0.065538577618'), 'buy_cnt': Decimal('46.578954911572'), 'sell_cnt_tot': Decimal('46.500000000000'), 'hold_cnt': Decimal('0.078954911572'), 'pocket_cnt': Decimal('0.078954911572'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.279096000000'), 'prc_curr': Decimal('0.284187750000'), 'prc_high': Decimal('0.318000000000'), 'prc_low': Decimal('0.272200000000'), 'prc_chg_pct': Decimal('1.824400000000'), 'prc_chg_pct_high': Decimal('13.366000000000'), 'prc_chg_pct_low': Decimal('-2.470800000000'), 'prc_chg_pct_drop': Decimal('-11.540000000000'), 'prc_sell_avg': Decimal('0.284187750000'), 'val_curr': Decimal('0.022438018671'), 'val_tot': Decimal('0.022438018671'), 'gain_loss_amt_est': Decimal('0.321581104710'), 'gain_loss_amt_est_high': Decimal('1.737581334021'), 'gain_loss_amt_est_low': Decimal('-0.321208473070'), 'gain_loss_amt': Decimal('0.237168393671'), 'gain_loss_amt_net': Decimal('0.259606412342'), 'gain_loss_pct_est': Decimal('2.473700000000'), 'gain_loss_pct_est_high': Decimal('13.366000000000'), 'gain_loss_pct_est_low': Decimal('-2.470800000000'), 'gain_loss_pct': Decimal('1.824400000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29651, 'bo_uuid': '5cc571a9-574d-4db6-8aca-32f0c513f569', 'buy_curr_symb': 'LCX', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'LCX', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'LCX', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('117651200.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 8, 29, 18), 'check_last_dttm': datetime.datetime(2024, 11, 27, 10, 6, 51), 'add_dttm': datetime.datetime(2024, 11, 27, 8, 29, 18), 'upd_dttm': datetime.datetime(2024, 11, 27, 8, 29, 18), 'del_dttm': datetime.datetime(2024, 11, 27, 8, 29, 18), 'dlm': datetime.datetime(2024, 11, 27, 10, 8, 19)},
		{'pos_id': 29858, 'symb': 'USDC', 'prod_id': 'WELL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 5, 4), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 13, 42, 10), 'age_mins': 218, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('2.805743062500'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007031937500'), 'fees_cnt_tot': Decimal('0.014513234258'), 'buy_cnt': Decimal('35.406880229559'), 'sell_cnt_tot': Decimal('35.000000000000'), 'hold_cnt': Decimal('0.406880229559'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.406880229559'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.084729290000'), 'prc_curr': Decimal('0.080164090000'), 'prc_high': Decimal('0.085000000000'), 'prc_low': Decimal('0.080164090000'), 'prc_chg_pct': Decimal('-5.388000000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-5.388000000000'), 'prc_chg_pct_drop': Decimal('-5.390000000000'), 'prc_sell_avg': Decimal('0.080164090000'), 'val_curr': Decimal('0.032617183342'), 'val_tot': Decimal('0.032617183342'), 'gain_loss_amt_est': Decimal('-0.154596884112'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-0.154596884112'), 'gain_loss_amt': Decimal('-0.161639754158'), 'gain_loss_amt_net': Decimal('-0.129022570817'), 'gain_loss_pct_est': Decimal('-5.153200000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-5.153200000000'), 'gain_loss_pct': Decimal('-5.388000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '1h', 'sell_strat_type': 'strat', 'sell_strat_name': 'sha', 'sell_strat_freq': '1h', 'bo_id': 29512, 'bo_uuid': '3531b41d-7881-4e94-9326-cbb7ecad6a5f', 'buy_curr_symb': 'WELL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'WELL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'WELL', 'base_size_incr': Decimal('1.000000000000'), 'base_size_min': Decimal('1.000000000000'), 'base_size_max': Decimal('289584259.672040800000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000001000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 5, 5, 1), 'check_last_dttm': datetime.datetime(2024, 11, 27, 8, 42, 3), 'add_dttm': datetime.datetime(2024, 11, 27, 5, 5, 1), 'upd_dttm': datetime.datetime(2024, 11, 27, 5, 5, 1), 'del_dttm': datetime.datetime(2024, 11, 27, 5, 5, 1), 'dlm': datetime.datetime(2024, 11, 27, 8, 42, 57)},
		{'pos_id': 29916, 'symb': 'USDC', 'prod_id': 'LCX-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 6, 36, 44), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 13, 10, 45), 'age_mins': 94, 'tot_out_cnt': Decimal('17.000000000000'), 'tot_in_cnt': Decimal('17.401886250000'), 'buy_fees_cnt': Decimal('0.042394014963'), 'sell_fees_cnt_tot': Decimal('0.043613750000'), 'fees_cnt_tot': Decimal('0.086007764963'), 'buy_cnt': Decimal('64.946786614467'), 'sell_cnt_tot': Decimal('64.900000000000'), 'hold_cnt': Decimal('0.046786614467'), 'pocket_cnt': Decimal('0.046786614467'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.261752750000'), 'prc_curr': Decimal('0.268133840000'), 'prc_high': Decimal('0.278500000000'), 'prc_low': Decimal('0.252700000000'), 'prc_chg_pct': Decimal('2.437800000000'), 'prc_chg_pct_high': Decimal('5.901500000000'), 'prc_chg_pct_low': Decimal('-3.458500000000'), 'prc_chg_pct_drop': Decimal('-3.460000000000'), 'prc_sell_avg': Decimal('0.268133840000'), 'val_curr': Decimal('0.012545074598'), 'val_tot': Decimal('0.012545074598'), 'gain_loss_amt_est': Decimal('0.457696241969'), 'gain_loss_amt_est_high': Decimal('1.003249249530'), 'gain_loss_amt_est_low': Decimal('-0.587947022524'), 'gain_loss_amt': Decimal('0.414431324598'), 'gain_loss_amt_net': Decimal('0.426976399195'), 'gain_loss_pct_est': Decimal('2.692300000000'), 'gain_loss_pct_est_high': Decimal('5.901500000000'), 'gain_loss_pct_est_low': Decimal('-3.458500000000'), 'gain_loss_pct': Decimal('2.437800000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29570, 'bo_uuid': 'cda529fa-ae9b-411b-bb41-f7f68b199c64', 'buy_curr_symb': 'LCX', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'LCX', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'LCX', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('117651200.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 6, 37, 23), 'check_last_dttm': datetime.datetime(2024, 11, 27, 8, 10, 39), 'add_dttm': datetime.datetime(2024, 11, 27, 6, 37, 23), 'upd_dttm': datetime.datetime(2024, 11, 27, 6, 37, 23), 'del_dttm': datetime.datetime(2024, 11, 27, 6, 37, 23), 'dlm': datetime.datetime(2024, 11, 27, 8, 11, 12)},
		{'pos_id': 29824, 'symb': 'USDC', 'prod_id': 'DOGE-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 3, 47, 28), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 12, 8, 3), 'age_mins': 201, 'tot_out_cnt': Decimal('10.000000000000'), 'tot_in_cnt': Decimal('9.720125782500'), 'buy_fees_cnt': Decimal('0.024937655860'), 'sell_fees_cnt_tot': Decimal('0.024361217500'), 'fees_cnt_tot': Decimal('0.049298873360'), 'buy_cnt': Decimal('25.173023631302'), 'sell_cnt_tot': Decimal('25.100000000000'), 'hold_cnt': Decimal('0.073023631302'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.073023631302'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.397250650000'), 'prc_curr': Decimal('0.387256010000'), 'prc_high': Decimal('0.397520000000'), 'prc_low': Decimal('0.387256010000'), 'prc_chg_pct': Decimal('-2.516000000000'), 'prc_chg_pct_high': Decimal('0E-12'), 'prc_chg_pct_low': Decimal('-2.516000000000'), 'prc_chg_pct_drop': Decimal('-2.520000000000'), 'prc_sell_avg': Decimal('0.387256010000'), 'val_curr': Decimal('0.028278840094'), 'val_tot': Decimal('0.028278840094'), 'gain_loss_amt_est': Decimal('-0.234377212473'), 'gain_loss_amt_est_high': Decimal('0E-12'), 'gain_loss_amt_est_low': Decimal('-0.234377212473'), 'gain_loss_amt': Decimal('-0.251595377406'), 'gain_loss_amt_net': Decimal('-0.223316537313'), 'gain_loss_pct_est': Decimal('-2.343800000000'), 'gain_loss_pct_est_high': Decimal('0E-12'), 'gain_loss_pct_est_low': Decimal('-2.343800000000'), 'gain_loss_pct': Decimal('-2.516000000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'sha', 'buy_strat_freq': '1h', 'sell_strat_type': 'strat', 'sell_strat_name': 'sha', 'sell_strat_freq': '1h', 'bo_id': 29478, 'bo_uuid': '7d1e602c-a789-43dc-bc4e-ab9add7ba588', 'buy_curr_symb': 'DOGE', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOGE', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOGE', 'base_size_incr': Decimal('0.100000000000'), 'base_size_min': Decimal('0.100000000000'), 'base_size_max': Decimal('141834300.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000010000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 3, 48, 39), 'check_last_dttm': datetime.datetime(2024, 11, 27, 7, 7, 56), 'add_dttm': datetime.datetime(2024, 11, 27, 3, 48, 39), 'upd_dttm': datetime.datetime(2024, 11, 27, 3, 48, 39), 'del_dttm': datetime.datetime(2024, 11, 27, 3, 48, 39), 'dlm': datetime.datetime(2024, 11, 27, 7, 8, 18)},
		{'pos_id': 29862, 'symb': 'USDC', 'prod_id': '00-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 5, 10), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 11, 48, 31), 'age_mins': 98, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('2.741973885000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.006872115000'), 'fees_cnt_tot': Decimal('0.014353411758'), 'buy_cnt': Decimal('38.024379964954'), 'sell_cnt_tot': Decimal('38.020000000000'), 'hold_cnt': Decimal('0.004379964954'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.004379964954'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.078896750000'), 'prc_curr': Decimal('0.072119250000'), 'prc_high': Decimal('0.079500000000'), 'prc_low': Decimal('0.072119250000'), 'prc_chg_pct': Decimal('-8.590300000000'), 'prc_chg_pct_high': Decimal('0.257600000000'), 'prc_chg_pct_low': Decimal('-8.590300000000'), 'prc_chg_pct_drop': Decimal('-8.850000000000'), 'prc_sell_avg': Decimal('0.072119250000'), 'val_curr': Decimal('0.000315879788'), 'val_tot': Decimal('0.000315879788'), 'gain_loss_amt_est': Decimal('-0.209010510572'), 'gain_loss_amt_est_high': Decimal('0.007728455228'), 'gain_loss_amt_est_low': Decimal('-0.209010510572'), 'gain_loss_amt': Decimal('-0.257710235212'), 'gain_loss_amt_net': Decimal('-0.257394355425'), 'gain_loss_pct_est': Decimal('-6.967000000000'), 'gain_loss_pct_est_high': Decimal('0.257600000000'), 'gain_loss_pct_est_low': Decimal('-6.967000000000'), 'gain_loss_pct': Decimal('-8.590300000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '15min', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'trail_stop', 'sell_strat_freq': '', 'bo_id': 29516, 'bo_uuid': '0b23a441-1b8a-4f37-981b-4e3c80ec418a', 'buy_curr_symb': '00', 'spend_curr_symb': 'USDC', 'sell_curr_symb': '00', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': '00', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('145572316.105043830000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 5, 11, 8), 'check_last_dttm': datetime.datetime(2024, 11, 27, 6, 48, 24), 'add_dttm': datetime.datetime(2024, 11, 27, 5, 11, 8), 'upd_dttm': datetime.datetime(2024, 11, 27, 5, 11, 8), 'del_dttm': datetime.datetime(2024, 11, 27, 5, 11, 8), 'dlm': datetime.datetime(2024, 11, 27, 6, 49, 55)},
		{'pos_id': 29849, 'symb': 'USDC', 'prod_id': '00-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 27, 5, 2, 50), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 11, 48, 17), 'age_mins': 105, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('5.576260410000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.013975590000'), 'fees_cnt_tot': Decimal('0.028938183516'), 'buy_cnt': Decimal('77.326064683253'), 'sell_cnt_tot': Decimal('77.320000000000'), 'hold_cnt': Decimal('0.006064683253'), 'pocket_cnt': Decimal('0E-12'), 'clip_cnt': Decimal('0.006064683253'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.077593500000'), 'prc_curr': Decimal('0.072119250000'), 'prc_high': Decimal('0.079500000000'), 'prc_low': Decimal('0.072119250000'), 'prc_chg_pct': Decimal('-7.055000000000'), 'prc_chg_pct_high': Decimal('1.941500000000'), 'prc_chg_pct_low': Decimal('-7.055000000000'), 'prc_chg_pct_drop': Decimal('-9.000000000000'), 'prc_sell_avg': Decimal('0.072119250000'), 'val_curr': Decimal('0.000437380408'), 'val_tot': Decimal('0.000437380408'), 'gain_loss_amt_est': Decimal('-0.324266852249'), 'gain_loss_amt_est_high': Decimal('0.116491716445'), 'gain_loss_amt_est_low': Decimal('-0.324266852249'), 'gain_loss_amt': Decimal('-0.423302209592'), 'gain_loss_amt_net': Decimal('-0.422864829185'), 'gain_loss_pct_est': Decimal('-5.404400000000'), 'gain_loss_pct_est_high': Decimal('1.941500000000'), 'gain_loss_pct_est_low': Decimal('-5.404400000000'), 'gain_loss_pct': Decimal('-7.055000000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '4h', 'sell_strat_type': 'stop_loss', 'sell_strat_name': 'trail_stop', 'sell_strat_freq': '', 'bo_id': 29503, 'bo_uuid': 'bcf0f205-71b2-4ec7-a865-bd88c1e033f8', 'buy_curr_symb': '00', 'spend_curr_symb': 'USDC', 'sell_curr_symb': '00', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': '00', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('145572316.105043830000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 27, 5, 3, 27), 'check_last_dttm': datetime.datetime(2024, 11, 27, 6, 48, 11), 'add_dttm': datetime.datetime(2024, 11, 27, 5, 3, 27), 'upd_dttm': datetime.datetime(2024, 11, 27, 5, 3, 27), 'del_dttm': datetime.datetime(2024, 11, 27, 5, 3, 27), 'dlm': datetime.datetime(2024, 11, 27, 6, 49, 42)},
		{'pos_id': 29224, 'symb': 'USDC', 'prod_id': 'RENDER-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 26, 8, 24, 12), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 11, 24, 13), 'age_mins': 1320, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.079172775000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007717225000'), 'fees_cnt_tot': Decimal('0.015198521758'), 'buy_cnt': Decimal('0.412192658849'), 'sell_cnt_tot': Decimal('0.410000000000'), 'hold_cnt': Decimal('0.002192658849'), 'pocket_cnt': Decimal('0.002192658849'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('7.278150000000'), 'prc_curr': Decimal('7.510177500000'), 'prc_high': Decimal('7.659000000000'), 'prc_low': Decimal('7.147000000000'), 'prc_chg_pct': Decimal('3.188000000000'), 'prc_chg_pct_high': Decimal('4.806900000000'), 'prc_chg_pct_low': Decimal('-1.802000000000'), 'prc_chg_pct_drop': Decimal('-1.620000000000'), 'prc_sell_avg': Decimal('7.510177500000'), 'val_curr': Decimal('0.016467257153'), 'val_tot': Decimal('0.016467257153'), 'gain_loss_amt_est': Decimal('0.104635106451'), 'gain_loss_amt_est_high': Decimal('0.144205601700'), 'gain_loss_amt_est_low': Decimal('-0.054059067206'), 'gain_loss_amt': Decimal('0.095640032153'), 'gain_loss_amt_net': Decimal('0.112107289306'), 'gain_loss_pct_est': Decimal('3.487800000000'), 'gain_loss_pct_est_high': Decimal('4.806900000000'), 'gain_loss_pct_est_low': Decimal('-1.802000000000'), 'gain_loss_pct': Decimal('3.188000000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28877, 'bo_uuid': '61dfa3ae-401c-41c2-9225-6df6d0a5f06d', 'buy_curr_symb': 'RENDER', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'RENDER', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'RENDER', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('1982431.303836978700'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 26, 8, 32, 41), 'check_last_dttm': datetime.datetime(2024, 11, 27, 6, 24, 7), 'add_dttm': datetime.datetime(2024, 11, 26, 8, 32, 41), 'upd_dttm': datetime.datetime(2024, 11, 26, 8, 32, 41), 'del_dttm': datetime.datetime(2024, 11, 26, 8, 32, 41), 'dlm': datetime.datetime(2024, 11, 27, 6, 25, 5)},
		{'pos_id': 29207, 'symb': 'USDC', 'prod_id': 'RENDER-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 26, 8, 12, 33), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 11, 24, 1), 'age_mins': 1332, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.154255800000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015424200000'), 'fees_cnt_tot': Decimal('0.030386793516'), 'buy_cnt': Decimal('0.829412057440'), 'sell_cnt_tot': Decimal('0.820000000000'), 'hold_cnt': Decimal('0.009412057440'), 'pocket_cnt': Decimal('0.009412057440'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('7.234040000000'), 'prc_curr': Decimal('7.505190000000'), 'prc_high': Decimal('7.659000000000'), 'prc_low': Decimal('7.147000000000'), 'prc_chg_pct': Decimal('3.748300000000'), 'prc_chg_pct_high': Decimal('5.445900000000'), 'prc_chg_pct_low': Decimal('-1.203200000000'), 'prc_chg_pct_drop': Decimal('-1.700000000000'), 'prc_sell_avg': Decimal('7.505190000000'), 'val_curr': Decimal('0.070639279378'), 'val_tot': Decimal('0.070639279378'), 'gain_loss_amt_est': Decimal('0.247131616638'), 'gain_loss_amt_est_high': Decimal('0.326755174152'), 'gain_loss_amt_est_low': Decimal('-0.072192025476'), 'gain_loss_amt': Decimal('0.224895079378'), 'gain_loss_amt_net': Decimal('0.295534358756'), 'gain_loss_pct_est': Decimal('4.118900000000'), 'gain_loss_pct_est_high': Decimal('5.445900000000'), 'gain_loss_pct_est_low': Decimal('-1.203200000000'), 'gain_loss_pct': Decimal('3.748300000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '30min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28860, 'bo_uuid': '7f7d6bd9-a044-4672-b824-f834015f5c06', 'buy_curr_symb': 'RENDER', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'RENDER', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'RENDER', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('1982431.303836978700'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 26, 8, 31, 50), 'check_last_dttm': datetime.datetime(2024, 11, 27, 6, 23, 55), 'add_dttm': datetime.datetime(2024, 11, 26, 8, 31, 50), 'upd_dttm': datetime.datetime(2024, 11, 26, 8, 31, 50), 'del_dttm': datetime.datetime(2024, 11, 26, 8, 31, 50), 'dlm': datetime.datetime(2024, 11, 27, 6, 24, 56)},
		{'pos_id': 29113, 'symb': 'USDC', 'prod_id': 'RNDR-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 26, 6, 56, 35), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 11, 23, 51), 'age_mins': 1408, 'tot_out_cnt': Decimal('2.000000000000'), 'tot_in_cnt': Decimal('2.026670625000'), 'buy_fees_cnt': Decimal('0.004987531172'), 'sell_fees_cnt_tot': Decimal('0.005079375000'), 'fees_cnt_tot': Decimal('0.010066906172'), 'buy_cnt': Decimal('0.274867040800'), 'sell_cnt_tot': Decimal('0.270000000000'), 'hold_cnt': Decimal('0.004867040800'), 'pocket_cnt': Decimal('0.004867040800'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('7.276245250000'), 'prc_curr': Decimal('7.506187500000'), 'prc_high': Decimal('7.820000000000'), 'prc_low': Decimal('7.117800000000'), 'prc_chg_pct': Decimal('3.160200000000'), 'prc_chg_pct_high': Decimal('4.768300000000'), 'prc_chg_pct_low': Decimal('-2.177600000000'), 'prc_chg_pct_drop': Decimal('-1.610000000000'), 'prc_sell_avg': Decimal('7.506187500000'), 'val_curr': Decimal('0.036532920815'), 'val_tot': Decimal('0.036532920815'), 'gain_loss_amt_est': Decimal('0.069144109734'), 'gain_loss_amt_est_high': Decimal('0.095366425427'), 'gain_loss_amt_est_low': Decimal('-0.043551376994'), 'gain_loss_amt': Decimal('0.063203545815'), 'gain_loss_amt_net': Decimal('0.099736466630'), 'gain_loss_pct_est': Decimal('3.457200000000'), 'gain_loss_pct_est_high': Decimal('4.768300000000'), 'gain_loss_pct_est_low': Decimal('-2.177600000000'), 'gain_loss_pct': Decimal('3.160200000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28766, 'bo_uuid': 'c415d390-ff82-4d67-8da7-a57a7c44ff0e', 'buy_curr_symb': 'RNDR', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'RNDR', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'RNDR', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('8235000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 26, 6, 59, 15), 'check_last_dttm': datetime.datetime(2024, 11, 27, 6, 23, 45), 'add_dttm': datetime.datetime(2024, 11, 26, 6, 59, 15), 'upd_dttm': datetime.datetime(2024, 11, 26, 6, 59, 15), 'del_dttm': datetime.datetime(2024, 11, 26, 6, 59, 15), 'dlm': datetime.datetime(2024, 11, 27, 6, 24, 51)},
		{'pos_id': 29216, 'symb': 'USDC', 'prod_id': 'RENDER-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 26, 8, 18, 21), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 11, 11, 45), 'age_mins': 1314, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.095122800000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007757200000'), 'fees_cnt_tot': Decimal('0.015238496758'), 'buy_cnt': Decimal('0.413331312602'), 'sell_cnt_tot': Decimal('0.410000000000'), 'hold_cnt': Decimal('0.003331312602'), 'pocket_cnt': Decimal('0.003331312602'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('7.258100000000'), 'prc_curr': Decimal('7.549080000000'), 'prc_high': Decimal('7.659000000000'), 'prc_low': Decimal('7.147000000000'), 'prc_chg_pct': Decimal('4.009000000000'), 'prc_chg_pct_high': Decimal('5.096400000000'), 'prc_chg_pct_low': Decimal('-1.530700000000'), 'prc_chg_pct_drop': Decimal('-1.090000000000'), 'prc_sell_avg': Decimal('7.549080000000'), 'val_curr': Decimal('0.025148345338'), 'val_tot': Decimal('0.025148345338'), 'gain_loss_amt_est': Decimal('0.126851379834'), 'gain_loss_amt_est_high': Decimal('0.152891252528'), 'gain_loss_amt_est_low': Decimal('-0.045921108834'), 'gain_loss_amt': Decimal('0.120271145338'), 'gain_loss_amt_net': Decimal('0.145419490675'), 'gain_loss_pct_est': Decimal('4.228400000000'), 'gain_loss_pct_est_high': Decimal('5.096400000000'), 'gain_loss_pct_est_low': Decimal('-1.530700000000'), 'gain_loss_pct': Decimal('4.009000000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28869, 'bo_uuid': 'd66a393b-54d3-4a0b-babc-b54bdcf20a51', 'buy_curr_symb': 'RENDER', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'RENDER', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'RENDER', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('1982431.303836978700'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 26, 8, 32, 16), 'check_last_dttm': datetime.datetime(2024, 11, 27, 6, 11, 39), 'add_dttm': datetime.datetime(2024, 11, 26, 8, 32, 16), 'upd_dttm': datetime.datetime(2024, 11, 26, 8, 32, 16), 'del_dttm': datetime.datetime(2024, 11, 26, 8, 32, 16), 'dlm': datetime.datetime(2024, 11, 27, 6, 12, 16)},
		{'pos_id': 29003, 'symb': 'USDC', 'prod_id': 'DOT-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 26, 4, 6, 45), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 11, 9, 37), 'age_mins': 1563, 'tot_out_cnt': Decimal('5.000000000000'), 'tot_in_cnt': Decimal('5.140092614689'), 'buy_fees_cnt': Decimal('0.012468827930'), 'sell_fees_cnt_tot': Decimal('0.012882437631'), 'fees_cnt_tot': Decimal('0.025351265561'), 'buy_cnt': Decimal('0.622429947843'), 'sell_cnt_tot': Decimal('0.622039480000'), 'hold_cnt': Decimal('0.000390467843'), 'pocket_cnt': Decimal('0.000390467843'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('8.033032500000'), 'prc_curr': Decimal('8.263290000000'), 'prc_high': Decimal('8.446000000000'), 'prc_low': Decimal('7.575000000000'), 'prc_chg_pct': Decimal('2.866400000000'), 'prc_chg_pct_high': Decimal('4.817200000000'), 'prc_chg_pct_low': Decimal('-5.701900000000'), 'prc_chg_pct_drop': Decimal('-1.950000000000'), 'prc_sell_avg': Decimal('8.263290000000'), 'val_curr': Decimal('0.003226549022'), 'val_tot': Decimal('0.003226549022'), 'gain_loss_amt_est': Decimal('0.156832117879'), 'gain_loss_amt_est_high': Decimal('0.240860160838'), 'gain_loss_amt_est_low': Decimal('-0.285093145089'), 'gain_loss_amt': Decimal('0.143319163711'), 'gain_loss_amt_net': Decimal('0.146545712734'), 'gain_loss_pct_est': Decimal('3.136600000000'), 'gain_loss_pct_est_high': Decimal('4.817200000000'), 'gain_loss_pct_est_low': Decimal('-5.701900000000'), 'gain_loss_pct': Decimal('2.866400000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28656, 'bo_uuid': '5ef4a992-2c3c-4d0a-8cc3-3cbae219f2cb', 'buy_curr_symb': 'DOT', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'DOT', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'DOT', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('1732900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.001000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 26, 4, 7, 45), 'check_last_dttm': datetime.datetime(2024, 11, 27, 6, 9, 31), 'add_dttm': datetime.datetime(2024, 11, 26, 4, 7, 45), 'upd_dttm': datetime.datetime(2024, 11, 26, 4, 7, 45), 'del_dttm': datetime.datetime(2024, 11, 26, 4, 7, 45), 'dlm': datetime.datetime(2024, 11, 27, 6, 10, 4)},
		{'pos_id': 28621, 'symb': 'USDC', 'prod_id': 'ADA-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 25, 17, 59, 17), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 10, 57, 39), 'age_mins': 2159, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.273369747678'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015722731197'), 'fees_cnt_tot': Decimal('0.030685324713'), 'buy_cnt': Decimal('6.242216736007'), 'sell_cnt_tot': Decimal('6.236085750000'), 'hold_cnt': Decimal('0.006130986007'), 'pocket_cnt': Decimal('0.006130986007'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.961197000000'), 'prc_curr': Decimal('1.005978750000'), 'prc_high': Decimal('1.022000000000'), 'prc_low': Decimal('0.879800000000'), 'prc_chg_pct': Decimal('4.659000000000'), 'prc_chg_pct_high': Decimal('6.128100000000'), 'prc_chg_pct_low': Decimal('-8.468300000000'), 'prc_chg_pct_drop': Decimal('-1.470000000000'), 'prc_sell_avg': Decimal('1.005978750000'), 'val_curr': Decimal('0.006167641640'), 'val_tot': Decimal('0.006167641640'), 'gain_loss_amt_est': Decimal('0.294651356589'), 'gain_loss_amt_est_high': Decimal('0.367685292401'), 'gain_loss_amt_est_low': Decimal('-0.508097715661'), 'gain_loss_amt': Decimal('0.279537389318'), 'gain_loss_amt_net': Decimal('0.285705030957'), 'gain_loss_pct_est': Decimal('4.910900000000'), 'gain_loss_pct_est_high': Decimal('6.128100000000'), 'gain_loss_pct_est_low': Decimal('-8.468300000000'), 'gain_loss_pct': Decimal('4.659000000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '4h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28274, 'bo_uuid': 'b81571d8-9497-4ac4-b736-4fe390f3c5fc', 'buy_curr_symb': 'ADA', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'ADA', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'ADA', 'base_size_incr': Decimal('1.0000E-8'), 'base_size_min': Decimal('1.0000E-8'), 'base_size_max': Decimal('30176000.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 25, 18, 0, 31), 'check_last_dttm': datetime.datetime(2024, 11, 27, 5, 57, 32), 'add_dttm': datetime.datetime(2024, 11, 25, 18, 0, 31), 'upd_dttm': datetime.datetime(2024, 11, 25, 18, 0, 31), 'del_dttm': datetime.datetime(2024, 11, 25, 18, 0, 31), 'dlm': datetime.datetime(2024, 11, 27, 5, 57, 58)},
		{'pos_id': 29411, 'symb': 'USDC', 'prod_id': 'TRB-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 26, 12, 35, 55), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 9, 17, 25), 'age_mins': 941, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.073177800000'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007702200000'), 'fees_cnt_tot': Decimal('0.015183496758'), 'buy_cnt': Decimal('0.044143954908'), 'sell_cnt_tot': Decimal('0.044000000000'), 'hold_cnt': Decimal('0.000143954908'), 'pocket_cnt': Decimal('0.000143954908'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('67.959475000000'), 'prc_curr': Decimal('69.844950000000'), 'prc_high': Decimal('70.980000000000'), 'prc_low': Decimal('65.500000000000'), 'prc_chg_pct': Decimal('2.774400000000'), 'prc_chg_pct_high': Decimal('4.003200000000'), 'prc_chg_pct_low': Decimal('-3.619000000000'), 'prc_chg_pct_drop': Decimal('-1.230000000000'), 'prc_sell_avg': Decimal('69.844950000000'), 'val_curr': Decimal('0.010054523352'), 'val_tot': Decimal('0.010054523352'), 'gain_loss_amt_est': Decimal('0.089193964462'), 'gain_loss_amt_est_high': Decimal('0.120094732897'), 'gain_loss_amt_est_low': Decimal('-0.108570953526'), 'gain_loss_amt': Decimal('0.083232323352'), 'gain_loss_amt_net': Decimal('0.093286846703'), 'gain_loss_pct_est': Decimal('2.973100000000'), 'gain_loss_pct_est_high': Decimal('4.003200000000'), 'gain_loss_pct_est_low': Decimal('-3.619000000000'), 'gain_loss_pct': Decimal('2.774400000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29065, 'bo_uuid': '79d0d98f-12e3-4426-a75d-6496b1e654d4', 'buy_curr_symb': 'TRB', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'TRB', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'TRB', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('719900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 26, 12, 36, 43), 'check_last_dttm': datetime.datetime(2024, 11, 27, 4, 17, 19), 'add_dttm': datetime.datetime(2024, 11, 26, 12, 36, 43), 'upd_dttm': datetime.datetime(2024, 11, 26, 12, 36, 43), 'del_dttm': datetime.datetime(2024, 11, 26, 12, 36, 43), 'dlm': datetime.datetime(2024, 11, 27, 4, 18, 3)},
		{'pos_id': 29400, 'symb': 'USDC', 'prod_id': 'TRB-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 26, 12, 29, 31), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 9, 17, 12), 'age_mins': 948, 'tot_out_cnt': Decimal('6.000000000000'), 'tot_in_cnt': Decimal('6.144600000000'), 'buy_fees_cnt': Decimal('0.014962593516'), 'sell_fees_cnt_tot': Decimal('0.015400000000'), 'fees_cnt_tot': Decimal('0.030362593516'), 'buy_cnt': Decimal('0.088366121460'), 'sell_cnt_tot': Decimal('0.088000000000'), 'hold_cnt': Decimal('0.000366121460'), 'pocket_cnt': Decimal('0.000366121460'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('67.899325000000'), 'prc_curr': Decimal('69.825000000000'), 'prc_high': Decimal('70.980000000000'), 'prc_low': Decimal('65.500000000000'), 'prc_chg_pct': Decimal('2.836100000000'), 'prc_chg_pct_high': Decimal('4.095300000000'), 'prc_chg_pct_low': Decimal('-3.533700000000'), 'prc_chg_pct_drop': Decimal('-1.260000000000'), 'prc_sell_avg': Decimal('69.825000000000'), 'val_curr': Decimal('0.025564430945'), 'val_tot': Decimal('0.025564430945'), 'gain_loss_amt_est': Decimal('0.183861179771'), 'gain_loss_amt_est_high': Decimal('0.245717464793'), 'gain_loss_amt_est_low': Decimal('-0.212019044370'), 'gain_loss_amt': Decimal('0.170164430945'), 'gain_loss_amt_net': Decimal('0.195728861889'), 'gain_loss_pct_est': Decimal('3.064400000000'), 'gain_loss_pct_est_high': Decimal('4.095300000000'), 'gain_loss_pct_est_low': Decimal('-3.533700000000'), 'gain_loss_pct': Decimal('2.836100000000'), 'buy_strat_type': 'up', 'buy_strat_name': 'nwe_3row', 'buy_strat_freq': '15min', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 29054, 'bo_uuid': 'fbf2fe66-f2b4-4cca-b623-ae69885dbbd6', 'buy_curr_symb': 'TRB', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'TRB', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'TRB', 'base_size_incr': Decimal('0.001000000000'), 'base_size_min': Decimal('0.001000000000'), 'base_size_max': Decimal('719900.000000000000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.010000000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 26, 12, 29, 53), 'check_last_dttm': datetime.datetime(2024, 11, 27, 4, 17, 6), 'add_dttm': datetime.datetime(2024, 11, 26, 12, 29, 53), 'upd_dttm': datetime.datetime(2024, 11, 26, 12, 29, 53), 'del_dttm': datetime.datetime(2024, 11, 26, 12, 29, 53), 'dlm': datetime.datetime(2024, 11, 27, 4, 17, 58)},
		{'pos_id': 29112, 'symb': 'USDC', 'prod_id': 'POL-USDC', 'pos_stat': 'CLOSE', 'pos_begin_dttm': datetime.datetime(2024, 11, 26, 6, 55, 39), 'pos_end_dttm': datetime.datetime(2024, 11, 27, 9, 3, 55), 'age_mins': 1268, 'tot_out_cnt': Decimal('3.000000000000'), 'tot_in_cnt': Decimal('3.173022562500'), 'buy_fees_cnt': Decimal('0.007481296758'), 'sell_fees_cnt_tot': Decimal('0.007952437500'), 'fees_cnt_tot': Decimal('0.015433734258'), 'buy_cnt': Decimal('5.826555107558'), 'sell_cnt_tot': Decimal('5.810000000000'), 'hold_cnt': Decimal('0.016555107558'), 'pocket_cnt': Decimal('0.016555107558'), 'clip_cnt': Decimal('0E-12'), 'pocket_pct': Decimal('2.000000000000'), 'clip_pct': Decimal('0E-12'), 'sell_order_cnt': 1, 'sell_order_attempt_cnt': 1, 'prc_buy': Decimal('0.514884000000'), 'prc_curr': Decimal('0.546131250000'), 'prc_high': Decimal('0.559900000000'), 'prc_low': Decimal('0.509000000000'), 'prc_chg_pct': Decimal('6.068800000000'), 'prc_chg_pct_high': Decimal('8.276800000000'), 'prc_chg_pct_low': Decimal('-1.142800000000'), 'prc_chg_pct_drop': Decimal('-2.210000000000'), 'prc_sell_avg': Decimal('0.546131250000'), 'val_curr': Decimal('0.009041261585'), 'val_tot': Decimal('0.009041261585'), 'gain_loss_amt_est': Decimal('0.197030787517'), 'gain_loss_amt_est_high': Decimal('0.248304472464'), 'gain_loss_amt_est_low': Decimal('-0.034283450253'), 'gain_loss_amt': Decimal('0.182063824085'), 'gain_loss_amt_net': Decimal('0.191105085669'), 'gain_loss_pct_est': Decimal('6.567700000000'), 'gain_loss_pct_est_high': Decimal('8.276800000000'), 'gain_loss_pct_est_low': Decimal('-1.142800000000'), 'gain_loss_pct': Decimal('6.068800000000'), 'buy_strat_type': 'dn', 'buy_strat_name': 'drop', 'buy_strat_freq': '1h', 'sell_strat_type': 'profit', 'sell_strat_name': 'trail_profit', 'sell_strat_freq': '', 'bo_id': 28765, 'bo_uuid': '2a0f5898-c283-4891-8f71-116dbb3d2072', 'buy_curr_symb': 'POL', 'spend_curr_symb': 'USDC', 'sell_curr_symb': 'POL', 'recv_curr_symb': 'USDC', 'fees_curr_symb': 'USDC', 'base_curr_symb': 'POL', 'base_size_incr': Decimal('0.010000000000'), 'base_size_min': Decimal('0.010000000000'), 'base_size_max': Decimal('26020434.110127510000'), 'quote_curr_symb': 'USDC', 'quote_size_incr': Decimal('0.000100000000'), 'quote_size_min': Decimal('1.000000000000'), 'quote_size_max': Decimal('10000000.000000000000'), 'sell_yn': 'Y', 'hodl_yn': 'N', 'sell_block_yn': 'N', 'sell_force_yn': 'N', 'test_tf': 0, 'test_txn_yn': 'N', 'force_sell_tf': 0, 'ignore_tf': 0, 'error_tf': 0, 'reason': '', 'note1': None, 'note2': None, 'note3': None, 'mkt_name': None, 'mkt_venue': None, 'pos_type': 'SPOT', 'buy_asset_type': None, 'check_mkt_dttm': datetime.datetime(2024, 11, 26, 6, 56, 48), 'check_last_dttm': datetime.datetime(2024, 11, 27, 4, 3, 49), 'add_dttm': datetime.datetime(2024, 11, 26, 6, 56, 48), 'upd_dttm': datetime.datetime(2024, 11, 26, 6, 56, 48), 'del_dttm': datetime.datetime(2024, 11, 26, 6, 56, 48), 'dlm': datetime.datetime(2024, 11, 27, 4, 4, 59)}
	]

	hmsg = ""
	hmsg += f"{'mkt':^12} | "
	hmsg += f"{'T':^1} | "
	hmsg += f"{'pos_id':^6} | "
	hmsg += f"{'strat':^10} | "
	hmsg += f"{'freq':^5} | "
	hmsg += f"{'sell':^14} | "
	hmsg += f"{'age':^10} | "
	hmsg += f"{'sold @ est':^16} | "
	hmsg += f"{'buy_val':^16} | "
	hmsg += f"{'buy_prc':^14} | "
	hmsg += f"{'curr_prc':^14} | "
	hmsg += f"{'gain_pct':^8} % | "
	hmsg += f"{'gain_est':^8} % | "
	hmsg += f"{'gain_top':^8} % | "
	hmsg += f"{'drop_pct':^8} % | "
	hmsg += f"{'gain_loss':^14} | "
	hmsg += f"{'gain_loss_est':^14} | "
	hmsg += f"{'gain_loss_high':^14}"

	title = 'Last 25 Closed Positions - Live'
	headers = hmsg
	len_cnt = len(hmsg)

	chart_top(in_str=title, len_cnt=len_cnt, align='center')
	chart_headers(in_str=hmsg, len_cnt=len_cnt, align='center')
	chart_mid(len_cnt=len_cnt)
	for pos in poss:
		age_mins = pos['age_mins']
		age_days    = age_mins // (24 * 60)
		age_hours   = (age_mins // 60) % 24
		age_minutes = age_mins % 60
		if age_mins < 60:
			disp_age = f"{'':>3} {'':>2} {age_minutes:>2}"
		elif age_mins >= 60 and age_mins < 600:
			disp_age = f"{'':>3} {age_hours:>2}:{age_minutes:>02}"
		elif age_mins >= 600 and age_days < 1:
			disp_age = f"{'':>3} {age_hours:>02}:{age_minutes:>02}"
		else:
			disp_age = f"{age_days:>3} {age_hours:02}:{age_minutes:02}"
		pos['report_age'] = disp_age

		utc_timezone = pytz.utc
		pos['pos_end_dttm'] = utc_timezone.localize(pos['pos_end_dttm'])
		local_timezone = pytz.timezone('America/New_York')
		pos['pos_end_dttm'] = pos['pos_end_dttm'].astimezone(local_timezone)
		pos['pos_end_dttm'] = datetime.datetime.strftime(pos['pos_end_dttm'], "%m-%d-%Y %H:%M")
#		pos['pos_end_dttm'] = pos['pos_end_dttm'].strftime('%m-%d-%Y %H:%M')
		if pos['test_txn_yn'] == 'N': pos['test_txn_yn'] = ''

		msg = ''
		msg += f'{pos['prod_id']:<12} | '
		msg += f'{pos['test_txn_yn']:^1} | '
		msg += f'{pos['pos_id']:<6} | '
		msg += f'{pos['buy_strat_name']:>10} | '
		msg += f'{pos['buy_strat_freq']:>5} | '
		msg += f'{pos['sell_strat_name']:>14} | '
		msg += f'{pos['report_age']:>10} | '
		msg += f"{pos['pos_end_dttm']:>16} | "
		msg += f'$ {pos['tot_out_cnt']:>14.8f} | '
		msg += f'{pos['prc_buy']:>14.8f} | '
		msg += f'{pos['prc_curr']:>14.8f} | '
		msg += f'{pos['gain_loss_pct']:>8.2f} % | '
		msg += f'{pos['gain_loss_pct_est']:>8.2f} % | '
		msg += f'{pos['gain_loss_pct_est_high']:>8.2f} % | '
		msg += f'{pos['prc_chg_pct_drop']:>8.2f} % | '
		msg += f'{pos['gain_loss_amt']:>14.8f} | '
		msg += f'{pos['gain_loss_amt_est']:>14.8f} | '
		msg += f'{pos['gain_loss_amt_est_high']:>14.8f}'

		if pos['gain_loss_pct'] >= 22:
			msg = cs(msg, font_color='#FFFFFF', bg_color='#228B22')
		elif pos['gain_loss_pct'] >= 16 and  pos['gain_loss_pct'] < 22:
			msg = cs(msg, font_color='#FFFFFF', bg_color='#32CD32')
		elif pos['gain_loss_pct'] >= 11 and  pos['gain_loss_pct'] < 16:
			msg = cs(msg, font_color='#FFFFFF', bg_color='#9ACD32')
		elif pos['gain_loss_pct'] >= 7 and  pos['gain_loss_pct'] < 11:
			msg = cs(msg, font_color='#9ACD32', bg_color='#000000')
		elif pos['gain_loss_pct'] >= 4 and  pos['gain_loss_pct'] < 7:
			msg = cs(msg, font_color='#32CD32', bg_color='#000000')
		elif pos['gain_loss_pct'] >= 2 and  pos['gain_loss_pct'] < 4:
			msg = cs(msg, font_color='#228B22', bg_color='#000000')
		elif pos['gain_loss_pct'] > -2 and pos['gain_loss_pct'] < 2:
			msg = msg
		elif pos['gain_loss_pct'] <= -2 and pos['gain_loss_pct'] > -4:
			msg = cs(msg, font_color='#FFA500', bg_color='#000000')
		elif pos['gain_loss_pct'] <= -4 and pos['gain_loss_pct'] > -7:
			msg = cs(msg, font_color='#FF4500', bg_color='#000000')
		elif pos['gain_loss_pct'] <= -7 and pos['gain_loss_pct'] > -11:
			msg = cs(msg, font_color='#FF0000', bg_color='#000000')
		elif pos['gain_loss_pct'] <= -11 and pos['gain_loss_pct'] > -16:
			msg = cs(msg, font_color='#FFFFFF', bg_color='#FFA500')
		elif pos['gain_loss_pct'] <= -16 and pos['gain_loss_pct'] > -22:
			msg = cs(msg, font_color='#FFFFFF', bg_color='#FF4500')
		elif pos['gain_loss_pct'] <= -22:
			msg = cs(msg, font_color='#FFFFFF', bg_color='#FF0000')

		chart_row(msg, len_cnt=len_cnt)

	chart_bottom(len_cnt=len_cnt)

#<=====>#

def demo():
	demo1()
	demo2()
	demo3()

#<=====>#
# Post Variables
#<=====>#


#<=====>#
# Default Run
#<=====>#

if __name__ == "__main__":
	demo()

#<=====>#

