# Copyright 2016- Game Server Services, Inc. or its affiliates. All Rights
# Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License").
# You may not use this file except in compliance with the License.
# A copy of the License is located at
#
#  http://www.apache.org/licenses/LICENSE-2.0
#
# or in the "license" file accompanying this file. This file is distributed
# on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
# express or implied. See the License for the specific language governing
# permissions and limitations under the License.
from __future__ import annotations
from typing import *

from ...core.model import CdkResource, Stack
from ...core.func import GetAttr
from ...core.model import TransactionSetting
from ...core.model import LogSetting

from ..ref.NamespaceRef import NamespaceRef
from .CurrentMasterData import CurrentMasterData
from .LotteryModel import LotteryModel
from .PrizeTable import PrizeTable

from .options.NamespaceOptions import NamespaceOptions


class Namespace(CdkResource):
    stack: Stack
    name: str
    transaction_setting: TransactionSetting
    description: Optional[str] = None
    lottery_trigger_script_id: Optional[str] = None
    choice_prize_table_script_id: Optional[str] = None
    log_setting: Optional[LogSetting] = None

    def __init__(
        self,
        stack: Stack,
        name: str,
        transaction_setting: TransactionSetting,
        options: Optional[NamespaceOptions] = NamespaceOptions(),
    ):
        super().__init__(
            "Lottery_Namespace_" + name
        )

        self.stack = stack
        self.name = name
        self.transaction_setting = transaction_setting
        self.description = options.description if options.description else None
        self.lottery_trigger_script_id = options.lottery_trigger_script_id if options.lottery_trigger_script_id else None
        self.choice_prize_table_script_id = options.choice_prize_table_script_id if options.choice_prize_table_script_id else None
        self.log_setting = options.log_setting if options.log_setting else None
        stack.add_resource(
            self,
        )


    def alternate_keys(
        self,
    ):
        return "name"

    def resource_type(
        self,
    ) -> str:
        return "GS2::Lottery::Namespace"

    def properties(
        self,
    ) -> Dict[str, Any]:
        properties: Dict[str, Any] = {}

        if self.name is not None:
            properties["Name"] = self.name
        if self.description is not None:
            properties["Description"] = self.description
        if self.transaction_setting is not None:
            properties["TransactionSetting"] = self.transaction_setting.properties(
            )
        if self.lottery_trigger_script_id is not None:
            properties["LotteryTriggerScriptId"] = self.lottery_trigger_script_id
        if self.choice_prize_table_script_id is not None:
            properties["ChoicePrizeTableScriptId"] = self.choice_prize_table_script_id
        if self.log_setting is not None:
            properties["LogSetting"] = self.log_setting.properties(
            )

        return properties

    def ref(
        self,
    ) -> NamespaceRef:
        return NamespaceRef(
            self.name,
        )

    def get_attr_namespace_id(
        self,
    ) -> GetAttr:
        return GetAttr(
            None,
            None,
            "Item.NamespaceId",
        )

    def master_data(
        self,
        lottery_models: List[LotteryModel],
        prize_tables: List[PrizeTable],
    ) -> Namespace:
        CurrentMasterData(
            self.stack,
            self.name,
            lottery_models,
            prize_tables,
        ).add_depends_on(
            self,
        )
        return self
