# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ids_validator',
 'ids_validator.checks',
 'ids_validator.checks.generic',
 'ids_validator.checks.v1',
 'ids_validator.checks.v1.rules',
 'ids_validator.checks.v1.rules.samples']

package_data = \
{'': ['*'], 'ids_validator': ['templates/*']}

install_requires = \
['jsonref', 'jsonschema>4.0.0', 'pydash', 'rich', 'ts-ids-es-json-generator']

setup_kwargs = {
    'name': 'ts-ids-validator',
    'version': '0.9.12',
    'description': 'Python utility for validating IDS',
    'long_description': '# TetraScience IDS Validator <!-- omit in toc -->\n\n## Table of Contents <!-- omit in toc -->\n\n- [Overview](#overview)\n- [Usage](#usage)\n- [Components](#components)\n  - [Node](#node)\n  - [Checker Classes](#checker-classes)\n  - [Validator](#validator)\n  - [List of Checker Classes](#list-of-checker-classes)\n    - [Base Classes](#base-classes)\n    - [Generic](#generic)\n    - [V1](#v1)\n  - [Writing New Checks](#writing-new-checks)\n  - [Extending Checkers Classes](#extending-checkers-classes)\n    - [Pattern 1](#pattern-1)\n    - [Pattern 2](#pattern-2)\n  - [Running Checks for Specific Nodes](#running-checks-for-specific-nodes)\n  - [List of Checks for Validator](#list-of-checks-for-validator)\n- [Changelog](#changelog)\n- [v0.9.12](#v0912)\n- [v0.9.11](#v0911)\n- [v0.9.10](#v0910)\n- [v0.9.9](#v099)\n- [v0.9.8](#v098)\n- [v0.9.7](#v097)\n- [v0.9.6](#v096)\n\n## Overview\n\nThe TetraScience IDS Validator checks that IDS artifacts follow a set of rules which\nmake them compatible with the Tetra Data Platform, and optionally that they are\ncompatible with with additional IDS design conventions.\nThe validator either passes or fails with a list of the checks which led to the failure.\n\nThe validator checks these files in an IDS folder:\n\n- schema.json\n- elasticsearch.json\n- athena.json\n\nYou can find the validation rules in:\n\n- [IDS Design Conventions - schema.json](https://developers.tetrascience.com/docs/ids-design-conventions-schemajson)\n- [IDS Design Conventions - elasticsearch.json](https://developers.tetrascience.com/docs/ids-design-conventions-elasticsearchjson)\n- [IDS Design Conventions - athena.json](https://developers.tetrascience.com/docs/ids-design-conventions-athenajson)\n\n## Usage\n\n```bash\npoetry run python -m ids_validator --ids_dir=path/to/ids/folder\n```\n\nThis will run the required checks for the `@idsConventionVersion` mentioned in `schema.json`.\n\nIf `@idsConventionVersion` is missing in `schema.json` or if it is not supported by `schema_validator`, only `generic` checks will be run.\n\n## Components\n\n### Node\n\n- `Node: UserDict` class is an abstraction for `dict` in `schema.json`\n- When crawling `schema.json`, each `key-value` pair where `value` is a `dict`, is casted into `Node` type.\n- For each K_V pair, `Node` has following attributes\n  - `name (default=root)`: The `key`\n  - `data`: The `value:dict`\n  - `path (default=root)`: The fully-qualified path for the `key` in `schema.json`\n- File: [ids_node.py](src/ids_node.py)\n\n### Checker Classes\n\n- A checker class must implement `AbstractChecker`\n- When crawling `schema.json`, its `run()` method will be called for each node.\n- `run()` implements the rules/condition to be checked for validating the node.\n- `run()` accepts two arguments:\n  - `node: Node`: `Node` for which we are running the checks\n  - `context: dict`\n    - It contains python dicts for `schema.json`, `athena.json` and `convention_version`.\n    - It is used to supplementary data required for running complex checks.\n\n### Validator\n\n- `Validator` class is the one that implements the crawler.\n- It has following attributes:\n  - `ids: dict`: `schema.json` converted to python `dict`\n  - `athena: dict`: `athena.json` converted to python `dict`\n  - `checks_list`: A list of instantiated checker classes.\n    These list of checks will be run for each node\n- `Validator.traverse_ids()` crawls from `Node` to `Node` in `ids:dict`, Calling `run()` for each checker in the checks_list on the node\n\n### List of Checker Classes\n\n#### Base Classes\n\n- `AbstractChecker`\n\n  - Every checker class must implement it.\n  - File: [abstract_checker.py](src/checks/abstract_checker.py)\n\n- `RuleBasedChecker`\n  - It is base class that allows validating `Node` against a set of `rules`\n  - It comes in handy for implementing checks for property Nodes that has predefined template\n  - The child class inheriting `RulesBasedChecker` must define `rules`\n  - `rules` is a `dict` that maps `Node.path` to `set of rules:dict`\n  - The `set of rules` for a `Node.path` may contain following items:\n    - `"type"`, `Union[List[str], str]`: defines what should be the `type` value for the `Node`\n    - `"compatible_type"`, `ids_validator.checks.rules_checker.BackwardCompatibleType`:\n      defines the allowable `type` values for a `Node`, matching either a `preferred`\n      type, or one of a list of `deprecated` types which will lead to a warning.\n    - `"min_properties"`, `List[str]`: defines minimum set of property names that must exist for the Node. More properties can exist in addition to `min_properties`\n    - `"properties"`, `List[str]`: defines a set of property names that must must exactly match the property list of the `Node`\n    - `"min_required"`, `List[str]`: The required list of the `Node` must at least contain the values mentioned in `min_required`\n    - `"required"`, `List[str]`: The required list of the `Node` must only contain values listed in `required`\n- Rules based checkers defined for v1 conventions can be found [here](src/checks/v1/nodes_checker.py)\n\n#### Generic\n\n- `AdditionalPropertyChecker`: [additional_property.py](src/checks/generic/additional_property.py)\n- `RequiredPropertiesChecker`: [required_property.py](src/checks/generic/required_property.py)\n- `DatacubesChecker`: [datacubes.py](src/checks/generic/datacubes.py)\n- `RootNodeChecker`: [root_node.py](src/checks/generic/root_node.py)\n- `TypeChecker`: [type_check.py](src/checks/generic/type_check.py)\n- `AthenaChecker`: [athena.py](src/checks/generic/athena.py)\n\n#### V1\n\n- `V1ChildNameChecker`: [child_name.py](src/checks/v1/child_name.py)\n- `V1ConventionVersionChecker`: [convention_version_check.py](src/checks/v1/convention_version_check.py)\n- `V1SystemNodeChecker`: [nodes_checker.py](src/checks/v1/nodes_checker.py)\n- `V1SampleNodeChecker`: [nodes_checker.py](src/checks/v1/nodes_checker.py)\n- `V1UserNodeChecker`: [nodes_checker.py](src/checks/v1/nodes_checker.py)\n- `V1RootNodeChecker`: [root_node.py](src/checks/v1/root_node.py)\n- `V1SnakeCaseChecker`: [snake_case.py](src/checks/v1/snake_case.py)\n\n### Writing New Checks\n\n- Checkers must implement `AbstractCheckers`\n- `run()` method implement one or more checks for the node\n- In case of no failure an empty list must be returned\n- In case of failures, it must return a list of one or more tuple\n- The tuple will contain two values\n  - `log message:str`: The message to be logged when check fails\n  - `criticality`: either `Log.CRITICAL` or `Log.WARNING`\n\n### Extending Checkers Classes\n\n#### Pattern 1\n\n```python\nclass ChildChecker(ParentChecker):\n    def run(node: Node, context: dict):\n        logs = []\n        # Implement new checks and append failure to logs\n\n        # Run Parent checkers and append logs\n        logs += super().run(node, context)\n        return logs\n```\n\nIf `check_list` passed to `Validator` contains the `ChildChecker`, then it must not contain `ParentChecker` in the same list.\nDoing so will cause ParentCheck to run twice and populate failures logs if any, twice.\n\nTODO: Instead of `return logs`, we can `return set(logs)` to remove duplicates, but we cannot avoid executing same code twice\n\n#### Pattern 2\n\n```python\nclass ChildChecker(ParentChecker):\n    def run(node: Node, context: dict):\n        logs = []\n        # Implement new checks and append failure to logs\n        # use or override helper function of the parent class\n        return logs\n```\n\n### Running Checks for Specific Nodes\n\n```python\nclass AdhocChecker(AbstractChecker):\n    def run(node: Node, context: dict):\n        logs = []\n        paths = []\n        # paths is a list of fully qualified path to a key in schema.json\n        # each path must start form root\n        # eg: root.samples\n        # eg: root.samples.items.properties.property_name\n        if node.path in  paths:\n            # Implement new checks and append failure to logs\n            logs += perform_new_checks(node, context)\n        return logs\n```\n\n### List of Checks for Validator\n\n- `checks_dict`, defined [here](src/checks/__init__.py), maps the `type of validation` that we want to perform to the `list the of checks` needed to be run for the validation\n- The list off checks is actually a list of instantiated checker objects\n\n## Changelog\n\n## v0.9.12\n\n- Update check for `samples[*].labels[*].source.name` type: previously the type was\n  required to be `"string"`, now it is required to be either `["string", "null"]` or\n  `"string"`, with `"string"` leading to a deprecation warning. This change makes this\n  `source` definition the same as `samples[*].properties[*].source` in a\n  backward-compatible way.\n\n## v0.9.11\n\n- Fix bug in `AthenaChecker` to allow root level IDS properties as partition paths.\n- Update `TypeChecker` to catch errors related to undefined/misspelled `type` key.\n- Update `jsonschema` version to fix package installation error\n\n## v0.9.10\n\n- Modify `V1SnakeCaseChecker` to ignore checks for keys present in `definitions` object.\n- Add temporary allowance for `@link` in `*.properties`\n\n## v0.9.9\n\n- Lock `jsonschema` version in requirements.txt\n\n## v0.9.8\n\n- Modify `RulesChecker` to log missing and extra properties\n\n## v0.9.7\n\n- Allow properties with `const` values to have non-nullable `type`\n\n## v0.9.6\n\n- Add checker classes for generic validation\n- Add checker classes for v1.0.0 convention validation\n',
    'author': 'TetraScience',
    'author_email': 'developers@tetrascience.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://developers.tetrascience.com',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
