# -*- coding: utf-8 -*-
from distutils.core import setup

packages = \
['iso4217parse']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'iso4217parse',
    'version': '0.4.2',
    'description': 'Parse currencies (symbols and codes) from and to ISO4217.',
    'long_description': "[![Build Status](https://travis-ci.org/tammoippen/iso4217parse.svg?branch=master)](https://travis-ci.org/tammoippen/iso4217parse)\n[![Coverage Status](https://coveralls.io/repos/github/tammoippen/iso4217parse/badge.svg?branch=master)](https://coveralls.io/github/tammoippen/iso4217parse?branch=master)\n[![Tested CPython Versions](https://img.shields.io/badge/cpython-2.7%2C%203.5%2C%203.6%2C%20nightly-brightgreen.svg)](https://img.shields.io/badge/cpython-2.7%2C%203.5%2C%203.6%2C%20nightly-brightgreen.svg)\n[![Tested PyPy Versions](https://img.shields.io/badge/pypy-2.7--5.8.0%2C%203.5--5.8.0-brightgreen.svg)](https://img.shields.io/badge/pypy-2.7--5.8.0%2C%203.5--5.8.0-brightgreen.svg)\n[![PyPi version](https://img.shields.io/pypi/v/iso4217parse.svg)](https://pypi.python.org/pypi/iso4217parse)\n[![PyPi license](https://img.shields.io/pypi/l/iso4217parse.svg)](https://pypi.python.org/pypi/iso4217parse)\n\n# ISO4217 Currency Parser\n\nParse currencies (symbols and codes) from and to [ISO4217](https://en.wikipedia.org/wiki/ISO_4217).\n\nSimilar to [iso4217](https://github.com/spoqa/iso4217) package, but\n\n * data is aquired by scraping wikipedia (see [below](#data-aquisition)) - this is repeatable and you stay on the most current data\n * currency symbols are currated by hand - this allows some fuzzy currency matching\n * no download and parsing during install\n * no external dependancies (`enum34`)\n\nWhen you want to *reuse* the [*data.json*](https://github.com/tammoippen/iso4217parse/blob/master/iso4217parse/data.json) file for your projects, please leave a attribution note. I licence the file under (CC BY 4.0).\n\nInstall:\n```\npip install iso4217parse\n```\n\n## Documentation\n\nEach currency is modeled as a `collections.namedtuple`:\n```python\nCurrency = namedtuple('Currency', [\n    'alpha3',     # unicode:       the ISO4217 alpha3 code\n    'code_num',   # int:           the ISO4217 numeric code\n    'name',       # unicode:       the currency name\n    'symbols',    # List[unicode]: list of possible symbols;\n                  #                first is opinionated choice for representation\n    'minor',      # int:           number of decimal digits to round\n    'countries',  # List[unicode]: list of countries that use this currency.\n])\n```\n\n**parse:**  Try to parse the input in a best effort approach by using `by_alpha3()`, `by_code_num()`, ... functions:\n```python\nIn [1]: import iso4217parse\n\nIn [2]: iso4217parse.parse('CHF')\nOut[2]: [Currency(alpha3='CHF', code_num=756, name='Swiss franc', \n                  symbols=['SFr.', 'fr', 'Fr.', 'F', 'franc', 'francs', 'Franc', 'Francs'],\n                  minor=2, countries=['CH', 'LI'])]\n\nIn [3]: iso4217parse.parse(192)\nOut[3]:\n[Currency(alpha3='CUP', code_num=192, name='Cuban peso', \n          symbols=['₱', '＄', '﹩', '$', 'dollar', 'dollars', 'Dollar', 'Dollars', '＄MN', '﹩MN', '$MN'], \n          minor=2, countries=['CU'])]\n\nIn [4]: iso4217parse.parse('Price is 5 €')\nOut[4]: [Currency(alpha3='EUR', code_num=978, name='Euro', \n         symbols=['€', 'euro', 'euros'], minor=2, \n         countries=['AD', 'AT', 'AX', 'BE', 'BL', 'CY', 'DE', 'EE', 'ES', 'FI',\n                   'FR', 'GF', 'GP', 'GR', 'IE', 'IT', 'LT', 'LU', 'LV', 'MC',\n                   'ME', 'MF', 'MQ', 'MT', 'NL', 'PM', 'PT', 'RE', 'SI', 'SK',\n                   'SM', 'TF', 'VA', 'XK', 'YT'])]\n\nIn [5]: iso4217parse.parse('CA﹩15.76')\nOut[5]: [Currency(alpha3='CAD', code_num=124, name='Canadian dollar', \n         symbols=['CA$', 'CA＄', '＄', '$', 'dollar', 'dollars', 'Dollar', 'Dollars', 'CA﹩', '﹩'], \n         minor=2, countries=['CA'])]\n\nIn [6]: iso4217parse.parse?\nSignature: iso4217parse.parse(v, country_code=None)\nDocstring:\nTry parse `v` to currencies; filter by country_code\n\nIf `v` is a number, try `by_code_num()`; otherwise try:\n    1) if `v` is 3 character uppercase: `by_alpha3()`\n    2) Exact symbol match: `by_symbol()`\n    3) Exact country code match: `by_country()`\n    4) Fuzzy by symbol match heuristic: `by_symbol_match()`\n\nParameters:\n    v: Union[unicode, int]           Either a iso4217 numeric code or some string\n    country_code: Optional[unicode]  Iso3166 alpha2 country code.\n\nReturns:\n    List[Currency]: found Currency objects.\n```\n\n**by_alpha3:** Get the currency by its iso4217 alpha3 code:\n```python\nIn [1]: import iso4217parse\n\nIn [2]: iso4217parse.by_alpha3('CHF')\nOut[2]: Currency(alpha3='CHF', code_num=756, name='Swiss franc', \n                 symbols=['SFr.', 'fr', 'Fr.', 'F', 'franc', 'francs', 'Franc', 'Francs'],\n                 minor=2, countries=['CH', 'LI'])\n\nIn [3]: iso4217parse.by_alpha3?\nSignature: iso4217parse.by_alpha3(code)\nDocstring:\nGet Currency for ISO4217 alpha3 code\n\nParameters:\n    code: unicode  An alpha3 iso4217 code.\n\nReturns:\n    Currency: Currency object for `code`, if available.\n```\n\n**by_code_num:** Get the currency by its iso4217 numeric code:\n```python\nIn [1]: import iso4217parse\n\nIn [2]: iso4217parse.by_code_num(51)\nOut[2]: Currency(alpha3='AMD', code_num=51, name='Armenian dram', \n                 symbols=['֏', 'դր', 'dram'], minor=2, countries=['AM'])\n\nIn [3]: iso4217parse.by_code_num?\nSignature: iso4217parse.by_code_num(code_num)\nDocstring:\nGet Currency for ISO4217 numeric code\n\nParameters:\n    code_num: int  An iso4217 numeric code.\n\nReturns:\n    Currency: Currency object for `code_num`, if available.\n```\n\n**by_country:** Get currencies used in a country:\n```python\nIn [1]: import iso4217parse\n\nIn [2]: iso4217parse.country('HK')\nOut[2]: \n[ \n  Currency(alpha3='HKD', code_num=344, name='Hong Kong dollar', \n           symbols=['HK$', 'HK＄', '＄', '$', 'dollar', 'dollars', 'Dollar', 'Dollars', 'HK﹩', '﹩', '元'], \n           minor=2, countries=['HK']),\n  Currency(alpha3='CNH', code_num=None, name='Chinese yuan (when traded offshore)', \n           symbols=['CN¥', '￥', 'CN￥', '¥', 'RMB', '元'], \n           minor=2, countries=['HK'])\n]\n\nIn [3]: iso4217parse.country?\nSignature: iso4217parse.by_country(country_code)\nDocstring:\nGet all currencies used in country\n\nParameters:\n    country_code: unicode  iso3166 alpha2 country code\n\nReturns:\n    List[Currency]: Currency objects used in country.\n```\n\n**by_symbol:** Get currencies that use the given symbol:\n```python\nIn [1]: import iso4217parse\n\nIn [2]: iso4217parse.by_symbol('＄MN')\nOut[2]: \n[\n  Currency(alpha3='CUP', code_num=192, name='Cuban peso', \n           symbols=['₱', '＄', '﹩', '$', 'dollar', 'dollars', 'Dollar', 'Dollars', '＄MN', '﹩MN', '$MN'], \n           minor=2, countries=['CU'])\n]\n\nIn [3]: iso4217parse.by_symbol('＄')\nOut[3]: [...] # 35 different currencies\n\nIn [4]: [c.alpha3 for c in iso4217parse.by_symbol('＄')]\nOut[4]:\n['ARS', 'AUD', 'BBD', 'BMD', 'BZD', 'SBD', 'BND', 'CAD', 'CVE', 'KYD', 'CLP', \n 'COP', 'CUP', 'DOP', 'FJD', 'GYD', 'HKD', 'JMD', 'LRD', 'MXN', 'NAD', 'NZD', \n 'SGD', 'TTD', 'USD', 'UYU', 'TWD', 'CUC', 'ZWL', 'XCD', 'SRD', 'BRL', 'KID', \n 'NTD', 'TVD']\n\nIn [5]: iso4217parse.by_symbol('＄', country_code='US')\nOut[5]: \n[\n  Currency(alpha3='USD', code_num=840, name='United States dollar', \n           symbols=['US$', '$', '＄', '﹩', 'dollar', 'dollars', 'Dollar', 'Dollars', 'US＄', 'US﹩'], \n           minor=2, \n           countries=['AS', 'EC', 'GU', 'HT', 'MH', 'MP', 'PR', 'PW', 'SV', 'TC', 'TL', 'UM', 'US'])\n]\n\nIn [6]: iso4217parse.by_symbol?\nSignature: iso4217parse.by_symbol(symbol, country_code=None)\nDocstring:\nGet list of possible currencies for symbol; filter by country_code\n\nLook for all currencies that use the `symbol`. If there are currencies used\nin the country of `country_code`, return only those; otherwise return all\nfound currencies.\n\nParameters:\n    symbol: unicode                  Currency symbol.\n    country_code: Optional[unicode]  Iso3166 alpha2 country code.\n\nReturns:\n    List[Currency]: Currency objects for `symbol`; filter by country_code.\n```\n\n**by_symbol_match:** Look for currency symbol occurence in input string:\n```python\nIn [1]: import iso4217parse\n\nIn [2]: iso4217parse.by_symbol_match('RD$35.8')\nOut[2]: \n[\n  Currency(alpha3='DOP', code_num=214, name='Dominican peso', \n           symbols=['RD$', '＄', '﹩', '$', 'dollar', 'dollars', 'Dollar', 'Dollars', 'RD＄', 'RD﹩'], \n           minor=2, countries=['DO'])\n]\n\nIn [3]: iso4217parse.by_symbol_match('The price is ₨ 35.8 !')\nOut[3]: \n[\n  Currency(alpha3='LKR', code_num=144, name='Sri Lankan rupee', \n           symbols=['රු', '₨', 'Rs', 'ரூ', 'SLRs', 'rupees', 'rupee'], \n           minor=2, countries=['LK']),\n  Currency(alpha3='MUR', code_num=480, name='Mauritian rupee', \n           symbols=['₨', 'rupees', 'rupee'], minor=2, countries=['MU']),\n  Currency(alpha3='NPR', code_num=524, name='Nepalese rupee', \n           symbols=['रु', '₨', 'Rs', 'Re', 'rupees', 'rupee'], minor=2, countries=['NP']),\n  Currency(alpha3='PKR', code_num=586, name='Pakistani rupee', \n           symbols=['₨', 'Rs', 'rupees', 'rupee'], \n           minor=2, countries=['PK'])\n]\n\nIn [4]: iso4217parse.by_symbol_match('The price is ₨ 35.8 !', country_code='NP')\nOut[4]: \n[\n  Currency(alpha3='NPR', code_num=524, name='Nepalese rupee', \n           symbols=['रु', '₨', 'Rs', 'Re', 'rupees', 'rupee'], \n           minor=2, countries=['NP'])\n]\n\nIn [5]: iso4217parse.by_symbol_match?\nSignature: iso4217parse.by_symbol_match(value, country_code=None)\nDocstring:\nGet list of possible currencies where the symbol is in value; filter by country_code (iso3166 alpha2 code)\n\nLook for first matching currency symbol in `value`. Filter similar to `by_symbol`.\nSymbols are sorted by length and relevance of first character (see `_symbols()`).\n\nNote: This is a [heuristic](https://en.wikipedia.org/wiki/Heuristic) !\n\nParameters:\n    value: unicode                   Some input string.\n    country_code: Optional[unicode]  Iso3166 alpha2 country code.\n\nReturns:\n    List[Currency]: Currency objects found in `value`; filter by country_code.\n```\n\n\n## Data aquisition\n\nBasic ISO4217 currency information is gathered from wikipedia: https://en.wikipedia.org/wiki/ISO_4217 . The tables are parsed with `gen_data.py` and stored in `iso4217parse/data.json`. This gives information for `alpha3`, `code_num`, `name`, `minor` and `countries`. The currency symbol information is hand gathered from:\n\n* individuel wikipedia pages, i.e. [EUR](https://en.wikipedia.org/wiki/Euro) has a `Denominations` -> `Symbol` section.\n* http://www.iotafinance.com/en/ISO-4217-Currency-Codes.html\n* http://www.xe.com/currency/ , i.e. [GBP](http://www.xe.com/currency/gbp-british-pound) has a `Currency Facts` -> `Symbol` section\n\nand stored in `iso4217parse/symbols.json`. Each currency can have multiple currency symbols - the first symbol in the list is the (opinionated) choice\nfor the currency.\n\n**Contribution Note**: Possible ways to contribute here:\n\n* hand check symbols for currency code.\n* automatic generation of the `iso4217parse/symbols.json` file.\n",
    'author': 'Tammo Ippen',
    'author_email': 'tammo.ippen@posteo.de',
    'url': 'https://github.com/tammoippen/iso4217parse',
    'packages': packages,
    'package_data': package_data,
}


setup(**setup_kwargs)
