"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getOrCreateKubectlLayer = exports.KubectlProvider = void 0;
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const cluster_1 = require("./cluster");
const kubectl_layer_1 = require("./kubectl-layer");
class KubectlProvider extends core_1.NestedStack {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const cluster = props.cluster;
        if (!cluster.kubectlRole) {
            throw new Error('"kubectlRole" is not defined, cannot issue kubectl commands against this cluster');
        }
        if (cluster.kubectlPrivateSubnets && !cluster.kubectlSecurityGroup) {
            throw new Error('"kubectlSecurityGroup" is required if "kubectlSubnets" is specified');
        }
        const layer = (_a = cluster.kubectlLayer) !== null && _a !== void 0 ? _a : getOrCreateKubectlLayer(this);
        const handler = new lambda.Function(this, 'Handler', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'kubectl-handler')),
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            timeout: core_1.Duration.minutes(15),
            description: 'onEvent handler for EKS kubectl resource provider',
            layers: [layer],
            memorySize: 256,
            environment: cluster.kubectlEnvironment,
            // defined only when using private access
            vpc: cluster.kubectlPrivateSubnets ? cluster.vpc : undefined,
            securityGroups: cluster.kubectlSecurityGroup ? [cluster.kubectlSecurityGroup] : undefined,
            vpcSubnets: cluster.kubectlPrivateSubnets ? { subnets: cluster.kubectlPrivateSubnets } : undefined,
        });
        this.handlerRole = handler.role;
        this.handlerRole.addToPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeCluster'],
            resources: [cluster.clusterArn],
        }));
        // allow this handler to assume the kubectl role
        cluster.kubectlRole.grant(this.handlerRole, 'sts:AssumeRole');
        const provider = new cr.Provider(this, 'Provider', {
            onEventHandler: handler,
        });
        this.serviceToken = provider.serviceToken;
        this.roleArn = cluster.kubectlRole.roleArn;
    }
    static getOrCreate(scope, cluster) {
        // if this is an "owned" cluster, it has a provider associated with it
        if (cluster instanceof cluster_1.Cluster) {
            return cluster._attachKubectlResourceScope(scope);
        }
        // if this is an imported cluster, we need to provision a custom resource provider in this stack
        // we will define one per stack for each cluster based on the cluster uniqueid
        const uid = `${cluster.node.uniqueId}-KubectlProvider`;
        const stack = core_1.Stack.of(scope);
        let provider = stack.node.tryFindChild(uid);
        if (!provider) {
            provider = new KubectlProvider(stack, uid, { cluster });
        }
        return provider;
    }
}
exports.KubectlProvider = KubectlProvider;
/**
 * Gets or create a singleton instance of KubectlLayer.
 *
 * (exported for unit tests).
 */
function getOrCreateKubectlLayer(scope, props = {}) {
    const stack = core_1.Stack.of(scope);
    const id = 'kubectl-layer-' + (props.version ? props.version : '8C2542BC-BF2B-4DFE-B765-E181FD30A9A0');
    const exists = stack.node.tryFindChild(id);
    if (exists) {
        return exists;
    }
    return new kubectl_layer_1.KubectlLayer(stack, id, props);
}
exports.getOrCreateKubectlLayer = getOrCreateKubectlLayer;
//# sourceMappingURL=data:application/json;base64,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