"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const lambda = require("@aws-cdk/aws-lambda");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const tasks = require("../../lib");
let stack;
let fn;
beforeEach(() => {
    stack = new core_1.Stack();
    fn = new lambda.Function(stack, 'Fn', {
        code: lambda.Code.fromInline('hello'),
        handler: 'index.hello',
        runtime: lambda.Runtime.PYTHON_2_7,
    });
});
test('Invoke lambda with default magic ARN', () => {
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunLambdaTask(fn, {
            payload: sfn.TaskInput.fromObject({
                foo: 'bar',
            }),
            invocationType: tasks.InvocationType.REQUEST_RESPONSE,
            clientContext: 'eyJoZWxsbyI6IndvcmxkIn0=',
            qualifier: '1',
        }),
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task,
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::lambda:invoke',
                ],
            ],
        },
        End: true,
        Parameters: {
            FunctionName: {
                Ref: 'Fn9270CBC0',
            },
            Payload: {
                foo: 'bar',
            },
            InvocationType: 'RequestResponse',
            ClientContext: 'eyJoZWxsbyI6IndvcmxkIn0=',
            Qualifier: '1',
        },
    });
});
test('Lambda function can be used in a Task with Task Token', () => {
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunLambdaTask(fn, {
            integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            payload: sfn.TaskInput.fromObject({
                token: sfn.Context.taskToken,
            }),
        }),
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task,
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::lambda:invoke.waitForTaskToken',
                ],
            ],
        },
        End: true,
        Parameters: {
            FunctionName: {
                Ref: 'Fn9270CBC0',
            },
            Payload: {
                'token.$': '$$.Task.Token',
            },
        },
    });
});
test('Lambda function is invoked with the state input as payload by default', () => {
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunLambdaTask(fn),
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task,
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::lambda:invoke',
                ],
            ],
        },
        End: true,
        Parameters: {
            'FunctionName': {
                Ref: 'Fn9270CBC0',
            },
            'Payload.$': '$',
        },
    });
});
test('Lambda function can be provided with the state input as the payload', () => {
    const task = new sfn.Task(stack, 'Task', {
        task: new tasks.RunLambdaTask(fn, {
            payload: sfn.TaskInput.fromDataAt('$'),
        }),
    });
    new sfn.StateMachine(stack, 'SM', {
        definition: task,
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::lambda:invoke',
                ],
            ],
        },
        End: true,
        Parameters: {
            'FunctionName': {
                Ref: 'Fn9270CBC0',
            },
            'Payload.$': '$',
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied but task token is not included in payLoad', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunLambdaTask(fn, {
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            }),
        });
    }).toThrow(/Task Token is missing in payload/i);
});
test('Task throws if SYNC is supplied as service integration pattern', () => {
    expect(() => {
        new sfn.Task(stack, 'Task', {
            task: new tasks.RunLambdaTask(fn, {
                integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
            }),
        });
    }).toThrow(/Invalid Service Integration Pattern: SYNC is not supported to call Lambda./i);
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicnVuLWxhbWJkYS10YXNrLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJydW4tbGFtYmRhLXRhc2sudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLGdDQUE4QjtBQUM5Qiw4Q0FBOEM7QUFDOUMsa0RBQWtEO0FBQ2xELHdDQUFzQztBQUN0QyxtQ0FBbUM7QUFFbkMsSUFBSSxLQUFZLENBQUM7QUFDakIsSUFBSSxFQUFtQixDQUFDO0FBQ3hCLFVBQVUsQ0FBQyxHQUFHLEVBQUU7SUFDZCxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUNwQixFQUFFLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUU7UUFDcEMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQztRQUNyQyxPQUFPLEVBQUUsYUFBYTtRQUN0QixPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxVQUFVO0tBQ25DLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHNDQUFzQyxFQUFFLEdBQUcsRUFBRTtJQUNoRCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtRQUN2QyxJQUFJLEVBQUUsSUFBSSxLQUFLLENBQUMsYUFBYSxDQUFDLEVBQUUsRUFBRTtZQUNoQyxPQUFPLEVBQUUsR0FBRyxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUM7Z0JBQ2hDLEdBQUcsRUFBRSxLQUFLO2FBQ1gsQ0FBQztZQUNGLGNBQWMsRUFBRSxLQUFLLENBQUMsY0FBYyxDQUFDLGdCQUFnQjtZQUNyRCxhQUFhLEVBQUUsMEJBQTBCO1lBQ3pDLFNBQVMsRUFBRSxHQUFHO1NBQ2YsQ0FBQztLQUNILENBQUMsQ0FBQztJQUNILElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFO1FBQ2hDLFVBQVUsRUFBRSxJQUFJO0tBQ2pCLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDO1FBQ2hELElBQUksRUFBRSxNQUFNO1FBQ1osUUFBUSxFQUFFO1lBQ1IsVUFBVSxFQUFFO2dCQUNWLEVBQUU7Z0JBQ0Y7b0JBQ0UsTUFBTTtvQkFDTjt3QkFDRSxHQUFHLEVBQUUsZ0JBQWdCO3FCQUN0QjtvQkFDRCx5QkFBeUI7aUJBQzFCO2FBQ0Y7U0FDRjtRQUNELEdBQUcsRUFBRSxJQUFJO1FBQ1QsVUFBVSxFQUFFO1lBQ1YsWUFBWSxFQUFFO2dCQUNaLEdBQUcsRUFBRSxZQUFZO2FBQ2xCO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLEdBQUcsRUFBRSxLQUFLO2FBQ1g7WUFDRCxjQUFjLEVBQUUsaUJBQWlCO1lBQ2pDLGFBQWEsRUFBRSwwQkFBMEI7WUFDekMsU0FBUyxFQUFFLEdBQUc7U0FDZjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHVEQUF1RCxFQUFFLEdBQUcsRUFBRTtJQUNqRSxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtRQUN2QyxJQUFJLEVBQUUsSUFBSSxLQUFLLENBQUMsYUFBYSxDQUFDLEVBQUUsRUFBRTtZQUNoQyxrQkFBa0IsRUFBRSxHQUFHLENBQUMseUJBQXlCLENBQUMsbUJBQW1CO1lBQ3JFLE9BQU8sRUFBRSxHQUFHLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQztnQkFDaEMsS0FBSyxFQUFFLEdBQUcsQ0FBQyxPQUFPLENBQUMsU0FBUzthQUM3QixDQUFDO1NBQ0gsQ0FBQztLQUNILENBQUMsQ0FBQztJQUNILElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFO1FBQ2hDLFVBQVUsRUFBRSxJQUFJO0tBQ2pCLENBQUMsQ0FBQztJQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDO1FBQ2hELElBQUksRUFBRSxNQUFNO1FBQ1osUUFBUSxFQUFFO1lBQ1IsVUFBVSxFQUFFO2dCQUNWLEVBQUU7Z0JBQ0Y7b0JBQ0UsTUFBTTtvQkFDTjt3QkFDRSxHQUFHLEVBQUUsZ0JBQWdCO3FCQUN0QjtvQkFDRCwwQ0FBMEM7aUJBQzNDO2FBQ0Y7U0FDRjtRQUNELEdBQUcsRUFBRSxJQUFJO1FBQ1QsVUFBVSxFQUFFO1lBQ1YsWUFBWSxFQUFFO2dCQUNaLEdBQUcsRUFBRSxZQUFZO2FBQ2xCO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLFNBQVMsRUFBRSxlQUFlO2FBQzNCO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyx1RUFBdUUsRUFBRSxHQUFHLEVBQUU7SUFDakYsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7UUFDdkMsSUFBSSxFQUFFLElBQUksS0FBSyxDQUFDLGFBQWEsQ0FBQyxFQUFFLENBQUM7S0FDbEMsQ0FBQyxDQUFDO0lBQ0gsSUFBSSxHQUFHLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUU7UUFDaEMsVUFBVSxFQUFFLElBQUk7S0FDakIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDaEQsSUFBSSxFQUFFLE1BQU07UUFDWixRQUFRLEVBQUU7WUFDUixVQUFVLEVBQUU7Z0JBQ1YsRUFBRTtnQkFDRjtvQkFDRSxNQUFNO29CQUNOO3dCQUNFLEdBQUcsRUFBRSxnQkFBZ0I7cUJBQ3RCO29CQUNELHlCQUF5QjtpQkFDMUI7YUFDRjtTQUNGO1FBQ0QsR0FBRyxFQUFFLElBQUk7UUFDVCxVQUFVLEVBQUU7WUFDVixjQUFjLEVBQUU7Z0JBQ2QsR0FBRyxFQUFFLFlBQVk7YUFDbEI7WUFDRCxXQUFXLEVBQUUsR0FBRztTQUNqQjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHFFQUFxRSxFQUFFLEdBQUcsRUFBRTtJQUMvRSxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtRQUN2QyxJQUFJLEVBQUUsSUFBSSxLQUFLLENBQUMsYUFBYSxDQUFDLEVBQUUsRUFBRTtZQUNoQyxPQUFPLEVBQUUsR0FBRyxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDO1NBQ3ZDLENBQUM7S0FDSCxDQUFDLENBQUM7SUFDSCxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRTtRQUNoQyxVQUFVLEVBQUUsSUFBSTtLQUNqQixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQztRQUNoRCxJQUFJLEVBQUUsTUFBTTtRQUNaLFFBQVEsRUFBRTtZQUNSLFVBQVUsRUFBRTtnQkFDVixFQUFFO2dCQUNGO29CQUNFLE1BQU07b0JBQ047d0JBQ0UsR0FBRyxFQUFFLGdCQUFnQjtxQkFDdEI7b0JBQ0QseUJBQXlCO2lCQUMxQjthQUNGO1NBQ0Y7UUFDRCxHQUFHLEVBQUUsSUFBSTtRQUNULFVBQVUsRUFBRTtZQUNWLGNBQWMsRUFBRTtnQkFDZCxHQUFHLEVBQUUsWUFBWTthQUNsQjtZQUNELFdBQVcsRUFBRSxHQUFHO1NBQ2pCO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsMEZBQTBGLEVBQUUsR0FBRyxFQUFFO0lBQ3BHLE1BQU0sQ0FBQyxHQUFHLEVBQUU7UUFDVixJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtZQUMxQixJQUFJLEVBQUUsSUFBSSxLQUFLLENBQUMsYUFBYSxDQUFDLEVBQUUsRUFBRTtnQkFDaEMsa0JBQWtCLEVBQUUsR0FBRyxDQUFDLHlCQUF5QixDQUFDLG1CQUFtQjthQUN0RSxDQUFDO1NBQ0gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLG1DQUFtQyxDQUFDLENBQUM7QUFDbEQsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsZ0VBQWdFLEVBQUUsR0FBRyxFQUFFO0lBQzFFLE1BQU0sQ0FBQyxHQUFHLEVBQUU7UUFDVixJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtZQUMxQixJQUFJLEVBQUUsSUFBSSxLQUFLLENBQUMsYUFBYSxDQUFDLEVBQUUsRUFBRTtnQkFDaEMsa0JBQWtCLEVBQUUsR0FBRyxDQUFDLHlCQUF5QixDQUFDLElBQUk7YUFDdkQsQ0FBQztTQUNILENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyw2RUFBNkUsQ0FBQyxDQUFDO0FBQzVGLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICdAYXdzLWNkay9hc3NlcnQvamVzdCc7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQgKiBhcyBzZm4gZnJvbSAnQGF3cy1jZGsvYXdzLXN0ZXBmdW5jdGlvbnMnO1xuaW1wb3J0IHsgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIHRhc2tzIGZyb20gJy4uLy4uL2xpYic7XG5cbmxldCBzdGFjazogU3RhY2s7XG5sZXQgZm46IGxhbWJkYS5GdW5jdGlvbjtcbmJlZm9yZUVhY2goKCkgPT4ge1xuICBzdGFjayA9IG5ldyBTdGFjaygpO1xuICBmbiA9IG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdGbicsIHtcbiAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tSW5saW5lKCdoZWxsbycpLFxuICAgIGhhbmRsZXI6ICdpbmRleC5oZWxsbycsXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuUFlUSE9OXzJfNyxcbiAgfSk7XG59KTtcblxudGVzdCgnSW52b2tlIGxhbWJkYSB3aXRoIGRlZmF1bHQgbWFnaWMgQVJOJywgKCkgPT4ge1xuICBjb25zdCB0YXNrID0gbmV3IHNmbi5UYXNrKHN0YWNrLCAnVGFzaycsIHtcbiAgICB0YXNrOiBuZXcgdGFza3MuUnVuTGFtYmRhVGFzayhmbiwge1xuICAgICAgcGF5bG9hZDogc2ZuLlRhc2tJbnB1dC5mcm9tT2JqZWN0KHtcbiAgICAgICAgZm9vOiAnYmFyJyxcbiAgICAgIH0pLFxuICAgICAgaW52b2NhdGlvblR5cGU6IHRhc2tzLkludm9jYXRpb25UeXBlLlJFUVVFU1RfUkVTUE9OU0UsXG4gICAgICBjbGllbnRDb250ZXh0OiAnZXlKb1pXeHNieUk2SW5kdmNteGtJbjA9JyxcbiAgICAgIHF1YWxpZmllcjogJzEnLFxuICAgIH0pLFxuICB9KTtcbiAgbmV3IHNmbi5TdGF0ZU1hY2hpbmUoc3RhY2ssICdTTScsIHtcbiAgICBkZWZpbml0aW9uOiB0YXNrLFxuICB9KTtcblxuICBleHBlY3Qoc3RhY2sucmVzb2x2ZSh0YXNrLnRvU3RhdGVKc29uKCkpKS50b0VxdWFsKHtcbiAgICBUeXBlOiAnVGFzaycsXG4gICAgUmVzb3VyY2U6IHtcbiAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgJycsXG4gICAgICAgIFtcbiAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgJzpzdGF0ZXM6OjpsYW1iZGE6aW52b2tlJyxcbiAgICAgICAgXSxcbiAgICAgIF0sXG4gICAgfSxcbiAgICBFbmQ6IHRydWUsXG4gICAgUGFyYW1ldGVyczoge1xuICAgICAgRnVuY3Rpb25OYW1lOiB7XG4gICAgICAgIFJlZjogJ0ZuOTI3MENCQzAnLFxuICAgICAgfSxcbiAgICAgIFBheWxvYWQ6IHtcbiAgICAgICAgZm9vOiAnYmFyJyxcbiAgICAgIH0sXG4gICAgICBJbnZvY2F0aW9uVHlwZTogJ1JlcXVlc3RSZXNwb25zZScsXG4gICAgICBDbGllbnRDb250ZXh0OiAnZXlKb1pXeHNieUk2SW5kdmNteGtJbjA9JyxcbiAgICAgIFF1YWxpZmllcjogJzEnLFxuICAgIH0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ0xhbWJkYSBmdW5jdGlvbiBjYW4gYmUgdXNlZCBpbiBhIFRhc2sgd2l0aCBUYXNrIFRva2VuJywgKCkgPT4ge1xuICBjb25zdCB0YXNrID0gbmV3IHNmbi5UYXNrKHN0YWNrLCAnVGFzaycsIHtcbiAgICB0YXNrOiBuZXcgdGFza3MuUnVuTGFtYmRhVGFzayhmbiwge1xuICAgICAgaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5XQUlUX0ZPUl9UQVNLX1RPS0VOLFxuICAgICAgcGF5bG9hZDogc2ZuLlRhc2tJbnB1dC5mcm9tT2JqZWN0KHtcbiAgICAgICAgdG9rZW46IHNmbi5Db250ZXh0LnRhc2tUb2tlbixcbiAgICAgIH0pLFxuICAgIH0pLFxuICB9KTtcbiAgbmV3IHNmbi5TdGF0ZU1hY2hpbmUoc3RhY2ssICdTTScsIHtcbiAgICBkZWZpbml0aW9uOiB0YXNrLFxuICB9KTtcblxuICBleHBlY3Qoc3RhY2sucmVzb2x2ZSh0YXNrLnRvU3RhdGVKc29uKCkpKS50b0VxdWFsKHtcbiAgICBUeXBlOiAnVGFzaycsXG4gICAgUmVzb3VyY2U6IHtcbiAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgJycsXG4gICAgICAgIFtcbiAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgJzpzdGF0ZXM6OjpsYW1iZGE6aW52b2tlLndhaXRGb3JUYXNrVG9rZW4nLFxuICAgICAgICBdLFxuICAgICAgXSxcbiAgICB9LFxuICAgIEVuZDogdHJ1ZSxcbiAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICBGdW5jdGlvbk5hbWU6IHtcbiAgICAgICAgUmVmOiAnRm45MjcwQ0JDMCcsXG4gICAgICB9LFxuICAgICAgUGF5bG9hZDoge1xuICAgICAgICAndG9rZW4uJCc6ICckJC5UYXNrLlRva2VuJyxcbiAgICAgIH0sXG4gICAgfSxcbiAgfSk7XG59KTtcblxudGVzdCgnTGFtYmRhIGZ1bmN0aW9uIGlzIGludm9rZWQgd2l0aCB0aGUgc3RhdGUgaW5wdXQgYXMgcGF5bG9hZCBieSBkZWZhdWx0JywgKCkgPT4ge1xuICBjb25zdCB0YXNrID0gbmV3IHNmbi5UYXNrKHN0YWNrLCAnVGFzaycsIHtcbiAgICB0YXNrOiBuZXcgdGFza3MuUnVuTGFtYmRhVGFzayhmbiksXG4gIH0pO1xuICBuZXcgc2ZuLlN0YXRlTWFjaGluZShzdGFjaywgJ1NNJywge1xuICAgIGRlZmluaXRpb246IHRhc2ssXG4gIH0pO1xuXG4gIGV4cGVjdChzdGFjay5yZXNvbHZlKHRhc2sudG9TdGF0ZUpzb24oKSkpLnRvRXF1YWwoe1xuICAgIFR5cGU6ICdUYXNrJyxcbiAgICBSZXNvdXJjZToge1xuICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAnJyxcbiAgICAgICAgW1xuICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBSZWY6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgfSxcbiAgICAgICAgICAnOnN0YXRlczo6OmxhbWJkYTppbnZva2UnLFxuICAgICAgICBdLFxuICAgICAgXSxcbiAgICB9LFxuICAgIEVuZDogdHJ1ZSxcbiAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICAnRnVuY3Rpb25OYW1lJzoge1xuICAgICAgICBSZWY6ICdGbjkyNzBDQkMwJyxcbiAgICAgIH0sXG4gICAgICAnUGF5bG9hZC4kJzogJyQnLFxuICAgIH0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ0xhbWJkYSBmdW5jdGlvbiBjYW4gYmUgcHJvdmlkZWQgd2l0aCB0aGUgc3RhdGUgaW5wdXQgYXMgdGhlIHBheWxvYWQnLCAoKSA9PiB7XG4gIGNvbnN0IHRhc2sgPSBuZXcgc2ZuLlRhc2soc3RhY2ssICdUYXNrJywge1xuICAgIHRhc2s6IG5ldyB0YXNrcy5SdW5MYW1iZGFUYXNrKGZuLCB7XG4gICAgICBwYXlsb2FkOiBzZm4uVGFza0lucHV0LmZyb21EYXRhQXQoJyQnKSxcbiAgICB9KSxcbiAgfSk7XG4gIG5ldyBzZm4uU3RhdGVNYWNoaW5lKHN0YWNrLCAnU00nLCB7XG4gICAgZGVmaW5pdGlvbjogdGFzayxcbiAgfSk7XG5cbiAgZXhwZWN0KHN0YWNrLnJlc29sdmUodGFzay50b1N0YXRlSnNvbigpKSkudG9FcXVhbCh7XG4gICAgVHlwZTogJ1Rhc2snLFxuICAgIFJlc291cmNlOiB7XG4gICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICcnLFxuICAgICAgICBbXG4gICAgICAgICAgJ2FybjonLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFJlZjogJ0FXUzo6UGFydGl0aW9uJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgICc6c3RhdGVzOjo6bGFtYmRhOmludm9rZScsXG4gICAgICAgIF0sXG4gICAgICBdLFxuICAgIH0sXG4gICAgRW5kOiB0cnVlLFxuICAgIFBhcmFtZXRlcnM6IHtcbiAgICAgICdGdW5jdGlvbk5hbWUnOiB7XG4gICAgICAgIFJlZjogJ0ZuOTI3MENCQzAnLFxuICAgICAgfSxcbiAgICAgICdQYXlsb2FkLiQnOiAnJCcsXG4gICAgfSxcbiAgfSk7XG59KTtcblxudGVzdCgnVGFzayB0aHJvd3MgaWYgV0FJVF9GT1JfVEFTS19UT0tFTiBpcyBzdXBwbGllZCBidXQgdGFzayB0b2tlbiBpcyBub3QgaW5jbHVkZWQgaW4gcGF5TG9hZCcsICgpID0+IHtcbiAgZXhwZWN0KCgpID0+IHtcbiAgICBuZXcgc2ZuLlRhc2soc3RhY2ssICdUYXNrJywge1xuICAgICAgdGFzazogbmV3IHRhc2tzLlJ1bkxhbWJkYVRhc2soZm4sIHtcbiAgICAgICAgaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uU2VydmljZUludGVncmF0aW9uUGF0dGVybi5XQUlUX0ZPUl9UQVNLX1RPS0VOLFxuICAgICAgfSksXG4gICAgfSk7XG4gIH0pLnRvVGhyb3coL1Rhc2sgVG9rZW4gaXMgbWlzc2luZyBpbiBwYXlsb2FkL2kpO1xufSk7XG5cbnRlc3QoJ1Rhc2sgdGhyb3dzIGlmIFNZTkMgaXMgc3VwcGxpZWQgYXMgc2VydmljZSBpbnRlZ3JhdGlvbiBwYXR0ZXJuJywgKCkgPT4ge1xuICBleHBlY3QoKCkgPT4ge1xuICAgIG5ldyBzZm4uVGFzayhzdGFjaywgJ1Rhc2snLCB7XG4gICAgICB0YXNrOiBuZXcgdGFza3MuUnVuTGFtYmRhVGFzayhmbiwge1xuICAgICAgICBpbnRlZ3JhdGlvblBhdHRlcm46IHNmbi5TZXJ2aWNlSW50ZWdyYXRpb25QYXR0ZXJuLlNZTkMsXG4gICAgICB9KSxcbiAgICB9KTtcbiAgfSkudG9UaHJvdygvSW52YWxpZCBTZXJ2aWNlIEludGVncmF0aW9uIFBhdHRlcm46IFNZTkMgaXMgbm90IHN1cHBvcnRlZCB0byBjYWxsIExhbWJkYS4vaSk7XG59KTtcbiJdfQ==