"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("./resource-arn-suffix");
/**
 * A Step Functions Task to call StartExecution on another state machine.
 *
 * It supports three service integration patterns: FIRE_AND_FORGET, SYNC and WAIT_FOR_TASK_TOKEN.
 */
class StartExecution {
    constructor(stateMachine, props = {}) {
        this.stateMachine = stateMachine;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call Step Functions.`);
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is missing in input (pass Context.taskToken somewhere in input)');
        }
    }
    bind(task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('states', 'startExecution', this.integrationPattern),
            policyStatements: this.createScopedAccessPolicy(task),
            parameters: {
                Input: this.props.input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            },
        };
    }
    /**
     * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
     *
     * the scoped access policy should be generated accordingly.
     *
     * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
     */
    createScopedAccessPolicy(task) {
        const stack = core_1.Stack.of(task);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['states:DescribeExecution', 'states:StopExecution'],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        sep: ':',
                        resourceName: `${stack.parseArn(this.stateMachine.stateMachineArn, ':').resourceName}*`,
                    })],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    })],
            }));
        }
        return policyStatements;
    }
}
exports.StartExecution = StartExecution;
//# sourceMappingURL=data:application/json;base64,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