"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * Determines the level of detail about provisioned throughput consumption that is returned.
 */
var DynamoConsumedCapacity;
(function (DynamoConsumedCapacity) {
    /**
     * The response includes the aggregate ConsumedCapacity for the operation,
     * together with ConsumedCapacity for each table and secondary index that was accessed
     */
    DynamoConsumedCapacity["INDEXES"] = "INDEXES";
    /**
     * The response includes only the aggregate ConsumedCapacity for the operation.
     */
    DynamoConsumedCapacity["TOTAL"] = "TOTAL";
    /**
     * No ConsumedCapacity details are included in the response.
     */
    DynamoConsumedCapacity["NONE"] = "NONE";
})(DynamoConsumedCapacity = exports.DynamoConsumedCapacity || (exports.DynamoConsumedCapacity = {}));
/**
 * Determines whether item collection metrics are returned.
 */
var DynamoItemCollectionMetrics;
(function (DynamoItemCollectionMetrics) {
    /**
     * If set to SIZE, the response includes statistics about item collections,
     * if any, that were modified during the operation.
     */
    DynamoItemCollectionMetrics["SIZE"] = "SIZE";
    /**
     * If set to NONE, no statistics are returned.
     */
    DynamoItemCollectionMetrics["NONE"] = "NONE";
})(DynamoItemCollectionMetrics = exports.DynamoItemCollectionMetrics || (exports.DynamoItemCollectionMetrics = {}));
/**
 * Use ReturnValues if you want to get the item attributes as they appear before or after they are changed
 */
var DynamoReturnValues;
(function (DynamoReturnValues) {
    /**
     * Nothing is returned
     */
    DynamoReturnValues["NONE"] = "NONE";
    /**
     * Returns all of the attributes of the item
     */
    DynamoReturnValues["ALL_OLD"] = "ALL_OLD";
    /**
     * Returns only the updated attributes
     */
    DynamoReturnValues["UPDATED_OLD"] = "UPDATED_OLD";
    /**
     * Returns all of the attributes of the item
     */
    DynamoReturnValues["ALL_NEW"] = "ALL_NEW";
    /**
     * Returns only the updated attributes
     */
    DynamoReturnValues["UPDATED_NEW"] = "UPDATED_NEW";
})(DynamoReturnValues = exports.DynamoReturnValues || (exports.DynamoReturnValues = {}));
/**
 * Class to generate AttributeValue
 */
class DynamoAttributeValue {
    constructor() {
        this.attributeValue = {};
    }
    /**
     * Sets an attribute of type String. For example:  "S": "Hello"
     */
    withS(value) {
        this.attributeValue.S = value;
        return this;
    }
    /**
     * Sets an attribute of type Number. For example:  "N": "123.45"
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    withN(value) {
        this.attributeValue.N = value;
        return this;
    }
    /**
     * Sets an attribute of type Binary. For example:  "B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"
     */
    withB(value) {
        this.attributeValue.B = value;
        return this;
    }
    /**
     * Sets an attribute of type String Set. For example:  "SS": ["Giraffe", "Hippo" ,"Zebra"]
     */
    withSS(value) {
        this.attributeValue.SS = value;
        return this;
    }
    /**
     * Sets an attribute of type Number Set. For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    withNS(value) {
        this.attributeValue.NS = value;
        return this;
    }
    /**
     * Sets an attribute of type Binary Set. For example:  "BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]
     */
    withBS(value) {
        this.attributeValue.BS = value;
        return this;
    }
    /**
     * Sets an attribute of type Map. For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     */
    withM(value) {
        this.attributeValue.M = transformAttributeValueMap(value);
        return this;
    }
    /**
     * Sets an attribute of type List. For example:  "L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N", "3.14159"}]
     */
    withL(value) {
        this.attributeValue.L = value.map(val => val.toObject());
        return this;
    }
    /**
     * Sets an attribute of type Null. For example:  "NULL": true
     */
    withNULL(value) {
        this.attributeValue.NULL = value;
        return this;
    }
    /**
     * Sets an attribute of type Boolean. For example:  "BOOL": true
     */
    withBOOL(value) {
        this.attributeValue.BOOL = value;
        return this;
    }
    /**
     * Return the attributeValue object
     */
    toObject() {
        return this.attributeValue;
    }
}
exports.DynamoAttributeValue = DynamoAttributeValue;
/**
 * Class to generate projection expression
 */
class DynamoProjectionExpression {
    constructor() {
        this.expression = [];
    }
    /**
     * Adds the passed attribute to the chain
     *
     * @param attr Attribute name
     */
    withAttribute(attr) {
        if (this.expression.length) {
            this.expression.push(`.${attr}`);
        }
        else {
            this.expression.push(attr);
        }
        return this;
    }
    /**
     * Adds the array literal access for passed index
     *
     * @param index array index
     */
    atIndex(index) {
        if (!this.expression.length) {
            throw new Error('Expression must start with an attribute');
        }
        this.expression.push(`[${index}]`);
        return this;
    }
    /**
     * converts and return the string expression
     */
    toString() {
        return this.expression.join('');
    }
}
exports.DynamoProjectionExpression = DynamoProjectionExpression;
/**
 * A StepFunctions task to call DynamoGetItem
 */
class DynamoGetItem {
    constructor(props) {
        this.props = props;
        core_1.withResolved(props.tableName, validateTableName);
    }
    bind(_task) {
        var _a;
        return {
            resourceArn: getDynamoResourceArn(DynamoMethod.GET),
            policyStatements: getDynamoPolicyStatements(_task, this.props.tableName, DynamoMethod.GET),
            parameters: {
                Key: configurePrimaryKey(this.props.partitionKey, this.props.sortKey),
                TableName: this.props.tableName,
                ConsistentRead: (_a = this.props.consistentRead) !== null && _a !== void 0 ? _a : false,
                ExpressionAttributeNames: this.props.expressionAttributeNames,
                ProjectionExpression: this.configureProjectionExpression(this.props.projectionExpression),
                ReturnConsumedCapacity: this.props.returnConsumedCapacity,
            },
        };
    }
    configureProjectionExpression(expressions) {
        return expressions
            ? expressions.map(expression => expression.toString()).join(',')
            : undefined;
    }
}
exports.DynamoGetItem = DynamoGetItem;
/**
 * A StepFunctions task to call DynamoPutItem
 */
class DynamoPutItem {
    constructor(props) {
        this.props = props;
        core_1.withResolved(props.tableName, validateTableName);
    }
    bind(_task) {
        return {
            resourceArn: getDynamoResourceArn(DynamoMethod.PUT),
            policyStatements: getDynamoPolicyStatements(_task, this.props.tableName, DynamoMethod.PUT),
            parameters: {
                Item: transformAttributeValueMap(this.props.item),
                TableName: this.props.tableName,
                ConditionExpression: this.props.conditionExpression,
                ExpressionAttributeNames: this.props.expressionAttributeNames,
                ExpressionAttributeValues: transformAttributeValueMap(this.props.expressionAttributeValues),
                ReturnConsumedCapacity: this.props.returnConsumedCapacity,
                ReturnItemCollectionMetrics: this.props.returnItemCollectionMetrics,
                ReturnValues: this.props.returnValues,
            },
        };
    }
}
exports.DynamoPutItem = DynamoPutItem;
/**
 * A StepFunctions task to call DynamoDeleteItem
 */
class DynamoDeleteItem {
    constructor(props) {
        this.props = props;
        core_1.withResolved(props.tableName, validateTableName);
    }
    bind(_task) {
        return {
            resourceArn: getDynamoResourceArn(DynamoMethod.DELETE),
            policyStatements: getDynamoPolicyStatements(_task, this.props.tableName, DynamoMethod.DELETE),
            parameters: {
                Key: configurePrimaryKey(this.props.partitionKey, this.props.sortKey),
                TableName: this.props.tableName,
                ConditionExpression: this.props.conditionExpression,
                ExpressionAttributeNames: this.props.expressionAttributeNames,
                ExpressionAttributeValues: transformAttributeValueMap(this.props.expressionAttributeValues),
                ReturnConsumedCapacity: this.props.returnConsumedCapacity,
                ReturnItemCollectionMetrics: this.props.returnItemCollectionMetrics,
                ReturnValues: this.props.returnValues,
            },
        };
    }
}
exports.DynamoDeleteItem = DynamoDeleteItem;
/**
 * A StepFunctions task to call DynamoUpdateItem
 */
class DynamoUpdateItem {
    constructor(props) {
        this.props = props;
        core_1.withResolved(props.tableName, validateTableName);
    }
    bind(_task) {
        return {
            resourceArn: getDynamoResourceArn(DynamoMethod.UPDATE),
            policyStatements: getDynamoPolicyStatements(_task, this.props.tableName, DynamoMethod.UPDATE),
            parameters: {
                Key: configurePrimaryKey(this.props.partitionKey, this.props.sortKey),
                TableName: this.props.tableName,
                ConditionExpression: this.props.conditionExpression,
                ExpressionAttributeNames: this.props.expressionAttributeNames,
                ExpressionAttributeValues: transformAttributeValueMap(this.props.expressionAttributeValues),
                ReturnConsumedCapacity: this.props.returnConsumedCapacity,
                ReturnItemCollectionMetrics: this.props.returnItemCollectionMetrics,
                ReturnValues: this.props.returnValues,
                UpdateExpression: this.props.updateExpression,
            },
        };
    }
}
exports.DynamoUpdateItem = DynamoUpdateItem;
/**
 * A helper wrapper class to call all DynamoDB APIs
 */
class CallDynamoDB {
    /**
     * Method to get DynamoGetItem task
     *
     * @param props DynamoGetItemProps
     */
    static getItem(props) {
        return new DynamoGetItem(props);
    }
    /**
     * Method to get DynamoPutItem task
     *
     * @param props DynamoPutItemProps
     */
    static putItem(props) {
        return new DynamoPutItem(props);
    }
    /**
     * Method to get DynamoDeleteItem task
     *
     * @param props DynamoDeleteItemProps
     */
    static deleteItem(props) {
        return new DynamoDeleteItem(props);
    }
    /**
     * Method to get DynamoUpdateItem task
     *
     * @param props DynamoUpdateItemProps
     */
    static updateItem(props) {
        return new DynamoUpdateItem(props);
    }
}
exports.CallDynamoDB = CallDynamoDB;
var DynamoMethod;
(function (DynamoMethod) {
    DynamoMethod["GET"] = "Get";
    DynamoMethod["PUT"] = "Put";
    DynamoMethod["DELETE"] = "Delete";
    DynamoMethod["UPDATE"] = "Update";
})(DynamoMethod || (DynamoMethod = {}));
function validateTableName(tableName) {
    if (tableName.length < 3 ||
        tableName.length > 255 ||
        !new RegExp(/[a-zA-Z0-9_.-]+$/).test(tableName)) {
        throw new Error(`TableName should not contain alphanumeric characters and should be between 3-255 characters long. Received: ${tableName}`);
    }
}
function getDynamoResourceArn(method) {
    return resource_arn_suffix_1.getResourceArn('dynamodb', `${method.toLowerCase()}Item`, sfn.ServiceIntegrationPattern.FIRE_AND_FORGET);
}
function getDynamoPolicyStatements(task, tableName, method) {
    return [
        new iam.PolicyStatement({
            resources: [
                core_1.Stack.of(task).formatArn({
                    service: 'dynamodb',
                    resource: 'table',
                    resourceName: tableName,
                }),
            ],
            actions: [`dynamodb:${method}Item`],
        }),
    ];
}
function configurePrimaryKey(partitionKey, sortKey) {
    const key = {
        [partitionKey.name]: partitionKey.value.toObject(),
    };
    if (sortKey) {
        key[sortKey.name] = sortKey.value.toObject();
    }
    return key;
}
function transformAttributeValueMap(attrMap) {
    const transformedValue = {};
    for (const key in attrMap) {
        if (key) {
            transformedValue[key] = attrMap[key].toObject();
        }
    }
    return attrMap ? transformedValue : undefined;
}
//# sourceMappingURL=data:application/json;base64,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