import datetime
import logging
import string
import urllib

import requests
import time
import hmac
import hashlib
from .configuration import RTPPlayConfig

logger = logging.getLogger()


class RTPPlayAPI:
    def __init__(self):
        self.config = self.__get_config()
        self.AUTH_TOKEN = ""
        self.AUTH_TOKEN_LIFETIME = 0

    def __get_config(self):
        r = requests.get("https://www.rtp.pt/common/config/apps/RTPPlay/config_mobile.json").json()
        return RTPPlayConfig(r)

    def __get_key(self, current_time_millis):
        RTP_PLAY_AUTH_KEY = "bdaafc5451445b65a10dc615e63d7ffe"
        key = ""
        i = 0
        for j in range(len(RTP_PLAY_AUTH_KEY)):
            if j % 5 == 3:
                key += current_time_millis[i]
                i += 1
            key += RTP_PLAY_AUTH_KEY[j]
        return key

    def __encode(self, key, msg):
        # this took me longer to reverse than I'd like to admit...
        signature = hmac.new(key.encode("US_ASCII"), msg.encode("US_ASCII"), hashlib.sha256).digest()
        encoded = ""
        digits = string.digits + string.ascii_lowercase
        radix = 16
        for b in signature:
            i = (int(b) & 0xff) + 0x100
            buf = ""
            while i >= radix:
                buf += digits[i % radix]
                i = i // radix
            buf += digits[i]
            encoded += buf[-2::-1]
        return encoded

    def __set_token(self):
        current_time_millis = str(round(time.time() * 1000))
        key = self.__get_key(current_time_millis)
        msg = current_time_millis + self.config.AUTH_URL_ENDPOINT
        headers = {'RTP-Play-Auth': 'RTPPLAY_MOBILE_ANDROID',
                   'RTP-Play-Auth-Hash': self.__encode(key, msg),
                   'RTP-Play-Auth-Timestamp': current_time_millis}
        logger.debug("Token headers:", headers)
        r = requests.post(self.config.AUTH_URL_ENDPOINT, headers=headers).json()
        if "error" not in r:
            self.AUTH_TOKEN = r["token"]["token"]
            self.AUTH_TOKEN_LIFETIME = r["token"]["expire"] * 1000
            return
        logger.error("Auth error:", r["error"])

    def __request(self, url, retry=True):
        current_time_millis = round(time.time() * 1000)
        if not self.config or not self.config.AUTH_URL_ENDPOINT:
            self.config = self.__get_config()
        if not self.AUTH_TOKEN or self.AUTH_TOKEN_LIFETIME - current_time_millis < 30000:
            self.__set_token()
        if not self.AUTH_TOKEN or self.AUTH_TOKEN_LIFETIME <= 0:
            # unable to fetch token
            return
        headers = {'Authorization': 'Bearer ' + self.AUTH_TOKEN,
                   'RTP-Play-Auth-Timestamp': str(current_time_millis)}
        logger.debug(url)
        r = requests.get(self.config.BASE_API_URL_ENDPOINT + url, headers=headers).json()
        if "error" not in r or r["error"] != "Not Authenticated" or not retry:
            return r["result"]
        self.AUTH_TOKEN = ""
        self.AUTH_TOKEN_LIFETIME = 0
        return self.__request(url, False)

    def get_live_tv_channels(self):
        """
            Example: [{'channel_id': '5', 'channel_name': 'RTP1', 'channel_summary': 'Emissão em direto RTP1\xa0com o melhor entretenimento, ficção nacional e internacional, informação, desporto e muito mais. Conteúdos para toda a família, disponível em várias plataformas.', 'channel_console_title': 'RTP1', 'channel_console_summary': '', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtp1.png?fm=pn32&w=300', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/EPG/tv/epg-dia.php?canal=1&ac=d&sem=e', 'channel_card_logo': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/5-563718101410.png', 'channel_card_logo_1x1': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/5-065319091505.png', 'channel_card_logo_9x16': '', 'channel_list_order': '1', 'channel_color': '#1072fc', 'channel_live': '1', 'channel_ondemand': '1', 'channel_rewrite': 'rtp1', 'channel_onairlive_site': '', 'channel_epg': '1'}, {'channel_id': '3', 'channel_name': 'RTP2', 'channel_summary': 'Emissão em direto RTP2\xa0com o melhor da cultura, informação, artes e\xa0espetáculos, desporto, conteúdos para os mais novos e muito mais. Disponível em várias plataformas.', 'channel_console_title': 'Titulo RTP2', 'channel_console_summary': 'Descrição RTP2', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtp2.png?fm=pn32&w=300', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/rtp2/', 'channel_card_logo': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/3-363718101410.png', 'channel_card_logo_1x1': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/3-014819141705.png', 'channel_card_logo_9x16': '', 'channel_list_order': '2', 'channel_color': '#f9f910', 'channel_live': '1', 'channel_ondemand': '1', 'channel_rewrite': 'rtp2', 'channel_onairlive_site': '', 'channel_epg': '1'}, {'channel_id': '64', 'channel_name': 'RTP3', 'channel_summary': 'Emissão em direto RTP3 com toda a informação nacional e internacional, análise, política, economia, ciência, desporto e sociedade.  Informação de confiança disponível em várias plataformas.', 'channel_console_title': '', 'channel_console_summary': '', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtp3.png?fm=pn32&w=300', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/rtp3/', 'channel_card_logo': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/64-393818101410.png', 'channel_card_logo_1x1': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/64-174819141705.png', 'channel_card_logo_9x16': '', 'channel_list_order': '3', 'channel_color': '#0037e6', 'channel_live': '1', 'channel_ondemand': '1', 'channel_rewrite': 'rtp3', 'channel_onairlive_site': '', 'channel_epg': '1'}, {'channel_id': '120', 'channel_name': 'RTP Internacional', 'channel_summary': 'Emissão em direto RTP Internacional com a melhor informação, entretenimento e ficção para que se mantenha ligado a Portugal. Disponível em múltiplas plataformas.', 'channel_console_title': '', 'channel_console_summary': '', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtpinternacional.png?fm=pn32&w=300', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/rtpi/', 'channel_card_logo': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/120-344318101410.png', 'channel_card_logo_1x1': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/120-233619141305.png', 'channel_card_logo_9x16': '', 'channel_list_order': '10', 'channel_color': '#0202af', 'channel_live': '1', 'channel_ondemand': '1', 'channel_rewrite': 'rtpinternacional', 'channel_onairlive_site': '', 'channel_epg': '1'}, {'channel_id': '80', 'channel_name': 'RTP Memória', 'channel_summary': "Emissão em direto RTP Memória com uma seleção de programas e marcos históricos da televisão que traz p'rá frente os melhores conteúdos. Disponível em múltiplas plataformas.", 'channel_console_title': '', 'channel_console_summary': '', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtpmemoria.png?fm=pn32&w=300', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/rtpmemoria/', 'channel_card_logo': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/80-584819141705.png', 'channel_card_logo_1x1': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/80-344819141705.png', 'channel_card_logo_9x16': '', 'channel_list_order': '12', 'channel_color': '#57666e', 'channel_live': '1', 'channel_ondemand': '1', 'channel_rewrite': 'rtpmemoria', 'channel_onairlive_site': '', 'channel_epg': '1'}, {'channel_id': '107', 'channel_name': 'RTP Madeira', 'channel_summary': 'Emissão em direto\xa0RTP Madeira com o melhor da informação, reportagem, desporto, debate, cultura, educação e entretenimento. Disponível em múltiplas plataformas.', 'channel_console_title': '', 'channel_console_summary': '', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtpmadeira.png?fm=pn32&w=300', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/rtpmadeira', 'channel_card_logo': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/107-443519141305.png', 'channel_card_logo_1x1': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/107-523519141305.png', 'channel_card_logo_9x16': '', 'channel_list_order': '14', 'channel_color': '#cc33cc', 'channel_live': '1', 'channel_ondemand': '1', 'channel_rewrite': 'rtpmadeira', 'channel_onairlive_site': '', 'channel_epg': '1'}, {'channel_id': '106', 'channel_name': 'RTP Açores', 'channel_summary': 'Emissão em direto RTP Açores com o melhor da informação, reportagem, desporto, debate, cultura, educação e entretenimento. Disponível em múltiplas plataformas.', 'channel_console_title': '', 'channel_console_summary': '', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtpacores.png?fm=pn32&w=300', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/acores/', 'channel_card_logo': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/106-563419141305.png', 'channel_card_logo_1x1': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/106-083519141305.png', 'channel_card_logo_9x16': '', 'channel_list_order': '16', 'channel_color': '#55af31', 'channel_live': '1', 'channel_ondemand': '1', 'channel_rewrite': 'rtpacores', 'channel_onairlive_site': '', 'channel_epg': '1'}, {'channel_id': '27', 'channel_name': 'RTP África', 'channel_summary': 'Emissão em direto RTP África com as notícias que\xa0marcam a atualidade no continente Africano, música, cultura, entretenimento e muito mais. Disponível em múltiplas plataformas.', 'channel_console_title': '', 'channel_console_summary': '', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtpafrica.png?fm=pn32&w=300', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/EPG/tv/epg-dia.php?canal=6&ac=d&sem=e', 'channel_card_logo': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/27-363219141305.png', 'channel_card_logo_1x1': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/27-173219141305.png', 'channel_card_logo_9x16': '', 'channel_list_order': '18', 'channel_color': '#ef7f17', 'channel_live': '1', 'channel_ondemand': '1', 'channel_rewrite': 'rtpafrica', 'channel_onairlive_site': '', 'channel_epg': '1'}]
        :return:
        """
        return self.__request("list-channels?broadcaster=tvlinear&live=true")

    def get_live_radio_channels(self):
        """
            Example: [{'channel_id': '5', 'channel_name': 'RTP1', 'channel_summary': 'Emissão em direto RTP1\xa0com o melhor entretenimento, ficção nacional e internacional, informação, desporto e muito mais. Conteúdos para toda a família, disponível em várias plataformas.', 'channel_console_title': 'RTP1', 'channel_console_summary': '', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtp1.png?fm=pn32&w=300', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/EPG/tv/epg-dia.php?canal=1&ac=d&sem=e', 'channel_card_logo': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/5-563718101410.png', 'channel_card_logo_1x1': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/5-065319091505.png', 'channel_card_logo_9x16': '', 'channel_list_order': '1', 'channel_color': '#1072fc', 'channel_live': '1', 'channel_ondemand': '1', 'channel_rewrite': 'rtp1', 'channel_onairlive_site': '', 'channel_epg': '1'}, {'channel_id': '3', 'channel_name': 'RTP2', 'channel_summary': 'Emissão em direto RTP2\xa0com o melhor da cultura, informação, artes e\xa0espetáculos, desporto, conteúdos para os mais novos e muito mais. Disponível em várias plataformas.', 'channel_console_title': 'Titulo RTP2', 'channel_console_summary': 'Descrição RTP2', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtp2.png?fm=pn32&w=300', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/rtp2/', 'channel_card_logo': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/3-363718101410.png', 'channel_card_logo_1x1': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/3-014819141705.png', 'channel_card_logo_9x16': '', 'channel_list_order': '2', 'channel_color': '#f9f910', 'channel_live': '1', 'channel_ondemand': '1', 'channel_rewrite': 'rtp2', 'channel_onairlive_site': '', 'channel_epg': '1'}, {'channel_id': '64', 'channel_name': 'RTP3', 'channel_summary': 'Emissão em direto RTP3 com toda a informação nacional e internacional, análise, política, economia, ciência, desporto e sociedade.  Informação de confiança disponível em várias plataformas.', 'channel_console_title': '', 'channel_console_summary': '', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtp3.png?fm=pn32&w=300', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/rtp3/', 'channel_card_logo': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/64-393818101410.png', 'channel_card_logo_1x1': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/64-174819141705.png', 'channel_card_logo_9x16': '', 'channel_list_order': '3', 'channel_color': '#0037e6', 'channel_live': '1', 'channel_ondemand': '1', 'channel_rewrite': 'rtp3', 'channel_onairlive_site': '', 'channel_epg': '1'}, {'channel_id': '120', 'channel_name': 'RTP Internacional', 'channel_summary': 'Emissão em direto RTP Internacional com a melhor informação, entretenimento e ficção para que se mantenha ligado a Portugal. Disponível em múltiplas plataformas.', 'channel_console_title': '', 'channel_console_summary': '', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtpinternacional.png?fm=pn32&w=300', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/rtpi/', 'channel_card_logo': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/120-344318101410.png', 'channel_card_logo_1x1': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/120-233619141305.png', 'channel_card_logo_9x16': '', 'channel_list_order': '10', 'channel_color': '#0202af', 'channel_live': '1', 'channel_ondemand': '1', 'channel_rewrite': 'rtpinternacional', 'channel_onairlive_site': '', 'channel_epg': '1'}, {'channel_id': '80', 'channel_name': 'RTP Memória', 'channel_summary': "Emissão em direto RTP Memória com uma seleção de programas e marcos históricos da televisão que traz p'rá frente os melhores conteúdos. Disponível em múltiplas plataformas.", 'channel_console_title': '', 'channel_console_summary': '', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtpmemoria.png?fm=pn32&w=300', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/rtpmemoria/', 'channel_card_logo': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/80-584819141705.png', 'channel_card_logo_1x1': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/80-344819141705.png', 'channel_card_logo_9x16': '', 'channel_list_order': '12', 'channel_color': '#57666e', 'channel_live': '1', 'channel_ondemand': '1', 'channel_rewrite': 'rtpmemoria', 'channel_onairlive_site': '', 'channel_epg': '1'}, {'channel_id': '107', 'channel_name': 'RTP Madeira', 'channel_summary': 'Emissão em direto\xa0RTP Madeira com o melhor da informação, reportagem, desporto, debate, cultura, educação e entretenimento. Disponível em múltiplas plataformas.', 'channel_console_title': '', 'channel_console_summary': '', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtpmadeira.png?fm=pn32&w=300', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/rtpmadeira', 'channel_card_logo': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/107-443519141305.png', 'channel_card_logo_1x1': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/107-523519141305.png', 'channel_card_logo_9x16': '', 'channel_list_order': '14', 'channel_color': '#cc33cc', 'channel_live': '1', 'channel_ondemand': '1', 'channel_rewrite': 'rtpmadeira', 'channel_onairlive_site': '', 'channel_epg': '1'}, {'channel_id': '106', 'channel_name': 'RTP Açores', 'channel_summary': 'Emissão em direto RTP Açores com o melhor da informação, reportagem, desporto, debate, cultura, educação e entretenimento. Disponível em múltiplas plataformas.', 'channel_console_title': '', 'channel_console_summary': '', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtpacores.png?fm=pn32&w=300', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/acores/', 'channel_card_logo': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/106-563419141305.png', 'channel_card_logo_1x1': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/106-083519141305.png', 'channel_card_logo_9x16': '', 'channel_list_order': '16', 'channel_color': '#55af31', 'channel_live': '1', 'channel_ondemand': '1', 'channel_rewrite': 'rtpacores', 'channel_onairlive_site': '', 'channel_epg': '1'}, {'channel_id': '27', 'channel_name': 'RTP África', 'channel_summary': 'Emissão em direto RTP África com as notícias que\xa0marcam a atualidade no continente Africano, música, cultura, entretenimento e muito mais. Disponível em múltiplas plataformas.', 'channel_console_title': '', 'channel_console_summary': '', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtpafrica.png?fm=pn32&w=300', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/EPG/tv/epg-dia.php?canal=6&ac=d&sem=e', 'channel_card_logo': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/27-363219141305.png', 'channel_card_logo_1x1': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/27-173219141305.png', 'channel_card_logo_9x16': '', 'channel_list_order': '18', 'channel_color': '#ef7f17', 'channel_live': '1', 'channel_ondemand': '1', 'channel_rewrite': 'rtpafrica', 'channel_onairlive_site': '', 'channel_epg': '1'}]
        :return:
        """
        return self.__request("list-channels?broadcaster=tvlinear&live=true")

    def get_slideshow(self):
        """
            Example: {'total': 5, 'slideshow': [{'channel_id': '', 'program_id': '8617', 'episode_id': '', 'type': 'tv', 'title': 'Não Te Esqueças da Letra!', 'description': 'O novo concurso onde a música é a rainha da noite!', 'image_desktop': 'https://cdn-images.rtp.pt/icm/destaques/images/9e/9e3054906eefdd5df7c1d0c5acac7b23?1500&amp;rect=2&amp;sy=0,1997,1125', 'image_app': 'https://cdn-images.rtp.pt/icm/destaques/images/eb/ebccdfeda7bba364fabc3dbea3010802?1024&amp;rect=0&amp;sy=0,750,1000', 'link': 'https://www.rtp.pt/play/p8617/nao-te-esquecas-da-letra', 'video_download': '', 'card_logo': '', 'image_wide': ''}, {'channel_id': '', 'program_id': '8589', 'episode_id': '', 'type': 'tv', 'title': 'Conspiração Terrorista', 'description': "Realizado por Michael Apted, 'Conspiração Terrorista' conta com John Malkovich, Noomi Rapace, Orlando Bloom, Michael Douglas e Toni Collette nos papéis principais.", 'image_desktop': 'https://cdn-images.rtp.pt/icm/destaques/images/b7/b7acc38a44214036feedb29db02ef940?1500&amp;rect=2&amp;sy=0,1997,1125', 'image_app': 'https://cdn-images.rtp.pt/icm/destaques/images/f0/f0332b12982df5184ac82f9c849eb238?1024&amp;rect=0&amp;sy=0,750,1000', 'link': 'https://www.rtp.pt/play/p8589/conspiracao-terrorista', 'video_download': '', 'card_logo': '', 'image_wide': ''}, {'channel_id': '', 'program_id': '8499', 'episode_id': '', 'type': 'tv', 'title': 'Gigantes', 'description': '2ª temporada da série espanhola que conta uma história de amor e ódio fraternal no submundo do crime de Madrid.', 'image_desktop': 'https://cdn-images.rtp.pt/icm/destaques/images/7f/7fcae13d74aacf2b1926832e9fac2a78?1500&amp;rect=2&amp;sy=0,1997,1125', 'image_app': 'https://cdn-images.rtp.pt/icm/destaques/images/da/da9b393b969a8296417c224debc61670?1024&amp;rect=0&amp;sy=0,750,1000', 'link': 'https://www.rtp.pt/play/p8499/gigantes', 'video_download': 'https://cdn-ondemand.rtp.pt/nas2.share/mcm/mp4/03d/03dcd2955d45bf5b136780b7d9155ce8_2mbrtp.mp4', 'card_logo': 'https://cdn-images.rtp.pt/icm/destaques/images/53/53590dc26f23cc003e48e47ea476fc77?1024&amp;rect=0&amp;sy=0,3682,1752', 'image_wide': ''}, {'channel_id': '', 'program_id': '8511', 'episode_id': '', 'type': 'tv', 'title': 'O Sítio da Mulher Morta', 'description': 'Adaptação cinematográfica do conto homónimo de Manuel Teixeira Gomes.', 'image_desktop': 'https://cdn-images.rtp.pt/icm/destaques/images/26/2679c50a093acfb464f9e3b46afa62f5?1500&amp;rect=2&amp;sy=0,1997,1125', 'image_app': 'https://cdn-images.rtp.pt/icm/destaques/images/6f/6f0b82bf9e86a9d8e28d00bc22bd1237?1024&amp;rect=0&amp;sy=0,750,1000', 'link': 'https://www.rtp.pt/play/p8511/o-sitio-da-mulher-morta', 'video_download': '', 'card_logo': '', 'image_wide': ''}, {'channel_id': '', 'program_id': '8601', 'episode_id': '', 'type': 'tv', 'title': 'Ottilie Von Faber-Castell - Uma Mulher Ousada', 'description': 'Uma jovem brilhante, herdeira da famosa marca de lápis, enfrenta o mundo dos negócios, da família e do amor proibido. Filme em 2 partes da cineasta alemã Claudia Garde.', 'image_desktop': 'https://cdn-images.rtp.pt/icm/destaques/images/5e/5e97f7cf78f0ab24677663864392d5ba?1500&amp;rect=2&amp;sy=0,1997,1125', 'image_app': 'https://cdn-images.rtp.pt/icm/destaques/images/3b/3b7056d53916a59f18ae289bddeb1c3d?1024&amp;rect=0&amp;sy=0,750,1000', 'link': 'https://www.rtp.pt/play/p8601/ottilie-von-faber-castell', 'video_download': '', 'card_logo': '', 'image_wide': ''}]}
        :return:
        """
        return self.__request("extends-get-slideshow")

    def get_collection(self, collection_id):
        """
            Example: {'name': '64 Anos de RTP', 'total': 12, 'collection_id': '178', 'collection_name': '64 Anos de RTP', 'collection_total': 12, 'collection_type': '', 'collection_description': '', 'collection_video': '', 'collection_rewrite': '64-anos-de-rtp', 'collection_key': 'collection_178', 'collection_category': '', 'collection_image': '', 'programs': [{'channel_id': '5', 'channel_name': 'RTP1', 'channel_type': 'tvlinear', 'program_id': '8505', 'program_is_serie': '2', 'program_rewrite': 'festival-da-cancao-2021', 'program_title': 'Festival da Canção 2021', 'program_lead': 'O Festival da Canção 2021 é constituído por duas semifinais e pela grande final, marcada para 6 de março.<br />A edição de 2021 do Festival da Canção retoma em traços gerais o modelo dos anos anteriores e contará com 20 canções a concurso. Dessas, 18 resultam de convites diretos da RTP, uma vez mais tendo em conta a representação da diversidade de géneros musicais e artistas com obra pública. A novidade está nas restantes entradas, com a seleção de dois autores através da livre submissão de canções.', 'program_promotion': 'Quem irá representar Portugal no Festival Eurovisão da Canção 2021, em Roterdão?', 'program_image': 'https://cdn-images.rtp.pt/EPG/imagens/40289_61049_53403.jpg?format=pjpg&w=400', 'program_image_dstk': '', 'program_season': '', 'program_type_serie': '0', 'program_serie_img': 'https://cdn-images.rtp.pt/EPG/imagens/S40289_1764_7437.jpg?format=pjpg&w=400', 'program_image_169_wide': 'https://cdn-images.rtp.pt/EPG/imagens/16x940289_1128_60244.jpg?format=pjpg&w=400', 'program_image_podcast': '', 'program_media_type': 'video', 'program_onair_site': '', 'program_onair_site_exclusive': '0', 'episode_id': '0', 'program_serie_label': ''}, {'channel_id': '5', 'channel_name': 'RTP1', 'channel_type': 'tvlinear', 'program_id': '1222', 'program_is_serie': '1', 'program_rewrite': 'bem-vindos-a-beirais', 'program_title': 'Bem-vindos a Beirais', 'program_lead': 'Diogo Almada, um bem sucedido gestor de contas numa empresa de telecomunicações confronta-se com problemas graves de stress e ansiedade, originados pela constante pressão em que vive. A situação piora quando sofre um ataque cardíaco. No hospital, é alertado para o risco que corre: se não abrandar o ritmo, poderá vir a ter graves consequências. Numa tentativa de reaver a sua qualidade de vida, Diogo muda-se para a pequena aldeia de Beirais, em Vila Real arriscando uma nova vida como agricultor. No entanto, a sua namorada Teresa recusa deixar Lisboa e a sua carreira e, mesmo assim, Diogo prossegue com os seus planos.\r<br />A distância que o separa de Teresa, leva Diogo a aproximar-se de Clara, uma das sócias da casa de turismo de habitação.', 'program_promotion': 'Quando não podes voltar a falhar, o sucesso é o único caminho possível.', 'program_image': 'https://cdn-images.rtp.pt/EPG/imagens/30134_34353_39667.jpg?format=pjpg&w=400', 'program_image_dstk': '', 'program_season': '1', 'program_type_serie': '1', 'program_serie_img': 'https://cdn-images.rtp.pt/EPG/imagens/S30134_35_1118.jpg?format=pjpg&w=400', 'program_image_169_wide': '', 'program_image_podcast': '', 'program_media_type': '', 'program_onair_site': '', 'program_onair_site_exclusive': '0', 'episode_id': '0', 'program_serie_label': 'Série Completa'}, {'channel_id': '5', 'channel_name': 'RTP1', 'channel_type': 'tvlinear', 'program_id': '8157', 'program_is_serie': '2', 'program_rewrite': 'telejornal', 'program_title': 'Telejornal', 'program_lead': 'A mais rigorosa seleção de notícias, todos os dias às 20h00. De segunda a domingo, João Adelino Faria e José Rodrigues dos Santos mostram-lhe o que de mais relevante se passou o país e no mundo. Se é importante está no Telejornal.', 'program_promotion': 'A mais rigorosa seleção de notícias, todos os dias às 20h00', 'program_image': 'https://cdn-images.rtp.pt/EPG/imagens/38087_57379_89633.png?format=pjpg&w=400', 'program_image_dstk': '', 'program_season': '2021', 'program_type_serie': '0', 'program_serie_img': 'https://cdn-images.rtp.pt/EPG/imagens/S39831_1766_30806.jpg?format=pjpg&w=400', 'program_image_169_wide': 'https://cdn-images.rtp.pt/EPG/imagens/16x939831_1026_62291.jpg?format=pjpg&w=400', 'program_image_podcast': '', 'program_media_type': 'video', 'program_onair_site': '', 'program_onair_site_exclusive': '0', 'episode_id': '0', 'program_serie_label': ''}, {'channel_id': '5', 'channel_name': 'RTP1', 'channel_type': 'tvlinear', 'program_id': '7759', 'program_is_serie': '1', 'program_rewrite': 'conta-me-como-foi', 'program_title': 'Conta-me Como Foi', 'program_lead': 'Dezembro de 1984. Com o Natal à porta, a família Lopes tenta recuperar das feridas deixadas pelo falhanço do casamento de Isabel e Inácio. Toni e Carlos mal se falam e Isabel não quer ver Luz. Tudo isto acontece num país a meio caminho entre o passado colonial e o futuro que está à porta:  a adesão à CEE. Ao longo desta temporada, a família Lopes testemunha e participa em alguns dos acontecimentos mais marcantes da década de 80: a ascensão de Cavaco Silva ao poder, a inauguração das Amoreiras, o flagelo dos incêndios, o acidente de Alcafache e o aparecimento do Totoloto.<br /><br /><br />', 'program_promotion': 'Reencontre os Lopes a viver o turbilhão em que está Portugal nos anos 80', 'program_image': 'https://cdn-images.rtp.pt/EPG/imagens/38041_56132_37654.jpg?format=pjpg&w=400', 'program_image_dstk': '', 'program_season': '7', 'program_type_serie': '1', 'program_serie_img': 'https://cdn-images.rtp.pt/EPG/imagens/S39533_1299_1295.jpg?format=pjpg&w=400', 'program_image_169_wide': '', 'program_image_podcast': '', 'program_media_type': 'video', 'program_onair_site': '', 'program_onair_site_exclusive': '0', 'episode_id': '0', 'program_serie_label': 'Série Completa'}, {'channel_id': '5', 'channel_name': 'RTP1', 'channel_type': 'tvlinear', 'program_id': '8250', 'program_is_serie': '2', 'program_rewrite': 'praca-da-alegria', 'program_title': 'Praça da Alegria', 'program_lead': 'Jorge Gabriel e Sónia Araújo dão-lhe as boas vindas à Praça da Alegria! Porque sabemos que gosta da nossa companhia, oferecemos boa disposição até à hora de almoço!\r<br />De segunda a sexta-feira, a Praça da Alegria leva até si a melhor música, as últimas tendências da moda, conselhos úteis e muitas dicas que facilitam o seu dia-a-dia. Portugal inteiro marca presença nesta Praça da Alegria com reportagens e diretos onde ficamos mais perto de si!', 'program_promotion': 'Otimismo,  entusiasmo e muita energia as manhãs de Portugal!', 'program_image': 'https://cdn-images.rtp.pt/EPG/imagens/36375_51571_10521.png?format=pjpg&w=400', 'program_image_dstk': '', 'program_season': '2021', 'program_type_serie': '0', 'program_serie_img': 'https://cdn-images.rtp.pt/EPG/imagens/S39965_1763_89760.jpg?format=pjpg&w=400', 'program_image_169_wide': '', 'program_image_podcast': '', 'program_media_type': 'video', 'program_onair_site': '', 'program_onair_site_exclusive': '0', 'episode_id': '0', 'program_serie_label': ''}, {'channel_id': '5', 'channel_name': 'RTP1', 'channel_type': 'tvlinear', 'program_id': '8248', 'program_is_serie': '2', 'program_rewrite': 'a-nossa-tarde', 'program_title': 'A Nossa Tarde', 'program_lead': 'Pensado a partir da essência da apresentadora, Tânia Ribas de Oliveira, o programa "A Nossa Tarde" tem, por isso, um lado mais emocional, com base em histórias com final feliz, e um lado muito divertido, ou não fosse a nossa Tânia uma pessoa que gosta de dar umas belas e sonoras gargalhadas. \r<br />As emoções vão chegar através de histórias de vida, de pessoas que se superam perante as adversidades de quem luta diariamente para mudar o que a vida, muitas vezes, teima em querer dar. As emoções estarão patentes também nas homenagens que vamos fazer. O lado mais divertido vai chegar-nos pelos jogos e pelas conversas com figuras públicas, que vêm contar-nos um lado menos conhecido. Não vão faltar momentos musicais e surpresas por parte da nossa Tânia.', 'program_promotion': 'A Nossa Tarde é o seu lugar', 'program_image': 'https://cdn-images.rtp.pt/EPG/imagens/37127_53815_22501.png?format=pjpg&w=400', 'program_image_dstk': '', 'program_season': '2021', 'program_type_serie': '0', 'program_serie_img': 'https://cdn-images.rtp.pt/EPG/imagens/S40034_1776_93335.jpg?format=pjpg&w=400', 'program_image_169_wide': '', 'program_image_podcast': '', 'program_media_type': 'video', 'program_onair_site': '', 'program_onair_site_exclusive': '0', 'episode_id': '0', 'program_serie_label': ''}, {'channel_id': '80', 'channel_name': 'RTP Memória', 'channel_type': 'tvlinear', 'program_id': '1366', 'program_is_serie': '1', 'program_rewrite': 'os-filhos-do-rock', 'program_title': 'Os Filhos do Rock', 'program_lead': 'Ah... os anos 80. Dos penteados indesculpáveis e do Passeio dos Alegres. Da SIDA e do Bairro Alto. Dos UHF, Xutos, GNR, Heróis do Mar e Rui Veloso. E d"Os Barões, convém não esquecer.<br /><br />Todas as revoluções precisam de um porta-voz. Na do rock português é Xavier Bastos quem assume esse papel. Amigo de Jorge Palma e conhecedor de música e dos músicos, faz do seu programa "Dias do Rock" o epicentro do terremoto sonoro que abala a rádio nacional.<br /><br />Outro trio que vai fazer histórias começa ao mesmo tempo. João Pedro, Zé Paulo e Garrafa são rockeiros a valer, e o seu talento começa a ser reconhecido por todos. Por Xavier, o radialista rock que lhes propõe um contrato; por Pedro, o empresário sem escrúpulos que se antecipa e os rouba para a sua editora; pelos amigos, namoradas, até pelas famílias. Pelo mundo da música e pelo grande público.<br />', 'program_promotion': 'O rock volta a fazer história!... na nova série de época da RTP com a história da geração que atravessou os melhores anos do rock em Portugal...<br />', 'program_image': 'https://cdn-images.rtp.pt/EPG/imagens/30539_34094_1397.png?format=pjpg&w=400', 'program_image_dstk': '', 'program_season': '', 'program_type_serie': '1', 'program_serie_img': 'https://cdn-images.rtp.pt/EPG/imagens/S30539_30_51020.jpg?format=pjpg&w=400', 'program_image_169_wide': '', 'program_image_podcast': '', 'program_media_type': '', 'program_onair_site': '', 'program_onair_site_exclusive': '0', 'episode_id': '0', 'program_serie_label': 'Série Completa'}, {'channel_id': '5', 'channel_name': 'RTP1', 'channel_type': 'tvlinear', 'program_id': '7434', 'program_is_serie': '2', 'program_rewrite': 'o-preco-certo', 'program_title': 'O Preço Certo', 'program_lead': 'Há mais de uma década em emissão contínua na RTP1, "O Preço Certo", é o gameshow de maior longevidade da televisão mundial. Estreado pela primeira vez em 1956 nos Estados Unidos, já foi transmitido em mais de 30 países.<br />O sucesso por todo o mundo é testemunho da sua contínua popularidade e vitalidade, provando ser um clássico e intemporal formato de programas de entretenimento.<br />Estreado em 2002 com o nome "O Preço Certo em Euros", para familiarizar os portugueses com a nova moeda, foi primeiro apresentado por Jorge Gabriel e depois por Fernando Mendes, sempre com Miguel Vital a dar voz aos fabulosos prémios. Em 2006 foi relançado com o nome original de O Preço Certo mantendo o apresentador e toda a boa disposição.', 'program_promotion': 'No "Preço Certo", as emoções são garantidas!', 'program_image': 'https://cdn-images.rtp.pt/EPG/imagens/13007_45883_49885.png?format=pjpg&w=400', 'program_image_dstk': '', 'program_season': '25', 'program_type_serie': '0', 'program_serie_img': 'https://cdn-images.rtp.pt/EPG/imagens/S39155_1765_67387.jpg?format=pjpg&w=400', 'program_image_169_wide': 'https://cdn-images.rtp.pt/EPG/imagens/16x939155_662_75844.jpg?format=pjpg&w=400', 'program_image_podcast': '', 'program_media_type': 'video', 'program_onair_site': '', 'program_onair_site_exclusive': '0', 'episode_id': '0', 'program_serie_label': ''}, {'channel_id': '5', 'channel_name': 'RTP1', 'channel_type': 'tvlinear', 'program_id': '7805', 'program_is_serie': '2', 'program_rewrite': '5-para-a-meia-noite', 'program_title': '5 Para a Meia-Noite', 'program_lead': 'O 5 Para a Meia-Noite está de volta agora com Inês Lopes Gonçalves a assumir o papel de anfitriã deste hotel sem pequeno-almoço incluído. <br />O talk-show mais imprevisível da televisão portuguesa vai receber convidados que marcam a atualidade e que não vão escapar às acutilantes perguntas da "Pressão no Ar".<br /> <br />O humorista Carlos Pereira reforça o plantel de colaboradores e protagoniza novas rubricas como "Prós e Contras dos Pobres". Quem também traz rubricas frescas é Miguel Rocha e a irreverente Beatriz Gosta, que vai surpreender tudo e todos.<br />  <br />Mas há mais! O 5 estreia outras rubricas como "Tenho uma Consulta às 5", "Perguntas Muito Infantis" e "Correio Safadão", em que celebridades partilham as mensagens privadas que recebem nas redes sociais.<br />', 'program_promotion': 'Inês Lopes Gonçalves recebe os convidados que marcam a semana<br />', 'program_image': 'https://cdn-images.rtp.pt/EPG/imagens/39585_59684_78374.png?format=pjpg&w=400', 'program_image_dstk': '', 'program_season': '17', 'program_type_serie': '0', 'program_serie_img': 'https://cdn-images.rtp.pt/EPG/imagens/S39585_1762_139.jpg?format=pjpg&w=400', 'program_image_169_wide': 'https://cdn-images.rtp.pt/EPG/imagens/16x939585_810_37868.jpg?format=pjpg&w=400', 'program_image_podcast': '', 'program_media_type': 'video', 'program_onair_site': '', 'program_onair_site_exclusive': '0', 'episode_id': '0', 'program_serie_label': ''}, {'channel_id': '5', 'channel_name': 'RTP1', 'channel_type': 'tvlinear', 'program_id': '7479', 'program_is_serie': '2', 'program_rewrite': 'portugueses-pelo-mundo-comunidades', 'program_title': 'Portugueses pelo Mundo - Comunidades', 'program_lead': 'No mundo não é difícil encontrar casos de portugueses de sucesso nas mais diversas áreas, como empresários, artistas, entre outros. São eles que levam o nome de Portugal longe, fazendo sempre questão de mostrar o orgulho das suas origens e na portugalidade. Nesta décima temporada, vamos acompanhar gerações de emigrantes, conhecer famílias, histórias e casos de sucesso, sem nunca descurar a imagem de marca do programa: mostrar o mundo e a comunidade através do olhar dos portugueses. São eles que nos vão levar numa viagem pelos Estados Unidos da América, Brasil e Europa, numa série de episódios especiais. Este regresso do programa marca a aproximação às comunidades portuguesas espalhadas pelo mundo. Em cada programa três histórias irão cruzar-se, sempre com dois pontos em comum: Portugal e a comunidade portuguesa: um português que esteja na comunidade há várias décadas, um lusodescendente e um emigrante mais recente na comunidade. Juntos vão contar-nos as suas experiências, partilhando a forma como chegaram, como se adaptaram e como vivem hoje em dia. Com eles vamos mostrar a forma como a portugalidade está espalhada pelo mundo, provando assim que somos bem mais do que os 11 milhões que as estatísticas nos apresentam. Prepare-se porque a viagem vai começar...<br />', 'program_promotion': 'Por todo o lado, há sempre Portugueses pelo Mundo...<br />', 'program_image': 'https://cdn-images.rtp.pt/EPG/imagens/39201_58760_27903.jpg?format=pjpg&w=400', 'program_image_dstk': '', 'program_season': '10', 'program_type_serie': '0', 'program_serie_img': 'https://cdn-images.rtp.pt/EPG/imagens/S39201_1775_7941.jpg?format=pjpg&w=400', 'program_image_169_wide': '', 'program_image_podcast': '', 'program_media_type': '', 'program_onair_site': '', 'program_onair_site_exclusive': '0', 'episode_id': '0', 'program_serie_label': ''}, {'channel_id': '80', 'channel_name': 'RTP Memória', 'channel_type': 'tvlinear', 'program_id': '8529', 'program_is_serie': '2', 'program_rewrite': 'traz-pra-frente', 'program_title': 'Traz prá Frente', 'program_lead': 'Inês Lopes Gonçalves modera um debate com um painel de luxo: Fernando Alvim, Nuno Markl, Álvaro Costa, Júlio Isidro e um convidado especial. Numa conversa desempoeirada, lança-se a semana da RTP Memória à mesa, com os melhores conteúdos e surpresas em destaque. <br /><br />Reciclamos o Cartaz TV e trazemos para a frente um debate divertido, mas doutorado, sobre o imaginário da televisão.', 'program_promotion': 'Reciclamos o Cartaz TV e trazemos para a frente um debate divertido, mas doutorado, sobre o imaginário da televisão<br />', 'program_image': 'https://cdn-images.rtp.pt/EPG/imagens/36710_52499_92090.jpg?format=pjpg&w=400', 'program_image_dstk': '', 'program_season': '6', 'program_type_serie': '0', 'program_serie_img': 'https://cdn-images.rtp.pt/EPG/imagens/S40033_1774_35815.jpg?format=pjpg&w=400', 'program_image_169_wide': 'https://cdn-images.rtp.pt/EPG/imagens/16x940033_1124_75939.jpg?format=pjpg&w=400', 'program_image_podcast': '', 'program_media_type': 'video', 'program_onair_site': '', 'program_onair_site_exclusive': '0', 'episode_id': '0', 'program_serie_label': ''}, {'channel_id': '5', 'channel_name': 'RTP1', 'channel_type': 'tvlinear', 'program_id': '7401', 'program_is_serie': '2', 'program_rewrite': 'quem-quer-ser-milionario-alta-pressao', 'program_title': 'Quem Quer Ser Milionário - Alta Pressão', 'program_lead': 'O Quem Quer Ser Milionário - Alta Pressão está de regresso à RTP, no ano em que se assinalam os 20 anos de existência do formato. E com uma novidade: a apresentação fica a cargo de Filomena Cautela. <br /><br />No concurso de cultura geral e de conhecimento mais conhecido no mundo e de maior longevidade todos ganham prémios! O objetivo final é acertar nas 15 perguntas de escolha múltipla para ganhar o prémio máximo de 50.000 euros. E isto não é tarefa fácil!<br /><br />Em Quem Quer Ser Milionário - Alta Pressão seis concorrentes disputam entre si a oportunidade de jogar com a nossa apresentadora Filomena Cautela na Cadeira Vermelha! A esta primeira fase chamamos "Dedo Rápido". Os concorrentes terão de responder a três perguntas de escolha múltipla tendo apenas 15 segundos para responder a cada uma delas. O jogador mais rápido vence esta primeira ronda e ganha um cheque no valor de 500 euros e uma vantagem única. Este cheque dá-lhe a oportunidade de comprar uma das duas ajudas que Filomena Cautela irá disponibilizar para continuar a prova!<br />', 'program_promotion': 'Um concurso de conhecimento, tática e... muitos nervos!', 'program_image': 'https://cdn-images.rtp.pt/EPG/imagens/39100_58492_79692.jpg?format=pjpg&w=400', 'program_image_dstk': '', 'program_season': '', 'program_type_serie': '0', 'program_serie_img': 'https://cdn-images.rtp.pt/EPG/imagens/S39100_1772_78807.jpg?format=pjpg&w=400', 'program_image_169_wide': 'https://cdn-images.rtp.pt/EPG/imagens/16x939100_594_16142.jpg?format=pjpg&w=400', 'program_image_podcast': '', 'program_media_type': 'video', 'program_onair_site': '', 'program_onair_site_exclusive': '0', 'episode_id': '0', 'program_serie_label': ''}]}
        :return:
        """
        return self.__request("get-collection/?collection_id=" + str(collection_id))

    def get_channel_epg(self, channel_id, date=None):
        """
            Example: {'channel_id': '5', 'channel_name': 'RTP1', 'alignment_date': '14-03-2021', 'alignment': [{'time': '06:30', 'date_time': '2021-03-14 06:30:00', 'date_time_utc': '2021-03-14T06:30:00+00:00', 'title': 'Zig Zag', 'comment': '', 'description': 'O Gabi e a Filó entraram na casa do Zig Zag e parece que tão cedo não a vão abandonar! Filó, uma gata muito inteligente que sabe muitas coisas e quer ensiná-las, Gabi, um cão muito divertido, que o que mais gosta é de brincar. Os dois são grandes amigos e vão viver muitas situações engraçadas.', 'homepage': 'https://www.rtp.pt/programa/tv/p31901', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/31901_36640_67908.jpg?format=pjpg&w=400', 'categories': ['Infantis e Juvenis', 'PalPlus']}, {'time': '06:31', 'date_time': '2021-03-14 06:31:00', 'date_time_utc': '2021-03-14T06:31:00+00:00', 'title': 'Olá, Como te Sentes?', 'comment': '', 'description': 'Olá, como te sentes?Numa fase em que estás mais tempo em casa e longe de quem gostas é importante perceberes o que sentes com todas estas mudanças.As emoções e sentimentos são importantes para nos conhecermos e quando as partilhamos é muito mais fácil sermos compreendidos!', 'homepage': 'https://www.rtp.pt/programa/tv/p39019/e4', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/39019_58152_80634.jpg?format=pjpg&w=400', 'rtpplay': {'id': '7306', 'rewrite': 'ola-como-te-sentes'}, 'categories': ['Infantis e Juvenis', 'HD', 'PalPlus']}, {'time': '06:32', 'date_time': '2021-03-14 06:32:00', 'date_time_utc': '2021-03-14T06:32:00+00:00', 'title': 'Lulu Cambalhota', 'comment': '', 'description': 'Lulu Vroumette, uma série alegre e agradável para os pequenos exploradores da vida e da natureza. Lulu e os seus amigos adoram o campo, as florestas, o riacho, todas as estações do ano. Para eles o melhor parque infantil é a natureza!', 'homepage': 'https://www.rtp.pt/programa/tv/p27843/e49', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/27843_42595_98940.jpg?format=pjpg&w=400', 'rtpplay': {'id': '2648', 'rewrite': 'lulu-cambalhota'}, 'categories': ['Infantis e Juvenis', 'PalPlus']}, {'time': '06:45', 'date_time': '2021-03-14 06:45:00', 'date_time_utc': '2021-03-14T06:45:00+00:00', 'title': 'As Aventuras do Max - Atlantos', 'comment': '', 'description': 'O nosso corajoso Max não desiste de salvar o mundo do terrível Sombra!', 'homepage': 'https://www.rtp.pt/programa/tv/p31932/e22', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/31932_58211_52792.jpg?format=pjpg&w=400', 'categories': ['Infantis e Juvenis', 'PalPlus']}, {'time': '06:58', 'date_time': '2021-03-14 06:58:00', 'date_time_utc': '2021-03-14T06:58:00+00:00', 'title': 'Gawayn', 'comment': '', 'description': 'Esta é uma aventura entre homens valentes que querem salvar a princesa Gwendolina!', 'homepage': 'https://www.rtp.pt/programa/tv/p25259/e16', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/25259_45176_61957.jpg?format=pjpg&w=400', 'categories': ['Infantis e Juvenis', 'PalPlus']}, {'time': '07:11', 'date_time': '2021-03-14 07:11:00', 'date_time_utc': '2021-03-14T07:11:00+00:00', 'title': 'Nutri Ventures - Em Busca dos 7 Reinos', 'comment': '', 'description': 'Nutris são as aventuras de 4 amigos no reino da nutrição.O mau da fita fez desaparecer da face da terra todos os alimentos e sabores transformando-os num comprimido.Os Nutris vão salvar o paladar terrestre, descobrir onde esconderam os alimentos e aprender sobre como comer sem adoecer.', 'homepage': 'https://www.rtp.pt/programa/tv/p40073/e12', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/30769_41243_72535.jpg?format=pjpg&w=400', 'categories': ['Infantis e Juvenis', 'PalPlus']}, {'time': '07:33', 'date_time': '2021-03-14 07:33:00', 'date_time_utc': '2021-03-14T07:33:00+00:00', 'title': 'Desafio #FicoEmCasa', 'comment': '', 'description': 'O Desafio #FicoEmCasa continua, mas agora o ecrã foi ocupado por um habitante rabugento, monstruoso, vindo de outra galáxia. O Vladimir vai tentar trazer alegria às vossas casas, atenção vai tentar, porque ele não é lá muito alegre, e vai mostrar todos os desafios que temos recebido das meninas e meninos.', 'homepage': 'https://www.rtp.pt/programa/tv/p38995/e8', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/38995_58114_72026.jpg?format=pjpg&w=400', 'rtpplay': {'id': '7326', 'rewrite': 'desafio-fico-em-casa-2'}, 'categories': ['Infantis e Juvenis', 'HD', 'PalPlus']}, {'time': '07:43', 'date_time': '2021-03-14 07:43:00', 'date_time_utc': '2021-03-14T07:43:00+00:00', 'title': 'Radar XS', 'comment': '', 'description': 'Radar XS é um telejornal para crianças dos 8 aos 12 anos onde se pretende incentivar o pensamento crítico e a cidadania. Em estúdio vamos juntar a política e a cultura, a economia e o desporto, a ciência e a literatura infantil, a atualidade nacional e internacional, numa linguagem simples e objetiva.Todas as semanas, o Radar XS sai à rua à procura de histórias, projetos, escolas, pessoas que nos inspirem, para alimentar a vossa curiosidade. Queremos ouvir as crianças! Dar voz às vossas dúvidas, esclarecer questões, promover o diálogo e o debate, em casa, na escola, no estúdio, em todo o lado!', 'homepage': 'https://www.rtp.pt/programa/tv/p39383/e78', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/39383_61229_15371.jpg?format=pjpg&w=400', 'rtpplay': {'id': '7655', 'rewrite': 'radar-xs-3'}, 'categories': ['Infantis e Juvenis', 'HD', 'PalPlus']}, {'time': '08:00', 'date_time': '2021-03-14 08:00:00', 'date_time_utc': '2021-03-14T08:00:00+00:00', 'title': 'Bom Dia Portugal Fim de Semana', 'comment': '', 'description': 'Um espaço informativo em que se dá relevo às notícias da atualidade nacional e internacional, desporto, meteorologia, trânsito e economia.', 'homepage': 'https://www.rtp.pt/programa/tv/p39834', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/38084_57380_28384.png?format=pjpg&w=400', 'rtpplay': {'id': '8154', 'rewrite': 'bom-dia-portugal-fim-de-semana', 'episode': {'id': '530734', 'number': '', 'url': 'https://www.rtp.pt/play/p8154/e530734/bom-dia-portugal-fim-de-semana'}}, 'categories': ['Informação', 'Directo', 'Teletexto', 'PalPlus']}, {'time': '10:30', 'date_time': '2021-03-14 10:30:00', 'date_time_utc': '2021-03-14T10:30:00+00:00', 'title': 'Eucaristia Dominical', 'comment': 'Lisboa: IV Domingo da Quaresma', 'description': 'A celebração dominical do Dia e da Eucaristia do Senhor está no centro da vida da Igreja Católica.', 'homepage': 'https://www.rtp.pt/programa/tv/p39847/e11', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/35203_47492_60760.jpg?format=pjpg&w=400', 'rtpplay': {'id': '8347', 'rewrite': 'eucaristia-dominical', 'episode': {'id': '530749', 'number': '11', 'url': 'https://www.rtp.pt/play/p8347/e530749/eucaristia-dominical'}}, 'categories': ['Religiosos', 'Directo', 'PalPlus']}, {'time': '11:31', 'date_time': '2021-03-14 11:31:00', 'date_time_utc': '2021-03-14T11:31:00+00:00', 'title': 'Céus Ancestrais', 'comment': '', 'description': 'Desde os primórdios da Humanidade que os céus influenciam todos os aspectos da nossa vida. Ajudaram-nos a orientar, a construir e a definir o próprio conceito de tempo. A série documental ?Céus Ancestrais? olha para cosmos através dos olhos dos nossos antepassados, percebendo por que mutações passou a sua concepção a longo da história. E um grupo de historiadores e astrónomos explica a ciência por trás dos fenómenos que os nossos ancestrais procuraram explicar por meio da mitologia. Esta é a história da obsessão da humanidade pelos céus e das formas como os nossos antepassados explicavam o universo.', 'homepage': 'https://www.rtp.pt/programa/tv/p40336/e2', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/40336_61180_53513.jpg?format=pjpg&w=400', 'rtpplay': {'id': '8555', 'rewrite': 'ceus-ancestrais', 'episode': {'id': '530706', 'number': '2', 'url': 'https://www.rtp.pt/play/p8555/e530706/ceus-ancestrais'}}, 'categories': ['Documentários', 'Teletexto', 'HD', 'PalPlus']}, {'time': '12:33', 'date_time': '2021-03-14 12:33:00', 'date_time_utc': '2021-03-14T12:33:00+00:00', 'title': 'Sem Fronteiras', 'comment': 'Salvaterra de Miño e Vieira do Minho', 'description': 'Beatriz Gosta, apresentadora portuguesa, viajará  a uma terra galega para a conhecer a fundo. Javier Varela, apresentador galego, visitará terras portuguesas para embrenhar-se na sua cultura. Ambos estarão, isso sim, em lugares que partilham o mesmo, ou semelhante, nome. Assim sendo, Beatriz estará na Pontenova e Javier em Ponte de Lima. Ela em Baiona e ele em Baião. Programa a programa irão descobrir as vidas dos habitantes do país vizinho, a sua História, a sua gastronomia e, consequentemente, o quão parecidos realmente somos nos dois lados da raia. Sem Fronteiras, um espaço televisivo para nos conhecermos melhor.', 'homepage': 'https://www.rtp.pt/programa/tv/p40147/e9', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/40147_60810_11387.png?format=pjpg&w=400', 'rtpplay': {'id': '8373', 'rewrite': 'sem-fronteiras', 'episode': {'id': '530704', 'number': '9', 'url': 'https://www.rtp.pt/play/p8373/e530704/sem-fronteiras'}}, 'categories': ['Magazines', 'Teletexto', 'HD', 'PalPlus']}, {'time': '12:59', 'date_time': '2021-03-14 12:59:00', 'date_time_utc': '2021-03-14T12:59:00+00:00', 'title': 'Jornal da Tarde', 'comment': '', 'description': 'Todos os dias, às 13 horas, a sua hora de almoço é completada com a atualização das notícias que marcam a atualidade nacional e mundial, desde a política, a economia, a sociedade e sem esquecer o desporto. Reportagens no local, entrevistas em direto e sempre a isenção e o respeito pela verdade que marcam a informação na RTP.', 'homepage': 'https://www.rtp.pt/programa/tv/p39835', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/32531_39576_60708.jpg?format=pjpg&w=400', 'rtpplay': {'id': '8155', 'rewrite': 'jornal-da-tarde', 'episode': {'id': '530764', 'number': '', 'url': 'https://www.rtp.pt/play/p8155/e530764/jornal-da-tarde'}}, 'categories': ['Informação', 'Directo', 'Teletexto', 'PalPlus']}, {'time': '14:25', 'date_time': '2021-03-14 14:25:00', 'date_time_utc': '2021-03-14T14:25:00+00:00', 'title': 'Faz Faísca', 'comment': '', 'description': 'No Faz Faísca, tudo pode acontecer! Neste dinâmico magazine de lifestyle, todas as semanas recebemos convidados das mais distintas áreas, sempre com um desafio na manga! No Faz Faísca, estamos prontos para vos surpreender: seja à boleia com o fantástico Toy, seja com a nossa "caixa louca" recheada de presentes inesperados ou num Talk Show pequeníssimo, mas muito especial! Vamos levar-vos aos bastidores da nossa RTP, vamos em reportagem pelo país fora, não esquecendo nunca a música, o cinema, a moda, o desporto. Contamos com os criativos repórteres Diana Taveira, Inês Faria, Catarina Palma, Rodrigo Gomes, Idevor Mendonça, Rita La Rochezoire, Ana Moreira e Teresa Peres, sempre com boa disposição e irreverência. A este grupo dinâmico e bem humorado, podem juntar-se, a qualquer momento, novos reforços com a mesma missão!', 'homepage': 'https://www.rtp.pt/programa/tv/p40304/e3', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/40304_61294_95951.png?format=pjpg&w=400', 'categories': ['Magazines', 'HD', 'PalPlus']}, {'time': '15:28', 'date_time': '2021-03-14 15:28:00', 'date_time_utc': '2021-03-14T15:28:00+00:00', 'title': 'Hoje é Domingo!', 'comment': '', 'description': '"Hoje é Domingo!" apresentado por Vera Fernandes e João Paulo Rodrigues que lhe fazem companhia nas suas tardes de domingo. Boa música, humor, convidados especiais e jogos inovadores. Aqui nunca saberemos o que vai acontecer, quem irá aparecer ou quais os jogos e atuações surpreendentes reservados para cada programa.', 'homepage': 'https://www.rtp.pt/programa/tv/p40267/e6', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/39662_59900_9601.png?format=pjpg&w=400', 'rtpplay': {'id': '8460', 'rewrite': 'hoje-e-domingo', 'episode': {'id': '530777', 'number': '6', 'url': 'https://www.rtp.pt/play/p8460/e530777/hoje-e-domingo'}}, 'categories': ['Entretenimento', 'Directo', 'HD', 'PalPlus']}, {'time': '19:59', 'date_time': '2021-03-14 19:59:00', 'date_time_utc': '2021-03-14T19:59:00+00:00', 'title': 'Telejornal', 'comment': '', 'description': 'A mais rigorosa seleção de notícias, todos os dias às 20h00. De segunda a domingo, João Adelino Faria e José Rodrigues dos Santos mostram-lhe o que de mais relevante se passou o país e no mundo. Se é importante está no Telejornal.', 'homepage': 'https://www.rtp.pt/programa/tv/p39831', 'isLive': '1', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/38087_57379_89633.png?format=pjpg&w=400', 'rtpplay': {'id': '8157', 'rewrite': 'telejornal', 'episode': {'id': '530800', 'number': '', 'url': 'https://www.rtp.pt/play/p8157/e530800/telejornal'}}, 'categories': ['Informação', 'Directo', 'Teletexto', 'PalPlus']}, {'time': '21:23', 'date_time': '2021-03-14 21:23:00', 'date_time_utc': '2021-03-14T21:23:00+00:00', 'title': 'The Voice Kids', 'comment': '', 'description': 'O palco do The Voice vai receber crianças de todo o país, com mais de 7 anos, para nos mostrar que o talento não tem idade. Catarina Furtado é a apresentadora Catarina Furtado e vai ajudar os mais novos a gerir as emoções e os sonhos no The Voice Kids.As cadeiras mais famosas de Portugal vão receber Marisa Liz, Fernando Daniel, Carolina Deslandes e Carlão. Nos bastidores, Bárbara Lourenço é a apresentadora digital que, na fase de Provas Cegas, vai acompanhar os candidatos no momento antes de subirem ao palco.', 'homepage': 'https://www.rtp.pt/programa/tv/p40096/e9', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/40096_60652_67122.jpg?format=pjpg&w=400', 'rtpplay': {'id': '8256', 'rewrite': 'the-voice-kids'}, 'categories': ['Concursos', 'HD', 'PalPlus']}, {'time': '23:37', 'date_time': '2021-03-14 23:37:00', 'date_time_utc': '2021-03-14T23:37:00+00:00', 'title': 'The Voice Kids - As Equipas/Batalhas', 'comment': '', 'description': 'Relembre os melhores momentos das Batalhas dos nossos cantores de palmo e meio.Marisa Liz, Fernando Daniel, Carolina Deslandes e Carlão procuraram as melhores vozes e um deles será o vencedor com a eleição da melhor voz de Portugal.', 'homepage': 'https://www.rtp.pt/programa/tv/p40437', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/40096_60652_67122.jpg?format=pjpg&w=400', 'categories': ['Concursos', 'HD', 'PalPlus']}, {'time': '00:46', 'date_time': '2021-03-14 00:46:00', 'date_time_utc': '2021-03-14T00:46:00+00:00', 'title': 'Amar Pablo, Odiar Escobar', 'comment': '', 'description': 'Considerado o mais impiedoso, calculista e poderoso traficante de sempre, Pablo Emilio Escobar Gaviria (1949-1993) foi um colombiano que, graças ao comércio de cocaína, se transformou num dos homens mais ricos do mundo. Ficou também célebre por subornar governantes, juízes e todos os que lhe pudessem ser úteis, e pela falta de misericórdia por quem lhe desobedecesse. "Amar Pablo, Odiar Escobar" relata um momento crucial da sua vida através dos olhos de Virgínia Vallejo, uma reputada jornalista da televisão colombiana que enfrentou o maior conflito moral da sua existência ao apaixonar-se por ele e tornar-se sua amante.', 'homepage': 'https://www.rtp.pt/programa/tv/p38950', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/38950_57990_61334.jpg?format=pjpg&w=400', 'rtpplay': {'id': '7266', 'rewrite': 'amar-pablo-odiar-escobar'}, 'categories': ['Filmes ', 'Teletexto', 'HD', 'PalPlus']}, {'time': '02:46', 'date_time': '2021-03-14 02:46:00', 'date_time_utc': '2021-03-14T02:46:00+00:00', 'title': 'Faz Faísca', 'comment': '', 'description': 'No Faz Faísca, tudo pode acontecer! Neste dinâmico magazine de lifestyle, todas as semanas recebemos convidados das mais distintas áreas, sempre com um desafio na manga! No Faz Faísca, estamos prontos para vos surpreender: seja à boleia com o fantástico Toy, seja com a nossa "caixa louca" recheada de presentes inesperados ou num Talk Show pequeníssimo, mas muito especial! Vamos levar-vos aos bastidores da nossa RTP, vamos em reportagem pelo país fora, não esquecendo nunca a música, o cinema, a moda, o desporto. Contamos com os criativos repórteres Diana Taveira, Inês Faria, Catarina Palma, Rodrigo Gomes, Idevor Mendonça, Rita La Rochezoire, Ana Moreira e Teresa Peres, sempre com boa disposição e irreverência. A este grupo dinâmico e bem humorado, podem juntar-se, a qualquer momento, novos reforços com a mesma missão!', 'homepage': 'https://www.rtp.pt/programa/tv/p40304/e3', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/40304_61294_95951.png?format=pjpg&w=400', 'categories': ['Magazines', 'HD', 'PalPlus']}, {'time': '03:41', 'date_time': '2021-03-14 03:41:00', 'date_time_utc': '2021-03-14T03:41:00+00:00', 'title': 'Janela Indiscreta', 'comment': '', 'description': 'O Janela Indiscreta é o seu programa de cinema! Apresenta, em primeira mão, estreias cinematográficas e entrevistas exclusivas aos protagonistas das produções nacionais e internacionais, dando a conhecer curiosidades e histórias de bastidores.', 'homepage': 'https://www.rtp.pt/programa/tv/p39839/e10', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/33838_42773_63937.jpg?format=pjpg&w=400', 'rtpplay': {'id': '8349', 'rewrite': 'janela-indiscreta', 'episode': {'id': '530616', 'number': '11', 'url': 'https://www.rtp.pt/play/p8349/e530616/janela-indiscreta'}}, 'categories': ['Magazines', 'PalPlus']}, {'time': '04:28', 'date_time': '2021-03-14 04:28:00', 'date_time_utc': '2021-03-14T04:28:00+00:00', 'title': 'Televendas', 'comment': '', 'description': 'Todos os dias, um espaço dedicado a vendas por televisão, em que são feitas demonstrações dos produtos.', 'homepage': 'https://www.rtp.pt/programa/tv/p1132', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/1132_35764_19125.jpg?format=pjpg&w=400', 'categories': ['Outros']}, {'time': '06:14', 'date_time': '2021-03-14 06:14:00', 'date_time_utc': '2021-03-14T06:14:00+00:00', 'title': 'Manchetes 3', 'comment': '', 'description': 'Todos os dias as manchetes dos Jornais de referência estão na RTP.', 'homepage': 'https://www.rtp.pt/programa/tv/p39838', 'image': 'https://cdn-images.rtp.pt/EPG/imagens/36655_52400_89059.png?format=pjpg&w=400', 'categories': ['Informação', 'PalPlus']}]}

        :param channel_id: channel id as returned by get_live_tv_channels or get_live_radio_channels
        :param date: data in the format dd-mm-yyyy for which epg is requested
        :return:
        """
        if not date:
            date = datetime.date.today().strftime('%d-%m-%Y')
        return self.__request("get-channel-epg/?channel_id={}&date={}".format(channel_id, date))

    def get_channel(self, channel_id):
        """
            Example: [{'channel_id': '5', 'channel_rewrite': 'rtp1', 'channel_name': 'RTP1', 'channel_summary': 'Emissão em direto RTP1\xa0com o melhor entretenimento, ficção nacional e internacional, informação, desporto e muito mais. Conteúdos para toda a família, disponível em várias plataformas.', 'channel_console_title': 'RTP1', 'channel_console_summary': '', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtp1.png?fm=pn32&w=300', 'channel_card_logo': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/5-563718101410.png', 'channel_card_logo_1x1': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/5-065319091505.png', 'channel_card_logo_9x16': '', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/EPG/tv/epg-dia.php?canal=1&ac=d&sem=e', 'channel_color': '#1072fc', 'channel_alignment': 'https://www.rtp.pt/rtp1/?headline=19&visual=6', 'channel_list_order': '1', 'channel_key': 'rtp1', 'channel_live': '1', 'channel_dvr': '0', 'channel_hd': '1', 'channel_drm_widevine': '1', 'channel_drm_fairplay': '1', 'channel_drm_playready': '1', 'channel_ondemand': '1', 'channel_stream_name': '2ch5h264.mp4', 'channel_android_stream_name': 'rtp1_5ch5h264.stream/playlist.m3u8', 'channel_onairlive_site': '', 'channel_epg': '1', 'web': {'url': 'https://www.rtp.pt/play/direto/rtp1'}, 'onair': {'previous': {'title': 'Hoje é Domingo!', 'comment': '', 'description': '"Hoje é Domingo!" apresentado por Vera Fernandes e João Paulo Rodrigues que lhe fazem companhia nas suas tardes de domingo. Boa música, humor, convidados especiais e jogos inovadores. Aqui nunca saberemos o que vai acontecer, quem irá aparecer ou quais os jogos e atuações surpreendentes reservados para cada programa.', 'duration': '04:30', 'real_date_time': '2021-03-14 15:28:00', 'real_date_time_utc': '2021-03-14T15:28:00+00:00', 'real_end_date_time': '2021-03-14 19:58:59', 'real_end_date_time_utc': '2021-03-14T19:58:59+00:00', 'homepage': '', 'rtpplay': {'id': '8460', 'rewrite': 'hoje-e-domingo'}, 'program_epg': {'title': 'Hoje é Domingo!', 'original_title': '', 'director': '', 'interpreters': 'Apresentação de Vera Fernandes e João Paulo Rodrigues', 'producers': '', 'classification': 'Todos', 'year': '2021', 'color': 'Cor', 'emission': '16:9'}, 'image': 'https://cdn-images.rtp.pt/EPG/imagens/39662_59900_9601.png?format=pjpg&w=400', 'categories': ['Entretenimento', 'Directo', 'PalPlus', 'HD']}, 'current': {'title': 'Telejornal', 'comment': '', 'description': 'A mais rigorosa seleção de notícias, todos os dias às 20h00. De segunda a domingo, João Adelino Faria e José Rodrigues dos Santos mostram-lhe o que de mais relevante se passou o país e no mundo. Se é importante está no Telejornal.', 'duration': '01:23', 'real_date_time': '2021-03-14 19:59:00', 'real_date_time_utc': '2021-03-14T19:59:00+00:00', 'real_end_date_time': '2021-03-14 21:22:59', 'real_end_date_time_utc': '2021-03-14T21:22:59+00:00', 'homepage': '', 'live': '1', 'nolive_image': None, 'rtpplay': {'id': '8157', 'rewrite': 'telejornal'}, 'program_epg': {'title': 'Telejornal', 'original_title': 'Telejornal 2021', 'director': '', 'interpreters': 'Apresentação: João Adelino Faria, José Rodrigues dos Santos', 'producers': '', 'classification': 'Todos', 'year': '2021', 'color': 'Cor', 'emission': '16:9'}, 'image': 'https://cdn-images.rtp.pt/EPG/imagens/38087_57379_89633.png?format=pjpg&w=400', 'categories': ['Informação', 'Directo', 'Teletexto', 'PalPlus']}, 'next': {'title': 'The Voice Kids', 'comment': '', 'description': 'O palco do The Voice vai receber crianças de todo o país, com mais de 7 anos, para nos mostrar que o talento não tem idade. Catarina Furtado é a apresentadora Catarina Furtado e vai ajudar os mais novos a gerir as emoções e os sonhos no The Voice Kids.As cadeiras mais famosas de Portugal vão receber Marisa Liz, Fernando Daniel, Carolina Deslandes e Carlão. Nos bastidores, Bárbara Lourenço é a apresentadora digital que, na fase de Provas Cegas, vai acompanhar os candidatos no momento antes de subirem ao palco.', 'duration': '02:13', 'real_date_time': '2021-03-14 21:23:00', 'real_date_time_utc': '2021-03-14T21:23:00+00:00', 'real_end_date_time': '2021-03-14 23:36:59', 'real_end_date_time_utc': '2021-03-14T23:36:59+00:00', 'homepage': '', 'rtpplay': {'id': '8256', 'rewrite': 'the-voice-kids'}, 'program_epg': {'title': 'The Voice Kids', 'original_title': 'The Voice Kids - 2021', 'director': '', 'interpreters': 'Apresentação de Catarina Furtado', 'producers': 'Shine Iberia Portugal', 'classification': 'Todos', 'year': '2021', 'color': 'Cor', 'emission': '16:9'}, 'image': 'https://cdn-images.rtp.pt/EPG/imagens/40096_60652_67122.jpg?format=pjpg&w=400', 'categories': ['Concursos', 'HD']}}, 'hls_url': 'https://streaming-live-app.rtp.pt/liverepeater/smil:rtp1.smil/rtp_playlist.m3u8?token=1615784400_7867dd72cf244700585ce7ded0af15a5fc0c4752', 'stream_url': {'http': {'smil': 'https://streaming-live-app.rtp.pt/liverepeater/smil:rtp1.smil/rtp_playlist.m3u8?token=1615784400_7867dd72cf244700585ce7ded0af15a5fc0c4752', 'standard': 'https://streaming-live-app.rtp.pt/liverepeater/rtp1_5ch5h264.stream/rtp_playlist.m3u8?token=1615784400_f193ce2a9e00e6f586902b1f7caa744e6767b1ef'}}, 'stream_url_drm': {'http': {'smil': 'https://streaming-live-app.rtp.pt/liverepeater/smil:rtp1_FPS.smil/rtp_playlist.m3u8?token=1615784400_a5e2fbcb9f22e19d688a8dd3f7fdbf28362c3a7a', 'standard': ''}, 'token': 'rtp|2021-03-14T21:21:03Z|pB4uhre+Wbw2+lxMRvf3njby8BgZExkhnoKhnu2f5S9dnw+RYtN4J8LDivslldGgDHMjD0PvXk9enMz9k5mcIWKdYZufokoYv3hhgHGgpE4pYAxg7a7eKDZ4yX+WIljxnlt2d4tVpBiPtwWwFPgNkBBAcP5EcD5cOoN6UWq42Vw=|3a641d7111f858f73b03e928cb39ff193bee0ac9'}, 'stream': {'clean': {'smil': 'https://streaming-live-app.rtp.pt/liverepeater/smil:rtp1HD.smil/rtp_playlist.m3u8?token=1615784400_0d56c60b30c12d2971f193ed5ff712df95890ecf', 'standard': 'https://streaming-live-app.rtp.pt/liverepeater/rtp1_9ch5h264.stream/rtp_playlist.m3u8?token=1615784400_8e245ed588086bdb2494b6c2b93ab15eb3c63ccd'}, 'token': 'rtp|2021-03-14T21:21:03Z|pB4uhre+Wbw2+lxMRvf3njby8BgZExkhnoKhnu2f5S9dnw+RYtN4J8LDivslldGgDHMjD0PvXk9enMz9k5mcIWKdYZufokoYv3hhgHGgpE4pYAxg7a7eKDZ4yX+WIljxnlt2d4tVpBiPtwWwFPgNkBBAcP5EcD5cOoN6UWq42Vw=|3a641d7111f858f73b03e928cb39ff193bee0ac9', 'hls': {'drm': '0', 'stream_url': 'https://streaming-live-app.rtp.pt/liverepeater/smil:rtp1HD.smil/rtp_playlist.m3u8?token=1615784400_0d56c60b30c12d2971f193ed5ff712df95890ecf', 'dvr': '0'}, 'dash': {'drm': '0', 'stream_url': 'https://streaming-live-app.rtp.pt/liverepeater/rtp1_9ch5h264.stream/rtp_playlist.m3u8?token=1615784400_8e245ed588086bdb2494b6c2b93ab15eb3c63ccd', 'dvr': '0'}}, 'dash_stream_name': 'https://streaming-live-app.rtp.pt/liverepeater/smil:rtp1HD.smil/rtp_playlist.m3u8?token=1615784400_0d56c60b30c12d2971f193ed5ff712df95890ecf', 'rights_api': 'https://www.rtp.pt/services/rtpplay?v=2&ch_k=rtp1', 'onair_api': 'https://www.rtp.pt/play/onairnow.php?channel_id=5'}]

        :param channel_id:
        :return:
        """
        if str(channel_id) in self.config.DVR_CHANNELS:
            return self.__request("get-channel/{}?dvr=1&include_epg=true&include_webparams=true".format(channel_id))
        return self.__request("get-channel/{}?include_epg=true&include_webparams=true".format(channel_id))

    def get_program(self, program_id):
        """
            Example: [{'channel_id': '5', 'channel_name': 'RTP1', 'channel_rewrite': 'rtp1', 'channel_summary': 'Emissão em direto RTP1\xa0com o melhor entretenimento, ficção nacional e internacional, informação, desporto e muito mais. Conteúdos para toda a família, disponível em várias plataformas.', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtp1.png?fm=pn32&w=300', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/EPG/tv/epg-dia.php?canal=1&ac=d&sem=e', 'channel_list_order': '1', 'channel_lastchanged': '2021-03-07 00:33:02', 'program_id': '8505', 'program_technical_metadata': '', 'program_is_serie': '2', 'program_podcast': '0', 'program_offline': '0', 'program_download': '0', 'program_rewrite': 'festival-da-cancao-2021', 'program_title': 'Festival da Canção 2021', 'program_lead': 'O Festival da Canção 2021 é constituído por duas semifinais e pela grande final, marcada para 6 de março.A edição de 2021 do Festival da Canção retoma em traços gerais o modelo dos anos anteriores e contará com 20 canções a concurso. Dessas, 18 resultam de convites diretos da RTP, uma vez mais tendo em conta a representação da diversidade de géneros musicais e artistas com obra pública. A novidade está nas restantes entradas, com a seleção de dois autores através da livre submissão de canções.', 'program_promotion': 'Quem irá representar Portugal no Festival Eurovisão da Canção 2021, em Roterdão?', 'program_image': 'https://cdn-images.rtp.pt/EPG/imagens/40289_61049_53403.jpg?format=pjpg&w=400', 'program_info': '', 'program_lastchanged': '2021-03-09 12:14:46', 'program_season': '', 'program_type_serie': '0', 'program_serie_img': 'https://cdn-images.rtp.pt/EPG/imagens/S40289_1764_7437.jpg?format=pjpg&w=400', 'program_image_169_wide': 'https://cdn-images.rtp.pt/EPG/imagens/16x940289_1128_60244.jpg?format=pjpg&w=400', 'program_order_by': '1', 'program_media_type': 'video', 'program_ids_old': '7193', 'program_ids_platform': '1', 'program_onair_site': 'rtpplay', 'program_related_link': '', 'program_category_id': '4', 'program_category_name': 'Música', 'program_epg': {'title': 'Festival da Canção 2021', 'original_title': 'Festival da Canção 2021', 'director': '', 'interpreters': '', 'producers': '', 'classification': 'Todos', 'year': '2021', 'color': 'Cor', 'emission': '16:9'}, 'program_total_episodes': '5', 'program_serie_label': '', 'web': {'url': 'https://www.rtp.pt/play/p8505/festival-da-cancao-2021'}, 'program_collections': [{'collection_id': '32', 'collection_title': 'Para toda a família', 'collection_type': ''}, {'collection_id': '178', 'collection_title': '64 Anos de RTP', 'collection_type': ''}], 'program_categories': [{'category_id': '4', 'category_name': 'Música'}]}]
        :param program_id:
        :return:
        """
        return self.__request("get-program/{}?include_webparams=true&include_epg=1".format(program_id))

    def get_episode(self, program_id, episode_id):
        """
            Example: {'episode': {'channel_id': '5', 'channel_name': 'RTP1', 'channel_rewrite': 'rtp1', 'channel_summary': 'Emissão em direto RTP1\xa0com o melhor entretenimento, ficção nacional e internacional, informação, desporto e muito mais. Conteúdos para toda a família, disponível em várias plataformas.', 'channel_image': 'https://cdn-images.rtp.pt/common/img/channels/logos/color-negative/horizontal/rtp1.png?fm=pn32&w=300', 'channel_card_logo': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/5-563718101410.png', 'channel_card_logo_1x1': 'https://cdn-images.rtp.pt/common/img/channels/logos/color/horizontal/5-065319091505.png', 'channel_card_logo_9x16': '', 'channel_type': 'tvlinear', 'channel_homepage': 'https://www.rtp.pt/EPG/tv/epg-dia.php?canal=1&ac=d&sem=e', 'channel_list_order': '1', 'channel_lastchanged': '2021-03-07 00:33:02', 'program_id': '8505', 'program_technical_metadata': '', 'program_lifetime': '0', 'program_lifedate': '0000-00-00 00:00:00', 'program_podcast': '0', 'program_offline': '0', 'program_download': '0', 'program_rewrite': 'festival-da-cancao-2021', 'program_title': 'Festival da Canção 2021', 'program_lead': 'O Festival da Canção 2021 é constituído por duas semifinais e pela grande final, marcada para 6 de março.A edição de 2021 do Festival da Canção retoma em traços gerais o modelo dos anos anteriores e contará com 20 canções a concurso. Dessas, 18 resultam de convites diretos da RTP, uma vez mais tendo em conta a representação da diversidade de géneros musicais e artistas com obra pública. A novidade está nas restantes entradas, com a seleção de dois autores através da livre submissão de canções.', 'program_promotion': 'Quem irá representar Portugal no Festival Eurovisão da Canção 2021, em Roterdão?', 'program_image': 'https://cdn-images.rtp.pt/EPG/imagens/40289_61049_53403.jpg?format=pjpg&w=400', 'program_info': '', 'program_lastchanged': '2021-03-09 12:14:46', 'program_season': '', 'program_type_serie': '0', 'program_media_type': 'video', 'program_onair_site': 'rtpplay', 'program_related_link': '', 'program_is_serie': '2', 'program_category': '4', 'program_category_name': 'Música', 'episode_id': '529073', 'program_type': 'Concursos', 'episode_date': '2021-03-06', 'episode_title': 'Final', 'episode_subtitle': None, 'episode_description': 'Final Quem irá representar Portugal no Festival Eurovisão da Canção 2021, em Roterdão?\nO Festival da Canção 2021 é constituído por duas semifinais e pela grande final, marcada para 6 de março.A edição de 2021 do Festival da Canção retoma em traços gerais o modelo dos anos anteriores e contará com 20 canções a concurso. Dessas, 18 resultam de convites diretos da RTP, uma vez mais tendo em conta a representação da diversidade de géneros musicais e artistas com obra pública. A novidade está nas restantes entradas, com a seleção de dois autores através da livre submissão de canções.', 'episode_summary': 'Filomena Cautela e Vasco Palmerim apresentam a Final do Festival da Canção 2021.Entre os finalistas estão:Karetus e Romeu Bairos,  Joana Alegre, Fábia Maia, Valéria, Carolina Deslandes, NEEV, Pedro Gonçalves, Sara Afonso, EU.CLIDES, The Black Mamba. <br />O vencedor vai representar Portugal no Festival da Eurovisão que este ano tem lugar em Roterdão, nos Países Baixos, nos dias 18, 20 e 22 de maio.', 'episode_air_date': '2021-03-06 23:56:27', 'episode_date_onair': '2021-03-06 23:56:27', 'episode_is_live': '1', 'episode_lastchanged': '2021-03-07 00:48:10', 'episode_number': '3', 'episode_related_link': '', 'episode_image': '', 'program_epg': '40289', 'program_category_ads_key': None, 'episode_duration_complete': '03:24:20', 'program_collections': [{'collection_id': '32', 'collection_title': 'Para toda a família', 'collection_type': ''}, {'collection_id': '178', 'collection_title': '64 Anos de RTP', 'collection_type': ''}], 'episode_collections': [], 'program_categories': [{'category_id': '4', 'category_name': 'Música'}]}, 'assets': [{'asset_id': '909889', 'asset_internal_id': '385068', 'asset_duration': '01:39:37', 'asset_is_live': '1', 'asset_thumbnail': 'https://cdn-images.rtp.pt/multimedia/screenshots/p8505/p8505_1_202103062249180729.jpg?format=pjpg&w=400', 'asset_video_preview': '//cdn-ondemand.rtp.pt/nas2.share/h264/512x384/p8505/preview/p8505_1_202103062249180729_videoprv.mp4', 'asset_date': '2021-03-06 22:36:31', 'asset_base_dir': 'nas2.share/videos/auto/', 'asset_internal_id_info': '385265', 'asset_lastchanged': '2021-03-06 22:59:10', 'asset_geoblock_info': '2', 'asset_screenshot_preview': '//cdn-images.rtp.pt/multimedia/screenshots/p8505/preview/p8505_1_202103062249180729_preview.vtt', 'asset_type': 'video', 'asset_size': '1639745473', 'asset_path': '/h264/512x384/p8505/p8505_1_202103062249180729.mp4', 'hls_url': 'http://streaming-ondemand.rtp.pt/nas2.share/h264/512x384/p8505/index.m3u8?tlm=hls&streams=p8505_1_202103062249180729.mp4.m3u8:1024', 'hls_url_new': 'http://streaming-ondemand.rtp.pt/nas2.share/h264/512x384/p8505/p8505_1_202103062249180729/master.m3u8', 'dash_url': 'https://streaming-ondemand.rtp.pt/nas2.share/h264/512x384/p8505/p8505_1_202103062249180729/master.mpd', 'rights_api': 'https://www.rtp.pt/services/rtpplay?v=5&file=nas2.share/h264/512x384/p8505/p8505_1_202103062249180729.mp4', 'stream_url': {'http': {'standard': 'http://streaming-ondemand.rtp.pt/nas2.share/h264/512x384/p8505/index.m3u8?tlm=hls&streams=p8505_1_202103062249180729.mp4.m3u8:1024'}, 'rtmp': {'standard': {'streamer': 'rtpvodfs.fplive.net', 'application': 'rtpvod', 'file': '/nas2.share/h264/512x384/p8505/p8505_1_202103062249180729.mp4'}}}, 'download_url': 'http://cdn-ondemand.rtp.pt/nas2.share/h264/512x384/p8505/p8505_1_202103062249180729.mp4?d=1', 'subtitles': {'vtt': '', 'ebutt': ''}}, {'asset_id': '909894', 'asset_internal_id': '385070', 'asset_duration': '00:55:50', 'asset_is_live': '1', 'asset_thumbnail': 'https://cdn-images.rtp.pt/multimedia/screenshots/p8505/p8505_2_202103062353197726.jpg?format=pjpg&w=400', 'asset_video_preview': '//cdn-ondemand.rtp.pt/nas2.share/h264/512x384/p8505/preview/p8505_2_202103062353197726_videoprv.mp4', 'asset_date': '2021-03-06 23:40:53', 'asset_base_dir': 'nas2.share/videos/auto/', 'asset_internal_id_info': '385267', 'asset_lastchanged': '2021-03-06 23:55:10', 'asset_geoblock_info': '2', 'asset_screenshot_preview': '//cdn-images.rtp.pt/multimedia/screenshots/p8505/preview/p8505_2_202103062353197726_preview.vtt', 'asset_type': 'video', 'asset_size': '925421114', 'asset_path': '/h264/512x384/p8505/p8505_2_202103062353197726.mp4', 'hls_url': 'http://streaming-ondemand.rtp.pt/nas2.share/h264/512x384/p8505/index.m3u8?tlm=hls&streams=p8505_2_202103062353197726.mp4.m3u8:1024', 'hls_url_new': 'http://streaming-ondemand.rtp.pt/nas2.share/h264/512x384/p8505/p8505_2_202103062353197726/master.m3u8', 'dash_url': 'https://streaming-ondemand.rtp.pt/nas2.share/h264/512x384/p8505/p8505_2_202103062353197726/master.mpd', 'rights_api': 'https://www.rtp.pt/services/rtpplay?v=5&file=nas2.share/h264/512x384/p8505/p8505_2_202103062353197726.mp4', 'stream_url': {'http': {'standard': 'http://streaming-ondemand.rtp.pt/nas2.share/h264/512x384/p8505/index.m3u8?tlm=hls&streams=p8505_2_202103062353197726.mp4.m3u8:1024'}, 'rtmp': {'standard': {'streamer': 'rtpvodfs.fplive.net', 'application': 'rtpvod', 'file': '/nas2.share/h264/512x384/p8505/p8505_2_202103062353197726.mp4'}}}, 'download_url': 'http://cdn-ondemand.rtp.pt/nas2.share/h264/512x384/p8505/p8505_2_202103062353197726.mp4?d=1', 'subtitles': {'vtt': '', 'ebutt': ''}}, {'asset_id': '909902', 'asset_internal_id': '385075', 'asset_duration': '00:48:53', 'asset_is_live': '1', 'asset_thumbnail': 'https://cdn-images.rtp.pt/multimedia/screenshots/p8505/p8505_3_202103060000000574.jpg?format=pjpg&w=400', 'asset_video_preview': '//cdn-ondemand.rtp.pt/nas2.share/h264/512x384/p8505/preview/p8505_3_202103060000000574_videoprv.mp4', 'asset_date': '2021-03-07 00:36:32', 'asset_base_dir': 'nas2.share/videos/auto/', 'asset_internal_id_info': '385272', 'asset_lastchanged': '2021-03-07 07:09:39', 'asset_geoblock_info': '2', 'asset_screenshot_preview': '//cdn-images.rtp.pt/multimedia/screenshots/p8505/preview/p8505_3_202103060000000574_preview.vtt', 'asset_type': 'video', 'asset_size': '807529665', 'asset_path': '/h264/512x384/p8505/p8505_3_202103060000000574.mp4', 'hls_url': 'http://streaming-ondemand.rtp.pt/nas2.share/h264/512x384/p8505/index.m3u8?tlm=hls&streams=p8505_3_202103060000000574.mp4.m3u8:1024', 'hls_url_new': 'http://streaming-ondemand.rtp.pt/nas2.share/h264/512x384/p8505/p8505_3_202103060000000574/master.m3u8', 'dash_url': 'https://streaming-ondemand.rtp.pt/nas2.share/h264/512x384/p8505/p8505_3_202103060000000574/master.mpd', 'rights_api': 'https://www.rtp.pt/services/rtpplay?v=5&file=nas2.share/h264/512x384/p8505/p8505_3_202103060000000574.mp4', 'stream_url': {'http': {'standard': 'http://streaming-ondemand.rtp.pt/nas2.share/h264/512x384/p8505/index.m3u8?tlm=hls&streams=p8505_3_202103060000000574.mp4.m3u8:1024'}, 'rtmp': {'standard': {'streamer': 'rtpvodfs.fplive.net', 'application': 'rtpvod', 'file': '/nas2.share/h264/512x384/p8505/p8505_3_202103060000000574.mp4'}}}, 'download_url': 'http://cdn-ondemand.rtp.pt/nas2.share/h264/512x384/p8505/p8505_3_202103060000000574.mp4?d=1', 'subtitles': {'vtt': '', 'ebutt': ''}}]}
        :param program_id:
        :param episode_id:
        :return:
        """
        return self.__request("get-episode/{}/{}?include_assets=true&include_webparams".format(program_id, episode_id))

    def list_episodes(self, program_id, per_page=30, one_page=0):
        """
            Example: [[{'paging': {'rpp': 100, 'current_page': 8505, 'total': 0}, 'program': {'program_order_by': 'ASC', 'program_image': 'https://cdn-images.rtp.pt/?format=pjpg&w=400', 'program_serie_label': '', 'program_onair_site': 'rtpplay'}, 'episodes': ''}]]
        :param per_page:
        :param program_id:
        :param one_page: optional - page number (returns only this page)
        :return: a list of pages, including all the entries up to 3 pages, unless a page number is supplied
        """
        return self.__request_paginated("list-episodes/{}?page={{}}&rpp={}".format(program_id, per_page), one_page)

    def list_programs(self, category_id=None, per_page=30, one_page=0):
        """
            Example: [[{'paging': {'rpp': 100, 'current_page': 8505, 'total': 0}, 'program': {'program_order_by': 'ASC', 'program_image': 'https://cdn-images.rtp.pt/?format=pjpg&w=400', 'program_serie_label': '', 'program_onair_site': 'rtpplay'}, 'episodes': ''}]]
        :param per_page:
        :param category_id:
        :param one_page: optional - page number (returns only this page)
        :return: a list of pages, including all the entries up to 3 pages, unless a page number is supplied
        """
        if category_id:
            return self.__request_paginated("list-programs/?category_id={}&page={{}}&rpp=100".format(category_id), one_page)
        return self.__request_paginated("list-programs/?page={{}}&rpp={}".format(category_id, per_page), one_page)

    def search(self, search_term, per_page=30, one_page=0):
        """

        :param search_term: query string
        :param per_page:
        :param one_page:
        :return:
        """
        return self.__request_paginated("search/?query={}&page={{}}&rpp={}".format(
            urllib.parse.quote_plus(search_term), per_page), one_page)

    def __request_paginated(self, request, one_page):
        """
            stops at 3 pages
        :param request:
        :param one_page:
        :return:
        """
        pages = []
        page = max(one_page, 1)
        while True:
            current_page = self.__request(request.format(page))
            pages.append([current_page])
            if current_page["paging"]["total"] <= current_page["paging"]["rpp"] * page or one_page != 0 or page == 3:
                break
            page += 1
        return pages
