import inspect
import json
import os
import pathlib
import sys
import threading

from g4epy.console_run_sink import ConsoleRunSink

from .test_env import is_notebook
from .runner import run, do_execute, sink_printer, stream_subprocess, console_printer


class UserPlugin(object):
    def __init__(self, plugin_path, **kwargs):
        self.config = {}
        self.user_commands = []    # Direct execution commands user added
        self.commands = []         # Total list of commands, generated by g4epy + user_commands
        self.user_macros = []      # a list of user macroses to run

        self.config.update(kwargs)  # update configs

        sink = self.config.get('sink', 'auto')
        if not sink:
            self.is_notebook = False
            self.sink = ConsoleRunSink()
        else:
            self.is_notebook = is_notebook()
            if (sink == 'auto' and self.is_notebook) or sink == 'notebook':
                # NotebookRunSink requires IPython which might not be installed if one only wants to work with console
                from g4epy.notebook_run_sink import NotebookRunSink
                self.sink = NotebookRunSink()
            else:
                self.sink = ConsoleRunSink()

        if not plugin_path:
            raise RuntimeError("")
        default_build_prefix = os.path.join(plugin_path, 'cmake-build')
        self.config['build_prefix'] = self.config.get('build_prefix', default_build_prefix)

        self.config['install_prefix'] = self.config.get('install_prefix', plugin_path)
        self.runner = None

    def run_cmake_target(self, target, suffix=''):
        """Builds G4E"""
        # Generate execution file        

        command = f"cmake --build {self.config['build_prefix']} --target {target} {suffix}"
        
        self.sink.to_show = ["%]", "Error", "ERROR", "FATAL"]   # "[" - Cmake like "[9%]"
        self.sink.show_running_command(command)
        # if not self.sink.is_displayed:
        self.sink.display()
        run(command, self.sink, cwd=self.config['build_prefix'])

    def build(self, threads='auto'):
        self.ensure_build_dir_exist()
        if threads == 'auto':
            import multiprocessing
            threads = multiprocessing.cpu_count()
            if threads > 2:
                threads -= 1
                
        suffix = f' -- -j {threads} -w '
        

        self.run_cmake_target('g4e', suffix)

    def clean(self):
        self.run_cmake_target('clean')

    def ensure_build_dir_exist(self):
        if not os.path.exists(self.config['build_prefix']):
            err_msg = f"Error! CMake build path {self.config['build_prefix']} does not exist.\n"\
                      f"You may create it (if you sure it is right) with the command:\n"\
                      f"  mkdir -p {self.config['build_prefix']}"
            raise ValueError(err_msg)
        

    def cmake_configure(self, build_type='Debug', silence_warnings=True):
        """
        RelwithDebInfo
        :param build_type:
        :param silence_warnings:
        :return:
        """
        self.ensure_build_dir_exist()

        flags = ""
        if silence_warnings:
            flags += " -DG4E_SILENCE_WARNINGS=1"

        command = f"cmake {flags} -DCMAKE_BUILD_TYPE={build_type} {self.config['g4e_home']}"
        self.sink.to_show = [">>>", "-- Configuring done", "-- Generating done", "Error"]  # "[" - Cmake like "[9%]"
        self.sink.show_running_command(command)

        # if not self.sink.is_displayed:
        self.sink.display()
        run(command, self.sink, cwd=self.config['build_prefix'])


    def _repr_html_(self):
        result_str = f"<strong>eJANA plugin builder</strong><br>"
        return result_str
