import asyncio
from asyncio import exceptions
import datetime
from http import HTTPStatus
import logging
import aiohttp
import requests
from requests.models import Response
from typing import Optional, Any, Callable
from jls_utils.abstract.abstract import *
from jls_utils.secondary.const import *
from jls_utils.secondary import model_from_json, Config, show_logs_func

# DocsController


def get_docs(*args, **kwargs) -> ResultIds:
    return model_from_json(send_to_core_get(DOCS(None), *args, **kwargs), ResultIds)


def get_docs_id(id: str, *args, **kwargs) -> ResultDoc:
    return model_from_json(send_to_core_get(DOCS_ID(id, None), *args, **kwargs), ResultDoc)


def post_docs(data: DocWithName, wait: bool, *args, **kwargs) -> Alias.Id:
    return send_to_core_modify(DOCS(wait), data, *args, **kwargs)


def post_docs_id(id: str, data: DocWithName, wait: bool, *args, **kwargs) -> Alias.Id:
    return send_to_core_modify(DOCS_ID(id, wait), data, *args, **kwargs)


def delete_docs_id(id: str, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(DOCS_ID(id, wait), *args, **kwargs, is_post=False)


def delete_docs(wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(DOCS(wait), *args, **kwargs, is_post=False)

# CollectionsController


def get_collections(*args, **kwargs) -> ResultOwnAndSharedIds:
    return model_from_json(send_to_core_get(COLLECTIONS(None), *args, **kwargs), ResultOwnAndSharedIds)


def get_collections_name(name: str, operation_id: Optional[str], run_id: Optional[str], *args, **kwargs) -> ResultOwnAndSharedIds:
    return model_from_json(send_to_core_get(COLLECTIONS_IDS_NAME(name, operation_id, run_id), *args, **kwargs), ResultOwnAndSharedIds)


def get_collection_name(name: str, operation_id: Optional[str], run_id: Optional[str], offset: int, limit: int, *args, **kwargs) -> ResultCollection:
    return model_from_json(send_to_core_get(COLLECTIONS_NAME(name, operation_id, run_id, offset, limit), *args, **kwargs), ResultCollection)


def get_collections_id(id: str, offset: int, limit: int, *args, **kwargs) -> ResultCollection:
    return model_from_json(send_to_core_get(COLLECTIONS_ID(id, offset, limit), *args, **kwargs), ResultCollection)


def post_collections(data: DocsCollection, wait: bool, *args, **kwargs) -> Alias.Id:
    return send_to_core_modify(COLLECTIONS(wait), data, *args, **kwargs)


def post_collections_id(id: str, data: DocsCollection, wait: bool, *args, **kwargs) -> Alias.Id:
    return send_to_core_modify(COLLECTIONS_ID_MODIFY(id, wait), data, *args, **kwargs)


def post_collections_id_s3(id: str, data: PostS3Settings, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(COLLECTIONS_ID_S3(id, wait), data, *args, **kwargs)


def post_collections_data(data: DocsDataCollection, wait: bool=True, *args, **kwargs) -> Alias.Id:
    return send_to_core_modify(COLLECTIONS_DATA(wait), data, *args, **kwargs)


def post_collections_id_add(id: str, data: DocsCollectionChange, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(COLLECTIONS_ID_ADD(id, wait), data, *args, **kwargs)


def post_collections_id_copy(id: str, full_copy: bool, wait: bool, *args, **kwargs) -> Alias.Id:
    return send_to_core_modify(COLLECTIONS_ID_COPY(id, full_copy, wait), *args, **kwargs)


def post_collections_id_applyScheme(id: str, data: FixScheme, wait: bool, *args, **kwargs) -> Alias.Id:
    return send_to_core_modify(COLLECTIONS_APPLY_SCHEME(id, wait), data, *args, **kwargs)


def post_collections_id_fixScheme(id: str, data: FixScheme, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(COLLECTIONS_FIX_SCHEME(id, wait), data, *args, **kwargs)


def post_collections_id_unfixScheme(id: str, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(COLLECTIONS_UNFIX_SCHEME(id, wait), *args, **kwargs)


def post_collections_metadata(id: str, data: CollectionMetadata, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(COLLECTIONS_METADATA(id, wait), data, *args, **kwargs)


def delete_collections(wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(COLLECTIONS(wait), *args, **kwargs, is_post=False)


def delete_collections_id(id: str, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(COLLECTIONS_ID_MODIFY(id, wait), *args, **kwargs, is_post=False)


def delete_collections_id_s3(key: str, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(COLLECTIONS_ID_S3(key, wait), *args, **kwargs, is_post=False)


def delete_collections_id_del(id: str, data: DocsCollectionChange, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(COLLECTIONS_ID_DEL(id, wait), data, *args, **kwargs, is_post=False)

# SchemeController


def get_schemes(*args, **kwargs) -> ResultOwnAndSharedIds:
    return model_from_json(send_to_core_get(SCHEMES(None), *args, **kwargs), ResultOwnAndSharedIds)


def get_schemes_id(id: str, *args, **kwargs) -> ResultScheme:
    return model_from_json(send_to_core_get(SCHEMES_ID(id, None), *args, **kwargs), ResultScheme)


def get_schemes_id_raw(id: str, *args, **kwargs) -> Alias.Json:
    return send_to_core_get(SCHEMES_ID_RAW(id), *args, **kwargs)


def get_schemes_mapping(scheme_from_id: str, scheme_to_id: str, *args, **kwargs) -> ResultMapping:
    return model_from_json(send_to_core_get(SCHEMES_MAPPING_IDS(scheme_from_id, scheme_to_id), *args, **kwargs), ResultMapping)


def post_schemes(data: SchemeWithName, wait: bool, *args, **kwargs) -> Alias.Id:
    return send_to_core_modify(SCHEMES(wait), data, *args, **kwargs)


def post_schemes_id(id: str, data: SchemeWithName, wait: bool, *args, **kwargs) -> Alias.Id:
    return send_to_core_modify(SCHEMES_ID(id, wait), data, *args, **kwargs)


def post_schemes_mapping(data: SchemesFixMapping, wait: bool, *args, **kwargs) -> Alias.Id:
    return send_to_core_modify(SCHEMES_MAPPING(wait), data, *args, **kwargs)


def delete_schemes(wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(SCHEMES(wait), *args, **kwargs, is_post=False)


def delete_schemes_id(id: str, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(SCHEMES_ID(id, wait), *args, **kwargs, is_post=False)


def delete_schemes_mapping(data: SchemesIds, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(SCHEMES_MAPPING(wait), data, *args, **kwargs, is_post=False)

# CommonController


def get_check_auth(*args, **kwargs) -> Alias.Info:
    return send_to_core_get(CHECK, *args, **kwargs, is_text=True)


def get_ping(*args, **kwargs) -> Alias.Info:
    return send_to_core_get(PING, *args, **kwargs, is_text=True)


# def get_mapping(wait: bool, *args, **kwargs) -> ResultDocMappingsFull:
#     return model_from_json(send_to_core_get(MAPPING, *args, **kwargs), ResultDocMappingsFull)


# def get_mapping_id(id: str, wait: bool, *args, **kwargs) -> ResultMapping:
#     return model_from_json(send_to_core_get(MAPPING_ID(id, None), *args, **kwargs), ResultMapping)


# def post_mapping(data: DocsAndScheme, wait: bool, *args, **kwargs) -> Alias.Info:
#     return send_to_core_modify(MAPPING(wait), data, *args, **kwargs)


# def delete_mapping_id(id: str, wait: bool, *args, **kwargs) -> Alias.Info:
#     return send_to_core_modify(MAPPING_ID(id, wait), *args, **kwargs, is_post=False)


# def delete_all(wait: bool, *args, **kwargs) -> Alias.Info:
#     return send_to_core_modify(COMMON_ALL(wait), *args, **kwargs, is_post=False)

# UserShareController


def get_share_collection_id(id: str, *args, **kwargs) -> ResultLogins:
    return model_from_json(send_to_core_get(SHARE_COLLECTION_ID(id, None), *args, **kwargs), ResultLogins)


def get_share_scheme_id(id: str, *args, **kwargs) -> ResultLogins:
    return model_from_json(send_to_core_get(SHARE_SCHEME_ID(id, None), *args, **kwargs), ResultLogins)


def get_share_userApp_id(id: str, *args, **kwargs) -> ResultLogins:
    return model_from_json(send_to_core_get(SHARE_USER_APP_ID(id, None), *args, **kwargs), ResultLogins)


def get_share_login(login: str, *args, **kwargs) -> ResultSharedForLogin:
    return model_from_json(send_to_core_get(SHARE_LOGIN(login), *args, **kwargs), ResultSharedForLogin)


def post_share_collection_id(id: str, data: SharedWithUsers, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(SHARE_COLLECTION_ID(id, wait), data, *args, **kwargs)


def post_share_scheme_id(id: str, data: SharedWithUsers, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(SHARE_SCHEME_ID(id, wait), data, *args, **kwargs)


def post_share_userApp_id(id: str, data: SharedWithUsers, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(SHARE_USER_APP_ID(id, wait), data, *args, **kwargs)


def post_share(data: Shared, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(SHARE(wait), data, *args, **kwargs)


def delete_share(data: Shared, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(SHARE(wait), data, *args, **kwargs, is_post=False)


def delete_share_all(wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(SHARE_ALL(wait), *args, **kwargs, is_post=False)

# RegistrationController


def get_register_login(login: str, *args, **kwargs) -> Alias.Info:
    return send_to_core_get(REGISTER_LOGIN(login, None), is_text=True, *args, **kwargs)


def get_register_all(*args, **kwargs) -> ResultLogins:
    return model_from_json(send_to_core_get(REGISTER_ALL, *args, **kwargs), ResultLogins)


def post_register(auth: Optional[AUTH]=None, conn_url: Optional[str]=None) -> Alias.Info:
    if auth is None:
        auth = (Config.CORE_USERNAME, Config.CORE_PASSWORD)
    send_to_core_modify(REGISTER, User(login=auth[0], password=auth[1]), with_auth=False, with_show=False)
    info = f"user {auth[0]} created"
    if Config.VERBOSE:
        logging.info(info)
    return info


def delete_register(auth: Optional[AUTH]=None, conn_url: Optional[str]=None, *args, **kwargs) -> Alias.Info:
    send_to_core_modify(REGISTER, *args, **kwargs, auth=auth, with_show=False, is_post=False)
    login = auth[0] if auth is not None else Config.CORE_USERNAME
    info = f"user {login} removed"
    if Config.VERBOSE:
        logging.info(info)
    return info


def delete_register_login(login: str, wait: bool, *args, **kwargs) -> Alias.Info:
    send_to_core_modify(REGISTER_LOGIN(login, wait), *args, **kwargs, with_show=False, is_post=False)
    info = f"user {login} removed"
    if Config.VERBOSE:
        logging.info(info)
    return info

# UserAppsController


def get_userApps(*args, **kwargs) -> ResultOwnAndSharedIds:
    return model_from_json(send_to_core_get(USER_APPS(None), *args, **kwargs), ResultOwnAndSharedIds)


def get_userApps_realIds(*args, **kwargs) -> ResultOwnAndSharedIds:
    return model_from_json(send_to_core_get(USER_APPS_REAL_IDS, *args, **kwargs), ResultOwnAndSharedIds)


def get_userApps_mapIds(*args, **kwargs) -> ResultOwnAndSharedIdsMap:
    return model_from_json(send_to_core_get(USER_APPS_MAP_IDS, *args, **kwargs), ResultOwnAndSharedIdsMap)


def get_userApps_id(id: str, *args, **kwargs) -> UserApp:
    return model_from_json(send_to_core_get(USER_APPS_ID(id, None), *args, **kwargs), UserApp)


def get_userApps_realId(id: str, *args, **kwargs) -> UserApp:
    return model_from_json(send_to_core_get(USER_APPS_REAL_ID(id), *args, **kwargs), UserApp)


def post_userApps(data: UserApp, wait: bool, *args, **kwargs) -> Alias.Id:
    return send_to_core_modify(USER_APPS(wait), data, *args, **kwargs)


def post_userApps_id(id: str, data: UserApp, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(USER_APPS_ID(id, wait), data, *args, **kwargs)


def delete_userApps(wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(USER_APPS(wait), *args, **kwargs, is_post=False)


def delete_userApps_id(id: str, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(USER_APPS_ID(id, wait), *args, **kwargs, is_post=False)

# UserTasksController


def get_userTasks(*args, **kwargs) -> ResultIds:
    return model_from_json(send_to_core_get(USER_TASKS(None), *args, **kwargs), ResultIds)


def get_userTasks_realIds(*args, **kwargs) -> ResultIds:
    return model_from_json(send_to_core_get(USER_TASKS_REAL_IDS, *args, **kwargs), ResultIds)


def get_userTasks_mapIds(*args, **kwargs) -> ResultIdsMap:
    return model_from_json(send_to_core_get(USER_TASKS_MAP_IDS, *args, **kwargs), ResultIdsMap)


def get_userTasks_id(id: str, *args, **kwargs) -> UserTask:
    return model_from_json(send_to_core_get(USER_TASKS_ID(id, None), *args, **kwargs), UserTask)


def get_userTasks_realId(id: str, *args, **kwargs) -> UserTask:
    return model_from_json(send_to_core_get(USER_TASKS_REAL_ID(id), *args, **kwargs), UserTask)


def post_userTasks(data: UserTask, wait: bool, *args, **kwargs) -> Alias.Id:
    return send_to_core_modify(USER_TASKS(wait), data, *args, **kwargs)


def post_userTasks_id(id: str, data: UserTask, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(USER_TASKS_ID(id, wait), data, *args, **kwargs)


def delete_userTasks(wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(USER_TASKS(wait), *args, **kwargs, is_post=False)


def delete_userTasks_id(id: str, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(USER_TASKS_ID(id, wait), *args, **kwargs, is_post=False)

# UserCfgsController


def get_userCfgs(*args, **kwargs) -> ResultIds:
    return model_from_json(send_to_core_get(USER_CFGS(None), *args, **kwargs), ResultIds)


def get_userCfgs_realIds(*args, **kwargs) -> ResultIds:
    return model_from_json(send_to_core_get(USER_CFGS_REAL_IDS, *args, **kwargs), ResultIds)


def get_userCfgs_mapIds(*args, **kwargs) -> ResultIdsMap:
    return model_from_json(send_to_core_get(USER_CFGS_MAP_IDS, *args, **kwargs), ResultIdsMap)


def get_userCfgs_id(id: str, *args, **kwargs) -> ResultUserCfg:
    return model_from_json(send_to_core_get(USER_CFGS_ID(id, None), *args, **kwargs), ResultUserCfg)


def get_userCfgs_realId(id: str, *args, **kwargs) -> ResultUserCfg:
    return model_from_json(send_to_core_get(USER_CFGS_REAL_ID(id), *args, **kwargs), ResultUserCfg)


def post_userCfgs(data: UserCfg, wait: bool, *args, **kwargs) -> Alias.Id:
    return send_to_core_modify(USER_CFGS(wait), data, *args, **kwargs)


def post_userCfgs_id(id: str, data: UserCfg, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(USER_CFGS_ID(id, wait), data, *args, **kwargs)


def delete_userCfgs(wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(USER_CFGS(wait), *args, **kwargs, is_post=False)


def delete_userCfgs_id(id: str, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(USER_CFGS_ID(id, wait), *args, **kwargs, is_post=False)

# OperationResultsController


def get_operationResults(*args, **kwargs) -> ResultIds:
    return model_from_json(send_to_core_get(OPERATION_RESULTS(None), *args, **kwargs), ResultIds)


def get_operationResults_id(id: str, is_text=True, *args, **kwargs) -> Optional[str]:
    return send_to_core_get(OPERATION_RESULTS_ID(id, None), is_text=is_text, *args, **kwargs)


def delete_operationResults(wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(OPERATION_RESULTS(wait), *args, **kwargs, is_post=False)


def delete_operationResults_id(id: str, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(OPERATION_RESULTS_ID(id, wait), *args, **kwargs, is_post=False)

# TempRunController


def get_run_condition(id: str, *args, **kwargs) -> Condition:
    return model_from_json(send_to_core_get(TEMP_RUN_CONDITION(id), *args, **kwargs), Condition)


def get_run_activeRuns(*args, **kwargs) -> ResultIds:
    return model_from_json(send_to_core_get(TEMP_RUN_ACTIVE_RUNS, *args, **kwargs), ResultIds)


def get_run_mainTaskCfg(id: str, *args, **kwargs) -> MainTaskCfg:
    return model_from_json(send_to_core_get(TEMP_RUN_MAIN_TASK_CFG(id), *args, **kwargs), MainTaskCfg)


def get_run_operationsIds(task_id: str, cfg_id: Optional[str]=None, *args, **kwargs) -> ResultIds:
    return model_from_json(send_to_core_get(TEMP_RUN_OPERATIONS_IDS(task_id, cfg_id), *args, **kwargs), ResultIds)

# ManagerController


def get_manager_logs(data: LogsTask, with_show: bool, *args, **kwargs) -> AppLogs:
    return model_from_json(send_to_core_modify(MANAGER_LOGS, data, with_show=with_show, show_func=show_logs_func, *args, **kwargs), AppLogs)


def get_clickhouse_all(*args, **kwargs) -> str:
    return send_to_core_get(MANAGER_CLICKHOUSE_ALL, is_text=True, *args, **kwargs)


def get_clickhouse_id(id: str, *args, **kwargs) -> str:
    return send_to_core_get(MANAGER_CLICKHOUSE_ID(id), is_text=True, *args, **kwargs)


def get_manager_dagKeyValue_operationId(id: str, *args, **kwargs) -> Alias.Json:
    return send_to_core_get(MANAGER_DAG_KEY_VALUE_OPERATION_ID(id), *args, **kwargs)


def post_manager_dagKeyValue(data: KeysValues, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(MANAGER_DAG_KEY_VALUE(wait), data, *args, **kwargs)


def get_app_info(id: str, parse: bool, *args, **kwargs) -> Union[Alias.Json, AppFunctionsInfo]:
    res = send_to_core_get(MANAGER_APP_INFO(id), *args, **kwargs)
    if parse:
        return model_from_json(res, AppFunctionsInfo)
    return res


def get_app_info_by_real_id(id: str, *args, **kwargs) -> Alias.Json:
    return send_to_core_get(MANAGER_APP_INFO_REAL_ID(id), *args, **kwargs)


def get_task_schedules(data: Operation, with_show: bool, *args, **kwargs) -> Schedules:
    res = model_from_json(send_to_core_modify(MANAGER_TASK_SCHEDULES, data, with_show=False, *args, **kwargs), Schedules)
    if with_show:
        logging.info(res)
    return res


def post_manager_task(data: MainTask, with_show: bool, long: bool, long_timeout: int, wait: bool, auth: Optional[AUTH], conn_url: Optional[str]=None, *args, **kwargs) -> AppLogs:
    check_profile_mode(data.profileMode)
    res = send_to_core_modify(MANAGER_TASK(wait and not long), data, with_show=with_show, show_func=show_logs_func, auth=auth, *args, **kwargs)
    if wait and long:
        res = asyncio.run(__get_result(res, timeout=long_timeout, auth=auth))
    return AppLogs.parse_raw(res)


def post_manager_task_run(data: RunTask, with_show: bool, long: bool, long_timeout: int, wait: bool, auth: Optional[AUTH], conn_url: Optional[str]=None, *args, **kwargs) -> Optional[AppLogs]:
    check_profile_mode(data.profileMode)
    res = send_to_core_modify(MANAGER_TASK_RUN(wait and not long), data, with_show=with_show, show_func=show_logs_func, auth=auth, *args, **kwargs)
    if wait and long:
        res = asyncio.run(__get_result(res, timeout=long_timeout, auth=auth))
    if data.withLogs or data.schedule is not None:
        return AppLogs.parse_raw(res)


def post_manager_task_unschedule(data: UnscheduleOperation, with_show: bool, wait: bool, *args, **kwargs) -> Alias.Info:
    return send_to_core_modify(MANAGER_TASK_UNSCHEDULE(wait), data, with_show=with_show, *args, **kwargs)


def post_manager_task_stop(data: StopOperation, with_show: bool, wait: bool, *args, **kwargs) -> Optional[AppLogs]:
    show_func = show_logs_func if data.withLogs else None
    res = send_to_core_modify(MANAGER_TASK_STOP(wait), data, with_show=with_show, show_func=show_func, *args, **kwargs)
    if data.withLogs:
        return AppLogs.parse_raw(res)


def post_manager_task_stop_all(data: StopOperationMany, wait: bool, *args, **kwargs) -> Alias.Json: # FIXME show
    return send_to_core_modify(MANAGER_TASK_STOP_ALL(wait), data, *args, **kwargs)


def post_manager_task_resume(data: Operation, wait: bool, *args, **kwargs) -> Alias.Empty:
    return send_to_core_modify(MANAGER_TASK_RESUME(wait), data, *args, **kwargs)


def post_manager_task_pause(data: Operation, wait: bool, *args, **kwargs) -> Alias.Empty:
    return send_to_core_modify(MANAGER_TASK_PAUSE(wait), data, *args, **kwargs)


def post_manager_app_stop(data: AppManage, wait: bool, *args, **kwargs) -> Alias.Empty:
    return send_to_core_modify(MANAGER_APP_STOP(wait), data, *args, **kwargs)


def post_manager_app_resume(data: AppManage, wait: bool, *args, **kwargs) -> Alias.Empty:
    return send_to_core_modify(MANAGER_APP_RESUME(wait), data, *args, **kwargs)


def post_manager_app_pause(data: AppManage, wait: bool, *args, **kwargs) -> Alias.Empty:
    return send_to_core_modify(MANAGER_APP_PAUSE(wait), data, *args, **kwargs)


async def kafka_send(data: KafkaMsg, *args, **kwargs) -> Union[Alias.Info, KafkaMsg]:
    result = await send_to_core_post_async(KAFKA_SEND, data, *args, **kwargs)
    try:
        return KafkaMsg.parse_raw(result)
    except:
        return result


##################################


async def __get_result(id: str, is_text: bool = True, check_time: float = LONG_SLEEP_TIME, timeout: int = WAIT_RESULT_TIMEOUT, auth: Optional[AUTH]=None, conn_url: Optional[str]=None) -> str:      # FIXME
    host = Config.HOST_PORT if conn_url is None else conn_url
    assert host is not None, "host port not set"
    if auth is None:
        auth = aiohttp.BasicAuth(login=Config.CORE_USERNAME, password=Config.CORE_PASSWORD, encoding='utf-8')
    
    timeout_deadline = None if timeout is None else datetime.datetime.now() + datetime.timedelta(0, timeout)
    while True:
        try:
            async with aiohttp.ClientSession(auth=auth, connector=aiohttp.TCPConnector(verify_ssl=False), timeout=AIOHTTP_TIMEOUT_MINI) as session:
                while True:
                    async with session.get(f"{host}{OPERATION_RESULTS_ID(id, None)}", headers=HEADERS) as response:
                        if response.ok:
                            if is_text:
                                result = await response.text()
                            else:
                                result = await response.json()
                            return result    
                        else:
                            if timeout_deadline is not None:
                                now = datetime.datetime.now()
                                if now >= timeout_deadline:
                                    raise Exception("wait get result timeout")
                            await asyncio.sleep(check_time)
        except exceptions.TimeoutError:
            pass    # recreate ClientSession


async def async_check_response(response: aiohttp.ClientResponse, show_func: Optional[Callable]=None):
    if not response.ok:
        if show_func is None:
            logging.error(await response.text())
        else:
            show_func(await response.text(), err=True)
    response.raise_for_status()


def check_response(path: str, response: Response, show_func: Optional[Callable]=None):
    if response.status_code >= 400:
        if show_func is None:
            text = response.text
            msg = f"failed: {text}" if len(text) > 0 else "failed"
            logging.error(f"{path} {msg}")
        else:
            logging.error(f"{path} failed")
            show_func(response.text, err=True)
    response.raise_for_status()


def send_to_core_get(path: str, with_auth=True, is_text=False, auth: Optional[AUTH]=None, conn_url: Optional[str]=None) -> Optional[str]:
    host = Config.HOST_PORT if conn_url is None else conn_url
    assert host is not None, "host port not set"
    if auth is None or not with_auth:
        auth = (Config.CORE_USERNAME, Config.CORE_PASSWORD) if with_auth else None
    response = requests.get(f"{host}{path}", headers=HEADERS, auth=auth)
    check_response(f"{host}{path}", response)
    if response.status_code == HTTPStatus.NO_CONTENT:
        return None
    if is_text:
        return response.text
    else:
        return response.json()


def send_to_core_modify(path: str, operation: Optional[Any] = None, with_auth: bool=True, with_show: Optional[bool]=None, show_func: Optional[Callable]=None, is_post: bool=True, auth: Optional[AUTH]=None, conn_url: Optional[str]=None) -> str:
    """modify: post by default, else - delete"""
    host = Config.HOST_PORT if conn_url is None else conn_url
    assert host is not None, "host port not set"
    if auth is None:
        auth = (Config.CORE_USERNAME, Config.CORE_PASSWORD)
    if operation is not None:
        operation = operation.json()
    if is_post:
        response = requests.post(f"{host}{path}", data=operation, headers=HEADERS, auth=auth if with_auth else None)
    else:   # delete
        response = requests.delete(f"{host}{path}", data=operation, headers=HEADERS, auth=auth if with_auth else None)
    check_response(f"{host}{path}", response, show_func=show_func)
    if response.status_code == HTTPStatus.NO_CONTENT:
        return ""
    result = response.text
    if with_show is None:
        with_show = Config.VERBOSE
    if with_show:
        if show_func is None:
            logging.info(result)
        else:
            show_func(result)
    return result


async def send_to_core_post_async(path: str, operation: Optional[str] = None, with_auth=True, with_show=True, show_func: Optional[Callable]=None, auth: Optional[AUTH]=None, conn_url: Optional[str]=None) -> str:
    host = Config.HOST_PORT if conn_url is None else conn_url
    assert host is not None, "host port not set"
    if auth is None or not with_auth:
        auth = aiohttp.BasicAuth(login=Config.CORE_USERNAME, password=Config.CORE_PASSWORD, encoding='utf-8') if with_auth else None
    if operation is not None:
        operation = operation.json()

    async with aiohttp.ClientSession(auth=auth, connector=aiohttp.TCPConnector(verify_ssl=False), timeout=AIOHTTP_TIMEOUT) as session:
        async with session.post(f"{host}{path}", data=operation, headers=HEADERS) as response:
            await async_check_response(response, show_func=show_func)
            result = await response.text()
    if with_show:
        if show_func is None:
            logging.info(result)
        else:
            show_func(result)
    return result


def check_profile_mode(profile_mode: Optional[str]):
    assert profile_mode in [None, "no", "all", "time", "df_info", "df_show"], f"wrong profile_mode: {profile_mode}"
