"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const eks = require("../lib");
const CLUSTER_VERSION = eks.KubernetesVersion.V1_16;
module.exports = {
    'can be added to a cluster'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            selectors: [{ namespace: 'default' }],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfilePodExecutionRole4795C054', 'Arn'] },
                selectors: [{ namespace: 'default' }],
            },
        }));
        test.done();
    },
    'supports specifying a profile name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            fargateProfileName: 'MyProfileName',
            selectors: [{ namespace: 'default' }],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfilePodExecutionRole4795C054', 'Arn'] },
                selectors: [{ namespace: 'default' }],
                fargateProfileName: 'MyProfileName',
            },
        }));
        test.done();
    },
    'supports custom execution role'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
        const myRole = new iam.Role(stack, 'MyRole', { assumedBy: new iam.AnyPrincipal() });
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            podExecutionRole: myRole,
            selectors: [{ namespace: 'default' }],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyRoleF48FFE04', 'Arn'] },
                selectors: [{ namespace: 'default' }],
            },
        }));
        test.done();
    },
    'supports tags through aspects'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            selectors: [{ namespace: 'default' }],
        });
        core_1.Tag.add(stack, 'aspectTag', 'hello');
        core_1.Tag.add(cluster, 'propTag', '123');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                selectors: [{ namespace: 'default' }],
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfilePodExecutionRole4795C054', 'Arn'] },
                tags: {
                    propTag: '123',
                    aspectTag: 'hello',
                },
            },
        }));
        test.done();
    },
    'supports specifying vpc'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
        const vpc = ec2.Vpc.fromVpcAttributes(stack, 'MyVpc', {
            vpcId: 'vpc123',
            availabilityZones: ['az1'],
            privateSubnetIds: ['priv1'],
        });
        // WHEN
        cluster.addFargateProfile('MyProfile', {
            selectors: [{ namespace: 'default' }],
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfilePodExecutionRole4795C054', 'Arn'] },
                selectors: [{ namespace: 'default' }],
                subnets: ['priv1'],
            },
        }));
        test.done();
    },
    'fails if there are no selectors or if there are more than 5'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
        // THEN
        test.throws(() => cluster.addFargateProfile('MyProfile', { selectors: [] }));
        test.throws(() => cluster.addFargateProfile('MyProfile', {
            selectors: [
                { namespace: '1' },
                { namespace: '2' },
                { namespace: '3' },
                { namespace: '4' },
                { namespace: '5' },
                { namespace: '6' },
            ],
        }));
        test.done();
    },
    'FargateCluster creates an EKS cluster fully managed by Fargate'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new eks.FargateCluster(stack, 'FargateCluster', { version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesPatch', {
            ResourceName: 'deployment/coredns',
            ResourceNamespace: 'kube-system',
            ApplyPatchJson: '{"spec":{"template":{"metadata":{"annotations":{"eks.amazonaws.com/compute-type":"fargate"}}}}}',
            RestorePatchJson: '{"spec":{"template":{"metadata":{"annotations":{"eks.amazonaws.com/compute-type":"ec2"}}}}}',
            ClusterName: {
                Ref: 'FargateCluster019F03E8',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: {
                    Ref: 'FargateCluster019F03E8',
                },
                podExecutionRoleArn: {
                    'Fn::GetAtt': [
                        'FargateClusterfargateprofiledefaultPodExecutionRole66F2610E',
                        'Arn',
                    ],
                },
                selectors: [
                    { namespace: 'default' },
                    { namespace: 'kube-system' },
                ],
            },
        }));
        test.done();
    },
    'can create FargateCluster with a custom profile'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new eks.FargateCluster(stack, 'FargateCluster', {
            defaultProfile: {
                fargateProfileName: 'my-app', selectors: [{ namespace: 'foo' }, { namespace: 'bar' }],
            },
            version: CLUSTER_VERSION,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: {
                    Ref: 'FargateCluster019F03E8',
                },
                fargateProfileName: 'my-app',
                podExecutionRoleArn: {
                    'Fn::GetAtt': [
                        'FargateClusterfargateprofilemyappPodExecutionRole875B4635',
                        'Arn',
                    ],
                },
                selectors: [
                    { namespace: 'foo' },
                    { namespace: 'bar' },
                ],
            },
        }));
        test.done();
    },
    'custom profile name is "custom" if no custom profile name is provided'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new eks.FargateCluster(stack, 'FargateCluster', {
            defaultProfile: {
                selectors: [{ namespace: 'foo' }, { namespace: 'bar' }],
            },
            version: CLUSTER_VERSION,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: {
                    Ref: 'FargateCluster019F03E8',
                },
                podExecutionRoleArn: {
                    'Fn::GetAtt': [
                        'FargateClusterfargateprofilecustomPodExecutionRoleDB415F19',
                        'Arn',
                    ],
                },
                selectors: [
                    { namespace: 'foo' },
                    { namespace: 'bar' },
                ],
            },
        }));
        test.done();
    },
    'multiple Fargate profiles added to a cluster are processed sequentially'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { version: CLUSTER_VERSION });
        // WHEN
        cluster.addFargateProfile('MyProfile1', {
            selectors: [{ namespace: 'namespace1' }],
        });
        cluster.addFargateProfile('MyProfile2', {
            selectors: [{ namespace: 'namespace2' }],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Config: {
                clusterName: { Ref: 'MyCluster8AD82BF8' },
                podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfile1PodExecutionRole794E9E37', 'Arn'] },
                selectors: [{ namespace: 'namespace1' }],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-FargateProfile', {
            Properties: {
                ServiceToken: { 'Fn::GetAtt': [
                        'awscdkawseksClusterResourceProviderNestedStackawscdkawseksClusterResourceProviderNestedStackResource9827C454',
                        'Outputs.awscdkawseksClusterResourceProviderframeworkonEventEA97AA31Arn',
                    ] },
                AssumeRoleArn: { 'Fn::GetAtt': ['MyClusterCreationRoleB5FA4FF3', 'Arn'] },
                Config: {
                    clusterName: { Ref: 'MyCluster8AD82BF8' },
                    podExecutionRoleArn: { 'Fn::GetAtt': ['MyClusterfargateprofileMyProfile2PodExecutionRoleD1151CCF', 'Arn'] },
                    selectors: [{ namespace: 'namespace2' }],
                },
            },
            DependsOn: [
                'MyClusterfargateprofileMyProfile1PodExecutionRole794E9E37',
                'MyClusterfargateprofileMyProfile1879D501A',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'fargate role is added to RBAC'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new eks.FargateCluster(stack, 'FargateCluster', { version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::AWSCDK-EKS-KubernetesResource', {
            Manifest: {
                'Fn::Join': [
                    '',
                    [
                        '[{"apiVersion":"v1","kind":"ConfigMap","metadata":{"name":"aws-auth","namespace":"kube-system"},"data":{"mapRoles":"[{\\"rolearn\\":\\"',
                        {
                            'Fn::GetAtt': [
                                'FargateClusterfargateprofiledefaultPodExecutionRole66F2610E',
                                'Arn',
                            ],
                        },
                        '\\",\\"username\\":\\"system:node:{{SessionName}}\\",\\"groups\\":[\\"system:bootstrappers\\",\\"system:nodes\\",\\"system:node-proxier\\"]}]","mapUsers":"[]","mapAccounts":"[]"}}]',
                    ],
                ],
            },
        }));
        test.done();
    },
    'cannot be added to a cluster without kubectl enabled'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cluster = new eks.Cluster(stack, 'MyCluster', { kubectlEnabled: false, version: CLUSTER_VERSION });
        // WHEN
        test.throws(() => new eks.FargateProfile(stack, 'MyFargateProfile', {
            cluster,
            selectors: [{ namespace: 'default' }],
        }), /unsupported/);
        test.done();
    },
    'allow cluster creation role to iam:PassRole on fargate pod execution role'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new eks.FargateCluster(stack, 'FargateCluster', { version: CLUSTER_VERSION });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'iam:PassRole',
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'FargateClusterRole8E36B33A',
                                'Arn',
                            ],
                        },
                    },
                    {
                        Action: [
                            'ec2:DescribeSubnets',
                            'ec2:DescribeRouteTables',
                        ],
                        Effect: 'Allow',
                        Resource: '*',
                    },
                    {
                        Action: [
                            'eks:CreateCluster',
                            'eks:DescribeCluster',
                            'eks:DescribeUpdate',
                            'eks:DeleteCluster',
                            'eks:UpdateClusterVersion',
                            'eks:UpdateClusterConfig',
                            'eks:CreateFargateProfile',
                            'eks:TagResource',
                            'eks:UntagResource',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            '*',
                        ],
                    },
                    {
                        Action: [
                            'eks:DescribeFargateProfile',
                            'eks:DeleteFargateProfile',
                        ],
                        Effect: 'Allow',
                        Resource: '*',
                    },
                    {
                        Action: ['iam:GetRole', 'iam:listAttachedRolePolicies'],
                        Effect: 'Allow',
                        Resource: '*',
                    },
                    {
                        Action: 'iam:CreateServiceLinkedRole',
                        Effect: 'Allow',
                        Resource: '*',
                    },
                    {
                        Action: 'ec2:DescribeVpcs',
                        Effect: 'Allow',
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':ec2:',
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    ':',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    ':vpc/',
                                    {
                                        Ref: 'FargateClusterDefaultVpcE69D3A13',
                                    },
                                ],
                            ],
                        },
                    },
                    {
                        Action: 'iam:PassRole',
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'FargateClusterfargateprofiledefaultPodExecutionRole66F2610E',
                                'Arn',
                            ],
                        },
                    },
                ],
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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