"use strict";
const cluster_1 = require("../lib/cluster-resource-handler/cluster");
const mocks = require("./cluster-resource-handler-mocks");
module.exports = {
    setUp(callback) {
        mocks.reset();
        callback();
    },
    create: {
        async 'onCreate: minimal defaults (vpc + role)'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create', mocks.MOCK_PROPS));
            await handler.onEvent();
            test.deepEqual(mocks.actualRequest.configureAssumeRoleRequest, {
                RoleArn: mocks.MOCK_ASSUME_ROLE_ARN,
                RoleSessionName: 'AWSCDK.EKSCluster.Create.fake-request-id',
            });
            test.deepEqual(mocks.actualRequest.createClusterRequest, {
                roleArn: 'arn:of:role',
                resourcesVpcConfig: {
                    subnetIds: ['subnet1', 'subnet2'],
                    securityGroupIds: ['sg1', 'sg2', 'sg3'],
                },
                name: 'MyResourceId-fakerequestid',
            });
            test.done();
        },
        async 'generated cluster name does not exceed 100 characters'(test) {
            var _a, _b;
            // GIVEN
            const req = {
                StackId: 'fake-stack-id',
                RequestId: '602c078a-6181-4352-9676-4f00352445aa',
                ResourceType: 'Custom::EKSCluster',
                ServiceToken: 'boom',
                LogicalResourceId: 'hello'.repeat(30),
                ResponseURL: 'http://response-url',
                RequestType: 'Create',
                ResourceProperties: {
                    ServiceToken: 'boom',
                    Config: mocks.MOCK_PROPS,
                    AssumeRoleArn: mocks.MOCK_ASSUME_ROLE_ARN,
                },
            };
            // WHEN
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, req);
            await handler.onEvent();
            // THEN
            test.equal((_a = mocks.actualRequest.createClusterRequest) === null || _a === void 0 ? void 0 : _a.name.length, 100);
            test.deepEqual((_b = mocks.actualRequest.createClusterRequest) === null || _b === void 0 ? void 0 : _b.name, 'hellohellohellohellohellohellohellohellohellohellohellohellohellohe-602c078a6181435296764f00352445aa');
            test.done();
        },
        async 'onCreate: explicit cluster name'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create', {
                ...mocks.MOCK_PROPS,
                name: 'ExplicitCustomName',
            }));
            await handler.onEvent();
            test.deepEqual(mocks.actualRequest.createClusterRequest.name, 'ExplicitCustomName');
            test.done();
        },
        async 'with no specific version'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create', {
                ...mocks.MOCK_PROPS,
                version: '12.34.56',
            }));
            await handler.onEvent();
            test.deepEqual(mocks.actualRequest.createClusterRequest.version, '12.34.56');
            test.done();
        },
        async 'isCreateComplete still not complete if cluster is not ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'CREATING';
            const resp = await handler.isComplete();
            test.deepEqual(mocks.actualRequest.describeClusterRequest.name, 'physical-resource-id');
            test.deepEqual(resp, { IsComplete: false });
            test.done();
        },
        async 'isCreateComplete is complete when cluster is ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Create'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'ACTIVE';
            const resp = await handler.isComplete();
            test.deepEqual(resp, {
                IsComplete: true,
                Data: {
                    Name: 'physical-resource-id',
                    Endpoint: 'http://endpoint',
                    Arn: 'arn:cluster-arn',
                    CertificateAuthorityData: 'certificateAuthority-data',
                    ClusterSecurityGroupId: '',
                    EncryptionConfigKeyArn: '',
                    OpenIdConnectIssuerUrl: '',
                    OpenIdConnectIssuer: '',
                },
            });
            test.done();
        },
    },
    delete: {
        async 'returns correct physical name'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            const resp = await handler.onEvent();
            test.deepEqual(mocks.actualRequest.deleteClusterRequest.name, 'physical-resource-id');
            test.deepEqual(resp, { PhysicalResourceId: 'physical-resource-id' });
            test.done();
        },
        async 'onDelete ignores ResourceNotFoundException'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            mocks.simulateResponse.deleteClusterErrorCode = 'ResourceNotFoundException';
            await handler.onEvent();
            test.done();
        },
        async 'isDeleteComplete returns false as long as describeCluster succeeds'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            const resp = await handler.isComplete();
            test.deepEqual(mocks.actualRequest.describeClusterRequest.name, 'physical-resource-id');
            test.ok(!resp.IsComplete);
            test.done();
        },
        async 'isDeleteComplete returns true when describeCluster throws a ResourceNotFound exception'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            mocks.simulateResponse.describeClusterExceptionCode = 'ResourceNotFoundException';
            const resp = await handler.isComplete();
            test.ok(resp.IsComplete);
            test.done();
        },
        async 'isDeleteComplete propagates other errors'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Delete'));
            mocks.simulateResponse.describeClusterExceptionCode = 'OtherException';
            let error;
            try {
                await handler.isComplete();
            }
            catch (e) {
                error = e;
            }
            test.equal(error.code, 'OtherException');
            test.done();
        },
    },
    update: {
        async 'no change'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', mocks.MOCK_PROPS, mocks.MOCK_PROPS));
            const resp = await handler.onEvent();
            test.equal(resp, undefined);
            test.equal(mocks.actualRequest.createClusterRequest, undefined);
            test.equal(mocks.actualRequest.updateClusterConfigRequest, undefined);
            test.equal(mocks.actualRequest.updateClusterVersionRequest, undefined);
            test.done();
        },
        async 'isUpdateComplete is not complete when status is not ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'UPDATING';
            const resp = await handler.isComplete();
            test.deepEqual(resp.IsComplete, false);
            test.done();
        },
        async 'isUpdateComplete waits for ACTIVE'(test) {
            const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update'));
            mocks.simulateResponse.describeClusterResponseMockStatus = 'ACTIVE';
            const resp = await handler.isComplete();
            test.deepEqual(resp.IsComplete, true);
            test.done();
        },
        'requires replacement': {
            'name change': {
                async 'explicit name change'(test) {
                    // GIVEN
                    const req = mocks.newRequest('Update', {
                        ...mocks.MOCK_PROPS,
                        name: 'new-cluster-name-1234',
                    }, {
                        name: 'old-cluster-name',
                    });
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, req);
                    // WHEN
                    const resp = await handler.onEvent();
                    // THEN
                    test.deepEqual(mocks.actualRequest.createClusterRequest, {
                        name: 'new-cluster-name-1234',
                        roleArn: 'arn:of:role',
                        resourcesVpcConfig: {
                            subnetIds: ['subnet1', 'subnet2'],
                            securityGroupIds: ['sg1', 'sg2', 'sg3'],
                        },
                    });
                    test.deepEqual(resp, { PhysicalResourceId: 'new-cluster-name-1234' });
                    test.done();
                },
                async 'from auto-gen name to explicit name'(test) {
                    // GIVEN
                    const req = mocks.newRequest('Update', {
                        ...mocks.MOCK_PROPS,
                        name: undefined,
                    }, {
                        name: 'explicit',
                    });
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, req);
                    // WHEN
                    const resp = await handler.onEvent();
                    // THEN
                    test.deepEqual(mocks.actualRequest.createClusterRequest, {
                        name: 'MyResourceId-fakerequestid',
                        roleArn: 'arn:of:role',
                        resourcesVpcConfig: {
                            subnetIds: ['subnet1', 'subnet2'],
                            securityGroupIds: ['sg1', 'sg2', 'sg3'],
                        },
                    });
                    test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                    test.done();
                },
            },
            async 'subnets or security groups requires a replacement'(test) {
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    ...mocks.MOCK_PROPS,
                    resourcesVpcConfig: {
                        subnetIds: ['subnet1', 'subnet2'],
                        securityGroupIds: ['sg1'],
                    },
                }, {
                    ...mocks.MOCK_PROPS,
                    resourcesVpcConfig: {
                        subnetIds: ['subnet1'],
                        securityGroupIds: ['sg2'],
                    },
                }));
                const resp = await handler.onEvent();
                test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'MyResourceId-fakerequestid',
                    roleArn: 'arn:of:role',
                    resourcesVpcConfig: {
                        subnetIds: ['subnet1', 'subnet2'],
                        securityGroupIds: ['sg1'],
                    },
                });
                test.done();
            },
            async '"roleArn" requires a replacement'(test) {
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                }, {
                    roleArn: 'old-arn',
                }));
                const resp = await handler.onEvent();
                test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'MyResourceId-fakerequestid',
                    roleArn: 'new-arn',
                });
                test.done();
            },
            async 'fails if cluster has an explicit "name" that is the same as the old "name"'(test) {
                // GIVEN
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                    name: 'explicit-cluster-name',
                }, {
                    roleArn: 'old-arn',
                    name: 'explicit-cluster-name',
                }));
                // THEN
                let err;
                try {
                    await handler.onEvent();
                }
                catch (e) {
                    err = e;
                }
                test.equal(err === null || err === void 0 ? void 0 : err.message, 'Cannot replace cluster "explicit-cluster-name" since it has an explicit physical name. Either rename the cluster or remove the "name" configuration');
                test.done();
            },
            async 'succeeds if cluster had an explicit "name" and now it does not'(test) {
                // GIVEN
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                    name: undefined,
                }, {
                    roleArn: 'old-arn',
                    name: 'explicit-cluster-name',
                }));
                // WHEN
                const resp = await handler.onEvent();
                // THEN
                test.deepEqual(resp, { PhysicalResourceId: 'MyResourceId-fakerequestid' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'MyResourceId-fakerequestid',
                    roleArn: 'new-arn',
                });
                test.done();
            },
            async 'succeeds if cluster had an explicit "name" and now it has a different name'(test) {
                // GIVEN
                const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                    roleArn: 'new-arn',
                    name: 'new-explicit-cluster-name',
                }, {
                    roleArn: 'old-arn',
                    name: 'old-explicit-cluster-name',
                }));
                // WHEN
                const resp = await handler.onEvent();
                // THEN
                test.deepEqual(resp, { PhysicalResourceId: 'new-explicit-cluster-name' });
                test.deepEqual(mocks.actualRequest.createClusterRequest, {
                    name: 'new-explicit-cluster-name',
                    roleArn: 'new-arn',
                });
                test.done();
            },
        },
        'isUpdateComplete with EKS update ID': {
            async 'with "Failed" status'(test) {
                const event = mocks.newRequest('Update');
                const isCompleteHandler = new cluster_1.ClusterResourceHandler(mocks.client, {
                    ...event,
                    EksUpdateId: 'foobar',
                });
                mocks.simulateResponse.describeUpdateResponseMockStatus = 'Failed';
                mocks.simulateResponse.describeUpdateResponseMockErrors = [
                    {
                        errorMessage: 'errorMessageMock',
                        errorCode: 'errorCodeMock',
                        resourceIds: [
                            'foo', 'bar',
                        ],
                    },
                ];
                let error;
                try {
                    await isCompleteHandler.isComplete();
                }
                catch (e) {
                    error = e;
                }
                test.ok(error);
                test.deepEqual(mocks.actualRequest.describeUpdateRequest, { name: 'physical-resource-id', updateId: 'foobar' });
                test.equal(error.message, 'cluster update id "foobar" failed with errors: [{"errorMessage":"errorMessageMock","errorCode":"errorCodeMock","resourceIds":["foo","bar"]}]');
                test.done();
            },
            async 'with "InProgress" status, returns IsComplete=false'(test) {
                const event = mocks.newRequest('Update');
                const isCompleteHandler = new cluster_1.ClusterResourceHandler(mocks.client, {
                    ...event,
                    EksUpdateId: 'foobar',
                });
                mocks.simulateResponse.describeUpdateResponseMockStatus = 'InProgress';
                const response = await isCompleteHandler.isComplete();
                test.deepEqual(mocks.actualRequest.describeUpdateRequest, { name: 'physical-resource-id', updateId: 'foobar' });
                test.equal(response.IsComplete, false);
                test.done();
            },
            async 'with "Successful" status, returns IsComplete=true with "Data"'(test) {
                const event = mocks.newRequest('Update');
                const isCompleteHandler = new cluster_1.ClusterResourceHandler(mocks.client, {
                    ...event,
                    EksUpdateId: 'foobar',
                });
                mocks.simulateResponse.describeUpdateResponseMockStatus = 'Successful';
                const response = await isCompleteHandler.isComplete();
                test.deepEqual(mocks.actualRequest.describeUpdateRequest, { name: 'physical-resource-id', updateId: 'foobar' });
                test.deepEqual(response, {
                    IsComplete: true,
                    Data: {
                        Name: 'physical-resource-id',
                        Endpoint: 'http://endpoint',
                        Arn: 'arn:cluster-arn',
                        CertificateAuthorityData: 'certificateAuthority-data',
                        ClusterSecurityGroupId: '',
                        EncryptionConfigKeyArn: '',
                        OpenIdConnectIssuerUrl: '',
                        OpenIdConnectIssuer: '',
                    },
                });
                test.done();
            },
        },
        'in-place': {
            'version change': {
                async 'from undefined to a specific value'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: '12.34',
                    }, {
                        version: undefined,
                    }));
                    const resp = await handler.onEvent();
                    test.deepEqual(resp, { EksUpdateId: mocks.MOCK_UPDATE_STATUS_ID });
                    test.deepEqual(mocks.actualRequest.updateClusterVersionRequest, {
                        name: 'physical-resource-id',
                        version: '12.34',
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'from a specific value to another value'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: '2.0',
                    }, {
                        version: '1.1',
                    }));
                    const resp = await handler.onEvent();
                    test.deepEqual(resp, { EksUpdateId: mocks.MOCK_UPDATE_STATUS_ID });
                    test.deepEqual(mocks.actualRequest.updateClusterVersionRequest, {
                        name: 'physical-resource-id',
                        version: '2.0',
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'to a new value that is already the current version'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: '1.0',
                    }));
                    const resp = await handler.onEvent();
                    test.equal(resp, undefined);
                    test.deepEqual(mocks.actualRequest.describeClusterRequest, { name: 'physical-resource-id' });
                    test.equal(mocks.actualRequest.updateClusterVersionRequest, undefined);
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'fails from specific value to undefined'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        version: undefined,
                    }, {
                        version: '1.2',
                    }));
                    let error;
                    try {
                        await handler.onEvent();
                    }
                    catch (e) {
                        error = e;
                    }
                    test.equal(error.message, 'Cannot remove cluster version configuration. Current version is 1.2');
                    test.done();
                },
            },
            'logging or access change': {
                async 'from undefined to partial logging enabled'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        logging: {
                            clusterLogging: [
                                {
                                    types: ['api'],
                                    enabled: true,
                                },
                            ],
                        },
                    }, {
                        logging: undefined,
                    }));
                    const resp = await handler.onEvent();
                    test.deepEqual(resp, { EksUpdateId: mocks.MOCK_UPDATE_STATUS_ID });
                    test.deepEqual(mocks.actualRequest.updateClusterConfigRequest, {
                        name: 'physical-resource-id',
                        logging: {
                            clusterLogging: [
                                {
                                    types: ['api'],
                                    enabled: true,
                                },
                            ],
                        },
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'from partial vpc configuration to only private access enabled'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        resourcesVpcConfig: {
                            securityGroupIds: ['sg1', 'sg2', 'sg3'],
                            endpointPrivateAccess: true,
                        },
                    }, {
                        resourcesVpcConfig: {
                            securityGroupIds: ['sg1', 'sg2', 'sg3'],
                        },
                    }));
                    const resp = await handler.onEvent();
                    test.deepEqual(resp, { EksUpdateId: mocks.MOCK_UPDATE_STATUS_ID });
                    test.deepEqual(mocks.actualRequest.updateClusterConfigRequest, {
                        name: 'physical-resource-id',
                        logging: undefined,
                        resourcesVpcConfig: {
                            endpointPrivateAccess: true,
                            endpointPublicAccess: undefined,
                            publicAccessCidrs: undefined,
                        },
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
                async 'from undefined to both logging and access fully enabled'(test) {
                    const handler = new cluster_1.ClusterResourceHandler(mocks.client, mocks.newRequest('Update', {
                        logging: {
                            clusterLogging: [
                                {
                                    types: ['api', 'audit', 'authenticator', 'controllerManager', 'scheduler'],
                                    enabled: true,
                                },
                            ],
                        },
                        resourcesVpcConfig: {
                            endpointPrivateAccess: true,
                            endpointPublicAccess: true,
                            publicAccessCidrs: ['0.0.0.0/0'],
                        },
                    }, {
                        logging: undefined,
                        resourcesVpcConfig: undefined,
                    }));
                    const resp = await handler.onEvent();
                    test.deepEqual(resp, { EksUpdateId: mocks.MOCK_UPDATE_STATUS_ID });
                    test.deepEqual(mocks.actualRequest.updateClusterConfigRequest, {
                        name: 'physical-resource-id',
                        logging: {
                            clusterLogging: [
                                {
                                    types: ['api', 'audit', 'authenticator', 'controllerManager', 'scheduler'],
                                    enabled: true,
                                },
                            ],
                        },
                        resourcesVpcConfig: {
                            endpointPrivateAccess: true,
                            endpointPublicAccess: true,
                            publicAccessCidrs: ['0.0.0.0/0'],
                        },
                    });
                    test.equal(mocks.actualRequest.createClusterRequest, undefined);
                    test.done();
                },
            },
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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