"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Nodegroup = exports.NodegroupAmiType = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const eks_generated_1 = require("./eks.generated");
/**
 * The AMI type for your node group. GPU instance types should use the `AL2_x86_64_GPU` AMI type, which uses the
 * Amazon EKS-optimized Linux AMI with GPU support. Non-GPU instances should use the `AL2_x86_64` AMI type, which
 * uses the Amazon EKS-optimized Linux AMI.
 */
var NodegroupAmiType;
(function (NodegroupAmiType) {
    /**
     * Amazon Linux 2
     */
    NodegroupAmiType["AL2_X86_64"] = "AL2_x86_64";
    /**
     * Amazon Linux 2 with GPU support
     */
    NodegroupAmiType["AL2_X86_64_GPU"] = "AL2_x86_64_GPU";
})(NodegroupAmiType = exports.NodegroupAmiType || (exports.NodegroupAmiType = {}));
/**
 * The Nodegroup resource class
 */
class Nodegroup extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.nodegroupName,
        });
        this.cluster = props.cluster;
        this.desiredSize = (_b = (_a = props.desiredSize) !== null && _a !== void 0 ? _a : props.minSize) !== null && _b !== void 0 ? _b : 2;
        this.maxSize = (_c = props.maxSize) !== null && _c !== void 0 ? _c : this.desiredSize;
        this.minSize = (_d = props.minSize) !== null && _d !== void 0 ? _d : 1;
        if (this.desiredSize > this.maxSize) {
            throw new Error(`Desired capacity ${this.desiredSize} can't be greater than max size ${this.maxSize}`);
        }
        if (this.desiredSize < this.minSize) {
            throw new Error(`Minimum capacity ${this.minSize} can't be greater than desired size ${this.desiredSize}`);
        }
        if (!props.nodeRole) {
            const ngRole = new aws_iam_1.Role(this, 'NodeGroupRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            });
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
            ngRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
            this.role = ngRole;
        }
        else {
            this.role = props.nodeRole;
        }
        const resource = new eks_generated_1.CfnNodegroup(this, 'Resource', {
            clusterName: this.cluster.clusterName,
            nodegroupName: props.nodegroupName,
            nodeRole: this.role.roleArn,
            subnets: this.cluster.vpc.selectSubnets(props.subnets).subnetIds,
            amiType: props.amiType,
            diskSize: props.diskSize,
            forceUpdateEnabled: (_e = props.forceUpdate) !== null && _e !== void 0 ? _e : true,
            instanceTypes: props.instanceType ? [props.instanceType.toString()] : undefined,
            labels: props.labels,
            releaseVersion: props.releaseVersion,
            remoteAccess: props.remoteAccess ? {
                ec2SshKey: props.remoteAccess.sshKeyName,
                sourceSecurityGroups: props.remoteAccess.sourceSecurityGroups ?
                    props.remoteAccess.sourceSecurityGroups.map(m => m.securityGroupId) : undefined,
            } : undefined,
            scalingConfig: {
                desiredSize: this.desiredSize,
                maxSize: this.maxSize,
                minSize: this.minSize,
            },
            tags: props.tags,
        });
        // managed nodegroups update the `aws-auth` on creation, but we still need to track
        // its state for consistency.
        if (this.cluster.kubectlEnabled) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.cluster.awsAuth.addRoleMapping(this.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes',
                ],
            });
        }
        this.nodegroupArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'eks',
            resource: 'nodegroup',
            resourceName: this.physicalName,
        });
        this.nodegroupName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Import the Nodegroup from attributes
     */
    static fromNodegroupName(scope, id, nodegroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.nodegroupName = nodegroupName;
            }
        }
        return new Import(scope, id);
    }
}
exports.Nodegroup = Nodegroup;
//# sourceMappingURL=data:application/json;base64,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