"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FargateProfile = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
/**
 * Fargate profiles allows an administrator to declare which pods run on
 * Fargate. This declaration is done through the profile’s selectors. Each
 * profile can have up to five selectors that contain a namespace and optional
 * labels. You must define a namespace for every selector. The label field
 * consists of multiple optional key-value pairs. Pods that match a selector (by
 * matching a namespace for the selector and all of the labels specified in the
 * selector) are scheduled on Fargate. If a namespace selector is defined
 * without any labels, Amazon EKS will attempt to schedule all pods that run in
 * that namespace onto Fargate using the profile. If a to-be-scheduled pod
 * matches any of the selectors in the Fargate profile, then that pod is
 * scheduled on Fargate.
 *
 * If a pod matches multiple Fargate profiles, Amazon EKS picks one of the
 * matches at random. In this case, you can specify which profile a pod should
 * use by adding the following Kubernetes label to the pod specification:
 * eks.amazonaws.com/fargate-profile: profile_name. However, the pod must still
 * match a selector in that profile in order to be scheduled onto Fargate.
 */
class FargateProfile extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        // currently the custom resource requires a role to assume when interacting with the cluster
        // and we only have this role when kubectl is enabled.
        if (!props.cluster.kubectlEnabled) {
            throw new Error('adding Faregate Profiles to clusters without kubectl enabled is currently unsupported');
        }
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this);
        this.podExecutionRole = (_a = props.podExecutionRole) !== null && _a !== void 0 ? _a : new iam.Role(this, 'PodExecutionRole', {
            assumedBy: new iam.ServicePrincipal('eks-fargate-pods.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSFargatePodExecutionRolePolicy')],
        });
        this.podExecutionRole.grantPassRole(props.cluster._kubectlCreationRole);
        let subnets;
        if (props.vpc) {
            const selection = (_b = props.subnetSelection) !== null && _b !== void 0 ? _b : { subnetType: ec2.SubnetType.PRIVATE };
            subnets = props.vpc.selectSubnets(selection).subnetIds;
        }
        if (props.selectors.length < 1) {
            throw new Error('Fargate profile requires at least one selector');
        }
        if (props.selectors.length > 5) {
            throw new Error('Fargate profile supports up to five selectors');
        }
        this.tags = new core_1.TagManager(core_1.TagType.MAP, 'AWS::EKS::FargateProfile');
        const resource = new core_1.CustomResource(this, 'Resource', {
            serviceToken: provider.serviceToken,
            resourceType: consts_1.FARGATE_PROFILE_RESOURCE_TYPE,
            properties: {
                AssumeRoleArn: props.cluster._kubectlCreationRole.roleArn,
                Config: {
                    clusterName: props.cluster.clusterName,
                    fargateProfileName: props.fargateProfileName,
                    podExecutionRoleArn: this.podExecutionRole.roleArn,
                    selectors: props.selectors,
                    subnets,
                    tags: core_1.Lazy.anyValue({ produce: () => this.tags.renderTags() }),
                },
            },
        });
        this.fargateProfileArn = resource.getAttString('fargateProfileArn');
        this.fargateProfileName = resource.ref;
        // Fargate profiles must be created sequentially. If other profile(s) already
        // exist on the same cluster, create a dependency to force sequential creation.
        const clusterFargateProfiles = props.cluster._attachFargateProfile(this);
        if (clusterFargateProfiles.length > 1) {
            const previousProfile = clusterFargateProfiles[clusterFargateProfiles.length - 2];
            resource.node.addDependency(previousProfile);
        }
        // map the fargate pod execution role to the relevant groups in rbac
        // see https://github.com/aws/aws-cdk/issues/7981
        props.cluster.awsAuth.addRoleMapping(this.podExecutionRole, {
            username: 'system:node:{{SessionName}}',
            groups: [
                'system:bootstrappers',
                'system:nodes',
                'system:node-proxier',
            ],
        });
    }
}
exports.FargateProfile = FargateProfile;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZmFyZ2F0ZS1wcm9maWxlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiZmFyZ2F0ZS1wcm9maWxlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHdDQUF3QztBQUN4Qyx3Q0FBd0M7QUFDeEMsd0NBQWdHO0FBRWhHLDhEQUFrRjtBQUNsRiwyRUFBc0U7QUFzRnRFOzs7Ozs7Ozs7Ozs7Ozs7Ozs7R0FrQkc7QUFDSCxNQUFhLGNBQWUsU0FBUSxnQkFBUztJQTZCM0MsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUEwQjs7UUFDbEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQiw0RkFBNEY7UUFDNUYsc0RBQXNEO1FBQ3RELElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRTtZQUNqQyxNQUFNLElBQUksS0FBSyxDQUFDLHVGQUF1RixDQUFDLENBQUM7U0FDMUc7UUFFRCxNQUFNLFFBQVEsR0FBRyxtREFBdUIsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFM0QsSUFBSSxDQUFDLGdCQUFnQixTQUFHLEtBQUssQ0FBQyxnQkFBZ0IsbUNBQUksSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtZQUN2RixTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsZ0NBQWdDLENBQUM7WUFDckUsZUFBZSxFQUFFLENBQUUsR0FBRyxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFFO1NBQzFHLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBRXhFLElBQUksT0FBNkIsQ0FBQztRQUNsQyxJQUFJLEtBQUssQ0FBQyxHQUFHLEVBQUU7WUFDYixNQUFNLFNBQVMsU0FBd0IsS0FBSyxDQUFDLGVBQWUsbUNBQUksRUFBRSxVQUFVLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsQ0FBQztZQUN2RyxPQUFPLEdBQUcsS0FBSyxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDLENBQUMsU0FBUyxDQUFDO1NBQ3hEO1FBRUQsSUFBSSxLQUFLLENBQUMsU0FBUyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDOUIsTUFBTSxJQUFJLEtBQUssQ0FBQyxnREFBZ0QsQ0FBQyxDQUFDO1NBQ25FO1FBRUQsSUFBSSxLQUFLLENBQUMsU0FBUyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDOUIsTUFBTSxJQUFJLEtBQUssQ0FBQywrQ0FBK0MsQ0FBQyxDQUFDO1NBQ2xFO1FBRUQsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLGlCQUFVLENBQUMsY0FBTyxDQUFDLEdBQUcsRUFBRSwwQkFBMEIsQ0FBQyxDQUFDO1FBRXBFLE1BQU0sUUFBUSxHQUFHLElBQUkscUJBQWMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3BELFlBQVksRUFBRSxRQUFRLENBQUMsWUFBWTtZQUNuQyxZQUFZLEVBQUUsc0NBQTZCO1lBQzNDLFVBQVUsRUFBRTtnQkFDVixhQUFhLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsQ0FBQyxPQUFPO2dCQUN6RCxNQUFNLEVBQUU7b0JBQ04sV0FBVyxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsV0FBVztvQkFDdEMsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQjtvQkFDNUMsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLE9BQU87b0JBQ2xELFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztvQkFDMUIsT0FBTztvQkFDUCxJQUFJLEVBQUUsV0FBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQUM7aUJBQy9EO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsUUFBUSxDQUFDLFlBQVksQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQ3BFLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDO1FBRXZDLDZFQUE2RTtRQUM3RSwrRUFBK0U7UUFDL0UsTUFBTSxzQkFBc0IsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLHFCQUFxQixDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3pFLElBQUksc0JBQXNCLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNyQyxNQUFNLGVBQWUsR0FBRyxzQkFBc0IsQ0FBQyxzQkFBc0IsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUM7WUFDbEYsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsZUFBZSxDQUFDLENBQUM7U0FDOUM7UUFFRCxvRUFBb0U7UUFDcEUsaURBQWlEO1FBQ2pELEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUU7WUFDMUQsUUFBUSxFQUFFLDZCQUE2QjtZQUN2QyxNQUFNLEVBQUU7Z0JBQ04sc0JBQXNCO2dCQUN0QixjQUFjO2dCQUNkLHFCQUFxQjthQUN0QjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUM7Q0FDRjtBQXJHRCx3Q0FxR0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBlYzIgZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QsIEN1c3RvbVJlc291cmNlLCBJVGFnZ2FibGUsIExhenksIFRhZ01hbmFnZXIsIFRhZ1R5cGUgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENsdXN0ZXIgfSBmcm9tICcuL2NsdXN0ZXInO1xuaW1wb3J0IHsgRkFSR0FURV9QUk9GSUxFX1JFU09VUkNFX1RZUEUgfSBmcm9tICcuL2NsdXN0ZXItcmVzb3VyY2UtaGFuZGxlci9jb25zdHMnO1xuaW1wb3J0IHsgQ2x1c3RlclJlc291cmNlUHJvdmlkZXIgfSBmcm9tICcuL2NsdXN0ZXItcmVzb3VyY2UtcHJvdmlkZXInO1xuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGRlZmluaW5nIEVLUyBGYXJnYXRlIFByb2ZpbGVzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEZhcmdhdGVQcm9maWxlT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgRmFyZ2F0ZSBwcm9maWxlLlxuICAgKiBAZGVmYXVsdCAtIGdlbmVyYXRlZFxuICAgKi9cbiAgcmVhZG9ubHkgZmFyZ2F0ZVByb2ZpbGVOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcG9kIGV4ZWN1dGlvbiByb2xlIHRvIHVzZSBmb3IgcG9kcyB0aGF0IG1hdGNoIHRoZSBzZWxlY3RvcnMgaW4gdGhlXG4gICAqIEZhcmdhdGUgcHJvZmlsZS4gVGhlIHBvZCBleGVjdXRpb24gcm9sZSBhbGxvd3MgRmFyZ2F0ZSBpbmZyYXN0cnVjdHVyZSB0b1xuICAgKiByZWdpc3RlciB3aXRoIHlvdXIgY2x1c3RlciBhcyBhIG5vZGUsIGFuZCBpdCBwcm92aWRlcyByZWFkIGFjY2VzcyB0byBBbWF6b25cbiAgICogRUNSIGltYWdlIHJlcG9zaXRvcmllcy5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWtzL2xhdGVzdC91c2VyZ3VpZGUvcG9kLWV4ZWN1dGlvbi1yb2xlLmh0bWxcbiAgICogQGRlZmF1bHQgLSBhIHJvbGUgd2lsbCBiZSBhdXRvbWF0aWNhbGx5IGNyZWF0ZWRcbiAgICovXG4gIHJlYWRvbmx5IHBvZEV4ZWN1dGlvblJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAqIFRoZSBzZWxlY3RvcnMgdG8gbWF0Y2ggZm9yIHBvZHMgdG8gdXNlIHRoaXMgRmFyZ2F0ZSBwcm9maWxlLiBFYWNoIHNlbGVjdG9yXG4gICAqIG11c3QgaGF2ZSBhbiBhc3NvY2lhdGVkIG5hbWVzcGFjZS4gT3B0aW9uYWxseSwgeW91IGNhbiBhbHNvIHNwZWNpZnkgbGFiZWxzXG4gICAqIGZvciBhIG5hbWVzcGFjZS5cbiAgICpcbiAgICogQXQgbGVhc3Qgb25lIHNlbGVjdG9yIGlzIHJlcXVpcmVkIGFuZCB5b3UgbWF5IHNwZWNpZnkgdXAgdG8gZml2ZSBzZWxlY3RvcnMuXG4gICAqL1xuICByZWFkb25seSBzZWxlY3RvcnM6IFNlbGVjdG9yW107XG5cbiAgLyoqXG4gICAqIFRoZSBWUEMgZnJvbSB3aGljaCB0byBzZWxlY3Qgc3VibmV0cyB0byBsYXVuY2ggeW91ciBwb2RzIGludG8uXG4gICAqXG4gICAqIEJ5IGRlZmF1bHQsIGFsbCBwcml2YXRlIHN1Ym5ldHMgYXJlIHNlbGVjdGVkLiBZb3UgY2FuIGN1c3RvbWl6ZSB0aGlzIHVzaW5nXG4gICAqIGBzdWJuZXRTZWxlY3Rpb25gLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGFsbCBwcml2YXRlIHN1Ym5ldHMgdXNlZCBieSB0aGVFS1MgY2x1c3RlclxuICAgKi9cbiAgcmVhZG9ubHkgdnBjPzogZWMyLklWcGM7XG5cbiAgLyoqXG4gICAqIFNlbGVjdCB3aGljaCBzdWJuZXRzIHRvIGxhdW5jaCB5b3VyIHBvZHMgaW50by4gQXQgdGhpcyB0aW1lLCBwb2RzIHJ1bm5pbmdcbiAgICogb24gRmFyZ2F0ZSBhcmUgbm90IGFzc2lnbmVkIHB1YmxpYyBJUCBhZGRyZXNzZXMsIHNvIG9ubHkgcHJpdmF0ZSBzdWJuZXRzXG4gICAqICh3aXRoIG5vIGRpcmVjdCByb3V0ZSB0byBhbiBJbnRlcm5ldCBHYXRld2F5KSBhcmUgYWxsb3dlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBhbGwgcHJpdmF0ZSBzdWJuZXRzIG9mIHRoZSBWUEMgYXJlIHNlbGVjdGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgc3VibmV0U2VsZWN0aW9uPzogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcbn1cblxuLyoqXG4gKiBDb25maWd1cmF0aW9uIHByb3BzIGZvciBFS1MgRmFyZ2F0ZSBQcm9maWxlcy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBGYXJnYXRlUHJvZmlsZVByb3BzIGV4dGVuZHMgRmFyZ2F0ZVByb2ZpbGVPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBFS1MgY2x1c3RlciB0byBhcHBseSB0aGUgRmFyZ2F0ZSBwcm9maWxlIHRvLlxuICAgKiBbZGlzYWJsZS1hd3NsaW50OnJlZi12aWEtaW50ZXJmYWNlXVxuICAgKi9cbiAgcmVhZG9ubHkgY2x1c3RlcjogQ2x1c3Rlcjtcbn1cblxuLyoqXG4gKiBGYXJnYXRlIHByb2ZpbGUgc2VsZWN0b3IuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU2VsZWN0b3Ige1xuICAvKipcbiAgICogVGhlIEt1YmVybmV0ZXMgbmFtZXNwYWNlIHRoYXQgdGhlIHNlbGVjdG9yIHNob3VsZCBtYXRjaC5cbiAgICpcbiAgICogWW91IG11c3Qgc3BlY2lmeSBhIG5hbWVzcGFjZSBmb3IgYSBzZWxlY3Rvci4gVGhlIHNlbGVjdG9yIG9ubHkgbWF0Y2hlcyBwb2RzXG4gICAqIHRoYXQgYXJlIGNyZWF0ZWQgaW4gdGhpcyBuYW1lc3BhY2UsIGJ1dCB5b3UgY2FuIGNyZWF0ZSBtdWx0aXBsZSBzZWxlY3RvcnNcbiAgICogdG8gdGFyZ2V0IG11bHRpcGxlIG5hbWVzcGFjZXMuXG4gICAqL1xuICByZWFkb25seSBuYW1lc3BhY2U6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEt1YmVybmV0ZXMgbGFiZWxzIHRoYXQgdGhlIHNlbGVjdG9yIHNob3VsZCBtYXRjaC4gQSBwb2QgbXVzdCBjb250YWluXG4gICAqIGFsbCBvZiB0aGUgbGFiZWxzIHRoYXQgYXJlIHNwZWNpZmllZCBpbiB0aGUgc2VsZWN0b3IgZm9yIGl0IHRvIGJlXG4gICAqIGNvbnNpZGVyZWQgYSBtYXRjaC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBhbGwgcG9kcyB3aXRoaW4gdGhlIG5hbWVzcGFjZSB3aWxsIGJlIHNlbGVjdGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgbGFiZWxzPzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcbn1cblxuLyoqXG4gKiBGYXJnYXRlIHByb2ZpbGVzIGFsbG93cyBhbiBhZG1pbmlzdHJhdG9yIHRvIGRlY2xhcmUgd2hpY2ggcG9kcyBydW4gb25cbiAqIEZhcmdhdGUuIFRoaXMgZGVjbGFyYXRpb24gaXMgZG9uZSB0aHJvdWdoIHRoZSBwcm9maWxl4oCZcyBzZWxlY3RvcnMuIEVhY2hcbiAqIHByb2ZpbGUgY2FuIGhhdmUgdXAgdG8gZml2ZSBzZWxlY3RvcnMgdGhhdCBjb250YWluIGEgbmFtZXNwYWNlIGFuZCBvcHRpb25hbFxuICogbGFiZWxzLiBZb3UgbXVzdCBkZWZpbmUgYSBuYW1lc3BhY2UgZm9yIGV2ZXJ5IHNlbGVjdG9yLiBUaGUgbGFiZWwgZmllbGRcbiAqIGNvbnNpc3RzIG9mIG11bHRpcGxlIG9wdGlvbmFsIGtleS12YWx1ZSBwYWlycy4gUG9kcyB0aGF0IG1hdGNoIGEgc2VsZWN0b3IgKGJ5XG4gKiBtYXRjaGluZyBhIG5hbWVzcGFjZSBmb3IgdGhlIHNlbGVjdG9yIGFuZCBhbGwgb2YgdGhlIGxhYmVscyBzcGVjaWZpZWQgaW4gdGhlXG4gKiBzZWxlY3RvcikgYXJlIHNjaGVkdWxlZCBvbiBGYXJnYXRlLiBJZiBhIG5hbWVzcGFjZSBzZWxlY3RvciBpcyBkZWZpbmVkXG4gKiB3aXRob3V0IGFueSBsYWJlbHMsIEFtYXpvbiBFS1Mgd2lsbCBhdHRlbXB0IHRvIHNjaGVkdWxlIGFsbCBwb2RzIHRoYXQgcnVuIGluXG4gKiB0aGF0IG5hbWVzcGFjZSBvbnRvIEZhcmdhdGUgdXNpbmcgdGhlIHByb2ZpbGUuIElmIGEgdG8tYmUtc2NoZWR1bGVkIHBvZFxuICogbWF0Y2hlcyBhbnkgb2YgdGhlIHNlbGVjdG9ycyBpbiB0aGUgRmFyZ2F0ZSBwcm9maWxlLCB0aGVuIHRoYXQgcG9kIGlzXG4gKiBzY2hlZHVsZWQgb24gRmFyZ2F0ZS5cbiAqXG4gKiBJZiBhIHBvZCBtYXRjaGVzIG11bHRpcGxlIEZhcmdhdGUgcHJvZmlsZXMsIEFtYXpvbiBFS1MgcGlja3Mgb25lIG9mIHRoZVxuICogbWF0Y2hlcyBhdCByYW5kb20uIEluIHRoaXMgY2FzZSwgeW91IGNhbiBzcGVjaWZ5IHdoaWNoIHByb2ZpbGUgYSBwb2Qgc2hvdWxkXG4gKiB1c2UgYnkgYWRkaW5nIHRoZSBmb2xsb3dpbmcgS3ViZXJuZXRlcyBsYWJlbCB0byB0aGUgcG9kIHNwZWNpZmljYXRpb246XG4gKiBla3MuYW1hem9uYXdzLmNvbS9mYXJnYXRlLXByb2ZpbGU6IHByb2ZpbGVfbmFtZS4gSG93ZXZlciwgdGhlIHBvZCBtdXN0IHN0aWxsXG4gKiBtYXRjaCBhIHNlbGVjdG9yIGluIHRoYXQgcHJvZmlsZSBpbiBvcmRlciB0byBiZSBzY2hlZHVsZWQgb250byBGYXJnYXRlLlxuICovXG5leHBvcnQgY2xhc3MgRmFyZ2F0ZVByb2ZpbGUgZXh0ZW5kcyBDb25zdHJ1Y3QgaW1wbGVtZW50cyBJVGFnZ2FibGUge1xuXG4gIC8qKlxuICAgKiBUaGUgZnVsbCBBbWF6b24gUmVzb3VyY2UgTmFtZSAoQVJOKSBvZiB0aGUgRmFyZ2F0ZSBwcm9maWxlLlxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZmFyZ2F0ZVByb2ZpbGVBcm46IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIEZhcmdhdGUgcHJvZmlsZS5cbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGZhcmdhdGVQcm9maWxlTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBSZXNvdXJjZSB0YWdzLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHRhZ3M6IFRhZ01hbmFnZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBwb2QgZXhlY3V0aW9uIHJvbGUgdG8gdXNlIGZvciBwb2RzIHRoYXQgbWF0Y2ggdGhlIHNlbGVjdG9ycyBpbiB0aGVcbiAgICogRmFyZ2F0ZSBwcm9maWxlLiBUaGUgcG9kIGV4ZWN1dGlvbiByb2xlIGFsbG93cyBGYXJnYXRlIGluZnJhc3RydWN0dXJlIHRvXG4gICAqIHJlZ2lzdGVyIHdpdGggeW91ciBjbHVzdGVyIGFzIGEgbm9kZSwgYW5kIGl0IHByb3ZpZGVzIHJlYWQgYWNjZXNzIHRvIEFtYXpvblxuICAgKiBFQ1IgaW1hZ2UgcmVwb3NpdG9yaWVzLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHBvZEV4ZWN1dGlvblJvbGU6IGlhbS5JUm9sZTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRmFyZ2F0ZVByb2ZpbGVQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICAvLyBjdXJyZW50bHkgdGhlIGN1c3RvbSByZXNvdXJjZSByZXF1aXJlcyBhIHJvbGUgdG8gYXNzdW1lIHdoZW4gaW50ZXJhY3Rpbmcgd2l0aCB0aGUgY2x1c3RlclxuICAgIC8vIGFuZCB3ZSBvbmx5IGhhdmUgdGhpcyByb2xlIHdoZW4ga3ViZWN0bCBpcyBlbmFibGVkLlxuICAgIGlmICghcHJvcHMuY2x1c3Rlci5rdWJlY3RsRW5hYmxlZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdhZGRpbmcgRmFyZWdhdGUgUHJvZmlsZXMgdG8gY2x1c3RlcnMgd2l0aG91dCBrdWJlY3RsIGVuYWJsZWQgaXMgY3VycmVudGx5IHVuc3VwcG9ydGVkJyk7XG4gICAgfVxuXG4gICAgY29uc3QgcHJvdmlkZXIgPSBDbHVzdGVyUmVzb3VyY2VQcm92aWRlci5nZXRPckNyZWF0ZSh0aGlzKTtcblxuICAgIHRoaXMucG9kRXhlY3V0aW9uUm9sZSA9IHByb3BzLnBvZEV4ZWN1dGlvblJvbGUgPz8gbmV3IGlhbS5Sb2xlKHRoaXMsICdQb2RFeGVjdXRpb25Sb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2Vrcy1mYXJnYXRlLXBvZHMuYW1hem9uYXdzLmNvbScpLFxuICAgICAgbWFuYWdlZFBvbGljaWVzOiBbIGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnQW1hem9uRUtTRmFyZ2F0ZVBvZEV4ZWN1dGlvblJvbGVQb2xpY3knKSBdLFxuICAgIH0pO1xuXG4gICAgdGhpcy5wb2RFeGVjdXRpb25Sb2xlLmdyYW50UGFzc1JvbGUocHJvcHMuY2x1c3Rlci5fa3ViZWN0bENyZWF0aW9uUm9sZSk7XG5cbiAgICBsZXQgc3VibmV0czogc3RyaW5nW10gfCB1bmRlZmluZWQ7XG4gICAgaWYgKHByb3BzLnZwYykge1xuICAgICAgY29uc3Qgc2VsZWN0aW9uOiBlYzIuU3VibmV0U2VsZWN0aW9uID0gcHJvcHMuc3VibmV0U2VsZWN0aW9uID8/IHsgc3VibmV0VHlwZTogZWMyLlN1Ym5ldFR5cGUuUFJJVkFURSB9O1xuICAgICAgc3VibmV0cyA9IHByb3BzLnZwYy5zZWxlY3RTdWJuZXRzKHNlbGVjdGlvbikuc3VibmV0SWRzO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5zZWxlY3RvcnMubGVuZ3RoIDwgMSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdGYXJnYXRlIHByb2ZpbGUgcmVxdWlyZXMgYXQgbGVhc3Qgb25lIHNlbGVjdG9yJyk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLnNlbGVjdG9ycy5sZW5ndGggPiA1KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0ZhcmdhdGUgcHJvZmlsZSBzdXBwb3J0cyB1cCB0byBmaXZlIHNlbGVjdG9ycycpO1xuICAgIH1cblxuICAgIHRoaXMudGFncyA9IG5ldyBUYWdNYW5hZ2VyKFRhZ1R5cGUuTUFQLCAnQVdTOjpFS1M6OkZhcmdhdGVQcm9maWxlJyk7XG5cbiAgICBjb25zdCByZXNvdXJjZSA9IG5ldyBDdXN0b21SZXNvdXJjZSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBzZXJ2aWNlVG9rZW46IHByb3ZpZGVyLnNlcnZpY2VUb2tlbixcbiAgICAgIHJlc291cmNlVHlwZTogRkFSR0FURV9QUk9GSUxFX1JFU09VUkNFX1RZUEUsXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIEFzc3VtZVJvbGVBcm46IHByb3BzLmNsdXN0ZXIuX2t1YmVjdGxDcmVhdGlvblJvbGUucm9sZUFybixcbiAgICAgICAgQ29uZmlnOiB7XG4gICAgICAgICAgY2x1c3Rlck5hbWU6IHByb3BzLmNsdXN0ZXIuY2x1c3Rlck5hbWUsXG4gICAgICAgICAgZmFyZ2F0ZVByb2ZpbGVOYW1lOiBwcm9wcy5mYXJnYXRlUHJvZmlsZU5hbWUsXG4gICAgICAgICAgcG9kRXhlY3V0aW9uUm9sZUFybjogdGhpcy5wb2RFeGVjdXRpb25Sb2xlLnJvbGVBcm4sXG4gICAgICAgICAgc2VsZWN0b3JzOiBwcm9wcy5zZWxlY3RvcnMsXG4gICAgICAgICAgc3VibmV0cyxcbiAgICAgICAgICB0YWdzOiBMYXp5LmFueVZhbHVlKHsgcHJvZHVjZTogKCkgPT4gdGhpcy50YWdzLnJlbmRlclRhZ3MoKSB9KSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICB0aGlzLmZhcmdhdGVQcm9maWxlQXJuID0gcmVzb3VyY2UuZ2V0QXR0U3RyaW5nKCdmYXJnYXRlUHJvZmlsZUFybicpO1xuICAgIHRoaXMuZmFyZ2F0ZVByb2ZpbGVOYW1lID0gcmVzb3VyY2UucmVmO1xuXG4gICAgLy8gRmFyZ2F0ZSBwcm9maWxlcyBtdXN0IGJlIGNyZWF0ZWQgc2VxdWVudGlhbGx5LiBJZiBvdGhlciBwcm9maWxlKHMpIGFscmVhZHlcbiAgICAvLyBleGlzdCBvbiB0aGUgc2FtZSBjbHVzdGVyLCBjcmVhdGUgYSBkZXBlbmRlbmN5IHRvIGZvcmNlIHNlcXVlbnRpYWwgY3JlYXRpb24uXG4gICAgY29uc3QgY2x1c3RlckZhcmdhdGVQcm9maWxlcyA9IHByb3BzLmNsdXN0ZXIuX2F0dGFjaEZhcmdhdGVQcm9maWxlKHRoaXMpO1xuICAgIGlmIChjbHVzdGVyRmFyZ2F0ZVByb2ZpbGVzLmxlbmd0aCA+IDEpIHtcbiAgICAgIGNvbnN0IHByZXZpb3VzUHJvZmlsZSA9IGNsdXN0ZXJGYXJnYXRlUHJvZmlsZXNbY2x1c3RlckZhcmdhdGVQcm9maWxlcy5sZW5ndGggLSAyXTtcbiAgICAgIHJlc291cmNlLm5vZGUuYWRkRGVwZW5kZW5jeShwcmV2aW91c1Byb2ZpbGUpO1xuICAgIH1cblxuICAgIC8vIG1hcCB0aGUgZmFyZ2F0ZSBwb2QgZXhlY3V0aW9uIHJvbGUgdG8gdGhlIHJlbGV2YW50IGdyb3VwcyBpbiByYmFjXG4gICAgLy8gc2VlIGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvYXdzLWNkay9pc3N1ZXMvNzk4MVxuICAgIHByb3BzLmNsdXN0ZXIuYXdzQXV0aC5hZGRSb2xlTWFwcGluZyh0aGlzLnBvZEV4ZWN1dGlvblJvbGUsIHtcbiAgICAgIHVzZXJuYW1lOiAnc3lzdGVtOm5vZGU6e3tTZXNzaW9uTmFtZX19JyxcbiAgICAgIGdyb3VwczogW1xuICAgICAgICAnc3lzdGVtOmJvb3RzdHJhcHBlcnMnLFxuICAgICAgICAnc3lzdGVtOm5vZGVzJyxcbiAgICAgICAgJ3N5c3RlbTpub2RlLXByb3hpZXInLFxuICAgICAgXSxcbiAgICB9KTtcbiAgfVxufVxuIl19