"use strict";
const assert_1 = require("@aws-cdk/assert");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const nodeunit_1 = require("nodeunit");
const path = require("path");
const lambda = require("../lib");
module.exports = nodeunit_1.testCase({
    'creating a layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        // WHEN
        new lambda.LayerVersion(stack, 'LayerVersion', {
            code,
            compatibleRuntimes: [lambda.Runtime.NODEJS_8_10]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::LayerVersion', {
            Content: {
                S3Bucket: stack.resolve(bucket.bucketName),
                S3Key: 'ObjectKey',
            },
            CompatibleRuntimes: ['nodejs8.10']
        }));
        test.done();
    },
    'granting access to a layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        const layer = new lambda.LayerVersion(stack, 'LayerVersion', {
            code,
            compatibleRuntimes: [lambda.Runtime.NODEJS_10_X]
        });
        // WHEN
        layer.addPermission('GrantUsage-123456789012', { accountId: '123456789012' });
        layer.addPermission('GrantUsage-o-123456', { accountId: '*', organizationId: 'o-123456' });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::LayerVersionPermission', {
            Action: 'lambda:GetLayerVersion',
            LayerVersionArn: stack.resolve(layer.layerVersionArn),
            Principal: '123456789012',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::LayerVersionPermission', {
            Action: 'lambda:GetLayerVersion',
            LayerVersionArn: stack.resolve(layer.layerVersionArn),
            Principal: '*',
            OrganizationId: 'o-123456'
        }));
        test.done();
    },
    'creating a layer with no runtimes compatible'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        // THEN
        test.throws(() => new lambda.LayerVersion(stack, 'LayerVersion', { code, compatibleRuntimes: [] }), /supports no runtime/);
        test.done();
    },
    'asset metadata is added to the cloudformation resource'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        stack.node.setContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT, true);
        // WHEN
        new lambda.LayerVersion(stack, 'layer', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'layer-code'))
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::LayerVersion', {
            Metadata: {
                'aws:asset:path': 'asset.45f085ecc03a1a22cf003fba3fab28e660c92bcfcd4d0c01b62c7cd191070a2d',
                'aws:asset:property': 'Content'
            }
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    }
});
//# sourceMappingURL=data:application/json;base64,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