from Bio.Align import PairwiseAligner
import itertools
from typing import List



alignment_refs: dict[str, str] = {}
# >A/chicken/Italy/21VIR1151-2/2021_seg_ver=20
alignment_refs['NA'] = 'ATGAATCCAAATCAGAAAATAGCGACCATTGGCTCCATCTCATTGGGACTAGTTGTATTCAATGTTCTACTGCATGCCTTGAGCATCATATTAATGGTGTTAGCCCTGGGGAAAAGTGAAAACAATGGAATCTGCAAGGGAACTATAGTAAGGGAATATAATGAAACAGTTAGGATAGAGAAAGTGACCCAGTGGTACAACACTAGTGTAGTCGAATATGTACCGCATTGGAACGAGGGCGCTTATATAAACAACACCGAACCAATATGTGATGTCAAGGGCTTTGCACCTTTTTCCAAGGACAACGGAATAAGAATTGGCTCCAGAGGACATATTTTTGTCATAAGGGAGCCTTTCGTCTCTTGTTCACCTGTAGAGTGCAGAACTTTCTTCCTCACTCAGGGAGCTCTACTCAATGACAAACACTCAAATGGAACAGTGAAGGATAGGAGCCCATTCAGAACTCTCATGAGTGTCGAAGTGGGTCAATCACCCAATGTGTATCAAGCAAGGTTTGAAGCTGTAGCATGGTCAGCAACAGCCTGTCATGATGGTAAGAAATGGATGACGATTGGTGTGACAGGGCCAGATTCGAAAGCAATAGCAGTAGTCCATTACGGAGGAGTGCCTACTGATATTGTTAACTCCTGGGCAGGAGACATATTACGGACTCAGGAGTCATCTTGTACTTGCATTCAAGGTAATTGTTATTGGGTAATGACTGACGGTCCATCCAATAGACAGGCGCAGTATAGAATATACAAAGCAAATCAAGGCAAAATAATTGACCAAGCAGATGTCAGCTTTAGTGGAGGGCATATTGAGGAATGCTCTTGTTATCCAAATGATGGTAAAGTGGAATGCGTGTGTAGAGACAACTGGATGGGAACTAACAGGCCTGTGCTAATTATCTCGCCTGACCTCTCTTACAGGGTTGGGTATTTATGTGCGGGATTGCCCAGTGACACTCCAAGAGGGGAAGATGCCCAATTTGTCGGTTCGTGCACTAGTCCCATGGGAAATCAGGGGTATGGCGTAAAAGGTTTCGGGTTCCGACAGGGATCTGATGTGTGGATGGGGCGGACAATTAGTCGAACCTCCAGGTCAGGGTTTGAAATAATAAGGATAAAGAATGGTTGGACGCAGACAAGCAAAGAACGGATTAGAAGGCAGGTGGTTGTTGATAATTTGAATTGGTCGGGATACAGTGGGTCTTTCACTTTACCAGTAGAATTGTCTGGGAGGGAATGTTTAGTCCCCTGTTTTTGGGTCGAAATGATCAGAGGCAGGCCAGAAGAAAGAACAATCTGGACCTCTAGTAGCTCCATTGTAATGTGTGGAGTTGATCATGAAATTGCCGATTGGTCATGGCACGATGGAGCTATTCTTCCCTTTGACATCGATGGGATGTAA'
# >A/avian/Italy/21VIR10913-3/2021_seg_ver=1
alignment_refs['NP'] = 'ATGGCGTCTCAAGGCACCAAACGATCCTATGAACAGATGGAGACTGGTGGAGAGCGCCAGAATGCCACTGAGATCAGAGCATCTGTTGGACGAATGGTTGGTGGAATTGGGAGGTTCTACATACAGATGTGCACTGAGCTCAAACTCAGCGACCATGAAGGAAGGCTGATCCAGAACAGCATAACAATAGAGAGAATGGTTCTCTCTGCATTTGATGAAAGGAGGAACAAATATCTGGAAGAACACCCCAGTGCGGGGAAGGACCCGAAGAAAACTGGAGGTCCAATTTATCGAAGGAGAGATGGGAAATGGGTGAGAGAACTGATCCTGTATGACAAAGAGGAGATCAGGAGAATCTGGCGTCAAGCGAACAATGGAGAAGACGCAACTGCTGGTCTCACTCACCTGATGATCTGGCATTCTAATTTAAATGATGCCACATACCAGAGGACGAGAGCTCTCGTGCGTACTGGGATGGACCCCAGGATGTGCTCTCTTATGCAAGGATCAACTCTCCCAAGGAGGTCTGGAGCTGCTGGTGCAGCAGTAAAGGGAGTCGGGACGATGGTGATGGAACTAATTCGGATGATAAAGCGAGGAATTAATGATCGGAACTTCTGGAGAGGCGAGAACGGACGAAGGACAAGGATTGCATATGAGAGAATGTGCAACATCCTCAAAGGGAAATTCCAAACAGCAGCGCAAAGAGCAATGATGGACCAGGTGCGTGAAAGCAGGAATCCTGGCAATGCTGAAATTGAAGATCTCATCTTTCTGGCACGGTCTGCACTCATCCTGAGAGGATCAGTGGCTCATAAGTCCTGCTTGCCTGCTTGTGTGTACGGACTCGCTGTAGCCAGTGGATACGACTTTGAGAGAGAAGGGTACTCTCTAGTTGGGATAGATCCTTTCCGTCTGCTTCAAAACAGTCAGGTCTTCAGCCTCATTAGACCAAATGAGAATCCAGCACACAAGAGTCAATTGGTATGGATGGCATGTCATTCCGCAGCATTCGAGGATCTGAGAGTCTCAAGTTTCATCAGAGGAACAAGAGTAGTTCCAAGAGGACAACTATCCACAAGAGGGGTTCAAATTGCTTCAAATGAGAACATGGAAACAATGGACTCCAGCACTCTTGAACTGAGAAGCAGATATTGGGCTATAAGAACCAGGAGTGGAGGGAACACCAACCAACAGAGAGCATCTGCAGGACAGATCAGTGTACAGCCCACTTTTTCGGTACAGAGAAATCTTCCCTTTGAAAGAGCGACCATTATGGCGGCGTTCACAGGGAATACTGAGGGCAGAACATCCGACATGAGGACTGAAATCATAAGAATGATGGAAAGTGCCAGACCAGAAGATGTGTCTTTCCAGGGGCGGGGAGTCTTCGAGCTTTCGGACGAAAAGGCAACGAACCCGATCGTGCCTTCCTTTGACATGAGTAATGAAGGATCTTATTTCTTCGGAGACAATGCAGAGGAGTATGACAATTAA'
# >A/chicken/Italy/21VIR1151-2_MP
alignment_refs['MP'] = 'ATGAGTCTTCTAACCGAGGTCGAAACGTACGTTCTCTCTATCGTCCCGTCAGGCCCCCTCAAAGCCGAGATCGCGCAGAGACTTGAAGATGTCTTTGCAGGGAAGAACACCGATCTTGAGGCTCTCATGGAATGGCTAAAGACAAGACCAATCCTGTCACCTCTGACTAAGGGGATTTTGGGATTTGTGTTCACGCTCACCGTGCCCAGTGAGCGAGGACTGCAGCGTAGACGCTTTGTCCAAAATGCTCTAAATGGAAATGGAGACCCAAACAACATGGACAGGGCAGTAAAACTGTACAGGAAATTGAAGAGAGAGATAACATTCCATGGGGCTAAAGAAGTTGCACTCAGTTACTCAACCGGTGCACTTGCCAGTTGTATGGGTCTCATATACAACAGAATGGGGACGGTGACCGCAGAAGTGGCATTGGGCCTAGTGTGTGCCACCTGTGAGCAGATTGCTGATTCACAGCATCGGTCTCACAGACAGATAGCAACCACCACCAACCCACTAATCAGACATGAAAACAGAATGGTGTTGGCCAGTACTACAGCTAAGGCTATGGAGCAGATGGCTGGATCGACTGAGCAGGCAGCGGAAGCCATGGAGGTTGCTAGTCAGGCTAGGCAGATGGTGCAGGCAATGAGGACCATTGGAACTCATCCTAGCTCCAGTGCTGGTCTGAGAGATGATCTCCTTGAAAATTTGCAGGCCTACCAAAAACGGATGGGAGTGCAACTGCAGCGATTCAAGTGATCCTCTCGTTATTGCCGCAAGTATCATTGGGATCTTGCACTTGATATTGTGGATTCTTGATCGCCTTTTCTTCAAATGCGTTTATCGTCGCCTTAAATACGGTTTGAAAAGAGGGCCTTCTACGGAAGGGGTACCTGAGTCCATGAGGGAAGAGTACCGGCAGGAACAGCAGAGTGCTGTGGATGTTGACGATGGTCATTTTGTCAACATAGAGCTGGAGTAA'
# >A/avian/Italy/21VIR10913-3/2021_seg_ver=1 
alignment_refs['NS'] = 'ATGGATTCCAACACTGTGTCAAGCTTTCAGGTAGACTGCTTTCTTTGGCATGTCCGCAAACGATTTGCAGACCAAGAACTGGGTGATGCCCCATTCCTTGACCGGCTTCGCCGAGATCAGAAATCCCTGAGAGGAAGAGGCAGCACTCTTGGTCTGGACATCGAAACAGCCACCCGTGTGGGAAAGCAGATAGTGGAGCGGATTCTGGAAGAAGAATCTGATGAGGCACTTAAAATGACTATTGCCCCCGTGCCAGCTTCACGCTACCTAACTGACATGACTCTTGAGGAGATGTCAAGGGACTGGTTCATGCTCATGCCCAAACAGAAAGTGGCAGGTTCCCTTTGCATCAGAATGGACCAGGCAATAATGGATAAAAACATCATATTGAAAGCAAACTTCAGTGTGATTTTTGACCGGCTGAAAACCCTAATACTAATTAGAGCTTTCACAGAAGAAGGAGCAATTGTGGGAGAAATCTCACCATTACCTTCTCTTCCAGGACATACTAATGAGGATGTCAAAAATGCAATTGGGGTCCTCATCGGAGGACTTGAATGGAATGATAACACAGTTCGAGTCTCTGAAACTCTACAGAGATTCGCTTGGAGAAGCAGTAATGAGGATGGGAGACCTCCACTCCCTCCAAAGCAGAAACGGAAAATGGCGAAGACAATTGAGTCAGAAGTTTGAAGAAATAAGATGGCTGATTGAAGAAGTGCGGCACAGATTGAAGATTACAGAGAACAGTTTCGAACAGATAACTTTCATGCAAGCCTTACAACTATTGCTTGAAGTGGAGCAAGAGATAAGAACTTTCTCGTTTCAGCTTATTTAA'
# >A/avian/Italy/21VIR10913-3/2021_seg_ver=1
alignment_refs['PA'] = 'ATGGAAGACTTTGTGCGACAATGCTTCAATCCAATGATTGTCGAGCTTGCGGAAAAAGCAATGAAAGAATATGGGGAAGATCCGAAAATCGAGACAAACAAATTTGCCGCAATATGCACACACTTAGAAGTCTGTTTCATGTATTCGGATTTCCATTTTATTGATGAACGAGGCGAATCAATGATTGTAGAATCTGGCGATCCAAATGCATTATTGAAACACCGATTTGAGATAATCGAAGGGAGAGACCGAGCAATGGCCTGGACAGTGGTGAATAGTATCTGCAACACCACAGGAGTCGAAAAGCCCAAATTCCTCCCTGATTTGTATGACTACAAAGAGAACCGATTCATTGAAATTGGAGTAACGCGAAGGGAAGTTCACATATACTATTTAGAAAAAGCCAACAAGATAAAATCGGAGAAAACACACATTCACATATTCTCATTCACTGGAGAGGAAATGGCCACCAAGGCGGAATACACCCTTGATGAAGAGAGCAGAGCAAGAATAAAAACCAGACTGTTCACTATAAGACAAGAAATGGCCAGTAGAGGTCTATGGGATTCCTTTCGTCAGTCCGAGAGAGGCGAAGAGACAACTGAAGAAAGATTTGAAATCACAGGAACCATGCGCAGGCTTGCCGACCAAAGTCTCCCACCGAACTTCTCCAGCCTTGAAAACTTTAGAGCCTATGTGGATGGATTCGAACCGAACGGCTGCATTGAGGGCAAGCTTTCTCAAATGTCAAAAGAAGTGAACGCCAGAATTGAGCCATTCCTGAAGACAACACCACGCCCTCTCAGATTACCTGATGGGCCTCCTTGTTCTCAGCGGTCGAAGTTCTTGCTGATGGATGCCCTTAAGTTGAGCATCGAAGACCCTAGCCATGAGGGGGAGGGCATACCGCTGTATGATGCAATCAAATGCATGAAGACATTTTTTGGCTGGAAAGAGCCCAACCTCGTAAAGCCGCATGAGAAAGGCATAAACCCTAATTACCTCCTGGCTTGGAAGCAGGTGCTGGCAGAACTTCAAGATATTGAGAATGAGGAAAAATTTCCAAAAACAAAGAACATGAAGAAAACAAGCCAATTGAAGTGGGCACTTGGTGAGAACATGGCACCAGAAAAAGTGGACTTTGAGGACTGCAAAGATGTTAGCGATCTAAGACAGTACGACAGTGACGAACCAGAGCCTAGATCACTAGCAAGCTGGATTCAGAGTGAATTCAACAAGGCATGCGAATTGACAGATTCGAGTTGGATTGAACTTGATGAGATAGGGGAAGACGTTGCTCCAATCGAACACATTGCGAGTATGAGGAGGAACTATTTCACAGCGGAGGTATCCCATTGCAGGGCCACTGAATACATAATGAAGGGAGTATACATAAACACAGCCCTATTGAATGCATCCTGTGCAGCCATGGATGACTTCCAACTGATTCCAATGATAAGCAAGTGCAGAACCAAAGAAGGAAGACGGAGGACAAATCTGTATGGATTCATTATAAAAGGAAGATCCCATTTGAGGAATGACACCGATGTGGTAAACTTTGTGAGCATGGAATTCTCTCTAACTGACCCGAGGCTAGAGCCACACAAATGGGAAAAGTACTGTGTTCTTGAGATAGGAGACATGCTCCTACGGACTGCAATAGGCCAAGTGTCGAGGCCCATGTTCCTGTATGTGAGAACCAATGGGACTTCCAAGATCAAAATGAAATGGGGCATGGAGATGAGACGATGCCTTCTTCAGTCCCTTCAACAAATTGAGAGCATGATTGAGGCCGAATCTTCTGTCAAAGAGAAGGACATGACCAAGGAATTCTTTGAAAACAAATCAGAAACATGGCCAATTGGAGAATCACCCAAAGGGGTTGAGGAAGGCTCTATTGGGAAAGTATGCAGAACATTGCTAGCAAAGTCTGTGTTCAACAGCCTATATGCATCTCCACAACTCGAGGGATTTTCAGCTGAATCAAGAAAATTGCTTCTCATTGTTCAGGCACTTAGGGACAACCTGGAACCTGGAACCTTCGATCTTGGGGGGCTATATGAAGCAATTGAGGAGTGCCTGATTAACGATCCCTGGGTTTTGCTTAATGCGTCTTGGTTCAACTCCTTCCTCACACATGCACTGAAATAG'
# >A/avian/Italy/21VIR10913-3/2021_seg_ver=1
alignment_refs['PB1'] = 'ATGGATGTCAATCCGACTTTACTTTTCTTAAAAGTGCCAGCGCAAGATGCCATAAGTACCACATTCCCTTACACTGGAGATCCTCCATACAGCCATGGAACAGGGACAGGATACACAATGGACACAGTCAACAGAACACATCAATACTCAGAGAAGGGAAAATGGACAACAAACACAGAAACCGGAGCACCTCAACTCAACCCAATTGATGGGCCACTACCTGAGGACAACGAACCGAGCGGATATGCACAAACAGATTGCGTGTTGGAAGCAATGGCTTTCCTTGAAGAGTCCCACCCAGGGATCTTTGAAAACTCTTGTCTTGAAACGATGGAAGTCGTTCAGCAAACAAGAGTGGACAAACTAACTCAAGGTCGCCAGACATATGACTGGACACTGAATAGAAACCAACCAGCTGCAACTGCCCTGGCCAACACTATAGAGGTCTTCAGATCAAACGGTCTAACAGCCAATGAATCGGGGAGACTAATAGATTTCCTCAAGGATGTGATGGACTCAATGGATAAAGAAGAAATGGAAATAACAACACATTTCCAGAGAAAGAGAAGAGTAAGGGACAACATGACCAAGAAAATGGTCACACAAAGAACAATAGGAAAGAAGAAACAAAGGCTAAACAAGAGGAGCTACTTAATAAGAGCACTGACACTGAATACAATGACAAAAGATGCAGAAAGAGGCAAATTGAAGAGACGGGCGATTGCAACACCAGGGATGCAGATTAGAGGATTTGTGTACTTTGTCGAAACACTGGCAAGGAGCATCTGTGAAAAACTTGAGCAATCTGGACTCCCCGTTGGAGGGAATGAGAAGAAGGCTAAATTGGCAAATGTCGTGAGAAAAATGATGACTAACTCACAAGATACAGAGCTCTCCTTCACAATTACTGGAGACAACACCAAATGGAATGAGAATCAAAATCCTCGGATGTTTCTGGCAATGATAACATACATTACAAGAAACCAACCTGAATGGTTTAGAAATGTCTTAAGTATTGCCCCTATAATGTTCTCGAACAAAATGGCGAGATTGGGAAAAGGGTACATGTTTGAAAGTAAGAGCATGAAGTTACGGACACAAATACCTGCAGAATTGCTTGCAAACATTGACTTAAAATACTTCAATGAATCAACAAGAAAGAAAATCGAAAAGATAAGGCCTCTACTAATAGATGGCACTGCCTCATTGAGTCCTGGAATGATGATGGGCATGTTCAATATGCTGAGTACAGTATTAGGAGTTTCAATCCTAAATCTTGGGCAAAAGAATTACACCAAAACCACATACTGGTGGGATGGACTCCAATCCTCTGATGATTTCGCCCTCATAGTAAATGCACCGAATCATGAGGGAATACAAGCAGGAGTGGATAGGTTCTATAGGACCTGCAAACTGGTCGGGATCAATATGAGCAAAAAGAAGTCTTACATAAACCGGACTGGAACATTTGAGTTCACAAGCTTTTTCTATCGCTATGGATTTGTGGCTAACTTCAGTATGGAGCTGCCCAGCTTTGGGGTTTCTGGGATCAATGAATCAGCTGACATGAGCATTGGCGTCACAGTGATAAAGAACAACATGATAAACAATGACCTTGGACCAGCAACAGCTCAAATGGCCCTTCAACTATTCATCAAAGATTACAGGTACACGTACCGATGCCACAGAGGTGACACACAAATTCAAACGAGGAGATCATTCGAGCTGAAGAAGCTGTGGGAACAGACCCGTTCAAAGGCAGGACTGTTGGTGTCAGATGGAGGACCAAATCTATACAACATCCGGAATCTCCATATCCCAGAGGTCTGCCTGAAGTGGGAGCTGATGGACGAAGATTACCAGGGCAGGTTGTGTAATCCTCTGAACCCATTTGTCAGCCATAAAGAAATTGAGTCCGTAAACAATGCTGTGGTGATGCCAGCTCACGGTCCAGCCAAAAGCATGGAATATGATGCCGTTGCGACTACACACTCATGGACTCCTAAAAGGAATCGTTCCATTCTCAATACCAGTCAAAGGGGAATTCTTGAGGATGAACAGATGTACCAGAAATGCTGCAGTCTATTCGAGAAATTCTTCCCCAGTAGTTCATACAGGAGACCAGTTGGAATTTCCAGCATGGTGGAGGCCATGGTGTCTAGGGCCCGAATCGATGCACGCATTGATTTCGAATCTGGAAGGATCAAGAAGGAAGAGTTTGCTGAGATCATGAAGATCTGTTCCACCATTGAAGAGCTCAGACGGCAAAAATAG'
# >A/avian/Italy/21VIR10913-3/2021_seg_ver=1
alignment_refs['PB2'] = 'ATGGAGAGAATAAAAGAGCTAAGAGATTTGATGTCGCAGTCTCGCACTCGCGAGATACTGACAAAAACCACCGTGGACCATATGGCCATAATCAAGAAATATACATCAGGAAGACAGGAGAAGAACCCTGCACTTAGGATGAAGTGGATGATGGCAATGAAATATCCGATTACAGCAGACAAAAGGATAATGGAGATGATCCCTGAAAGAAACGAGCAAGGTCAGACTCTTTGGAGCAAAACAAATGATGCTGGATCGGATAGAGTAATGGTGTCACCTCTGGCTGTGACGTGGTGGAATAGAAATGGACCAACAACAAGTACAGTCCATTACCCAAAGGTCTATAAAACTTACTTTGAAAAGGTTGAAAGGTTAAAGCATGGAACCTTCGGCCCTGTCCATTTCCGGAATCAGGTTAAGATACGCCGCAGAGTTGACATAAACCCGGGCCATGCAGACCTCAGTGCTAAAGAAGCACAAGACGTCATCATGGAGGTCGTTTTCCCAAATGAAGTCGGAGCCAGAATATTGACATCAGAGTCACAGTTAACAATTACAAAAGAAAAGAAGGAGGAACTCCAGGACTGTAAGATTGCCCCTCTAATGGTGGCATACATGTTGGAGAGAGAACTGGTTCGAAAAACCAGATTCCTGCCAGTAGCTGGCGGAACAAGTAGCGTATATATCGAAGTGTTGCACCTGACTCAAGGAACCTGCTGGGAACAAATGTATACGCCAGGAGGAGAAGTGAGAAATGATGACATTGACCAAAGTTTAATTATTGCTGCCAGAAATATCGTTAGGAGAGCAACAGTATCAGCAGACCCATTGGCTTCACTACTGGAGATGTGCCATAGTACACAGATTGGCGGGATAAGAATGGTAGACATTCTTAGACAGAACCCAACAGAAGAGCAAGCCGTGGATATATGCAAAGCAGCAATGGGTTTAAGAATCAGTTCATCCTTCAGTTTTGGAGGTTTCACTTTCAAAAGGACAAGCGGATCATCTGTCAAAAGAGAAGAGGAAGTGCTCACCGGCAACCTCCAAACATTGAAAATAAGAGTACATGAAGGGTATGAGGAATTCACAATGGTTGGGCGGAGAGCAACAGCCATTCTAAGGAAAGCAACCAGAAGGCTGATCCAATTGATAGTAAGTGGAAAAGACGAGCAGTCAATCGCCGAAGCGATCATAGTGGCAATGGTGTTCTCTCAAGAGGATTGCATGATAAAGGCTGTACGAGGTGATTTAAATTTTGTCAATAGAGCGAATCAGCGGCTCAATCCTATGCATCAGCTCCTGAGGCATTTCCAAAAGGATGCAAAGGTACTATTCCAAAACTGGGGAATTGAACCCATTGACAATGTCATGGGAATGATAGGAATATTGCCTGATATGACTCCCAGCACAGAGATGTCACTAAGAGGAGTGAGGGTCAGTAAAATGGGAGTGGATGAATATTCCAGTACTGAGAGGGTGGTCGTGAGTATTGATCGCTTCTTGAGGGTACGAGACCAGAGAGGAAATGTACTCTTGTCTCCCGAAGAGGTCAGTGAAACACAGGGAACAGAGAAGCTAACGATAACATATTCATCATCCATGATGTGGGAAATTAATGGCCCTGAGTCAGTGCTAGTTAACACATATCAATGGGTCATCAGAAACTGGGAAACTGTGAAGATTCAGTGGTCCCAAGACCCTACAATGCTATACAACAAGATGGAGTTTGAGCCTTTTCAGTCCTTGGTGCCCAAGGCAGCCAGAGGCCAGTACAGTGGATTTGTAAGGACCTTATTCCAGCAGATGCGTGATGTGCTGGGAACCTTTGACACTGTCCAGATAATAAAGCTACTTCCATTTGCAGCAGCACCACCGGAACAGAGTAGGATGCAGTTCTCTTCTCTAACTGTAAACGTAAGGGGTTCAGGAATGAGAATACTTGTGAGAGGAAACTCCCCTGTGTTCAACTATAATAAGGCAACCAAGAGGCTCATAGTCCTTGGAAAGGATGCTGGTGCATTGACAGGAGACCCAGGTGAGGGGACAGCAGGAGTGGAGTCTGCGGTATTGAGAGGGTTCCTAATTCTGGGCAAAGAGGACAAAAGATATGGACCAGCGCTGAGCATCAATGAATTGAGCAATCTTGCGAAAGGGGAGAAGGCTAATGTGTTGATAGGGCAAGGAGACGTGGTGTTGGTGATGAAACGGAAACGGGACTCTAGCATACTTACTGACAGCCAGACAGCGACCAAAAGAATTCGGATGGCCATCAATTA'


def read_fasta(file):
    sample_name = None
    seq = []
    
    for line in file:
        if not (line := line.strip()):
            continue

        if line.startswith('>'):
            if sample_name is not None:
                yield sample_name, ''.join(seq).upper()
            sample_name = line[1:]
            seq.clear()
        else:
            seq.append(line)
    
    if sample_name is not None:
        yield sample_name, ''.join(seq).upper()


def pairwise_alignment(ref_seq, q_seq):
    aligner = PairwiseAligner()
    aligner.match_score = 1
    aligner.mismatch_score = -1
    aligner.open_gap_score = -4
    aligner.extend_gap_score = -1
    ref_al, q_al = aligner.align(ref_seq, q_seq)[0]
    cut_al = cut_alignment(ref_al, q_al)
    return cut_al


def cut_alignment(ref, qry):
    gap_idxs = [i for i, nt in enumerate(ref) if nt == '-']
    return ''.join(nt for i, nt in enumerate(qry) if i not in gap_idxs)


def encode_sequence(seq: str) -> List[bool]:
    T, F = True, False
    encoding_dict = {
        'A': [T, F, F, F], 'C': [F, T, F, F], 'G': [F, F, T, F], 'T': [F, F, F, T], 'U': [F, F, F, T], 'W': [T, F, F, T],
        'S': [F, T, T, F], 'M': [T, T, F, F], 'K': [F, F, T, T], 'R': [T, F, T, F], 'Y': [F, T, F, T], 'B': [F, T, T, T],
        'D': [T, F, T, T], 'H': [T, T, F, T], 'V': [T, T, T, F], 'N': [T, T, T, T], 'Z': [F, F, F, F], '-': [F, F, F, F]
    }
    return list(itertools.chain(*[encoding_dict[base] for base in seq]))
