# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['protogen']

package_data = \
{'': ['*']}

install_requires = \
['protobuf>=3.17.3,<4.0.0']

setup_kwargs = {
    'name': 'protogen',
    'version': '0.1.0',
    'description': 'protogen makes writing protoc plugins easy.',
    'long_description': '# `protogen`\n\nPackage `protogen` makes writing `protoc` plugins easier.\nWorking with the raw protobuf descriptor messages can be cumbersome.\n`protogen` resolves and links the dependencies and references between the raw Protobuf descriptors and turns them into their corresponding `protogen` classes that are easier to work with.\nIt also provides mechanisms that are espacially useful to generate Python code like dealing with Python imports.\n\n## Installation\n\nPackage `protogen` is available via `pip`. To install run:\n\n```\npip install protogen\n```\n\n## API\n\nMost classes in `protogen` are simply replacements of their corresponding Protobuf descriptors: `protogen.File` represents a FileDescriptor, `protogen.Message` a Descriptor, `protogen.Field` a FieldDescriptor and so on. They should be self explanatory. You can read their docstrings for more information about them.\n\nThe classes `protogen.Options`, `protogen.Plugin` and `protogen.GeneratedFile` make up a framework to generate files.\nYou can see these in action in the following example plugin:\n\n```python\n#!/usr/bin/env python\n"""An example plugin."""\n\nimport protogen\n\ndef generate(gen: protogen.Plugin):\n    for f in gen.files_to_generate:\n        g = gen.new_generated_file(\n            f.proto.name.replace(".proto", ".py"), \n            f.py_import_path,\n        )\n        g.P("# Generated code ahead.")\n        g.P()\n        g.print_imports()\n        g.P()\n        for m in f.message:\n            g.P("class ", m.py_ident, ":")\n            for ff in m.fields:\n                # ...\n        for s in f.services:\n            g.P("class ", s.py_ident, ":")\n            for m in f.methods:\n                g.P("  def ", m.py_name, "(request):")\n                g.P("    pass")\n\nif __name__ == "__main__":\n    opts = protogen.Options()\n    opts.run(generate)\n```\n\n## class `protogen.Options`\n\nThe `protogen.Options` class can be used to specify options for the resolution process (resolution from plain proto descriptors to `protogen` classes).\n`Option.run(f: func(Plugin))` waits for `protoc` to write the CodeGeneratorRequest to `stdin`, resolves the descriptors contained in it to their corresponding `protogen` classes and initializes a new `Plugin` with the resolved classes.  \n`f` is then called with the `Plugin` as argument.\n\nOnce `f` returns, `Options` will collect the CodeGeneratorResponse from the `Plugin` that contains the all created `GeneratedFile`s and write it to `stdout` for `protoc` to pick it up.\n`protoc` writes the generated files to disk.\n\n## class `protogen.Plugin`\n\nThe `Plugin` class holds the files code generation is requested for in the `Plugin.files_to_generate` attribute. These are the files that were provided as command line arguments to `protoc`.\nAny options/parameters passed to the plugin via the `protoc --plugin_opt=<param>` command line flag are accessible via `Plugin.parameter`.\nWith `Plugin.new_generated_file` a new `GeneratedFile` gets created that is automatically added to the CodeGeneratorResponse of the plugin.\nTypically, but not necessarily, one file for each file in `Plugin.files_to_generate` is created.\n\n## class `protogen.GeneratedFile`\n\nThe `GeneratedFile` is just a buffer you can add lines to using the `g.P` (print) method.\nA `GeneratedFile` is created with `Plugin.new_generated_file(filename, py_import_path)`.\nThe `filename` is obviously the name of the file to be created.\nThe `py_import_path` is used for *import resolution*.\n\nNote that the following assumes the plugin generates Python code. For other kinds of plugins, the following is not relevant:\n\nIt is often necessary to import Python identifiers that are defined in different Python modules.\nFor example, a Protobuf messages might reference `google.protobuf.Timestamp` in one of its fields.\nThe corresponding Python class `google.protobuf.timestamp_pb2.Timestamp` needs to be imported before its mentioned in the generated code.\n\nThe `protogen.PyImportPath` class represent a Python import path. Is just a wrapper around an import path (for example `"google.protobuf.timestamp_pb2"`).\nThe `PyIdent` class represent a Python identifier. It holds a `PyImportPath` together with a name (e.g. a class name like `"Timestamp"`).\n\nThe `protogen.GeneratedFile` provides mechanisms to handle Python imports.\nInternally it maintains a list of `PyImportPath`s that it needs to import.\n`PyImportPaths` might be added to this list implictly when calling `GeneratedFile.P(*args)` or rather explicitly when calling `GeneratedFile.qualified_py_ident(PyIdent)`.\nWhen any of the arguments to `GeneratedFile.P` is a `protogen.PyIdent`, the `py_import_path` of the `GeneratedFile` gets compared to the arguments `PyIdent.py_import_path`. \n\nIf they are from different Python modules, the arguments import path will be added to the list of imports and the fully qualified name of the `PyIdent` will be printed. \n\nIf both files are from the same `PyImportPath`, then the import path is not added to the list of imports. In that case it is sufficient to reference the `PyIdent` by its simple name (e.g. `Timestamp`), thus only the `PyIdent.py_name` will be printed.\n\nTo place the import statements in the buffer of the `GeneratedFile` use `GeneratedFile.print_imports`. This will put a line `"import <path>"` for each `PyImportPath` that the generated file needs to import (e.g `"import google.protobuf.timestamp_pb2"`) in the buffer.\n\nThe following example shows how the `GeneratedFile.P` function behaves for different `PyImportPaths`::\n\n```python\n# g is of type protogen.GeneratedFile\n# message_a and message_b are of type protogen.Message\n\n>>> g.py_import_path\n{ "mypackage.mymodule" }\n\n>>> message_a.py_ident\n{ py_import_path: "google.protobuf.timestamp_pb2", py_name: "Timestamp" }\n>>> g.P("hello ", message_a.py_ident) \n# adds "hello google.protobuf.timestamp_pb2.Timestamp" to g\'s line buffer and "google.protobuf.timestamp_pb2" to the imports\n\n>>> message_b.py_ident\n{ py_import_path: "mypackage.mymodule", py_name: "MyMessage" }\n>>> g.P("hello ", message_b.py_ident) \n# adds "hello MyMessage" to g\'s line buffer (and nothing to the imports)\n```\n\nNote that you can provide a custom `py_import_func` in the `Options` constructor.\nThis function is used in the resolution process to calculate the `PyImportPath` for `protogen.File`s.\n`protogen.Message`s, `protogen.Service`s and `protogen.Enum`s inherit the `PyImportPath` (that is part of their `PyIdent`) from the file they are defined in.\nBy default the `protogen.default_py_import_func` is used. \nIt is compatible with the style of the offical Python `protoc` plugin that generates for each input file `path/to/file.proto` a corresponding `path/to/file_pb2.py` file.\n\nFor example, assume you know that code generation for proto definitions that are part of the `mypackage.**` proto package happens with a `protoc` plugin that generates one `.py` file per proto package. \nThat `plugin` also omits the `_pb2` suffix.\nFor the proto package `mypackage.api.a`, that might contain any number of files, it creates a `mypackage/api/a.py` file.\nFor the proto package `mypackage.api.b`, a `mypackage/api/b.py` file.\n\nA `py_import_func` describing this would be:\n\n```python\ndef py_import_func(\n    proto_filename: str, \n    proto_package:str,\n) -> protogen.PyImportPath:\n    if proto_package.split(".")[0] == "mypackage":\n        # Python import path is simply the package name.\n        return protogen.PyImportPath(proto_package) \n    # For every other package, assume its generated with the offical Python plugin.\n    return protogen.default_py_import_func(proto_filename, proto_package)\n```\n\n# Misc\n\n## What is a protoc plugin anyway?\n\n`protoc`, the **Proto**buf **c**ompiler, is used to generate code derived from Protobuf definitions (`.proto` files).\nUnder the hood, `protoc`\'s job is to read and parse the definitions into their *Descriptor* types (see [google/protobuf/descriptor.proto](https://github.com/protocolbuffers/protobuf/blob/4f49062a95f18a6c7e21ba17715a2b0a4608151a/src/google/protobuf/descriptor.proto)).\nWhen `protoc` is run (with a plugin) it creates a CodeGeneratorRequest (see [google/protobuf/compiler/plugin.proto#L68](https://github.com/protocolbuffers/protobuf/blob/4f49062a95f18a6c7e21ba17715a2b0a4608151a/src/google/protobuf/compiler/plugin.proto#L68)) that contains the descriptors for the files to generate and everything they import and passes it to the plugin via `stdin`.\n\nA *protoc plugin* is an executable. It reads the CodeGeneratorRequest from `stdin` and returns a CodeGeneratorResponse (see [google/protobuf/compiler/plugin.proto#L99](https://github.com/protocolbuffers/protobuf/blob/4f49062a95f18a6c7e21ba17715a2b0a4608151a/src/google/protobuf/compiler/plugin.proto#L99)) via `stdout`.\nThe plugin can use the descriptors from the CodeGeneratorRequest to create output files (in memory).\nIt returns these output files (consisting of name and content as string) in the CodeGeneratorResponse to `protoc`.\n\n`protoc` then writes these files to disk.\n\n## Run `protoc` with your plugin\n\nAssume you have an executable plugin under `path/to/plugin/main.py`.\nYou can invoke it via:\n\n```\nprotoc \n    --plugin=protoc-gen-myplugin=path/to/plugin/main.py \\\n    --plugin_out=./output_root \\\n    myproto.proto myproto2.proto\n```\n\nCaveats:\n- you must use the `--plugin=protoc-gen-<plugin_name>` prefix, otherwise `protoc` fails with "plugin not executable"\n- your plugin must be executable (`chmod +x path/to/plugin/main.py` and put a `#!/usr/bin/env python` at the top of the file)\n\n# See also\n\n- if you want to write protoc plugins with JavaScript/TypeScript: [github.com/fischor/protogen-javascript](https://github.com/fischor/protogen-javascript)\n- if you want to write protoc plugins with Golang: [google.golang.org/protobuf/compiler/protogen](https://google.golang.org/protobuf/compiler/protogen)\n\n# Credits\n\nThis package is inspired by the [google.golang.org/protobuf/compiler/protogen Golang](https://pkg.go.dev/google.golang.org/protobuf@v1.27.1/compiler/protogen) package.\n',
    'author': 'fischor',
    'author_email': 'fischor.sh@gmail.com',
    'maintainer': 'fischor',
    'maintainer_email': 'fischor.sh@gmail.com',
    'url': 'https://github.com/fischor/protogen-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>3.6.2,<4.0',
}


setup(**setup_kwargs)
