"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComputeEnvironment = exports.AllocationStrategy = exports.ComputeResourceType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const batch_generated_1 = require("./batch.generated");
/**
 * (experimental) Property to specify if the compute environment uses On-Demand or SpotFleet compute resources.
 *
 * @experimental
 */
var ComputeResourceType;
(function (ComputeResourceType) {
    ComputeResourceType["ON_DEMAND"] = "EC2";
    ComputeResourceType["SPOT"] = "SPOT";
})(ComputeResourceType = exports.ComputeResourceType || (exports.ComputeResourceType = {}));
/**
 * (experimental) Properties for how to prepare compute resources that are provisioned for a compute environment.
 *
 * @experimental
 */
var AllocationStrategy;
(function (AllocationStrategy) {
    AllocationStrategy["BEST_FIT"] = "BEST_FIT";
    AllocationStrategy["BEST_FIT_PROGRESSIVE"] = "BEST_FIT_PROGRESSIVE";
    AllocationStrategy["SPOT_CAPACITY_OPTIMIZED"] = "SPOT_CAPACITY_OPTIMIZED";
})(AllocationStrategy = exports.AllocationStrategy || (exports.AllocationStrategy = {}));
/**
 * (experimental) Batch Compute Environment.
 *
 * Defines a batch compute environment to run batch jobs on.
 *
 * @experimental
 */
class ComputeEnvironment extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props = { enabled: true, managed: true }) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.computeEnvironmentName,
        });
        this.validateProps(props);
        const spotFleetRole = this.getSpotFleetRole(props);
        let computeResources;
        // Only allow compute resources to be set when using MANAGED type
        if (props.computeResources && this.isManaged(props)) {
            computeResources = {
                allocationStrategy: props.computeResources.allocationStrategy
                    || (props.computeResources.type === ComputeResourceType.SPOT
                        ? AllocationStrategy.SPOT_CAPACITY_OPTIMIZED
                        : AllocationStrategy.BEST_FIT),
                bidPercentage: props.computeResources.bidPercentage,
                desiredvCpus: props.computeResources.desiredvCpus,
                ec2KeyPair: props.computeResources.ec2KeyPair,
                imageId: props.computeResources.image && props.computeResources.image.getImage(this).imageId,
                instanceRole: props.computeResources.instanceRole
                    ? props.computeResources.instanceRole
                    : new iam.CfnInstanceProfile(this, 'Instance-Profile', {
                        roles: [new iam.Role(this, 'Ecs-Instance-Role', {
                                assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                                managedPolicies: [
                                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'),
                                ],
                            }).roleName],
                    }).attrArn,
                instanceTypes: this.buildInstanceTypes(props.computeResources.instanceTypes),
                launchTemplate: props.computeResources.launchTemplate,
                maxvCpus: props.computeResources.maxvCpus || 256,
                minvCpus: props.computeResources.minvCpus || 0,
                placementGroup: props.computeResources.placementGroup,
                securityGroupIds: this.buildSecurityGroupIds(props.computeResources.vpc, props.computeResources.securityGroups),
                spotIamFleetRole: spotFleetRole === null || spotFleetRole === void 0 ? void 0 : spotFleetRole.roleArn,
                subnets: props.computeResources.vpc.selectSubnets(props.computeResources.vpcSubnets).subnetIds,
                tags: props.computeResources.computeResourcesTags,
                type: props.computeResources.type || ComputeResourceType.ON_DEMAND,
            };
        }
        const computeEnvironment = new batch_generated_1.CfnComputeEnvironment(this, 'Resource', {
            computeEnvironmentName: this.physicalName,
            computeResources,
            serviceRole: (_c = (_b = props.serviceRole) === null || _b === void 0 ? void 0 : _b.roleArn) !== null && _c !== void 0 ? _c : new iam.Role(this, 'Resource-Service-Instance-Role', {
                managedPolicies: [
                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBatchServiceRole'),
                ],
                assumedBy: new iam.ServicePrincipal('batch.amazonaws.com'),
            }).roleArn,
            state: props.enabled === undefined ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            type: this.isManaged(props) ? 'MANAGED' : 'UNMANAGED',
        });
        if (props.computeResources && props.computeResources.vpc) {
            this.node.addDependency(props.computeResources.vpc);
        }
        this.computeEnvironmentArn = this.getResourceArnAttribute(computeEnvironment.ref, {
            service: 'batch',
            resource: 'compute-environment',
            resourceName: this.physicalName,
        });
        this.computeEnvironmentName = this.getResourceNameAttribute(computeEnvironment.ref);
    }
    /**
     * (experimental) Fetches an existing batch compute environment by its amazon resource name.
     *
     * @experimental
     */
    static fromComputeEnvironmentArn(scope, id, computeEnvironmentArn) {
        const stack = core_1.Stack.of(scope);
        const computeEnvironmentName = stack.parseArn(computeEnvironmentArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.computeEnvironmentArn = computeEnvironmentArn;
                this.computeEnvironmentName = computeEnvironmentName;
            }
        }
        return new Import(scope, id);
    }
    isManaged(props) {
        var _b;
        return (_b = props.managed) !== null && _b !== void 0 ? _b : true;
    }
    /**
     * Validates the properties provided for a new batch compute environment.
     */
    validateProps(props) {
        if (props === undefined) {
            return;
        }
        if (!this.isManaged(props) && props.computeResources !== undefined) {
            throw new Error('It is not allowed to set computeResources on an AWS unmanaged compute environment');
        }
        if (this.isManaged(props) && props.computeResources === undefined) {
            throw new Error('computeResources is missing but required on a managed compute environment');
        }
        // Setting a bid percentage is only allowed on SPOT resources +
        // Cannot use SPOT_CAPACITY_OPTIMIZED when using ON_DEMAND
        if (props.computeResources) {
            if (props.computeResources.type === ComputeResourceType.ON_DEMAND) {
                // VALIDATE FOR ON_DEMAND
                // Bid percentage is not allowed
                if (props.computeResources.bidPercentage !== undefined) {
                    throw new Error('Setting the bid percentage is only allowed for SPOT type resources on a batch compute environment');
                }
                // SPOT_CAPACITY_OPTIMIZED allocation is not allowed
                if (props.computeResources.allocationStrategy && props.computeResources.allocationStrategy === AllocationStrategy.SPOT_CAPACITY_OPTIMIZED) {
                    throw new Error('The SPOT_CAPACITY_OPTIMIZED allocation strategy is only allowed if the environment is a SPOT type compute environment');
                }
            }
            else {
                // VALIDATE FOR SPOT
                // Bid percentage must be from 0 - 100
                if (props.computeResources.bidPercentage !== undefined &&
                    (props.computeResources.bidPercentage < 0 || props.computeResources.bidPercentage > 100)) {
                    throw new Error('Bid percentage can only be a value between 0 and 100');
                }
            }
            if (props.computeResources.minvCpus) {
                // minvCpus cannot be less than 0
                if (props.computeResources.minvCpus < 0) {
                    throw new Error('Minimum vCpus for a batch compute environment cannot be less than 0');
                }
                // minvCpus cannot exceed max vCpus
                if (props.computeResources.maxvCpus &&
                    props.computeResources.minvCpus > props.computeResources.maxvCpus) {
                    throw new Error('Minimum vCpus cannot be greater than the maximum vCpus');
                }
            }
        }
    }
    buildInstanceTypes(instanceTypes) {
        if (instanceTypes === undefined) {
            return [
                'optimal',
            ];
        }
        return instanceTypes.map((type) => type.toString());
    }
    buildSecurityGroupIds(vpc, securityGroups) {
        if (securityGroups === undefined) {
            return [
                new ec2.SecurityGroup(this, 'Resource-Security-Group', { vpc }).securityGroupId,
            ];
        }
        return securityGroups.map((group) => group.securityGroupId);
    }
    /**
     * Generates an AWS IAM role for provisioning spotfleet resources
     * if the allocation strategy is set to BEST_FIT or not defined.
     *
     * @param props - the compute environment construct properties
     */
    getSpotFleetRole(props) {
        var _b;
        if (((_b = props.computeResources) === null || _b === void 0 ? void 0 : _b.allocationStrategy) && props.computeResources.allocationStrategy !== AllocationStrategy.BEST_FIT) {
            return undefined;
        }
        if (props.computeResources) {
            if (props.computeResources.spotFleetRole) {
                return props.computeResources.spotFleetRole;
            }
            else if (props.computeResources.type === ComputeResourceType.SPOT) {
                return iam.Role.fromRoleArn(this, 'Resource-SpotFleet-Role', `arn:${this.stack.partition}:iam::${this.stack.account}:role/aws-service-role/spotfleet.amazonaws.com/AWSServiceRoleForEC2SpotFleet`);
            }
        }
        return undefined;
    }
}
exports.ComputeEnvironment = ComputeEnvironment;
_a = JSII_RTTI_SYMBOL_1;
ComputeEnvironment[_a] = { fqn: "@aws-cdk/aws-batch.ComputeEnvironment", version: "1.108.1" };
//# sourceMappingURL=data:application/json;base64,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