import re
from argparse import ArgumentParser, FileType, Action
from json import load
from logging import DEBUG, INFO, debug, info, error
from os.path import split, dirname
from junitparser import JUnitXml, TestCase, TestSuite

from unitrail.logging_utils import initialize_logger
from unitrail.results_formatter import push_results
from unitrail.testrail_api import APIClient
from unitrail.testrail_filter import (apply_case_filter, apply_section_filter,
                                      collect_children_sections, create_testrun)

from unitrail.__utils import (collect_full_mapping,
                              read_test_results_from_report, map_results_to_cases)


def __main(options):
    try:
        initialize_logger(loglevel=DEBUG if options.verbose else INFO)
        debug('Verbose logging is enabled')
        mapping = collect_full_mapping(options)
        debug(mapping)
        project_id = mapping["project"]
        client = APIClient(project_id, url=options.server, user=options.username,
                           password=options.password, notls=options.noverify, cacert=options.ca)

        if not options.testrun:
            info('Creating new test run since there is no testrun provided')
            testrun = create_testrun(client, mapping)
        else:
            info('Loading testrun {}'.format(options.testrun))
            testrun = client.get_run(options.testrun)

        testrun_id = testrun['id']
        info(
            ' -> Working with testrun ({}) "{}"'.format(testrun_id, testrun['name']))
        info(' -> See it in browser {}'.format(testrun['url']))

        tests = client.get_tests(testrun_id)

        test_results = read_test_results_from_report(options.reports)

        info('Mapping execution results to TestRail cases')
        mapped_results = map_results_to_cases(
            test_results, tests, mapping['mapping'])
        info('Pushing execution results to TestRail...')
        push_results(mapped_results, client)
        info('Test execution results are pushed to test run {}'.format(
            testrun['url']))
        testrail_results = client.get_run(testrun_id)
        if options.leaveopen:
            info('Test run will not be closed')
        else:
            if options.forceclose:
                info('Closing testrun due to forced close option set')
                client.close_run(testrun_id)
            elif testrail_results['failed_count'] == 0:
                info('Since there are no failed tests, test run will be closed..')
                client.close_run(testrun_id)

    except Exception as err:
        error(err)


class StoreNameValuePair(Action):
    def __call__(self, parser, namespace, values, option_string=None):
        if not getattr(namespace, 'defines'):
            setattr(namespace, 'defines', {})
        defines = getattr(namespace, 'defines')
        if type(values) != list:
            values = [values]
        for value in values:
            k, v = value.split('=')
            defines[k] = v


def make_parser():
    parser = ArgumentParser(
        prog='unitrail', description='Fill test run in Testrail using xUnit XML report generated by automated tests')
    parser.set_defaults(func=main)

    parser.add_argument(
        '-r', '--reports',
        required=True,
        nargs="+",
        type=str,
        help='xUnit reports to handle')

    parser.add_argument(
        '-v', '--verbose',
        action='store_true',
        help='Make logs verbose')

    parser.add_argument(
        '-s', '--server',
        type=str,
        default='http://testrail/index.php?/api/v2/',
        help='Set TestRail server address (default is http://testrail/index.php?/api/v2/)')

    parser.add_argument(
        '-u', '--username',
        type=str,
        required=True,
        help='Username to authenticate in TestRail')

    parser.add_argument(
        '-p', '--password',
        type=str,
        required=True,
        help='Password or API key to authenticate in TestRail')

    parser.add_argument(
        '-m', '--mapping',
        type=FileType('r', encoding='UTF-8'),
        required=True,
        help='JSON file with mapping of the testcases in report to scenarios in testrail')

    parser.add_argument(
        '-t', '--testrun',
        type=str,
        help='Existing testrun ID, if not exists - new one will be created'
    )

    parser.add_argument(
        '-c', '--ca',
        type=str,
        help='Path to the root CA for self-signed certificates'
    )

    parser.add_argument(
        '-k', '--noverify',
        action='store_true',
        help='Disable TLS verification (not recommended)'
    )

    parser.add_argument(
        '-O', '--leaveopen',
        action='store_true',
        help='Do not close test run after execution. Defaults to false'
    )

    parser.add_argument(
        '-C', '--forceclose',
        action='store_true',
        help='Close test run in the end, nevermind of tests execution results. Defaults to false'
    )

    parser.add_argument(
        '-D', '--defines',
        action=StoreNameValuePair,
        nargs='+',
        required=False,
        default={},
        help='Define mapping parameters in dynamic from commandline'
    )
    return parser


def main():
    parser = make_parser()
    __main(parser.parse_args())


if __name__ == '__main__':
    main()
